// Copyright (C) 2018 ycmd contributors
//
// This file is part of ycmd.
//
// ycmd is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// ycmd is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with ycmd.  If not, see <http://www.gnu.org/licenses/>.

#include "Character.h"
#include "CharacterRepository.h"
#include "CodePoint.h"
#include "TestUtils.h"

#include <array>
#include <gtest/gtest.h>
#include <gmock/gmock.h>

using ::testing::TestWithParam;
using ::testing::ValuesIn;

namespace YouCompleteMe {

struct NormalizationTuple {
  const char* source;
  const char* nfc;
  const char* nfd;
  const char* nfkc;
  const char* nfkd;
};


std::ostream& operator<<( std::ostream& os,
                          const NormalizationTuple &tuple ) {
  os << "{ " << PrintToString( tuple.source ) << ", "
             << PrintToString( tuple.nfc    ) << ", "
             << PrintToString( tuple.nfd    ) << ", "
             << PrintToString( tuple.nfkc   ) << ", "
             << PrintToString( tuple.nfkd   ) << " }";
  return os;
}


class NormalizationTest : public TestWithParam< NormalizationTuple > {
protected:
  NormalizationTest()
    : repo_( CharacterRepository::Instance() ) {
  }

  virtual void SetUp() {
    repo_.ClearCharacters();
    tuple_ = GetParam();
  }

  CharacterRepository &repo_;
  NormalizationTuple tuple_;
};


TEST_P( NormalizationTest, NormalizationFormDecompositionIsConform ) {
  EXPECT_THAT( Character( tuple_.source ).Normal(), Equals( tuple_.nfd  ) );
  EXPECT_THAT( Character( tuple_.nfc    ).Normal(), Equals( tuple_.nfd  ) );
  EXPECT_THAT( Character( tuple_.nfd    ).Normal(), Equals( tuple_.nfd  ) );
  EXPECT_THAT( Character( tuple_.nfkc   ).Normal(), Equals( tuple_.nfkd ) );
  EXPECT_THAT( Character( tuple_.nfkd   ).Normal(), Equals( tuple_.nfkd ) );
}


// Tests generated from
// https://unicode.org/Public/UCD/latest/ucd/NormalizationTest.txt
const NormalizationTuple tests[] = {
  { "Ḋ", "Ḋ", "Ḋ", "Ḋ", "Ḋ" },
  { "Ḍ", "Ḍ", "Ḍ", "Ḍ", "Ḍ" },
  { "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇" },
  { "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇" },
  { "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇" },
  { "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇", "Ḍ̇" },
  { "Ḋ̛", "Ḋ̛", "Ḋ̛", "Ḋ̛", "Ḋ̛" },
  { "Ḍ̛", "Ḍ̛", "Ḍ̛", "Ḍ̛", "Ḍ̛" },
  { "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇" },
  { "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇" },
  { "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇" },
  { "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇", "Ḍ̛̇" },
  { "È", "È", "È", "È", "È" },
  { "Ē", "Ē", "Ē", "Ē", "Ē" },
  { "È", "È", "È", "È", "È" },
  { "Ē", "Ē", "Ē", "Ē", "Ē" },
  { "Ḕ", "Ḕ", "Ḕ", "Ḕ", "Ḕ" },
  { "Ḕ", "Ḕ", "Ḕ", "Ḕ", "Ḕ" },
  { "Ḕ̄", "Ḕ̄", "Ḕ̄", "Ḕ̄", "Ḕ̄" },
  { "Ḕ", "Ḕ", "Ḕ", "Ḕ", "Ḕ" },
  { "È̄", "È̄", "È̄", "È̄", "È̄" },
  { "ֱָֹ֑׃ְ֬֟", "ֱָֹ֑׃ְ֬֟", "ֱָֹ֑׃ְ֬֟", "ֱָֹ֑׃ְ֬֟", "ֱָֹ֑׃ְ֬֟" },
  { "ְַּ֥֒׀֭ׄ", "ְַּ֥֒׀֭ׄ", "ְַּ֥֒׀֭ׄ", "ְַּ֥֒׀֭ׄ", "ְַּ֥֒׀֭ׄ" },
  { "ᄀ각", "ᄀ각", "ᄀ각", "ᄀ각", "ᄀ각" },
  { "ᄀ각ᆨ", "ᄀ각ᆨ", "ᄀ각ᆨ", "ᄀ각ᆨ", "ᄀ각ᆨ" },
  { " ", " ", " ", " ", " " },
  { "¨", "¨", "¨", " ̈", " ̈" },
  { "ª", "ª", "ª", "a", "a" },
  { "¯", "¯", "¯", " ̄", " ̄" },
  { "²", "²", "²", "2", "2" },
  { "³", "³", "³", "3", "3" },
  { "´", "´", "´", " ́", " ́" },
  { "µ", "µ", "µ", "μ", "μ" },
  { "¸", "¸", "¸", " ̧", " ̧" },
  { "¹", "¹", "¹", "1", "1" },
  { "º", "º", "º", "o", "o" },
  { "¼", "¼", "¼", "1⁄4", "1⁄4" },
  { "½", "½", "½", "1⁄2", "1⁄2" },
  { "¾", "¾", "¾", "3⁄4", "3⁄4" },
  { "À", "À", "À", "À", "À" },
  { "Á", "Á", "Á", "Á", "Á" },
  { "Â", "Â", "Â", "Â", "Â" },
  { "Ã", "Ã", "Ã", "Ã", "Ã" },
  { "Ä", "Ä", "Ä", "Ä", "Ä" },
  { "Å", "Å", "Å", "Å", "Å" },
  { "Ç", "Ç", "Ç", "Ç", "Ç" },
  { "È", "È", "È", "È", "È" },
  { "É", "É", "É", "É", "É" },
  { "Ê", "Ê", "Ê", "Ê", "Ê" },
  { "Ë", "Ë", "Ë", "Ë", "Ë" },
  { "Ì", "Ì", "Ì", "Ì", "Ì" },
  { "Í", "Í", "Í", "Í", "Í" },
  { "Î", "Î", "Î", "Î", "Î" },
  { "Ï", "Ï", "Ï", "Ï", "Ï" },
  { "Ñ", "Ñ", "Ñ", "Ñ", "Ñ" },
  { "Ò", "Ò", "Ò", "Ò", "Ò" },
  { "Ó", "Ó", "Ó", "Ó", "Ó" },
  { "Ô", "Ô", "Ô", "Ô", "Ô" },
  { "Õ", "Õ", "Õ", "Õ", "Õ" },
  { "Ö", "Ö", "Ö", "Ö", "Ö" },
  { "Ù", "Ù", "Ù", "Ù", "Ù" },
  { "Ú", "Ú", "Ú", "Ú", "Ú" },
  { "Û", "Û", "Û", "Û", "Û" },
  { "Ü", "Ü", "Ü", "Ü", "Ü" },
  { "Ý", "Ý", "Ý", "Ý", "Ý" },
  { "à", "à", "à", "à", "à" },
  { "á", "á", "á", "á", "á" },
  { "â", "â", "â", "â", "â" },
  { "ã", "ã", "ã", "ã", "ã" },
  { "ä", "ä", "ä", "ä", "ä" },
  { "å", "å", "å", "å", "å" },
  { "ç", "ç", "ç", "ç", "ç" },
  { "è", "è", "è", "è", "è" },
  { "é", "é", "é", "é", "é" },
  { "ê", "ê", "ê", "ê", "ê" },
  { "ë", "ë", "ë", "ë", "ë" },
  { "ì", "ì", "ì", "ì", "ì" },
  { "í", "í", "í", "í", "í" },
  { "î", "î", "î", "î", "î" },
  { "ï", "ï", "ï", "ï", "ï" },
  { "ñ", "ñ", "ñ", "ñ", "ñ" },
  { "ò", "ò", "ò", "ò", "ò" },
  { "ó", "ó", "ó", "ó", "ó" },
  { "ô", "ô", "ô", "ô", "ô" },
  { "õ", "õ", "õ", "õ", "õ" },
  { "ö", "ö", "ö", "ö", "ö" },
  { "ù", "ù", "ù", "ù", "ù" },
  { "ú", "ú", "ú", "ú", "ú" },
  { "û", "û", "û", "û", "û" },
  { "ü", "ü", "ü", "ü", "ü" },
  { "ý", "ý", "ý", "ý", "ý" },
  { "ÿ", "ÿ", "ÿ", "ÿ", "ÿ" },
  { "Ā", "Ā", "Ā", "Ā", "Ā" },
  { "ā", "ā", "ā", "ā", "ā" },
  { "Ă", "Ă", "Ă", "Ă", "Ă" },
  { "ă", "ă", "ă", "ă", "ă" },
  { "Ą", "Ą", "Ą", "Ą", "Ą" },
  { "ą", "ą", "ą", "ą", "ą" },
  { "Ć", "Ć", "Ć", "Ć", "Ć" },
  { "ć", "ć", "ć", "ć", "ć" },
  { "Ĉ", "Ĉ", "Ĉ", "Ĉ", "Ĉ" },
  { "ĉ", "ĉ", "ĉ", "ĉ", "ĉ" },
  { "Ċ", "Ċ", "Ċ", "Ċ", "Ċ" },
  { "ċ", "ċ", "ċ", "ċ", "ċ" },
  { "Č", "Č", "Č", "Č", "Č" },
  { "č", "č", "č", "č", "č" },
  { "Ď", "Ď", "Ď", "Ď", "Ď" },
  { "ď", "ď", "ď", "ď", "ď" },
  { "Ē", "Ē", "Ē", "Ē", "Ē" },
  { "ē", "ē", "ē", "ē", "ē" },
  { "Ĕ", "Ĕ", "Ĕ", "Ĕ", "Ĕ" },
  { "ĕ", "ĕ", "ĕ", "ĕ", "ĕ" },
  { "Ė", "Ė", "Ė", "Ė", "Ė" },
  { "ė", "ė", "ė", "ė", "ė" },
  { "Ę", "Ę", "Ę", "Ę", "Ę" },
  { "ę", "ę", "ę", "ę", "ę" },
  { "Ě", "Ě", "Ě", "Ě", "Ě" },
  { "ě", "ě", "ě", "ě", "ě" },
  { "Ĝ", "Ĝ", "Ĝ", "Ĝ", "Ĝ" },
  { "ĝ", "ĝ", "ĝ", "ĝ", "ĝ" },
  { "Ğ", "Ğ", "Ğ", "Ğ", "Ğ" },
  { "ğ", "ğ", "ğ", "ğ", "ğ" },
  { "Ġ", "Ġ", "Ġ", "Ġ", "Ġ" },
  { "ġ", "ġ", "ġ", "ġ", "ġ" },
  { "Ģ", "Ģ", "Ģ", "Ģ", "Ģ" },
  { "ģ", "ģ", "ģ", "ģ", "ģ" },
  { "Ĥ", "Ĥ", "Ĥ", "Ĥ", "Ĥ" },
  { "ĥ", "ĥ", "ĥ", "ĥ", "ĥ" },
  { "Ĩ", "Ĩ", "Ĩ", "Ĩ", "Ĩ" },
  { "ĩ", "ĩ", "ĩ", "ĩ", "ĩ" },
  { "Ī", "Ī", "Ī", "Ī", "Ī" },
  { "ī", "ī", "ī", "ī", "ī" },
  { "Ĭ", "Ĭ", "Ĭ", "Ĭ", "Ĭ" },
  { "ĭ", "ĭ", "ĭ", "ĭ", "ĭ" },
  { "Į", "Į", "Į", "Į", "Į" },
  { "į", "į", "į", "į", "į" },
  { "İ", "İ", "İ", "İ", "İ" },
  { "Ĳ", "Ĳ", "Ĳ", "IJ", "IJ" },
  { "ĳ", "ĳ", "ĳ", "ij", "ij" },
  { "Ĵ", "Ĵ", "Ĵ", "Ĵ", "Ĵ" },
  { "ĵ", "ĵ", "ĵ", "ĵ", "ĵ" },
  { "Ķ", "Ķ", "Ķ", "Ķ", "Ķ" },
  { "ķ", "ķ", "ķ", "ķ", "ķ" },
  { "Ĺ", "Ĺ", "Ĺ", "Ĺ", "Ĺ" },
  { "ĺ", "ĺ", "ĺ", "ĺ", "ĺ" },
  { "Ļ", "Ļ", "Ļ", "Ļ", "Ļ" },
  { "ļ", "ļ", "ļ", "ļ", "ļ" },
  { "Ľ", "Ľ", "Ľ", "Ľ", "Ľ" },
  { "ľ", "ľ", "ľ", "ľ", "ľ" },
  { "Ŀ", "Ŀ", "Ŀ", "L·", "L·" },
  { "ŀ", "ŀ", "ŀ", "l·", "l·" },
  { "Ń", "Ń", "Ń", "Ń", "Ń" },
  { "ń", "ń", "ń", "ń", "ń" },
  { "Ņ", "Ņ", "Ņ", "Ņ", "Ņ" },
  { "ņ", "ņ", "ņ", "ņ", "ņ" },
  { "Ň", "Ň", "Ň", "Ň", "Ň" },
  { "ň", "ň", "ň", "ň", "ň" },
  { "ŉ", "ŉ", "ŉ", "ʼn", "ʼn" },
  { "Ō", "Ō", "Ō", "Ō", "Ō" },
  { "ō", "ō", "ō", "ō", "ō" },
  { "Ŏ", "Ŏ", "Ŏ", "Ŏ", "Ŏ" },
  { "ŏ", "ŏ", "ŏ", "ŏ", "ŏ" },
  { "Ő", "Ő", "Ő", "Ő", "Ő" },
  { "ő", "ő", "ő", "ő", "ő" },
  { "Ŕ", "Ŕ", "Ŕ", "Ŕ", "Ŕ" },
  { "ŕ", "ŕ", "ŕ", "ŕ", "ŕ" },
  { "Ŗ", "Ŗ", "Ŗ", "Ŗ", "Ŗ" },
  { "ŗ", "ŗ", "ŗ", "ŗ", "ŗ" },
  { "Ř", "Ř", "Ř", "Ř", "Ř" },
  { "ř", "ř", "ř", "ř", "ř" },
  { "Ś", "Ś", "Ś", "Ś", "Ś" },
  { "ś", "ś", "ś", "ś", "ś" },
  { "Ŝ", "Ŝ", "Ŝ", "Ŝ", "Ŝ" },
  { "ŝ", "ŝ", "ŝ", "ŝ", "ŝ" },
  { "Ş", "Ş", "Ş", "Ş", "Ş" },
  { "ş", "ş", "ş", "ş", "ş" },
  { "Š", "Š", "Š", "Š", "Š" },
  { "š", "š", "š", "š", "š" },
  { "Ţ", "Ţ", "Ţ", "Ţ", "Ţ" },
  { "ţ", "ţ", "ţ", "ţ", "ţ" },
  { "Ť", "Ť", "Ť", "Ť", "Ť" },
  { "ť", "ť", "ť", "ť", "ť" },
  { "Ũ", "Ũ", "Ũ", "Ũ", "Ũ" },
  { "ũ", "ũ", "ũ", "ũ", "ũ" },
  { "Ū", "Ū", "Ū", "Ū", "Ū" },
  { "ū", "ū", "ū", "ū", "ū" },
  { "Ŭ", "Ŭ", "Ŭ", "Ŭ", "Ŭ" },
  { "ŭ", "ŭ", "ŭ", "ŭ", "ŭ" },
  { "Ů", "Ů", "Ů", "Ů", "Ů" },
  { "ů", "ů", "ů", "ů", "ů" },
  { "Ű", "Ű", "Ű", "Ű", "Ű" },
  { "ű", "ű", "ű", "ű", "ű" },
  { "Ų", "Ų", "Ų", "Ų", "Ų" },
  { "ų", "ų", "ų", "ų", "ų" },
  { "Ŵ", "Ŵ", "Ŵ", "Ŵ", "Ŵ" },
  { "ŵ", "ŵ", "ŵ", "ŵ", "ŵ" },
  { "Ŷ", "Ŷ", "Ŷ", "Ŷ", "Ŷ" },
  { "ŷ", "ŷ", "ŷ", "ŷ", "ŷ" },
  { "Ÿ", "Ÿ", "Ÿ", "Ÿ", "Ÿ" },
  { "Ź", "Ź", "Ź", "Ź", "Ź" },
  { "ź", "ź", "ź", "ź", "ź" },
  { "Ż", "Ż", "Ż", "Ż", "Ż" },
  { "ż", "ż", "ż", "ż", "ż" },
  { "Ž", "Ž", "Ž", "Ž", "Ž" },
  { "ž", "ž", "ž", "ž", "ž" },
  { "ſ", "ſ", "ſ", "s", "s" },
  { "Ơ", "Ơ", "Ơ", "Ơ", "Ơ" },
  { "ơ", "ơ", "ơ", "ơ", "ơ" },
  { "Ư", "Ư", "Ư", "Ư", "Ư" },
  { "ư", "ư", "ư", "ư", "ư" },
  { "Ǆ", "Ǆ", "Ǆ", "DŽ", "DŽ" },
  { "ǅ", "ǅ", "ǅ", "Dž", "Dž" },
  { "ǆ", "ǆ", "ǆ", "dž", "dž" },
  { "Ǉ", "Ǉ", "Ǉ", "LJ", "LJ" },
  { "ǈ", "ǈ", "ǈ", "Lj", "Lj" },
  { "ǉ", "ǉ", "ǉ", "lj", "lj" },
  { "Ǌ", "Ǌ", "Ǌ", "NJ", "NJ" },
  { "ǋ", "ǋ", "ǋ", "Nj", "Nj" },
  { "ǌ", "ǌ", "ǌ", "nj", "nj" },
  { "Ǎ", "Ǎ", "Ǎ", "Ǎ", "Ǎ" },
  { "ǎ", "ǎ", "ǎ", "ǎ", "ǎ" },
  { "Ǐ", "Ǐ", "Ǐ", "Ǐ", "Ǐ" },
  { "ǐ", "ǐ", "ǐ", "ǐ", "ǐ" },
  { "Ǒ", "Ǒ", "Ǒ", "Ǒ", "Ǒ" },
  { "ǒ", "ǒ", "ǒ", "ǒ", "ǒ" },
  { "Ǔ", "Ǔ", "Ǔ", "Ǔ", "Ǔ" },
  { "ǔ", "ǔ", "ǔ", "ǔ", "ǔ" },
  { "Ǖ", "Ǖ", "Ǖ", "Ǖ", "Ǖ" },
  { "ǖ", "ǖ", "ǖ", "ǖ", "ǖ" },
  { "Ǘ", "Ǘ", "Ǘ", "Ǘ", "Ǘ" },
  { "ǘ", "ǘ", "ǘ", "ǘ", "ǘ" },
  { "Ǚ", "Ǚ", "Ǚ", "Ǚ", "Ǚ" },
  { "ǚ", "ǚ", "ǚ", "ǚ", "ǚ" },
  { "Ǜ", "Ǜ", "Ǜ", "Ǜ", "Ǜ" },
  { "ǜ", "ǜ", "ǜ", "ǜ", "ǜ" },
  { "Ǟ", "Ǟ", "Ǟ", "Ǟ", "Ǟ" },
  { "ǟ", "ǟ", "ǟ", "ǟ", "ǟ" },
  { "Ǡ", "Ǡ", "Ǡ", "Ǡ", "Ǡ" },
  { "ǡ", "ǡ", "ǡ", "ǡ", "ǡ" },
  { "Ǣ", "Ǣ", "Ǣ", "Ǣ", "Ǣ" },
  { "ǣ", "ǣ", "ǣ", "ǣ", "ǣ" },
  { "Ǧ", "Ǧ", "Ǧ", "Ǧ", "Ǧ" },
  { "ǧ", "ǧ", "ǧ", "ǧ", "ǧ" },
  { "Ǩ", "Ǩ", "Ǩ", "Ǩ", "Ǩ" },
  { "ǩ", "ǩ", "ǩ", "ǩ", "ǩ" },
  { "Ǫ", "Ǫ", "Ǫ", "Ǫ", "Ǫ" },
  { "ǫ", "ǫ", "ǫ", "ǫ", "ǫ" },
  { "Ǭ", "Ǭ", "Ǭ", "Ǭ", "Ǭ" },
  { "ǭ", "ǭ", "ǭ", "ǭ", "ǭ" },
  { "Ǯ", "Ǯ", "Ǯ", "Ǯ", "Ǯ" },
  { "ǯ", "ǯ", "ǯ", "ǯ", "ǯ" },
  { "ǰ", "ǰ", "ǰ", "ǰ", "ǰ" },
  { "Ǳ", "Ǳ", "Ǳ", "DZ", "DZ" },
  { "ǲ", "ǲ", "ǲ", "Dz", "Dz" },
  { "ǳ", "ǳ", "ǳ", "dz", "dz" },
  { "Ǵ", "Ǵ", "Ǵ", "Ǵ", "Ǵ" },
  { "ǵ", "ǵ", "ǵ", "ǵ", "ǵ" },
  { "Ǹ", "Ǹ", "Ǹ", "Ǹ", "Ǹ" },
  { "ǹ", "ǹ", "ǹ", "ǹ", "ǹ" },
  { "Ǻ", "Ǻ", "Ǻ", "Ǻ", "Ǻ" },
  { "ǻ", "ǻ", "ǻ", "ǻ", "ǻ" },
  { "Ǽ", "Ǽ", "Ǽ", "Ǽ", "Ǽ" },
  { "ǽ", "ǽ", "ǽ", "ǽ", "ǽ" },
  { "Ǿ", "Ǿ", "Ǿ", "Ǿ", "Ǿ" },
  { "ǿ", "ǿ", "ǿ", "ǿ", "ǿ" },
  { "Ȁ", "Ȁ", "Ȁ", "Ȁ", "Ȁ" },
  { "ȁ", "ȁ", "ȁ", "ȁ", "ȁ" },
  { "Ȃ", "Ȃ", "Ȃ", "Ȃ", "Ȃ" },
  { "ȃ", "ȃ", "ȃ", "ȃ", "ȃ" },
  { "Ȅ", "Ȅ", "Ȅ", "Ȅ", "Ȅ" },
  { "ȅ", "ȅ", "ȅ", "ȅ", "ȅ" },
  { "Ȇ", "Ȇ", "Ȇ", "Ȇ", "Ȇ" },
  { "ȇ", "ȇ", "ȇ", "ȇ", "ȇ" },
  { "Ȉ", "Ȉ", "Ȉ", "Ȉ", "Ȉ" },
  { "ȉ", "ȉ", "ȉ", "ȉ", "ȉ" },
  { "Ȋ", "Ȋ", "Ȋ", "Ȋ", "Ȋ" },
  { "ȋ", "ȋ", "ȋ", "ȋ", "ȋ" },
  { "Ȍ", "Ȍ", "Ȍ", "Ȍ", "Ȍ" },
  { "ȍ", "ȍ", "ȍ", "ȍ", "ȍ" },
  { "Ȏ", "Ȏ", "Ȏ", "Ȏ", "Ȏ" },
  { "ȏ", "ȏ", "ȏ", "ȏ", "ȏ" },
  { "Ȑ", "Ȑ", "Ȑ", "Ȑ", "Ȑ" },
  { "ȑ", "ȑ", "ȑ", "ȑ", "ȑ" },
  { "Ȓ", "Ȓ", "Ȓ", "Ȓ", "Ȓ" },
  { "ȓ", "ȓ", "ȓ", "ȓ", "ȓ" },
  { "Ȕ", "Ȕ", "Ȕ", "Ȕ", "Ȕ" },
  { "ȕ", "ȕ", "ȕ", "ȕ", "ȕ" },
  { "Ȗ", "Ȗ", "Ȗ", "Ȗ", "Ȗ" },
  { "ȗ", "ȗ", "ȗ", "ȗ", "ȗ" },
  { "Ș", "Ș", "Ș", "Ș", "Ș" },
  { "ș", "ș", "ș", "ș", "ș" },
  { "Ț", "Ț", "Ț", "Ț", "Ț" },
  { "ț", "ț", "ț", "ț", "ț" },
  { "Ȟ", "Ȟ", "Ȟ", "Ȟ", "Ȟ" },
  { "ȟ", "ȟ", "ȟ", "ȟ", "ȟ" },
  { "Ȧ", "Ȧ", "Ȧ", "Ȧ", "Ȧ" },
  { "ȧ", "ȧ", "ȧ", "ȧ", "ȧ" },
  { "Ȩ", "Ȩ", "Ȩ", "Ȩ", "Ȩ" },
  { "ȩ", "ȩ", "ȩ", "ȩ", "ȩ" },
  { "Ȫ", "Ȫ", "Ȫ", "Ȫ", "Ȫ" },
  { "ȫ", "ȫ", "ȫ", "ȫ", "ȫ" },
  { "Ȭ", "Ȭ", "Ȭ", "Ȭ", "Ȭ" },
  { "ȭ", "ȭ", "ȭ", "ȭ", "ȭ" },
  { "Ȯ", "Ȯ", "Ȯ", "Ȯ", "Ȯ" },
  { "ȯ", "ȯ", "ȯ", "ȯ", "ȯ" },
  { "Ȱ", "Ȱ", "Ȱ", "Ȱ", "Ȱ" },
  { "ȱ", "ȱ", "ȱ", "ȱ", "ȱ" },
  { "Ȳ", "Ȳ", "Ȳ", "Ȳ", "Ȳ" },
  { "ȳ", "ȳ", "ȳ", "ȳ", "ȳ" },
  { "ʰ", "ʰ", "ʰ", "h", "h" },
  { "ʱ", "ʱ", "ʱ", "ɦ", "ɦ" },
  { "ʲ", "ʲ", "ʲ", "j", "j" },
  { "ʳ", "ʳ", "ʳ", "r", "r" },
  { "ʴ", "ʴ", "ʴ", "ɹ", "ɹ" },
  { "ʵ", "ʵ", "ʵ", "ɻ", "ɻ" },
  { "ʶ", "ʶ", "ʶ", "ʁ", "ʁ" },
  { "ʷ", "ʷ", "ʷ", "w", "w" },
  { "ʸ", "ʸ", "ʸ", "y", "y" },
  { "˘", "˘", "˘", " ̆", " ̆" },
  { "˙", "˙", "˙", " ̇", " ̇" },
  { "˚", "˚", "˚", " ̊", " ̊" },
  { "˛", "˛", "˛", " ̨", " ̨" },
  { "˜", "˜", "˜", " ̃", " ̃" },
  { "˝", "˝", "˝", " ̋", " ̋" },
  { "ˠ", "ˠ", "ˠ", "ɣ", "ɣ" },
  { "ˡ", "ˡ", "ˡ", "l", "l" },
  { "ˢ", "ˢ", "ˢ", "s", "s" },
  { "ˣ", "ˣ", "ˣ", "x", "x" },
  { "ˤ", "ˤ", "ˤ", "ʕ", "ʕ" },
  { "̀", "̀", "̀", "̀", "̀" },
  { "́", "́", "́", "́", "́" },
  { "̓", "̓", "̓", "̓", "̓" },
  { "̈́", "̈́", "̈́", "̈́", "̈́" },
  { "ʹ", "ʹ", "ʹ", "ʹ", "ʹ" },
  { "ͺ", "ͺ", "ͺ", " ͅ", " ͅ" },
  { ";", ";", ";", ";", ";" },
  { "΄", "΄", "΄", " ́", " ́" },
  { "΅", "΅", "΅", " ̈́", " ̈́" },
  { "Ά", "Ά", "Ά", "Ά", "Ά" },
  { "·", "·", "·", "·", "·" },
  { "Έ", "Έ", "Έ", "Έ", "Έ" },
  { "Ή", "Ή", "Ή", "Ή", "Ή" },
  { "Ί", "Ί", "Ί", "Ί", "Ί" },
  { "Ό", "Ό", "Ό", "Ό", "Ό" },
  { "Ύ", "Ύ", "Ύ", "Ύ", "Ύ" },
  { "Ώ", "Ώ", "Ώ", "Ώ", "Ώ" },
  { "ΐ", "ΐ", "ΐ", "ΐ", "ΐ" },
  { "Ϊ", "Ϊ", "Ϊ", "Ϊ", "Ϊ" },
  { "Ϋ", "Ϋ", "Ϋ", "Ϋ", "Ϋ" },
  { "ά", "ά", "ά", "ά", "ά" },
  { "έ", "έ", "έ", "έ", "έ" },
  { "ή", "ή", "ή", "ή", "ή" },
  { "ί", "ί", "ί", "ί", "ί" },
  { "ΰ", "ΰ", "ΰ", "ΰ", "ΰ" },
  { "ϊ", "ϊ", "ϊ", "ϊ", "ϊ" },
  { "ϋ", "ϋ", "ϋ", "ϋ", "ϋ" },
  { "ό", "ό", "ό", "ό", "ό" },
  { "ύ", "ύ", "ύ", "ύ", "ύ" },
  { "ώ", "ώ", "ώ", "ώ", "ώ" },
  { "ϐ", "ϐ", "ϐ", "β", "β" },
  { "ϑ", "ϑ", "ϑ", "θ", "θ" },
  { "ϒ", "ϒ", "ϒ", "Υ", "Υ" },
  { "ϓ", "ϓ", "ϓ", "Ύ", "Ύ" },
  { "ϔ", "ϔ", "ϔ", "Ϋ", "Ϋ" },
  { "ϕ", "ϕ", "ϕ", "φ", "φ" },
  { "ϖ", "ϖ", "ϖ", "π", "π" },
  { "ϰ", "ϰ", "ϰ", "κ", "κ" },
  { "ϱ", "ϱ", "ϱ", "ρ", "ρ" },
  { "ϲ", "ϲ", "ϲ", "ς", "ς" },
  { "ϴ", "ϴ", "ϴ", "Θ", "Θ" },
  { "ϵ", "ϵ", "ϵ", "ε", "ε" },
  { "Ϲ", "Ϲ", "Ϲ", "Σ", "Σ" },
  { "Ѐ", "Ѐ", "Ѐ", "Ѐ", "Ѐ" },
  { "Ё", "Ё", "Ё", "Ё", "Ё" },
  { "Ѓ", "Ѓ", "Ѓ", "Ѓ", "Ѓ" },
  { "Ї", "Ї", "Ї", "Ї", "Ї" },
  { "Ќ", "Ќ", "Ќ", "Ќ", "Ќ" },
  { "Ѝ", "Ѝ", "Ѝ", "Ѝ", "Ѝ" },
  { "Ў", "Ў", "Ў", "Ў", "Ў" },
  { "Й", "Й", "Й", "Й", "Й" },
  { "й", "й", "й", "й", "й" },
  { "ѐ", "ѐ", "ѐ", "ѐ", "ѐ" },
  { "ё", "ё", "ё", "ё", "ё" },
  { "ѓ", "ѓ", "ѓ", "ѓ", "ѓ" },
  { "ї", "ї", "ї", "ї", "ї" },
  { "ќ", "ќ", "ќ", "ќ", "ќ" },
  { "ѝ", "ѝ", "ѝ", "ѝ", "ѝ" },
  { "ў", "ў", "ў", "ў", "ў" },
  { "Ѷ", "Ѷ", "Ѷ", "Ѷ", "Ѷ" },
  { "ѷ", "ѷ", "ѷ", "ѷ", "ѷ" },
  { "Ӂ", "Ӂ", "Ӂ", "Ӂ", "Ӂ" },
  { "ӂ", "ӂ", "ӂ", "ӂ", "ӂ" },
  { "Ӑ", "Ӑ", "Ӑ", "Ӑ", "Ӑ" },
  { "ӑ", "ӑ", "ӑ", "ӑ", "ӑ" },
  { "Ӓ", "Ӓ", "Ӓ", "Ӓ", "Ӓ" },
  { "ӓ", "ӓ", "ӓ", "ӓ", "ӓ" },
  { "Ӗ", "Ӗ", "Ӗ", "Ӗ", "Ӗ" },
  { "ӗ", "ӗ", "ӗ", "ӗ", "ӗ" },
  { "Ӛ", "Ӛ", "Ӛ", "Ӛ", "Ӛ" },
  { "ӛ", "ӛ", "ӛ", "ӛ", "ӛ" },
  { "Ӝ", "Ӝ", "Ӝ", "Ӝ", "Ӝ" },
  { "ӝ", "ӝ", "ӝ", "ӝ", "ӝ" },
  { "Ӟ", "Ӟ", "Ӟ", "Ӟ", "Ӟ" },
  { "ӟ", "ӟ", "ӟ", "ӟ", "ӟ" },
  { "Ӣ", "Ӣ", "Ӣ", "Ӣ", "Ӣ" },
  { "ӣ", "ӣ", "ӣ", "ӣ", "ӣ" },
  { "Ӥ", "Ӥ", "Ӥ", "Ӥ", "Ӥ" },
  { "ӥ", "ӥ", "ӥ", "ӥ", "ӥ" },
  { "Ӧ", "Ӧ", "Ӧ", "Ӧ", "Ӧ" },
  { "ӧ", "ӧ", "ӧ", "ӧ", "ӧ" },
  { "Ӫ", "Ӫ", "Ӫ", "Ӫ", "Ӫ" },
  { "ӫ", "ӫ", "ӫ", "ӫ", "ӫ" },
  { "Ӭ", "Ӭ", "Ӭ", "Ӭ", "Ӭ" },
  { "ӭ", "ӭ", "ӭ", "ӭ", "ӭ" },
  { "Ӯ", "Ӯ", "Ӯ", "Ӯ", "Ӯ" },
  { "ӯ", "ӯ", "ӯ", "ӯ", "ӯ" },
  { "Ӱ", "Ӱ", "Ӱ", "Ӱ", "Ӱ" },
  { "ӱ", "ӱ", "ӱ", "ӱ", "ӱ" },
  { "Ӳ", "Ӳ", "Ӳ", "Ӳ", "Ӳ" },
  { "ӳ", "ӳ", "ӳ", "ӳ", "ӳ" },
  { "Ӵ", "Ӵ", "Ӵ", "Ӵ", "Ӵ" },
  { "ӵ", "ӵ", "ӵ", "ӵ", "ӵ" },
  { "Ӹ", "Ӹ", "Ӹ", "Ӹ", "Ӹ" },
  { "ӹ", "ӹ", "ӹ", "ӹ", "ӹ" },
  { "և", "և", "և", "եւ", "եւ" },
  { "آ", "آ", "آ", "آ", "آ" },
  { "أ", "أ", "أ", "أ", "أ" },
  { "ؤ", "ؤ", "ؤ", "ؤ", "ؤ" },
  { "إ", "إ", "إ", "إ", "إ" },
  { "ئ", "ئ", "ئ", "ئ", "ئ" },
  { "ٵ", "ٵ", "ٵ", "اٴ", "اٴ" },
  { "ٶ", "ٶ", "ٶ", "وٴ", "وٴ" },
  { "ٷ", "ٷ", "ٷ", "ۇٴ", "ۇٴ" },
  { "ٸ", "ٸ", "ٸ", "يٴ", "يٴ" },
  { "ۀ", "ۀ", "ۀ", "ۀ", "ۀ" },
  { "ۂ", "ۂ", "ۂ", "ۂ", "ۂ" },
  { "ۓ", "ۓ", "ۓ", "ۓ", "ۓ" },
  { "ऩ", "ऩ", "ऩ", "ऩ", "ऩ" },
  { "ऱ", "ऱ", "ऱ", "ऱ", "ऱ" },
  { "ऴ", "ऴ", "ऴ", "ऴ", "ऴ" },
  { "क़", "क़", "क़", "क़", "क़" },
  { "ख़", "ख़", "ख़", "ख़", "ख़" },
  { "ग़", "ग़", "ग़", "ग़", "ग़" },
  { "ज़", "ज़", "ज़", "ज़", "ज़" },
  { "ड़", "ड़", "ड़", "ड़", "ड़" },
  { "ढ़", "ढ़", "ढ़", "ढ़", "ढ़" },
  { "फ़", "फ़", "फ़", "फ़", "फ़" },
  { "य़", "य़", "य़", "य़", "य़" },
  { "ো", "ো", "ো", "ো", "ো" },
  { "ৌ", "ৌ", "ৌ", "ৌ", "ৌ" },
  { "ড়", "ড়", "ড়", "ড়", "ড়" },
  { "ঢ়", "ঢ়", "ঢ়", "ঢ়", "ঢ়" },
  { "য়", "য়", "য়", "য়", "য়" },
  { "ਲ਼", "ਲ਼", "ਲ਼", "ਲ਼", "ਲ਼" },
  { "ਸ਼", "ਸ਼", "ਸ਼", "ਸ਼", "ਸ਼" },
  { "ਖ਼", "ਖ਼", "ਖ਼", "ਖ਼", "ਖ਼" },
  { "ਗ਼", "ਗ਼", "ਗ਼", "ਗ਼", "ਗ਼" },
  { "ਜ਼", "ਜ਼", "ਜ਼", "ਜ਼", "ਜ਼" },
  { "ਫ਼", "ਫ਼", "ਫ਼", "ਫ਼", "ਫ਼" },
  { "ୈ", "ୈ", "ୈ", "ୈ", "ୈ" },
  { "ୋ", "ୋ", "ୋ", "ୋ", "ୋ" },
  { "ୌ", "ୌ", "ୌ", "ୌ", "ୌ" },
  { "ଡ଼", "ଡ଼", "ଡ଼", "ଡ଼", "ଡ଼" },
  { "ଢ଼", "ଢ଼", "ଢ଼", "ଢ଼", "ଢ଼" },
  { "ஔ", "ஔ", "ஔ", "ஔ", "ஔ" },
  { "ொ", "ொ", "ொ", "ொ", "ொ" },
  { "ோ", "ோ", "ோ", "ோ", "ோ" },
  { "ௌ", "ௌ", "ௌ", "ௌ", "ௌ" },
  { "ై", "ై", "ై", "ై", "ై" },
  { "ೀ", "ೀ", "ೀ", "ೀ", "ೀ" },
  { "ೇ", "ೇ", "ೇ", "ೇ", "ೇ" },
  { "ೈ", "ೈ", "ೈ", "ೈ", "ೈ" },
  { "ೊ", "ೊ", "ೊ", "ೊ", "ೊ" },
  { "ೋ", "ೋ", "ೋ", "ೋ", "ೋ" },
  { "ൊ", "ൊ", "ൊ", "ൊ", "ൊ" },
  { "ോ", "ോ", "ോ", "ോ", "ോ" },
  { "ൌ", "ൌ", "ൌ", "ൌ", "ൌ" },
  { "ේ", "ේ", "ේ", "ේ", "ේ" },
  { "ො", "ො", "ො", "ො", "ො" },
  { "ෝ", "ෝ", "ෝ", "ෝ", "ෝ" },
  { "ෞ", "ෞ", "ෞ", "ෞ", "ෞ" },
  { "ำ", "ำ", "ำ", "ํา", "ํา" },
  { "ຳ", "ຳ", "ຳ", "ໍາ", "ໍາ" },
  { "ໜ", "ໜ", "ໜ", "ຫນ", "ຫນ" },
  { "ໝ", "ໝ", "ໝ", "ຫມ", "ຫມ" },
  { "༌", "༌", "༌", "་", "་" },
  { "གྷ", "གྷ", "གྷ", "གྷ", "གྷ" },
  { "ཌྷ", "ཌྷ", "ཌྷ", "ཌྷ", "ཌྷ" },
  { "དྷ", "དྷ", "དྷ", "དྷ", "དྷ" },
  { "བྷ", "བྷ", "བྷ", "བྷ", "བྷ" },
  { "ཛྷ", "ཛྷ", "ཛྷ", "ཛྷ", "ཛྷ" },
  { "ཀྵ", "ཀྵ", "ཀྵ", "ཀྵ", "ཀྵ" },
  { "ཱི", "ཱི", "ཱི", "ཱི", "ཱི" },
  { "ཱུ", "ཱུ", "ཱུ", "ཱུ", "ཱུ" },
  { "ྲྀ", "ྲྀ", "ྲྀ", "ྲྀ", "ྲྀ" },
  { "ཷ", "ཷ", "ཷ", "ྲཱྀ", "ྲཱྀ" },
  { "ླྀ", "ླྀ", "ླྀ", "ླྀ", "ླྀ" },
  { "ཹ", "ཹ", "ཹ", "ླཱྀ", "ླཱྀ" },
  { "ཱྀ", "ཱྀ", "ཱྀ", "ཱྀ", "ཱྀ" },
  { "ྒྷ", "ྒྷ", "ྒྷ", "ྒྷ", "ྒྷ" },
  { "ྜྷ", "ྜྷ", "ྜྷ", "ྜྷ", "ྜྷ" },
  { "ྡྷ", "ྡྷ", "ྡྷ", "ྡྷ", "ྡྷ" },
  { "ྦྷ", "ྦྷ", "ྦྷ", "ྦྷ", "ྦྷ" },
  { "ྫྷ", "ྫྷ", "ྫྷ", "ྫྷ", "ྫྷ" },
  { "ྐྵ", "ྐྵ", "ྐྵ", "ྐྵ", "ྐྵ" },
  { "ဦ", "ဦ", "ဦ", "ဦ", "ဦ" },
  { "ჼ", "ჼ", "ჼ", "ნ", "ნ" },
  { "ᬆ", "ᬆ", "ᬆ", "ᬆ", "ᬆ" },
  { "ᬈ", "ᬈ", "ᬈ", "ᬈ", "ᬈ" },
  { "ᬊ", "ᬊ", "ᬊ", "ᬊ", "ᬊ" },
  { "ᬌ", "ᬌ", "ᬌ", "ᬌ", "ᬌ" },
  { "ᬎ", "ᬎ", "ᬎ", "ᬎ", "ᬎ" },
  { "ᬒ", "ᬒ", "ᬒ", "ᬒ", "ᬒ" },
  { "ᬻ", "ᬻ", "ᬻ", "ᬻ", "ᬻ" },
  { "ᬽ", "ᬽ", "ᬽ", "ᬽ", "ᬽ" },
  { "ᭀ", "ᭀ", "ᭀ", "ᭀ", "ᭀ" },
  { "ᭁ", "ᭁ", "ᭁ", "ᭁ", "ᭁ" },
  { "ᭃ", "ᭃ", "ᭃ", "ᭃ", "ᭃ" },
  { "ᴬ", "ᴬ", "ᴬ", "A", "A" },
  { "ᴭ", "ᴭ", "ᴭ", "Æ", "Æ" },
  { "ᴮ", "ᴮ", "ᴮ", "B", "B" },
  { "ᴰ", "ᴰ", "ᴰ", "D", "D" },
  { "ᴱ", "ᴱ", "ᴱ", "E", "E" },
  { "ᴲ", "ᴲ", "ᴲ", "Ǝ", "Ǝ" },
  { "ᴳ", "ᴳ", "ᴳ", "G", "G" },
  { "ᴴ", "ᴴ", "ᴴ", "H", "H" },
  { "ᴵ", "ᴵ", "ᴵ", "I", "I" },
  { "ᴶ", "ᴶ", "ᴶ", "J", "J" },
  { "ᴷ", "ᴷ", "ᴷ", "K", "K" },
  { "ᴸ", "ᴸ", "ᴸ", "L", "L" },
  { "ᴹ", "ᴹ", "ᴹ", "M", "M" },
  { "ᴺ", "ᴺ", "ᴺ", "N", "N" },
  { "ᴼ", "ᴼ", "ᴼ", "O", "O" },
  { "ᴽ", "ᴽ", "ᴽ", "Ȣ", "Ȣ" },
  { "ᴾ", "ᴾ", "ᴾ", "P", "P" },
  { "ᴿ", "ᴿ", "ᴿ", "R", "R" },
  { "ᵀ", "ᵀ", "ᵀ", "T", "T" },
  { "ᵁ", "ᵁ", "ᵁ", "U", "U" },
  { "ᵂ", "ᵂ", "ᵂ", "W", "W" },
  { "ᵃ", "ᵃ", "ᵃ", "a", "a" },
  { "ᵄ", "ᵄ", "ᵄ", "ɐ", "ɐ" },
  { "ᵅ", "ᵅ", "ᵅ", "ɑ", "ɑ" },
  { "ᵆ", "ᵆ", "ᵆ", "ᴂ", "ᴂ" },
  { "ᵇ", "ᵇ", "ᵇ", "b", "b" },
  { "ᵈ", "ᵈ", "ᵈ", "d", "d" },
  { "ᵉ", "ᵉ", "ᵉ", "e", "e" },
  { "ᵊ", "ᵊ", "ᵊ", "ə", "ə" },
  { "ᵋ", "ᵋ", "ᵋ", "ɛ", "ɛ" },
  { "ᵌ", "ᵌ", "ᵌ", "ɜ", "ɜ" },
  { "ᵍ", "ᵍ", "ᵍ", "g", "g" },
  { "ᵏ", "ᵏ", "ᵏ", "k", "k" },
  { "ᵐ", "ᵐ", "ᵐ", "m", "m" },
  { "ᵑ", "ᵑ", "ᵑ", "ŋ", "ŋ" },
  { "ᵒ", "ᵒ", "ᵒ", "o", "o" },
  { "ᵓ", "ᵓ", "ᵓ", "ɔ", "ɔ" },
  { "ᵔ", "ᵔ", "ᵔ", "ᴖ", "ᴖ" },
  { "ᵕ", "ᵕ", "ᵕ", "ᴗ", "ᴗ" },
  { "ᵖ", "ᵖ", "ᵖ", "p", "p" },
  { "ᵗ", "ᵗ", "ᵗ", "t", "t" },
  { "ᵘ", "ᵘ", "ᵘ", "u", "u" },
  { "ᵙ", "ᵙ", "ᵙ", "ᴝ", "ᴝ" },
  { "ᵚ", "ᵚ", "ᵚ", "ɯ", "ɯ" },
  { "ᵛ", "ᵛ", "ᵛ", "v", "v" },
  { "ᵜ", "ᵜ", "ᵜ", "ᴥ", "ᴥ" },
  { "ᵝ", "ᵝ", "ᵝ", "β", "β" },
  { "ᵞ", "ᵞ", "ᵞ", "γ", "γ" },
  { "ᵟ", "ᵟ", "ᵟ", "δ", "δ" },
  { "ᵠ", "ᵠ", "ᵠ", "φ", "φ" },
  { "ᵡ", "ᵡ", "ᵡ", "χ", "χ" },
  { "ᵢ", "ᵢ", "ᵢ", "i", "i" },
  { "ᵣ", "ᵣ", "ᵣ", "r", "r" },
  { "ᵤ", "ᵤ", "ᵤ", "u", "u" },
  { "ᵥ", "ᵥ", "ᵥ", "v", "v" },
  { "ᵦ", "ᵦ", "ᵦ", "β", "β" },
  { "ᵧ", "ᵧ", "ᵧ", "γ", "γ" },
  { "ᵨ", "ᵨ", "ᵨ", "ρ", "ρ" },
  { "ᵩ", "ᵩ", "ᵩ", "φ", "φ" },
  { "ᵪ", "ᵪ", "ᵪ", "χ", "χ" },
  { "ᵸ", "ᵸ", "ᵸ", "н", "н" },
  { "ᶛ", "ᶛ", "ᶛ", "ɒ", "ɒ" },
  { "ᶜ", "ᶜ", "ᶜ", "c", "c" },
  { "ᶝ", "ᶝ", "ᶝ", "ɕ", "ɕ" },
  { "ᶞ", "ᶞ", "ᶞ", "ð", "ð" },
  { "ᶟ", "ᶟ", "ᶟ", "ɜ", "ɜ" },
  { "ᶠ", "ᶠ", "ᶠ", "f", "f" },
  { "ᶡ", "ᶡ", "ᶡ", "ɟ", "ɟ" },
  { "ᶢ", "ᶢ", "ᶢ", "ɡ", "ɡ" },
  { "ᶣ", "ᶣ", "ᶣ", "ɥ", "ɥ" },
  { "ᶤ", "ᶤ", "ᶤ", "ɨ", "ɨ" },
  { "ᶥ", "ᶥ", "ᶥ", "ɩ", "ɩ" },
  { "ᶦ", "ᶦ", "ᶦ", "ɪ", "ɪ" },
  { "ᶧ", "ᶧ", "ᶧ", "ᵻ", "ᵻ" },
  { "ᶨ", "ᶨ", "ᶨ", "ʝ", "ʝ" },
  { "ᶩ", "ᶩ", "ᶩ", "ɭ", "ɭ" },
  { "ᶪ", "ᶪ", "ᶪ", "ᶅ", "ᶅ" },
  { "ᶫ", "ᶫ", "ᶫ", "ʟ", "ʟ" },
  { "ᶬ", "ᶬ", "ᶬ", "ɱ", "ɱ" },
  { "ᶭ", "ᶭ", "ᶭ", "ɰ", "ɰ" },
  { "ᶮ", "ᶮ", "ᶮ", "ɲ", "ɲ" },
  { "ᶯ", "ᶯ", "ᶯ", "ɳ", "ɳ" },
  { "ᶰ", "ᶰ", "ᶰ", "ɴ", "ɴ" },
  { "ᶱ", "ᶱ", "ᶱ", "ɵ", "ɵ" },
  { "ᶲ", "ᶲ", "ᶲ", "ɸ", "ɸ" },
  { "ᶳ", "ᶳ", "ᶳ", "ʂ", "ʂ" },
  { "ᶴ", "ᶴ", "ᶴ", "ʃ", "ʃ" },
  { "ᶵ", "ᶵ", "ᶵ", "ƫ", "ƫ" },
  { "ᶶ", "ᶶ", "ᶶ", "ʉ", "ʉ" },
  { "ᶷ", "ᶷ", "ᶷ", "ʊ", "ʊ" },
  { "ᶸ", "ᶸ", "ᶸ", "ᴜ", "ᴜ" },
  { "ᶹ", "ᶹ", "ᶹ", "ʋ", "ʋ" },
  { "ᶺ", "ᶺ", "ᶺ", "ʌ", "ʌ" },
  { "ᶻ", "ᶻ", "ᶻ", "z", "z" },
  { "ᶼ", "ᶼ", "ᶼ", "ʐ", "ʐ" },
  { "ᶽ", "ᶽ", "ᶽ", "ʑ", "ʑ" },
  { "ᶾ", "ᶾ", "ᶾ", "ʒ", "ʒ" },
  { "ᶿ", "ᶿ", "ᶿ", "θ", "θ" },
  { "Ḁ", "Ḁ", "Ḁ", "Ḁ", "Ḁ" },
  { "ḁ", "ḁ", "ḁ", "ḁ", "ḁ" },
  { "Ḃ", "Ḃ", "Ḃ", "Ḃ", "Ḃ" },
  { "ḃ", "ḃ", "ḃ", "ḃ", "ḃ" },
  { "Ḅ", "Ḅ", "Ḅ", "Ḅ", "Ḅ" },
  { "ḅ", "ḅ", "ḅ", "ḅ", "ḅ" },
  { "Ḇ", "Ḇ", "Ḇ", "Ḇ", "Ḇ" },
  { "ḇ", "ḇ", "ḇ", "ḇ", "ḇ" },
  { "Ḉ", "Ḉ", "Ḉ", "Ḉ", "Ḉ" },
  { "ḉ", "ḉ", "ḉ", "ḉ", "ḉ" },
  { "Ḋ", "Ḋ", "Ḋ", "Ḋ", "Ḋ" },
  { "ḋ", "ḋ", "ḋ", "ḋ", "ḋ" },
  { "Ḍ", "Ḍ", "Ḍ", "Ḍ", "Ḍ" },
  { "ḍ", "ḍ", "ḍ", "ḍ", "ḍ" },
  { "Ḏ", "Ḏ", "Ḏ", "Ḏ", "Ḏ" },
  { "ḏ", "ḏ", "ḏ", "ḏ", "ḏ" },
  { "Ḑ", "Ḑ", "Ḑ", "Ḑ", "Ḑ" },
  { "ḑ", "ḑ", "ḑ", "ḑ", "ḑ" },
  { "Ḓ", "Ḓ", "Ḓ", "Ḓ", "Ḓ" },
  { "ḓ", "ḓ", "ḓ", "ḓ", "ḓ" },
  { "Ḕ", "Ḕ", "Ḕ", "Ḕ", "Ḕ" },
  { "ḕ", "ḕ", "ḕ", "ḕ", "ḕ" },
  { "Ḗ", "Ḗ", "Ḗ", "Ḗ", "Ḗ" },
  { "ḗ", "ḗ", "ḗ", "ḗ", "ḗ" },
  { "Ḙ", "Ḙ", "Ḙ", "Ḙ", "Ḙ" },
  { "ḙ", "ḙ", "ḙ", "ḙ", "ḙ" },
  { "Ḛ", "Ḛ", "Ḛ", "Ḛ", "Ḛ" },
  { "ḛ", "ḛ", "ḛ", "ḛ", "ḛ" },
  { "Ḝ", "Ḝ", "Ḝ", "Ḝ", "Ḝ" },
  { "ḝ", "ḝ", "ḝ", "ḝ", "ḝ" },
  { "Ḟ", "Ḟ", "Ḟ", "Ḟ", "Ḟ" },
  { "ḟ", "ḟ", "ḟ", "ḟ", "ḟ" },
  { "Ḡ", "Ḡ", "Ḡ", "Ḡ", "Ḡ" },
  { "ḡ", "ḡ", "ḡ", "ḡ", "ḡ" },
  { "Ḣ", "Ḣ", "Ḣ", "Ḣ", "Ḣ" },
  { "ḣ", "ḣ", "ḣ", "ḣ", "ḣ" },
  { "Ḥ", "Ḥ", "Ḥ", "Ḥ", "Ḥ" },
  { "ḥ", "ḥ", "ḥ", "ḥ", "ḥ" },
  { "Ḧ", "Ḧ", "Ḧ", "Ḧ", "Ḧ" },
  { "ḧ", "ḧ", "ḧ", "ḧ", "ḧ" },
  { "Ḩ", "Ḩ", "Ḩ", "Ḩ", "Ḩ" },
  { "ḩ", "ḩ", "ḩ", "ḩ", "ḩ" },
  { "Ḫ", "Ḫ", "Ḫ", "Ḫ", "Ḫ" },
  { "ḫ", "ḫ", "ḫ", "ḫ", "ḫ" },
  { "Ḭ", "Ḭ", "Ḭ", "Ḭ", "Ḭ" },
  { "ḭ", "ḭ", "ḭ", "ḭ", "ḭ" },
  { "Ḯ", "Ḯ", "Ḯ", "Ḯ", "Ḯ" },
  { "ḯ", "ḯ", "ḯ", "ḯ", "ḯ" },
  { "Ḱ", "Ḱ", "Ḱ", "Ḱ", "Ḱ" },
  { "ḱ", "ḱ", "ḱ", "ḱ", "ḱ" },
  { "Ḳ", "Ḳ", "Ḳ", "Ḳ", "Ḳ" },
  { "ḳ", "ḳ", "ḳ", "ḳ", "ḳ" },
  { "Ḵ", "Ḵ", "Ḵ", "Ḵ", "Ḵ" },
  { "ḵ", "ḵ", "ḵ", "ḵ", "ḵ" },
  { "Ḷ", "Ḷ", "Ḷ", "Ḷ", "Ḷ" },
  { "ḷ", "ḷ", "ḷ", "ḷ", "ḷ" },
  { "Ḹ", "Ḹ", "Ḹ", "Ḹ", "Ḹ" },
  { "ḹ", "ḹ", "ḹ", "ḹ", "ḹ" },
  { "Ḻ", "Ḻ", "Ḻ", "Ḻ", "Ḻ" },
  { "ḻ", "ḻ", "ḻ", "ḻ", "ḻ" },
  { "Ḽ", "Ḽ", "Ḽ", "Ḽ", "Ḽ" },
  { "ḽ", "ḽ", "ḽ", "ḽ", "ḽ" },
  { "Ḿ", "Ḿ", "Ḿ", "Ḿ", "Ḿ" },
  { "ḿ", "ḿ", "ḿ", "ḿ", "ḿ" },
  { "Ṁ", "Ṁ", "Ṁ", "Ṁ", "Ṁ" },
  { "ṁ", "ṁ", "ṁ", "ṁ", "ṁ" },
  { "Ṃ", "Ṃ", "Ṃ", "Ṃ", "Ṃ" },
  { "ṃ", "ṃ", "ṃ", "ṃ", "ṃ" },
  { "Ṅ", "Ṅ", "Ṅ", "Ṅ", "Ṅ" },
  { "ṅ", "ṅ", "ṅ", "ṅ", "ṅ" },
  { "Ṇ", "Ṇ", "Ṇ", "Ṇ", "Ṇ" },
  { "ṇ", "ṇ", "ṇ", "ṇ", "ṇ" },
  { "Ṉ", "Ṉ", "Ṉ", "Ṉ", "Ṉ" },
  { "ṉ", "ṉ", "ṉ", "ṉ", "ṉ" },
  { "Ṋ", "Ṋ", "Ṋ", "Ṋ", "Ṋ" },
  { "ṋ", "ṋ", "ṋ", "ṋ", "ṋ" },
  { "Ṍ", "Ṍ", "Ṍ", "Ṍ", "Ṍ" },
  { "ṍ", "ṍ", "ṍ", "ṍ", "ṍ" },
  { "Ṏ", "Ṏ", "Ṏ", "Ṏ", "Ṏ" },
  { "ṏ", "ṏ", "ṏ", "ṏ", "ṏ" },
  { "Ṑ", "Ṑ", "Ṑ", "Ṑ", "Ṑ" },
  { "ṑ", "ṑ", "ṑ", "ṑ", "ṑ" },
  { "Ṓ", "Ṓ", "Ṓ", "Ṓ", "Ṓ" },
  { "ṓ", "ṓ", "ṓ", "ṓ", "ṓ" },
  { "Ṕ", "Ṕ", "Ṕ", "Ṕ", "Ṕ" },
  { "ṕ", "ṕ", "ṕ", "ṕ", "ṕ" },
  { "Ṗ", "Ṗ", "Ṗ", "Ṗ", "Ṗ" },
  { "ṗ", "ṗ", "ṗ", "ṗ", "ṗ" },
  { "Ṙ", "Ṙ", "Ṙ", "Ṙ", "Ṙ" },
  { "ṙ", "ṙ", "ṙ", "ṙ", "ṙ" },
  { "Ṛ", "Ṛ", "Ṛ", "Ṛ", "Ṛ" },
  { "ṛ", "ṛ", "ṛ", "ṛ", "ṛ" },
  { "Ṝ", "Ṝ", "Ṝ", "Ṝ", "Ṝ" },
  { "ṝ", "ṝ", "ṝ", "ṝ", "ṝ" },
  { "Ṟ", "Ṟ", "Ṟ", "Ṟ", "Ṟ" },
  { "ṟ", "ṟ", "ṟ", "ṟ", "ṟ" },
  { "Ṡ", "Ṡ", "Ṡ", "Ṡ", "Ṡ" },
  { "ṡ", "ṡ", "ṡ", "ṡ", "ṡ" },
  { "Ṣ", "Ṣ", "Ṣ", "Ṣ", "Ṣ" },
  { "ṣ", "ṣ", "ṣ", "ṣ", "ṣ" },
  { "Ṥ", "Ṥ", "Ṥ", "Ṥ", "Ṥ" },
  { "ṥ", "ṥ", "ṥ", "ṥ", "ṥ" },
  { "Ṧ", "Ṧ", "Ṧ", "Ṧ", "Ṧ" },
  { "ṧ", "ṧ", "ṧ", "ṧ", "ṧ" },
  { "Ṩ", "Ṩ", "Ṩ", "Ṩ", "Ṩ" },
  { "ṩ", "ṩ", "ṩ", "ṩ", "ṩ" },
  { "Ṫ", "Ṫ", "Ṫ", "Ṫ", "Ṫ" },
  { "ṫ", "ṫ", "ṫ", "ṫ", "ṫ" },
  { "Ṭ", "Ṭ", "Ṭ", "Ṭ", "Ṭ" },
  { "ṭ", "ṭ", "ṭ", "ṭ", "ṭ" },
  { "Ṯ", "Ṯ", "Ṯ", "Ṯ", "Ṯ" },
  { "ṯ", "ṯ", "ṯ", "ṯ", "ṯ" },
  { "Ṱ", "Ṱ", "Ṱ", "Ṱ", "Ṱ" },
  { "ṱ", "ṱ", "ṱ", "ṱ", "ṱ" },
  { "Ṳ", "Ṳ", "Ṳ", "Ṳ", "Ṳ" },
  { "ṳ", "ṳ", "ṳ", "ṳ", "ṳ" },
  { "Ṵ", "Ṵ", "Ṵ", "Ṵ", "Ṵ" },
  { "ṵ", "ṵ", "ṵ", "ṵ", "ṵ" },
  { "Ṷ", "Ṷ", "Ṷ", "Ṷ", "Ṷ" },
  { "ṷ", "ṷ", "ṷ", "ṷ", "ṷ" },
  { "Ṹ", "Ṹ", "Ṹ", "Ṹ", "Ṹ" },
  { "ṹ", "ṹ", "ṹ", "ṹ", "ṹ" },
  { "Ṻ", "Ṻ", "Ṻ", "Ṻ", "Ṻ" },
  { "ṻ", "ṻ", "ṻ", "ṻ", "ṻ" },
  { "Ṽ", "Ṽ", "Ṽ", "Ṽ", "Ṽ" },
  { "ṽ", "ṽ", "ṽ", "ṽ", "ṽ" },
  { "Ṿ", "Ṿ", "Ṿ", "Ṿ", "Ṿ" },
  { "ṿ", "ṿ", "ṿ", "ṿ", "ṿ" },
  { "Ẁ", "Ẁ", "Ẁ", "Ẁ", "Ẁ" },
  { "ẁ", "ẁ", "ẁ", "ẁ", "ẁ" },
  { "Ẃ", "Ẃ", "Ẃ", "Ẃ", "Ẃ" },
  { "ẃ", "ẃ", "ẃ", "ẃ", "ẃ" },
  { "Ẅ", "Ẅ", "Ẅ", "Ẅ", "Ẅ" },
  { "ẅ", "ẅ", "ẅ", "ẅ", "ẅ" },
  { "Ẇ", "Ẇ", "Ẇ", "Ẇ", "Ẇ" },
  { "ẇ", "ẇ", "ẇ", "ẇ", "ẇ" },
  { "Ẉ", "Ẉ", "Ẉ", "Ẉ", "Ẉ" },
  { "ẉ", "ẉ", "ẉ", "ẉ", "ẉ" },
  { "Ẋ", "Ẋ", "Ẋ", "Ẋ", "Ẋ" },
  { "ẋ", "ẋ", "ẋ", "ẋ", "ẋ" },
  { "Ẍ", "Ẍ", "Ẍ", "Ẍ", "Ẍ" },
  { "ẍ", "ẍ", "ẍ", "ẍ", "ẍ" },
  { "Ẏ", "Ẏ", "Ẏ", "Ẏ", "Ẏ" },
  { "ẏ", "ẏ", "ẏ", "ẏ", "ẏ" },
  { "Ẑ", "Ẑ", "Ẑ", "Ẑ", "Ẑ" },
  { "ẑ", "ẑ", "ẑ", "ẑ", "ẑ" },
  { "Ẓ", "Ẓ", "Ẓ", "Ẓ", "Ẓ" },
  { "ẓ", "ẓ", "ẓ", "ẓ", "ẓ" },
  { "Ẕ", "Ẕ", "Ẕ", "Ẕ", "Ẕ" },
  { "ẕ", "ẕ", "ẕ", "ẕ", "ẕ" },
  { "ẖ", "ẖ", "ẖ", "ẖ", "ẖ" },
  { "ẗ", "ẗ", "ẗ", "ẗ", "ẗ" },
  { "ẘ", "ẘ", "ẘ", "ẘ", "ẘ" },
  { "ẙ", "ẙ", "ẙ", "ẙ", "ẙ" },
  { "ẚ", "ẚ", "ẚ", "aʾ", "aʾ" },
  { "ẛ", "ẛ", "ẛ", "ṡ", "ṡ" },
  { "Ạ", "Ạ", "Ạ", "Ạ", "Ạ" },
  { "ạ", "ạ", "ạ", "ạ", "ạ" },
  { "Ả", "Ả", "Ả", "Ả", "Ả" },
  { "ả", "ả", "ả", "ả", "ả" },
  { "Ấ", "Ấ", "Ấ", "Ấ", "Ấ" },
  { "ấ", "ấ", "ấ", "ấ", "ấ" },
  { "Ầ", "Ầ", "Ầ", "Ầ", "Ầ" },
  { "ầ", "ầ", "ầ", "ầ", "ầ" },
  { "Ẩ", "Ẩ", "Ẩ", "Ẩ", "Ẩ" },
  { "ẩ", "ẩ", "ẩ", "ẩ", "ẩ" },
  { "Ẫ", "Ẫ", "Ẫ", "Ẫ", "Ẫ" },
  { "ẫ", "ẫ", "ẫ", "ẫ", "ẫ" },
  { "Ậ", "Ậ", "Ậ", "Ậ", "Ậ" },
  { "ậ", "ậ", "ậ", "ậ", "ậ" },
  { "Ắ", "Ắ", "Ắ", "Ắ", "Ắ" },
  { "ắ", "ắ", "ắ", "ắ", "ắ" },
  { "Ằ", "Ằ", "Ằ", "Ằ", "Ằ" },
  { "ằ", "ằ", "ằ", "ằ", "ằ" },
  { "Ẳ", "Ẳ", "Ẳ", "Ẳ", "Ẳ" },
  { "ẳ", "ẳ", "ẳ", "ẳ", "ẳ" },
  { "Ẵ", "Ẵ", "Ẵ", "Ẵ", "Ẵ" },
  { "ẵ", "ẵ", "ẵ", "ẵ", "ẵ" },
  { "Ặ", "Ặ", "Ặ", "Ặ", "Ặ" },
  { "ặ", "ặ", "ặ", "ặ", "ặ" },
  { "Ẹ", "Ẹ", "Ẹ", "Ẹ", "Ẹ" },
  { "ẹ", "ẹ", "ẹ", "ẹ", "ẹ" },
  { "Ẻ", "Ẻ", "Ẻ", "Ẻ", "Ẻ" },
  { "ẻ", "ẻ", "ẻ", "ẻ", "ẻ" },
  { "Ẽ", "Ẽ", "Ẽ", "Ẽ", "Ẽ" },
  { "ẽ", "ẽ", "ẽ", "ẽ", "ẽ" },
  { "Ế", "Ế", "Ế", "Ế", "Ế" },
  { "ế", "ế", "ế", "ế", "ế" },
  { "Ề", "Ề", "Ề", "Ề", "Ề" },
  { "ề", "ề", "ề", "ề", "ề" },
  { "Ể", "Ể", "Ể", "Ể", "Ể" },
  { "ể", "ể", "ể", "ể", "ể" },
  { "Ễ", "Ễ", "Ễ", "Ễ", "Ễ" },
  { "ễ", "ễ", "ễ", "ễ", "ễ" },
  { "Ệ", "Ệ", "Ệ", "Ệ", "Ệ" },
  { "ệ", "ệ", "ệ", "ệ", "ệ" },
  { "Ỉ", "Ỉ", "Ỉ", "Ỉ", "Ỉ" },
  { "ỉ", "ỉ", "ỉ", "ỉ", "ỉ" },
  { "Ị", "Ị", "Ị", "Ị", "Ị" },
  { "ị", "ị", "ị", "ị", "ị" },
  { "Ọ", "Ọ", "Ọ", "Ọ", "Ọ" },
  { "ọ", "ọ", "ọ", "ọ", "ọ" },
  { "Ỏ", "Ỏ", "Ỏ", "Ỏ", "Ỏ" },
  { "ỏ", "ỏ", "ỏ", "ỏ", "ỏ" },
  { "Ố", "Ố", "Ố", "Ố", "Ố" },
  { "ố", "ố", "ố", "ố", "ố" },
  { "Ồ", "Ồ", "Ồ", "Ồ", "Ồ" },
  { "ồ", "ồ", "ồ", "ồ", "ồ" },
  { "Ổ", "Ổ", "Ổ", "Ổ", "Ổ" },
  { "ổ", "ổ", "ổ", "ổ", "ổ" },
  { "Ỗ", "Ỗ", "Ỗ", "Ỗ", "Ỗ" },
  { "ỗ", "ỗ", "ỗ", "ỗ", "ỗ" },
  { "Ộ", "Ộ", "Ộ", "Ộ", "Ộ" },
  { "ộ", "ộ", "ộ", "ộ", "ộ" },
  { "Ớ", "Ớ", "Ớ", "Ớ", "Ớ" },
  { "ớ", "ớ", "ớ", "ớ", "ớ" },
  { "Ờ", "Ờ", "Ờ", "Ờ", "Ờ" },
  { "ờ", "ờ", "ờ", "ờ", "ờ" },
  { "Ở", "Ở", "Ở", "Ở", "Ở" },
  { "ở", "ở", "ở", "ở", "ở" },
  { "Ỡ", "Ỡ", "Ỡ", "Ỡ", "Ỡ" },
  { "ỡ", "ỡ", "ỡ", "ỡ", "ỡ" },
  { "Ợ", "Ợ", "Ợ", "Ợ", "Ợ" },
  { "ợ", "ợ", "ợ", "ợ", "ợ" },
  { "Ụ", "Ụ", "Ụ", "Ụ", "Ụ" },
  { "ụ", "ụ", "ụ", "ụ", "ụ" },
  { "Ủ", "Ủ", "Ủ", "Ủ", "Ủ" },
  { "ủ", "ủ", "ủ", "ủ", "ủ" },
  { "Ứ", "Ứ", "Ứ", "Ứ", "Ứ" },
  { "ứ", "ứ", "ứ", "ứ", "ứ" },
  { "Ừ", "Ừ", "Ừ", "Ừ", "Ừ" },
  { "ừ", "ừ", "ừ", "ừ", "ừ" },
  { "Ử", "Ử", "Ử", "Ử", "Ử" },
  { "ử", "ử", "ử", "ử", "ử" },
  { "Ữ", "Ữ", "Ữ", "Ữ", "Ữ" },
  { "ữ", "ữ", "ữ", "ữ", "ữ" },
  { "Ự", "Ự", "Ự", "Ự", "Ự" },
  { "ự", "ự", "ự", "ự", "ự" },
  { "Ỳ", "Ỳ", "Ỳ", "Ỳ", "Ỳ" },
  { "ỳ", "ỳ", "ỳ", "ỳ", "ỳ" },
  { "Ỵ", "Ỵ", "Ỵ", "Ỵ", "Ỵ" },
  { "ỵ", "ỵ", "ỵ", "ỵ", "ỵ" },
  { "Ỷ", "Ỷ", "Ỷ", "Ỷ", "Ỷ" },
  { "ỷ", "ỷ", "ỷ", "ỷ", "ỷ" },
  { "Ỹ", "Ỹ", "Ỹ", "Ỹ", "Ỹ" },
  { "ỹ", "ỹ", "ỹ", "ỹ", "ỹ" },
  { "ἀ", "ἀ", "ἀ", "ἀ", "ἀ" },
  { "ἁ", "ἁ", "ἁ", "ἁ", "ἁ" },
  { "ἂ", "ἂ", "ἂ", "ἂ", "ἂ" },
  { "ἃ", "ἃ", "ἃ", "ἃ", "ἃ" },
  { "ἄ", "ἄ", "ἄ", "ἄ", "ἄ" },
  { "ἅ", "ἅ", "ἅ", "ἅ", "ἅ" },
  { "ἆ", "ἆ", "ἆ", "ἆ", "ἆ" },
  { "ἇ", "ἇ", "ἇ", "ἇ", "ἇ" },
  { "Ἀ", "Ἀ", "Ἀ", "Ἀ", "Ἀ" },
  { "Ἁ", "Ἁ", "Ἁ", "Ἁ", "Ἁ" },
  { "Ἂ", "Ἂ", "Ἂ", "Ἂ", "Ἂ" },
  { "Ἃ", "Ἃ", "Ἃ", "Ἃ", "Ἃ" },
  { "Ἄ", "Ἄ", "Ἄ", "Ἄ", "Ἄ" },
  { "Ἅ", "Ἅ", "Ἅ", "Ἅ", "Ἅ" },
  { "Ἆ", "Ἆ", "Ἆ", "Ἆ", "Ἆ" },
  { "Ἇ", "Ἇ", "Ἇ", "Ἇ", "Ἇ" },
  { "ἐ", "ἐ", "ἐ", "ἐ", "ἐ" },
  { "ἑ", "ἑ", "ἑ", "ἑ", "ἑ" },
  { "ἒ", "ἒ", "ἒ", "ἒ", "ἒ" },
  { "ἓ", "ἓ", "ἓ", "ἓ", "ἓ" },
  { "ἔ", "ἔ", "ἔ", "ἔ", "ἔ" },
  { "ἕ", "ἕ", "ἕ", "ἕ", "ἕ" },
  { "Ἐ", "Ἐ", "Ἐ", "Ἐ", "Ἐ" },
  { "Ἑ", "Ἑ", "Ἑ", "Ἑ", "Ἑ" },
  { "Ἒ", "Ἒ", "Ἒ", "Ἒ", "Ἒ" },
  { "Ἓ", "Ἓ", "Ἓ", "Ἓ", "Ἓ" },
  { "Ἔ", "Ἔ", "Ἔ", "Ἔ", "Ἔ" },
  { "Ἕ", "Ἕ", "Ἕ", "Ἕ", "Ἕ" },
  { "ἠ", "ἠ", "ἠ", "ἠ", "ἠ" },
  { "ἡ", "ἡ", "ἡ", "ἡ", "ἡ" },
  { "ἢ", "ἢ", "ἢ", "ἢ", "ἢ" },
  { "ἣ", "ἣ", "ἣ", "ἣ", "ἣ" },
  { "ἤ", "ἤ", "ἤ", "ἤ", "ἤ" },
  { "ἥ", "ἥ", "ἥ", "ἥ", "ἥ" },
  { "ἦ", "ἦ", "ἦ", "ἦ", "ἦ" },
  { "ἧ", "ἧ", "ἧ", "ἧ", "ἧ" },
  { "Ἠ", "Ἠ", "Ἠ", "Ἠ", "Ἠ" },
  { "Ἡ", "Ἡ", "Ἡ", "Ἡ", "Ἡ" },
  { "Ἢ", "Ἢ", "Ἢ", "Ἢ", "Ἢ" },
  { "Ἣ", "Ἣ", "Ἣ", "Ἣ", "Ἣ" },
  { "Ἤ", "Ἤ", "Ἤ", "Ἤ", "Ἤ" },
  { "Ἥ", "Ἥ", "Ἥ", "Ἥ", "Ἥ" },
  { "Ἦ", "Ἦ", "Ἦ", "Ἦ", "Ἦ" },
  { "Ἧ", "Ἧ", "Ἧ", "Ἧ", "Ἧ" },
  { "ἰ", "ἰ", "ἰ", "ἰ", "ἰ" },
  { "ἱ", "ἱ", "ἱ", "ἱ", "ἱ" },
  { "ἲ", "ἲ", "ἲ", "ἲ", "ἲ" },
  { "ἳ", "ἳ", "ἳ", "ἳ", "ἳ" },
  { "ἴ", "ἴ", "ἴ", "ἴ", "ἴ" },
  { "ἵ", "ἵ", "ἵ", "ἵ", "ἵ" },
  { "ἶ", "ἶ", "ἶ", "ἶ", "ἶ" },
  { "ἷ", "ἷ", "ἷ", "ἷ", "ἷ" },
  { "Ἰ", "Ἰ", "Ἰ", "Ἰ", "Ἰ" },
  { "Ἱ", "Ἱ", "Ἱ", "Ἱ", "Ἱ" },
  { "Ἲ", "Ἲ", "Ἲ", "Ἲ", "Ἲ" },
  { "Ἳ", "Ἳ", "Ἳ", "Ἳ", "Ἳ" },
  { "Ἴ", "Ἴ", "Ἴ", "Ἴ", "Ἴ" },
  { "Ἵ", "Ἵ", "Ἵ", "Ἵ", "Ἵ" },
  { "Ἶ", "Ἶ", "Ἶ", "Ἶ", "Ἶ" },
  { "Ἷ", "Ἷ", "Ἷ", "Ἷ", "Ἷ" },
  { "ὀ", "ὀ", "ὀ", "ὀ", "ὀ" },
  { "ὁ", "ὁ", "ὁ", "ὁ", "ὁ" },
  { "ὂ", "ὂ", "ὂ", "ὂ", "ὂ" },
  { "ὃ", "ὃ", "ὃ", "ὃ", "ὃ" },
  { "ὄ", "ὄ", "ὄ", "ὄ", "ὄ" },
  { "ὅ", "ὅ", "ὅ", "ὅ", "ὅ" },
  { "Ὀ", "Ὀ", "Ὀ", "Ὀ", "Ὀ" },
  { "Ὁ", "Ὁ", "Ὁ", "Ὁ", "Ὁ" },
  { "Ὂ", "Ὂ", "Ὂ", "Ὂ", "Ὂ" },
  { "Ὃ", "Ὃ", "Ὃ", "Ὃ", "Ὃ" },
  { "Ὄ", "Ὄ", "Ὄ", "Ὄ", "Ὄ" },
  { "Ὅ", "Ὅ", "Ὅ", "Ὅ", "Ὅ" },
  { "ὐ", "ὐ", "ὐ", "ὐ", "ὐ" },
  { "ὑ", "ὑ", "ὑ", "ὑ", "ὑ" },
  { "ὒ", "ὒ", "ὒ", "ὒ", "ὒ" },
  { "ὓ", "ὓ", "ὓ", "ὓ", "ὓ" },
  { "ὔ", "ὔ", "ὔ", "ὔ", "ὔ" },
  { "ὕ", "ὕ", "ὕ", "ὕ", "ὕ" },
  { "ὖ", "ὖ", "ὖ", "ὖ", "ὖ" },
  { "ὗ", "ὗ", "ὗ", "ὗ", "ὗ" },
  { "Ὑ", "Ὑ", "Ὑ", "Ὑ", "Ὑ" },
  { "Ὓ", "Ὓ", "Ὓ", "Ὓ", "Ὓ" },
  { "Ὕ", "Ὕ", "Ὕ", "Ὕ", "Ὕ" },
  { "Ὗ", "Ὗ", "Ὗ", "Ὗ", "Ὗ" },
  { "ὠ", "ὠ", "ὠ", "ὠ", "ὠ" },
  { "ὡ", "ὡ", "ὡ", "ὡ", "ὡ" },
  { "ὢ", "ὢ", "ὢ", "ὢ", "ὢ" },
  { "ὣ", "ὣ", "ὣ", "ὣ", "ὣ" },
  { "ὤ", "ὤ", "ὤ", "ὤ", "ὤ" },
  { "ὥ", "ὥ", "ὥ", "ὥ", "ὥ" },
  { "ὦ", "ὦ", "ὦ", "ὦ", "ὦ" },
  { "ὧ", "ὧ", "ὧ", "ὧ", "ὧ" },
  { "Ὠ", "Ὠ", "Ὠ", "Ὠ", "Ὠ" },
  { "Ὡ", "Ὡ", "Ὡ", "Ὡ", "Ὡ" },
  { "Ὢ", "Ὢ", "Ὢ", "Ὢ", "Ὢ" },
  { "Ὣ", "Ὣ", "Ὣ", "Ὣ", "Ὣ" },
  { "Ὤ", "Ὤ", "Ὤ", "Ὤ", "Ὤ" },
  { "Ὥ", "Ὥ", "Ὥ", "Ὥ", "Ὥ" },
  { "Ὦ", "Ὦ", "Ὦ", "Ὦ", "Ὦ" },
  { "Ὧ", "Ὧ", "Ὧ", "Ὧ", "Ὧ" },
  { "ὰ", "ὰ", "ὰ", "ὰ", "ὰ" },
  { "ά", "ά", "ά", "ά", "ά" },
  { "ὲ", "ὲ", "ὲ", "ὲ", "ὲ" },
  { "έ", "έ", "έ", "έ", "έ" },
  { "ὴ", "ὴ", "ὴ", "ὴ", "ὴ" },
  { "ή", "ή", "ή", "ή", "ή" },
  { "ὶ", "ὶ", "ὶ", "ὶ", "ὶ" },
  { "ί", "ί", "ί", "ί", "ί" },
  { "ὸ", "ὸ", "ὸ", "ὸ", "ὸ" },
  { "ό", "ό", "ό", "ό", "ό" },
  { "ὺ", "ὺ", "ὺ", "ὺ", "ὺ" },
  { "ύ", "ύ", "ύ", "ύ", "ύ" },
  { "ὼ", "ὼ", "ὼ", "ὼ", "ὼ" },
  { "ώ", "ώ", "ώ", "ώ", "ώ" },
  { "ᾀ", "ᾀ", "ᾀ", "ᾀ", "ᾀ" },
  { "ᾁ", "ᾁ", "ᾁ", "ᾁ", "ᾁ" },
  { "ᾂ", "ᾂ", "ᾂ", "ᾂ", "ᾂ" },
  { "ᾃ", "ᾃ", "ᾃ", "ᾃ", "ᾃ" },
  { "ᾄ", "ᾄ", "ᾄ", "ᾄ", "ᾄ" },
  { "ᾅ", "ᾅ", "ᾅ", "ᾅ", "ᾅ" },
  { "ᾆ", "ᾆ", "ᾆ", "ᾆ", "ᾆ" },
  { "ᾇ", "ᾇ", "ᾇ", "ᾇ", "ᾇ" },
  { "ᾈ", "ᾈ", "ᾈ", "ᾈ", "ᾈ" },
  { "ᾉ", "ᾉ", "ᾉ", "ᾉ", "ᾉ" },
  { "ᾊ", "ᾊ", "ᾊ", "ᾊ", "ᾊ" },
  { "ᾋ", "ᾋ", "ᾋ", "ᾋ", "ᾋ" },
  { "ᾌ", "ᾌ", "ᾌ", "ᾌ", "ᾌ" },
  { "ᾍ", "ᾍ", "ᾍ", "ᾍ", "ᾍ" },
  { "ᾎ", "ᾎ", "ᾎ", "ᾎ", "ᾎ" },
  { "ᾏ", "ᾏ", "ᾏ", "ᾏ", "ᾏ" },
  { "ᾐ", "ᾐ", "ᾐ", "ᾐ", "ᾐ" },
  { "ᾑ", "ᾑ", "ᾑ", "ᾑ", "ᾑ" },
  { "ᾒ", "ᾒ", "ᾒ", "ᾒ", "ᾒ" },
  { "ᾓ", "ᾓ", "ᾓ", "ᾓ", "ᾓ" },
  { "ᾔ", "ᾔ", "ᾔ", "ᾔ", "ᾔ" },
  { "ᾕ", "ᾕ", "ᾕ", "ᾕ", "ᾕ" },
  { "ᾖ", "ᾖ", "ᾖ", "ᾖ", "ᾖ" },
  { "ᾗ", "ᾗ", "ᾗ", "ᾗ", "ᾗ" },
  { "ᾘ", "ᾘ", "ᾘ", "ᾘ", "ᾘ" },
  { "ᾙ", "ᾙ", "ᾙ", "ᾙ", "ᾙ" },
  { "ᾚ", "ᾚ", "ᾚ", "ᾚ", "ᾚ" },
  { "ᾛ", "ᾛ", "ᾛ", "ᾛ", "ᾛ" },
  { "ᾜ", "ᾜ", "ᾜ", "ᾜ", "ᾜ" },
  { "ᾝ", "ᾝ", "ᾝ", "ᾝ", "ᾝ" },
  { "ᾞ", "ᾞ", "ᾞ", "ᾞ", "ᾞ" },
  { "ᾟ", "ᾟ", "ᾟ", "ᾟ", "ᾟ" },
  { "ᾠ", "ᾠ", "ᾠ", "ᾠ", "ᾠ" },
  { "ᾡ", "ᾡ", "ᾡ", "ᾡ", "ᾡ" },
  { "ᾢ", "ᾢ", "ᾢ", "ᾢ", "ᾢ" },
  { "ᾣ", "ᾣ", "ᾣ", "ᾣ", "ᾣ" },
  { "ᾤ", "ᾤ", "ᾤ", "ᾤ", "ᾤ" },
  { "ᾥ", "ᾥ", "ᾥ", "ᾥ", "ᾥ" },
  { "ᾦ", "ᾦ", "ᾦ", "ᾦ", "ᾦ" },
  { "ᾧ", "ᾧ", "ᾧ", "ᾧ", "ᾧ" },
  { "ᾨ", "ᾨ", "ᾨ", "ᾨ", "ᾨ" },
  { "ᾩ", "ᾩ", "ᾩ", "ᾩ", "ᾩ" },
  { "ᾪ", "ᾪ", "ᾪ", "ᾪ", "ᾪ" },
  { "ᾫ", "ᾫ", "ᾫ", "ᾫ", "ᾫ" },
  { "ᾬ", "ᾬ", "ᾬ", "ᾬ", "ᾬ" },
  { "ᾭ", "ᾭ", "ᾭ", "ᾭ", "ᾭ" },
  { "ᾮ", "ᾮ", "ᾮ", "ᾮ", "ᾮ" },
  { "ᾯ", "ᾯ", "ᾯ", "ᾯ", "ᾯ" },
  { "ᾰ", "ᾰ", "ᾰ", "ᾰ", "ᾰ" },
  { "ᾱ", "ᾱ", "ᾱ", "ᾱ", "ᾱ" },
  { "ᾲ", "ᾲ", "ᾲ", "ᾲ", "ᾲ" },
  { "ᾳ", "ᾳ", "ᾳ", "ᾳ", "ᾳ" },
  { "ᾴ", "ᾴ", "ᾴ", "ᾴ", "ᾴ" },
  { "ᾶ", "ᾶ", "ᾶ", "ᾶ", "ᾶ" },
  { "ᾷ", "ᾷ", "ᾷ", "ᾷ", "ᾷ" },
  { "Ᾰ", "Ᾰ", "Ᾰ", "Ᾰ", "Ᾰ" },
  { "Ᾱ", "Ᾱ", "Ᾱ", "Ᾱ", "Ᾱ" },
  { "Ὰ", "Ὰ", "Ὰ", "Ὰ", "Ὰ" },
  { "Ά", "Ά", "Ά", "Ά", "Ά" },
  { "ᾼ", "ᾼ", "ᾼ", "ᾼ", "ᾼ" },
  { "᾽", "᾽", "᾽", " ̓", " ̓" },
  { "ι", "ι", "ι", "ι", "ι" },
  { "᾿", "᾿", "᾿", " ̓", " ̓" },
  { "῀", "῀", "῀", " ͂", " ͂" },
  { "῁", "῁", "῁", " ̈͂", " ̈͂" },
  { "ῂ", "ῂ", "ῂ", "ῂ", "ῂ" },
  { "ῃ", "ῃ", "ῃ", "ῃ", "ῃ" },
  { "ῄ", "ῄ", "ῄ", "ῄ", "ῄ" },
  { "ῆ", "ῆ", "ῆ", "ῆ", "ῆ" },
  { "ῇ", "ῇ", "ῇ", "ῇ", "ῇ" },
  { "Ὲ", "Ὲ", "Ὲ", "Ὲ", "Ὲ" },
  { "Έ", "Έ", "Έ", "Έ", "Έ" },
  { "Ὴ", "Ὴ", "Ὴ", "Ὴ", "Ὴ" },
  { "Ή", "Ή", "Ή", "Ή", "Ή" },
  { "ῌ", "ῌ", "ῌ", "ῌ", "ῌ" },
  { "῍", "῍", "῍", " ̓̀", " ̓̀" },
  { "῎", "῎", "῎", " ̓́", " ̓́" },
  { "῏", "῏", "῏", " ̓͂", " ̓͂" },
  { "ῐ", "ῐ", "ῐ", "ῐ", "ῐ" },
  { "ῑ", "ῑ", "ῑ", "ῑ", "ῑ" },
  { "ῒ", "ῒ", "ῒ", "ῒ", "ῒ" },
  { "ΐ", "ΐ", "ΐ", "ΐ", "ΐ" },
  { "ῖ", "ῖ", "ῖ", "ῖ", "ῖ" },
  { "ῗ", "ῗ", "ῗ", "ῗ", "ῗ" },
  { "Ῐ", "Ῐ", "Ῐ", "Ῐ", "Ῐ" },
  { "Ῑ", "Ῑ", "Ῑ", "Ῑ", "Ῑ" },
  { "Ὶ", "Ὶ", "Ὶ", "Ὶ", "Ὶ" },
  { "Ί", "Ί", "Ί", "Ί", "Ί" },
  { "῝", "῝", "῝", " ̔̀", " ̔̀" },
  { "῞", "῞", "῞", " ̔́", " ̔́" },
  { "῟", "῟", "῟", " ̔͂", " ̔͂" },
  { "ῠ", "ῠ", "ῠ", "ῠ", "ῠ" },
  { "ῡ", "ῡ", "ῡ", "ῡ", "ῡ" },
  { "ῢ", "ῢ", "ῢ", "ῢ", "ῢ" },
  { "ΰ", "ΰ", "ΰ", "ΰ", "ΰ" },
  { "ῤ", "ῤ", "ῤ", "ῤ", "ῤ" },
  { "ῥ", "ῥ", "ῥ", "ῥ", "ῥ" },
  { "ῦ", "ῦ", "ῦ", "ῦ", "ῦ" },
  { "ῧ", "ῧ", "ῧ", "ῧ", "ῧ" },
  { "Ῠ", "Ῠ", "Ῠ", "Ῠ", "Ῠ" },
  { "Ῡ", "Ῡ", "Ῡ", "Ῡ", "Ῡ" },
  { "Ὺ", "Ὺ", "Ὺ", "Ὺ", "Ὺ" },
  { "Ύ", "Ύ", "Ύ", "Ύ", "Ύ" },
  { "Ῥ", "Ῥ", "Ῥ", "Ῥ", "Ῥ" },
  { "῭", "῭", "῭", " ̈̀", " ̈̀" },
  { "΅", "΅", "΅", " ̈́", " ̈́" },
  { "`", "`", "`", "`", "`" },
  { "ῲ", "ῲ", "ῲ", "ῲ", "ῲ" },
  { "ῳ", "ῳ", "ῳ", "ῳ", "ῳ" },
  { "ῴ", "ῴ", "ῴ", "ῴ", "ῴ" },
  { "ῶ", "ῶ", "ῶ", "ῶ", "ῶ" },
  { "ῷ", "ῷ", "ῷ", "ῷ", "ῷ" },
  { "Ὸ", "Ὸ", "Ὸ", "Ὸ", "Ὸ" },
  { "Ό", "Ό", "Ό", "Ό", "Ό" },
  { "Ὼ", "Ὼ", "Ὼ", "Ὼ", "Ὼ" },
  { "Ώ", "Ώ", "Ώ", "Ώ", "Ώ" },
  { "ῼ", "ῼ", "ῼ", "ῼ", "ῼ" },
  { "´", "´", "´", " ́", " ́" },
  { "῾", "῾", "῾", " ̔", " ̔" },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { " ", " ", " ", " ", " " },
  { "‑", "‑", "‑", "‐", "‐" },
  { "‗", "‗", "‗", " ̳", " ̳" },
  { "․", "․", "․", ".", "." },
  { "‥", "‥", "‥", "..", ".." },
  { "…", "…", "…", "...", "..." },
  { " ", " ", " ", " ", " " },
  { "″", "″", "″", "′′", "′′" },
  { "‴", "‴", "‴", "′′′", "′′′" },
  { "‶", "‶", "‶", "‵‵", "‵‵" },
  { "‷", "‷", "‷", "‵‵‵", "‵‵‵" },
  { "‼", "‼", "‼", "!!", "!!" },
  { "‾", "‾", "‾", " ̅", " ̅" },
  { "⁇", "⁇", "⁇", "??", "??" },
  { "⁈", "⁈", "⁈", "?!", "?!" },
  { "⁉", "⁉", "⁉", "!?", "!?" },
  { "⁗", "⁗", "⁗", "′′′′", "′′′′" },
  { " ", " ", " ", " ", " " },
  { "⁰", "⁰", "⁰", "0", "0" },
  { "ⁱ", "ⁱ", "ⁱ", "i", "i" },
  { "⁴", "⁴", "⁴", "4", "4" },
  { "⁵", "⁵", "⁵", "5", "5" },
  { "⁶", "⁶", "⁶", "6", "6" },
  { "⁷", "⁷", "⁷", "7", "7" },
  { "⁸", "⁸", "⁸", "8", "8" },
  { "⁹", "⁹", "⁹", "9", "9" },
  { "⁺", "⁺", "⁺", "+", "+" },
  { "⁻", "⁻", "⁻", "−", "−" },
  { "⁼", "⁼", "⁼", "=", "=" },
  { "⁽", "⁽", "⁽", "(", "(" },
  { "⁾", "⁾", "⁾", ")", ")" },
  { "ⁿ", "ⁿ", "ⁿ", "n", "n" },
  { "₀", "₀", "₀", "0", "0" },
  { "₁", "₁", "₁", "1", "1" },
  { "₂", "₂", "₂", "2", "2" },
  { "₃", "₃", "₃", "3", "3" },
  { "₄", "₄", "₄", "4", "4" },
  { "₅", "₅", "₅", "5", "5" },
  { "₆", "₆", "₆", "6", "6" },
  { "₇", "₇", "₇", "7", "7" },
  { "₈", "₈", "₈", "8", "8" },
  { "₉", "₉", "₉", "9", "9" },
  { "₊", "₊", "₊", "+", "+" },
  { "₋", "₋", "₋", "−", "−" },
  { "₌", "₌", "₌", "=", "=" },
  { "₍", "₍", "₍", "(", "(" },
  { "₎", "₎", "₎", ")", ")" },
  { "ₐ", "ₐ", "ₐ", "a", "a" },
  { "ₑ", "ₑ", "ₑ", "e", "e" },
  { "ₒ", "ₒ", "ₒ", "o", "o" },
  { "ₓ", "ₓ", "ₓ", "x", "x" },
  { "ₔ", "ₔ", "ₔ", "ə", "ə" },
  { "ₕ", "ₕ", "ₕ", "h", "h" },
  { "ₖ", "ₖ", "ₖ", "k", "k" },
  { "ₗ", "ₗ", "ₗ", "l", "l" },
  { "ₘ", "ₘ", "ₘ", "m", "m" },
  { "ₙ", "ₙ", "ₙ", "n", "n" },
  { "ₚ", "ₚ", "ₚ", "p", "p" },
  { "ₛ", "ₛ", "ₛ", "s", "s" },
  { "ₜ", "ₜ", "ₜ", "t", "t" },
  { "₨", "₨", "₨", "Rs", "Rs" },
  { "℀", "℀", "℀", "a/c", "a/c" },
  { "℁", "℁", "℁", "a/s", "a/s" },
  { "ℂ", "ℂ", "ℂ", "C", "C" },
  { "℃", "℃", "℃", "°C", "°C" },
  { "℅", "℅", "℅", "c/o", "c/o" },
  { "℆", "℆", "℆", "c/u", "c/u" },
  { "ℇ", "ℇ", "ℇ", "Ɛ", "Ɛ" },
  { "℉", "℉", "℉", "°F", "°F" },
  { "ℊ", "ℊ", "ℊ", "g", "g" },
  { "ℋ", "ℋ", "ℋ", "H", "H" },
  { "ℌ", "ℌ", "ℌ", "H", "H" },
  { "ℍ", "ℍ", "ℍ", "H", "H" },
  { "ℎ", "ℎ", "ℎ", "h", "h" },
  { "ℏ", "ℏ", "ℏ", "ħ", "ħ" },
  { "ℐ", "ℐ", "ℐ", "I", "I" },
  { "ℑ", "ℑ", "ℑ", "I", "I" },
  { "ℒ", "ℒ", "ℒ", "L", "L" },
  { "ℓ", "ℓ", "ℓ", "l", "l" },
  { "ℕ", "ℕ", "ℕ", "N", "N" },
  { "№", "№", "№", "No", "No" },
  { "ℙ", "ℙ", "ℙ", "P", "P" },
  { "ℚ", "ℚ", "ℚ", "Q", "Q" },
  { "ℛ", "ℛ", "ℛ", "R", "R" },
  { "ℜ", "ℜ", "ℜ", "R", "R" },
  { "ℝ", "ℝ", "ℝ", "R", "R" },
  { "℠", "℠", "℠", "SM", "SM" },
  { "℡", "℡", "℡", "TEL", "TEL" },
  { "™", "™", "™", "TM", "TM" },
  { "ℤ", "ℤ", "ℤ", "Z", "Z" },
  { "Ω", "Ω", "Ω", "Ω", "Ω" },
  { "ℨ", "ℨ", "ℨ", "Z", "Z" },
  { "K", "K", "K", "K", "K" },
  { "Å", "Å", "Å", "Å", "Å" },
  { "ℬ", "ℬ", "ℬ", "B", "B" },
  { "ℭ", "ℭ", "ℭ", "C", "C" },
  { "ℯ", "ℯ", "ℯ", "e", "e" },
  { "ℰ", "ℰ", "ℰ", "E", "E" },
  { "ℱ", "ℱ", "ℱ", "F", "F" },
  { "ℳ", "ℳ", "ℳ", "M", "M" },
  { "ℴ", "ℴ", "ℴ", "o", "o" },
  { "ℵ", "ℵ", "ℵ", "א", "א" },
  { "ℶ", "ℶ", "ℶ", "ב", "ב" },
  { "ℷ", "ℷ", "ℷ", "ג", "ג" },
  { "ℸ", "ℸ", "ℸ", "ד", "ד" },
  { "ℹ", "ℹ", "ℹ", "i", "i" },
  { "℻", "℻", "℻", "FAX", "FAX" },
  { "ℼ", "ℼ", "ℼ", "π", "π" },
  { "ℽ", "ℽ", "ℽ", "γ", "γ" },
  { "ℾ", "ℾ", "ℾ", "Γ", "Γ" },
  { "ℿ", "ℿ", "ℿ", "Π", "Π" },
  { "⅀", "⅀", "⅀", "∑", "∑" },
  { "ⅅ", "ⅅ", "ⅅ", "D", "D" },
  { "ⅆ", "ⅆ", "ⅆ", "d", "d" },
  { "ⅇ", "ⅇ", "ⅇ", "e", "e" },
  { "ⅈ", "ⅈ", "ⅈ", "i", "i" },
  { "ⅉ", "ⅉ", "ⅉ", "j", "j" },
  { "⅐", "⅐", "⅐", "1⁄7", "1⁄7" },
  { "⅑", "⅑", "⅑", "1⁄9", "1⁄9" },
  { "⅒", "⅒", "⅒", "1⁄10", "1⁄10" },
  { "⅓", "⅓", "⅓", "1⁄3", "1⁄3" },
  { "⅔", "⅔", "⅔", "2⁄3", "2⁄3" },
  { "⅕", "⅕", "⅕", "1⁄5", "1⁄5" },
  { "⅖", "⅖", "⅖", "2⁄5", "2⁄5" },
  { "⅗", "⅗", "⅗", "3⁄5", "3⁄5" },
  { "⅘", "⅘", "⅘", "4⁄5", "4⁄5" },
  { "⅙", "⅙", "⅙", "1⁄6", "1⁄6" },
  { "⅚", "⅚", "⅚", "5⁄6", "5⁄6" },
  { "⅛", "⅛", "⅛", "1⁄8", "1⁄8" },
  { "⅜", "⅜", "⅜", "3⁄8", "3⁄8" },
  { "⅝", "⅝", "⅝", "5⁄8", "5⁄8" },
  { "⅞", "⅞", "⅞", "7⁄8", "7⁄8" },
  { "⅟", "⅟", "⅟", "1⁄", "1⁄" },
  { "Ⅰ", "Ⅰ", "Ⅰ", "I", "I" },
  { "Ⅱ", "Ⅱ", "Ⅱ", "II", "II" },
  { "Ⅲ", "Ⅲ", "Ⅲ", "III", "III" },
  { "Ⅳ", "Ⅳ", "Ⅳ", "IV", "IV" },
  { "Ⅴ", "Ⅴ", "Ⅴ", "V", "V" },
  { "Ⅵ", "Ⅵ", "Ⅵ", "VI", "VI" },
  { "Ⅶ", "Ⅶ", "Ⅶ", "VII", "VII" },
  { "Ⅷ", "Ⅷ", "Ⅷ", "VIII", "VIII" },
  { "Ⅸ", "Ⅸ", "Ⅸ", "IX", "IX" },
  { "Ⅹ", "Ⅹ", "Ⅹ", "X", "X" },
  { "Ⅺ", "Ⅺ", "Ⅺ", "XI", "XI" },
  { "Ⅻ", "Ⅻ", "Ⅻ", "XII", "XII" },
  { "Ⅼ", "Ⅼ", "Ⅼ", "L", "L" },
  { "Ⅽ", "Ⅽ", "Ⅽ", "C", "C" },
  { "Ⅾ", "Ⅾ", "Ⅾ", "D", "D" },
  { "Ⅿ", "Ⅿ", "Ⅿ", "M", "M" },
  { "ⅰ", "ⅰ", "ⅰ", "i", "i" },
  { "ⅱ", "ⅱ", "ⅱ", "ii", "ii" },
  { "ⅲ", "ⅲ", "ⅲ", "iii", "iii" },
  { "ⅳ", "ⅳ", "ⅳ", "iv", "iv" },
  { "ⅴ", "ⅴ", "ⅴ", "v", "v" },
  { "ⅵ", "ⅵ", "ⅵ", "vi", "vi" },
  { "ⅶ", "ⅶ", "ⅶ", "vii", "vii" },
  { "ⅷ", "ⅷ", "ⅷ", "viii", "viii" },
  { "ⅸ", "ⅸ", "ⅸ", "ix", "ix" },
  { "ⅹ", "ⅹ", "ⅹ", "x", "x" },
  { "ⅺ", "ⅺ", "ⅺ", "xi", "xi" },
  { "ⅻ", "ⅻ", "ⅻ", "xii", "xii" },
  { "ⅼ", "ⅼ", "ⅼ", "l", "l" },
  { "ⅽ", "ⅽ", "ⅽ", "c", "c" },
  { "ⅾ", "ⅾ", "ⅾ", "d", "d" },
  { "ⅿ", "ⅿ", "ⅿ", "m", "m" },
  { "↉", "↉", "↉", "0⁄3", "0⁄3" },
  { "↚", "↚", "↚", "↚", "↚" },
  { "↛", "↛", "↛", "↛", "↛" },
  { "↮", "↮", "↮", "↮", "↮" },
  { "⇍", "⇍", "⇍", "⇍", "⇍" },
  { "⇎", "⇎", "⇎", "⇎", "⇎" },
  { "⇏", "⇏", "⇏", "⇏", "⇏" },
  { "∄", "∄", "∄", "∄", "∄" },
  { "∉", "∉", "∉", "∉", "∉" },
  { "∌", "∌", "∌", "∌", "∌" },
  { "∤", "∤", "∤", "∤", "∤" },
  { "∦", "∦", "∦", "∦", "∦" },
  { "∬", "∬", "∬", "∫∫", "∫∫" },
  { "∭", "∭", "∭", "∫∫∫", "∫∫∫" },
  { "∯", "∯", "∯", "∮∮", "∮∮" },
  { "∰", "∰", "∰", "∮∮∮", "∮∮∮" },
  { "≁", "≁", "≁", "≁", "≁" },
  { "≄", "≄", "≄", "≄", "≄" },
  { "≇", "≇", "≇", "≇", "≇" },
  { "≉", "≉", "≉", "≉", "≉" },
  { "≠", "≠", "≠", "≠", "≠" },
  { "≢", "≢", "≢", "≢", "≢" },
  { "≭", "≭", "≭", "≭", "≭" },
  { "≮", "≮", "≮", "≮", "≮" },
  { "≯", "≯", "≯", "≯", "≯" },
  { "≰", "≰", "≰", "≰", "≰" },
  { "≱", "≱", "≱", "≱", "≱" },
  { "≴", "≴", "≴", "≴", "≴" },
  { "≵", "≵", "≵", "≵", "≵" },
  { "≸", "≸", "≸", "≸", "≸" },
  { "≹", "≹", "≹", "≹", "≹" },
  { "⊀", "⊀", "⊀", "⊀", "⊀" },
  { "⊁", "⊁", "⊁", "⊁", "⊁" },
  { "⊄", "⊄", "⊄", "⊄", "⊄" },
  { "⊅", "⊅", "⊅", "⊅", "⊅" },
  { "⊈", "⊈", "⊈", "⊈", "⊈" },
  { "⊉", "⊉", "⊉", "⊉", "⊉" },
  { "⊬", "⊬", "⊬", "⊬", "⊬" },
  { "⊭", "⊭", "⊭", "⊭", "⊭" },
  { "⊮", "⊮", "⊮", "⊮", "⊮" },
  { "⊯", "⊯", "⊯", "⊯", "⊯" },
  { "⋠", "⋠", "⋠", "⋠", "⋠" },
  { "⋡", "⋡", "⋡", "⋡", "⋡" },
  { "⋢", "⋢", "⋢", "⋢", "⋢" },
  { "⋣", "⋣", "⋣", "⋣", "⋣" },
  { "⋪", "⋪", "⋪", "⋪", "⋪" },
  { "⋫", "⋫", "⋫", "⋫", "⋫" },
  { "⋬", "⋬", "⋬", "⋬", "⋬" },
  { "⋭", "⋭", "⋭", "⋭", "⋭" },
  { "〈", "〈", "〈", "〈", "〈" },
  { "〉", "〉", "〉", "〉", "〉" },
  { "①", "①", "①", "1", "1" },
  { "②", "②", "②", "2", "2" },
  { "③", "③", "③", "3", "3" },
  { "④", "④", "④", "4", "4" },
  { "⑤", "⑤", "⑤", "5", "5" },
  { "⑥", "⑥", "⑥", "6", "6" },
  { "⑦", "⑦", "⑦", "7", "7" },
  { "⑧", "⑧", "⑧", "8", "8" },
  { "⑨", "⑨", "⑨", "9", "9" },
  { "⑩", "⑩", "⑩", "10", "10" },
  { "⑪", "⑪", "⑪", "11", "11" },
  { "⑫", "⑫", "⑫", "12", "12" },
  { "⑬", "⑬", "⑬", "13", "13" },
  { "⑭", "⑭", "⑭", "14", "14" },
  { "⑮", "⑮", "⑮", "15", "15" },
  { "⑯", "⑯", "⑯", "16", "16" },
  { "⑰", "⑰", "⑰", "17", "17" },
  { "⑱", "⑱", "⑱", "18", "18" },
  { "⑲", "⑲", "⑲", "19", "19" },
  { "⑳", "⑳", "⑳", "20", "20" },
  { "⑴", "⑴", "⑴", "(1)", "(1)" },
  { "⑵", "⑵", "⑵", "(2)", "(2)" },
  { "⑶", "⑶", "⑶", "(3)", "(3)" },
  { "⑷", "⑷", "⑷", "(4)", "(4)" },
  { "⑸", "⑸", "⑸", "(5)", "(5)" },
  { "⑹", "⑹", "⑹", "(6)", "(6)" },
  { "⑺", "⑺", "⑺", "(7)", "(7)" },
  { "⑻", "⑻", "⑻", "(8)", "(8)" },
  { "⑼", "⑼", "⑼", "(9)", "(9)" },
  { "⑽", "⑽", "⑽", "(10)", "(10)" },
  { "⑾", "⑾", "⑾", "(11)", "(11)" },
  { "⑿", "⑿", "⑿", "(12)", "(12)" },
  { "⒀", "⒀", "⒀", "(13)", "(13)" },
  { "⒁", "⒁", "⒁", "(14)", "(14)" },
  { "⒂", "⒂", "⒂", "(15)", "(15)" },
  { "⒃", "⒃", "⒃", "(16)", "(16)" },
  { "⒄", "⒄", "⒄", "(17)", "(17)" },
  { "⒅", "⒅", "⒅", "(18)", "(18)" },
  { "⒆", "⒆", "⒆", "(19)", "(19)" },
  { "⒇", "⒇", "⒇", "(20)", "(20)" },
  { "⒈", "⒈", "⒈", "1.", "1." },
  { "⒉", "⒉", "⒉", "2.", "2." },
  { "⒊", "⒊", "⒊", "3.", "3." },
  { "⒋", "⒋", "⒋", "4.", "4." },
  { "⒌", "⒌", "⒌", "5.", "5." },
  { "⒍", "⒍", "⒍", "6.", "6." },
  { "⒎", "⒎", "⒎", "7.", "7." },
  { "⒏", "⒏", "⒏", "8.", "8." },
  { "⒐", "⒐", "⒐", "9.", "9." },
  { "⒑", "⒑", "⒑", "10.", "10." },
  { "⒒", "⒒", "⒒", "11.", "11." },
  { "⒓", "⒓", "⒓", "12.", "12." },
  { "⒔", "⒔", "⒔", "13.", "13." },
  { "⒕", "⒕", "⒕", "14.", "14." },
  { "⒖", "⒖", "⒖", "15.", "15." },
  { "⒗", "⒗", "⒗", "16.", "16." },
  { "⒘", "⒘", "⒘", "17.", "17." },
  { "⒙", "⒙", "⒙", "18.", "18." },
  { "⒚", "⒚", "⒚", "19.", "19." },
  { "⒛", "⒛", "⒛", "20.", "20." },
  { "⒜", "⒜", "⒜", "(a)", "(a)" },
  { "⒝", "⒝", "⒝", "(b)", "(b)" },
  { "⒞", "⒞", "⒞", "(c)", "(c)" },
  { "⒟", "⒟", "⒟", "(d)", "(d)" },
  { "⒠", "⒠", "⒠", "(e)", "(e)" },
  { "⒡", "⒡", "⒡", "(f)", "(f)" },
  { "⒢", "⒢", "⒢", "(g)", "(g)" },
  { "⒣", "⒣", "⒣", "(h)", "(h)" },
  { "⒤", "⒤", "⒤", "(i)", "(i)" },
  { "⒥", "⒥", "⒥", "(j)", "(j)" },
  { "⒦", "⒦", "⒦", "(k)", "(k)" },
  { "⒧", "⒧", "⒧", "(l)", "(l)" },
  { "⒨", "⒨", "⒨", "(m)", "(m)" },
  { "⒩", "⒩", "⒩", "(n)", "(n)" },
  { "⒪", "⒪", "⒪", "(o)", "(o)" },
  { "⒫", "⒫", "⒫", "(p)", "(p)" },
  { "⒬", "⒬", "⒬", "(q)", "(q)" },
  { "⒭", "⒭", "⒭", "(r)", "(r)" },
  { "⒮", "⒮", "⒮", "(s)", "(s)" },
  { "⒯", "⒯", "⒯", "(t)", "(t)" },
  { "⒰", "⒰", "⒰", "(u)", "(u)" },
  { "⒱", "⒱", "⒱", "(v)", "(v)" },
  { "⒲", "⒲", "⒲", "(w)", "(w)" },
  { "⒳", "⒳", "⒳", "(x)", "(x)" },
  { "⒴", "⒴", "⒴", "(y)", "(y)" },
  { "⒵", "⒵", "⒵", "(z)", "(z)" },
  { "Ⓐ", "Ⓐ", "Ⓐ", "A", "A" },
  { "Ⓑ", "Ⓑ", "Ⓑ", "B", "B" },
  { "Ⓒ", "Ⓒ", "Ⓒ", "C", "C" },
  { "Ⓓ", "Ⓓ", "Ⓓ", "D", "D" },
  { "Ⓔ", "Ⓔ", "Ⓔ", "E", "E" },
  { "Ⓕ", "Ⓕ", "Ⓕ", "F", "F" },
  { "Ⓖ", "Ⓖ", "Ⓖ", "G", "G" },
  { "Ⓗ", "Ⓗ", "Ⓗ", "H", "H" },
  { "Ⓘ", "Ⓘ", "Ⓘ", "I", "I" },
  { "Ⓙ", "Ⓙ", "Ⓙ", "J", "J" },
  { "Ⓚ", "Ⓚ", "Ⓚ", "K", "K" },
  { "Ⓛ", "Ⓛ", "Ⓛ", "L", "L" },
  { "Ⓜ", "Ⓜ", "Ⓜ", "M", "M" },
  { "Ⓝ", "Ⓝ", "Ⓝ", "N", "N" },
  { "Ⓞ", "Ⓞ", "Ⓞ", "O", "O" },
  { "Ⓟ", "Ⓟ", "Ⓟ", "P", "P" },
  { "Ⓠ", "Ⓠ", "Ⓠ", "Q", "Q" },
  { "Ⓡ", "Ⓡ", "Ⓡ", "R", "R" },
  { "Ⓢ", "Ⓢ", "Ⓢ", "S", "S" },
  { "Ⓣ", "Ⓣ", "Ⓣ", "T", "T" },
  { "Ⓤ", "Ⓤ", "Ⓤ", "U", "U" },
  { "Ⓥ", "Ⓥ", "Ⓥ", "V", "V" },
  { "Ⓦ", "Ⓦ", "Ⓦ", "W", "W" },
  { "Ⓧ", "Ⓧ", "Ⓧ", "X", "X" },
  { "Ⓨ", "Ⓨ", "Ⓨ", "Y", "Y" },
  { "Ⓩ", "Ⓩ", "Ⓩ", "Z", "Z" },
  { "ⓐ", "ⓐ", "ⓐ", "a", "a" },
  { "ⓑ", "ⓑ", "ⓑ", "b", "b" },
  { "ⓒ", "ⓒ", "ⓒ", "c", "c" },
  { "ⓓ", "ⓓ", "ⓓ", "d", "d" },
  { "ⓔ", "ⓔ", "ⓔ", "e", "e" },
  { "ⓕ", "ⓕ", "ⓕ", "f", "f" },
  { "ⓖ", "ⓖ", "ⓖ", "g", "g" },
  { "ⓗ", "ⓗ", "ⓗ", "h", "h" },
  { "ⓘ", "ⓘ", "ⓘ", "i", "i" },
  { "ⓙ", "ⓙ", "ⓙ", "j", "j" },
  { "ⓚ", "ⓚ", "ⓚ", "k", "k" },
  { "ⓛ", "ⓛ", "ⓛ", "l", "l" },
  { "ⓜ", "ⓜ", "ⓜ", "m", "m" },
  { "ⓝ", "ⓝ", "ⓝ", "n", "n" },
  { "ⓞ", "ⓞ", "ⓞ", "o", "o" },
  { "ⓟ", "ⓟ", "ⓟ", "p", "p" },
  { "ⓠ", "ⓠ", "ⓠ", "q", "q" },
  { "ⓡ", "ⓡ", "ⓡ", "r", "r" },
  { "ⓢ", "ⓢ", "ⓢ", "s", "s" },
  { "ⓣ", "ⓣ", "ⓣ", "t", "t" },
  { "ⓤ", "ⓤ", "ⓤ", "u", "u" },
  { "ⓥ", "ⓥ", "ⓥ", "v", "v" },
  { "ⓦ", "ⓦ", "ⓦ", "w", "w" },
  { "ⓧ", "ⓧ", "ⓧ", "x", "x" },
  { "ⓨ", "ⓨ", "ⓨ", "y", "y" },
  { "ⓩ", "ⓩ", "ⓩ", "z", "z" },
  { "⓪", "⓪", "⓪", "0", "0" },
  { "⨌", "⨌", "⨌", "∫∫∫∫", "∫∫∫∫" },
  { "⩴", "⩴", "⩴", "::=", "::=" },
  { "⩵", "⩵", "⩵", "==", "==" },
  { "⩶", "⩶", "⩶", "===", "===" },
  { "⫝̸", "⫝̸", "⫝̸", "⫝̸", "⫝̸" },
  { "ⱼ", "ⱼ", "ⱼ", "j", "j" },
  { "ⱽ", "ⱽ", "ⱽ", "V", "V" },
  { "ⵯ", "ⵯ", "ⵯ", "ⵡ", "ⵡ" },
  { "⺟", "⺟", "⺟", "母", "母" },
  { "⻳", "⻳", "⻳", "龟", "龟" },
  { "⼀", "⼀", "⼀", "一", "一" },
  { "⼁", "⼁", "⼁", "丨", "丨" },
  { "⼂", "⼂", "⼂", "丶", "丶" },
  { "⼃", "⼃", "⼃", "丿", "丿" },
  { "⼄", "⼄", "⼄", "乙", "乙" },
  { "⼅", "⼅", "⼅", "亅", "亅" },
  { "⼆", "⼆", "⼆", "二", "二" },
  { "⼇", "⼇", "⼇", "亠", "亠" },
  { "⼈", "⼈", "⼈", "人", "人" },
  { "⼉", "⼉", "⼉", "儿", "儿" },
  { "⼊", "⼊", "⼊", "入", "入" },
  { "⼋", "⼋", "⼋", "八", "八" },
  { "⼌", "⼌", "⼌", "冂", "冂" },
  { "⼍", "⼍", "⼍", "冖", "冖" },
  { "⼎", "⼎", "⼎", "冫", "冫" },
  { "⼏", "⼏", "⼏", "几", "几" },
  { "⼐", "⼐", "⼐", "凵", "凵" },
  { "⼑", "⼑", "⼑", "刀", "刀" },
  { "⼒", "⼒", "⼒", "力", "力" },
  { "⼓", "⼓", "⼓", "勹", "勹" },
  { "⼔", "⼔", "⼔", "匕", "匕" },
  { "⼕", "⼕", "⼕", "匚", "匚" },
  { "⼖", "⼖", "⼖", "匸", "匸" },
  { "⼗", "⼗", "⼗", "十", "十" },
  { "⼘", "⼘", "⼘", "卜", "卜" },
  { "⼙", "⼙", "⼙", "卩", "卩" },
  { "⼚", "⼚", "⼚", "厂", "厂" },
  { "⼛", "⼛", "⼛", "厶", "厶" },
  { "⼜", "⼜", "⼜", "又", "又" },
  { "⼝", "⼝", "⼝", "口", "口" },
  { "⼞", "⼞", "⼞", "囗", "囗" },
  { "⼟", "⼟", "⼟", "土", "土" },
  { "⼠", "⼠", "⼠", "士", "士" },
  { "⼡", "⼡", "⼡", "夂", "夂" },
  { "⼢", "⼢", "⼢", "夊", "夊" },
  { "⼣", "⼣", "⼣", "夕", "夕" },
  { "⼤", "⼤", "⼤", "大", "大" },
  { "⼥", "⼥", "⼥", "女", "女" },
  { "⼦", "⼦", "⼦", "子", "子" },
  { "⼧", "⼧", "⼧", "宀", "宀" },
  { "⼨", "⼨", "⼨", "寸", "寸" },
  { "⼩", "⼩", "⼩", "小", "小" },
  { "⼪", "⼪", "⼪", "尢", "尢" },
  { "⼫", "⼫", "⼫", "尸", "尸" },
  { "⼬", "⼬", "⼬", "屮", "屮" },
  { "⼭", "⼭", "⼭", "山", "山" },
  { "⼮", "⼮", "⼮", "巛", "巛" },
  { "⼯", "⼯", "⼯", "工", "工" },
  { "⼰", "⼰", "⼰", "己", "己" },
  { "⼱", "⼱", "⼱", "巾", "巾" },
  { "⼲", "⼲", "⼲", "干", "干" },
  { "⼳", "⼳", "⼳", "幺", "幺" },
  { "⼴", "⼴", "⼴", "广", "广" },
  { "⼵", "⼵", "⼵", "廴", "廴" },
  { "⼶", "⼶", "⼶", "廾", "廾" },
  { "⼷", "⼷", "⼷", "弋", "弋" },
  { "⼸", "⼸", "⼸", "弓", "弓" },
  { "⼹", "⼹", "⼹", "彐", "彐" },
  { "⼺", "⼺", "⼺", "彡", "彡" },
  { "⼻", "⼻", "⼻", "彳", "彳" },
  { "⼼", "⼼", "⼼", "心", "心" },
  { "⼽", "⼽", "⼽", "戈", "戈" },
  { "⼾", "⼾", "⼾", "戶", "戶" },
  { "⼿", "⼿", "⼿", "手", "手" },
  { "⽀", "⽀", "⽀", "支", "支" },
  { "⽁", "⽁", "⽁", "攴", "攴" },
  { "⽂", "⽂", "⽂", "文", "文" },
  { "⽃", "⽃", "⽃", "斗", "斗" },
  { "⽄", "⽄", "⽄", "斤", "斤" },
  { "⽅", "⽅", "⽅", "方", "方" },
  { "⽆", "⽆", "⽆", "无", "无" },
  { "⽇", "⽇", "⽇", "日", "日" },
  { "⽈", "⽈", "⽈", "曰", "曰" },
  { "⽉", "⽉", "⽉", "月", "月" },
  { "⽊", "⽊", "⽊", "木", "木" },
  { "⽋", "⽋", "⽋", "欠", "欠" },
  { "⽌", "⽌", "⽌", "止", "止" },
  { "⽍", "⽍", "⽍", "歹", "歹" },
  { "⽎", "⽎", "⽎", "殳", "殳" },
  { "⽏", "⽏", "⽏", "毋", "毋" },
  { "⽐", "⽐", "⽐", "比", "比" },
  { "⽑", "⽑", "⽑", "毛", "毛" },
  { "⽒", "⽒", "⽒", "氏", "氏" },
  { "⽓", "⽓", "⽓", "气", "气" },
  { "⽔", "⽔", "⽔", "水", "水" },
  { "⽕", "⽕", "⽕", "火", "火" },
  { "⽖", "⽖", "⽖", "爪", "爪" },
  { "⽗", "⽗", "⽗", "父", "父" },
  { "⽘", "⽘", "⽘", "爻", "爻" },
  { "⽙", "⽙", "⽙", "爿", "爿" },
  { "⽚", "⽚", "⽚", "片", "片" },
  { "⽛", "⽛", "⽛", "牙", "牙" },
  { "⽜", "⽜", "⽜", "牛", "牛" },
  { "⽝", "⽝", "⽝", "犬", "犬" },
  { "⽞", "⽞", "⽞", "玄", "玄" },
  { "⽟", "⽟", "⽟", "玉", "玉" },
  { "⽠", "⽠", "⽠", "瓜", "瓜" },
  { "⽡", "⽡", "⽡", "瓦", "瓦" },
  { "⽢", "⽢", "⽢", "甘", "甘" },
  { "⽣", "⽣", "⽣", "生", "生" },
  { "⽤", "⽤", "⽤", "用", "用" },
  { "⽥", "⽥", "⽥", "田", "田" },
  { "⽦", "⽦", "⽦", "疋", "疋" },
  { "⽧", "⽧", "⽧", "疒", "疒" },
  { "⽨", "⽨", "⽨", "癶", "癶" },
  { "⽩", "⽩", "⽩", "白", "白" },
  { "⽪", "⽪", "⽪", "皮", "皮" },
  { "⽫", "⽫", "⽫", "皿", "皿" },
  { "⽬", "⽬", "⽬", "目", "目" },
  { "⽭", "⽭", "⽭", "矛", "矛" },
  { "⽮", "⽮", "⽮", "矢", "矢" },
  { "⽯", "⽯", "⽯", "石", "石" },
  { "⽰", "⽰", "⽰", "示", "示" },
  { "⽱", "⽱", "⽱", "禸", "禸" },
  { "⽲", "⽲", "⽲", "禾", "禾" },
  { "⽳", "⽳", "⽳", "穴", "穴" },
  { "⽴", "⽴", "⽴", "立", "立" },
  { "⽵", "⽵", "⽵", "竹", "竹" },
  { "⽶", "⽶", "⽶", "米", "米" },
  { "⽷", "⽷", "⽷", "糸", "糸" },
  { "⽸", "⽸", "⽸", "缶", "缶" },
  { "⽹", "⽹", "⽹", "网", "网" },
  { "⽺", "⽺", "⽺", "羊", "羊" },
  { "⽻", "⽻", "⽻", "羽", "羽" },
  { "⽼", "⽼", "⽼", "老", "老" },
  { "⽽", "⽽", "⽽", "而", "而" },
  { "⽾", "⽾", "⽾", "耒", "耒" },
  { "⽿", "⽿", "⽿", "耳", "耳" },
  { "⾀", "⾀", "⾀", "聿", "聿" },
  { "⾁", "⾁", "⾁", "肉", "肉" },
  { "⾂", "⾂", "⾂", "臣", "臣" },
  { "⾃", "⾃", "⾃", "自", "自" },
  { "⾄", "⾄", "⾄", "至", "至" },
  { "⾅", "⾅", "⾅", "臼", "臼" },
  { "⾆", "⾆", "⾆", "舌", "舌" },
  { "⾇", "⾇", "⾇", "舛", "舛" },
  { "⾈", "⾈", "⾈", "舟", "舟" },
  { "⾉", "⾉", "⾉", "艮", "艮" },
  { "⾊", "⾊", "⾊", "色", "色" },
  { "⾋", "⾋", "⾋", "艸", "艸" },
  { "⾌", "⾌", "⾌", "虍", "虍" },
  { "⾍", "⾍", "⾍", "虫", "虫" },
  { "⾎", "⾎", "⾎", "血", "血" },
  { "⾏", "⾏", "⾏", "行", "行" },
  { "⾐", "⾐", "⾐", "衣", "衣" },
  { "⾑", "⾑", "⾑", "襾", "襾" },
  { "⾒", "⾒", "⾒", "見", "見" },
  { "⾓", "⾓", "⾓", "角", "角" },
  { "⾔", "⾔", "⾔", "言", "言" },
  { "⾕", "⾕", "⾕", "谷", "谷" },
  { "⾖", "⾖", "⾖", "豆", "豆" },
  { "⾗", "⾗", "⾗", "豕", "豕" },
  { "⾘", "⾘", "⾘", "豸", "豸" },
  { "⾙", "⾙", "⾙", "貝", "貝" },
  { "⾚", "⾚", "⾚", "赤", "赤" },
  { "⾛", "⾛", "⾛", "走", "走" },
  { "⾜", "⾜", "⾜", "足", "足" },
  { "⾝", "⾝", "⾝", "身", "身" },
  { "⾞", "⾞", "⾞", "車", "車" },
  { "⾟", "⾟", "⾟", "辛", "辛" },
  { "⾠", "⾠", "⾠", "辰", "辰" },
  { "⾡", "⾡", "⾡", "辵", "辵" },
  { "⾢", "⾢", "⾢", "邑", "邑" },
  { "⾣", "⾣", "⾣", "酉", "酉" },
  { "⾤", "⾤", "⾤", "釆", "釆" },
  { "⾥", "⾥", "⾥", "里", "里" },
  { "⾦", "⾦", "⾦", "金", "金" },
  { "⾧", "⾧", "⾧", "長", "長" },
  { "⾨", "⾨", "⾨", "門", "門" },
  { "⾩", "⾩", "⾩", "阜", "阜" },
  { "⾪", "⾪", "⾪", "隶", "隶" },
  { "⾫", "⾫", "⾫", "隹", "隹" },
  { "⾬", "⾬", "⾬", "雨", "雨" },
  { "⾭", "⾭", "⾭", "靑", "靑" },
  { "⾮", "⾮", "⾮", "非", "非" },
  { "⾯", "⾯", "⾯", "面", "面" },
  { "⾰", "⾰", "⾰", "革", "革" },
  { "⾱", "⾱", "⾱", "韋", "韋" },
  { "⾲", "⾲", "⾲", "韭", "韭" },
  { "⾳", "⾳", "⾳", "音", "音" },
  { "⾴", "⾴", "⾴", "頁", "頁" },
  { "⾵", "⾵", "⾵", "風", "風" },
  { "⾶", "⾶", "⾶", "飛", "飛" },
  { "⾷", "⾷", "⾷", "食", "食" },
  { "⾸", "⾸", "⾸", "首", "首" },
  { "⾹", "⾹", "⾹", "香", "香" },
  { "⾺", "⾺", "⾺", "馬", "馬" },
  { "⾻", "⾻", "⾻", "骨", "骨" },
  { "⾼", "⾼", "⾼", "高", "高" },
  { "⾽", "⾽", "⾽", "髟", "髟" },
  { "⾾", "⾾", "⾾", "鬥", "鬥" },
  { "⾿", "⾿", "⾿", "鬯", "鬯" },
  { "⿀", "⿀", "⿀", "鬲", "鬲" },
  { "⿁", "⿁", "⿁", "鬼", "鬼" },
  { "⿂", "⿂", "⿂", "魚", "魚" },
  { "⿃", "⿃", "⿃", "鳥", "鳥" },
  { "⿄", "⿄", "⿄", "鹵", "鹵" },
  { "⿅", "⿅", "⿅", "鹿", "鹿" },
  { "⿆", "⿆", "⿆", "麥", "麥" },
  { "⿇", "⿇", "⿇", "麻", "麻" },
  { "⿈", "⿈", "⿈", "黃", "黃" },
  { "⿉", "⿉", "⿉", "黍", "黍" },
  { "⿊", "⿊", "⿊", "黑", "黑" },
  { "⿋", "⿋", "⿋", "黹", "黹" },
  { "⿌", "⿌", "⿌", "黽", "黽" },
  { "⿍", "⿍", "⿍", "鼎", "鼎" },
  { "⿎", "⿎", "⿎", "鼓", "鼓" },
  { "⿏", "⿏", "⿏", "鼠", "鼠" },
  { "⿐", "⿐", "⿐", "鼻", "鼻" },
  { "⿑", "⿑", "⿑", "齊", "齊" },
  { "⿒", "⿒", "⿒", "齒", "齒" },
  { "⿓", "⿓", "⿓", "龍", "龍" },
  { "⿔", "⿔", "⿔", "龜", "龜" },
  { "⿕", "⿕", "⿕", "龠", "龠" },
  { "　", "　", "　", " ", " " },
  { "〶", "〶", "〶", "〒", "〒" },
  { "〸", "〸", "〸", "十", "十" },
  { "〹", "〹", "〹", "卄", "卄" },
  { "〺", "〺", "〺", "卅", "卅" },
  { "が", "が", "が", "が", "が" },
  { "ぎ", "ぎ", "ぎ", "ぎ", "ぎ" },
  { "ぐ", "ぐ", "ぐ", "ぐ", "ぐ" },
  { "げ", "げ", "げ", "げ", "げ" },
  { "ご", "ご", "ご", "ご", "ご" },
  { "ざ", "ざ", "ざ", "ざ", "ざ" },
  { "じ", "じ", "じ", "じ", "じ" },
  { "ず", "ず", "ず", "ず", "ず" },
  { "ぜ", "ぜ", "ぜ", "ぜ", "ぜ" },
  { "ぞ", "ぞ", "ぞ", "ぞ", "ぞ" },
  { "だ", "だ", "だ", "だ", "だ" },
  { "ぢ", "ぢ", "ぢ", "ぢ", "ぢ" },
  { "づ", "づ", "づ", "づ", "づ" },
  { "で", "で", "で", "で", "で" },
  { "ど", "ど", "ど", "ど", "ど" },
  { "ば", "ば", "ば", "ば", "ば" },
  { "ぱ", "ぱ", "ぱ", "ぱ", "ぱ" },
  { "び", "び", "び", "び", "び" },
  { "ぴ", "ぴ", "ぴ", "ぴ", "ぴ" },
  { "ぶ", "ぶ", "ぶ", "ぶ", "ぶ" },
  { "ぷ", "ぷ", "ぷ", "ぷ", "ぷ" },
  { "べ", "べ", "べ", "べ", "べ" },
  { "ぺ", "ぺ", "ぺ", "ぺ", "ぺ" },
  { "ぼ", "ぼ", "ぼ", "ぼ", "ぼ" },
  { "ぽ", "ぽ", "ぽ", "ぽ", "ぽ" },
  { "ゔ", "ゔ", "ゔ", "ゔ", "ゔ" },
  { "゛", "゛", "゛", " ゙", " ゙" },
  { "゜", "゜", "゜", " ゚", " ゚" },
  { "ゞ", "ゞ", "ゞ", "ゞ", "ゞ" },
  { "ゟ", "ゟ", "ゟ", "より", "より" },
  { "ガ", "ガ", "ガ", "ガ", "ガ" },
  { "ギ", "ギ", "ギ", "ギ", "ギ" },
  { "グ", "グ", "グ", "グ", "グ" },
  { "ゲ", "ゲ", "ゲ", "ゲ", "ゲ" },
  { "ゴ", "ゴ", "ゴ", "ゴ", "ゴ" },
  { "ザ", "ザ", "ザ", "ザ", "ザ" },
  { "ジ", "ジ", "ジ", "ジ", "ジ" },
  { "ズ", "ズ", "ズ", "ズ", "ズ" },
  { "ゼ", "ゼ", "ゼ", "ゼ", "ゼ" },
  { "ゾ", "ゾ", "ゾ", "ゾ", "ゾ" },
  { "ダ", "ダ", "ダ", "ダ", "ダ" },
  { "ヂ", "ヂ", "ヂ", "ヂ", "ヂ" },
  { "ヅ", "ヅ", "ヅ", "ヅ", "ヅ" },
  { "デ", "デ", "デ", "デ", "デ" },
  { "ド", "ド", "ド", "ド", "ド" },
  { "バ", "バ", "バ", "バ", "バ" },
  { "パ", "パ", "パ", "パ", "パ" },
  { "ビ", "ビ", "ビ", "ビ", "ビ" },
  { "ピ", "ピ", "ピ", "ピ", "ピ" },
  { "ブ", "ブ", "ブ", "ブ", "ブ" },
  { "プ", "プ", "プ", "プ", "プ" },
  { "ベ", "ベ", "ベ", "ベ", "ベ" },
  { "ペ", "ペ", "ペ", "ペ", "ペ" },
  { "ボ", "ボ", "ボ", "ボ", "ボ" },
  { "ポ", "ポ", "ポ", "ポ", "ポ" },
  { "ヴ", "ヴ", "ヴ", "ヴ", "ヴ" },
  { "ヷ", "ヷ", "ヷ", "ヷ", "ヷ" },
  { "ヸ", "ヸ", "ヸ", "ヸ", "ヸ" },
  { "ヹ", "ヹ", "ヹ", "ヹ", "ヹ" },
  { "ヺ", "ヺ", "ヺ", "ヺ", "ヺ" },
  { "ヾ", "ヾ", "ヾ", "ヾ", "ヾ" },
  { "ヿ", "ヿ", "ヿ", "コト", "コト" },
  { "ㄱ", "ㄱ", "ㄱ", "ᄀ", "ᄀ" },
  { "ㄲ", "ㄲ", "ㄲ", "ᄁ", "ᄁ" },
  { "ㄳ", "ㄳ", "ㄳ", "ᆪ", "ᆪ" },
  { "ㄴ", "ㄴ", "ㄴ", "ᄂ", "ᄂ" },
  { "ㄵ", "ㄵ", "ㄵ", "ᆬ", "ᆬ" },
  { "ㄶ", "ㄶ", "ㄶ", "ᆭ", "ᆭ" },
  { "ㄷ", "ㄷ", "ㄷ", "ᄃ", "ᄃ" },
  { "ㄸ", "ㄸ", "ㄸ", "ᄄ", "ᄄ" },
  { "ㄹ", "ㄹ", "ㄹ", "ᄅ", "ᄅ" },
  { "ㄺ", "ㄺ", "ㄺ", "ᆰ", "ᆰ" },
  { "ㄻ", "ㄻ", "ㄻ", "ᆱ", "ᆱ" },
  { "ㄼ", "ㄼ", "ㄼ", "ᆲ", "ᆲ" },
  { "ㄽ", "ㄽ", "ㄽ", "ᆳ", "ᆳ" },
  { "ㄾ", "ㄾ", "ㄾ", "ᆴ", "ᆴ" },
  { "ㄿ", "ㄿ", "ㄿ", "ᆵ", "ᆵ" },
  { "ㅀ", "ㅀ", "ㅀ", "ᄚ", "ᄚ" },
  { "ㅁ", "ㅁ", "ㅁ", "ᄆ", "ᄆ" },
  { "ㅂ", "ㅂ", "ㅂ", "ᄇ", "ᄇ" },
  { "ㅃ", "ㅃ", "ㅃ", "ᄈ", "ᄈ" },
  { "ㅄ", "ㅄ", "ㅄ", "ᄡ", "ᄡ" },
  { "ㅅ", "ㅅ", "ㅅ", "ᄉ", "ᄉ" },
  { "ㅆ", "ㅆ", "ㅆ", "ᄊ", "ᄊ" },
  { "ㅇ", "ㅇ", "ㅇ", "ᄋ", "ᄋ" },
  { "ㅈ", "ㅈ", "ㅈ", "ᄌ", "ᄌ" },
  { "ㅉ", "ㅉ", "ㅉ", "ᄍ", "ᄍ" },
  { "ㅊ", "ㅊ", "ㅊ", "ᄎ", "ᄎ" },
  { "ㅋ", "ㅋ", "ㅋ", "ᄏ", "ᄏ" },
  { "ㅌ", "ㅌ", "ㅌ", "ᄐ", "ᄐ" },
  { "ㅍ", "ㅍ", "ㅍ", "ᄑ", "ᄑ" },
  { "ㅎ", "ㅎ", "ㅎ", "ᄒ", "ᄒ" },
  { "ㅏ", "ㅏ", "ㅏ", "ᅡ", "ᅡ" },
  { "ㅐ", "ㅐ", "ㅐ", "ᅢ", "ᅢ" },
  { "ㅑ", "ㅑ", "ㅑ", "ᅣ", "ᅣ" },
  { "ㅒ", "ㅒ", "ㅒ", "ᅤ", "ᅤ" },
  { "ㅓ", "ㅓ", "ㅓ", "ᅥ", "ᅥ" },
  { "ㅔ", "ㅔ", "ㅔ", "ᅦ", "ᅦ" },
  { "ㅕ", "ㅕ", "ㅕ", "ᅧ", "ᅧ" },
  { "ㅖ", "ㅖ", "ㅖ", "ᅨ", "ᅨ" },
  { "ㅗ", "ㅗ", "ㅗ", "ᅩ", "ᅩ" },
  { "ㅘ", "ㅘ", "ㅘ", "ᅪ", "ᅪ" },
  { "ㅙ", "ㅙ", "ㅙ", "ᅫ", "ᅫ" },
  { "ㅚ", "ㅚ", "ㅚ", "ᅬ", "ᅬ" },
  { "ㅛ", "ㅛ", "ㅛ", "ᅭ", "ᅭ" },
  { "ㅜ", "ㅜ", "ㅜ", "ᅮ", "ᅮ" },
  { "ㅝ", "ㅝ", "ㅝ", "ᅯ", "ᅯ" },
  { "ㅞ", "ㅞ", "ㅞ", "ᅰ", "ᅰ" },
  { "ㅟ", "ㅟ", "ㅟ", "ᅱ", "ᅱ" },
  { "ㅠ", "ㅠ", "ㅠ", "ᅲ", "ᅲ" },
  { "ㅡ", "ㅡ", "ㅡ", "ᅳ", "ᅳ" },
  { "ㅢ", "ㅢ", "ㅢ", "ᅴ", "ᅴ" },
  { "ㅣ", "ㅣ", "ㅣ", "ᅵ", "ᅵ" },
  { "ㅤ", "ㅤ", "ㅤ", "ᅠ", "ᅠ" },
  { "ㅥ", "ㅥ", "ㅥ", "ᄔ", "ᄔ" },
  { "ㅦ", "ㅦ", "ㅦ", "ᄕ", "ᄕ" },
  { "ㅧ", "ㅧ", "ㅧ", "ᇇ", "ᇇ" },
  { "ㅨ", "ㅨ", "ㅨ", "ᇈ", "ᇈ" },
  { "ㅩ", "ㅩ", "ㅩ", "ᇌ", "ᇌ" },
  { "ㅪ", "ㅪ", "ㅪ", "ᇎ", "ᇎ" },
  { "ㅫ", "ㅫ", "ㅫ", "ᇓ", "ᇓ" },
  { "ㅬ", "ㅬ", "ㅬ", "ᇗ", "ᇗ" },
  { "ㅭ", "ㅭ", "ㅭ", "ᇙ", "ᇙ" },
  { "ㅮ", "ㅮ", "ㅮ", "ᄜ", "ᄜ" },
  { "ㅯ", "ㅯ", "ㅯ", "ᇝ", "ᇝ" },
  { "ㅰ", "ㅰ", "ㅰ", "ᇟ", "ᇟ" },
  { "ㅱ", "ㅱ", "ㅱ", "ᄝ", "ᄝ" },
  { "ㅲ", "ㅲ", "ㅲ", "ᄞ", "ᄞ" },
  { "ㅳ", "ㅳ", "ㅳ", "ᄠ", "ᄠ" },
  { "ㅴ", "ㅴ", "ㅴ", "ᄢ", "ᄢ" },
  { "ㅵ", "ㅵ", "ㅵ", "ᄣ", "ᄣ" },
  { "ㅶ", "ㅶ", "ㅶ", "ᄧ", "ᄧ" },
  { "ㅷ", "ㅷ", "ㅷ", "ᄩ", "ᄩ" },
  { "ㅸ", "ㅸ", "ㅸ", "ᄫ", "ᄫ" },
  { "ㅹ", "ㅹ", "ㅹ", "ᄬ", "ᄬ" },
  { "ㅺ", "ㅺ", "ㅺ", "ᄭ", "ᄭ" },
  { "ㅻ", "ㅻ", "ㅻ", "ᄮ", "ᄮ" },
  { "ㅼ", "ㅼ", "ㅼ", "ᄯ", "ᄯ" },
  { "ㅽ", "ㅽ", "ㅽ", "ᄲ", "ᄲ" },
  { "ㅾ", "ㅾ", "ㅾ", "ᄶ", "ᄶ" },
  { "ㅿ", "ㅿ", "ㅿ", "ᅀ", "ᅀ" },
  { "ㆀ", "ㆀ", "ㆀ", "ᅇ", "ᅇ" },
  { "ㆁ", "ㆁ", "ㆁ", "ᅌ", "ᅌ" },
  { "ㆂ", "ㆂ", "ㆂ", "ᇱ", "ᇱ" },
  { "ㆃ", "ㆃ", "ㆃ", "ᇲ", "ᇲ" },
  { "ㆄ", "ㆄ", "ㆄ", "ᅗ", "ᅗ" },
  { "ㆅ", "ㆅ", "ㆅ", "ᅘ", "ᅘ" },
  { "ㆆ", "ㆆ", "ㆆ", "ᅙ", "ᅙ" },
  { "ㆇ", "ㆇ", "ㆇ", "ᆄ", "ᆄ" },
  { "ㆈ", "ㆈ", "ㆈ", "ᆅ", "ᆅ" },
  { "ㆉ", "ㆉ", "ㆉ", "ᆈ", "ᆈ" },
  { "ㆊ", "ㆊ", "ㆊ", "ᆑ", "ᆑ" },
  { "ㆋ", "ㆋ", "ㆋ", "ᆒ", "ᆒ" },
  { "ㆌ", "ㆌ", "ㆌ", "ᆔ", "ᆔ" },
  { "ㆍ", "ㆍ", "ㆍ", "ᆞ", "ᆞ" },
  { "ㆎ", "ㆎ", "ㆎ", "ᆡ", "ᆡ" },
  { "㆒", "㆒", "㆒", "一", "一" },
  { "㆓", "㆓", "㆓", "二", "二" },
  { "㆔", "㆔", "㆔", "三", "三" },
  { "㆕", "㆕", "㆕", "四", "四" },
  { "㆖", "㆖", "㆖", "上", "上" },
  { "㆗", "㆗", "㆗", "中", "中" },
  { "㆘", "㆘", "㆘", "下", "下" },
  { "㆙", "㆙", "㆙", "甲", "甲" },
  { "㆚", "㆚", "㆚", "乙", "乙" },
  { "㆛", "㆛", "㆛", "丙", "丙" },
  { "㆜", "㆜", "㆜", "丁", "丁" },
  { "㆝", "㆝", "㆝", "天", "天" },
  { "㆞", "㆞", "㆞", "地", "地" },
  { "㆟", "㆟", "㆟", "人", "人" },
  { "㈀", "㈀", "㈀", "(ᄀ)", "(ᄀ)" },
  { "㈁", "㈁", "㈁", "(ᄂ)", "(ᄂ)" },
  { "㈂", "㈂", "㈂", "(ᄃ)", "(ᄃ)" },
  { "㈃", "㈃", "㈃", "(ᄅ)", "(ᄅ)" },
  { "㈄", "㈄", "㈄", "(ᄆ)", "(ᄆ)" },
  { "㈅", "㈅", "㈅", "(ᄇ)", "(ᄇ)" },
  { "㈆", "㈆", "㈆", "(ᄉ)", "(ᄉ)" },
  { "㈇", "㈇", "㈇", "(ᄋ)", "(ᄋ)" },
  { "㈈", "㈈", "㈈", "(ᄌ)", "(ᄌ)" },
  { "㈉", "㈉", "㈉", "(ᄎ)", "(ᄎ)" },
  { "㈊", "㈊", "㈊", "(ᄏ)", "(ᄏ)" },
  { "㈋", "㈋", "㈋", "(ᄐ)", "(ᄐ)" },
  { "㈌", "㈌", "㈌", "(ᄑ)", "(ᄑ)" },
  { "㈍", "㈍", "㈍", "(ᄒ)", "(ᄒ)" },
  { "㈎", "㈎", "㈎", "(가)", "(가)" },
  { "㈏", "㈏", "㈏", "(나)", "(나)" },
  { "㈐", "㈐", "㈐", "(다)", "(다)" },
  { "㈑", "㈑", "㈑", "(라)", "(라)" },
  { "㈒", "㈒", "㈒", "(마)", "(마)" },
  { "㈓", "㈓", "㈓", "(바)", "(바)" },
  { "㈔", "㈔", "㈔", "(사)", "(사)" },
  { "㈕", "㈕", "㈕", "(아)", "(아)" },
  { "㈖", "㈖", "㈖", "(자)", "(자)" },
  { "㈗", "㈗", "㈗", "(차)", "(차)" },
  { "㈘", "㈘", "㈘", "(카)", "(카)" },
  { "㈙", "㈙", "㈙", "(타)", "(타)" },
  { "㈚", "㈚", "㈚", "(파)", "(파)" },
  { "㈛", "㈛", "㈛", "(하)", "(하)" },
  { "㈜", "㈜", "㈜", "(주)", "(주)" },
  { "㈝", "㈝", "㈝", "(오전)", "(오전)" },
  { "㈞", "㈞", "㈞", "(오후)", "(오후)" },
  { "㈠", "㈠", "㈠", "(一)", "(一)" },
  { "㈡", "㈡", "㈡", "(二)", "(二)" },
  { "㈢", "㈢", "㈢", "(三)", "(三)" },
  { "㈣", "㈣", "㈣", "(四)", "(四)" },
  { "㈤", "㈤", "㈤", "(五)", "(五)" },
  { "㈥", "㈥", "㈥", "(六)", "(六)" },
  { "㈦", "㈦", "㈦", "(七)", "(七)" },
  { "㈧", "㈧", "㈧", "(八)", "(八)" },
  { "㈨", "㈨", "㈨", "(九)", "(九)" },
  { "㈩", "㈩", "㈩", "(十)", "(十)" },
  { "㈪", "㈪", "㈪", "(月)", "(月)" },
  { "㈫", "㈫", "㈫", "(火)", "(火)" },
  { "㈬", "㈬", "㈬", "(水)", "(水)" },
  { "㈭", "㈭", "㈭", "(木)", "(木)" },
  { "㈮", "㈮", "㈮", "(金)", "(金)" },
  { "㈯", "㈯", "㈯", "(土)", "(土)" },
  { "㈰", "㈰", "㈰", "(日)", "(日)" },
  { "㈱", "㈱", "㈱", "(株)", "(株)" },
  { "㈲", "㈲", "㈲", "(有)", "(有)" },
  { "㈳", "㈳", "㈳", "(社)", "(社)" },
  { "㈴", "㈴", "㈴", "(名)", "(名)" },
  { "㈵", "㈵", "㈵", "(特)", "(特)" },
  { "㈶", "㈶", "㈶", "(財)", "(財)" },
  { "㈷", "㈷", "㈷", "(祝)", "(祝)" },
  { "㈸", "㈸", "㈸", "(労)", "(労)" },
  { "㈹", "㈹", "㈹", "(代)", "(代)" },
  { "㈺", "㈺", "㈺", "(呼)", "(呼)" },
  { "㈻", "㈻", "㈻", "(学)", "(学)" },
  { "㈼", "㈼", "㈼", "(監)", "(監)" },
  { "㈽", "㈽", "㈽", "(企)", "(企)" },
  { "㈾", "㈾", "㈾", "(資)", "(資)" },
  { "㈿", "㈿", "㈿", "(協)", "(協)" },
  { "㉀", "㉀", "㉀", "(祭)", "(祭)" },
  { "㉁", "㉁", "㉁", "(休)", "(休)" },
  { "㉂", "㉂", "㉂", "(自)", "(自)" },
  { "㉃", "㉃", "㉃", "(至)", "(至)" },
  { "㉄", "㉄", "㉄", "問", "問" },
  { "㉅", "㉅", "㉅", "幼", "幼" },
  { "㉆", "㉆", "㉆", "文", "文" },
  { "㉇", "㉇", "㉇", "箏", "箏" },
  { "㉐", "㉐", "㉐", "PTE", "PTE" },
  { "㉑", "㉑", "㉑", "21", "21" },
  { "㉒", "㉒", "㉒", "22", "22" },
  { "㉓", "㉓", "㉓", "23", "23" },
  { "㉔", "㉔", "㉔", "24", "24" },
  { "㉕", "㉕", "㉕", "25", "25" },
  { "㉖", "㉖", "㉖", "26", "26" },
  { "㉗", "㉗", "㉗", "27", "27" },
  { "㉘", "㉘", "㉘", "28", "28" },
  { "㉙", "㉙", "㉙", "29", "29" },
  { "㉚", "㉚", "㉚", "30", "30" },
  { "㉛", "㉛", "㉛", "31", "31" },
  { "㉜", "㉜", "㉜", "32", "32" },
  { "㉝", "㉝", "㉝", "33", "33" },
  { "㉞", "㉞", "㉞", "34", "34" },
  { "㉟", "㉟", "㉟", "35", "35" },
  { "㉠", "㉠", "㉠", "ᄀ", "ᄀ" },
  { "㉡", "㉡", "㉡", "ᄂ", "ᄂ" },
  { "㉢", "㉢", "㉢", "ᄃ", "ᄃ" },
  { "㉣", "㉣", "㉣", "ᄅ", "ᄅ" },
  { "㉤", "㉤", "㉤", "ᄆ", "ᄆ" },
  { "㉥", "㉥", "㉥", "ᄇ", "ᄇ" },
  { "㉦", "㉦", "㉦", "ᄉ", "ᄉ" },
  { "㉧", "㉧", "㉧", "ᄋ", "ᄋ" },
  { "㉨", "㉨", "㉨", "ᄌ", "ᄌ" },
  { "㉩", "㉩", "㉩", "ᄎ", "ᄎ" },
  { "㉪", "㉪", "㉪", "ᄏ", "ᄏ" },
  { "㉫", "㉫", "㉫", "ᄐ", "ᄐ" },
  { "㉬", "㉬", "㉬", "ᄑ", "ᄑ" },
  { "㉭", "㉭", "㉭", "ᄒ", "ᄒ" },
  { "㉮", "㉮", "㉮", "가", "가" },
  { "㉯", "㉯", "㉯", "나", "나" },
  { "㉰", "㉰", "㉰", "다", "다" },
  { "㉱", "㉱", "㉱", "라", "라" },
  { "㉲", "㉲", "㉲", "마", "마" },
  { "㉳", "㉳", "㉳", "바", "바" },
  { "㉴", "㉴", "㉴", "사", "사" },
  { "㉵", "㉵", "㉵", "아", "아" },
  { "㉶", "㉶", "㉶", "자", "자" },
  { "㉷", "㉷", "㉷", "차", "차" },
  { "㉸", "㉸", "㉸", "카", "카" },
  { "㉹", "㉹", "㉹", "타", "타" },
  { "㉺", "㉺", "㉺", "파", "파" },
  { "㉻", "㉻", "㉻", "하", "하" },
  { "㉼", "㉼", "㉼", "참고", "참고" },
  { "㉽", "㉽", "㉽", "주의", "주의" },
  { "㉾", "㉾", "㉾", "우", "우" },
  { "㊀", "㊀", "㊀", "一", "一" },
  { "㊁", "㊁", "㊁", "二", "二" },
  { "㊂", "㊂", "㊂", "三", "三" },
  { "㊃", "㊃", "㊃", "四", "四" },
  { "㊄", "㊄", "㊄", "五", "五" },
  { "㊅", "㊅", "㊅", "六", "六" },
  { "㊆", "㊆", "㊆", "七", "七" },
  { "㊇", "㊇", "㊇", "八", "八" },
  { "㊈", "㊈", "㊈", "九", "九" },
  { "㊉", "㊉", "㊉", "十", "十" },
  { "㊊", "㊊", "㊊", "月", "月" },
  { "㊋", "㊋", "㊋", "火", "火" },
  { "㊌", "㊌", "㊌", "水", "水" },
  { "㊍", "㊍", "㊍", "木", "木" },
  { "㊎", "㊎", "㊎", "金", "金" },
  { "㊏", "㊏", "㊏", "土", "土" },
  { "㊐", "㊐", "㊐", "日", "日" },
  { "㊑", "㊑", "㊑", "株", "株" },
  { "㊒", "㊒", "㊒", "有", "有" },
  { "㊓", "㊓", "㊓", "社", "社" },
  { "㊔", "㊔", "㊔", "名", "名" },
  { "㊕", "㊕", "㊕", "特", "特" },
  { "㊖", "㊖", "㊖", "財", "財" },
  { "㊗", "㊗", "㊗", "祝", "祝" },
  { "㊘", "㊘", "㊘", "労", "労" },
  { "㊙", "㊙", "㊙", "秘", "秘" },
  { "㊚", "㊚", "㊚", "男", "男" },
  { "㊛", "㊛", "㊛", "女", "女" },
  { "㊜", "㊜", "㊜", "適", "適" },
  { "㊝", "㊝", "㊝", "優", "優" },
  { "㊞", "㊞", "㊞", "印", "印" },
  { "㊟", "㊟", "㊟", "注", "注" },
  { "㊠", "㊠", "㊠", "項", "項" },
  { "㊡", "㊡", "㊡", "休", "休" },
  { "㊢", "㊢", "㊢", "写", "写" },
  { "㊣", "㊣", "㊣", "正", "正" },
  { "㊤", "㊤", "㊤", "上", "上" },
  { "㊥", "㊥", "㊥", "中", "中" },
  { "㊦", "㊦", "㊦", "下", "下" },
  { "㊧", "㊧", "㊧", "左", "左" },
  { "㊨", "㊨", "㊨", "右", "右" },
  { "㊩", "㊩", "㊩", "医", "医" },
  { "㊪", "㊪", "㊪", "宗", "宗" },
  { "㊫", "㊫", "㊫", "学", "学" },
  { "㊬", "㊬", "㊬", "監", "監" },
  { "㊭", "㊭", "㊭", "企", "企" },
  { "㊮", "㊮", "㊮", "資", "資" },
  { "㊯", "㊯", "㊯", "協", "協" },
  { "㊰", "㊰", "㊰", "夜", "夜" },
  { "㊱", "㊱", "㊱", "36", "36" },
  { "㊲", "㊲", "㊲", "37", "37" },
  { "㊳", "㊳", "㊳", "38", "38" },
  { "㊴", "㊴", "㊴", "39", "39" },
  { "㊵", "㊵", "㊵", "40", "40" },
  { "㊶", "㊶", "㊶", "41", "41" },
  { "㊷", "㊷", "㊷", "42", "42" },
  { "㊸", "㊸", "㊸", "43", "43" },
  { "㊹", "㊹", "㊹", "44", "44" },
  { "㊺", "㊺", "㊺", "45", "45" },
  { "㊻", "㊻", "㊻", "46", "46" },
  { "㊼", "㊼", "㊼", "47", "47" },
  { "㊽", "㊽", "㊽", "48", "48" },
  { "㊾", "㊾", "㊾", "49", "49" },
  { "㊿", "㊿", "㊿", "50", "50" },
  { "㋀", "㋀", "㋀", "1月", "1月" },
  { "㋁", "㋁", "㋁", "2月", "2月" },
  { "㋂", "㋂", "㋂", "3月", "3月" },
  { "㋃", "㋃", "㋃", "4月", "4月" },
  { "㋄", "㋄", "㋄", "5月", "5月" },
  { "㋅", "㋅", "㋅", "6月", "6月" },
  { "㋆", "㋆", "㋆", "7月", "7月" },
  { "㋇", "㋇", "㋇", "8月", "8月" },
  { "㋈", "㋈", "㋈", "9月", "9月" },
  { "㋉", "㋉", "㋉", "10月", "10月" },
  { "㋊", "㋊", "㋊", "11月", "11月" },
  { "㋋", "㋋", "㋋", "12月", "12月" },
  { "㋌", "㋌", "㋌", "Hg", "Hg" },
  { "㋍", "㋍", "㋍", "erg", "erg" },
  { "㋎", "㋎", "㋎", "eV", "eV" },
  { "㋏", "㋏", "㋏", "LTD", "LTD" },
  { "㋐", "㋐", "㋐", "ア", "ア" },
  { "㋑", "㋑", "㋑", "イ", "イ" },
  { "㋒", "㋒", "㋒", "ウ", "ウ" },
  { "㋓", "㋓", "㋓", "エ", "エ" },
  { "㋔", "㋔", "㋔", "オ", "オ" },
  { "㋕", "㋕", "㋕", "カ", "カ" },
  { "㋖", "㋖", "㋖", "キ", "キ" },
  { "㋗", "㋗", "㋗", "ク", "ク" },
  { "㋘", "㋘", "㋘", "ケ", "ケ" },
  { "㋙", "㋙", "㋙", "コ", "コ" },
  { "㋚", "㋚", "㋚", "サ", "サ" },
  { "㋛", "㋛", "㋛", "シ", "シ" },
  { "㋜", "㋜", "㋜", "ス", "ス" },
  { "㋝", "㋝", "㋝", "セ", "セ" },
  { "㋞", "㋞", "㋞", "ソ", "ソ" },
  { "㋟", "㋟", "㋟", "タ", "タ" },
  { "㋠", "㋠", "㋠", "チ", "チ" },
  { "㋡", "㋡", "㋡", "ツ", "ツ" },
  { "㋢", "㋢", "㋢", "テ", "テ" },
  { "㋣", "㋣", "㋣", "ト", "ト" },
  { "㋤", "㋤", "㋤", "ナ", "ナ" },
  { "㋥", "㋥", "㋥", "ニ", "ニ" },
  { "㋦", "㋦", "㋦", "ヌ", "ヌ" },
  { "㋧", "㋧", "㋧", "ネ", "ネ" },
  { "㋨", "㋨", "㋨", "ノ", "ノ" },
  { "㋩", "㋩", "㋩", "ハ", "ハ" },
  { "㋪", "㋪", "㋪", "ヒ", "ヒ" },
  { "㋫", "㋫", "㋫", "フ", "フ" },
  { "㋬", "㋬", "㋬", "ヘ", "ヘ" },
  { "㋭", "㋭", "㋭", "ホ", "ホ" },
  { "㋮", "㋮", "㋮", "マ", "マ" },
  { "㋯", "㋯", "㋯", "ミ", "ミ" },
  { "㋰", "㋰", "㋰", "ム", "ム" },
  { "㋱", "㋱", "㋱", "メ", "メ" },
  { "㋲", "㋲", "㋲", "モ", "モ" },
  { "㋳", "㋳", "㋳", "ヤ", "ヤ" },
  { "㋴", "㋴", "㋴", "ユ", "ユ" },
  { "㋵", "㋵", "㋵", "ヨ", "ヨ" },
  { "㋶", "㋶", "㋶", "ラ", "ラ" },
  { "㋷", "㋷", "㋷", "リ", "リ" },
  { "㋸", "㋸", "㋸", "ル", "ル" },
  { "㋹", "㋹", "㋹", "レ", "レ" },
  { "㋺", "㋺", "㋺", "ロ", "ロ" },
  { "㋻", "㋻", "㋻", "ワ", "ワ" },
  { "㋼", "㋼", "㋼", "ヰ", "ヰ" },
  { "㋽", "㋽", "㋽", "ヱ", "ヱ" },
  { "㋾", "㋾", "㋾", "ヲ", "ヲ" },
  { "㌀", "㌀", "㌀", "アパート", "アパート" },
  { "㌁", "㌁", "㌁", "アルファ", "アルファ" },
  { "㌂", "㌂", "㌂", "アンペア", "アンペア" },
  { "㌃", "㌃", "㌃", "アール", "アール" },
  { "㌄", "㌄", "㌄", "イニング", "イニング" },
  { "㌅", "㌅", "㌅", "インチ", "インチ" },
  { "㌆", "㌆", "㌆", "ウォン", "ウォン" },
  { "㌇", "㌇", "㌇", "エスクード", "エスクード" },
  { "㌈", "㌈", "㌈", "エーカー", "エーカー" },
  { "㌉", "㌉", "㌉", "オンス", "オンス" },
  { "㌊", "㌊", "㌊", "オーム", "オーム" },
  { "㌋", "㌋", "㌋", "カイリ", "カイリ" },
  { "㌌", "㌌", "㌌", "カラット", "カラット" },
  { "㌍", "㌍", "㌍", "カロリー", "カロリー" },
  { "㌎", "㌎", "㌎", "ガロン", "ガロン" },
  { "㌏", "㌏", "㌏", "ガンマ", "ガンマ" },
  { "㌐", "㌐", "㌐", "ギガ", "ギガ" },
  { "㌑", "㌑", "㌑", "ギニー", "ギニー" },
  { "㌒", "㌒", "㌒", "キュリー", "キュリー" },
  { "㌓", "㌓", "㌓", "ギルダー", "ギルダー" },
  { "㌔", "㌔", "㌔", "キロ", "キロ" },
  { "㌕", "㌕", "㌕", "キログラム", "キログラム" },
  { "㌖", "㌖", "㌖", "キロメートル", "キロメートル" },
  { "㌗", "㌗", "㌗", "キロワット", "キロワット" },
  { "㌘", "㌘", "㌘", "グラム", "グラム" },
  { "㌙", "㌙", "㌙", "グラムトン", "グラムトン" },
  { "㌚", "㌚", "㌚", "クルゼイロ", "クルゼイロ" },
  { "㌛", "㌛", "㌛", "クローネ", "クローネ" },
  { "㌜", "㌜", "㌜", "ケース", "ケース" },
  { "㌝", "㌝", "㌝", "コルナ", "コルナ" },
  { "㌞", "㌞", "㌞", "コーポ", "コーポ" },
  { "㌟", "㌟", "㌟", "サイクル", "サイクル" },
  { "㌠", "㌠", "㌠", "サンチーム", "サンチーム" },
  { "㌡", "㌡", "㌡", "シリング", "シリング" },
  { "㌢", "㌢", "㌢", "センチ", "センチ" },
  { "㌣", "㌣", "㌣", "セント", "セント" },
  { "㌤", "㌤", "㌤", "ダース", "ダース" },
  { "㌥", "㌥", "㌥", "デシ", "デシ" },
  { "㌦", "㌦", "㌦", "ドル", "ドル" },
  { "㌧", "㌧", "㌧", "トン", "トン" },
  { "㌨", "㌨", "㌨", "ナノ", "ナノ" },
  { "㌩", "㌩", "㌩", "ノット", "ノット" },
  { "㌪", "㌪", "㌪", "ハイツ", "ハイツ" },
  { "㌫", "㌫", "㌫", "パーセント", "パーセント" },
  { "㌬", "㌬", "㌬", "パーツ", "パーツ" },
  { "㌭", "㌭", "㌭", "バーレル", "バーレル" },
  { "㌮", "㌮", "㌮", "ピアストル", "ピアストル" },
  { "㌯", "㌯", "㌯", "ピクル", "ピクル" },
  { "㌰", "㌰", "㌰", "ピコ", "ピコ" },
  { "㌱", "㌱", "㌱", "ビル", "ビル" },
  { "㌲", "㌲", "㌲", "ファラッド", "ファラッド" },
  { "㌳", "㌳", "㌳", "フィート", "フィート" },
  { "㌴", "㌴", "㌴", "ブッシェル", "ブッシェル" },
  { "㌵", "㌵", "㌵", "フラン", "フラン" },
  { "㌶", "㌶", "㌶", "ヘクタール", "ヘクタール" },
  { "㌷", "㌷", "㌷", "ペソ", "ペソ" },
  { "㌸", "㌸", "㌸", "ペニヒ", "ペニヒ" },
  { "㌹", "㌹", "㌹", "ヘルツ", "ヘルツ" },
  { "㌺", "㌺", "㌺", "ペンス", "ペンス" },
  { "㌻", "㌻", "㌻", "ページ", "ページ" },
  { "㌼", "㌼", "㌼", "ベータ", "ベータ" },
  { "㌽", "㌽", "㌽", "ポイント", "ポイント" },
  { "㌾", "㌾", "㌾", "ボルト", "ボルト" },
  { "㌿", "㌿", "㌿", "ホン", "ホン" },
  { "㍀", "㍀", "㍀", "ポンド", "ポンド" },
  { "㍁", "㍁", "㍁", "ホール", "ホール" },
  { "㍂", "㍂", "㍂", "ホーン", "ホーン" },
  { "㍃", "㍃", "㍃", "マイクロ", "マイクロ" },
  { "㍄", "㍄", "㍄", "マイル", "マイル" },
  { "㍅", "㍅", "㍅", "マッハ", "マッハ" },
  { "㍆", "㍆", "㍆", "マルク", "マルク" },
  { "㍇", "㍇", "㍇", "マンション", "マンション" },
  { "㍈", "㍈", "㍈", "ミクロン", "ミクロン" },
  { "㍉", "㍉", "㍉", "ミリ", "ミリ" },
  { "㍊", "㍊", "㍊", "ミリバール", "ミリバール" },
  { "㍋", "㍋", "㍋", "メガ", "メガ" },
  { "㍌", "㍌", "㍌", "メガトン", "メガトン" },
  { "㍍", "㍍", "㍍", "メートル", "メートル" },
  { "㍎", "㍎", "㍎", "ヤード", "ヤード" },
  { "㍏", "㍏", "㍏", "ヤール", "ヤール" },
  { "㍐", "㍐", "㍐", "ユアン", "ユアン" },
  { "㍑", "㍑", "㍑", "リットル", "リットル" },
  { "㍒", "㍒", "㍒", "リラ", "リラ" },
  { "㍓", "㍓", "㍓", "ルピー", "ルピー" },
  { "㍔", "㍔", "㍔", "ルーブル", "ルーブル" },
  { "㍕", "㍕", "㍕", "レム", "レム" },
  { "㍖", "㍖", "㍖", "レントゲン", "レントゲン" },
  { "㍗", "㍗", "㍗", "ワット", "ワット" },
  { "㍘", "㍘", "㍘", "0点", "0点" },
  { "㍙", "㍙", "㍙", "1点", "1点" },
  { "㍚", "㍚", "㍚", "2点", "2点" },
  { "㍛", "㍛", "㍛", "3点", "3点" },
  { "㍜", "㍜", "㍜", "4点", "4点" },
  { "㍝", "㍝", "㍝", "5点", "5点" },
  { "㍞", "㍞", "㍞", "6点", "6点" },
  { "㍟", "㍟", "㍟", "7点", "7点" },
  { "㍠", "㍠", "㍠", "8点", "8点" },
  { "㍡", "㍡", "㍡", "9点", "9点" },
  { "㍢", "㍢", "㍢", "10点", "10点" },
  { "㍣", "㍣", "㍣", "11点", "11点" },
  { "㍤", "㍤", "㍤", "12点", "12点" },
  { "㍥", "㍥", "㍥", "13点", "13点" },
  { "㍦", "㍦", "㍦", "14点", "14点" },
  { "㍧", "㍧", "㍧", "15点", "15点" },
  { "㍨", "㍨", "㍨", "16点", "16点" },
  { "㍩", "㍩", "㍩", "17点", "17点" },
  { "㍪", "㍪", "㍪", "18点", "18点" },
  { "㍫", "㍫", "㍫", "19点", "19点" },
  { "㍬", "㍬", "㍬", "20点", "20点" },
  { "㍭", "㍭", "㍭", "21点", "21点" },
  { "㍮", "㍮", "㍮", "22点", "22点" },
  { "㍯", "㍯", "㍯", "23点", "23点" },
  { "㍰", "㍰", "㍰", "24点", "24点" },
  { "㍱", "㍱", "㍱", "hPa", "hPa" },
  { "㍲", "㍲", "㍲", "da", "da" },
  { "㍳", "㍳", "㍳", "AU", "AU" },
  { "㍴", "㍴", "㍴", "bar", "bar" },
  { "㍵", "㍵", "㍵", "oV", "oV" },
  { "㍶", "㍶", "㍶", "pc", "pc" },
  { "㍷", "㍷", "㍷", "dm", "dm" },
  { "㍸", "㍸", "㍸", "dm2", "dm2" },
  { "㍹", "㍹", "㍹", "dm3", "dm3" },
  { "㍺", "㍺", "㍺", "IU", "IU" },
  { "㍻", "㍻", "㍻", "平成", "平成" },
  { "㍼", "㍼", "㍼", "昭和", "昭和" },
  { "㍽", "㍽", "㍽", "大正", "大正" },
  { "㍾", "㍾", "㍾", "明治", "明治" },
  { "㍿", "㍿", "㍿", "株式会社", "株式会社" },
  { "㎀", "㎀", "㎀", "pA", "pA" },
  { "㎁", "㎁", "㎁", "nA", "nA" },
  { "㎂", "㎂", "㎂", "μA", "μA" },
  { "㎃", "㎃", "㎃", "mA", "mA" },
  { "㎄", "㎄", "㎄", "kA", "kA" },
  { "㎅", "㎅", "㎅", "KB", "KB" },
  { "㎆", "㎆", "㎆", "MB", "MB" },
  { "㎇", "㎇", "㎇", "GB", "GB" },
  { "㎈", "㎈", "㎈", "cal", "cal" },
  { "㎉", "㎉", "㎉", "kcal", "kcal" },
  { "㎊", "㎊", "㎊", "pF", "pF" },
  { "㎋", "㎋", "㎋", "nF", "nF" },
  { "㎌", "㎌", "㎌", "μF", "μF" },
  { "㎍", "㎍", "㎍", "μg", "μg" },
  { "㎎", "㎎", "㎎", "mg", "mg" },
  { "㎏", "㎏", "㎏", "kg", "kg" },
  { "㎐", "㎐", "㎐", "Hz", "Hz" },
  { "㎑", "㎑", "㎑", "kHz", "kHz" },
  { "㎒", "㎒", "㎒", "MHz", "MHz" },
  { "㎓", "㎓", "㎓", "GHz", "GHz" },
  { "㎔", "㎔", "㎔", "THz", "THz" },
  { "㎕", "㎕", "㎕", "μl", "μl" },
  { "㎖", "㎖", "㎖", "ml", "ml" },
  { "㎗", "㎗", "㎗", "dl", "dl" },
  { "㎘", "㎘", "㎘", "kl", "kl" },
  { "㎙", "㎙", "㎙", "fm", "fm" },
  { "㎚", "㎚", "㎚", "nm", "nm" },
  { "㎛", "㎛", "㎛", "μm", "μm" },
  { "㎜", "㎜", "㎜", "mm", "mm" },
  { "㎝", "㎝", "㎝", "cm", "cm" },
  { "㎞", "㎞", "㎞", "km", "km" },
  { "㎟", "㎟", "㎟", "mm2", "mm2" },
  { "㎠", "㎠", "㎠", "cm2", "cm2" },
  { "㎡", "㎡", "㎡", "m2", "m2" },
  { "㎢", "㎢", "㎢", "km2", "km2" },
  { "㎣", "㎣", "㎣", "mm3", "mm3" },
  { "㎤", "㎤", "㎤", "cm3", "cm3" },
  { "㎥", "㎥", "㎥", "m3", "m3" },
  { "㎦", "㎦", "㎦", "km3", "km3" },
  { "㎧", "㎧", "㎧", "m∕s", "m∕s" },
  { "㎨", "㎨", "㎨", "m∕s2", "m∕s2" },
  { "㎩", "㎩", "㎩", "Pa", "Pa" },
  { "㎪", "㎪", "㎪", "kPa", "kPa" },
  { "㎫", "㎫", "㎫", "MPa", "MPa" },
  { "㎬", "㎬", "㎬", "GPa", "GPa" },
  { "㎭", "㎭", "㎭", "rad", "rad" },
  { "㎮", "㎮", "㎮", "rad∕s", "rad∕s" },
  { "㎯", "㎯", "㎯", "rad∕s2", "rad∕s2" },
  { "㎰", "㎰", "㎰", "ps", "ps" },
  { "㎱", "㎱", "㎱", "ns", "ns" },
  { "㎲", "㎲", "㎲", "μs", "μs" },
  { "㎳", "㎳", "㎳", "ms", "ms" },
  { "㎴", "㎴", "㎴", "pV", "pV" },
  { "㎵", "㎵", "㎵", "nV", "nV" },
  { "㎶", "㎶", "㎶", "μV", "μV" },
  { "㎷", "㎷", "㎷", "mV", "mV" },
  { "㎸", "㎸", "㎸", "kV", "kV" },
  { "㎹", "㎹", "㎹", "MV", "MV" },
  { "㎺", "㎺", "㎺", "pW", "pW" },
  { "㎻", "㎻", "㎻", "nW", "nW" },
  { "㎼", "㎼", "㎼", "μW", "μW" },
  { "㎽", "㎽", "㎽", "mW", "mW" },
  { "㎾", "㎾", "㎾", "kW", "kW" },
  { "㎿", "㎿", "㎿", "MW", "MW" },
  { "㏀", "㏀", "㏀", "kΩ", "kΩ" },
  { "㏁", "㏁", "㏁", "MΩ", "MΩ" },
  { "㏂", "㏂", "㏂", "a.m.", "a.m." },
  { "㏃", "㏃", "㏃", "Bq", "Bq" },
  { "㏄", "㏄", "㏄", "cc", "cc" },
  { "㏅", "㏅", "㏅", "cd", "cd" },
  { "㏆", "㏆", "㏆", "C∕kg", "C∕kg" },
  { "㏇", "㏇", "㏇", "Co.", "Co." },
  { "㏈", "㏈", "㏈", "dB", "dB" },
  { "㏉", "㏉", "㏉", "Gy", "Gy" },
  { "㏊", "㏊", "㏊", "ha", "ha" },
  { "㏋", "㏋", "㏋", "HP", "HP" },
  { "㏌", "㏌", "㏌", "in", "in" },
  { "㏍", "㏍", "㏍", "KK", "KK" },
  { "㏎", "㏎", "㏎", "KM", "KM" },
  { "㏏", "㏏", "㏏", "kt", "kt" },
  { "㏐", "㏐", "㏐", "lm", "lm" },
  { "㏑", "㏑", "㏑", "ln", "ln" },
  { "㏒", "㏒", "㏒", "log", "log" },
  { "㏓", "㏓", "㏓", "lx", "lx" },
  { "㏔", "㏔", "㏔", "mb", "mb" },
  { "㏕", "㏕", "㏕", "mil", "mil" },
  { "㏖", "㏖", "㏖", "mol", "mol" },
  { "㏗", "㏗", "㏗", "PH", "PH" },
  { "㏘", "㏘", "㏘", "p.m.", "p.m." },
  { "㏙", "㏙", "㏙", "PPM", "PPM" },
  { "㏚", "㏚", "㏚", "PR", "PR" },
  { "㏛", "㏛", "㏛", "sr", "sr" },
  { "㏜", "㏜", "㏜", "Sv", "Sv" },
  { "㏝", "㏝", "㏝", "Wb", "Wb" },
  { "㏞", "㏞", "㏞", "V∕m", "V∕m" },
  { "㏟", "㏟", "㏟", "A∕m", "A∕m" },
  { "㏠", "㏠", "㏠", "1日", "1日" },
  { "㏡", "㏡", "㏡", "2日", "2日" },
  { "㏢", "㏢", "㏢", "3日", "3日" },
  { "㏣", "㏣", "㏣", "4日", "4日" },
  { "㏤", "㏤", "㏤", "5日", "5日" },
  { "㏥", "㏥", "㏥", "6日", "6日" },
  { "㏦", "㏦", "㏦", "7日", "7日" },
  { "㏧", "㏧", "㏧", "8日", "8日" },
  { "㏨", "㏨", "㏨", "9日", "9日" },
  { "㏩", "㏩", "㏩", "10日", "10日" },
  { "㏪", "㏪", "㏪", "11日", "11日" },
  { "㏫", "㏫", "㏫", "12日", "12日" },
  { "㏬", "㏬", "㏬", "13日", "13日" },
  { "㏭", "㏭", "㏭", "14日", "14日" },
  { "㏮", "㏮", "㏮", "15日", "15日" },
  { "㏯", "㏯", "㏯", "16日", "16日" },
  { "㏰", "㏰", "㏰", "17日", "17日" },
  { "㏱", "㏱", "㏱", "18日", "18日" },
  { "㏲", "㏲", "㏲", "19日", "19日" },
  { "㏳", "㏳", "㏳", "20日", "20日" },
  { "㏴", "㏴", "㏴", "21日", "21日" },
  { "㏵", "㏵", "㏵", "22日", "22日" },
  { "㏶", "㏶", "㏶", "23日", "23日" },
  { "㏷", "㏷", "㏷", "24日", "24日" },
  { "㏸", "㏸", "㏸", "25日", "25日" },
  { "㏹", "㏹", "㏹", "26日", "26日" },
  { "㏺", "㏺", "㏺", "27日", "27日" },
  { "㏻", "㏻", "㏻", "28日", "28日" },
  { "㏼", "㏼", "㏼", "29日", "29日" },
  { "㏽", "㏽", "㏽", "30日", "30日" },
  { "㏾", "㏾", "㏾", "31日", "31日" },
  { "㏿", "㏿", "㏿", "gal", "gal" },
  { "ꚜ", "ꚜ", "ꚜ", "ъ", "ъ" },
  { "ꚝ", "ꚝ", "ꚝ", "ь", "ь" },
  { "ꝰ", "ꝰ", "ꝰ", "ꝯ", "ꝯ" },
  { "ꟸ", "ꟸ", "ꟸ", "Ħ", "Ħ" },
  { "ꟹ", "ꟹ", "ꟹ", "œ", "œ" },
  { "ꭜ", "ꭜ", "ꭜ", "ꜧ", "ꜧ" },
  { "ꭝ", "ꭝ", "ꭝ", "ꬷ", "ꬷ" },
  { "ꭞ", "ꭞ", "ꭞ", "ɫ", "ɫ" },
  { "ꭟ", "ꭟ", "ꭟ", "ꭒ", "ꭒ" },
  { "가", "가", "가", "가", "가" },
  { "각", "각", "각", "각", "각" },
  { "갂", "갂", "갂", "갂", "갂" },
  { "갃", "갃", "갃", "갃", "갃" },
  { "간", "간", "간", "간", "간" },
  { "갅", "갅", "갅", "갅", "갅" },
  { "갆", "갆", "갆", "갆", "갆" },
  { "갇", "갇", "갇", "갇", "갇" },
  { "갈", "갈", "갈", "갈", "갈" },
  { "갉", "갉", "갉", "갉", "갉" },
  { "갊", "갊", "갊", "갊", "갊" },
  { "갋", "갋", "갋", "갋", "갋" },
  { "갌", "갌", "갌", "갌", "갌" },
  { "갍", "갍", "갍", "갍", "갍" },
  { "갎", "갎", "갎", "갎", "갎" },
  { "갏", "갏", "갏", "갏", "갏" },
  { "감", "감", "감", "감", "감" },
  { "갑", "갑", "갑", "갑", "갑" },
  { "값", "값", "값", "값", "값" },
  { "갓", "갓", "갓", "갓", "갓" },
  { "갔", "갔", "갔", "갔", "갔" },
  { "강", "강", "강", "강", "강" },
  { "갖", "갖", "갖", "갖", "갖" },
  { "갗", "갗", "갗", "갗", "갗" },
  { "갘", "갘", "갘", "갘", "갘" },
  { "같", "같", "같", "같", "같" },
  { "갚", "갚", "갚", "갚", "갚" },
  { "갛", "갛", "갛", "갛", "갛" },
  { "개", "개", "개", "개", "개" },
  { "객", "객", "객", "객", "객" },
  { "갞", "갞", "갞", "갞", "갞" },
  { "갟", "갟", "갟", "갟", "갟" },
  { "갠", "갠", "갠", "갠", "갠" },
  { "갡", "갡", "갡", "갡", "갡" },
  { "갢", "갢", "갢", "갢", "갢" },
  { "갣", "갣", "갣", "갣", "갣" },
  { "갤", "갤", "갤", "갤", "갤" },
  { "갥", "갥", "갥", "갥", "갥" },
  { "갦", "갦", "갦", "갦", "갦" },
  { "갧", "갧", "갧", "갧", "갧" },
  { "갨", "갨", "갨", "갨", "갨" },
  { "갩", "갩", "갩", "갩", "갩" },
  { "갪", "갪", "갪", "갪", "갪" },
  { "갫", "갫", "갫", "갫", "갫" },
  { "갬", "갬", "갬", "갬", "갬" },
  { "갭", "갭", "갭", "갭", "갭" },
  { "갮", "갮", "갮", "갮", "갮" },
  { "갯", "갯", "갯", "갯", "갯" },
  { "갰", "갰", "갰", "갰", "갰" },
  { "갱", "갱", "갱", "갱", "갱" },
  { "갲", "갲", "갲", "갲", "갲" },
  { "갳", "갳", "갳", "갳", "갳" },
  { "갴", "갴", "갴", "갴", "갴" },
  { "갵", "갵", "갵", "갵", "갵" },
  { "갶", "갶", "갶", "갶", "갶" },
  { "갷", "갷", "갷", "갷", "갷" },
  { "갸", "갸", "갸", "갸", "갸" },
  { "갹", "갹", "갹", "갹", "갹" },
  { "갺", "갺", "갺", "갺", "갺" },
  { "갻", "갻", "갻", "갻", "갻" },
  { "갼", "갼", "갼", "갼", "갼" },
  { "갽", "갽", "갽", "갽", "갽" },
  { "갾", "갾", "갾", "갾", "갾" },
  { "갿", "갿", "갿", "갿", "갿" },
  { "걀", "걀", "걀", "걀", "걀" },
  { "걁", "걁", "걁", "걁", "걁" },
  { "걂", "걂", "걂", "걂", "걂" },
  { "걃", "걃", "걃", "걃", "걃" },
  { "걄", "걄", "걄", "걄", "걄" },
  { "걅", "걅", "걅", "걅", "걅" },
  { "걆", "걆", "걆", "걆", "걆" },
  { "걇", "걇", "걇", "걇", "걇" },
  { "걈", "걈", "걈", "걈", "걈" },
  { "걉", "걉", "걉", "걉", "걉" },
  { "걊", "걊", "걊", "걊", "걊" },
  { "걋", "걋", "걋", "걋", "걋" },
  { "걌", "걌", "걌", "걌", "걌" },
  { "걍", "걍", "걍", "걍", "걍" },
  { "걎", "걎", "걎", "걎", "걎" },
  { "걏", "걏", "걏", "걏", "걏" },
  { "걐", "걐", "걐", "걐", "걐" },
  { "걑", "걑", "걑", "걑", "걑" },
  { "걒", "걒", "걒", "걒", "걒" },
  { "걓", "걓", "걓", "걓", "걓" },
  { "걔", "걔", "걔", "걔", "걔" },
  { "걕", "걕", "걕", "걕", "걕" },
  { "걖", "걖", "걖", "걖", "걖" },
  { "걗", "걗", "걗", "걗", "걗" },
  { "걘", "걘", "걘", "걘", "걘" },
  { "걙", "걙", "걙", "걙", "걙" },
  { "걚", "걚", "걚", "걚", "걚" },
  { "걛", "걛", "걛", "걛", "걛" },
  { "걜", "걜", "걜", "걜", "걜" },
  { "걝", "걝", "걝", "걝", "걝" },
  { "걞", "걞", "걞", "걞", "걞" },
  { "걟", "걟", "걟", "걟", "걟" },
  { "걠", "걠", "걠", "걠", "걠" },
  { "걡", "걡", "걡", "걡", "걡" },
  { "걢", "걢", "걢", "걢", "걢" },
  { "걣", "걣", "걣", "걣", "걣" },
  { "걤", "걤", "걤", "걤", "걤" },
  { "걥", "걥", "걥", "걥", "걥" },
  { "걦", "걦", "걦", "걦", "걦" },
  { "걧", "걧", "걧", "걧", "걧" },
  { "걨", "걨", "걨", "걨", "걨" },
  { "걩", "걩", "걩", "걩", "걩" },
  { "걪", "걪", "걪", "걪", "걪" },
  { "걫", "걫", "걫", "걫", "걫" },
  { "걬", "걬", "걬", "걬", "걬" },
  { "걭", "걭", "걭", "걭", "걭" },
  { "걮", "걮", "걮", "걮", "걮" },
  { "걯", "걯", "걯", "걯", "걯" },
  { "거", "거", "거", "거", "거" },
  { "걱", "걱", "걱", "걱", "걱" },
  { "걲", "걲", "걲", "걲", "걲" },
  { "걳", "걳", "걳", "걳", "걳" },
  { "건", "건", "건", "건", "건" },
  { "걵", "걵", "걵", "걵", "걵" },
  { "걶", "걶", "걶", "걶", "걶" },
  { "걷", "걷", "걷", "걷", "걷" },
  { "걸", "걸", "걸", "걸", "걸" },
  { "걹", "걹", "걹", "걹", "걹" },
  { "걺", "걺", "걺", "걺", "걺" },
  { "걻", "걻", "걻", "걻", "걻" },
  { "걼", "걼", "걼", "걼", "걼" },
  { "걽", "걽", "걽", "걽", "걽" },
  { "걾", "걾", "걾", "걾", "걾" },
  { "걿", "걿", "걿", "걿", "걿" },
  { "검", "검", "검", "검", "검" },
  { "겁", "겁", "겁", "겁", "겁" },
  { "겂", "겂", "겂", "겂", "겂" },
  { "것", "것", "것", "것", "것" },
  { "겄", "겄", "겄", "겄", "겄" },
  { "겅", "겅", "겅", "겅", "겅" },
  { "겆", "겆", "겆", "겆", "겆" },
  { "겇", "겇", "겇", "겇", "겇" },
  { "겈", "겈", "겈", "겈", "겈" },
  { "겉", "겉", "겉", "겉", "겉" },
  { "겊", "겊", "겊", "겊", "겊" },
  { "겋", "겋", "겋", "겋", "겋" },
  { "게", "게", "게", "게", "게" },
  { "겍", "겍", "겍", "겍", "겍" },
  { "겎", "겎", "겎", "겎", "겎" },
  { "겏", "겏", "겏", "겏", "겏" },
  { "겐", "겐", "겐", "겐", "겐" },
  { "겑", "겑", "겑", "겑", "겑" },
  { "겒", "겒", "겒", "겒", "겒" },
  { "겓", "겓", "겓", "겓", "겓" },
  { "겔", "겔", "겔", "겔", "겔" },
  { "겕", "겕", "겕", "겕", "겕" },
  { "겖", "겖", "겖", "겖", "겖" },
  { "겗", "겗", "겗", "겗", "겗" },
  { "겘", "겘", "겘", "겘", "겘" },
  { "겙", "겙", "겙", "겙", "겙" },
  { "겚", "겚", "겚", "겚", "겚" },
  { "겛", "겛", "겛", "겛", "겛" },
  { "겜", "겜", "겜", "겜", "겜" },
  { "겝", "겝", "겝", "겝", "겝" },
  { "겞", "겞", "겞", "겞", "겞" },
  { "겟", "겟", "겟", "겟", "겟" },
  { "겠", "겠", "겠", "겠", "겠" },
  { "겡", "겡", "겡", "겡", "겡" },
  { "겢", "겢", "겢", "겢", "겢" },
  { "겣", "겣", "겣", "겣", "겣" },
  { "겤", "겤", "겤", "겤", "겤" },
  { "겥", "겥", "겥", "겥", "겥" },
  { "겦", "겦", "겦", "겦", "겦" },
  { "겧", "겧", "겧", "겧", "겧" },
  { "겨", "겨", "겨", "겨", "겨" },
  { "격", "격", "격", "격", "격" },
  { "겪", "겪", "겪", "겪", "겪" },
  { "겫", "겫", "겫", "겫", "겫" },
  { "견", "견", "견", "견", "견" },
  { "겭", "겭", "겭", "겭", "겭" },
  { "겮", "겮", "겮", "겮", "겮" },
  { "겯", "겯", "겯", "겯", "겯" },
  { "결", "결", "결", "결", "결" },
  { "겱", "겱", "겱", "겱", "겱" },
  { "겲", "겲", "겲", "겲", "겲" },
  { "겳", "겳", "겳", "겳", "겳" },
  { "겴", "겴", "겴", "겴", "겴" },
  { "겵", "겵", "겵", "겵", "겵" },
  { "겶", "겶", "겶", "겶", "겶" },
  { "겷", "겷", "겷", "겷", "겷" },
  { "겸", "겸", "겸", "겸", "겸" },
  { "겹", "겹", "겹", "겹", "겹" },
  { "겺", "겺", "겺", "겺", "겺" },
  { "겻", "겻", "겻", "겻", "겻" },
  { "겼", "겼", "겼", "겼", "겼" },
  { "경", "경", "경", "경", "경" },
  { "겾", "겾", "겾", "겾", "겾" },
  { "겿", "겿", "겿", "겿", "겿" },
  { "곀", "곀", "곀", "곀", "곀" },
  { "곁", "곁", "곁", "곁", "곁" },
  { "곂", "곂", "곂", "곂", "곂" },
  { "곃", "곃", "곃", "곃", "곃" },
  { "계", "계", "계", "계", "계" },
  { "곅", "곅", "곅", "곅", "곅" },
  { "곆", "곆", "곆", "곆", "곆" },
  { "곇", "곇", "곇", "곇", "곇" },
  { "곈", "곈", "곈", "곈", "곈" },
  { "곉", "곉", "곉", "곉", "곉" },
  { "곊", "곊", "곊", "곊", "곊" },
  { "곋", "곋", "곋", "곋", "곋" },
  { "곌", "곌", "곌", "곌", "곌" },
  { "곍", "곍", "곍", "곍", "곍" },
  { "곎", "곎", "곎", "곎", "곎" },
  { "곏", "곏", "곏", "곏", "곏" },
  { "곐", "곐", "곐", "곐", "곐" },
  { "곑", "곑", "곑", "곑", "곑" },
  { "곒", "곒", "곒", "곒", "곒" },
  { "곓", "곓", "곓", "곓", "곓" },
  { "곔", "곔", "곔", "곔", "곔" },
  { "곕", "곕", "곕", "곕", "곕" },
  { "곖", "곖", "곖", "곖", "곖" },
  { "곗", "곗", "곗", "곗", "곗" },
  { "곘", "곘", "곘", "곘", "곘" },
  { "곙", "곙", "곙", "곙", "곙" },
  { "곚", "곚", "곚", "곚", "곚" },
  { "곛", "곛", "곛", "곛", "곛" },
  { "곜", "곜", "곜", "곜", "곜" },
  { "곝", "곝", "곝", "곝", "곝" },
  { "곞", "곞", "곞", "곞", "곞" },
  { "곟", "곟", "곟", "곟", "곟" },
  { "고", "고", "고", "고", "고" },
  { "곡", "곡", "곡", "곡", "곡" },
  { "곢", "곢", "곢", "곢", "곢" },
  { "곣", "곣", "곣", "곣", "곣" },
  { "곤", "곤", "곤", "곤", "곤" },
  { "곥", "곥", "곥", "곥", "곥" },
  { "곦", "곦", "곦", "곦", "곦" },
  { "곧", "곧", "곧", "곧", "곧" },
  { "골", "골", "골", "골", "골" },
  { "곩", "곩", "곩", "곩", "곩" },
  { "곪", "곪", "곪", "곪", "곪" },
  { "곫", "곫", "곫", "곫", "곫" },
  { "곬", "곬", "곬", "곬", "곬" },
  { "곭", "곭", "곭", "곭", "곭" },
  { "곮", "곮", "곮", "곮", "곮" },
  { "곯", "곯", "곯", "곯", "곯" },
  { "곰", "곰", "곰", "곰", "곰" },
  { "곱", "곱", "곱", "곱", "곱" },
  { "곲", "곲", "곲", "곲", "곲" },
  { "곳", "곳", "곳", "곳", "곳" },
  { "곴", "곴", "곴", "곴", "곴" },
  { "공", "공", "공", "공", "공" },
  { "곶", "곶", "곶", "곶", "곶" },
  { "곷", "곷", "곷", "곷", "곷" },
  { "곸", "곸", "곸", "곸", "곸" },
  { "곹", "곹", "곹", "곹", "곹" },
  { "곺", "곺", "곺", "곺", "곺" },
  { "곻", "곻", "곻", "곻", "곻" },
  { "과", "과", "과", "과", "과" },
  { "곽", "곽", "곽", "곽", "곽" },
  { "곾", "곾", "곾", "곾", "곾" },
  { "곿", "곿", "곿", "곿", "곿" },
  { "관", "관", "관", "관", "관" },
  { "괁", "괁", "괁", "괁", "괁" },
  { "괂", "괂", "괂", "괂", "괂" },
  { "괃", "괃", "괃", "괃", "괃" },
  { "괄", "괄", "괄", "괄", "괄" },
  { "괅", "괅", "괅", "괅", "괅" },
  { "괆", "괆", "괆", "괆", "괆" },
  { "괇", "괇", "괇", "괇", "괇" },
  { "괈", "괈", "괈", "괈", "괈" },
  { "괉", "괉", "괉", "괉", "괉" },
  { "괊", "괊", "괊", "괊", "괊" },
  { "괋", "괋", "괋", "괋", "괋" },
  { "괌", "괌", "괌", "괌", "괌" },
  { "괍", "괍", "괍", "괍", "괍" },
  { "괎", "괎", "괎", "괎", "괎" },
  { "괏", "괏", "괏", "괏", "괏" },
  { "괐", "괐", "괐", "괐", "괐" },
  { "광", "광", "광", "광", "광" },
  { "괒", "괒", "괒", "괒", "괒" },
  { "괓", "괓", "괓", "괓", "괓" },
  { "괔", "괔", "괔", "괔", "괔" },
  { "괕", "괕", "괕", "괕", "괕" },
  { "괖", "괖", "괖", "괖", "괖" },
  { "괗", "괗", "괗", "괗", "괗" },
  { "괘", "괘", "괘", "괘", "괘" },
  { "괙", "괙", "괙", "괙", "괙" },
  { "괚", "괚", "괚", "괚", "괚" },
  { "괛", "괛", "괛", "괛", "괛" },
  { "괜", "괜", "괜", "괜", "괜" },
  { "괝", "괝", "괝", "괝", "괝" },
  { "괞", "괞", "괞", "괞", "괞" },
  { "괟", "괟", "괟", "괟", "괟" },
  { "괠", "괠", "괠", "괠", "괠" },
  { "괡", "괡", "괡", "괡", "괡" },
  { "괢", "괢", "괢", "괢", "괢" },
  { "괣", "괣", "괣", "괣", "괣" },
  { "괤", "괤", "괤", "괤", "괤" },
  { "괥", "괥", "괥", "괥", "괥" },
  { "괦", "괦", "괦", "괦", "괦" },
  { "괧", "괧", "괧", "괧", "괧" },
  { "괨", "괨", "괨", "괨", "괨" },
  { "괩", "괩", "괩", "괩", "괩" },
  { "괪", "괪", "괪", "괪", "괪" },
  { "괫", "괫", "괫", "괫", "괫" },
  { "괬", "괬", "괬", "괬", "괬" },
  { "괭", "괭", "괭", "괭", "괭" },
  { "괮", "괮", "괮", "괮", "괮" },
  { "괯", "괯", "괯", "괯", "괯" },
  { "괰", "괰", "괰", "괰", "괰" },
  { "괱", "괱", "괱", "괱", "괱" },
  { "괲", "괲", "괲", "괲", "괲" },
  { "괳", "괳", "괳", "괳", "괳" },
  { "괴", "괴", "괴", "괴", "괴" },
  { "괵", "괵", "괵", "괵", "괵" },
  { "괶", "괶", "괶", "괶", "괶" },
  { "괷", "괷", "괷", "괷", "괷" },
  { "괸", "괸", "괸", "괸", "괸" },
  { "괹", "괹", "괹", "괹", "괹" },
  { "괺", "괺", "괺", "괺", "괺" },
  { "괻", "괻", "괻", "괻", "괻" },
  { "괼", "괼", "괼", "괼", "괼" },
  { "괽", "괽", "괽", "괽", "괽" },
  { "괾", "괾", "괾", "괾", "괾" },
  { "괿", "괿", "괿", "괿", "괿" },
  { "굀", "굀", "굀", "굀", "굀" },
  { "굁", "굁", "굁", "굁", "굁" },
  { "굂", "굂", "굂", "굂", "굂" },
  { "굃", "굃", "굃", "굃", "굃" },
  { "굄", "굄", "굄", "굄", "굄" },
  { "굅", "굅", "굅", "굅", "굅" },
  { "굆", "굆", "굆", "굆", "굆" },
  { "굇", "굇", "굇", "굇", "굇" },
  { "굈", "굈", "굈", "굈", "굈" },
  { "굉", "굉", "굉", "굉", "굉" },
  { "굊", "굊", "굊", "굊", "굊" },
  { "굋", "굋", "굋", "굋", "굋" },
  { "굌", "굌", "굌", "굌", "굌" },
  { "굍", "굍", "굍", "굍", "굍" },
  { "굎", "굎", "굎", "굎", "굎" },
  { "굏", "굏", "굏", "굏", "굏" },
  { "교", "교", "교", "교", "교" },
  { "굑", "굑", "굑", "굑", "굑" },
  { "굒", "굒", "굒", "굒", "굒" },
  { "굓", "굓", "굓", "굓", "굓" },
  { "굔", "굔", "굔", "굔", "굔" },
  { "굕", "굕", "굕", "굕", "굕" },
  { "굖", "굖", "굖", "굖", "굖" },
  { "굗", "굗", "굗", "굗", "굗" },
  { "굘", "굘", "굘", "굘", "굘" },
  { "굙", "굙", "굙", "굙", "굙" },
  { "굚", "굚", "굚", "굚", "굚" },
  { "굛", "굛", "굛", "굛", "굛" },
  { "굜", "굜", "굜", "굜", "굜" },
  { "굝", "굝", "굝", "굝", "굝" },
  { "굞", "굞", "굞", "굞", "굞" },
  { "굟", "굟", "굟", "굟", "굟" },
  { "굠", "굠", "굠", "굠", "굠" },
  { "굡", "굡", "굡", "굡", "굡" },
  { "굢", "굢", "굢", "굢", "굢" },
  { "굣", "굣", "굣", "굣", "굣" },
  { "굤", "굤", "굤", "굤", "굤" },
  { "굥", "굥", "굥", "굥", "굥" },
  { "굦", "굦", "굦", "굦", "굦" },
  { "굧", "굧", "굧", "굧", "굧" },
  { "굨", "굨", "굨", "굨", "굨" },
  { "굩", "굩", "굩", "굩", "굩" },
  { "굪", "굪", "굪", "굪", "굪" },
  { "굫", "굫", "굫", "굫", "굫" },
  { "구", "구", "구", "구", "구" },
  { "국", "국", "국", "국", "국" },
  { "굮", "굮", "굮", "굮", "굮" },
  { "굯", "굯", "굯", "굯", "굯" },
  { "군", "군", "군", "군", "군" },
  { "굱", "굱", "굱", "굱", "굱" },
  { "굲", "굲", "굲", "굲", "굲" },
  { "굳", "굳", "굳", "굳", "굳" },
  { "굴", "굴", "굴", "굴", "굴" },
  { "굵", "굵", "굵", "굵", "굵" },
  { "굶", "굶", "굶", "굶", "굶" },
  { "굷", "굷", "굷", "굷", "굷" },
  { "굸", "굸", "굸", "굸", "굸" },
  { "굹", "굹", "굹", "굹", "굹" },
  { "굺", "굺", "굺", "굺", "굺" },
  { "굻", "굻", "굻", "굻", "굻" },
  { "굼", "굼", "굼", "굼", "굼" },
  { "굽", "굽", "굽", "굽", "굽" },
  { "굾", "굾", "굾", "굾", "굾" },
  { "굿", "굿", "굿", "굿", "굿" },
  { "궀", "궀", "궀", "궀", "궀" },
  { "궁", "궁", "궁", "궁", "궁" },
  { "궂", "궂", "궂", "궂", "궂" },
  { "궃", "궃", "궃", "궃", "궃" },
  { "궄", "궄", "궄", "궄", "궄" },
  { "궅", "궅", "궅", "궅", "궅" },
  { "궆", "궆", "궆", "궆", "궆" },
  { "궇", "궇", "궇", "궇", "궇" },
  { "궈", "궈", "궈", "궈", "궈" },
  { "궉", "궉", "궉", "궉", "궉" },
  { "궊", "궊", "궊", "궊", "궊" },
  { "궋", "궋", "궋", "궋", "궋" },
  { "권", "권", "권", "권", "권" },
  { "궍", "궍", "궍", "궍", "궍" },
  { "궎", "궎", "궎", "궎", "궎" },
  { "궏", "궏", "궏", "궏", "궏" },
  { "궐", "궐", "궐", "궐", "궐" },
  { "궑", "궑", "궑", "궑", "궑" },
  { "궒", "궒", "궒", "궒", "궒" },
  { "궓", "궓", "궓", "궓", "궓" },
  { "궔", "궔", "궔", "궔", "궔" },
  { "궕", "궕", "궕", "궕", "궕" },
  { "궖", "궖", "궖", "궖", "궖" },
  { "궗", "궗", "궗", "궗", "궗" },
  { "궘", "궘", "궘", "궘", "궘" },
  { "궙", "궙", "궙", "궙", "궙" },
  { "궚", "궚", "궚", "궚", "궚" },
  { "궛", "궛", "궛", "궛", "궛" },
  { "궜", "궜", "궜", "궜", "궜" },
  { "궝", "궝", "궝", "궝", "궝" },
  { "궞", "궞", "궞", "궞", "궞" },
  { "궟", "궟", "궟", "궟", "궟" },
  { "궠", "궠", "궠", "궠", "궠" },
  { "궡", "궡", "궡", "궡", "궡" },
  { "궢", "궢", "궢", "궢", "궢" },
  { "궣", "궣", "궣", "궣", "궣" },
  { "궤", "궤", "궤", "궤", "궤" },
  { "궥", "궥", "궥", "궥", "궥" },
  { "궦", "궦", "궦", "궦", "궦" },
  { "궧", "궧", "궧", "궧", "궧" },
  { "궨", "궨", "궨", "궨", "궨" },
  { "궩", "궩", "궩", "궩", "궩" },
  { "궪", "궪", "궪", "궪", "궪" },
  { "궫", "궫", "궫", "궫", "궫" },
  { "궬", "궬", "궬", "궬", "궬" },
  { "궭", "궭", "궭", "궭", "궭" },
  { "궮", "궮", "궮", "궮", "궮" },
  { "궯", "궯", "궯", "궯", "궯" },
  { "궰", "궰", "궰", "궰", "궰" },
  { "궱", "궱", "궱", "궱", "궱" },
  { "궲", "궲", "궲", "궲", "궲" },
  { "궳", "궳", "궳", "궳", "궳" },
  { "궴", "궴", "궴", "궴", "궴" },
  { "궵", "궵", "궵", "궵", "궵" },
  { "궶", "궶", "궶", "궶", "궶" },
  { "궷", "궷", "궷", "궷", "궷" },
  { "궸", "궸", "궸", "궸", "궸" },
  { "궹", "궹", "궹", "궹", "궹" },
  { "궺", "궺", "궺", "궺", "궺" },
  { "궻", "궻", "궻", "궻", "궻" },
  { "궼", "궼", "궼", "궼", "궼" },
  { "궽", "궽", "궽", "궽", "궽" },
  { "궾", "궾", "궾", "궾", "궾" },
  { "궿", "궿", "궿", "궿", "궿" },
  { "귀", "귀", "귀", "귀", "귀" },
  { "귁", "귁", "귁", "귁", "귁" },
  { "귂", "귂", "귂", "귂", "귂" },
  { "귃", "귃", "귃", "귃", "귃" },
  { "귄", "귄", "귄", "귄", "귄" },
  { "귅", "귅", "귅", "귅", "귅" },
  { "귆", "귆", "귆", "귆", "귆" },
  { "귇", "귇", "귇", "귇", "귇" },
  { "귈", "귈", "귈", "귈", "귈" },
  { "귉", "귉", "귉", "귉", "귉" },
  { "귊", "귊", "귊", "귊", "귊" },
  { "귋", "귋", "귋", "귋", "귋" },
  { "귌", "귌", "귌", "귌", "귌" },
  { "귍", "귍", "귍", "귍", "귍" },
  { "귎", "귎", "귎", "귎", "귎" },
  { "귏", "귏", "귏", "귏", "귏" },
  { "귐", "귐", "귐", "귐", "귐" },
  { "귑", "귑", "귑", "귑", "귑" },
  { "귒", "귒", "귒", "귒", "귒" },
  { "귓", "귓", "귓", "귓", "귓" },
  { "귔", "귔", "귔", "귔", "귔" },
  { "귕", "귕", "귕", "귕", "귕" },
  { "귖", "귖", "귖", "귖", "귖" },
  { "귗", "귗", "귗", "귗", "귗" },
  { "귘", "귘", "귘", "귘", "귘" },
  { "귙", "귙", "귙", "귙", "귙" },
  { "귚", "귚", "귚", "귚", "귚" },
  { "귛", "귛", "귛", "귛", "귛" },
  { "규", "규", "규", "규", "규" },
  { "귝", "귝", "귝", "귝", "귝" },
  { "귞", "귞", "귞", "귞", "귞" },
  { "귟", "귟", "귟", "귟", "귟" },
  { "균", "균", "균", "균", "균" },
  { "귡", "귡", "귡", "귡", "귡" },
  { "귢", "귢", "귢", "귢", "귢" },
  { "귣", "귣", "귣", "귣", "귣" },
  { "귤", "귤", "귤", "귤", "귤" },
  { "귥", "귥", "귥", "귥", "귥" },
  { "귦", "귦", "귦", "귦", "귦" },
  { "귧", "귧", "귧", "귧", "귧" },
  { "귨", "귨", "귨", "귨", "귨" },
  { "귩", "귩", "귩", "귩", "귩" },
  { "귪", "귪", "귪", "귪", "귪" },
  { "귫", "귫", "귫", "귫", "귫" },
  { "귬", "귬", "귬", "귬", "귬" },
  { "귭", "귭", "귭", "귭", "귭" },
  { "귮", "귮", "귮", "귮", "귮" },
  { "귯", "귯", "귯", "귯", "귯" },
  { "귰", "귰", "귰", "귰", "귰" },
  { "귱", "귱", "귱", "귱", "귱" },
  { "귲", "귲", "귲", "귲", "귲" },
  { "귳", "귳", "귳", "귳", "귳" },
  { "귴", "귴", "귴", "귴", "귴" },
  { "귵", "귵", "귵", "귵", "귵" },
  { "귶", "귶", "귶", "귶", "귶" },
  { "귷", "귷", "귷", "귷", "귷" },
  { "그", "그", "그", "그", "그" },
  { "극", "극", "극", "극", "극" },
  { "귺", "귺", "귺", "귺", "귺" },
  { "귻", "귻", "귻", "귻", "귻" },
  { "근", "근", "근", "근", "근" },
  { "귽", "귽", "귽", "귽", "귽" },
  { "귾", "귾", "귾", "귾", "귾" },
  { "귿", "귿", "귿", "귿", "귿" },
  { "글", "글", "글", "글", "글" },
  { "긁", "긁", "긁", "긁", "긁" },
  { "긂", "긂", "긂", "긂", "긂" },
  { "긃", "긃", "긃", "긃", "긃" },
  { "긄", "긄", "긄", "긄", "긄" },
  { "긅", "긅", "긅", "긅", "긅" },
  { "긆", "긆", "긆", "긆", "긆" },
  { "긇", "긇", "긇", "긇", "긇" },
  { "금", "금", "금", "금", "금" },
  { "급", "급", "급", "급", "급" },
  { "긊", "긊", "긊", "긊", "긊" },
  { "긋", "긋", "긋", "긋", "긋" },
  { "긌", "긌", "긌", "긌", "긌" },
  { "긍", "긍", "긍", "긍", "긍" },
  { "긎", "긎", "긎", "긎", "긎" },
  { "긏", "긏", "긏", "긏", "긏" },
  { "긐", "긐", "긐", "긐", "긐" },
  { "긑", "긑", "긑", "긑", "긑" },
  { "긒", "긒", "긒", "긒", "긒" },
  { "긓", "긓", "긓", "긓", "긓" },
  { "긔", "긔", "긔", "긔", "긔" },
  { "긕", "긕", "긕", "긕", "긕" },
  { "긖", "긖", "긖", "긖", "긖" },
  { "긗", "긗", "긗", "긗", "긗" },
  { "긘", "긘", "긘", "긘", "긘" },
  { "긙", "긙", "긙", "긙", "긙" },
  { "긚", "긚", "긚", "긚", "긚" },
  { "긛", "긛", "긛", "긛", "긛" },
  { "긜", "긜", "긜", "긜", "긜" },
  { "긝", "긝", "긝", "긝", "긝" },
  { "긞", "긞", "긞", "긞", "긞" },
  { "긟", "긟", "긟", "긟", "긟" },
  { "긠", "긠", "긠", "긠", "긠" },
  { "긡", "긡", "긡", "긡", "긡" },
  { "긢", "긢", "긢", "긢", "긢" },
  { "긣", "긣", "긣", "긣", "긣" },
  { "긤", "긤", "긤", "긤", "긤" },
  { "긥", "긥", "긥", "긥", "긥" },
  { "긦", "긦", "긦", "긦", "긦" },
  { "긧", "긧", "긧", "긧", "긧" },
  { "긨", "긨", "긨", "긨", "긨" },
  { "긩", "긩", "긩", "긩", "긩" },
  { "긪", "긪", "긪", "긪", "긪" },
  { "긫", "긫", "긫", "긫", "긫" },
  { "긬", "긬", "긬", "긬", "긬" },
  { "긭", "긭", "긭", "긭", "긭" },
  { "긮", "긮", "긮", "긮", "긮" },
  { "긯", "긯", "긯", "긯", "긯" },
  { "기", "기", "기", "기", "기" },
  { "긱", "긱", "긱", "긱", "긱" },
  { "긲", "긲", "긲", "긲", "긲" },
  { "긳", "긳", "긳", "긳", "긳" },
  { "긴", "긴", "긴", "긴", "긴" },
  { "긵", "긵", "긵", "긵", "긵" },
  { "긶", "긶", "긶", "긶", "긶" },
  { "긷", "긷", "긷", "긷", "긷" },
  { "길", "길", "길", "길", "길" },
  { "긹", "긹", "긹", "긹", "긹" },
  { "긺", "긺", "긺", "긺", "긺" },
  { "긻", "긻", "긻", "긻", "긻" },
  { "긼", "긼", "긼", "긼", "긼" },
  { "긽", "긽", "긽", "긽", "긽" },
  { "긾", "긾", "긾", "긾", "긾" },
  { "긿", "긿", "긿", "긿", "긿" },
  { "김", "김", "김", "김", "김" },
  { "깁", "깁", "깁", "깁", "깁" },
  { "깂", "깂", "깂", "깂", "깂" },
  { "깃", "깃", "깃", "깃", "깃" },
  { "깄", "깄", "깄", "깄", "깄" },
  { "깅", "깅", "깅", "깅", "깅" },
  { "깆", "깆", "깆", "깆", "깆" },
  { "깇", "깇", "깇", "깇", "깇" },
  { "깈", "깈", "깈", "깈", "깈" },
  { "깉", "깉", "깉", "깉", "깉" },
  { "깊", "깊", "깊", "깊", "깊" },
  { "깋", "깋", "깋", "깋", "깋" },
  { "까", "까", "까", "까", "까" },
  { "깍", "깍", "깍", "깍", "깍" },
  { "깎", "깎", "깎", "깎", "깎" },
  { "깏", "깏", "깏", "깏", "깏" },
  { "깐", "깐", "깐", "깐", "깐" },
  { "깑", "깑", "깑", "깑", "깑" },
  { "깒", "깒", "깒", "깒", "깒" },
  { "깓", "깓", "깓", "깓", "깓" },
  { "깔", "깔", "깔", "깔", "깔" },
  { "깕", "깕", "깕", "깕", "깕" },
  { "깖", "깖", "깖", "깖", "깖" },
  { "깗", "깗", "깗", "깗", "깗" },
  { "깘", "깘", "깘", "깘", "깘" },
  { "깙", "깙", "깙", "깙", "깙" },
  { "깚", "깚", "깚", "깚", "깚" },
  { "깛", "깛", "깛", "깛", "깛" },
  { "깜", "깜", "깜", "깜", "깜" },
  { "깝", "깝", "깝", "깝", "깝" },
  { "깞", "깞", "깞", "깞", "깞" },
  { "깟", "깟", "깟", "깟", "깟" },
  { "깠", "깠", "깠", "깠", "깠" },
  { "깡", "깡", "깡", "깡", "깡" },
  { "깢", "깢", "깢", "깢", "깢" },
  { "깣", "깣", "깣", "깣", "깣" },
  { "깤", "깤", "깤", "깤", "깤" },
  { "깥", "깥", "깥", "깥", "깥" },
  { "깦", "깦", "깦", "깦", "깦" },
  { "깧", "깧", "깧", "깧", "깧" },
  { "깨", "깨", "깨", "깨", "깨" },
  { "깩", "깩", "깩", "깩", "깩" },
  { "깪", "깪", "깪", "깪", "깪" },
  { "깫", "깫", "깫", "깫", "깫" },
  { "깬", "깬", "깬", "깬", "깬" },
  { "깭", "깭", "깭", "깭", "깭" },
  { "깮", "깮", "깮", "깮", "깮" },
  { "깯", "깯", "깯", "깯", "깯" },
  { "깰", "깰", "깰", "깰", "깰" },
  { "깱", "깱", "깱", "깱", "깱" },
  { "깲", "깲", "깲", "깲", "깲" },
  { "깳", "깳", "깳", "깳", "깳" },
  { "깴", "깴", "깴", "깴", "깴" },
  { "깵", "깵", "깵", "깵", "깵" },
  { "깶", "깶", "깶", "깶", "깶" },
  { "깷", "깷", "깷", "깷", "깷" },
  { "깸", "깸", "깸", "깸", "깸" },
  { "깹", "깹", "깹", "깹", "깹" },
  { "깺", "깺", "깺", "깺", "깺" },
  { "깻", "깻", "깻", "깻", "깻" },
  { "깼", "깼", "깼", "깼", "깼" },
  { "깽", "깽", "깽", "깽", "깽" },
  { "깾", "깾", "깾", "깾", "깾" },
  { "깿", "깿", "깿", "깿", "깿" },
  { "꺀", "꺀", "꺀", "꺀", "꺀" },
  { "꺁", "꺁", "꺁", "꺁", "꺁" },
  { "꺂", "꺂", "꺂", "꺂", "꺂" },
  { "꺃", "꺃", "꺃", "꺃", "꺃" },
  { "꺄", "꺄", "꺄", "꺄", "꺄" },
  { "꺅", "꺅", "꺅", "꺅", "꺅" },
  { "꺆", "꺆", "꺆", "꺆", "꺆" },
  { "꺇", "꺇", "꺇", "꺇", "꺇" },
  { "꺈", "꺈", "꺈", "꺈", "꺈" },
  { "꺉", "꺉", "꺉", "꺉", "꺉" },
  { "꺊", "꺊", "꺊", "꺊", "꺊" },
  { "꺋", "꺋", "꺋", "꺋", "꺋" },
  { "꺌", "꺌", "꺌", "꺌", "꺌" },
  { "꺍", "꺍", "꺍", "꺍", "꺍" },
  { "꺎", "꺎", "꺎", "꺎", "꺎" },
  { "꺏", "꺏", "꺏", "꺏", "꺏" },
  { "꺐", "꺐", "꺐", "꺐", "꺐" },
  { "꺑", "꺑", "꺑", "꺑", "꺑" },
  { "꺒", "꺒", "꺒", "꺒", "꺒" },
  { "꺓", "꺓", "꺓", "꺓", "꺓" },
  { "꺔", "꺔", "꺔", "꺔", "꺔" },
  { "꺕", "꺕", "꺕", "꺕", "꺕" },
  { "꺖", "꺖", "꺖", "꺖", "꺖" },
  { "꺗", "꺗", "꺗", "꺗", "꺗" },
  { "꺘", "꺘", "꺘", "꺘", "꺘" },
  { "꺙", "꺙", "꺙", "꺙", "꺙" },
  { "꺚", "꺚", "꺚", "꺚", "꺚" },
  { "꺛", "꺛", "꺛", "꺛", "꺛" },
  { "꺜", "꺜", "꺜", "꺜", "꺜" },
  { "꺝", "꺝", "꺝", "꺝", "꺝" },
  { "꺞", "꺞", "꺞", "꺞", "꺞" },
  { "꺟", "꺟", "꺟", "꺟", "꺟" },
  { "꺠", "꺠", "꺠", "꺠", "꺠" },
  { "꺡", "꺡", "꺡", "꺡", "꺡" },
  { "꺢", "꺢", "꺢", "꺢", "꺢" },
  { "꺣", "꺣", "꺣", "꺣", "꺣" },
  { "꺤", "꺤", "꺤", "꺤", "꺤" },
  { "꺥", "꺥", "꺥", "꺥", "꺥" },
  { "꺦", "꺦", "꺦", "꺦", "꺦" },
  { "꺧", "꺧", "꺧", "꺧", "꺧" },
  { "꺨", "꺨", "꺨", "꺨", "꺨" },
  { "꺩", "꺩", "꺩", "꺩", "꺩" },
  { "꺪", "꺪", "꺪", "꺪", "꺪" },
  { "꺫", "꺫", "꺫", "꺫", "꺫" },
  { "꺬", "꺬", "꺬", "꺬", "꺬" },
  { "꺭", "꺭", "꺭", "꺭", "꺭" },
  { "꺮", "꺮", "꺮", "꺮", "꺮" },
  { "꺯", "꺯", "꺯", "꺯", "꺯" },
  { "꺰", "꺰", "꺰", "꺰", "꺰" },
  { "꺱", "꺱", "꺱", "꺱", "꺱" },
  { "꺲", "꺲", "꺲", "꺲", "꺲" },
  { "꺳", "꺳", "꺳", "꺳", "꺳" },
  { "꺴", "꺴", "꺴", "꺴", "꺴" },
  { "꺵", "꺵", "꺵", "꺵", "꺵" },
  { "꺶", "꺶", "꺶", "꺶", "꺶" },
  { "꺷", "꺷", "꺷", "꺷", "꺷" },
  { "꺸", "꺸", "꺸", "꺸", "꺸" },
  { "꺹", "꺹", "꺹", "꺹", "꺹" },
  { "꺺", "꺺", "꺺", "꺺", "꺺" },
  { "꺻", "꺻", "꺻", "꺻", "꺻" },
  { "꺼", "꺼", "꺼", "꺼", "꺼" },
  { "꺽", "꺽", "꺽", "꺽", "꺽" },
  { "꺾", "꺾", "꺾", "꺾", "꺾" },
  { "꺿", "꺿", "꺿", "꺿", "꺿" },
  { "껀", "껀", "껀", "껀", "껀" },
  { "껁", "껁", "껁", "껁", "껁" },
  { "껂", "껂", "껂", "껂", "껂" },
  { "껃", "껃", "껃", "껃", "껃" },
  { "껄", "껄", "껄", "껄", "껄" },
  { "껅", "껅", "껅", "껅", "껅" },
  { "껆", "껆", "껆", "껆", "껆" },
  { "껇", "껇", "껇", "껇", "껇" },
  { "껈", "껈", "껈", "껈", "껈" },
  { "껉", "껉", "껉", "껉", "껉" },
  { "껊", "껊", "껊", "껊", "껊" },
  { "껋", "껋", "껋", "껋", "껋" },
  { "껌", "껌", "껌", "껌", "껌" },
  { "껍", "껍", "껍", "껍", "껍" },
  { "껎", "껎", "껎", "껎", "껎" },
  { "껏", "껏", "껏", "껏", "껏" },
  { "껐", "껐", "껐", "껐", "껐" },
  { "껑", "껑", "껑", "껑", "껑" },
  { "껒", "껒", "껒", "껒", "껒" },
  { "껓", "껓", "껓", "껓", "껓" },
  { "껔", "껔", "껔", "껔", "껔" },
  { "껕", "껕", "껕", "껕", "껕" },
  { "껖", "껖", "껖", "껖", "껖" },
  { "껗", "껗", "껗", "껗", "껗" },
  { "께", "께", "께", "께", "께" },
  { "껙", "껙", "껙", "껙", "껙" },
  { "껚", "껚", "껚", "껚", "껚" },
  { "껛", "껛", "껛", "껛", "껛" },
  { "껜", "껜", "껜", "껜", "껜" },
  { "껝", "껝", "껝", "껝", "껝" },
  { "껞", "껞", "껞", "껞", "껞" },
  { "껟", "껟", "껟", "껟", "껟" },
  { "껠", "껠", "껠", "껠", "껠" },
  { "껡", "껡", "껡", "껡", "껡" },
  { "껢", "껢", "껢", "껢", "껢" },
  { "껣", "껣", "껣", "껣", "껣" },
  { "껤", "껤", "껤", "껤", "껤" },
  { "껥", "껥", "껥", "껥", "껥" },
  { "껦", "껦", "껦", "껦", "껦" },
  { "껧", "껧", "껧", "껧", "껧" },
  { "껨", "껨", "껨", "껨", "껨" },
  { "껩", "껩", "껩", "껩", "껩" },
  { "껪", "껪", "껪", "껪", "껪" },
  { "껫", "껫", "껫", "껫", "껫" },
  { "껬", "껬", "껬", "껬", "껬" },
  { "껭", "껭", "껭", "껭", "껭" },
  { "껮", "껮", "껮", "껮", "껮" },
  { "껯", "껯", "껯", "껯", "껯" },
  { "껰", "껰", "껰", "껰", "껰" },
  { "껱", "껱", "껱", "껱", "껱" },
  { "껲", "껲", "껲", "껲", "껲" },
  { "껳", "껳", "껳", "껳", "껳" },
  { "껴", "껴", "껴", "껴", "껴" },
  { "껵", "껵", "껵", "껵", "껵" },
  { "껶", "껶", "껶", "껶", "껶" },
  { "껷", "껷", "껷", "껷", "껷" },
  { "껸", "껸", "껸", "껸", "껸" },
  { "껹", "껹", "껹", "껹", "껹" },
  { "껺", "껺", "껺", "껺", "껺" },
  { "껻", "껻", "껻", "껻", "껻" },
  { "껼", "껼", "껼", "껼", "껼" },
  { "껽", "껽", "껽", "껽", "껽" },
  { "껾", "껾", "껾", "껾", "껾" },
  { "껿", "껿", "껿", "껿", "껿" },
  { "꼀", "꼀", "꼀", "꼀", "꼀" },
  { "꼁", "꼁", "꼁", "꼁", "꼁" },
  { "꼂", "꼂", "꼂", "꼂", "꼂" },
  { "꼃", "꼃", "꼃", "꼃", "꼃" },
  { "꼄", "꼄", "꼄", "꼄", "꼄" },
  { "꼅", "꼅", "꼅", "꼅", "꼅" },
  { "꼆", "꼆", "꼆", "꼆", "꼆" },
  { "꼇", "꼇", "꼇", "꼇", "꼇" },
  { "꼈", "꼈", "꼈", "꼈", "꼈" },
  { "꼉", "꼉", "꼉", "꼉", "꼉" },
  { "꼊", "꼊", "꼊", "꼊", "꼊" },
  { "꼋", "꼋", "꼋", "꼋", "꼋" },
  { "꼌", "꼌", "꼌", "꼌", "꼌" },
  { "꼍", "꼍", "꼍", "꼍", "꼍" },
  { "꼎", "꼎", "꼎", "꼎", "꼎" },
  { "꼏", "꼏", "꼏", "꼏", "꼏" },
  { "꼐", "꼐", "꼐", "꼐", "꼐" },
  { "꼑", "꼑", "꼑", "꼑", "꼑" },
  { "꼒", "꼒", "꼒", "꼒", "꼒" },
  { "꼓", "꼓", "꼓", "꼓", "꼓" },
  { "꼔", "꼔", "꼔", "꼔", "꼔" },
  { "꼕", "꼕", "꼕", "꼕", "꼕" },
  { "꼖", "꼖", "꼖", "꼖", "꼖" },
  { "꼗", "꼗", "꼗", "꼗", "꼗" },
  { "꼘", "꼘", "꼘", "꼘", "꼘" },
  { "꼙", "꼙", "꼙", "꼙", "꼙" },
  { "꼚", "꼚", "꼚", "꼚", "꼚" },
  { "꼛", "꼛", "꼛", "꼛", "꼛" },
  { "꼜", "꼜", "꼜", "꼜", "꼜" },
  { "꼝", "꼝", "꼝", "꼝", "꼝" },
  { "꼞", "꼞", "꼞", "꼞", "꼞" },
  { "꼟", "꼟", "꼟", "꼟", "꼟" },
  { "꼠", "꼠", "꼠", "꼠", "꼠" },
  { "꼡", "꼡", "꼡", "꼡", "꼡" },
  { "꼢", "꼢", "꼢", "꼢", "꼢" },
  { "꼣", "꼣", "꼣", "꼣", "꼣" },
  { "꼤", "꼤", "꼤", "꼤", "꼤" },
  { "꼥", "꼥", "꼥", "꼥", "꼥" },
  { "꼦", "꼦", "꼦", "꼦", "꼦" },
  { "꼧", "꼧", "꼧", "꼧", "꼧" },
  { "꼨", "꼨", "꼨", "꼨", "꼨" },
  { "꼩", "꼩", "꼩", "꼩", "꼩" },
  { "꼪", "꼪", "꼪", "꼪", "꼪" },
  { "꼫", "꼫", "꼫", "꼫", "꼫" },
  { "꼬", "꼬", "꼬", "꼬", "꼬" },
  { "꼭", "꼭", "꼭", "꼭", "꼭" },
  { "꼮", "꼮", "꼮", "꼮", "꼮" },
  { "꼯", "꼯", "꼯", "꼯", "꼯" },
  { "꼰", "꼰", "꼰", "꼰", "꼰" },
  { "꼱", "꼱", "꼱", "꼱", "꼱" },
  { "꼲", "꼲", "꼲", "꼲", "꼲" },
  { "꼳", "꼳", "꼳", "꼳", "꼳" },
  { "꼴", "꼴", "꼴", "꼴", "꼴" },
  { "꼵", "꼵", "꼵", "꼵", "꼵" },
  { "꼶", "꼶", "꼶", "꼶", "꼶" },
  { "꼷", "꼷", "꼷", "꼷", "꼷" },
  { "꼸", "꼸", "꼸", "꼸", "꼸" },
  { "꼹", "꼹", "꼹", "꼹", "꼹" },
  { "꼺", "꼺", "꼺", "꼺", "꼺" },
  { "꼻", "꼻", "꼻", "꼻", "꼻" },
  { "꼼", "꼼", "꼼", "꼼", "꼼" },
  { "꼽", "꼽", "꼽", "꼽", "꼽" },
  { "꼾", "꼾", "꼾", "꼾", "꼾" },
  { "꼿", "꼿", "꼿", "꼿", "꼿" },
  { "꽀", "꽀", "꽀", "꽀", "꽀" },
  { "꽁", "꽁", "꽁", "꽁", "꽁" },
  { "꽂", "꽂", "꽂", "꽂", "꽂" },
  { "꽃", "꽃", "꽃", "꽃", "꽃" },
  { "꽄", "꽄", "꽄", "꽄", "꽄" },
  { "꽅", "꽅", "꽅", "꽅", "꽅" },
  { "꽆", "꽆", "꽆", "꽆", "꽆" },
  { "꽇", "꽇", "꽇", "꽇", "꽇" },
  { "꽈", "꽈", "꽈", "꽈", "꽈" },
  { "꽉", "꽉", "꽉", "꽉", "꽉" },
  { "꽊", "꽊", "꽊", "꽊", "꽊" },
  { "꽋", "꽋", "꽋", "꽋", "꽋" },
  { "꽌", "꽌", "꽌", "꽌", "꽌" },
  { "꽍", "꽍", "꽍", "꽍", "꽍" },
  { "꽎", "꽎", "꽎", "꽎", "꽎" },
  { "꽏", "꽏", "꽏", "꽏", "꽏" },
  { "꽐", "꽐", "꽐", "꽐", "꽐" },
  { "꽑", "꽑", "꽑", "꽑", "꽑" },
  { "꽒", "꽒", "꽒", "꽒", "꽒" },
  { "꽓", "꽓", "꽓", "꽓", "꽓" },
  { "꽔", "꽔", "꽔", "꽔", "꽔" },
  { "꽕", "꽕", "꽕", "꽕", "꽕" },
  { "꽖", "꽖", "꽖", "꽖", "꽖" },
  { "꽗", "꽗", "꽗", "꽗", "꽗" },
  { "꽘", "꽘", "꽘", "꽘", "꽘" },
  { "꽙", "꽙", "꽙", "꽙", "꽙" },
  { "꽚", "꽚", "꽚", "꽚", "꽚" },
  { "꽛", "꽛", "꽛", "꽛", "꽛" },
  { "꽜", "꽜", "꽜", "꽜", "꽜" },
  { "꽝", "꽝", "꽝", "꽝", "꽝" },
  { "꽞", "꽞", "꽞", "꽞", "꽞" },
  { "꽟", "꽟", "꽟", "꽟", "꽟" },
  { "꽠", "꽠", "꽠", "꽠", "꽠" },
  { "꽡", "꽡", "꽡", "꽡", "꽡" },
  { "꽢", "꽢", "꽢", "꽢", "꽢" },
  { "꽣", "꽣", "꽣", "꽣", "꽣" },
  { "꽤", "꽤", "꽤", "꽤", "꽤" },
  { "꽥", "꽥", "꽥", "꽥", "꽥" },
  { "꽦", "꽦", "꽦", "꽦", "꽦" },
  { "꽧", "꽧", "꽧", "꽧", "꽧" },
  { "꽨", "꽨", "꽨", "꽨", "꽨" },
  { "꽩", "꽩", "꽩", "꽩", "꽩" },
  { "꽪", "꽪", "꽪", "꽪", "꽪" },
  { "꽫", "꽫", "꽫", "꽫", "꽫" },
  { "꽬", "꽬", "꽬", "꽬", "꽬" },
  { "꽭", "꽭", "꽭", "꽭", "꽭" },
  { "꽮", "꽮", "꽮", "꽮", "꽮" },
  { "꽯", "꽯", "꽯", "꽯", "꽯" },
  { "꽰", "꽰", "꽰", "꽰", "꽰" },
  { "꽱", "꽱", "꽱", "꽱", "꽱" },
  { "꽲", "꽲", "꽲", "꽲", "꽲" },
  { "꽳", "꽳", "꽳", "꽳", "꽳" },
  { "꽴", "꽴", "꽴", "꽴", "꽴" },
  { "꽵", "꽵", "꽵", "꽵", "꽵" },
  { "꽶", "꽶", "꽶", "꽶", "꽶" },
  { "꽷", "꽷", "꽷", "꽷", "꽷" },
  { "꽸", "꽸", "꽸", "꽸", "꽸" },
  { "꽹", "꽹", "꽹", "꽹", "꽹" },
  { "꽺", "꽺", "꽺", "꽺", "꽺" },
  { "꽻", "꽻", "꽻", "꽻", "꽻" },
  { "꽼", "꽼", "꽼", "꽼", "꽼" },
  { "꽽", "꽽", "꽽", "꽽", "꽽" },
  { "꽾", "꽾", "꽾", "꽾", "꽾" },
  { "꽿", "꽿", "꽿", "꽿", "꽿" },
  { "꾀", "꾀", "꾀", "꾀", "꾀" },
  { "꾁", "꾁", "꾁", "꾁", "꾁" },
  { "꾂", "꾂", "꾂", "꾂", "꾂" },
  { "꾃", "꾃", "꾃", "꾃", "꾃" },
  { "꾄", "꾄", "꾄", "꾄", "꾄" },
  { "꾅", "꾅", "꾅", "꾅", "꾅" },
  { "꾆", "꾆", "꾆", "꾆", "꾆" },
  { "꾇", "꾇", "꾇", "꾇", "꾇" },
  { "꾈", "꾈", "꾈", "꾈", "꾈" },
  { "꾉", "꾉", "꾉", "꾉", "꾉" },
  { "꾊", "꾊", "꾊", "꾊", "꾊" },
  { "꾋", "꾋", "꾋", "꾋", "꾋" },
  { "꾌", "꾌", "꾌", "꾌", "꾌" },
  { "꾍", "꾍", "꾍", "꾍", "꾍" },
  { "꾎", "꾎", "꾎", "꾎", "꾎" },
  { "꾏", "꾏", "꾏", "꾏", "꾏" },
  { "꾐", "꾐", "꾐", "꾐", "꾐" },
  { "꾑", "꾑", "꾑", "꾑", "꾑" },
  { "꾒", "꾒", "꾒", "꾒", "꾒" },
  { "꾓", "꾓", "꾓", "꾓", "꾓" },
  { "꾔", "꾔", "꾔", "꾔", "꾔" },
  { "꾕", "꾕", "꾕", "꾕", "꾕" },
  { "꾖", "꾖", "꾖", "꾖", "꾖" },
  { "꾗", "꾗", "꾗", "꾗", "꾗" },
  { "꾘", "꾘", "꾘", "꾘", "꾘" },
  { "꾙", "꾙", "꾙", "꾙", "꾙" },
  { "꾚", "꾚", "꾚", "꾚", "꾚" },
  { "꾛", "꾛", "꾛", "꾛", "꾛" },
  { "꾜", "꾜", "꾜", "꾜", "꾜" },
  { "꾝", "꾝", "꾝", "꾝", "꾝" },
  { "꾞", "꾞", "꾞", "꾞", "꾞" },
  { "꾟", "꾟", "꾟", "꾟", "꾟" },
  { "꾠", "꾠", "꾠", "꾠", "꾠" },
  { "꾡", "꾡", "꾡", "꾡", "꾡" },
  { "꾢", "꾢", "꾢", "꾢", "꾢" },
  { "꾣", "꾣", "꾣", "꾣", "꾣" },
  { "꾤", "꾤", "꾤", "꾤", "꾤" },
  { "꾥", "꾥", "꾥", "꾥", "꾥" },
  { "꾦", "꾦", "꾦", "꾦", "꾦" },
  { "꾧", "꾧", "꾧", "꾧", "꾧" },
  { "꾨", "꾨", "꾨", "꾨", "꾨" },
  { "꾩", "꾩", "꾩", "꾩", "꾩" },
  { "꾪", "꾪", "꾪", "꾪", "꾪" },
  { "꾫", "꾫", "꾫", "꾫", "꾫" },
  { "꾬", "꾬", "꾬", "꾬", "꾬" },
  { "꾭", "꾭", "꾭", "꾭", "꾭" },
  { "꾮", "꾮", "꾮", "꾮", "꾮" },
  { "꾯", "꾯", "꾯", "꾯", "꾯" },
  { "꾰", "꾰", "꾰", "꾰", "꾰" },
  { "꾱", "꾱", "꾱", "꾱", "꾱" },
  { "꾲", "꾲", "꾲", "꾲", "꾲" },
  { "꾳", "꾳", "꾳", "꾳", "꾳" },
  { "꾴", "꾴", "꾴", "꾴", "꾴" },
  { "꾵", "꾵", "꾵", "꾵", "꾵" },
  { "꾶", "꾶", "꾶", "꾶", "꾶" },
  { "꾷", "꾷", "꾷", "꾷", "꾷" },
  { "꾸", "꾸", "꾸", "꾸", "꾸" },
  { "꾹", "꾹", "꾹", "꾹", "꾹" },
  { "꾺", "꾺", "꾺", "꾺", "꾺" },
  { "꾻", "꾻", "꾻", "꾻", "꾻" },
  { "꾼", "꾼", "꾼", "꾼", "꾼" },
  { "꾽", "꾽", "꾽", "꾽", "꾽" },
  { "꾾", "꾾", "꾾", "꾾", "꾾" },
  { "꾿", "꾿", "꾿", "꾿", "꾿" },
  { "꿀", "꿀", "꿀", "꿀", "꿀" },
  { "꿁", "꿁", "꿁", "꿁", "꿁" },
  { "꿂", "꿂", "꿂", "꿂", "꿂" },
  { "꿃", "꿃", "꿃", "꿃", "꿃" },
  { "꿄", "꿄", "꿄", "꿄", "꿄" },
  { "꿅", "꿅", "꿅", "꿅", "꿅" },
  { "꿆", "꿆", "꿆", "꿆", "꿆" },
  { "꿇", "꿇", "꿇", "꿇", "꿇" },
  { "꿈", "꿈", "꿈", "꿈", "꿈" },
  { "꿉", "꿉", "꿉", "꿉", "꿉" },
  { "꿊", "꿊", "꿊", "꿊", "꿊" },
  { "꿋", "꿋", "꿋", "꿋", "꿋" },
  { "꿌", "꿌", "꿌", "꿌", "꿌" },
  { "꿍", "꿍", "꿍", "꿍", "꿍" },
  { "꿎", "꿎", "꿎", "꿎", "꿎" },
  { "꿏", "꿏", "꿏", "꿏", "꿏" },
  { "꿐", "꿐", "꿐", "꿐", "꿐" },
  { "꿑", "꿑", "꿑", "꿑", "꿑" },
  { "꿒", "꿒", "꿒", "꿒", "꿒" },
  { "꿓", "꿓", "꿓", "꿓", "꿓" },
  { "꿔", "꿔", "꿔", "꿔", "꿔" },
  { "꿕", "꿕", "꿕", "꿕", "꿕" },
  { "꿖", "꿖", "꿖", "꿖", "꿖" },
  { "꿗", "꿗", "꿗", "꿗", "꿗" },
  { "꿘", "꿘", "꿘", "꿘", "꿘" },
  { "꿙", "꿙", "꿙", "꿙", "꿙" },
  { "꿚", "꿚", "꿚", "꿚", "꿚" },
  { "꿛", "꿛", "꿛", "꿛", "꿛" },
  { "꿜", "꿜", "꿜", "꿜", "꿜" },
  { "꿝", "꿝", "꿝", "꿝", "꿝" },
  { "꿞", "꿞", "꿞", "꿞", "꿞" },
  { "꿟", "꿟", "꿟", "꿟", "꿟" },
  { "꿠", "꿠", "꿠", "꿠", "꿠" },
  { "꿡", "꿡", "꿡", "꿡", "꿡" },
  { "꿢", "꿢", "꿢", "꿢", "꿢" },
  { "꿣", "꿣", "꿣", "꿣", "꿣" },
  { "꿤", "꿤", "꿤", "꿤", "꿤" },
  { "꿥", "꿥", "꿥", "꿥", "꿥" },
  { "꿦", "꿦", "꿦", "꿦", "꿦" },
  { "꿧", "꿧", "꿧", "꿧", "꿧" },
  { "꿨", "꿨", "꿨", "꿨", "꿨" },
  { "꿩", "꿩", "꿩", "꿩", "꿩" },
  { "꿪", "꿪", "꿪", "꿪", "꿪" },
  { "꿫", "꿫", "꿫", "꿫", "꿫" },
  { "꿬", "꿬", "꿬", "꿬", "꿬" },
  { "꿭", "꿭", "꿭", "꿭", "꿭" },
  { "꿮", "꿮", "꿮", "꿮", "꿮" },
  { "꿯", "꿯", "꿯", "꿯", "꿯" },
  { "꿰", "꿰", "꿰", "꿰", "꿰" },
  { "꿱", "꿱", "꿱", "꿱", "꿱" },
  { "꿲", "꿲", "꿲", "꿲", "꿲" },
  { "꿳", "꿳", "꿳", "꿳", "꿳" },
  { "꿴", "꿴", "꿴", "꿴", "꿴" },
  { "꿵", "꿵", "꿵", "꿵", "꿵" },
  { "꿶", "꿶", "꿶", "꿶", "꿶" },
  { "꿷", "꿷", "꿷", "꿷", "꿷" },
  { "꿸", "꿸", "꿸", "꿸", "꿸" },
  { "꿹", "꿹", "꿹", "꿹", "꿹" },
  { "꿺", "꿺", "꿺", "꿺", "꿺" },
  { "꿻", "꿻", "꿻", "꿻", "꿻" },
  { "꿼", "꿼", "꿼", "꿼", "꿼" },
  { "꿽", "꿽", "꿽", "꿽", "꿽" },
  { "꿾", "꿾", "꿾", "꿾", "꿾" },
  { "꿿", "꿿", "꿿", "꿿", "꿿" },
  { "뀀", "뀀", "뀀", "뀀", "뀀" },
  { "뀁", "뀁", "뀁", "뀁", "뀁" },
  { "뀂", "뀂", "뀂", "뀂", "뀂" },
  { "뀃", "뀃", "뀃", "뀃", "뀃" },
  { "뀄", "뀄", "뀄", "뀄", "뀄" },
  { "뀅", "뀅", "뀅", "뀅", "뀅" },
  { "뀆", "뀆", "뀆", "뀆", "뀆" },
  { "뀇", "뀇", "뀇", "뀇", "뀇" },
  { "뀈", "뀈", "뀈", "뀈", "뀈" },
  { "뀉", "뀉", "뀉", "뀉", "뀉" },
  { "뀊", "뀊", "뀊", "뀊", "뀊" },
  { "뀋", "뀋", "뀋", "뀋", "뀋" },
  { "뀌", "뀌", "뀌", "뀌", "뀌" },
  { "뀍", "뀍", "뀍", "뀍", "뀍" },
  { "뀎", "뀎", "뀎", "뀎", "뀎" },
  { "뀏", "뀏", "뀏", "뀏", "뀏" },
  { "뀐", "뀐", "뀐", "뀐", "뀐" },
  { "뀑", "뀑", "뀑", "뀑", "뀑" },
  { "뀒", "뀒", "뀒", "뀒", "뀒" },
  { "뀓", "뀓", "뀓", "뀓", "뀓" },
  { "뀔", "뀔", "뀔", "뀔", "뀔" },
  { "뀕", "뀕", "뀕", "뀕", "뀕" },
  { "뀖", "뀖", "뀖", "뀖", "뀖" },
  { "뀗", "뀗", "뀗", "뀗", "뀗" },
  { "뀘", "뀘", "뀘", "뀘", "뀘" },
  { "뀙", "뀙", "뀙", "뀙", "뀙" },
  { "뀚", "뀚", "뀚", "뀚", "뀚" },
  { "뀛", "뀛", "뀛", "뀛", "뀛" },
  { "뀜", "뀜", "뀜", "뀜", "뀜" },
  { "뀝", "뀝", "뀝", "뀝", "뀝" },
  { "뀞", "뀞", "뀞", "뀞", "뀞" },
  { "뀟", "뀟", "뀟", "뀟", "뀟" },
  { "뀠", "뀠", "뀠", "뀠", "뀠" },
  { "뀡", "뀡", "뀡", "뀡", "뀡" },
  { "뀢", "뀢", "뀢", "뀢", "뀢" },
  { "뀣", "뀣", "뀣", "뀣", "뀣" },
  { "뀤", "뀤", "뀤", "뀤", "뀤" },
  { "뀥", "뀥", "뀥", "뀥", "뀥" },
  { "뀦", "뀦", "뀦", "뀦", "뀦" },
  { "뀧", "뀧", "뀧", "뀧", "뀧" },
  { "뀨", "뀨", "뀨", "뀨", "뀨" },
  { "뀩", "뀩", "뀩", "뀩", "뀩" },
  { "뀪", "뀪", "뀪", "뀪", "뀪" },
  { "뀫", "뀫", "뀫", "뀫", "뀫" },
  { "뀬", "뀬", "뀬", "뀬", "뀬" },
  { "뀭", "뀭", "뀭", "뀭", "뀭" },
  { "뀮", "뀮", "뀮", "뀮", "뀮" },
  { "뀯", "뀯", "뀯", "뀯", "뀯" },
  { "뀰", "뀰", "뀰", "뀰", "뀰" },
  { "뀱", "뀱", "뀱", "뀱", "뀱" },
  { "뀲", "뀲", "뀲", "뀲", "뀲" },
  { "뀳", "뀳", "뀳", "뀳", "뀳" },
  { "뀴", "뀴", "뀴", "뀴", "뀴" },
  { "뀵", "뀵", "뀵", "뀵", "뀵" },
  { "뀶", "뀶", "뀶", "뀶", "뀶" },
  { "뀷", "뀷", "뀷", "뀷", "뀷" },
  { "뀸", "뀸", "뀸", "뀸", "뀸" },
  { "뀹", "뀹", "뀹", "뀹", "뀹" },
  { "뀺", "뀺", "뀺", "뀺", "뀺" },
  { "뀻", "뀻", "뀻", "뀻", "뀻" },
  { "뀼", "뀼", "뀼", "뀼", "뀼" },
  { "뀽", "뀽", "뀽", "뀽", "뀽" },
  { "뀾", "뀾", "뀾", "뀾", "뀾" },
  { "뀿", "뀿", "뀿", "뀿", "뀿" },
  { "끀", "끀", "끀", "끀", "끀" },
  { "끁", "끁", "끁", "끁", "끁" },
  { "끂", "끂", "끂", "끂", "끂" },
  { "끃", "끃", "끃", "끃", "끃" },
  { "끄", "끄", "끄", "끄", "끄" },
  { "끅", "끅", "끅", "끅", "끅" },
  { "끆", "끆", "끆", "끆", "끆" },
  { "끇", "끇", "끇", "끇", "끇" },
  { "끈", "끈", "끈", "끈", "끈" },
  { "끉", "끉", "끉", "끉", "끉" },
  { "끊", "끊", "끊", "끊", "끊" },
  { "끋", "끋", "끋", "끋", "끋" },
  { "끌", "끌", "끌", "끌", "끌" },
  { "끍", "끍", "끍", "끍", "끍" },
  { "끎", "끎", "끎", "끎", "끎" },
  { "끏", "끏", "끏", "끏", "끏" },
  { "끐", "끐", "끐", "끐", "끐" },
  { "끑", "끑", "끑", "끑", "끑" },
  { "끒", "끒", "끒", "끒", "끒" },
  { "끓", "끓", "끓", "끓", "끓" },
  { "끔", "끔", "끔", "끔", "끔" },
  { "끕", "끕", "끕", "끕", "끕" },
  { "끖", "끖", "끖", "끖", "끖" },
  { "끗", "끗", "끗", "끗", "끗" },
  { "끘", "끘", "끘", "끘", "끘" },
  { "끙", "끙", "끙", "끙", "끙" },
  { "끚", "끚", "끚", "끚", "끚" },
  { "끛", "끛", "끛", "끛", "끛" },
  { "끜", "끜", "끜", "끜", "끜" },
  { "끝", "끝", "끝", "끝", "끝" },
  { "끞", "끞", "끞", "끞", "끞" },
  { "끟", "끟", "끟", "끟", "끟" },
  { "끠", "끠", "끠", "끠", "끠" },
  { "끡", "끡", "끡", "끡", "끡" },
  { "끢", "끢", "끢", "끢", "끢" },
  { "끣", "끣", "끣", "끣", "끣" },
  { "끤", "끤", "끤", "끤", "끤" },
  { "끥", "끥", "끥", "끥", "끥" },
  { "끦", "끦", "끦", "끦", "끦" },
  { "끧", "끧", "끧", "끧", "끧" },
  { "끨", "끨", "끨", "끨", "끨" },
  { "끩", "끩", "끩", "끩", "끩" },
  { "끪", "끪", "끪", "끪", "끪" },
  { "끫", "끫", "끫", "끫", "끫" },
  { "끬", "끬", "끬", "끬", "끬" },
  { "끭", "끭", "끭", "끭", "끭" },
  { "끮", "끮", "끮", "끮", "끮" },
  { "끯", "끯", "끯", "끯", "끯" },
  { "끰", "끰", "끰", "끰", "끰" },
  { "끱", "끱", "끱", "끱", "끱" },
  { "끲", "끲", "끲", "끲", "끲" },
  { "끳", "끳", "끳", "끳", "끳" },
  { "끴", "끴", "끴", "끴", "끴" },
  { "끵", "끵", "끵", "끵", "끵" },
  { "끶", "끶", "끶", "끶", "끶" },
  { "끷", "끷", "끷", "끷", "끷" },
  { "끸", "끸", "끸", "끸", "끸" },
  { "끹", "끹", "끹", "끹", "끹" },
  { "끺", "끺", "끺", "끺", "끺" },
  { "끻", "끻", "끻", "끻", "끻" },
  { "끼", "끼", "끼", "끼", "끼" },
  { "끽", "끽", "끽", "끽", "끽" },
  { "끾", "끾", "끾", "끾", "끾" },
  { "끿", "끿", "끿", "끿", "끿" },
  { "낀", "낀", "낀", "낀", "낀" },
  { "낁", "낁", "낁", "낁", "낁" },
  { "낂", "낂", "낂", "낂", "낂" },
  { "낃", "낃", "낃", "낃", "낃" },
  { "낄", "낄", "낄", "낄", "낄" },
  { "낅", "낅", "낅", "낅", "낅" },
  { "낆", "낆", "낆", "낆", "낆" },
  { "낇", "낇", "낇", "낇", "낇" },
  { "낈", "낈", "낈", "낈", "낈" },
  { "낉", "낉", "낉", "낉", "낉" },
  { "낊", "낊", "낊", "낊", "낊" },
  { "낋", "낋", "낋", "낋", "낋" },
  { "낌", "낌", "낌", "낌", "낌" },
  { "낍", "낍", "낍", "낍", "낍" },
  { "낎", "낎", "낎", "낎", "낎" },
  { "낏", "낏", "낏", "낏", "낏" },
  { "낐", "낐", "낐", "낐", "낐" },
  { "낑", "낑", "낑", "낑", "낑" },
  { "낒", "낒", "낒", "낒", "낒" },
  { "낓", "낓", "낓", "낓", "낓" },
  { "낔", "낔", "낔", "낔", "낔" },
  { "낕", "낕", "낕", "낕", "낕" },
  { "낖", "낖", "낖", "낖", "낖" },
  { "낗", "낗", "낗", "낗", "낗" },
  { "나", "나", "나", "나", "나" },
  { "낙", "낙", "낙", "낙", "낙" },
  { "낚", "낚", "낚", "낚", "낚" },
  { "낛", "낛", "낛", "낛", "낛" },
  { "난", "난", "난", "난", "난" },
  { "낝", "낝", "낝", "낝", "낝" },
  { "낞", "낞", "낞", "낞", "낞" },
  { "낟", "낟", "낟", "낟", "낟" },
  { "날", "날", "날", "날", "날" },
  { "낡", "낡", "낡", "낡", "낡" },
  { "낢", "낢", "낢", "낢", "낢" },
  { "낣", "낣", "낣", "낣", "낣" },
  { "낤", "낤", "낤", "낤", "낤" },
  { "낥", "낥", "낥", "낥", "낥" },
  { "낦", "낦", "낦", "낦", "낦" },
  { "낧", "낧", "낧", "낧", "낧" },
  { "남", "남", "남", "남", "남" },
  { "납", "납", "납", "납", "납" },
  { "낪", "낪", "낪", "낪", "낪" },
  { "낫", "낫", "낫", "낫", "낫" },
  { "났", "났", "났", "났", "났" },
  { "낭", "낭", "낭", "낭", "낭" },
  { "낮", "낮", "낮", "낮", "낮" },
  { "낯", "낯", "낯", "낯", "낯" },
  { "낰", "낰", "낰", "낰", "낰" },
  { "낱", "낱", "낱", "낱", "낱" },
  { "낲", "낲", "낲", "낲", "낲" },
  { "낳", "낳", "낳", "낳", "낳" },
  { "내", "내", "내", "내", "내" },
  { "낵", "낵", "낵", "낵", "낵" },
  { "낶", "낶", "낶", "낶", "낶" },
  { "낷", "낷", "낷", "낷", "낷" },
  { "낸", "낸", "낸", "낸", "낸" },
  { "낹", "낹", "낹", "낹", "낹" },
  { "낺", "낺", "낺", "낺", "낺" },
  { "낻", "낻", "낻", "낻", "낻" },
  { "낼", "낼", "낼", "낼", "낼" },
  { "낽", "낽", "낽", "낽", "낽" },
  { "낾", "낾", "낾", "낾", "낾" },
  { "낿", "낿", "낿", "낿", "낿" },
  { "냀", "냀", "냀", "냀", "냀" },
  { "냁", "냁", "냁", "냁", "냁" },
  { "냂", "냂", "냂", "냂", "냂" },
  { "냃", "냃", "냃", "냃", "냃" },
  { "냄", "냄", "냄", "냄", "냄" },
  { "냅", "냅", "냅", "냅", "냅" },
  { "냆", "냆", "냆", "냆", "냆" },
  { "냇", "냇", "냇", "냇", "냇" },
  { "냈", "냈", "냈", "냈", "냈" },
  { "냉", "냉", "냉", "냉", "냉" },
  { "냊", "냊", "냊", "냊", "냊" },
  { "냋", "냋", "냋", "냋", "냋" },
  { "냌", "냌", "냌", "냌", "냌" },
  { "냍", "냍", "냍", "냍", "냍" },
  { "냎", "냎", "냎", "냎", "냎" },
  { "냏", "냏", "냏", "냏", "냏" },
  { "냐", "냐", "냐", "냐", "냐" },
  { "냑", "냑", "냑", "냑", "냑" },
  { "냒", "냒", "냒", "냒", "냒" },
  { "냓", "냓", "냓", "냓", "냓" },
  { "냔", "냔", "냔", "냔", "냔" },
  { "냕", "냕", "냕", "냕", "냕" },
  { "냖", "냖", "냖", "냖", "냖" },
  { "냗", "냗", "냗", "냗", "냗" },
  { "냘", "냘", "냘", "냘", "냘" },
  { "냙", "냙", "냙", "냙", "냙" },
  { "냚", "냚", "냚", "냚", "냚" },
  { "냛", "냛", "냛", "냛", "냛" },
  { "냜", "냜", "냜", "냜", "냜" },
  { "냝", "냝", "냝", "냝", "냝" },
  { "냞", "냞", "냞", "냞", "냞" },
  { "냟", "냟", "냟", "냟", "냟" },
  { "냠", "냠", "냠", "냠", "냠" },
  { "냡", "냡", "냡", "냡", "냡" },
  { "냢", "냢", "냢", "냢", "냢" },
  { "냣", "냣", "냣", "냣", "냣" },
  { "냤", "냤", "냤", "냤", "냤" },
  { "냥", "냥", "냥", "냥", "냥" },
  { "냦", "냦", "냦", "냦", "냦" },
  { "냧", "냧", "냧", "냧", "냧" },
  { "냨", "냨", "냨", "냨", "냨" },
  { "냩", "냩", "냩", "냩", "냩" },
  { "냪", "냪", "냪", "냪", "냪" },
  { "냫", "냫", "냫", "냫", "냫" },
  { "냬", "냬", "냬", "냬", "냬" },
  { "냭", "냭", "냭", "냭", "냭" },
  { "냮", "냮", "냮", "냮", "냮" },
  { "냯", "냯", "냯", "냯", "냯" },
  { "냰", "냰", "냰", "냰", "냰" },
  { "냱", "냱", "냱", "냱", "냱" },
  { "냲", "냲", "냲", "냲", "냲" },
  { "냳", "냳", "냳", "냳", "냳" },
  { "냴", "냴", "냴", "냴", "냴" },
  { "냵", "냵", "냵", "냵", "냵" },
  { "냶", "냶", "냶", "냶", "냶" },
  { "냷", "냷", "냷", "냷", "냷" },
  { "냸", "냸", "냸", "냸", "냸" },
  { "냹", "냹", "냹", "냹", "냹" },
  { "냺", "냺", "냺", "냺", "냺" },
  { "냻", "냻", "냻", "냻", "냻" },
  { "냼", "냼", "냼", "냼", "냼" },
  { "냽", "냽", "냽", "냽", "냽" },
  { "냾", "냾", "냾", "냾", "냾" },
  { "냿", "냿", "냿", "냿", "냿" },
  { "넀", "넀", "넀", "넀", "넀" },
  { "넁", "넁", "넁", "넁", "넁" },
  { "넂", "넂", "넂", "넂", "넂" },
  { "넃", "넃", "넃", "넃", "넃" },
  { "넄", "넄", "넄", "넄", "넄" },
  { "넅", "넅", "넅", "넅", "넅" },
  { "넆", "넆", "넆", "넆", "넆" },
  { "넇", "넇", "넇", "넇", "넇" },
  { "너", "너", "너", "너", "너" },
  { "넉", "넉", "넉", "넉", "넉" },
  { "넊", "넊", "넊", "넊", "넊" },
  { "넋", "넋", "넋", "넋", "넋" },
  { "넌", "넌", "넌", "넌", "넌" },
  { "넍", "넍", "넍", "넍", "넍" },
  { "넎", "넎", "넎", "넎", "넎" },
  { "넏", "넏", "넏", "넏", "넏" },
  { "널", "널", "널", "널", "널" },
  { "넑", "넑", "넑", "넑", "넑" },
  { "넒", "넒", "넒", "넒", "넒" },
  { "넓", "넓", "넓", "넓", "넓" },
  { "넔", "넔", "넔", "넔", "넔" },
  { "넕", "넕", "넕", "넕", "넕" },
  { "넖", "넖", "넖", "넖", "넖" },
  { "넗", "넗", "넗", "넗", "넗" },
  { "넘", "넘", "넘", "넘", "넘" },
  { "넙", "넙", "넙", "넙", "넙" },
  { "넚", "넚", "넚", "넚", "넚" },
  { "넛", "넛", "넛", "넛", "넛" },
  { "넜", "넜", "넜", "넜", "넜" },
  { "넝", "넝", "넝", "넝", "넝" },
  { "넞", "넞", "넞", "넞", "넞" },
  { "넟", "넟", "넟", "넟", "넟" },
  { "넠", "넠", "넠", "넠", "넠" },
  { "넡", "넡", "넡", "넡", "넡" },
  { "넢", "넢", "넢", "넢", "넢" },
  { "넣", "넣", "넣", "넣", "넣" },
  { "네", "네", "네", "네", "네" },
  { "넥", "넥", "넥", "넥", "넥" },
  { "넦", "넦", "넦", "넦", "넦" },
  { "넧", "넧", "넧", "넧", "넧" },
  { "넨", "넨", "넨", "넨", "넨" },
  { "넩", "넩", "넩", "넩", "넩" },
  { "넪", "넪", "넪", "넪", "넪" },
  { "넫", "넫", "넫", "넫", "넫" },
  { "넬", "넬", "넬", "넬", "넬" },
  { "넭", "넭", "넭", "넭", "넭" },
  { "넮", "넮", "넮", "넮", "넮" },
  { "넯", "넯", "넯", "넯", "넯" },
  { "넰", "넰", "넰", "넰", "넰" },
  { "넱", "넱", "넱", "넱", "넱" },
  { "넲", "넲", "넲", "넲", "넲" },
  { "넳", "넳", "넳", "넳", "넳" },
  { "넴", "넴", "넴", "넴", "넴" },
  { "넵", "넵", "넵", "넵", "넵" },
  { "넶", "넶", "넶", "넶", "넶" },
  { "넷", "넷", "넷", "넷", "넷" },
  { "넸", "넸", "넸", "넸", "넸" },
  { "넹", "넹", "넹", "넹", "넹" },
  { "넺", "넺", "넺", "넺", "넺" },
  { "넻", "넻", "넻", "넻", "넻" },
  { "넼", "넼", "넼", "넼", "넼" },
  { "넽", "넽", "넽", "넽", "넽" },
  { "넾", "넾", "넾", "넾", "넾" },
  { "넿", "넿", "넿", "넿", "넿" },
  { "녀", "녀", "녀", "녀", "녀" },
  { "녁", "녁", "녁", "녁", "녁" },
  { "녂", "녂", "녂", "녂", "녂" },
  { "녃", "녃", "녃", "녃", "녃" },
  { "년", "년", "년", "년", "년" },
  { "녅", "녅", "녅", "녅", "녅" },
  { "녆", "녆", "녆", "녆", "녆" },
  { "녇", "녇", "녇", "녇", "녇" },
  { "녈", "녈", "녈", "녈", "녈" },
  { "녉", "녉", "녉", "녉", "녉" },
  { "녊", "녊", "녊", "녊", "녊" },
  { "녋", "녋", "녋", "녋", "녋" },
  { "녌", "녌", "녌", "녌", "녌" },
  { "녍", "녍", "녍", "녍", "녍" },
  { "녎", "녎", "녎", "녎", "녎" },
  { "녏", "녏", "녏", "녏", "녏" },
  { "념", "념", "념", "념", "념" },
  { "녑", "녑", "녑", "녑", "녑" },
  { "녒", "녒", "녒", "녒", "녒" },
  { "녓", "녓", "녓", "녓", "녓" },
  { "녔", "녔", "녔", "녔", "녔" },
  { "녕", "녕", "녕", "녕", "녕" },
  { "녖", "녖", "녖", "녖", "녖" },
  { "녗", "녗", "녗", "녗", "녗" },
  { "녘", "녘", "녘", "녘", "녘" },
  { "녙", "녙", "녙", "녙", "녙" },
  { "녚", "녚", "녚", "녚", "녚" },
  { "녛", "녛", "녛", "녛", "녛" },
  { "녜", "녜", "녜", "녜", "녜" },
  { "녝", "녝", "녝", "녝", "녝" },
  { "녞", "녞", "녞", "녞", "녞" },
  { "녟", "녟", "녟", "녟", "녟" },
  { "녠", "녠", "녠", "녠", "녠" },
  { "녡", "녡", "녡", "녡", "녡" },
  { "녢", "녢", "녢", "녢", "녢" },
  { "녣", "녣", "녣", "녣", "녣" },
  { "녤", "녤", "녤", "녤", "녤" },
  { "녥", "녥", "녥", "녥", "녥" },
  { "녦", "녦", "녦", "녦", "녦" },
  { "녧", "녧", "녧", "녧", "녧" },
  { "녨", "녨", "녨", "녨", "녨" },
  { "녩", "녩", "녩", "녩", "녩" },
  { "녪", "녪", "녪", "녪", "녪" },
  { "녫", "녫", "녫", "녫", "녫" },
  { "녬", "녬", "녬", "녬", "녬" },
  { "녭", "녭", "녭", "녭", "녭" },
  { "녮", "녮", "녮", "녮", "녮" },
  { "녯", "녯", "녯", "녯", "녯" },
  { "녰", "녰", "녰", "녰", "녰" },
  { "녱", "녱", "녱", "녱", "녱" },
  { "녲", "녲", "녲", "녲", "녲" },
  { "녳", "녳", "녳", "녳", "녳" },
  { "녴", "녴", "녴", "녴", "녴" },
  { "녵", "녵", "녵", "녵", "녵" },
  { "녶", "녶", "녶", "녶", "녶" },
  { "녷", "녷", "녷", "녷", "녷" },
  { "노", "노", "노", "노", "노" },
  { "녹", "녹", "녹", "녹", "녹" },
  { "녺", "녺", "녺", "녺", "녺" },
  { "녻", "녻", "녻", "녻", "녻" },
  { "논", "논", "논", "논", "논" },
  { "녽", "녽", "녽", "녽", "녽" },
  { "녾", "녾", "녾", "녾", "녾" },
  { "녿", "녿", "녿", "녿", "녿" },
  { "놀", "놀", "놀", "놀", "놀" },
  { "놁", "놁", "놁", "놁", "놁" },
  { "놂", "놂", "놂", "놂", "놂" },
  { "놃", "놃", "놃", "놃", "놃" },
  { "놄", "놄", "놄", "놄", "놄" },
  { "놅", "놅", "놅", "놅", "놅" },
  { "놆", "놆", "놆", "놆", "놆" },
  { "놇", "놇", "놇", "놇", "놇" },
  { "놈", "놈", "놈", "놈", "놈" },
  { "놉", "놉", "놉", "놉", "놉" },
  { "놊", "놊", "놊", "놊", "놊" },
  { "놋", "놋", "놋", "놋", "놋" },
  { "놌", "놌", "놌", "놌", "놌" },
  { "농", "농", "농", "농", "농" },
  { "놎", "놎", "놎", "놎", "놎" },
  { "놏", "놏", "놏", "놏", "놏" },
  { "놐", "놐", "놐", "놐", "놐" },
  { "놑", "놑", "놑", "놑", "놑" },
  { "높", "높", "높", "높", "높" },
  { "놓", "놓", "놓", "놓", "놓" },
  { "놔", "놔", "놔", "놔", "놔" },
  { "놕", "놕", "놕", "놕", "놕" },
  { "놖", "놖", "놖", "놖", "놖" },
  { "놗", "놗", "놗", "놗", "놗" },
  { "놘", "놘", "놘", "놘", "놘" },
  { "놙", "놙", "놙", "놙", "놙" },
  { "놚", "놚", "놚", "놚", "놚" },
  { "놛", "놛", "놛", "놛", "놛" },
  { "놜", "놜", "놜", "놜", "놜" },
  { "놝", "놝", "놝", "놝", "놝" },
  { "놞", "놞", "놞", "놞", "놞" },
  { "놟", "놟", "놟", "놟", "놟" },
  { "놠", "놠", "놠", "놠", "놠" },
  { "놡", "놡", "놡", "놡", "놡" },
  { "놢", "놢", "놢", "놢", "놢" },
  { "놣", "놣", "놣", "놣", "놣" },
  { "놤", "놤", "놤", "놤", "놤" },
  { "놥", "놥", "놥", "놥", "놥" },
  { "놦", "놦", "놦", "놦", "놦" },
  { "놧", "놧", "놧", "놧", "놧" },
  { "놨", "놨", "놨", "놨", "놨" },
  { "놩", "놩", "놩", "놩", "놩" },
  { "놪", "놪", "놪", "놪", "놪" },
  { "놫", "놫", "놫", "놫", "놫" },
  { "놬", "놬", "놬", "놬", "놬" },
  { "놭", "놭", "놭", "놭", "놭" },
  { "놮", "놮", "놮", "놮", "놮" },
  { "놯", "놯", "놯", "놯", "놯" },
  { "놰", "놰", "놰", "놰", "놰" },
  { "놱", "놱", "놱", "놱", "놱" },
  { "놲", "놲", "놲", "놲", "놲" },
  { "놳", "놳", "놳", "놳", "놳" },
  { "놴", "놴", "놴", "놴", "놴" },
  { "놵", "놵", "놵", "놵", "놵" },
  { "놶", "놶", "놶", "놶", "놶" },
  { "놷", "놷", "놷", "놷", "놷" },
  { "놸", "놸", "놸", "놸", "놸" },
  { "놹", "놹", "놹", "놹", "놹" },
  { "놺", "놺", "놺", "놺", "놺" },
  { "놻", "놻", "놻", "놻", "놻" },
  { "놼", "놼", "놼", "놼", "놼" },
  { "놽", "놽", "놽", "놽", "놽" },
  { "놾", "놾", "놾", "놾", "놾" },
  { "놿", "놿", "놿", "놿", "놿" },
  { "뇀", "뇀", "뇀", "뇀", "뇀" },
  { "뇁", "뇁", "뇁", "뇁", "뇁" },
  { "뇂", "뇂", "뇂", "뇂", "뇂" },
  { "뇃", "뇃", "뇃", "뇃", "뇃" },
  { "뇄", "뇄", "뇄", "뇄", "뇄" },
  { "뇅", "뇅", "뇅", "뇅", "뇅" },
  { "뇆", "뇆", "뇆", "뇆", "뇆" },
  { "뇇", "뇇", "뇇", "뇇", "뇇" },
  { "뇈", "뇈", "뇈", "뇈", "뇈" },
  { "뇉", "뇉", "뇉", "뇉", "뇉" },
  { "뇊", "뇊", "뇊", "뇊", "뇊" },
  { "뇋", "뇋", "뇋", "뇋", "뇋" },
  { "뇌", "뇌", "뇌", "뇌", "뇌" },
  { "뇍", "뇍", "뇍", "뇍", "뇍" },
  { "뇎", "뇎", "뇎", "뇎", "뇎" },
  { "뇏", "뇏", "뇏", "뇏", "뇏" },
  { "뇐", "뇐", "뇐", "뇐", "뇐" },
  { "뇑", "뇑", "뇑", "뇑", "뇑" },
  { "뇒", "뇒", "뇒", "뇒", "뇒" },
  { "뇓", "뇓", "뇓", "뇓", "뇓" },
  { "뇔", "뇔", "뇔", "뇔", "뇔" },
  { "뇕", "뇕", "뇕", "뇕", "뇕" },
  { "뇖", "뇖", "뇖", "뇖", "뇖" },
  { "뇗", "뇗", "뇗", "뇗", "뇗" },
  { "뇘", "뇘", "뇘", "뇘", "뇘" },
  { "뇙", "뇙", "뇙", "뇙", "뇙" },
  { "뇚", "뇚", "뇚", "뇚", "뇚" },
  { "뇛", "뇛", "뇛", "뇛", "뇛" },
  { "뇜", "뇜", "뇜", "뇜", "뇜" },
  { "뇝", "뇝", "뇝", "뇝", "뇝" },
  { "뇞", "뇞", "뇞", "뇞", "뇞" },
  { "뇟", "뇟", "뇟", "뇟", "뇟" },
  { "뇠", "뇠", "뇠", "뇠", "뇠" },
  { "뇡", "뇡", "뇡", "뇡", "뇡" },
  { "뇢", "뇢", "뇢", "뇢", "뇢" },
  { "뇣", "뇣", "뇣", "뇣", "뇣" },
  { "뇤", "뇤", "뇤", "뇤", "뇤" },
  { "뇥", "뇥", "뇥", "뇥", "뇥" },
  { "뇦", "뇦", "뇦", "뇦", "뇦" },
  { "뇧", "뇧", "뇧", "뇧", "뇧" },
  { "뇨", "뇨", "뇨", "뇨", "뇨" },
  { "뇩", "뇩", "뇩", "뇩", "뇩" },
  { "뇪", "뇪", "뇪", "뇪", "뇪" },
  { "뇫", "뇫", "뇫", "뇫", "뇫" },
  { "뇬", "뇬", "뇬", "뇬", "뇬" },
  { "뇭", "뇭", "뇭", "뇭", "뇭" },
  { "뇮", "뇮", "뇮", "뇮", "뇮" },
  { "뇯", "뇯", "뇯", "뇯", "뇯" },
  { "뇰", "뇰", "뇰", "뇰", "뇰" },
  { "뇱", "뇱", "뇱", "뇱", "뇱" },
  { "뇲", "뇲", "뇲", "뇲", "뇲" },
  { "뇳", "뇳", "뇳", "뇳", "뇳" },
  { "뇴", "뇴", "뇴", "뇴", "뇴" },
  { "뇵", "뇵", "뇵", "뇵", "뇵" },
  { "뇶", "뇶", "뇶", "뇶", "뇶" },
  { "뇷", "뇷", "뇷", "뇷", "뇷" },
  { "뇸", "뇸", "뇸", "뇸", "뇸" },
  { "뇹", "뇹", "뇹", "뇹", "뇹" },
  { "뇺", "뇺", "뇺", "뇺", "뇺" },
  { "뇻", "뇻", "뇻", "뇻", "뇻" },
  { "뇼", "뇼", "뇼", "뇼", "뇼" },
  { "뇽", "뇽", "뇽", "뇽", "뇽" },
  { "뇾", "뇾", "뇾", "뇾", "뇾" },
  { "뇿", "뇿", "뇿", "뇿", "뇿" },
  { "눀", "눀", "눀", "눀", "눀" },
  { "눁", "눁", "눁", "눁", "눁" },
  { "눂", "눂", "눂", "눂", "눂" },
  { "눃", "눃", "눃", "눃", "눃" },
  { "누", "누", "누", "누", "누" },
  { "눅", "눅", "눅", "눅", "눅" },
  { "눆", "눆", "눆", "눆", "눆" },
  { "눇", "눇", "눇", "눇", "눇" },
  { "눈", "눈", "눈", "눈", "눈" },
  { "눉", "눉", "눉", "눉", "눉" },
  { "눊", "눊", "눊", "눊", "눊" },
  { "눋", "눋", "눋", "눋", "눋" },
  { "눌", "눌", "눌", "눌", "눌" },
  { "눍", "눍", "눍", "눍", "눍" },
  { "눎", "눎", "눎", "눎", "눎" },
  { "눏", "눏", "눏", "눏", "눏" },
  { "눐", "눐", "눐", "눐", "눐" },
  { "눑", "눑", "눑", "눑", "눑" },
  { "눒", "눒", "눒", "눒", "눒" },
  { "눓", "눓", "눓", "눓", "눓" },
  { "눔", "눔", "눔", "눔", "눔" },
  { "눕", "눕", "눕", "눕", "눕" },
  { "눖", "눖", "눖", "눖", "눖" },
  { "눗", "눗", "눗", "눗", "눗" },
  { "눘", "눘", "눘", "눘", "눘" },
  { "눙", "눙", "눙", "눙", "눙" },
  { "눚", "눚", "눚", "눚", "눚" },
  { "눛", "눛", "눛", "눛", "눛" },
  { "눜", "눜", "눜", "눜", "눜" },
  { "눝", "눝", "눝", "눝", "눝" },
  { "눞", "눞", "눞", "눞", "눞" },
  { "눟", "눟", "눟", "눟", "눟" },
  { "눠", "눠", "눠", "눠", "눠" },
  { "눡", "눡", "눡", "눡", "눡" },
  { "눢", "눢", "눢", "눢", "눢" },
  { "눣", "눣", "눣", "눣", "눣" },
  { "눤", "눤", "눤", "눤", "눤" },
  { "눥", "눥", "눥", "눥", "눥" },
  { "눦", "눦", "눦", "눦", "눦" },
  { "눧", "눧", "눧", "눧", "눧" },
  { "눨", "눨", "눨", "눨", "눨" },
  { "눩", "눩", "눩", "눩", "눩" },
  { "눪", "눪", "눪", "눪", "눪" },
  { "눫", "눫", "눫", "눫", "눫" },
  { "눬", "눬", "눬", "눬", "눬" },
  { "눭", "눭", "눭", "눭", "눭" },
  { "눮", "눮", "눮", "눮", "눮" },
  { "눯", "눯", "눯", "눯", "눯" },
  { "눰", "눰", "눰", "눰", "눰" },
  { "눱", "눱", "눱", "눱", "눱" },
  { "눲", "눲", "눲", "눲", "눲" },
  { "눳", "눳", "눳", "눳", "눳" },
  { "눴", "눴", "눴", "눴", "눴" },
  { "눵", "눵", "눵", "눵", "눵" },
  { "눶", "눶", "눶", "눶", "눶" },
  { "눷", "눷", "눷", "눷", "눷" },
  { "눸", "눸", "눸", "눸", "눸" },
  { "눹", "눹", "눹", "눹", "눹" },
  { "눺", "눺", "눺", "눺", "눺" },
  { "눻", "눻", "눻", "눻", "눻" },
  { "눼", "눼", "눼", "눼", "눼" },
  { "눽", "눽", "눽", "눽", "눽" },
  { "눾", "눾", "눾", "눾", "눾" },
  { "눿", "눿", "눿", "눿", "눿" },
  { "뉀", "뉀", "뉀", "뉀", "뉀" },
  { "뉁", "뉁", "뉁", "뉁", "뉁" },
  { "뉂", "뉂", "뉂", "뉂", "뉂" },
  { "뉃", "뉃", "뉃", "뉃", "뉃" },
  { "뉄", "뉄", "뉄", "뉄", "뉄" },
  { "뉅", "뉅", "뉅", "뉅", "뉅" },
  { "뉆", "뉆", "뉆", "뉆", "뉆" },
  { "뉇", "뉇", "뉇", "뉇", "뉇" },
  { "뉈", "뉈", "뉈", "뉈", "뉈" },
  { "뉉", "뉉", "뉉", "뉉", "뉉" },
  { "뉊", "뉊", "뉊", "뉊", "뉊" },
  { "뉋", "뉋", "뉋", "뉋", "뉋" },
  { "뉌", "뉌", "뉌", "뉌", "뉌" },
  { "뉍", "뉍", "뉍", "뉍", "뉍" },
  { "뉎", "뉎", "뉎", "뉎", "뉎" },
  { "뉏", "뉏", "뉏", "뉏", "뉏" },
  { "뉐", "뉐", "뉐", "뉐", "뉐" },
  { "뉑", "뉑", "뉑", "뉑", "뉑" },
  { "뉒", "뉒", "뉒", "뉒", "뉒" },
  { "뉓", "뉓", "뉓", "뉓", "뉓" },
  { "뉔", "뉔", "뉔", "뉔", "뉔" },
  { "뉕", "뉕", "뉕", "뉕", "뉕" },
  { "뉖", "뉖", "뉖", "뉖", "뉖" },
  { "뉗", "뉗", "뉗", "뉗", "뉗" },
  { "뉘", "뉘", "뉘", "뉘", "뉘" },
  { "뉙", "뉙", "뉙", "뉙", "뉙" },
  { "뉚", "뉚", "뉚", "뉚", "뉚" },
  { "뉛", "뉛", "뉛", "뉛", "뉛" },
  { "뉜", "뉜", "뉜", "뉜", "뉜" },
  { "뉝", "뉝", "뉝", "뉝", "뉝" },
  { "뉞", "뉞", "뉞", "뉞", "뉞" },
  { "뉟", "뉟", "뉟", "뉟", "뉟" },
  { "뉠", "뉠", "뉠", "뉠", "뉠" },
  { "뉡", "뉡", "뉡", "뉡", "뉡" },
  { "뉢", "뉢", "뉢", "뉢", "뉢" },
  { "뉣", "뉣", "뉣", "뉣", "뉣" },
  { "뉤", "뉤", "뉤", "뉤", "뉤" },
  { "뉥", "뉥", "뉥", "뉥", "뉥" },
  { "뉦", "뉦", "뉦", "뉦", "뉦" },
  { "뉧", "뉧", "뉧", "뉧", "뉧" },
  { "뉨", "뉨", "뉨", "뉨", "뉨" },
  { "뉩", "뉩", "뉩", "뉩", "뉩" },
  { "뉪", "뉪", "뉪", "뉪", "뉪" },
  { "뉫", "뉫", "뉫", "뉫", "뉫" },
  { "뉬", "뉬", "뉬", "뉬", "뉬" },
  { "뉭", "뉭", "뉭", "뉭", "뉭" },
  { "뉮", "뉮", "뉮", "뉮", "뉮" },
  { "뉯", "뉯", "뉯", "뉯", "뉯" },
  { "뉰", "뉰", "뉰", "뉰", "뉰" },
  { "뉱", "뉱", "뉱", "뉱", "뉱" },
  { "뉲", "뉲", "뉲", "뉲", "뉲" },
  { "뉳", "뉳", "뉳", "뉳", "뉳" },
  { "뉴", "뉴", "뉴", "뉴", "뉴" },
  { "뉵", "뉵", "뉵", "뉵", "뉵" },
  { "뉶", "뉶", "뉶", "뉶", "뉶" },
  { "뉷", "뉷", "뉷", "뉷", "뉷" },
  { "뉸", "뉸", "뉸", "뉸", "뉸" },
  { "뉹", "뉹", "뉹", "뉹", "뉹" },
  { "뉺", "뉺", "뉺", "뉺", "뉺" },
  { "뉻", "뉻", "뉻", "뉻", "뉻" },
  { "뉼", "뉼", "뉼", "뉼", "뉼" },
  { "뉽", "뉽", "뉽", "뉽", "뉽" },
  { "뉾", "뉾", "뉾", "뉾", "뉾" },
  { "뉿", "뉿", "뉿", "뉿", "뉿" },
  { "늀", "늀", "늀", "늀", "늀" },
  { "늁", "늁", "늁", "늁", "늁" },
  { "늂", "늂", "늂", "늂", "늂" },
  { "늃", "늃", "늃", "늃", "늃" },
  { "늄", "늄", "늄", "늄", "늄" },
  { "늅", "늅", "늅", "늅", "늅" },
  { "늆", "늆", "늆", "늆", "늆" },
  { "늇", "늇", "늇", "늇", "늇" },
  { "늈", "늈", "늈", "늈", "늈" },
  { "늉", "늉", "늉", "늉", "늉" },
  { "늊", "늊", "늊", "늊", "늊" },
  { "늋", "늋", "늋", "늋", "늋" },
  { "늌", "늌", "늌", "늌", "늌" },
  { "늍", "늍", "늍", "늍", "늍" },
  { "늎", "늎", "늎", "늎", "늎" },
  { "늏", "늏", "늏", "늏", "늏" },
  { "느", "느", "느", "느", "느" },
  { "늑", "늑", "늑", "늑", "늑" },
  { "늒", "늒", "늒", "늒", "늒" },
  { "늓", "늓", "늓", "늓", "늓" },
  { "는", "는", "는", "는", "는" },
  { "늕", "늕", "늕", "늕", "늕" },
  { "늖", "늖", "늖", "늖", "늖" },
  { "늗", "늗", "늗", "늗", "늗" },
  { "늘", "늘", "늘", "늘", "늘" },
  { "늙", "늙", "늙", "늙", "늙" },
  { "늚", "늚", "늚", "늚", "늚" },
  { "늛", "늛", "늛", "늛", "늛" },
  { "늜", "늜", "늜", "늜", "늜" },
  { "늝", "늝", "늝", "늝", "늝" },
  { "늞", "늞", "늞", "늞", "늞" },
  { "늟", "늟", "늟", "늟", "늟" },
  { "늠", "늠", "늠", "늠", "늠" },
  { "늡", "늡", "늡", "늡", "늡" },
  { "늢", "늢", "늢", "늢", "늢" },
  { "늣", "늣", "늣", "늣", "늣" },
  { "늤", "늤", "늤", "늤", "늤" },
  { "능", "능", "능", "능", "능" },
  { "늦", "늦", "늦", "늦", "늦" },
  { "늧", "늧", "늧", "늧", "늧" },
  { "늨", "늨", "늨", "늨", "늨" },
  { "늩", "늩", "늩", "늩", "늩" },
  { "늪", "늪", "늪", "늪", "늪" },
  { "늫", "늫", "늫", "늫", "늫" },
  { "늬", "늬", "늬", "늬", "늬" },
  { "늭", "늭", "늭", "늭", "늭" },
  { "늮", "늮", "늮", "늮", "늮" },
  { "늯", "늯", "늯", "늯", "늯" },
  { "늰", "늰", "늰", "늰", "늰" },
  { "늱", "늱", "늱", "늱", "늱" },
  { "늲", "늲", "늲", "늲", "늲" },
  { "늳", "늳", "늳", "늳", "늳" },
  { "늴", "늴", "늴", "늴", "늴" },
  { "늵", "늵", "늵", "늵", "늵" },
  { "늶", "늶", "늶", "늶", "늶" },
  { "늷", "늷", "늷", "늷", "늷" },
  { "늸", "늸", "늸", "늸", "늸" },
  { "늹", "늹", "늹", "늹", "늹" },
  { "늺", "늺", "늺", "늺", "늺" },
  { "늻", "늻", "늻", "늻", "늻" },
  { "늼", "늼", "늼", "늼", "늼" },
  { "늽", "늽", "늽", "늽", "늽" },
  { "늾", "늾", "늾", "늾", "늾" },
  { "늿", "늿", "늿", "늿", "늿" },
  { "닀", "닀", "닀", "닀", "닀" },
  { "닁", "닁", "닁", "닁", "닁" },
  { "닂", "닂", "닂", "닂", "닂" },
  { "닃", "닃", "닃", "닃", "닃" },
  { "닄", "닄", "닄", "닄", "닄" },
  { "닅", "닅", "닅", "닅", "닅" },
  { "닆", "닆", "닆", "닆", "닆" },
  { "닇", "닇", "닇", "닇", "닇" },
  { "니", "니", "니", "니", "니" },
  { "닉", "닉", "닉", "닉", "닉" },
  { "닊", "닊", "닊", "닊", "닊" },
  { "닋", "닋", "닋", "닋", "닋" },
  { "닌", "닌", "닌", "닌", "닌" },
  { "닍", "닍", "닍", "닍", "닍" },
  { "닎", "닎", "닎", "닎", "닎" },
  { "닏", "닏", "닏", "닏", "닏" },
  { "닐", "닐", "닐", "닐", "닐" },
  { "닑", "닑", "닑", "닑", "닑" },
  { "닒", "닒", "닒", "닒", "닒" },
  { "닓", "닓", "닓", "닓", "닓" },
  { "닔", "닔", "닔", "닔", "닔" },
  { "닕", "닕", "닕", "닕", "닕" },
  { "닖", "닖", "닖", "닖", "닖" },
  { "닗", "닗", "닗", "닗", "닗" },
  { "님", "님", "님", "님", "님" },
  { "닙", "닙", "닙", "닙", "닙" },
  { "닚", "닚", "닚", "닚", "닚" },
  { "닛", "닛", "닛", "닛", "닛" },
  { "닜", "닜", "닜", "닜", "닜" },
  { "닝", "닝", "닝", "닝", "닝" },
  { "닞", "닞", "닞", "닞", "닞" },
  { "닟", "닟", "닟", "닟", "닟" },
  { "닠", "닠", "닠", "닠", "닠" },
  { "닡", "닡", "닡", "닡", "닡" },
  { "닢", "닢", "닢", "닢", "닢" },
  { "닣", "닣", "닣", "닣", "닣" },
  { "다", "다", "다", "다", "다" },
  { "닥", "닥", "닥", "닥", "닥" },
  { "닦", "닦", "닦", "닦", "닦" },
  { "닧", "닧", "닧", "닧", "닧" },
  { "단", "단", "단", "단", "단" },
  { "닩", "닩", "닩", "닩", "닩" },
  { "닪", "닪", "닪", "닪", "닪" },
  { "닫", "닫", "닫", "닫", "닫" },
  { "달", "달", "달", "달", "달" },
  { "닭", "닭", "닭", "닭", "닭" },
  { "닮", "닮", "닮", "닮", "닮" },
  { "닯", "닯", "닯", "닯", "닯" },
  { "닰", "닰", "닰", "닰", "닰" },
  { "닱", "닱", "닱", "닱", "닱" },
  { "닲", "닲", "닲", "닲", "닲" },
  { "닳", "닳", "닳", "닳", "닳" },
  { "담", "담", "담", "담", "담" },
  { "답", "답", "답", "답", "답" },
  { "닶", "닶", "닶", "닶", "닶" },
  { "닷", "닷", "닷", "닷", "닷" },
  { "닸", "닸", "닸", "닸", "닸" },
  { "당", "당", "당", "당", "당" },
  { "닺", "닺", "닺", "닺", "닺" },
  { "닻", "닻", "닻", "닻", "닻" },
  { "닼", "닼", "닼", "닼", "닼" },
  { "닽", "닽", "닽", "닽", "닽" },
  { "닾", "닾", "닾", "닾", "닾" },
  { "닿", "닿", "닿", "닿", "닿" },
  { "대", "대", "대", "대", "대" },
  { "댁", "댁", "댁", "댁", "댁" },
  { "댂", "댂", "댂", "댂", "댂" },
  { "댃", "댃", "댃", "댃", "댃" },
  { "댄", "댄", "댄", "댄", "댄" },
  { "댅", "댅", "댅", "댅", "댅" },
  { "댆", "댆", "댆", "댆", "댆" },
  { "댇", "댇", "댇", "댇", "댇" },
  { "댈", "댈", "댈", "댈", "댈" },
  { "댉", "댉", "댉", "댉", "댉" },
  { "댊", "댊", "댊", "댊", "댊" },
  { "댋", "댋", "댋", "댋", "댋" },
  { "댌", "댌", "댌", "댌", "댌" },
  { "댍", "댍", "댍", "댍", "댍" },
  { "댎", "댎", "댎", "댎", "댎" },
  { "댏", "댏", "댏", "댏", "댏" },
  { "댐", "댐", "댐", "댐", "댐" },
  { "댑", "댑", "댑", "댑", "댑" },
  { "댒", "댒", "댒", "댒", "댒" },
  { "댓", "댓", "댓", "댓", "댓" },
  { "댔", "댔", "댔", "댔", "댔" },
  { "댕", "댕", "댕", "댕", "댕" },
  { "댖", "댖", "댖", "댖", "댖" },
  { "댗", "댗", "댗", "댗", "댗" },
  { "댘", "댘", "댘", "댘", "댘" },
  { "댙", "댙", "댙", "댙", "댙" },
  { "댚", "댚", "댚", "댚", "댚" },
  { "댛", "댛", "댛", "댛", "댛" },
  { "댜", "댜", "댜", "댜", "댜" },
  { "댝", "댝", "댝", "댝", "댝" },
  { "댞", "댞", "댞", "댞", "댞" },
  { "댟", "댟", "댟", "댟", "댟" },
  { "댠", "댠", "댠", "댠", "댠" },
  { "댡", "댡", "댡", "댡", "댡" },
  { "댢", "댢", "댢", "댢", "댢" },
  { "댣", "댣", "댣", "댣", "댣" },
  { "댤", "댤", "댤", "댤", "댤" },
  { "댥", "댥", "댥", "댥", "댥" },
  { "댦", "댦", "댦", "댦", "댦" },
  { "댧", "댧", "댧", "댧", "댧" },
  { "댨", "댨", "댨", "댨", "댨" },
  { "댩", "댩", "댩", "댩", "댩" },
  { "댪", "댪", "댪", "댪", "댪" },
  { "댫", "댫", "댫", "댫", "댫" },
  { "댬", "댬", "댬", "댬", "댬" },
  { "댭", "댭", "댭", "댭", "댭" },
  { "댮", "댮", "댮", "댮", "댮" },
  { "댯", "댯", "댯", "댯", "댯" },
  { "댰", "댰", "댰", "댰", "댰" },
  { "댱", "댱", "댱", "댱", "댱" },
  { "댲", "댲", "댲", "댲", "댲" },
  { "댳", "댳", "댳", "댳", "댳" },
  { "댴", "댴", "댴", "댴", "댴" },
  { "댵", "댵", "댵", "댵", "댵" },
  { "댶", "댶", "댶", "댶", "댶" },
  { "댷", "댷", "댷", "댷", "댷" },
  { "댸", "댸", "댸", "댸", "댸" },
  { "댹", "댹", "댹", "댹", "댹" },
  { "댺", "댺", "댺", "댺", "댺" },
  { "댻", "댻", "댻", "댻", "댻" },
  { "댼", "댼", "댼", "댼", "댼" },
  { "댽", "댽", "댽", "댽", "댽" },
  { "댾", "댾", "댾", "댾", "댾" },
  { "댿", "댿", "댿", "댿", "댿" },
  { "덀", "덀", "덀", "덀", "덀" },
  { "덁", "덁", "덁", "덁", "덁" },
  { "덂", "덂", "덂", "덂", "덂" },
  { "덃", "덃", "덃", "덃", "덃" },
  { "덄", "덄", "덄", "덄", "덄" },
  { "덅", "덅", "덅", "덅", "덅" },
  { "덆", "덆", "덆", "덆", "덆" },
  { "덇", "덇", "덇", "덇", "덇" },
  { "덈", "덈", "덈", "덈", "덈" },
  { "덉", "덉", "덉", "덉", "덉" },
  { "덊", "덊", "덊", "덊", "덊" },
  { "덋", "덋", "덋", "덋", "덋" },
  { "덌", "덌", "덌", "덌", "덌" },
  { "덍", "덍", "덍", "덍", "덍" },
  { "덎", "덎", "덎", "덎", "덎" },
  { "덏", "덏", "덏", "덏", "덏" },
  { "덐", "덐", "덐", "덐", "덐" },
  { "덑", "덑", "덑", "덑", "덑" },
  { "덒", "덒", "덒", "덒", "덒" },
  { "덓", "덓", "덓", "덓", "덓" },
  { "더", "더", "더", "더", "더" },
  { "덕", "덕", "덕", "덕", "덕" },
  { "덖", "덖", "덖", "덖", "덖" },
  { "덗", "덗", "덗", "덗", "덗" },
  { "던", "던", "던", "던", "던" },
  { "덙", "덙", "덙", "덙", "덙" },
  { "덚", "덚", "덚", "덚", "덚" },
  { "덛", "덛", "덛", "덛", "덛" },
  { "덜", "덜", "덜", "덜", "덜" },
  { "덝", "덝", "덝", "덝", "덝" },
  { "덞", "덞", "덞", "덞", "덞" },
  { "덟", "덟", "덟", "덟", "덟" },
  { "덠", "덠", "덠", "덠", "덠" },
  { "덡", "덡", "덡", "덡", "덡" },
  { "덢", "덢", "덢", "덢", "덢" },
  { "덣", "덣", "덣", "덣", "덣" },
  { "덤", "덤", "덤", "덤", "덤" },
  { "덥", "덥", "덥", "덥", "덥" },
  { "덦", "덦", "덦", "덦", "덦" },
  { "덧", "덧", "덧", "덧", "덧" },
  { "덨", "덨", "덨", "덨", "덨" },
  { "덩", "덩", "덩", "덩", "덩" },
  { "덪", "덪", "덪", "덪", "덪" },
  { "덫", "덫", "덫", "덫", "덫" },
  { "덬", "덬", "덬", "덬", "덬" },
  { "덭", "덭", "덭", "덭", "덭" },
  { "덮", "덮", "덮", "덮", "덮" },
  { "덯", "덯", "덯", "덯", "덯" },
  { "데", "데", "데", "데", "데" },
  { "덱", "덱", "덱", "덱", "덱" },
  { "덲", "덲", "덲", "덲", "덲" },
  { "덳", "덳", "덳", "덳", "덳" },
  { "덴", "덴", "덴", "덴", "덴" },
  { "덵", "덵", "덵", "덵", "덵" },
  { "덶", "덶", "덶", "덶", "덶" },
  { "덷", "덷", "덷", "덷", "덷" },
  { "델", "델", "델", "델", "델" },
  { "덹", "덹", "덹", "덹", "덹" },
  { "덺", "덺", "덺", "덺", "덺" },
  { "덻", "덻", "덻", "덻", "덻" },
  { "덼", "덼", "덼", "덼", "덼" },
  { "덽", "덽", "덽", "덽", "덽" },
  { "덾", "덾", "덾", "덾", "덾" },
  { "덿", "덿", "덿", "덿", "덿" },
  { "뎀", "뎀", "뎀", "뎀", "뎀" },
  { "뎁", "뎁", "뎁", "뎁", "뎁" },
  { "뎂", "뎂", "뎂", "뎂", "뎂" },
  { "뎃", "뎃", "뎃", "뎃", "뎃" },
  { "뎄", "뎄", "뎄", "뎄", "뎄" },
  { "뎅", "뎅", "뎅", "뎅", "뎅" },
  { "뎆", "뎆", "뎆", "뎆", "뎆" },
  { "뎇", "뎇", "뎇", "뎇", "뎇" },
  { "뎈", "뎈", "뎈", "뎈", "뎈" },
  { "뎉", "뎉", "뎉", "뎉", "뎉" },
  { "뎊", "뎊", "뎊", "뎊", "뎊" },
  { "뎋", "뎋", "뎋", "뎋", "뎋" },
  { "뎌", "뎌", "뎌", "뎌", "뎌" },
  { "뎍", "뎍", "뎍", "뎍", "뎍" },
  { "뎎", "뎎", "뎎", "뎎", "뎎" },
  { "뎏", "뎏", "뎏", "뎏", "뎏" },
  { "뎐", "뎐", "뎐", "뎐", "뎐" },
  { "뎑", "뎑", "뎑", "뎑", "뎑" },
  { "뎒", "뎒", "뎒", "뎒", "뎒" },
  { "뎓", "뎓", "뎓", "뎓", "뎓" },
  { "뎔", "뎔", "뎔", "뎔", "뎔" },
  { "뎕", "뎕", "뎕", "뎕", "뎕" },
  { "뎖", "뎖", "뎖", "뎖", "뎖" },
  { "뎗", "뎗", "뎗", "뎗", "뎗" },
  { "뎘", "뎘", "뎘", "뎘", "뎘" },
  { "뎙", "뎙", "뎙", "뎙", "뎙" },
  { "뎚", "뎚", "뎚", "뎚", "뎚" },
  { "뎛", "뎛", "뎛", "뎛", "뎛" },
  { "뎜", "뎜", "뎜", "뎜", "뎜" },
  { "뎝", "뎝", "뎝", "뎝", "뎝" },
  { "뎞", "뎞", "뎞", "뎞", "뎞" },
  { "뎟", "뎟", "뎟", "뎟", "뎟" },
  { "뎠", "뎠", "뎠", "뎠", "뎠" },
  { "뎡", "뎡", "뎡", "뎡", "뎡" },
  { "뎢", "뎢", "뎢", "뎢", "뎢" },
  { "뎣", "뎣", "뎣", "뎣", "뎣" },
  { "뎤", "뎤", "뎤", "뎤", "뎤" },
  { "뎥", "뎥", "뎥", "뎥", "뎥" },
  { "뎦", "뎦", "뎦", "뎦", "뎦" },
  { "뎧", "뎧", "뎧", "뎧", "뎧" },
  { "뎨", "뎨", "뎨", "뎨", "뎨" },
  { "뎩", "뎩", "뎩", "뎩", "뎩" },
  { "뎪", "뎪", "뎪", "뎪", "뎪" },
  { "뎫", "뎫", "뎫", "뎫", "뎫" },
  { "뎬", "뎬", "뎬", "뎬", "뎬" },
  { "뎭", "뎭", "뎭", "뎭", "뎭" },
  { "뎮", "뎮", "뎮", "뎮", "뎮" },
  { "뎯", "뎯", "뎯", "뎯", "뎯" },
  { "뎰", "뎰", "뎰", "뎰", "뎰" },
  { "뎱", "뎱", "뎱", "뎱", "뎱" },
  { "뎲", "뎲", "뎲", "뎲", "뎲" },
  { "뎳", "뎳", "뎳", "뎳", "뎳" },
  { "뎴", "뎴", "뎴", "뎴", "뎴" },
  { "뎵", "뎵", "뎵", "뎵", "뎵" },
  { "뎶", "뎶", "뎶", "뎶", "뎶" },
  { "뎷", "뎷", "뎷", "뎷", "뎷" },
  { "뎸", "뎸", "뎸", "뎸", "뎸" },
  { "뎹", "뎹", "뎹", "뎹", "뎹" },
  { "뎺", "뎺", "뎺", "뎺", "뎺" },
  { "뎻", "뎻", "뎻", "뎻", "뎻" },
  { "뎼", "뎼", "뎼", "뎼", "뎼" },
  { "뎽", "뎽", "뎽", "뎽", "뎽" },
  { "뎾", "뎾", "뎾", "뎾", "뎾" },
  { "뎿", "뎿", "뎿", "뎿", "뎿" },
  { "돀", "돀", "돀", "돀", "돀" },
  { "돁", "돁", "돁", "돁", "돁" },
  { "돂", "돂", "돂", "돂", "돂" },
  { "돃", "돃", "돃", "돃", "돃" },
  { "도", "도", "도", "도", "도" },
  { "독", "독", "독", "독", "독" },
  { "돆", "돆", "돆", "돆", "돆" },
  { "돇", "돇", "돇", "돇", "돇" },
  { "돈", "돈", "돈", "돈", "돈" },
  { "돉", "돉", "돉", "돉", "돉" },
  { "돊", "돊", "돊", "돊", "돊" },
  { "돋", "돋", "돋", "돋", "돋" },
  { "돌", "돌", "돌", "돌", "돌" },
  { "돍", "돍", "돍", "돍", "돍" },
  { "돎", "돎", "돎", "돎", "돎" },
  { "돏", "돏", "돏", "돏", "돏" },
  { "돐", "돐", "돐", "돐", "돐" },
  { "돑", "돑", "돑", "돑", "돑" },
  { "돒", "돒", "돒", "돒", "돒" },
  { "돓", "돓", "돓", "돓", "돓" },
  { "돔", "돔", "돔", "돔", "돔" },
  { "돕", "돕", "돕", "돕", "돕" },
  { "돖", "돖", "돖", "돖", "돖" },
  { "돗", "돗", "돗", "돗", "돗" },
  { "돘", "돘", "돘", "돘", "돘" },
  { "동", "동", "동", "동", "동" },
  { "돚", "돚", "돚", "돚", "돚" },
  { "돛", "돛", "돛", "돛", "돛" },
  { "돜", "돜", "돜", "돜", "돜" },
  { "돝", "돝", "돝", "돝", "돝" },
  { "돞", "돞", "돞", "돞", "돞" },
  { "돟", "돟", "돟", "돟", "돟" },
  { "돠", "돠", "돠", "돠", "돠" },
  { "돡", "돡", "돡", "돡", "돡" },
  { "돢", "돢", "돢", "돢", "돢" },
  { "돣", "돣", "돣", "돣", "돣" },
  { "돤", "돤", "돤", "돤", "돤" },
  { "돥", "돥", "돥", "돥", "돥" },
  { "돦", "돦", "돦", "돦", "돦" },
  { "돧", "돧", "돧", "돧", "돧" },
  { "돨", "돨", "돨", "돨", "돨" },
  { "돩", "돩", "돩", "돩", "돩" },
  { "돪", "돪", "돪", "돪", "돪" },
  { "돫", "돫", "돫", "돫", "돫" },
  { "돬", "돬", "돬", "돬", "돬" },
  { "돭", "돭", "돭", "돭", "돭" },
  { "돮", "돮", "돮", "돮", "돮" },
  { "돯", "돯", "돯", "돯", "돯" },
  { "돰", "돰", "돰", "돰", "돰" },
  { "돱", "돱", "돱", "돱", "돱" },
  { "돲", "돲", "돲", "돲", "돲" },
  { "돳", "돳", "돳", "돳", "돳" },
  { "돴", "돴", "돴", "돴", "돴" },
  { "돵", "돵", "돵", "돵", "돵" },
  { "돶", "돶", "돶", "돶", "돶" },
  { "돷", "돷", "돷", "돷", "돷" },
  { "돸", "돸", "돸", "돸", "돸" },
  { "돹", "돹", "돹", "돹", "돹" },
  { "돺", "돺", "돺", "돺", "돺" },
  { "돻", "돻", "돻", "돻", "돻" },
  { "돼", "돼", "돼", "돼", "돼" },
  { "돽", "돽", "돽", "돽", "돽" },
  { "돾", "돾", "돾", "돾", "돾" },
  { "돿", "돿", "돿", "돿", "돿" },
  { "됀", "됀", "됀", "됀", "됀" },
  { "됁", "됁", "됁", "됁", "됁" },
  { "됂", "됂", "됂", "됂", "됂" },
  { "됃", "됃", "됃", "됃", "됃" },
  { "됄", "됄", "됄", "됄", "됄" },
  { "됅", "됅", "됅", "됅", "됅" },
  { "됆", "됆", "됆", "됆", "됆" },
  { "됇", "됇", "됇", "됇", "됇" },
  { "됈", "됈", "됈", "됈", "됈" },
  { "됉", "됉", "됉", "됉", "됉" },
  { "됊", "됊", "됊", "됊", "됊" },
  { "됋", "됋", "됋", "됋", "됋" },
  { "됌", "됌", "됌", "됌", "됌" },
  { "됍", "됍", "됍", "됍", "됍" },
  { "됎", "됎", "됎", "됎", "됎" },
  { "됏", "됏", "됏", "됏", "됏" },
  { "됐", "됐", "됐", "됐", "됐" },
  { "됑", "됑", "됑", "됑", "됑" },
  { "됒", "됒", "됒", "됒", "됒" },
  { "됓", "됓", "됓", "됓", "됓" },
  { "됔", "됔", "됔", "됔", "됔" },
  { "됕", "됕", "됕", "됕", "됕" },
  { "됖", "됖", "됖", "됖", "됖" },
  { "됗", "됗", "됗", "됗", "됗" },
  { "되", "되", "되", "되", "되" },
  { "됙", "됙", "됙", "됙", "됙" },
  { "됚", "됚", "됚", "됚", "됚" },
  { "됛", "됛", "됛", "됛", "됛" },
  { "된", "된", "된", "된", "된" },
  { "됝", "됝", "됝", "됝", "됝" },
  { "됞", "됞", "됞", "됞", "됞" },
  { "됟", "됟", "됟", "됟", "됟" },
  { "될", "될", "될", "될", "될" },
  { "됡", "됡", "됡", "됡", "됡" },
  { "됢", "됢", "됢", "됢", "됢" },
  { "됣", "됣", "됣", "됣", "됣" },
  { "됤", "됤", "됤", "됤", "됤" },
  { "됥", "됥", "됥", "됥", "됥" },
  { "됦", "됦", "됦", "됦", "됦" },
  { "됧", "됧", "됧", "됧", "됧" },
  { "됨", "됨", "됨", "됨", "됨" },
  { "됩", "됩", "됩", "됩", "됩" },
  { "됪", "됪", "됪", "됪", "됪" },
  { "됫", "됫", "됫", "됫", "됫" },
  { "됬", "됬", "됬", "됬", "됬" },
  { "됭", "됭", "됭", "됭", "됭" },
  { "됮", "됮", "됮", "됮", "됮" },
  { "됯", "됯", "됯", "됯", "됯" },
  { "됰", "됰", "됰", "됰", "됰" },
  { "됱", "됱", "됱", "됱", "됱" },
  { "됲", "됲", "됲", "됲", "됲" },
  { "됳", "됳", "됳", "됳", "됳" },
  { "됴", "됴", "됴", "됴", "됴" },
  { "됵", "됵", "됵", "됵", "됵" },
  { "됶", "됶", "됶", "됶", "됶" },
  { "됷", "됷", "됷", "됷", "됷" },
  { "됸", "됸", "됸", "됸", "됸" },
  { "됹", "됹", "됹", "됹", "됹" },
  { "됺", "됺", "됺", "됺", "됺" },
  { "됻", "됻", "됻", "됻", "됻" },
  { "됼", "됼", "됼", "됼", "됼" },
  { "됽", "됽", "됽", "됽", "됽" },
  { "됾", "됾", "됾", "됾", "됾" },
  { "됿", "됿", "됿", "됿", "됿" },
  { "둀", "둀", "둀", "둀", "둀" },
  { "둁", "둁", "둁", "둁", "둁" },
  { "둂", "둂", "둂", "둂", "둂" },
  { "둃", "둃", "둃", "둃", "둃" },
  { "둄", "둄", "둄", "둄", "둄" },
  { "둅", "둅", "둅", "둅", "둅" },
  { "둆", "둆", "둆", "둆", "둆" },
  { "둇", "둇", "둇", "둇", "둇" },
  { "둈", "둈", "둈", "둈", "둈" },
  { "둉", "둉", "둉", "둉", "둉" },
  { "둊", "둊", "둊", "둊", "둊" },
  { "둋", "둋", "둋", "둋", "둋" },
  { "둌", "둌", "둌", "둌", "둌" },
  { "둍", "둍", "둍", "둍", "둍" },
  { "둎", "둎", "둎", "둎", "둎" },
  { "둏", "둏", "둏", "둏", "둏" },
  { "두", "두", "두", "두", "두" },
  { "둑", "둑", "둑", "둑", "둑" },
  { "둒", "둒", "둒", "둒", "둒" },
  { "둓", "둓", "둓", "둓", "둓" },
  { "둔", "둔", "둔", "둔", "둔" },
  { "둕", "둕", "둕", "둕", "둕" },
  { "둖", "둖", "둖", "둖", "둖" },
  { "둗", "둗", "둗", "둗", "둗" },
  { "둘", "둘", "둘", "둘", "둘" },
  { "둙", "둙", "둙", "둙", "둙" },
  { "둚", "둚", "둚", "둚", "둚" },
  { "둛", "둛", "둛", "둛", "둛" },
  { "둜", "둜", "둜", "둜", "둜" },
  { "둝", "둝", "둝", "둝", "둝" },
  { "둞", "둞", "둞", "둞", "둞" },
  { "둟", "둟", "둟", "둟", "둟" },
  { "둠", "둠", "둠", "둠", "둠" },
  { "둡", "둡", "둡", "둡", "둡" },
  { "둢", "둢", "둢", "둢", "둢" },
  { "둣", "둣", "둣", "둣", "둣" },
  { "둤", "둤", "둤", "둤", "둤" },
  { "둥", "둥", "둥", "둥", "둥" },
  { "둦", "둦", "둦", "둦", "둦" },
  { "둧", "둧", "둧", "둧", "둧" },
  { "둨", "둨", "둨", "둨", "둨" },
  { "둩", "둩", "둩", "둩", "둩" },
  { "둪", "둪", "둪", "둪", "둪" },
  { "둫", "둫", "둫", "둫", "둫" },
  { "둬", "둬", "둬", "둬", "둬" },
  { "둭", "둭", "둭", "둭", "둭" },
  { "둮", "둮", "둮", "둮", "둮" },
  { "둯", "둯", "둯", "둯", "둯" },
  { "둰", "둰", "둰", "둰", "둰" },
  { "둱", "둱", "둱", "둱", "둱" },
  { "둲", "둲", "둲", "둲", "둲" },
  { "둳", "둳", "둳", "둳", "둳" },
  { "둴", "둴", "둴", "둴", "둴" },
  { "둵", "둵", "둵", "둵", "둵" },
  { "둶", "둶", "둶", "둶", "둶" },
  { "둷", "둷", "둷", "둷", "둷" },
  { "둸", "둸", "둸", "둸", "둸" },
  { "둹", "둹", "둹", "둹", "둹" },
  { "둺", "둺", "둺", "둺", "둺" },
  { "둻", "둻", "둻", "둻", "둻" },
  { "둼", "둼", "둼", "둼", "둼" },
  { "둽", "둽", "둽", "둽", "둽" },
  { "둾", "둾", "둾", "둾", "둾" },
  { "둿", "둿", "둿", "둿", "둿" },
  { "뒀", "뒀", "뒀", "뒀", "뒀" },
  { "뒁", "뒁", "뒁", "뒁", "뒁" },
  { "뒂", "뒂", "뒂", "뒂", "뒂" },
  { "뒃", "뒃", "뒃", "뒃", "뒃" },
  { "뒄", "뒄", "뒄", "뒄", "뒄" },
  { "뒅", "뒅", "뒅", "뒅", "뒅" },
  { "뒆", "뒆", "뒆", "뒆", "뒆" },
  { "뒇", "뒇", "뒇", "뒇", "뒇" },
  { "뒈", "뒈", "뒈", "뒈", "뒈" },
  { "뒉", "뒉", "뒉", "뒉", "뒉" },
  { "뒊", "뒊", "뒊", "뒊", "뒊" },
  { "뒋", "뒋", "뒋", "뒋", "뒋" },
  { "뒌", "뒌", "뒌", "뒌", "뒌" },
  { "뒍", "뒍", "뒍", "뒍", "뒍" },
  { "뒎", "뒎", "뒎", "뒎", "뒎" },
  { "뒏", "뒏", "뒏", "뒏", "뒏" },
  { "뒐", "뒐", "뒐", "뒐", "뒐" },
  { "뒑", "뒑", "뒑", "뒑", "뒑" },
  { "뒒", "뒒", "뒒", "뒒", "뒒" },
  { "뒓", "뒓", "뒓", "뒓", "뒓" },
  { "뒔", "뒔", "뒔", "뒔", "뒔" },
  { "뒕", "뒕", "뒕", "뒕", "뒕" },
  { "뒖", "뒖", "뒖", "뒖", "뒖" },
  { "뒗", "뒗", "뒗", "뒗", "뒗" },
  { "뒘", "뒘", "뒘", "뒘", "뒘" },
  { "뒙", "뒙", "뒙", "뒙", "뒙" },
  { "뒚", "뒚", "뒚", "뒚", "뒚" },
  { "뒛", "뒛", "뒛", "뒛", "뒛" },
  { "뒜", "뒜", "뒜", "뒜", "뒜" },
  { "뒝", "뒝", "뒝", "뒝", "뒝" },
  { "뒞", "뒞", "뒞", "뒞", "뒞" },
  { "뒟", "뒟", "뒟", "뒟", "뒟" },
  { "뒠", "뒠", "뒠", "뒠", "뒠" },
  { "뒡", "뒡", "뒡", "뒡", "뒡" },
  { "뒢", "뒢", "뒢", "뒢", "뒢" },
  { "뒣", "뒣", "뒣", "뒣", "뒣" },
  { "뒤", "뒤", "뒤", "뒤", "뒤" },
  { "뒥", "뒥", "뒥", "뒥", "뒥" },
  { "뒦", "뒦", "뒦", "뒦", "뒦" },
  { "뒧", "뒧", "뒧", "뒧", "뒧" },
  { "뒨", "뒨", "뒨", "뒨", "뒨" },
  { "뒩", "뒩", "뒩", "뒩", "뒩" },
  { "뒪", "뒪", "뒪", "뒪", "뒪" },
  { "뒫", "뒫", "뒫", "뒫", "뒫" },
  { "뒬", "뒬", "뒬", "뒬", "뒬" },
  { "뒭", "뒭", "뒭", "뒭", "뒭" },
  { "뒮", "뒮", "뒮", "뒮", "뒮" },
  { "뒯", "뒯", "뒯", "뒯", "뒯" },
  { "뒰", "뒰", "뒰", "뒰", "뒰" },
  { "뒱", "뒱", "뒱", "뒱", "뒱" },
  { "뒲", "뒲", "뒲", "뒲", "뒲" },
  { "뒳", "뒳", "뒳", "뒳", "뒳" },
  { "뒴", "뒴", "뒴", "뒴", "뒴" },
  { "뒵", "뒵", "뒵", "뒵", "뒵" },
  { "뒶", "뒶", "뒶", "뒶", "뒶" },
  { "뒷", "뒷", "뒷", "뒷", "뒷" },
  { "뒸", "뒸", "뒸", "뒸", "뒸" },
  { "뒹", "뒹", "뒹", "뒹", "뒹" },
  { "뒺", "뒺", "뒺", "뒺", "뒺" },
  { "뒻", "뒻", "뒻", "뒻", "뒻" },
  { "뒼", "뒼", "뒼", "뒼", "뒼" },
  { "뒽", "뒽", "뒽", "뒽", "뒽" },
  { "뒾", "뒾", "뒾", "뒾", "뒾" },
  { "뒿", "뒿", "뒿", "뒿", "뒿" },
  { "듀", "듀", "듀", "듀", "듀" },
  { "듁", "듁", "듁", "듁", "듁" },
  { "듂", "듂", "듂", "듂", "듂" },
  { "듃", "듃", "듃", "듃", "듃" },
  { "듄", "듄", "듄", "듄", "듄" },
  { "듅", "듅", "듅", "듅", "듅" },
  { "듆", "듆", "듆", "듆", "듆" },
  { "듇", "듇", "듇", "듇", "듇" },
  { "듈", "듈", "듈", "듈", "듈" },
  { "듉", "듉", "듉", "듉", "듉" },
  { "듊", "듊", "듊", "듊", "듊" },
  { "듋", "듋", "듋", "듋", "듋" },
  { "듌", "듌", "듌", "듌", "듌" },
  { "듍", "듍", "듍", "듍", "듍" },
  { "듎", "듎", "듎", "듎", "듎" },
  { "듏", "듏", "듏", "듏", "듏" },
  { "듐", "듐", "듐", "듐", "듐" },
  { "듑", "듑", "듑", "듑", "듑" },
  { "듒", "듒", "듒", "듒", "듒" },
  { "듓", "듓", "듓", "듓", "듓" },
  { "듔", "듔", "듔", "듔", "듔" },
  { "듕", "듕", "듕", "듕", "듕" },
  { "듖", "듖", "듖", "듖", "듖" },
  { "듗", "듗", "듗", "듗", "듗" },
  { "듘", "듘", "듘", "듘", "듘" },
  { "듙", "듙", "듙", "듙", "듙" },
  { "듚", "듚", "듚", "듚", "듚" },
  { "듛", "듛", "듛", "듛", "듛" },
  { "드", "드", "드", "드", "드" },
  { "득", "득", "득", "득", "득" },
  { "듞", "듞", "듞", "듞", "듞" },
  { "듟", "듟", "듟", "듟", "듟" },
  { "든", "든", "든", "든", "든" },
  { "듡", "듡", "듡", "듡", "듡" },
  { "듢", "듢", "듢", "듢", "듢" },
  { "듣", "듣", "듣", "듣", "듣" },
  { "들", "들", "들", "들", "들" },
  { "듥", "듥", "듥", "듥", "듥" },
  { "듦", "듦", "듦", "듦", "듦" },
  { "듧", "듧", "듧", "듧", "듧" },
  { "듨", "듨", "듨", "듨", "듨" },
  { "듩", "듩", "듩", "듩", "듩" },
  { "듪", "듪", "듪", "듪", "듪" },
  { "듫", "듫", "듫", "듫", "듫" },
  { "듬", "듬", "듬", "듬", "듬" },
  { "듭", "듭", "듭", "듭", "듭" },
  { "듮", "듮", "듮", "듮", "듮" },
  { "듯", "듯", "듯", "듯", "듯" },
  { "듰", "듰", "듰", "듰", "듰" },
  { "등", "등", "등", "등", "등" },
  { "듲", "듲", "듲", "듲", "듲" },
  { "듳", "듳", "듳", "듳", "듳" },
  { "듴", "듴", "듴", "듴", "듴" },
  { "듵", "듵", "듵", "듵", "듵" },
  { "듶", "듶", "듶", "듶", "듶" },
  { "듷", "듷", "듷", "듷", "듷" },
  { "듸", "듸", "듸", "듸", "듸" },
  { "듹", "듹", "듹", "듹", "듹" },
  { "듺", "듺", "듺", "듺", "듺" },
  { "듻", "듻", "듻", "듻", "듻" },
  { "듼", "듼", "듼", "듼", "듼" },
  { "듽", "듽", "듽", "듽", "듽" },
  { "듾", "듾", "듾", "듾", "듾" },
  { "듿", "듿", "듿", "듿", "듿" },
  { "딀", "딀", "딀", "딀", "딀" },
  { "딁", "딁", "딁", "딁", "딁" },
  { "딂", "딂", "딂", "딂", "딂" },
  { "딃", "딃", "딃", "딃", "딃" },
  { "딄", "딄", "딄", "딄", "딄" },
  { "딅", "딅", "딅", "딅", "딅" },
  { "딆", "딆", "딆", "딆", "딆" },
  { "딇", "딇", "딇", "딇", "딇" },
  { "딈", "딈", "딈", "딈", "딈" },
  { "딉", "딉", "딉", "딉", "딉" },
  { "딊", "딊", "딊", "딊", "딊" },
  { "딋", "딋", "딋", "딋", "딋" },
  { "딌", "딌", "딌", "딌", "딌" },
  { "딍", "딍", "딍", "딍", "딍" },
  { "딎", "딎", "딎", "딎", "딎" },
  { "딏", "딏", "딏", "딏", "딏" },
  { "딐", "딐", "딐", "딐", "딐" },
  { "딑", "딑", "딑", "딑", "딑" },
  { "딒", "딒", "딒", "딒", "딒" },
  { "딓", "딓", "딓", "딓", "딓" },
  { "디", "디", "디", "디", "디" },
  { "딕", "딕", "딕", "딕", "딕" },
  { "딖", "딖", "딖", "딖", "딖" },
  { "딗", "딗", "딗", "딗", "딗" },
  { "딘", "딘", "딘", "딘", "딘" },
  { "딙", "딙", "딙", "딙", "딙" },
  { "딚", "딚", "딚", "딚", "딚" },
  { "딛", "딛", "딛", "딛", "딛" },
  { "딜", "딜", "딜", "딜", "딜" },
  { "딝", "딝", "딝", "딝", "딝" },
  { "딞", "딞", "딞", "딞", "딞" },
  { "딟", "딟", "딟", "딟", "딟" },
  { "딠", "딠", "딠", "딠", "딠" },
  { "딡", "딡", "딡", "딡", "딡" },
  { "딢", "딢", "딢", "딢", "딢" },
  { "딣", "딣", "딣", "딣", "딣" },
  { "딤", "딤", "딤", "딤", "딤" },
  { "딥", "딥", "딥", "딥", "딥" },
  { "딦", "딦", "딦", "딦", "딦" },
  { "딧", "딧", "딧", "딧", "딧" },
  { "딨", "딨", "딨", "딨", "딨" },
  { "딩", "딩", "딩", "딩", "딩" },
  { "딪", "딪", "딪", "딪", "딪" },
  { "딫", "딫", "딫", "딫", "딫" },
  { "딬", "딬", "딬", "딬", "딬" },
  { "딭", "딭", "딭", "딭", "딭" },
  { "딮", "딮", "딮", "딮", "딮" },
  { "딯", "딯", "딯", "딯", "딯" },
  { "따", "따", "따", "따", "따" },
  { "딱", "딱", "딱", "딱", "딱" },
  { "딲", "딲", "딲", "딲", "딲" },
  { "딳", "딳", "딳", "딳", "딳" },
  { "딴", "딴", "딴", "딴", "딴" },
  { "딵", "딵", "딵", "딵", "딵" },
  { "딶", "딶", "딶", "딶", "딶" },
  { "딷", "딷", "딷", "딷", "딷" },
  { "딸", "딸", "딸", "딸", "딸" },
  { "딹", "딹", "딹", "딹", "딹" },
  { "딺", "딺", "딺", "딺", "딺" },
  { "딻", "딻", "딻", "딻", "딻" },
  { "딼", "딼", "딼", "딼", "딼" },
  { "딽", "딽", "딽", "딽", "딽" },
  { "딾", "딾", "딾", "딾", "딾" },
  { "딿", "딿", "딿", "딿", "딿" },
  { "땀", "땀", "땀", "땀", "땀" },
  { "땁", "땁", "땁", "땁", "땁" },
  { "땂", "땂", "땂", "땂", "땂" },
  { "땃", "땃", "땃", "땃", "땃" },
  { "땄", "땄", "땄", "땄", "땄" },
  { "땅", "땅", "땅", "땅", "땅" },
  { "땆", "땆", "땆", "땆", "땆" },
  { "땇", "땇", "땇", "땇", "땇" },
  { "땈", "땈", "땈", "땈", "땈" },
  { "땉", "땉", "땉", "땉", "땉" },
  { "땊", "땊", "땊", "땊", "땊" },
  { "땋", "땋", "땋", "땋", "땋" },
  { "때", "때", "때", "때", "때" },
  { "땍", "땍", "땍", "땍", "땍" },
  { "땎", "땎", "땎", "땎", "땎" },
  { "땏", "땏", "땏", "땏", "땏" },
  { "땐", "땐", "땐", "땐", "땐" },
  { "땑", "땑", "땑", "땑", "땑" },
  { "땒", "땒", "땒", "땒", "땒" },
  { "땓", "땓", "땓", "땓", "땓" },
  { "땔", "땔", "땔", "땔", "땔" },
  { "땕", "땕", "땕", "땕", "땕" },
  { "땖", "땖", "땖", "땖", "땖" },
  { "땗", "땗", "땗", "땗", "땗" },
  { "땘", "땘", "땘", "땘", "땘" },
  { "땙", "땙", "땙", "땙", "땙" },
  { "땚", "땚", "땚", "땚", "땚" },
  { "땛", "땛", "땛", "땛", "땛" },
  { "땜", "땜", "땜", "땜", "땜" },
  { "땝", "땝", "땝", "땝", "땝" },
  { "땞", "땞", "땞", "땞", "땞" },
  { "땟", "땟", "땟", "땟", "땟" },
  { "땠", "땠", "땠", "땠", "땠" },
  { "땡", "땡", "땡", "땡", "땡" },
  { "땢", "땢", "땢", "땢", "땢" },
  { "땣", "땣", "땣", "땣", "땣" },
  { "땤", "땤", "땤", "땤", "땤" },
  { "땥", "땥", "땥", "땥", "땥" },
  { "땦", "땦", "땦", "땦", "땦" },
  { "땧", "땧", "땧", "땧", "땧" },
  { "땨", "땨", "땨", "땨", "땨" },
  { "땩", "땩", "땩", "땩", "땩" },
  { "땪", "땪", "땪", "땪", "땪" },
  { "땫", "땫", "땫", "땫", "땫" },
  { "땬", "땬", "땬", "땬", "땬" },
  { "땭", "땭", "땭", "땭", "땭" },
  { "땮", "땮", "땮", "땮", "땮" },
  { "땯", "땯", "땯", "땯", "땯" },
  { "땰", "땰", "땰", "땰", "땰" },
  { "땱", "땱", "땱", "땱", "땱" },
  { "땲", "땲", "땲", "땲", "땲" },
  { "땳", "땳", "땳", "땳", "땳" },
  { "땴", "땴", "땴", "땴", "땴" },
  { "땵", "땵", "땵", "땵", "땵" },
  { "땶", "땶", "땶", "땶", "땶" },
  { "땷", "땷", "땷", "땷", "땷" },
  { "땸", "땸", "땸", "땸", "땸" },
  { "땹", "땹", "땹", "땹", "땹" },
  { "땺", "땺", "땺", "땺", "땺" },
  { "땻", "땻", "땻", "땻", "땻" },
  { "땼", "땼", "땼", "땼", "땼" },
  { "땽", "땽", "땽", "땽", "땽" },
  { "땾", "땾", "땾", "땾", "땾" },
  { "땿", "땿", "땿", "땿", "땿" },
  { "떀", "떀", "떀", "떀", "떀" },
  { "떁", "떁", "떁", "떁", "떁" },
  { "떂", "떂", "떂", "떂", "떂" },
  { "떃", "떃", "떃", "떃", "떃" },
  { "떄", "떄", "떄", "떄", "떄" },
  { "떅", "떅", "떅", "떅", "떅" },
  { "떆", "떆", "떆", "떆", "떆" },
  { "떇", "떇", "떇", "떇", "떇" },
  { "떈", "떈", "떈", "떈", "떈" },
  { "떉", "떉", "떉", "떉", "떉" },
  { "떊", "떊", "떊", "떊", "떊" },
  { "떋", "떋", "떋", "떋", "떋" },
  { "떌", "떌", "떌", "떌", "떌" },
  { "떍", "떍", "떍", "떍", "떍" },
  { "떎", "떎", "떎", "떎", "떎" },
  { "떏", "떏", "떏", "떏", "떏" },
  { "떐", "떐", "떐", "떐", "떐" },
  { "떑", "떑", "떑", "떑", "떑" },
  { "떒", "떒", "떒", "떒", "떒" },
  { "떓", "떓", "떓", "떓", "떓" },
  { "떔", "떔", "떔", "떔", "떔" },
  { "떕", "떕", "떕", "떕", "떕" },
  { "떖", "떖", "떖", "떖", "떖" },
  { "떗", "떗", "떗", "떗", "떗" },
  { "떘", "떘", "떘", "떘", "떘" },
  { "떙", "떙", "떙", "떙", "떙" },
  { "떚", "떚", "떚", "떚", "떚" },
  { "떛", "떛", "떛", "떛", "떛" },
  { "떜", "떜", "떜", "떜", "떜" },
  { "떝", "떝", "떝", "떝", "떝" },
  { "떞", "떞", "떞", "떞", "떞" },
  { "떟", "떟", "떟", "떟", "떟" },
  { "떠", "떠", "떠", "떠", "떠" },
  { "떡", "떡", "떡", "떡", "떡" },
  { "떢", "떢", "떢", "떢", "떢" },
  { "떣", "떣", "떣", "떣", "떣" },
  { "떤", "떤", "떤", "떤", "떤" },
  { "떥", "떥", "떥", "떥", "떥" },
  { "떦", "떦", "떦", "떦", "떦" },
  { "떧", "떧", "떧", "떧", "떧" },
  { "떨", "떨", "떨", "떨", "떨" },
  { "떩", "떩", "떩", "떩", "떩" },
  { "떪", "떪", "떪", "떪", "떪" },
  { "떫", "떫", "떫", "떫", "떫" },
  { "떬", "떬", "떬", "떬", "떬" },
  { "떭", "떭", "떭", "떭", "떭" },
  { "떮", "떮", "떮", "떮", "떮" },
  { "떯", "떯", "떯", "떯", "떯" },
  { "떰", "떰", "떰", "떰", "떰" },
  { "떱", "떱", "떱", "떱", "떱" },
  { "떲", "떲", "떲", "떲", "떲" },
  { "떳", "떳", "떳", "떳", "떳" },
  { "떴", "떴", "떴", "떴", "떴" },
  { "떵", "떵", "떵", "떵", "떵" },
  { "떶", "떶", "떶", "떶", "떶" },
  { "떷", "떷", "떷", "떷", "떷" },
  { "떸", "떸", "떸", "떸", "떸" },
  { "떹", "떹", "떹", "떹", "떹" },
  { "떺", "떺", "떺", "떺", "떺" },
  { "떻", "떻", "떻", "떻", "떻" },
  { "떼", "떼", "떼", "떼", "떼" },
  { "떽", "떽", "떽", "떽", "떽" },
  { "떾", "떾", "떾", "떾", "떾" },
  { "떿", "떿", "떿", "떿", "떿" },
  { "뗀", "뗀", "뗀", "뗀", "뗀" },
  { "뗁", "뗁", "뗁", "뗁", "뗁" },
  { "뗂", "뗂", "뗂", "뗂", "뗂" },
  { "뗃", "뗃", "뗃", "뗃", "뗃" },
  { "뗄", "뗄", "뗄", "뗄", "뗄" },
  { "뗅", "뗅", "뗅", "뗅", "뗅" },
  { "뗆", "뗆", "뗆", "뗆", "뗆" },
  { "뗇", "뗇", "뗇", "뗇", "뗇" },
  { "뗈", "뗈", "뗈", "뗈", "뗈" },
  { "뗉", "뗉", "뗉", "뗉", "뗉" },
  { "뗊", "뗊", "뗊", "뗊", "뗊" },
  { "뗋", "뗋", "뗋", "뗋", "뗋" },
  { "뗌", "뗌", "뗌", "뗌", "뗌" },
  { "뗍", "뗍", "뗍", "뗍", "뗍" },
  { "뗎", "뗎", "뗎", "뗎", "뗎" },
  { "뗏", "뗏", "뗏", "뗏", "뗏" },
  { "뗐", "뗐", "뗐", "뗐", "뗐" },
  { "뗑", "뗑", "뗑", "뗑", "뗑" },
  { "뗒", "뗒", "뗒", "뗒", "뗒" },
  { "뗓", "뗓", "뗓", "뗓", "뗓" },
  { "뗔", "뗔", "뗔", "뗔", "뗔" },
  { "뗕", "뗕", "뗕", "뗕", "뗕" },
  { "뗖", "뗖", "뗖", "뗖", "뗖" },
  { "뗗", "뗗", "뗗", "뗗", "뗗" },
  { "뗘", "뗘", "뗘", "뗘", "뗘" },
  { "뗙", "뗙", "뗙", "뗙", "뗙" },
  { "뗚", "뗚", "뗚", "뗚", "뗚" },
  { "뗛", "뗛", "뗛", "뗛", "뗛" },
  { "뗜", "뗜", "뗜", "뗜", "뗜" },
  { "뗝", "뗝", "뗝", "뗝", "뗝" },
  { "뗞", "뗞", "뗞", "뗞", "뗞" },
  { "뗟", "뗟", "뗟", "뗟", "뗟" },
  { "뗠", "뗠", "뗠", "뗠", "뗠" },
  { "뗡", "뗡", "뗡", "뗡", "뗡" },
  { "뗢", "뗢", "뗢", "뗢", "뗢" },
  { "뗣", "뗣", "뗣", "뗣", "뗣" },
  { "뗤", "뗤", "뗤", "뗤", "뗤" },
  { "뗥", "뗥", "뗥", "뗥", "뗥" },
  { "뗦", "뗦", "뗦", "뗦", "뗦" },
  { "뗧", "뗧", "뗧", "뗧", "뗧" },
  { "뗨", "뗨", "뗨", "뗨", "뗨" },
  { "뗩", "뗩", "뗩", "뗩", "뗩" },
  { "뗪", "뗪", "뗪", "뗪", "뗪" },
  { "뗫", "뗫", "뗫", "뗫", "뗫" },
  { "뗬", "뗬", "뗬", "뗬", "뗬" },
  { "뗭", "뗭", "뗭", "뗭", "뗭" },
  { "뗮", "뗮", "뗮", "뗮", "뗮" },
  { "뗯", "뗯", "뗯", "뗯", "뗯" },
  { "뗰", "뗰", "뗰", "뗰", "뗰" },
  { "뗱", "뗱", "뗱", "뗱", "뗱" },
  { "뗲", "뗲", "뗲", "뗲", "뗲" },
  { "뗳", "뗳", "뗳", "뗳", "뗳" },
  { "뗴", "뗴", "뗴", "뗴", "뗴" },
  { "뗵", "뗵", "뗵", "뗵", "뗵" },
  { "뗶", "뗶", "뗶", "뗶", "뗶" },
  { "뗷", "뗷", "뗷", "뗷", "뗷" },
  { "뗸", "뗸", "뗸", "뗸", "뗸" },
  { "뗹", "뗹", "뗹", "뗹", "뗹" },
  { "뗺", "뗺", "뗺", "뗺", "뗺" },
  { "뗻", "뗻", "뗻", "뗻", "뗻" },
  { "뗼", "뗼", "뗼", "뗼", "뗼" },
  { "뗽", "뗽", "뗽", "뗽", "뗽" },
  { "뗾", "뗾", "뗾", "뗾", "뗾" },
  { "뗿", "뗿", "뗿", "뗿", "뗿" },
  { "똀", "똀", "똀", "똀", "똀" },
  { "똁", "똁", "똁", "똁", "똁" },
  { "똂", "똂", "똂", "똂", "똂" },
  { "똃", "똃", "똃", "똃", "똃" },
  { "똄", "똄", "똄", "똄", "똄" },
  { "똅", "똅", "똅", "똅", "똅" },
  { "똆", "똆", "똆", "똆", "똆" },
  { "똇", "똇", "똇", "똇", "똇" },
  { "똈", "똈", "똈", "똈", "똈" },
  { "똉", "똉", "똉", "똉", "똉" },
  { "똊", "똊", "똊", "똊", "똊" },
  { "똋", "똋", "똋", "똋", "똋" },
  { "똌", "똌", "똌", "똌", "똌" },
  { "똍", "똍", "똍", "똍", "똍" },
  { "똎", "똎", "똎", "똎", "똎" },
  { "똏", "똏", "똏", "똏", "똏" },
  { "또", "또", "또", "또", "또" },
  { "똑", "똑", "똑", "똑", "똑" },
  { "똒", "똒", "똒", "똒", "똒" },
  { "똓", "똓", "똓", "똓", "똓" },
  { "똔", "똔", "똔", "똔", "똔" },
  { "똕", "똕", "똕", "똕", "똕" },
  { "똖", "똖", "똖", "똖", "똖" },
  { "똗", "똗", "똗", "똗", "똗" },
  { "똘", "똘", "똘", "똘", "똘" },
  { "똙", "똙", "똙", "똙", "똙" },
  { "똚", "똚", "똚", "똚", "똚" },
  { "똛", "똛", "똛", "똛", "똛" },
  { "똜", "똜", "똜", "똜", "똜" },
  { "똝", "똝", "똝", "똝", "똝" },
  { "똞", "똞", "똞", "똞", "똞" },
  { "똟", "똟", "똟", "똟", "똟" },
  { "똠", "똠", "똠", "똠", "똠" },
  { "똡", "똡", "똡", "똡", "똡" },
  { "똢", "똢", "똢", "똢", "똢" },
  { "똣", "똣", "똣", "똣", "똣" },
  { "똤", "똤", "똤", "똤", "똤" },
  { "똥", "똥", "똥", "똥", "똥" },
  { "똦", "똦", "똦", "똦", "똦" },
  { "똧", "똧", "똧", "똧", "똧" },
  { "똨", "똨", "똨", "똨", "똨" },
  { "똩", "똩", "똩", "똩", "똩" },
  { "똪", "똪", "똪", "똪", "똪" },
  { "똫", "똫", "똫", "똫", "똫" },
  { "똬", "똬", "똬", "똬", "똬" },
  { "똭", "똭", "똭", "똭", "똭" },
  { "똮", "똮", "똮", "똮", "똮" },
  { "똯", "똯", "똯", "똯", "똯" },
  { "똰", "똰", "똰", "똰", "똰" },
  { "똱", "똱", "똱", "똱", "똱" },
  { "똲", "똲", "똲", "똲", "똲" },
  { "똳", "똳", "똳", "똳", "똳" },
  { "똴", "똴", "똴", "똴", "똴" },
  { "똵", "똵", "똵", "똵", "똵" },
  { "똶", "똶", "똶", "똶", "똶" },
  { "똷", "똷", "똷", "똷", "똷" },
  { "똸", "똸", "똸", "똸", "똸" },
  { "똹", "똹", "똹", "똹", "똹" },
  { "똺", "똺", "똺", "똺", "똺" },
  { "똻", "똻", "똻", "똻", "똻" },
  { "똼", "똼", "똼", "똼", "똼" },
  { "똽", "똽", "똽", "똽", "똽" },
  { "똾", "똾", "똾", "똾", "똾" },
  { "똿", "똿", "똿", "똿", "똿" },
  { "뙀", "뙀", "뙀", "뙀", "뙀" },
  { "뙁", "뙁", "뙁", "뙁", "뙁" },
  { "뙂", "뙂", "뙂", "뙂", "뙂" },
  { "뙃", "뙃", "뙃", "뙃", "뙃" },
  { "뙄", "뙄", "뙄", "뙄", "뙄" },
  { "뙅", "뙅", "뙅", "뙅", "뙅" },
  { "뙆", "뙆", "뙆", "뙆", "뙆" },
  { "뙇", "뙇", "뙇", "뙇", "뙇" },
  { "뙈", "뙈", "뙈", "뙈", "뙈" },
  { "뙉", "뙉", "뙉", "뙉", "뙉" },
  { "뙊", "뙊", "뙊", "뙊", "뙊" },
  { "뙋", "뙋", "뙋", "뙋", "뙋" },
  { "뙌", "뙌", "뙌", "뙌", "뙌" },
  { "뙍", "뙍", "뙍", "뙍", "뙍" },
  { "뙎", "뙎", "뙎", "뙎", "뙎" },
  { "뙏", "뙏", "뙏", "뙏", "뙏" },
  { "뙐", "뙐", "뙐", "뙐", "뙐" },
  { "뙑", "뙑", "뙑", "뙑", "뙑" },
  { "뙒", "뙒", "뙒", "뙒", "뙒" },
  { "뙓", "뙓", "뙓", "뙓", "뙓" },
  { "뙔", "뙔", "뙔", "뙔", "뙔" },
  { "뙕", "뙕", "뙕", "뙕", "뙕" },
  { "뙖", "뙖", "뙖", "뙖", "뙖" },
  { "뙗", "뙗", "뙗", "뙗", "뙗" },
  { "뙘", "뙘", "뙘", "뙘", "뙘" },
  { "뙙", "뙙", "뙙", "뙙", "뙙" },
  { "뙚", "뙚", "뙚", "뙚", "뙚" },
  { "뙛", "뙛", "뙛", "뙛", "뙛" },
  { "뙜", "뙜", "뙜", "뙜", "뙜" },
  { "뙝", "뙝", "뙝", "뙝", "뙝" },
  { "뙞", "뙞", "뙞", "뙞", "뙞" },
  { "뙟", "뙟", "뙟", "뙟", "뙟" },
  { "뙠", "뙠", "뙠", "뙠", "뙠" },
  { "뙡", "뙡", "뙡", "뙡", "뙡" },
  { "뙢", "뙢", "뙢", "뙢", "뙢" },
  { "뙣", "뙣", "뙣", "뙣", "뙣" },
  { "뙤", "뙤", "뙤", "뙤", "뙤" },
  { "뙥", "뙥", "뙥", "뙥", "뙥" },
  { "뙦", "뙦", "뙦", "뙦", "뙦" },
  { "뙧", "뙧", "뙧", "뙧", "뙧" },
  { "뙨", "뙨", "뙨", "뙨", "뙨" },
  { "뙩", "뙩", "뙩", "뙩", "뙩" },
  { "뙪", "뙪", "뙪", "뙪", "뙪" },
  { "뙫", "뙫", "뙫", "뙫", "뙫" },
  { "뙬", "뙬", "뙬", "뙬", "뙬" },
  { "뙭", "뙭", "뙭", "뙭", "뙭" },
  { "뙮", "뙮", "뙮", "뙮", "뙮" },
  { "뙯", "뙯", "뙯", "뙯", "뙯" },
  { "뙰", "뙰", "뙰", "뙰", "뙰" },
  { "뙱", "뙱", "뙱", "뙱", "뙱" },
  { "뙲", "뙲", "뙲", "뙲", "뙲" },
  { "뙳", "뙳", "뙳", "뙳", "뙳" },
  { "뙴", "뙴", "뙴", "뙴", "뙴" },
  { "뙵", "뙵", "뙵", "뙵", "뙵" },
  { "뙶", "뙶", "뙶", "뙶", "뙶" },
  { "뙷", "뙷", "뙷", "뙷", "뙷" },
  { "뙸", "뙸", "뙸", "뙸", "뙸" },
  { "뙹", "뙹", "뙹", "뙹", "뙹" },
  { "뙺", "뙺", "뙺", "뙺", "뙺" },
  { "뙻", "뙻", "뙻", "뙻", "뙻" },
  { "뙼", "뙼", "뙼", "뙼", "뙼" },
  { "뙽", "뙽", "뙽", "뙽", "뙽" },
  { "뙾", "뙾", "뙾", "뙾", "뙾" },
  { "뙿", "뙿", "뙿", "뙿", "뙿" },
  { "뚀", "뚀", "뚀", "뚀", "뚀" },
  { "뚁", "뚁", "뚁", "뚁", "뚁" },
  { "뚂", "뚂", "뚂", "뚂", "뚂" },
  { "뚃", "뚃", "뚃", "뚃", "뚃" },
  { "뚄", "뚄", "뚄", "뚄", "뚄" },
  { "뚅", "뚅", "뚅", "뚅", "뚅" },
  { "뚆", "뚆", "뚆", "뚆", "뚆" },
  { "뚇", "뚇", "뚇", "뚇", "뚇" },
  { "뚈", "뚈", "뚈", "뚈", "뚈" },
  { "뚉", "뚉", "뚉", "뚉", "뚉" },
  { "뚊", "뚊", "뚊", "뚊", "뚊" },
  { "뚋", "뚋", "뚋", "뚋", "뚋" },
  { "뚌", "뚌", "뚌", "뚌", "뚌" },
  { "뚍", "뚍", "뚍", "뚍", "뚍" },
  { "뚎", "뚎", "뚎", "뚎", "뚎" },
  { "뚏", "뚏", "뚏", "뚏", "뚏" },
  { "뚐", "뚐", "뚐", "뚐", "뚐" },
  { "뚑", "뚑", "뚑", "뚑", "뚑" },
  { "뚒", "뚒", "뚒", "뚒", "뚒" },
  { "뚓", "뚓", "뚓", "뚓", "뚓" },
  { "뚔", "뚔", "뚔", "뚔", "뚔" },
  { "뚕", "뚕", "뚕", "뚕", "뚕" },
  { "뚖", "뚖", "뚖", "뚖", "뚖" },
  { "뚗", "뚗", "뚗", "뚗", "뚗" },
  { "뚘", "뚘", "뚘", "뚘", "뚘" },
  { "뚙", "뚙", "뚙", "뚙", "뚙" },
  { "뚚", "뚚", "뚚", "뚚", "뚚" },
  { "뚛", "뚛", "뚛", "뚛", "뚛" },
  { "뚜", "뚜", "뚜", "뚜", "뚜" },
  { "뚝", "뚝", "뚝", "뚝", "뚝" },
  { "뚞", "뚞", "뚞", "뚞", "뚞" },
  { "뚟", "뚟", "뚟", "뚟", "뚟" },
  { "뚠", "뚠", "뚠", "뚠", "뚠" },
  { "뚡", "뚡", "뚡", "뚡", "뚡" },
  { "뚢", "뚢", "뚢", "뚢", "뚢" },
  { "뚣", "뚣", "뚣", "뚣", "뚣" },
  { "뚤", "뚤", "뚤", "뚤", "뚤" },
  { "뚥", "뚥", "뚥", "뚥", "뚥" },
  { "뚦", "뚦", "뚦", "뚦", "뚦" },
  { "뚧", "뚧", "뚧", "뚧", "뚧" },
  { "뚨", "뚨", "뚨", "뚨", "뚨" },
  { "뚩", "뚩", "뚩", "뚩", "뚩" },
  { "뚪", "뚪", "뚪", "뚪", "뚪" },
  { "뚫", "뚫", "뚫", "뚫", "뚫" },
  { "뚬", "뚬", "뚬", "뚬", "뚬" },
  { "뚭", "뚭", "뚭", "뚭", "뚭" },
  { "뚮", "뚮", "뚮", "뚮", "뚮" },
  { "뚯", "뚯", "뚯", "뚯", "뚯" },
  { "뚰", "뚰", "뚰", "뚰", "뚰" },
  { "뚱", "뚱", "뚱", "뚱", "뚱" },
  { "뚲", "뚲", "뚲", "뚲", "뚲" },
  { "뚳", "뚳", "뚳", "뚳", "뚳" },
  { "뚴", "뚴", "뚴", "뚴", "뚴" },
  { "뚵", "뚵", "뚵", "뚵", "뚵" },
  { "뚶", "뚶", "뚶", "뚶", "뚶" },
  { "뚷", "뚷", "뚷", "뚷", "뚷" },
  { "뚸", "뚸", "뚸", "뚸", "뚸" },
  { "뚹", "뚹", "뚹", "뚹", "뚹" },
  { "뚺", "뚺", "뚺", "뚺", "뚺" },
  { "뚻", "뚻", "뚻", "뚻", "뚻" },
  { "뚼", "뚼", "뚼", "뚼", "뚼" },
  { "뚽", "뚽", "뚽", "뚽", "뚽" },
  { "뚾", "뚾", "뚾", "뚾", "뚾" },
  { "뚿", "뚿", "뚿", "뚿", "뚿" },
  { "뛀", "뛀", "뛀", "뛀", "뛀" },
  { "뛁", "뛁", "뛁", "뛁", "뛁" },
  { "뛂", "뛂", "뛂", "뛂", "뛂" },
  { "뛃", "뛃", "뛃", "뛃", "뛃" },
  { "뛄", "뛄", "뛄", "뛄", "뛄" },
  { "뛅", "뛅", "뛅", "뛅", "뛅" },
  { "뛆", "뛆", "뛆", "뛆", "뛆" },
  { "뛇", "뛇", "뛇", "뛇", "뛇" },
  { "뛈", "뛈", "뛈", "뛈", "뛈" },
  { "뛉", "뛉", "뛉", "뛉", "뛉" },
  { "뛊", "뛊", "뛊", "뛊", "뛊" },
  { "뛋", "뛋", "뛋", "뛋", "뛋" },
  { "뛌", "뛌", "뛌", "뛌", "뛌" },
  { "뛍", "뛍", "뛍", "뛍", "뛍" },
  { "뛎", "뛎", "뛎", "뛎", "뛎" },
  { "뛏", "뛏", "뛏", "뛏", "뛏" },
  { "뛐", "뛐", "뛐", "뛐", "뛐" },
  { "뛑", "뛑", "뛑", "뛑", "뛑" },
  { "뛒", "뛒", "뛒", "뛒", "뛒" },
  { "뛓", "뛓", "뛓", "뛓", "뛓" },
  { "뛔", "뛔", "뛔", "뛔", "뛔" },
  { "뛕", "뛕", "뛕", "뛕", "뛕" },
  { "뛖", "뛖", "뛖", "뛖", "뛖" },
  { "뛗", "뛗", "뛗", "뛗", "뛗" },
  { "뛘", "뛘", "뛘", "뛘", "뛘" },
  { "뛙", "뛙", "뛙", "뛙", "뛙" },
  { "뛚", "뛚", "뛚", "뛚", "뛚" },
  { "뛛", "뛛", "뛛", "뛛", "뛛" },
  { "뛜", "뛜", "뛜", "뛜", "뛜" },
  { "뛝", "뛝", "뛝", "뛝", "뛝" },
  { "뛞", "뛞", "뛞", "뛞", "뛞" },
  { "뛟", "뛟", "뛟", "뛟", "뛟" },
  { "뛠", "뛠", "뛠", "뛠", "뛠" },
  { "뛡", "뛡", "뛡", "뛡", "뛡" },
  { "뛢", "뛢", "뛢", "뛢", "뛢" },
  { "뛣", "뛣", "뛣", "뛣", "뛣" },
  { "뛤", "뛤", "뛤", "뛤", "뛤" },
  { "뛥", "뛥", "뛥", "뛥", "뛥" },
  { "뛦", "뛦", "뛦", "뛦", "뛦" },
  { "뛧", "뛧", "뛧", "뛧", "뛧" },
  { "뛨", "뛨", "뛨", "뛨", "뛨" },
  { "뛩", "뛩", "뛩", "뛩", "뛩" },
  { "뛪", "뛪", "뛪", "뛪", "뛪" },
  { "뛫", "뛫", "뛫", "뛫", "뛫" },
  { "뛬", "뛬", "뛬", "뛬", "뛬" },
  { "뛭", "뛭", "뛭", "뛭", "뛭" },
  { "뛮", "뛮", "뛮", "뛮", "뛮" },
  { "뛯", "뛯", "뛯", "뛯", "뛯" },
  { "뛰", "뛰", "뛰", "뛰", "뛰" },
  { "뛱", "뛱", "뛱", "뛱", "뛱" },
  { "뛲", "뛲", "뛲", "뛲", "뛲" },
  { "뛳", "뛳", "뛳", "뛳", "뛳" },
  { "뛴", "뛴", "뛴", "뛴", "뛴" },
  { "뛵", "뛵", "뛵", "뛵", "뛵" },
  { "뛶", "뛶", "뛶", "뛶", "뛶" },
  { "뛷", "뛷", "뛷", "뛷", "뛷" },
  { "뛸", "뛸", "뛸", "뛸", "뛸" },
  { "뛹", "뛹", "뛹", "뛹", "뛹" },
  { "뛺", "뛺", "뛺", "뛺", "뛺" },
  { "뛻", "뛻", "뛻", "뛻", "뛻" },
  { "뛼", "뛼", "뛼", "뛼", "뛼" },
  { "뛽", "뛽", "뛽", "뛽", "뛽" },
  { "뛾", "뛾", "뛾", "뛾", "뛾" },
  { "뛿", "뛿", "뛿", "뛿", "뛿" },
  { "뜀", "뜀", "뜀", "뜀", "뜀" },
  { "뜁", "뜁", "뜁", "뜁", "뜁" },
  { "뜂", "뜂", "뜂", "뜂", "뜂" },
  { "뜃", "뜃", "뜃", "뜃", "뜃" },
  { "뜄", "뜄", "뜄", "뜄", "뜄" },
  { "뜅", "뜅", "뜅", "뜅", "뜅" },
  { "뜆", "뜆", "뜆", "뜆", "뜆" },
  { "뜇", "뜇", "뜇", "뜇", "뜇" },
  { "뜈", "뜈", "뜈", "뜈", "뜈" },
  { "뜉", "뜉", "뜉", "뜉", "뜉" },
  { "뜊", "뜊", "뜊", "뜊", "뜊" },
  { "뜋", "뜋", "뜋", "뜋", "뜋" },
  { "뜌", "뜌", "뜌", "뜌", "뜌" },
  { "뜍", "뜍", "뜍", "뜍", "뜍" },
  { "뜎", "뜎", "뜎", "뜎", "뜎" },
  { "뜏", "뜏", "뜏", "뜏", "뜏" },
  { "뜐", "뜐", "뜐", "뜐", "뜐" },
  { "뜑", "뜑", "뜑", "뜑", "뜑" },
  { "뜒", "뜒", "뜒", "뜒", "뜒" },
  { "뜓", "뜓", "뜓", "뜓", "뜓" },
  { "뜔", "뜔", "뜔", "뜔", "뜔" },
  { "뜕", "뜕", "뜕", "뜕", "뜕" },
  { "뜖", "뜖", "뜖", "뜖", "뜖" },
  { "뜗", "뜗", "뜗", "뜗", "뜗" },
  { "뜘", "뜘", "뜘", "뜘", "뜘" },
  { "뜙", "뜙", "뜙", "뜙", "뜙" },
  { "뜚", "뜚", "뜚", "뜚", "뜚" },
  { "뜛", "뜛", "뜛", "뜛", "뜛" },
  { "뜜", "뜜", "뜜", "뜜", "뜜" },
  { "뜝", "뜝", "뜝", "뜝", "뜝" },
  { "뜞", "뜞", "뜞", "뜞", "뜞" },
  { "뜟", "뜟", "뜟", "뜟", "뜟" },
  { "뜠", "뜠", "뜠", "뜠", "뜠" },
  { "뜡", "뜡", "뜡", "뜡", "뜡" },
  { "뜢", "뜢", "뜢", "뜢", "뜢" },
  { "뜣", "뜣", "뜣", "뜣", "뜣" },
  { "뜤", "뜤", "뜤", "뜤", "뜤" },
  { "뜥", "뜥", "뜥", "뜥", "뜥" },
  { "뜦", "뜦", "뜦", "뜦", "뜦" },
  { "뜧", "뜧", "뜧", "뜧", "뜧" },
  { "뜨", "뜨", "뜨", "뜨", "뜨" },
  { "뜩", "뜩", "뜩", "뜩", "뜩" },
  { "뜪", "뜪", "뜪", "뜪", "뜪" },
  { "뜫", "뜫", "뜫", "뜫", "뜫" },
  { "뜬", "뜬", "뜬", "뜬", "뜬" },
  { "뜭", "뜭", "뜭", "뜭", "뜭" },
  { "뜮", "뜮", "뜮", "뜮", "뜮" },
  { "뜯", "뜯", "뜯", "뜯", "뜯" },
  { "뜰", "뜰", "뜰", "뜰", "뜰" },
  { "뜱", "뜱", "뜱", "뜱", "뜱" },
  { "뜲", "뜲", "뜲", "뜲", "뜲" },
  { "뜳", "뜳", "뜳", "뜳", "뜳" },
  { "뜴", "뜴", "뜴", "뜴", "뜴" },
  { "뜵", "뜵", "뜵", "뜵", "뜵" },
  { "뜶", "뜶", "뜶", "뜶", "뜶" },
  { "뜷", "뜷", "뜷", "뜷", "뜷" },
  { "뜸", "뜸", "뜸", "뜸", "뜸" },
  { "뜹", "뜹", "뜹", "뜹", "뜹" },
  { "뜺", "뜺", "뜺", "뜺", "뜺" },
  { "뜻", "뜻", "뜻", "뜻", "뜻" },
  { "뜼", "뜼", "뜼", "뜼", "뜼" },
  { "뜽", "뜽", "뜽", "뜽", "뜽" },
  { "뜾", "뜾", "뜾", "뜾", "뜾" },
  { "뜿", "뜿", "뜿", "뜿", "뜿" },
  { "띀", "띀", "띀", "띀", "띀" },
  { "띁", "띁", "띁", "띁", "띁" },
  { "띂", "띂", "띂", "띂", "띂" },
  { "띃", "띃", "띃", "띃", "띃" },
  { "띄", "띄", "띄", "띄", "띄" },
  { "띅", "띅", "띅", "띅", "띅" },
  { "띆", "띆", "띆", "띆", "띆" },
  { "띇", "띇", "띇", "띇", "띇" },
  { "띈", "띈", "띈", "띈", "띈" },
  { "띉", "띉", "띉", "띉", "띉" },
  { "띊", "띊", "띊", "띊", "띊" },
  { "띋", "띋", "띋", "띋", "띋" },
  { "띌", "띌", "띌", "띌", "띌" },
  { "띍", "띍", "띍", "띍", "띍" },
  { "띎", "띎", "띎", "띎", "띎" },
  { "띏", "띏", "띏", "띏", "띏" },
  { "띐", "띐", "띐", "띐", "띐" },
  { "띑", "띑", "띑", "띑", "띑" },
  { "띒", "띒", "띒", "띒", "띒" },
  { "띓", "띓", "띓", "띓", "띓" },
  { "띔", "띔", "띔", "띔", "띔" },
  { "띕", "띕", "띕", "띕", "띕" },
  { "띖", "띖", "띖", "띖", "띖" },
  { "띗", "띗", "띗", "띗", "띗" },
  { "띘", "띘", "띘", "띘", "띘" },
  { "띙", "띙", "띙", "띙", "띙" },
  { "띚", "띚", "띚", "띚", "띚" },
  { "띛", "띛", "띛", "띛", "띛" },
  { "띜", "띜", "띜", "띜", "띜" },
  { "띝", "띝", "띝", "띝", "띝" },
  { "띞", "띞", "띞", "띞", "띞" },
  { "띟", "띟", "띟", "띟", "띟" },
  { "띠", "띠", "띠", "띠", "띠" },
  { "띡", "띡", "띡", "띡", "띡" },
  { "띢", "띢", "띢", "띢", "띢" },
  { "띣", "띣", "띣", "띣", "띣" },
  { "띤", "띤", "띤", "띤", "띤" },
  { "띥", "띥", "띥", "띥", "띥" },
  { "띦", "띦", "띦", "띦", "띦" },
  { "띧", "띧", "띧", "띧", "띧" },
  { "띨", "띨", "띨", "띨", "띨" },
  { "띩", "띩", "띩", "띩", "띩" },
  { "띪", "띪", "띪", "띪", "띪" },
  { "띫", "띫", "띫", "띫", "띫" },
  { "띬", "띬", "띬", "띬", "띬" },
  { "띭", "띭", "띭", "띭", "띭" },
  { "띮", "띮", "띮", "띮", "띮" },
  { "띯", "띯", "띯", "띯", "띯" },
  { "띰", "띰", "띰", "띰", "띰" },
  { "띱", "띱", "띱", "띱", "띱" },
  { "띲", "띲", "띲", "띲", "띲" },
  { "띳", "띳", "띳", "띳", "띳" },
  { "띴", "띴", "띴", "띴", "띴" },
  { "띵", "띵", "띵", "띵", "띵" },
  { "띶", "띶", "띶", "띶", "띶" },
  { "띷", "띷", "띷", "띷", "띷" },
  { "띸", "띸", "띸", "띸", "띸" },
  { "띹", "띹", "띹", "띹", "띹" },
  { "띺", "띺", "띺", "띺", "띺" },
  { "띻", "띻", "띻", "띻", "띻" },
  { "라", "라", "라", "라", "라" },
  { "락", "락", "락", "락", "락" },
  { "띾", "띾", "띾", "띾", "띾" },
  { "띿", "띿", "띿", "띿", "띿" },
  { "란", "란", "란", "란", "란" },
  { "랁", "랁", "랁", "랁", "랁" },
  { "랂", "랂", "랂", "랂", "랂" },
  { "랃", "랃", "랃", "랃", "랃" },
  { "랄", "랄", "랄", "랄", "랄" },
  { "랅", "랅", "랅", "랅", "랅" },
  { "랆", "랆", "랆", "랆", "랆" },
  { "랇", "랇", "랇", "랇", "랇" },
  { "랈", "랈", "랈", "랈", "랈" },
  { "랉", "랉", "랉", "랉", "랉" },
  { "랊", "랊", "랊", "랊", "랊" },
  { "랋", "랋", "랋", "랋", "랋" },
  { "람", "람", "람", "람", "람" },
  { "랍", "랍", "랍", "랍", "랍" },
  { "랎", "랎", "랎", "랎", "랎" },
  { "랏", "랏", "랏", "랏", "랏" },
  { "랐", "랐", "랐", "랐", "랐" },
  { "랑", "랑", "랑", "랑", "랑" },
  { "랒", "랒", "랒", "랒", "랒" },
  { "랓", "랓", "랓", "랓", "랓" },
  { "랔", "랔", "랔", "랔", "랔" },
  { "랕", "랕", "랕", "랕", "랕" },
  { "랖", "랖", "랖", "랖", "랖" },
  { "랗", "랗", "랗", "랗", "랗" },
  { "래", "래", "래", "래", "래" },
  { "랙", "랙", "랙", "랙", "랙" },
  { "랚", "랚", "랚", "랚", "랚" },
  { "랛", "랛", "랛", "랛", "랛" },
  { "랜", "랜", "랜", "랜", "랜" },
  { "랝", "랝", "랝", "랝", "랝" },
  { "랞", "랞", "랞", "랞", "랞" },
  { "랟", "랟", "랟", "랟", "랟" },
  { "랠", "랠", "랠", "랠", "랠" },
  { "랡", "랡", "랡", "랡", "랡" },
  { "랢", "랢", "랢", "랢", "랢" },
  { "랣", "랣", "랣", "랣", "랣" },
  { "랤", "랤", "랤", "랤", "랤" },
  { "랥", "랥", "랥", "랥", "랥" },
  { "랦", "랦", "랦", "랦", "랦" },
  { "랧", "랧", "랧", "랧", "랧" },
  { "램", "램", "램", "램", "램" },
  { "랩", "랩", "랩", "랩", "랩" },
  { "랪", "랪", "랪", "랪", "랪" },
  { "랫", "랫", "랫", "랫", "랫" },
  { "랬", "랬", "랬", "랬", "랬" },
  { "랭", "랭", "랭", "랭", "랭" },
  { "랮", "랮", "랮", "랮", "랮" },
  { "랯", "랯", "랯", "랯", "랯" },
  { "랰", "랰", "랰", "랰", "랰" },
  { "랱", "랱", "랱", "랱", "랱" },
  { "랲", "랲", "랲", "랲", "랲" },
  { "랳", "랳", "랳", "랳", "랳" },
  { "랴", "랴", "랴", "랴", "랴" },
  { "략", "략", "략", "략", "략" },
  { "랶", "랶", "랶", "랶", "랶" },
  { "랷", "랷", "랷", "랷", "랷" },
  { "랸", "랸", "랸", "랸", "랸" },
  { "랹", "랹", "랹", "랹", "랹" },
  { "랺", "랺", "랺", "랺", "랺" },
  { "랻", "랻", "랻", "랻", "랻" },
  { "랼", "랼", "랼", "랼", "랼" },
  { "랽", "랽", "랽", "랽", "랽" },
  { "랾", "랾", "랾", "랾", "랾" },
  { "랿", "랿", "랿", "랿", "랿" },
  { "럀", "럀", "럀", "럀", "럀" },
  { "럁", "럁", "럁", "럁", "럁" },
  { "럂", "럂", "럂", "럂", "럂" },
  { "럃", "럃", "럃", "럃", "럃" },
  { "럄", "럄", "럄", "럄", "럄" },
  { "럅", "럅", "럅", "럅", "럅" },
  { "럆", "럆", "럆", "럆", "럆" },
  { "럇", "럇", "럇", "럇", "럇" },
  { "럈", "럈", "럈", "럈", "럈" },
  { "량", "량", "량", "량", "량" },
  { "럊", "럊", "럊", "럊", "럊" },
  { "럋", "럋", "럋", "럋", "럋" },
  { "럌", "럌", "럌", "럌", "럌" },
  { "럍", "럍", "럍", "럍", "럍" },
  { "럎", "럎", "럎", "럎", "럎" },
  { "럏", "럏", "럏", "럏", "럏" },
  { "럐", "럐", "럐", "럐", "럐" },
  { "럑", "럑", "럑", "럑", "럑" },
  { "럒", "럒", "럒", "럒", "럒" },
  { "럓", "럓", "럓", "럓", "럓" },
  { "럔", "럔", "럔", "럔", "럔" },
  { "럕", "럕", "럕", "럕", "럕" },
  { "럖", "럖", "럖", "럖", "럖" },
  { "럗", "럗", "럗", "럗", "럗" },
  { "럘", "럘", "럘", "럘", "럘" },
  { "럙", "럙", "럙", "럙", "럙" },
  { "럚", "럚", "럚", "럚", "럚" },
  { "럛", "럛", "럛", "럛", "럛" },
  { "럜", "럜", "럜", "럜", "럜" },
  { "럝", "럝", "럝", "럝", "럝" },
  { "럞", "럞", "럞", "럞", "럞" },
  { "럟", "럟", "럟", "럟", "럟" },
  { "럠", "럠", "럠", "럠", "럠" },
  { "럡", "럡", "럡", "럡", "럡" },
  { "럢", "럢", "럢", "럢", "럢" },
  { "럣", "럣", "럣", "럣", "럣" },
  { "럤", "럤", "럤", "럤", "럤" },
  { "럥", "럥", "럥", "럥", "럥" },
  { "럦", "럦", "럦", "럦", "럦" },
  { "럧", "럧", "럧", "럧", "럧" },
  { "럨", "럨", "럨", "럨", "럨" },
  { "럩", "럩", "럩", "럩", "럩" },
  { "럪", "럪", "럪", "럪", "럪" },
  { "럫", "럫", "럫", "럫", "럫" },
  { "러", "러", "러", "러", "러" },
  { "럭", "럭", "럭", "럭", "럭" },
  { "럮", "럮", "럮", "럮", "럮" },
  { "럯", "럯", "럯", "럯", "럯" },
  { "런", "런", "런", "런", "런" },
  { "럱", "럱", "럱", "럱", "럱" },
  { "럲", "럲", "럲", "럲", "럲" },
  { "럳", "럳", "럳", "럳", "럳" },
  { "럴", "럴", "럴", "럴", "럴" },
  { "럵", "럵", "럵", "럵", "럵" },
  { "럶", "럶", "럶", "럶", "럶" },
  { "럷", "럷", "럷", "럷", "럷" },
  { "럸", "럸", "럸", "럸", "럸" },
  { "럹", "럹", "럹", "럹", "럹" },
  { "럺", "럺", "럺", "럺", "럺" },
  { "럻", "럻", "럻", "럻", "럻" },
  { "럼", "럼", "럼", "럼", "럼" },
  { "럽", "럽", "럽", "럽", "럽" },
  { "럾", "럾", "럾", "럾", "럾" },
  { "럿", "럿", "럿", "럿", "럿" },
  { "렀", "렀", "렀", "렀", "렀" },
  { "렁", "렁", "렁", "렁", "렁" },
  { "렂", "렂", "렂", "렂", "렂" },
  { "렃", "렃", "렃", "렃", "렃" },
  { "렄", "렄", "렄", "렄", "렄" },
  { "렅", "렅", "렅", "렅", "렅" },
  { "렆", "렆", "렆", "렆", "렆" },
  { "렇", "렇", "렇", "렇", "렇" },
  { "레", "레", "레", "레", "레" },
  { "렉", "렉", "렉", "렉", "렉" },
  { "렊", "렊", "렊", "렊", "렊" },
  { "렋", "렋", "렋", "렋", "렋" },
  { "렌", "렌", "렌", "렌", "렌" },
  { "렍", "렍", "렍", "렍", "렍" },
  { "렎", "렎", "렎", "렎", "렎" },
  { "렏", "렏", "렏", "렏", "렏" },
  { "렐", "렐", "렐", "렐", "렐" },
  { "렑", "렑", "렑", "렑", "렑" },
  { "렒", "렒", "렒", "렒", "렒" },
  { "렓", "렓", "렓", "렓", "렓" },
  { "렔", "렔", "렔", "렔", "렔" },
  { "렕", "렕", "렕", "렕", "렕" },
  { "렖", "렖", "렖", "렖", "렖" },
  { "렗", "렗", "렗", "렗", "렗" },
  { "렘", "렘", "렘", "렘", "렘" },
  { "렙", "렙", "렙", "렙", "렙" },
  { "렚", "렚", "렚", "렚", "렚" },
  { "렛", "렛", "렛", "렛", "렛" },
  { "렜", "렜", "렜", "렜", "렜" },
  { "렝", "렝", "렝", "렝", "렝" },
  { "렞", "렞", "렞", "렞", "렞" },
  { "렟", "렟", "렟", "렟", "렟" },
  { "렠", "렠", "렠", "렠", "렠" },
  { "렡", "렡", "렡", "렡", "렡" },
  { "렢", "렢", "렢", "렢", "렢" },
  { "렣", "렣", "렣", "렣", "렣" },
  { "려", "려", "려", "려", "려" },
  { "력", "력", "력", "력", "력" },
  { "렦", "렦", "렦", "렦", "렦" },
  { "렧", "렧", "렧", "렧", "렧" },
  { "련", "련", "련", "련", "련" },
  { "렩", "렩", "렩", "렩", "렩" },
  { "렪", "렪", "렪", "렪", "렪" },
  { "렫", "렫", "렫", "렫", "렫" },
  { "렬", "렬", "렬", "렬", "렬" },
  { "렭", "렭", "렭", "렭", "렭" },
  { "렮", "렮", "렮", "렮", "렮" },
  { "렯", "렯", "렯", "렯", "렯" },
  { "렰", "렰", "렰", "렰", "렰" },
  { "렱", "렱", "렱", "렱", "렱" },
  { "렲", "렲", "렲", "렲", "렲" },
  { "렳", "렳", "렳", "렳", "렳" },
  { "렴", "렴", "렴", "렴", "렴" },
  { "렵", "렵", "렵", "렵", "렵" },
  { "렶", "렶", "렶", "렶", "렶" },
  { "렷", "렷", "렷", "렷", "렷" },
  { "렸", "렸", "렸", "렸", "렸" },
  { "령", "령", "령", "령", "령" },
  { "렺", "렺", "렺", "렺", "렺" },
  { "렻", "렻", "렻", "렻", "렻" },
  { "렼", "렼", "렼", "렼", "렼" },
  { "렽", "렽", "렽", "렽", "렽" },
  { "렾", "렾", "렾", "렾", "렾" },
  { "렿", "렿", "렿", "렿", "렿" },
  { "례", "례", "례", "례", "례" },
  { "롁", "롁", "롁", "롁", "롁" },
  { "롂", "롂", "롂", "롂", "롂" },
  { "롃", "롃", "롃", "롃", "롃" },
  { "롄", "롄", "롄", "롄", "롄" },
  { "롅", "롅", "롅", "롅", "롅" },
  { "롆", "롆", "롆", "롆", "롆" },
  { "롇", "롇", "롇", "롇", "롇" },
  { "롈", "롈", "롈", "롈", "롈" },
  { "롉", "롉", "롉", "롉", "롉" },
  { "롊", "롊", "롊", "롊", "롊" },
  { "롋", "롋", "롋", "롋", "롋" },
  { "롌", "롌", "롌", "롌", "롌" },
  { "롍", "롍", "롍", "롍", "롍" },
  { "롎", "롎", "롎", "롎", "롎" },
  { "롏", "롏", "롏", "롏", "롏" },
  { "롐", "롐", "롐", "롐", "롐" },
  { "롑", "롑", "롑", "롑", "롑" },
  { "롒", "롒", "롒", "롒", "롒" },
  { "롓", "롓", "롓", "롓", "롓" },
  { "롔", "롔", "롔", "롔", "롔" },
  { "롕", "롕", "롕", "롕", "롕" },
  { "롖", "롖", "롖", "롖", "롖" },
  { "롗", "롗", "롗", "롗", "롗" },
  { "롘", "롘", "롘", "롘", "롘" },
  { "롙", "롙", "롙", "롙", "롙" },
  { "롚", "롚", "롚", "롚", "롚" },
  { "롛", "롛", "롛", "롛", "롛" },
  { "로", "로", "로", "로", "로" },
  { "록", "록", "록", "록", "록" },
  { "롞", "롞", "롞", "롞", "롞" },
  { "롟", "롟", "롟", "롟", "롟" },
  { "론", "론", "론", "론", "론" },
  { "롡", "롡", "롡", "롡", "롡" },
  { "롢", "롢", "롢", "롢", "롢" },
  { "롣", "롣", "롣", "롣", "롣" },
  { "롤", "롤", "롤", "롤", "롤" },
  { "롥", "롥", "롥", "롥", "롥" },
  { "롦", "롦", "롦", "롦", "롦" },
  { "롧", "롧", "롧", "롧", "롧" },
  { "롨", "롨", "롨", "롨", "롨" },
  { "롩", "롩", "롩", "롩", "롩" },
  { "롪", "롪", "롪", "롪", "롪" },
  { "롫", "롫", "롫", "롫", "롫" },
  { "롬", "롬", "롬", "롬", "롬" },
  { "롭", "롭", "롭", "롭", "롭" },
  { "롮", "롮", "롮", "롮", "롮" },
  { "롯", "롯", "롯", "롯", "롯" },
  { "롰", "롰", "롰", "롰", "롰" },
  { "롱", "롱", "롱", "롱", "롱" },
  { "롲", "롲", "롲", "롲", "롲" },
  { "롳", "롳", "롳", "롳", "롳" },
  { "롴", "롴", "롴", "롴", "롴" },
  { "롵", "롵", "롵", "롵", "롵" },
  { "롶", "롶", "롶", "롶", "롶" },
  { "롷", "롷", "롷", "롷", "롷" },
  { "롸", "롸", "롸", "롸", "롸" },
  { "롹", "롹", "롹", "롹", "롹" },
  { "롺", "롺", "롺", "롺", "롺" },
  { "롻", "롻", "롻", "롻", "롻" },
  { "롼", "롼", "롼", "롼", "롼" },
  { "롽", "롽", "롽", "롽", "롽" },
  { "롾", "롾", "롾", "롾", "롾" },
  { "롿", "롿", "롿", "롿", "롿" },
  { "뢀", "뢀", "뢀", "뢀", "뢀" },
  { "뢁", "뢁", "뢁", "뢁", "뢁" },
  { "뢂", "뢂", "뢂", "뢂", "뢂" },
  { "뢃", "뢃", "뢃", "뢃", "뢃" },
  { "뢄", "뢄", "뢄", "뢄", "뢄" },
  { "뢅", "뢅", "뢅", "뢅", "뢅" },
  { "뢆", "뢆", "뢆", "뢆", "뢆" },
  { "뢇", "뢇", "뢇", "뢇", "뢇" },
  { "뢈", "뢈", "뢈", "뢈", "뢈" },
  { "뢉", "뢉", "뢉", "뢉", "뢉" },
  { "뢊", "뢊", "뢊", "뢊", "뢊" },
  { "뢋", "뢋", "뢋", "뢋", "뢋" },
  { "뢌", "뢌", "뢌", "뢌", "뢌" },
  { "뢍", "뢍", "뢍", "뢍", "뢍" },
  { "뢎", "뢎", "뢎", "뢎", "뢎" },
  { "뢏", "뢏", "뢏", "뢏", "뢏" },
  { "뢐", "뢐", "뢐", "뢐", "뢐" },
  { "뢑", "뢑", "뢑", "뢑", "뢑" },
  { "뢒", "뢒", "뢒", "뢒", "뢒" },
  { "뢓", "뢓", "뢓", "뢓", "뢓" },
  { "뢔", "뢔", "뢔", "뢔", "뢔" },
  { "뢕", "뢕", "뢕", "뢕", "뢕" },
  { "뢖", "뢖", "뢖", "뢖", "뢖" },
  { "뢗", "뢗", "뢗", "뢗", "뢗" },
  { "뢘", "뢘", "뢘", "뢘", "뢘" },
  { "뢙", "뢙", "뢙", "뢙", "뢙" },
  { "뢚", "뢚", "뢚", "뢚", "뢚" },
  { "뢛", "뢛", "뢛", "뢛", "뢛" },
  { "뢜", "뢜", "뢜", "뢜", "뢜" },
  { "뢝", "뢝", "뢝", "뢝", "뢝" },
  { "뢞", "뢞", "뢞", "뢞", "뢞" },
  { "뢟", "뢟", "뢟", "뢟", "뢟" },
  { "뢠", "뢠", "뢠", "뢠", "뢠" },
  { "뢡", "뢡", "뢡", "뢡", "뢡" },
  { "뢢", "뢢", "뢢", "뢢", "뢢" },
  { "뢣", "뢣", "뢣", "뢣", "뢣" },
  { "뢤", "뢤", "뢤", "뢤", "뢤" },
  { "뢥", "뢥", "뢥", "뢥", "뢥" },
  { "뢦", "뢦", "뢦", "뢦", "뢦" },
  { "뢧", "뢧", "뢧", "뢧", "뢧" },
  { "뢨", "뢨", "뢨", "뢨", "뢨" },
  { "뢩", "뢩", "뢩", "뢩", "뢩" },
  { "뢪", "뢪", "뢪", "뢪", "뢪" },
  { "뢫", "뢫", "뢫", "뢫", "뢫" },
  { "뢬", "뢬", "뢬", "뢬", "뢬" },
  { "뢭", "뢭", "뢭", "뢭", "뢭" },
  { "뢮", "뢮", "뢮", "뢮", "뢮" },
  { "뢯", "뢯", "뢯", "뢯", "뢯" },
  { "뢰", "뢰", "뢰", "뢰", "뢰" },
  { "뢱", "뢱", "뢱", "뢱", "뢱" },
  { "뢲", "뢲", "뢲", "뢲", "뢲" },
  { "뢳", "뢳", "뢳", "뢳", "뢳" },
  { "뢴", "뢴", "뢴", "뢴", "뢴" },
  { "뢵", "뢵", "뢵", "뢵", "뢵" },
  { "뢶", "뢶", "뢶", "뢶", "뢶" },
  { "뢷", "뢷", "뢷", "뢷", "뢷" },
  { "뢸", "뢸", "뢸", "뢸", "뢸" },
  { "뢹", "뢹", "뢹", "뢹", "뢹" },
  { "뢺", "뢺", "뢺", "뢺", "뢺" },
  { "뢻", "뢻", "뢻", "뢻", "뢻" },
  { "뢼", "뢼", "뢼", "뢼", "뢼" },
  { "뢽", "뢽", "뢽", "뢽", "뢽" },
  { "뢾", "뢾", "뢾", "뢾", "뢾" },
  { "뢿", "뢿", "뢿", "뢿", "뢿" },
  { "룀", "룀", "룀", "룀", "룀" },
  { "룁", "룁", "룁", "룁", "룁" },
  { "룂", "룂", "룂", "룂", "룂" },
  { "룃", "룃", "룃", "룃", "룃" },
  { "룄", "룄", "룄", "룄", "룄" },
  { "룅", "룅", "룅", "룅", "룅" },
  { "룆", "룆", "룆", "룆", "룆" },
  { "룇", "룇", "룇", "룇", "룇" },
  { "룈", "룈", "룈", "룈", "룈" },
  { "룉", "룉", "룉", "룉", "룉" },
  { "룊", "룊", "룊", "룊", "룊" },
  { "룋", "룋", "룋", "룋", "룋" },
  { "료", "료", "료", "료", "료" },
  { "룍", "룍", "룍", "룍", "룍" },
  { "룎", "룎", "룎", "룎", "룎" },
  { "룏", "룏", "룏", "룏", "룏" },
  { "룐", "룐", "룐", "룐", "룐" },
  { "룑", "룑", "룑", "룑", "룑" },
  { "룒", "룒", "룒", "룒", "룒" },
  { "룓", "룓", "룓", "룓", "룓" },
  { "룔", "룔", "룔", "룔", "룔" },
  { "룕", "룕", "룕", "룕", "룕" },
  { "룖", "룖", "룖", "룖", "룖" },
  { "룗", "룗", "룗", "룗", "룗" },
  { "룘", "룘", "룘", "룘", "룘" },
  { "룙", "룙", "룙", "룙", "룙" },
  { "룚", "룚", "룚", "룚", "룚" },
  { "룛", "룛", "룛", "룛", "룛" },
  { "룜", "룜", "룜", "룜", "룜" },
  { "룝", "룝", "룝", "룝", "룝" },
  { "룞", "룞", "룞", "룞", "룞" },
  { "룟", "룟", "룟", "룟", "룟" },
  { "룠", "룠", "룠", "룠", "룠" },
  { "룡", "룡", "룡", "룡", "룡" },
  { "룢", "룢", "룢", "룢", "룢" },
  { "룣", "룣", "룣", "룣", "룣" },
  { "룤", "룤", "룤", "룤", "룤" },
  { "룥", "룥", "룥", "룥", "룥" },
  { "룦", "룦", "룦", "룦", "룦" },
  { "룧", "룧", "룧", "룧", "룧" },
  { "루", "루", "루", "루", "루" },
  { "룩", "룩", "룩", "룩", "룩" },
  { "룪", "룪", "룪", "룪", "룪" },
  { "룫", "룫", "룫", "룫", "룫" },
  { "룬", "룬", "룬", "룬", "룬" },
  { "룭", "룭", "룭", "룭", "룭" },
  { "룮", "룮", "룮", "룮", "룮" },
  { "룯", "룯", "룯", "룯", "룯" },
  { "룰", "룰", "룰", "룰", "룰" },
  { "룱", "룱", "룱", "룱", "룱" },
  { "룲", "룲", "룲", "룲", "룲" },
  { "룳", "룳", "룳", "룳", "룳" },
  { "룴", "룴", "룴", "룴", "룴" },
  { "룵", "룵", "룵", "룵", "룵" },
  { "룶", "룶", "룶", "룶", "룶" },
  { "룷", "룷", "룷", "룷", "룷" },
  { "룸", "룸", "룸", "룸", "룸" },
  { "룹", "룹", "룹", "룹", "룹" },
  { "룺", "룺", "룺", "룺", "룺" },
  { "룻", "룻", "룻", "룻", "룻" },
  { "룼", "룼", "룼", "룼", "룼" },
  { "룽", "룽", "룽", "룽", "룽" },
  { "룾", "룾", "룾", "룾", "룾" },
  { "룿", "룿", "룿", "룿", "룿" },
  { "뤀", "뤀", "뤀", "뤀", "뤀" },
  { "뤁", "뤁", "뤁", "뤁", "뤁" },
  { "뤂", "뤂", "뤂", "뤂", "뤂" },
  { "뤃", "뤃", "뤃", "뤃", "뤃" },
  { "뤄", "뤄", "뤄", "뤄", "뤄" },
  { "뤅", "뤅", "뤅", "뤅", "뤅" },
  { "뤆", "뤆", "뤆", "뤆", "뤆" },
  { "뤇", "뤇", "뤇", "뤇", "뤇" },
  { "뤈", "뤈", "뤈", "뤈", "뤈" },
  { "뤉", "뤉", "뤉", "뤉", "뤉" },
  { "뤊", "뤊", "뤊", "뤊", "뤊" },
  { "뤋", "뤋", "뤋", "뤋", "뤋" },
  { "뤌", "뤌", "뤌", "뤌", "뤌" },
  { "뤍", "뤍", "뤍", "뤍", "뤍" },
  { "뤎", "뤎", "뤎", "뤎", "뤎" },
  { "뤏", "뤏", "뤏", "뤏", "뤏" },
  { "뤐", "뤐", "뤐", "뤐", "뤐" },
  { "뤑", "뤑", "뤑", "뤑", "뤑" },
  { "뤒", "뤒", "뤒", "뤒", "뤒" },
  { "뤓", "뤓", "뤓", "뤓", "뤓" },
  { "뤔", "뤔", "뤔", "뤔", "뤔" },
  { "뤕", "뤕", "뤕", "뤕", "뤕" },
  { "뤖", "뤖", "뤖", "뤖", "뤖" },
  { "뤗", "뤗", "뤗", "뤗", "뤗" },
  { "뤘", "뤘", "뤘", "뤘", "뤘" },
  { "뤙", "뤙", "뤙", "뤙", "뤙" },
  { "뤚", "뤚", "뤚", "뤚", "뤚" },
  { "뤛", "뤛", "뤛", "뤛", "뤛" },
  { "뤜", "뤜", "뤜", "뤜", "뤜" },
  { "뤝", "뤝", "뤝", "뤝", "뤝" },
  { "뤞", "뤞", "뤞", "뤞", "뤞" },
  { "뤟", "뤟", "뤟", "뤟", "뤟" },
  { "뤠", "뤠", "뤠", "뤠", "뤠" },
  { "뤡", "뤡", "뤡", "뤡", "뤡" },
  { "뤢", "뤢", "뤢", "뤢", "뤢" },
  { "뤣", "뤣", "뤣", "뤣", "뤣" },
  { "뤤", "뤤", "뤤", "뤤", "뤤" },
  { "뤥", "뤥", "뤥", "뤥", "뤥" },
  { "뤦", "뤦", "뤦", "뤦", "뤦" },
  { "뤧", "뤧", "뤧", "뤧", "뤧" },
  { "뤨", "뤨", "뤨", "뤨", "뤨" },
  { "뤩", "뤩", "뤩", "뤩", "뤩" },
  { "뤪", "뤪", "뤪", "뤪", "뤪" },
  { "뤫", "뤫", "뤫", "뤫", "뤫" },
  { "뤬", "뤬", "뤬", "뤬", "뤬" },
  { "뤭", "뤭", "뤭", "뤭", "뤭" },
  { "뤮", "뤮", "뤮", "뤮", "뤮" },
  { "뤯", "뤯", "뤯", "뤯", "뤯" },
  { "뤰", "뤰", "뤰", "뤰", "뤰" },
  { "뤱", "뤱", "뤱", "뤱", "뤱" },
  { "뤲", "뤲", "뤲", "뤲", "뤲" },
  { "뤳", "뤳", "뤳", "뤳", "뤳" },
  { "뤴", "뤴", "뤴", "뤴", "뤴" },
  { "뤵", "뤵", "뤵", "뤵", "뤵" },
  { "뤶", "뤶", "뤶", "뤶", "뤶" },
  { "뤷", "뤷", "뤷", "뤷", "뤷" },
  { "뤸", "뤸", "뤸", "뤸", "뤸" },
  { "뤹", "뤹", "뤹", "뤹", "뤹" },
  { "뤺", "뤺", "뤺", "뤺", "뤺" },
  { "뤻", "뤻", "뤻", "뤻", "뤻" },
  { "뤼", "뤼", "뤼", "뤼", "뤼" },
  { "뤽", "뤽", "뤽", "뤽", "뤽" },
  { "뤾", "뤾", "뤾", "뤾", "뤾" },
  { "뤿", "뤿", "뤿", "뤿", "뤿" },
  { "륀", "륀", "륀", "륀", "륀" },
  { "륁", "륁", "륁", "륁", "륁" },
  { "륂", "륂", "륂", "륂", "륂" },
  { "륃", "륃", "륃", "륃", "륃" },
  { "륄", "륄", "륄", "륄", "륄" },
  { "륅", "륅", "륅", "륅", "륅" },
  { "륆", "륆", "륆", "륆", "륆" },
  { "륇", "륇", "륇", "륇", "륇" },
  { "륈", "륈", "륈", "륈", "륈" },
  { "륉", "륉", "륉", "륉", "륉" },
  { "륊", "륊", "륊", "륊", "륊" },
  { "륋", "륋", "륋", "륋", "륋" },
  { "륌", "륌", "륌", "륌", "륌" },
  { "륍", "륍", "륍", "륍", "륍" },
  { "륎", "륎", "륎", "륎", "륎" },
  { "륏", "륏", "륏", "륏", "륏" },
  { "륐", "륐", "륐", "륐", "륐" },
  { "륑", "륑", "륑", "륑", "륑" },
  { "륒", "륒", "륒", "륒", "륒" },
  { "륓", "륓", "륓", "륓", "륓" },
  { "륔", "륔", "륔", "륔", "륔" },
  { "륕", "륕", "륕", "륕", "륕" },
  { "륖", "륖", "륖", "륖", "륖" },
  { "륗", "륗", "륗", "륗", "륗" },
  { "류", "류", "류", "류", "류" },
  { "륙", "륙", "륙", "륙", "륙" },
  { "륚", "륚", "륚", "륚", "륚" },
  { "륛", "륛", "륛", "륛", "륛" },
  { "륜", "륜", "륜", "륜", "륜" },
  { "륝", "륝", "륝", "륝", "륝" },
  { "륞", "륞", "륞", "륞", "륞" },
  { "륟", "륟", "륟", "륟", "륟" },
  { "률", "률", "률", "률", "률" },
  { "륡", "륡", "륡", "륡", "륡" },
  { "륢", "륢", "륢", "륢", "륢" },
  { "륣", "륣", "륣", "륣", "륣" },
  { "륤", "륤", "륤", "륤", "륤" },
  { "륥", "륥", "륥", "륥", "륥" },
  { "륦", "륦", "륦", "륦", "륦" },
  { "륧", "륧", "륧", "륧", "륧" },
  { "륨", "륨", "륨", "륨", "륨" },
  { "륩", "륩", "륩", "륩", "륩" },
  { "륪", "륪", "륪", "륪", "륪" },
  { "륫", "륫", "륫", "륫", "륫" },
  { "륬", "륬", "륬", "륬", "륬" },
  { "륭", "륭", "륭", "륭", "륭" },
  { "륮", "륮", "륮", "륮", "륮" },
  { "륯", "륯", "륯", "륯", "륯" },
  { "륰", "륰", "륰", "륰", "륰" },
  { "륱", "륱", "륱", "륱", "륱" },
  { "륲", "륲", "륲", "륲", "륲" },
  { "륳", "륳", "륳", "륳", "륳" },
  { "르", "르", "르", "르", "르" },
  { "륵", "륵", "륵", "륵", "륵" },
  { "륶", "륶", "륶", "륶", "륶" },
  { "륷", "륷", "륷", "륷", "륷" },
  { "른", "른", "른", "른", "른" },
  { "륹", "륹", "륹", "륹", "륹" },
  { "륺", "륺", "륺", "륺", "륺" },
  { "륻", "륻", "륻", "륻", "륻" },
  { "를", "를", "를", "를", "를" },
  { "륽", "륽", "륽", "륽", "륽" },
  { "륾", "륾", "륾", "륾", "륾" },
  { "륿", "륿", "륿", "륿", "륿" },
  { "릀", "릀", "릀", "릀", "릀" },
  { "릁", "릁", "릁", "릁", "릁" },
  { "릂", "릂", "릂", "릂", "릂" },
  { "릃", "릃", "릃", "릃", "릃" },
  { "름", "름", "름", "름", "름" },
  { "릅", "릅", "릅", "릅", "릅" },
  { "릆", "릆", "릆", "릆", "릆" },
  { "릇", "릇", "릇", "릇", "릇" },
  { "릈", "릈", "릈", "릈", "릈" },
  { "릉", "릉", "릉", "릉", "릉" },
  { "릊", "릊", "릊", "릊", "릊" },
  { "릋", "릋", "릋", "릋", "릋" },
  { "릌", "릌", "릌", "릌", "릌" },
  { "릍", "릍", "릍", "릍", "릍" },
  { "릎", "릎", "릎", "릎", "릎" },
  { "릏", "릏", "릏", "릏", "릏" },
  { "릐", "릐", "릐", "릐", "릐" },
  { "릑", "릑", "릑", "릑", "릑" },
  { "릒", "릒", "릒", "릒", "릒" },
  { "릓", "릓", "릓", "릓", "릓" },
  { "릔", "릔", "릔", "릔", "릔" },
  { "릕", "릕", "릕", "릕", "릕" },
  { "릖", "릖", "릖", "릖", "릖" },
  { "릗", "릗", "릗", "릗", "릗" },
  { "릘", "릘", "릘", "릘", "릘" },
  { "릙", "릙", "릙", "릙", "릙" },
  { "릚", "릚", "릚", "릚", "릚" },
  { "릛", "릛", "릛", "릛", "릛" },
  { "릜", "릜", "릜", "릜", "릜" },
  { "릝", "릝", "릝", "릝", "릝" },
  { "릞", "릞", "릞", "릞", "릞" },
  { "릟", "릟", "릟", "릟", "릟" },
  { "릠", "릠", "릠", "릠", "릠" },
  { "릡", "릡", "릡", "릡", "릡" },
  { "릢", "릢", "릢", "릢", "릢" },
  { "릣", "릣", "릣", "릣", "릣" },
  { "릤", "릤", "릤", "릤", "릤" },
  { "릥", "릥", "릥", "릥", "릥" },
  { "릦", "릦", "릦", "릦", "릦" },
  { "릧", "릧", "릧", "릧", "릧" },
  { "릨", "릨", "릨", "릨", "릨" },
  { "릩", "릩", "릩", "릩", "릩" },
  { "릪", "릪", "릪", "릪", "릪" },
  { "릫", "릫", "릫", "릫", "릫" },
  { "리", "리", "리", "리", "리" },
  { "릭", "릭", "릭", "릭", "릭" },
  { "릮", "릮", "릮", "릮", "릮" },
  { "릯", "릯", "릯", "릯", "릯" },
  { "린", "린", "린", "린", "린" },
  { "릱", "릱", "릱", "릱", "릱" },
  { "릲", "릲", "릲", "릲", "릲" },
  { "릳", "릳", "릳", "릳", "릳" },
  { "릴", "릴", "릴", "릴", "릴" },
  { "릵", "릵", "릵", "릵", "릵" },
  { "릶", "릶", "릶", "릶", "릶" },
  { "릷", "릷", "릷", "릷", "릷" },
  { "릸", "릸", "릸", "릸", "릸" },
  { "릹", "릹", "릹", "릹", "릹" },
  { "릺", "릺", "릺", "릺", "릺" },
  { "릻", "릻", "릻", "릻", "릻" },
  { "림", "림", "림", "림", "림" },
  { "립", "립", "립", "립", "립" },
  { "릾", "릾", "릾", "릾", "릾" },
  { "릿", "릿", "릿", "릿", "릿" },
  { "맀", "맀", "맀", "맀", "맀" },
  { "링", "링", "링", "링", "링" },
  { "맂", "맂", "맂", "맂", "맂" },
  { "맃", "맃", "맃", "맃", "맃" },
  { "맄", "맄", "맄", "맄", "맄" },
  { "맅", "맅", "맅", "맅", "맅" },
  { "맆", "맆", "맆", "맆", "맆" },
  { "맇", "맇", "맇", "맇", "맇" },
  { "마", "마", "마", "마", "마" },
  { "막", "막", "막", "막", "막" },
  { "맊", "맊", "맊", "맊", "맊" },
  { "맋", "맋", "맋", "맋", "맋" },
  { "만", "만", "만", "만", "만" },
  { "맍", "맍", "맍", "맍", "맍" },
  { "많", "많", "많", "많", "많" },
  { "맏", "맏", "맏", "맏", "맏" },
  { "말", "말", "말", "말", "말" },
  { "맑", "맑", "맑", "맑", "맑" },
  { "맒", "맒", "맒", "맒", "맒" },
  { "맓", "맓", "맓", "맓", "맓" },
  { "맔", "맔", "맔", "맔", "맔" },
  { "맕", "맕", "맕", "맕", "맕" },
  { "맖", "맖", "맖", "맖", "맖" },
  { "맗", "맗", "맗", "맗", "맗" },
  { "맘", "맘", "맘", "맘", "맘" },
  { "맙", "맙", "맙", "맙", "맙" },
  { "맚", "맚", "맚", "맚", "맚" },
  { "맛", "맛", "맛", "맛", "맛" },
  { "맜", "맜", "맜", "맜", "맜" },
  { "망", "망", "망", "망", "망" },
  { "맞", "맞", "맞", "맞", "맞" },
  { "맟", "맟", "맟", "맟", "맟" },
  { "맠", "맠", "맠", "맠", "맠" },
  { "맡", "맡", "맡", "맡", "맡" },
  { "맢", "맢", "맢", "맢", "맢" },
  { "맣", "맣", "맣", "맣", "맣" },
  { "매", "매", "매", "매", "매" },
  { "맥", "맥", "맥", "맥", "맥" },
  { "맦", "맦", "맦", "맦", "맦" },
  { "맧", "맧", "맧", "맧", "맧" },
  { "맨", "맨", "맨", "맨", "맨" },
  { "맩", "맩", "맩", "맩", "맩" },
  { "맪", "맪", "맪", "맪", "맪" },
  { "맫", "맫", "맫", "맫", "맫" },
  { "맬", "맬", "맬", "맬", "맬" },
  { "맭", "맭", "맭", "맭", "맭" },
  { "맮", "맮", "맮", "맮", "맮" },
  { "맯", "맯", "맯", "맯", "맯" },
  { "맰", "맰", "맰", "맰", "맰" },
  { "맱", "맱", "맱", "맱", "맱" },
  { "맲", "맲", "맲", "맲", "맲" },
  { "맳", "맳", "맳", "맳", "맳" },
  { "맴", "맴", "맴", "맴", "맴" },
  { "맵", "맵", "맵", "맵", "맵" },
  { "맶", "맶", "맶", "맶", "맶" },
  { "맷", "맷", "맷", "맷", "맷" },
  { "맸", "맸", "맸", "맸", "맸" },
  { "맹", "맹", "맹", "맹", "맹" },
  { "맺", "맺", "맺", "맺", "맺" },
  { "맻", "맻", "맻", "맻", "맻" },
  { "맼", "맼", "맼", "맼", "맼" },
  { "맽", "맽", "맽", "맽", "맽" },
  { "맾", "맾", "맾", "맾", "맾" },
  { "맿", "맿", "맿", "맿", "맿" },
  { "먀", "먀", "먀", "먀", "먀" },
  { "먁", "먁", "먁", "먁", "먁" },
  { "먂", "먂", "먂", "먂", "먂" },
  { "먃", "먃", "먃", "먃", "먃" },
  { "먄", "먄", "먄", "먄", "먄" },
  { "먅", "먅", "먅", "먅", "먅" },
  { "먆", "먆", "먆", "먆", "먆" },
  { "먇", "먇", "먇", "먇", "먇" },
  { "먈", "먈", "먈", "먈", "먈" },
  { "먉", "먉", "먉", "먉", "먉" },
  { "먊", "먊", "먊", "먊", "먊" },
  { "먋", "먋", "먋", "먋", "먋" },
  { "먌", "먌", "먌", "먌", "먌" },
  { "먍", "먍", "먍", "먍", "먍" },
  { "먎", "먎", "먎", "먎", "먎" },
  { "먏", "먏", "먏", "먏", "먏" },
  { "먐", "먐", "먐", "먐", "먐" },
  { "먑", "먑", "먑", "먑", "먑" },
  { "먒", "먒", "먒", "먒", "먒" },
  { "먓", "먓", "먓", "먓", "먓" },
  { "먔", "먔", "먔", "먔", "먔" },
  { "먕", "먕", "먕", "먕", "먕" },
  { "먖", "먖", "먖", "먖", "먖" },
  { "먗", "먗", "먗", "먗", "먗" },
  { "먘", "먘", "먘", "먘", "먘" },
  { "먙", "먙", "먙", "먙", "먙" },
  { "먚", "먚", "먚", "먚", "먚" },
  { "먛", "먛", "먛", "먛", "먛" },
  { "먜", "먜", "먜", "먜", "먜" },
  { "먝", "먝", "먝", "먝", "먝" },
  { "먞", "먞", "먞", "먞", "먞" },
  { "먟", "먟", "먟", "먟", "먟" },
  { "먠", "먠", "먠", "먠", "먠" },
  { "먡", "먡", "먡", "먡", "먡" },
  { "먢", "먢", "먢", "먢", "먢" },
  { "먣", "먣", "먣", "먣", "먣" },
  { "먤", "먤", "먤", "먤", "먤" },
  { "먥", "먥", "먥", "먥", "먥" },
  { "먦", "먦", "먦", "먦", "먦" },
  { "먧", "먧", "먧", "먧", "먧" },
  { "먨", "먨", "먨", "먨", "먨" },
  { "먩", "먩", "먩", "먩", "먩" },
  { "먪", "먪", "먪", "먪", "먪" },
  { "먫", "먫", "먫", "먫", "먫" },
  { "먬", "먬", "먬", "먬", "먬" },
  { "먭", "먭", "먭", "먭", "먭" },
  { "먮", "먮", "먮", "먮", "먮" },
  { "먯", "먯", "먯", "먯", "먯" },
  { "먰", "먰", "먰", "먰", "먰" },
  { "먱", "먱", "먱", "먱", "먱" },
  { "먲", "먲", "먲", "먲", "먲" },
  { "먳", "먳", "먳", "먳", "먳" },
  { "먴", "먴", "먴", "먴", "먴" },
  { "먵", "먵", "먵", "먵", "먵" },
  { "먶", "먶", "먶", "먶", "먶" },
  { "먷", "먷", "먷", "먷", "먷" },
  { "머", "머", "머", "머", "머" },
  { "먹", "먹", "먹", "먹", "먹" },
  { "먺", "먺", "먺", "먺", "먺" },
  { "먻", "먻", "먻", "먻", "먻" },
  { "먼", "먼", "먼", "먼", "먼" },
  { "먽", "먽", "먽", "먽", "먽" },
  { "먾", "먾", "먾", "먾", "먾" },
  { "먿", "먿", "먿", "먿", "먿" },
  { "멀", "멀", "멀", "멀", "멀" },
  { "멁", "멁", "멁", "멁", "멁" },
  { "멂", "멂", "멂", "멂", "멂" },
  { "멃", "멃", "멃", "멃", "멃" },
  { "멄", "멄", "멄", "멄", "멄" },
  { "멅", "멅", "멅", "멅", "멅" },
  { "멆", "멆", "멆", "멆", "멆" },
  { "멇", "멇", "멇", "멇", "멇" },
  { "멈", "멈", "멈", "멈", "멈" },
  { "멉", "멉", "멉", "멉", "멉" },
  { "멊", "멊", "멊", "멊", "멊" },
  { "멋", "멋", "멋", "멋", "멋" },
  { "멌", "멌", "멌", "멌", "멌" },
  { "멍", "멍", "멍", "멍", "멍" },
  { "멎", "멎", "멎", "멎", "멎" },
  { "멏", "멏", "멏", "멏", "멏" },
  { "멐", "멐", "멐", "멐", "멐" },
  { "멑", "멑", "멑", "멑", "멑" },
  { "멒", "멒", "멒", "멒", "멒" },
  { "멓", "멓", "멓", "멓", "멓" },
  { "메", "메", "메", "메", "메" },
  { "멕", "멕", "멕", "멕", "멕" },
  { "멖", "멖", "멖", "멖", "멖" },
  { "멗", "멗", "멗", "멗", "멗" },
  { "멘", "멘", "멘", "멘", "멘" },
  { "멙", "멙", "멙", "멙", "멙" },
  { "멚", "멚", "멚", "멚", "멚" },
  { "멛", "멛", "멛", "멛", "멛" },
  { "멜", "멜", "멜", "멜", "멜" },
  { "멝", "멝", "멝", "멝", "멝" },
  { "멞", "멞", "멞", "멞", "멞" },
  { "멟", "멟", "멟", "멟", "멟" },
  { "멠", "멠", "멠", "멠", "멠" },
  { "멡", "멡", "멡", "멡", "멡" },
  { "멢", "멢", "멢", "멢", "멢" },
  { "멣", "멣", "멣", "멣", "멣" },
  { "멤", "멤", "멤", "멤", "멤" },
  { "멥", "멥", "멥", "멥", "멥" },
  { "멦", "멦", "멦", "멦", "멦" },
  { "멧", "멧", "멧", "멧", "멧" },
  { "멨", "멨", "멨", "멨", "멨" },
  { "멩", "멩", "멩", "멩", "멩" },
  { "멪", "멪", "멪", "멪", "멪" },
  { "멫", "멫", "멫", "멫", "멫" },
  { "멬", "멬", "멬", "멬", "멬" },
  { "멭", "멭", "멭", "멭", "멭" },
  { "멮", "멮", "멮", "멮", "멮" },
  { "멯", "멯", "멯", "멯", "멯" },
  { "며", "며", "며", "며", "며" },
  { "멱", "멱", "멱", "멱", "멱" },
  { "멲", "멲", "멲", "멲", "멲" },
  { "멳", "멳", "멳", "멳", "멳" },
  { "면", "면", "면", "면", "면" },
  { "멵", "멵", "멵", "멵", "멵" },
  { "멶", "멶", "멶", "멶", "멶" },
  { "멷", "멷", "멷", "멷", "멷" },
  { "멸", "멸", "멸", "멸", "멸" },
  { "멹", "멹", "멹", "멹", "멹" },
  { "멺", "멺", "멺", "멺", "멺" },
  { "멻", "멻", "멻", "멻", "멻" },
  { "멼", "멼", "멼", "멼", "멼" },
  { "멽", "멽", "멽", "멽", "멽" },
  { "멾", "멾", "멾", "멾", "멾" },
  { "멿", "멿", "멿", "멿", "멿" },
  { "몀", "몀", "몀", "몀", "몀" },
  { "몁", "몁", "몁", "몁", "몁" },
  { "몂", "몂", "몂", "몂", "몂" },
  { "몃", "몃", "몃", "몃", "몃" },
  { "몄", "몄", "몄", "몄", "몄" },
  { "명", "명", "명", "명", "명" },
  { "몆", "몆", "몆", "몆", "몆" },
  { "몇", "몇", "몇", "몇", "몇" },
  { "몈", "몈", "몈", "몈", "몈" },
  { "몉", "몉", "몉", "몉", "몉" },
  { "몊", "몊", "몊", "몊", "몊" },
  { "몋", "몋", "몋", "몋", "몋" },
  { "몌", "몌", "몌", "몌", "몌" },
  { "몍", "몍", "몍", "몍", "몍" },
  { "몎", "몎", "몎", "몎", "몎" },
  { "몏", "몏", "몏", "몏", "몏" },
  { "몐", "몐", "몐", "몐", "몐" },
  { "몑", "몑", "몑", "몑", "몑" },
  { "몒", "몒", "몒", "몒", "몒" },
  { "몓", "몓", "몓", "몓", "몓" },
  { "몔", "몔", "몔", "몔", "몔" },
  { "몕", "몕", "몕", "몕", "몕" },
  { "몖", "몖", "몖", "몖", "몖" },
  { "몗", "몗", "몗", "몗", "몗" },
  { "몘", "몘", "몘", "몘", "몘" },
  { "몙", "몙", "몙", "몙", "몙" },
  { "몚", "몚", "몚", "몚", "몚" },
  { "몛", "몛", "몛", "몛", "몛" },
  { "몜", "몜", "몜", "몜", "몜" },
  { "몝", "몝", "몝", "몝", "몝" },
  { "몞", "몞", "몞", "몞", "몞" },
  { "몟", "몟", "몟", "몟", "몟" },
  { "몠", "몠", "몠", "몠", "몠" },
  { "몡", "몡", "몡", "몡", "몡" },
  { "몢", "몢", "몢", "몢", "몢" },
  { "몣", "몣", "몣", "몣", "몣" },
  { "몤", "몤", "몤", "몤", "몤" },
  { "몥", "몥", "몥", "몥", "몥" },
  { "몦", "몦", "몦", "몦", "몦" },
  { "몧", "몧", "몧", "몧", "몧" },
  { "모", "모", "모", "모", "모" },
  { "목", "목", "목", "목", "목" },
  { "몪", "몪", "몪", "몪", "몪" },
  { "몫", "몫", "몫", "몫", "몫" },
  { "몬", "몬", "몬", "몬", "몬" },
  { "몭", "몭", "몭", "몭", "몭" },
  { "몮", "몮", "몮", "몮", "몮" },
  { "몯", "몯", "몯", "몯", "몯" },
  { "몰", "몰", "몰", "몰", "몰" },
  { "몱", "몱", "몱", "몱", "몱" },
  { "몲", "몲", "몲", "몲", "몲" },
  { "몳", "몳", "몳", "몳", "몳" },
  { "몴", "몴", "몴", "몴", "몴" },
  { "몵", "몵", "몵", "몵", "몵" },
  { "몶", "몶", "몶", "몶", "몶" },
  { "몷", "몷", "몷", "몷", "몷" },
  { "몸", "몸", "몸", "몸", "몸" },
  { "몹", "몹", "몹", "몹", "몹" },
  { "몺", "몺", "몺", "몺", "몺" },
  { "못", "못", "못", "못", "못" },
  { "몼", "몼", "몼", "몼", "몼" },
  { "몽", "몽", "몽", "몽", "몽" },
  { "몾", "몾", "몾", "몾", "몾" },
  { "몿", "몿", "몿", "몿", "몿" },
  { "뫀", "뫀", "뫀", "뫀", "뫀" },
  { "뫁", "뫁", "뫁", "뫁", "뫁" },
  { "뫂", "뫂", "뫂", "뫂", "뫂" },
  { "뫃", "뫃", "뫃", "뫃", "뫃" },
  { "뫄", "뫄", "뫄", "뫄", "뫄" },
  { "뫅", "뫅", "뫅", "뫅", "뫅" },
  { "뫆", "뫆", "뫆", "뫆", "뫆" },
  { "뫇", "뫇", "뫇", "뫇", "뫇" },
  { "뫈", "뫈", "뫈", "뫈", "뫈" },
  { "뫉", "뫉", "뫉", "뫉", "뫉" },
  { "뫊", "뫊", "뫊", "뫊", "뫊" },
  { "뫋", "뫋", "뫋", "뫋", "뫋" },
  { "뫌", "뫌", "뫌", "뫌", "뫌" },
  { "뫍", "뫍", "뫍", "뫍", "뫍" },
  { "뫎", "뫎", "뫎", "뫎", "뫎" },
  { "뫏", "뫏", "뫏", "뫏", "뫏" },
  { "뫐", "뫐", "뫐", "뫐", "뫐" },
  { "뫑", "뫑", "뫑", "뫑", "뫑" },
  { "뫒", "뫒", "뫒", "뫒", "뫒" },
  { "뫓", "뫓", "뫓", "뫓", "뫓" },
  { "뫔", "뫔", "뫔", "뫔", "뫔" },
  { "뫕", "뫕", "뫕", "뫕", "뫕" },
  { "뫖", "뫖", "뫖", "뫖", "뫖" },
  { "뫗", "뫗", "뫗", "뫗", "뫗" },
  { "뫘", "뫘", "뫘", "뫘", "뫘" },
  { "뫙", "뫙", "뫙", "뫙", "뫙" },
  { "뫚", "뫚", "뫚", "뫚", "뫚" },
  { "뫛", "뫛", "뫛", "뫛", "뫛" },
  { "뫜", "뫜", "뫜", "뫜", "뫜" },
  { "뫝", "뫝", "뫝", "뫝", "뫝" },
  { "뫞", "뫞", "뫞", "뫞", "뫞" },
  { "뫟", "뫟", "뫟", "뫟", "뫟" },
  { "뫠", "뫠", "뫠", "뫠", "뫠" },
  { "뫡", "뫡", "뫡", "뫡", "뫡" },
  { "뫢", "뫢", "뫢", "뫢", "뫢" },
  { "뫣", "뫣", "뫣", "뫣", "뫣" },
  { "뫤", "뫤", "뫤", "뫤", "뫤" },
  { "뫥", "뫥", "뫥", "뫥", "뫥" },
  { "뫦", "뫦", "뫦", "뫦", "뫦" },
  { "뫧", "뫧", "뫧", "뫧", "뫧" },
  { "뫨", "뫨", "뫨", "뫨", "뫨" },
  { "뫩", "뫩", "뫩", "뫩", "뫩" },
  { "뫪", "뫪", "뫪", "뫪", "뫪" },
  { "뫫", "뫫", "뫫", "뫫", "뫫" },
  { "뫬", "뫬", "뫬", "뫬", "뫬" },
  { "뫭", "뫭", "뫭", "뫭", "뫭" },
  { "뫮", "뫮", "뫮", "뫮", "뫮" },
  { "뫯", "뫯", "뫯", "뫯", "뫯" },
  { "뫰", "뫰", "뫰", "뫰", "뫰" },
  { "뫱", "뫱", "뫱", "뫱", "뫱" },
  { "뫲", "뫲", "뫲", "뫲", "뫲" },
  { "뫳", "뫳", "뫳", "뫳", "뫳" },
  { "뫴", "뫴", "뫴", "뫴", "뫴" },
  { "뫵", "뫵", "뫵", "뫵", "뫵" },
  { "뫶", "뫶", "뫶", "뫶", "뫶" },
  { "뫷", "뫷", "뫷", "뫷", "뫷" },
  { "뫸", "뫸", "뫸", "뫸", "뫸" },
  { "뫹", "뫹", "뫹", "뫹", "뫹" },
  { "뫺", "뫺", "뫺", "뫺", "뫺" },
  { "뫻", "뫻", "뫻", "뫻", "뫻" },
  { "뫼", "뫼", "뫼", "뫼", "뫼" },
  { "뫽", "뫽", "뫽", "뫽", "뫽" },
  { "뫾", "뫾", "뫾", "뫾", "뫾" },
  { "뫿", "뫿", "뫿", "뫿", "뫿" },
  { "묀", "묀", "묀", "묀", "묀" },
  { "묁", "묁", "묁", "묁", "묁" },
  { "묂", "묂", "묂", "묂", "묂" },
  { "묃", "묃", "묃", "묃", "묃" },
  { "묄", "묄", "묄", "묄", "묄" },
  { "묅", "묅", "묅", "묅", "묅" },
  { "묆", "묆", "묆", "묆", "묆" },
  { "묇", "묇", "묇", "묇", "묇" },
  { "묈", "묈", "묈", "묈", "묈" },
  { "묉", "묉", "묉", "묉", "묉" },
  { "묊", "묊", "묊", "묊", "묊" },
  { "묋", "묋", "묋", "묋", "묋" },
  { "묌", "묌", "묌", "묌", "묌" },
  { "묍", "묍", "묍", "묍", "묍" },
  { "묎", "묎", "묎", "묎", "묎" },
  { "묏", "묏", "묏", "묏", "묏" },
  { "묐", "묐", "묐", "묐", "묐" },
  { "묑", "묑", "묑", "묑", "묑" },
  { "묒", "묒", "묒", "묒", "묒" },
  { "묓", "묓", "묓", "묓", "묓" },
  { "묔", "묔", "묔", "묔", "묔" },
  { "묕", "묕", "묕", "묕", "묕" },
  { "묖", "묖", "묖", "묖", "묖" },
  { "묗", "묗", "묗", "묗", "묗" },
  { "묘", "묘", "묘", "묘", "묘" },
  { "묙", "묙", "묙", "묙", "묙" },
  { "묚", "묚", "묚", "묚", "묚" },
  { "묛", "묛", "묛", "묛", "묛" },
  { "묜", "묜", "묜", "묜", "묜" },
  { "묝", "묝", "묝", "묝", "묝" },
  { "묞", "묞", "묞", "묞", "묞" },
  { "묟", "묟", "묟", "묟", "묟" },
  { "묠", "묠", "묠", "묠", "묠" },
  { "묡", "묡", "묡", "묡", "묡" },
  { "묢", "묢", "묢", "묢", "묢" },
  { "묣", "묣", "묣", "묣", "묣" },
  { "묤", "묤", "묤", "묤", "묤" },
  { "묥", "묥", "묥", "묥", "묥" },
  { "묦", "묦", "묦", "묦", "묦" },
  { "묧", "묧", "묧", "묧", "묧" },
  { "묨", "묨", "묨", "묨", "묨" },
  { "묩", "묩", "묩", "묩", "묩" },
  { "묪", "묪", "묪", "묪", "묪" },
  { "묫", "묫", "묫", "묫", "묫" },
  { "묬", "묬", "묬", "묬", "묬" },
  { "묭", "묭", "묭", "묭", "묭" },
  { "묮", "묮", "묮", "묮", "묮" },
  { "묯", "묯", "묯", "묯", "묯" },
  { "묰", "묰", "묰", "묰", "묰" },
  { "묱", "묱", "묱", "묱", "묱" },
  { "묲", "묲", "묲", "묲", "묲" },
  { "묳", "묳", "묳", "묳", "묳" },
  { "무", "무", "무", "무", "무" },
  { "묵", "묵", "묵", "묵", "묵" },
  { "묶", "묶", "묶", "묶", "묶" },
  { "묷", "묷", "묷", "묷", "묷" },
  { "문", "문", "문", "문", "문" },
  { "묹", "묹", "묹", "묹", "묹" },
  { "묺", "묺", "묺", "묺", "묺" },
  { "묻", "묻", "묻", "묻", "묻" },
  { "물", "물", "물", "물", "물" },
  { "묽", "묽", "묽", "묽", "묽" },
  { "묾", "묾", "묾", "묾", "묾" },
  { "묿", "묿", "묿", "묿", "묿" },
  { "뭀", "뭀", "뭀", "뭀", "뭀" },
  { "뭁", "뭁", "뭁", "뭁", "뭁" },
  { "뭂", "뭂", "뭂", "뭂", "뭂" },
  { "뭃", "뭃", "뭃", "뭃", "뭃" },
  { "뭄", "뭄", "뭄", "뭄", "뭄" },
  { "뭅", "뭅", "뭅", "뭅", "뭅" },
  { "뭆", "뭆", "뭆", "뭆", "뭆" },
  { "뭇", "뭇", "뭇", "뭇", "뭇" },
  { "뭈", "뭈", "뭈", "뭈", "뭈" },
  { "뭉", "뭉", "뭉", "뭉", "뭉" },
  { "뭊", "뭊", "뭊", "뭊", "뭊" },
  { "뭋", "뭋", "뭋", "뭋", "뭋" },
  { "뭌", "뭌", "뭌", "뭌", "뭌" },
  { "뭍", "뭍", "뭍", "뭍", "뭍" },
  { "뭎", "뭎", "뭎", "뭎", "뭎" },
  { "뭏", "뭏", "뭏", "뭏", "뭏" },
  { "뭐", "뭐", "뭐", "뭐", "뭐" },
  { "뭑", "뭑", "뭑", "뭑", "뭑" },
  { "뭒", "뭒", "뭒", "뭒", "뭒" },
  { "뭓", "뭓", "뭓", "뭓", "뭓" },
  { "뭔", "뭔", "뭔", "뭔", "뭔" },
  { "뭕", "뭕", "뭕", "뭕", "뭕" },
  { "뭖", "뭖", "뭖", "뭖", "뭖" },
  { "뭗", "뭗", "뭗", "뭗", "뭗" },
  { "뭘", "뭘", "뭘", "뭘", "뭘" },
  { "뭙", "뭙", "뭙", "뭙", "뭙" },
  { "뭚", "뭚", "뭚", "뭚", "뭚" },
  { "뭛", "뭛", "뭛", "뭛", "뭛" },
  { "뭜", "뭜", "뭜", "뭜", "뭜" },
  { "뭝", "뭝", "뭝", "뭝", "뭝" },
  { "뭞", "뭞", "뭞", "뭞", "뭞" },
  { "뭟", "뭟", "뭟", "뭟", "뭟" },
  { "뭠", "뭠", "뭠", "뭠", "뭠" },
  { "뭡", "뭡", "뭡", "뭡", "뭡" },
  { "뭢", "뭢", "뭢", "뭢", "뭢" },
  { "뭣", "뭣", "뭣", "뭣", "뭣" },
  { "뭤", "뭤", "뭤", "뭤", "뭤" },
  { "뭥", "뭥", "뭥", "뭥", "뭥" },
  { "뭦", "뭦", "뭦", "뭦", "뭦" },
  { "뭧", "뭧", "뭧", "뭧", "뭧" },
  { "뭨", "뭨", "뭨", "뭨", "뭨" },
  { "뭩", "뭩", "뭩", "뭩", "뭩" },
  { "뭪", "뭪", "뭪", "뭪", "뭪" },
  { "뭫", "뭫", "뭫", "뭫", "뭫" },
  { "뭬", "뭬", "뭬", "뭬", "뭬" },
  { "뭭", "뭭", "뭭", "뭭", "뭭" },
  { "뭮", "뭮", "뭮", "뭮", "뭮" },
  { "뭯", "뭯", "뭯", "뭯", "뭯" },
  { "뭰", "뭰", "뭰", "뭰", "뭰" },
  { "뭱", "뭱", "뭱", "뭱", "뭱" },
  { "뭲", "뭲", "뭲", "뭲", "뭲" },
  { "뭳", "뭳", "뭳", "뭳", "뭳" },
  { "뭴", "뭴", "뭴", "뭴", "뭴" },
  { "뭵", "뭵", "뭵", "뭵", "뭵" },
  { "뭶", "뭶", "뭶", "뭶", "뭶" },
  { "뭷", "뭷", "뭷", "뭷", "뭷" },
  { "뭸", "뭸", "뭸", "뭸", "뭸" },
  { "뭹", "뭹", "뭹", "뭹", "뭹" },
  { "뭺", "뭺", "뭺", "뭺", "뭺" },
  { "뭻", "뭻", "뭻", "뭻", "뭻" },
  { "뭼", "뭼", "뭼", "뭼", "뭼" },
  { "뭽", "뭽", "뭽", "뭽", "뭽" },
  { "뭾", "뭾", "뭾", "뭾", "뭾" },
  { "뭿", "뭿", "뭿", "뭿", "뭿" },
  { "뮀", "뮀", "뮀", "뮀", "뮀" },
  { "뮁", "뮁", "뮁", "뮁", "뮁" },
  { "뮂", "뮂", "뮂", "뮂", "뮂" },
  { "뮃", "뮃", "뮃", "뮃", "뮃" },
  { "뮄", "뮄", "뮄", "뮄", "뮄" },
  { "뮅", "뮅", "뮅", "뮅", "뮅" },
  { "뮆", "뮆", "뮆", "뮆", "뮆" },
  { "뮇", "뮇", "뮇", "뮇", "뮇" },
  { "뮈", "뮈", "뮈", "뮈", "뮈" },
  { "뮉", "뮉", "뮉", "뮉", "뮉" },
  { "뮊", "뮊", "뮊", "뮊", "뮊" },
  { "뮋", "뮋", "뮋", "뮋", "뮋" },
  { "뮌", "뮌", "뮌", "뮌", "뮌" },
  { "뮍", "뮍", "뮍", "뮍", "뮍" },
  { "뮎", "뮎", "뮎", "뮎", "뮎" },
  { "뮏", "뮏", "뮏", "뮏", "뮏" },
  { "뮐", "뮐", "뮐", "뮐", "뮐" },
  { "뮑", "뮑", "뮑", "뮑", "뮑" },
  { "뮒", "뮒", "뮒", "뮒", "뮒" },
  { "뮓", "뮓", "뮓", "뮓", "뮓" },
  { "뮔", "뮔", "뮔", "뮔", "뮔" },
  { "뮕", "뮕", "뮕", "뮕", "뮕" },
  { "뮖", "뮖", "뮖", "뮖", "뮖" },
  { "뮗", "뮗", "뮗", "뮗", "뮗" },
  { "뮘", "뮘", "뮘", "뮘", "뮘" },
  { "뮙", "뮙", "뮙", "뮙", "뮙" },
  { "뮚", "뮚", "뮚", "뮚", "뮚" },
  { "뮛", "뮛", "뮛", "뮛", "뮛" },
  { "뮜", "뮜", "뮜", "뮜", "뮜" },
  { "뮝", "뮝", "뮝", "뮝", "뮝" },
  { "뮞", "뮞", "뮞", "뮞", "뮞" },
  { "뮟", "뮟", "뮟", "뮟", "뮟" },
  { "뮠", "뮠", "뮠", "뮠", "뮠" },
  { "뮡", "뮡", "뮡", "뮡", "뮡" },
  { "뮢", "뮢", "뮢", "뮢", "뮢" },
  { "뮣", "뮣", "뮣", "뮣", "뮣" },
  { "뮤", "뮤", "뮤", "뮤", "뮤" },
  { "뮥", "뮥", "뮥", "뮥", "뮥" },
  { "뮦", "뮦", "뮦", "뮦", "뮦" },
  { "뮧", "뮧", "뮧", "뮧", "뮧" },
  { "뮨", "뮨", "뮨", "뮨", "뮨" },
  { "뮩", "뮩", "뮩", "뮩", "뮩" },
  { "뮪", "뮪", "뮪", "뮪", "뮪" },
  { "뮫", "뮫", "뮫", "뮫", "뮫" },
  { "뮬", "뮬", "뮬", "뮬", "뮬" },
  { "뮭", "뮭", "뮭", "뮭", "뮭" },
  { "뮮", "뮮", "뮮", "뮮", "뮮" },
  { "뮯", "뮯", "뮯", "뮯", "뮯" },
  { "뮰", "뮰", "뮰", "뮰", "뮰" },
  { "뮱", "뮱", "뮱", "뮱", "뮱" },
  { "뮲", "뮲", "뮲", "뮲", "뮲" },
  { "뮳", "뮳", "뮳", "뮳", "뮳" },
  { "뮴", "뮴", "뮴", "뮴", "뮴" },
  { "뮵", "뮵", "뮵", "뮵", "뮵" },
  { "뮶", "뮶", "뮶", "뮶", "뮶" },
  { "뮷", "뮷", "뮷", "뮷", "뮷" },
  { "뮸", "뮸", "뮸", "뮸", "뮸" },
  { "뮹", "뮹", "뮹", "뮹", "뮹" },
  { "뮺", "뮺", "뮺", "뮺", "뮺" },
  { "뮻", "뮻", "뮻", "뮻", "뮻" },
  { "뮼", "뮼", "뮼", "뮼", "뮼" },
  { "뮽", "뮽", "뮽", "뮽", "뮽" },
  { "뮾", "뮾", "뮾", "뮾", "뮾" },
  { "뮿", "뮿", "뮿", "뮿", "뮿" },
  { "므", "므", "므", "므", "므" },
  { "믁", "믁", "믁", "믁", "믁" },
  { "믂", "믂", "믂", "믂", "믂" },
  { "믃", "믃", "믃", "믃", "믃" },
  { "믄", "믄", "믄", "믄", "믄" },
  { "믅", "믅", "믅", "믅", "믅" },
  { "믆", "믆", "믆", "믆", "믆" },
  { "믇", "믇", "믇", "믇", "믇" },
  { "믈", "믈", "믈", "믈", "믈" },
  { "믉", "믉", "믉", "믉", "믉" },
  { "믊", "믊", "믊", "믊", "믊" },
  { "믋", "믋", "믋", "믋", "믋" },
  { "믌", "믌", "믌", "믌", "믌" },
  { "믍", "믍", "믍", "믍", "믍" },
  { "믎", "믎", "믎", "믎", "믎" },
  { "믏", "믏", "믏", "믏", "믏" },
  { "믐", "믐", "믐", "믐", "믐" },
  { "믑", "믑", "믑", "믑", "믑" },
  { "믒", "믒", "믒", "믒", "믒" },
  { "믓", "믓", "믓", "믓", "믓" },
  { "믔", "믔", "믔", "믔", "믔" },
  { "믕", "믕", "믕", "믕", "믕" },
  { "믖", "믖", "믖", "믖", "믖" },
  { "믗", "믗", "믗", "믗", "믗" },
  { "믘", "믘", "믘", "믘", "믘" },
  { "믙", "믙", "믙", "믙", "믙" },
  { "믚", "믚", "믚", "믚", "믚" },
  { "믛", "믛", "믛", "믛", "믛" },
  { "믜", "믜", "믜", "믜", "믜" },
  { "믝", "믝", "믝", "믝", "믝" },
  { "믞", "믞", "믞", "믞", "믞" },
  { "믟", "믟", "믟", "믟", "믟" },
  { "믠", "믠", "믠", "믠", "믠" },
  { "믡", "믡", "믡", "믡", "믡" },
  { "믢", "믢", "믢", "믢", "믢" },
  { "믣", "믣", "믣", "믣", "믣" },
  { "믤", "믤", "믤", "믤", "믤" },
  { "믥", "믥", "믥", "믥", "믥" },
  { "믦", "믦", "믦", "믦", "믦" },
  { "믧", "믧", "믧", "믧", "믧" },
  { "믨", "믨", "믨", "믨", "믨" },
  { "믩", "믩", "믩", "믩", "믩" },
  { "믪", "믪", "믪", "믪", "믪" },
  { "믫", "믫", "믫", "믫", "믫" },
  { "믬", "믬", "믬", "믬", "믬" },
  { "믭", "믭", "믭", "믭", "믭" },
  { "믮", "믮", "믮", "믮", "믮" },
  { "믯", "믯", "믯", "믯", "믯" },
  { "믰", "믰", "믰", "믰", "믰" },
  { "믱", "믱", "믱", "믱", "믱" },
  { "믲", "믲", "믲", "믲", "믲" },
  { "믳", "믳", "믳", "믳", "믳" },
  { "믴", "믴", "믴", "믴", "믴" },
  { "믵", "믵", "믵", "믵", "믵" },
  { "믶", "믶", "믶", "믶", "믶" },
  { "믷", "믷", "믷", "믷", "믷" },
  { "미", "미", "미", "미", "미" },
  { "믹", "믹", "믹", "믹", "믹" },
  { "믺", "믺", "믺", "믺", "믺" },
  { "믻", "믻", "믻", "믻", "믻" },
  { "민", "민", "민", "민", "민" },
  { "믽", "믽", "믽", "믽", "믽" },
  { "믾", "믾", "믾", "믾", "믾" },
  { "믿", "믿", "믿", "믿", "믿" },
  { "밀", "밀", "밀", "밀", "밀" },
  { "밁", "밁", "밁", "밁", "밁" },
  { "밂", "밂", "밂", "밂", "밂" },
  { "밃", "밃", "밃", "밃", "밃" },
  { "밄", "밄", "밄", "밄", "밄" },
  { "밅", "밅", "밅", "밅", "밅" },
  { "밆", "밆", "밆", "밆", "밆" },
  { "밇", "밇", "밇", "밇", "밇" },
  { "밈", "밈", "밈", "밈", "밈" },
  { "밉", "밉", "밉", "밉", "밉" },
  { "밊", "밊", "밊", "밊", "밊" },
  { "밋", "밋", "밋", "밋", "밋" },
  { "밌", "밌", "밌", "밌", "밌" },
  { "밍", "밍", "밍", "밍", "밍" },
  { "밎", "밎", "밎", "밎", "밎" },
  { "및", "및", "및", "및", "및" },
  { "밐", "밐", "밐", "밐", "밐" },
  { "밑", "밑", "밑", "밑", "밑" },
  { "밒", "밒", "밒", "밒", "밒" },
  { "밓", "밓", "밓", "밓", "밓" },
  { "바", "바", "바", "바", "바" },
  { "박", "박", "박", "박", "박" },
  { "밖", "밖", "밖", "밖", "밖" },
  { "밗", "밗", "밗", "밗", "밗" },
  { "반", "반", "반", "반", "반" },
  { "밙", "밙", "밙", "밙", "밙" },
  { "밚", "밚", "밚", "밚", "밚" },
  { "받", "받", "받", "받", "받" },
  { "발", "발", "발", "발", "발" },
  { "밝", "밝", "밝", "밝", "밝" },
  { "밞", "밞", "밞", "밞", "밞" },
  { "밟", "밟", "밟", "밟", "밟" },
  { "밠", "밠", "밠", "밠", "밠" },
  { "밡", "밡", "밡", "밡", "밡" },
  { "밢", "밢", "밢", "밢", "밢" },
  { "밣", "밣", "밣", "밣", "밣" },
  { "밤", "밤", "밤", "밤", "밤" },
  { "밥", "밥", "밥", "밥", "밥" },
  { "밦", "밦", "밦", "밦", "밦" },
  { "밧", "밧", "밧", "밧", "밧" },
  { "밨", "밨", "밨", "밨", "밨" },
  { "방", "방", "방", "방", "방" },
  { "밪", "밪", "밪", "밪", "밪" },
  { "밫", "밫", "밫", "밫", "밫" },
  { "밬", "밬", "밬", "밬", "밬" },
  { "밭", "밭", "밭", "밭", "밭" },
  { "밮", "밮", "밮", "밮", "밮" },
  { "밯", "밯", "밯", "밯", "밯" },
  { "배", "배", "배", "배", "배" },
  { "백", "백", "백", "백", "백" },
  { "밲", "밲", "밲", "밲", "밲" },
  { "밳", "밳", "밳", "밳", "밳" },
  { "밴", "밴", "밴", "밴", "밴" },
  { "밵", "밵", "밵", "밵", "밵" },
  { "밶", "밶", "밶", "밶", "밶" },
  { "밷", "밷", "밷", "밷", "밷" },
  { "밸", "밸", "밸", "밸", "밸" },
  { "밹", "밹", "밹", "밹", "밹" },
  { "밺", "밺", "밺", "밺", "밺" },
  { "밻", "밻", "밻", "밻", "밻" },
  { "밼", "밼", "밼", "밼", "밼" },
  { "밽", "밽", "밽", "밽", "밽" },
  { "밾", "밾", "밾", "밾", "밾" },
  { "밿", "밿", "밿", "밿", "밿" },
  { "뱀", "뱀", "뱀", "뱀", "뱀" },
  { "뱁", "뱁", "뱁", "뱁", "뱁" },
  { "뱂", "뱂", "뱂", "뱂", "뱂" },
  { "뱃", "뱃", "뱃", "뱃", "뱃" },
  { "뱄", "뱄", "뱄", "뱄", "뱄" },
  { "뱅", "뱅", "뱅", "뱅", "뱅" },
  { "뱆", "뱆", "뱆", "뱆", "뱆" },
  { "뱇", "뱇", "뱇", "뱇", "뱇" },
  { "뱈", "뱈", "뱈", "뱈", "뱈" },
  { "뱉", "뱉", "뱉", "뱉", "뱉" },
  { "뱊", "뱊", "뱊", "뱊", "뱊" },
  { "뱋", "뱋", "뱋", "뱋", "뱋" },
  { "뱌", "뱌", "뱌", "뱌", "뱌" },
  { "뱍", "뱍", "뱍", "뱍", "뱍" },
  { "뱎", "뱎", "뱎", "뱎", "뱎" },
  { "뱏", "뱏", "뱏", "뱏", "뱏" },
  { "뱐", "뱐", "뱐", "뱐", "뱐" },
  { "뱑", "뱑", "뱑", "뱑", "뱑" },
  { "뱒", "뱒", "뱒", "뱒", "뱒" },
  { "뱓", "뱓", "뱓", "뱓", "뱓" },
  { "뱔", "뱔", "뱔", "뱔", "뱔" },
  { "뱕", "뱕", "뱕", "뱕", "뱕" },
  { "뱖", "뱖", "뱖", "뱖", "뱖" },
  { "뱗", "뱗", "뱗", "뱗", "뱗" },
  { "뱘", "뱘", "뱘", "뱘", "뱘" },
  { "뱙", "뱙", "뱙", "뱙", "뱙" },
  { "뱚", "뱚", "뱚", "뱚", "뱚" },
  { "뱛", "뱛", "뱛", "뱛", "뱛" },
  { "뱜", "뱜", "뱜", "뱜", "뱜" },
  { "뱝", "뱝", "뱝", "뱝", "뱝" },
  { "뱞", "뱞", "뱞", "뱞", "뱞" },
  { "뱟", "뱟", "뱟", "뱟", "뱟" },
  { "뱠", "뱠", "뱠", "뱠", "뱠" },
  { "뱡", "뱡", "뱡", "뱡", "뱡" },
  { "뱢", "뱢", "뱢", "뱢", "뱢" },
  { "뱣", "뱣", "뱣", "뱣", "뱣" },
  { "뱤", "뱤", "뱤", "뱤", "뱤" },
  { "뱥", "뱥", "뱥", "뱥", "뱥" },
  { "뱦", "뱦", "뱦", "뱦", "뱦" },
  { "뱧", "뱧", "뱧", "뱧", "뱧" },
  { "뱨", "뱨", "뱨", "뱨", "뱨" },
  { "뱩", "뱩", "뱩", "뱩", "뱩" },
  { "뱪", "뱪", "뱪", "뱪", "뱪" },
  { "뱫", "뱫", "뱫", "뱫", "뱫" },
  { "뱬", "뱬", "뱬", "뱬", "뱬" },
  { "뱭", "뱭", "뱭", "뱭", "뱭" },
  { "뱮", "뱮", "뱮", "뱮", "뱮" },
  { "뱯", "뱯", "뱯", "뱯", "뱯" },
  { "뱰", "뱰", "뱰", "뱰", "뱰" },
  { "뱱", "뱱", "뱱", "뱱", "뱱" },
  { "뱲", "뱲", "뱲", "뱲", "뱲" },
  { "뱳", "뱳", "뱳", "뱳", "뱳" },
  { "뱴", "뱴", "뱴", "뱴", "뱴" },
  { "뱵", "뱵", "뱵", "뱵", "뱵" },
  { "뱶", "뱶", "뱶", "뱶", "뱶" },
  { "뱷", "뱷", "뱷", "뱷", "뱷" },
  { "뱸", "뱸", "뱸", "뱸", "뱸" },
  { "뱹", "뱹", "뱹", "뱹", "뱹" },
  { "뱺", "뱺", "뱺", "뱺", "뱺" },
  { "뱻", "뱻", "뱻", "뱻", "뱻" },
  { "뱼", "뱼", "뱼", "뱼", "뱼" },
  { "뱽", "뱽", "뱽", "뱽", "뱽" },
  { "뱾", "뱾", "뱾", "뱾", "뱾" },
  { "뱿", "뱿", "뱿", "뱿", "뱿" },
  { "벀", "벀", "벀", "벀", "벀" },
  { "벁", "벁", "벁", "벁", "벁" },
  { "벂", "벂", "벂", "벂", "벂" },
  { "벃", "벃", "벃", "벃", "벃" },
  { "버", "버", "버", "버", "버" },
  { "벅", "벅", "벅", "벅", "벅" },
  { "벆", "벆", "벆", "벆", "벆" },
  { "벇", "벇", "벇", "벇", "벇" },
  { "번", "번", "번", "번", "번" },
  { "벉", "벉", "벉", "벉", "벉" },
  { "벊", "벊", "벊", "벊", "벊" },
  { "벋", "벋", "벋", "벋", "벋" },
  { "벌", "벌", "벌", "벌", "벌" },
  { "벍", "벍", "벍", "벍", "벍" },
  { "벎", "벎", "벎", "벎", "벎" },
  { "벏", "벏", "벏", "벏", "벏" },
  { "벐", "벐", "벐", "벐", "벐" },
  { "벑", "벑", "벑", "벑", "벑" },
  { "벒", "벒", "벒", "벒", "벒" },
  { "벓", "벓", "벓", "벓", "벓" },
  { "범", "범", "범", "범", "범" },
  { "법", "법", "법", "법", "법" },
  { "벖", "벖", "벖", "벖", "벖" },
  { "벗", "벗", "벗", "벗", "벗" },
  { "벘", "벘", "벘", "벘", "벘" },
  { "벙", "벙", "벙", "벙", "벙" },
  { "벚", "벚", "벚", "벚", "벚" },
  { "벛", "벛", "벛", "벛", "벛" },
  { "벜", "벜", "벜", "벜", "벜" },
  { "벝", "벝", "벝", "벝", "벝" },
  { "벞", "벞", "벞", "벞", "벞" },
  { "벟", "벟", "벟", "벟", "벟" },
  { "베", "베", "베", "베", "베" },
  { "벡", "벡", "벡", "벡", "벡" },
  { "벢", "벢", "벢", "벢", "벢" },
  { "벣", "벣", "벣", "벣", "벣" },
  { "벤", "벤", "벤", "벤", "벤" },
  { "벥", "벥", "벥", "벥", "벥" },
  { "벦", "벦", "벦", "벦", "벦" },
  { "벧", "벧", "벧", "벧", "벧" },
  { "벨", "벨", "벨", "벨", "벨" },
  { "벩", "벩", "벩", "벩", "벩" },
  { "벪", "벪", "벪", "벪", "벪" },
  { "벫", "벫", "벫", "벫", "벫" },
  { "벬", "벬", "벬", "벬", "벬" },
  { "벭", "벭", "벭", "벭", "벭" },
  { "벮", "벮", "벮", "벮", "벮" },
  { "벯", "벯", "벯", "벯", "벯" },
  { "벰", "벰", "벰", "벰", "벰" },
  { "벱", "벱", "벱", "벱", "벱" },
  { "벲", "벲", "벲", "벲", "벲" },
  { "벳", "벳", "벳", "벳", "벳" },
  { "벴", "벴", "벴", "벴", "벴" },
  { "벵", "벵", "벵", "벵", "벵" },
  { "벶", "벶", "벶", "벶", "벶" },
  { "벷", "벷", "벷", "벷", "벷" },
  { "벸", "벸", "벸", "벸", "벸" },
  { "벹", "벹", "벹", "벹", "벹" },
  { "벺", "벺", "벺", "벺", "벺" },
  { "벻", "벻", "벻", "벻", "벻" },
  { "벼", "벼", "벼", "벼", "벼" },
  { "벽", "벽", "벽", "벽", "벽" },
  { "벾", "벾", "벾", "벾", "벾" },
  { "벿", "벿", "벿", "벿", "벿" },
  { "변", "변", "변", "변", "변" },
  { "볁", "볁", "볁", "볁", "볁" },
  { "볂", "볂", "볂", "볂", "볂" },
  { "볃", "볃", "볃", "볃", "볃" },
  { "별", "별", "별", "별", "별" },
  { "볅", "볅", "볅", "볅", "볅" },
  { "볆", "볆", "볆", "볆", "볆" },
  { "볇", "볇", "볇", "볇", "볇" },
  { "볈", "볈", "볈", "볈", "볈" },
  { "볉", "볉", "볉", "볉", "볉" },
  { "볊", "볊", "볊", "볊", "볊" },
  { "볋", "볋", "볋", "볋", "볋" },
  { "볌", "볌", "볌", "볌", "볌" },
  { "볍", "볍", "볍", "볍", "볍" },
  { "볎", "볎", "볎", "볎", "볎" },
  { "볏", "볏", "볏", "볏", "볏" },
  { "볐", "볐", "볐", "볐", "볐" },
  { "병", "병", "병", "병", "병" },
  { "볒", "볒", "볒", "볒", "볒" },
  { "볓", "볓", "볓", "볓", "볓" },
  { "볔", "볔", "볔", "볔", "볔" },
  { "볕", "볕", "볕", "볕", "볕" },
  { "볖", "볖", "볖", "볖", "볖" },
  { "볗", "볗", "볗", "볗", "볗" },
  { "볘", "볘", "볘", "볘", "볘" },
  { "볙", "볙", "볙", "볙", "볙" },
  { "볚", "볚", "볚", "볚", "볚" },
  { "볛", "볛", "볛", "볛", "볛" },
  { "볜", "볜", "볜", "볜", "볜" },
  { "볝", "볝", "볝", "볝", "볝" },
  { "볞", "볞", "볞", "볞", "볞" },
  { "볟", "볟", "볟", "볟", "볟" },
  { "볠", "볠", "볠", "볠", "볠" },
  { "볡", "볡", "볡", "볡", "볡" },
  { "볢", "볢", "볢", "볢", "볢" },
  { "볣", "볣", "볣", "볣", "볣" },
  { "볤", "볤", "볤", "볤", "볤" },
  { "볥", "볥", "볥", "볥", "볥" },
  { "볦", "볦", "볦", "볦", "볦" },
  { "볧", "볧", "볧", "볧", "볧" },
  { "볨", "볨", "볨", "볨", "볨" },
  { "볩", "볩", "볩", "볩", "볩" },
  { "볪", "볪", "볪", "볪", "볪" },
  { "볫", "볫", "볫", "볫", "볫" },
  { "볬", "볬", "볬", "볬", "볬" },
  { "볭", "볭", "볭", "볭", "볭" },
  { "볮", "볮", "볮", "볮", "볮" },
  { "볯", "볯", "볯", "볯", "볯" },
  { "볰", "볰", "볰", "볰", "볰" },
  { "볱", "볱", "볱", "볱", "볱" },
  { "볲", "볲", "볲", "볲", "볲" },
  { "볳", "볳", "볳", "볳", "볳" },
  { "보", "보", "보", "보", "보" },
  { "복", "복", "복", "복", "복" },
  { "볶", "볶", "볶", "볶", "볶" },
  { "볷", "볷", "볷", "볷", "볷" },
  { "본", "본", "본", "본", "본" },
  { "볹", "볹", "볹", "볹", "볹" },
  { "볺", "볺", "볺", "볺", "볺" },
  { "볻", "볻", "볻", "볻", "볻" },
  { "볼", "볼", "볼", "볼", "볼" },
  { "볽", "볽", "볽", "볽", "볽" },
  { "볾", "볾", "볾", "볾", "볾" },
  { "볿", "볿", "볿", "볿", "볿" },
  { "봀", "봀", "봀", "봀", "봀" },
  { "봁", "봁", "봁", "봁", "봁" },
  { "봂", "봂", "봂", "봂", "봂" },
  { "봃", "봃", "봃", "봃", "봃" },
  { "봄", "봄", "봄", "봄", "봄" },
  { "봅", "봅", "봅", "봅", "봅" },
  { "봆", "봆", "봆", "봆", "봆" },
  { "봇", "봇", "봇", "봇", "봇" },
  { "봈", "봈", "봈", "봈", "봈" },
  { "봉", "봉", "봉", "봉", "봉" },
  { "봊", "봊", "봊", "봊", "봊" },
  { "봋", "봋", "봋", "봋", "봋" },
  { "봌", "봌", "봌", "봌", "봌" },
  { "봍", "봍", "봍", "봍", "봍" },
  { "봎", "봎", "봎", "봎", "봎" },
  { "봏", "봏", "봏", "봏", "봏" },
  { "봐", "봐", "봐", "봐", "봐" },
  { "봑", "봑", "봑", "봑", "봑" },
  { "봒", "봒", "봒", "봒", "봒" },
  { "봓", "봓", "봓", "봓", "봓" },
  { "봔", "봔", "봔", "봔", "봔" },
  { "봕", "봕", "봕", "봕", "봕" },
  { "봖", "봖", "봖", "봖", "봖" },
  { "봗", "봗", "봗", "봗", "봗" },
  { "봘", "봘", "봘", "봘", "봘" },
  { "봙", "봙", "봙", "봙", "봙" },
  { "봚", "봚", "봚", "봚", "봚" },
  { "봛", "봛", "봛", "봛", "봛" },
  { "봜", "봜", "봜", "봜", "봜" },
  { "봝", "봝", "봝", "봝", "봝" },
  { "봞", "봞", "봞", "봞", "봞" },
  { "봟", "봟", "봟", "봟", "봟" },
  { "봠", "봠", "봠", "봠", "봠" },
  { "봡", "봡", "봡", "봡", "봡" },
  { "봢", "봢", "봢", "봢", "봢" },
  { "봣", "봣", "봣", "봣", "봣" },
  { "봤", "봤", "봤", "봤", "봤" },
  { "봥", "봥", "봥", "봥", "봥" },
  { "봦", "봦", "봦", "봦", "봦" },
  { "봧", "봧", "봧", "봧", "봧" },
  { "봨", "봨", "봨", "봨", "봨" },
  { "봩", "봩", "봩", "봩", "봩" },
  { "봪", "봪", "봪", "봪", "봪" },
  { "봫", "봫", "봫", "봫", "봫" },
  { "봬", "봬", "봬", "봬", "봬" },
  { "봭", "봭", "봭", "봭", "봭" },
  { "봮", "봮", "봮", "봮", "봮" },
  { "봯", "봯", "봯", "봯", "봯" },
  { "봰", "봰", "봰", "봰", "봰" },
  { "봱", "봱", "봱", "봱", "봱" },
  { "봲", "봲", "봲", "봲", "봲" },
  { "봳", "봳", "봳", "봳", "봳" },
  { "봴", "봴", "봴", "봴", "봴" },
  { "봵", "봵", "봵", "봵", "봵" },
  { "봶", "봶", "봶", "봶", "봶" },
  { "봷", "봷", "봷", "봷", "봷" },
  { "봸", "봸", "봸", "봸", "봸" },
  { "봹", "봹", "봹", "봹", "봹" },
  { "봺", "봺", "봺", "봺", "봺" },
  { "봻", "봻", "봻", "봻", "봻" },
  { "봼", "봼", "봼", "봼", "봼" },
  { "봽", "봽", "봽", "봽", "봽" },
  { "봾", "봾", "봾", "봾", "봾" },
  { "봿", "봿", "봿", "봿", "봿" },
  { "뵀", "뵀", "뵀", "뵀", "뵀" },
  { "뵁", "뵁", "뵁", "뵁", "뵁" },
  { "뵂", "뵂", "뵂", "뵂", "뵂" },
  { "뵃", "뵃", "뵃", "뵃", "뵃" },
  { "뵄", "뵄", "뵄", "뵄", "뵄" },
  { "뵅", "뵅", "뵅", "뵅", "뵅" },
  { "뵆", "뵆", "뵆", "뵆", "뵆" },
  { "뵇", "뵇", "뵇", "뵇", "뵇" },
  { "뵈", "뵈", "뵈", "뵈", "뵈" },
  { "뵉", "뵉", "뵉", "뵉", "뵉" },
  { "뵊", "뵊", "뵊", "뵊", "뵊" },
  { "뵋", "뵋", "뵋", "뵋", "뵋" },
  { "뵌", "뵌", "뵌", "뵌", "뵌" },
  { "뵍", "뵍", "뵍", "뵍", "뵍" },
  { "뵎", "뵎", "뵎", "뵎", "뵎" },
  { "뵏", "뵏", "뵏", "뵏", "뵏" },
  { "뵐", "뵐", "뵐", "뵐", "뵐" },
  { "뵑", "뵑", "뵑", "뵑", "뵑" },
  { "뵒", "뵒", "뵒", "뵒", "뵒" },
  { "뵓", "뵓", "뵓", "뵓", "뵓" },
  { "뵔", "뵔", "뵔", "뵔", "뵔" },
  { "뵕", "뵕", "뵕", "뵕", "뵕" },
  { "뵖", "뵖", "뵖", "뵖", "뵖" },
  { "뵗", "뵗", "뵗", "뵗", "뵗" },
  { "뵘", "뵘", "뵘", "뵘", "뵘" },
  { "뵙", "뵙", "뵙", "뵙", "뵙" },
  { "뵚", "뵚", "뵚", "뵚", "뵚" },
  { "뵛", "뵛", "뵛", "뵛", "뵛" },
  { "뵜", "뵜", "뵜", "뵜", "뵜" },
  { "뵝", "뵝", "뵝", "뵝", "뵝" },
  { "뵞", "뵞", "뵞", "뵞", "뵞" },
  { "뵟", "뵟", "뵟", "뵟", "뵟" },
  { "뵠", "뵠", "뵠", "뵠", "뵠" },
  { "뵡", "뵡", "뵡", "뵡", "뵡" },
  { "뵢", "뵢", "뵢", "뵢", "뵢" },
  { "뵣", "뵣", "뵣", "뵣", "뵣" },
  { "뵤", "뵤", "뵤", "뵤", "뵤" },
  { "뵥", "뵥", "뵥", "뵥", "뵥" },
  { "뵦", "뵦", "뵦", "뵦", "뵦" },
  { "뵧", "뵧", "뵧", "뵧", "뵧" },
  { "뵨", "뵨", "뵨", "뵨", "뵨" },
  { "뵩", "뵩", "뵩", "뵩", "뵩" },
  { "뵪", "뵪", "뵪", "뵪", "뵪" },
  { "뵫", "뵫", "뵫", "뵫", "뵫" },
  { "뵬", "뵬", "뵬", "뵬", "뵬" },
  { "뵭", "뵭", "뵭", "뵭", "뵭" },
  { "뵮", "뵮", "뵮", "뵮", "뵮" },
  { "뵯", "뵯", "뵯", "뵯", "뵯" },
  { "뵰", "뵰", "뵰", "뵰", "뵰" },
  { "뵱", "뵱", "뵱", "뵱", "뵱" },
  { "뵲", "뵲", "뵲", "뵲", "뵲" },
  { "뵳", "뵳", "뵳", "뵳", "뵳" },
  { "뵴", "뵴", "뵴", "뵴", "뵴" },
  { "뵵", "뵵", "뵵", "뵵", "뵵" },
  { "뵶", "뵶", "뵶", "뵶", "뵶" },
  { "뵷", "뵷", "뵷", "뵷", "뵷" },
  { "뵸", "뵸", "뵸", "뵸", "뵸" },
  { "뵹", "뵹", "뵹", "뵹", "뵹" },
  { "뵺", "뵺", "뵺", "뵺", "뵺" },
  { "뵻", "뵻", "뵻", "뵻", "뵻" },
  { "뵼", "뵼", "뵼", "뵼", "뵼" },
  { "뵽", "뵽", "뵽", "뵽", "뵽" },
  { "뵾", "뵾", "뵾", "뵾", "뵾" },
  { "뵿", "뵿", "뵿", "뵿", "뵿" },
  { "부", "부", "부", "부", "부" },
  { "북", "북", "북", "북", "북" },
  { "붂", "붂", "붂", "붂", "붂" },
  { "붃", "붃", "붃", "붃", "붃" },
  { "분", "분", "분", "분", "분" },
  { "붅", "붅", "붅", "붅", "붅" },
  { "붆", "붆", "붆", "붆", "붆" },
  { "붇", "붇", "붇", "붇", "붇" },
  { "불", "불", "불", "불", "불" },
  { "붉", "붉", "붉", "붉", "붉" },
  { "붊", "붊", "붊", "붊", "붊" },
  { "붋", "붋", "붋", "붋", "붋" },
  { "붌", "붌", "붌", "붌", "붌" },
  { "붍", "붍", "붍", "붍", "붍" },
  { "붎", "붎", "붎", "붎", "붎" },
  { "붏", "붏", "붏", "붏", "붏" },
  { "붐", "붐", "붐", "붐", "붐" },
  { "붑", "붑", "붑", "붑", "붑" },
  { "붒", "붒", "붒", "붒", "붒" },
  { "붓", "붓", "붓", "붓", "붓" },
  { "붔", "붔", "붔", "붔", "붔" },
  { "붕", "붕", "붕", "붕", "붕" },
  { "붖", "붖", "붖", "붖", "붖" },
  { "붗", "붗", "붗", "붗", "붗" },
  { "붘", "붘", "붘", "붘", "붘" },
  { "붙", "붙", "붙", "붙", "붙" },
  { "붚", "붚", "붚", "붚", "붚" },
  { "붛", "붛", "붛", "붛", "붛" },
  { "붜", "붜", "붜", "붜", "붜" },
  { "붝", "붝", "붝", "붝", "붝" },
  { "붞", "붞", "붞", "붞", "붞" },
  { "붟", "붟", "붟", "붟", "붟" },
  { "붠", "붠", "붠", "붠", "붠" },
  { "붡", "붡", "붡", "붡", "붡" },
  { "붢", "붢", "붢", "붢", "붢" },
  { "붣", "붣", "붣", "붣", "붣" },
  { "붤", "붤", "붤", "붤", "붤" },
  { "붥", "붥", "붥", "붥", "붥" },
  { "붦", "붦", "붦", "붦", "붦" },
  { "붧", "붧", "붧", "붧", "붧" },
  { "붨", "붨", "붨", "붨", "붨" },
  { "붩", "붩", "붩", "붩", "붩" },
  { "붪", "붪", "붪", "붪", "붪" },
  { "붫", "붫", "붫", "붫", "붫" },
  { "붬", "붬", "붬", "붬", "붬" },
  { "붭", "붭", "붭", "붭", "붭" },
  { "붮", "붮", "붮", "붮", "붮" },
  { "붯", "붯", "붯", "붯", "붯" },
  { "붰", "붰", "붰", "붰", "붰" },
  { "붱", "붱", "붱", "붱", "붱" },
  { "붲", "붲", "붲", "붲", "붲" },
  { "붳", "붳", "붳", "붳", "붳" },
  { "붴", "붴", "붴", "붴", "붴" },
  { "붵", "붵", "붵", "붵", "붵" },
  { "붶", "붶", "붶", "붶", "붶" },
  { "붷", "붷", "붷", "붷", "붷" },
  { "붸", "붸", "붸", "붸", "붸" },
  { "붹", "붹", "붹", "붹", "붹" },
  { "붺", "붺", "붺", "붺", "붺" },
  { "붻", "붻", "붻", "붻", "붻" },
  { "붼", "붼", "붼", "붼", "붼" },
  { "붽", "붽", "붽", "붽", "붽" },
  { "붾", "붾", "붾", "붾", "붾" },
  { "붿", "붿", "붿", "붿", "붿" },
  { "뷀", "뷀", "뷀", "뷀", "뷀" },
  { "뷁", "뷁", "뷁", "뷁", "뷁" },
  { "뷂", "뷂", "뷂", "뷂", "뷂" },
  { "뷃", "뷃", "뷃", "뷃", "뷃" },
  { "뷄", "뷄", "뷄", "뷄", "뷄" },
  { "뷅", "뷅", "뷅", "뷅", "뷅" },
  { "뷆", "뷆", "뷆", "뷆", "뷆" },
  { "뷇", "뷇", "뷇", "뷇", "뷇" },
  { "뷈", "뷈", "뷈", "뷈", "뷈" },
  { "뷉", "뷉", "뷉", "뷉", "뷉" },
  { "뷊", "뷊", "뷊", "뷊", "뷊" },
  { "뷋", "뷋", "뷋", "뷋", "뷋" },
  { "뷌", "뷌", "뷌", "뷌", "뷌" },
  { "뷍", "뷍", "뷍", "뷍", "뷍" },
  { "뷎", "뷎", "뷎", "뷎", "뷎" },
  { "뷏", "뷏", "뷏", "뷏", "뷏" },
  { "뷐", "뷐", "뷐", "뷐", "뷐" },
  { "뷑", "뷑", "뷑", "뷑", "뷑" },
  { "뷒", "뷒", "뷒", "뷒", "뷒" },
  { "뷓", "뷓", "뷓", "뷓", "뷓" },
  { "뷔", "뷔", "뷔", "뷔", "뷔" },
  { "뷕", "뷕", "뷕", "뷕", "뷕" },
  { "뷖", "뷖", "뷖", "뷖", "뷖" },
  { "뷗", "뷗", "뷗", "뷗", "뷗" },
  { "뷘", "뷘", "뷘", "뷘", "뷘" },
  { "뷙", "뷙", "뷙", "뷙", "뷙" },
  { "뷚", "뷚", "뷚", "뷚", "뷚" },
  { "뷛", "뷛", "뷛", "뷛", "뷛" },
  { "뷜", "뷜", "뷜", "뷜", "뷜" },
  { "뷝", "뷝", "뷝", "뷝", "뷝" },
  { "뷞", "뷞", "뷞", "뷞", "뷞" },
  { "뷟", "뷟", "뷟", "뷟", "뷟" },
  { "뷠", "뷠", "뷠", "뷠", "뷠" },
  { "뷡", "뷡", "뷡", "뷡", "뷡" },
  { "뷢", "뷢", "뷢", "뷢", "뷢" },
  { "뷣", "뷣", "뷣", "뷣", "뷣" },
  { "뷤", "뷤", "뷤", "뷤", "뷤" },
  { "뷥", "뷥", "뷥", "뷥", "뷥" },
  { "뷦", "뷦", "뷦", "뷦", "뷦" },
  { "뷧", "뷧", "뷧", "뷧", "뷧" },
  { "뷨", "뷨", "뷨", "뷨", "뷨" },
  { "뷩", "뷩", "뷩", "뷩", "뷩" },
  { "뷪", "뷪", "뷪", "뷪", "뷪" },
  { "뷫", "뷫", "뷫", "뷫", "뷫" },
  { "뷬", "뷬", "뷬", "뷬", "뷬" },
  { "뷭", "뷭", "뷭", "뷭", "뷭" },
  { "뷮", "뷮", "뷮", "뷮", "뷮" },
  { "뷯", "뷯", "뷯", "뷯", "뷯" },
  { "뷰", "뷰", "뷰", "뷰", "뷰" },
  { "뷱", "뷱", "뷱", "뷱", "뷱" },
  { "뷲", "뷲", "뷲", "뷲", "뷲" },
  { "뷳", "뷳", "뷳", "뷳", "뷳" },
  { "뷴", "뷴", "뷴", "뷴", "뷴" },
  { "뷵", "뷵", "뷵", "뷵", "뷵" },
  { "뷶", "뷶", "뷶", "뷶", "뷶" },
  { "뷷", "뷷", "뷷", "뷷", "뷷" },
  { "뷸", "뷸", "뷸", "뷸", "뷸" },
  { "뷹", "뷹", "뷹", "뷹", "뷹" },
  { "뷺", "뷺", "뷺", "뷺", "뷺" },
  { "뷻", "뷻", "뷻", "뷻", "뷻" },
  { "뷼", "뷼", "뷼", "뷼", "뷼" },
  { "뷽", "뷽", "뷽", "뷽", "뷽" },
  { "뷾", "뷾", "뷾", "뷾", "뷾" },
  { "뷿", "뷿", "뷿", "뷿", "뷿" },
  { "븀", "븀", "븀", "븀", "븀" },
  { "븁", "븁", "븁", "븁", "븁" },
  { "븂", "븂", "븂", "븂", "븂" },
  { "븃", "븃", "븃", "븃", "븃" },
  { "븄", "븄", "븄", "븄", "븄" },
  { "븅", "븅", "븅", "븅", "븅" },
  { "븆", "븆", "븆", "븆", "븆" },
  { "븇", "븇", "븇", "븇", "븇" },
  { "븈", "븈", "븈", "븈", "븈" },
  { "븉", "븉", "븉", "븉", "븉" },
  { "븊", "븊", "븊", "븊", "븊" },
  { "븋", "븋", "븋", "븋", "븋" },
  { "브", "브", "브", "브", "브" },
  { "븍", "븍", "븍", "븍", "븍" },
  { "븎", "븎", "븎", "븎", "븎" },
  { "븏", "븏", "븏", "븏", "븏" },
  { "븐", "븐", "븐", "븐", "븐" },
  { "븑", "븑", "븑", "븑", "븑" },
  { "븒", "븒", "븒", "븒", "븒" },
  { "븓", "븓", "븓", "븓", "븓" },
  { "블", "블", "블", "블", "블" },
  { "븕", "븕", "븕", "븕", "븕" },
  { "븖", "븖", "븖", "븖", "븖" },
  { "븗", "븗", "븗", "븗", "븗" },
  { "븘", "븘", "븘", "븘", "븘" },
  { "븙", "븙", "븙", "븙", "븙" },
  { "븚", "븚", "븚", "븚", "븚" },
  { "븛", "븛", "븛", "븛", "븛" },
  { "븜", "븜", "븜", "븜", "븜" },
  { "븝", "븝", "븝", "븝", "븝" },
  { "븞", "븞", "븞", "븞", "븞" },
  { "븟", "븟", "븟", "븟", "븟" },
  { "븠", "븠", "븠", "븠", "븠" },
  { "븡", "븡", "븡", "븡", "븡" },
  { "븢", "븢", "븢", "븢", "븢" },
  { "븣", "븣", "븣", "븣", "븣" },
  { "븤", "븤", "븤", "븤", "븤" },
  { "븥", "븥", "븥", "븥", "븥" },
  { "븦", "븦", "븦", "븦", "븦" },
  { "븧", "븧", "븧", "븧", "븧" },
  { "븨", "븨", "븨", "븨", "븨" },
  { "븩", "븩", "븩", "븩", "븩" },
  { "븪", "븪", "븪", "븪", "븪" },
  { "븫", "븫", "븫", "븫", "븫" },
  { "븬", "븬", "븬", "븬", "븬" },
  { "븭", "븭", "븭", "븭", "븭" },
  { "븮", "븮", "븮", "븮", "븮" },
  { "븯", "븯", "븯", "븯", "븯" },
  { "븰", "븰", "븰", "븰", "븰" },
  { "븱", "븱", "븱", "븱", "븱" },
  { "븲", "븲", "븲", "븲", "븲" },
  { "븳", "븳", "븳", "븳", "븳" },
  { "븴", "븴", "븴", "븴", "븴" },
  { "븵", "븵", "븵", "븵", "븵" },
  { "븶", "븶", "븶", "븶", "븶" },
  { "븷", "븷", "븷", "븷", "븷" },
  { "븸", "븸", "븸", "븸", "븸" },
  { "븹", "븹", "븹", "븹", "븹" },
  { "븺", "븺", "븺", "븺", "븺" },
  { "븻", "븻", "븻", "븻", "븻" },
  { "븼", "븼", "븼", "븼", "븼" },
  { "븽", "븽", "븽", "븽", "븽" },
  { "븾", "븾", "븾", "븾", "븾" },
  { "븿", "븿", "븿", "븿", "븿" },
  { "빀", "빀", "빀", "빀", "빀" },
  { "빁", "빁", "빁", "빁", "빁" },
  { "빂", "빂", "빂", "빂", "빂" },
  { "빃", "빃", "빃", "빃", "빃" },
  { "비", "비", "비", "비", "비" },
  { "빅", "빅", "빅", "빅", "빅" },
  { "빆", "빆", "빆", "빆", "빆" },
  { "빇", "빇", "빇", "빇", "빇" },
  { "빈", "빈", "빈", "빈", "빈" },
  { "빉", "빉", "빉", "빉", "빉" },
  { "빊", "빊", "빊", "빊", "빊" },
  { "빋", "빋", "빋", "빋", "빋" },
  { "빌", "빌", "빌", "빌", "빌" },
  { "빍", "빍", "빍", "빍", "빍" },
  { "빎", "빎", "빎", "빎", "빎" },
  { "빏", "빏", "빏", "빏", "빏" },
  { "빐", "빐", "빐", "빐", "빐" },
  { "빑", "빑", "빑", "빑", "빑" },
  { "빒", "빒", "빒", "빒", "빒" },
  { "빓", "빓", "빓", "빓", "빓" },
  { "빔", "빔", "빔", "빔", "빔" },
  { "빕", "빕", "빕", "빕", "빕" },
  { "빖", "빖", "빖", "빖", "빖" },
  { "빗", "빗", "빗", "빗", "빗" },
  { "빘", "빘", "빘", "빘", "빘" },
  { "빙", "빙", "빙", "빙", "빙" },
  { "빚", "빚", "빚", "빚", "빚" },
  { "빛", "빛", "빛", "빛", "빛" },
  { "빜", "빜", "빜", "빜", "빜" },
  { "빝", "빝", "빝", "빝", "빝" },
  { "빞", "빞", "빞", "빞", "빞" },
  { "빟", "빟", "빟", "빟", "빟" },
  { "빠", "빠", "빠", "빠", "빠" },
  { "빡", "빡", "빡", "빡", "빡" },
  { "빢", "빢", "빢", "빢", "빢" },
  { "빣", "빣", "빣", "빣", "빣" },
  { "빤", "빤", "빤", "빤", "빤" },
  { "빥", "빥", "빥", "빥", "빥" },
  { "빦", "빦", "빦", "빦", "빦" },
  { "빧", "빧", "빧", "빧", "빧" },
  { "빨", "빨", "빨", "빨", "빨" },
  { "빩", "빩", "빩", "빩", "빩" },
  { "빪", "빪", "빪", "빪", "빪" },
  { "빫", "빫", "빫", "빫", "빫" },
  { "빬", "빬", "빬", "빬", "빬" },
  { "빭", "빭", "빭", "빭", "빭" },
  { "빮", "빮", "빮", "빮", "빮" },
  { "빯", "빯", "빯", "빯", "빯" },
  { "빰", "빰", "빰", "빰", "빰" },
  { "빱", "빱", "빱", "빱", "빱" },
  { "빲", "빲", "빲", "빲", "빲" },
  { "빳", "빳", "빳", "빳", "빳" },
  { "빴", "빴", "빴", "빴", "빴" },
  { "빵", "빵", "빵", "빵", "빵" },
  { "빶", "빶", "빶", "빶", "빶" },
  { "빷", "빷", "빷", "빷", "빷" },
  { "빸", "빸", "빸", "빸", "빸" },
  { "빹", "빹", "빹", "빹", "빹" },
  { "빺", "빺", "빺", "빺", "빺" },
  { "빻", "빻", "빻", "빻", "빻" },
  { "빼", "빼", "빼", "빼", "빼" },
  { "빽", "빽", "빽", "빽", "빽" },
  { "빾", "빾", "빾", "빾", "빾" },
  { "빿", "빿", "빿", "빿", "빿" },
  { "뺀", "뺀", "뺀", "뺀", "뺀" },
  { "뺁", "뺁", "뺁", "뺁", "뺁" },
  { "뺂", "뺂", "뺂", "뺂", "뺂" },
  { "뺃", "뺃", "뺃", "뺃", "뺃" },
  { "뺄", "뺄", "뺄", "뺄", "뺄" },
  { "뺅", "뺅", "뺅", "뺅", "뺅" },
  { "뺆", "뺆", "뺆", "뺆", "뺆" },
  { "뺇", "뺇", "뺇", "뺇", "뺇" },
  { "뺈", "뺈", "뺈", "뺈", "뺈" },
  { "뺉", "뺉", "뺉", "뺉", "뺉" },
  { "뺊", "뺊", "뺊", "뺊", "뺊" },
  { "뺋", "뺋", "뺋", "뺋", "뺋" },
  { "뺌", "뺌", "뺌", "뺌", "뺌" },
  { "뺍", "뺍", "뺍", "뺍", "뺍" },
  { "뺎", "뺎", "뺎", "뺎", "뺎" },
  { "뺏", "뺏", "뺏", "뺏", "뺏" },
  { "뺐", "뺐", "뺐", "뺐", "뺐" },
  { "뺑", "뺑", "뺑", "뺑", "뺑" },
  { "뺒", "뺒", "뺒", "뺒", "뺒" },
  { "뺓", "뺓", "뺓", "뺓", "뺓" },
  { "뺔", "뺔", "뺔", "뺔", "뺔" },
  { "뺕", "뺕", "뺕", "뺕", "뺕" },
  { "뺖", "뺖", "뺖", "뺖", "뺖" },
  { "뺗", "뺗", "뺗", "뺗", "뺗" },
  { "뺘", "뺘", "뺘", "뺘", "뺘" },
  { "뺙", "뺙", "뺙", "뺙", "뺙" },
  { "뺚", "뺚", "뺚", "뺚", "뺚" },
  { "뺛", "뺛", "뺛", "뺛", "뺛" },
  { "뺜", "뺜", "뺜", "뺜", "뺜" },
  { "뺝", "뺝", "뺝", "뺝", "뺝" },
  { "뺞", "뺞", "뺞", "뺞", "뺞" },
  { "뺟", "뺟", "뺟", "뺟", "뺟" },
  { "뺠", "뺠", "뺠", "뺠", "뺠" },
  { "뺡", "뺡", "뺡", "뺡", "뺡" },
  { "뺢", "뺢", "뺢", "뺢", "뺢" },
  { "뺣", "뺣", "뺣", "뺣", "뺣" },
  { "뺤", "뺤", "뺤", "뺤", "뺤" },
  { "뺥", "뺥", "뺥", "뺥", "뺥" },
  { "뺦", "뺦", "뺦", "뺦", "뺦" },
  { "뺧", "뺧", "뺧", "뺧", "뺧" },
  { "뺨", "뺨", "뺨", "뺨", "뺨" },
  { "뺩", "뺩", "뺩", "뺩", "뺩" },
  { "뺪", "뺪", "뺪", "뺪", "뺪" },
  { "뺫", "뺫", "뺫", "뺫", "뺫" },
  { "뺬", "뺬", "뺬", "뺬", "뺬" },
  { "뺭", "뺭", "뺭", "뺭", "뺭" },
  { "뺮", "뺮", "뺮", "뺮", "뺮" },
  { "뺯", "뺯", "뺯", "뺯", "뺯" },
  { "뺰", "뺰", "뺰", "뺰", "뺰" },
  { "뺱", "뺱", "뺱", "뺱", "뺱" },
  { "뺲", "뺲", "뺲", "뺲", "뺲" },
  { "뺳", "뺳", "뺳", "뺳", "뺳" },
  { "뺴", "뺴", "뺴", "뺴", "뺴" },
  { "뺵", "뺵", "뺵", "뺵", "뺵" },
  { "뺶", "뺶", "뺶", "뺶", "뺶" },
  { "뺷", "뺷", "뺷", "뺷", "뺷" },
  { "뺸", "뺸", "뺸", "뺸", "뺸" },
  { "뺹", "뺹", "뺹", "뺹", "뺹" },
  { "뺺", "뺺", "뺺", "뺺", "뺺" },
  { "뺻", "뺻", "뺻", "뺻", "뺻" },
  { "뺼", "뺼", "뺼", "뺼", "뺼" },
  { "뺽", "뺽", "뺽", "뺽", "뺽" },
  { "뺾", "뺾", "뺾", "뺾", "뺾" },
  { "뺿", "뺿", "뺿", "뺿", "뺿" },
  { "뻀", "뻀", "뻀", "뻀", "뻀" },
  { "뻁", "뻁", "뻁", "뻁", "뻁" },
  { "뻂", "뻂", "뻂", "뻂", "뻂" },
  { "뻃", "뻃", "뻃", "뻃", "뻃" },
  { "뻄", "뻄", "뻄", "뻄", "뻄" },
  { "뻅", "뻅", "뻅", "뻅", "뻅" },
  { "뻆", "뻆", "뻆", "뻆", "뻆" },
  { "뻇", "뻇", "뻇", "뻇", "뻇" },
  { "뻈", "뻈", "뻈", "뻈", "뻈" },
  { "뻉", "뻉", "뻉", "뻉", "뻉" },
  { "뻊", "뻊", "뻊", "뻊", "뻊" },
  { "뻋", "뻋", "뻋", "뻋", "뻋" },
  { "뻌", "뻌", "뻌", "뻌", "뻌" },
  { "뻍", "뻍", "뻍", "뻍", "뻍" },
  { "뻎", "뻎", "뻎", "뻎", "뻎" },
  { "뻏", "뻏", "뻏", "뻏", "뻏" },
  { "뻐", "뻐", "뻐", "뻐", "뻐" },
  { "뻑", "뻑", "뻑", "뻑", "뻑" },
  { "뻒", "뻒", "뻒", "뻒", "뻒" },
  { "뻓", "뻓", "뻓", "뻓", "뻓" },
  { "뻔", "뻔", "뻔", "뻔", "뻔" },
  { "뻕", "뻕", "뻕", "뻕", "뻕" },
  { "뻖", "뻖", "뻖", "뻖", "뻖" },
  { "뻗", "뻗", "뻗", "뻗", "뻗" },
  { "뻘", "뻘", "뻘", "뻘", "뻘" },
  { "뻙", "뻙", "뻙", "뻙", "뻙" },
  { "뻚", "뻚", "뻚", "뻚", "뻚" },
  { "뻛", "뻛", "뻛", "뻛", "뻛" },
  { "뻜", "뻜", "뻜", "뻜", "뻜" },
  { "뻝", "뻝", "뻝", "뻝", "뻝" },
  { "뻞", "뻞", "뻞", "뻞", "뻞" },
  { "뻟", "뻟", "뻟", "뻟", "뻟" },
  { "뻠", "뻠", "뻠", "뻠", "뻠" },
  { "뻡", "뻡", "뻡", "뻡", "뻡" },
  { "뻢", "뻢", "뻢", "뻢", "뻢" },
  { "뻣", "뻣", "뻣", "뻣", "뻣" },
  { "뻤", "뻤", "뻤", "뻤", "뻤" },
  { "뻥", "뻥", "뻥", "뻥", "뻥" },
  { "뻦", "뻦", "뻦", "뻦", "뻦" },
  { "뻧", "뻧", "뻧", "뻧", "뻧" },
  { "뻨", "뻨", "뻨", "뻨", "뻨" },
  { "뻩", "뻩", "뻩", "뻩", "뻩" },
  { "뻪", "뻪", "뻪", "뻪", "뻪" },
  { "뻫", "뻫", "뻫", "뻫", "뻫" },
  { "뻬", "뻬", "뻬", "뻬", "뻬" },
  { "뻭", "뻭", "뻭", "뻭", "뻭" },
  { "뻮", "뻮", "뻮", "뻮", "뻮" },
  { "뻯", "뻯", "뻯", "뻯", "뻯" },
  { "뻰", "뻰", "뻰", "뻰", "뻰" },
  { "뻱", "뻱", "뻱", "뻱", "뻱" },
  { "뻲", "뻲", "뻲", "뻲", "뻲" },
  { "뻳", "뻳", "뻳", "뻳", "뻳" },
  { "뻴", "뻴", "뻴", "뻴", "뻴" },
  { "뻵", "뻵", "뻵", "뻵", "뻵" },
  { "뻶", "뻶", "뻶", "뻶", "뻶" },
  { "뻷", "뻷", "뻷", "뻷", "뻷" },
  { "뻸", "뻸", "뻸", "뻸", "뻸" },
  { "뻹", "뻹", "뻹", "뻹", "뻹" },
  { "뻺", "뻺", "뻺", "뻺", "뻺" },
  { "뻻", "뻻", "뻻", "뻻", "뻻" },
  { "뻼", "뻼", "뻼", "뻼", "뻼" },
  { "뻽", "뻽", "뻽", "뻽", "뻽" },
  { "뻾", "뻾", "뻾", "뻾", "뻾" },
  { "뻿", "뻿", "뻿", "뻿", "뻿" },
  { "뼀", "뼀", "뼀", "뼀", "뼀" },
  { "뼁", "뼁", "뼁", "뼁", "뼁" },
  { "뼂", "뼂", "뼂", "뼂", "뼂" },
  { "뼃", "뼃", "뼃", "뼃", "뼃" },
  { "뼄", "뼄", "뼄", "뼄", "뼄" },
  { "뼅", "뼅", "뼅", "뼅", "뼅" },
  { "뼆", "뼆", "뼆", "뼆", "뼆" },
  { "뼇", "뼇", "뼇", "뼇", "뼇" },
  { "뼈", "뼈", "뼈", "뼈", "뼈" },
  { "뼉", "뼉", "뼉", "뼉", "뼉" },
  { "뼊", "뼊", "뼊", "뼊", "뼊" },
  { "뼋", "뼋", "뼋", "뼋", "뼋" },
  { "뼌", "뼌", "뼌", "뼌", "뼌" },
  { "뼍", "뼍", "뼍", "뼍", "뼍" },
  { "뼎", "뼎", "뼎", "뼎", "뼎" },
  { "뼏", "뼏", "뼏", "뼏", "뼏" },
  { "뼐", "뼐", "뼐", "뼐", "뼐" },
  { "뼑", "뼑", "뼑", "뼑", "뼑" },
  { "뼒", "뼒", "뼒", "뼒", "뼒" },
  { "뼓", "뼓", "뼓", "뼓", "뼓" },
  { "뼔", "뼔", "뼔", "뼔", "뼔" },
  { "뼕", "뼕", "뼕", "뼕", "뼕" },
  { "뼖", "뼖", "뼖", "뼖", "뼖" },
  { "뼗", "뼗", "뼗", "뼗", "뼗" },
  { "뼘", "뼘", "뼘", "뼘", "뼘" },
  { "뼙", "뼙", "뼙", "뼙", "뼙" },
  { "뼚", "뼚", "뼚", "뼚", "뼚" },
  { "뼛", "뼛", "뼛", "뼛", "뼛" },
  { "뼜", "뼜", "뼜", "뼜", "뼜" },
  { "뼝", "뼝", "뼝", "뼝", "뼝" },
  { "뼞", "뼞", "뼞", "뼞", "뼞" },
  { "뼟", "뼟", "뼟", "뼟", "뼟" },
  { "뼠", "뼠", "뼠", "뼠", "뼠" },
  { "뼡", "뼡", "뼡", "뼡", "뼡" },
  { "뼢", "뼢", "뼢", "뼢", "뼢" },
  { "뼣", "뼣", "뼣", "뼣", "뼣" },
  { "뼤", "뼤", "뼤", "뼤", "뼤" },
  { "뼥", "뼥", "뼥", "뼥", "뼥" },
  { "뼦", "뼦", "뼦", "뼦", "뼦" },
  { "뼧", "뼧", "뼧", "뼧", "뼧" },
  { "뼨", "뼨", "뼨", "뼨", "뼨" },
  { "뼩", "뼩", "뼩", "뼩", "뼩" },
  { "뼪", "뼪", "뼪", "뼪", "뼪" },
  { "뼫", "뼫", "뼫", "뼫", "뼫" },
  { "뼬", "뼬", "뼬", "뼬", "뼬" },
  { "뼭", "뼭", "뼭", "뼭", "뼭" },
  { "뼮", "뼮", "뼮", "뼮", "뼮" },
  { "뼯", "뼯", "뼯", "뼯", "뼯" },
  { "뼰", "뼰", "뼰", "뼰", "뼰" },
  { "뼱", "뼱", "뼱", "뼱", "뼱" },
  { "뼲", "뼲", "뼲", "뼲", "뼲" },
  { "뼳", "뼳", "뼳", "뼳", "뼳" },
  { "뼴", "뼴", "뼴", "뼴", "뼴" },
  { "뼵", "뼵", "뼵", "뼵", "뼵" },
  { "뼶", "뼶", "뼶", "뼶", "뼶" },
  { "뼷", "뼷", "뼷", "뼷", "뼷" },
  { "뼸", "뼸", "뼸", "뼸", "뼸" },
  { "뼹", "뼹", "뼹", "뼹", "뼹" },
  { "뼺", "뼺", "뼺", "뼺", "뼺" },
  { "뼻", "뼻", "뼻", "뼻", "뼻" },
  { "뼼", "뼼", "뼼", "뼼", "뼼" },
  { "뼽", "뼽", "뼽", "뼽", "뼽" },
  { "뼾", "뼾", "뼾", "뼾", "뼾" },
  { "뼿", "뼿", "뼿", "뼿", "뼿" },
  { "뽀", "뽀", "뽀", "뽀", "뽀" },
  { "뽁", "뽁", "뽁", "뽁", "뽁" },
  { "뽂", "뽂", "뽂", "뽂", "뽂" },
  { "뽃", "뽃", "뽃", "뽃", "뽃" },
  { "뽄", "뽄", "뽄", "뽄", "뽄" },
  { "뽅", "뽅", "뽅", "뽅", "뽅" },
  { "뽆", "뽆", "뽆", "뽆", "뽆" },
  { "뽇", "뽇", "뽇", "뽇", "뽇" },
  { "뽈", "뽈", "뽈", "뽈", "뽈" },
  { "뽉", "뽉", "뽉", "뽉", "뽉" },
  { "뽊", "뽊", "뽊", "뽊", "뽊" },
  { "뽋", "뽋", "뽋", "뽋", "뽋" },
  { "뽌", "뽌", "뽌", "뽌", "뽌" },
  { "뽍", "뽍", "뽍", "뽍", "뽍" },
  { "뽎", "뽎", "뽎", "뽎", "뽎" },
  { "뽏", "뽏", "뽏", "뽏", "뽏" },
  { "뽐", "뽐", "뽐", "뽐", "뽐" },
  { "뽑", "뽑", "뽑", "뽑", "뽑" },
  { "뽒", "뽒", "뽒", "뽒", "뽒" },
  { "뽓", "뽓", "뽓", "뽓", "뽓" },
  { "뽔", "뽔", "뽔", "뽔", "뽔" },
  { "뽕", "뽕", "뽕", "뽕", "뽕" },
  { "뽖", "뽖", "뽖", "뽖", "뽖" },
  { "뽗", "뽗", "뽗", "뽗", "뽗" },
  { "뽘", "뽘", "뽘", "뽘", "뽘" },
  { "뽙", "뽙", "뽙", "뽙", "뽙" },
  { "뽚", "뽚", "뽚", "뽚", "뽚" },
  { "뽛", "뽛", "뽛", "뽛", "뽛" },
  { "뽜", "뽜", "뽜", "뽜", "뽜" },
  { "뽝", "뽝", "뽝", "뽝", "뽝" },
  { "뽞", "뽞", "뽞", "뽞", "뽞" },
  { "뽟", "뽟", "뽟", "뽟", "뽟" },
  { "뽠", "뽠", "뽠", "뽠", "뽠" },
  { "뽡", "뽡", "뽡", "뽡", "뽡" },
  { "뽢", "뽢", "뽢", "뽢", "뽢" },
  { "뽣", "뽣", "뽣", "뽣", "뽣" },
  { "뽤", "뽤", "뽤", "뽤", "뽤" },
  { "뽥", "뽥", "뽥", "뽥", "뽥" },
  { "뽦", "뽦", "뽦", "뽦", "뽦" },
  { "뽧", "뽧", "뽧", "뽧", "뽧" },
  { "뽨", "뽨", "뽨", "뽨", "뽨" },
  { "뽩", "뽩", "뽩", "뽩", "뽩" },
  { "뽪", "뽪", "뽪", "뽪", "뽪" },
  { "뽫", "뽫", "뽫", "뽫", "뽫" },
  { "뽬", "뽬", "뽬", "뽬", "뽬" },
  { "뽭", "뽭", "뽭", "뽭", "뽭" },
  { "뽮", "뽮", "뽮", "뽮", "뽮" },
  { "뽯", "뽯", "뽯", "뽯", "뽯" },
  { "뽰", "뽰", "뽰", "뽰", "뽰" },
  { "뽱", "뽱", "뽱", "뽱", "뽱" },
  { "뽲", "뽲", "뽲", "뽲", "뽲" },
  { "뽳", "뽳", "뽳", "뽳", "뽳" },
  { "뽴", "뽴", "뽴", "뽴", "뽴" },
  { "뽵", "뽵", "뽵", "뽵", "뽵" },
  { "뽶", "뽶", "뽶", "뽶", "뽶" },
  { "뽷", "뽷", "뽷", "뽷", "뽷" },
  { "뽸", "뽸", "뽸", "뽸", "뽸" },
  { "뽹", "뽹", "뽹", "뽹", "뽹" },
  { "뽺", "뽺", "뽺", "뽺", "뽺" },
  { "뽻", "뽻", "뽻", "뽻", "뽻" },
  { "뽼", "뽼", "뽼", "뽼", "뽼" },
  { "뽽", "뽽", "뽽", "뽽", "뽽" },
  { "뽾", "뽾", "뽾", "뽾", "뽾" },
  { "뽿", "뽿", "뽿", "뽿", "뽿" },
  { "뾀", "뾀", "뾀", "뾀", "뾀" },
  { "뾁", "뾁", "뾁", "뾁", "뾁" },
  { "뾂", "뾂", "뾂", "뾂", "뾂" },
  { "뾃", "뾃", "뾃", "뾃", "뾃" },
  { "뾄", "뾄", "뾄", "뾄", "뾄" },
  { "뾅", "뾅", "뾅", "뾅", "뾅" },
  { "뾆", "뾆", "뾆", "뾆", "뾆" },
  { "뾇", "뾇", "뾇", "뾇", "뾇" },
  { "뾈", "뾈", "뾈", "뾈", "뾈" },
  { "뾉", "뾉", "뾉", "뾉", "뾉" },
  { "뾊", "뾊", "뾊", "뾊", "뾊" },
  { "뾋", "뾋", "뾋", "뾋", "뾋" },
  { "뾌", "뾌", "뾌", "뾌", "뾌" },
  { "뾍", "뾍", "뾍", "뾍", "뾍" },
  { "뾎", "뾎", "뾎", "뾎", "뾎" },
  { "뾏", "뾏", "뾏", "뾏", "뾏" },
  { "뾐", "뾐", "뾐", "뾐", "뾐" },
  { "뾑", "뾑", "뾑", "뾑", "뾑" },
  { "뾒", "뾒", "뾒", "뾒", "뾒" },
  { "뾓", "뾓", "뾓", "뾓", "뾓" },
  { "뾔", "뾔", "뾔", "뾔", "뾔" },
  { "뾕", "뾕", "뾕", "뾕", "뾕" },
  { "뾖", "뾖", "뾖", "뾖", "뾖" },
  { "뾗", "뾗", "뾗", "뾗", "뾗" },
  { "뾘", "뾘", "뾘", "뾘", "뾘" },
  { "뾙", "뾙", "뾙", "뾙", "뾙" },
  { "뾚", "뾚", "뾚", "뾚", "뾚" },
  { "뾛", "뾛", "뾛", "뾛", "뾛" },
  { "뾜", "뾜", "뾜", "뾜", "뾜" },
  { "뾝", "뾝", "뾝", "뾝", "뾝" },
  { "뾞", "뾞", "뾞", "뾞", "뾞" },
  { "뾟", "뾟", "뾟", "뾟", "뾟" },
  { "뾠", "뾠", "뾠", "뾠", "뾠" },
  { "뾡", "뾡", "뾡", "뾡", "뾡" },
  { "뾢", "뾢", "뾢", "뾢", "뾢" },
  { "뾣", "뾣", "뾣", "뾣", "뾣" },
  { "뾤", "뾤", "뾤", "뾤", "뾤" },
  { "뾥", "뾥", "뾥", "뾥", "뾥" },
  { "뾦", "뾦", "뾦", "뾦", "뾦" },
  { "뾧", "뾧", "뾧", "뾧", "뾧" },
  { "뾨", "뾨", "뾨", "뾨", "뾨" },
  { "뾩", "뾩", "뾩", "뾩", "뾩" },
  { "뾪", "뾪", "뾪", "뾪", "뾪" },
  { "뾫", "뾫", "뾫", "뾫", "뾫" },
  { "뾬", "뾬", "뾬", "뾬", "뾬" },
  { "뾭", "뾭", "뾭", "뾭", "뾭" },
  { "뾮", "뾮", "뾮", "뾮", "뾮" },
  { "뾯", "뾯", "뾯", "뾯", "뾯" },
  { "뾰", "뾰", "뾰", "뾰", "뾰" },
  { "뾱", "뾱", "뾱", "뾱", "뾱" },
  { "뾲", "뾲", "뾲", "뾲", "뾲" },
  { "뾳", "뾳", "뾳", "뾳", "뾳" },
  { "뾴", "뾴", "뾴", "뾴", "뾴" },
  { "뾵", "뾵", "뾵", "뾵", "뾵" },
  { "뾶", "뾶", "뾶", "뾶", "뾶" },
  { "뾷", "뾷", "뾷", "뾷", "뾷" },
  { "뾸", "뾸", "뾸", "뾸", "뾸" },
  { "뾹", "뾹", "뾹", "뾹", "뾹" },
  { "뾺", "뾺", "뾺", "뾺", "뾺" },
  { "뾻", "뾻", "뾻", "뾻", "뾻" },
  { "뾼", "뾼", "뾼", "뾼", "뾼" },
  { "뾽", "뾽", "뾽", "뾽", "뾽" },
  { "뾾", "뾾", "뾾", "뾾", "뾾" },
  { "뾿", "뾿", "뾿", "뾿", "뾿" },
  { "뿀", "뿀", "뿀", "뿀", "뿀" },
  { "뿁", "뿁", "뿁", "뿁", "뿁" },
  { "뿂", "뿂", "뿂", "뿂", "뿂" },
  { "뿃", "뿃", "뿃", "뿃", "뿃" },
  { "뿄", "뿄", "뿄", "뿄", "뿄" },
  { "뿅", "뿅", "뿅", "뿅", "뿅" },
  { "뿆", "뿆", "뿆", "뿆", "뿆" },
  { "뿇", "뿇", "뿇", "뿇", "뿇" },
  { "뿈", "뿈", "뿈", "뿈", "뿈" },
  { "뿉", "뿉", "뿉", "뿉", "뿉" },
  { "뿊", "뿊", "뿊", "뿊", "뿊" },
  { "뿋", "뿋", "뿋", "뿋", "뿋" },
  { "뿌", "뿌", "뿌", "뿌", "뿌" },
  { "뿍", "뿍", "뿍", "뿍", "뿍" },
  { "뿎", "뿎", "뿎", "뿎", "뿎" },
  { "뿏", "뿏", "뿏", "뿏", "뿏" },
  { "뿐", "뿐", "뿐", "뿐", "뿐" },
  { "뿑", "뿑", "뿑", "뿑", "뿑" },
  { "뿒", "뿒", "뿒", "뿒", "뿒" },
  { "뿓", "뿓", "뿓", "뿓", "뿓" },
  { "뿔", "뿔", "뿔", "뿔", "뿔" },
  { "뿕", "뿕", "뿕", "뿕", "뿕" },
  { "뿖", "뿖", "뿖", "뿖", "뿖" },
  { "뿗", "뿗", "뿗", "뿗", "뿗" },
  { "뿘", "뿘", "뿘", "뿘", "뿘" },
  { "뿙", "뿙", "뿙", "뿙", "뿙" },
  { "뿚", "뿚", "뿚", "뿚", "뿚" },
  { "뿛", "뿛", "뿛", "뿛", "뿛" },
  { "뿜", "뿜", "뿜", "뿜", "뿜" },
  { "뿝", "뿝", "뿝", "뿝", "뿝" },
  { "뿞", "뿞", "뿞", "뿞", "뿞" },
  { "뿟", "뿟", "뿟", "뿟", "뿟" },
  { "뿠", "뿠", "뿠", "뿠", "뿠" },
  { "뿡", "뿡", "뿡", "뿡", "뿡" },
  { "뿢", "뿢", "뿢", "뿢", "뿢" },
  { "뿣", "뿣", "뿣", "뿣", "뿣" },
  { "뿤", "뿤", "뿤", "뿤", "뿤" },
  { "뿥", "뿥", "뿥", "뿥", "뿥" },
  { "뿦", "뿦", "뿦", "뿦", "뿦" },
  { "뿧", "뿧", "뿧", "뿧", "뿧" },
  { "뿨", "뿨", "뿨", "뿨", "뿨" },
  { "뿩", "뿩", "뿩", "뿩", "뿩" },
  { "뿪", "뿪", "뿪", "뿪", "뿪" },
  { "뿫", "뿫", "뿫", "뿫", "뿫" },
  { "뿬", "뿬", "뿬", "뿬", "뿬" },
  { "뿭", "뿭", "뿭", "뿭", "뿭" },
  { "뿮", "뿮", "뿮", "뿮", "뿮" },
  { "뿯", "뿯", "뿯", "뿯", "뿯" },
  { "뿰", "뿰", "뿰", "뿰", "뿰" },
  { "뿱", "뿱", "뿱", "뿱", "뿱" },
  { "뿲", "뿲", "뿲", "뿲", "뿲" },
  { "뿳", "뿳", "뿳", "뿳", "뿳" },
  { "뿴", "뿴", "뿴", "뿴", "뿴" },
  { "뿵", "뿵", "뿵", "뿵", "뿵" },
  { "뿶", "뿶", "뿶", "뿶", "뿶" },
  { "뿷", "뿷", "뿷", "뿷", "뿷" },
  { "뿸", "뿸", "뿸", "뿸", "뿸" },
  { "뿹", "뿹", "뿹", "뿹", "뿹" },
  { "뿺", "뿺", "뿺", "뿺", "뿺" },
  { "뿻", "뿻", "뿻", "뿻", "뿻" },
  { "뿼", "뿼", "뿼", "뿼", "뿼" },
  { "뿽", "뿽", "뿽", "뿽", "뿽" },
  { "뿾", "뿾", "뿾", "뿾", "뿾" },
  { "뿿", "뿿", "뿿", "뿿", "뿿" },
  { "쀀", "쀀", "쀀", "쀀", "쀀" },
  { "쀁", "쀁", "쀁", "쀁", "쀁" },
  { "쀂", "쀂", "쀂", "쀂", "쀂" },
  { "쀃", "쀃", "쀃", "쀃", "쀃" },
  { "쀄", "쀄", "쀄", "쀄", "쀄" },
  { "쀅", "쀅", "쀅", "쀅", "쀅" },
  { "쀆", "쀆", "쀆", "쀆", "쀆" },
  { "쀇", "쀇", "쀇", "쀇", "쀇" },
  { "쀈", "쀈", "쀈", "쀈", "쀈" },
  { "쀉", "쀉", "쀉", "쀉", "쀉" },
  { "쀊", "쀊", "쀊", "쀊", "쀊" },
  { "쀋", "쀋", "쀋", "쀋", "쀋" },
  { "쀌", "쀌", "쀌", "쀌", "쀌" },
  { "쀍", "쀍", "쀍", "쀍", "쀍" },
  { "쀎", "쀎", "쀎", "쀎", "쀎" },
  { "쀏", "쀏", "쀏", "쀏", "쀏" },
  { "쀐", "쀐", "쀐", "쀐", "쀐" },
  { "쀑", "쀑", "쀑", "쀑", "쀑" },
  { "쀒", "쀒", "쀒", "쀒", "쀒" },
  { "쀓", "쀓", "쀓", "쀓", "쀓" },
  { "쀔", "쀔", "쀔", "쀔", "쀔" },
  { "쀕", "쀕", "쀕", "쀕", "쀕" },
  { "쀖", "쀖", "쀖", "쀖", "쀖" },
  { "쀗", "쀗", "쀗", "쀗", "쀗" },
  { "쀘", "쀘", "쀘", "쀘", "쀘" },
  { "쀙", "쀙", "쀙", "쀙", "쀙" },
  { "쀚", "쀚", "쀚", "쀚", "쀚" },
  { "쀛", "쀛", "쀛", "쀛", "쀛" },
  { "쀜", "쀜", "쀜", "쀜", "쀜" },
  { "쀝", "쀝", "쀝", "쀝", "쀝" },
  { "쀞", "쀞", "쀞", "쀞", "쀞" },
  { "쀟", "쀟", "쀟", "쀟", "쀟" },
  { "쀠", "쀠", "쀠", "쀠", "쀠" },
  { "쀡", "쀡", "쀡", "쀡", "쀡" },
  { "쀢", "쀢", "쀢", "쀢", "쀢" },
  { "쀣", "쀣", "쀣", "쀣", "쀣" },
  { "쀤", "쀤", "쀤", "쀤", "쀤" },
  { "쀥", "쀥", "쀥", "쀥", "쀥" },
  { "쀦", "쀦", "쀦", "쀦", "쀦" },
  { "쀧", "쀧", "쀧", "쀧", "쀧" },
  { "쀨", "쀨", "쀨", "쀨", "쀨" },
  { "쀩", "쀩", "쀩", "쀩", "쀩" },
  { "쀪", "쀪", "쀪", "쀪", "쀪" },
  { "쀫", "쀫", "쀫", "쀫", "쀫" },
  { "쀬", "쀬", "쀬", "쀬", "쀬" },
  { "쀭", "쀭", "쀭", "쀭", "쀭" },
  { "쀮", "쀮", "쀮", "쀮", "쀮" },
  { "쀯", "쀯", "쀯", "쀯", "쀯" },
  { "쀰", "쀰", "쀰", "쀰", "쀰" },
  { "쀱", "쀱", "쀱", "쀱", "쀱" },
  { "쀲", "쀲", "쀲", "쀲", "쀲" },
  { "쀳", "쀳", "쀳", "쀳", "쀳" },
  { "쀴", "쀴", "쀴", "쀴", "쀴" },
  { "쀵", "쀵", "쀵", "쀵", "쀵" },
  { "쀶", "쀶", "쀶", "쀶", "쀶" },
  { "쀷", "쀷", "쀷", "쀷", "쀷" },
  { "쀸", "쀸", "쀸", "쀸", "쀸" },
  { "쀹", "쀹", "쀹", "쀹", "쀹" },
  { "쀺", "쀺", "쀺", "쀺", "쀺" },
  { "쀻", "쀻", "쀻", "쀻", "쀻" },
  { "쀼", "쀼", "쀼", "쀼", "쀼" },
  { "쀽", "쀽", "쀽", "쀽", "쀽" },
  { "쀾", "쀾", "쀾", "쀾", "쀾" },
  { "쀿", "쀿", "쀿", "쀿", "쀿" },
  { "쁀", "쁀", "쁀", "쁀", "쁀" },
  { "쁁", "쁁", "쁁", "쁁", "쁁" },
  { "쁂", "쁂", "쁂", "쁂", "쁂" },
  { "쁃", "쁃", "쁃", "쁃", "쁃" },
  { "쁄", "쁄", "쁄", "쁄", "쁄" },
  { "쁅", "쁅", "쁅", "쁅", "쁅" },
  { "쁆", "쁆", "쁆", "쁆", "쁆" },
  { "쁇", "쁇", "쁇", "쁇", "쁇" },
  { "쁈", "쁈", "쁈", "쁈", "쁈" },
  { "쁉", "쁉", "쁉", "쁉", "쁉" },
  { "쁊", "쁊", "쁊", "쁊", "쁊" },
  { "쁋", "쁋", "쁋", "쁋", "쁋" },
  { "쁌", "쁌", "쁌", "쁌", "쁌" },
  { "쁍", "쁍", "쁍", "쁍", "쁍" },
  { "쁎", "쁎", "쁎", "쁎", "쁎" },
  { "쁏", "쁏", "쁏", "쁏", "쁏" },
  { "쁐", "쁐", "쁐", "쁐", "쁐" },
  { "쁑", "쁑", "쁑", "쁑", "쁑" },
  { "쁒", "쁒", "쁒", "쁒", "쁒" },
  { "쁓", "쁓", "쁓", "쁓", "쁓" },
  { "쁔", "쁔", "쁔", "쁔", "쁔" },
  { "쁕", "쁕", "쁕", "쁕", "쁕" },
  { "쁖", "쁖", "쁖", "쁖", "쁖" },
  { "쁗", "쁗", "쁗", "쁗", "쁗" },
  { "쁘", "쁘", "쁘", "쁘", "쁘" },
  { "쁙", "쁙", "쁙", "쁙", "쁙" },
  { "쁚", "쁚", "쁚", "쁚", "쁚" },
  { "쁛", "쁛", "쁛", "쁛", "쁛" },
  { "쁜", "쁜", "쁜", "쁜", "쁜" },
  { "쁝", "쁝", "쁝", "쁝", "쁝" },
  { "쁞", "쁞", "쁞", "쁞", "쁞" },
  { "쁟", "쁟", "쁟", "쁟", "쁟" },
  { "쁠", "쁠", "쁠", "쁠", "쁠" },
  { "쁡", "쁡", "쁡", "쁡", "쁡" },
  { "쁢", "쁢", "쁢", "쁢", "쁢" },
  { "쁣", "쁣", "쁣", "쁣", "쁣" },
  { "쁤", "쁤", "쁤", "쁤", "쁤" },
  { "쁥", "쁥", "쁥", "쁥", "쁥" },
  { "쁦", "쁦", "쁦", "쁦", "쁦" },
  { "쁧", "쁧", "쁧", "쁧", "쁧" },
  { "쁨", "쁨", "쁨", "쁨", "쁨" },
  { "쁩", "쁩", "쁩", "쁩", "쁩" },
  { "쁪", "쁪", "쁪", "쁪", "쁪" },
  { "쁫", "쁫", "쁫", "쁫", "쁫" },
  { "쁬", "쁬", "쁬", "쁬", "쁬" },
  { "쁭", "쁭", "쁭", "쁭", "쁭" },
  { "쁮", "쁮", "쁮", "쁮", "쁮" },
  { "쁯", "쁯", "쁯", "쁯", "쁯" },
  { "쁰", "쁰", "쁰", "쁰", "쁰" },
  { "쁱", "쁱", "쁱", "쁱", "쁱" },
  { "쁲", "쁲", "쁲", "쁲", "쁲" },
  { "쁳", "쁳", "쁳", "쁳", "쁳" },
  { "쁴", "쁴", "쁴", "쁴", "쁴" },
  { "쁵", "쁵", "쁵", "쁵", "쁵" },
  { "쁶", "쁶", "쁶", "쁶", "쁶" },
  { "쁷", "쁷", "쁷", "쁷", "쁷" },
  { "쁸", "쁸", "쁸", "쁸", "쁸" },
  { "쁹", "쁹", "쁹", "쁹", "쁹" },
  { "쁺", "쁺", "쁺", "쁺", "쁺" },
  { "쁻", "쁻", "쁻", "쁻", "쁻" },
  { "쁼", "쁼", "쁼", "쁼", "쁼" },
  { "쁽", "쁽", "쁽", "쁽", "쁽" },
  { "쁾", "쁾", "쁾", "쁾", "쁾" },
  { "쁿", "쁿", "쁿", "쁿", "쁿" },
  { "삀", "삀", "삀", "삀", "삀" },
  { "삁", "삁", "삁", "삁", "삁" },
  { "삂", "삂", "삂", "삂", "삂" },
  { "삃", "삃", "삃", "삃", "삃" },
  { "삄", "삄", "삄", "삄", "삄" },
  { "삅", "삅", "삅", "삅", "삅" },
  { "삆", "삆", "삆", "삆", "삆" },
  { "삇", "삇", "삇", "삇", "삇" },
  { "삈", "삈", "삈", "삈", "삈" },
  { "삉", "삉", "삉", "삉", "삉" },
  { "삊", "삊", "삊", "삊", "삊" },
  { "삋", "삋", "삋", "삋", "삋" },
  { "삌", "삌", "삌", "삌", "삌" },
  { "삍", "삍", "삍", "삍", "삍" },
  { "삎", "삎", "삎", "삎", "삎" },
  { "삏", "삏", "삏", "삏", "삏" },
  { "삐", "삐", "삐", "삐", "삐" },
  { "삑", "삑", "삑", "삑", "삑" },
  { "삒", "삒", "삒", "삒", "삒" },
  { "삓", "삓", "삓", "삓", "삓" },
  { "삔", "삔", "삔", "삔", "삔" },
  { "삕", "삕", "삕", "삕", "삕" },
  { "삖", "삖", "삖", "삖", "삖" },
  { "삗", "삗", "삗", "삗", "삗" },
  { "삘", "삘", "삘", "삘", "삘" },
  { "삙", "삙", "삙", "삙", "삙" },
  { "삚", "삚", "삚", "삚", "삚" },
  { "삛", "삛", "삛", "삛", "삛" },
  { "삜", "삜", "삜", "삜", "삜" },
  { "삝", "삝", "삝", "삝", "삝" },
  { "삞", "삞", "삞", "삞", "삞" },
  { "삟", "삟", "삟", "삟", "삟" },
  { "삠", "삠", "삠", "삠", "삠" },
  { "삡", "삡", "삡", "삡", "삡" },
  { "삢", "삢", "삢", "삢", "삢" },
  { "삣", "삣", "삣", "삣", "삣" },
  { "삤", "삤", "삤", "삤", "삤" },
  { "삥", "삥", "삥", "삥", "삥" },
  { "삦", "삦", "삦", "삦", "삦" },
  { "삧", "삧", "삧", "삧", "삧" },
  { "삨", "삨", "삨", "삨", "삨" },
  { "삩", "삩", "삩", "삩", "삩" },
  { "삪", "삪", "삪", "삪", "삪" },
  { "삫", "삫", "삫", "삫", "삫" },
  { "사", "사", "사", "사", "사" },
  { "삭", "삭", "삭", "삭", "삭" },
  { "삮", "삮", "삮", "삮", "삮" },
  { "삯", "삯", "삯", "삯", "삯" },
  { "산", "산", "산", "산", "산" },
  { "삱", "삱", "삱", "삱", "삱" },
  { "삲", "삲", "삲", "삲", "삲" },
  { "삳", "삳", "삳", "삳", "삳" },
  { "살", "살", "살", "살", "살" },
  { "삵", "삵", "삵", "삵", "삵" },
  { "삶", "삶", "삶", "삶", "삶" },
  { "삷", "삷", "삷", "삷", "삷" },
  { "삸", "삸", "삸", "삸", "삸" },
  { "삹", "삹", "삹", "삹", "삹" },
  { "삺", "삺", "삺", "삺", "삺" },
  { "삻", "삻", "삻", "삻", "삻" },
  { "삼", "삼", "삼", "삼", "삼" },
  { "삽", "삽", "삽", "삽", "삽" },
  { "삾", "삾", "삾", "삾", "삾" },
  { "삿", "삿", "삿", "삿", "삿" },
  { "샀", "샀", "샀", "샀", "샀" },
  { "상", "상", "상", "상", "상" },
  { "샂", "샂", "샂", "샂", "샂" },
  { "샃", "샃", "샃", "샃", "샃" },
  { "샄", "샄", "샄", "샄", "샄" },
  { "샅", "샅", "샅", "샅", "샅" },
  { "샆", "샆", "샆", "샆", "샆" },
  { "샇", "샇", "샇", "샇", "샇" },
  { "새", "새", "새", "새", "새" },
  { "색", "색", "색", "색", "색" },
  { "샊", "샊", "샊", "샊", "샊" },
  { "샋", "샋", "샋", "샋", "샋" },
  { "샌", "샌", "샌", "샌", "샌" },
  { "샍", "샍", "샍", "샍", "샍" },
  { "샎", "샎", "샎", "샎", "샎" },
  { "샏", "샏", "샏", "샏", "샏" },
  { "샐", "샐", "샐", "샐", "샐" },
  { "샑", "샑", "샑", "샑", "샑" },
  { "샒", "샒", "샒", "샒", "샒" },
  { "샓", "샓", "샓", "샓", "샓" },
  { "샔", "샔", "샔", "샔", "샔" },
  { "샕", "샕", "샕", "샕", "샕" },
  { "샖", "샖", "샖", "샖", "샖" },
  { "샗", "샗", "샗", "샗", "샗" },
  { "샘", "샘", "샘", "샘", "샘" },
  { "샙", "샙", "샙", "샙", "샙" },
  { "샚", "샚", "샚", "샚", "샚" },
  { "샛", "샛", "샛", "샛", "샛" },
  { "샜", "샜", "샜", "샜", "샜" },
  { "생", "생", "생", "생", "생" },
  { "샞", "샞", "샞", "샞", "샞" },
  { "샟", "샟", "샟", "샟", "샟" },
  { "샠", "샠", "샠", "샠", "샠" },
  { "샡", "샡", "샡", "샡", "샡" },
  { "샢", "샢", "샢", "샢", "샢" },
  { "샣", "샣", "샣", "샣", "샣" },
  { "샤", "샤", "샤", "샤", "샤" },
  { "샥", "샥", "샥", "샥", "샥" },
  { "샦", "샦", "샦", "샦", "샦" },
  { "샧", "샧", "샧", "샧", "샧" },
  { "샨", "샨", "샨", "샨", "샨" },
  { "샩", "샩", "샩", "샩", "샩" },
  { "샪", "샪", "샪", "샪", "샪" },
  { "샫", "샫", "샫", "샫", "샫" },
  { "샬", "샬", "샬", "샬", "샬" },
  { "샭", "샭", "샭", "샭", "샭" },
  { "샮", "샮", "샮", "샮", "샮" },
  { "샯", "샯", "샯", "샯", "샯" },
  { "샰", "샰", "샰", "샰", "샰" },
  { "샱", "샱", "샱", "샱", "샱" },
  { "샲", "샲", "샲", "샲", "샲" },
  { "샳", "샳", "샳", "샳", "샳" },
  { "샴", "샴", "샴", "샴", "샴" },
  { "샵", "샵", "샵", "샵", "샵" },
  { "샶", "샶", "샶", "샶", "샶" },
  { "샷", "샷", "샷", "샷", "샷" },
  { "샸", "샸", "샸", "샸", "샸" },
  { "샹", "샹", "샹", "샹", "샹" },
  { "샺", "샺", "샺", "샺", "샺" },
  { "샻", "샻", "샻", "샻", "샻" },
  { "샼", "샼", "샼", "샼", "샼" },
  { "샽", "샽", "샽", "샽", "샽" },
  { "샾", "샾", "샾", "샾", "샾" },
  { "샿", "샿", "샿", "샿", "샿" },
  { "섀", "섀", "섀", "섀", "섀" },
  { "섁", "섁", "섁", "섁", "섁" },
  { "섂", "섂", "섂", "섂", "섂" },
  { "섃", "섃", "섃", "섃", "섃" },
  { "섄", "섄", "섄", "섄", "섄" },
  { "섅", "섅", "섅", "섅", "섅" },
  { "섆", "섆", "섆", "섆", "섆" },
  { "섇", "섇", "섇", "섇", "섇" },
  { "섈", "섈", "섈", "섈", "섈" },
  { "섉", "섉", "섉", "섉", "섉" },
  { "섊", "섊", "섊", "섊", "섊" },
  { "섋", "섋", "섋", "섋", "섋" },
  { "섌", "섌", "섌", "섌", "섌" },
  { "섍", "섍", "섍", "섍", "섍" },
  { "섎", "섎", "섎", "섎", "섎" },
  { "섏", "섏", "섏", "섏", "섏" },
  { "섐", "섐", "섐", "섐", "섐" },
  { "섑", "섑", "섑", "섑", "섑" },
  { "섒", "섒", "섒", "섒", "섒" },
  { "섓", "섓", "섓", "섓", "섓" },
  { "섔", "섔", "섔", "섔", "섔" },
  { "섕", "섕", "섕", "섕", "섕" },
  { "섖", "섖", "섖", "섖", "섖" },
  { "섗", "섗", "섗", "섗", "섗" },
  { "섘", "섘", "섘", "섘", "섘" },
  { "섙", "섙", "섙", "섙", "섙" },
  { "섚", "섚", "섚", "섚", "섚" },
  { "섛", "섛", "섛", "섛", "섛" },
  { "서", "서", "서", "서", "서" },
  { "석", "석", "석", "석", "석" },
  { "섞", "섞", "섞", "섞", "섞" },
  { "섟", "섟", "섟", "섟", "섟" },
  { "선", "선", "선", "선", "선" },
  { "섡", "섡", "섡", "섡", "섡" },
  { "섢", "섢", "섢", "섢", "섢" },
  { "섣", "섣", "섣", "섣", "섣" },
  { "설", "설", "설", "설", "설" },
  { "섥", "섥", "섥", "섥", "섥" },
  { "섦", "섦", "섦", "섦", "섦" },
  { "섧", "섧", "섧", "섧", "섧" },
  { "섨", "섨", "섨", "섨", "섨" },
  { "섩", "섩", "섩", "섩", "섩" },
  { "섪", "섪", "섪", "섪", "섪" },
  { "섫", "섫", "섫", "섫", "섫" },
  { "섬", "섬", "섬", "섬", "섬" },
  { "섭", "섭", "섭", "섭", "섭" },
  { "섮", "섮", "섮", "섮", "섮" },
  { "섯", "섯", "섯", "섯", "섯" },
  { "섰", "섰", "섰", "섰", "섰" },
  { "성", "성", "성", "성", "성" },
  { "섲", "섲", "섲", "섲", "섲" },
  { "섳", "섳", "섳", "섳", "섳" },
  { "섴", "섴", "섴", "섴", "섴" },
  { "섵", "섵", "섵", "섵", "섵" },
  { "섶", "섶", "섶", "섶", "섶" },
  { "섷", "섷", "섷", "섷", "섷" },
  { "세", "세", "세", "세", "세" },
  { "섹", "섹", "섹", "섹", "섹" },
  { "섺", "섺", "섺", "섺", "섺" },
  { "섻", "섻", "섻", "섻", "섻" },
  { "센", "센", "센", "센", "센" },
  { "섽", "섽", "섽", "섽", "섽" },
  { "섾", "섾", "섾", "섾", "섾" },
  { "섿", "섿", "섿", "섿", "섿" },
  { "셀", "셀", "셀", "셀", "셀" },
  { "셁", "셁", "셁", "셁", "셁" },
  { "셂", "셂", "셂", "셂", "셂" },
  { "셃", "셃", "셃", "셃", "셃" },
  { "셄", "셄", "셄", "셄", "셄" },
  { "셅", "셅", "셅", "셅", "셅" },
  { "셆", "셆", "셆", "셆", "셆" },
  { "셇", "셇", "셇", "셇", "셇" },
  { "셈", "셈", "셈", "셈", "셈" },
  { "셉", "셉", "셉", "셉", "셉" },
  { "셊", "셊", "셊", "셊", "셊" },
  { "셋", "셋", "셋", "셋", "셋" },
  { "셌", "셌", "셌", "셌", "셌" },
  { "셍", "셍", "셍", "셍", "셍" },
  { "셎", "셎", "셎", "셎", "셎" },
  { "셏", "셏", "셏", "셏", "셏" },
  { "셐", "셐", "셐", "셐", "셐" },
  { "셑", "셑", "셑", "셑", "셑" },
  { "셒", "셒", "셒", "셒", "셒" },
  { "셓", "셓", "셓", "셓", "셓" },
  { "셔", "셔", "셔", "셔", "셔" },
  { "셕", "셕", "셕", "셕", "셕" },
  { "셖", "셖", "셖", "셖", "셖" },
  { "셗", "셗", "셗", "셗", "셗" },
  { "션", "션", "션", "션", "션" },
  { "셙", "셙", "셙", "셙", "셙" },
  { "셚", "셚", "셚", "셚", "셚" },
  { "셛", "셛", "셛", "셛", "셛" },
  { "셜", "셜", "셜", "셜", "셜" },
  { "셝", "셝", "셝", "셝", "셝" },
  { "셞", "셞", "셞", "셞", "셞" },
  { "셟", "셟", "셟", "셟", "셟" },
  { "셠", "셠", "셠", "셠", "셠" },
  { "셡", "셡", "셡", "셡", "셡" },
  { "셢", "셢", "셢", "셢", "셢" },
  { "셣", "셣", "셣", "셣", "셣" },
  { "셤", "셤", "셤", "셤", "셤" },
  { "셥", "셥", "셥", "셥", "셥" },
  { "셦", "셦", "셦", "셦", "셦" },
  { "셧", "셧", "셧", "셧", "셧" },
  { "셨", "셨", "셨", "셨", "셨" },
  { "셩", "셩", "셩", "셩", "셩" },
  { "셪", "셪", "셪", "셪", "셪" },
  { "셫", "셫", "셫", "셫", "셫" },
  { "셬", "셬", "셬", "셬", "셬" },
  { "셭", "셭", "셭", "셭", "셭" },
  { "셮", "셮", "셮", "셮", "셮" },
  { "셯", "셯", "셯", "셯", "셯" },
  { "셰", "셰", "셰", "셰", "셰" },
  { "셱", "셱", "셱", "셱", "셱" },
  { "셲", "셲", "셲", "셲", "셲" },
  { "셳", "셳", "셳", "셳", "셳" },
  { "셴", "셴", "셴", "셴", "셴" },
  { "셵", "셵", "셵", "셵", "셵" },
  { "셶", "셶", "셶", "셶", "셶" },
  { "셷", "셷", "셷", "셷", "셷" },
  { "셸", "셸", "셸", "셸", "셸" },
  { "셹", "셹", "셹", "셹", "셹" },
  { "셺", "셺", "셺", "셺", "셺" },
  { "셻", "셻", "셻", "셻", "셻" },
  { "셼", "셼", "셼", "셼", "셼" },
  { "셽", "셽", "셽", "셽", "셽" },
  { "셾", "셾", "셾", "셾", "셾" },
  { "셿", "셿", "셿", "셿", "셿" },
  { "솀", "솀", "솀", "솀", "솀" },
  { "솁", "솁", "솁", "솁", "솁" },
  { "솂", "솂", "솂", "솂", "솂" },
  { "솃", "솃", "솃", "솃", "솃" },
  { "솄", "솄", "솄", "솄", "솄" },
  { "솅", "솅", "솅", "솅", "솅" },
  { "솆", "솆", "솆", "솆", "솆" },
  { "솇", "솇", "솇", "솇", "솇" },
  { "솈", "솈", "솈", "솈", "솈" },
  { "솉", "솉", "솉", "솉", "솉" },
  { "솊", "솊", "솊", "솊", "솊" },
  { "솋", "솋", "솋", "솋", "솋" },
  { "소", "소", "소", "소", "소" },
  { "속", "속", "속", "속", "속" },
  { "솎", "솎", "솎", "솎", "솎" },
  { "솏", "솏", "솏", "솏", "솏" },
  { "손", "손", "손", "손", "손" },
  { "솑", "솑", "솑", "솑", "솑" },
  { "솒", "솒", "솒", "솒", "솒" },
  { "솓", "솓", "솓", "솓", "솓" },
  { "솔", "솔", "솔", "솔", "솔" },
  { "솕", "솕", "솕", "솕", "솕" },
  { "솖", "솖", "솖", "솖", "솖" },
  { "솗", "솗", "솗", "솗", "솗" },
  { "솘", "솘", "솘", "솘", "솘" },
  { "솙", "솙", "솙", "솙", "솙" },
  { "솚", "솚", "솚", "솚", "솚" },
  { "솛", "솛", "솛", "솛", "솛" },
  { "솜", "솜", "솜", "솜", "솜" },
  { "솝", "솝", "솝", "솝", "솝" },
  { "솞", "솞", "솞", "솞", "솞" },
  { "솟", "솟", "솟", "솟", "솟" },
  { "솠", "솠", "솠", "솠", "솠" },
  { "송", "송", "송", "송", "송" },
  { "솢", "솢", "솢", "솢", "솢" },
  { "솣", "솣", "솣", "솣", "솣" },
  { "솤", "솤", "솤", "솤", "솤" },
  { "솥", "솥", "솥", "솥", "솥" },
  { "솦", "솦", "솦", "솦", "솦" },
  { "솧", "솧", "솧", "솧", "솧" },
  { "솨", "솨", "솨", "솨", "솨" },
  { "솩", "솩", "솩", "솩", "솩" },
  { "솪", "솪", "솪", "솪", "솪" },
  { "솫", "솫", "솫", "솫", "솫" },
  { "솬", "솬", "솬", "솬", "솬" },
  { "솭", "솭", "솭", "솭", "솭" },
  { "솮", "솮", "솮", "솮", "솮" },
  { "솯", "솯", "솯", "솯", "솯" },
  { "솰", "솰", "솰", "솰", "솰" },
  { "솱", "솱", "솱", "솱", "솱" },
  { "솲", "솲", "솲", "솲", "솲" },
  { "솳", "솳", "솳", "솳", "솳" },
  { "솴", "솴", "솴", "솴", "솴" },
  { "솵", "솵", "솵", "솵", "솵" },
  { "솶", "솶", "솶", "솶", "솶" },
  { "솷", "솷", "솷", "솷", "솷" },
  { "솸", "솸", "솸", "솸", "솸" },
  { "솹", "솹", "솹", "솹", "솹" },
  { "솺", "솺", "솺", "솺", "솺" },
  { "솻", "솻", "솻", "솻", "솻" },
  { "솼", "솼", "솼", "솼", "솼" },
  { "솽", "솽", "솽", "솽", "솽" },
  { "솾", "솾", "솾", "솾", "솾" },
  { "솿", "솿", "솿", "솿", "솿" },
  { "쇀", "쇀", "쇀", "쇀", "쇀" },
  { "쇁", "쇁", "쇁", "쇁", "쇁" },
  { "쇂", "쇂", "쇂", "쇂", "쇂" },
  { "쇃", "쇃", "쇃", "쇃", "쇃" },
  { "쇄", "쇄", "쇄", "쇄", "쇄" },
  { "쇅", "쇅", "쇅", "쇅", "쇅" },
  { "쇆", "쇆", "쇆", "쇆", "쇆" },
  { "쇇", "쇇", "쇇", "쇇", "쇇" },
  { "쇈", "쇈", "쇈", "쇈", "쇈" },
  { "쇉", "쇉", "쇉", "쇉", "쇉" },
  { "쇊", "쇊", "쇊", "쇊", "쇊" },
  { "쇋", "쇋", "쇋", "쇋", "쇋" },
  { "쇌", "쇌", "쇌", "쇌", "쇌" },
  { "쇍", "쇍", "쇍", "쇍", "쇍" },
  { "쇎", "쇎", "쇎", "쇎", "쇎" },
  { "쇏", "쇏", "쇏", "쇏", "쇏" },
  { "쇐", "쇐", "쇐", "쇐", "쇐" },
  { "쇑", "쇑", "쇑", "쇑", "쇑" },
  { "쇒", "쇒", "쇒", "쇒", "쇒" },
  { "쇓", "쇓", "쇓", "쇓", "쇓" },
  { "쇔", "쇔", "쇔", "쇔", "쇔" },
  { "쇕", "쇕", "쇕", "쇕", "쇕" },
  { "쇖", "쇖", "쇖", "쇖", "쇖" },
  { "쇗", "쇗", "쇗", "쇗", "쇗" },
  { "쇘", "쇘", "쇘", "쇘", "쇘" },
  { "쇙", "쇙", "쇙", "쇙", "쇙" },
  { "쇚", "쇚", "쇚", "쇚", "쇚" },
  { "쇛", "쇛", "쇛", "쇛", "쇛" },
  { "쇜", "쇜", "쇜", "쇜", "쇜" },
  { "쇝", "쇝", "쇝", "쇝", "쇝" },
  { "쇞", "쇞", "쇞", "쇞", "쇞" },
  { "쇟", "쇟", "쇟", "쇟", "쇟" },
  { "쇠", "쇠", "쇠", "쇠", "쇠" },
  { "쇡", "쇡", "쇡", "쇡", "쇡" },
  { "쇢", "쇢", "쇢", "쇢", "쇢" },
  { "쇣", "쇣", "쇣", "쇣", "쇣" },
  { "쇤", "쇤", "쇤", "쇤", "쇤" },
  { "쇥", "쇥", "쇥", "쇥", "쇥" },
  { "쇦", "쇦", "쇦", "쇦", "쇦" },
  { "쇧", "쇧", "쇧", "쇧", "쇧" },
  { "쇨", "쇨", "쇨", "쇨", "쇨" },
  { "쇩", "쇩", "쇩", "쇩", "쇩" },
  { "쇪", "쇪", "쇪", "쇪", "쇪" },
  { "쇫", "쇫", "쇫", "쇫", "쇫" },
  { "쇬", "쇬", "쇬", "쇬", "쇬" },
  { "쇭", "쇭", "쇭", "쇭", "쇭" },
  { "쇮", "쇮", "쇮", "쇮", "쇮" },
  { "쇯", "쇯", "쇯", "쇯", "쇯" },
  { "쇰", "쇰", "쇰", "쇰", "쇰" },
  { "쇱", "쇱", "쇱", "쇱", "쇱" },
  { "쇲", "쇲", "쇲", "쇲", "쇲" },
  { "쇳", "쇳", "쇳", "쇳", "쇳" },
  { "쇴", "쇴", "쇴", "쇴", "쇴" },
  { "쇵", "쇵", "쇵", "쇵", "쇵" },
  { "쇶", "쇶", "쇶", "쇶", "쇶" },
  { "쇷", "쇷", "쇷", "쇷", "쇷" },
  { "쇸", "쇸", "쇸", "쇸", "쇸" },
  { "쇹", "쇹", "쇹", "쇹", "쇹" },
  { "쇺", "쇺", "쇺", "쇺", "쇺" },
  { "쇻", "쇻", "쇻", "쇻", "쇻" },
  { "쇼", "쇼", "쇼", "쇼", "쇼" },
  { "쇽", "쇽", "쇽", "쇽", "쇽" },
  { "쇾", "쇾", "쇾", "쇾", "쇾" },
  { "쇿", "쇿", "쇿", "쇿", "쇿" },
  { "숀", "숀", "숀", "숀", "숀" },
  { "숁", "숁", "숁", "숁", "숁" },
  { "숂", "숂", "숂", "숂", "숂" },
  { "숃", "숃", "숃", "숃", "숃" },
  { "숄", "숄", "숄", "숄", "숄" },
  { "숅", "숅", "숅", "숅", "숅" },
  { "숆", "숆", "숆", "숆", "숆" },
  { "숇", "숇", "숇", "숇", "숇" },
  { "숈", "숈", "숈", "숈", "숈" },
  { "숉", "숉", "숉", "숉", "숉" },
  { "숊", "숊", "숊", "숊", "숊" },
  { "숋", "숋", "숋", "숋", "숋" },
  { "숌", "숌", "숌", "숌", "숌" },
  { "숍", "숍", "숍", "숍", "숍" },
  { "숎", "숎", "숎", "숎", "숎" },
  { "숏", "숏", "숏", "숏", "숏" },
  { "숐", "숐", "숐", "숐", "숐" },
  { "숑", "숑", "숑", "숑", "숑" },
  { "숒", "숒", "숒", "숒", "숒" },
  { "숓", "숓", "숓", "숓", "숓" },
  { "숔", "숔", "숔", "숔", "숔" },
  { "숕", "숕", "숕", "숕", "숕" },
  { "숖", "숖", "숖", "숖", "숖" },
  { "숗", "숗", "숗", "숗", "숗" },
  { "수", "수", "수", "수", "수" },
  { "숙", "숙", "숙", "숙", "숙" },
  { "숚", "숚", "숚", "숚", "숚" },
  { "숛", "숛", "숛", "숛", "숛" },
  { "순", "순", "순", "순", "순" },
  { "숝", "숝", "숝", "숝", "숝" },
  { "숞", "숞", "숞", "숞", "숞" },
  { "숟", "숟", "숟", "숟", "숟" },
  { "술", "술", "술", "술", "술" },
  { "숡", "숡", "숡", "숡", "숡" },
  { "숢", "숢", "숢", "숢", "숢" },
  { "숣", "숣", "숣", "숣", "숣" },
  { "숤", "숤", "숤", "숤", "숤" },
  { "숥", "숥", "숥", "숥", "숥" },
  { "숦", "숦", "숦", "숦", "숦" },
  { "숧", "숧", "숧", "숧", "숧" },
  { "숨", "숨", "숨", "숨", "숨" },
  { "숩", "숩", "숩", "숩", "숩" },
  { "숪", "숪", "숪", "숪", "숪" },
  { "숫", "숫", "숫", "숫", "숫" },
  { "숬", "숬", "숬", "숬", "숬" },
  { "숭", "숭", "숭", "숭", "숭" },
  { "숮", "숮", "숮", "숮", "숮" },
  { "숯", "숯", "숯", "숯", "숯" },
  { "숰", "숰", "숰", "숰", "숰" },
  { "숱", "숱", "숱", "숱", "숱" },
  { "숲", "숲", "숲", "숲", "숲" },
  { "숳", "숳", "숳", "숳", "숳" },
  { "숴", "숴", "숴", "숴", "숴" },
  { "숵", "숵", "숵", "숵", "숵" },
  { "숶", "숶", "숶", "숶", "숶" },
  { "숷", "숷", "숷", "숷", "숷" },
  { "숸", "숸", "숸", "숸", "숸" },
  { "숹", "숹", "숹", "숹", "숹" },
  { "숺", "숺", "숺", "숺", "숺" },
  { "숻", "숻", "숻", "숻", "숻" },
  { "숼", "숼", "숼", "숼", "숼" },
  { "숽", "숽", "숽", "숽", "숽" },
  { "숾", "숾", "숾", "숾", "숾" },
  { "숿", "숿", "숿", "숿", "숿" },
  { "쉀", "쉀", "쉀", "쉀", "쉀" },
  { "쉁", "쉁", "쉁", "쉁", "쉁" },
  { "쉂", "쉂", "쉂", "쉂", "쉂" },
  { "쉃", "쉃", "쉃", "쉃", "쉃" },
  { "쉄", "쉄", "쉄", "쉄", "쉄" },
  { "쉅", "쉅", "쉅", "쉅", "쉅" },
  { "쉆", "쉆", "쉆", "쉆", "쉆" },
  { "쉇", "쉇", "쉇", "쉇", "쉇" },
  { "쉈", "쉈", "쉈", "쉈", "쉈" },
  { "쉉", "쉉", "쉉", "쉉", "쉉" },
  { "쉊", "쉊", "쉊", "쉊", "쉊" },
  { "쉋", "쉋", "쉋", "쉋", "쉋" },
  { "쉌", "쉌", "쉌", "쉌", "쉌" },
  { "쉍", "쉍", "쉍", "쉍", "쉍" },
  { "쉎", "쉎", "쉎", "쉎", "쉎" },
  { "쉏", "쉏", "쉏", "쉏", "쉏" },
  { "쉐", "쉐", "쉐", "쉐", "쉐" },
  { "쉑", "쉑", "쉑", "쉑", "쉑" },
  { "쉒", "쉒", "쉒", "쉒", "쉒" },
  { "쉓", "쉓", "쉓", "쉓", "쉓" },
  { "쉔", "쉔", "쉔", "쉔", "쉔" },
  { "쉕", "쉕", "쉕", "쉕", "쉕" },
  { "쉖", "쉖", "쉖", "쉖", "쉖" },
  { "쉗", "쉗", "쉗", "쉗", "쉗" },
  { "쉘", "쉘", "쉘", "쉘", "쉘" },
  { "쉙", "쉙", "쉙", "쉙", "쉙" },
  { "쉚", "쉚", "쉚", "쉚", "쉚" },
  { "쉛", "쉛", "쉛", "쉛", "쉛" },
  { "쉜", "쉜", "쉜", "쉜", "쉜" },
  { "쉝", "쉝", "쉝", "쉝", "쉝" },
  { "쉞", "쉞", "쉞", "쉞", "쉞" },
  { "쉟", "쉟", "쉟", "쉟", "쉟" },
  { "쉠", "쉠", "쉠", "쉠", "쉠" },
  { "쉡", "쉡", "쉡", "쉡", "쉡" },
  { "쉢", "쉢", "쉢", "쉢", "쉢" },
  { "쉣", "쉣", "쉣", "쉣", "쉣" },
  { "쉤", "쉤", "쉤", "쉤", "쉤" },
  { "쉥", "쉥", "쉥", "쉥", "쉥" },
  { "쉦", "쉦", "쉦", "쉦", "쉦" },
  { "쉧", "쉧", "쉧", "쉧", "쉧" },
  { "쉨", "쉨", "쉨", "쉨", "쉨" },
  { "쉩", "쉩", "쉩", "쉩", "쉩" },
  { "쉪", "쉪", "쉪", "쉪", "쉪" },
  { "쉫", "쉫", "쉫", "쉫", "쉫" },
  { "쉬", "쉬", "쉬", "쉬", "쉬" },
  { "쉭", "쉭", "쉭", "쉭", "쉭" },
  { "쉮", "쉮", "쉮", "쉮", "쉮" },
  { "쉯", "쉯", "쉯", "쉯", "쉯" },
  { "쉰", "쉰", "쉰", "쉰", "쉰" },
  { "쉱", "쉱", "쉱", "쉱", "쉱" },
  { "쉲", "쉲", "쉲", "쉲", "쉲" },
  { "쉳", "쉳", "쉳", "쉳", "쉳" },
  { "쉴", "쉴", "쉴", "쉴", "쉴" },
  { "쉵", "쉵", "쉵", "쉵", "쉵" },
  { "쉶", "쉶", "쉶", "쉶", "쉶" },
  { "쉷", "쉷", "쉷", "쉷", "쉷" },
  { "쉸", "쉸", "쉸", "쉸", "쉸" },
  { "쉹", "쉹", "쉹", "쉹", "쉹" },
  { "쉺", "쉺", "쉺", "쉺", "쉺" },
  { "쉻", "쉻", "쉻", "쉻", "쉻" },
  { "쉼", "쉼", "쉼", "쉼", "쉼" },
  { "쉽", "쉽", "쉽", "쉽", "쉽" },
  { "쉾", "쉾", "쉾", "쉾", "쉾" },
  { "쉿", "쉿", "쉿", "쉿", "쉿" },
  { "슀", "슀", "슀", "슀", "슀" },
  { "슁", "슁", "슁", "슁", "슁" },
  { "슂", "슂", "슂", "슂", "슂" },
  { "슃", "슃", "슃", "슃", "슃" },
  { "슄", "슄", "슄", "슄", "슄" },
  { "슅", "슅", "슅", "슅", "슅" },
  { "슆", "슆", "슆", "슆", "슆" },
  { "슇", "슇", "슇", "슇", "슇" },
  { "슈", "슈", "슈", "슈", "슈" },
  { "슉", "슉", "슉", "슉", "슉" },
  { "슊", "슊", "슊", "슊", "슊" },
  { "슋", "슋", "슋", "슋", "슋" },
  { "슌", "슌", "슌", "슌", "슌" },
  { "슍", "슍", "슍", "슍", "슍" },
  { "슎", "슎", "슎", "슎", "슎" },
  { "슏", "슏", "슏", "슏", "슏" },
  { "슐", "슐", "슐", "슐", "슐" },
  { "슑", "슑", "슑", "슑", "슑" },
  { "슒", "슒", "슒", "슒", "슒" },
  { "슓", "슓", "슓", "슓", "슓" },
  { "슔", "슔", "슔", "슔", "슔" },
  { "슕", "슕", "슕", "슕", "슕" },
  { "슖", "슖", "슖", "슖", "슖" },
  { "슗", "슗", "슗", "슗", "슗" },
  { "슘", "슘", "슘", "슘", "슘" },
  { "슙", "슙", "슙", "슙", "슙" },
  { "슚", "슚", "슚", "슚", "슚" },
  { "슛", "슛", "슛", "슛", "슛" },
  { "슜", "슜", "슜", "슜", "슜" },
  { "슝", "슝", "슝", "슝", "슝" },
  { "슞", "슞", "슞", "슞", "슞" },
  { "슟", "슟", "슟", "슟", "슟" },
  { "슠", "슠", "슠", "슠", "슠" },
  { "슡", "슡", "슡", "슡", "슡" },
  { "슢", "슢", "슢", "슢", "슢" },
  { "슣", "슣", "슣", "슣", "슣" },
  { "스", "스", "스", "스", "스" },
  { "슥", "슥", "슥", "슥", "슥" },
  { "슦", "슦", "슦", "슦", "슦" },
  { "슧", "슧", "슧", "슧", "슧" },
  { "슨", "슨", "슨", "슨", "슨" },
  { "슩", "슩", "슩", "슩", "슩" },
  { "슪", "슪", "슪", "슪", "슪" },
  { "슫", "슫", "슫", "슫", "슫" },
  { "슬", "슬", "슬", "슬", "슬" },
  { "슭", "슭", "슭", "슭", "슭" },
  { "슮", "슮", "슮", "슮", "슮" },
  { "슯", "슯", "슯", "슯", "슯" },
  { "슰", "슰", "슰", "슰", "슰" },
  { "슱", "슱", "슱", "슱", "슱" },
  { "슲", "슲", "슲", "슲", "슲" },
  { "슳", "슳", "슳", "슳", "슳" },
  { "슴", "슴", "슴", "슴", "슴" },
  { "습", "습", "습", "습", "습" },
  { "슶", "슶", "슶", "슶", "슶" },
  { "슷", "슷", "슷", "슷", "슷" },
  { "슸", "슸", "슸", "슸", "슸" },
  { "승", "승", "승", "승", "승" },
  { "슺", "슺", "슺", "슺", "슺" },
  { "슻", "슻", "슻", "슻", "슻" },
  { "슼", "슼", "슼", "슼", "슼" },
  { "슽", "슽", "슽", "슽", "슽" },
  { "슾", "슾", "슾", "슾", "슾" },
  { "슿", "슿", "슿", "슿", "슿" },
  { "싀", "싀", "싀", "싀", "싀" },
  { "싁", "싁", "싁", "싁", "싁" },
  { "싂", "싂", "싂", "싂", "싂" },
  { "싃", "싃", "싃", "싃", "싃" },
  { "싄", "싄", "싄", "싄", "싄" },
  { "싅", "싅", "싅", "싅", "싅" },
  { "싆", "싆", "싆", "싆", "싆" },
  { "싇", "싇", "싇", "싇", "싇" },
  { "싈", "싈", "싈", "싈", "싈" },
  { "싉", "싉", "싉", "싉", "싉" },
  { "싊", "싊", "싊", "싊", "싊" },
  { "싋", "싋", "싋", "싋", "싋" },
  { "싌", "싌", "싌", "싌", "싌" },
  { "싍", "싍", "싍", "싍", "싍" },
  { "싎", "싎", "싎", "싎", "싎" },
  { "싏", "싏", "싏", "싏", "싏" },
  { "싐", "싐", "싐", "싐", "싐" },
  { "싑", "싑", "싑", "싑", "싑" },
  { "싒", "싒", "싒", "싒", "싒" },
  { "싓", "싓", "싓", "싓", "싓" },
  { "싔", "싔", "싔", "싔", "싔" },
  { "싕", "싕", "싕", "싕", "싕" },
  { "싖", "싖", "싖", "싖", "싖" },
  { "싗", "싗", "싗", "싗", "싗" },
  { "싘", "싘", "싘", "싘", "싘" },
  { "싙", "싙", "싙", "싙", "싙" },
  { "싚", "싚", "싚", "싚", "싚" },
  { "싛", "싛", "싛", "싛", "싛" },
  { "시", "시", "시", "시", "시" },
  { "식", "식", "식", "식", "식" },
  { "싞", "싞", "싞", "싞", "싞" },
  { "싟", "싟", "싟", "싟", "싟" },
  { "신", "신", "신", "신", "신" },
  { "싡", "싡", "싡", "싡", "싡" },
  { "싢", "싢", "싢", "싢", "싢" },
  { "싣", "싣", "싣", "싣", "싣" },
  { "실", "실", "실", "실", "실" },
  { "싥", "싥", "싥", "싥", "싥" },
  { "싦", "싦", "싦", "싦", "싦" },
  { "싧", "싧", "싧", "싧", "싧" },
  { "싨", "싨", "싨", "싨", "싨" },
  { "싩", "싩", "싩", "싩", "싩" },
  { "싪", "싪", "싪", "싪", "싪" },
  { "싫", "싫", "싫", "싫", "싫" },
  { "심", "심", "심", "심", "심" },
  { "십", "십", "십", "십", "십" },
  { "싮", "싮", "싮", "싮", "싮" },
  { "싯", "싯", "싯", "싯", "싯" },
  { "싰", "싰", "싰", "싰", "싰" },
  { "싱", "싱", "싱", "싱", "싱" },
  { "싲", "싲", "싲", "싲", "싲" },
  { "싳", "싳", "싳", "싳", "싳" },
  { "싴", "싴", "싴", "싴", "싴" },
  { "싵", "싵", "싵", "싵", "싵" },
  { "싶", "싶", "싶", "싶", "싶" },
  { "싷", "싷", "싷", "싷", "싷" },
  { "싸", "싸", "싸", "싸", "싸" },
  { "싹", "싹", "싹", "싹", "싹" },
  { "싺", "싺", "싺", "싺", "싺" },
  { "싻", "싻", "싻", "싻", "싻" },
  { "싼", "싼", "싼", "싼", "싼" },
  { "싽", "싽", "싽", "싽", "싽" },
  { "싾", "싾", "싾", "싾", "싾" },
  { "싿", "싿", "싿", "싿", "싿" },
  { "쌀", "쌀", "쌀", "쌀", "쌀" },
  { "쌁", "쌁", "쌁", "쌁", "쌁" },
  { "쌂", "쌂", "쌂", "쌂", "쌂" },
  { "쌃", "쌃", "쌃", "쌃", "쌃" },
  { "쌄", "쌄", "쌄", "쌄", "쌄" },
  { "쌅", "쌅", "쌅", "쌅", "쌅" },
  { "쌆", "쌆", "쌆", "쌆", "쌆" },
  { "쌇", "쌇", "쌇", "쌇", "쌇" },
  { "쌈", "쌈", "쌈", "쌈", "쌈" },
  { "쌉", "쌉", "쌉", "쌉", "쌉" },
  { "쌊", "쌊", "쌊", "쌊", "쌊" },
  { "쌋", "쌋", "쌋", "쌋", "쌋" },
  { "쌌", "쌌", "쌌", "쌌", "쌌" },
  { "쌍", "쌍", "쌍", "쌍", "쌍" },
  { "쌎", "쌎", "쌎", "쌎", "쌎" },
  { "쌏", "쌏", "쌏", "쌏", "쌏" },
  { "쌐", "쌐", "쌐", "쌐", "쌐" },
  { "쌑", "쌑", "쌑", "쌑", "쌑" },
  { "쌒", "쌒", "쌒", "쌒", "쌒" },
  { "쌓", "쌓", "쌓", "쌓", "쌓" },
  { "쌔", "쌔", "쌔", "쌔", "쌔" },
  { "쌕", "쌕", "쌕", "쌕", "쌕" },
  { "쌖", "쌖", "쌖", "쌖", "쌖" },
  { "쌗", "쌗", "쌗", "쌗", "쌗" },
  { "쌘", "쌘", "쌘", "쌘", "쌘" },
  { "쌙", "쌙", "쌙", "쌙", "쌙" },
  { "쌚", "쌚", "쌚", "쌚", "쌚" },
  { "쌛", "쌛", "쌛", "쌛", "쌛" },
  { "쌜", "쌜", "쌜", "쌜", "쌜" },
  { "쌝", "쌝", "쌝", "쌝", "쌝" },
  { "쌞", "쌞", "쌞", "쌞", "쌞" },
  { "쌟", "쌟", "쌟", "쌟", "쌟" },
  { "쌠", "쌠", "쌠", "쌠", "쌠" },
  { "쌡", "쌡", "쌡", "쌡", "쌡" },
  { "쌢", "쌢", "쌢", "쌢", "쌢" },
  { "쌣", "쌣", "쌣", "쌣", "쌣" },
  { "쌤", "쌤", "쌤", "쌤", "쌤" },
  { "쌥", "쌥", "쌥", "쌥", "쌥" },
  { "쌦", "쌦", "쌦", "쌦", "쌦" },
  { "쌧", "쌧", "쌧", "쌧", "쌧" },
  { "쌨", "쌨", "쌨", "쌨", "쌨" },
  { "쌩", "쌩", "쌩", "쌩", "쌩" },
  { "쌪", "쌪", "쌪", "쌪", "쌪" },
  { "쌫", "쌫", "쌫", "쌫", "쌫" },
  { "쌬", "쌬", "쌬", "쌬", "쌬" },
  { "쌭", "쌭", "쌭", "쌭", "쌭" },
  { "쌮", "쌮", "쌮", "쌮", "쌮" },
  { "쌯", "쌯", "쌯", "쌯", "쌯" },
  { "쌰", "쌰", "쌰", "쌰", "쌰" },
  { "쌱", "쌱", "쌱", "쌱", "쌱" },
  { "쌲", "쌲", "쌲", "쌲", "쌲" },
  { "쌳", "쌳", "쌳", "쌳", "쌳" },
  { "쌴", "쌴", "쌴", "쌴", "쌴" },
  { "쌵", "쌵", "쌵", "쌵", "쌵" },
  { "쌶", "쌶", "쌶", "쌶", "쌶" },
  { "쌷", "쌷", "쌷", "쌷", "쌷" },
  { "쌸", "쌸", "쌸", "쌸", "쌸" },
  { "쌹", "쌹", "쌹", "쌹", "쌹" },
  { "쌺", "쌺", "쌺", "쌺", "쌺" },
  { "쌻", "쌻", "쌻", "쌻", "쌻" },
  { "쌼", "쌼", "쌼", "쌼", "쌼" },
  { "쌽", "쌽", "쌽", "쌽", "쌽" },
  { "쌾", "쌾", "쌾", "쌾", "쌾" },
  { "쌿", "쌿", "쌿", "쌿", "쌿" },
  { "썀", "썀", "썀", "썀", "썀" },
  { "썁", "썁", "썁", "썁", "썁" },
  { "썂", "썂", "썂", "썂", "썂" },
  { "썃", "썃", "썃", "썃", "썃" },
  { "썄", "썄", "썄", "썄", "썄" },
  { "썅", "썅", "썅", "썅", "썅" },
  { "썆", "썆", "썆", "썆", "썆" },
  { "썇", "썇", "썇", "썇", "썇" },
  { "썈", "썈", "썈", "썈", "썈" },
  { "썉", "썉", "썉", "썉", "썉" },
  { "썊", "썊", "썊", "썊", "썊" },
  { "썋", "썋", "썋", "썋", "썋" },
  { "썌", "썌", "썌", "썌", "썌" },
  { "썍", "썍", "썍", "썍", "썍" },
  { "썎", "썎", "썎", "썎", "썎" },
  { "썏", "썏", "썏", "썏", "썏" },
  { "썐", "썐", "썐", "썐", "썐" },
  { "썑", "썑", "썑", "썑", "썑" },
  { "썒", "썒", "썒", "썒", "썒" },
  { "썓", "썓", "썓", "썓", "썓" },
  { "썔", "썔", "썔", "썔", "썔" },
  { "썕", "썕", "썕", "썕", "썕" },
  { "썖", "썖", "썖", "썖", "썖" },
  { "썗", "썗", "썗", "썗", "썗" },
  { "썘", "썘", "썘", "썘", "썘" },
  { "썙", "썙", "썙", "썙", "썙" },
  { "썚", "썚", "썚", "썚", "썚" },
  { "썛", "썛", "썛", "썛", "썛" },
  { "썜", "썜", "썜", "썜", "썜" },
  { "썝", "썝", "썝", "썝", "썝" },
  { "썞", "썞", "썞", "썞", "썞" },
  { "썟", "썟", "썟", "썟", "썟" },
  { "썠", "썠", "썠", "썠", "썠" },
  { "썡", "썡", "썡", "썡", "썡" },
  { "썢", "썢", "썢", "썢", "썢" },
  { "썣", "썣", "썣", "썣", "썣" },
  { "썤", "썤", "썤", "썤", "썤" },
  { "썥", "썥", "썥", "썥", "썥" },
  { "썦", "썦", "썦", "썦", "썦" },
  { "썧", "썧", "썧", "썧", "썧" },
  { "써", "써", "써", "써", "써" },
  { "썩", "썩", "썩", "썩", "썩" },
  { "썪", "썪", "썪", "썪", "썪" },
  { "썫", "썫", "썫", "썫", "썫" },
  { "썬", "썬", "썬", "썬", "썬" },
  { "썭", "썭", "썭", "썭", "썭" },
  { "썮", "썮", "썮", "썮", "썮" },
  { "썯", "썯", "썯", "썯", "썯" },
  { "썰", "썰", "썰", "썰", "썰" },
  { "썱", "썱", "썱", "썱", "썱" },
  { "썲", "썲", "썲", "썲", "썲" },
  { "썳", "썳", "썳", "썳", "썳" },
  { "썴", "썴", "썴", "썴", "썴" },
  { "썵", "썵", "썵", "썵", "썵" },
  { "썶", "썶", "썶", "썶", "썶" },
  { "썷", "썷", "썷", "썷", "썷" },
  { "썸", "썸", "썸", "썸", "썸" },
  { "썹", "썹", "썹", "썹", "썹" },
  { "썺", "썺", "썺", "썺", "썺" },
  { "썻", "썻", "썻", "썻", "썻" },
  { "썼", "썼", "썼", "썼", "썼" },
  { "썽", "썽", "썽", "썽", "썽" },
  { "썾", "썾", "썾", "썾", "썾" },
  { "썿", "썿", "썿", "썿", "썿" },
  { "쎀", "쎀", "쎀", "쎀", "쎀" },
  { "쎁", "쎁", "쎁", "쎁", "쎁" },
  { "쎂", "쎂", "쎂", "쎂", "쎂" },
  { "쎃", "쎃", "쎃", "쎃", "쎃" },
  { "쎄", "쎄", "쎄", "쎄", "쎄" },
  { "쎅", "쎅", "쎅", "쎅", "쎅" },
  { "쎆", "쎆", "쎆", "쎆", "쎆" },
  { "쎇", "쎇", "쎇", "쎇", "쎇" },
  { "쎈", "쎈", "쎈", "쎈", "쎈" },
  { "쎉", "쎉", "쎉", "쎉", "쎉" },
  { "쎊", "쎊", "쎊", "쎊", "쎊" },
  { "쎋", "쎋", "쎋", "쎋", "쎋" },
  { "쎌", "쎌", "쎌", "쎌", "쎌" },
  { "쎍", "쎍", "쎍", "쎍", "쎍" },
  { "쎎", "쎎", "쎎", "쎎", "쎎" },
  { "쎏", "쎏", "쎏", "쎏", "쎏" },
  { "쎐", "쎐", "쎐", "쎐", "쎐" },
  { "쎑", "쎑", "쎑", "쎑", "쎑" },
  { "쎒", "쎒", "쎒", "쎒", "쎒" },
  { "쎓", "쎓", "쎓", "쎓", "쎓" },
  { "쎔", "쎔", "쎔", "쎔", "쎔" },
  { "쎕", "쎕", "쎕", "쎕", "쎕" },
  { "쎖", "쎖", "쎖", "쎖", "쎖" },
  { "쎗", "쎗", "쎗", "쎗", "쎗" },
  { "쎘", "쎘", "쎘", "쎘", "쎘" },
  { "쎙", "쎙", "쎙", "쎙", "쎙" },
  { "쎚", "쎚", "쎚", "쎚", "쎚" },
  { "쎛", "쎛", "쎛", "쎛", "쎛" },
  { "쎜", "쎜", "쎜", "쎜", "쎜" },
  { "쎝", "쎝", "쎝", "쎝", "쎝" },
  { "쎞", "쎞", "쎞", "쎞", "쎞" },
  { "쎟", "쎟", "쎟", "쎟", "쎟" },
  { "쎠", "쎠", "쎠", "쎠", "쎠" },
  { "쎡", "쎡", "쎡", "쎡", "쎡" },
  { "쎢", "쎢", "쎢", "쎢", "쎢" },
  { "쎣", "쎣", "쎣", "쎣", "쎣" },
  { "쎤", "쎤", "쎤", "쎤", "쎤" },
  { "쎥", "쎥", "쎥", "쎥", "쎥" },
  { "쎦", "쎦", "쎦", "쎦", "쎦" },
  { "쎧", "쎧", "쎧", "쎧", "쎧" },
  { "쎨", "쎨", "쎨", "쎨", "쎨" },
  { "쎩", "쎩", "쎩", "쎩", "쎩" },
  { "쎪", "쎪", "쎪", "쎪", "쎪" },
  { "쎫", "쎫", "쎫", "쎫", "쎫" },
  { "쎬", "쎬", "쎬", "쎬", "쎬" },
  { "쎭", "쎭", "쎭", "쎭", "쎭" },
  { "쎮", "쎮", "쎮", "쎮", "쎮" },
  { "쎯", "쎯", "쎯", "쎯", "쎯" },
  { "쎰", "쎰", "쎰", "쎰", "쎰" },
  { "쎱", "쎱", "쎱", "쎱", "쎱" },
  { "쎲", "쎲", "쎲", "쎲", "쎲" },
  { "쎳", "쎳", "쎳", "쎳", "쎳" },
  { "쎴", "쎴", "쎴", "쎴", "쎴" },
  { "쎵", "쎵", "쎵", "쎵", "쎵" },
  { "쎶", "쎶", "쎶", "쎶", "쎶" },
  { "쎷", "쎷", "쎷", "쎷", "쎷" },
  { "쎸", "쎸", "쎸", "쎸", "쎸" },
  { "쎹", "쎹", "쎹", "쎹", "쎹" },
  { "쎺", "쎺", "쎺", "쎺", "쎺" },
  { "쎻", "쎻", "쎻", "쎻", "쎻" },
  { "쎼", "쎼", "쎼", "쎼", "쎼" },
  { "쎽", "쎽", "쎽", "쎽", "쎽" },
  { "쎾", "쎾", "쎾", "쎾", "쎾" },
  { "쎿", "쎿", "쎿", "쎿", "쎿" },
  { "쏀", "쏀", "쏀", "쏀", "쏀" },
  { "쏁", "쏁", "쏁", "쏁", "쏁" },
  { "쏂", "쏂", "쏂", "쏂", "쏂" },
  { "쏃", "쏃", "쏃", "쏃", "쏃" },
  { "쏄", "쏄", "쏄", "쏄", "쏄" },
  { "쏅", "쏅", "쏅", "쏅", "쏅" },
  { "쏆", "쏆", "쏆", "쏆", "쏆" },
  { "쏇", "쏇", "쏇", "쏇", "쏇" },
  { "쏈", "쏈", "쏈", "쏈", "쏈" },
  { "쏉", "쏉", "쏉", "쏉", "쏉" },
  { "쏊", "쏊", "쏊", "쏊", "쏊" },
  { "쏋", "쏋", "쏋", "쏋", "쏋" },
  { "쏌", "쏌", "쏌", "쏌", "쏌" },
  { "쏍", "쏍", "쏍", "쏍", "쏍" },
  { "쏎", "쏎", "쏎", "쏎", "쏎" },
  { "쏏", "쏏", "쏏", "쏏", "쏏" },
  { "쏐", "쏐", "쏐", "쏐", "쏐" },
  { "쏑", "쏑", "쏑", "쏑", "쏑" },
  { "쏒", "쏒", "쏒", "쏒", "쏒" },
  { "쏓", "쏓", "쏓", "쏓", "쏓" },
  { "쏔", "쏔", "쏔", "쏔", "쏔" },
  { "쏕", "쏕", "쏕", "쏕", "쏕" },
  { "쏖", "쏖", "쏖", "쏖", "쏖" },
  { "쏗", "쏗", "쏗", "쏗", "쏗" },
  { "쏘", "쏘", "쏘", "쏘", "쏘" },
  { "쏙", "쏙", "쏙", "쏙", "쏙" },
  { "쏚", "쏚", "쏚", "쏚", "쏚" },
  { "쏛", "쏛", "쏛", "쏛", "쏛" },
  { "쏜", "쏜", "쏜", "쏜", "쏜" },
  { "쏝", "쏝", "쏝", "쏝", "쏝" },
  { "쏞", "쏞", "쏞", "쏞", "쏞" },
  { "쏟", "쏟", "쏟", "쏟", "쏟" },
  { "쏠", "쏠", "쏠", "쏠", "쏠" },
  { "쏡", "쏡", "쏡", "쏡", "쏡" },
  { "쏢", "쏢", "쏢", "쏢", "쏢" },
  { "쏣", "쏣", "쏣", "쏣", "쏣" },
  { "쏤", "쏤", "쏤", "쏤", "쏤" },
  { "쏥", "쏥", "쏥", "쏥", "쏥" },
  { "쏦", "쏦", "쏦", "쏦", "쏦" },
  { "쏧", "쏧", "쏧", "쏧", "쏧" },
  { "쏨", "쏨", "쏨", "쏨", "쏨" },
  { "쏩", "쏩", "쏩", "쏩", "쏩" },
  { "쏪", "쏪", "쏪", "쏪", "쏪" },
  { "쏫", "쏫", "쏫", "쏫", "쏫" },
  { "쏬", "쏬", "쏬", "쏬", "쏬" },
  { "쏭", "쏭", "쏭", "쏭", "쏭" },
  { "쏮", "쏮", "쏮", "쏮", "쏮" },
  { "쏯", "쏯", "쏯", "쏯", "쏯" },
  { "쏰", "쏰", "쏰", "쏰", "쏰" },
  { "쏱", "쏱", "쏱", "쏱", "쏱" },
  { "쏲", "쏲", "쏲", "쏲", "쏲" },
  { "쏳", "쏳", "쏳", "쏳", "쏳" },
  { "쏴", "쏴", "쏴", "쏴", "쏴" },
  { "쏵", "쏵", "쏵", "쏵", "쏵" },
  { "쏶", "쏶", "쏶", "쏶", "쏶" },
  { "쏷", "쏷", "쏷", "쏷", "쏷" },
  { "쏸", "쏸", "쏸", "쏸", "쏸" },
  { "쏹", "쏹", "쏹", "쏹", "쏹" },
  { "쏺", "쏺", "쏺", "쏺", "쏺" },
  { "쏻", "쏻", "쏻", "쏻", "쏻" },
  { "쏼", "쏼", "쏼", "쏼", "쏼" },
  { "쏽", "쏽", "쏽", "쏽", "쏽" },
  { "쏾", "쏾", "쏾", "쏾", "쏾" },
  { "쏿", "쏿", "쏿", "쏿", "쏿" },
  { "쐀", "쐀", "쐀", "쐀", "쐀" },
  { "쐁", "쐁", "쐁", "쐁", "쐁" },
  { "쐂", "쐂", "쐂", "쐂", "쐂" },
  { "쐃", "쐃", "쐃", "쐃", "쐃" },
  { "쐄", "쐄", "쐄", "쐄", "쐄" },
  { "쐅", "쐅", "쐅", "쐅", "쐅" },
  { "쐆", "쐆", "쐆", "쐆", "쐆" },
  { "쐇", "쐇", "쐇", "쐇", "쐇" },
  { "쐈", "쐈", "쐈", "쐈", "쐈" },
  { "쐉", "쐉", "쐉", "쐉", "쐉" },
  { "쐊", "쐊", "쐊", "쐊", "쐊" },
  { "쐋", "쐋", "쐋", "쐋", "쐋" },
  { "쐌", "쐌", "쐌", "쐌", "쐌" },
  { "쐍", "쐍", "쐍", "쐍", "쐍" },
  { "쐎", "쐎", "쐎", "쐎", "쐎" },
  { "쐏", "쐏", "쐏", "쐏", "쐏" },
  { "쐐", "쐐", "쐐", "쐐", "쐐" },
  { "쐑", "쐑", "쐑", "쐑", "쐑" },
  { "쐒", "쐒", "쐒", "쐒", "쐒" },
  { "쐓", "쐓", "쐓", "쐓", "쐓" },
  { "쐔", "쐔", "쐔", "쐔", "쐔" },
  { "쐕", "쐕", "쐕", "쐕", "쐕" },
  { "쐖", "쐖", "쐖", "쐖", "쐖" },
  { "쐗", "쐗", "쐗", "쐗", "쐗" },
  { "쐘", "쐘", "쐘", "쐘", "쐘" },
  { "쐙", "쐙", "쐙", "쐙", "쐙" },
  { "쐚", "쐚", "쐚", "쐚", "쐚" },
  { "쐛", "쐛", "쐛", "쐛", "쐛" },
  { "쐜", "쐜", "쐜", "쐜", "쐜" },
  { "쐝", "쐝", "쐝", "쐝", "쐝" },
  { "쐞", "쐞", "쐞", "쐞", "쐞" },
  { "쐟", "쐟", "쐟", "쐟", "쐟" },
  { "쐠", "쐠", "쐠", "쐠", "쐠" },
  { "쐡", "쐡", "쐡", "쐡", "쐡" },
  { "쐢", "쐢", "쐢", "쐢", "쐢" },
  { "쐣", "쐣", "쐣", "쐣", "쐣" },
  { "쐤", "쐤", "쐤", "쐤", "쐤" },
  { "쐥", "쐥", "쐥", "쐥", "쐥" },
  { "쐦", "쐦", "쐦", "쐦", "쐦" },
  { "쐧", "쐧", "쐧", "쐧", "쐧" },
  { "쐨", "쐨", "쐨", "쐨", "쐨" },
  { "쐩", "쐩", "쐩", "쐩", "쐩" },
  { "쐪", "쐪", "쐪", "쐪", "쐪" },
  { "쐫", "쐫", "쐫", "쐫", "쐫" },
  { "쐬", "쐬", "쐬", "쐬", "쐬" },
  { "쐭", "쐭", "쐭", "쐭", "쐭" },
  { "쐮", "쐮", "쐮", "쐮", "쐮" },
  { "쐯", "쐯", "쐯", "쐯", "쐯" },
  { "쐰", "쐰", "쐰", "쐰", "쐰" },
  { "쐱", "쐱", "쐱", "쐱", "쐱" },
  { "쐲", "쐲", "쐲", "쐲", "쐲" },
  { "쐳", "쐳", "쐳", "쐳", "쐳" },
  { "쐴", "쐴", "쐴", "쐴", "쐴" },
  { "쐵", "쐵", "쐵", "쐵", "쐵" },
  { "쐶", "쐶", "쐶", "쐶", "쐶" },
  { "쐷", "쐷", "쐷", "쐷", "쐷" },
  { "쐸", "쐸", "쐸", "쐸", "쐸" },
  { "쐹", "쐹", "쐹", "쐹", "쐹" },
  { "쐺", "쐺", "쐺", "쐺", "쐺" },
  { "쐻", "쐻", "쐻", "쐻", "쐻" },
  { "쐼", "쐼", "쐼", "쐼", "쐼" },
  { "쐽", "쐽", "쐽", "쐽", "쐽" },
  { "쐾", "쐾", "쐾", "쐾", "쐾" },
  { "쐿", "쐿", "쐿", "쐿", "쐿" },
  { "쑀", "쑀", "쑀", "쑀", "쑀" },
  { "쑁", "쑁", "쑁", "쑁", "쑁" },
  { "쑂", "쑂", "쑂", "쑂", "쑂" },
  { "쑃", "쑃", "쑃", "쑃", "쑃" },
  { "쑄", "쑄", "쑄", "쑄", "쑄" },
  { "쑅", "쑅", "쑅", "쑅", "쑅" },
  { "쑆", "쑆", "쑆", "쑆", "쑆" },
  { "쑇", "쑇", "쑇", "쑇", "쑇" },
  { "쑈", "쑈", "쑈", "쑈", "쑈" },
  { "쑉", "쑉", "쑉", "쑉", "쑉" },
  { "쑊", "쑊", "쑊", "쑊", "쑊" },
  { "쑋", "쑋", "쑋", "쑋", "쑋" },
  { "쑌", "쑌", "쑌", "쑌", "쑌" },
  { "쑍", "쑍", "쑍", "쑍", "쑍" },
  { "쑎", "쑎", "쑎", "쑎", "쑎" },
  { "쑏", "쑏", "쑏", "쑏", "쑏" },
  { "쑐", "쑐", "쑐", "쑐", "쑐" },
  { "쑑", "쑑", "쑑", "쑑", "쑑" },
  { "쑒", "쑒", "쑒", "쑒", "쑒" },
  { "쑓", "쑓", "쑓", "쑓", "쑓" },
  { "쑔", "쑔", "쑔", "쑔", "쑔" },
  { "쑕", "쑕", "쑕", "쑕", "쑕" },
  { "쑖", "쑖", "쑖", "쑖", "쑖" },
  { "쑗", "쑗", "쑗", "쑗", "쑗" },
  { "쑘", "쑘", "쑘", "쑘", "쑘" },
  { "쑙", "쑙", "쑙", "쑙", "쑙" },
  { "쑚", "쑚", "쑚", "쑚", "쑚" },
  { "쑛", "쑛", "쑛", "쑛", "쑛" },
  { "쑜", "쑜", "쑜", "쑜", "쑜" },
  { "쑝", "쑝", "쑝", "쑝", "쑝" },
  { "쑞", "쑞", "쑞", "쑞", "쑞" },
  { "쑟", "쑟", "쑟", "쑟", "쑟" },
  { "쑠", "쑠", "쑠", "쑠", "쑠" },
  { "쑡", "쑡", "쑡", "쑡", "쑡" },
  { "쑢", "쑢", "쑢", "쑢", "쑢" },
  { "쑣", "쑣", "쑣", "쑣", "쑣" },
  { "쑤", "쑤", "쑤", "쑤", "쑤" },
  { "쑥", "쑥", "쑥", "쑥", "쑥" },
  { "쑦", "쑦", "쑦", "쑦", "쑦" },
  { "쑧", "쑧", "쑧", "쑧", "쑧" },
  { "쑨", "쑨", "쑨", "쑨", "쑨" },
  { "쑩", "쑩", "쑩", "쑩", "쑩" },
  { "쑪", "쑪", "쑪", "쑪", "쑪" },
  { "쑫", "쑫", "쑫", "쑫", "쑫" },
  { "쑬", "쑬", "쑬", "쑬", "쑬" },
  { "쑭", "쑭", "쑭", "쑭", "쑭" },
  { "쑮", "쑮", "쑮", "쑮", "쑮" },
  { "쑯", "쑯", "쑯", "쑯", "쑯" },
  { "쑰", "쑰", "쑰", "쑰", "쑰" },
  { "쑱", "쑱", "쑱", "쑱", "쑱" },
  { "쑲", "쑲", "쑲", "쑲", "쑲" },
  { "쑳", "쑳", "쑳", "쑳", "쑳" },
  { "쑴", "쑴", "쑴", "쑴", "쑴" },
  { "쑵", "쑵", "쑵", "쑵", "쑵" },
  { "쑶", "쑶", "쑶", "쑶", "쑶" },
  { "쑷", "쑷", "쑷", "쑷", "쑷" },
  { "쑸", "쑸", "쑸", "쑸", "쑸" },
  { "쑹", "쑹", "쑹", "쑹", "쑹" },
  { "쑺", "쑺", "쑺", "쑺", "쑺" },
  { "쑻", "쑻", "쑻", "쑻", "쑻" },
  { "쑼", "쑼", "쑼", "쑼", "쑼" },
  { "쑽", "쑽", "쑽", "쑽", "쑽" },
  { "쑾", "쑾", "쑾", "쑾", "쑾" },
  { "쑿", "쑿", "쑿", "쑿", "쑿" },
  { "쒀", "쒀", "쒀", "쒀", "쒀" },
  { "쒁", "쒁", "쒁", "쒁", "쒁" },
  { "쒂", "쒂", "쒂", "쒂", "쒂" },
  { "쒃", "쒃", "쒃", "쒃", "쒃" },
  { "쒄", "쒄", "쒄", "쒄", "쒄" },
  { "쒅", "쒅", "쒅", "쒅", "쒅" },
  { "쒆", "쒆", "쒆", "쒆", "쒆" },
  { "쒇", "쒇", "쒇", "쒇", "쒇" },
  { "쒈", "쒈", "쒈", "쒈", "쒈" },
  { "쒉", "쒉", "쒉", "쒉", "쒉" },
  { "쒊", "쒊", "쒊", "쒊", "쒊" },
  { "쒋", "쒋", "쒋", "쒋", "쒋" },
  { "쒌", "쒌", "쒌", "쒌", "쒌" },
  { "쒍", "쒍", "쒍", "쒍", "쒍" },
  { "쒎", "쒎", "쒎", "쒎", "쒎" },
  { "쒏", "쒏", "쒏", "쒏", "쒏" },
  { "쒐", "쒐", "쒐", "쒐", "쒐" },
  { "쒑", "쒑", "쒑", "쒑", "쒑" },
  { "쒒", "쒒", "쒒", "쒒", "쒒" },
  { "쒓", "쒓", "쒓", "쒓", "쒓" },
  { "쒔", "쒔", "쒔", "쒔", "쒔" },
  { "쒕", "쒕", "쒕", "쒕", "쒕" },
  { "쒖", "쒖", "쒖", "쒖", "쒖" },
  { "쒗", "쒗", "쒗", "쒗", "쒗" },
  { "쒘", "쒘", "쒘", "쒘", "쒘" },
  { "쒙", "쒙", "쒙", "쒙", "쒙" },
  { "쒚", "쒚", "쒚", "쒚", "쒚" },
  { "쒛", "쒛", "쒛", "쒛", "쒛" },
  { "쒜", "쒜", "쒜", "쒜", "쒜" },
  { "쒝", "쒝", "쒝", "쒝", "쒝" },
  { "쒞", "쒞", "쒞", "쒞", "쒞" },
  { "쒟", "쒟", "쒟", "쒟", "쒟" },
  { "쒠", "쒠", "쒠", "쒠", "쒠" },
  { "쒡", "쒡", "쒡", "쒡", "쒡" },
  { "쒢", "쒢", "쒢", "쒢", "쒢" },
  { "쒣", "쒣", "쒣", "쒣", "쒣" },
  { "쒤", "쒤", "쒤", "쒤", "쒤" },
  { "쒥", "쒥", "쒥", "쒥", "쒥" },
  { "쒦", "쒦", "쒦", "쒦", "쒦" },
  { "쒧", "쒧", "쒧", "쒧", "쒧" },
  { "쒨", "쒨", "쒨", "쒨", "쒨" },
  { "쒩", "쒩", "쒩", "쒩", "쒩" },
  { "쒪", "쒪", "쒪", "쒪", "쒪" },
  { "쒫", "쒫", "쒫", "쒫", "쒫" },
  { "쒬", "쒬", "쒬", "쒬", "쒬" },
  { "쒭", "쒭", "쒭", "쒭", "쒭" },
  { "쒮", "쒮", "쒮", "쒮", "쒮" },
  { "쒯", "쒯", "쒯", "쒯", "쒯" },
  { "쒰", "쒰", "쒰", "쒰", "쒰" },
  { "쒱", "쒱", "쒱", "쒱", "쒱" },
  { "쒲", "쒲", "쒲", "쒲", "쒲" },
  { "쒳", "쒳", "쒳", "쒳", "쒳" },
  { "쒴", "쒴", "쒴", "쒴", "쒴" },
  { "쒵", "쒵", "쒵", "쒵", "쒵" },
  { "쒶", "쒶", "쒶", "쒶", "쒶" },
  { "쒷", "쒷", "쒷", "쒷", "쒷" },
  { "쒸", "쒸", "쒸", "쒸", "쒸" },
  { "쒹", "쒹", "쒹", "쒹", "쒹" },
  { "쒺", "쒺", "쒺", "쒺", "쒺" },
  { "쒻", "쒻", "쒻", "쒻", "쒻" },
  { "쒼", "쒼", "쒼", "쒼", "쒼" },
  { "쒽", "쒽", "쒽", "쒽", "쒽" },
  { "쒾", "쒾", "쒾", "쒾", "쒾" },
  { "쒿", "쒿", "쒿", "쒿", "쒿" },
  { "쓀", "쓀", "쓀", "쓀", "쓀" },
  { "쓁", "쓁", "쓁", "쓁", "쓁" },
  { "쓂", "쓂", "쓂", "쓂", "쓂" },
  { "쓃", "쓃", "쓃", "쓃", "쓃" },
  { "쓄", "쓄", "쓄", "쓄", "쓄" },
  { "쓅", "쓅", "쓅", "쓅", "쓅" },
  { "쓆", "쓆", "쓆", "쓆", "쓆" },
  { "쓇", "쓇", "쓇", "쓇", "쓇" },
  { "쓈", "쓈", "쓈", "쓈", "쓈" },
  { "쓉", "쓉", "쓉", "쓉", "쓉" },
  { "쓊", "쓊", "쓊", "쓊", "쓊" },
  { "쓋", "쓋", "쓋", "쓋", "쓋" },
  { "쓌", "쓌", "쓌", "쓌", "쓌" },
  { "쓍", "쓍", "쓍", "쓍", "쓍" },
  { "쓎", "쓎", "쓎", "쓎", "쓎" },
  { "쓏", "쓏", "쓏", "쓏", "쓏" },
  { "쓐", "쓐", "쓐", "쓐", "쓐" },
  { "쓑", "쓑", "쓑", "쓑", "쓑" },
  { "쓒", "쓒", "쓒", "쓒", "쓒" },
  { "쓓", "쓓", "쓓", "쓓", "쓓" },
  { "쓔", "쓔", "쓔", "쓔", "쓔" },
  { "쓕", "쓕", "쓕", "쓕", "쓕" },
  { "쓖", "쓖", "쓖", "쓖", "쓖" },
  { "쓗", "쓗", "쓗", "쓗", "쓗" },
  { "쓘", "쓘", "쓘", "쓘", "쓘" },
  { "쓙", "쓙", "쓙", "쓙", "쓙" },
  { "쓚", "쓚", "쓚", "쓚", "쓚" },
  { "쓛", "쓛", "쓛", "쓛", "쓛" },
  { "쓜", "쓜", "쓜", "쓜", "쓜" },
  { "쓝", "쓝", "쓝", "쓝", "쓝" },
  { "쓞", "쓞", "쓞", "쓞", "쓞" },
  { "쓟", "쓟", "쓟", "쓟", "쓟" },
  { "쓠", "쓠", "쓠", "쓠", "쓠" },
  { "쓡", "쓡", "쓡", "쓡", "쓡" },
  { "쓢", "쓢", "쓢", "쓢", "쓢" },
  { "쓣", "쓣", "쓣", "쓣", "쓣" },
  { "쓤", "쓤", "쓤", "쓤", "쓤" },
  { "쓥", "쓥", "쓥", "쓥", "쓥" },
  { "쓦", "쓦", "쓦", "쓦", "쓦" },
  { "쓧", "쓧", "쓧", "쓧", "쓧" },
  { "쓨", "쓨", "쓨", "쓨", "쓨" },
  { "쓩", "쓩", "쓩", "쓩", "쓩" },
  { "쓪", "쓪", "쓪", "쓪", "쓪" },
  { "쓫", "쓫", "쓫", "쓫", "쓫" },
  { "쓬", "쓬", "쓬", "쓬", "쓬" },
  { "쓭", "쓭", "쓭", "쓭", "쓭" },
  { "쓮", "쓮", "쓮", "쓮", "쓮" },
  { "쓯", "쓯", "쓯", "쓯", "쓯" },
  { "쓰", "쓰", "쓰", "쓰", "쓰" },
  { "쓱", "쓱", "쓱", "쓱", "쓱" },
  { "쓲", "쓲", "쓲", "쓲", "쓲" },
  { "쓳", "쓳", "쓳", "쓳", "쓳" },
  { "쓴", "쓴", "쓴", "쓴", "쓴" },
  { "쓵", "쓵", "쓵", "쓵", "쓵" },
  { "쓶", "쓶", "쓶", "쓶", "쓶" },
  { "쓷", "쓷", "쓷", "쓷", "쓷" },
  { "쓸", "쓸", "쓸", "쓸", "쓸" },
  { "쓹", "쓹", "쓹", "쓹", "쓹" },
  { "쓺", "쓺", "쓺", "쓺", "쓺" },
  { "쓻", "쓻", "쓻", "쓻", "쓻" },
  { "쓼", "쓼", "쓼", "쓼", "쓼" },
  { "쓽", "쓽", "쓽", "쓽", "쓽" },
  { "쓾", "쓾", "쓾", "쓾", "쓾" },
  { "쓿", "쓿", "쓿", "쓿", "쓿" },
  { "씀", "씀", "씀", "씀", "씀" },
  { "씁", "씁", "씁", "씁", "씁" },
  { "씂", "씂", "씂", "씂", "씂" },
  { "씃", "씃", "씃", "씃", "씃" },
  { "씄", "씄", "씄", "씄", "씄" },
  { "씅", "씅", "씅", "씅", "씅" },
  { "씆", "씆", "씆", "씆", "씆" },
  { "씇", "씇", "씇", "씇", "씇" },
  { "씈", "씈", "씈", "씈", "씈" },
  { "씉", "씉", "씉", "씉", "씉" },
  { "씊", "씊", "씊", "씊", "씊" },
  { "씋", "씋", "씋", "씋", "씋" },
  { "씌", "씌", "씌", "씌", "씌" },
  { "씍", "씍", "씍", "씍", "씍" },
  { "씎", "씎", "씎", "씎", "씎" },
  { "씏", "씏", "씏", "씏", "씏" },
  { "씐", "씐", "씐", "씐", "씐" },
  { "씑", "씑", "씑", "씑", "씑" },
  { "씒", "씒", "씒", "씒", "씒" },
  { "씓", "씓", "씓", "씓", "씓" },
  { "씔", "씔", "씔", "씔", "씔" },
  { "씕", "씕", "씕", "씕", "씕" },
  { "씖", "씖", "씖", "씖", "씖" },
  { "씗", "씗", "씗", "씗", "씗" },
  { "씘", "씘", "씘", "씘", "씘" },
  { "씙", "씙", "씙", "씙", "씙" },
  { "씚", "씚", "씚", "씚", "씚" },
  { "씛", "씛", "씛", "씛", "씛" },
  { "씜", "씜", "씜", "씜", "씜" },
  { "씝", "씝", "씝", "씝", "씝" },
  { "씞", "씞", "씞", "씞", "씞" },
  { "씟", "씟", "씟", "씟", "씟" },
  { "씠", "씠", "씠", "씠", "씠" },
  { "씡", "씡", "씡", "씡", "씡" },
  { "씢", "씢", "씢", "씢", "씢" },
  { "씣", "씣", "씣", "씣", "씣" },
  { "씤", "씤", "씤", "씤", "씤" },
  { "씥", "씥", "씥", "씥", "씥" },
  { "씦", "씦", "씦", "씦", "씦" },
  { "씧", "씧", "씧", "씧", "씧" },
  { "씨", "씨", "씨", "씨", "씨" },
  { "씩", "씩", "씩", "씩", "씩" },
  { "씪", "씪", "씪", "씪", "씪" },
  { "씫", "씫", "씫", "씫", "씫" },
  { "씬", "씬", "씬", "씬", "씬" },
  { "씭", "씭", "씭", "씭", "씭" },
  { "씮", "씮", "씮", "씮", "씮" },
  { "씯", "씯", "씯", "씯", "씯" },
  { "씰", "씰", "씰", "씰", "씰" },
  { "씱", "씱", "씱", "씱", "씱" },
  { "씲", "씲", "씲", "씲", "씲" },
  { "씳", "씳", "씳", "씳", "씳" },
  { "씴", "씴", "씴", "씴", "씴" },
  { "씵", "씵", "씵", "씵", "씵" },
  { "씶", "씶", "씶", "씶", "씶" },
  { "씷", "씷", "씷", "씷", "씷" },
  { "씸", "씸", "씸", "씸", "씸" },
  { "씹", "씹", "씹", "씹", "씹" },
  { "씺", "씺", "씺", "씺", "씺" },
  { "씻", "씻", "씻", "씻", "씻" },
  { "씼", "씼", "씼", "씼", "씼" },
  { "씽", "씽", "씽", "씽", "씽" },
  { "씾", "씾", "씾", "씾", "씾" },
  { "씿", "씿", "씿", "씿", "씿" },
  { "앀", "앀", "앀", "앀", "앀" },
  { "앁", "앁", "앁", "앁", "앁" },
  { "앂", "앂", "앂", "앂", "앂" },
  { "앃", "앃", "앃", "앃", "앃" },
  { "아", "아", "아", "아", "아" },
  { "악", "악", "악", "악", "악" },
  { "앆", "앆", "앆", "앆", "앆" },
  { "앇", "앇", "앇", "앇", "앇" },
  { "안", "안", "안", "안", "안" },
  { "앉", "앉", "앉", "앉", "앉" },
  { "않", "않", "않", "않", "않" },
  { "앋", "앋", "앋", "앋", "앋" },
  { "알", "알", "알", "알", "알" },
  { "앍", "앍", "앍", "앍", "앍" },
  { "앎", "앎", "앎", "앎", "앎" },
  { "앏", "앏", "앏", "앏", "앏" },
  { "앐", "앐", "앐", "앐", "앐" },
  { "앑", "앑", "앑", "앑", "앑" },
  { "앒", "앒", "앒", "앒", "앒" },
  { "앓", "앓", "앓", "앓", "앓" },
  { "암", "암", "암", "암", "암" },
  { "압", "압", "압", "압", "압" },
  { "앖", "앖", "앖", "앖", "앖" },
  { "앗", "앗", "앗", "앗", "앗" },
  { "았", "았", "았", "았", "았" },
  { "앙", "앙", "앙", "앙", "앙" },
  { "앚", "앚", "앚", "앚", "앚" },
  { "앛", "앛", "앛", "앛", "앛" },
  { "앜", "앜", "앜", "앜", "앜" },
  { "앝", "앝", "앝", "앝", "앝" },
  { "앞", "앞", "앞", "앞", "앞" },
  { "앟", "앟", "앟", "앟", "앟" },
  { "애", "애", "애", "애", "애" },
  { "액", "액", "액", "액", "액" },
  { "앢", "앢", "앢", "앢", "앢" },
  { "앣", "앣", "앣", "앣", "앣" },
  { "앤", "앤", "앤", "앤", "앤" },
  { "앥", "앥", "앥", "앥", "앥" },
  { "앦", "앦", "앦", "앦", "앦" },
  { "앧", "앧", "앧", "앧", "앧" },
  { "앨", "앨", "앨", "앨", "앨" },
  { "앩", "앩", "앩", "앩", "앩" },
  { "앪", "앪", "앪", "앪", "앪" },
  { "앫", "앫", "앫", "앫", "앫" },
  { "앬", "앬", "앬", "앬", "앬" },
  { "앭", "앭", "앭", "앭", "앭" },
  { "앮", "앮", "앮", "앮", "앮" },
  { "앯", "앯", "앯", "앯", "앯" },
  { "앰", "앰", "앰", "앰", "앰" },
  { "앱", "앱", "앱", "앱", "앱" },
  { "앲", "앲", "앲", "앲", "앲" },
  { "앳", "앳", "앳", "앳", "앳" },
  { "앴", "앴", "앴", "앴", "앴" },
  { "앵", "앵", "앵", "앵", "앵" },
  { "앶", "앶", "앶", "앶", "앶" },
  { "앷", "앷", "앷", "앷", "앷" },
  { "앸", "앸", "앸", "앸", "앸" },
  { "앹", "앹", "앹", "앹", "앹" },
  { "앺", "앺", "앺", "앺", "앺" },
  { "앻", "앻", "앻", "앻", "앻" },
  { "야", "야", "야", "야", "야" },
  { "약", "약", "약", "약", "약" },
  { "앾", "앾", "앾", "앾", "앾" },
  { "앿", "앿", "앿", "앿", "앿" },
  { "얀", "얀", "얀", "얀", "얀" },
  { "얁", "얁", "얁", "얁", "얁" },
  { "얂", "얂", "얂", "얂", "얂" },
  { "얃", "얃", "얃", "얃", "얃" },
  { "얄", "얄", "얄", "얄", "얄" },
  { "얅", "얅", "얅", "얅", "얅" },
  { "얆", "얆", "얆", "얆", "얆" },
  { "얇", "얇", "얇", "얇", "얇" },
  { "얈", "얈", "얈", "얈", "얈" },
  { "얉", "얉", "얉", "얉", "얉" },
  { "얊", "얊", "얊", "얊", "얊" },
  { "얋", "얋", "얋", "얋", "얋" },
  { "얌", "얌", "얌", "얌", "얌" },
  { "얍", "얍", "얍", "얍", "얍" },
  { "얎", "얎", "얎", "얎", "얎" },
  { "얏", "얏", "얏", "얏", "얏" },
  { "얐", "얐", "얐", "얐", "얐" },
  { "양", "양", "양", "양", "양" },
  { "얒", "얒", "얒", "얒", "얒" },
  { "얓", "얓", "얓", "얓", "얓" },
  { "얔", "얔", "얔", "얔", "얔" },
  { "얕", "얕", "얕", "얕", "얕" },
  { "얖", "얖", "얖", "얖", "얖" },
  { "얗", "얗", "얗", "얗", "얗" },
  { "얘", "얘", "얘", "얘", "얘" },
  { "얙", "얙", "얙", "얙", "얙" },
  { "얚", "얚", "얚", "얚", "얚" },
  { "얛", "얛", "얛", "얛", "얛" },
  { "얜", "얜", "얜", "얜", "얜" },
  { "얝", "얝", "얝", "얝", "얝" },
  { "얞", "얞", "얞", "얞", "얞" },
  { "얟", "얟", "얟", "얟", "얟" },
  { "얠", "얠", "얠", "얠", "얠" },
  { "얡", "얡", "얡", "얡", "얡" },
  { "얢", "얢", "얢", "얢", "얢" },
  { "얣", "얣", "얣", "얣", "얣" },
  { "얤", "얤", "얤", "얤", "얤" },
  { "얥", "얥", "얥", "얥", "얥" },
  { "얦", "얦", "얦", "얦", "얦" },
  { "얧", "얧", "얧", "얧", "얧" },
  { "얨", "얨", "얨", "얨", "얨" },
  { "얩", "얩", "얩", "얩", "얩" },
  { "얪", "얪", "얪", "얪", "얪" },
  { "얫", "얫", "얫", "얫", "얫" },
  { "얬", "얬", "얬", "얬", "얬" },
  { "얭", "얭", "얭", "얭", "얭" },
  { "얮", "얮", "얮", "얮", "얮" },
  { "얯", "얯", "얯", "얯", "얯" },
  { "얰", "얰", "얰", "얰", "얰" },
  { "얱", "얱", "얱", "얱", "얱" },
  { "얲", "얲", "얲", "얲", "얲" },
  { "얳", "얳", "얳", "얳", "얳" },
  { "어", "어", "어", "어", "어" },
  { "억", "억", "억", "억", "억" },
  { "얶", "얶", "얶", "얶", "얶" },
  { "얷", "얷", "얷", "얷", "얷" },
  { "언", "언", "언", "언", "언" },
  { "얹", "얹", "얹", "얹", "얹" },
  { "얺", "얺", "얺", "얺", "얺" },
  { "얻", "얻", "얻", "얻", "얻" },
  { "얼", "얼", "얼", "얼", "얼" },
  { "얽", "얽", "얽", "얽", "얽" },
  { "얾", "얾", "얾", "얾", "얾" },
  { "얿", "얿", "얿", "얿", "얿" },
  { "엀", "엀", "엀", "엀", "엀" },
  { "엁", "엁", "엁", "엁", "엁" },
  { "엂", "엂", "엂", "엂", "엂" },
  { "엃", "엃", "엃", "엃", "엃" },
  { "엄", "엄", "엄", "엄", "엄" },
  { "업", "업", "업", "업", "업" },
  { "없", "없", "없", "없", "없" },
  { "엇", "엇", "엇", "엇", "엇" },
  { "었", "었", "었", "었", "었" },
  { "엉", "엉", "엉", "엉", "엉" },
  { "엊", "엊", "엊", "엊", "엊" },
  { "엋", "엋", "엋", "엋", "엋" },
  { "엌", "엌", "엌", "엌", "엌" },
  { "엍", "엍", "엍", "엍", "엍" },
  { "엎", "엎", "엎", "엎", "엎" },
  { "엏", "엏", "엏", "엏", "엏" },
  { "에", "에", "에", "에", "에" },
  { "엑", "엑", "엑", "엑", "엑" },
  { "엒", "엒", "엒", "엒", "엒" },
  { "엓", "엓", "엓", "엓", "엓" },
  { "엔", "엔", "엔", "엔", "엔" },
  { "엕", "엕", "엕", "엕", "엕" },
  { "엖", "엖", "엖", "엖", "엖" },
  { "엗", "엗", "엗", "엗", "엗" },
  { "엘", "엘", "엘", "엘", "엘" },
  { "엙", "엙", "엙", "엙", "엙" },
  { "엚", "엚", "엚", "엚", "엚" },
  { "엛", "엛", "엛", "엛", "엛" },
  { "엜", "엜", "엜", "엜", "엜" },
  { "엝", "엝", "엝", "엝", "엝" },
  { "엞", "엞", "엞", "엞", "엞" },
  { "엟", "엟", "엟", "엟", "엟" },
  { "엠", "엠", "엠", "엠", "엠" },
  { "엡", "엡", "엡", "엡", "엡" },
  { "엢", "엢", "엢", "엢", "엢" },
  { "엣", "엣", "엣", "엣", "엣" },
  { "엤", "엤", "엤", "엤", "엤" },
  { "엥", "엥", "엥", "엥", "엥" },
  { "엦", "엦", "엦", "엦", "엦" },
  { "엧", "엧", "엧", "엧", "엧" },
  { "엨", "엨", "엨", "엨", "엨" },
  { "엩", "엩", "엩", "엩", "엩" },
  { "엪", "엪", "엪", "엪", "엪" },
  { "엫", "엫", "엫", "엫", "엫" },
  { "여", "여", "여", "여", "여" },
  { "역", "역", "역", "역", "역" },
  { "엮", "엮", "엮", "엮", "엮" },
  { "엯", "엯", "엯", "엯", "엯" },
  { "연", "연", "연", "연", "연" },
  { "엱", "엱", "엱", "엱", "엱" },
  { "엲", "엲", "엲", "엲", "엲" },
  { "엳", "엳", "엳", "엳", "엳" },
  { "열", "열", "열", "열", "열" },
  { "엵", "엵", "엵", "엵", "엵" },
  { "엶", "엶", "엶", "엶", "엶" },
  { "엷", "엷", "엷", "엷", "엷" },
  { "엸", "엸", "엸", "엸", "엸" },
  { "엹", "엹", "엹", "엹", "엹" },
  { "엺", "엺", "엺", "엺", "엺" },
  { "엻", "엻", "엻", "엻", "엻" },
  { "염", "염", "염", "염", "염" },
  { "엽", "엽", "엽", "엽", "엽" },
  { "엾", "엾", "엾", "엾", "엾" },
  { "엿", "엿", "엿", "엿", "엿" },
  { "였", "였", "였", "였", "였" },
  { "영", "영", "영", "영", "영" },
  { "옂", "옂", "옂", "옂", "옂" },
  { "옃", "옃", "옃", "옃", "옃" },
  { "옄", "옄", "옄", "옄", "옄" },
  { "옅", "옅", "옅", "옅", "옅" },
  { "옆", "옆", "옆", "옆", "옆" },
  { "옇", "옇", "옇", "옇", "옇" },
  { "예", "예", "예", "예", "예" },
  { "옉", "옉", "옉", "옉", "옉" },
  { "옊", "옊", "옊", "옊", "옊" },
  { "옋", "옋", "옋", "옋", "옋" },
  { "옌", "옌", "옌", "옌", "옌" },
  { "옍", "옍", "옍", "옍", "옍" },
  { "옎", "옎", "옎", "옎", "옎" },
  { "옏", "옏", "옏", "옏", "옏" },
  { "옐", "옐", "옐", "옐", "옐" },
  { "옑", "옑", "옑", "옑", "옑" },
  { "옒", "옒", "옒", "옒", "옒" },
  { "옓", "옓", "옓", "옓", "옓" },
  { "옔", "옔", "옔", "옔", "옔" },
  { "옕", "옕", "옕", "옕", "옕" },
  { "옖", "옖", "옖", "옖", "옖" },
  { "옗", "옗", "옗", "옗", "옗" },
  { "옘", "옘", "옘", "옘", "옘" },
  { "옙", "옙", "옙", "옙", "옙" },
  { "옚", "옚", "옚", "옚", "옚" },
  { "옛", "옛", "옛", "옛", "옛" },
  { "옜", "옜", "옜", "옜", "옜" },
  { "옝", "옝", "옝", "옝", "옝" },
  { "옞", "옞", "옞", "옞", "옞" },
  { "옟", "옟", "옟", "옟", "옟" },
  { "옠", "옠", "옠", "옠", "옠" },
  { "옡", "옡", "옡", "옡", "옡" },
  { "옢", "옢", "옢", "옢", "옢" },
  { "옣", "옣", "옣", "옣", "옣" },
  { "오", "오", "오", "오", "오" },
  { "옥", "옥", "옥", "옥", "옥" },
  { "옦", "옦", "옦", "옦", "옦" },
  { "옧", "옧", "옧", "옧", "옧" },
  { "온", "온", "온", "온", "온" },
  { "옩", "옩", "옩", "옩", "옩" },
  { "옪", "옪", "옪", "옪", "옪" },
  { "옫", "옫", "옫", "옫", "옫" },
  { "올", "올", "올", "올", "올" },
  { "옭", "옭", "옭", "옭", "옭" },
  { "옮", "옮", "옮", "옮", "옮" },
  { "옯", "옯", "옯", "옯", "옯" },
  { "옰", "옰", "옰", "옰", "옰" },
  { "옱", "옱", "옱", "옱", "옱" },
  { "옲", "옲", "옲", "옲", "옲" },
  { "옳", "옳", "옳", "옳", "옳" },
  { "옴", "옴", "옴", "옴", "옴" },
  { "옵", "옵", "옵", "옵", "옵" },
  { "옶", "옶", "옶", "옶", "옶" },
  { "옷", "옷", "옷", "옷", "옷" },
  { "옸", "옸", "옸", "옸", "옸" },
  { "옹", "옹", "옹", "옹", "옹" },
  { "옺", "옺", "옺", "옺", "옺" },
  { "옻", "옻", "옻", "옻", "옻" },
  { "옼", "옼", "옼", "옼", "옼" },
  { "옽", "옽", "옽", "옽", "옽" },
  { "옾", "옾", "옾", "옾", "옾" },
  { "옿", "옿", "옿", "옿", "옿" },
  { "와", "와", "와", "와", "와" },
  { "왁", "왁", "왁", "왁", "왁" },
  { "왂", "왂", "왂", "왂", "왂" },
  { "왃", "왃", "왃", "왃", "왃" },
  { "완", "완", "완", "완", "완" },
  { "왅", "왅", "왅", "왅", "왅" },
  { "왆", "왆", "왆", "왆", "왆" },
  { "왇", "왇", "왇", "왇", "왇" },
  { "왈", "왈", "왈", "왈", "왈" },
  { "왉", "왉", "왉", "왉", "왉" },
  { "왊", "왊", "왊", "왊", "왊" },
  { "왋", "왋", "왋", "왋", "왋" },
  { "왌", "왌", "왌", "왌", "왌" },
  { "왍", "왍", "왍", "왍", "왍" },
  { "왎", "왎", "왎", "왎", "왎" },
  { "왏", "왏", "왏", "왏", "왏" },
  { "왐", "왐", "왐", "왐", "왐" },
  { "왑", "왑", "왑", "왑", "왑" },
  { "왒", "왒", "왒", "왒", "왒" },
  { "왓", "왓", "왓", "왓", "왓" },
  { "왔", "왔", "왔", "왔", "왔" },
  { "왕", "왕", "왕", "왕", "왕" },
  { "왖", "왖", "왖", "왖", "왖" },
  { "왗", "왗", "왗", "왗", "왗" },
  { "왘", "왘", "왘", "왘", "왘" },
  { "왙", "왙", "왙", "왙", "왙" },
  { "왚", "왚", "왚", "왚", "왚" },
  { "왛", "왛", "왛", "왛", "왛" },
  { "왜", "왜", "왜", "왜", "왜" },
  { "왝", "왝", "왝", "왝", "왝" },
  { "왞", "왞", "왞", "왞", "왞" },
  { "왟", "왟", "왟", "왟", "왟" },
  { "왠", "왠", "왠", "왠", "왠" },
  { "왡", "왡", "왡", "왡", "왡" },
  { "왢", "왢", "왢", "왢", "왢" },
  { "왣", "왣", "왣", "왣", "왣" },
  { "왤", "왤", "왤", "왤", "왤" },
  { "왥", "왥", "왥", "왥", "왥" },
  { "왦", "왦", "왦", "왦", "왦" },
  { "왧", "왧", "왧", "왧", "왧" },
  { "왨", "왨", "왨", "왨", "왨" },
  { "왩", "왩", "왩", "왩", "왩" },
  { "왪", "왪", "왪", "왪", "왪" },
  { "왫", "왫", "왫", "왫", "왫" },
  { "왬", "왬", "왬", "왬", "왬" },
  { "왭", "왭", "왭", "왭", "왭" },
  { "왮", "왮", "왮", "왮", "왮" },
  { "왯", "왯", "왯", "왯", "왯" },
  { "왰", "왰", "왰", "왰", "왰" },
  { "왱", "왱", "왱", "왱", "왱" },
  { "왲", "왲", "왲", "왲", "왲" },
  { "왳", "왳", "왳", "왳", "왳" },
  { "왴", "왴", "왴", "왴", "왴" },
  { "왵", "왵", "왵", "왵", "왵" },
  { "왶", "왶", "왶", "왶", "왶" },
  { "왷", "왷", "왷", "왷", "왷" },
  { "외", "외", "외", "외", "외" },
  { "왹", "왹", "왹", "왹", "왹" },
  { "왺", "왺", "왺", "왺", "왺" },
  { "왻", "왻", "왻", "왻", "왻" },
  { "왼", "왼", "왼", "왼", "왼" },
  { "왽", "왽", "왽", "왽", "왽" },
  { "왾", "왾", "왾", "왾", "왾" },
  { "왿", "왿", "왿", "왿", "왿" },
  { "욀", "욀", "욀", "욀", "욀" },
  { "욁", "욁", "욁", "욁", "욁" },
  { "욂", "욂", "욂", "욂", "욂" },
  { "욃", "욃", "욃", "욃", "욃" },
  { "욄", "욄", "욄", "욄", "욄" },
  { "욅", "욅", "욅", "욅", "욅" },
  { "욆", "욆", "욆", "욆", "욆" },
  { "욇", "욇", "욇", "욇", "욇" },
  { "욈", "욈", "욈", "욈", "욈" },
  { "욉", "욉", "욉", "욉", "욉" },
  { "욊", "욊", "욊", "욊", "욊" },
  { "욋", "욋", "욋", "욋", "욋" },
  { "욌", "욌", "욌", "욌", "욌" },
  { "욍", "욍", "욍", "욍", "욍" },
  { "욎", "욎", "욎", "욎", "욎" },
  { "욏", "욏", "욏", "욏", "욏" },
  { "욐", "욐", "욐", "욐", "욐" },
  { "욑", "욑", "욑", "욑", "욑" },
  { "욒", "욒", "욒", "욒", "욒" },
  { "욓", "욓", "욓", "욓", "욓" },
  { "요", "요", "요", "요", "요" },
  { "욕", "욕", "욕", "욕", "욕" },
  { "욖", "욖", "욖", "욖", "욖" },
  { "욗", "욗", "욗", "욗", "욗" },
  { "욘", "욘", "욘", "욘", "욘" },
  { "욙", "욙", "욙", "욙", "욙" },
  { "욚", "욚", "욚", "욚", "욚" },
  { "욛", "욛", "욛", "욛", "욛" },
  { "욜", "욜", "욜", "욜", "욜" },
  { "욝", "욝", "욝", "욝", "욝" },
  { "욞", "욞", "욞", "욞", "욞" },
  { "욟", "욟", "욟", "욟", "욟" },
  { "욠", "욠", "욠", "욠", "욠" },
  { "욡", "욡", "욡", "욡", "욡" },
  { "욢", "욢", "욢", "욢", "욢" },
  { "욣", "욣", "욣", "욣", "욣" },
  { "욤", "욤", "욤", "욤", "욤" },
  { "욥", "욥", "욥", "욥", "욥" },
  { "욦", "욦", "욦", "욦", "욦" },
  { "욧", "욧", "욧", "욧", "욧" },
  { "욨", "욨", "욨", "욨", "욨" },
  { "용", "용", "용", "용", "용" },
  { "욪", "욪", "욪", "욪", "욪" },
  { "욫", "욫", "욫", "욫", "욫" },
  { "욬", "욬", "욬", "욬", "욬" },
  { "욭", "욭", "욭", "욭", "욭" },
  { "욮", "욮", "욮", "욮", "욮" },
  { "욯", "욯", "욯", "욯", "욯" },
  { "우", "우", "우", "우", "우" },
  { "욱", "욱", "욱", "욱", "욱" },
  { "욲", "욲", "욲", "욲", "욲" },
  { "욳", "욳", "욳", "욳", "욳" },
  { "운", "운", "운", "운", "운" },
  { "욵", "욵", "욵", "욵", "욵" },
  { "욶", "욶", "욶", "욶", "욶" },
  { "욷", "욷", "욷", "욷", "욷" },
  { "울", "울", "울", "울", "울" },
  { "욹", "욹", "욹", "욹", "욹" },
  { "욺", "욺", "욺", "욺", "욺" },
  { "욻", "욻", "욻", "욻", "욻" },
  { "욼", "욼", "욼", "욼", "욼" },
  { "욽", "욽", "욽", "욽", "욽" },
  { "욾", "욾", "욾", "욾", "욾" },
  { "욿", "욿", "욿", "욿", "욿" },
  { "움", "움", "움", "움", "움" },
  { "웁", "웁", "웁", "웁", "웁" },
  { "웂", "웂", "웂", "웂", "웂" },
  { "웃", "웃", "웃", "웃", "웃" },
  { "웄", "웄", "웄", "웄", "웄" },
  { "웅", "웅", "웅", "웅", "웅" },
  { "웆", "웆", "웆", "웆", "웆" },
  { "웇", "웇", "웇", "웇", "웇" },
  { "웈", "웈", "웈", "웈", "웈" },
  { "웉", "웉", "웉", "웉", "웉" },
  { "웊", "웊", "웊", "웊", "웊" },
  { "웋", "웋", "웋", "웋", "웋" },
  { "워", "워", "워", "워", "워" },
  { "웍", "웍", "웍", "웍", "웍" },
  { "웎", "웎", "웎", "웎", "웎" },
  { "웏", "웏", "웏", "웏", "웏" },
  { "원", "원", "원", "원", "원" },
  { "웑", "웑", "웑", "웑", "웑" },
  { "웒", "웒", "웒", "웒", "웒" },
  { "웓", "웓", "웓", "웓", "웓" },
  { "월", "월", "월", "월", "월" },
  { "웕", "웕", "웕", "웕", "웕" },
  { "웖", "웖", "웖", "웖", "웖" },
  { "웗", "웗", "웗", "웗", "웗" },
  { "웘", "웘", "웘", "웘", "웘" },
  { "웙", "웙", "웙", "웙", "웙" },
  { "웚", "웚", "웚", "웚", "웚" },
  { "웛", "웛", "웛", "웛", "웛" },
  { "웜", "웜", "웜", "웜", "웜" },
  { "웝", "웝", "웝", "웝", "웝" },
  { "웞", "웞", "웞", "웞", "웞" },
  { "웟", "웟", "웟", "웟", "웟" },
  { "웠", "웠", "웠", "웠", "웠" },
  { "웡", "웡", "웡", "웡", "웡" },
  { "웢", "웢", "웢", "웢", "웢" },
  { "웣", "웣", "웣", "웣", "웣" },
  { "웤", "웤", "웤", "웤", "웤" },
  { "웥", "웥", "웥", "웥", "웥" },
  { "웦", "웦", "웦", "웦", "웦" },
  { "웧", "웧", "웧", "웧", "웧" },
  { "웨", "웨", "웨", "웨", "웨" },
  { "웩", "웩", "웩", "웩", "웩" },
  { "웪", "웪", "웪", "웪", "웪" },
  { "웫", "웫", "웫", "웫", "웫" },
  { "웬", "웬", "웬", "웬", "웬" },
  { "웭", "웭", "웭", "웭", "웭" },
  { "웮", "웮", "웮", "웮", "웮" },
  { "웯", "웯", "웯", "웯", "웯" },
  { "웰", "웰", "웰", "웰", "웰" },
  { "웱", "웱", "웱", "웱", "웱" },
  { "웲", "웲", "웲", "웲", "웲" },
  { "웳", "웳", "웳", "웳", "웳" },
  { "웴", "웴", "웴", "웴", "웴" },
  { "웵", "웵", "웵", "웵", "웵" },
  { "웶", "웶", "웶", "웶", "웶" },
  { "웷", "웷", "웷", "웷", "웷" },
  { "웸", "웸", "웸", "웸", "웸" },
  { "웹", "웹", "웹", "웹", "웹" },
  { "웺", "웺", "웺", "웺", "웺" },
  { "웻", "웻", "웻", "웻", "웻" },
  { "웼", "웼", "웼", "웼", "웼" },
  { "웽", "웽", "웽", "웽", "웽" },
  { "웾", "웾", "웾", "웾", "웾" },
  { "웿", "웿", "웿", "웿", "웿" },
  { "윀", "윀", "윀", "윀", "윀" },
  { "윁", "윁", "윁", "윁", "윁" },
  { "윂", "윂", "윂", "윂", "윂" },
  { "윃", "윃", "윃", "윃", "윃" },
  { "위", "위", "위", "위", "위" },
  { "윅", "윅", "윅", "윅", "윅" },
  { "윆", "윆", "윆", "윆", "윆" },
  { "윇", "윇", "윇", "윇", "윇" },
  { "윈", "윈", "윈", "윈", "윈" },
  { "윉", "윉", "윉", "윉", "윉" },
  { "윊", "윊", "윊", "윊", "윊" },
  { "윋", "윋", "윋", "윋", "윋" },
  { "윌", "윌", "윌", "윌", "윌" },
  { "윍", "윍", "윍", "윍", "윍" },
  { "윎", "윎", "윎", "윎", "윎" },
  { "윏", "윏", "윏", "윏", "윏" },
  { "윐", "윐", "윐", "윐", "윐" },
  { "윑", "윑", "윑", "윑", "윑" },
  { "윒", "윒", "윒", "윒", "윒" },
  { "윓", "윓", "윓", "윓", "윓" },
  { "윔", "윔", "윔", "윔", "윔" },
  { "윕", "윕", "윕", "윕", "윕" },
  { "윖", "윖", "윖", "윖", "윖" },
  { "윗", "윗", "윗", "윗", "윗" },
  { "윘", "윘", "윘", "윘", "윘" },
  { "윙", "윙", "윙", "윙", "윙" },
  { "윚", "윚", "윚", "윚", "윚" },
  { "윛", "윛", "윛", "윛", "윛" },
  { "윜", "윜", "윜", "윜", "윜" },
  { "윝", "윝", "윝", "윝", "윝" },
  { "윞", "윞", "윞", "윞", "윞" },
  { "윟", "윟", "윟", "윟", "윟" },
  { "유", "유", "유", "유", "유" },
  { "육", "육", "육", "육", "육" },
  { "윢", "윢", "윢", "윢", "윢" },
  { "윣", "윣", "윣", "윣", "윣" },
  { "윤", "윤", "윤", "윤", "윤" },
  { "윥", "윥", "윥", "윥", "윥" },
  { "윦", "윦", "윦", "윦", "윦" },
  { "윧", "윧", "윧", "윧", "윧" },
  { "율", "율", "율", "율", "율" },
  { "윩", "윩", "윩", "윩", "윩" },
  { "윪", "윪", "윪", "윪", "윪" },
  { "윫", "윫", "윫", "윫", "윫" },
  { "윬", "윬", "윬", "윬", "윬" },
  { "윭", "윭", "윭", "윭", "윭" },
  { "윮", "윮", "윮", "윮", "윮" },
  { "윯", "윯", "윯", "윯", "윯" },
  { "윰", "윰", "윰", "윰", "윰" },
  { "윱", "윱", "윱", "윱", "윱" },
  { "윲", "윲", "윲", "윲", "윲" },
  { "윳", "윳", "윳", "윳", "윳" },
  { "윴", "윴", "윴", "윴", "윴" },
  { "융", "융", "융", "융", "융" },
  { "윶", "윶", "윶", "윶", "윶" },
  { "윷", "윷", "윷", "윷", "윷" },
  { "윸", "윸", "윸", "윸", "윸" },
  { "윹", "윹", "윹", "윹", "윹" },
  { "윺", "윺", "윺", "윺", "윺" },
  { "윻", "윻", "윻", "윻", "윻" },
  { "으", "으", "으", "으", "으" },
  { "윽", "윽", "윽", "윽", "윽" },
  { "윾", "윾", "윾", "윾", "윾" },
  { "윿", "윿", "윿", "윿", "윿" },
  { "은", "은", "은", "은", "은" },
  { "읁", "읁", "읁", "읁", "읁" },
  { "읂", "읂", "읂", "읂", "읂" },
  { "읃", "읃", "읃", "읃", "읃" },
  { "을", "을", "을", "을", "을" },
  { "읅", "읅", "읅", "읅", "읅" },
  { "읆", "읆", "읆", "읆", "읆" },
  { "읇", "읇", "읇", "읇", "읇" },
  { "읈", "읈", "읈", "읈", "읈" },
  { "읉", "읉", "읉", "읉", "읉" },
  { "읊", "읊", "읊", "읊", "읊" },
  { "읋", "읋", "읋", "읋", "읋" },
  { "음", "음", "음", "음", "음" },
  { "읍", "읍", "읍", "읍", "읍" },
  { "읎", "읎", "읎", "읎", "읎" },
  { "읏", "읏", "읏", "읏", "읏" },
  { "읐", "읐", "읐", "읐", "읐" },
  { "응", "응", "응", "응", "응" },
  { "읒", "읒", "읒", "읒", "읒" },
  { "읓", "읓", "읓", "읓", "읓" },
  { "읔", "읔", "읔", "읔", "읔" },
  { "읕", "읕", "읕", "읕", "읕" },
  { "읖", "읖", "읖", "읖", "읖" },
  { "읗", "읗", "읗", "읗", "읗" },
  { "의", "의", "의", "의", "의" },
  { "읙", "읙", "읙", "읙", "읙" },
  { "읚", "읚", "읚", "읚", "읚" },
  { "읛", "읛", "읛", "읛", "읛" },
  { "읜", "읜", "읜", "읜", "읜" },
  { "읝", "읝", "읝", "읝", "읝" },
  { "읞", "읞", "읞", "읞", "읞" },
  { "읟", "읟", "읟", "읟", "읟" },
  { "읠", "읠", "읠", "읠", "읠" },
  { "읡", "읡", "읡", "읡", "읡" },
  { "읢", "읢", "읢", "읢", "읢" },
  { "읣", "읣", "읣", "읣", "읣" },
  { "읤", "읤", "읤", "읤", "읤" },
  { "읥", "읥", "읥", "읥", "읥" },
  { "읦", "읦", "읦", "읦", "읦" },
  { "읧", "읧", "읧", "읧", "읧" },
  { "읨", "읨", "읨", "읨", "읨" },
  { "읩", "읩", "읩", "읩", "읩" },
  { "읪", "읪", "읪", "읪", "읪" },
  { "읫", "읫", "읫", "읫", "읫" },
  { "읬", "읬", "읬", "읬", "읬" },
  { "읭", "읭", "읭", "읭", "읭" },
  { "읮", "읮", "읮", "읮", "읮" },
  { "읯", "읯", "읯", "읯", "읯" },
  { "읰", "읰", "읰", "읰", "읰" },
  { "읱", "읱", "읱", "읱", "읱" },
  { "읲", "읲", "읲", "읲", "읲" },
  { "읳", "읳", "읳", "읳", "읳" },
  { "이", "이", "이", "이", "이" },
  { "익", "익", "익", "익", "익" },
  { "읶", "읶", "읶", "읶", "읶" },
  { "읷", "읷", "읷", "읷", "읷" },
  { "인", "인", "인", "인", "인" },
  { "읹", "읹", "읹", "읹", "읹" },
  { "읺", "읺", "읺", "읺", "읺" },
  { "읻", "읻", "읻", "읻", "읻" },
  { "일", "일", "일", "일", "일" },
  { "읽", "읽", "읽", "읽", "읽" },
  { "읾", "읾", "읾", "읾", "읾" },
  { "읿", "읿", "읿", "읿", "읿" },
  { "잀", "잀", "잀", "잀", "잀" },
  { "잁", "잁", "잁", "잁", "잁" },
  { "잂", "잂", "잂", "잂", "잂" },
  { "잃", "잃", "잃", "잃", "잃" },
  { "임", "임", "임", "임", "임" },
  { "입", "입", "입", "입", "입" },
  { "잆", "잆", "잆", "잆", "잆" },
  { "잇", "잇", "잇", "잇", "잇" },
  { "있", "있", "있", "있", "있" },
  { "잉", "잉", "잉", "잉", "잉" },
  { "잊", "잊", "잊", "잊", "잊" },
  { "잋", "잋", "잋", "잋", "잋" },
  { "잌", "잌", "잌", "잌", "잌" },
  { "잍", "잍", "잍", "잍", "잍" },
  { "잎", "잎", "잎", "잎", "잎" },
  { "잏", "잏", "잏", "잏", "잏" },
  { "자", "자", "자", "자", "자" },
  { "작", "작", "작", "작", "작" },
  { "잒", "잒", "잒", "잒", "잒" },
  { "잓", "잓", "잓", "잓", "잓" },
  { "잔", "잔", "잔", "잔", "잔" },
  { "잕", "잕", "잕", "잕", "잕" },
  { "잖", "잖", "잖", "잖", "잖" },
  { "잗", "잗", "잗", "잗", "잗" },
  { "잘", "잘", "잘", "잘", "잘" },
  { "잙", "잙", "잙", "잙", "잙" },
  { "잚", "잚", "잚", "잚", "잚" },
  { "잛", "잛", "잛", "잛", "잛" },
  { "잜", "잜", "잜", "잜", "잜" },
  { "잝", "잝", "잝", "잝", "잝" },
  { "잞", "잞", "잞", "잞", "잞" },
  { "잟", "잟", "잟", "잟", "잟" },
  { "잠", "잠", "잠", "잠", "잠" },
  { "잡", "잡", "잡", "잡", "잡" },
  { "잢", "잢", "잢", "잢", "잢" },
  { "잣", "잣", "잣", "잣", "잣" },
  { "잤", "잤", "잤", "잤", "잤" },
  { "장", "장", "장", "장", "장" },
  { "잦", "잦", "잦", "잦", "잦" },
  { "잧", "잧", "잧", "잧", "잧" },
  { "잨", "잨", "잨", "잨", "잨" },
  { "잩", "잩", "잩", "잩", "잩" },
  { "잪", "잪", "잪", "잪", "잪" },
  { "잫", "잫", "잫", "잫", "잫" },
  { "재", "재", "재", "재", "재" },
  { "잭", "잭", "잭", "잭", "잭" },
  { "잮", "잮", "잮", "잮", "잮" },
  { "잯", "잯", "잯", "잯", "잯" },
  { "잰", "잰", "잰", "잰", "잰" },
  { "잱", "잱", "잱", "잱", "잱" },
  { "잲", "잲", "잲", "잲", "잲" },
  { "잳", "잳", "잳", "잳", "잳" },
  { "잴", "잴", "잴", "잴", "잴" },
  { "잵", "잵", "잵", "잵", "잵" },
  { "잶", "잶", "잶", "잶", "잶" },
  { "잷", "잷", "잷", "잷", "잷" },
  { "잸", "잸", "잸", "잸", "잸" },
  { "잹", "잹", "잹", "잹", "잹" },
  { "잺", "잺", "잺", "잺", "잺" },
  { "잻", "잻", "잻", "잻", "잻" },
  { "잼", "잼", "잼", "잼", "잼" },
  { "잽", "잽", "잽", "잽", "잽" },
  { "잾", "잾", "잾", "잾", "잾" },
  { "잿", "잿", "잿", "잿", "잿" },
  { "쟀", "쟀", "쟀", "쟀", "쟀" },
  { "쟁", "쟁", "쟁", "쟁", "쟁" },
  { "쟂", "쟂", "쟂", "쟂", "쟂" },
  { "쟃", "쟃", "쟃", "쟃", "쟃" },
  { "쟄", "쟄", "쟄", "쟄", "쟄" },
  { "쟅", "쟅", "쟅", "쟅", "쟅" },
  { "쟆", "쟆", "쟆", "쟆", "쟆" },
  { "쟇", "쟇", "쟇", "쟇", "쟇" },
  { "쟈", "쟈", "쟈", "쟈", "쟈" },
  { "쟉", "쟉", "쟉", "쟉", "쟉" },
  { "쟊", "쟊", "쟊", "쟊", "쟊" },
  { "쟋", "쟋", "쟋", "쟋", "쟋" },
  { "쟌", "쟌", "쟌", "쟌", "쟌" },
  { "쟍", "쟍", "쟍", "쟍", "쟍" },
  { "쟎", "쟎", "쟎", "쟎", "쟎" },
  { "쟏", "쟏", "쟏", "쟏", "쟏" },
  { "쟐", "쟐", "쟐", "쟐", "쟐" },
  { "쟑", "쟑", "쟑", "쟑", "쟑" },
  { "쟒", "쟒", "쟒", "쟒", "쟒" },
  { "쟓", "쟓", "쟓", "쟓", "쟓" },
  { "쟔", "쟔", "쟔", "쟔", "쟔" },
  { "쟕", "쟕", "쟕", "쟕", "쟕" },
  { "쟖", "쟖", "쟖", "쟖", "쟖" },
  { "쟗", "쟗", "쟗", "쟗", "쟗" },
  { "쟘", "쟘", "쟘", "쟘", "쟘" },
  { "쟙", "쟙", "쟙", "쟙", "쟙" },
  { "쟚", "쟚", "쟚", "쟚", "쟚" },
  { "쟛", "쟛", "쟛", "쟛", "쟛" },
  { "쟜", "쟜", "쟜", "쟜", "쟜" },
  { "쟝", "쟝", "쟝", "쟝", "쟝" },
  { "쟞", "쟞", "쟞", "쟞", "쟞" },
  { "쟟", "쟟", "쟟", "쟟", "쟟" },
  { "쟠", "쟠", "쟠", "쟠", "쟠" },
  { "쟡", "쟡", "쟡", "쟡", "쟡" },
  { "쟢", "쟢", "쟢", "쟢", "쟢" },
  { "쟣", "쟣", "쟣", "쟣", "쟣" },
  { "쟤", "쟤", "쟤", "쟤", "쟤" },
  { "쟥", "쟥", "쟥", "쟥", "쟥" },
  { "쟦", "쟦", "쟦", "쟦", "쟦" },
  { "쟧", "쟧", "쟧", "쟧", "쟧" },
  { "쟨", "쟨", "쟨", "쟨", "쟨" },
  { "쟩", "쟩", "쟩", "쟩", "쟩" },
  { "쟪", "쟪", "쟪", "쟪", "쟪" },
  { "쟫", "쟫", "쟫", "쟫", "쟫" },
  { "쟬", "쟬", "쟬", "쟬", "쟬" },
  { "쟭", "쟭", "쟭", "쟭", "쟭" },
  { "쟮", "쟮", "쟮", "쟮", "쟮" },
  { "쟯", "쟯", "쟯", "쟯", "쟯" },
  { "쟰", "쟰", "쟰", "쟰", "쟰" },
  { "쟱", "쟱", "쟱", "쟱", "쟱" },
  { "쟲", "쟲", "쟲", "쟲", "쟲" },
  { "쟳", "쟳", "쟳", "쟳", "쟳" },
  { "쟴", "쟴", "쟴", "쟴", "쟴" },
  { "쟵", "쟵", "쟵", "쟵", "쟵" },
  { "쟶", "쟶", "쟶", "쟶", "쟶" },
  { "쟷", "쟷", "쟷", "쟷", "쟷" },
  { "쟸", "쟸", "쟸", "쟸", "쟸" },
  { "쟹", "쟹", "쟹", "쟹", "쟹" },
  { "쟺", "쟺", "쟺", "쟺", "쟺" },
  { "쟻", "쟻", "쟻", "쟻", "쟻" },
  { "쟼", "쟼", "쟼", "쟼", "쟼" },
  { "쟽", "쟽", "쟽", "쟽", "쟽" },
  { "쟾", "쟾", "쟾", "쟾", "쟾" },
  { "쟿", "쟿", "쟿", "쟿", "쟿" },
  { "저", "저", "저", "저", "저" },
  { "적", "적", "적", "적", "적" },
  { "젂", "젂", "젂", "젂", "젂" },
  { "젃", "젃", "젃", "젃", "젃" },
  { "전", "전", "전", "전", "전" },
  { "젅", "젅", "젅", "젅", "젅" },
  { "젆", "젆", "젆", "젆", "젆" },
  { "젇", "젇", "젇", "젇", "젇" },
  { "절", "절", "절", "절", "절" },
  { "젉", "젉", "젉", "젉", "젉" },
  { "젊", "젊", "젊", "젊", "젊" },
  { "젋", "젋", "젋", "젋", "젋" },
  { "젌", "젌", "젌", "젌", "젌" },
  { "젍", "젍", "젍", "젍", "젍" },
  { "젎", "젎", "젎", "젎", "젎" },
  { "젏", "젏", "젏", "젏", "젏" },
  { "점", "점", "점", "점", "점" },
  { "접", "접", "접", "접", "접" },
  { "젒", "젒", "젒", "젒", "젒" },
  { "젓", "젓", "젓", "젓", "젓" },
  { "젔", "젔", "젔", "젔", "젔" },
  { "정", "정", "정", "정", "정" },
  { "젖", "젖", "젖", "젖", "젖" },
  { "젗", "젗", "젗", "젗", "젗" },
  { "젘", "젘", "젘", "젘", "젘" },
  { "젙", "젙", "젙", "젙", "젙" },
  { "젚", "젚", "젚", "젚", "젚" },
  { "젛", "젛", "젛", "젛", "젛" },
  { "제", "제", "제", "제", "제" },
  { "젝", "젝", "젝", "젝", "젝" },
  { "젞", "젞", "젞", "젞", "젞" },
  { "젟", "젟", "젟", "젟", "젟" },
  { "젠", "젠", "젠", "젠", "젠" },
  { "젡", "젡", "젡", "젡", "젡" },
  { "젢", "젢", "젢", "젢", "젢" },
  { "젣", "젣", "젣", "젣", "젣" },
  { "젤", "젤", "젤", "젤", "젤" },
  { "젥", "젥", "젥", "젥", "젥" },
  { "젦", "젦", "젦", "젦", "젦" },
  { "젧", "젧", "젧", "젧", "젧" },
  { "젨", "젨", "젨", "젨", "젨" },
  { "젩", "젩", "젩", "젩", "젩" },
  { "젪", "젪", "젪", "젪", "젪" },
  { "젫", "젫", "젫", "젫", "젫" },
  { "젬", "젬", "젬", "젬", "젬" },
  { "젭", "젭", "젭", "젭", "젭" },
  { "젮", "젮", "젮", "젮", "젮" },
  { "젯", "젯", "젯", "젯", "젯" },
  { "젰", "젰", "젰", "젰", "젰" },
  { "젱", "젱", "젱", "젱", "젱" },
  { "젲", "젲", "젲", "젲", "젲" },
  { "젳", "젳", "젳", "젳", "젳" },
  { "젴", "젴", "젴", "젴", "젴" },
  { "젵", "젵", "젵", "젵", "젵" },
  { "젶", "젶", "젶", "젶", "젶" },
  { "젷", "젷", "젷", "젷", "젷" },
  { "져", "져", "져", "져", "져" },
  { "젹", "젹", "젹", "젹", "젹" },
  { "젺", "젺", "젺", "젺", "젺" },
  { "젻", "젻", "젻", "젻", "젻" },
  { "젼", "젼", "젼", "젼", "젼" },
  { "젽", "젽", "젽", "젽", "젽" },
  { "젾", "젾", "젾", "젾", "젾" },
  { "젿", "젿", "젿", "젿", "젿" },
  { "졀", "졀", "졀", "졀", "졀" },
  { "졁", "졁", "졁", "졁", "졁" },
  { "졂", "졂", "졂", "졂", "졂" },
  { "졃", "졃", "졃", "졃", "졃" },
  { "졄", "졄", "졄", "졄", "졄" },
  { "졅", "졅", "졅", "졅", "졅" },
  { "졆", "졆", "졆", "졆", "졆" },
  { "졇", "졇", "졇", "졇", "졇" },
  { "졈", "졈", "졈", "졈", "졈" },
  { "졉", "졉", "졉", "졉", "졉" },
  { "졊", "졊", "졊", "졊", "졊" },
  { "졋", "졋", "졋", "졋", "졋" },
  { "졌", "졌", "졌", "졌", "졌" },
  { "졍", "졍", "졍", "졍", "졍" },
  { "졎", "졎", "졎", "졎", "졎" },
  { "졏", "졏", "졏", "졏", "졏" },
  { "졐", "졐", "졐", "졐", "졐" },
  { "졑", "졑", "졑", "졑", "졑" },
  { "졒", "졒", "졒", "졒", "졒" },
  { "졓", "졓", "졓", "졓", "졓" },
  { "졔", "졔", "졔", "졔", "졔" },
  { "졕", "졕", "졕", "졕", "졕" },
  { "졖", "졖", "졖", "졖", "졖" },
  { "졗", "졗", "졗", "졗", "졗" },
  { "졘", "졘", "졘", "졘", "졘" },
  { "졙", "졙", "졙", "졙", "졙" },
  { "졚", "졚", "졚", "졚", "졚" },
  { "졛", "졛", "졛", "졛", "졛" },
  { "졜", "졜", "졜", "졜", "졜" },
  { "졝", "졝", "졝", "졝", "졝" },
  { "졞", "졞", "졞", "졞", "졞" },
  { "졟", "졟", "졟", "졟", "졟" },
  { "졠", "졠", "졠", "졠", "졠" },
  { "졡", "졡", "졡", "졡", "졡" },
  { "졢", "졢", "졢", "졢", "졢" },
  { "졣", "졣", "졣", "졣", "졣" },
  { "졤", "졤", "졤", "졤", "졤" },
  { "졥", "졥", "졥", "졥", "졥" },
  { "졦", "졦", "졦", "졦", "졦" },
  { "졧", "졧", "졧", "졧", "졧" },
  { "졨", "졨", "졨", "졨", "졨" },
  { "졩", "졩", "졩", "졩", "졩" },
  { "졪", "졪", "졪", "졪", "졪" },
  { "졫", "졫", "졫", "졫", "졫" },
  { "졬", "졬", "졬", "졬", "졬" },
  { "졭", "졭", "졭", "졭", "졭" },
  { "졮", "졮", "졮", "졮", "졮" },
  { "졯", "졯", "졯", "졯", "졯" },
  { "조", "조", "조", "조", "조" },
  { "족", "족", "족", "족", "족" },
  { "졲", "졲", "졲", "졲", "졲" },
  { "졳", "졳", "졳", "졳", "졳" },
  { "존", "존", "존", "존", "존" },
  { "졵", "졵", "졵", "졵", "졵" },
  { "졶", "졶", "졶", "졶", "졶" },
  { "졷", "졷", "졷", "졷", "졷" },
  { "졸", "졸", "졸", "졸", "졸" },
  { "졹", "졹", "졹", "졹", "졹" },
  { "졺", "졺", "졺", "졺", "졺" },
  { "졻", "졻", "졻", "졻", "졻" },
  { "졼", "졼", "졼", "졼", "졼" },
  { "졽", "졽", "졽", "졽", "졽" },
  { "졾", "졾", "졾", "졾", "졾" },
  { "졿", "졿", "졿", "졿", "졿" },
  { "좀", "좀", "좀", "좀", "좀" },
  { "좁", "좁", "좁", "좁", "좁" },
  { "좂", "좂", "좂", "좂", "좂" },
  { "좃", "좃", "좃", "좃", "좃" },
  { "좄", "좄", "좄", "좄", "좄" },
  { "종", "종", "종", "종", "종" },
  { "좆", "좆", "좆", "좆", "좆" },
  { "좇", "좇", "좇", "좇", "좇" },
  { "좈", "좈", "좈", "좈", "좈" },
  { "좉", "좉", "좉", "좉", "좉" },
  { "좊", "좊", "좊", "좊", "좊" },
  { "좋", "좋", "좋", "좋", "좋" },
  { "좌", "좌", "좌", "좌", "좌" },
  { "좍", "좍", "좍", "좍", "좍" },
  { "좎", "좎", "좎", "좎", "좎" },
  { "좏", "좏", "좏", "좏", "좏" },
  { "좐", "좐", "좐", "좐", "좐" },
  { "좑", "좑", "좑", "좑", "좑" },
  { "좒", "좒", "좒", "좒", "좒" },
  { "좓", "좓", "좓", "좓", "좓" },
  { "좔", "좔", "좔", "좔", "좔" },
  { "좕", "좕", "좕", "좕", "좕" },
  { "좖", "좖", "좖", "좖", "좖" },
  { "좗", "좗", "좗", "좗", "좗" },
  { "좘", "좘", "좘", "좘", "좘" },
  { "좙", "좙", "좙", "좙", "좙" },
  { "좚", "좚", "좚", "좚", "좚" },
  { "좛", "좛", "좛", "좛", "좛" },
  { "좜", "좜", "좜", "좜", "좜" },
  { "좝", "좝", "좝", "좝", "좝" },
  { "좞", "좞", "좞", "좞", "좞" },
  { "좟", "좟", "좟", "좟", "좟" },
  { "좠", "좠", "좠", "좠", "좠" },
  { "좡", "좡", "좡", "좡", "좡" },
  { "좢", "좢", "좢", "좢", "좢" },
  { "좣", "좣", "좣", "좣", "좣" },
  { "좤", "좤", "좤", "좤", "좤" },
  { "좥", "좥", "좥", "좥", "좥" },
  { "좦", "좦", "좦", "좦", "좦" },
  { "좧", "좧", "좧", "좧", "좧" },
  { "좨", "좨", "좨", "좨", "좨" },
  { "좩", "좩", "좩", "좩", "좩" },
  { "좪", "좪", "좪", "좪", "좪" },
  { "좫", "좫", "좫", "좫", "좫" },
  { "좬", "좬", "좬", "좬", "좬" },
  { "좭", "좭", "좭", "좭", "좭" },
  { "좮", "좮", "좮", "좮", "좮" },
  { "좯", "좯", "좯", "좯", "좯" },
  { "좰", "좰", "좰", "좰", "좰" },
  { "좱", "좱", "좱", "좱", "좱" },
  { "좲", "좲", "좲", "좲", "좲" },
  { "좳", "좳", "좳", "좳", "좳" },
  { "좴", "좴", "좴", "좴", "좴" },
  { "좵", "좵", "좵", "좵", "좵" },
  { "좶", "좶", "좶", "좶", "좶" },
  { "좷", "좷", "좷", "좷", "좷" },
  { "좸", "좸", "좸", "좸", "좸" },
  { "좹", "좹", "좹", "좹", "좹" },
  { "좺", "좺", "좺", "좺", "좺" },
  { "좻", "좻", "좻", "좻", "좻" },
  { "좼", "좼", "좼", "좼", "좼" },
  { "좽", "좽", "좽", "좽", "좽" },
  { "좾", "좾", "좾", "좾", "좾" },
  { "좿", "좿", "좿", "좿", "좿" },
  { "죀", "죀", "죀", "죀", "죀" },
  { "죁", "죁", "죁", "죁", "죁" },
  { "죂", "죂", "죂", "죂", "죂" },
  { "죃", "죃", "죃", "죃", "죃" },
  { "죄", "죄", "죄", "죄", "죄" },
  { "죅", "죅", "죅", "죅", "죅" },
  { "죆", "죆", "죆", "죆", "죆" },
  { "죇", "죇", "죇", "죇", "죇" },
  { "죈", "죈", "죈", "죈", "죈" },
  { "죉", "죉", "죉", "죉", "죉" },
  { "죊", "죊", "죊", "죊", "죊" },
  { "죋", "죋", "죋", "죋", "죋" },
  { "죌", "죌", "죌", "죌", "죌" },
  { "죍", "죍", "죍", "죍", "죍" },
  { "죎", "죎", "죎", "죎", "죎" },
  { "죏", "죏", "죏", "죏", "죏" },
  { "죐", "죐", "죐", "죐", "죐" },
  { "죑", "죑", "죑", "죑", "죑" },
  { "죒", "죒", "죒", "죒", "죒" },
  { "죓", "죓", "죓", "죓", "죓" },
  { "죔", "죔", "죔", "죔", "죔" },
  { "죕", "죕", "죕", "죕", "죕" },
  { "죖", "죖", "죖", "죖", "죖" },
  { "죗", "죗", "죗", "죗", "죗" },
  { "죘", "죘", "죘", "죘", "죘" },
  { "죙", "죙", "죙", "죙", "죙" },
  { "죚", "죚", "죚", "죚", "죚" },
  { "죛", "죛", "죛", "죛", "죛" },
  { "죜", "죜", "죜", "죜", "죜" },
  { "죝", "죝", "죝", "죝", "죝" },
  { "죞", "죞", "죞", "죞", "죞" },
  { "죟", "죟", "죟", "죟", "죟" },
  { "죠", "죠", "죠", "죠", "죠" },
  { "죡", "죡", "죡", "죡", "죡" },
  { "죢", "죢", "죢", "죢", "죢" },
  { "죣", "죣", "죣", "죣", "죣" },
  { "죤", "죤", "죤", "죤", "죤" },
  { "죥", "죥", "죥", "죥", "죥" },
  { "죦", "죦", "죦", "죦", "죦" },
  { "죧", "죧", "죧", "죧", "죧" },
  { "죨", "죨", "죨", "죨", "죨" },
  { "죩", "죩", "죩", "죩", "죩" },
  { "죪", "죪", "죪", "죪", "죪" },
  { "죫", "죫", "죫", "죫", "죫" },
  { "죬", "죬", "죬", "죬", "죬" },
  { "죭", "죭", "죭", "죭", "죭" },
  { "죮", "죮", "죮", "죮", "죮" },
  { "죯", "죯", "죯", "죯", "죯" },
  { "죰", "죰", "죰", "죰", "죰" },
  { "죱", "죱", "죱", "죱", "죱" },
  { "죲", "죲", "죲", "죲", "죲" },
  { "죳", "죳", "죳", "죳", "죳" },
  { "죴", "죴", "죴", "죴", "죴" },
  { "죵", "죵", "죵", "죵", "죵" },
  { "죶", "죶", "죶", "죶", "죶" },
  { "죷", "죷", "죷", "죷", "죷" },
  { "죸", "죸", "죸", "죸", "죸" },
  { "죹", "죹", "죹", "죹", "죹" },
  { "죺", "죺", "죺", "죺", "죺" },
  { "죻", "죻", "죻", "죻", "죻" },
  { "주", "주", "주", "주", "주" },
  { "죽", "죽", "죽", "죽", "죽" },
  { "죾", "죾", "죾", "죾", "죾" },
  { "죿", "죿", "죿", "죿", "죿" },
  { "준", "준", "준", "준", "준" },
  { "줁", "줁", "줁", "줁", "줁" },
  { "줂", "줂", "줂", "줂", "줂" },
  { "줃", "줃", "줃", "줃", "줃" },
  { "줄", "줄", "줄", "줄", "줄" },
  { "줅", "줅", "줅", "줅", "줅" },
  { "줆", "줆", "줆", "줆", "줆" },
  { "줇", "줇", "줇", "줇", "줇" },
  { "줈", "줈", "줈", "줈", "줈" },
  { "줉", "줉", "줉", "줉", "줉" },
  { "줊", "줊", "줊", "줊", "줊" },
  { "줋", "줋", "줋", "줋", "줋" },
  { "줌", "줌", "줌", "줌", "줌" },
  { "줍", "줍", "줍", "줍", "줍" },
  { "줎", "줎", "줎", "줎", "줎" },
  { "줏", "줏", "줏", "줏", "줏" },
  { "줐", "줐", "줐", "줐", "줐" },
  { "중", "중", "중", "중", "중" },
  { "줒", "줒", "줒", "줒", "줒" },
  { "줓", "줓", "줓", "줓", "줓" },
  { "줔", "줔", "줔", "줔", "줔" },
  { "줕", "줕", "줕", "줕", "줕" },
  { "줖", "줖", "줖", "줖", "줖" },
  { "줗", "줗", "줗", "줗", "줗" },
  { "줘", "줘", "줘", "줘", "줘" },
  { "줙", "줙", "줙", "줙", "줙" },
  { "줚", "줚", "줚", "줚", "줚" },
  { "줛", "줛", "줛", "줛", "줛" },
  { "줜", "줜", "줜", "줜", "줜" },
  { "줝", "줝", "줝", "줝", "줝" },
  { "줞", "줞", "줞", "줞", "줞" },
  { "줟", "줟", "줟", "줟", "줟" },
  { "줠", "줠", "줠", "줠", "줠" },
  { "줡", "줡", "줡", "줡", "줡" },
  { "줢", "줢", "줢", "줢", "줢" },
  { "줣", "줣", "줣", "줣", "줣" },
  { "줤", "줤", "줤", "줤", "줤" },
  { "줥", "줥", "줥", "줥", "줥" },
  { "줦", "줦", "줦", "줦", "줦" },
  { "줧", "줧", "줧", "줧", "줧" },
  { "줨", "줨", "줨", "줨", "줨" },
  { "줩", "줩", "줩", "줩", "줩" },
  { "줪", "줪", "줪", "줪", "줪" },
  { "줫", "줫", "줫", "줫", "줫" },
  { "줬", "줬", "줬", "줬", "줬" },
  { "줭", "줭", "줭", "줭", "줭" },
  { "줮", "줮", "줮", "줮", "줮" },
  { "줯", "줯", "줯", "줯", "줯" },
  { "줰", "줰", "줰", "줰", "줰" },
  { "줱", "줱", "줱", "줱", "줱" },
  { "줲", "줲", "줲", "줲", "줲" },
  { "줳", "줳", "줳", "줳", "줳" },
  { "줴", "줴", "줴", "줴", "줴" },
  { "줵", "줵", "줵", "줵", "줵" },
  { "줶", "줶", "줶", "줶", "줶" },
  { "줷", "줷", "줷", "줷", "줷" },
  { "줸", "줸", "줸", "줸", "줸" },
  { "줹", "줹", "줹", "줹", "줹" },
  { "줺", "줺", "줺", "줺", "줺" },
  { "줻", "줻", "줻", "줻", "줻" },
  { "줼", "줼", "줼", "줼", "줼" },
  { "줽", "줽", "줽", "줽", "줽" },
  { "줾", "줾", "줾", "줾", "줾" },
  { "줿", "줿", "줿", "줿", "줿" },
  { "쥀", "쥀", "쥀", "쥀", "쥀" },
  { "쥁", "쥁", "쥁", "쥁", "쥁" },
  { "쥂", "쥂", "쥂", "쥂", "쥂" },
  { "쥃", "쥃", "쥃", "쥃", "쥃" },
  { "쥄", "쥄", "쥄", "쥄", "쥄" },
  { "쥅", "쥅", "쥅", "쥅", "쥅" },
  { "쥆", "쥆", "쥆", "쥆", "쥆" },
  { "쥇", "쥇", "쥇", "쥇", "쥇" },
  { "쥈", "쥈", "쥈", "쥈", "쥈" },
  { "쥉", "쥉", "쥉", "쥉", "쥉" },
  { "쥊", "쥊", "쥊", "쥊", "쥊" },
  { "쥋", "쥋", "쥋", "쥋", "쥋" },
  { "쥌", "쥌", "쥌", "쥌", "쥌" },
  { "쥍", "쥍", "쥍", "쥍", "쥍" },
  { "쥎", "쥎", "쥎", "쥎", "쥎" },
  { "쥏", "쥏", "쥏", "쥏", "쥏" },
  { "쥐", "쥐", "쥐", "쥐", "쥐" },
  { "쥑", "쥑", "쥑", "쥑", "쥑" },
  { "쥒", "쥒", "쥒", "쥒", "쥒" },
  { "쥓", "쥓", "쥓", "쥓", "쥓" },
  { "쥔", "쥔", "쥔", "쥔", "쥔" },
  { "쥕", "쥕", "쥕", "쥕", "쥕" },
  { "쥖", "쥖", "쥖", "쥖", "쥖" },
  { "쥗", "쥗", "쥗", "쥗", "쥗" },
  { "쥘", "쥘", "쥘", "쥘", "쥘" },
  { "쥙", "쥙", "쥙", "쥙", "쥙" },
  { "쥚", "쥚", "쥚", "쥚", "쥚" },
  { "쥛", "쥛", "쥛", "쥛", "쥛" },
  { "쥜", "쥜", "쥜", "쥜", "쥜" },
  { "쥝", "쥝", "쥝", "쥝", "쥝" },
  { "쥞", "쥞", "쥞", "쥞", "쥞" },
  { "쥟", "쥟", "쥟", "쥟", "쥟" },
  { "쥠", "쥠", "쥠", "쥠", "쥠" },
  { "쥡", "쥡", "쥡", "쥡", "쥡" },
  { "쥢", "쥢", "쥢", "쥢", "쥢" },
  { "쥣", "쥣", "쥣", "쥣", "쥣" },
  { "쥤", "쥤", "쥤", "쥤", "쥤" },
  { "쥥", "쥥", "쥥", "쥥", "쥥" },
  { "쥦", "쥦", "쥦", "쥦", "쥦" },
  { "쥧", "쥧", "쥧", "쥧", "쥧" },
  { "쥨", "쥨", "쥨", "쥨", "쥨" },
  { "쥩", "쥩", "쥩", "쥩", "쥩" },
  { "쥪", "쥪", "쥪", "쥪", "쥪" },
  { "쥫", "쥫", "쥫", "쥫", "쥫" },
  { "쥬", "쥬", "쥬", "쥬", "쥬" },
  { "쥭", "쥭", "쥭", "쥭", "쥭" },
  { "쥮", "쥮", "쥮", "쥮", "쥮" },
  { "쥯", "쥯", "쥯", "쥯", "쥯" },
  { "쥰", "쥰", "쥰", "쥰", "쥰" },
  { "쥱", "쥱", "쥱", "쥱", "쥱" },
  { "쥲", "쥲", "쥲", "쥲", "쥲" },
  { "쥳", "쥳", "쥳", "쥳", "쥳" },
  { "쥴", "쥴", "쥴", "쥴", "쥴" },
  { "쥵", "쥵", "쥵", "쥵", "쥵" },
  { "쥶", "쥶", "쥶", "쥶", "쥶" },
  { "쥷", "쥷", "쥷", "쥷", "쥷" },
  { "쥸", "쥸", "쥸", "쥸", "쥸" },
  { "쥹", "쥹", "쥹", "쥹", "쥹" },
  { "쥺", "쥺", "쥺", "쥺", "쥺" },
  { "쥻", "쥻", "쥻", "쥻", "쥻" },
  { "쥼", "쥼", "쥼", "쥼", "쥼" },
  { "쥽", "쥽", "쥽", "쥽", "쥽" },
  { "쥾", "쥾", "쥾", "쥾", "쥾" },
  { "쥿", "쥿", "쥿", "쥿", "쥿" },
  { "즀", "즀", "즀", "즀", "즀" },
  { "즁", "즁", "즁", "즁", "즁" },
  { "즂", "즂", "즂", "즂", "즂" },
  { "즃", "즃", "즃", "즃", "즃" },
  { "즄", "즄", "즄", "즄", "즄" },
  { "즅", "즅", "즅", "즅", "즅" },
  { "즆", "즆", "즆", "즆", "즆" },
  { "즇", "즇", "즇", "즇", "즇" },
  { "즈", "즈", "즈", "즈", "즈" },
  { "즉", "즉", "즉", "즉", "즉" },
  { "즊", "즊", "즊", "즊", "즊" },
  { "즋", "즋", "즋", "즋", "즋" },
  { "즌", "즌", "즌", "즌", "즌" },
  { "즍", "즍", "즍", "즍", "즍" },
  { "즎", "즎", "즎", "즎", "즎" },
  { "즏", "즏", "즏", "즏", "즏" },
  { "즐", "즐", "즐", "즐", "즐" },
  { "즑", "즑", "즑", "즑", "즑" },
  { "즒", "즒", "즒", "즒", "즒" },
  { "즓", "즓", "즓", "즓", "즓" },
  { "즔", "즔", "즔", "즔", "즔" },
  { "즕", "즕", "즕", "즕", "즕" },
  { "즖", "즖", "즖", "즖", "즖" },
  { "즗", "즗", "즗", "즗", "즗" },
  { "즘", "즘", "즘", "즘", "즘" },
  { "즙", "즙", "즙", "즙", "즙" },
  { "즚", "즚", "즚", "즚", "즚" },
  { "즛", "즛", "즛", "즛", "즛" },
  { "즜", "즜", "즜", "즜", "즜" },
  { "증", "증", "증", "증", "증" },
  { "즞", "즞", "즞", "즞", "즞" },
  { "즟", "즟", "즟", "즟", "즟" },
  { "즠", "즠", "즠", "즠", "즠" },
  { "즡", "즡", "즡", "즡", "즡" },
  { "즢", "즢", "즢", "즢", "즢" },
  { "즣", "즣", "즣", "즣", "즣" },
  { "즤", "즤", "즤", "즤", "즤" },
  { "즥", "즥", "즥", "즥", "즥" },
  { "즦", "즦", "즦", "즦", "즦" },
  { "즧", "즧", "즧", "즧", "즧" },
  { "즨", "즨", "즨", "즨", "즨" },
  { "즩", "즩", "즩", "즩", "즩" },
  { "즪", "즪", "즪", "즪", "즪" },
  { "즫", "즫", "즫", "즫", "즫" },
  { "즬", "즬", "즬", "즬", "즬" },
  { "즭", "즭", "즭", "즭", "즭" },
  { "즮", "즮", "즮", "즮", "즮" },
  { "즯", "즯", "즯", "즯", "즯" },
  { "즰", "즰", "즰", "즰", "즰" },
  { "즱", "즱", "즱", "즱", "즱" },
  { "즲", "즲", "즲", "즲", "즲" },
  { "즳", "즳", "즳", "즳", "즳" },
  { "즴", "즴", "즴", "즴", "즴" },
  { "즵", "즵", "즵", "즵", "즵" },
  { "즶", "즶", "즶", "즶", "즶" },
  { "즷", "즷", "즷", "즷", "즷" },
  { "즸", "즸", "즸", "즸", "즸" },
  { "즹", "즹", "즹", "즹", "즹" },
  { "즺", "즺", "즺", "즺", "즺" },
  { "즻", "즻", "즻", "즻", "즻" },
  { "즼", "즼", "즼", "즼", "즼" },
  { "즽", "즽", "즽", "즽", "즽" },
  { "즾", "즾", "즾", "즾", "즾" },
  { "즿", "즿", "즿", "즿", "즿" },
  { "지", "지", "지", "지", "지" },
  { "직", "직", "직", "직", "직" },
  { "짂", "짂", "짂", "짂", "짂" },
  { "짃", "짃", "짃", "짃", "짃" },
  { "진", "진", "진", "진", "진" },
  { "짅", "짅", "짅", "짅", "짅" },
  { "짆", "짆", "짆", "짆", "짆" },
  { "짇", "짇", "짇", "짇", "짇" },
  { "질", "질", "질", "질", "질" },
  { "짉", "짉", "짉", "짉", "짉" },
  { "짊", "짊", "짊", "짊", "짊" },
  { "짋", "짋", "짋", "짋", "짋" },
  { "짌", "짌", "짌", "짌", "짌" },
  { "짍", "짍", "짍", "짍", "짍" },
  { "짎", "짎", "짎", "짎", "짎" },
  { "짏", "짏", "짏", "짏", "짏" },
  { "짐", "짐", "짐", "짐", "짐" },
  { "집", "집", "집", "집", "집" },
  { "짒", "짒", "짒", "짒", "짒" },
  { "짓", "짓", "짓", "짓", "짓" },
  { "짔", "짔", "짔", "짔", "짔" },
  { "징", "징", "징", "징", "징" },
  { "짖", "짖", "짖", "짖", "짖" },
  { "짗", "짗", "짗", "짗", "짗" },
  { "짘", "짘", "짘", "짘", "짘" },
  { "짙", "짙", "짙", "짙", "짙" },
  { "짚", "짚", "짚", "짚", "짚" },
  { "짛", "짛", "짛", "짛", "짛" },
  { "짜", "짜", "짜", "짜", "짜" },
  { "짝", "짝", "짝", "짝", "짝" },
  { "짞", "짞", "짞", "짞", "짞" },
  { "짟", "짟", "짟", "짟", "짟" },
  { "짠", "짠", "짠", "짠", "짠" },
  { "짡", "짡", "짡", "짡", "짡" },
  { "짢", "짢", "짢", "짢", "짢" },
  { "짣", "짣", "짣", "짣", "짣" },
  { "짤", "짤", "짤", "짤", "짤" },
  { "짥", "짥", "짥", "짥", "짥" },
  { "짦", "짦", "짦", "짦", "짦" },
  { "짧", "짧", "짧", "짧", "짧" },
  { "짨", "짨", "짨", "짨", "짨" },
  { "짩", "짩", "짩", "짩", "짩" },
  { "짪", "짪", "짪", "짪", "짪" },
  { "짫", "짫", "짫", "짫", "짫" },
  { "짬", "짬", "짬", "짬", "짬" },
  { "짭", "짭", "짭", "짭", "짭" },
  { "짮", "짮", "짮", "짮", "짮" },
  { "짯", "짯", "짯", "짯", "짯" },
  { "짰", "짰", "짰", "짰", "짰" },
  { "짱", "짱", "짱", "짱", "짱" },
  { "짲", "짲", "짲", "짲", "짲" },
  { "짳", "짳", "짳", "짳", "짳" },
  { "짴", "짴", "짴", "짴", "짴" },
  { "짵", "짵", "짵", "짵", "짵" },
  { "짶", "짶", "짶", "짶", "짶" },
  { "짷", "짷", "짷", "짷", "짷" },
  { "째", "째", "째", "째", "째" },
  { "짹", "짹", "짹", "짹", "짹" },
  { "짺", "짺", "짺", "짺", "짺" },
  { "짻", "짻", "짻", "짻", "짻" },
  { "짼", "짼", "짼", "짼", "짼" },
  { "짽", "짽", "짽", "짽", "짽" },
  { "짾", "짾", "짾", "짾", "짾" },
  { "짿", "짿", "짿", "짿", "짿" },
  { "쨀", "쨀", "쨀", "쨀", "쨀" },
  { "쨁", "쨁", "쨁", "쨁", "쨁" },
  { "쨂", "쨂", "쨂", "쨂", "쨂" },
  { "쨃", "쨃", "쨃", "쨃", "쨃" },
  { "쨄", "쨄", "쨄", "쨄", "쨄" },
  { "쨅", "쨅", "쨅", "쨅", "쨅" },
  { "쨆", "쨆", "쨆", "쨆", "쨆" },
  { "쨇", "쨇", "쨇", "쨇", "쨇" },
  { "쨈", "쨈", "쨈", "쨈", "쨈" },
  { "쨉", "쨉", "쨉", "쨉", "쨉" },
  { "쨊", "쨊", "쨊", "쨊", "쨊" },
  { "쨋", "쨋", "쨋", "쨋", "쨋" },
  { "쨌", "쨌", "쨌", "쨌", "쨌" },
  { "쨍", "쨍", "쨍", "쨍", "쨍" },
  { "쨎", "쨎", "쨎", "쨎", "쨎" },
  { "쨏", "쨏", "쨏", "쨏", "쨏" },
  { "쨐", "쨐", "쨐", "쨐", "쨐" },
  { "쨑", "쨑", "쨑", "쨑", "쨑" },
  { "쨒", "쨒", "쨒", "쨒", "쨒" },
  { "쨓", "쨓", "쨓", "쨓", "쨓" },
  { "쨔", "쨔", "쨔", "쨔", "쨔" },
  { "쨕", "쨕", "쨕", "쨕", "쨕" },
  { "쨖", "쨖", "쨖", "쨖", "쨖" },
  { "쨗", "쨗", "쨗", "쨗", "쨗" },
  { "쨘", "쨘", "쨘", "쨘", "쨘" },
  { "쨙", "쨙", "쨙", "쨙", "쨙" },
  { "쨚", "쨚", "쨚", "쨚", "쨚" },
  { "쨛", "쨛", "쨛", "쨛", "쨛" },
  { "쨜", "쨜", "쨜", "쨜", "쨜" },
  { "쨝", "쨝", "쨝", "쨝", "쨝" },
  { "쨞", "쨞", "쨞", "쨞", "쨞" },
  { "쨟", "쨟", "쨟", "쨟", "쨟" },
  { "쨠", "쨠", "쨠", "쨠", "쨠" },
  { "쨡", "쨡", "쨡", "쨡", "쨡" },
  { "쨢", "쨢", "쨢", "쨢", "쨢" },
  { "쨣", "쨣", "쨣", "쨣", "쨣" },
  { "쨤", "쨤", "쨤", "쨤", "쨤" },
  { "쨥", "쨥", "쨥", "쨥", "쨥" },
  { "쨦", "쨦", "쨦", "쨦", "쨦" },
  { "쨧", "쨧", "쨧", "쨧", "쨧" },
  { "쨨", "쨨", "쨨", "쨨", "쨨" },
  { "쨩", "쨩", "쨩", "쨩", "쨩" },
  { "쨪", "쨪", "쨪", "쨪", "쨪" },
  { "쨫", "쨫", "쨫", "쨫", "쨫" },
  { "쨬", "쨬", "쨬", "쨬", "쨬" },
  { "쨭", "쨭", "쨭", "쨭", "쨭" },
  { "쨮", "쨮", "쨮", "쨮", "쨮" },
  { "쨯", "쨯", "쨯", "쨯", "쨯" },
  { "쨰", "쨰", "쨰", "쨰", "쨰" },
  { "쨱", "쨱", "쨱", "쨱", "쨱" },
  { "쨲", "쨲", "쨲", "쨲", "쨲" },
  { "쨳", "쨳", "쨳", "쨳", "쨳" },
  { "쨴", "쨴", "쨴", "쨴", "쨴" },
  { "쨵", "쨵", "쨵", "쨵", "쨵" },
  { "쨶", "쨶", "쨶", "쨶", "쨶" },
  { "쨷", "쨷", "쨷", "쨷", "쨷" },
  { "쨸", "쨸", "쨸", "쨸", "쨸" },
  { "쨹", "쨹", "쨹", "쨹", "쨹" },
  { "쨺", "쨺", "쨺", "쨺", "쨺" },
  { "쨻", "쨻", "쨻", "쨻", "쨻" },
  { "쨼", "쨼", "쨼", "쨼", "쨼" },
  { "쨽", "쨽", "쨽", "쨽", "쨽" },
  { "쨾", "쨾", "쨾", "쨾", "쨾" },
  { "쨿", "쨿", "쨿", "쨿", "쨿" },
  { "쩀", "쩀", "쩀", "쩀", "쩀" },
  { "쩁", "쩁", "쩁", "쩁", "쩁" },
  { "쩂", "쩂", "쩂", "쩂", "쩂" },
  { "쩃", "쩃", "쩃", "쩃", "쩃" },
  { "쩄", "쩄", "쩄", "쩄", "쩄" },
  { "쩅", "쩅", "쩅", "쩅", "쩅" },
  { "쩆", "쩆", "쩆", "쩆", "쩆" },
  { "쩇", "쩇", "쩇", "쩇", "쩇" },
  { "쩈", "쩈", "쩈", "쩈", "쩈" },
  { "쩉", "쩉", "쩉", "쩉", "쩉" },
  { "쩊", "쩊", "쩊", "쩊", "쩊" },
  { "쩋", "쩋", "쩋", "쩋", "쩋" },
  { "쩌", "쩌", "쩌", "쩌", "쩌" },
  { "쩍", "쩍", "쩍", "쩍", "쩍" },
  { "쩎", "쩎", "쩎", "쩎", "쩎" },
  { "쩏", "쩏", "쩏", "쩏", "쩏" },
  { "쩐", "쩐", "쩐", "쩐", "쩐" },
  { "쩑", "쩑", "쩑", "쩑", "쩑" },
  { "쩒", "쩒", "쩒", "쩒", "쩒" },
  { "쩓", "쩓", "쩓", "쩓", "쩓" },
  { "쩔", "쩔", "쩔", "쩔", "쩔" },
  { "쩕", "쩕", "쩕", "쩕", "쩕" },
  { "쩖", "쩖", "쩖", "쩖", "쩖" },
  { "쩗", "쩗", "쩗", "쩗", "쩗" },
  { "쩘", "쩘", "쩘", "쩘", "쩘" },
  { "쩙", "쩙", "쩙", "쩙", "쩙" },
  { "쩚", "쩚", "쩚", "쩚", "쩚" },
  { "쩛", "쩛", "쩛", "쩛", "쩛" },
  { "쩜", "쩜", "쩜", "쩜", "쩜" },
  { "쩝", "쩝", "쩝", "쩝", "쩝" },
  { "쩞", "쩞", "쩞", "쩞", "쩞" },
  { "쩟", "쩟", "쩟", "쩟", "쩟" },
  { "쩠", "쩠", "쩠", "쩠", "쩠" },
  { "쩡", "쩡", "쩡", "쩡", "쩡" },
  { "쩢", "쩢", "쩢", "쩢", "쩢" },
  { "쩣", "쩣", "쩣", "쩣", "쩣" },
  { "쩤", "쩤", "쩤", "쩤", "쩤" },
  { "쩥", "쩥", "쩥", "쩥", "쩥" },
  { "쩦", "쩦", "쩦", "쩦", "쩦" },
  { "쩧", "쩧", "쩧", "쩧", "쩧" },
  { "쩨", "쩨", "쩨", "쩨", "쩨" },
  { "쩩", "쩩", "쩩", "쩩", "쩩" },
  { "쩪", "쩪", "쩪", "쩪", "쩪" },
  { "쩫", "쩫", "쩫", "쩫", "쩫" },
  { "쩬", "쩬", "쩬", "쩬", "쩬" },
  { "쩭", "쩭", "쩭", "쩭", "쩭" },
  { "쩮", "쩮", "쩮", "쩮", "쩮" },
  { "쩯", "쩯", "쩯", "쩯", "쩯" },
  { "쩰", "쩰", "쩰", "쩰", "쩰" },
  { "쩱", "쩱", "쩱", "쩱", "쩱" },
  { "쩲", "쩲", "쩲", "쩲", "쩲" },
  { "쩳", "쩳", "쩳", "쩳", "쩳" },
  { "쩴", "쩴", "쩴", "쩴", "쩴" },
  { "쩵", "쩵", "쩵", "쩵", "쩵" },
  { "쩶", "쩶", "쩶", "쩶", "쩶" },
  { "쩷", "쩷", "쩷", "쩷", "쩷" },
  { "쩸", "쩸", "쩸", "쩸", "쩸" },
  { "쩹", "쩹", "쩹", "쩹", "쩹" },
  { "쩺", "쩺", "쩺", "쩺", "쩺" },
  { "쩻", "쩻", "쩻", "쩻", "쩻" },
  { "쩼", "쩼", "쩼", "쩼", "쩼" },
  { "쩽", "쩽", "쩽", "쩽", "쩽" },
  { "쩾", "쩾", "쩾", "쩾", "쩾" },
  { "쩿", "쩿", "쩿", "쩿", "쩿" },
  { "쪀", "쪀", "쪀", "쪀", "쪀" },
  { "쪁", "쪁", "쪁", "쪁", "쪁" },
  { "쪂", "쪂", "쪂", "쪂", "쪂" },
  { "쪃", "쪃", "쪃", "쪃", "쪃" },
  { "쪄", "쪄", "쪄", "쪄", "쪄" },
  { "쪅", "쪅", "쪅", "쪅", "쪅" },
  { "쪆", "쪆", "쪆", "쪆", "쪆" },
  { "쪇", "쪇", "쪇", "쪇", "쪇" },
  { "쪈", "쪈", "쪈", "쪈", "쪈" },
  { "쪉", "쪉", "쪉", "쪉", "쪉" },
  { "쪊", "쪊", "쪊", "쪊", "쪊" },
  { "쪋", "쪋", "쪋", "쪋", "쪋" },
  { "쪌", "쪌", "쪌", "쪌", "쪌" },
  { "쪍", "쪍", "쪍", "쪍", "쪍" },
  { "쪎", "쪎", "쪎", "쪎", "쪎" },
  { "쪏", "쪏", "쪏", "쪏", "쪏" },
  { "쪐", "쪐", "쪐", "쪐", "쪐" },
  { "쪑", "쪑", "쪑", "쪑", "쪑" },
  { "쪒", "쪒", "쪒", "쪒", "쪒" },
  { "쪓", "쪓", "쪓", "쪓", "쪓" },
  { "쪔", "쪔", "쪔", "쪔", "쪔" },
  { "쪕", "쪕", "쪕", "쪕", "쪕" },
  { "쪖", "쪖", "쪖", "쪖", "쪖" },
  { "쪗", "쪗", "쪗", "쪗", "쪗" },
  { "쪘", "쪘", "쪘", "쪘", "쪘" },
  { "쪙", "쪙", "쪙", "쪙", "쪙" },
  { "쪚", "쪚", "쪚", "쪚", "쪚" },
  { "쪛", "쪛", "쪛", "쪛", "쪛" },
  { "쪜", "쪜", "쪜", "쪜", "쪜" },
  { "쪝", "쪝", "쪝", "쪝", "쪝" },
  { "쪞", "쪞", "쪞", "쪞", "쪞" },
  { "쪟", "쪟", "쪟", "쪟", "쪟" },
  { "쪠", "쪠", "쪠", "쪠", "쪠" },
  { "쪡", "쪡", "쪡", "쪡", "쪡" },
  { "쪢", "쪢", "쪢", "쪢", "쪢" },
  { "쪣", "쪣", "쪣", "쪣", "쪣" },
  { "쪤", "쪤", "쪤", "쪤", "쪤" },
  { "쪥", "쪥", "쪥", "쪥", "쪥" },
  { "쪦", "쪦", "쪦", "쪦", "쪦" },
  { "쪧", "쪧", "쪧", "쪧", "쪧" },
  { "쪨", "쪨", "쪨", "쪨", "쪨" },
  { "쪩", "쪩", "쪩", "쪩", "쪩" },
  { "쪪", "쪪", "쪪", "쪪", "쪪" },
  { "쪫", "쪫", "쪫", "쪫", "쪫" },
  { "쪬", "쪬", "쪬", "쪬", "쪬" },
  { "쪭", "쪭", "쪭", "쪭", "쪭" },
  { "쪮", "쪮", "쪮", "쪮", "쪮" },
  { "쪯", "쪯", "쪯", "쪯", "쪯" },
  { "쪰", "쪰", "쪰", "쪰", "쪰" },
  { "쪱", "쪱", "쪱", "쪱", "쪱" },
  { "쪲", "쪲", "쪲", "쪲", "쪲" },
  { "쪳", "쪳", "쪳", "쪳", "쪳" },
  { "쪴", "쪴", "쪴", "쪴", "쪴" },
  { "쪵", "쪵", "쪵", "쪵", "쪵" },
  { "쪶", "쪶", "쪶", "쪶", "쪶" },
  { "쪷", "쪷", "쪷", "쪷", "쪷" },
  { "쪸", "쪸", "쪸", "쪸", "쪸" },
  { "쪹", "쪹", "쪹", "쪹", "쪹" },
  { "쪺", "쪺", "쪺", "쪺", "쪺" },
  { "쪻", "쪻", "쪻", "쪻", "쪻" },
  { "쪼", "쪼", "쪼", "쪼", "쪼" },
  { "쪽", "쪽", "쪽", "쪽", "쪽" },
  { "쪾", "쪾", "쪾", "쪾", "쪾" },
  { "쪿", "쪿", "쪿", "쪿", "쪿" },
  { "쫀", "쫀", "쫀", "쫀", "쫀" },
  { "쫁", "쫁", "쫁", "쫁", "쫁" },
  { "쫂", "쫂", "쫂", "쫂", "쫂" },
  { "쫃", "쫃", "쫃", "쫃", "쫃" },
  { "쫄", "쫄", "쫄", "쫄", "쫄" },
  { "쫅", "쫅", "쫅", "쫅", "쫅" },
  { "쫆", "쫆", "쫆", "쫆", "쫆" },
  { "쫇", "쫇", "쫇", "쫇", "쫇" },
  { "쫈", "쫈", "쫈", "쫈", "쫈" },
  { "쫉", "쫉", "쫉", "쫉", "쫉" },
  { "쫊", "쫊", "쫊", "쫊", "쫊" },
  { "쫋", "쫋", "쫋", "쫋", "쫋" },
  { "쫌", "쫌", "쫌", "쫌", "쫌" },
  { "쫍", "쫍", "쫍", "쫍", "쫍" },
  { "쫎", "쫎", "쫎", "쫎", "쫎" },
  { "쫏", "쫏", "쫏", "쫏", "쫏" },
  { "쫐", "쫐", "쫐", "쫐", "쫐" },
  { "쫑", "쫑", "쫑", "쫑", "쫑" },
  { "쫒", "쫒", "쫒", "쫒", "쫒" },
  { "쫓", "쫓", "쫓", "쫓", "쫓" },
  { "쫔", "쫔", "쫔", "쫔", "쫔" },
  { "쫕", "쫕", "쫕", "쫕", "쫕" },
  { "쫖", "쫖", "쫖", "쫖", "쫖" },
  { "쫗", "쫗", "쫗", "쫗", "쫗" },
  { "쫘", "쫘", "쫘", "쫘", "쫘" },
  { "쫙", "쫙", "쫙", "쫙", "쫙" },
  { "쫚", "쫚", "쫚", "쫚", "쫚" },
  { "쫛", "쫛", "쫛", "쫛", "쫛" },
  { "쫜", "쫜", "쫜", "쫜", "쫜" },
  { "쫝", "쫝", "쫝", "쫝", "쫝" },
  { "쫞", "쫞", "쫞", "쫞", "쫞" },
  { "쫟", "쫟", "쫟", "쫟", "쫟" },
  { "쫠", "쫠", "쫠", "쫠", "쫠" },
  { "쫡", "쫡", "쫡", "쫡", "쫡" },
  { "쫢", "쫢", "쫢", "쫢", "쫢" },
  { "쫣", "쫣", "쫣", "쫣", "쫣" },
  { "쫤", "쫤", "쫤", "쫤", "쫤" },
  { "쫥", "쫥", "쫥", "쫥", "쫥" },
  { "쫦", "쫦", "쫦", "쫦", "쫦" },
  { "쫧", "쫧", "쫧", "쫧", "쫧" },
  { "쫨", "쫨", "쫨", "쫨", "쫨" },
  { "쫩", "쫩", "쫩", "쫩", "쫩" },
  { "쫪", "쫪", "쫪", "쫪", "쫪" },
  { "쫫", "쫫", "쫫", "쫫", "쫫" },
  { "쫬", "쫬", "쫬", "쫬", "쫬" },
  { "쫭", "쫭", "쫭", "쫭", "쫭" },
  { "쫮", "쫮", "쫮", "쫮", "쫮" },
  { "쫯", "쫯", "쫯", "쫯", "쫯" },
  { "쫰", "쫰", "쫰", "쫰", "쫰" },
  { "쫱", "쫱", "쫱", "쫱", "쫱" },
  { "쫲", "쫲", "쫲", "쫲", "쫲" },
  { "쫳", "쫳", "쫳", "쫳", "쫳" },
  { "쫴", "쫴", "쫴", "쫴", "쫴" },
  { "쫵", "쫵", "쫵", "쫵", "쫵" },
  { "쫶", "쫶", "쫶", "쫶", "쫶" },
  { "쫷", "쫷", "쫷", "쫷", "쫷" },
  { "쫸", "쫸", "쫸", "쫸", "쫸" },
  { "쫹", "쫹", "쫹", "쫹", "쫹" },
  { "쫺", "쫺", "쫺", "쫺", "쫺" },
  { "쫻", "쫻", "쫻", "쫻", "쫻" },
  { "쫼", "쫼", "쫼", "쫼", "쫼" },
  { "쫽", "쫽", "쫽", "쫽", "쫽" },
  { "쫾", "쫾", "쫾", "쫾", "쫾" },
  { "쫿", "쫿", "쫿", "쫿", "쫿" },
  { "쬀", "쬀", "쬀", "쬀", "쬀" },
  { "쬁", "쬁", "쬁", "쬁", "쬁" },
  { "쬂", "쬂", "쬂", "쬂", "쬂" },
  { "쬃", "쬃", "쬃", "쬃", "쬃" },
  { "쬄", "쬄", "쬄", "쬄", "쬄" },
  { "쬅", "쬅", "쬅", "쬅", "쬅" },
  { "쬆", "쬆", "쬆", "쬆", "쬆" },
  { "쬇", "쬇", "쬇", "쬇", "쬇" },
  { "쬈", "쬈", "쬈", "쬈", "쬈" },
  { "쬉", "쬉", "쬉", "쬉", "쬉" },
  { "쬊", "쬊", "쬊", "쬊", "쬊" },
  { "쬋", "쬋", "쬋", "쬋", "쬋" },
  { "쬌", "쬌", "쬌", "쬌", "쬌" },
  { "쬍", "쬍", "쬍", "쬍", "쬍" },
  { "쬎", "쬎", "쬎", "쬎", "쬎" },
  { "쬏", "쬏", "쬏", "쬏", "쬏" },
  { "쬐", "쬐", "쬐", "쬐", "쬐" },
  { "쬑", "쬑", "쬑", "쬑", "쬑" },
  { "쬒", "쬒", "쬒", "쬒", "쬒" },
  { "쬓", "쬓", "쬓", "쬓", "쬓" },
  { "쬔", "쬔", "쬔", "쬔", "쬔" },
  { "쬕", "쬕", "쬕", "쬕", "쬕" },
  { "쬖", "쬖", "쬖", "쬖", "쬖" },
  { "쬗", "쬗", "쬗", "쬗", "쬗" },
  { "쬘", "쬘", "쬘", "쬘", "쬘" },
  { "쬙", "쬙", "쬙", "쬙", "쬙" },
  { "쬚", "쬚", "쬚", "쬚", "쬚" },
  { "쬛", "쬛", "쬛", "쬛", "쬛" },
  { "쬜", "쬜", "쬜", "쬜", "쬜" },
  { "쬝", "쬝", "쬝", "쬝", "쬝" },
  { "쬞", "쬞", "쬞", "쬞", "쬞" },
  { "쬟", "쬟", "쬟", "쬟", "쬟" },
  { "쬠", "쬠", "쬠", "쬠", "쬠" },
  { "쬡", "쬡", "쬡", "쬡", "쬡" },
  { "쬢", "쬢", "쬢", "쬢", "쬢" },
  { "쬣", "쬣", "쬣", "쬣", "쬣" },
  { "쬤", "쬤", "쬤", "쬤", "쬤" },
  { "쬥", "쬥", "쬥", "쬥", "쬥" },
  { "쬦", "쬦", "쬦", "쬦", "쬦" },
  { "쬧", "쬧", "쬧", "쬧", "쬧" },
  { "쬨", "쬨", "쬨", "쬨", "쬨" },
  { "쬩", "쬩", "쬩", "쬩", "쬩" },
  { "쬪", "쬪", "쬪", "쬪", "쬪" },
  { "쬫", "쬫", "쬫", "쬫", "쬫" },
  { "쬬", "쬬", "쬬", "쬬", "쬬" },
  { "쬭", "쬭", "쬭", "쬭", "쬭" },
  { "쬮", "쬮", "쬮", "쬮", "쬮" },
  { "쬯", "쬯", "쬯", "쬯", "쬯" },
  { "쬰", "쬰", "쬰", "쬰", "쬰" },
  { "쬱", "쬱", "쬱", "쬱", "쬱" },
  { "쬲", "쬲", "쬲", "쬲", "쬲" },
  { "쬳", "쬳", "쬳", "쬳", "쬳" },
  { "쬴", "쬴", "쬴", "쬴", "쬴" },
  { "쬵", "쬵", "쬵", "쬵", "쬵" },
  { "쬶", "쬶", "쬶", "쬶", "쬶" },
  { "쬷", "쬷", "쬷", "쬷", "쬷" },
  { "쬸", "쬸", "쬸", "쬸", "쬸" },
  { "쬹", "쬹", "쬹", "쬹", "쬹" },
  { "쬺", "쬺", "쬺", "쬺", "쬺" },
  { "쬻", "쬻", "쬻", "쬻", "쬻" },
  { "쬼", "쬼", "쬼", "쬼", "쬼" },
  { "쬽", "쬽", "쬽", "쬽", "쬽" },
  { "쬾", "쬾", "쬾", "쬾", "쬾" },
  { "쬿", "쬿", "쬿", "쬿", "쬿" },
  { "쭀", "쭀", "쭀", "쭀", "쭀" },
  { "쭁", "쭁", "쭁", "쭁", "쭁" },
  { "쭂", "쭂", "쭂", "쭂", "쭂" },
  { "쭃", "쭃", "쭃", "쭃", "쭃" },
  { "쭄", "쭄", "쭄", "쭄", "쭄" },
  { "쭅", "쭅", "쭅", "쭅", "쭅" },
  { "쭆", "쭆", "쭆", "쭆", "쭆" },
  { "쭇", "쭇", "쭇", "쭇", "쭇" },
  { "쭈", "쭈", "쭈", "쭈", "쭈" },
  { "쭉", "쭉", "쭉", "쭉", "쭉" },
  { "쭊", "쭊", "쭊", "쭊", "쭊" },
  { "쭋", "쭋", "쭋", "쭋", "쭋" },
  { "쭌", "쭌", "쭌", "쭌", "쭌" },
  { "쭍", "쭍", "쭍", "쭍", "쭍" },
  { "쭎", "쭎", "쭎", "쭎", "쭎" },
  { "쭏", "쭏", "쭏", "쭏", "쭏" },
  { "쭐", "쭐", "쭐", "쭐", "쭐" },
  { "쭑", "쭑", "쭑", "쭑", "쭑" },
  { "쭒", "쭒", "쭒", "쭒", "쭒" },
  { "쭓", "쭓", "쭓", "쭓", "쭓" },
  { "쭔", "쭔", "쭔", "쭔", "쭔" },
  { "쭕", "쭕", "쭕", "쭕", "쭕" },
  { "쭖", "쭖", "쭖", "쭖", "쭖" },
  { "쭗", "쭗", "쭗", "쭗", "쭗" },
  { "쭘", "쭘", "쭘", "쭘", "쭘" },
  { "쭙", "쭙", "쭙", "쭙", "쭙" },
  { "쭚", "쭚", "쭚", "쭚", "쭚" },
  { "쭛", "쭛", "쭛", "쭛", "쭛" },
  { "쭜", "쭜", "쭜", "쭜", "쭜" },
  { "쭝", "쭝", "쭝", "쭝", "쭝" },
  { "쭞", "쭞", "쭞", "쭞", "쭞" },
  { "쭟", "쭟", "쭟", "쭟", "쭟" },
  { "쭠", "쭠", "쭠", "쭠", "쭠" },
  { "쭡", "쭡", "쭡", "쭡", "쭡" },
  { "쭢", "쭢", "쭢", "쭢", "쭢" },
  { "쭣", "쭣", "쭣", "쭣", "쭣" },
  { "쭤", "쭤", "쭤", "쭤", "쭤" },
  { "쭥", "쭥", "쭥", "쭥", "쭥" },
  { "쭦", "쭦", "쭦", "쭦", "쭦" },
  { "쭧", "쭧", "쭧", "쭧", "쭧" },
  { "쭨", "쭨", "쭨", "쭨", "쭨" },
  { "쭩", "쭩", "쭩", "쭩", "쭩" },
  { "쭪", "쭪", "쭪", "쭪", "쭪" },
  { "쭫", "쭫", "쭫", "쭫", "쭫" },
  { "쭬", "쭬", "쭬", "쭬", "쭬" },
  { "쭭", "쭭", "쭭", "쭭", "쭭" },
  { "쭮", "쭮", "쭮", "쭮", "쭮" },
  { "쭯", "쭯", "쭯", "쭯", "쭯" },
  { "쭰", "쭰", "쭰", "쭰", "쭰" },
  { "쭱", "쭱", "쭱", "쭱", "쭱" },
  { "쭲", "쭲", "쭲", "쭲", "쭲" },
  { "쭳", "쭳", "쭳", "쭳", "쭳" },
  { "쭴", "쭴", "쭴", "쭴", "쭴" },
  { "쭵", "쭵", "쭵", "쭵", "쭵" },
  { "쭶", "쭶", "쭶", "쭶", "쭶" },
  { "쭷", "쭷", "쭷", "쭷", "쭷" },
  { "쭸", "쭸", "쭸", "쭸", "쭸" },
  { "쭹", "쭹", "쭹", "쭹", "쭹" },
  { "쭺", "쭺", "쭺", "쭺", "쭺" },
  { "쭻", "쭻", "쭻", "쭻", "쭻" },
  { "쭼", "쭼", "쭼", "쭼", "쭼" },
  { "쭽", "쭽", "쭽", "쭽", "쭽" },
  { "쭾", "쭾", "쭾", "쭾", "쭾" },
  { "쭿", "쭿", "쭿", "쭿", "쭿" },
  { "쮀", "쮀", "쮀", "쮀", "쮀" },
  { "쮁", "쮁", "쮁", "쮁", "쮁" },
  { "쮂", "쮂", "쮂", "쮂", "쮂" },
  { "쮃", "쮃", "쮃", "쮃", "쮃" },
  { "쮄", "쮄", "쮄", "쮄", "쮄" },
  { "쮅", "쮅", "쮅", "쮅", "쮅" },
  { "쮆", "쮆", "쮆", "쮆", "쮆" },
  { "쮇", "쮇", "쮇", "쮇", "쮇" },
  { "쮈", "쮈", "쮈", "쮈", "쮈" },
  { "쮉", "쮉", "쮉", "쮉", "쮉" },
  { "쮊", "쮊", "쮊", "쮊", "쮊" },
  { "쮋", "쮋", "쮋", "쮋", "쮋" },
  { "쮌", "쮌", "쮌", "쮌", "쮌" },
  { "쮍", "쮍", "쮍", "쮍", "쮍" },
  { "쮎", "쮎", "쮎", "쮎", "쮎" },
  { "쮏", "쮏", "쮏", "쮏", "쮏" },
  { "쮐", "쮐", "쮐", "쮐", "쮐" },
  { "쮑", "쮑", "쮑", "쮑", "쮑" },
  { "쮒", "쮒", "쮒", "쮒", "쮒" },
  { "쮓", "쮓", "쮓", "쮓", "쮓" },
  { "쮔", "쮔", "쮔", "쮔", "쮔" },
  { "쮕", "쮕", "쮕", "쮕", "쮕" },
  { "쮖", "쮖", "쮖", "쮖", "쮖" },
  { "쮗", "쮗", "쮗", "쮗", "쮗" },
  { "쮘", "쮘", "쮘", "쮘", "쮘" },
  { "쮙", "쮙", "쮙", "쮙", "쮙" },
  { "쮚", "쮚", "쮚", "쮚", "쮚" },
  { "쮛", "쮛", "쮛", "쮛", "쮛" },
  { "쮜", "쮜", "쮜", "쮜", "쮜" },
  { "쮝", "쮝", "쮝", "쮝", "쮝" },
  { "쮞", "쮞", "쮞", "쮞", "쮞" },
  { "쮟", "쮟", "쮟", "쮟", "쮟" },
  { "쮠", "쮠", "쮠", "쮠", "쮠" },
  { "쮡", "쮡", "쮡", "쮡", "쮡" },
  { "쮢", "쮢", "쮢", "쮢", "쮢" },
  { "쮣", "쮣", "쮣", "쮣", "쮣" },
  { "쮤", "쮤", "쮤", "쮤", "쮤" },
  { "쮥", "쮥", "쮥", "쮥", "쮥" },
  { "쮦", "쮦", "쮦", "쮦", "쮦" },
  { "쮧", "쮧", "쮧", "쮧", "쮧" },
  { "쮨", "쮨", "쮨", "쮨", "쮨" },
  { "쮩", "쮩", "쮩", "쮩", "쮩" },
  { "쮪", "쮪", "쮪", "쮪", "쮪" },
  { "쮫", "쮫", "쮫", "쮫", "쮫" },
  { "쮬", "쮬", "쮬", "쮬", "쮬" },
  { "쮭", "쮭", "쮭", "쮭", "쮭" },
  { "쮮", "쮮", "쮮", "쮮", "쮮" },
  { "쮯", "쮯", "쮯", "쮯", "쮯" },
  { "쮰", "쮰", "쮰", "쮰", "쮰" },
  { "쮱", "쮱", "쮱", "쮱", "쮱" },
  { "쮲", "쮲", "쮲", "쮲", "쮲" },
  { "쮳", "쮳", "쮳", "쮳", "쮳" },
  { "쮴", "쮴", "쮴", "쮴", "쮴" },
  { "쮵", "쮵", "쮵", "쮵", "쮵" },
  { "쮶", "쮶", "쮶", "쮶", "쮶" },
  { "쮷", "쮷", "쮷", "쮷", "쮷" },
  { "쮸", "쮸", "쮸", "쮸", "쮸" },
  { "쮹", "쮹", "쮹", "쮹", "쮹" },
  { "쮺", "쮺", "쮺", "쮺", "쮺" },
  { "쮻", "쮻", "쮻", "쮻", "쮻" },
  { "쮼", "쮼", "쮼", "쮼", "쮼" },
  { "쮽", "쮽", "쮽", "쮽", "쮽" },
  { "쮾", "쮾", "쮾", "쮾", "쮾" },
  { "쮿", "쮿", "쮿", "쮿", "쮿" },
  { "쯀", "쯀", "쯀", "쯀", "쯀" },
  { "쯁", "쯁", "쯁", "쯁", "쯁" },
  { "쯂", "쯂", "쯂", "쯂", "쯂" },
  { "쯃", "쯃", "쯃", "쯃", "쯃" },
  { "쯄", "쯄", "쯄", "쯄", "쯄" },
  { "쯅", "쯅", "쯅", "쯅", "쯅" },
  { "쯆", "쯆", "쯆", "쯆", "쯆" },
  { "쯇", "쯇", "쯇", "쯇", "쯇" },
  { "쯈", "쯈", "쯈", "쯈", "쯈" },
  { "쯉", "쯉", "쯉", "쯉", "쯉" },
  { "쯊", "쯊", "쯊", "쯊", "쯊" },
  { "쯋", "쯋", "쯋", "쯋", "쯋" },
  { "쯌", "쯌", "쯌", "쯌", "쯌" },
  { "쯍", "쯍", "쯍", "쯍", "쯍" },
  { "쯎", "쯎", "쯎", "쯎", "쯎" },
  { "쯏", "쯏", "쯏", "쯏", "쯏" },
  { "쯐", "쯐", "쯐", "쯐", "쯐" },
  { "쯑", "쯑", "쯑", "쯑", "쯑" },
  { "쯒", "쯒", "쯒", "쯒", "쯒" },
  { "쯓", "쯓", "쯓", "쯓", "쯓" },
  { "쯔", "쯔", "쯔", "쯔", "쯔" },
  { "쯕", "쯕", "쯕", "쯕", "쯕" },
  { "쯖", "쯖", "쯖", "쯖", "쯖" },
  { "쯗", "쯗", "쯗", "쯗", "쯗" },
  { "쯘", "쯘", "쯘", "쯘", "쯘" },
  { "쯙", "쯙", "쯙", "쯙", "쯙" },
  { "쯚", "쯚", "쯚", "쯚", "쯚" },
  { "쯛", "쯛", "쯛", "쯛", "쯛" },
  { "쯜", "쯜", "쯜", "쯜", "쯜" },
  { "쯝", "쯝", "쯝", "쯝", "쯝" },
  { "쯞", "쯞", "쯞", "쯞", "쯞" },
  { "쯟", "쯟", "쯟", "쯟", "쯟" },
  { "쯠", "쯠", "쯠", "쯠", "쯠" },
  { "쯡", "쯡", "쯡", "쯡", "쯡" },
  { "쯢", "쯢", "쯢", "쯢", "쯢" },
  { "쯣", "쯣", "쯣", "쯣", "쯣" },
  { "쯤", "쯤", "쯤", "쯤", "쯤" },
  { "쯥", "쯥", "쯥", "쯥", "쯥" },
  { "쯦", "쯦", "쯦", "쯦", "쯦" },
  { "쯧", "쯧", "쯧", "쯧", "쯧" },
  { "쯨", "쯨", "쯨", "쯨", "쯨" },
  { "쯩", "쯩", "쯩", "쯩", "쯩" },
  { "쯪", "쯪", "쯪", "쯪", "쯪" },
  { "쯫", "쯫", "쯫", "쯫", "쯫" },
  { "쯬", "쯬", "쯬", "쯬", "쯬" },
  { "쯭", "쯭", "쯭", "쯭", "쯭" },
  { "쯮", "쯮", "쯮", "쯮", "쯮" },
  { "쯯", "쯯", "쯯", "쯯", "쯯" },
  { "쯰", "쯰", "쯰", "쯰", "쯰" },
  { "쯱", "쯱", "쯱", "쯱", "쯱" },
  { "쯲", "쯲", "쯲", "쯲", "쯲" },
  { "쯳", "쯳", "쯳", "쯳", "쯳" },
  { "쯴", "쯴", "쯴", "쯴", "쯴" },
  { "쯵", "쯵", "쯵", "쯵", "쯵" },
  { "쯶", "쯶", "쯶", "쯶", "쯶" },
  { "쯷", "쯷", "쯷", "쯷", "쯷" },
  { "쯸", "쯸", "쯸", "쯸", "쯸" },
  { "쯹", "쯹", "쯹", "쯹", "쯹" },
  { "쯺", "쯺", "쯺", "쯺", "쯺" },
  { "쯻", "쯻", "쯻", "쯻", "쯻" },
  { "쯼", "쯼", "쯼", "쯼", "쯼" },
  { "쯽", "쯽", "쯽", "쯽", "쯽" },
  { "쯾", "쯾", "쯾", "쯾", "쯾" },
  { "쯿", "쯿", "쯿", "쯿", "쯿" },
  { "찀", "찀", "찀", "찀", "찀" },
  { "찁", "찁", "찁", "찁", "찁" },
  { "찂", "찂", "찂", "찂", "찂" },
  { "찃", "찃", "찃", "찃", "찃" },
  { "찄", "찄", "찄", "찄", "찄" },
  { "찅", "찅", "찅", "찅", "찅" },
  { "찆", "찆", "찆", "찆", "찆" },
  { "찇", "찇", "찇", "찇", "찇" },
  { "찈", "찈", "찈", "찈", "찈" },
  { "찉", "찉", "찉", "찉", "찉" },
  { "찊", "찊", "찊", "찊", "찊" },
  { "찋", "찋", "찋", "찋", "찋" },
  { "찌", "찌", "찌", "찌", "찌" },
  { "찍", "찍", "찍", "찍", "찍" },
  { "찎", "찎", "찎", "찎", "찎" },
  { "찏", "찏", "찏", "찏", "찏" },
  { "찐", "찐", "찐", "찐", "찐" },
  { "찑", "찑", "찑", "찑", "찑" },
  { "찒", "찒", "찒", "찒", "찒" },
  { "찓", "찓", "찓", "찓", "찓" },
  { "찔", "찔", "찔", "찔", "찔" },
  { "찕", "찕", "찕", "찕", "찕" },
  { "찖", "찖", "찖", "찖", "찖" },
  { "찗", "찗", "찗", "찗", "찗" },
  { "찘", "찘", "찘", "찘", "찘" },
  { "찙", "찙", "찙", "찙", "찙" },
  { "찚", "찚", "찚", "찚", "찚" },
  { "찛", "찛", "찛", "찛", "찛" },
  { "찜", "찜", "찜", "찜", "찜" },
  { "찝", "찝", "찝", "찝", "찝" },
  { "찞", "찞", "찞", "찞", "찞" },
  { "찟", "찟", "찟", "찟", "찟" },
  { "찠", "찠", "찠", "찠", "찠" },
  { "찡", "찡", "찡", "찡", "찡" },
  { "찢", "찢", "찢", "찢", "찢" },
  { "찣", "찣", "찣", "찣", "찣" },
  { "찤", "찤", "찤", "찤", "찤" },
  { "찥", "찥", "찥", "찥", "찥" },
  { "찦", "찦", "찦", "찦", "찦" },
  { "찧", "찧", "찧", "찧", "찧" },
  { "차", "차", "차", "차", "차" },
  { "착", "착", "착", "착", "착" },
  { "찪", "찪", "찪", "찪", "찪" },
  { "찫", "찫", "찫", "찫", "찫" },
  { "찬", "찬", "찬", "찬", "찬" },
  { "찭", "찭", "찭", "찭", "찭" },
  { "찮", "찮", "찮", "찮", "찮" },
  { "찯", "찯", "찯", "찯", "찯" },
  { "찰", "찰", "찰", "찰", "찰" },
  { "찱", "찱", "찱", "찱", "찱" },
  { "찲", "찲", "찲", "찲", "찲" },
  { "찳", "찳", "찳", "찳", "찳" },
  { "찴", "찴", "찴", "찴", "찴" },
  { "찵", "찵", "찵", "찵", "찵" },
  { "찶", "찶", "찶", "찶", "찶" },
  { "찷", "찷", "찷", "찷", "찷" },
  { "참", "참", "참", "참", "참" },
  { "찹", "찹", "찹", "찹", "찹" },
  { "찺", "찺", "찺", "찺", "찺" },
  { "찻", "찻", "찻", "찻", "찻" },
  { "찼", "찼", "찼", "찼", "찼" },
  { "창", "창", "창", "창", "창" },
  { "찾", "찾", "찾", "찾", "찾" },
  { "찿", "찿", "찿", "찿", "찿" },
  { "챀", "챀", "챀", "챀", "챀" },
  { "챁", "챁", "챁", "챁", "챁" },
  { "챂", "챂", "챂", "챂", "챂" },
  { "챃", "챃", "챃", "챃", "챃" },
  { "채", "채", "채", "채", "채" },
  { "책", "책", "책", "책", "책" },
  { "챆", "챆", "챆", "챆", "챆" },
  { "챇", "챇", "챇", "챇", "챇" },
  { "챈", "챈", "챈", "챈", "챈" },
  { "챉", "챉", "챉", "챉", "챉" },
  { "챊", "챊", "챊", "챊", "챊" },
  { "챋", "챋", "챋", "챋", "챋" },
  { "챌", "챌", "챌", "챌", "챌" },
  { "챍", "챍", "챍", "챍", "챍" },
  { "챎", "챎", "챎", "챎", "챎" },
  { "챏", "챏", "챏", "챏", "챏" },
  { "챐", "챐", "챐", "챐", "챐" },
  { "챑", "챑", "챑", "챑", "챑" },
  { "챒", "챒", "챒", "챒", "챒" },
  { "챓", "챓", "챓", "챓", "챓" },
  { "챔", "챔", "챔", "챔", "챔" },
  { "챕", "챕", "챕", "챕", "챕" },
  { "챖", "챖", "챖", "챖", "챖" },
  { "챗", "챗", "챗", "챗", "챗" },
  { "챘", "챘", "챘", "챘", "챘" },
  { "챙", "챙", "챙", "챙", "챙" },
  { "챚", "챚", "챚", "챚", "챚" },
  { "챛", "챛", "챛", "챛", "챛" },
  { "챜", "챜", "챜", "챜", "챜" },
  { "챝", "챝", "챝", "챝", "챝" },
  { "챞", "챞", "챞", "챞", "챞" },
  { "챟", "챟", "챟", "챟", "챟" },
  { "챠", "챠", "챠", "챠", "챠" },
  { "챡", "챡", "챡", "챡", "챡" },
  { "챢", "챢", "챢", "챢", "챢" },
  { "챣", "챣", "챣", "챣", "챣" },
  { "챤", "챤", "챤", "챤", "챤" },
  { "챥", "챥", "챥", "챥", "챥" },
  { "챦", "챦", "챦", "챦", "챦" },
  { "챧", "챧", "챧", "챧", "챧" },
  { "챨", "챨", "챨", "챨", "챨" },
  { "챩", "챩", "챩", "챩", "챩" },
  { "챪", "챪", "챪", "챪", "챪" },
  { "챫", "챫", "챫", "챫", "챫" },
  { "챬", "챬", "챬", "챬", "챬" },
  { "챭", "챭", "챭", "챭", "챭" },
  { "챮", "챮", "챮", "챮", "챮" },
  { "챯", "챯", "챯", "챯", "챯" },
  { "챰", "챰", "챰", "챰", "챰" },
  { "챱", "챱", "챱", "챱", "챱" },
  { "챲", "챲", "챲", "챲", "챲" },
  { "챳", "챳", "챳", "챳", "챳" },
  { "챴", "챴", "챴", "챴", "챴" },
  { "챵", "챵", "챵", "챵", "챵" },
  { "챶", "챶", "챶", "챶", "챶" },
  { "챷", "챷", "챷", "챷", "챷" },
  { "챸", "챸", "챸", "챸", "챸" },
  { "챹", "챹", "챹", "챹", "챹" },
  { "챺", "챺", "챺", "챺", "챺" },
  { "챻", "챻", "챻", "챻", "챻" },
  { "챼", "챼", "챼", "챼", "챼" },
  { "챽", "챽", "챽", "챽", "챽" },
  { "챾", "챾", "챾", "챾", "챾" },
  { "챿", "챿", "챿", "챿", "챿" },
  { "첀", "첀", "첀", "첀", "첀" },
  { "첁", "첁", "첁", "첁", "첁" },
  { "첂", "첂", "첂", "첂", "첂" },
  { "첃", "첃", "첃", "첃", "첃" },
  { "첄", "첄", "첄", "첄", "첄" },
  { "첅", "첅", "첅", "첅", "첅" },
  { "첆", "첆", "첆", "첆", "첆" },
  { "첇", "첇", "첇", "첇", "첇" },
  { "첈", "첈", "첈", "첈", "첈" },
  { "첉", "첉", "첉", "첉", "첉" },
  { "첊", "첊", "첊", "첊", "첊" },
  { "첋", "첋", "첋", "첋", "첋" },
  { "첌", "첌", "첌", "첌", "첌" },
  { "첍", "첍", "첍", "첍", "첍" },
  { "첎", "첎", "첎", "첎", "첎" },
  { "첏", "첏", "첏", "첏", "첏" },
  { "첐", "첐", "첐", "첐", "첐" },
  { "첑", "첑", "첑", "첑", "첑" },
  { "첒", "첒", "첒", "첒", "첒" },
  { "첓", "첓", "첓", "첓", "첓" },
  { "첔", "첔", "첔", "첔", "첔" },
  { "첕", "첕", "첕", "첕", "첕" },
  { "첖", "첖", "첖", "첖", "첖" },
  { "첗", "첗", "첗", "첗", "첗" },
  { "처", "처", "처", "처", "처" },
  { "척", "척", "척", "척", "척" },
  { "첚", "첚", "첚", "첚", "첚" },
  { "첛", "첛", "첛", "첛", "첛" },
  { "천", "천", "천", "천", "천" },
  { "첝", "첝", "첝", "첝", "첝" },
  { "첞", "첞", "첞", "첞", "첞" },
  { "첟", "첟", "첟", "첟", "첟" },
  { "철", "철", "철", "철", "철" },
  { "첡", "첡", "첡", "첡", "첡" },
  { "첢", "첢", "첢", "첢", "첢" },
  { "첣", "첣", "첣", "첣", "첣" },
  { "첤", "첤", "첤", "첤", "첤" },
  { "첥", "첥", "첥", "첥", "첥" },
  { "첦", "첦", "첦", "첦", "첦" },
  { "첧", "첧", "첧", "첧", "첧" },
  { "첨", "첨", "첨", "첨", "첨" },
  { "첩", "첩", "첩", "첩", "첩" },
  { "첪", "첪", "첪", "첪", "첪" },
  { "첫", "첫", "첫", "첫", "첫" },
  { "첬", "첬", "첬", "첬", "첬" },
  { "청", "청", "청", "청", "청" },
  { "첮", "첮", "첮", "첮", "첮" },
  { "첯", "첯", "첯", "첯", "첯" },
  { "첰", "첰", "첰", "첰", "첰" },
  { "첱", "첱", "첱", "첱", "첱" },
  { "첲", "첲", "첲", "첲", "첲" },
  { "첳", "첳", "첳", "첳", "첳" },
  { "체", "체", "체", "체", "체" },
  { "첵", "첵", "첵", "첵", "첵" },
  { "첶", "첶", "첶", "첶", "첶" },
  { "첷", "첷", "첷", "첷", "첷" },
  { "첸", "첸", "첸", "첸", "첸" },
  { "첹", "첹", "첹", "첹", "첹" },
  { "첺", "첺", "첺", "첺", "첺" },
  { "첻", "첻", "첻", "첻", "첻" },
  { "첼", "첼", "첼", "첼", "첼" },
  { "첽", "첽", "첽", "첽", "첽" },
  { "첾", "첾", "첾", "첾", "첾" },
  { "첿", "첿", "첿", "첿", "첿" },
  { "쳀", "쳀", "쳀", "쳀", "쳀" },
  { "쳁", "쳁", "쳁", "쳁", "쳁" },
  { "쳂", "쳂", "쳂", "쳂", "쳂" },
  { "쳃", "쳃", "쳃", "쳃", "쳃" },
  { "쳄", "쳄", "쳄", "쳄", "쳄" },
  { "쳅", "쳅", "쳅", "쳅", "쳅" },
  { "쳆", "쳆", "쳆", "쳆", "쳆" },
  { "쳇", "쳇", "쳇", "쳇", "쳇" },
  { "쳈", "쳈", "쳈", "쳈", "쳈" },
  { "쳉", "쳉", "쳉", "쳉", "쳉" },
  { "쳊", "쳊", "쳊", "쳊", "쳊" },
  { "쳋", "쳋", "쳋", "쳋", "쳋" },
  { "쳌", "쳌", "쳌", "쳌", "쳌" },
  { "쳍", "쳍", "쳍", "쳍", "쳍" },
  { "쳎", "쳎", "쳎", "쳎", "쳎" },
  { "쳏", "쳏", "쳏", "쳏", "쳏" },
  { "쳐", "쳐", "쳐", "쳐", "쳐" },
  { "쳑", "쳑", "쳑", "쳑", "쳑" },
  { "쳒", "쳒", "쳒", "쳒", "쳒" },
  { "쳓", "쳓", "쳓", "쳓", "쳓" },
  { "쳔", "쳔", "쳔", "쳔", "쳔" },
  { "쳕", "쳕", "쳕", "쳕", "쳕" },
  { "쳖", "쳖", "쳖", "쳖", "쳖" },
  { "쳗", "쳗", "쳗", "쳗", "쳗" },
  { "쳘", "쳘", "쳘", "쳘", "쳘" },
  { "쳙", "쳙", "쳙", "쳙", "쳙" },
  { "쳚", "쳚", "쳚", "쳚", "쳚" },
  { "쳛", "쳛", "쳛", "쳛", "쳛" },
  { "쳜", "쳜", "쳜", "쳜", "쳜" },
  { "쳝", "쳝", "쳝", "쳝", "쳝" },
  { "쳞", "쳞", "쳞", "쳞", "쳞" },
  { "쳟", "쳟", "쳟", "쳟", "쳟" },
  { "쳠", "쳠", "쳠", "쳠", "쳠" },
  { "쳡", "쳡", "쳡", "쳡", "쳡" },
  { "쳢", "쳢", "쳢", "쳢", "쳢" },
  { "쳣", "쳣", "쳣", "쳣", "쳣" },
  { "쳤", "쳤", "쳤", "쳤", "쳤" },
  { "쳥", "쳥", "쳥", "쳥", "쳥" },
  { "쳦", "쳦", "쳦", "쳦", "쳦" },
  { "쳧", "쳧", "쳧", "쳧", "쳧" },
  { "쳨", "쳨", "쳨", "쳨", "쳨" },
  { "쳩", "쳩", "쳩", "쳩", "쳩" },
  { "쳪", "쳪", "쳪", "쳪", "쳪" },
  { "쳫", "쳫", "쳫", "쳫", "쳫" },
  { "쳬", "쳬", "쳬", "쳬", "쳬" },
  { "쳭", "쳭", "쳭", "쳭", "쳭" },
  { "쳮", "쳮", "쳮", "쳮", "쳮" },
  { "쳯", "쳯", "쳯", "쳯", "쳯" },
  { "쳰", "쳰", "쳰", "쳰", "쳰" },
  { "쳱", "쳱", "쳱", "쳱", "쳱" },
  { "쳲", "쳲", "쳲", "쳲", "쳲" },
  { "쳳", "쳳", "쳳", "쳳", "쳳" },
  { "쳴", "쳴", "쳴", "쳴", "쳴" },
  { "쳵", "쳵", "쳵", "쳵", "쳵" },
  { "쳶", "쳶", "쳶", "쳶", "쳶" },
  { "쳷", "쳷", "쳷", "쳷", "쳷" },
  { "쳸", "쳸", "쳸", "쳸", "쳸" },
  { "쳹", "쳹", "쳹", "쳹", "쳹" },
  { "쳺", "쳺", "쳺", "쳺", "쳺" },
  { "쳻", "쳻", "쳻", "쳻", "쳻" },
  { "쳼", "쳼", "쳼", "쳼", "쳼" },
  { "쳽", "쳽", "쳽", "쳽", "쳽" },
  { "쳾", "쳾", "쳾", "쳾", "쳾" },
  { "쳿", "쳿", "쳿", "쳿", "쳿" },
  { "촀", "촀", "촀", "촀", "촀" },
  { "촁", "촁", "촁", "촁", "촁" },
  { "촂", "촂", "촂", "촂", "촂" },
  { "촃", "촃", "촃", "촃", "촃" },
  { "촄", "촄", "촄", "촄", "촄" },
  { "촅", "촅", "촅", "촅", "촅" },
  { "촆", "촆", "촆", "촆", "촆" },
  { "촇", "촇", "촇", "촇", "촇" },
  { "초", "초", "초", "초", "초" },
  { "촉", "촉", "촉", "촉", "촉" },
  { "촊", "촊", "촊", "촊", "촊" },
  { "촋", "촋", "촋", "촋", "촋" },
  { "촌", "촌", "촌", "촌", "촌" },
  { "촍", "촍", "촍", "촍", "촍" },
  { "촎", "촎", "촎", "촎", "촎" },
  { "촏", "촏", "촏", "촏", "촏" },
  { "촐", "촐", "촐", "촐", "촐" },
  { "촑", "촑", "촑", "촑", "촑" },
  { "촒", "촒", "촒", "촒", "촒" },
  { "촓", "촓", "촓", "촓", "촓" },
  { "촔", "촔", "촔", "촔", "촔" },
  { "촕", "촕", "촕", "촕", "촕" },
  { "촖", "촖", "촖", "촖", "촖" },
  { "촗", "촗", "촗", "촗", "촗" },
  { "촘", "촘", "촘", "촘", "촘" },
  { "촙", "촙", "촙", "촙", "촙" },
  { "촚", "촚", "촚", "촚", "촚" },
  { "촛", "촛", "촛", "촛", "촛" },
  { "촜", "촜", "촜", "촜", "촜" },
  { "총", "총", "총", "총", "총" },
  { "촞", "촞", "촞", "촞", "촞" },
  { "촟", "촟", "촟", "촟", "촟" },
  { "촠", "촠", "촠", "촠", "촠" },
  { "촡", "촡", "촡", "촡", "촡" },
  { "촢", "촢", "촢", "촢", "촢" },
  { "촣", "촣", "촣", "촣", "촣" },
  { "촤", "촤", "촤", "촤", "촤" },
  { "촥", "촥", "촥", "촥", "촥" },
  { "촦", "촦", "촦", "촦", "촦" },
  { "촧", "촧", "촧", "촧", "촧" },
  { "촨", "촨", "촨", "촨", "촨" },
  { "촩", "촩", "촩", "촩", "촩" },
  { "촪", "촪", "촪", "촪", "촪" },
  { "촫", "촫", "촫", "촫", "촫" },
  { "촬", "촬", "촬", "촬", "촬" },
  { "촭", "촭", "촭", "촭", "촭" },
  { "촮", "촮", "촮", "촮", "촮" },
  { "촯", "촯", "촯", "촯", "촯" },
  { "촰", "촰", "촰", "촰", "촰" },
  { "촱", "촱", "촱", "촱", "촱" },
  { "촲", "촲", "촲", "촲", "촲" },
  { "촳", "촳", "촳", "촳", "촳" },
  { "촴", "촴", "촴", "촴", "촴" },
  { "촵", "촵", "촵", "촵", "촵" },
  { "촶", "촶", "촶", "촶", "촶" },
  { "촷", "촷", "촷", "촷", "촷" },
  { "촸", "촸", "촸", "촸", "촸" },
  { "촹", "촹", "촹", "촹", "촹" },
  { "촺", "촺", "촺", "촺", "촺" },
  { "촻", "촻", "촻", "촻", "촻" },
  { "촼", "촼", "촼", "촼", "촼" },
  { "촽", "촽", "촽", "촽", "촽" },
  { "촾", "촾", "촾", "촾", "촾" },
  { "촿", "촿", "촿", "촿", "촿" },
  { "쵀", "쵀", "쵀", "쵀", "쵀" },
  { "쵁", "쵁", "쵁", "쵁", "쵁" },
  { "쵂", "쵂", "쵂", "쵂", "쵂" },
  { "쵃", "쵃", "쵃", "쵃", "쵃" },
  { "쵄", "쵄", "쵄", "쵄", "쵄" },
  { "쵅", "쵅", "쵅", "쵅", "쵅" },
  { "쵆", "쵆", "쵆", "쵆", "쵆" },
  { "쵇", "쵇", "쵇", "쵇", "쵇" },
  { "쵈", "쵈", "쵈", "쵈", "쵈" },
  { "쵉", "쵉", "쵉", "쵉", "쵉" },
  { "쵊", "쵊", "쵊", "쵊", "쵊" },
  { "쵋", "쵋", "쵋", "쵋", "쵋" },
  { "쵌", "쵌", "쵌", "쵌", "쵌" },
  { "쵍", "쵍", "쵍", "쵍", "쵍" },
  { "쵎", "쵎", "쵎", "쵎", "쵎" },
  { "쵏", "쵏", "쵏", "쵏", "쵏" },
  { "쵐", "쵐", "쵐", "쵐", "쵐" },
  { "쵑", "쵑", "쵑", "쵑", "쵑" },
  { "쵒", "쵒", "쵒", "쵒", "쵒" },
  { "쵓", "쵓", "쵓", "쵓", "쵓" },
  { "쵔", "쵔", "쵔", "쵔", "쵔" },
  { "쵕", "쵕", "쵕", "쵕", "쵕" },
  { "쵖", "쵖", "쵖", "쵖", "쵖" },
  { "쵗", "쵗", "쵗", "쵗", "쵗" },
  { "쵘", "쵘", "쵘", "쵘", "쵘" },
  { "쵙", "쵙", "쵙", "쵙", "쵙" },
  { "쵚", "쵚", "쵚", "쵚", "쵚" },
  { "쵛", "쵛", "쵛", "쵛", "쵛" },
  { "최", "최", "최", "최", "최" },
  { "쵝", "쵝", "쵝", "쵝", "쵝" },
  { "쵞", "쵞", "쵞", "쵞", "쵞" },
  { "쵟", "쵟", "쵟", "쵟", "쵟" },
  { "쵠", "쵠", "쵠", "쵠", "쵠" },
  { "쵡", "쵡", "쵡", "쵡", "쵡" },
  { "쵢", "쵢", "쵢", "쵢", "쵢" },
  { "쵣", "쵣", "쵣", "쵣", "쵣" },
  { "쵤", "쵤", "쵤", "쵤", "쵤" },
  { "쵥", "쵥", "쵥", "쵥", "쵥" },
  { "쵦", "쵦", "쵦", "쵦", "쵦" },
  { "쵧", "쵧", "쵧", "쵧", "쵧" },
  { "쵨", "쵨", "쵨", "쵨", "쵨" },
  { "쵩", "쵩", "쵩", "쵩", "쵩" },
  { "쵪", "쵪", "쵪", "쵪", "쵪" },
  { "쵫", "쵫", "쵫", "쵫", "쵫" },
  { "쵬", "쵬", "쵬", "쵬", "쵬" },
  { "쵭", "쵭", "쵭", "쵭", "쵭" },
  { "쵮", "쵮", "쵮", "쵮", "쵮" },
  { "쵯", "쵯", "쵯", "쵯", "쵯" },
  { "쵰", "쵰", "쵰", "쵰", "쵰" },
  { "쵱", "쵱", "쵱", "쵱", "쵱" },
  { "쵲", "쵲", "쵲", "쵲", "쵲" },
  { "쵳", "쵳", "쵳", "쵳", "쵳" },
  { "쵴", "쵴", "쵴", "쵴", "쵴" },
  { "쵵", "쵵", "쵵", "쵵", "쵵" },
  { "쵶", "쵶", "쵶", "쵶", "쵶" },
  { "쵷", "쵷", "쵷", "쵷", "쵷" },
  { "쵸", "쵸", "쵸", "쵸", "쵸" },
  { "쵹", "쵹", "쵹", "쵹", "쵹" },
  { "쵺", "쵺", "쵺", "쵺", "쵺" },
  { "쵻", "쵻", "쵻", "쵻", "쵻" },
  { "쵼", "쵼", "쵼", "쵼", "쵼" },
  { "쵽", "쵽", "쵽", "쵽", "쵽" },
  { "쵾", "쵾", "쵾", "쵾", "쵾" },
  { "쵿", "쵿", "쵿", "쵿", "쵿" },
  { "춀", "춀", "춀", "춀", "춀" },
  { "춁", "춁", "춁", "춁", "춁" },
  { "춂", "춂", "춂", "춂", "춂" },
  { "춃", "춃", "춃", "춃", "춃" },
  { "춄", "춄", "춄", "춄", "춄" },
  { "춅", "춅", "춅", "춅", "춅" },
  { "춆", "춆", "춆", "춆", "춆" },
  { "춇", "춇", "춇", "춇", "춇" },
  { "춈", "춈", "춈", "춈", "춈" },
  { "춉", "춉", "춉", "춉", "춉" },
  { "춊", "춊", "춊", "춊", "춊" },
  { "춋", "춋", "춋", "춋", "춋" },
  { "춌", "춌", "춌", "춌", "춌" },
  { "춍", "춍", "춍", "춍", "춍" },
  { "춎", "춎", "춎", "춎", "춎" },
  { "춏", "춏", "춏", "춏", "춏" },
  { "춐", "춐", "춐", "춐", "춐" },
  { "춑", "춑", "춑", "춑", "춑" },
  { "춒", "춒", "춒", "춒", "춒" },
  { "춓", "춓", "춓", "춓", "춓" },
  { "추", "추", "추", "추", "추" },
  { "축", "축", "축", "축", "축" },
  { "춖", "춖", "춖", "춖", "춖" },
  { "춗", "춗", "춗", "춗", "춗" },
  { "춘", "춘", "춘", "춘", "춘" },
  { "춙", "춙", "춙", "춙", "춙" },
  { "춚", "춚", "춚", "춚", "춚" },
  { "춛", "춛", "춛", "춛", "춛" },
  { "출", "출", "출", "출", "출" },
  { "춝", "춝", "춝", "춝", "춝" },
  { "춞", "춞", "춞", "춞", "춞" },
  { "춟", "춟", "춟", "춟", "춟" },
  { "춠", "춠", "춠", "춠", "춠" },
  { "춡", "춡", "춡", "춡", "춡" },
  { "춢", "춢", "춢", "춢", "춢" },
  { "춣", "춣", "춣", "춣", "춣" },
  { "춤", "춤", "춤", "춤", "춤" },
  { "춥", "춥", "춥", "춥", "춥" },
  { "춦", "춦", "춦", "춦", "춦" },
  { "춧", "춧", "춧", "춧", "춧" },
  { "춨", "춨", "춨", "춨", "춨" },
  { "충", "충", "충", "충", "충" },
  { "춪", "춪", "춪", "춪", "춪" },
  { "춫", "춫", "춫", "춫", "춫" },
  { "춬", "춬", "춬", "춬", "춬" },
  { "춭", "춭", "춭", "춭", "춭" },
  { "춮", "춮", "춮", "춮", "춮" },
  { "춯", "춯", "춯", "춯", "춯" },
  { "춰", "춰", "춰", "춰", "춰" },
  { "춱", "춱", "춱", "춱", "춱" },
  { "춲", "춲", "춲", "춲", "춲" },
  { "춳", "춳", "춳", "춳", "춳" },
  { "춴", "춴", "춴", "춴", "춴" },
  { "춵", "춵", "춵", "춵", "춵" },
  { "춶", "춶", "춶", "춶", "춶" },
  { "춷", "춷", "춷", "춷", "춷" },
  { "춸", "춸", "춸", "춸", "춸" },
  { "춹", "춹", "춹", "춹", "춹" },
  { "춺", "춺", "춺", "춺", "춺" },
  { "춻", "춻", "춻", "춻", "춻" },
  { "춼", "춼", "춼", "춼", "춼" },
  { "춽", "춽", "춽", "춽", "춽" },
  { "춾", "춾", "춾", "춾", "춾" },
  { "춿", "춿", "춿", "춿", "춿" },
  { "췀", "췀", "췀", "췀", "췀" },
  { "췁", "췁", "췁", "췁", "췁" },
  { "췂", "췂", "췂", "췂", "췂" },
  { "췃", "췃", "췃", "췃", "췃" },
  { "췄", "췄", "췄", "췄", "췄" },
  { "췅", "췅", "췅", "췅", "췅" },
  { "췆", "췆", "췆", "췆", "췆" },
  { "췇", "췇", "췇", "췇", "췇" },
  { "췈", "췈", "췈", "췈", "췈" },
  { "췉", "췉", "췉", "췉", "췉" },
  { "췊", "췊", "췊", "췊", "췊" },
  { "췋", "췋", "췋", "췋", "췋" },
  { "췌", "췌", "췌", "췌", "췌" },
  { "췍", "췍", "췍", "췍", "췍" },
  { "췎", "췎", "췎", "췎", "췎" },
  { "췏", "췏", "췏", "췏", "췏" },
  { "췐", "췐", "췐", "췐", "췐" },
  { "췑", "췑", "췑", "췑", "췑" },
  { "췒", "췒", "췒", "췒", "췒" },
  { "췓", "췓", "췓", "췓", "췓" },
  { "췔", "췔", "췔", "췔", "췔" },
  { "췕", "췕", "췕", "췕", "췕" },
  { "췖", "췖", "췖", "췖", "췖" },
  { "췗", "췗", "췗", "췗", "췗" },
  { "췘", "췘", "췘", "췘", "췘" },
  { "췙", "췙", "췙", "췙", "췙" },
  { "췚", "췚", "췚", "췚", "췚" },
  { "췛", "췛", "췛", "췛", "췛" },
  { "췜", "췜", "췜", "췜", "췜" },
  { "췝", "췝", "췝", "췝", "췝" },
  { "췞", "췞", "췞", "췞", "췞" },
  { "췟", "췟", "췟", "췟", "췟" },
  { "췠", "췠", "췠", "췠", "췠" },
  { "췡", "췡", "췡", "췡", "췡" },
  { "췢", "췢", "췢", "췢", "췢" },
  { "췣", "췣", "췣", "췣", "췣" },
  { "췤", "췤", "췤", "췤", "췤" },
  { "췥", "췥", "췥", "췥", "췥" },
  { "췦", "췦", "췦", "췦", "췦" },
  { "췧", "췧", "췧", "췧", "췧" },
  { "취", "취", "취", "취", "취" },
  { "췩", "췩", "췩", "췩", "췩" },
  { "췪", "췪", "췪", "췪", "췪" },
  { "췫", "췫", "췫", "췫", "췫" },
  { "췬", "췬", "췬", "췬", "췬" },
  { "췭", "췭", "췭", "췭", "췭" },
  { "췮", "췮", "췮", "췮", "췮" },
  { "췯", "췯", "췯", "췯", "췯" },
  { "췰", "췰", "췰", "췰", "췰" },
  { "췱", "췱", "췱", "췱", "췱" },
  { "췲", "췲", "췲", "췲", "췲" },
  { "췳", "췳", "췳", "췳", "췳" },
  { "췴", "췴", "췴", "췴", "췴" },
  { "췵", "췵", "췵", "췵", "췵" },
  { "췶", "췶", "췶", "췶", "췶" },
  { "췷", "췷", "췷", "췷", "췷" },
  { "췸", "췸", "췸", "췸", "췸" },
  { "췹", "췹", "췹", "췹", "췹" },
  { "췺", "췺", "췺", "췺", "췺" },
  { "췻", "췻", "췻", "췻", "췻" },
  { "췼", "췼", "췼", "췼", "췼" },
  { "췽", "췽", "췽", "췽", "췽" },
  { "췾", "췾", "췾", "췾", "췾" },
  { "췿", "췿", "췿", "췿", "췿" },
  { "츀", "츀", "츀", "츀", "츀" },
  { "츁", "츁", "츁", "츁", "츁" },
  { "츂", "츂", "츂", "츂", "츂" },
  { "츃", "츃", "츃", "츃", "츃" },
  { "츄", "츄", "츄", "츄", "츄" },
  { "츅", "츅", "츅", "츅", "츅" },
  { "츆", "츆", "츆", "츆", "츆" },
  { "츇", "츇", "츇", "츇", "츇" },
  { "츈", "츈", "츈", "츈", "츈" },
  { "츉", "츉", "츉", "츉", "츉" },
  { "츊", "츊", "츊", "츊", "츊" },
  { "츋", "츋", "츋", "츋", "츋" },
  { "츌", "츌", "츌", "츌", "츌" },
  { "츍", "츍", "츍", "츍", "츍" },
  { "츎", "츎", "츎", "츎", "츎" },
  { "츏", "츏", "츏", "츏", "츏" },
  { "츐", "츐", "츐", "츐", "츐" },
  { "츑", "츑", "츑", "츑", "츑" },
  { "츒", "츒", "츒", "츒", "츒" },
  { "츓", "츓", "츓", "츓", "츓" },
  { "츔", "츔", "츔", "츔", "츔" },
  { "츕", "츕", "츕", "츕", "츕" },
  { "츖", "츖", "츖", "츖", "츖" },
  { "츗", "츗", "츗", "츗", "츗" },
  { "츘", "츘", "츘", "츘", "츘" },
  { "츙", "츙", "츙", "츙", "츙" },
  { "츚", "츚", "츚", "츚", "츚" },
  { "츛", "츛", "츛", "츛", "츛" },
  { "츜", "츜", "츜", "츜", "츜" },
  { "츝", "츝", "츝", "츝", "츝" },
  { "츞", "츞", "츞", "츞", "츞" },
  { "츟", "츟", "츟", "츟", "츟" },
  { "츠", "츠", "츠", "츠", "츠" },
  { "측", "측", "측", "측", "측" },
  { "츢", "츢", "츢", "츢", "츢" },
  { "츣", "츣", "츣", "츣", "츣" },
  { "츤", "츤", "츤", "츤", "츤" },
  { "츥", "츥", "츥", "츥", "츥" },
  { "츦", "츦", "츦", "츦", "츦" },
  { "츧", "츧", "츧", "츧", "츧" },
  { "츨", "츨", "츨", "츨", "츨" },
  { "츩", "츩", "츩", "츩", "츩" },
  { "츪", "츪", "츪", "츪", "츪" },
  { "츫", "츫", "츫", "츫", "츫" },
  { "츬", "츬", "츬", "츬", "츬" },
  { "츭", "츭", "츭", "츭", "츭" },
  { "츮", "츮", "츮", "츮", "츮" },
  { "츯", "츯", "츯", "츯", "츯" },
  { "츰", "츰", "츰", "츰", "츰" },
  { "츱", "츱", "츱", "츱", "츱" },
  { "츲", "츲", "츲", "츲", "츲" },
  { "츳", "츳", "츳", "츳", "츳" },
  { "츴", "츴", "츴", "츴", "츴" },
  { "층", "층", "층", "층", "층" },
  { "츶", "츶", "츶", "츶", "츶" },
  { "츷", "츷", "츷", "츷", "츷" },
  { "츸", "츸", "츸", "츸", "츸" },
  { "츹", "츹", "츹", "츹", "츹" },
  { "츺", "츺", "츺", "츺", "츺" },
  { "츻", "츻", "츻", "츻", "츻" },
  { "츼", "츼", "츼", "츼", "츼" },
  { "츽", "츽", "츽", "츽", "츽" },
  { "츾", "츾", "츾", "츾", "츾" },
  { "츿", "츿", "츿", "츿", "츿" },
  { "칀", "칀", "칀", "칀", "칀" },
  { "칁", "칁", "칁", "칁", "칁" },
  { "칂", "칂", "칂", "칂", "칂" },
  { "칃", "칃", "칃", "칃", "칃" },
  { "칄", "칄", "칄", "칄", "칄" },
  { "칅", "칅", "칅", "칅", "칅" },
  { "칆", "칆", "칆", "칆", "칆" },
  { "칇", "칇", "칇", "칇", "칇" },
  { "칈", "칈", "칈", "칈", "칈" },
  { "칉", "칉", "칉", "칉", "칉" },
  { "칊", "칊", "칊", "칊", "칊" },
  { "칋", "칋", "칋", "칋", "칋" },
  { "칌", "칌", "칌", "칌", "칌" },
  { "칍", "칍", "칍", "칍", "칍" },
  { "칎", "칎", "칎", "칎", "칎" },
  { "칏", "칏", "칏", "칏", "칏" },
  { "칐", "칐", "칐", "칐", "칐" },
  { "칑", "칑", "칑", "칑", "칑" },
  { "칒", "칒", "칒", "칒", "칒" },
  { "칓", "칓", "칓", "칓", "칓" },
  { "칔", "칔", "칔", "칔", "칔" },
  { "칕", "칕", "칕", "칕", "칕" },
  { "칖", "칖", "칖", "칖", "칖" },
  { "칗", "칗", "칗", "칗", "칗" },
  { "치", "치", "치", "치", "치" },
  { "칙", "칙", "칙", "칙", "칙" },
  { "칚", "칚", "칚", "칚", "칚" },
  { "칛", "칛", "칛", "칛", "칛" },
  { "친", "친", "친", "친", "친" },
  { "칝", "칝", "칝", "칝", "칝" },
  { "칞", "칞", "칞", "칞", "칞" },
  { "칟", "칟", "칟", "칟", "칟" },
  { "칠", "칠", "칠", "칠", "칠" },
  { "칡", "칡", "칡", "칡", "칡" },
  { "칢", "칢", "칢", "칢", "칢" },
  { "칣", "칣", "칣", "칣", "칣" },
  { "칤", "칤", "칤", "칤", "칤" },
  { "칥", "칥", "칥", "칥", "칥" },
  { "칦", "칦", "칦", "칦", "칦" },
  { "칧", "칧", "칧", "칧", "칧" },
  { "침", "침", "침", "침", "침" },
  { "칩", "칩", "칩", "칩", "칩" },
  { "칪", "칪", "칪", "칪", "칪" },
  { "칫", "칫", "칫", "칫", "칫" },
  { "칬", "칬", "칬", "칬", "칬" },
  { "칭", "칭", "칭", "칭", "칭" },
  { "칮", "칮", "칮", "칮", "칮" },
  { "칯", "칯", "칯", "칯", "칯" },
  { "칰", "칰", "칰", "칰", "칰" },
  { "칱", "칱", "칱", "칱", "칱" },
  { "칲", "칲", "칲", "칲", "칲" },
  { "칳", "칳", "칳", "칳", "칳" },
  { "카", "카", "카", "카", "카" },
  { "칵", "칵", "칵", "칵", "칵" },
  { "칶", "칶", "칶", "칶", "칶" },
  { "칷", "칷", "칷", "칷", "칷" },
  { "칸", "칸", "칸", "칸", "칸" },
  { "칹", "칹", "칹", "칹", "칹" },
  { "칺", "칺", "칺", "칺", "칺" },
  { "칻", "칻", "칻", "칻", "칻" },
  { "칼", "칼", "칼", "칼", "칼" },
  { "칽", "칽", "칽", "칽", "칽" },
  { "칾", "칾", "칾", "칾", "칾" },
  { "칿", "칿", "칿", "칿", "칿" },
  { "캀", "캀", "캀", "캀", "캀" },
  { "캁", "캁", "캁", "캁", "캁" },
  { "캂", "캂", "캂", "캂", "캂" },
  { "캃", "캃", "캃", "캃", "캃" },
  { "캄", "캄", "캄", "캄", "캄" },
  { "캅", "캅", "캅", "캅", "캅" },
  { "캆", "캆", "캆", "캆", "캆" },
  { "캇", "캇", "캇", "캇", "캇" },
  { "캈", "캈", "캈", "캈", "캈" },
  { "캉", "캉", "캉", "캉", "캉" },
  { "캊", "캊", "캊", "캊", "캊" },
  { "캋", "캋", "캋", "캋", "캋" },
  { "캌", "캌", "캌", "캌", "캌" },
  { "캍", "캍", "캍", "캍", "캍" },
  { "캎", "캎", "캎", "캎", "캎" },
  { "캏", "캏", "캏", "캏", "캏" },
  { "캐", "캐", "캐", "캐", "캐" },
  { "캑", "캑", "캑", "캑", "캑" },
  { "캒", "캒", "캒", "캒", "캒" },
  { "캓", "캓", "캓", "캓", "캓" },
  { "캔", "캔", "캔", "캔", "캔" },
  { "캕", "캕", "캕", "캕", "캕" },
  { "캖", "캖", "캖", "캖", "캖" },
  { "캗", "캗", "캗", "캗", "캗" },
  { "캘", "캘", "캘", "캘", "캘" },
  { "캙", "캙", "캙", "캙", "캙" },
  { "캚", "캚", "캚", "캚", "캚" },
  { "캛", "캛", "캛", "캛", "캛" },
  { "캜", "캜", "캜", "캜", "캜" },
  { "캝", "캝", "캝", "캝", "캝" },
  { "캞", "캞", "캞", "캞", "캞" },
  { "캟", "캟", "캟", "캟", "캟" },
  { "캠", "캠", "캠", "캠", "캠" },
  { "캡", "캡", "캡", "캡", "캡" },
  { "캢", "캢", "캢", "캢", "캢" },
  { "캣", "캣", "캣", "캣", "캣" },
  { "캤", "캤", "캤", "캤", "캤" },
  { "캥", "캥", "캥", "캥", "캥" },
  { "캦", "캦", "캦", "캦", "캦" },
  { "캧", "캧", "캧", "캧", "캧" },
  { "캨", "캨", "캨", "캨", "캨" },
  { "캩", "캩", "캩", "캩", "캩" },
  { "캪", "캪", "캪", "캪", "캪" },
  { "캫", "캫", "캫", "캫", "캫" },
  { "캬", "캬", "캬", "캬", "캬" },
  { "캭", "캭", "캭", "캭", "캭" },
  { "캮", "캮", "캮", "캮", "캮" },
  { "캯", "캯", "캯", "캯", "캯" },
  { "캰", "캰", "캰", "캰", "캰" },
  { "캱", "캱", "캱", "캱", "캱" },
  { "캲", "캲", "캲", "캲", "캲" },
  { "캳", "캳", "캳", "캳", "캳" },
  { "캴", "캴", "캴", "캴", "캴" },
  { "캵", "캵", "캵", "캵", "캵" },
  { "캶", "캶", "캶", "캶", "캶" },
  { "캷", "캷", "캷", "캷", "캷" },
  { "캸", "캸", "캸", "캸", "캸" },
  { "캹", "캹", "캹", "캹", "캹" },
  { "캺", "캺", "캺", "캺", "캺" },
  { "캻", "캻", "캻", "캻", "캻" },
  { "캼", "캼", "캼", "캼", "캼" },
  { "캽", "캽", "캽", "캽", "캽" },
  { "캾", "캾", "캾", "캾", "캾" },
  { "캿", "캿", "캿", "캿", "캿" },
  { "컀", "컀", "컀", "컀", "컀" },
  { "컁", "컁", "컁", "컁", "컁" },
  { "컂", "컂", "컂", "컂", "컂" },
  { "컃", "컃", "컃", "컃", "컃" },
  { "컄", "컄", "컄", "컄", "컄" },
  { "컅", "컅", "컅", "컅", "컅" },
  { "컆", "컆", "컆", "컆", "컆" },
  { "컇", "컇", "컇", "컇", "컇" },
  { "컈", "컈", "컈", "컈", "컈" },
  { "컉", "컉", "컉", "컉", "컉" },
  { "컊", "컊", "컊", "컊", "컊" },
  { "컋", "컋", "컋", "컋", "컋" },
  { "컌", "컌", "컌", "컌", "컌" },
  { "컍", "컍", "컍", "컍", "컍" },
  { "컎", "컎", "컎", "컎", "컎" },
  { "컏", "컏", "컏", "컏", "컏" },
  { "컐", "컐", "컐", "컐", "컐" },
  { "컑", "컑", "컑", "컑", "컑" },
  { "컒", "컒", "컒", "컒", "컒" },
  { "컓", "컓", "컓", "컓", "컓" },
  { "컔", "컔", "컔", "컔", "컔" },
  { "컕", "컕", "컕", "컕", "컕" },
  { "컖", "컖", "컖", "컖", "컖" },
  { "컗", "컗", "컗", "컗", "컗" },
  { "컘", "컘", "컘", "컘", "컘" },
  { "컙", "컙", "컙", "컙", "컙" },
  { "컚", "컚", "컚", "컚", "컚" },
  { "컛", "컛", "컛", "컛", "컛" },
  { "컜", "컜", "컜", "컜", "컜" },
  { "컝", "컝", "컝", "컝", "컝" },
  { "컞", "컞", "컞", "컞", "컞" },
  { "컟", "컟", "컟", "컟", "컟" },
  { "컠", "컠", "컠", "컠", "컠" },
  { "컡", "컡", "컡", "컡", "컡" },
  { "컢", "컢", "컢", "컢", "컢" },
  { "컣", "컣", "컣", "컣", "컣" },
  { "커", "커", "커", "커", "커" },
  { "컥", "컥", "컥", "컥", "컥" },
  { "컦", "컦", "컦", "컦", "컦" },
  { "컧", "컧", "컧", "컧", "컧" },
  { "컨", "컨", "컨", "컨", "컨" },
  { "컩", "컩", "컩", "컩", "컩" },
  { "컪", "컪", "컪", "컪", "컪" },
  { "컫", "컫", "컫", "컫", "컫" },
  { "컬", "컬", "컬", "컬", "컬" },
  { "컭", "컭", "컭", "컭", "컭" },
  { "컮", "컮", "컮", "컮", "컮" },
  { "컯", "컯", "컯", "컯", "컯" },
  { "컰", "컰", "컰", "컰", "컰" },
  { "컱", "컱", "컱", "컱", "컱" },
  { "컲", "컲", "컲", "컲", "컲" },
  { "컳", "컳", "컳", "컳", "컳" },
  { "컴", "컴", "컴", "컴", "컴" },
  { "컵", "컵", "컵", "컵", "컵" },
  { "컶", "컶", "컶", "컶", "컶" },
  { "컷", "컷", "컷", "컷", "컷" },
  { "컸", "컸", "컸", "컸", "컸" },
  { "컹", "컹", "컹", "컹", "컹" },
  { "컺", "컺", "컺", "컺", "컺" },
  { "컻", "컻", "컻", "컻", "컻" },
  { "컼", "컼", "컼", "컼", "컼" },
  { "컽", "컽", "컽", "컽", "컽" },
  { "컾", "컾", "컾", "컾", "컾" },
  { "컿", "컿", "컿", "컿", "컿" },
  { "케", "케", "케", "케", "케" },
  { "켁", "켁", "켁", "켁", "켁" },
  { "켂", "켂", "켂", "켂", "켂" },
  { "켃", "켃", "켃", "켃", "켃" },
  { "켄", "켄", "켄", "켄", "켄" },
  { "켅", "켅", "켅", "켅", "켅" },
  { "켆", "켆", "켆", "켆", "켆" },
  { "켇", "켇", "켇", "켇", "켇" },
  { "켈", "켈", "켈", "켈", "켈" },
  { "켉", "켉", "켉", "켉", "켉" },
  { "켊", "켊", "켊", "켊", "켊" },
  { "켋", "켋", "켋", "켋", "켋" },
  { "켌", "켌", "켌", "켌", "켌" },
  { "켍", "켍", "켍", "켍", "켍" },
  { "켎", "켎", "켎", "켎", "켎" },
  { "켏", "켏", "켏", "켏", "켏" },
  { "켐", "켐", "켐", "켐", "켐" },
  { "켑", "켑", "켑", "켑", "켑" },
  { "켒", "켒", "켒", "켒", "켒" },
  { "켓", "켓", "켓", "켓", "켓" },
  { "켔", "켔", "켔", "켔", "켔" },
  { "켕", "켕", "켕", "켕", "켕" },
  { "켖", "켖", "켖", "켖", "켖" },
  { "켗", "켗", "켗", "켗", "켗" },
  { "켘", "켘", "켘", "켘", "켘" },
  { "켙", "켙", "켙", "켙", "켙" },
  { "켚", "켚", "켚", "켚", "켚" },
  { "켛", "켛", "켛", "켛", "켛" },
  { "켜", "켜", "켜", "켜", "켜" },
  { "켝", "켝", "켝", "켝", "켝" },
  { "켞", "켞", "켞", "켞", "켞" },
  { "켟", "켟", "켟", "켟", "켟" },
  { "켠", "켠", "켠", "켠", "켠" },
  { "켡", "켡", "켡", "켡", "켡" },
  { "켢", "켢", "켢", "켢", "켢" },
  { "켣", "켣", "켣", "켣", "켣" },
  { "켤", "켤", "켤", "켤", "켤" },
  { "켥", "켥", "켥", "켥", "켥" },
  { "켦", "켦", "켦", "켦", "켦" },
  { "켧", "켧", "켧", "켧", "켧" },
  { "켨", "켨", "켨", "켨", "켨" },
  { "켩", "켩", "켩", "켩", "켩" },
  { "켪", "켪", "켪", "켪", "켪" },
  { "켫", "켫", "켫", "켫", "켫" },
  { "켬", "켬", "켬", "켬", "켬" },
  { "켭", "켭", "켭", "켭", "켭" },
  { "켮", "켮", "켮", "켮", "켮" },
  { "켯", "켯", "켯", "켯", "켯" },
  { "켰", "켰", "켰", "켰", "켰" },
  { "켱", "켱", "켱", "켱", "켱" },
  { "켲", "켲", "켲", "켲", "켲" },
  { "켳", "켳", "켳", "켳", "켳" },
  { "켴", "켴", "켴", "켴", "켴" },
  { "켵", "켵", "켵", "켵", "켵" },
  { "켶", "켶", "켶", "켶", "켶" },
  { "켷", "켷", "켷", "켷", "켷" },
  { "켸", "켸", "켸", "켸", "켸" },
  { "켹", "켹", "켹", "켹", "켹" },
  { "켺", "켺", "켺", "켺", "켺" },
  { "켻", "켻", "켻", "켻", "켻" },
  { "켼", "켼", "켼", "켼", "켼" },
  { "켽", "켽", "켽", "켽", "켽" },
  { "켾", "켾", "켾", "켾", "켾" },
  { "켿", "켿", "켿", "켿", "켿" },
  { "콀", "콀", "콀", "콀", "콀" },
  { "콁", "콁", "콁", "콁", "콁" },
  { "콂", "콂", "콂", "콂", "콂" },
  { "콃", "콃", "콃", "콃", "콃" },
  { "콄", "콄", "콄", "콄", "콄" },
  { "콅", "콅", "콅", "콅", "콅" },
  { "콆", "콆", "콆", "콆", "콆" },
  { "콇", "콇", "콇", "콇", "콇" },
  { "콈", "콈", "콈", "콈", "콈" },
  { "콉", "콉", "콉", "콉", "콉" },
  { "콊", "콊", "콊", "콊", "콊" },
  { "콋", "콋", "콋", "콋", "콋" },
  { "콌", "콌", "콌", "콌", "콌" },
  { "콍", "콍", "콍", "콍", "콍" },
  { "콎", "콎", "콎", "콎", "콎" },
  { "콏", "콏", "콏", "콏", "콏" },
  { "콐", "콐", "콐", "콐", "콐" },
  { "콑", "콑", "콑", "콑", "콑" },
  { "콒", "콒", "콒", "콒", "콒" },
  { "콓", "콓", "콓", "콓", "콓" },
  { "코", "코", "코", "코", "코" },
  { "콕", "콕", "콕", "콕", "콕" },
  { "콖", "콖", "콖", "콖", "콖" },
  { "콗", "콗", "콗", "콗", "콗" },
  { "콘", "콘", "콘", "콘", "콘" },
  { "콙", "콙", "콙", "콙", "콙" },
  { "콚", "콚", "콚", "콚", "콚" },
  { "콛", "콛", "콛", "콛", "콛" },
  { "콜", "콜", "콜", "콜", "콜" },
  { "콝", "콝", "콝", "콝", "콝" },
  { "콞", "콞", "콞", "콞", "콞" },
  { "콟", "콟", "콟", "콟", "콟" },
  { "콠", "콠", "콠", "콠", "콠" },
  { "콡", "콡", "콡", "콡", "콡" },
  { "콢", "콢", "콢", "콢", "콢" },
  { "콣", "콣", "콣", "콣", "콣" },
  { "콤", "콤", "콤", "콤", "콤" },
  { "콥", "콥", "콥", "콥", "콥" },
  { "콦", "콦", "콦", "콦", "콦" },
  { "콧", "콧", "콧", "콧", "콧" },
  { "콨", "콨", "콨", "콨", "콨" },
  { "콩", "콩", "콩", "콩", "콩" },
  { "콪", "콪", "콪", "콪", "콪" },
  { "콫", "콫", "콫", "콫", "콫" },
  { "콬", "콬", "콬", "콬", "콬" },
  { "콭", "콭", "콭", "콭", "콭" },
  { "콮", "콮", "콮", "콮", "콮" },
  { "콯", "콯", "콯", "콯", "콯" },
  { "콰", "콰", "콰", "콰", "콰" },
  { "콱", "콱", "콱", "콱", "콱" },
  { "콲", "콲", "콲", "콲", "콲" },
  { "콳", "콳", "콳", "콳", "콳" },
  { "콴", "콴", "콴", "콴", "콴" },
  { "콵", "콵", "콵", "콵", "콵" },
  { "콶", "콶", "콶", "콶", "콶" },
  { "콷", "콷", "콷", "콷", "콷" },
  { "콸", "콸", "콸", "콸", "콸" },
  { "콹", "콹", "콹", "콹", "콹" },
  { "콺", "콺", "콺", "콺", "콺" },
  { "콻", "콻", "콻", "콻", "콻" },
  { "콼", "콼", "콼", "콼", "콼" },
  { "콽", "콽", "콽", "콽", "콽" },
  { "콾", "콾", "콾", "콾", "콾" },
  { "콿", "콿", "콿", "콿", "콿" },
  { "쾀", "쾀", "쾀", "쾀", "쾀" },
  { "쾁", "쾁", "쾁", "쾁", "쾁" },
  { "쾂", "쾂", "쾂", "쾂", "쾂" },
  { "쾃", "쾃", "쾃", "쾃", "쾃" },
  { "쾄", "쾄", "쾄", "쾄", "쾄" },
  { "쾅", "쾅", "쾅", "쾅", "쾅" },
  { "쾆", "쾆", "쾆", "쾆", "쾆" },
  { "쾇", "쾇", "쾇", "쾇", "쾇" },
  { "쾈", "쾈", "쾈", "쾈", "쾈" },
  { "쾉", "쾉", "쾉", "쾉", "쾉" },
  { "쾊", "쾊", "쾊", "쾊", "쾊" },
  { "쾋", "쾋", "쾋", "쾋", "쾋" },
  { "쾌", "쾌", "쾌", "쾌", "쾌" },
  { "쾍", "쾍", "쾍", "쾍", "쾍" },
  { "쾎", "쾎", "쾎", "쾎", "쾎" },
  { "쾏", "쾏", "쾏", "쾏", "쾏" },
  { "쾐", "쾐", "쾐", "쾐", "쾐" },
  { "쾑", "쾑", "쾑", "쾑", "쾑" },
  { "쾒", "쾒", "쾒", "쾒", "쾒" },
  { "쾓", "쾓", "쾓", "쾓", "쾓" },
  { "쾔", "쾔", "쾔", "쾔", "쾔" },
  { "쾕", "쾕", "쾕", "쾕", "쾕" },
  { "쾖", "쾖", "쾖", "쾖", "쾖" },
  { "쾗", "쾗", "쾗", "쾗", "쾗" },
  { "쾘", "쾘", "쾘", "쾘", "쾘" },
  { "쾙", "쾙", "쾙", "쾙", "쾙" },
  { "쾚", "쾚", "쾚", "쾚", "쾚" },
  { "쾛", "쾛", "쾛", "쾛", "쾛" },
  { "쾜", "쾜", "쾜", "쾜", "쾜" },
  { "쾝", "쾝", "쾝", "쾝", "쾝" },
  { "쾞", "쾞", "쾞", "쾞", "쾞" },
  { "쾟", "쾟", "쾟", "쾟", "쾟" },
  { "쾠", "쾠", "쾠", "쾠", "쾠" },
  { "쾡", "쾡", "쾡", "쾡", "쾡" },
  { "쾢", "쾢", "쾢", "쾢", "쾢" },
  { "쾣", "쾣", "쾣", "쾣", "쾣" },
  { "쾤", "쾤", "쾤", "쾤", "쾤" },
  { "쾥", "쾥", "쾥", "쾥", "쾥" },
  { "쾦", "쾦", "쾦", "쾦", "쾦" },
  { "쾧", "쾧", "쾧", "쾧", "쾧" },
  { "쾨", "쾨", "쾨", "쾨", "쾨" },
  { "쾩", "쾩", "쾩", "쾩", "쾩" },
  { "쾪", "쾪", "쾪", "쾪", "쾪" },
  { "쾫", "쾫", "쾫", "쾫", "쾫" },
  { "쾬", "쾬", "쾬", "쾬", "쾬" },
  { "쾭", "쾭", "쾭", "쾭", "쾭" },
  { "쾮", "쾮", "쾮", "쾮", "쾮" },
  { "쾯", "쾯", "쾯", "쾯", "쾯" },
  { "쾰", "쾰", "쾰", "쾰", "쾰" },
  { "쾱", "쾱", "쾱", "쾱", "쾱" },
  { "쾲", "쾲", "쾲", "쾲", "쾲" },
  { "쾳", "쾳", "쾳", "쾳", "쾳" },
  { "쾴", "쾴", "쾴", "쾴", "쾴" },
  { "쾵", "쾵", "쾵", "쾵", "쾵" },
  { "쾶", "쾶", "쾶", "쾶", "쾶" },
  { "쾷", "쾷", "쾷", "쾷", "쾷" },
  { "쾸", "쾸", "쾸", "쾸", "쾸" },
  { "쾹", "쾹", "쾹", "쾹", "쾹" },
  { "쾺", "쾺", "쾺", "쾺", "쾺" },
  { "쾻", "쾻", "쾻", "쾻", "쾻" },
  { "쾼", "쾼", "쾼", "쾼", "쾼" },
  { "쾽", "쾽", "쾽", "쾽", "쾽" },
  { "쾾", "쾾", "쾾", "쾾", "쾾" },
  { "쾿", "쾿", "쾿", "쾿", "쾿" },
  { "쿀", "쿀", "쿀", "쿀", "쿀" },
  { "쿁", "쿁", "쿁", "쿁", "쿁" },
  { "쿂", "쿂", "쿂", "쿂", "쿂" },
  { "쿃", "쿃", "쿃", "쿃", "쿃" },
  { "쿄", "쿄", "쿄", "쿄", "쿄" },
  { "쿅", "쿅", "쿅", "쿅", "쿅" },
  { "쿆", "쿆", "쿆", "쿆", "쿆" },
  { "쿇", "쿇", "쿇", "쿇", "쿇" },
  { "쿈", "쿈", "쿈", "쿈", "쿈" },
  { "쿉", "쿉", "쿉", "쿉", "쿉" },
  { "쿊", "쿊", "쿊", "쿊", "쿊" },
  { "쿋", "쿋", "쿋", "쿋", "쿋" },
  { "쿌", "쿌", "쿌", "쿌", "쿌" },
  { "쿍", "쿍", "쿍", "쿍", "쿍" },
  { "쿎", "쿎", "쿎", "쿎", "쿎" },
  { "쿏", "쿏", "쿏", "쿏", "쿏" },
  { "쿐", "쿐", "쿐", "쿐", "쿐" },
  { "쿑", "쿑", "쿑", "쿑", "쿑" },
  { "쿒", "쿒", "쿒", "쿒", "쿒" },
  { "쿓", "쿓", "쿓", "쿓", "쿓" },
  { "쿔", "쿔", "쿔", "쿔", "쿔" },
  { "쿕", "쿕", "쿕", "쿕", "쿕" },
  { "쿖", "쿖", "쿖", "쿖", "쿖" },
  { "쿗", "쿗", "쿗", "쿗", "쿗" },
  { "쿘", "쿘", "쿘", "쿘", "쿘" },
  { "쿙", "쿙", "쿙", "쿙", "쿙" },
  { "쿚", "쿚", "쿚", "쿚", "쿚" },
  { "쿛", "쿛", "쿛", "쿛", "쿛" },
  { "쿜", "쿜", "쿜", "쿜", "쿜" },
  { "쿝", "쿝", "쿝", "쿝", "쿝" },
  { "쿞", "쿞", "쿞", "쿞", "쿞" },
  { "쿟", "쿟", "쿟", "쿟", "쿟" },
  { "쿠", "쿠", "쿠", "쿠", "쿠" },
  { "쿡", "쿡", "쿡", "쿡", "쿡" },
  { "쿢", "쿢", "쿢", "쿢", "쿢" },
  { "쿣", "쿣", "쿣", "쿣", "쿣" },
  { "쿤", "쿤", "쿤", "쿤", "쿤" },
  { "쿥", "쿥", "쿥", "쿥", "쿥" },
  { "쿦", "쿦", "쿦", "쿦", "쿦" },
  { "쿧", "쿧", "쿧", "쿧", "쿧" },
  { "쿨", "쿨", "쿨", "쿨", "쿨" },
  { "쿩", "쿩", "쿩", "쿩", "쿩" },
  { "쿪", "쿪", "쿪", "쿪", "쿪" },
  { "쿫", "쿫", "쿫", "쿫", "쿫" },
  { "쿬", "쿬", "쿬", "쿬", "쿬" },
  { "쿭", "쿭", "쿭", "쿭", "쿭" },
  { "쿮", "쿮", "쿮", "쿮", "쿮" },
  { "쿯", "쿯", "쿯", "쿯", "쿯" },
  { "쿰", "쿰", "쿰", "쿰", "쿰" },
  { "쿱", "쿱", "쿱", "쿱", "쿱" },
  { "쿲", "쿲", "쿲", "쿲", "쿲" },
  { "쿳", "쿳", "쿳", "쿳", "쿳" },
  { "쿴", "쿴", "쿴", "쿴", "쿴" },
  { "쿵", "쿵", "쿵", "쿵", "쿵" },
  { "쿶", "쿶", "쿶", "쿶", "쿶" },
  { "쿷", "쿷", "쿷", "쿷", "쿷" },
  { "쿸", "쿸", "쿸", "쿸", "쿸" },
  { "쿹", "쿹", "쿹", "쿹", "쿹" },
  { "쿺", "쿺", "쿺", "쿺", "쿺" },
  { "쿻", "쿻", "쿻", "쿻", "쿻" },
  { "쿼", "쿼", "쿼", "쿼", "쿼" },
  { "쿽", "쿽", "쿽", "쿽", "쿽" },
  { "쿾", "쿾", "쿾", "쿾", "쿾" },
  { "쿿", "쿿", "쿿", "쿿", "쿿" },
  { "퀀", "퀀", "퀀", "퀀", "퀀" },
  { "퀁", "퀁", "퀁", "퀁", "퀁" },
  { "퀂", "퀂", "퀂", "퀂", "퀂" },
  { "퀃", "퀃", "퀃", "퀃", "퀃" },
  { "퀄", "퀄", "퀄", "퀄", "퀄" },
  { "퀅", "퀅", "퀅", "퀅", "퀅" },
  { "퀆", "퀆", "퀆", "퀆", "퀆" },
  { "퀇", "퀇", "퀇", "퀇", "퀇" },
  { "퀈", "퀈", "퀈", "퀈", "퀈" },
  { "퀉", "퀉", "퀉", "퀉", "퀉" },
  { "퀊", "퀊", "퀊", "퀊", "퀊" },
  { "퀋", "퀋", "퀋", "퀋", "퀋" },
  { "퀌", "퀌", "퀌", "퀌", "퀌" },
  { "퀍", "퀍", "퀍", "퀍", "퀍" },
  { "퀎", "퀎", "퀎", "퀎", "퀎" },
  { "퀏", "퀏", "퀏", "퀏", "퀏" },
  { "퀐", "퀐", "퀐", "퀐", "퀐" },
  { "퀑", "퀑", "퀑", "퀑", "퀑" },
  { "퀒", "퀒", "퀒", "퀒", "퀒" },
  { "퀓", "퀓", "퀓", "퀓", "퀓" },
  { "퀔", "퀔", "퀔", "퀔", "퀔" },
  { "퀕", "퀕", "퀕", "퀕", "퀕" },
  { "퀖", "퀖", "퀖", "퀖", "퀖" },
  { "퀗", "퀗", "퀗", "퀗", "퀗" },
  { "퀘", "퀘", "퀘", "퀘", "퀘" },
  { "퀙", "퀙", "퀙", "퀙", "퀙" },
  { "퀚", "퀚", "퀚", "퀚", "퀚" },
  { "퀛", "퀛", "퀛", "퀛", "퀛" },
  { "퀜", "퀜", "퀜", "퀜", "퀜" },
  { "퀝", "퀝", "퀝", "퀝", "퀝" },
  { "퀞", "퀞", "퀞", "퀞", "퀞" },
  { "퀟", "퀟", "퀟", "퀟", "퀟" },
  { "퀠", "퀠", "퀠", "퀠", "퀠" },
  { "퀡", "퀡", "퀡", "퀡", "퀡" },
  { "퀢", "퀢", "퀢", "퀢", "퀢" },
  { "퀣", "퀣", "퀣", "퀣", "퀣" },
  { "퀤", "퀤", "퀤", "퀤", "퀤" },
  { "퀥", "퀥", "퀥", "퀥", "퀥" },
  { "퀦", "퀦", "퀦", "퀦", "퀦" },
  { "퀧", "퀧", "퀧", "퀧", "퀧" },
  { "퀨", "퀨", "퀨", "퀨", "퀨" },
  { "퀩", "퀩", "퀩", "퀩", "퀩" },
  { "퀪", "퀪", "퀪", "퀪", "퀪" },
  { "퀫", "퀫", "퀫", "퀫", "퀫" },
  { "퀬", "퀬", "퀬", "퀬", "퀬" },
  { "퀭", "퀭", "퀭", "퀭", "퀭" },
  { "퀮", "퀮", "퀮", "퀮", "퀮" },
  { "퀯", "퀯", "퀯", "퀯", "퀯" },
  { "퀰", "퀰", "퀰", "퀰", "퀰" },
  { "퀱", "퀱", "퀱", "퀱", "퀱" },
  { "퀲", "퀲", "퀲", "퀲", "퀲" },
  { "퀳", "퀳", "퀳", "퀳", "퀳" },
  { "퀴", "퀴", "퀴", "퀴", "퀴" },
  { "퀵", "퀵", "퀵", "퀵", "퀵" },
  { "퀶", "퀶", "퀶", "퀶", "퀶" },
  { "퀷", "퀷", "퀷", "퀷", "퀷" },
  { "퀸", "퀸", "퀸", "퀸", "퀸" },
  { "퀹", "퀹", "퀹", "퀹", "퀹" },
  { "퀺", "퀺", "퀺", "퀺", "퀺" },
  { "퀻", "퀻", "퀻", "퀻", "퀻" },
  { "퀼", "퀼", "퀼", "퀼", "퀼" },
  { "퀽", "퀽", "퀽", "퀽", "퀽" },
  { "퀾", "퀾", "퀾", "퀾", "퀾" },
  { "퀿", "퀿", "퀿", "퀿", "퀿" },
  { "큀", "큀", "큀", "큀", "큀" },
  { "큁", "큁", "큁", "큁", "큁" },
  { "큂", "큂", "큂", "큂", "큂" },
  { "큃", "큃", "큃", "큃", "큃" },
  { "큄", "큄", "큄", "큄", "큄" },
  { "큅", "큅", "큅", "큅", "큅" },
  { "큆", "큆", "큆", "큆", "큆" },
  { "큇", "큇", "큇", "큇", "큇" },
  { "큈", "큈", "큈", "큈", "큈" },
  { "큉", "큉", "큉", "큉", "큉" },
  { "큊", "큊", "큊", "큊", "큊" },
  { "큋", "큋", "큋", "큋", "큋" },
  { "큌", "큌", "큌", "큌", "큌" },
  { "큍", "큍", "큍", "큍", "큍" },
  { "큎", "큎", "큎", "큎", "큎" },
  { "큏", "큏", "큏", "큏", "큏" },
  { "큐", "큐", "큐", "큐", "큐" },
  { "큑", "큑", "큑", "큑", "큑" },
  { "큒", "큒", "큒", "큒", "큒" },
  { "큓", "큓", "큓", "큓", "큓" },
  { "큔", "큔", "큔", "큔", "큔" },
  { "큕", "큕", "큕", "큕", "큕" },
  { "큖", "큖", "큖", "큖", "큖" },
  { "큗", "큗", "큗", "큗", "큗" },
  { "큘", "큘", "큘", "큘", "큘" },
  { "큙", "큙", "큙", "큙", "큙" },
  { "큚", "큚", "큚", "큚", "큚" },
  { "큛", "큛", "큛", "큛", "큛" },
  { "큜", "큜", "큜", "큜", "큜" },
  { "큝", "큝", "큝", "큝", "큝" },
  { "큞", "큞", "큞", "큞", "큞" },
  { "큟", "큟", "큟", "큟", "큟" },
  { "큠", "큠", "큠", "큠", "큠" },
  { "큡", "큡", "큡", "큡", "큡" },
  { "큢", "큢", "큢", "큢", "큢" },
  { "큣", "큣", "큣", "큣", "큣" },
  { "큤", "큤", "큤", "큤", "큤" },
  { "큥", "큥", "큥", "큥", "큥" },
  { "큦", "큦", "큦", "큦", "큦" },
  { "큧", "큧", "큧", "큧", "큧" },
  { "큨", "큨", "큨", "큨", "큨" },
  { "큩", "큩", "큩", "큩", "큩" },
  { "큪", "큪", "큪", "큪", "큪" },
  { "큫", "큫", "큫", "큫", "큫" },
  { "크", "크", "크", "크", "크" },
  { "큭", "큭", "큭", "큭", "큭" },
  { "큮", "큮", "큮", "큮", "큮" },
  { "큯", "큯", "큯", "큯", "큯" },
  { "큰", "큰", "큰", "큰", "큰" },
  { "큱", "큱", "큱", "큱", "큱" },
  { "큲", "큲", "큲", "큲", "큲" },
  { "큳", "큳", "큳", "큳", "큳" },
  { "클", "클", "클", "클", "클" },
  { "큵", "큵", "큵", "큵", "큵" },
  { "큶", "큶", "큶", "큶", "큶" },
  { "큷", "큷", "큷", "큷", "큷" },
  { "큸", "큸", "큸", "큸", "큸" },
  { "큹", "큹", "큹", "큹", "큹" },
  { "큺", "큺", "큺", "큺", "큺" },
  { "큻", "큻", "큻", "큻", "큻" },
  { "큼", "큼", "큼", "큼", "큼" },
  { "큽", "큽", "큽", "큽", "큽" },
  { "큾", "큾", "큾", "큾", "큾" },
  { "큿", "큿", "큿", "큿", "큿" },
  { "킀", "킀", "킀", "킀", "킀" },
  { "킁", "킁", "킁", "킁", "킁" },
  { "킂", "킂", "킂", "킂", "킂" },
  { "킃", "킃", "킃", "킃", "킃" },
  { "킄", "킄", "킄", "킄", "킄" },
  { "킅", "킅", "킅", "킅", "킅" },
  { "킆", "킆", "킆", "킆", "킆" },
  { "킇", "킇", "킇", "킇", "킇" },
  { "킈", "킈", "킈", "킈", "킈" },
  { "킉", "킉", "킉", "킉", "킉" },
  { "킊", "킊", "킊", "킊", "킊" },
  { "킋", "킋", "킋", "킋", "킋" },
  { "킌", "킌", "킌", "킌", "킌" },
  { "킍", "킍", "킍", "킍", "킍" },
  { "킎", "킎", "킎", "킎", "킎" },
  { "킏", "킏", "킏", "킏", "킏" },
  { "킐", "킐", "킐", "킐", "킐" },
  { "킑", "킑", "킑", "킑", "킑" },
  { "킒", "킒", "킒", "킒", "킒" },
  { "킓", "킓", "킓", "킓", "킓" },
  { "킔", "킔", "킔", "킔", "킔" },
  { "킕", "킕", "킕", "킕", "킕" },
  { "킖", "킖", "킖", "킖", "킖" },
  { "킗", "킗", "킗", "킗", "킗" },
  { "킘", "킘", "킘", "킘", "킘" },
  { "킙", "킙", "킙", "킙", "킙" },
  { "킚", "킚", "킚", "킚", "킚" },
  { "킛", "킛", "킛", "킛", "킛" },
  { "킜", "킜", "킜", "킜", "킜" },
  { "킝", "킝", "킝", "킝", "킝" },
  { "킞", "킞", "킞", "킞", "킞" },
  { "킟", "킟", "킟", "킟", "킟" },
  { "킠", "킠", "킠", "킠", "킠" },
  { "킡", "킡", "킡", "킡", "킡" },
  { "킢", "킢", "킢", "킢", "킢" },
  { "킣", "킣", "킣", "킣", "킣" },
  { "키", "키", "키", "키", "키" },
  { "킥", "킥", "킥", "킥", "킥" },
  { "킦", "킦", "킦", "킦", "킦" },
  { "킧", "킧", "킧", "킧", "킧" },
  { "킨", "킨", "킨", "킨", "킨" },
  { "킩", "킩", "킩", "킩", "킩" },
  { "킪", "킪", "킪", "킪", "킪" },
  { "킫", "킫", "킫", "킫", "킫" },
  { "킬", "킬", "킬", "킬", "킬" },
  { "킭", "킭", "킭", "킭", "킭" },
  { "킮", "킮", "킮", "킮", "킮" },
  { "킯", "킯", "킯", "킯", "킯" },
  { "킰", "킰", "킰", "킰", "킰" },
  { "킱", "킱", "킱", "킱", "킱" },
  { "킲", "킲", "킲", "킲", "킲" },
  { "킳", "킳", "킳", "킳", "킳" },
  { "킴", "킴", "킴", "킴", "킴" },
  { "킵", "킵", "킵", "킵", "킵" },
  { "킶", "킶", "킶", "킶", "킶" },
  { "킷", "킷", "킷", "킷", "킷" },
  { "킸", "킸", "킸", "킸", "킸" },
  { "킹", "킹", "킹", "킹", "킹" },
  { "킺", "킺", "킺", "킺", "킺" },
  { "킻", "킻", "킻", "킻", "킻" },
  { "킼", "킼", "킼", "킼", "킼" },
  { "킽", "킽", "킽", "킽", "킽" },
  { "킾", "킾", "킾", "킾", "킾" },
  { "킿", "킿", "킿", "킿", "킿" },
  { "타", "타", "타", "타", "타" },
  { "탁", "탁", "탁", "탁", "탁" },
  { "탂", "탂", "탂", "탂", "탂" },
  { "탃", "탃", "탃", "탃", "탃" },
  { "탄", "탄", "탄", "탄", "탄" },
  { "탅", "탅", "탅", "탅", "탅" },
  { "탆", "탆", "탆", "탆", "탆" },
  { "탇", "탇", "탇", "탇", "탇" },
  { "탈", "탈", "탈", "탈", "탈" },
  { "탉", "탉", "탉", "탉", "탉" },
  { "탊", "탊", "탊", "탊", "탊" },
  { "탋", "탋", "탋", "탋", "탋" },
  { "탌", "탌", "탌", "탌", "탌" },
  { "탍", "탍", "탍", "탍", "탍" },
  { "탎", "탎", "탎", "탎", "탎" },
  { "탏", "탏", "탏", "탏", "탏" },
  { "탐", "탐", "탐", "탐", "탐" },
  { "탑", "탑", "탑", "탑", "탑" },
  { "탒", "탒", "탒", "탒", "탒" },
  { "탓", "탓", "탓", "탓", "탓" },
  { "탔", "탔", "탔", "탔", "탔" },
  { "탕", "탕", "탕", "탕", "탕" },
  { "탖", "탖", "탖", "탖", "탖" },
  { "탗", "탗", "탗", "탗", "탗" },
  { "탘", "탘", "탘", "탘", "탘" },
  { "탙", "탙", "탙", "탙", "탙" },
  { "탚", "탚", "탚", "탚", "탚" },
  { "탛", "탛", "탛", "탛", "탛" },
  { "태", "태", "태", "태", "태" },
  { "택", "택", "택", "택", "택" },
  { "탞", "탞", "탞", "탞", "탞" },
  { "탟", "탟", "탟", "탟", "탟" },
  { "탠", "탠", "탠", "탠", "탠" },
  { "탡", "탡", "탡", "탡", "탡" },
  { "탢", "탢", "탢", "탢", "탢" },
  { "탣", "탣", "탣", "탣", "탣" },
  { "탤", "탤", "탤", "탤", "탤" },
  { "탥", "탥", "탥", "탥", "탥" },
  { "탦", "탦", "탦", "탦", "탦" },
  { "탧", "탧", "탧", "탧", "탧" },
  { "탨", "탨", "탨", "탨", "탨" },
  { "탩", "탩", "탩", "탩", "탩" },
  { "탪", "탪", "탪", "탪", "탪" },
  { "탫", "탫", "탫", "탫", "탫" },
  { "탬", "탬", "탬", "탬", "탬" },
  { "탭", "탭", "탭", "탭", "탭" },
  { "탮", "탮", "탮", "탮", "탮" },
  { "탯", "탯", "탯", "탯", "탯" },
  { "탰", "탰", "탰", "탰", "탰" },
  { "탱", "탱", "탱", "탱", "탱" },
  { "탲", "탲", "탲", "탲", "탲" },
  { "탳", "탳", "탳", "탳", "탳" },
  { "탴", "탴", "탴", "탴", "탴" },
  { "탵", "탵", "탵", "탵", "탵" },
  { "탶", "탶", "탶", "탶", "탶" },
  { "탷", "탷", "탷", "탷", "탷" },
  { "탸", "탸", "탸", "탸", "탸" },
  { "탹", "탹", "탹", "탹", "탹" },
  { "탺", "탺", "탺", "탺", "탺" },
  { "탻", "탻", "탻", "탻", "탻" },
  { "탼", "탼", "탼", "탼", "탼" },
  { "탽", "탽", "탽", "탽", "탽" },
  { "탾", "탾", "탾", "탾", "탾" },
  { "탿", "탿", "탿", "탿", "탿" },
  { "턀", "턀", "턀", "턀", "턀" },
  { "턁", "턁", "턁", "턁", "턁" },
  { "턂", "턂", "턂", "턂", "턂" },
  { "턃", "턃", "턃", "턃", "턃" },
  { "턄", "턄", "턄", "턄", "턄" },
  { "턅", "턅", "턅", "턅", "턅" },
  { "턆", "턆", "턆", "턆", "턆" },
  { "턇", "턇", "턇", "턇", "턇" },
  { "턈", "턈", "턈", "턈", "턈" },
  { "턉", "턉", "턉", "턉", "턉" },
  { "턊", "턊", "턊", "턊", "턊" },
  { "턋", "턋", "턋", "턋", "턋" },
  { "턌", "턌", "턌", "턌", "턌" },
  { "턍", "턍", "턍", "턍", "턍" },
  { "턎", "턎", "턎", "턎", "턎" },
  { "턏", "턏", "턏", "턏", "턏" },
  { "턐", "턐", "턐", "턐", "턐" },
  { "턑", "턑", "턑", "턑", "턑" },
  { "턒", "턒", "턒", "턒", "턒" },
  { "턓", "턓", "턓", "턓", "턓" },
  { "턔", "턔", "턔", "턔", "턔" },
  { "턕", "턕", "턕", "턕", "턕" },
  { "턖", "턖", "턖", "턖", "턖" },
  { "턗", "턗", "턗", "턗", "턗" },
  { "턘", "턘", "턘", "턘", "턘" },
  { "턙", "턙", "턙", "턙", "턙" },
  { "턚", "턚", "턚", "턚", "턚" },
  { "턛", "턛", "턛", "턛", "턛" },
  { "턜", "턜", "턜", "턜", "턜" },
  { "턝", "턝", "턝", "턝", "턝" },
  { "턞", "턞", "턞", "턞", "턞" },
  { "턟", "턟", "턟", "턟", "턟" },
  { "턠", "턠", "턠", "턠", "턠" },
  { "턡", "턡", "턡", "턡", "턡" },
  { "턢", "턢", "턢", "턢", "턢" },
  { "턣", "턣", "턣", "턣", "턣" },
  { "턤", "턤", "턤", "턤", "턤" },
  { "턥", "턥", "턥", "턥", "턥" },
  { "턦", "턦", "턦", "턦", "턦" },
  { "턧", "턧", "턧", "턧", "턧" },
  { "턨", "턨", "턨", "턨", "턨" },
  { "턩", "턩", "턩", "턩", "턩" },
  { "턪", "턪", "턪", "턪", "턪" },
  { "턫", "턫", "턫", "턫", "턫" },
  { "턬", "턬", "턬", "턬", "턬" },
  { "턭", "턭", "턭", "턭", "턭" },
  { "턮", "턮", "턮", "턮", "턮" },
  { "턯", "턯", "턯", "턯", "턯" },
  { "터", "터", "터", "터", "터" },
  { "턱", "턱", "턱", "턱", "턱" },
  { "턲", "턲", "턲", "턲", "턲" },
  { "턳", "턳", "턳", "턳", "턳" },
  { "턴", "턴", "턴", "턴", "턴" },
  { "턵", "턵", "턵", "턵", "턵" },
  { "턶", "턶", "턶", "턶", "턶" },
  { "턷", "턷", "턷", "턷", "턷" },
  { "털", "털", "털", "털", "털" },
  { "턹", "턹", "턹", "턹", "턹" },
  { "턺", "턺", "턺", "턺", "턺" },
  { "턻", "턻", "턻", "턻", "턻" },
  { "턼", "턼", "턼", "턼", "턼" },
  { "턽", "턽", "턽", "턽", "턽" },
  { "턾", "턾", "턾", "턾", "턾" },
  { "턿", "턿", "턿", "턿", "턿" },
  { "텀", "텀", "텀", "텀", "텀" },
  { "텁", "텁", "텁", "텁", "텁" },
  { "텂", "텂", "텂", "텂", "텂" },
  { "텃", "텃", "텃", "텃", "텃" },
  { "텄", "텄", "텄", "텄", "텄" },
  { "텅", "텅", "텅", "텅", "텅" },
  { "텆", "텆", "텆", "텆", "텆" },
  { "텇", "텇", "텇", "텇", "텇" },
  { "텈", "텈", "텈", "텈", "텈" },
  { "텉", "텉", "텉", "텉", "텉" },
  { "텊", "텊", "텊", "텊", "텊" },
  { "텋", "텋", "텋", "텋", "텋" },
  { "테", "테", "테", "테", "테" },
  { "텍", "텍", "텍", "텍", "텍" },
  { "텎", "텎", "텎", "텎", "텎" },
  { "텏", "텏", "텏", "텏", "텏" },
  { "텐", "텐", "텐", "텐", "텐" },
  { "텑", "텑", "텑", "텑", "텑" },
  { "텒", "텒", "텒", "텒", "텒" },
  { "텓", "텓", "텓", "텓", "텓" },
  { "텔", "텔", "텔", "텔", "텔" },
  { "텕", "텕", "텕", "텕", "텕" },
  { "텖", "텖", "텖", "텖", "텖" },
  { "텗", "텗", "텗", "텗", "텗" },
  { "텘", "텘", "텘", "텘", "텘" },
  { "텙", "텙", "텙", "텙", "텙" },
  { "텚", "텚", "텚", "텚", "텚" },
  { "텛", "텛", "텛", "텛", "텛" },
  { "템", "템", "템", "템", "템" },
  { "텝", "텝", "텝", "텝", "텝" },
  { "텞", "텞", "텞", "텞", "텞" },
  { "텟", "텟", "텟", "텟", "텟" },
  { "텠", "텠", "텠", "텠", "텠" },
  { "텡", "텡", "텡", "텡", "텡" },
  { "텢", "텢", "텢", "텢", "텢" },
  { "텣", "텣", "텣", "텣", "텣" },
  { "텤", "텤", "텤", "텤", "텤" },
  { "텥", "텥", "텥", "텥", "텥" },
  { "텦", "텦", "텦", "텦", "텦" },
  { "텧", "텧", "텧", "텧", "텧" },
  { "텨", "텨", "텨", "텨", "텨" },
  { "텩", "텩", "텩", "텩", "텩" },
  { "텪", "텪", "텪", "텪", "텪" },
  { "텫", "텫", "텫", "텫", "텫" },
  { "텬", "텬", "텬", "텬", "텬" },
  { "텭", "텭", "텭", "텭", "텭" },
  { "텮", "텮", "텮", "텮", "텮" },
  { "텯", "텯", "텯", "텯", "텯" },
  { "텰", "텰", "텰", "텰", "텰" },
  { "텱", "텱", "텱", "텱", "텱" },
  { "텲", "텲", "텲", "텲", "텲" },
  { "텳", "텳", "텳", "텳", "텳" },
  { "텴", "텴", "텴", "텴", "텴" },
  { "텵", "텵", "텵", "텵", "텵" },
  { "텶", "텶", "텶", "텶", "텶" },
  { "텷", "텷", "텷", "텷", "텷" },
  { "텸", "텸", "텸", "텸", "텸" },
  { "텹", "텹", "텹", "텹", "텹" },
  { "텺", "텺", "텺", "텺", "텺" },
  { "텻", "텻", "텻", "텻", "텻" },
  { "텼", "텼", "텼", "텼", "텼" },
  { "텽", "텽", "텽", "텽", "텽" },
  { "텾", "텾", "텾", "텾", "텾" },
  { "텿", "텿", "텿", "텿", "텿" },
  { "톀", "톀", "톀", "톀", "톀" },
  { "톁", "톁", "톁", "톁", "톁" },
  { "톂", "톂", "톂", "톂", "톂" },
  { "톃", "톃", "톃", "톃", "톃" },
  { "톄", "톄", "톄", "톄", "톄" },
  { "톅", "톅", "톅", "톅", "톅" },
  { "톆", "톆", "톆", "톆", "톆" },
  { "톇", "톇", "톇", "톇", "톇" },
  { "톈", "톈", "톈", "톈", "톈" },
  { "톉", "톉", "톉", "톉", "톉" },
  { "톊", "톊", "톊", "톊", "톊" },
  { "톋", "톋", "톋", "톋", "톋" },
  { "톌", "톌", "톌", "톌", "톌" },
  { "톍", "톍", "톍", "톍", "톍" },
  { "톎", "톎", "톎", "톎", "톎" },
  { "톏", "톏", "톏", "톏", "톏" },
  { "톐", "톐", "톐", "톐", "톐" },
  { "톑", "톑", "톑", "톑", "톑" },
  { "톒", "톒", "톒", "톒", "톒" },
  { "톓", "톓", "톓", "톓", "톓" },
  { "톔", "톔", "톔", "톔", "톔" },
  { "톕", "톕", "톕", "톕", "톕" },
  { "톖", "톖", "톖", "톖", "톖" },
  { "톗", "톗", "톗", "톗", "톗" },
  { "톘", "톘", "톘", "톘", "톘" },
  { "톙", "톙", "톙", "톙", "톙" },
  { "톚", "톚", "톚", "톚", "톚" },
  { "톛", "톛", "톛", "톛", "톛" },
  { "톜", "톜", "톜", "톜", "톜" },
  { "톝", "톝", "톝", "톝", "톝" },
  { "톞", "톞", "톞", "톞", "톞" },
  { "톟", "톟", "톟", "톟", "톟" },
  { "토", "토", "토", "토", "토" },
  { "톡", "톡", "톡", "톡", "톡" },
  { "톢", "톢", "톢", "톢", "톢" },
  { "톣", "톣", "톣", "톣", "톣" },
  { "톤", "톤", "톤", "톤", "톤" },
  { "톥", "톥", "톥", "톥", "톥" },
  { "톦", "톦", "톦", "톦", "톦" },
  { "톧", "톧", "톧", "톧", "톧" },
  { "톨", "톨", "톨", "톨", "톨" },
  { "톩", "톩", "톩", "톩", "톩" },
  { "톪", "톪", "톪", "톪", "톪" },
  { "톫", "톫", "톫", "톫", "톫" },
  { "톬", "톬", "톬", "톬", "톬" },
  { "톭", "톭", "톭", "톭", "톭" },
  { "톮", "톮", "톮", "톮", "톮" },
  { "톯", "톯", "톯", "톯", "톯" },
  { "톰", "톰", "톰", "톰", "톰" },
  { "톱", "톱", "톱", "톱", "톱" },
  { "톲", "톲", "톲", "톲", "톲" },
  { "톳", "톳", "톳", "톳", "톳" },
  { "톴", "톴", "톴", "톴", "톴" },
  { "통", "통", "통", "통", "통" },
  { "톶", "톶", "톶", "톶", "톶" },
  { "톷", "톷", "톷", "톷", "톷" },
  { "톸", "톸", "톸", "톸", "톸" },
  { "톹", "톹", "톹", "톹", "톹" },
  { "톺", "톺", "톺", "톺", "톺" },
  { "톻", "톻", "톻", "톻", "톻" },
  { "톼", "톼", "톼", "톼", "톼" },
  { "톽", "톽", "톽", "톽", "톽" },
  { "톾", "톾", "톾", "톾", "톾" },
  { "톿", "톿", "톿", "톿", "톿" },
  { "퇀", "퇀", "퇀", "퇀", "퇀" },
  { "퇁", "퇁", "퇁", "퇁", "퇁" },
  { "퇂", "퇂", "퇂", "퇂", "퇂" },
  { "퇃", "퇃", "퇃", "퇃", "퇃" },
  { "퇄", "퇄", "퇄", "퇄", "퇄" },
  { "퇅", "퇅", "퇅", "퇅", "퇅" },
  { "퇆", "퇆", "퇆", "퇆", "퇆" },
  { "퇇", "퇇", "퇇", "퇇", "퇇" },
  { "퇈", "퇈", "퇈", "퇈", "퇈" },
  { "퇉", "퇉", "퇉", "퇉", "퇉" },
  { "퇊", "퇊", "퇊", "퇊", "퇊" },
  { "퇋", "퇋", "퇋", "퇋", "퇋" },
  { "퇌", "퇌", "퇌", "퇌", "퇌" },
  { "퇍", "퇍", "퇍", "퇍", "퇍" },
  { "퇎", "퇎", "퇎", "퇎", "퇎" },
  { "퇏", "퇏", "퇏", "퇏", "퇏" },
  { "퇐", "퇐", "퇐", "퇐", "퇐" },
  { "퇑", "퇑", "퇑", "퇑", "퇑" },
  { "퇒", "퇒", "퇒", "퇒", "퇒" },
  { "퇓", "퇓", "퇓", "퇓", "퇓" },
  { "퇔", "퇔", "퇔", "퇔", "퇔" },
  { "퇕", "퇕", "퇕", "퇕", "퇕" },
  { "퇖", "퇖", "퇖", "퇖", "퇖" },
  { "퇗", "퇗", "퇗", "퇗", "퇗" },
  { "퇘", "퇘", "퇘", "퇘", "퇘" },
  { "퇙", "퇙", "퇙", "퇙", "퇙" },
  { "퇚", "퇚", "퇚", "퇚", "퇚" },
  { "퇛", "퇛", "퇛", "퇛", "퇛" },
  { "퇜", "퇜", "퇜", "퇜", "퇜" },
  { "퇝", "퇝", "퇝", "퇝", "퇝" },
  { "퇞", "퇞", "퇞", "퇞", "퇞" },
  { "퇟", "퇟", "퇟", "퇟", "퇟" },
  { "퇠", "퇠", "퇠", "퇠", "퇠" },
  { "퇡", "퇡", "퇡", "퇡", "퇡" },
  { "퇢", "퇢", "퇢", "퇢", "퇢" },
  { "퇣", "퇣", "퇣", "퇣", "퇣" },
  { "퇤", "퇤", "퇤", "퇤", "퇤" },
  { "퇥", "퇥", "퇥", "퇥", "퇥" },
  { "퇦", "퇦", "퇦", "퇦", "퇦" },
  { "퇧", "퇧", "퇧", "퇧", "퇧" },
  { "퇨", "퇨", "퇨", "퇨", "퇨" },
  { "퇩", "퇩", "퇩", "퇩", "퇩" },
  { "퇪", "퇪", "퇪", "퇪", "퇪" },
  { "퇫", "퇫", "퇫", "퇫", "퇫" },
  { "퇬", "퇬", "퇬", "퇬", "퇬" },
  { "퇭", "퇭", "퇭", "퇭", "퇭" },
  { "퇮", "퇮", "퇮", "퇮", "퇮" },
  { "퇯", "퇯", "퇯", "퇯", "퇯" },
  { "퇰", "퇰", "퇰", "퇰", "퇰" },
  { "퇱", "퇱", "퇱", "퇱", "퇱" },
  { "퇲", "퇲", "퇲", "퇲", "퇲" },
  { "퇳", "퇳", "퇳", "퇳", "퇳" },
  { "퇴", "퇴", "퇴", "퇴", "퇴" },
  { "퇵", "퇵", "퇵", "퇵", "퇵" },
  { "퇶", "퇶", "퇶", "퇶", "퇶" },
  { "퇷", "퇷", "퇷", "퇷", "퇷" },
  { "퇸", "퇸", "퇸", "퇸", "퇸" },
  { "퇹", "퇹", "퇹", "퇹", "퇹" },
  { "퇺", "퇺", "퇺", "퇺", "퇺" },
  { "퇻", "퇻", "퇻", "퇻", "퇻" },
  { "퇼", "퇼", "퇼", "퇼", "퇼" },
  { "퇽", "퇽", "퇽", "퇽", "퇽" },
  { "퇾", "퇾", "퇾", "퇾", "퇾" },
  { "퇿", "퇿", "퇿", "퇿", "퇿" },
  { "툀", "툀", "툀", "툀", "툀" },
  { "툁", "툁", "툁", "툁", "툁" },
  { "툂", "툂", "툂", "툂", "툂" },
  { "툃", "툃", "툃", "툃", "툃" },
  { "툄", "툄", "툄", "툄", "툄" },
  { "툅", "툅", "툅", "툅", "툅" },
  { "툆", "툆", "툆", "툆", "툆" },
  { "툇", "툇", "툇", "툇", "툇" },
  { "툈", "툈", "툈", "툈", "툈" },
  { "툉", "툉", "툉", "툉", "툉" },
  { "툊", "툊", "툊", "툊", "툊" },
  { "툋", "툋", "툋", "툋", "툋" },
  { "툌", "툌", "툌", "툌", "툌" },
  { "툍", "툍", "툍", "툍", "툍" },
  { "툎", "툎", "툎", "툎", "툎" },
  { "툏", "툏", "툏", "툏", "툏" },
  { "툐", "툐", "툐", "툐", "툐" },
  { "툑", "툑", "툑", "툑", "툑" },
  { "툒", "툒", "툒", "툒", "툒" },
  { "툓", "툓", "툓", "툓", "툓" },
  { "툔", "툔", "툔", "툔", "툔" },
  { "툕", "툕", "툕", "툕", "툕" },
  { "툖", "툖", "툖", "툖", "툖" },
  { "툗", "툗", "툗", "툗", "툗" },
  { "툘", "툘", "툘", "툘", "툘" },
  { "툙", "툙", "툙", "툙", "툙" },
  { "툚", "툚", "툚", "툚", "툚" },
  { "툛", "툛", "툛", "툛", "툛" },
  { "툜", "툜", "툜", "툜", "툜" },
  { "툝", "툝", "툝", "툝", "툝" },
  { "툞", "툞", "툞", "툞", "툞" },
  { "툟", "툟", "툟", "툟", "툟" },
  { "툠", "툠", "툠", "툠", "툠" },
  { "툡", "툡", "툡", "툡", "툡" },
  { "툢", "툢", "툢", "툢", "툢" },
  { "툣", "툣", "툣", "툣", "툣" },
  { "툤", "툤", "툤", "툤", "툤" },
  { "툥", "툥", "툥", "툥", "툥" },
  { "툦", "툦", "툦", "툦", "툦" },
  { "툧", "툧", "툧", "툧", "툧" },
  { "툨", "툨", "툨", "툨", "툨" },
  { "툩", "툩", "툩", "툩", "툩" },
  { "툪", "툪", "툪", "툪", "툪" },
  { "툫", "툫", "툫", "툫", "툫" },
  { "투", "투", "투", "투", "투" },
  { "툭", "툭", "툭", "툭", "툭" },
  { "툮", "툮", "툮", "툮", "툮" },
  { "툯", "툯", "툯", "툯", "툯" },
  { "툰", "툰", "툰", "툰", "툰" },
  { "툱", "툱", "툱", "툱", "툱" },
  { "툲", "툲", "툲", "툲", "툲" },
  { "툳", "툳", "툳", "툳", "툳" },
  { "툴", "툴", "툴", "툴", "툴" },
  { "툵", "툵", "툵", "툵", "툵" },
  { "툶", "툶", "툶", "툶", "툶" },
  { "툷", "툷", "툷", "툷", "툷" },
  { "툸", "툸", "툸", "툸", "툸" },
  { "툹", "툹", "툹", "툹", "툹" },
  { "툺", "툺", "툺", "툺", "툺" },
  { "툻", "툻", "툻", "툻", "툻" },
  { "툼", "툼", "툼", "툼", "툼" },
  { "툽", "툽", "툽", "툽", "툽" },
  { "툾", "툾", "툾", "툾", "툾" },
  { "툿", "툿", "툿", "툿", "툿" },
  { "퉀", "퉀", "퉀", "퉀", "퉀" },
  { "퉁", "퉁", "퉁", "퉁", "퉁" },
  { "퉂", "퉂", "퉂", "퉂", "퉂" },
  { "퉃", "퉃", "퉃", "퉃", "퉃" },
  { "퉄", "퉄", "퉄", "퉄", "퉄" },
  { "퉅", "퉅", "퉅", "퉅", "퉅" },
  { "퉆", "퉆", "퉆", "퉆", "퉆" },
  { "퉇", "퉇", "퉇", "퉇", "퉇" },
  { "퉈", "퉈", "퉈", "퉈", "퉈" },
  { "퉉", "퉉", "퉉", "퉉", "퉉" },
  { "퉊", "퉊", "퉊", "퉊", "퉊" },
  { "퉋", "퉋", "퉋", "퉋", "퉋" },
  { "퉌", "퉌", "퉌", "퉌", "퉌" },
  { "퉍", "퉍", "퉍", "퉍", "퉍" },
  { "퉎", "퉎", "퉎", "퉎", "퉎" },
  { "퉏", "퉏", "퉏", "퉏", "퉏" },
  { "퉐", "퉐", "퉐", "퉐", "퉐" },
  { "퉑", "퉑", "퉑", "퉑", "퉑" },
  { "퉒", "퉒", "퉒", "퉒", "퉒" },
  { "퉓", "퉓", "퉓", "퉓", "퉓" },
  { "퉔", "퉔", "퉔", "퉔", "퉔" },
  { "퉕", "퉕", "퉕", "퉕", "퉕" },
  { "퉖", "퉖", "퉖", "퉖", "퉖" },
  { "퉗", "퉗", "퉗", "퉗", "퉗" },
  { "퉘", "퉘", "퉘", "퉘", "퉘" },
  { "퉙", "퉙", "퉙", "퉙", "퉙" },
  { "퉚", "퉚", "퉚", "퉚", "퉚" },
  { "퉛", "퉛", "퉛", "퉛", "퉛" },
  { "퉜", "퉜", "퉜", "퉜", "퉜" },
  { "퉝", "퉝", "퉝", "퉝", "퉝" },
  { "퉞", "퉞", "퉞", "퉞", "퉞" },
  { "퉟", "퉟", "퉟", "퉟", "퉟" },
  { "퉠", "퉠", "퉠", "퉠", "퉠" },
  { "퉡", "퉡", "퉡", "퉡", "퉡" },
  { "퉢", "퉢", "퉢", "퉢", "퉢" },
  { "퉣", "퉣", "퉣", "퉣", "퉣" },
  { "퉤", "퉤", "퉤", "퉤", "퉤" },
  { "퉥", "퉥", "퉥", "퉥", "퉥" },
  { "퉦", "퉦", "퉦", "퉦", "퉦" },
  { "퉧", "퉧", "퉧", "퉧", "퉧" },
  { "퉨", "퉨", "퉨", "퉨", "퉨" },
  { "퉩", "퉩", "퉩", "퉩", "퉩" },
  { "퉪", "퉪", "퉪", "퉪", "퉪" },
  { "퉫", "퉫", "퉫", "퉫", "퉫" },
  { "퉬", "퉬", "퉬", "퉬", "퉬" },
  { "퉭", "퉭", "퉭", "퉭", "퉭" },
  { "퉮", "퉮", "퉮", "퉮", "퉮" },
  { "퉯", "퉯", "퉯", "퉯", "퉯" },
  { "퉰", "퉰", "퉰", "퉰", "퉰" },
  { "퉱", "퉱", "퉱", "퉱", "퉱" },
  { "퉲", "퉲", "퉲", "퉲", "퉲" },
  { "퉳", "퉳", "퉳", "퉳", "퉳" },
  { "퉴", "퉴", "퉴", "퉴", "퉴" },
  { "퉵", "퉵", "퉵", "퉵", "퉵" },
  { "퉶", "퉶", "퉶", "퉶", "퉶" },
  { "퉷", "퉷", "퉷", "퉷", "퉷" },
  { "퉸", "퉸", "퉸", "퉸", "퉸" },
  { "퉹", "퉹", "퉹", "퉹", "퉹" },
  { "퉺", "퉺", "퉺", "퉺", "퉺" },
  { "퉻", "퉻", "퉻", "퉻", "퉻" },
  { "퉼", "퉼", "퉼", "퉼", "퉼" },
  { "퉽", "퉽", "퉽", "퉽", "퉽" },
  { "퉾", "퉾", "퉾", "퉾", "퉾" },
  { "퉿", "퉿", "퉿", "퉿", "퉿" },
  { "튀", "튀", "튀", "튀", "튀" },
  { "튁", "튁", "튁", "튁", "튁" },
  { "튂", "튂", "튂", "튂", "튂" },
  { "튃", "튃", "튃", "튃", "튃" },
  { "튄", "튄", "튄", "튄", "튄" },
  { "튅", "튅", "튅", "튅", "튅" },
  { "튆", "튆", "튆", "튆", "튆" },
  { "튇", "튇", "튇", "튇", "튇" },
  { "튈", "튈", "튈", "튈", "튈" },
  { "튉", "튉", "튉", "튉", "튉" },
  { "튊", "튊", "튊", "튊", "튊" },
  { "튋", "튋", "튋", "튋", "튋" },
  { "튌", "튌", "튌", "튌", "튌" },
  { "튍", "튍", "튍", "튍", "튍" },
  { "튎", "튎", "튎", "튎", "튎" },
  { "튏", "튏", "튏", "튏", "튏" },
  { "튐", "튐", "튐", "튐", "튐" },
  { "튑", "튑", "튑", "튑", "튑" },
  { "튒", "튒", "튒", "튒", "튒" },
  { "튓", "튓", "튓", "튓", "튓" },
  { "튔", "튔", "튔", "튔", "튔" },
  { "튕", "튕", "튕", "튕", "튕" },
  { "튖", "튖", "튖", "튖", "튖" },
  { "튗", "튗", "튗", "튗", "튗" },
  { "튘", "튘", "튘", "튘", "튘" },
  { "튙", "튙", "튙", "튙", "튙" },
  { "튚", "튚", "튚", "튚", "튚" },
  { "튛", "튛", "튛", "튛", "튛" },
  { "튜", "튜", "튜", "튜", "튜" },
  { "튝", "튝", "튝", "튝", "튝" },
  { "튞", "튞", "튞", "튞", "튞" },
  { "튟", "튟", "튟", "튟", "튟" },
  { "튠", "튠", "튠", "튠", "튠" },
  { "튡", "튡", "튡", "튡", "튡" },
  { "튢", "튢", "튢", "튢", "튢" },
  { "튣", "튣", "튣", "튣", "튣" },
  { "튤", "튤", "튤", "튤", "튤" },
  { "튥", "튥", "튥", "튥", "튥" },
  { "튦", "튦", "튦", "튦", "튦" },
  { "튧", "튧", "튧", "튧", "튧" },
  { "튨", "튨", "튨", "튨", "튨" },
  { "튩", "튩", "튩", "튩", "튩" },
  { "튪", "튪", "튪", "튪", "튪" },
  { "튫", "튫", "튫", "튫", "튫" },
  { "튬", "튬", "튬", "튬", "튬" },
  { "튭", "튭", "튭", "튭", "튭" },
  { "튮", "튮", "튮", "튮", "튮" },
  { "튯", "튯", "튯", "튯", "튯" },
  { "튰", "튰", "튰", "튰", "튰" },
  { "튱", "튱", "튱", "튱", "튱" },
  { "튲", "튲", "튲", "튲", "튲" },
  { "튳", "튳", "튳", "튳", "튳" },
  { "튴", "튴", "튴", "튴", "튴" },
  { "튵", "튵", "튵", "튵", "튵" },
  { "튶", "튶", "튶", "튶", "튶" },
  { "튷", "튷", "튷", "튷", "튷" },
  { "트", "트", "트", "트", "트" },
  { "특", "특", "특", "특", "특" },
  { "튺", "튺", "튺", "튺", "튺" },
  { "튻", "튻", "튻", "튻", "튻" },
  { "튼", "튼", "튼", "튼", "튼" },
  { "튽", "튽", "튽", "튽", "튽" },
  { "튾", "튾", "튾", "튾", "튾" },
  { "튿", "튿", "튿", "튿", "튿" },
  { "틀", "틀", "틀", "틀", "틀" },
  { "틁", "틁", "틁", "틁", "틁" },
  { "틂", "틂", "틂", "틂", "틂" },
  { "틃", "틃", "틃", "틃", "틃" },
  { "틄", "틄", "틄", "틄", "틄" },
  { "틅", "틅", "틅", "틅", "틅" },
  { "틆", "틆", "틆", "틆", "틆" },
  { "틇", "틇", "틇", "틇", "틇" },
  { "틈", "틈", "틈", "틈", "틈" },
  { "틉", "틉", "틉", "틉", "틉" },
  { "틊", "틊", "틊", "틊", "틊" },
  { "틋", "틋", "틋", "틋", "틋" },
  { "틌", "틌", "틌", "틌", "틌" },
  { "틍", "틍", "틍", "틍", "틍" },
  { "틎", "틎", "틎", "틎", "틎" },
  { "틏", "틏", "틏", "틏", "틏" },
  { "틐", "틐", "틐", "틐", "틐" },
  { "틑", "틑", "틑", "틑", "틑" },
  { "틒", "틒", "틒", "틒", "틒" },
  { "틓", "틓", "틓", "틓", "틓" },
  { "틔", "틔", "틔", "틔", "틔" },
  { "틕", "틕", "틕", "틕", "틕" },
  { "틖", "틖", "틖", "틖", "틖" },
  { "틗", "틗", "틗", "틗", "틗" },
  { "틘", "틘", "틘", "틘", "틘" },
  { "틙", "틙", "틙", "틙", "틙" },
  { "틚", "틚", "틚", "틚", "틚" },
  { "틛", "틛", "틛", "틛", "틛" },
  { "틜", "틜", "틜", "틜", "틜" },
  { "틝", "틝", "틝", "틝", "틝" },
  { "틞", "틞", "틞", "틞", "틞" },
  { "틟", "틟", "틟", "틟", "틟" },
  { "틠", "틠", "틠", "틠", "틠" },
  { "틡", "틡", "틡", "틡", "틡" },
  { "틢", "틢", "틢", "틢", "틢" },
  { "틣", "틣", "틣", "틣", "틣" },
  { "틤", "틤", "틤", "틤", "틤" },
  { "틥", "틥", "틥", "틥", "틥" },
  { "틦", "틦", "틦", "틦", "틦" },
  { "틧", "틧", "틧", "틧", "틧" },
  { "틨", "틨", "틨", "틨", "틨" },
  { "틩", "틩", "틩", "틩", "틩" },
  { "틪", "틪", "틪", "틪", "틪" },
  { "틫", "틫", "틫", "틫", "틫" },
  { "틬", "틬", "틬", "틬", "틬" },
  { "틭", "틭", "틭", "틭", "틭" },
  { "틮", "틮", "틮", "틮", "틮" },
  { "틯", "틯", "틯", "틯", "틯" },
  { "티", "티", "티", "티", "티" },
  { "틱", "틱", "틱", "틱", "틱" },
  { "틲", "틲", "틲", "틲", "틲" },
  { "틳", "틳", "틳", "틳", "틳" },
  { "틴", "틴", "틴", "틴", "틴" },
  { "틵", "틵", "틵", "틵", "틵" },
  { "틶", "틶", "틶", "틶", "틶" },
  { "틷", "틷", "틷", "틷", "틷" },
  { "틸", "틸", "틸", "틸", "틸" },
  { "틹", "틹", "틹", "틹", "틹" },
  { "틺", "틺", "틺", "틺", "틺" },
  { "틻", "틻", "틻", "틻", "틻" },
  { "틼", "틼", "틼", "틼", "틼" },
  { "틽", "틽", "틽", "틽", "틽" },
  { "틾", "틾", "틾", "틾", "틾" },
  { "틿", "틿", "틿", "틿", "틿" },
  { "팀", "팀", "팀", "팀", "팀" },
  { "팁", "팁", "팁", "팁", "팁" },
  { "팂", "팂", "팂", "팂", "팂" },
  { "팃", "팃", "팃", "팃", "팃" },
  { "팄", "팄", "팄", "팄", "팄" },
  { "팅", "팅", "팅", "팅", "팅" },
  { "팆", "팆", "팆", "팆", "팆" },
  { "팇", "팇", "팇", "팇", "팇" },
  { "팈", "팈", "팈", "팈", "팈" },
  { "팉", "팉", "팉", "팉", "팉" },
  { "팊", "팊", "팊", "팊", "팊" },
  { "팋", "팋", "팋", "팋", "팋" },
  { "파", "파", "파", "파", "파" },
  { "팍", "팍", "팍", "팍", "팍" },
  { "팎", "팎", "팎", "팎", "팎" },
  { "팏", "팏", "팏", "팏", "팏" },
  { "판", "판", "판", "판", "판" },
  { "팑", "팑", "팑", "팑", "팑" },
  { "팒", "팒", "팒", "팒", "팒" },
  { "팓", "팓", "팓", "팓", "팓" },
  { "팔", "팔", "팔", "팔", "팔" },
  { "팕", "팕", "팕", "팕", "팕" },
  { "팖", "팖", "팖", "팖", "팖" },
  { "팗", "팗", "팗", "팗", "팗" },
  { "팘", "팘", "팘", "팘", "팘" },
  { "팙", "팙", "팙", "팙", "팙" },
  { "팚", "팚", "팚", "팚", "팚" },
  { "팛", "팛", "팛", "팛", "팛" },
  { "팜", "팜", "팜", "팜", "팜" },
  { "팝", "팝", "팝", "팝", "팝" },
  { "팞", "팞", "팞", "팞", "팞" },
  { "팟", "팟", "팟", "팟", "팟" },
  { "팠", "팠", "팠", "팠", "팠" },
  { "팡", "팡", "팡", "팡", "팡" },
  { "팢", "팢", "팢", "팢", "팢" },
  { "팣", "팣", "팣", "팣", "팣" },
  { "팤", "팤", "팤", "팤", "팤" },
  { "팥", "팥", "팥", "팥", "팥" },
  { "팦", "팦", "팦", "팦", "팦" },
  { "팧", "팧", "팧", "팧", "팧" },
  { "패", "패", "패", "패", "패" },
  { "팩", "팩", "팩", "팩", "팩" },
  { "팪", "팪", "팪", "팪", "팪" },
  { "팫", "팫", "팫", "팫", "팫" },
  { "팬", "팬", "팬", "팬", "팬" },
  { "팭", "팭", "팭", "팭", "팭" },
  { "팮", "팮", "팮", "팮", "팮" },
  { "팯", "팯", "팯", "팯", "팯" },
  { "팰", "팰", "팰", "팰", "팰" },
  { "팱", "팱", "팱", "팱", "팱" },
  { "팲", "팲", "팲", "팲", "팲" },
  { "팳", "팳", "팳", "팳", "팳" },
  { "팴", "팴", "팴", "팴", "팴" },
  { "팵", "팵", "팵", "팵", "팵" },
  { "팶", "팶", "팶", "팶", "팶" },
  { "팷", "팷", "팷", "팷", "팷" },
  { "팸", "팸", "팸", "팸", "팸" },
  { "팹", "팹", "팹", "팹", "팹" },
  { "팺", "팺", "팺", "팺", "팺" },
  { "팻", "팻", "팻", "팻", "팻" },
  { "팼", "팼", "팼", "팼", "팼" },
  { "팽", "팽", "팽", "팽", "팽" },
  { "팾", "팾", "팾", "팾", "팾" },
  { "팿", "팿", "팿", "팿", "팿" },
  { "퍀", "퍀", "퍀", "퍀", "퍀" },
  { "퍁", "퍁", "퍁", "퍁", "퍁" },
  { "퍂", "퍂", "퍂", "퍂", "퍂" },
  { "퍃", "퍃", "퍃", "퍃", "퍃" },
  { "퍄", "퍄", "퍄", "퍄", "퍄" },
  { "퍅", "퍅", "퍅", "퍅", "퍅" },
  { "퍆", "퍆", "퍆", "퍆", "퍆" },
  { "퍇", "퍇", "퍇", "퍇", "퍇" },
  { "퍈", "퍈", "퍈", "퍈", "퍈" },
  { "퍉", "퍉", "퍉", "퍉", "퍉" },
  { "퍊", "퍊", "퍊", "퍊", "퍊" },
  { "퍋", "퍋", "퍋", "퍋", "퍋" },
  { "퍌", "퍌", "퍌", "퍌", "퍌" },
  { "퍍", "퍍", "퍍", "퍍", "퍍" },
  { "퍎", "퍎", "퍎", "퍎", "퍎" },
  { "퍏", "퍏", "퍏", "퍏", "퍏" },
  { "퍐", "퍐", "퍐", "퍐", "퍐" },
  { "퍑", "퍑", "퍑", "퍑", "퍑" },
  { "퍒", "퍒", "퍒", "퍒", "퍒" },
  { "퍓", "퍓", "퍓", "퍓", "퍓" },
  { "퍔", "퍔", "퍔", "퍔", "퍔" },
  { "퍕", "퍕", "퍕", "퍕", "퍕" },
  { "퍖", "퍖", "퍖", "퍖", "퍖" },
  { "퍗", "퍗", "퍗", "퍗", "퍗" },
  { "퍘", "퍘", "퍘", "퍘", "퍘" },
  { "퍙", "퍙", "퍙", "퍙", "퍙" },
  { "퍚", "퍚", "퍚", "퍚", "퍚" },
  { "퍛", "퍛", "퍛", "퍛", "퍛" },
  { "퍜", "퍜", "퍜", "퍜", "퍜" },
  { "퍝", "퍝", "퍝", "퍝", "퍝" },
  { "퍞", "퍞", "퍞", "퍞", "퍞" },
  { "퍟", "퍟", "퍟", "퍟", "퍟" },
  { "퍠", "퍠", "퍠", "퍠", "퍠" },
  { "퍡", "퍡", "퍡", "퍡", "퍡" },
  { "퍢", "퍢", "퍢", "퍢", "퍢" },
  { "퍣", "퍣", "퍣", "퍣", "퍣" },
  { "퍤", "퍤", "퍤", "퍤", "퍤" },
  { "퍥", "퍥", "퍥", "퍥", "퍥" },
  { "퍦", "퍦", "퍦", "퍦", "퍦" },
  { "퍧", "퍧", "퍧", "퍧", "퍧" },
  { "퍨", "퍨", "퍨", "퍨", "퍨" },
  { "퍩", "퍩", "퍩", "퍩", "퍩" },
  { "퍪", "퍪", "퍪", "퍪", "퍪" },
  { "퍫", "퍫", "퍫", "퍫", "퍫" },
  { "퍬", "퍬", "퍬", "퍬", "퍬" },
  { "퍭", "퍭", "퍭", "퍭", "퍭" },
  { "퍮", "퍮", "퍮", "퍮", "퍮" },
  { "퍯", "퍯", "퍯", "퍯", "퍯" },
  { "퍰", "퍰", "퍰", "퍰", "퍰" },
  { "퍱", "퍱", "퍱", "퍱", "퍱" },
  { "퍲", "퍲", "퍲", "퍲", "퍲" },
  { "퍳", "퍳", "퍳", "퍳", "퍳" },
  { "퍴", "퍴", "퍴", "퍴", "퍴" },
  { "퍵", "퍵", "퍵", "퍵", "퍵" },
  { "퍶", "퍶", "퍶", "퍶", "퍶" },
  { "퍷", "퍷", "퍷", "퍷", "퍷" },
  { "퍸", "퍸", "퍸", "퍸", "퍸" },
  { "퍹", "퍹", "퍹", "퍹", "퍹" },
  { "퍺", "퍺", "퍺", "퍺", "퍺" },
  { "퍻", "퍻", "퍻", "퍻", "퍻" },
  { "퍼", "퍼", "퍼", "퍼", "퍼" },
  { "퍽", "퍽", "퍽", "퍽", "퍽" },
  { "퍾", "퍾", "퍾", "퍾", "퍾" },
  { "퍿", "퍿", "퍿", "퍿", "퍿" },
  { "펀", "펀", "펀", "펀", "펀" },
  { "펁", "펁", "펁", "펁", "펁" },
  { "펂", "펂", "펂", "펂", "펂" },
  { "펃", "펃", "펃", "펃", "펃" },
  { "펄", "펄", "펄", "펄", "펄" },
  { "펅", "펅", "펅", "펅", "펅" },
  { "펆", "펆", "펆", "펆", "펆" },
  { "펇", "펇", "펇", "펇", "펇" },
  { "펈", "펈", "펈", "펈", "펈" },
  { "펉", "펉", "펉", "펉", "펉" },
  { "펊", "펊", "펊", "펊", "펊" },
  { "펋", "펋", "펋", "펋", "펋" },
  { "펌", "펌", "펌", "펌", "펌" },
  { "펍", "펍", "펍", "펍", "펍" },
  { "펎", "펎", "펎", "펎", "펎" },
  { "펏", "펏", "펏", "펏", "펏" },
  { "펐", "펐", "펐", "펐", "펐" },
  { "펑", "펑", "펑", "펑", "펑" },
  { "펒", "펒", "펒", "펒", "펒" },
  { "펓", "펓", "펓", "펓", "펓" },
  { "펔", "펔", "펔", "펔", "펔" },
  { "펕", "펕", "펕", "펕", "펕" },
  { "펖", "펖", "펖", "펖", "펖" },
  { "펗", "펗", "펗", "펗", "펗" },
  { "페", "페", "페", "페", "페" },
  { "펙", "펙", "펙", "펙", "펙" },
  { "펚", "펚", "펚", "펚", "펚" },
  { "펛", "펛", "펛", "펛", "펛" },
  { "펜", "펜", "펜", "펜", "펜" },
  { "펝", "펝", "펝", "펝", "펝" },
  { "펞", "펞", "펞", "펞", "펞" },
  { "펟", "펟", "펟", "펟", "펟" },
  { "펠", "펠", "펠", "펠", "펠" },
  { "펡", "펡", "펡", "펡", "펡" },
  { "펢", "펢", "펢", "펢", "펢" },
  { "펣", "펣", "펣", "펣", "펣" },
  { "펤", "펤", "펤", "펤", "펤" },
  { "펥", "펥", "펥", "펥", "펥" },
  { "펦", "펦", "펦", "펦", "펦" },
  { "펧", "펧", "펧", "펧", "펧" },
  { "펨", "펨", "펨", "펨", "펨" },
  { "펩", "펩", "펩", "펩", "펩" },
  { "펪", "펪", "펪", "펪", "펪" },
  { "펫", "펫", "펫", "펫", "펫" },
  { "펬", "펬", "펬", "펬", "펬" },
  { "펭", "펭", "펭", "펭", "펭" },
  { "펮", "펮", "펮", "펮", "펮" },
  { "펯", "펯", "펯", "펯", "펯" },
  { "펰", "펰", "펰", "펰", "펰" },
  { "펱", "펱", "펱", "펱", "펱" },
  { "펲", "펲", "펲", "펲", "펲" },
  { "펳", "펳", "펳", "펳", "펳" },
  { "펴", "펴", "펴", "펴", "펴" },
  { "펵", "펵", "펵", "펵", "펵" },
  { "펶", "펶", "펶", "펶", "펶" },
  { "펷", "펷", "펷", "펷", "펷" },
  { "편", "편", "편", "편", "편" },
  { "펹", "펹", "펹", "펹", "펹" },
  { "펺", "펺", "펺", "펺", "펺" },
  { "펻", "펻", "펻", "펻", "펻" },
  { "펼", "펼", "펼", "펼", "펼" },
  { "펽", "펽", "펽", "펽", "펽" },
  { "펾", "펾", "펾", "펾", "펾" },
  { "펿", "펿", "펿", "펿", "펿" },
  { "폀", "폀", "폀", "폀", "폀" },
  { "폁", "폁", "폁", "폁", "폁" },
  { "폂", "폂", "폂", "폂", "폂" },
  { "폃", "폃", "폃", "폃", "폃" },
  { "폄", "폄", "폄", "폄", "폄" },
  { "폅", "폅", "폅", "폅", "폅" },
  { "폆", "폆", "폆", "폆", "폆" },
  { "폇", "폇", "폇", "폇", "폇" },
  { "폈", "폈", "폈", "폈", "폈" },
  { "평", "평", "평", "평", "평" },
  { "폊", "폊", "폊", "폊", "폊" },
  { "폋", "폋", "폋", "폋", "폋" },
  { "폌", "폌", "폌", "폌", "폌" },
  { "폍", "폍", "폍", "폍", "폍" },
  { "폎", "폎", "폎", "폎", "폎" },
  { "폏", "폏", "폏", "폏", "폏" },
  { "폐", "폐", "폐", "폐", "폐" },
  { "폑", "폑", "폑", "폑", "폑" },
  { "폒", "폒", "폒", "폒", "폒" },
  { "폓", "폓", "폓", "폓", "폓" },
  { "폔", "폔", "폔", "폔", "폔" },
  { "폕", "폕", "폕", "폕", "폕" },
  { "폖", "폖", "폖", "폖", "폖" },
  { "폗", "폗", "폗", "폗", "폗" },
  { "폘", "폘", "폘", "폘", "폘" },
  { "폙", "폙", "폙", "폙", "폙" },
  { "폚", "폚", "폚", "폚", "폚" },
  { "폛", "폛", "폛", "폛", "폛" },
  { "폜", "폜", "폜", "폜", "폜" },
  { "폝", "폝", "폝", "폝", "폝" },
  { "폞", "폞", "폞", "폞", "폞" },
  { "폟", "폟", "폟", "폟", "폟" },
  { "폠", "폠", "폠", "폠", "폠" },
  { "폡", "폡", "폡", "폡", "폡" },
  { "폢", "폢", "폢", "폢", "폢" },
  { "폣", "폣", "폣", "폣", "폣" },
  { "폤", "폤", "폤", "폤", "폤" },
  { "폥", "폥", "폥", "폥", "폥" },
  { "폦", "폦", "폦", "폦", "폦" },
  { "폧", "폧", "폧", "폧", "폧" },
  { "폨", "폨", "폨", "폨", "폨" },
  { "폩", "폩", "폩", "폩", "폩" },
  { "폪", "폪", "폪", "폪", "폪" },
  { "폫", "폫", "폫", "폫", "폫" },
  { "포", "포", "포", "포", "포" },
  { "폭", "폭", "폭", "폭", "폭" },
  { "폮", "폮", "폮", "폮", "폮" },
  { "폯", "폯", "폯", "폯", "폯" },
  { "폰", "폰", "폰", "폰", "폰" },
  { "폱", "폱", "폱", "폱", "폱" },
  { "폲", "폲", "폲", "폲", "폲" },
  { "폳", "폳", "폳", "폳", "폳" },
  { "폴", "폴", "폴", "폴", "폴" },
  { "폵", "폵", "폵", "폵", "폵" },
  { "폶", "폶", "폶", "폶", "폶" },
  { "폷", "폷", "폷", "폷", "폷" },
  { "폸", "폸", "폸", "폸", "폸" },
  { "폹", "폹", "폹", "폹", "폹" },
  { "폺", "폺", "폺", "폺", "폺" },
  { "폻", "폻", "폻", "폻", "폻" },
  { "폼", "폼", "폼", "폼", "폼" },
  { "폽", "폽", "폽", "폽", "폽" },
  { "폾", "폾", "폾", "폾", "폾" },
  { "폿", "폿", "폿", "폿", "폿" },
  { "퐀", "퐀", "퐀", "퐀", "퐀" },
  { "퐁", "퐁", "퐁", "퐁", "퐁" },
  { "퐂", "퐂", "퐂", "퐂", "퐂" },
  { "퐃", "퐃", "퐃", "퐃", "퐃" },
  { "퐄", "퐄", "퐄", "퐄", "퐄" },
  { "퐅", "퐅", "퐅", "퐅", "퐅" },
  { "퐆", "퐆", "퐆", "퐆", "퐆" },
  { "퐇", "퐇", "퐇", "퐇", "퐇" },
  { "퐈", "퐈", "퐈", "퐈", "퐈" },
  { "퐉", "퐉", "퐉", "퐉", "퐉" },
  { "퐊", "퐊", "퐊", "퐊", "퐊" },
  { "퐋", "퐋", "퐋", "퐋", "퐋" },
  { "퐌", "퐌", "퐌", "퐌", "퐌" },
  { "퐍", "퐍", "퐍", "퐍", "퐍" },
  { "퐎", "퐎", "퐎", "퐎", "퐎" },
  { "퐏", "퐏", "퐏", "퐏", "퐏" },
  { "퐐", "퐐", "퐐", "퐐", "퐐" },
  { "퐑", "퐑", "퐑", "퐑", "퐑" },
  { "퐒", "퐒", "퐒", "퐒", "퐒" },
  { "퐓", "퐓", "퐓", "퐓", "퐓" },
  { "퐔", "퐔", "퐔", "퐔", "퐔" },
  { "퐕", "퐕", "퐕", "퐕", "퐕" },
  { "퐖", "퐖", "퐖", "퐖", "퐖" },
  { "퐗", "퐗", "퐗", "퐗", "퐗" },
  { "퐘", "퐘", "퐘", "퐘", "퐘" },
  { "퐙", "퐙", "퐙", "퐙", "퐙" },
  { "퐚", "퐚", "퐚", "퐚", "퐚" },
  { "퐛", "퐛", "퐛", "퐛", "퐛" },
  { "퐜", "퐜", "퐜", "퐜", "퐜" },
  { "퐝", "퐝", "퐝", "퐝", "퐝" },
  { "퐞", "퐞", "퐞", "퐞", "퐞" },
  { "퐟", "퐟", "퐟", "퐟", "퐟" },
  { "퐠", "퐠", "퐠", "퐠", "퐠" },
  { "퐡", "퐡", "퐡", "퐡", "퐡" },
  { "퐢", "퐢", "퐢", "퐢", "퐢" },
  { "퐣", "퐣", "퐣", "퐣", "퐣" },
  { "퐤", "퐤", "퐤", "퐤", "퐤" },
  { "퐥", "퐥", "퐥", "퐥", "퐥" },
  { "퐦", "퐦", "퐦", "퐦", "퐦" },
  { "퐧", "퐧", "퐧", "퐧", "퐧" },
  { "퐨", "퐨", "퐨", "퐨", "퐨" },
  { "퐩", "퐩", "퐩", "퐩", "퐩" },
  { "퐪", "퐪", "퐪", "퐪", "퐪" },
  { "퐫", "퐫", "퐫", "퐫", "퐫" },
  { "퐬", "퐬", "퐬", "퐬", "퐬" },
  { "퐭", "퐭", "퐭", "퐭", "퐭" },
  { "퐮", "퐮", "퐮", "퐮", "퐮" },
  { "퐯", "퐯", "퐯", "퐯", "퐯" },
  { "퐰", "퐰", "퐰", "퐰", "퐰" },
  { "퐱", "퐱", "퐱", "퐱", "퐱" },
  { "퐲", "퐲", "퐲", "퐲", "퐲" },
  { "퐳", "퐳", "퐳", "퐳", "퐳" },
  { "퐴", "퐴", "퐴", "퐴", "퐴" },
  { "퐵", "퐵", "퐵", "퐵", "퐵" },
  { "퐶", "퐶", "퐶", "퐶", "퐶" },
  { "퐷", "퐷", "퐷", "퐷", "퐷" },
  { "퐸", "퐸", "퐸", "퐸", "퐸" },
  { "퐹", "퐹", "퐹", "퐹", "퐹" },
  { "퐺", "퐺", "퐺", "퐺", "퐺" },
  { "퐻", "퐻", "퐻", "퐻", "퐻" },
  { "퐼", "퐼", "퐼", "퐼", "퐼" },
  { "퐽", "퐽", "퐽", "퐽", "퐽" },
  { "퐾", "퐾", "퐾", "퐾", "퐾" },
  { "퐿", "퐿", "퐿", "퐿", "퐿" },
  { "푀", "푀", "푀", "푀", "푀" },
  { "푁", "푁", "푁", "푁", "푁" },
  { "푂", "푂", "푂", "푂", "푂" },
  { "푃", "푃", "푃", "푃", "푃" },
  { "푄", "푄", "푄", "푄", "푄" },
  { "푅", "푅", "푅", "푅", "푅" },
  { "푆", "푆", "푆", "푆", "푆" },
  { "푇", "푇", "푇", "푇", "푇" },
  { "푈", "푈", "푈", "푈", "푈" },
  { "푉", "푉", "푉", "푉", "푉" },
  { "푊", "푊", "푊", "푊", "푊" },
  { "푋", "푋", "푋", "푋", "푋" },
  { "푌", "푌", "푌", "푌", "푌" },
  { "푍", "푍", "푍", "푍", "푍" },
  { "푎", "푎", "푎", "푎", "푎" },
  { "푏", "푏", "푏", "푏", "푏" },
  { "푐", "푐", "푐", "푐", "푐" },
  { "푑", "푑", "푑", "푑", "푑" },
  { "푒", "푒", "푒", "푒", "푒" },
  { "푓", "푓", "푓", "푓", "푓" },
  { "푔", "푔", "푔", "푔", "푔" },
  { "푕", "푕", "푕", "푕", "푕" },
  { "푖", "푖", "푖", "푖", "푖" },
  { "푗", "푗", "푗", "푗", "푗" },
  { "푘", "푘", "푘", "푘", "푘" },
  { "푙", "푙", "푙", "푙", "푙" },
  { "푚", "푚", "푚", "푚", "푚" },
  { "푛", "푛", "푛", "푛", "푛" },
  { "표", "표", "표", "표", "표" },
  { "푝", "푝", "푝", "푝", "푝" },
  { "푞", "푞", "푞", "푞", "푞" },
  { "푟", "푟", "푟", "푟", "푟" },
  { "푠", "푠", "푠", "푠", "푠" },
  { "푡", "푡", "푡", "푡", "푡" },
  { "푢", "푢", "푢", "푢", "푢" },
  { "푣", "푣", "푣", "푣", "푣" },
  { "푤", "푤", "푤", "푤", "푤" },
  { "푥", "푥", "푥", "푥", "푥" },
  { "푦", "푦", "푦", "푦", "푦" },
  { "푧", "푧", "푧", "푧", "푧" },
  { "푨", "푨", "푨", "푨", "푨" },
  { "푩", "푩", "푩", "푩", "푩" },
  { "푪", "푪", "푪", "푪", "푪" },
  { "푫", "푫", "푫", "푫", "푫" },
  { "푬", "푬", "푬", "푬", "푬" },
  { "푭", "푭", "푭", "푭", "푭" },
  { "푮", "푮", "푮", "푮", "푮" },
  { "푯", "푯", "푯", "푯", "푯" },
  { "푰", "푰", "푰", "푰", "푰" },
  { "푱", "푱", "푱", "푱", "푱" },
  { "푲", "푲", "푲", "푲", "푲" },
  { "푳", "푳", "푳", "푳", "푳" },
  { "푴", "푴", "푴", "푴", "푴" },
  { "푵", "푵", "푵", "푵", "푵" },
  { "푶", "푶", "푶", "푶", "푶" },
  { "푷", "푷", "푷", "푷", "푷" },
  { "푸", "푸", "푸", "푸", "푸" },
  { "푹", "푹", "푹", "푹", "푹" },
  { "푺", "푺", "푺", "푺", "푺" },
  { "푻", "푻", "푻", "푻", "푻" },
  { "푼", "푼", "푼", "푼", "푼" },
  { "푽", "푽", "푽", "푽", "푽" },
  { "푾", "푾", "푾", "푾", "푾" },
  { "푿", "푿", "푿", "푿", "푿" },
  { "풀", "풀", "풀", "풀", "풀" },
  { "풁", "풁", "풁", "풁", "풁" },
  { "풂", "풂", "풂", "풂", "풂" },
  { "풃", "풃", "풃", "풃", "풃" },
  { "풄", "풄", "풄", "풄", "풄" },
  { "풅", "풅", "풅", "풅", "풅" },
  { "풆", "풆", "풆", "풆", "풆" },
  { "풇", "풇", "풇", "풇", "풇" },
  { "품", "품", "품", "품", "품" },
  { "풉", "풉", "풉", "풉", "풉" },
  { "풊", "풊", "풊", "풊", "풊" },
  { "풋", "풋", "풋", "풋", "풋" },
  { "풌", "풌", "풌", "풌", "풌" },
  { "풍", "풍", "풍", "풍", "풍" },
  { "풎", "풎", "풎", "풎", "풎" },
  { "풏", "풏", "풏", "풏", "풏" },
  { "풐", "풐", "풐", "풐", "풐" },
  { "풑", "풑", "풑", "풑", "풑" },
  { "풒", "풒", "풒", "풒", "풒" },
  { "풓", "풓", "풓", "풓", "풓" },
  { "풔", "풔", "풔", "풔", "풔" },
  { "풕", "풕", "풕", "풕", "풕" },
  { "풖", "풖", "풖", "풖", "풖" },
  { "풗", "풗", "풗", "풗", "풗" },
  { "풘", "풘", "풘", "풘", "풘" },
  { "풙", "풙", "풙", "풙", "풙" },
  { "풚", "풚", "풚", "풚", "풚" },
  { "풛", "풛", "풛", "풛", "풛" },
  { "풜", "풜", "풜", "풜", "풜" },
  { "풝", "풝", "풝", "풝", "풝" },
  { "풞", "풞", "풞", "풞", "풞" },
  { "풟", "풟", "풟", "풟", "풟" },
  { "풠", "풠", "풠", "풠", "풠" },
  { "풡", "풡", "풡", "풡", "풡" },
  { "풢", "풢", "풢", "풢", "풢" },
  { "풣", "풣", "풣", "풣", "풣" },
  { "풤", "풤", "풤", "풤", "풤" },
  { "풥", "풥", "풥", "풥", "풥" },
  { "풦", "풦", "풦", "풦", "풦" },
  { "풧", "풧", "풧", "풧", "풧" },
  { "풨", "풨", "풨", "풨", "풨" },
  { "풩", "풩", "풩", "풩", "풩" },
  { "풪", "풪", "풪", "풪", "풪" },
  { "풫", "풫", "풫", "풫", "풫" },
  { "풬", "풬", "풬", "풬", "풬" },
  { "풭", "풭", "풭", "풭", "풭" },
  { "풮", "풮", "풮", "풮", "풮" },
  { "풯", "풯", "풯", "풯", "풯" },
  { "풰", "풰", "풰", "풰", "풰" },
  { "풱", "풱", "풱", "풱", "풱" },
  { "풲", "풲", "풲", "풲", "풲" },
  { "풳", "풳", "풳", "풳", "풳" },
  { "풴", "풴", "풴", "풴", "풴" },
  { "풵", "풵", "풵", "풵", "풵" },
  { "풶", "풶", "풶", "풶", "풶" },
  { "풷", "풷", "풷", "풷", "풷" },
  { "풸", "풸", "풸", "풸", "풸" },
  { "풹", "풹", "풹", "풹", "풹" },
  { "풺", "풺", "풺", "풺", "풺" },
  { "풻", "풻", "풻", "풻", "풻" },
  { "풼", "풼", "풼", "풼", "풼" },
  { "풽", "풽", "풽", "풽", "풽" },
  { "풾", "풾", "풾", "풾", "풾" },
  { "풿", "풿", "풿", "풿", "풿" },
  { "퓀", "퓀", "퓀", "퓀", "퓀" },
  { "퓁", "퓁", "퓁", "퓁", "퓁" },
  { "퓂", "퓂", "퓂", "퓂", "퓂" },
  { "퓃", "퓃", "퓃", "퓃", "퓃" },
  { "퓄", "퓄", "퓄", "퓄", "퓄" },
  { "퓅", "퓅", "퓅", "퓅", "퓅" },
  { "퓆", "퓆", "퓆", "퓆", "퓆" },
  { "퓇", "퓇", "퓇", "퓇", "퓇" },
  { "퓈", "퓈", "퓈", "퓈", "퓈" },
  { "퓉", "퓉", "퓉", "퓉", "퓉" },
  { "퓊", "퓊", "퓊", "퓊", "퓊" },
  { "퓋", "퓋", "퓋", "퓋", "퓋" },
  { "퓌", "퓌", "퓌", "퓌", "퓌" },
  { "퓍", "퓍", "퓍", "퓍", "퓍" },
  { "퓎", "퓎", "퓎", "퓎", "퓎" },
  { "퓏", "퓏", "퓏", "퓏", "퓏" },
  { "퓐", "퓐", "퓐", "퓐", "퓐" },
  { "퓑", "퓑", "퓑", "퓑", "퓑" },
  { "퓒", "퓒", "퓒", "퓒", "퓒" },
  { "퓓", "퓓", "퓓", "퓓", "퓓" },
  { "퓔", "퓔", "퓔", "퓔", "퓔" },
  { "퓕", "퓕", "퓕", "퓕", "퓕" },
  { "퓖", "퓖", "퓖", "퓖", "퓖" },
  { "퓗", "퓗", "퓗", "퓗", "퓗" },
  { "퓘", "퓘", "퓘", "퓘", "퓘" },
  { "퓙", "퓙", "퓙", "퓙", "퓙" },
  { "퓚", "퓚", "퓚", "퓚", "퓚" },
  { "퓛", "퓛", "퓛", "퓛", "퓛" },
  { "퓜", "퓜", "퓜", "퓜", "퓜" },
  { "퓝", "퓝", "퓝", "퓝", "퓝" },
  { "퓞", "퓞", "퓞", "퓞", "퓞" },
  { "퓟", "퓟", "퓟", "퓟", "퓟" },
  { "퓠", "퓠", "퓠", "퓠", "퓠" },
  { "퓡", "퓡", "퓡", "퓡", "퓡" },
  { "퓢", "퓢", "퓢", "퓢", "퓢" },
  { "퓣", "퓣", "퓣", "퓣", "퓣" },
  { "퓤", "퓤", "퓤", "퓤", "퓤" },
  { "퓥", "퓥", "퓥", "퓥", "퓥" },
  { "퓦", "퓦", "퓦", "퓦", "퓦" },
  { "퓧", "퓧", "퓧", "퓧", "퓧" },
  { "퓨", "퓨", "퓨", "퓨", "퓨" },
  { "퓩", "퓩", "퓩", "퓩", "퓩" },
  { "퓪", "퓪", "퓪", "퓪", "퓪" },
  { "퓫", "퓫", "퓫", "퓫", "퓫" },
  { "퓬", "퓬", "퓬", "퓬", "퓬" },
  { "퓭", "퓭", "퓭", "퓭", "퓭" },
  { "퓮", "퓮", "퓮", "퓮", "퓮" },
  { "퓯", "퓯", "퓯", "퓯", "퓯" },
  { "퓰", "퓰", "퓰", "퓰", "퓰" },
  { "퓱", "퓱", "퓱", "퓱", "퓱" },
  { "퓲", "퓲", "퓲", "퓲", "퓲" },
  { "퓳", "퓳", "퓳", "퓳", "퓳" },
  { "퓴", "퓴", "퓴", "퓴", "퓴" },
  { "퓵", "퓵", "퓵", "퓵", "퓵" },
  { "퓶", "퓶", "퓶", "퓶", "퓶" },
  { "퓷", "퓷", "퓷", "퓷", "퓷" },
  { "퓸", "퓸", "퓸", "퓸", "퓸" },
  { "퓹", "퓹", "퓹", "퓹", "퓹" },
  { "퓺", "퓺", "퓺", "퓺", "퓺" },
  { "퓻", "퓻", "퓻", "퓻", "퓻" },
  { "퓼", "퓼", "퓼", "퓼", "퓼" },
  { "퓽", "퓽", "퓽", "퓽", "퓽" },
  { "퓾", "퓾", "퓾", "퓾", "퓾" },
  { "퓿", "퓿", "퓿", "퓿", "퓿" },
  { "픀", "픀", "픀", "픀", "픀" },
  { "픁", "픁", "픁", "픁", "픁" },
  { "픂", "픂", "픂", "픂", "픂" },
  { "픃", "픃", "픃", "픃", "픃" },
  { "프", "프", "프", "프", "프" },
  { "픅", "픅", "픅", "픅", "픅" },
  { "픆", "픆", "픆", "픆", "픆" },
  { "픇", "픇", "픇", "픇", "픇" },
  { "픈", "픈", "픈", "픈", "픈" },
  { "픉", "픉", "픉", "픉", "픉" },
  { "픊", "픊", "픊", "픊", "픊" },
  { "픋", "픋", "픋", "픋", "픋" },
  { "플", "플", "플", "플", "플" },
  { "픍", "픍", "픍", "픍", "픍" },
  { "픎", "픎", "픎", "픎", "픎" },
  { "픏", "픏", "픏", "픏", "픏" },
  { "픐", "픐", "픐", "픐", "픐" },
  { "픑", "픑", "픑", "픑", "픑" },
  { "픒", "픒", "픒", "픒", "픒" },
  { "픓", "픓", "픓", "픓", "픓" },
  { "픔", "픔", "픔", "픔", "픔" },
  { "픕", "픕", "픕", "픕", "픕" },
  { "픖", "픖", "픖", "픖", "픖" },
  { "픗", "픗", "픗", "픗", "픗" },
  { "픘", "픘", "픘", "픘", "픘" },
  { "픙", "픙", "픙", "픙", "픙" },
  { "픚", "픚", "픚", "픚", "픚" },
  { "픛", "픛", "픛", "픛", "픛" },
  { "픜", "픜", "픜", "픜", "픜" },
  { "픝", "픝", "픝", "픝", "픝" },
  { "픞", "픞", "픞", "픞", "픞" },
  { "픟", "픟", "픟", "픟", "픟" },
  { "픠", "픠", "픠", "픠", "픠" },
  { "픡", "픡", "픡", "픡", "픡" },
  { "픢", "픢", "픢", "픢", "픢" },
  { "픣", "픣", "픣", "픣", "픣" },
  { "픤", "픤", "픤", "픤", "픤" },
  { "픥", "픥", "픥", "픥", "픥" },
  { "픦", "픦", "픦", "픦", "픦" },
  { "픧", "픧", "픧", "픧", "픧" },
  { "픨", "픨", "픨", "픨", "픨" },
  { "픩", "픩", "픩", "픩", "픩" },
  { "픪", "픪", "픪", "픪", "픪" },
  { "픫", "픫", "픫", "픫", "픫" },
  { "픬", "픬", "픬", "픬", "픬" },
  { "픭", "픭", "픭", "픭", "픭" },
  { "픮", "픮", "픮", "픮", "픮" },
  { "픯", "픯", "픯", "픯", "픯" },
  { "픰", "픰", "픰", "픰", "픰" },
  { "픱", "픱", "픱", "픱", "픱" },
  { "픲", "픲", "픲", "픲", "픲" },
  { "픳", "픳", "픳", "픳", "픳" },
  { "픴", "픴", "픴", "픴", "픴" },
  { "픵", "픵", "픵", "픵", "픵" },
  { "픶", "픶", "픶", "픶", "픶" },
  { "픷", "픷", "픷", "픷", "픷" },
  { "픸", "픸", "픸", "픸", "픸" },
  { "픹", "픹", "픹", "픹", "픹" },
  { "픺", "픺", "픺", "픺", "픺" },
  { "픻", "픻", "픻", "픻", "픻" },
  { "피", "피", "피", "피", "피" },
  { "픽", "픽", "픽", "픽", "픽" },
  { "픾", "픾", "픾", "픾", "픾" },
  { "픿", "픿", "픿", "픿", "픿" },
  { "핀", "핀", "핀", "핀", "핀" },
  { "핁", "핁", "핁", "핁", "핁" },
  { "핂", "핂", "핂", "핂", "핂" },
  { "핃", "핃", "핃", "핃", "핃" },
  { "필", "필", "필", "필", "필" },
  { "핅", "핅", "핅", "핅", "핅" },
  { "핆", "핆", "핆", "핆", "핆" },
  { "핇", "핇", "핇", "핇", "핇" },
  { "핈", "핈", "핈", "핈", "핈" },
  { "핉", "핉", "핉", "핉", "핉" },
  { "핊", "핊", "핊", "핊", "핊" },
  { "핋", "핋", "핋", "핋", "핋" },
  { "핌", "핌", "핌", "핌", "핌" },
  { "핍", "핍", "핍", "핍", "핍" },
  { "핎", "핎", "핎", "핎", "핎" },
  { "핏", "핏", "핏", "핏", "핏" },
  { "핐", "핐", "핐", "핐", "핐" },
  { "핑", "핑", "핑", "핑", "핑" },
  { "핒", "핒", "핒", "핒", "핒" },
  { "핓", "핓", "핓", "핓", "핓" },
  { "핔", "핔", "핔", "핔", "핔" },
  { "핕", "핕", "핕", "핕", "핕" },
  { "핖", "핖", "핖", "핖", "핖" },
  { "핗", "핗", "핗", "핗", "핗" },
  { "하", "하", "하", "하", "하" },
  { "학", "학", "학", "학", "학" },
  { "핚", "핚", "핚", "핚", "핚" },
  { "핛", "핛", "핛", "핛", "핛" },
  { "한", "한", "한", "한", "한" },
  { "핝", "핝", "핝", "핝", "핝" },
  { "핞", "핞", "핞", "핞", "핞" },
  { "핟", "핟", "핟", "핟", "핟" },
  { "할", "할", "할", "할", "할" },
  { "핡", "핡", "핡", "핡", "핡" },
  { "핢", "핢", "핢", "핢", "핢" },
  { "핣", "핣", "핣", "핣", "핣" },
  { "핤", "핤", "핤", "핤", "핤" },
  { "핥", "핥", "핥", "핥", "핥" },
  { "핦", "핦", "핦", "핦", "핦" },
  { "핧", "핧", "핧", "핧", "핧" },
  { "함", "함", "함", "함", "함" },
  { "합", "합", "합", "합", "합" },
  { "핪", "핪", "핪", "핪", "핪" },
  { "핫", "핫", "핫", "핫", "핫" },
  { "핬", "핬", "핬", "핬", "핬" },
  { "항", "항", "항", "항", "항" },
  { "핮", "핮", "핮", "핮", "핮" },
  { "핯", "핯", "핯", "핯", "핯" },
  { "핰", "핰", "핰", "핰", "핰" },
  { "핱", "핱", "핱", "핱", "핱" },
  { "핲", "핲", "핲", "핲", "핲" },
  { "핳", "핳", "핳", "핳", "핳" },
  { "해", "해", "해", "해", "해" },
  { "핵", "핵", "핵", "핵", "핵" },
  { "핶", "핶", "핶", "핶", "핶" },
  { "핷", "핷", "핷", "핷", "핷" },
  { "핸", "핸", "핸", "핸", "핸" },
  { "핹", "핹", "핹", "핹", "핹" },
  { "핺", "핺", "핺", "핺", "핺" },
  { "핻", "핻", "핻", "핻", "핻" },
  { "핼", "핼", "핼", "핼", "핼" },
  { "핽", "핽", "핽", "핽", "핽" },
  { "핾", "핾", "핾", "핾", "핾" },
  { "핿", "핿", "핿", "핿", "핿" },
  { "햀", "햀", "햀", "햀", "햀" },
  { "햁", "햁", "햁", "햁", "햁" },
  { "햂", "햂", "햂", "햂", "햂" },
  { "햃", "햃", "햃", "햃", "햃" },
  { "햄", "햄", "햄", "햄", "햄" },
  { "햅", "햅", "햅", "햅", "햅" },
  { "햆", "햆", "햆", "햆", "햆" },
  { "햇", "햇", "햇", "햇", "햇" },
  { "했", "했", "했", "했", "했" },
  { "행", "행", "행", "행", "행" },
  { "햊", "햊", "햊", "햊", "햊" },
  { "햋", "햋", "햋", "햋", "햋" },
  { "햌", "햌", "햌", "햌", "햌" },
  { "햍", "햍", "햍", "햍", "햍" },
  { "햎", "햎", "햎", "햎", "햎" },
  { "햏", "햏", "햏", "햏", "햏" },
  { "햐", "햐", "햐", "햐", "햐" },
  { "햑", "햑", "햑", "햑", "햑" },
  { "햒", "햒", "햒", "햒", "햒" },
  { "햓", "햓", "햓", "햓", "햓" },
  { "햔", "햔", "햔", "햔", "햔" },
  { "햕", "햕", "햕", "햕", "햕" },
  { "햖", "햖", "햖", "햖", "햖" },
  { "햗", "햗", "햗", "햗", "햗" },
  { "햘", "햘", "햘", "햘", "햘" },
  { "햙", "햙", "햙", "햙", "햙" },
  { "햚", "햚", "햚", "햚", "햚" },
  { "햛", "햛", "햛", "햛", "햛" },
  { "햜", "햜", "햜", "햜", "햜" },
  { "햝", "햝", "햝", "햝", "햝" },
  { "햞", "햞", "햞", "햞", "햞" },
  { "햟", "햟", "햟", "햟", "햟" },
  { "햠", "햠", "햠", "햠", "햠" },
  { "햡", "햡", "햡", "햡", "햡" },
  { "햢", "햢", "햢", "햢", "햢" },
  { "햣", "햣", "햣", "햣", "햣" },
  { "햤", "햤", "햤", "햤", "햤" },
  { "향", "향", "향", "향", "향" },
  { "햦", "햦", "햦", "햦", "햦" },
  { "햧", "햧", "햧", "햧", "햧" },
  { "햨", "햨", "햨", "햨", "햨" },
  { "햩", "햩", "햩", "햩", "햩" },
  { "햪", "햪", "햪", "햪", "햪" },
  { "햫", "햫", "햫", "햫", "햫" },
  { "햬", "햬", "햬", "햬", "햬" },
  { "햭", "햭", "햭", "햭", "햭" },
  { "햮", "햮", "햮", "햮", "햮" },
  { "햯", "햯", "햯", "햯", "햯" },
  { "햰", "햰", "햰", "햰", "햰" },
  { "햱", "햱", "햱", "햱", "햱" },
  { "햲", "햲", "햲", "햲", "햲" },
  { "햳", "햳", "햳", "햳", "햳" },
  { "햴", "햴", "햴", "햴", "햴" },
  { "햵", "햵", "햵", "햵", "햵" },
  { "햶", "햶", "햶", "햶", "햶" },
  { "햷", "햷", "햷", "햷", "햷" },
  { "햸", "햸", "햸", "햸", "햸" },
  { "햹", "햹", "햹", "햹", "햹" },
  { "햺", "햺", "햺", "햺", "햺" },
  { "햻", "햻", "햻", "햻", "햻" },
  { "햼", "햼", "햼", "햼", "햼" },
  { "햽", "햽", "햽", "햽", "햽" },
  { "햾", "햾", "햾", "햾", "햾" },
  { "햿", "햿", "햿", "햿", "햿" },
  { "헀", "헀", "헀", "헀", "헀" },
  { "헁", "헁", "헁", "헁", "헁" },
  { "헂", "헂", "헂", "헂", "헂" },
  { "헃", "헃", "헃", "헃", "헃" },
  { "헄", "헄", "헄", "헄", "헄" },
  { "헅", "헅", "헅", "헅", "헅" },
  { "헆", "헆", "헆", "헆", "헆" },
  { "헇", "헇", "헇", "헇", "헇" },
  { "허", "허", "허", "허", "허" },
  { "헉", "헉", "헉", "헉", "헉" },
  { "헊", "헊", "헊", "헊", "헊" },
  { "헋", "헋", "헋", "헋", "헋" },
  { "헌", "헌", "헌", "헌", "헌" },
  { "헍", "헍", "헍", "헍", "헍" },
  { "헎", "헎", "헎", "헎", "헎" },
  { "헏", "헏", "헏", "헏", "헏" },
  { "헐", "헐", "헐", "헐", "헐" },
  { "헑", "헑", "헑", "헑", "헑" },
  { "헒", "헒", "헒", "헒", "헒" },
  { "헓", "헓", "헓", "헓", "헓" },
  { "헔", "헔", "헔", "헔", "헔" },
  { "헕", "헕", "헕", "헕", "헕" },
  { "헖", "헖", "헖", "헖", "헖" },
  { "헗", "헗", "헗", "헗", "헗" },
  { "험", "험", "험", "험", "험" },
  { "헙", "헙", "헙", "헙", "헙" },
  { "헚", "헚", "헚", "헚", "헚" },
  { "헛", "헛", "헛", "헛", "헛" },
  { "헜", "헜", "헜", "헜", "헜" },
  { "헝", "헝", "헝", "헝", "헝" },
  { "헞", "헞", "헞", "헞", "헞" },
  { "헟", "헟", "헟", "헟", "헟" },
  { "헠", "헠", "헠", "헠", "헠" },
  { "헡", "헡", "헡", "헡", "헡" },
  { "헢", "헢", "헢", "헢", "헢" },
  { "헣", "헣", "헣", "헣", "헣" },
  { "헤", "헤", "헤", "헤", "헤" },
  { "헥", "헥", "헥", "헥", "헥" },
  { "헦", "헦", "헦", "헦", "헦" },
  { "헧", "헧", "헧", "헧", "헧" },
  { "헨", "헨", "헨", "헨", "헨" },
  { "헩", "헩", "헩", "헩", "헩" },
  { "헪", "헪", "헪", "헪", "헪" },
  { "헫", "헫", "헫", "헫", "헫" },
  { "헬", "헬", "헬", "헬", "헬" },
  { "헭", "헭", "헭", "헭", "헭" },
  { "헮", "헮", "헮", "헮", "헮" },
  { "헯", "헯", "헯", "헯", "헯" },
  { "헰", "헰", "헰", "헰", "헰" },
  { "헱", "헱", "헱", "헱", "헱" },
  { "헲", "헲", "헲", "헲", "헲" },
  { "헳", "헳", "헳", "헳", "헳" },
  { "헴", "헴", "헴", "헴", "헴" },
  { "헵", "헵", "헵", "헵", "헵" },
  { "헶", "헶", "헶", "헶", "헶" },
  { "헷", "헷", "헷", "헷", "헷" },
  { "헸", "헸", "헸", "헸", "헸" },
  { "헹", "헹", "헹", "헹", "헹" },
  { "헺", "헺", "헺", "헺", "헺" },
  { "헻", "헻", "헻", "헻", "헻" },
  { "헼", "헼", "헼", "헼", "헼" },
  { "헽", "헽", "헽", "헽", "헽" },
  { "헾", "헾", "헾", "헾", "헾" },
  { "헿", "헿", "헿", "헿", "헿" },
  { "혀", "혀", "혀", "혀", "혀" },
  { "혁", "혁", "혁", "혁", "혁" },
  { "혂", "혂", "혂", "혂", "혂" },
  { "혃", "혃", "혃", "혃", "혃" },
  { "현", "현", "현", "현", "현" },
  { "혅", "혅", "혅", "혅", "혅" },
  { "혆", "혆", "혆", "혆", "혆" },
  { "혇", "혇", "혇", "혇", "혇" },
  { "혈", "혈", "혈", "혈", "혈" },
  { "혉", "혉", "혉", "혉", "혉" },
  { "혊", "혊", "혊", "혊", "혊" },
  { "혋", "혋", "혋", "혋", "혋" },
  { "혌", "혌", "혌", "혌", "혌" },
  { "혍", "혍", "혍", "혍", "혍" },
  { "혎", "혎", "혎", "혎", "혎" },
  { "혏", "혏", "혏", "혏", "혏" },
  { "혐", "혐", "혐", "혐", "혐" },
  { "협", "협", "협", "협", "협" },
  { "혒", "혒", "혒", "혒", "혒" },
  { "혓", "혓", "혓", "혓", "혓" },
  { "혔", "혔", "혔", "혔", "혔" },
  { "형", "형", "형", "형", "형" },
  { "혖", "혖", "혖", "혖", "혖" },
  { "혗", "혗", "혗", "혗", "혗" },
  { "혘", "혘", "혘", "혘", "혘" },
  { "혙", "혙", "혙", "혙", "혙" },
  { "혚", "혚", "혚", "혚", "혚" },
  { "혛", "혛", "혛", "혛", "혛" },
  { "혜", "혜", "혜", "혜", "혜" },
  { "혝", "혝", "혝", "혝", "혝" },
  { "혞", "혞", "혞", "혞", "혞" },
  { "혟", "혟", "혟", "혟", "혟" },
  { "혠", "혠", "혠", "혠", "혠" },
  { "혡", "혡", "혡", "혡", "혡" },
  { "혢", "혢", "혢", "혢", "혢" },
  { "혣", "혣", "혣", "혣", "혣" },
  { "혤", "혤", "혤", "혤", "혤" },
  { "혥", "혥", "혥", "혥", "혥" },
  { "혦", "혦", "혦", "혦", "혦" },
  { "혧", "혧", "혧", "혧", "혧" },
  { "혨", "혨", "혨", "혨", "혨" },
  { "혩", "혩", "혩", "혩", "혩" },
  { "혪", "혪", "혪", "혪", "혪" },
  { "혫", "혫", "혫", "혫", "혫" },
  { "혬", "혬", "혬", "혬", "혬" },
  { "혭", "혭", "혭", "혭", "혭" },
  { "혮", "혮", "혮", "혮", "혮" },
  { "혯", "혯", "혯", "혯", "혯" },
  { "혰", "혰", "혰", "혰", "혰" },
  { "혱", "혱", "혱", "혱", "혱" },
  { "혲", "혲", "혲", "혲", "혲" },
  { "혳", "혳", "혳", "혳", "혳" },
  { "혴", "혴", "혴", "혴", "혴" },
  { "혵", "혵", "혵", "혵", "혵" },
  { "혶", "혶", "혶", "혶", "혶" },
  { "혷", "혷", "혷", "혷", "혷" },
  { "호", "호", "호", "호", "호" },
  { "혹", "혹", "혹", "혹", "혹" },
  { "혺", "혺", "혺", "혺", "혺" },
  { "혻", "혻", "혻", "혻", "혻" },
  { "혼", "혼", "혼", "혼", "혼" },
  { "혽", "혽", "혽", "혽", "혽" },
  { "혾", "혾", "혾", "혾", "혾" },
  { "혿", "혿", "혿", "혿", "혿" },
  { "홀", "홀", "홀", "홀", "홀" },
  { "홁", "홁", "홁", "홁", "홁" },
  { "홂", "홂", "홂", "홂", "홂" },
  { "홃", "홃", "홃", "홃", "홃" },
  { "홄", "홄", "홄", "홄", "홄" },
  { "홅", "홅", "홅", "홅", "홅" },
  { "홆", "홆", "홆", "홆", "홆" },
  { "홇", "홇", "홇", "홇", "홇" },
  { "홈", "홈", "홈", "홈", "홈" },
  { "홉", "홉", "홉", "홉", "홉" },
  { "홊", "홊", "홊", "홊", "홊" },
  { "홋", "홋", "홋", "홋", "홋" },
  { "홌", "홌", "홌", "홌", "홌" },
  { "홍", "홍", "홍", "홍", "홍" },
  { "홎", "홎", "홎", "홎", "홎" },
  { "홏", "홏", "홏", "홏", "홏" },
  { "홐", "홐", "홐", "홐", "홐" },
  { "홑", "홑", "홑", "홑", "홑" },
  { "홒", "홒", "홒", "홒", "홒" },
  { "홓", "홓", "홓", "홓", "홓" },
  { "화", "화", "화", "화", "화" },
  { "확", "확", "확", "확", "확" },
  { "홖", "홖", "홖", "홖", "홖" },
  { "홗", "홗", "홗", "홗", "홗" },
  { "환", "환", "환", "환", "환" },
  { "홙", "홙", "홙", "홙", "홙" },
  { "홚", "홚", "홚", "홚", "홚" },
  { "홛", "홛", "홛", "홛", "홛" },
  { "활", "활", "활", "활", "활" },
  { "홝", "홝", "홝", "홝", "홝" },
  { "홞", "홞", "홞", "홞", "홞" },
  { "홟", "홟", "홟", "홟", "홟" },
  { "홠", "홠", "홠", "홠", "홠" },
  { "홡", "홡", "홡", "홡", "홡" },
  { "홢", "홢", "홢", "홢", "홢" },
  { "홣", "홣", "홣", "홣", "홣" },
  { "홤", "홤", "홤", "홤", "홤" },
  { "홥", "홥", "홥", "홥", "홥" },
  { "홦", "홦", "홦", "홦", "홦" },
  { "홧", "홧", "홧", "홧", "홧" },
  { "홨", "홨", "홨", "홨", "홨" },
  { "황", "황", "황", "황", "황" },
  { "홪", "홪", "홪", "홪", "홪" },
  { "홫", "홫", "홫", "홫", "홫" },
  { "홬", "홬", "홬", "홬", "홬" },
  { "홭", "홭", "홭", "홭", "홭" },
  { "홮", "홮", "홮", "홮", "홮" },
  { "홯", "홯", "홯", "홯", "홯" },
  { "홰", "홰", "홰", "홰", "홰" },
  { "홱", "홱", "홱", "홱", "홱" },
  { "홲", "홲", "홲", "홲", "홲" },
  { "홳", "홳", "홳", "홳", "홳" },
  { "홴", "홴", "홴", "홴", "홴" },
  { "홵", "홵", "홵", "홵", "홵" },
  { "홶", "홶", "홶", "홶", "홶" },
  { "홷", "홷", "홷", "홷", "홷" },
  { "홸", "홸", "홸", "홸", "홸" },
  { "홹", "홹", "홹", "홹", "홹" },
  { "홺", "홺", "홺", "홺", "홺" },
  { "홻", "홻", "홻", "홻", "홻" },
  { "홼", "홼", "홼", "홼", "홼" },
  { "홽", "홽", "홽", "홽", "홽" },
  { "홾", "홾", "홾", "홾", "홾" },
  { "홿", "홿", "홿", "홿", "홿" },
  { "횀", "횀", "횀", "횀", "횀" },
  { "횁", "횁", "횁", "횁", "횁" },
  { "횂", "횂", "횂", "횂", "횂" },
  { "횃", "횃", "횃", "횃", "횃" },
  { "횄", "횄", "횄", "횄", "횄" },
  { "횅", "횅", "횅", "횅", "횅" },
  { "횆", "횆", "횆", "횆", "횆" },
  { "횇", "횇", "횇", "횇", "횇" },
  { "횈", "횈", "횈", "횈", "횈" },
  { "횉", "횉", "횉", "횉", "횉" },
  { "횊", "횊", "횊", "횊", "횊" },
  { "횋", "횋", "횋", "횋", "횋" },
  { "회", "회", "회", "회", "회" },
  { "획", "획", "획", "획", "획" },
  { "횎", "횎", "횎", "횎", "횎" },
  { "횏", "횏", "횏", "횏", "횏" },
  { "횐", "횐", "횐", "횐", "횐" },
  { "횑", "횑", "횑", "횑", "횑" },
  { "횒", "횒", "횒", "횒", "횒" },
  { "횓", "횓", "횓", "횓", "횓" },
  { "횔", "횔", "횔", "횔", "횔" },
  { "횕", "횕", "횕", "횕", "횕" },
  { "횖", "횖", "횖", "횖", "횖" },
  { "횗", "횗", "횗", "횗", "횗" },
  { "횘", "횘", "횘", "횘", "횘" },
  { "횙", "횙", "횙", "횙", "횙" },
  { "횚", "횚", "횚", "횚", "횚" },
  { "횛", "횛", "횛", "횛", "횛" },
  { "횜", "횜", "횜", "횜", "횜" },
  { "횝", "횝", "횝", "횝", "횝" },
  { "횞", "횞", "횞", "횞", "횞" },
  { "횟", "횟", "횟", "횟", "횟" },
  { "횠", "횠", "횠", "횠", "횠" },
  { "횡", "횡", "횡", "횡", "횡" },
  { "횢", "횢", "횢", "횢", "횢" },
  { "횣", "횣", "횣", "횣", "횣" },
  { "횤", "횤", "횤", "횤", "횤" },
  { "횥", "횥", "횥", "횥", "횥" },
  { "횦", "횦", "횦", "횦", "횦" },
  { "횧", "횧", "횧", "횧", "횧" },
  { "효", "효", "효", "효", "효" },
  { "횩", "횩", "횩", "횩", "횩" },
  { "횪", "횪", "횪", "횪", "횪" },
  { "횫", "횫", "횫", "횫", "횫" },
  { "횬", "횬", "횬", "횬", "횬" },
  { "횭", "횭", "횭", "횭", "횭" },
  { "횮", "횮", "횮", "횮", "횮" },
  { "횯", "횯", "횯", "횯", "횯" },
  { "횰", "횰", "횰", "횰", "횰" },
  { "횱", "횱", "횱", "횱", "횱" },
  { "횲", "횲", "횲", "횲", "횲" },
  { "횳", "횳", "횳", "횳", "횳" },
  { "횴", "횴", "횴", "횴", "횴" },
  { "횵", "횵", "횵", "횵", "횵" },
  { "횶", "횶", "횶", "횶", "횶" },
  { "횷", "횷", "횷", "횷", "횷" },
  { "횸", "횸", "횸", "횸", "횸" },
  { "횹", "횹", "횹", "횹", "횹" },
  { "횺", "횺", "횺", "횺", "횺" },
  { "횻", "횻", "횻", "횻", "횻" },
  { "횼", "횼", "횼", "횼", "횼" },
  { "횽", "횽", "횽", "횽", "횽" },
  { "횾", "횾", "횾", "횾", "횾" },
  { "횿", "횿", "횿", "횿", "횿" },
  { "훀", "훀", "훀", "훀", "훀" },
  { "훁", "훁", "훁", "훁", "훁" },
  { "훂", "훂", "훂", "훂", "훂" },
  { "훃", "훃", "훃", "훃", "훃" },
  { "후", "후", "후", "후", "후" },
  { "훅", "훅", "훅", "훅", "훅" },
  { "훆", "훆", "훆", "훆", "훆" },
  { "훇", "훇", "훇", "훇", "훇" },
  { "훈", "훈", "훈", "훈", "훈" },
  { "훉", "훉", "훉", "훉", "훉" },
  { "훊", "훊", "훊", "훊", "훊" },
  { "훋", "훋", "훋", "훋", "훋" },
  { "훌", "훌", "훌", "훌", "훌" },
  { "훍", "훍", "훍", "훍", "훍" },
  { "훎", "훎", "훎", "훎", "훎" },
  { "훏", "훏", "훏", "훏", "훏" },
  { "훐", "훐", "훐", "훐", "훐" },
  { "훑", "훑", "훑", "훑", "훑" },
  { "훒", "훒", "훒", "훒", "훒" },
  { "훓", "훓", "훓", "훓", "훓" },
  { "훔", "훔", "훔", "훔", "훔" },
  { "훕", "훕", "훕", "훕", "훕" },
  { "훖", "훖", "훖", "훖", "훖" },
  { "훗", "훗", "훗", "훗", "훗" },
  { "훘", "훘", "훘", "훘", "훘" },
  { "훙", "훙", "훙", "훙", "훙" },
  { "훚", "훚", "훚", "훚", "훚" },
  { "훛", "훛", "훛", "훛", "훛" },
  { "훜", "훜", "훜", "훜", "훜" },
  { "훝", "훝", "훝", "훝", "훝" },
  { "훞", "훞", "훞", "훞", "훞" },
  { "훟", "훟", "훟", "훟", "훟" },
  { "훠", "훠", "훠", "훠", "훠" },
  { "훡", "훡", "훡", "훡", "훡" },
  { "훢", "훢", "훢", "훢", "훢" },
  { "훣", "훣", "훣", "훣", "훣" },
  { "훤", "훤", "훤", "훤", "훤" },
  { "훥", "훥", "훥", "훥", "훥" },
  { "훦", "훦", "훦", "훦", "훦" },
  { "훧", "훧", "훧", "훧", "훧" },
  { "훨", "훨", "훨", "훨", "훨" },
  { "훩", "훩", "훩", "훩", "훩" },
  { "훪", "훪", "훪", "훪", "훪" },
  { "훫", "훫", "훫", "훫", "훫" },
  { "훬", "훬", "훬", "훬", "훬" },
  { "훭", "훭", "훭", "훭", "훭" },
  { "훮", "훮", "훮", "훮", "훮" },
  { "훯", "훯", "훯", "훯", "훯" },
  { "훰", "훰", "훰", "훰", "훰" },
  { "훱", "훱", "훱", "훱", "훱" },
  { "훲", "훲", "훲", "훲", "훲" },
  { "훳", "훳", "훳", "훳", "훳" },
  { "훴", "훴", "훴", "훴", "훴" },
  { "훵", "훵", "훵", "훵", "훵" },
  { "훶", "훶", "훶", "훶", "훶" },
  { "훷", "훷", "훷", "훷", "훷" },
  { "훸", "훸", "훸", "훸", "훸" },
  { "훹", "훹", "훹", "훹", "훹" },
  { "훺", "훺", "훺", "훺", "훺" },
  { "훻", "훻", "훻", "훻", "훻" },
  { "훼", "훼", "훼", "훼", "훼" },
  { "훽", "훽", "훽", "훽", "훽" },
  { "훾", "훾", "훾", "훾", "훾" },
  { "훿", "훿", "훿", "훿", "훿" },
  { "휀", "휀", "휀", "휀", "휀" },
  { "휁", "휁", "휁", "휁", "휁" },
  { "휂", "휂", "휂", "휂", "휂" },
  { "휃", "휃", "휃", "휃", "휃" },
  { "휄", "휄", "휄", "휄", "휄" },
  { "휅", "휅", "휅", "휅", "휅" },
  { "휆", "휆", "휆", "휆", "휆" },
  { "휇", "휇", "휇", "휇", "휇" },
  { "휈", "휈", "휈", "휈", "휈" },
  { "휉", "휉", "휉", "휉", "휉" },
  { "휊", "휊", "휊", "휊", "휊" },
  { "휋", "휋", "휋", "휋", "휋" },
  { "휌", "휌", "휌", "휌", "휌" },
  { "휍", "휍", "휍", "휍", "휍" },
  { "휎", "휎", "휎", "휎", "휎" },
  { "휏", "휏", "휏", "휏", "휏" },
  { "휐", "휐", "휐", "휐", "휐" },
  { "휑", "휑", "휑", "휑", "휑" },
  { "휒", "휒", "휒", "휒", "휒" },
  { "휓", "휓", "휓", "휓", "휓" },
  { "휔", "휔", "휔", "휔", "휔" },
  { "휕", "휕", "휕", "휕", "휕" },
  { "휖", "휖", "휖", "휖", "휖" },
  { "휗", "휗", "휗", "휗", "휗" },
  { "휘", "휘", "휘", "휘", "휘" },
  { "휙", "휙", "휙", "휙", "휙" },
  { "휚", "휚", "휚", "휚", "휚" },
  { "휛", "휛", "휛", "휛", "휛" },
  { "휜", "휜", "휜", "휜", "휜" },
  { "휝", "휝", "휝", "휝", "휝" },
  { "휞", "휞", "휞", "휞", "휞" },
  { "휟", "휟", "휟", "휟", "휟" },
  { "휠", "휠", "휠", "휠", "휠" },
  { "휡", "휡", "휡", "휡", "휡" },
  { "휢", "휢", "휢", "휢", "휢" },
  { "휣", "휣", "휣", "휣", "휣" },
  { "휤", "휤", "휤", "휤", "휤" },
  { "휥", "휥", "휥", "휥", "휥" },
  { "휦", "휦", "휦", "휦", "휦" },
  { "휧", "휧", "휧", "휧", "휧" },
  { "휨", "휨", "휨", "휨", "휨" },
  { "휩", "휩", "휩", "휩", "휩" },
  { "휪", "휪", "휪", "휪", "휪" },
  { "휫", "휫", "휫", "휫", "휫" },
  { "휬", "휬", "휬", "휬", "휬" },
  { "휭", "휭", "휭", "휭", "휭" },
  { "휮", "휮", "휮", "휮", "휮" },
  { "휯", "휯", "휯", "휯", "휯" },
  { "휰", "휰", "휰", "휰", "휰" },
  { "휱", "휱", "휱", "휱", "휱" },
  { "휲", "휲", "휲", "휲", "휲" },
  { "휳", "휳", "휳", "휳", "휳" },
  { "휴", "휴", "휴", "휴", "휴" },
  { "휵", "휵", "휵", "휵", "휵" },
  { "휶", "휶", "휶", "휶", "휶" },
  { "휷", "휷", "휷", "휷", "휷" },
  { "휸", "휸", "휸", "휸", "휸" },
  { "휹", "휹", "휹", "휹", "휹" },
  { "휺", "휺", "휺", "휺", "휺" },
  { "휻", "휻", "휻", "휻", "휻" },
  { "휼", "휼", "휼", "휼", "휼" },
  { "휽", "휽", "휽", "휽", "휽" },
  { "휾", "휾", "휾", "휾", "휾" },
  { "휿", "휿", "휿", "휿", "휿" },
  { "흀", "흀", "흀", "흀", "흀" },
  { "흁", "흁", "흁", "흁", "흁" },
  { "흂", "흂", "흂", "흂", "흂" },
  { "흃", "흃", "흃", "흃", "흃" },
  { "흄", "흄", "흄", "흄", "흄" },
  { "흅", "흅", "흅", "흅", "흅" },
  { "흆", "흆", "흆", "흆", "흆" },
  { "흇", "흇", "흇", "흇", "흇" },
  { "흈", "흈", "흈", "흈", "흈" },
  { "흉", "흉", "흉", "흉", "흉" },
  { "흊", "흊", "흊", "흊", "흊" },
  { "흋", "흋", "흋", "흋", "흋" },
  { "흌", "흌", "흌", "흌", "흌" },
  { "흍", "흍", "흍", "흍", "흍" },
  { "흎", "흎", "흎", "흎", "흎" },
  { "흏", "흏", "흏", "흏", "흏" },
  { "흐", "흐", "흐", "흐", "흐" },
  { "흑", "흑", "흑", "흑", "흑" },
  { "흒", "흒", "흒", "흒", "흒" },
  { "흓", "흓", "흓", "흓", "흓" },
  { "흔", "흔", "흔", "흔", "흔" },
  { "흕", "흕", "흕", "흕", "흕" },
  { "흖", "흖", "흖", "흖", "흖" },
  { "흗", "흗", "흗", "흗", "흗" },
  { "흘", "흘", "흘", "흘", "흘" },
  { "흙", "흙", "흙", "흙", "흙" },
  { "흚", "흚", "흚", "흚", "흚" },
  { "흛", "흛", "흛", "흛", "흛" },
  { "흜", "흜", "흜", "흜", "흜" },
  { "흝", "흝", "흝", "흝", "흝" },
  { "흞", "흞", "흞", "흞", "흞" },
  { "흟", "흟", "흟", "흟", "흟" },
  { "흠", "흠", "흠", "흠", "흠" },
  { "흡", "흡", "흡", "흡", "흡" },
  { "흢", "흢", "흢", "흢", "흢" },
  { "흣", "흣", "흣", "흣", "흣" },
  { "흤", "흤", "흤", "흤", "흤" },
  { "흥", "흥", "흥", "흥", "흥" },
  { "흦", "흦", "흦", "흦", "흦" },
  { "흧", "흧", "흧", "흧", "흧" },
  { "흨", "흨", "흨", "흨", "흨" },
  { "흩", "흩", "흩", "흩", "흩" },
  { "흪", "흪", "흪", "흪", "흪" },
  { "흫", "흫", "흫", "흫", "흫" },
  { "희", "희", "희", "희", "희" },
  { "흭", "흭", "흭", "흭", "흭" },
  { "흮", "흮", "흮", "흮", "흮" },
  { "흯", "흯", "흯", "흯", "흯" },
  { "흰", "흰", "흰", "흰", "흰" },
  { "흱", "흱", "흱", "흱", "흱" },
  { "흲", "흲", "흲", "흲", "흲" },
  { "흳", "흳", "흳", "흳", "흳" },
  { "흴", "흴", "흴", "흴", "흴" },
  { "흵", "흵", "흵", "흵", "흵" },
  { "흶", "흶", "흶", "흶", "흶" },
  { "흷", "흷", "흷", "흷", "흷" },
  { "흸", "흸", "흸", "흸", "흸" },
  { "흹", "흹", "흹", "흹", "흹" },
  { "흺", "흺", "흺", "흺", "흺" },
  { "흻", "흻", "흻", "흻", "흻" },
  { "흼", "흼", "흼", "흼", "흼" },
  { "흽", "흽", "흽", "흽", "흽" },
  { "흾", "흾", "흾", "흾", "흾" },
  { "흿", "흿", "흿", "흿", "흿" },
  { "힀", "힀", "힀", "힀", "힀" },
  { "힁", "힁", "힁", "힁", "힁" },
  { "힂", "힂", "힂", "힂", "힂" },
  { "힃", "힃", "힃", "힃", "힃" },
  { "힄", "힄", "힄", "힄", "힄" },
  { "힅", "힅", "힅", "힅", "힅" },
  { "힆", "힆", "힆", "힆", "힆" },
  { "힇", "힇", "힇", "힇", "힇" },
  { "히", "히", "히", "히", "히" },
  { "힉", "힉", "힉", "힉", "힉" },
  { "힊", "힊", "힊", "힊", "힊" },
  { "힋", "힋", "힋", "힋", "힋" },
  { "힌", "힌", "힌", "힌", "힌" },
  { "힍", "힍", "힍", "힍", "힍" },
  { "힎", "힎", "힎", "힎", "힎" },
  { "힏", "힏", "힏", "힏", "힏" },
  { "힐", "힐", "힐", "힐", "힐" },
  { "힑", "힑", "힑", "힑", "힑" },
  { "힒", "힒", "힒", "힒", "힒" },
  { "힓", "힓", "힓", "힓", "힓" },
  { "힔", "힔", "힔", "힔", "힔" },
  { "힕", "힕", "힕", "힕", "힕" },
  { "힖", "힖", "힖", "힖", "힖" },
  { "힗", "힗", "힗", "힗", "힗" },
  { "힘", "힘", "힘", "힘", "힘" },
  { "힙", "힙", "힙", "힙", "힙" },
  { "힚", "힚", "힚", "힚", "힚" },
  { "힛", "힛", "힛", "힛", "힛" },
  { "힜", "힜", "힜", "힜", "힜" },
  { "힝", "힝", "힝", "힝", "힝" },
  { "힞", "힞", "힞", "힞", "힞" },
  { "힟", "힟", "힟", "힟", "힟" },
  { "힠", "힠", "힠", "힠", "힠" },
  { "힡", "힡", "힡", "힡", "힡" },
  { "힢", "힢", "힢", "힢", "힢" },
  { "힣", "힣", "힣", "힣", "힣" },
  { "豈", "豈", "豈", "豈", "豈" },
  { "更", "更", "更", "更", "更" },
  { "車", "車", "車", "車", "車" },
  { "賈", "賈", "賈", "賈", "賈" },
  { "滑", "滑", "滑", "滑", "滑" },
  { "串", "串", "串", "串", "串" },
  { "句", "句", "句", "句", "句" },
  { "龜", "龜", "龜", "龜", "龜" },
  { "龜", "龜", "龜", "龜", "龜" },
  { "契", "契", "契", "契", "契" },
  { "金", "金", "金", "金", "金" },
  { "喇", "喇", "喇", "喇", "喇" },
  { "奈", "奈", "奈", "奈", "奈" },
  { "懶", "懶", "懶", "懶", "懶" },
  { "癩", "癩", "癩", "癩", "癩" },
  { "羅", "羅", "羅", "羅", "羅" },
  { "蘿", "蘿", "蘿", "蘿", "蘿" },
  { "螺", "螺", "螺", "螺", "螺" },
  { "裸", "裸", "裸", "裸", "裸" },
  { "邏", "邏", "邏", "邏", "邏" },
  { "樂", "樂", "樂", "樂", "樂" },
  { "洛", "洛", "洛", "洛", "洛" },
  { "烙", "烙", "烙", "烙", "烙" },
  { "珞", "珞", "珞", "珞", "珞" },
  { "落", "落", "落", "落", "落" },
  { "酪", "酪", "酪", "酪", "酪" },
  { "駱", "駱", "駱", "駱", "駱" },
  { "亂", "亂", "亂", "亂", "亂" },
  { "卵", "卵", "卵", "卵", "卵" },
  { "欄", "欄", "欄", "欄", "欄" },
  { "爛", "爛", "爛", "爛", "爛" },
  { "蘭", "蘭", "蘭", "蘭", "蘭" },
  { "鸞", "鸞", "鸞", "鸞", "鸞" },
  { "嵐", "嵐", "嵐", "嵐", "嵐" },
  { "濫", "濫", "濫", "濫", "濫" },
  { "藍", "藍", "藍", "藍", "藍" },
  { "襤", "襤", "襤", "襤", "襤" },
  { "拉", "拉", "拉", "拉", "拉" },
  { "臘", "臘", "臘", "臘", "臘" },
  { "蠟", "蠟", "蠟", "蠟", "蠟" },
  { "廊", "廊", "廊", "廊", "廊" },
  { "朗", "朗", "朗", "朗", "朗" },
  { "浪", "浪", "浪", "浪", "浪" },
  { "狼", "狼", "狼", "狼", "狼" },
  { "郎", "郎", "郎", "郎", "郎" },
  { "來", "來", "來", "來", "來" },
  { "冷", "冷", "冷", "冷", "冷" },
  { "勞", "勞", "勞", "勞", "勞" },
  { "擄", "擄", "擄", "擄", "擄" },
  { "櫓", "櫓", "櫓", "櫓", "櫓" },
  { "爐", "爐", "爐", "爐", "爐" },
  { "盧", "盧", "盧", "盧", "盧" },
  { "老", "老", "老", "老", "老" },
  { "蘆", "蘆", "蘆", "蘆", "蘆" },
  { "虜", "虜", "虜", "虜", "虜" },
  { "路", "路", "路", "路", "路" },
  { "露", "露", "露", "露", "露" },
  { "魯", "魯", "魯", "魯", "魯" },
  { "鷺", "鷺", "鷺", "鷺", "鷺" },
  { "碌", "碌", "碌", "碌", "碌" },
  { "祿", "祿", "祿", "祿", "祿" },
  { "綠", "綠", "綠", "綠", "綠" },
  { "菉", "菉", "菉", "菉", "菉" },
  { "錄", "錄", "錄", "錄", "錄" },
  { "鹿", "鹿", "鹿", "鹿", "鹿" },
  { "論", "論", "論", "論", "論" },
  { "壟", "壟", "壟", "壟", "壟" },
  { "弄", "弄", "弄", "弄", "弄" },
  { "籠", "籠", "籠", "籠", "籠" },
  { "聾", "聾", "聾", "聾", "聾" },
  { "牢", "牢", "牢", "牢", "牢" },
  { "磊", "磊", "磊", "磊", "磊" },
  { "賂", "賂", "賂", "賂", "賂" },
  { "雷", "雷", "雷", "雷", "雷" },
  { "壘", "壘", "壘", "壘", "壘" },
  { "屢", "屢", "屢", "屢", "屢" },
  { "樓", "樓", "樓", "樓", "樓" },
  { "淚", "淚", "淚", "淚", "淚" },
  { "漏", "漏", "漏", "漏", "漏" },
  { "累", "累", "累", "累", "累" },
  { "縷", "縷", "縷", "縷", "縷" },
  { "陋", "陋", "陋", "陋", "陋" },
  { "勒", "勒", "勒", "勒", "勒" },
  { "肋", "肋", "肋", "肋", "肋" },
  { "凜", "凜", "凜", "凜", "凜" },
  { "凌", "凌", "凌", "凌", "凌" },
  { "稜", "稜", "稜", "稜", "稜" },
  { "綾", "綾", "綾", "綾", "綾" },
  { "菱", "菱", "菱", "菱", "菱" },
  { "陵", "陵", "陵", "陵", "陵" },
  { "讀", "讀", "讀", "讀", "讀" },
  { "拏", "拏", "拏", "拏", "拏" },
  { "樂", "樂", "樂", "樂", "樂" },
  { "諾", "諾", "諾", "諾", "諾" },
  { "丹", "丹", "丹", "丹", "丹" },
  { "寧", "寧", "寧", "寧", "寧" },
  { "怒", "怒", "怒", "怒", "怒" },
  { "率", "率", "率", "率", "率" },
  { "異", "異", "異", "異", "異" },
  { "北", "北", "北", "北", "北" },
  { "磻", "磻", "磻", "磻", "磻" },
  { "便", "便", "便", "便", "便" },
  { "復", "復", "復", "復", "復" },
  { "不", "不", "不", "不", "不" },
  { "泌", "泌", "泌", "泌", "泌" },
  { "數", "數", "數", "數", "數" },
  { "索", "索", "索", "索", "索" },
  { "參", "參", "參", "參", "參" },
  { "塞", "塞", "塞", "塞", "塞" },
  { "省", "省", "省", "省", "省" },
  { "葉", "葉", "葉", "葉", "葉" },
  { "說", "說", "說", "說", "說" },
  { "殺", "殺", "殺", "殺", "殺" },
  { "辰", "辰", "辰", "辰", "辰" },
  { "沈", "沈", "沈", "沈", "沈" },
  { "拾", "拾", "拾", "拾", "拾" },
  { "若", "若", "若", "若", "若" },
  { "掠", "掠", "掠", "掠", "掠" },
  { "略", "略", "略", "略", "略" },
  { "亮", "亮", "亮", "亮", "亮" },
  { "兩", "兩", "兩", "兩", "兩" },
  { "凉", "凉", "凉", "凉", "凉" },
  { "梁", "梁", "梁", "梁", "梁" },
  { "糧", "糧", "糧", "糧", "糧" },
  { "良", "良", "良", "良", "良" },
  { "諒", "諒", "諒", "諒", "諒" },
  { "量", "量", "量", "量", "量" },
  { "勵", "勵", "勵", "勵", "勵" },
  { "呂", "呂", "呂", "呂", "呂" },
  { "女", "女", "女", "女", "女" },
  { "廬", "廬", "廬", "廬", "廬" },
  { "旅", "旅", "旅", "旅", "旅" },
  { "濾", "濾", "濾", "濾", "濾" },
  { "礪", "礪", "礪", "礪", "礪" },
  { "閭", "閭", "閭", "閭", "閭" },
  { "驪", "驪", "驪", "驪", "驪" },
  { "麗", "麗", "麗", "麗", "麗" },
  { "黎", "黎", "黎", "黎", "黎" },
  { "力", "力", "力", "力", "力" },
  { "曆", "曆", "曆", "曆", "曆" },
  { "歷", "歷", "歷", "歷", "歷" },
  { "轢", "轢", "轢", "轢", "轢" },
  { "年", "年", "年", "年", "年" },
  { "憐", "憐", "憐", "憐", "憐" },
  { "戀", "戀", "戀", "戀", "戀" },
  { "撚", "撚", "撚", "撚", "撚" },
  { "漣", "漣", "漣", "漣", "漣" },
  { "煉", "煉", "煉", "煉", "煉" },
  { "璉", "璉", "璉", "璉", "璉" },
  { "秊", "秊", "秊", "秊", "秊" },
  { "練", "練", "練", "練", "練" },
  { "聯", "聯", "聯", "聯", "聯" },
  { "輦", "輦", "輦", "輦", "輦" },
  { "蓮", "蓮", "蓮", "蓮", "蓮" },
  { "連", "連", "連", "連", "連" },
  { "鍊", "鍊", "鍊", "鍊", "鍊" },
  { "列", "列", "列", "列", "列" },
  { "劣", "劣", "劣", "劣", "劣" },
  { "咽", "咽", "咽", "咽", "咽" },
  { "烈", "烈", "烈", "烈", "烈" },
  { "裂", "裂", "裂", "裂", "裂" },
  { "說", "說", "說", "說", "說" },
  { "廉", "廉", "廉", "廉", "廉" },
  { "念", "念", "念", "念", "念" },
  { "捻", "捻", "捻", "捻", "捻" },
  { "殮", "殮", "殮", "殮", "殮" },
  { "簾", "簾", "簾", "簾", "簾" },
  { "獵", "獵", "獵", "獵", "獵" },
  { "令", "令", "令", "令", "令" },
  { "囹", "囹", "囹", "囹", "囹" },
  { "寧", "寧", "寧", "寧", "寧" },
  { "嶺", "嶺", "嶺", "嶺", "嶺" },
  { "怜", "怜", "怜", "怜", "怜" },
  { "玲", "玲", "玲", "玲", "玲" },
  { "瑩", "瑩", "瑩", "瑩", "瑩" },
  { "羚", "羚", "羚", "羚", "羚" },
  { "聆", "聆", "聆", "聆", "聆" },
  { "鈴", "鈴", "鈴", "鈴", "鈴" },
  { "零", "零", "零", "零", "零" },
  { "靈", "靈", "靈", "靈", "靈" },
  { "領", "領", "領", "領", "領" },
  { "例", "例", "例", "例", "例" },
  { "禮", "禮", "禮", "禮", "禮" },
  { "醴", "醴", "醴", "醴", "醴" },
  { "隸", "隸", "隸", "隸", "隸" },
  { "惡", "惡", "惡", "惡", "惡" },
  { "了", "了", "了", "了", "了" },
  { "僚", "僚", "僚", "僚", "僚" },
  { "寮", "寮", "寮", "寮", "寮" },
  { "尿", "尿", "尿", "尿", "尿" },
  { "料", "料", "料", "料", "料" },
  { "樂", "樂", "樂", "樂", "樂" },
  { "燎", "燎", "燎", "燎", "燎" },
  { "療", "療", "療", "療", "療" },
  { "蓼", "蓼", "蓼", "蓼", "蓼" },
  { "遼", "遼", "遼", "遼", "遼" },
  { "龍", "龍", "龍", "龍", "龍" },
  { "暈", "暈", "暈", "暈", "暈" },
  { "阮", "阮", "阮", "阮", "阮" },
  { "劉", "劉", "劉", "劉", "劉" },
  { "杻", "杻", "杻", "杻", "杻" },
  { "柳", "柳", "柳", "柳", "柳" },
  { "流", "流", "流", "流", "流" },
  { "溜", "溜", "溜", "溜", "溜" },
  { "琉", "琉", "琉", "琉", "琉" },
  { "留", "留", "留", "留", "留" },
  { "硫", "硫", "硫", "硫", "硫" },
  { "紐", "紐", "紐", "紐", "紐" },
  { "類", "類", "類", "類", "類" },
  { "六", "六", "六", "六", "六" },
  { "戮", "戮", "戮", "戮", "戮" },
  { "陸", "陸", "陸", "陸", "陸" },
  { "倫", "倫", "倫", "倫", "倫" },
  { "崙", "崙", "崙", "崙", "崙" },
  { "淪", "淪", "淪", "淪", "淪" },
  { "輪", "輪", "輪", "輪", "輪" },
  { "律", "律", "律", "律", "律" },
  { "慄", "慄", "慄", "慄", "慄" },
  { "栗", "栗", "栗", "栗", "栗" },
  { "率", "率", "率", "率", "率" },
  { "隆", "隆", "隆", "隆", "隆" },
  { "利", "利", "利", "利", "利" },
  { "吏", "吏", "吏", "吏", "吏" },
  { "履", "履", "履", "履", "履" },
  { "易", "易", "易", "易", "易" },
  { "李", "李", "李", "李", "李" },
  { "梨", "梨", "梨", "梨", "梨" },
  { "泥", "泥", "泥", "泥", "泥" },
  { "理", "理", "理", "理", "理" },
  { "痢", "痢", "痢", "痢", "痢" },
  { "罹", "罹", "罹", "罹", "罹" },
  { "裏", "裏", "裏", "裏", "裏" },
  { "裡", "裡", "裡", "裡", "裡" },
  { "里", "里", "里", "里", "里" },
  { "離", "離", "離", "離", "離" },
  { "匿", "匿", "匿", "匿", "匿" },
  { "溺", "溺", "溺", "溺", "溺" },
  { "吝", "吝", "吝", "吝", "吝" },
  { "燐", "燐", "燐", "燐", "燐" },
  { "璘", "璘", "璘", "璘", "璘" },
  { "藺", "藺", "藺", "藺", "藺" },
  { "隣", "隣", "隣", "隣", "隣" },
  { "鱗", "鱗", "鱗", "鱗", "鱗" },
  { "麟", "麟", "麟", "麟", "麟" },
  { "林", "林", "林", "林", "林" },
  { "淋", "淋", "淋", "淋", "淋" },
  { "臨", "臨", "臨", "臨", "臨" },
  { "立", "立", "立", "立", "立" },
  { "笠", "笠", "笠", "笠", "笠" },
  { "粒", "粒", "粒", "粒", "粒" },
  { "狀", "狀", "狀", "狀", "狀" },
  { "炙", "炙", "炙", "炙", "炙" },
  { "識", "識", "識", "識", "識" },
  { "什", "什", "什", "什", "什" },
  { "茶", "茶", "茶", "茶", "茶" },
  { "刺", "刺", "刺", "刺", "刺" },
  { "切", "切", "切", "切", "切" },
  { "度", "度", "度", "度", "度" },
  { "拓", "拓", "拓", "拓", "拓" },
  { "糖", "糖", "糖", "糖", "糖" },
  { "宅", "宅", "宅", "宅", "宅" },
  { "洞", "洞", "洞", "洞", "洞" },
  { "暴", "暴", "暴", "暴", "暴" },
  { "輻", "輻", "輻", "輻", "輻" },
  { "行", "行", "行", "行", "行" },
  { "降", "降", "降", "降", "降" },
  { "見", "見", "見", "見", "見" },
  { "廓", "廓", "廓", "廓", "廓" },
  { "兀", "兀", "兀", "兀", "兀" },
  { "嗀", "嗀", "嗀", "嗀", "嗀" },
  { "塚", "塚", "塚", "塚", "塚" },
  { "晴", "晴", "晴", "晴", "晴" },
  { "凞", "凞", "凞", "凞", "凞" },
  { "猪", "猪", "猪", "猪", "猪" },
  { "益", "益", "益", "益", "益" },
  { "礼", "礼", "礼", "礼", "礼" },
  { "神", "神", "神", "神", "神" },
  { "祥", "祥", "祥", "祥", "祥" },
  { "福", "福", "福", "福", "福" },
  { "靖", "靖", "靖", "靖", "靖" },
  { "精", "精", "精", "精", "精" },
  { "羽", "羽", "羽", "羽", "羽" },
  { "蘒", "蘒", "蘒", "蘒", "蘒" },
  { "諸", "諸", "諸", "諸", "諸" },
  { "逸", "逸", "逸", "逸", "逸" },
  { "都", "都", "都", "都", "都" },
  { "飯", "飯", "飯", "飯", "飯" },
  { "飼", "飼", "飼", "飼", "飼" },
  { "館", "館", "館", "館", "館" },
  { "鶴", "鶴", "鶴", "鶴", "鶴" },
  { "郞", "郞", "郞", "郞", "郞" },
  { "隷", "隷", "隷", "隷", "隷" },
  { "侮", "侮", "侮", "侮", "侮" },
  { "僧", "僧", "僧", "僧", "僧" },
  { "免", "免", "免", "免", "免" },
  { "勉", "勉", "勉", "勉", "勉" },
  { "勤", "勤", "勤", "勤", "勤" },
  { "卑", "卑", "卑", "卑", "卑" },
  { "喝", "喝", "喝", "喝", "喝" },
  { "嘆", "嘆", "嘆", "嘆", "嘆" },
  { "器", "器", "器", "器", "器" },
  { "塀", "塀", "塀", "塀", "塀" },
  { "墨", "墨", "墨", "墨", "墨" },
  { "層", "層", "層", "層", "層" },
  { "屮", "屮", "屮", "屮", "屮" },
  { "悔", "悔", "悔", "悔", "悔" },
  { "慨", "慨", "慨", "慨", "慨" },
  { "憎", "憎", "憎", "憎", "憎" },
  { "懲", "懲", "懲", "懲", "懲" },
  { "敏", "敏", "敏", "敏", "敏" },
  { "既", "既", "既", "既", "既" },
  { "暑", "暑", "暑", "暑", "暑" },
  { "梅", "梅", "梅", "梅", "梅" },
  { "海", "海", "海", "海", "海" },
  { "渚", "渚", "渚", "渚", "渚" },
  { "漢", "漢", "漢", "漢", "漢" },
  { "煮", "煮", "煮", "煮", "煮" },
  { "爫", "爫", "爫", "爫", "爫" },
  { "琢", "琢", "琢", "琢", "琢" },
  { "碑", "碑", "碑", "碑", "碑" },
  { "社", "社", "社", "社", "社" },
  { "祉", "祉", "祉", "祉", "祉" },
  { "祈", "祈", "祈", "祈", "祈" },
  { "祐", "祐", "祐", "祐", "祐" },
  { "祖", "祖", "祖", "祖", "祖" },
  { "祝", "祝", "祝", "祝", "祝" },
  { "禍", "禍", "禍", "禍", "禍" },
  { "禎", "禎", "禎", "禎", "禎" },
  { "穀", "穀", "穀", "穀", "穀" },
  { "突", "突", "突", "突", "突" },
  { "節", "節", "節", "節", "節" },
  { "練", "練", "練", "練", "練" },
  { "縉", "縉", "縉", "縉", "縉" },
  { "繁", "繁", "繁", "繁", "繁" },
  { "署", "署", "署", "署", "署" },
  { "者", "者", "者", "者", "者" },
  { "臭", "臭", "臭", "臭", "臭" },
  { "艹", "艹", "艹", "艹", "艹" },
  { "艹", "艹", "艹", "艹", "艹" },
  { "著", "著", "著", "著", "著" },
  { "褐", "褐", "褐", "褐", "褐" },
  { "視", "視", "視", "視", "視" },
  { "謁", "謁", "謁", "謁", "謁" },
  { "謹", "謹", "謹", "謹", "謹" },
  { "賓", "賓", "賓", "賓", "賓" },
  { "贈", "贈", "贈", "贈", "贈" },
  { "辶", "辶", "辶", "辶", "辶" },
  { "逸", "逸", "逸", "逸", "逸" },
  { "難", "難", "難", "難", "難" },
  { "響", "響", "響", "響", "響" },
  { "頻", "頻", "頻", "頻", "頻" },
  { "恵", "恵", "恵", "恵", "恵" },
  { "𤋮", "𤋮", "𤋮", "𤋮", "𤋮" },
  { "舘", "舘", "舘", "舘", "舘" },
  { "並", "並", "並", "並", "並" },
  { "况", "况", "况", "况", "况" },
  { "全", "全", "全", "全", "全" },
  { "侀", "侀", "侀", "侀", "侀" },
  { "充", "充", "充", "充", "充" },
  { "冀", "冀", "冀", "冀", "冀" },
  { "勇", "勇", "勇", "勇", "勇" },
  { "勺", "勺", "勺", "勺", "勺" },
  { "喝", "喝", "喝", "喝", "喝" },
  { "啕", "啕", "啕", "啕", "啕" },
  { "喙", "喙", "喙", "喙", "喙" },
  { "嗢", "嗢", "嗢", "嗢", "嗢" },
  { "塚", "塚", "塚", "塚", "塚" },
  { "墳", "墳", "墳", "墳", "墳" },
  { "奄", "奄", "奄", "奄", "奄" },
  { "奔", "奔", "奔", "奔", "奔" },
  { "婢", "婢", "婢", "婢", "婢" },
  { "嬨", "嬨", "嬨", "嬨", "嬨" },
  { "廒", "廒", "廒", "廒", "廒" },
  { "廙", "廙", "廙", "廙", "廙" },
  { "彩", "彩", "彩", "彩", "彩" },
  { "徭", "徭", "徭", "徭", "徭" },
  { "惘", "惘", "惘", "惘", "惘" },
  { "慎", "慎", "慎", "慎", "慎" },
  { "愈", "愈", "愈", "愈", "愈" },
  { "憎", "憎", "憎", "憎", "憎" },
  { "慠", "慠", "慠", "慠", "慠" },
  { "懲", "懲", "懲", "懲", "懲" },
  { "戴", "戴", "戴", "戴", "戴" },
  { "揄", "揄", "揄", "揄", "揄" },
  { "搜", "搜", "搜", "搜", "搜" },
  { "摒", "摒", "摒", "摒", "摒" },
  { "敖", "敖", "敖", "敖", "敖" },
  { "晴", "晴", "晴", "晴", "晴" },
  { "朗", "朗", "朗", "朗", "朗" },
  { "望", "望", "望", "望", "望" },
  { "杖", "杖", "杖", "杖", "杖" },
  { "歹", "歹", "歹", "歹", "歹" },
  { "殺", "殺", "殺", "殺", "殺" },
  { "流", "流", "流", "流", "流" },
  { "滛", "滛", "滛", "滛", "滛" },
  { "滋", "滋", "滋", "滋", "滋" },
  { "漢", "漢", "漢", "漢", "漢" },
  { "瀞", "瀞", "瀞", "瀞", "瀞" },
  { "煮", "煮", "煮", "煮", "煮" },
  { "瞧", "瞧", "瞧", "瞧", "瞧" },
  { "爵", "爵", "爵", "爵", "爵" },
  { "犯", "犯", "犯", "犯", "犯" },
  { "猪", "猪", "猪", "猪", "猪" },
  { "瑱", "瑱", "瑱", "瑱", "瑱" },
  { "甆", "甆", "甆", "甆", "甆" },
  { "画", "画", "画", "画", "画" },
  { "瘝", "瘝", "瘝", "瘝", "瘝" },
  { "瘟", "瘟", "瘟", "瘟", "瘟" },
  { "益", "益", "益", "益", "益" },
  { "盛", "盛", "盛", "盛", "盛" },
  { "直", "直", "直", "直", "直" },
  { "睊", "睊", "睊", "睊", "睊" },
  { "着", "着", "着", "着", "着" },
  { "磌", "磌", "磌", "磌", "磌" },
  { "窱", "窱", "窱", "窱", "窱" },
  { "節", "節", "節", "節", "節" },
  { "类", "类", "类", "类", "类" },
  { "絛", "絛", "絛", "絛", "絛" },
  { "練", "練", "練", "練", "練" },
  { "缾", "缾", "缾", "缾", "缾" },
  { "者", "者", "者", "者", "者" },
  { "荒", "荒", "荒", "荒", "荒" },
  { "華", "華", "華", "華", "華" },
  { "蝹", "蝹", "蝹", "蝹", "蝹" },
  { "襁", "襁", "襁", "襁", "襁" },
  { "覆", "覆", "覆", "覆", "覆" },
  { "視", "視", "視", "視", "視" },
  { "調", "調", "調", "調", "調" },
  { "諸", "諸", "諸", "諸", "諸" },
  { "請", "請", "請", "請", "請" },
  { "謁", "謁", "謁", "謁", "謁" },
  { "諾", "諾", "諾", "諾", "諾" },
  { "諭", "諭", "諭", "諭", "諭" },
  { "謹", "謹", "謹", "謹", "謹" },
  { "變", "變", "變", "變", "變" },
  { "贈", "贈", "贈", "贈", "贈" },
  { "輸", "輸", "輸", "輸", "輸" },
  { "遲", "遲", "遲", "遲", "遲" },
  { "醙", "醙", "醙", "醙", "醙" },
  { "鉶", "鉶", "鉶", "鉶", "鉶" },
  { "陼", "陼", "陼", "陼", "陼" },
  { "難", "難", "難", "難", "難" },
  { "靖", "靖", "靖", "靖", "靖" },
  { "韛", "韛", "韛", "韛", "韛" },
  { "響", "響", "響", "響", "響" },
  { "頋", "頋", "頋", "頋", "頋" },
  { "頻", "頻", "頻", "頻", "頻" },
  { "鬒", "鬒", "鬒", "鬒", "鬒" },
  { "龜", "龜", "龜", "龜", "龜" },
  { "𢡊", "𢡊", "𢡊", "𢡊", "𢡊" },
  { "𢡄", "𢡄", "𢡄", "𢡄", "𢡄" },
  { "𣏕", "𣏕", "𣏕", "𣏕", "𣏕" },
  { "㮝", "㮝", "㮝", "㮝", "㮝" },
  { "䀘", "䀘", "䀘", "䀘", "䀘" },
  { "䀹", "䀹", "䀹", "䀹", "䀹" },
  { "𥉉", "𥉉", "𥉉", "𥉉", "𥉉" },
  { "𥳐", "𥳐", "𥳐", "𥳐", "𥳐" },
  { "𧻓", "𧻓", "𧻓", "𧻓", "𧻓" },
  { "齃", "齃", "齃", "齃", "齃" },
  { "龎", "龎", "龎", "龎", "龎" },
  { "ﬀ", "ﬀ", "ﬀ", "ff", "ff" },
  { "ﬁ", "ﬁ", "ﬁ", "fi", "fi" },
  { "ﬂ", "ﬂ", "ﬂ", "fl", "fl" },
  { "ﬃ", "ﬃ", "ﬃ", "ffi", "ffi" },
  { "ﬄ", "ﬄ", "ﬄ", "ffl", "ffl" },
  { "ﬅ", "ﬅ", "ﬅ", "st", "st" },
  { "ﬆ", "ﬆ", "ﬆ", "st", "st" },
  { "ﬓ", "ﬓ", "ﬓ", "մն", "մն" },
  { "ﬔ", "ﬔ", "ﬔ", "մե", "մե" },
  { "ﬕ", "ﬕ", "ﬕ", "մի", "մի" },
  { "ﬖ", "ﬖ", "ﬖ", "վն", "վն" },
  { "ﬗ", "ﬗ", "ﬗ", "մխ", "մխ" },
  { "יִ", "יִ", "יִ", "יִ", "יִ" },
  { "ײַ", "ײַ", "ײַ", "ײַ", "ײַ" },
  { "ﬠ", "ﬠ", "ﬠ", "ע", "ע" },
  { "ﬡ", "ﬡ", "ﬡ", "א", "א" },
  { "ﬢ", "ﬢ", "ﬢ", "ד", "ד" },
  { "ﬣ", "ﬣ", "ﬣ", "ה", "ה" },
  { "ﬤ", "ﬤ", "ﬤ", "כ", "כ" },
  { "ﬥ", "ﬥ", "ﬥ", "ל", "ל" },
  { "ﬦ", "ﬦ", "ﬦ", "ם", "ם" },
  { "ﬧ", "ﬧ", "ﬧ", "ר", "ר" },
  { "ﬨ", "ﬨ", "ﬨ", "ת", "ת" },
  { "﬩", "﬩", "﬩", "+", "+" },
  { "שׁ", "שׁ", "שׁ", "שׁ", "שׁ" },
  { "שׂ", "שׂ", "שׂ", "שׂ", "שׂ" },
  { "שּׁ", "שּׁ", "שּׁ", "שּׁ", "שּׁ" },
  { "שּׂ", "שּׂ", "שּׂ", "שּׂ", "שּׂ" },
  { "אַ", "אַ", "אַ", "אַ", "אַ" },
  { "אָ", "אָ", "אָ", "אָ", "אָ" },
  { "אּ", "אּ", "אּ", "אּ", "אּ" },
  { "בּ", "בּ", "בּ", "בּ", "בּ" },
  { "גּ", "גּ", "גּ", "גּ", "גּ" },
  { "דּ", "דּ", "דּ", "דּ", "דּ" },
  { "הּ", "הּ", "הּ", "הּ", "הּ" },
  { "וּ", "וּ", "וּ", "וּ", "וּ" },
  { "זּ", "זּ", "זּ", "זּ", "זּ" },
  { "טּ", "טּ", "טּ", "טּ", "טּ" },
  { "יּ", "יּ", "יּ", "יּ", "יּ" },
  { "ךּ", "ךּ", "ךּ", "ךּ", "ךּ" },
  { "כּ", "כּ", "כּ", "כּ", "כּ" },
  { "לּ", "לּ", "לּ", "לּ", "לּ" },
  { "מּ", "מּ", "מּ", "מּ", "מּ" },
  { "נּ", "נּ", "נּ", "נּ", "נּ" },
  { "סּ", "סּ", "סּ", "סּ", "סּ" },
  { "ףּ", "ףּ", "ףּ", "ףּ", "ףּ" },
  { "פּ", "פּ", "פּ", "פּ", "פּ" },
  { "צּ", "צּ", "צּ", "צּ", "צּ" },
  { "קּ", "קּ", "קּ", "קּ", "קּ" },
  { "רּ", "רּ", "רּ", "רּ", "רּ" },
  { "שּ", "שּ", "שּ", "שּ", "שּ" },
  { "תּ", "תּ", "תּ", "תּ", "תּ" },
  { "וֹ", "וֹ", "וֹ", "וֹ", "וֹ" },
  { "בֿ", "בֿ", "בֿ", "בֿ", "בֿ" },
  { "כֿ", "כֿ", "כֿ", "כֿ", "כֿ" },
  { "פֿ", "פֿ", "פֿ", "פֿ", "פֿ" },
  { "ﭏ", "ﭏ", "ﭏ", "אל", "אל" },
  { "ﭐ", "ﭐ", "ﭐ", "ٱ", "ٱ" },
  { "ﭑ", "ﭑ", "ﭑ", "ٱ", "ٱ" },
  { "ﭒ", "ﭒ", "ﭒ", "ٻ", "ٻ" },
  { "ﭓ", "ﭓ", "ﭓ", "ٻ", "ٻ" },
  { "ﭔ", "ﭔ", "ﭔ", "ٻ", "ٻ" },
  { "ﭕ", "ﭕ", "ﭕ", "ٻ", "ٻ" },
  { "ﭖ", "ﭖ", "ﭖ", "پ", "پ" },
  { "ﭗ", "ﭗ", "ﭗ", "پ", "پ" },
  { "ﭘ", "ﭘ", "ﭘ", "پ", "پ" },
  { "ﭙ", "ﭙ", "ﭙ", "پ", "پ" },
  { "ﭚ", "ﭚ", "ﭚ", "ڀ", "ڀ" },
  { "ﭛ", "ﭛ", "ﭛ", "ڀ", "ڀ" },
  { "ﭜ", "ﭜ", "ﭜ", "ڀ", "ڀ" },
  { "ﭝ", "ﭝ", "ﭝ", "ڀ", "ڀ" },
  { "ﭞ", "ﭞ", "ﭞ", "ٺ", "ٺ" },
  { "ﭟ", "ﭟ", "ﭟ", "ٺ", "ٺ" },
  { "ﭠ", "ﭠ", "ﭠ", "ٺ", "ٺ" },
  { "ﭡ", "ﭡ", "ﭡ", "ٺ", "ٺ" },
  { "ﭢ", "ﭢ", "ﭢ", "ٿ", "ٿ" },
  { "ﭣ", "ﭣ", "ﭣ", "ٿ", "ٿ" },
  { "ﭤ", "ﭤ", "ﭤ", "ٿ", "ٿ" },
  { "ﭥ", "ﭥ", "ﭥ", "ٿ", "ٿ" },
  { "ﭦ", "ﭦ", "ﭦ", "ٹ", "ٹ" },
  { "ﭧ", "ﭧ", "ﭧ", "ٹ", "ٹ" },
  { "ﭨ", "ﭨ", "ﭨ", "ٹ", "ٹ" },
  { "ﭩ", "ﭩ", "ﭩ", "ٹ", "ٹ" },
  { "ﭪ", "ﭪ", "ﭪ", "ڤ", "ڤ" },
  { "ﭫ", "ﭫ", "ﭫ", "ڤ", "ڤ" },
  { "ﭬ", "ﭬ", "ﭬ", "ڤ", "ڤ" },
  { "ﭭ", "ﭭ", "ﭭ", "ڤ", "ڤ" },
  { "ﭮ", "ﭮ", "ﭮ", "ڦ", "ڦ" },
  { "ﭯ", "ﭯ", "ﭯ", "ڦ", "ڦ" },
  { "ﭰ", "ﭰ", "ﭰ", "ڦ", "ڦ" },
  { "ﭱ", "ﭱ", "ﭱ", "ڦ", "ڦ" },
  { "ﭲ", "ﭲ", "ﭲ", "ڄ", "ڄ" },
  { "ﭳ", "ﭳ", "ﭳ", "ڄ", "ڄ" },
  { "ﭴ", "ﭴ", "ﭴ", "ڄ", "ڄ" },
  { "ﭵ", "ﭵ", "ﭵ", "ڄ", "ڄ" },
  { "ﭶ", "ﭶ", "ﭶ", "ڃ", "ڃ" },
  { "ﭷ", "ﭷ", "ﭷ", "ڃ", "ڃ" },
  { "ﭸ", "ﭸ", "ﭸ", "ڃ", "ڃ" },
  { "ﭹ", "ﭹ", "ﭹ", "ڃ", "ڃ" },
  { "ﭺ", "ﭺ", "ﭺ", "چ", "چ" },
  { "ﭻ", "ﭻ", "ﭻ", "چ", "چ" },
  { "ﭼ", "ﭼ", "ﭼ", "چ", "چ" },
  { "ﭽ", "ﭽ", "ﭽ", "چ", "چ" },
  { "ﭾ", "ﭾ", "ﭾ", "ڇ", "ڇ" },
  { "ﭿ", "ﭿ", "ﭿ", "ڇ", "ڇ" },
  { "ﮀ", "ﮀ", "ﮀ", "ڇ", "ڇ" },
  { "ﮁ", "ﮁ", "ﮁ", "ڇ", "ڇ" },
  { "ﮂ", "ﮂ", "ﮂ", "ڍ", "ڍ" },
  { "ﮃ", "ﮃ", "ﮃ", "ڍ", "ڍ" },
  { "ﮄ", "ﮄ", "ﮄ", "ڌ", "ڌ" },
  { "ﮅ", "ﮅ", "ﮅ", "ڌ", "ڌ" },
  { "ﮆ", "ﮆ", "ﮆ", "ڎ", "ڎ" },
  { "ﮇ", "ﮇ", "ﮇ", "ڎ", "ڎ" },
  { "ﮈ", "ﮈ", "ﮈ", "ڈ", "ڈ" },
  { "ﮉ", "ﮉ", "ﮉ", "ڈ", "ڈ" },
  { "ﮊ", "ﮊ", "ﮊ", "ژ", "ژ" },
  { "ﮋ", "ﮋ", "ﮋ", "ژ", "ژ" },
  { "ﮌ", "ﮌ", "ﮌ", "ڑ", "ڑ" },
  { "ﮍ", "ﮍ", "ﮍ", "ڑ", "ڑ" },
  { "ﮎ", "ﮎ", "ﮎ", "ک", "ک" },
  { "ﮏ", "ﮏ", "ﮏ", "ک", "ک" },
  { "ﮐ", "ﮐ", "ﮐ", "ک", "ک" },
  { "ﮑ", "ﮑ", "ﮑ", "ک", "ک" },
  { "ﮒ", "ﮒ", "ﮒ", "گ", "گ" },
  { "ﮓ", "ﮓ", "ﮓ", "گ", "گ" },
  { "ﮔ", "ﮔ", "ﮔ", "گ", "گ" },
  { "ﮕ", "ﮕ", "ﮕ", "گ", "گ" },
  { "ﮖ", "ﮖ", "ﮖ", "ڳ", "ڳ" },
  { "ﮗ", "ﮗ", "ﮗ", "ڳ", "ڳ" },
  { "ﮘ", "ﮘ", "ﮘ", "ڳ", "ڳ" },
  { "ﮙ", "ﮙ", "ﮙ", "ڳ", "ڳ" },
  { "ﮚ", "ﮚ", "ﮚ", "ڱ", "ڱ" },
  { "ﮛ", "ﮛ", "ﮛ", "ڱ", "ڱ" },
  { "ﮜ", "ﮜ", "ﮜ", "ڱ", "ڱ" },
  { "ﮝ", "ﮝ", "ﮝ", "ڱ", "ڱ" },
  { "ﮞ", "ﮞ", "ﮞ", "ں", "ں" },
  { "ﮟ", "ﮟ", "ﮟ", "ں", "ں" },
  { "ﮠ", "ﮠ", "ﮠ", "ڻ", "ڻ" },
  { "ﮡ", "ﮡ", "ﮡ", "ڻ", "ڻ" },
  { "ﮢ", "ﮢ", "ﮢ", "ڻ", "ڻ" },
  { "ﮣ", "ﮣ", "ﮣ", "ڻ", "ڻ" },
  { "ﮤ", "ﮤ", "ﮤ", "ۀ", "ۀ" },
  { "ﮥ", "ﮥ", "ﮥ", "ۀ", "ۀ" },
  { "ﮦ", "ﮦ", "ﮦ", "ہ", "ہ" },
  { "ﮧ", "ﮧ", "ﮧ", "ہ", "ہ" },
  { "ﮨ", "ﮨ", "ﮨ", "ہ", "ہ" },
  { "ﮩ", "ﮩ", "ﮩ", "ہ", "ہ" },
  { "ﮪ", "ﮪ", "ﮪ", "ھ", "ھ" },
  { "ﮫ", "ﮫ", "ﮫ", "ھ", "ھ" },
  { "ﮬ", "ﮬ", "ﮬ", "ھ", "ھ" },
  { "ﮭ", "ﮭ", "ﮭ", "ھ", "ھ" },
  { "ﮮ", "ﮮ", "ﮮ", "ے", "ے" },
  { "ﮯ", "ﮯ", "ﮯ", "ے", "ے" },
  { "ﮰ", "ﮰ", "ﮰ", "ۓ", "ۓ" },
  { "ﮱ", "ﮱ", "ﮱ", "ۓ", "ۓ" },
  { "ﯓ", "ﯓ", "ﯓ", "ڭ", "ڭ" },
  { "ﯔ", "ﯔ", "ﯔ", "ڭ", "ڭ" },
  { "ﯕ", "ﯕ", "ﯕ", "ڭ", "ڭ" },
  { "ﯖ", "ﯖ", "ﯖ", "ڭ", "ڭ" },
  { "ﯗ", "ﯗ", "ﯗ", "ۇ", "ۇ" },
  { "ﯘ", "ﯘ", "ﯘ", "ۇ", "ۇ" },
  { "ﯙ", "ﯙ", "ﯙ", "ۆ", "ۆ" },
  { "ﯚ", "ﯚ", "ﯚ", "ۆ", "ۆ" },
  { "ﯛ", "ﯛ", "ﯛ", "ۈ", "ۈ" },
  { "ﯜ", "ﯜ", "ﯜ", "ۈ", "ۈ" },
  { "ﯝ", "ﯝ", "ﯝ", "ۇٴ", "ۇٴ" },
  { "ﯞ", "ﯞ", "ﯞ", "ۋ", "ۋ" },
  { "ﯟ", "ﯟ", "ﯟ", "ۋ", "ۋ" },
  { "ﯠ", "ﯠ", "ﯠ", "ۅ", "ۅ" },
  { "ﯡ", "ﯡ", "ﯡ", "ۅ", "ۅ" },
  { "ﯢ", "ﯢ", "ﯢ", "ۉ", "ۉ" },
  { "ﯣ", "ﯣ", "ﯣ", "ۉ", "ۉ" },
  { "ﯤ", "ﯤ", "ﯤ", "ې", "ې" },
  { "ﯥ", "ﯥ", "ﯥ", "ې", "ې" },
  { "ﯦ", "ﯦ", "ﯦ", "ې", "ې" },
  { "ﯧ", "ﯧ", "ﯧ", "ې", "ې" },
  { "ﯨ", "ﯨ", "ﯨ", "ى", "ى" },
  { "ﯩ", "ﯩ", "ﯩ", "ى", "ى" },
  { "ﯪ", "ﯪ", "ﯪ", "ئا", "ئا" },
  { "ﯫ", "ﯫ", "ﯫ", "ئا", "ئا" },
  { "ﯬ", "ﯬ", "ﯬ", "ئە", "ئە" },
  { "ﯭ", "ﯭ", "ﯭ", "ئە", "ئە" },
  { "ﯮ", "ﯮ", "ﯮ", "ئو", "ئو" },
  { "ﯯ", "ﯯ", "ﯯ", "ئو", "ئو" },
  { "ﯰ", "ﯰ", "ﯰ", "ئۇ", "ئۇ" },
  { "ﯱ", "ﯱ", "ﯱ", "ئۇ", "ئۇ" },
  { "ﯲ", "ﯲ", "ﯲ", "ئۆ", "ئۆ" },
  { "ﯳ", "ﯳ", "ﯳ", "ئۆ", "ئۆ" },
  { "ﯴ", "ﯴ", "ﯴ", "ئۈ", "ئۈ" },
  { "ﯵ", "ﯵ", "ﯵ", "ئۈ", "ئۈ" },
  { "ﯶ", "ﯶ", "ﯶ", "ئې", "ئې" },
  { "ﯷ", "ﯷ", "ﯷ", "ئې", "ئې" },
  { "ﯸ", "ﯸ", "ﯸ", "ئې", "ئې" },
  { "ﯹ", "ﯹ", "ﯹ", "ئى", "ئى" },
  { "ﯺ", "ﯺ", "ﯺ", "ئى", "ئى" },
  { "ﯻ", "ﯻ", "ﯻ", "ئى", "ئى" },
  { "ﯼ", "ﯼ", "ﯼ", "ی", "ی" },
  { "ﯽ", "ﯽ", "ﯽ", "ی", "ی" },
  { "ﯾ", "ﯾ", "ﯾ", "ی", "ی" },
  { "ﯿ", "ﯿ", "ﯿ", "ی", "ی" },
  { "ﰀ", "ﰀ", "ﰀ", "ئج", "ئج" },
  { "ﰁ", "ﰁ", "ﰁ", "ئح", "ئح" },
  { "ﰂ", "ﰂ", "ﰂ", "ئم", "ئم" },
  { "ﰃ", "ﰃ", "ﰃ", "ئى", "ئى" },
  { "ﰄ", "ﰄ", "ﰄ", "ئي", "ئي" },
  { "ﰅ", "ﰅ", "ﰅ", "بج", "بج" },
  { "ﰆ", "ﰆ", "ﰆ", "بح", "بح" },
  { "ﰇ", "ﰇ", "ﰇ", "بخ", "بخ" },
  { "ﰈ", "ﰈ", "ﰈ", "بم", "بم" },
  { "ﰉ", "ﰉ", "ﰉ", "بى", "بى" },
  { "ﰊ", "ﰊ", "ﰊ", "بي", "بي" },
  { "ﰋ", "ﰋ", "ﰋ", "تج", "تج" },
  { "ﰌ", "ﰌ", "ﰌ", "تح", "تح" },
  { "ﰍ", "ﰍ", "ﰍ", "تخ", "تخ" },
  { "ﰎ", "ﰎ", "ﰎ", "تم", "تم" },
  { "ﰏ", "ﰏ", "ﰏ", "تى", "تى" },
  { "ﰐ", "ﰐ", "ﰐ", "تي", "تي" },
  { "ﰑ", "ﰑ", "ﰑ", "ثج", "ثج" },
  { "ﰒ", "ﰒ", "ﰒ", "ثم", "ثم" },
  { "ﰓ", "ﰓ", "ﰓ", "ثى", "ثى" },
  { "ﰔ", "ﰔ", "ﰔ", "ثي", "ثي" },
  { "ﰕ", "ﰕ", "ﰕ", "جح", "جح" },
  { "ﰖ", "ﰖ", "ﰖ", "جم", "جم" },
  { "ﰗ", "ﰗ", "ﰗ", "حج", "حج" },
  { "ﰘ", "ﰘ", "ﰘ", "حم", "حم" },
  { "ﰙ", "ﰙ", "ﰙ", "خج", "خج" },
  { "ﰚ", "ﰚ", "ﰚ", "خح", "خح" },
  { "ﰛ", "ﰛ", "ﰛ", "خم", "خم" },
  { "ﰜ", "ﰜ", "ﰜ", "سج", "سج" },
  { "ﰝ", "ﰝ", "ﰝ", "سح", "سح" },
  { "ﰞ", "ﰞ", "ﰞ", "سخ", "سخ" },
  { "ﰟ", "ﰟ", "ﰟ", "سم", "سم" },
  { "ﰠ", "ﰠ", "ﰠ", "صح", "صح" },
  { "ﰡ", "ﰡ", "ﰡ", "صم", "صم" },
  { "ﰢ", "ﰢ", "ﰢ", "ضج", "ضج" },
  { "ﰣ", "ﰣ", "ﰣ", "ضح", "ضح" },
  { "ﰤ", "ﰤ", "ﰤ", "ضخ", "ضخ" },
  { "ﰥ", "ﰥ", "ﰥ", "ضم", "ضم" },
  { "ﰦ", "ﰦ", "ﰦ", "طح", "طح" },
  { "ﰧ", "ﰧ", "ﰧ", "طم", "طم" },
  { "ﰨ", "ﰨ", "ﰨ", "ظم", "ظم" },
  { "ﰩ", "ﰩ", "ﰩ", "عج", "عج" },
  { "ﰪ", "ﰪ", "ﰪ", "عم", "عم" },
  { "ﰫ", "ﰫ", "ﰫ", "غج", "غج" },
  { "ﰬ", "ﰬ", "ﰬ", "غم", "غم" },
  { "ﰭ", "ﰭ", "ﰭ", "فج", "فج" },
  { "ﰮ", "ﰮ", "ﰮ", "فح", "فح" },
  { "ﰯ", "ﰯ", "ﰯ", "فخ", "فخ" },
  { "ﰰ", "ﰰ", "ﰰ", "فم", "فم" },
  { "ﰱ", "ﰱ", "ﰱ", "فى", "فى" },
  { "ﰲ", "ﰲ", "ﰲ", "في", "في" },
  { "ﰳ", "ﰳ", "ﰳ", "قح", "قح" },
  { "ﰴ", "ﰴ", "ﰴ", "قم", "قم" },
  { "ﰵ", "ﰵ", "ﰵ", "قى", "قى" },
  { "ﰶ", "ﰶ", "ﰶ", "قي", "قي" },
  { "ﰷ", "ﰷ", "ﰷ", "كا", "كا" },
  { "ﰸ", "ﰸ", "ﰸ", "كج", "كج" },
  { "ﰹ", "ﰹ", "ﰹ", "كح", "كح" },
  { "ﰺ", "ﰺ", "ﰺ", "كخ", "كخ" },
  { "ﰻ", "ﰻ", "ﰻ", "كل", "كل" },
  { "ﰼ", "ﰼ", "ﰼ", "كم", "كم" },
  { "ﰽ", "ﰽ", "ﰽ", "كى", "كى" },
  { "ﰾ", "ﰾ", "ﰾ", "كي", "كي" },
  { "ﰿ", "ﰿ", "ﰿ", "لج", "لج" },
  { "ﱀ", "ﱀ", "ﱀ", "لح", "لح" },
  { "ﱁ", "ﱁ", "ﱁ", "لخ", "لخ" },
  { "ﱂ", "ﱂ", "ﱂ", "لم", "لم" },
  { "ﱃ", "ﱃ", "ﱃ", "لى", "لى" },
  { "ﱄ", "ﱄ", "ﱄ", "لي", "لي" },
  { "ﱅ", "ﱅ", "ﱅ", "مج", "مج" },
  { "ﱆ", "ﱆ", "ﱆ", "مح", "مح" },
  { "ﱇ", "ﱇ", "ﱇ", "مخ", "مخ" },
  { "ﱈ", "ﱈ", "ﱈ", "مم", "مم" },
  { "ﱉ", "ﱉ", "ﱉ", "مى", "مى" },
  { "ﱊ", "ﱊ", "ﱊ", "مي", "مي" },
  { "ﱋ", "ﱋ", "ﱋ", "نج", "نج" },
  { "ﱌ", "ﱌ", "ﱌ", "نح", "نح" },
  { "ﱍ", "ﱍ", "ﱍ", "نخ", "نخ" },
  { "ﱎ", "ﱎ", "ﱎ", "نم", "نم" },
  { "ﱏ", "ﱏ", "ﱏ", "نى", "نى" },
  { "ﱐ", "ﱐ", "ﱐ", "ني", "ني" },
  { "ﱑ", "ﱑ", "ﱑ", "هج", "هج" },
  { "ﱒ", "ﱒ", "ﱒ", "هم", "هم" },
  { "ﱓ", "ﱓ", "ﱓ", "هى", "هى" },
  { "ﱔ", "ﱔ", "ﱔ", "هي", "هي" },
  { "ﱕ", "ﱕ", "ﱕ", "يج", "يج" },
  { "ﱖ", "ﱖ", "ﱖ", "يح", "يح" },
  { "ﱗ", "ﱗ", "ﱗ", "يخ", "يخ" },
  { "ﱘ", "ﱘ", "ﱘ", "يم", "يم" },
  { "ﱙ", "ﱙ", "ﱙ", "يى", "يى" },
  { "ﱚ", "ﱚ", "ﱚ", "يي", "يي" },
  { "ﱛ", "ﱛ", "ﱛ", "ذٰ", "ذٰ" },
  { "ﱜ", "ﱜ", "ﱜ", "رٰ", "رٰ" },
  { "ﱝ", "ﱝ", "ﱝ", "ىٰ", "ىٰ" },
  { "ﱞ", "ﱞ", "ﱞ", " ٌّ", " ٌّ" },
  { "ﱟ", "ﱟ", "ﱟ", " ٍّ", " ٍّ" },
  { "ﱠ", "ﱠ", "ﱠ", " َّ", " َّ" },
  { "ﱡ", "ﱡ", "ﱡ", " ُّ", " ُّ" },
  { "ﱢ", "ﱢ", "ﱢ", " ِّ", " ِّ" },
  { "ﱣ", "ﱣ", "ﱣ", " ّٰ", " ّٰ" },
  { "ﱤ", "ﱤ", "ﱤ", "ئر", "ئر" },
  { "ﱥ", "ﱥ", "ﱥ", "ئز", "ئز" },
  { "ﱦ", "ﱦ", "ﱦ", "ئم", "ئم" },
  { "ﱧ", "ﱧ", "ﱧ", "ئن", "ئن" },
  { "ﱨ", "ﱨ", "ﱨ", "ئى", "ئى" },
  { "ﱩ", "ﱩ", "ﱩ", "ئي", "ئي" },
  { "ﱪ", "ﱪ", "ﱪ", "بر", "بر" },
  { "ﱫ", "ﱫ", "ﱫ", "بز", "بز" },
  { "ﱬ", "ﱬ", "ﱬ", "بم", "بم" },
  { "ﱭ", "ﱭ", "ﱭ", "بن", "بن" },
  { "ﱮ", "ﱮ", "ﱮ", "بى", "بى" },
  { "ﱯ", "ﱯ", "ﱯ", "بي", "بي" },
  { "ﱰ", "ﱰ", "ﱰ", "تر", "تر" },
  { "ﱱ", "ﱱ", "ﱱ", "تز", "تز" },
  { "ﱲ", "ﱲ", "ﱲ", "تم", "تم" },
  { "ﱳ", "ﱳ", "ﱳ", "تن", "تن" },
  { "ﱴ", "ﱴ", "ﱴ", "تى", "تى" },
  { "ﱵ", "ﱵ", "ﱵ", "تي", "تي" },
  { "ﱶ", "ﱶ", "ﱶ", "ثر", "ثر" },
  { "ﱷ", "ﱷ", "ﱷ", "ثز", "ثز" },
  { "ﱸ", "ﱸ", "ﱸ", "ثم", "ثم" },
  { "ﱹ", "ﱹ", "ﱹ", "ثن", "ثن" },
  { "ﱺ", "ﱺ", "ﱺ", "ثى", "ثى" },
  { "ﱻ", "ﱻ", "ﱻ", "ثي", "ثي" },
  { "ﱼ", "ﱼ", "ﱼ", "فى", "فى" },
  { "ﱽ", "ﱽ", "ﱽ", "في", "في" },
  { "ﱾ", "ﱾ", "ﱾ", "قى", "قى" },
  { "ﱿ", "ﱿ", "ﱿ", "قي", "قي" },
  { "ﲀ", "ﲀ", "ﲀ", "كا", "كا" },
  { "ﲁ", "ﲁ", "ﲁ", "كل", "كل" },
  { "ﲂ", "ﲂ", "ﲂ", "كم", "كم" },
  { "ﲃ", "ﲃ", "ﲃ", "كى", "كى" },
  { "ﲄ", "ﲄ", "ﲄ", "كي", "كي" },
  { "ﲅ", "ﲅ", "ﲅ", "لم", "لم" },
  { "ﲆ", "ﲆ", "ﲆ", "لى", "لى" },
  { "ﲇ", "ﲇ", "ﲇ", "لي", "لي" },
  { "ﲈ", "ﲈ", "ﲈ", "ما", "ما" },
  { "ﲉ", "ﲉ", "ﲉ", "مم", "مم" },
  { "ﲊ", "ﲊ", "ﲊ", "نر", "نر" },
  { "ﲋ", "ﲋ", "ﲋ", "نز", "نز" },
  { "ﲌ", "ﲌ", "ﲌ", "نم", "نم" },
  { "ﲍ", "ﲍ", "ﲍ", "نن", "نن" },
  { "ﲎ", "ﲎ", "ﲎ", "نى", "نى" },
  { "ﲏ", "ﲏ", "ﲏ", "ني", "ني" },
  { "ﲐ", "ﲐ", "ﲐ", "ىٰ", "ىٰ" },
  { "ﲑ", "ﲑ", "ﲑ", "ير", "ير" },
  { "ﲒ", "ﲒ", "ﲒ", "يز", "يز" },
  { "ﲓ", "ﲓ", "ﲓ", "يم", "يم" },
  { "ﲔ", "ﲔ", "ﲔ", "ين", "ين" },
  { "ﲕ", "ﲕ", "ﲕ", "يى", "يى" },
  { "ﲖ", "ﲖ", "ﲖ", "يي", "يي" },
  { "ﲗ", "ﲗ", "ﲗ", "ئج", "ئج" },
  { "ﲘ", "ﲘ", "ﲘ", "ئح", "ئح" },
  { "ﲙ", "ﲙ", "ﲙ", "ئخ", "ئخ" },
  { "ﲚ", "ﲚ", "ﲚ", "ئم", "ئم" },
  { "ﲛ", "ﲛ", "ﲛ", "ئه", "ئه" },
  { "ﲜ", "ﲜ", "ﲜ", "بج", "بج" },
  { "ﲝ", "ﲝ", "ﲝ", "بح", "بح" },
  { "ﲞ", "ﲞ", "ﲞ", "بخ", "بخ" },
  { "ﲟ", "ﲟ", "ﲟ", "بم", "بم" },
  { "ﲠ", "ﲠ", "ﲠ", "به", "به" },
  { "ﲡ", "ﲡ", "ﲡ", "تج", "تج" },
  { "ﲢ", "ﲢ", "ﲢ", "تح", "تح" },
  { "ﲣ", "ﲣ", "ﲣ", "تخ", "تخ" },
  { "ﲤ", "ﲤ", "ﲤ", "تم", "تم" },
  { "ﲥ", "ﲥ", "ﲥ", "ته", "ته" },
  { "ﲦ", "ﲦ", "ﲦ", "ثم", "ثم" },
  { "ﲧ", "ﲧ", "ﲧ", "جح", "جح" },
  { "ﲨ", "ﲨ", "ﲨ", "جم", "جم" },
  { "ﲩ", "ﲩ", "ﲩ", "حج", "حج" },
  { "ﲪ", "ﲪ", "ﲪ", "حم", "حم" },
  { "ﲫ", "ﲫ", "ﲫ", "خج", "خج" },
  { "ﲬ", "ﲬ", "ﲬ", "خم", "خم" },
  { "ﲭ", "ﲭ", "ﲭ", "سج", "سج" },
  { "ﲮ", "ﲮ", "ﲮ", "سح", "سح" },
  { "ﲯ", "ﲯ", "ﲯ", "سخ", "سخ" },
  { "ﲰ", "ﲰ", "ﲰ", "سم", "سم" },
  { "ﲱ", "ﲱ", "ﲱ", "صح", "صح" },
  { "ﲲ", "ﲲ", "ﲲ", "صخ", "صخ" },
  { "ﲳ", "ﲳ", "ﲳ", "صم", "صم" },
  { "ﲴ", "ﲴ", "ﲴ", "ضج", "ضج" },
  { "ﲵ", "ﲵ", "ﲵ", "ضح", "ضح" },
  { "ﲶ", "ﲶ", "ﲶ", "ضخ", "ضخ" },
  { "ﲷ", "ﲷ", "ﲷ", "ضم", "ضم" },
  { "ﲸ", "ﲸ", "ﲸ", "طح", "طح" },
  { "ﲹ", "ﲹ", "ﲹ", "ظم", "ظم" },
  { "ﲺ", "ﲺ", "ﲺ", "عج", "عج" },
  { "ﲻ", "ﲻ", "ﲻ", "عم", "عم" },
  { "ﲼ", "ﲼ", "ﲼ", "غج", "غج" },
  { "ﲽ", "ﲽ", "ﲽ", "غم", "غم" },
  { "ﲾ", "ﲾ", "ﲾ", "فج", "فج" },
  { "ﲿ", "ﲿ", "ﲿ", "فح", "فح" },
  { "ﳀ", "ﳀ", "ﳀ", "فخ", "فخ" },
  { "ﳁ", "ﳁ", "ﳁ", "فم", "فم" },
  { "ﳂ", "ﳂ", "ﳂ", "قح", "قح" },
  { "ﳃ", "ﳃ", "ﳃ", "قم", "قم" },
  { "ﳄ", "ﳄ", "ﳄ", "كج", "كج" },
  { "ﳅ", "ﳅ", "ﳅ", "كح", "كح" },
  { "ﳆ", "ﳆ", "ﳆ", "كخ", "كخ" },
  { "ﳇ", "ﳇ", "ﳇ", "كل", "كل" },
  { "ﳈ", "ﳈ", "ﳈ", "كم", "كم" },
  { "ﳉ", "ﳉ", "ﳉ", "لج", "لج" },
  { "ﳊ", "ﳊ", "ﳊ", "لح", "لح" },
  { "ﳋ", "ﳋ", "ﳋ", "لخ", "لخ" },
  { "ﳌ", "ﳌ", "ﳌ", "لم", "لم" },
  { "ﳍ", "ﳍ", "ﳍ", "له", "له" },
  { "ﳎ", "ﳎ", "ﳎ", "مج", "مج" },
  { "ﳏ", "ﳏ", "ﳏ", "مح", "مح" },
  { "ﳐ", "ﳐ", "ﳐ", "مخ", "مخ" },
  { "ﳑ", "ﳑ", "ﳑ", "مم", "مم" },
  { "ﳒ", "ﳒ", "ﳒ", "نج", "نج" },
  { "ﳓ", "ﳓ", "ﳓ", "نح", "نح" },
  { "ﳔ", "ﳔ", "ﳔ", "نخ", "نخ" },
  { "ﳕ", "ﳕ", "ﳕ", "نم", "نم" },
  { "ﳖ", "ﳖ", "ﳖ", "نه", "نه" },
  { "ﳗ", "ﳗ", "ﳗ", "هج", "هج" },
  { "ﳘ", "ﳘ", "ﳘ", "هم", "هم" },
  { "ﳙ", "ﳙ", "ﳙ", "هٰ", "هٰ" },
  { "ﳚ", "ﳚ", "ﳚ", "يج", "يج" },
  { "ﳛ", "ﳛ", "ﳛ", "يح", "يح" },
  { "ﳜ", "ﳜ", "ﳜ", "يخ", "يخ" },
  { "ﳝ", "ﳝ", "ﳝ", "يم", "يم" },
  { "ﳞ", "ﳞ", "ﳞ", "يه", "يه" },
  { "ﳟ", "ﳟ", "ﳟ", "ئم", "ئم" },
  { "ﳠ", "ﳠ", "ﳠ", "ئه", "ئه" },
  { "ﳡ", "ﳡ", "ﳡ", "بم", "بم" },
  { "ﳢ", "ﳢ", "ﳢ", "به", "به" },
  { "ﳣ", "ﳣ", "ﳣ", "تم", "تم" },
  { "ﳤ", "ﳤ", "ﳤ", "ته", "ته" },
  { "ﳥ", "ﳥ", "ﳥ", "ثم", "ثم" },
  { "ﳦ", "ﳦ", "ﳦ", "ثه", "ثه" },
  { "ﳧ", "ﳧ", "ﳧ", "سم", "سم" },
  { "ﳨ", "ﳨ", "ﳨ", "سه", "سه" },
  { "ﳩ", "ﳩ", "ﳩ", "شم", "شم" },
  { "ﳪ", "ﳪ", "ﳪ", "شه", "شه" },
  { "ﳫ", "ﳫ", "ﳫ", "كل", "كل" },
  { "ﳬ", "ﳬ", "ﳬ", "كم", "كم" },
  { "ﳭ", "ﳭ", "ﳭ", "لم", "لم" },
  { "ﳮ", "ﳮ", "ﳮ", "نم", "نم" },
  { "ﳯ", "ﳯ", "ﳯ", "نه", "نه" },
  { "ﳰ", "ﳰ", "ﳰ", "يم", "يم" },
  { "ﳱ", "ﳱ", "ﳱ", "يه", "يه" },
  { "ﳲ", "ﳲ", "ﳲ", "ـَّ", "ـَّ" },
  { "ﳳ", "ﳳ", "ﳳ", "ـُّ", "ـُّ" },
  { "ﳴ", "ﳴ", "ﳴ", "ـِّ", "ـِّ" },
  { "ﳵ", "ﳵ", "ﳵ", "طى", "طى" },
  { "ﳶ", "ﳶ", "ﳶ", "طي", "طي" },
  { "ﳷ", "ﳷ", "ﳷ", "عى", "عى" },
  { "ﳸ", "ﳸ", "ﳸ", "عي", "عي" },
  { "ﳹ", "ﳹ", "ﳹ", "غى", "غى" },
  { "ﳺ", "ﳺ", "ﳺ", "غي", "غي" },
  { "ﳻ", "ﳻ", "ﳻ", "سى", "سى" },
  { "ﳼ", "ﳼ", "ﳼ", "سي", "سي" },
  { "ﳽ", "ﳽ", "ﳽ", "شى", "شى" },
  { "ﳾ", "ﳾ", "ﳾ", "شي", "شي" },
  { "ﳿ", "ﳿ", "ﳿ", "حى", "حى" },
  { "ﴀ", "ﴀ", "ﴀ", "حي", "حي" },
  { "ﴁ", "ﴁ", "ﴁ", "جى", "جى" },
  { "ﴂ", "ﴂ", "ﴂ", "جي", "جي" },
  { "ﴃ", "ﴃ", "ﴃ", "خى", "خى" },
  { "ﴄ", "ﴄ", "ﴄ", "خي", "خي" },
  { "ﴅ", "ﴅ", "ﴅ", "صى", "صى" },
  { "ﴆ", "ﴆ", "ﴆ", "صي", "صي" },
  { "ﴇ", "ﴇ", "ﴇ", "ضى", "ضى" },
  { "ﴈ", "ﴈ", "ﴈ", "ضي", "ضي" },
  { "ﴉ", "ﴉ", "ﴉ", "شج", "شج" },
  { "ﴊ", "ﴊ", "ﴊ", "شح", "شح" },
  { "ﴋ", "ﴋ", "ﴋ", "شخ", "شخ" },
  { "ﴌ", "ﴌ", "ﴌ", "شم", "شم" },
  { "ﴍ", "ﴍ", "ﴍ", "شر", "شر" },
  { "ﴎ", "ﴎ", "ﴎ", "سر", "سر" },
  { "ﴏ", "ﴏ", "ﴏ", "صر", "صر" },
  { "ﴐ", "ﴐ", "ﴐ", "ضر", "ضر" },
  { "ﴑ", "ﴑ", "ﴑ", "طى", "طى" },
  { "ﴒ", "ﴒ", "ﴒ", "طي", "طي" },
  { "ﴓ", "ﴓ", "ﴓ", "عى", "عى" },
  { "ﴔ", "ﴔ", "ﴔ", "عي", "عي" },
  { "ﴕ", "ﴕ", "ﴕ", "غى", "غى" },
  { "ﴖ", "ﴖ", "ﴖ", "غي", "غي" },
  { "ﴗ", "ﴗ", "ﴗ", "سى", "سى" },
  { "ﴘ", "ﴘ", "ﴘ", "سي", "سي" },
  { "ﴙ", "ﴙ", "ﴙ", "شى", "شى" },
  { "ﴚ", "ﴚ", "ﴚ", "شي", "شي" },
  { "ﴛ", "ﴛ", "ﴛ", "حى", "حى" },
  { "ﴜ", "ﴜ", "ﴜ", "حي", "حي" },
  { "ﴝ", "ﴝ", "ﴝ", "جى", "جى" },
  { "ﴞ", "ﴞ", "ﴞ", "جي", "جي" },
  { "ﴟ", "ﴟ", "ﴟ", "خى", "خى" },
  { "ﴠ", "ﴠ", "ﴠ", "خي", "خي" },
  { "ﴡ", "ﴡ", "ﴡ", "صى", "صى" },
  { "ﴢ", "ﴢ", "ﴢ", "صي", "صي" },
  { "ﴣ", "ﴣ", "ﴣ", "ضى", "ضى" },
  { "ﴤ", "ﴤ", "ﴤ", "ضي", "ضي" },
  { "ﴥ", "ﴥ", "ﴥ", "شج", "شج" },
  { "ﴦ", "ﴦ", "ﴦ", "شح", "شح" },
  { "ﴧ", "ﴧ", "ﴧ", "شخ", "شخ" },
  { "ﴨ", "ﴨ", "ﴨ", "شم", "شم" },
  { "ﴩ", "ﴩ", "ﴩ", "شر", "شر" },
  { "ﴪ", "ﴪ", "ﴪ", "سر", "سر" },
  { "ﴫ", "ﴫ", "ﴫ", "صر", "صر" },
  { "ﴬ", "ﴬ", "ﴬ", "ضر", "ضر" },
  { "ﴭ", "ﴭ", "ﴭ", "شج", "شج" },
  { "ﴮ", "ﴮ", "ﴮ", "شح", "شح" },
  { "ﴯ", "ﴯ", "ﴯ", "شخ", "شخ" },
  { "ﴰ", "ﴰ", "ﴰ", "شم", "شم" },
  { "ﴱ", "ﴱ", "ﴱ", "سه", "سه" },
  { "ﴲ", "ﴲ", "ﴲ", "شه", "شه" },
  { "ﴳ", "ﴳ", "ﴳ", "طم", "طم" },
  { "ﴴ", "ﴴ", "ﴴ", "سج", "سج" },
  { "ﴵ", "ﴵ", "ﴵ", "سح", "سح" },
  { "ﴶ", "ﴶ", "ﴶ", "سخ", "سخ" },
  { "ﴷ", "ﴷ", "ﴷ", "شج", "شج" },
  { "ﴸ", "ﴸ", "ﴸ", "شح", "شح" },
  { "ﴹ", "ﴹ", "ﴹ", "شخ", "شخ" },
  { "ﴺ", "ﴺ", "ﴺ", "طم", "طم" },
  { "ﴻ", "ﴻ", "ﴻ", "ظم", "ظم" },
  { "ﴼ", "ﴼ", "ﴼ", "اً", "اً" },
  { "ﴽ", "ﴽ", "ﴽ", "اً", "اً" },
  { "ﵐ", "ﵐ", "ﵐ", "تجم", "تجم" },
  { "ﵑ", "ﵑ", "ﵑ", "تحج", "تحج" },
  { "ﵒ", "ﵒ", "ﵒ", "تحج", "تحج" },
  { "ﵓ", "ﵓ", "ﵓ", "تحم", "تحم" },
  { "ﵔ", "ﵔ", "ﵔ", "تخم", "تخم" },
  { "ﵕ", "ﵕ", "ﵕ", "تمج", "تمج" },
  { "ﵖ", "ﵖ", "ﵖ", "تمح", "تمح" },
  { "ﵗ", "ﵗ", "ﵗ", "تمخ", "تمخ" },
  { "ﵘ", "ﵘ", "ﵘ", "جمح", "جمح" },
  { "ﵙ", "ﵙ", "ﵙ", "جمح", "جمح" },
  { "ﵚ", "ﵚ", "ﵚ", "حمي", "حمي" },
  { "ﵛ", "ﵛ", "ﵛ", "حمى", "حمى" },
  { "ﵜ", "ﵜ", "ﵜ", "سحج", "سحج" },
  { "ﵝ", "ﵝ", "ﵝ", "سجح", "سجح" },
  { "ﵞ", "ﵞ", "ﵞ", "سجى", "سجى" },
  { "ﵟ", "ﵟ", "ﵟ", "سمح", "سمح" },
  { "ﵠ", "ﵠ", "ﵠ", "سمح", "سمح" },
  { "ﵡ", "ﵡ", "ﵡ", "سمج", "سمج" },
  { "ﵢ", "ﵢ", "ﵢ", "سمم", "سمم" },
  { "ﵣ", "ﵣ", "ﵣ", "سمم", "سمم" },
  { "ﵤ", "ﵤ", "ﵤ", "صحح", "صحح" },
  { "ﵥ", "ﵥ", "ﵥ", "صحح", "صحح" },
  { "ﵦ", "ﵦ", "ﵦ", "صمم", "صمم" },
  { "ﵧ", "ﵧ", "ﵧ", "شحم", "شحم" },
  { "ﵨ", "ﵨ", "ﵨ", "شحم", "شحم" },
  { "ﵩ", "ﵩ", "ﵩ", "شجي", "شجي" },
  { "ﵪ", "ﵪ", "ﵪ", "شمخ", "شمخ" },
  { "ﵫ", "ﵫ", "ﵫ", "شمخ", "شمخ" },
  { "ﵬ", "ﵬ", "ﵬ", "شمم", "شمم" },
  { "ﵭ", "ﵭ", "ﵭ", "شمم", "شمم" },
  { "ﵮ", "ﵮ", "ﵮ", "ضحى", "ضحى" },
  { "ﵯ", "ﵯ", "ﵯ", "ضخم", "ضخم" },
  { "ﵰ", "ﵰ", "ﵰ", "ضخم", "ضخم" },
  { "ﵱ", "ﵱ", "ﵱ", "طمح", "طمح" },
  { "ﵲ", "ﵲ", "ﵲ", "طمح", "طمح" },
  { "ﵳ", "ﵳ", "ﵳ", "طمم", "طمم" },
  { "ﵴ", "ﵴ", "ﵴ", "طمي", "طمي" },
  { "ﵵ", "ﵵ", "ﵵ", "عجم", "عجم" },
  { "ﵶ", "ﵶ", "ﵶ", "عمم", "عمم" },
  { "ﵷ", "ﵷ", "ﵷ", "عمم", "عمم" },
  { "ﵸ", "ﵸ", "ﵸ", "عمى", "عمى" },
  { "ﵹ", "ﵹ", "ﵹ", "غمم", "غمم" },
  { "ﵺ", "ﵺ", "ﵺ", "غمي", "غمي" },
  { "ﵻ", "ﵻ", "ﵻ", "غمى", "غمى" },
  { "ﵼ", "ﵼ", "ﵼ", "فخم", "فخم" },
  { "ﵽ", "ﵽ", "ﵽ", "فخم", "فخم" },
  { "ﵾ", "ﵾ", "ﵾ", "قمح", "قمح" },
  { "ﵿ", "ﵿ", "ﵿ", "قمم", "قمم" },
  { "ﶀ", "ﶀ", "ﶀ", "لحم", "لحم" },
  { "ﶁ", "ﶁ", "ﶁ", "لحي", "لحي" },
  { "ﶂ", "ﶂ", "ﶂ", "لحى", "لحى" },
  { "ﶃ", "ﶃ", "ﶃ", "لجج", "لجج" },
  { "ﶄ", "ﶄ", "ﶄ", "لجج", "لجج" },
  { "ﶅ", "ﶅ", "ﶅ", "لخم", "لخم" },
  { "ﶆ", "ﶆ", "ﶆ", "لخم", "لخم" },
  { "ﶇ", "ﶇ", "ﶇ", "لمح", "لمح" },
  { "ﶈ", "ﶈ", "ﶈ", "لمح", "لمح" },
  { "ﶉ", "ﶉ", "ﶉ", "محج", "محج" },
  { "ﶊ", "ﶊ", "ﶊ", "محم", "محم" },
  { "ﶋ", "ﶋ", "ﶋ", "محي", "محي" },
  { "ﶌ", "ﶌ", "ﶌ", "مجح", "مجح" },
  { "ﶍ", "ﶍ", "ﶍ", "مجم", "مجم" },
  { "ﶎ", "ﶎ", "ﶎ", "مخج", "مخج" },
  { "ﶏ", "ﶏ", "ﶏ", "مخم", "مخم" },
  { "ﶒ", "ﶒ", "ﶒ", "مجخ", "مجخ" },
  { "ﶓ", "ﶓ", "ﶓ", "همج", "همج" },
  { "ﶔ", "ﶔ", "ﶔ", "همم", "همم" },
  { "ﶕ", "ﶕ", "ﶕ", "نحم", "نحم" },
  { "ﶖ", "ﶖ", "ﶖ", "نحى", "نحى" },
  { "ﶗ", "ﶗ", "ﶗ", "نجم", "نجم" },
  { "ﶘ", "ﶘ", "ﶘ", "نجم", "نجم" },
  { "ﶙ", "ﶙ", "ﶙ", "نجى", "نجى" },
  { "ﶚ", "ﶚ", "ﶚ", "نمي", "نمي" },
  { "ﶛ", "ﶛ", "ﶛ", "نمى", "نمى" },
  { "ﶜ", "ﶜ", "ﶜ", "يمم", "يمم" },
  { "ﶝ", "ﶝ", "ﶝ", "يمم", "يمم" },
  { "ﶞ", "ﶞ", "ﶞ", "بخي", "بخي" },
  { "ﶟ", "ﶟ", "ﶟ", "تجي", "تجي" },
  { "ﶠ", "ﶠ", "ﶠ", "تجى", "تجى" },
  { "ﶡ", "ﶡ", "ﶡ", "تخي", "تخي" },
  { "ﶢ", "ﶢ", "ﶢ", "تخى", "تخى" },
  { "ﶣ", "ﶣ", "ﶣ", "تمي", "تمي" },
  { "ﶤ", "ﶤ", "ﶤ", "تمى", "تمى" },
  { "ﶥ", "ﶥ", "ﶥ", "جمي", "جمي" },
  { "ﶦ", "ﶦ", "ﶦ", "جحى", "جحى" },
  { "ﶧ", "ﶧ", "ﶧ", "جمى", "جمى" },
  { "ﶨ", "ﶨ", "ﶨ", "سخى", "سخى" },
  { "ﶩ", "ﶩ", "ﶩ", "صحي", "صحي" },
  { "ﶪ", "ﶪ", "ﶪ", "شحي", "شحي" },
  { "ﶫ", "ﶫ", "ﶫ", "ضحي", "ضحي" },
  { "ﶬ", "ﶬ", "ﶬ", "لجي", "لجي" },
  { "ﶭ", "ﶭ", "ﶭ", "لمي", "لمي" },
  { "ﶮ", "ﶮ", "ﶮ", "يحي", "يحي" },
  { "ﶯ", "ﶯ", "ﶯ", "يجي", "يجي" },
  { "ﶰ", "ﶰ", "ﶰ", "يمي", "يمي" },
  { "ﶱ", "ﶱ", "ﶱ", "ممي", "ممي" },
  { "ﶲ", "ﶲ", "ﶲ", "قمي", "قمي" },
  { "ﶳ", "ﶳ", "ﶳ", "نحي", "نحي" },
  { "ﶴ", "ﶴ", "ﶴ", "قمح", "قمح" },
  { "ﶵ", "ﶵ", "ﶵ", "لحم", "لحم" },
  { "ﶶ", "ﶶ", "ﶶ", "عمي", "عمي" },
  { "ﶷ", "ﶷ", "ﶷ", "كمي", "كمي" },
  { "ﶸ", "ﶸ", "ﶸ", "نجح", "نجح" },
  { "ﶹ", "ﶹ", "ﶹ", "مخي", "مخي" },
  { "ﶺ", "ﶺ", "ﶺ", "لجم", "لجم" },
  { "ﶻ", "ﶻ", "ﶻ", "كمم", "كمم" },
  { "ﶼ", "ﶼ", "ﶼ", "لجم", "لجم" },
  { "ﶽ", "ﶽ", "ﶽ", "نجح", "نجح" },
  { "ﶾ", "ﶾ", "ﶾ", "جحي", "جحي" },
  { "ﶿ", "ﶿ", "ﶿ", "حجي", "حجي" },
  { "ﷀ", "ﷀ", "ﷀ", "مجي", "مجي" },
  { "ﷁ", "ﷁ", "ﷁ", "فمي", "فمي" },
  { "ﷂ", "ﷂ", "ﷂ", "بحي", "بحي" },
  { "ﷃ", "ﷃ", "ﷃ", "كمم", "كمم" },
  { "ﷄ", "ﷄ", "ﷄ", "عجم", "عجم" },
  { "ﷅ", "ﷅ", "ﷅ", "صمم", "صمم" },
  { "ﷆ", "ﷆ", "ﷆ", "سخي", "سخي" },
  { "ﷇ", "ﷇ", "ﷇ", "نجي", "نجي" },
  { "ﷰ", "ﷰ", "ﷰ", "صلے", "صلے" },
  { "ﷱ", "ﷱ", "ﷱ", "قلے", "قلے" },
  { "ﷲ", "ﷲ", "ﷲ", "الله", "الله" },
  { "ﷳ", "ﷳ", "ﷳ", "اكبر", "اكبر" },
  { "ﷴ", "ﷴ", "ﷴ", "محمد", "محمد" },
  { "ﷵ", "ﷵ", "ﷵ", "صلعم", "صلعم" },
  { "ﷶ", "ﷶ", "ﷶ", "رسول", "رسول" },
  { "ﷷ", "ﷷ", "ﷷ", "عليه", "عليه" },
  { "ﷸ", "ﷸ", "ﷸ", "وسلم", "وسلم" },
  { "ﷹ", "ﷹ", "ﷹ", "صلى", "صلى" },
  { "ﷺ", "ﷺ", "ﷺ", "صلى الله عليه وسلم", "صلى الله عليه وسلم" },
  { "ﷻ", "ﷻ", "ﷻ", "جل جلاله", "جل جلاله" },
  { "﷼", "﷼", "﷼", "ریال", "ریال" },
  { "︐", "︐", "︐", ",", "," },
  { "︑", "︑", "︑", "、", "、" },
  { "︒", "︒", "︒", "。", "。" },
  { "︓", "︓", "︓", ":", ":" },
  { "︔", "︔", "︔", ";", ";" },
  { "︕", "︕", "︕", "!", "!" },
  { "︖", "︖", "︖", "?", "?" },
  { "︗", "︗", "︗", "〖", "〖" },
  { "︘", "︘", "︘", "〗", "〗" },
  { "︙", "︙", "︙", "...", "..." },
  { "︰", "︰", "︰", "..", ".." },
  { "︱", "︱", "︱", "—", "—" },
  { "︲", "︲", "︲", "–", "–" },
  { "︳", "︳", "︳", "_", "_" },
  { "︴", "︴", "︴", "_", "_" },
  { "︵", "︵", "︵", "(", "(" },
  { "︶", "︶", "︶", ")", ")" },
  { "︷", "︷", "︷", "{", "{" },
  { "︸", "︸", "︸", "}", "}" },
  { "︹", "︹", "︹", "〔", "〔" },
  { "︺", "︺", "︺", "〕", "〕" },
  { "︻", "︻", "︻", "【", "【" },
  { "︼", "︼", "︼", "】", "】" },
  { "︽", "︽", "︽", "《", "《" },
  { "︾", "︾", "︾", "》", "》" },
  { "︿", "︿", "︿", "〈", "〈" },
  { "﹀", "﹀", "﹀", "〉", "〉" },
  { "﹁", "﹁", "﹁", "「", "「" },
  { "﹂", "﹂", "﹂", "」", "」" },
  { "﹃", "﹃", "﹃", "『", "『" },
  { "﹄", "﹄", "﹄", "』", "』" },
  { "﹇", "﹇", "﹇", "[", "[" },
  { "﹈", "﹈", "﹈", "]", "]" },
  { "﹉", "﹉", "﹉", " ̅", " ̅" },
  { "﹊", "﹊", "﹊", " ̅", " ̅" },
  { "﹋", "﹋", "﹋", " ̅", " ̅" },
  { "﹌", "﹌", "﹌", " ̅", " ̅" },
  { "﹍", "﹍", "﹍", "_", "_" },
  { "﹎", "﹎", "﹎", "_", "_" },
  { "﹏", "﹏", "﹏", "_", "_" },
  { "﹐", "﹐", "﹐", ",", "," },
  { "﹑", "﹑", "﹑", "、", "、" },
  { "﹒", "﹒", "﹒", ".", "." },
  { "﹔", "﹔", "﹔", ";", ";" },
  { "﹕", "﹕", "﹕", ":", ":" },
  { "﹖", "﹖", "﹖", "?", "?" },
  { "﹗", "﹗", "﹗", "!", "!" },
  { "﹘", "﹘", "﹘", "—", "—" },
  { "﹙", "﹙", "﹙", "(", "(" },
  { "﹚", "﹚", "﹚", ")", ")" },
  { "﹛", "﹛", "﹛", "{", "{" },
  { "﹜", "﹜", "﹜", "}", "}" },
  { "﹝", "﹝", "﹝", "〔", "〔" },
  { "﹞", "﹞", "﹞", "〕", "〕" },
  { "﹟", "﹟", "﹟", "#", "#" },
  { "﹠", "﹠", "﹠", "&", "&" },
  { "﹡", "﹡", "﹡", "*", "*" },
  { "﹢", "﹢", "﹢", "+", "+" },
  { "﹣", "﹣", "﹣", "-", "-" },
  { "﹤", "﹤", "﹤", "<", "<" },
  { "﹥", "﹥", "﹥", ">", ">" },
  { "﹦", "﹦", "﹦", "=", "=" },
  { "﹨", "﹨", "﹨", "\\", "\\" },
  { "﹩", "﹩", "﹩", "$", "$" },
  { "﹪", "﹪", "﹪", "%", "%" },
  { "﹫", "﹫", "﹫", "@", "@" },
  { "ﹰ", "ﹰ", "ﹰ", " ً", " ً" },
  { "ﹱ", "ﹱ", "ﹱ", "ـً", "ـً" },
  { "ﹲ", "ﹲ", "ﹲ", " ٌ", " ٌ" },
  { "ﹴ", "ﹴ", "ﹴ", " ٍ", " ٍ" },
  { "ﹶ", "ﹶ", "ﹶ", " َ", " َ" },
  { "ﹷ", "ﹷ", "ﹷ", "ـَ", "ـَ" },
  { "ﹸ", "ﹸ", "ﹸ", " ُ", " ُ" },
  { "ﹹ", "ﹹ", "ﹹ", "ـُ", "ـُ" },
  { "ﹺ", "ﹺ", "ﹺ", " ِ", " ِ" },
  { "ﹻ", "ﹻ", "ﹻ", "ـِ", "ـِ" },
  { "ﹼ", "ﹼ", "ﹼ", " ّ", " ّ" },
  { "ﹽ", "ﹽ", "ﹽ", "ـّ", "ـّ" },
  { "ﹾ", "ﹾ", "ﹾ", " ْ", " ْ" },
  { "ﹿ", "ﹿ", "ﹿ", "ـْ", "ـْ" },
  { "ﺀ", "ﺀ", "ﺀ", "ء", "ء" },
  { "ﺁ", "ﺁ", "ﺁ", "آ", "آ" },
  { "ﺂ", "ﺂ", "ﺂ", "آ", "آ" },
  { "ﺃ", "ﺃ", "ﺃ", "أ", "أ" },
  { "ﺄ", "ﺄ", "ﺄ", "أ", "أ" },
  { "ﺅ", "ﺅ", "ﺅ", "ؤ", "ؤ" },
  { "ﺆ", "ﺆ", "ﺆ", "ؤ", "ؤ" },
  { "ﺇ", "ﺇ", "ﺇ", "إ", "إ" },
  { "ﺈ", "ﺈ", "ﺈ", "إ", "إ" },
  { "ﺉ", "ﺉ", "ﺉ", "ئ", "ئ" },
  { "ﺊ", "ﺊ", "ﺊ", "ئ", "ئ" },
  { "ﺋ", "ﺋ", "ﺋ", "ئ", "ئ" },
  { "ﺌ", "ﺌ", "ﺌ", "ئ", "ئ" },
  { "ﺍ", "ﺍ", "ﺍ", "ا", "ا" },
  { "ﺎ", "ﺎ", "ﺎ", "ا", "ا" },
  { "ﺏ", "ﺏ", "ﺏ", "ب", "ب" },
  { "ﺐ", "ﺐ", "ﺐ", "ب", "ب" },
  { "ﺑ", "ﺑ", "ﺑ", "ب", "ب" },
  { "ﺒ", "ﺒ", "ﺒ", "ب", "ب" },
  { "ﺓ", "ﺓ", "ﺓ", "ة", "ة" },
  { "ﺔ", "ﺔ", "ﺔ", "ة", "ة" },
  { "ﺕ", "ﺕ", "ﺕ", "ت", "ت" },
  { "ﺖ", "ﺖ", "ﺖ", "ت", "ت" },
  { "ﺗ", "ﺗ", "ﺗ", "ت", "ت" },
  { "ﺘ", "ﺘ", "ﺘ", "ت", "ت" },
  { "ﺙ", "ﺙ", "ﺙ", "ث", "ث" },
  { "ﺚ", "ﺚ", "ﺚ", "ث", "ث" },
  { "ﺛ", "ﺛ", "ﺛ", "ث", "ث" },
  { "ﺜ", "ﺜ", "ﺜ", "ث", "ث" },
  { "ﺝ", "ﺝ", "ﺝ", "ج", "ج" },
  { "ﺞ", "ﺞ", "ﺞ", "ج", "ج" },
  { "ﺟ", "ﺟ", "ﺟ", "ج", "ج" },
  { "ﺠ", "ﺠ", "ﺠ", "ج", "ج" },
  { "ﺡ", "ﺡ", "ﺡ", "ح", "ح" },
  { "ﺢ", "ﺢ", "ﺢ", "ح", "ح" },
  { "ﺣ", "ﺣ", "ﺣ", "ح", "ح" },
  { "ﺤ", "ﺤ", "ﺤ", "ح", "ح" },
  { "ﺥ", "ﺥ", "ﺥ", "خ", "خ" },
  { "ﺦ", "ﺦ", "ﺦ", "خ", "خ" },
  { "ﺧ", "ﺧ", "ﺧ", "خ", "خ" },
  { "ﺨ", "ﺨ", "ﺨ", "خ", "خ" },
  { "ﺩ", "ﺩ", "ﺩ", "د", "د" },
  { "ﺪ", "ﺪ", "ﺪ", "د", "د" },
  { "ﺫ", "ﺫ", "ﺫ", "ذ", "ذ" },
  { "ﺬ", "ﺬ", "ﺬ", "ذ", "ذ" },
  { "ﺭ", "ﺭ", "ﺭ", "ر", "ر" },
  { "ﺮ", "ﺮ", "ﺮ", "ر", "ر" },
  { "ﺯ", "ﺯ", "ﺯ", "ز", "ز" },
  { "ﺰ", "ﺰ", "ﺰ", "ز", "ز" },
  { "ﺱ", "ﺱ", "ﺱ", "س", "س" },
  { "ﺲ", "ﺲ", "ﺲ", "س", "س" },
  { "ﺳ", "ﺳ", "ﺳ", "س", "س" },
  { "ﺴ", "ﺴ", "ﺴ", "س", "س" },
  { "ﺵ", "ﺵ", "ﺵ", "ش", "ش" },
  { "ﺶ", "ﺶ", "ﺶ", "ش", "ش" },
  { "ﺷ", "ﺷ", "ﺷ", "ش", "ش" },
  { "ﺸ", "ﺸ", "ﺸ", "ش", "ش" },
  { "ﺹ", "ﺹ", "ﺹ", "ص", "ص" },
  { "ﺺ", "ﺺ", "ﺺ", "ص", "ص" },
  { "ﺻ", "ﺻ", "ﺻ", "ص", "ص" },
  { "ﺼ", "ﺼ", "ﺼ", "ص", "ص" },
  { "ﺽ", "ﺽ", "ﺽ", "ض", "ض" },
  { "ﺾ", "ﺾ", "ﺾ", "ض", "ض" },
  { "ﺿ", "ﺿ", "ﺿ", "ض", "ض" },
  { "ﻀ", "ﻀ", "ﻀ", "ض", "ض" },
  { "ﻁ", "ﻁ", "ﻁ", "ط", "ط" },
  { "ﻂ", "ﻂ", "ﻂ", "ط", "ط" },
  { "ﻃ", "ﻃ", "ﻃ", "ط", "ط" },
  { "ﻄ", "ﻄ", "ﻄ", "ط", "ط" },
  { "ﻅ", "ﻅ", "ﻅ", "ظ", "ظ" },
  { "ﻆ", "ﻆ", "ﻆ", "ظ", "ظ" },
  { "ﻇ", "ﻇ", "ﻇ", "ظ", "ظ" },
  { "ﻈ", "ﻈ", "ﻈ", "ظ", "ظ" },
  { "ﻉ", "ﻉ", "ﻉ", "ع", "ع" },
  { "ﻊ", "ﻊ", "ﻊ", "ع", "ع" },
  { "ﻋ", "ﻋ", "ﻋ", "ع", "ع" },
  { "ﻌ", "ﻌ", "ﻌ", "ع", "ع" },
  { "ﻍ", "ﻍ", "ﻍ", "غ", "غ" },
  { "ﻎ", "ﻎ", "ﻎ", "غ", "غ" },
  { "ﻏ", "ﻏ", "ﻏ", "غ", "غ" },
  { "ﻐ", "ﻐ", "ﻐ", "غ", "غ" },
  { "ﻑ", "ﻑ", "ﻑ", "ف", "ف" },
  { "ﻒ", "ﻒ", "ﻒ", "ف", "ف" },
  { "ﻓ", "ﻓ", "ﻓ", "ف", "ف" },
  { "ﻔ", "ﻔ", "ﻔ", "ف", "ف" },
  { "ﻕ", "ﻕ", "ﻕ", "ق", "ق" },
  { "ﻖ", "ﻖ", "ﻖ", "ق", "ق" },
  { "ﻗ", "ﻗ", "ﻗ", "ق", "ق" },
  { "ﻘ", "ﻘ", "ﻘ", "ق", "ق" },
  { "ﻙ", "ﻙ", "ﻙ", "ك", "ك" },
  { "ﻚ", "ﻚ", "ﻚ", "ك", "ك" },
  { "ﻛ", "ﻛ", "ﻛ", "ك", "ك" },
  { "ﻜ", "ﻜ", "ﻜ", "ك", "ك" },
  { "ﻝ", "ﻝ", "ﻝ", "ل", "ل" },
  { "ﻞ", "ﻞ", "ﻞ", "ل", "ل" },
  { "ﻟ", "ﻟ", "ﻟ", "ل", "ل" },
  { "ﻠ", "ﻠ", "ﻠ", "ل", "ل" },
  { "ﻡ", "ﻡ", "ﻡ", "م", "م" },
  { "ﻢ", "ﻢ", "ﻢ", "م", "م" },
  { "ﻣ", "ﻣ", "ﻣ", "م", "م" },
  { "ﻤ", "ﻤ", "ﻤ", "م", "م" },
  { "ﻥ", "ﻥ", "ﻥ", "ن", "ن" },
  { "ﻦ", "ﻦ", "ﻦ", "ن", "ن" },
  { "ﻧ", "ﻧ", "ﻧ", "ن", "ن" },
  { "ﻨ", "ﻨ", "ﻨ", "ن", "ن" },
  { "ﻩ", "ﻩ", "ﻩ", "ه", "ه" },
  { "ﻪ", "ﻪ", "ﻪ", "ه", "ه" },
  { "ﻫ", "ﻫ", "ﻫ", "ه", "ه" },
  { "ﻬ", "ﻬ", "ﻬ", "ه", "ه" },
  { "ﻭ", "ﻭ", "ﻭ", "و", "و" },
  { "ﻮ", "ﻮ", "ﻮ", "و", "و" },
  { "ﻯ", "ﻯ", "ﻯ", "ى", "ى" },
  { "ﻰ", "ﻰ", "ﻰ", "ى", "ى" },
  { "ﻱ", "ﻱ", "ﻱ", "ي", "ي" },
  { "ﻲ", "ﻲ", "ﻲ", "ي", "ي" },
  { "ﻳ", "ﻳ", "ﻳ", "ي", "ي" },
  { "ﻴ", "ﻴ", "ﻴ", "ي", "ي" },
  { "ﻵ", "ﻵ", "ﻵ", "لآ", "لآ" },
  { "ﻶ", "ﻶ", "ﻶ", "لآ", "لآ" },
  { "ﻷ", "ﻷ", "ﻷ", "لأ", "لأ" },
  { "ﻸ", "ﻸ", "ﻸ", "لأ", "لأ" },
  { "ﻹ", "ﻹ", "ﻹ", "لإ", "لإ" },
  { "ﻺ", "ﻺ", "ﻺ", "لإ", "لإ" },
  { "ﻻ", "ﻻ", "ﻻ", "لا", "لا" },
  { "ﻼ", "ﻼ", "ﻼ", "لا", "لا" },
  { "！", "！", "！", "!", "!" },
  { "＂", "＂", "＂", "\"", "\"" },
  { "＃", "＃", "＃", "#", "#" },
  { "＄", "＄", "＄", "$", "$" },
  { "％", "％", "％", "%", "%" },
  { "＆", "＆", "＆", "&", "&" },
  { "＇", "＇", "＇", "'", "'" },
  { "（", "（", "（", "(", "(" },
  { "）", "）", "）", ")", ")" },
  { "＊", "＊", "＊", "*", "*" },
  { "＋", "＋", "＋", "+", "+" },
  { "，", "，", "，", ",", "," },
  { "－", "－", "－", "-", "-" },
  { "．", "．", "．", ".", "." },
  { "／", "／", "／", "/", "/" },
  { "０", "０", "０", "0", "0" },
  { "１", "１", "１", "1", "1" },
  { "２", "２", "２", "2", "2" },
  { "３", "３", "３", "3", "3" },
  { "４", "４", "４", "4", "4" },
  { "５", "５", "５", "5", "5" },
  { "６", "６", "６", "6", "6" },
  { "７", "７", "７", "7", "7" },
  { "８", "８", "８", "8", "8" },
  { "９", "９", "９", "9", "9" },
  { "：", "：", "：", ":", ":" },
  { "；", "；", "；", ";", ";" },
  { "＜", "＜", "＜", "<", "<" },
  { "＝", "＝", "＝", "=", "=" },
  { "＞", "＞", "＞", ">", ">" },
  { "？", "？", "？", "?", "?" },
  { "＠", "＠", "＠", "@", "@" },
  { "Ａ", "Ａ", "Ａ", "A", "A" },
  { "Ｂ", "Ｂ", "Ｂ", "B", "B" },
  { "Ｃ", "Ｃ", "Ｃ", "C", "C" },
  { "Ｄ", "Ｄ", "Ｄ", "D", "D" },
  { "Ｅ", "Ｅ", "Ｅ", "E", "E" },
  { "Ｆ", "Ｆ", "Ｆ", "F", "F" },
  { "Ｇ", "Ｇ", "Ｇ", "G", "G" },
  { "Ｈ", "Ｈ", "Ｈ", "H", "H" },
  { "Ｉ", "Ｉ", "Ｉ", "I", "I" },
  { "Ｊ", "Ｊ", "Ｊ", "J", "J" },
  { "Ｋ", "Ｋ", "Ｋ", "K", "K" },
  { "Ｌ", "Ｌ", "Ｌ", "L", "L" },
  { "Ｍ", "Ｍ", "Ｍ", "M", "M" },
  { "Ｎ", "Ｎ", "Ｎ", "N", "N" },
  { "Ｏ", "Ｏ", "Ｏ", "O", "O" },
  { "Ｐ", "Ｐ", "Ｐ", "P", "P" },
  { "Ｑ", "Ｑ", "Ｑ", "Q", "Q" },
  { "Ｒ", "Ｒ", "Ｒ", "R", "R" },
  { "Ｓ", "Ｓ", "Ｓ", "S", "S" },
  { "Ｔ", "Ｔ", "Ｔ", "T", "T" },
  { "Ｕ", "Ｕ", "Ｕ", "U", "U" },
  { "Ｖ", "Ｖ", "Ｖ", "V", "V" },
  { "Ｗ", "Ｗ", "Ｗ", "W", "W" },
  { "Ｘ", "Ｘ", "Ｘ", "X", "X" },
  { "Ｙ", "Ｙ", "Ｙ", "Y", "Y" },
  { "Ｚ", "Ｚ", "Ｚ", "Z", "Z" },
  { "［", "［", "［", "[", "[" },
  { "＼", "＼", "＼", "\\", "\\" },
  { "］", "］", "］", "]", "]" },
  { "＾", "＾", "＾", "^", "^" },
  { "＿", "＿", "＿", "_", "_" },
  { "｀", "｀", "｀", "`", "`" },
  { "ａ", "ａ", "ａ", "a", "a" },
  { "ｂ", "ｂ", "ｂ", "b", "b" },
  { "ｃ", "ｃ", "ｃ", "c", "c" },
  { "ｄ", "ｄ", "ｄ", "d", "d" },
  { "ｅ", "ｅ", "ｅ", "e", "e" },
  { "ｆ", "ｆ", "ｆ", "f", "f" },
  { "ｇ", "ｇ", "ｇ", "g", "g" },
  { "ｈ", "ｈ", "ｈ", "h", "h" },
  { "ｉ", "ｉ", "ｉ", "i", "i" },
  { "ｊ", "ｊ", "ｊ", "j", "j" },
  { "ｋ", "ｋ", "ｋ", "k", "k" },
  { "ｌ", "ｌ", "ｌ", "l", "l" },
  { "ｍ", "ｍ", "ｍ", "m", "m" },
  { "ｎ", "ｎ", "ｎ", "n", "n" },
  { "ｏ", "ｏ", "ｏ", "o", "o" },
  { "ｐ", "ｐ", "ｐ", "p", "p" },
  { "ｑ", "ｑ", "ｑ", "q", "q" },
  { "ｒ", "ｒ", "ｒ", "r", "r" },
  { "ｓ", "ｓ", "ｓ", "s", "s" },
  { "ｔ", "ｔ", "ｔ", "t", "t" },
  { "ｕ", "ｕ", "ｕ", "u", "u" },
  { "ｖ", "ｖ", "ｖ", "v", "v" },
  { "ｗ", "ｗ", "ｗ", "w", "w" },
  { "ｘ", "ｘ", "ｘ", "x", "x" },
  { "ｙ", "ｙ", "ｙ", "y", "y" },
  { "ｚ", "ｚ", "ｚ", "z", "z" },
  { "｛", "｛", "｛", "{", "{" },
  { "｜", "｜", "｜", "|", "|" },
  { "｝", "｝", "｝", "}", "}" },
  { "～", "～", "～", "~", "~" },
  { "｟", "｟", "｟", "⦅", "⦅" },
  { "｠", "｠", "｠", "⦆", "⦆" },
  { "｡", "｡", "｡", "。", "。" },
  { "｢", "｢", "｢", "「", "「" },
  { "｣", "｣", "｣", "」", "」" },
  { "､", "､", "､", "、", "、" },
  { "･", "･", "･", "・", "・" },
  { "ｦ", "ｦ", "ｦ", "ヲ", "ヲ" },
  { "ｧ", "ｧ", "ｧ", "ァ", "ァ" },
  { "ｨ", "ｨ", "ｨ", "ィ", "ィ" },
  { "ｩ", "ｩ", "ｩ", "ゥ", "ゥ" },
  { "ｪ", "ｪ", "ｪ", "ェ", "ェ" },
  { "ｫ", "ｫ", "ｫ", "ォ", "ォ" },
  { "ｬ", "ｬ", "ｬ", "ャ", "ャ" },
  { "ｭ", "ｭ", "ｭ", "ュ", "ュ" },
  { "ｮ", "ｮ", "ｮ", "ョ", "ョ" },
  { "ｯ", "ｯ", "ｯ", "ッ", "ッ" },
  { "ｰ", "ｰ", "ｰ", "ー", "ー" },
  { "ｱ", "ｱ", "ｱ", "ア", "ア" },
  { "ｲ", "ｲ", "ｲ", "イ", "イ" },
  { "ｳ", "ｳ", "ｳ", "ウ", "ウ" },
  { "ｴ", "ｴ", "ｴ", "エ", "エ" },
  { "ｵ", "ｵ", "ｵ", "オ", "オ" },
  { "ｶ", "ｶ", "ｶ", "カ", "カ" },
  { "ｷ", "ｷ", "ｷ", "キ", "キ" },
  { "ｸ", "ｸ", "ｸ", "ク", "ク" },
  { "ｹ", "ｹ", "ｹ", "ケ", "ケ" },
  { "ｺ", "ｺ", "ｺ", "コ", "コ" },
  { "ｻ", "ｻ", "ｻ", "サ", "サ" },
  { "ｼ", "ｼ", "ｼ", "シ", "シ" },
  { "ｽ", "ｽ", "ｽ", "ス", "ス" },
  { "ｾ", "ｾ", "ｾ", "セ", "セ" },
  { "ｿ", "ｿ", "ｿ", "ソ", "ソ" },
  { "ﾀ", "ﾀ", "ﾀ", "タ", "タ" },
  { "ﾁ", "ﾁ", "ﾁ", "チ", "チ" },
  { "ﾂ", "ﾂ", "ﾂ", "ツ", "ツ" },
  { "ﾃ", "ﾃ", "ﾃ", "テ", "テ" },
  { "ﾄ", "ﾄ", "ﾄ", "ト", "ト" },
  { "ﾅ", "ﾅ", "ﾅ", "ナ", "ナ" },
  { "ﾆ", "ﾆ", "ﾆ", "ニ", "ニ" },
  { "ﾇ", "ﾇ", "ﾇ", "ヌ", "ヌ" },
  { "ﾈ", "ﾈ", "ﾈ", "ネ", "ネ" },
  { "ﾉ", "ﾉ", "ﾉ", "ノ", "ノ" },
  { "ﾊ", "ﾊ", "ﾊ", "ハ", "ハ" },
  { "ﾋ", "ﾋ", "ﾋ", "ヒ", "ヒ" },
  { "ﾌ", "ﾌ", "ﾌ", "フ", "フ" },
  { "ﾍ", "ﾍ", "ﾍ", "ヘ", "ヘ" },
  { "ﾎ", "ﾎ", "ﾎ", "ホ", "ホ" },
  { "ﾏ", "ﾏ", "ﾏ", "マ", "マ" },
  { "ﾐ", "ﾐ", "ﾐ", "ミ", "ミ" },
  { "ﾑ", "ﾑ", "ﾑ", "ム", "ム" },
  { "ﾒ", "ﾒ", "ﾒ", "メ", "メ" },
  { "ﾓ", "ﾓ", "ﾓ", "モ", "モ" },
  { "ﾔ", "ﾔ", "ﾔ", "ヤ", "ヤ" },
  { "ﾕ", "ﾕ", "ﾕ", "ユ", "ユ" },
  { "ﾖ", "ﾖ", "ﾖ", "ヨ", "ヨ" },
  { "ﾗ", "ﾗ", "ﾗ", "ラ", "ラ" },
  { "ﾘ", "ﾘ", "ﾘ", "リ", "リ" },
  { "ﾙ", "ﾙ", "ﾙ", "ル", "ル" },
  { "ﾚ", "ﾚ", "ﾚ", "レ", "レ" },
  { "ﾛ", "ﾛ", "ﾛ", "ロ", "ロ" },
  { "ﾜ", "ﾜ", "ﾜ", "ワ", "ワ" },
  { "ﾝ", "ﾝ", "ﾝ", "ン", "ン" },
  { "ﾞ", "ﾞ", "ﾞ", "゙", "゙" },
  { "ﾟ", "ﾟ", "ﾟ", "゚", "゚" },
  { "ﾠ", "ﾠ", "ﾠ", "ᅠ", "ᅠ" },
  { "ﾡ", "ﾡ", "ﾡ", "ᄀ", "ᄀ" },
  { "ﾢ", "ﾢ", "ﾢ", "ᄁ", "ᄁ" },
  { "ﾣ", "ﾣ", "ﾣ", "ᆪ", "ᆪ" },
  { "ﾤ", "ﾤ", "ﾤ", "ᄂ", "ᄂ" },
  { "ﾥ", "ﾥ", "ﾥ", "ᆬ", "ᆬ" },
  { "ﾦ", "ﾦ", "ﾦ", "ᆭ", "ᆭ" },
  { "ﾧ", "ﾧ", "ﾧ", "ᄃ", "ᄃ" },
  { "ﾨ", "ﾨ", "ﾨ", "ᄄ", "ᄄ" },
  { "ﾩ", "ﾩ", "ﾩ", "ᄅ", "ᄅ" },
  { "ﾪ", "ﾪ", "ﾪ", "ᆰ", "ᆰ" },
  { "ﾫ", "ﾫ", "ﾫ", "ᆱ", "ᆱ" },
  { "ﾬ", "ﾬ", "ﾬ", "ᆲ", "ᆲ" },
  { "ﾭ", "ﾭ", "ﾭ", "ᆳ", "ᆳ" },
  { "ﾮ", "ﾮ", "ﾮ", "ᆴ", "ᆴ" },
  { "ﾯ", "ﾯ", "ﾯ", "ᆵ", "ᆵ" },
  { "ﾰ", "ﾰ", "ﾰ", "ᄚ", "ᄚ" },
  { "ﾱ", "ﾱ", "ﾱ", "ᄆ", "ᄆ" },
  { "ﾲ", "ﾲ", "ﾲ", "ᄇ", "ᄇ" },
  { "ﾳ", "ﾳ", "ﾳ", "ᄈ", "ᄈ" },
  { "ﾴ", "ﾴ", "ﾴ", "ᄡ", "ᄡ" },
  { "ﾵ", "ﾵ", "ﾵ", "ᄉ", "ᄉ" },
  { "ﾶ", "ﾶ", "ﾶ", "ᄊ", "ᄊ" },
  { "ﾷ", "ﾷ", "ﾷ", "ᄋ", "ᄋ" },
  { "ﾸ", "ﾸ", "ﾸ", "ᄌ", "ᄌ" },
  { "ﾹ", "ﾹ", "ﾹ", "ᄍ", "ᄍ" },
  { "ﾺ", "ﾺ", "ﾺ", "ᄎ", "ᄎ" },
  { "ﾻ", "ﾻ", "ﾻ", "ᄏ", "ᄏ" },
  { "ﾼ", "ﾼ", "ﾼ", "ᄐ", "ᄐ" },
  { "ﾽ", "ﾽ", "ﾽ", "ᄑ", "ᄑ" },
  { "ﾾ", "ﾾ", "ﾾ", "ᄒ", "ᄒ" },
  { "ￂ", "ￂ", "ￂ", "ᅡ", "ᅡ" },
  { "ￃ", "ￃ", "ￃ", "ᅢ", "ᅢ" },
  { "ￄ", "ￄ", "ￄ", "ᅣ", "ᅣ" },
  { "ￅ", "ￅ", "ￅ", "ᅤ", "ᅤ" },
  { "ￆ", "ￆ", "ￆ", "ᅥ", "ᅥ" },
  { "ￇ", "ￇ", "ￇ", "ᅦ", "ᅦ" },
  { "ￊ", "ￊ", "ￊ", "ᅧ", "ᅧ" },
  { "ￋ", "ￋ", "ￋ", "ᅨ", "ᅨ" },
  { "ￌ", "ￌ", "ￌ", "ᅩ", "ᅩ" },
  { "ￍ", "ￍ", "ￍ", "ᅪ", "ᅪ" },
  { "ￎ", "ￎ", "ￎ", "ᅫ", "ᅫ" },
  { "ￏ", "ￏ", "ￏ", "ᅬ", "ᅬ" },
  { "ￒ", "ￒ", "ￒ", "ᅭ", "ᅭ" },
  { "ￓ", "ￓ", "ￓ", "ᅮ", "ᅮ" },
  { "ￔ", "ￔ", "ￔ", "ᅯ", "ᅯ" },
  { "ￕ", "ￕ", "ￕ", "ᅰ", "ᅰ" },
  { "ￖ", "ￖ", "ￖ", "ᅱ", "ᅱ" },
  { "ￗ", "ￗ", "ￗ", "ᅲ", "ᅲ" },
  { "ￚ", "ￚ", "ￚ", "ᅳ", "ᅳ" },
  { "ￛ", "ￛ", "ￛ", "ᅴ", "ᅴ" },
  { "ￜ", "ￜ", "ￜ", "ᅵ", "ᅵ" },
  { "￠", "￠", "￠", "¢", "¢" },
  { "￡", "￡", "￡", "£", "£" },
  { "￢", "￢", "￢", "¬", "¬" },
  { "￣", "￣", "￣", " ̄", " ̄" },
  { "￤", "￤", "￤", "¦", "¦" },
  { "￥", "￥", "￥", "¥", "¥" },
  { "￦", "￦", "￦", "₩", "₩" },
  { "￨", "￨", "￨", "│", "│" },
  { "￩", "￩", "￩", "←", "←" },
  { "￪", "￪", "￪", "↑", "↑" },
  { "￫", "￫", "￫", "→", "→" },
  { "￬", "￬", "￬", "↓", "↓" },
  { "￭", "￭", "￭", "■", "■" },
  { "￮", "￮", "￮", "○", "○" },
  { "𑂚", "𑂚", "𑂚", "𑂚", "𑂚" },
  { "𑂜", "𑂜", "𑂜", "𑂜", "𑂜" },
  { "𑂫", "𑂫", "𑂫", "𑂫", "𑂫" },
  { "𑄮", "𑄮", "𑄮", "𑄮", "𑄮" },
  { "𑄯", "𑄯", "𑄯", "𑄯", "𑄯" },
  { "𑍋", "𑍋", "𑍋", "𑍋", "𑍋" },
  { "𑍌", "𑍌", "𑍌", "𑍌", "𑍌" },
  { "𑒻", "𑒻", "𑒻", "𑒻", "𑒻" },
  { "𑒼", "𑒼", "𑒼", "𑒼", "𑒼" },
  { "𑒾", "𑒾", "𑒾", "𑒾", "𑒾" },
  { "𑖺", "𑖺", "𑖺", "𑖺", "𑖺" },
  { "𑖻", "𑖻", "𑖻", "𑖻", "𑖻" },
  { "𝅗𝅥", "𝅗𝅥", "𝅗𝅥", "𝅗𝅥", "𝅗𝅥" },
  { "𝅘𝅥", "𝅘𝅥", "𝅘𝅥", "𝅘𝅥", "𝅘𝅥" },
  { "𝅘𝅥𝅮", "𝅘𝅥𝅮", "𝅘𝅥𝅮", "𝅘𝅥𝅮", "𝅘𝅥𝅮" },
  { "𝅘𝅥𝅯", "𝅘𝅥𝅯", "𝅘𝅥𝅯", "𝅘𝅥𝅯", "𝅘𝅥𝅯" },
  { "𝅘𝅥𝅰", "𝅘𝅥𝅰", "𝅘𝅥𝅰", "𝅘𝅥𝅰", "𝅘𝅥𝅰" },
  { "𝅘𝅥𝅱", "𝅘𝅥𝅱", "𝅘𝅥𝅱", "𝅘𝅥𝅱", "𝅘𝅥𝅱" },
  { "𝅘𝅥𝅲", "𝅘𝅥𝅲", "𝅘𝅥𝅲", "𝅘𝅥𝅲", "𝅘𝅥𝅲" },
  { "𝆹𝅥", "𝆹𝅥", "𝆹𝅥", "𝆹𝅥", "𝆹𝅥" },
  { "𝆺𝅥", "𝆺𝅥", "𝆺𝅥", "𝆺𝅥", "𝆺𝅥" },
  { "𝆹𝅥𝅮", "𝆹𝅥𝅮", "𝆹𝅥𝅮", "𝆹𝅥𝅮", "𝆹𝅥𝅮" },
  { "𝆺𝅥𝅮", "𝆺𝅥𝅮", "𝆺𝅥𝅮", "𝆺𝅥𝅮", "𝆺𝅥𝅮" },
  { "𝆹𝅥𝅯", "𝆹𝅥𝅯", "𝆹𝅥𝅯", "𝆹𝅥𝅯", "𝆹𝅥𝅯" },
  { "𝆺𝅥𝅯", "𝆺𝅥𝅯", "𝆺𝅥𝅯", "𝆺𝅥𝅯", "𝆺𝅥𝅯" },
  { "𝐀", "𝐀", "𝐀", "A", "A" },
  { "𝐁", "𝐁", "𝐁", "B", "B" },
  { "𝐂", "𝐂", "𝐂", "C", "C" },
  { "𝐃", "𝐃", "𝐃", "D", "D" },
  { "𝐄", "𝐄", "𝐄", "E", "E" },
  { "𝐅", "𝐅", "𝐅", "F", "F" },
  { "𝐆", "𝐆", "𝐆", "G", "G" },
  { "𝐇", "𝐇", "𝐇", "H", "H" },
  { "𝐈", "𝐈", "𝐈", "I", "I" },
  { "𝐉", "𝐉", "𝐉", "J", "J" },
  { "𝐊", "𝐊", "𝐊", "K", "K" },
  { "𝐋", "𝐋", "𝐋", "L", "L" },
  { "𝐌", "𝐌", "𝐌", "M", "M" },
  { "𝐍", "𝐍", "𝐍", "N", "N" },
  { "𝐎", "𝐎", "𝐎", "O", "O" },
  { "𝐏", "𝐏", "𝐏", "P", "P" },
  { "𝐐", "𝐐", "𝐐", "Q", "Q" },
  { "𝐑", "𝐑", "𝐑", "R", "R" },
  { "𝐒", "𝐒", "𝐒", "S", "S" },
  { "𝐓", "𝐓", "𝐓", "T", "T" },
  { "𝐔", "𝐔", "𝐔", "U", "U" },
  { "𝐕", "𝐕", "𝐕", "V", "V" },
  { "𝐖", "𝐖", "𝐖", "W", "W" },
  { "𝐗", "𝐗", "𝐗", "X", "X" },
  { "𝐘", "𝐘", "𝐘", "Y", "Y" },
  { "𝐙", "𝐙", "𝐙", "Z", "Z" },
  { "𝐚", "𝐚", "𝐚", "a", "a" },
  { "𝐛", "𝐛", "𝐛", "b", "b" },
  { "𝐜", "𝐜", "𝐜", "c", "c" },
  { "𝐝", "𝐝", "𝐝", "d", "d" },
  { "𝐞", "𝐞", "𝐞", "e", "e" },
  { "𝐟", "𝐟", "𝐟", "f", "f" },
  { "𝐠", "𝐠", "𝐠", "g", "g" },
  { "𝐡", "𝐡", "𝐡", "h", "h" },
  { "𝐢", "𝐢", "𝐢", "i", "i" },
  { "𝐣", "𝐣", "𝐣", "j", "j" },
  { "𝐤", "𝐤", "𝐤", "k", "k" },
  { "𝐥", "𝐥", "𝐥", "l", "l" },
  { "𝐦", "𝐦", "𝐦", "m", "m" },
  { "𝐧", "𝐧", "𝐧", "n", "n" },
  { "𝐨", "𝐨", "𝐨", "o", "o" },
  { "𝐩", "𝐩", "𝐩", "p", "p" },
  { "𝐪", "𝐪", "𝐪", "q", "q" },
  { "𝐫", "𝐫", "𝐫", "r", "r" },
  { "𝐬", "𝐬", "𝐬", "s", "s" },
  { "𝐭", "𝐭", "𝐭", "t", "t" },
  { "𝐮", "𝐮", "𝐮", "u", "u" },
  { "𝐯", "𝐯", "𝐯", "v", "v" },
  { "𝐰", "𝐰", "𝐰", "w", "w" },
  { "𝐱", "𝐱", "𝐱", "x", "x" },
  { "𝐲", "𝐲", "𝐲", "y", "y" },
  { "𝐳", "𝐳", "𝐳", "z", "z" },
  { "𝐴", "𝐴", "𝐴", "A", "A" },
  { "𝐵", "𝐵", "𝐵", "B", "B" },
  { "𝐶", "𝐶", "𝐶", "C", "C" },
  { "𝐷", "𝐷", "𝐷", "D", "D" },
  { "𝐸", "𝐸", "𝐸", "E", "E" },
  { "𝐹", "𝐹", "𝐹", "F", "F" },
  { "𝐺", "𝐺", "𝐺", "G", "G" },
  { "𝐻", "𝐻", "𝐻", "H", "H" },
  { "𝐼", "𝐼", "𝐼", "I", "I" },
  { "𝐽", "𝐽", "𝐽", "J", "J" },
  { "𝐾", "𝐾", "𝐾", "K", "K" },
  { "𝐿", "𝐿", "𝐿", "L", "L" },
  { "𝑀", "𝑀", "𝑀", "M", "M" },
  { "𝑁", "𝑁", "𝑁", "N", "N" },
  { "𝑂", "𝑂", "𝑂", "O", "O" },
  { "𝑃", "𝑃", "𝑃", "P", "P" },
  { "𝑄", "𝑄", "𝑄", "Q", "Q" },
  { "𝑅", "𝑅", "𝑅", "R", "R" },
  { "𝑆", "𝑆", "𝑆", "S", "S" },
  { "𝑇", "𝑇", "𝑇", "T", "T" },
  { "𝑈", "𝑈", "𝑈", "U", "U" },
  { "𝑉", "𝑉", "𝑉", "V", "V" },
  { "𝑊", "𝑊", "𝑊", "W", "W" },
  { "𝑋", "𝑋", "𝑋", "X", "X" },
  { "𝑌", "𝑌", "𝑌", "Y", "Y" },
  { "𝑍", "𝑍", "𝑍", "Z", "Z" },
  { "𝑎", "𝑎", "𝑎", "a", "a" },
  { "𝑏", "𝑏", "𝑏", "b", "b" },
  { "𝑐", "𝑐", "𝑐", "c", "c" },
  { "𝑑", "𝑑", "𝑑", "d", "d" },
  { "𝑒", "𝑒", "𝑒", "e", "e" },
  { "𝑓", "𝑓", "𝑓", "f", "f" },
  { "𝑔", "𝑔", "𝑔", "g", "g" },
  { "𝑖", "𝑖", "𝑖", "i", "i" },
  { "𝑗", "𝑗", "𝑗", "j", "j" },
  { "𝑘", "𝑘", "𝑘", "k", "k" },
  { "𝑙", "𝑙", "𝑙", "l", "l" },
  { "𝑚", "𝑚", "𝑚", "m", "m" },
  { "𝑛", "𝑛", "𝑛", "n", "n" },
  { "𝑜", "𝑜", "𝑜", "o", "o" },
  { "𝑝", "𝑝", "𝑝", "p", "p" },
  { "𝑞", "𝑞", "𝑞", "q", "q" },
  { "𝑟", "𝑟", "𝑟", "r", "r" },
  { "𝑠", "𝑠", "𝑠", "s", "s" },
  { "𝑡", "𝑡", "𝑡", "t", "t" },
  { "𝑢", "𝑢", "𝑢", "u", "u" },
  { "𝑣", "𝑣", "𝑣", "v", "v" },
  { "𝑤", "𝑤", "𝑤", "w", "w" },
  { "𝑥", "𝑥", "𝑥", "x", "x" },
  { "𝑦", "𝑦", "𝑦", "y", "y" },
  { "𝑧", "𝑧", "𝑧", "z", "z" },
  { "𝑨", "𝑨", "𝑨", "A", "A" },
  { "𝑩", "𝑩", "𝑩", "B", "B" },
  { "𝑪", "𝑪", "𝑪", "C", "C" },
  { "𝑫", "𝑫", "𝑫", "D", "D" },
  { "𝑬", "𝑬", "𝑬", "E", "E" },
  { "𝑭", "𝑭", "𝑭", "F", "F" },
  { "𝑮", "𝑮", "𝑮", "G", "G" },
  { "𝑯", "𝑯", "𝑯", "H", "H" },
  { "𝑰", "𝑰", "𝑰", "I", "I" },
  { "𝑱", "𝑱", "𝑱", "J", "J" },
  { "𝑲", "𝑲", "𝑲", "K", "K" },
  { "𝑳", "𝑳", "𝑳", "L", "L" },
  { "𝑴", "𝑴", "𝑴", "M", "M" },
  { "𝑵", "𝑵", "𝑵", "N", "N" },
  { "𝑶", "𝑶", "𝑶", "O", "O" },
  { "𝑷", "𝑷", "𝑷", "P", "P" },
  { "𝑸", "𝑸", "𝑸", "Q", "Q" },
  { "𝑹", "𝑹", "𝑹", "R", "R" },
  { "𝑺", "𝑺", "𝑺", "S", "S" },
  { "𝑻", "𝑻", "𝑻", "T", "T" },
  { "𝑼", "𝑼", "𝑼", "U", "U" },
  { "𝑽", "𝑽", "𝑽", "V", "V" },
  { "𝑾", "𝑾", "𝑾", "W", "W" },
  { "𝑿", "𝑿", "𝑿", "X", "X" },
  { "𝒀", "𝒀", "𝒀", "Y", "Y" },
  { "𝒁", "𝒁", "𝒁", "Z", "Z" },
  { "𝒂", "𝒂", "𝒂", "a", "a" },
  { "𝒃", "𝒃", "𝒃", "b", "b" },
  { "𝒄", "𝒄", "𝒄", "c", "c" },
  { "𝒅", "𝒅", "𝒅", "d", "d" },
  { "𝒆", "𝒆", "𝒆", "e", "e" },
  { "𝒇", "𝒇", "𝒇", "f", "f" },
  { "𝒈", "𝒈", "𝒈", "g", "g" },
  { "𝒉", "𝒉", "𝒉", "h", "h" },
  { "𝒊", "𝒊", "𝒊", "i", "i" },
  { "𝒋", "𝒋", "𝒋", "j", "j" },
  { "𝒌", "𝒌", "𝒌", "k", "k" },
  { "𝒍", "𝒍", "𝒍", "l", "l" },
  { "𝒎", "𝒎", "𝒎", "m", "m" },
  { "𝒏", "𝒏", "𝒏", "n", "n" },
  { "𝒐", "𝒐", "𝒐", "o", "o" },
  { "𝒑", "𝒑", "𝒑", "p", "p" },
  { "𝒒", "𝒒", "𝒒", "q", "q" },
  { "𝒓", "𝒓", "𝒓", "r", "r" },
  { "𝒔", "𝒔", "𝒔", "s", "s" },
  { "𝒕", "𝒕", "𝒕", "t", "t" },
  { "𝒖", "𝒖", "𝒖", "u", "u" },
  { "𝒗", "𝒗", "𝒗", "v", "v" },
  { "𝒘", "𝒘", "𝒘", "w", "w" },
  { "𝒙", "𝒙", "𝒙", "x", "x" },
  { "𝒚", "𝒚", "𝒚", "y", "y" },
  { "𝒛", "𝒛", "𝒛", "z", "z" },
  { "𝒜", "𝒜", "𝒜", "A", "A" },
  { "𝒞", "𝒞", "𝒞", "C", "C" },
  { "𝒟", "𝒟", "𝒟", "D", "D" },
  { "𝒢", "𝒢", "𝒢", "G", "G" },
  { "𝒥", "𝒥", "𝒥", "J", "J" },
  { "𝒦", "𝒦", "𝒦", "K", "K" },
  { "𝒩", "𝒩", "𝒩", "N", "N" },
  { "𝒪", "𝒪", "𝒪", "O", "O" },
  { "𝒫", "𝒫", "𝒫", "P", "P" },
  { "𝒬", "𝒬", "𝒬", "Q", "Q" },
  { "𝒮", "𝒮", "𝒮", "S", "S" },
  { "𝒯", "𝒯", "𝒯", "T", "T" },
  { "𝒰", "𝒰", "𝒰", "U", "U" },
  { "𝒱", "𝒱", "𝒱", "V", "V" },
  { "𝒲", "𝒲", "𝒲", "W", "W" },
  { "𝒳", "𝒳", "𝒳", "X", "X" },
  { "𝒴", "𝒴", "𝒴", "Y", "Y" },
  { "𝒵", "𝒵", "𝒵", "Z", "Z" },
  { "𝒶", "𝒶", "𝒶", "a", "a" },
  { "𝒷", "𝒷", "𝒷", "b", "b" },
  { "𝒸", "𝒸", "𝒸", "c", "c" },
  { "𝒹", "𝒹", "𝒹", "d", "d" },
  { "𝒻", "𝒻", "𝒻", "f", "f" },
  { "𝒽", "𝒽", "𝒽", "h", "h" },
  { "𝒾", "𝒾", "𝒾", "i", "i" },
  { "𝒿", "𝒿", "𝒿", "j", "j" },
  { "𝓀", "𝓀", "𝓀", "k", "k" },
  { "𝓁", "𝓁", "𝓁", "l", "l" },
  { "𝓂", "𝓂", "𝓂", "m", "m" },
  { "𝓃", "𝓃", "𝓃", "n", "n" },
  { "𝓅", "𝓅", "𝓅", "p", "p" },
  { "𝓆", "𝓆", "𝓆", "q", "q" },
  { "𝓇", "𝓇", "𝓇", "r", "r" },
  { "𝓈", "𝓈", "𝓈", "s", "s" },
  { "𝓉", "𝓉", "𝓉", "t", "t" },
  { "𝓊", "𝓊", "𝓊", "u", "u" },
  { "𝓋", "𝓋", "𝓋", "v", "v" },
  { "𝓌", "𝓌", "𝓌", "w", "w" },
  { "𝓍", "𝓍", "𝓍", "x", "x" },
  { "𝓎", "𝓎", "𝓎", "y", "y" },
  { "𝓏", "𝓏", "𝓏", "z", "z" },
  { "𝓐", "𝓐", "𝓐", "A", "A" },
  { "𝓑", "𝓑", "𝓑", "B", "B" },
  { "𝓒", "𝓒", "𝓒", "C", "C" },
  { "𝓓", "𝓓", "𝓓", "D", "D" },
  { "𝓔", "𝓔", "𝓔", "E", "E" },
  { "𝓕", "𝓕", "𝓕", "F", "F" },
  { "𝓖", "𝓖", "𝓖", "G", "G" },
  { "𝓗", "𝓗", "𝓗", "H", "H" },
  { "𝓘", "𝓘", "𝓘", "I", "I" },
  { "𝓙", "𝓙", "𝓙", "J", "J" },
  { "𝓚", "𝓚", "𝓚", "K", "K" },
  { "𝓛", "𝓛", "𝓛", "L", "L" },
  { "𝓜", "𝓜", "𝓜", "M", "M" },
  { "𝓝", "𝓝", "𝓝", "N", "N" },
  { "𝓞", "𝓞", "𝓞", "O", "O" },
  { "𝓟", "𝓟", "𝓟", "P", "P" },
  { "𝓠", "𝓠", "𝓠", "Q", "Q" },
  { "𝓡", "𝓡", "𝓡", "R", "R" },
  { "𝓢", "𝓢", "𝓢", "S", "S" },
  { "𝓣", "𝓣", "𝓣", "T", "T" },
  { "𝓤", "𝓤", "𝓤", "U", "U" },
  { "𝓥", "𝓥", "𝓥", "V", "V" },
  { "𝓦", "𝓦", "𝓦", "W", "W" },
  { "𝓧", "𝓧", "𝓧", "X", "X" },
  { "𝓨", "𝓨", "𝓨", "Y", "Y" },
  { "𝓩", "𝓩", "𝓩", "Z", "Z" },
  { "𝓪", "𝓪", "𝓪", "a", "a" },
  { "𝓫", "𝓫", "𝓫", "b", "b" },
  { "𝓬", "𝓬", "𝓬", "c", "c" },
  { "𝓭", "𝓭", "𝓭", "d", "d" },
  { "𝓮", "𝓮", "𝓮", "e", "e" },
  { "𝓯", "𝓯", "𝓯", "f", "f" },
  { "𝓰", "𝓰", "𝓰", "g", "g" },
  { "𝓱", "𝓱", "𝓱", "h", "h" },
  { "𝓲", "𝓲", "𝓲", "i", "i" },
  { "𝓳", "𝓳", "𝓳", "j", "j" },
  { "𝓴", "𝓴", "𝓴", "k", "k" },
  { "𝓵", "𝓵", "𝓵", "l", "l" },
  { "𝓶", "𝓶", "𝓶", "m", "m" },
  { "𝓷", "𝓷", "𝓷", "n", "n" },
  { "𝓸", "𝓸", "𝓸", "o", "o" },
  { "𝓹", "𝓹", "𝓹", "p", "p" },
  { "𝓺", "𝓺", "𝓺", "q", "q" },
  { "𝓻", "𝓻", "𝓻", "r", "r" },
  { "𝓼", "𝓼", "𝓼", "s", "s" },
  { "𝓽", "𝓽", "𝓽", "t", "t" },
  { "𝓾", "𝓾", "𝓾", "u", "u" },
  { "𝓿", "𝓿", "𝓿", "v", "v" },
  { "𝔀", "𝔀", "𝔀", "w", "w" },
  { "𝔁", "𝔁", "𝔁", "x", "x" },
  { "𝔂", "𝔂", "𝔂", "y", "y" },
  { "𝔃", "𝔃", "𝔃", "z", "z" },
  { "𝔄", "𝔄", "𝔄", "A", "A" },
  { "𝔅", "𝔅", "𝔅", "B", "B" },
  { "𝔇", "𝔇", "𝔇", "D", "D" },
  { "𝔈", "𝔈", "𝔈", "E", "E" },
  { "𝔉", "𝔉", "𝔉", "F", "F" },
  { "𝔊", "𝔊", "𝔊", "G", "G" },
  { "𝔍", "𝔍", "𝔍", "J", "J" },
  { "𝔎", "𝔎", "𝔎", "K", "K" },
  { "𝔏", "𝔏", "𝔏", "L", "L" },
  { "𝔐", "𝔐", "𝔐", "M", "M" },
  { "𝔑", "𝔑", "𝔑", "N", "N" },
  { "𝔒", "𝔒", "𝔒", "O", "O" },
  { "𝔓", "𝔓", "𝔓", "P", "P" },
  { "𝔔", "𝔔", "𝔔", "Q", "Q" },
  { "𝔖", "𝔖", "𝔖", "S", "S" },
  { "𝔗", "𝔗", "𝔗", "T", "T" },
  { "𝔘", "𝔘", "𝔘", "U", "U" },
  { "𝔙", "𝔙", "𝔙", "V", "V" },
  { "𝔚", "𝔚", "𝔚", "W", "W" },
  { "𝔛", "𝔛", "𝔛", "X", "X" },
  { "𝔜", "𝔜", "𝔜", "Y", "Y" },
  { "𝔞", "𝔞", "𝔞", "a", "a" },
  { "𝔟", "𝔟", "𝔟", "b", "b" },
  { "𝔠", "𝔠", "𝔠", "c", "c" },
  { "𝔡", "𝔡", "𝔡", "d", "d" },
  { "𝔢", "𝔢", "𝔢", "e", "e" },
  { "𝔣", "𝔣", "𝔣", "f", "f" },
  { "𝔤", "𝔤", "𝔤", "g", "g" },
  { "𝔥", "𝔥", "𝔥", "h", "h" },
  { "𝔦", "𝔦", "𝔦", "i", "i" },
  { "𝔧", "𝔧", "𝔧", "j", "j" },
  { "𝔨", "𝔨", "𝔨", "k", "k" },
  { "𝔩", "𝔩", "𝔩", "l", "l" },
  { "𝔪", "𝔪", "𝔪", "m", "m" },
  { "𝔫", "𝔫", "𝔫", "n", "n" },
  { "𝔬", "𝔬", "𝔬", "o", "o" },
  { "𝔭", "𝔭", "𝔭", "p", "p" },
  { "𝔮", "𝔮", "𝔮", "q", "q" },
  { "𝔯", "𝔯", "𝔯", "r", "r" },
  { "𝔰", "𝔰", "𝔰", "s", "s" },
  { "𝔱", "𝔱", "𝔱", "t", "t" },
  { "𝔲", "𝔲", "𝔲", "u", "u" },
  { "𝔳", "𝔳", "𝔳", "v", "v" },
  { "𝔴", "𝔴", "𝔴", "w", "w" },
  { "𝔵", "𝔵", "𝔵", "x", "x" },
  { "𝔶", "𝔶", "𝔶", "y", "y" },
  { "𝔷", "𝔷", "𝔷", "z", "z" },
  { "𝔸", "𝔸", "𝔸", "A", "A" },
  { "𝔹", "𝔹", "𝔹", "B", "B" },
  { "𝔻", "𝔻", "𝔻", "D", "D" },
  { "𝔼", "𝔼", "𝔼", "E", "E" },
  { "𝔽", "𝔽", "𝔽", "F", "F" },
  { "𝔾", "𝔾", "𝔾", "G", "G" },
  { "𝕀", "𝕀", "𝕀", "I", "I" },
  { "𝕁", "𝕁", "𝕁", "J", "J" },
  { "𝕂", "𝕂", "𝕂", "K", "K" },
  { "𝕃", "𝕃", "𝕃", "L", "L" },
  { "𝕄", "𝕄", "𝕄", "M", "M" },
  { "𝕆", "𝕆", "𝕆", "O", "O" },
  { "𝕊", "𝕊", "𝕊", "S", "S" },
  { "𝕋", "𝕋", "𝕋", "T", "T" },
  { "𝕌", "𝕌", "𝕌", "U", "U" },
  { "𝕍", "𝕍", "𝕍", "V", "V" },
  { "𝕎", "𝕎", "𝕎", "W", "W" },
  { "𝕏", "𝕏", "𝕏", "X", "X" },
  { "𝕐", "𝕐", "𝕐", "Y", "Y" },
  { "𝕒", "𝕒", "𝕒", "a", "a" },
  { "𝕓", "𝕓", "𝕓", "b", "b" },
  { "𝕔", "𝕔", "𝕔", "c", "c" },
  { "𝕕", "𝕕", "𝕕", "d", "d" },
  { "𝕖", "𝕖", "𝕖", "e", "e" },
  { "𝕗", "𝕗", "𝕗", "f", "f" },
  { "𝕘", "𝕘", "𝕘", "g", "g" },
  { "𝕙", "𝕙", "𝕙", "h", "h" },
  { "𝕚", "𝕚", "𝕚", "i", "i" },
  { "𝕛", "𝕛", "𝕛", "j", "j" },
  { "𝕜", "𝕜", "𝕜", "k", "k" },
  { "𝕝", "𝕝", "𝕝", "l", "l" },
  { "𝕞", "𝕞", "𝕞", "m", "m" },
  { "𝕟", "𝕟", "𝕟", "n", "n" },
  { "𝕠", "𝕠", "𝕠", "o", "o" },
  { "𝕡", "𝕡", "𝕡", "p", "p" },
  { "𝕢", "𝕢", "𝕢", "q", "q" },
  { "𝕣", "𝕣", "𝕣", "r", "r" },
  { "𝕤", "𝕤", "𝕤", "s", "s" },
  { "𝕥", "𝕥", "𝕥", "t", "t" },
  { "𝕦", "𝕦", "𝕦", "u", "u" },
  { "𝕧", "𝕧", "𝕧", "v", "v" },
  { "𝕨", "𝕨", "𝕨", "w", "w" },
  { "𝕩", "𝕩", "𝕩", "x", "x" },
  { "𝕪", "𝕪", "𝕪", "y", "y" },
  { "𝕫", "𝕫", "𝕫", "z", "z" },
  { "𝕬", "𝕬", "𝕬", "A", "A" },
  { "𝕭", "𝕭", "𝕭", "B", "B" },
  { "𝕮", "𝕮", "𝕮", "C", "C" },
  { "𝕯", "𝕯", "𝕯", "D", "D" },
  { "𝕰", "𝕰", "𝕰", "E", "E" },
  { "𝕱", "𝕱", "𝕱", "F", "F" },
  { "𝕲", "𝕲", "𝕲", "G", "G" },
  { "𝕳", "𝕳", "𝕳", "H", "H" },
  { "𝕴", "𝕴", "𝕴", "I", "I" },
  { "𝕵", "𝕵", "𝕵", "J", "J" },
  { "𝕶", "𝕶", "𝕶", "K", "K" },
  { "𝕷", "𝕷", "𝕷", "L", "L" },
  { "𝕸", "𝕸", "𝕸", "M", "M" },
  { "𝕹", "𝕹", "𝕹", "N", "N" },
  { "𝕺", "𝕺", "𝕺", "O", "O" },
  { "𝕻", "𝕻", "𝕻", "P", "P" },
  { "𝕼", "𝕼", "𝕼", "Q", "Q" },
  { "𝕽", "𝕽", "𝕽", "R", "R" },
  { "𝕾", "𝕾", "𝕾", "S", "S" },
  { "𝕿", "𝕿", "𝕿", "T", "T" },
  { "𝖀", "𝖀", "𝖀", "U", "U" },
  { "𝖁", "𝖁", "𝖁", "V", "V" },
  { "𝖂", "𝖂", "𝖂", "W", "W" },
  { "𝖃", "𝖃", "𝖃", "X", "X" },
  { "𝖄", "𝖄", "𝖄", "Y", "Y" },
  { "𝖅", "𝖅", "𝖅", "Z", "Z" },
  { "𝖆", "𝖆", "𝖆", "a", "a" },
  { "𝖇", "𝖇", "𝖇", "b", "b" },
  { "𝖈", "𝖈", "𝖈", "c", "c" },
  { "𝖉", "𝖉", "𝖉", "d", "d" },
  { "𝖊", "𝖊", "𝖊", "e", "e" },
  { "𝖋", "𝖋", "𝖋", "f", "f" },
  { "𝖌", "𝖌", "𝖌", "g", "g" },
  { "𝖍", "𝖍", "𝖍", "h", "h" },
  { "𝖎", "𝖎", "𝖎", "i", "i" },
  { "𝖏", "𝖏", "𝖏", "j", "j" },
  { "𝖐", "𝖐", "𝖐", "k", "k" },
  { "𝖑", "𝖑", "𝖑", "l", "l" },
  { "𝖒", "𝖒", "𝖒", "m", "m" },
  { "𝖓", "𝖓", "𝖓", "n", "n" },
  { "𝖔", "𝖔", "𝖔", "o", "o" },
  { "𝖕", "𝖕", "𝖕", "p", "p" },
  { "𝖖", "𝖖", "𝖖", "q", "q" },
  { "𝖗", "𝖗", "𝖗", "r", "r" },
  { "𝖘", "𝖘", "𝖘", "s", "s" },
  { "𝖙", "𝖙", "𝖙", "t", "t" },
  { "𝖚", "𝖚", "𝖚", "u", "u" },
  { "𝖛", "𝖛", "𝖛", "v", "v" },
  { "𝖜", "𝖜", "𝖜", "w", "w" },
  { "𝖝", "𝖝", "𝖝", "x", "x" },
  { "𝖞", "𝖞", "𝖞", "y", "y" },
  { "𝖟", "𝖟", "𝖟", "z", "z" },
  { "𝖠", "𝖠", "𝖠", "A", "A" },
  { "𝖡", "𝖡", "𝖡", "B", "B" },
  { "𝖢", "𝖢", "𝖢", "C", "C" },
  { "𝖣", "𝖣", "𝖣", "D", "D" },
  { "𝖤", "𝖤", "𝖤", "E", "E" },
  { "𝖥", "𝖥", "𝖥", "F", "F" },
  { "𝖦", "𝖦", "𝖦", "G", "G" },
  { "𝖧", "𝖧", "𝖧", "H", "H" },
  { "𝖨", "𝖨", "𝖨", "I", "I" },
  { "𝖩", "𝖩", "𝖩", "J", "J" },
  { "𝖪", "𝖪", "𝖪", "K", "K" },
  { "𝖫", "𝖫", "𝖫", "L", "L" },
  { "𝖬", "𝖬", "𝖬", "M", "M" },
  { "𝖭", "𝖭", "𝖭", "N", "N" },
  { "𝖮", "𝖮", "𝖮", "O", "O" },
  { "𝖯", "𝖯", "𝖯", "P", "P" },
  { "𝖰", "𝖰", "𝖰", "Q", "Q" },
  { "𝖱", "𝖱", "𝖱", "R", "R" },
  { "𝖲", "𝖲", "𝖲", "S", "S" },
  { "𝖳", "𝖳", "𝖳", "T", "T" },
  { "𝖴", "𝖴", "𝖴", "U", "U" },
  { "𝖵", "𝖵", "𝖵", "V", "V" },
  { "𝖶", "𝖶", "𝖶", "W", "W" },
  { "𝖷", "𝖷", "𝖷", "X", "X" },
  { "𝖸", "𝖸", "𝖸", "Y", "Y" },
  { "𝖹", "𝖹", "𝖹", "Z", "Z" },
  { "𝖺", "𝖺", "𝖺", "a", "a" },
  { "𝖻", "𝖻", "𝖻", "b", "b" },
  { "𝖼", "𝖼", "𝖼", "c", "c" },
  { "𝖽", "𝖽", "𝖽", "d", "d" },
  { "𝖾", "𝖾", "𝖾", "e", "e" },
  { "𝖿", "𝖿", "𝖿", "f", "f" },
  { "𝗀", "𝗀", "𝗀", "g", "g" },
  { "𝗁", "𝗁", "𝗁", "h", "h" },
  { "𝗂", "𝗂", "𝗂", "i", "i" },
  { "𝗃", "𝗃", "𝗃", "j", "j" },
  { "𝗄", "𝗄", "𝗄", "k", "k" },
  { "𝗅", "𝗅", "𝗅", "l", "l" },
  { "𝗆", "𝗆", "𝗆", "m", "m" },
  { "𝗇", "𝗇", "𝗇", "n", "n" },
  { "𝗈", "𝗈", "𝗈", "o", "o" },
  { "𝗉", "𝗉", "𝗉", "p", "p" },
  { "𝗊", "𝗊", "𝗊", "q", "q" },
  { "𝗋", "𝗋", "𝗋", "r", "r" },
  { "𝗌", "𝗌", "𝗌", "s", "s" },
  { "𝗍", "𝗍", "𝗍", "t", "t" },
  { "𝗎", "𝗎", "𝗎", "u", "u" },
  { "𝗏", "𝗏", "𝗏", "v", "v" },
  { "𝗐", "𝗐", "𝗐", "w", "w" },
  { "𝗑", "𝗑", "𝗑", "x", "x" },
  { "𝗒", "𝗒", "𝗒", "y", "y" },
  { "𝗓", "𝗓", "𝗓", "z", "z" },
  { "𝗔", "𝗔", "𝗔", "A", "A" },
  { "𝗕", "𝗕", "𝗕", "B", "B" },
  { "𝗖", "𝗖", "𝗖", "C", "C" },
  { "𝗗", "𝗗", "𝗗", "D", "D" },
  { "𝗘", "𝗘", "𝗘", "E", "E" },
  { "𝗙", "𝗙", "𝗙", "F", "F" },
  { "𝗚", "𝗚", "𝗚", "G", "G" },
  { "𝗛", "𝗛", "𝗛", "H", "H" },
  { "𝗜", "𝗜", "𝗜", "I", "I" },
  { "𝗝", "𝗝", "𝗝", "J", "J" },
  { "𝗞", "𝗞", "𝗞", "K", "K" },
  { "𝗟", "𝗟", "𝗟", "L", "L" },
  { "𝗠", "𝗠", "𝗠", "M", "M" },
  { "𝗡", "𝗡", "𝗡", "N", "N" },
  { "𝗢", "𝗢", "𝗢", "O", "O" },
  { "𝗣", "𝗣", "𝗣", "P", "P" },
  { "𝗤", "𝗤", "𝗤", "Q", "Q" },
  { "𝗥", "𝗥", "𝗥", "R", "R" },
  { "𝗦", "𝗦", "𝗦", "S", "S" },
  { "𝗧", "𝗧", "𝗧", "T", "T" },
  { "𝗨", "𝗨", "𝗨", "U", "U" },
  { "𝗩", "𝗩", "𝗩", "V", "V" },
  { "𝗪", "𝗪", "𝗪", "W", "W" },
  { "𝗫", "𝗫", "𝗫", "X", "X" },
  { "𝗬", "𝗬", "𝗬", "Y", "Y" },
  { "𝗭", "𝗭", "𝗭", "Z", "Z" },
  { "𝗮", "𝗮", "𝗮", "a", "a" },
  { "𝗯", "𝗯", "𝗯", "b", "b" },
  { "𝗰", "𝗰", "𝗰", "c", "c" },
  { "𝗱", "𝗱", "𝗱", "d", "d" },
  { "𝗲", "𝗲", "𝗲", "e", "e" },
  { "𝗳", "𝗳", "𝗳", "f", "f" },
  { "𝗴", "𝗴", "𝗴", "g", "g" },
  { "𝗵", "𝗵", "𝗵", "h", "h" },
  { "𝗶", "𝗶", "𝗶", "i", "i" },
  { "𝗷", "𝗷", "𝗷", "j", "j" },
  { "𝗸", "𝗸", "𝗸", "k", "k" },
  { "𝗹", "𝗹", "𝗹", "l", "l" },
  { "𝗺", "𝗺", "𝗺", "m", "m" },
  { "𝗻", "𝗻", "𝗻", "n", "n" },
  { "𝗼", "𝗼", "𝗼", "o", "o" },
  { "𝗽", "𝗽", "𝗽", "p", "p" },
  { "𝗾", "𝗾", "𝗾", "q", "q" },
  { "𝗿", "𝗿", "𝗿", "r", "r" },
  { "𝘀", "𝘀", "𝘀", "s", "s" },
  { "𝘁", "𝘁", "𝘁", "t", "t" },
  { "𝘂", "𝘂", "𝘂", "u", "u" },
  { "𝘃", "𝘃", "𝘃", "v", "v" },
  { "𝘄", "𝘄", "𝘄", "w", "w" },
  { "𝘅", "𝘅", "𝘅", "x", "x" },
  { "𝘆", "𝘆", "𝘆", "y", "y" },
  { "𝘇", "𝘇", "𝘇", "z", "z" },
  { "𝘈", "𝘈", "𝘈", "A", "A" },
  { "𝘉", "𝘉", "𝘉", "B", "B" },
  { "𝘊", "𝘊", "𝘊", "C", "C" },
  { "𝘋", "𝘋", "𝘋", "D", "D" },
  { "𝘌", "𝘌", "𝘌", "E", "E" },
  { "𝘍", "𝘍", "𝘍", "F", "F" },
  { "𝘎", "𝘎", "𝘎", "G", "G" },
  { "𝘏", "𝘏", "𝘏", "H", "H" },
  { "𝘐", "𝘐", "𝘐", "I", "I" },
  { "𝘑", "𝘑", "𝘑", "J", "J" },
  { "𝘒", "𝘒", "𝘒", "K", "K" },
  { "𝘓", "𝘓", "𝘓", "L", "L" },
  { "𝘔", "𝘔", "𝘔", "M", "M" },
  { "𝘕", "𝘕", "𝘕", "N", "N" },
  { "𝘖", "𝘖", "𝘖", "O", "O" },
  { "𝘗", "𝘗", "𝘗", "P", "P" },
  { "𝘘", "𝘘", "𝘘", "Q", "Q" },
  { "𝘙", "𝘙", "𝘙", "R", "R" },
  { "𝘚", "𝘚", "𝘚", "S", "S" },
  { "𝘛", "𝘛", "𝘛", "T", "T" },
  { "𝘜", "𝘜", "𝘜", "U", "U" },
  { "𝘝", "𝘝", "𝘝", "V", "V" },
  { "𝘞", "𝘞", "𝘞", "W", "W" },
  { "𝘟", "𝘟", "𝘟", "X", "X" },
  { "𝘠", "𝘠", "𝘠", "Y", "Y" },
  { "𝘡", "𝘡", "𝘡", "Z", "Z" },
  { "𝘢", "𝘢", "𝘢", "a", "a" },
  { "𝘣", "𝘣", "𝘣", "b", "b" },
  { "𝘤", "𝘤", "𝘤", "c", "c" },
  { "𝘥", "𝘥", "𝘥", "d", "d" },
  { "𝘦", "𝘦", "𝘦", "e", "e" },
  { "𝘧", "𝘧", "𝘧", "f", "f" },
  { "𝘨", "𝘨", "𝘨", "g", "g" },
  { "𝘩", "𝘩", "𝘩", "h", "h" },
  { "𝘪", "𝘪", "𝘪", "i", "i" },
  { "𝘫", "𝘫", "𝘫", "j", "j" },
  { "𝘬", "𝘬", "𝘬", "k", "k" },
  { "𝘭", "𝘭", "𝘭", "l", "l" },
  { "𝘮", "𝘮", "𝘮", "m", "m" },
  { "𝘯", "𝘯", "𝘯", "n", "n" },
  { "𝘰", "𝘰", "𝘰", "o", "o" },
  { "𝘱", "𝘱", "𝘱", "p", "p" },
  { "𝘲", "𝘲", "𝘲", "q", "q" },
  { "𝘳", "𝘳", "𝘳", "r", "r" },
  { "𝘴", "𝘴", "𝘴", "s", "s" },
  { "𝘵", "𝘵", "𝘵", "t", "t" },
  { "𝘶", "𝘶", "𝘶", "u", "u" },
  { "𝘷", "𝘷", "𝘷", "v", "v" },
  { "𝘸", "𝘸", "𝘸", "w", "w" },
  { "𝘹", "𝘹", "𝘹", "x", "x" },
  { "𝘺", "𝘺", "𝘺", "y", "y" },
  { "𝘻", "𝘻", "𝘻", "z", "z" },
  { "𝘼", "𝘼", "𝘼", "A", "A" },
  { "𝘽", "𝘽", "𝘽", "B", "B" },
  { "𝘾", "𝘾", "𝘾", "C", "C" },
  { "𝘿", "𝘿", "𝘿", "D", "D" },
  { "𝙀", "𝙀", "𝙀", "E", "E" },
  { "𝙁", "𝙁", "𝙁", "F", "F" },
  { "𝙂", "𝙂", "𝙂", "G", "G" },
  { "𝙃", "𝙃", "𝙃", "H", "H" },
  { "𝙄", "𝙄", "𝙄", "I", "I" },
  { "𝙅", "𝙅", "𝙅", "J", "J" },
  { "𝙆", "𝙆", "𝙆", "K", "K" },
  { "𝙇", "𝙇", "𝙇", "L", "L" },
  { "𝙈", "𝙈", "𝙈", "M", "M" },
  { "𝙉", "𝙉", "𝙉", "N", "N" },
  { "𝙊", "𝙊", "𝙊", "O", "O" },
  { "𝙋", "𝙋", "𝙋", "P", "P" },
  { "𝙌", "𝙌", "𝙌", "Q", "Q" },
  { "𝙍", "𝙍", "𝙍", "R", "R" },
  { "𝙎", "𝙎", "𝙎", "S", "S" },
  { "𝙏", "𝙏", "𝙏", "T", "T" },
  { "𝙐", "𝙐", "𝙐", "U", "U" },
  { "𝙑", "𝙑", "𝙑", "V", "V" },
  { "𝙒", "𝙒", "𝙒", "W", "W" },
  { "𝙓", "𝙓", "𝙓", "X", "X" },
  { "𝙔", "𝙔", "𝙔", "Y", "Y" },
  { "𝙕", "𝙕", "𝙕", "Z", "Z" },
  { "𝙖", "𝙖", "𝙖", "a", "a" },
  { "𝙗", "𝙗", "𝙗", "b", "b" },
  { "𝙘", "𝙘", "𝙘", "c", "c" },
  { "𝙙", "𝙙", "𝙙", "d", "d" },
  { "𝙚", "𝙚", "𝙚", "e", "e" },
  { "𝙛", "𝙛", "𝙛", "f", "f" },
  { "𝙜", "𝙜", "𝙜", "g", "g" },
  { "𝙝", "𝙝", "𝙝", "h", "h" },
  { "𝙞", "𝙞", "𝙞", "i", "i" },
  { "𝙟", "𝙟", "𝙟", "j", "j" },
  { "𝙠", "𝙠", "𝙠", "k", "k" },
  { "𝙡", "𝙡", "𝙡", "l", "l" },
  { "𝙢", "𝙢", "𝙢", "m", "m" },
  { "𝙣", "𝙣", "𝙣", "n", "n" },
  { "𝙤", "𝙤", "𝙤", "o", "o" },
  { "𝙥", "𝙥", "𝙥", "p", "p" },
  { "𝙦", "𝙦", "𝙦", "q", "q" },
  { "𝙧", "𝙧", "𝙧", "r", "r" },
  { "𝙨", "𝙨", "𝙨", "s", "s" },
  { "𝙩", "𝙩", "𝙩", "t", "t" },
  { "𝙪", "𝙪", "𝙪", "u", "u" },
  { "𝙫", "𝙫", "𝙫", "v", "v" },
  { "𝙬", "𝙬", "𝙬", "w", "w" },
  { "𝙭", "𝙭", "𝙭", "x", "x" },
  { "𝙮", "𝙮", "𝙮", "y", "y" },
  { "𝙯", "𝙯", "𝙯", "z", "z" },
  { "𝙰", "𝙰", "𝙰", "A", "A" },
  { "𝙱", "𝙱", "𝙱", "B", "B" },
  { "𝙲", "𝙲", "𝙲", "C", "C" },
  { "𝙳", "𝙳", "𝙳", "D", "D" },
  { "𝙴", "𝙴", "𝙴", "E", "E" },
  { "𝙵", "𝙵", "𝙵", "F", "F" },
  { "𝙶", "𝙶", "𝙶", "G", "G" },
  { "𝙷", "𝙷", "𝙷", "H", "H" },
  { "𝙸", "𝙸", "𝙸", "I", "I" },
  { "𝙹", "𝙹", "𝙹", "J", "J" },
  { "𝙺", "𝙺", "𝙺", "K", "K" },
  { "𝙻", "𝙻", "𝙻", "L", "L" },
  { "𝙼", "𝙼", "𝙼", "M", "M" },
  { "𝙽", "𝙽", "𝙽", "N", "N" },
  { "𝙾", "𝙾", "𝙾", "O", "O" },
  { "𝙿", "𝙿", "𝙿", "P", "P" },
  { "𝚀", "𝚀", "𝚀", "Q", "Q" },
  { "𝚁", "𝚁", "𝚁", "R", "R" },
  { "𝚂", "𝚂", "𝚂", "S", "S" },
  { "𝚃", "𝚃", "𝚃", "T", "T" },
  { "𝚄", "𝚄", "𝚄", "U", "U" },
  { "𝚅", "𝚅", "𝚅", "V", "V" },
  { "𝚆", "𝚆", "𝚆", "W", "W" },
  { "𝚇", "𝚇", "𝚇", "X", "X" },
  { "𝚈", "𝚈", "𝚈", "Y", "Y" },
  { "𝚉", "𝚉", "𝚉", "Z", "Z" },
  { "𝚊", "𝚊", "𝚊", "a", "a" },
  { "𝚋", "𝚋", "𝚋", "b", "b" },
  { "𝚌", "𝚌", "𝚌", "c", "c" },
  { "𝚍", "𝚍", "𝚍", "d", "d" },
  { "𝚎", "𝚎", "𝚎", "e", "e" },
  { "𝚏", "𝚏", "𝚏", "f", "f" },
  { "𝚐", "𝚐", "𝚐", "g", "g" },
  { "𝚑", "𝚑", "𝚑", "h", "h" },
  { "𝚒", "𝚒", "𝚒", "i", "i" },
  { "𝚓", "𝚓", "𝚓", "j", "j" },
  { "𝚔", "𝚔", "𝚔", "k", "k" },
  { "𝚕", "𝚕", "𝚕", "l", "l" },
  { "𝚖", "𝚖", "𝚖", "m", "m" },
  { "𝚗", "𝚗", "𝚗", "n", "n" },
  { "𝚘", "𝚘", "𝚘", "o", "o" },
  { "𝚙", "𝚙", "𝚙", "p", "p" },
  { "𝚚", "𝚚", "𝚚", "q", "q" },
  { "𝚛", "𝚛", "𝚛", "r", "r" },
  { "𝚜", "𝚜", "𝚜", "s", "s" },
  { "𝚝", "𝚝", "𝚝", "t", "t" },
  { "𝚞", "𝚞", "𝚞", "u", "u" },
  { "𝚟", "𝚟", "𝚟", "v", "v" },
  { "𝚠", "𝚠", "𝚠", "w", "w" },
  { "𝚡", "𝚡", "𝚡", "x", "x" },
  { "𝚢", "𝚢", "𝚢", "y", "y" },
  { "𝚣", "𝚣", "𝚣", "z", "z" },
  { "𝚤", "𝚤", "𝚤", "ı", "ı" },
  { "𝚥", "𝚥", "𝚥", "ȷ", "ȷ" },
  { "𝚨", "𝚨", "𝚨", "Α", "Α" },
  { "𝚩", "𝚩", "𝚩", "Β", "Β" },
  { "𝚪", "𝚪", "𝚪", "Γ", "Γ" },
  { "𝚫", "𝚫", "𝚫", "Δ", "Δ" },
  { "𝚬", "𝚬", "𝚬", "Ε", "Ε" },
  { "𝚭", "𝚭", "𝚭", "Ζ", "Ζ" },
  { "𝚮", "𝚮", "𝚮", "Η", "Η" },
  { "𝚯", "𝚯", "𝚯", "Θ", "Θ" },
  { "𝚰", "𝚰", "𝚰", "Ι", "Ι" },
  { "𝚱", "𝚱", "𝚱", "Κ", "Κ" },
  { "𝚲", "𝚲", "𝚲", "Λ", "Λ" },
  { "𝚳", "𝚳", "𝚳", "Μ", "Μ" },
  { "𝚴", "𝚴", "𝚴", "Ν", "Ν" },
  { "𝚵", "𝚵", "𝚵", "Ξ", "Ξ" },
  { "𝚶", "𝚶", "𝚶", "Ο", "Ο" },
  { "𝚷", "𝚷", "𝚷", "Π", "Π" },
  { "𝚸", "𝚸", "𝚸", "Ρ", "Ρ" },
  { "𝚹", "𝚹", "𝚹", "Θ", "Θ" },
  { "𝚺", "𝚺", "𝚺", "Σ", "Σ" },
  { "𝚻", "𝚻", "𝚻", "Τ", "Τ" },
  { "𝚼", "𝚼", "𝚼", "Υ", "Υ" },
  { "𝚽", "𝚽", "𝚽", "Φ", "Φ" },
  { "𝚾", "𝚾", "𝚾", "Χ", "Χ" },
  { "𝚿", "𝚿", "𝚿", "Ψ", "Ψ" },
  { "𝛀", "𝛀", "𝛀", "Ω", "Ω" },
  { "𝛁", "𝛁", "𝛁", "∇", "∇" },
  { "𝛂", "𝛂", "𝛂", "α", "α" },
  { "𝛃", "𝛃", "𝛃", "β", "β" },
  { "𝛄", "𝛄", "𝛄", "γ", "γ" },
  { "𝛅", "𝛅", "𝛅", "δ", "δ" },
  { "𝛆", "𝛆", "𝛆", "ε", "ε" },
  { "𝛇", "𝛇", "𝛇", "ζ", "ζ" },
  { "𝛈", "𝛈", "𝛈", "η", "η" },
  { "𝛉", "𝛉", "𝛉", "θ", "θ" },
  { "𝛊", "𝛊", "𝛊", "ι", "ι" },
  { "𝛋", "𝛋", "𝛋", "κ", "κ" },
  { "𝛌", "𝛌", "𝛌", "λ", "λ" },
  { "𝛍", "𝛍", "𝛍", "μ", "μ" },
  { "𝛎", "𝛎", "𝛎", "ν", "ν" },
  { "𝛏", "𝛏", "𝛏", "ξ", "ξ" },
  { "𝛐", "𝛐", "𝛐", "ο", "ο" },
  { "𝛑", "𝛑", "𝛑", "π", "π" },
  { "𝛒", "𝛒", "𝛒", "ρ", "ρ" },
  { "𝛓", "𝛓", "𝛓", "ς", "ς" },
  { "𝛔", "𝛔", "𝛔", "σ", "σ" },
  { "𝛕", "𝛕", "𝛕", "τ", "τ" },
  { "𝛖", "𝛖", "𝛖", "υ", "υ" },
  { "𝛗", "𝛗", "𝛗", "φ", "φ" },
  { "𝛘", "𝛘", "𝛘", "χ", "χ" },
  { "𝛙", "𝛙", "𝛙", "ψ", "ψ" },
  { "𝛚", "𝛚", "𝛚", "ω", "ω" },
  { "𝛛", "𝛛", "𝛛", "∂", "∂" },
  { "𝛜", "𝛜", "𝛜", "ε", "ε" },
  { "𝛝", "𝛝", "𝛝", "θ", "θ" },
  { "𝛞", "𝛞", "𝛞", "κ", "κ" },
  { "𝛟", "𝛟", "𝛟", "φ", "φ" },
  { "𝛠", "𝛠", "𝛠", "ρ", "ρ" },
  { "𝛡", "𝛡", "𝛡", "π", "π" },
  { "𝛢", "𝛢", "𝛢", "Α", "Α" },
  { "𝛣", "𝛣", "𝛣", "Β", "Β" },
  { "𝛤", "𝛤", "𝛤", "Γ", "Γ" },
  { "𝛥", "𝛥", "𝛥", "Δ", "Δ" },
  { "𝛦", "𝛦", "𝛦", "Ε", "Ε" },
  { "𝛧", "𝛧", "𝛧", "Ζ", "Ζ" },
  { "𝛨", "𝛨", "𝛨", "Η", "Η" },
  { "𝛩", "𝛩", "𝛩", "Θ", "Θ" },
  { "𝛪", "𝛪", "𝛪", "Ι", "Ι" },
  { "𝛫", "𝛫", "𝛫", "Κ", "Κ" },
  { "𝛬", "𝛬", "𝛬", "Λ", "Λ" },
  { "𝛭", "𝛭", "𝛭", "Μ", "Μ" },
  { "𝛮", "𝛮", "𝛮", "Ν", "Ν" },
  { "𝛯", "𝛯", "𝛯", "Ξ", "Ξ" },
  { "𝛰", "𝛰", "𝛰", "Ο", "Ο" },
  { "𝛱", "𝛱", "𝛱", "Π", "Π" },
  { "𝛲", "𝛲", "𝛲", "Ρ", "Ρ" },
  { "𝛳", "𝛳", "𝛳", "Θ", "Θ" },
  { "𝛴", "𝛴", "𝛴", "Σ", "Σ" },
  { "𝛵", "𝛵", "𝛵", "Τ", "Τ" },
  { "𝛶", "𝛶", "𝛶", "Υ", "Υ" },
  { "𝛷", "𝛷", "𝛷", "Φ", "Φ" },
  { "𝛸", "𝛸", "𝛸", "Χ", "Χ" },
  { "𝛹", "𝛹", "𝛹", "Ψ", "Ψ" },
  { "𝛺", "𝛺", "𝛺", "Ω", "Ω" },
  { "𝛻", "𝛻", "𝛻", "∇", "∇" },
  { "𝛼", "𝛼", "𝛼", "α", "α" },
  { "𝛽", "𝛽", "𝛽", "β", "β" },
  { "𝛾", "𝛾", "𝛾", "γ", "γ" },
  { "𝛿", "𝛿", "𝛿", "δ", "δ" },
  { "𝜀", "𝜀", "𝜀", "ε", "ε" },
  { "𝜁", "𝜁", "𝜁", "ζ", "ζ" },
  { "𝜂", "𝜂", "𝜂", "η", "η" },
  { "𝜃", "𝜃", "𝜃", "θ", "θ" },
  { "𝜄", "𝜄", "𝜄", "ι", "ι" },
  { "𝜅", "𝜅", "𝜅", "κ", "κ" },
  { "𝜆", "𝜆", "𝜆", "λ", "λ" },
  { "𝜇", "𝜇", "𝜇", "μ", "μ" },
  { "𝜈", "𝜈", "𝜈", "ν", "ν" },
  { "𝜉", "𝜉", "𝜉", "ξ", "ξ" },
  { "𝜊", "𝜊", "𝜊", "ο", "ο" },
  { "𝜋", "𝜋", "𝜋", "π", "π" },
  { "𝜌", "𝜌", "𝜌", "ρ", "ρ" },
  { "𝜍", "𝜍", "𝜍", "ς", "ς" },
  { "𝜎", "𝜎", "𝜎", "σ", "σ" },
  { "𝜏", "𝜏", "𝜏", "τ", "τ" },
  { "𝜐", "𝜐", "𝜐", "υ", "υ" },
  { "𝜑", "𝜑", "𝜑", "φ", "φ" },
  { "𝜒", "𝜒", "𝜒", "χ", "χ" },
  { "𝜓", "𝜓", "𝜓", "ψ", "ψ" },
  { "𝜔", "𝜔", "𝜔", "ω", "ω" },
  { "𝜕", "𝜕", "𝜕", "∂", "∂" },
  { "𝜖", "𝜖", "𝜖", "ε", "ε" },
  { "𝜗", "𝜗", "𝜗", "θ", "θ" },
  { "𝜘", "𝜘", "𝜘", "κ", "κ" },
  { "𝜙", "𝜙", "𝜙", "φ", "φ" },
  { "𝜚", "𝜚", "𝜚", "ρ", "ρ" },
  { "𝜛", "𝜛", "𝜛", "π", "π" },
  { "𝜜", "𝜜", "𝜜", "Α", "Α" },
  { "𝜝", "𝜝", "𝜝", "Β", "Β" },
  { "𝜞", "𝜞", "𝜞", "Γ", "Γ" },
  { "𝜟", "𝜟", "𝜟", "Δ", "Δ" },
  { "𝜠", "𝜠", "𝜠", "Ε", "Ε" },
  { "𝜡", "𝜡", "𝜡", "Ζ", "Ζ" },
  { "𝜢", "𝜢", "𝜢", "Η", "Η" },
  { "𝜣", "𝜣", "𝜣", "Θ", "Θ" },
  { "𝜤", "𝜤", "𝜤", "Ι", "Ι" },
  { "𝜥", "𝜥", "𝜥", "Κ", "Κ" },
  { "𝜦", "𝜦", "𝜦", "Λ", "Λ" },
  { "𝜧", "𝜧", "𝜧", "Μ", "Μ" },
  { "𝜨", "𝜨", "𝜨", "Ν", "Ν" },
  { "𝜩", "𝜩", "𝜩", "Ξ", "Ξ" },
  { "𝜪", "𝜪", "𝜪", "Ο", "Ο" },
  { "𝜫", "𝜫", "𝜫", "Π", "Π" },
  { "𝜬", "𝜬", "𝜬", "Ρ", "Ρ" },
  { "𝜭", "𝜭", "𝜭", "Θ", "Θ" },
  { "𝜮", "𝜮", "𝜮", "Σ", "Σ" },
  { "𝜯", "𝜯", "𝜯", "Τ", "Τ" },
  { "𝜰", "𝜰", "𝜰", "Υ", "Υ" },
  { "𝜱", "𝜱", "𝜱", "Φ", "Φ" },
  { "𝜲", "𝜲", "𝜲", "Χ", "Χ" },
  { "𝜳", "𝜳", "𝜳", "Ψ", "Ψ" },
  { "𝜴", "𝜴", "𝜴", "Ω", "Ω" },
  { "𝜵", "𝜵", "𝜵", "∇", "∇" },
  { "𝜶", "𝜶", "𝜶", "α", "α" },
  { "𝜷", "𝜷", "𝜷", "β", "β" },
  { "𝜸", "𝜸", "𝜸", "γ", "γ" },
  { "𝜹", "𝜹", "𝜹", "δ", "δ" },
  { "𝜺", "𝜺", "𝜺", "ε", "ε" },
  { "𝜻", "𝜻", "𝜻", "ζ", "ζ" },
  { "𝜼", "𝜼", "𝜼", "η", "η" },
  { "𝜽", "𝜽", "𝜽", "θ", "θ" },
  { "𝜾", "𝜾", "𝜾", "ι", "ι" },
  { "𝜿", "𝜿", "𝜿", "κ", "κ" },
  { "𝝀", "𝝀", "𝝀", "λ", "λ" },
  { "𝝁", "𝝁", "𝝁", "μ", "μ" },
  { "𝝂", "𝝂", "𝝂", "ν", "ν" },
  { "𝝃", "𝝃", "𝝃", "ξ", "ξ" },
  { "𝝄", "𝝄", "𝝄", "ο", "ο" },
  { "𝝅", "𝝅", "𝝅", "π", "π" },
  { "𝝆", "𝝆", "𝝆", "ρ", "ρ" },
  { "𝝇", "𝝇", "𝝇", "ς", "ς" },
  { "𝝈", "𝝈", "𝝈", "σ", "σ" },
  { "𝝉", "𝝉", "𝝉", "τ", "τ" },
  { "𝝊", "𝝊", "𝝊", "υ", "υ" },
  { "𝝋", "𝝋", "𝝋", "φ", "φ" },
  { "𝝌", "𝝌", "𝝌", "χ", "χ" },
  { "𝝍", "𝝍", "𝝍", "ψ", "ψ" },
  { "𝝎", "𝝎", "𝝎", "ω", "ω" },
  { "𝝏", "𝝏", "𝝏", "∂", "∂" },
  { "𝝐", "𝝐", "𝝐", "ε", "ε" },
  { "𝝑", "𝝑", "𝝑", "θ", "θ" },
  { "𝝒", "𝝒", "𝝒", "κ", "κ" },
  { "𝝓", "𝝓", "𝝓", "φ", "φ" },
  { "𝝔", "𝝔", "𝝔", "ρ", "ρ" },
  { "𝝕", "𝝕", "𝝕", "π", "π" },
  { "𝝖", "𝝖", "𝝖", "Α", "Α" },
  { "𝝗", "𝝗", "𝝗", "Β", "Β" },
  { "𝝘", "𝝘", "𝝘", "Γ", "Γ" },
  { "𝝙", "𝝙", "𝝙", "Δ", "Δ" },
  { "𝝚", "𝝚", "𝝚", "Ε", "Ε" },
  { "𝝛", "𝝛", "𝝛", "Ζ", "Ζ" },
  { "𝝜", "𝝜", "𝝜", "Η", "Η" },
  { "𝝝", "𝝝", "𝝝", "Θ", "Θ" },
  { "𝝞", "𝝞", "𝝞", "Ι", "Ι" },
  { "𝝟", "𝝟", "𝝟", "Κ", "Κ" },
  { "𝝠", "𝝠", "𝝠", "Λ", "Λ" },
  { "𝝡", "𝝡", "𝝡", "Μ", "Μ" },
  { "𝝢", "𝝢", "𝝢", "Ν", "Ν" },
  { "𝝣", "𝝣", "𝝣", "Ξ", "Ξ" },
  { "𝝤", "𝝤", "𝝤", "Ο", "Ο" },
  { "𝝥", "𝝥", "𝝥", "Π", "Π" },
  { "𝝦", "𝝦", "𝝦", "Ρ", "Ρ" },
  { "𝝧", "𝝧", "𝝧", "Θ", "Θ" },
  { "𝝨", "𝝨", "𝝨", "Σ", "Σ" },
  { "𝝩", "𝝩", "𝝩", "Τ", "Τ" },
  { "𝝪", "𝝪", "𝝪", "Υ", "Υ" },
  { "𝝫", "𝝫", "𝝫", "Φ", "Φ" },
  { "𝝬", "𝝬", "𝝬", "Χ", "Χ" },
  { "𝝭", "𝝭", "𝝭", "Ψ", "Ψ" },
  { "𝝮", "𝝮", "𝝮", "Ω", "Ω" },
  { "𝝯", "𝝯", "𝝯", "∇", "∇" },
  { "𝝰", "𝝰", "𝝰", "α", "α" },
  { "𝝱", "𝝱", "𝝱", "β", "β" },
  { "𝝲", "𝝲", "𝝲", "γ", "γ" },
  { "𝝳", "𝝳", "𝝳", "δ", "δ" },
  { "𝝴", "𝝴", "𝝴", "ε", "ε" },
  { "𝝵", "𝝵", "𝝵", "ζ", "ζ" },
  { "𝝶", "𝝶", "𝝶", "η", "η" },
  { "𝝷", "𝝷", "𝝷", "θ", "θ" },
  { "𝝸", "𝝸", "𝝸", "ι", "ι" },
  { "𝝹", "𝝹", "𝝹", "κ", "κ" },
  { "𝝺", "𝝺", "𝝺", "λ", "λ" },
  { "𝝻", "𝝻", "𝝻", "μ", "μ" },
  { "𝝼", "𝝼", "𝝼", "ν", "ν" },
  { "𝝽", "𝝽", "𝝽", "ξ", "ξ" },
  { "𝝾", "𝝾", "𝝾", "ο", "ο" },
  { "𝝿", "𝝿", "𝝿", "π", "π" },
  { "𝞀", "𝞀", "𝞀", "ρ", "ρ" },
  { "𝞁", "𝞁", "𝞁", "ς", "ς" },
  { "𝞂", "𝞂", "𝞂", "σ", "σ" },
  { "𝞃", "𝞃", "𝞃", "τ", "τ" },
  { "𝞄", "𝞄", "𝞄", "υ", "υ" },
  { "𝞅", "𝞅", "𝞅", "φ", "φ" },
  { "𝞆", "𝞆", "𝞆", "χ", "χ" },
  { "𝞇", "𝞇", "𝞇", "ψ", "ψ" },
  { "𝞈", "𝞈", "𝞈", "ω", "ω" },
  { "𝞉", "𝞉", "𝞉", "∂", "∂" },
  { "𝞊", "𝞊", "𝞊", "ε", "ε" },
  { "𝞋", "𝞋", "𝞋", "θ", "θ" },
  { "𝞌", "𝞌", "𝞌", "κ", "κ" },
  { "𝞍", "𝞍", "𝞍", "φ", "φ" },
  { "𝞎", "𝞎", "𝞎", "ρ", "ρ" },
  { "𝞏", "𝞏", "𝞏", "π", "π" },
  { "𝞐", "𝞐", "𝞐", "Α", "Α" },
  { "𝞑", "𝞑", "𝞑", "Β", "Β" },
  { "𝞒", "𝞒", "𝞒", "Γ", "Γ" },
  { "𝞓", "𝞓", "𝞓", "Δ", "Δ" },
  { "𝞔", "𝞔", "𝞔", "Ε", "Ε" },
  { "𝞕", "𝞕", "𝞕", "Ζ", "Ζ" },
  { "𝞖", "𝞖", "𝞖", "Η", "Η" },
  { "𝞗", "𝞗", "𝞗", "Θ", "Θ" },
  { "𝞘", "𝞘", "𝞘", "Ι", "Ι" },
  { "𝞙", "𝞙", "𝞙", "Κ", "Κ" },
  { "𝞚", "𝞚", "𝞚", "Λ", "Λ" },
  { "𝞛", "𝞛", "𝞛", "Μ", "Μ" },
  { "𝞜", "𝞜", "𝞜", "Ν", "Ν" },
  { "𝞝", "𝞝", "𝞝", "Ξ", "Ξ" },
  { "𝞞", "𝞞", "𝞞", "Ο", "Ο" },
  { "𝞟", "𝞟", "𝞟", "Π", "Π" },
  { "𝞠", "𝞠", "𝞠", "Ρ", "Ρ" },
  { "𝞡", "𝞡", "𝞡", "Θ", "Θ" },
  { "𝞢", "𝞢", "𝞢", "Σ", "Σ" },
  { "𝞣", "𝞣", "𝞣", "Τ", "Τ" },
  { "𝞤", "𝞤", "𝞤", "Υ", "Υ" },
  { "𝞥", "𝞥", "𝞥", "Φ", "Φ" },
  { "𝞦", "𝞦", "𝞦", "Χ", "Χ" },
  { "𝞧", "𝞧", "𝞧", "Ψ", "Ψ" },
  { "𝞨", "𝞨", "𝞨", "Ω", "Ω" },
  { "𝞩", "𝞩", "𝞩", "∇", "∇" },
  { "𝞪", "𝞪", "𝞪", "α", "α" },
  { "𝞫", "𝞫", "𝞫", "β", "β" },
  { "𝞬", "𝞬", "𝞬", "γ", "γ" },
  { "𝞭", "𝞭", "𝞭", "δ", "δ" },
  { "𝞮", "𝞮", "𝞮", "ε", "ε" },
  { "𝞯", "𝞯", "𝞯", "ζ", "ζ" },
  { "𝞰", "𝞰", "𝞰", "η", "η" },
  { "𝞱", "𝞱", "𝞱", "θ", "θ" },
  { "𝞲", "𝞲", "𝞲", "ι", "ι" },
  { "𝞳", "𝞳", "𝞳", "κ", "κ" },
  { "𝞴", "𝞴", "𝞴", "λ", "λ" },
  { "𝞵", "𝞵", "𝞵", "μ", "μ" },
  { "𝞶", "𝞶", "𝞶", "ν", "ν" },
  { "𝞷", "𝞷", "𝞷", "ξ", "ξ" },
  { "𝞸", "𝞸", "𝞸", "ο", "ο" },
  { "𝞹", "𝞹", "𝞹", "π", "π" },
  { "𝞺", "𝞺", "𝞺", "ρ", "ρ" },
  { "𝞻", "𝞻", "𝞻", "ς", "ς" },
  { "𝞼", "𝞼", "𝞼", "σ", "σ" },
  { "𝞽", "𝞽", "𝞽", "τ", "τ" },
  { "𝞾", "𝞾", "𝞾", "υ", "υ" },
  { "𝞿", "𝞿", "𝞿", "φ", "φ" },
  { "𝟀", "𝟀", "𝟀", "χ", "χ" },
  { "𝟁", "𝟁", "𝟁", "ψ", "ψ" },
  { "𝟂", "𝟂", "𝟂", "ω", "ω" },
  { "𝟃", "𝟃", "𝟃", "∂", "∂" },
  { "𝟄", "𝟄", "𝟄", "ε", "ε" },
  { "𝟅", "𝟅", "𝟅", "θ", "θ" },
  { "𝟆", "𝟆", "𝟆", "κ", "κ" },
  { "𝟇", "𝟇", "𝟇", "φ", "φ" },
  { "𝟈", "𝟈", "𝟈", "ρ", "ρ" },
  { "𝟉", "𝟉", "𝟉", "π", "π" },
  { "𝟊", "𝟊", "𝟊", "Ϝ", "Ϝ" },
  { "𝟋", "𝟋", "𝟋", "ϝ", "ϝ" },
  { "𝟎", "𝟎", "𝟎", "0", "0" },
  { "𝟏", "𝟏", "𝟏", "1", "1" },
  { "𝟐", "𝟐", "𝟐", "2", "2" },
  { "𝟑", "𝟑", "𝟑", "3", "3" },
  { "𝟒", "𝟒", "𝟒", "4", "4" },
  { "𝟓", "𝟓", "𝟓", "5", "5" },
  { "𝟔", "𝟔", "𝟔", "6", "6" },
  { "𝟕", "𝟕", "𝟕", "7", "7" },
  { "𝟖", "𝟖", "𝟖", "8", "8" },
  { "𝟗", "𝟗", "𝟗", "9", "9" },
  { "𝟘", "𝟘", "𝟘", "0", "0" },
  { "𝟙", "𝟙", "𝟙", "1", "1" },
  { "𝟚", "𝟚", "𝟚", "2", "2" },
  { "𝟛", "𝟛", "𝟛", "3", "3" },
  { "𝟜", "𝟜", "𝟜", "4", "4" },
  { "𝟝", "𝟝", "𝟝", "5", "5" },
  { "𝟞", "𝟞", "𝟞", "6", "6" },
  { "𝟟", "𝟟", "𝟟", "7", "7" },
  { "𝟠", "𝟠", "𝟠", "8", "8" },
  { "𝟡", "𝟡", "𝟡", "9", "9" },
  { "𝟢", "𝟢", "𝟢", "0", "0" },
  { "𝟣", "𝟣", "𝟣", "1", "1" },
  { "𝟤", "𝟤", "𝟤", "2", "2" },
  { "𝟥", "𝟥", "𝟥", "3", "3" },
  { "𝟦", "𝟦", "𝟦", "4", "4" },
  { "𝟧", "𝟧", "𝟧", "5", "5" },
  { "𝟨", "𝟨", "𝟨", "6", "6" },
  { "𝟩", "𝟩", "𝟩", "7", "7" },
  { "𝟪", "𝟪", "𝟪", "8", "8" },
  { "𝟫", "𝟫", "𝟫", "9", "9" },
  { "𝟬", "𝟬", "𝟬", "0", "0" },
  { "𝟭", "𝟭", "𝟭", "1", "1" },
  { "𝟮", "𝟮", "𝟮", "2", "2" },
  { "𝟯", "𝟯", "𝟯", "3", "3" },
  { "𝟰", "𝟰", "𝟰", "4", "4" },
  { "𝟱", "𝟱", "𝟱", "5", "5" },
  { "𝟲", "𝟲", "𝟲", "6", "6" },
  { "𝟳", "𝟳", "𝟳", "7", "7" },
  { "𝟴", "𝟴", "𝟴", "8", "8" },
  { "𝟵", "𝟵", "𝟵", "9", "9" },
  { "𝟶", "𝟶", "𝟶", "0", "0" },
  { "𝟷", "𝟷", "𝟷", "1", "1" },
  { "𝟸", "𝟸", "𝟸", "2", "2" },
  { "𝟹", "𝟹", "𝟹", "3", "3" },
  { "𝟺", "𝟺", "𝟺", "4", "4" },
  { "𝟻", "𝟻", "𝟻", "5", "5" },
  { "𝟼", "𝟼", "𝟼", "6", "6" },
  { "𝟽", "𝟽", "𝟽", "7", "7" },
  { "𝟾", "𝟾", "𝟾", "8", "8" },
  { "𝟿", "𝟿", "𝟿", "9", "9" },
  { "𞸀", "𞸀", "𞸀", "ا", "ا" },
  { "𞸁", "𞸁", "𞸁", "ب", "ب" },
  { "𞸂", "𞸂", "𞸂", "ج", "ج" },
  { "𞸃", "𞸃", "𞸃", "د", "د" },
  { "𞸅", "𞸅", "𞸅", "و", "و" },
  { "𞸆", "𞸆", "𞸆", "ز", "ز" },
  { "𞸇", "𞸇", "𞸇", "ح", "ح" },
  { "𞸈", "𞸈", "𞸈", "ط", "ط" },
  { "𞸉", "𞸉", "𞸉", "ي", "ي" },
  { "𞸊", "𞸊", "𞸊", "ك", "ك" },
  { "𞸋", "𞸋", "𞸋", "ل", "ل" },
  { "𞸌", "𞸌", "𞸌", "م", "م" },
  { "𞸍", "𞸍", "𞸍", "ن", "ن" },
  { "𞸎", "𞸎", "𞸎", "س", "س" },
  { "𞸏", "𞸏", "𞸏", "ع", "ع" },
  { "𞸐", "𞸐", "𞸐", "ف", "ف" },
  { "𞸑", "𞸑", "𞸑", "ص", "ص" },
  { "𞸒", "𞸒", "𞸒", "ق", "ق" },
  { "𞸓", "𞸓", "𞸓", "ر", "ر" },
  { "𞸔", "𞸔", "𞸔", "ش", "ش" },
  { "𞸕", "𞸕", "𞸕", "ت", "ت" },
  { "𞸖", "𞸖", "𞸖", "ث", "ث" },
  { "𞸗", "𞸗", "𞸗", "خ", "خ" },
  { "𞸘", "𞸘", "𞸘", "ذ", "ذ" },
  { "𞸙", "𞸙", "𞸙", "ض", "ض" },
  { "𞸚", "𞸚", "𞸚", "ظ", "ظ" },
  { "𞸛", "𞸛", "𞸛", "غ", "غ" },
  { "𞸜", "𞸜", "𞸜", "ٮ", "ٮ" },
  { "𞸝", "𞸝", "𞸝", "ں", "ں" },
  { "𞸞", "𞸞", "𞸞", "ڡ", "ڡ" },
  { "𞸟", "𞸟", "𞸟", "ٯ", "ٯ" },
  { "𞸡", "𞸡", "𞸡", "ب", "ب" },
  { "𞸢", "𞸢", "𞸢", "ج", "ج" },
  { "𞸤", "𞸤", "𞸤", "ه", "ه" },
  { "𞸧", "𞸧", "𞸧", "ح", "ح" },
  { "𞸩", "𞸩", "𞸩", "ي", "ي" },
  { "𞸪", "𞸪", "𞸪", "ك", "ك" },
  { "𞸫", "𞸫", "𞸫", "ل", "ل" },
  { "𞸬", "𞸬", "𞸬", "م", "م" },
  { "𞸭", "𞸭", "𞸭", "ن", "ن" },
  { "𞸮", "𞸮", "𞸮", "س", "س" },
  { "𞸯", "𞸯", "𞸯", "ع", "ع" },
  { "𞸰", "𞸰", "𞸰", "ف", "ف" },
  { "𞸱", "𞸱", "𞸱", "ص", "ص" },
  { "𞸲", "𞸲", "𞸲", "ق", "ق" },
  { "𞸴", "𞸴", "𞸴", "ش", "ش" },
  { "𞸵", "𞸵", "𞸵", "ت", "ت" },
  { "𞸶", "𞸶", "𞸶", "ث", "ث" },
  { "𞸷", "𞸷", "𞸷", "خ", "خ" },
  { "𞸹", "𞸹", "𞸹", "ض", "ض" },
  { "𞸻", "𞸻", "𞸻", "غ", "غ" },
  { "𞹂", "𞹂", "𞹂", "ج", "ج" },
  { "𞹇", "𞹇", "𞹇", "ح", "ح" },
  { "𞹉", "𞹉", "𞹉", "ي", "ي" },
  { "𞹋", "𞹋", "𞹋", "ل", "ل" },
  { "𞹍", "𞹍", "𞹍", "ن", "ن" },
  { "𞹎", "𞹎", "𞹎", "س", "س" },
  { "𞹏", "𞹏", "𞹏", "ع", "ع" },
  { "𞹑", "𞹑", "𞹑", "ص", "ص" },
  { "𞹒", "𞹒", "𞹒", "ق", "ق" },
  { "𞹔", "𞹔", "𞹔", "ش", "ش" },
  { "𞹗", "𞹗", "𞹗", "خ", "خ" },
  { "𞹙", "𞹙", "𞹙", "ض", "ض" },
  { "𞹛", "𞹛", "𞹛", "غ", "غ" },
  { "𞹝", "𞹝", "𞹝", "ں", "ں" },
  { "𞹟", "𞹟", "𞹟", "ٯ", "ٯ" },
  { "𞹡", "𞹡", "𞹡", "ب", "ب" },
  { "𞹢", "𞹢", "𞹢", "ج", "ج" },
  { "𞹤", "𞹤", "𞹤", "ه", "ه" },
  { "𞹧", "𞹧", "𞹧", "ح", "ح" },
  { "𞹨", "𞹨", "𞹨", "ط", "ط" },
  { "𞹩", "𞹩", "𞹩", "ي", "ي" },
  { "𞹪", "𞹪", "𞹪", "ك", "ك" },
  { "𞹬", "𞹬", "𞹬", "م", "م" },
  { "𞹭", "𞹭", "𞹭", "ن", "ن" },
  { "𞹮", "𞹮", "𞹮", "س", "س" },
  { "𞹯", "𞹯", "𞹯", "ع", "ع" },
  { "𞹰", "𞹰", "𞹰", "ف", "ف" },
  { "𞹱", "𞹱", "𞹱", "ص", "ص" },
  { "𞹲", "𞹲", "𞹲", "ق", "ق" },
  { "𞹴", "𞹴", "𞹴", "ش", "ش" },
  { "𞹵", "𞹵", "𞹵", "ت", "ت" },
  { "𞹶", "𞹶", "𞹶", "ث", "ث" },
  { "𞹷", "𞹷", "𞹷", "خ", "خ" },
  { "𞹹", "𞹹", "𞹹", "ض", "ض" },
  { "𞹺", "𞹺", "𞹺", "ظ", "ظ" },
  { "𞹻", "𞹻", "𞹻", "غ", "غ" },
  { "𞹼", "𞹼", "𞹼", "ٮ", "ٮ" },
  { "𞹾", "𞹾", "𞹾", "ڡ", "ڡ" },
  { "𞺀", "𞺀", "𞺀", "ا", "ا" },
  { "𞺁", "𞺁", "𞺁", "ب", "ب" },
  { "𞺂", "𞺂", "𞺂", "ج", "ج" },
  { "𞺃", "𞺃", "𞺃", "د", "د" },
  { "𞺄", "𞺄", "𞺄", "ه", "ه" },
  { "𞺅", "𞺅", "𞺅", "و", "و" },
  { "𞺆", "𞺆", "𞺆", "ز", "ز" },
  { "𞺇", "𞺇", "𞺇", "ح", "ح" },
  { "𞺈", "𞺈", "𞺈", "ط", "ط" },
  { "𞺉", "𞺉", "𞺉", "ي", "ي" },
  { "𞺋", "𞺋", "𞺋", "ل", "ل" },
  { "𞺌", "𞺌", "𞺌", "م", "م" },
  { "𞺍", "𞺍", "𞺍", "ن", "ن" },
  { "𞺎", "𞺎", "𞺎", "س", "س" },
  { "𞺏", "𞺏", "𞺏", "ع", "ع" },
  { "𞺐", "𞺐", "𞺐", "ف", "ف" },
  { "𞺑", "𞺑", "𞺑", "ص", "ص" },
  { "𞺒", "𞺒", "𞺒", "ق", "ق" },
  { "𞺓", "𞺓", "𞺓", "ر", "ر" },
  { "𞺔", "𞺔", "𞺔", "ش", "ش" },
  { "𞺕", "𞺕", "𞺕", "ت", "ت" },
  { "𞺖", "𞺖", "𞺖", "ث", "ث" },
  { "𞺗", "𞺗", "𞺗", "خ", "خ" },
  { "𞺘", "𞺘", "𞺘", "ذ", "ذ" },
  { "𞺙", "𞺙", "𞺙", "ض", "ض" },
  { "𞺚", "𞺚", "𞺚", "ظ", "ظ" },
  { "𞺛", "𞺛", "𞺛", "غ", "غ" },
  { "𞺡", "𞺡", "𞺡", "ب", "ب" },
  { "𞺢", "𞺢", "𞺢", "ج", "ج" },
  { "𞺣", "𞺣", "𞺣", "د", "د" },
  { "𞺥", "𞺥", "𞺥", "و", "و" },
  { "𞺦", "𞺦", "𞺦", "ز", "ز" },
  { "𞺧", "𞺧", "𞺧", "ح", "ح" },
  { "𞺨", "𞺨", "𞺨", "ط", "ط" },
  { "𞺩", "𞺩", "𞺩", "ي", "ي" },
  { "𞺫", "𞺫", "𞺫", "ل", "ل" },
  { "𞺬", "𞺬", "𞺬", "م", "م" },
  { "𞺭", "𞺭", "𞺭", "ن", "ن" },
  { "𞺮", "𞺮", "𞺮", "س", "س" },
  { "𞺯", "𞺯", "𞺯", "ع", "ع" },
  { "𞺰", "𞺰", "𞺰", "ف", "ف" },
  { "𞺱", "𞺱", "𞺱", "ص", "ص" },
  { "𞺲", "𞺲", "𞺲", "ق", "ق" },
  { "𞺳", "𞺳", "𞺳", "ر", "ر" },
  { "𞺴", "𞺴", "𞺴", "ش", "ش" },
  { "𞺵", "𞺵", "𞺵", "ت", "ت" },
  { "𞺶", "𞺶", "𞺶", "ث", "ث" },
  { "𞺷", "𞺷", "𞺷", "خ", "خ" },
  { "𞺸", "𞺸", "𞺸", "ذ", "ذ" },
  { "𞺹", "𞺹", "𞺹", "ض", "ض" },
  { "𞺺", "𞺺", "𞺺", "ظ", "ظ" },
  { "𞺻", "𞺻", "𞺻", "غ", "غ" },
  { "🄀", "🄀", "🄀", "0.", "0." },
  { "🄁", "🄁", "🄁", "0,", "0," },
  { "🄂", "🄂", "🄂", "1,", "1," },
  { "🄃", "🄃", "🄃", "2,", "2," },
  { "🄄", "🄄", "🄄", "3,", "3," },
  { "🄅", "🄅", "🄅", "4,", "4," },
  { "🄆", "🄆", "🄆", "5,", "5," },
  { "🄇", "🄇", "🄇", "6,", "6," },
  { "🄈", "🄈", "🄈", "7,", "7," },
  { "🄉", "🄉", "🄉", "8,", "8," },
  { "🄊", "🄊", "🄊", "9,", "9," },
  { "🄐", "🄐", "🄐", "(A)", "(A)" },
  { "🄑", "🄑", "🄑", "(B)", "(B)" },
  { "🄒", "🄒", "🄒", "(C)", "(C)" },
  { "🄓", "🄓", "🄓", "(D)", "(D)" },
  { "🄔", "🄔", "🄔", "(E)", "(E)" },
  { "🄕", "🄕", "🄕", "(F)", "(F)" },
  { "🄖", "🄖", "🄖", "(G)", "(G)" },
  { "🄗", "🄗", "🄗", "(H)", "(H)" },
  { "🄘", "🄘", "🄘", "(I)", "(I)" },
  { "🄙", "🄙", "🄙", "(J)", "(J)" },
  { "🄚", "🄚", "🄚", "(K)", "(K)" },
  { "🄛", "🄛", "🄛", "(L)", "(L)" },
  { "🄜", "🄜", "🄜", "(M)", "(M)" },
  { "🄝", "🄝", "🄝", "(N)", "(N)" },
  { "🄞", "🄞", "🄞", "(O)", "(O)" },
  { "🄟", "🄟", "🄟", "(P)", "(P)" },
  { "🄠", "🄠", "🄠", "(Q)", "(Q)" },
  { "🄡", "🄡", "🄡", "(R)", "(R)" },
  { "🄢", "🄢", "🄢", "(S)", "(S)" },
  { "🄣", "🄣", "🄣", "(T)", "(T)" },
  { "🄤", "🄤", "🄤", "(U)", "(U)" },
  { "🄥", "🄥", "🄥", "(V)", "(V)" },
  { "🄦", "🄦", "🄦", "(W)", "(W)" },
  { "🄧", "🄧", "🄧", "(X)", "(X)" },
  { "🄨", "🄨", "🄨", "(Y)", "(Y)" },
  { "🄩", "🄩", "🄩", "(Z)", "(Z)" },
  { "🄪", "🄪", "🄪", "〔S〕", "〔S〕" },
  { "🄫", "🄫", "🄫", "C", "C" },
  { "🄬", "🄬", "🄬", "R", "R" },
  { "🄭", "🄭", "🄭", "CD", "CD" },
  { "🄮", "🄮", "🄮", "WZ", "WZ" },
  { "🄰", "🄰", "🄰", "A", "A" },
  { "🄱", "🄱", "🄱", "B", "B" },
  { "🄲", "🄲", "🄲", "C", "C" },
  { "🄳", "🄳", "🄳", "D", "D" },
  { "🄴", "🄴", "🄴", "E", "E" },
  { "🄵", "🄵", "🄵", "F", "F" },
  { "🄶", "🄶", "🄶", "G", "G" },
  { "🄷", "🄷", "🄷", "H", "H" },
  { "🄸", "🄸", "🄸", "I", "I" },
  { "🄹", "🄹", "🄹", "J", "J" },
  { "🄺", "🄺", "🄺", "K", "K" },
  { "🄻", "🄻", "🄻", "L", "L" },
  { "🄼", "🄼", "🄼", "M", "M" },
  { "🄽", "🄽", "🄽", "N", "N" },
  { "🄾", "🄾", "🄾", "O", "O" },
  { "🄿", "🄿", "🄿", "P", "P" },
  { "🅀", "🅀", "🅀", "Q", "Q" },
  { "🅁", "🅁", "🅁", "R", "R" },
  { "🅂", "🅂", "🅂", "S", "S" },
  { "🅃", "🅃", "🅃", "T", "T" },
  { "🅄", "🅄", "🅄", "U", "U" },
  { "🅅", "🅅", "🅅", "V", "V" },
  { "🅆", "🅆", "🅆", "W", "W" },
  { "🅇", "🅇", "🅇", "X", "X" },
  { "🅈", "🅈", "🅈", "Y", "Y" },
  { "🅉", "🅉", "🅉", "Z", "Z" },
  { "🅊", "🅊", "🅊", "HV", "HV" },
  { "🅋", "🅋", "🅋", "MV", "MV" },
  { "🅌", "🅌", "🅌", "SD", "SD" },
  { "🅍", "🅍", "🅍", "SS", "SS" },
  { "🅎", "🅎", "🅎", "PPV", "PPV" },
  { "🅏", "🅏", "🅏", "WC", "WC" },
  { "🅪", "🅪", "🅪", "MC", "MC" },
  { "🅫", "🅫", "🅫", "MD", "MD" },
  { "🅬", "🅬", "🅬", "MR", "MR" },
  { "🆐", "🆐", "🆐", "DJ", "DJ" },
  { "🈀", "🈀", "🈀", "ほか", "ほか" },
  { "🈁", "🈁", "🈁", "ココ", "ココ" },
  { "🈂", "🈂", "🈂", "サ", "サ" },
  { "🈐", "🈐", "🈐", "手", "手" },
  { "🈑", "🈑", "🈑", "字", "字" },
  { "🈒", "🈒", "🈒", "双", "双" },
  { "🈓", "🈓", "🈓", "デ", "デ" },
  { "🈔", "🈔", "🈔", "二", "二" },
  { "🈕", "🈕", "🈕", "多", "多" },
  { "🈖", "🈖", "🈖", "解", "解" },
  { "🈗", "🈗", "🈗", "天", "天" },
  { "🈘", "🈘", "🈘", "交", "交" },
  { "🈙", "🈙", "🈙", "映", "映" },
  { "🈚", "🈚", "🈚", "無", "無" },
  { "🈛", "🈛", "🈛", "料", "料" },
  { "🈜", "🈜", "🈜", "前", "前" },
  { "🈝", "🈝", "🈝", "後", "後" },
  { "🈞", "🈞", "🈞", "再", "再" },
  { "🈟", "🈟", "🈟", "新", "新" },
  { "🈠", "🈠", "🈠", "初", "初" },
  { "🈡", "🈡", "🈡", "終", "終" },
  { "🈢", "🈢", "🈢", "生", "生" },
  { "🈣", "🈣", "🈣", "販", "販" },
  { "🈤", "🈤", "🈤", "声", "声" },
  { "🈥", "🈥", "🈥", "吹", "吹" },
  { "🈦", "🈦", "🈦", "演", "演" },
  { "🈧", "🈧", "🈧", "投", "投" },
  { "🈨", "🈨", "🈨", "捕", "捕" },
  { "🈩", "🈩", "🈩", "一", "一" },
  { "🈪", "🈪", "🈪", "三", "三" },
  { "🈫", "🈫", "🈫", "遊", "遊" },
  { "🈬", "🈬", "🈬", "左", "左" },
  { "🈭", "🈭", "🈭", "中", "中" },
  { "🈮", "🈮", "🈮", "右", "右" },
  { "🈯", "🈯", "🈯", "指", "指" },
  { "🈰", "🈰", "🈰", "走", "走" },
  { "🈱", "🈱", "🈱", "打", "打" },
  { "🈲", "🈲", "🈲", "禁", "禁" },
  { "🈳", "🈳", "🈳", "空", "空" },
  { "🈴", "🈴", "🈴", "合", "合" },
  { "🈵", "🈵", "🈵", "満", "満" },
  { "🈶", "🈶", "🈶", "有", "有" },
  { "🈷", "🈷", "🈷", "月", "月" },
  { "🈸", "🈸", "🈸", "申", "申" },
  { "🈹", "🈹", "🈹", "割", "割" },
  { "🈺", "🈺", "🈺", "営", "営" },
  { "🈻", "🈻", "🈻", "配", "配" },
  { "🉀", "🉀", "🉀", "〔本〕", "〔本〕" },
  { "🉁", "🉁", "🉁", "〔三〕", "〔三〕" },
  { "🉂", "🉂", "🉂", "〔二〕", "〔二〕" },
  { "🉃", "🉃", "🉃", "〔安〕", "〔安〕" },
  { "🉄", "🉄", "🉄", "〔点〕", "〔点〕" },
  { "🉅", "🉅", "🉅", "〔打〕", "〔打〕" },
  { "🉆", "🉆", "🉆", "〔盗〕", "〔盗〕" },
  { "🉇", "🉇", "🉇", "〔勝〕", "〔勝〕" },
  { "🉈", "🉈", "🉈", "〔敗〕", "〔敗〕" },
  { "🉐", "🉐", "🉐", "得", "得" },
  { "🉑", "🉑", "🉑", "可", "可" },
  { "丽", "丽", "丽", "丽", "丽" },
  { "丸", "丸", "丸", "丸", "丸" },
  { "乁", "乁", "乁", "乁", "乁" },
  { "𠄢", "𠄢", "𠄢", "𠄢", "𠄢" },
  { "你", "你", "你", "你", "你" },
  { "侮", "侮", "侮", "侮", "侮" },
  { "侻", "侻", "侻", "侻", "侻" },
  { "倂", "倂", "倂", "倂", "倂" },
  { "偺", "偺", "偺", "偺", "偺" },
  { "備", "備", "備", "備", "備" },
  { "僧", "僧", "僧", "僧", "僧" },
  { "像", "像", "像", "像", "像" },
  { "㒞", "㒞", "㒞", "㒞", "㒞" },
  { "𠘺", "𠘺", "𠘺", "𠘺", "𠘺" },
  { "免", "免", "免", "免", "免" },
  { "兔", "兔", "兔", "兔", "兔" },
  { "兤", "兤", "兤", "兤", "兤" },
  { "具", "具", "具", "具", "具" },
  { "𠔜", "𠔜", "𠔜", "𠔜", "𠔜" },
  { "㒹", "㒹", "㒹", "㒹", "㒹" },
  { "內", "內", "內", "內", "內" },
  { "再", "再", "再", "再", "再" },
  { "𠕋", "𠕋", "𠕋", "𠕋", "𠕋" },
  { "冗", "冗", "冗", "冗", "冗" },
  { "冤", "冤", "冤", "冤", "冤" },
  { "仌", "仌", "仌", "仌", "仌" },
  { "冬", "冬", "冬", "冬", "冬" },
  { "况", "况", "况", "况", "况" },
  { "𩇟", "𩇟", "𩇟", "𩇟", "𩇟" },
  { "凵", "凵", "凵", "凵", "凵" },
  { "刃", "刃", "刃", "刃", "刃" },
  { "㓟", "㓟", "㓟", "㓟", "㓟" },
  { "刻", "刻", "刻", "刻", "刻" },
  { "剆", "剆", "剆", "剆", "剆" },
  { "割", "割", "割", "割", "割" },
  { "剷", "剷", "剷", "剷", "剷" },
  { "㔕", "㔕", "㔕", "㔕", "㔕" },
  { "勇", "勇", "勇", "勇", "勇" },
  { "勉", "勉", "勉", "勉", "勉" },
  { "勤", "勤", "勤", "勤", "勤" },
  { "勺", "勺", "勺", "勺", "勺" },
  { "包", "包", "包", "包", "包" },
  { "匆", "匆", "匆", "匆", "匆" },
  { "北", "北", "北", "北", "北" },
  { "卉", "卉", "卉", "卉", "卉" },
  { "卑", "卑", "卑", "卑", "卑" },
  { "博", "博", "博", "博", "博" },
  { "即", "即", "即", "即", "即" },
  { "卽", "卽", "卽", "卽", "卽" },
  { "卿", "卿", "卿", "卿", "卿" },
  { "卿", "卿", "卿", "卿", "卿" },
  { "卿", "卿", "卿", "卿", "卿" },
  { "𠨬", "𠨬", "𠨬", "𠨬", "𠨬" },
  { "灰", "灰", "灰", "灰", "灰" },
  { "及", "及", "及", "及", "及" },
  { "叟", "叟", "叟", "叟", "叟" },
  { "𠭣", "𠭣", "𠭣", "𠭣", "𠭣" },
  { "叫", "叫", "叫", "叫", "叫" },
  { "叱", "叱", "叱", "叱", "叱" },
  { "吆", "吆", "吆", "吆", "吆" },
  { "咞", "咞", "咞", "咞", "咞" },
  { "吸", "吸", "吸", "吸", "吸" },
  { "呈", "呈", "呈", "呈", "呈" },
  { "周", "周", "周", "周", "周" },
  { "咢", "咢", "咢", "咢", "咢" },
  { "哶", "哶", "哶", "哶", "哶" },
  { "唐", "唐", "唐", "唐", "唐" },
  { "啓", "啓", "啓", "啓", "啓" },
  { "啣", "啣", "啣", "啣", "啣" },
  { "善", "善", "善", "善", "善" },
  { "善", "善", "善", "善", "善" },
  { "喙", "喙", "喙", "喙", "喙" },
  { "喫", "喫", "喫", "喫", "喫" },
  { "喳", "喳", "喳", "喳", "喳" },
  { "嗂", "嗂", "嗂", "嗂", "嗂" },
  { "圖", "圖", "圖", "圖", "圖" },
  { "嘆", "嘆", "嘆", "嘆", "嘆" },
  { "圗", "圗", "圗", "圗", "圗" },
  { "噑", "噑", "噑", "噑", "噑" },
  { "噴", "噴", "噴", "噴", "噴" },
  { "切", "切", "切", "切", "切" },
  { "壮", "壮", "壮", "壮", "壮" },
  { "城", "城", "城", "城", "城" },
  { "埴", "埴", "埴", "埴", "埴" },
  { "堍", "堍", "堍", "堍", "堍" },
  { "型", "型", "型", "型", "型" },
  { "堲", "堲", "堲", "堲", "堲" },
  { "報", "報", "報", "報", "報" },
  { "墬", "墬", "墬", "墬", "墬" },
  { "𡓤", "𡓤", "𡓤", "𡓤", "𡓤" },
  { "売", "売", "売", "売", "売" },
  { "壷", "壷", "壷", "壷", "壷" },
  { "夆", "夆", "夆", "夆", "夆" },
  { "多", "多", "多", "多", "多" },
  { "夢", "夢", "夢", "夢", "夢" },
  { "奢", "奢", "奢", "奢", "奢" },
  { "𡚨", "𡚨", "𡚨", "𡚨", "𡚨" },
  { "𡛪", "𡛪", "𡛪", "𡛪", "𡛪" },
  { "姬", "姬", "姬", "姬", "姬" },
  { "娛", "娛", "娛", "娛", "娛" },
  { "娧", "娧", "娧", "娧", "娧" },
  { "姘", "姘", "姘", "姘", "姘" },
  { "婦", "婦", "婦", "婦", "婦" },
  { "㛮", "㛮", "㛮", "㛮", "㛮" },
  { "㛼", "㛼", "㛼", "㛼", "㛼" },
  { "嬈", "嬈", "嬈", "嬈", "嬈" },
  { "嬾", "嬾", "嬾", "嬾", "嬾" },
  { "嬾", "嬾", "嬾", "嬾", "嬾" },
  { "𡧈", "𡧈", "𡧈", "𡧈", "𡧈" },
  { "寃", "寃", "寃", "寃", "寃" },
  { "寘", "寘", "寘", "寘", "寘" },
  { "寧", "寧", "寧", "寧", "寧" },
  { "寳", "寳", "寳", "寳", "寳" },
  { "𡬘", "𡬘", "𡬘", "𡬘", "𡬘" },
  { "寿", "寿", "寿", "寿", "寿" },
  { "将", "将", "将", "将", "将" },
  { "当", "当", "当", "当", "当" },
  { "尢", "尢", "尢", "尢", "尢" },
  { "㞁", "㞁", "㞁", "㞁", "㞁" },
  { "屠", "屠", "屠", "屠", "屠" },
  { "屮", "屮", "屮", "屮", "屮" },
  { "峀", "峀", "峀", "峀", "峀" },
  { "岍", "岍", "岍", "岍", "岍" },
  { "𡷤", "𡷤", "𡷤", "𡷤", "𡷤" },
  { "嵃", "嵃", "嵃", "嵃", "嵃" },
  { "𡷦", "𡷦", "𡷦", "𡷦", "𡷦" },
  { "嵮", "嵮", "嵮", "嵮", "嵮" },
  { "嵫", "嵫", "嵫", "嵫", "嵫" },
  { "嵼", "嵼", "嵼", "嵼", "嵼" },
  { "巡", "巡", "巡", "巡", "巡" },
  { "巢", "巢", "巢", "巢", "巢" },
  { "㠯", "㠯", "㠯", "㠯", "㠯" },
  { "巽", "巽", "巽", "巽", "巽" },
  { "帨", "帨", "帨", "帨", "帨" },
  { "帽", "帽", "帽", "帽", "帽" },
  { "幩", "幩", "幩", "幩", "幩" },
  { "㡢", "㡢", "㡢", "㡢", "㡢" },
  { "𢆃", "𢆃", "𢆃", "𢆃", "𢆃" },
  { "㡼", "㡼", "㡼", "㡼", "㡼" },
  { "庰", "庰", "庰", "庰", "庰" },
  { "庳", "庳", "庳", "庳", "庳" },
  { "庶", "庶", "庶", "庶", "庶" },
  { "廊", "廊", "廊", "廊", "廊" },
  { "𪎒", "𪎒", "𪎒", "𪎒", "𪎒" },
  { "廾", "廾", "廾", "廾", "廾" },
  { "𢌱", "𢌱", "𢌱", "𢌱", "𢌱" },
  { "𢌱", "𢌱", "𢌱", "𢌱", "𢌱" },
  { "舁", "舁", "舁", "舁", "舁" },
  { "弢", "弢", "弢", "弢", "弢" },
  { "弢", "弢", "弢", "弢", "弢" },
  { "㣇", "㣇", "㣇", "㣇", "㣇" },
  { "𣊸", "𣊸", "𣊸", "𣊸", "𣊸" },
  { "𦇚", "𦇚", "𦇚", "𦇚", "𦇚" },
  { "形", "形", "形", "形", "形" },
  { "彫", "彫", "彫", "彫", "彫" },
  { "㣣", "㣣", "㣣", "㣣", "㣣" },
  { "徚", "徚", "徚", "徚", "徚" },
  { "忍", "忍", "忍", "忍", "忍" },
  { "志", "志", "志", "志", "志" },
  { "忹", "忹", "忹", "忹", "忹" },
  { "悁", "悁", "悁", "悁", "悁" },
  { "㤺", "㤺", "㤺", "㤺", "㤺" },
  { "㤜", "㤜", "㤜", "㤜", "㤜" },
  { "悔", "悔", "悔", "悔", "悔" },
  { "𢛔", "𢛔", "𢛔", "𢛔", "𢛔" },
  { "惇", "惇", "惇", "惇", "惇" },
  { "慈", "慈", "慈", "慈", "慈" },
  { "慌", "慌", "慌", "慌", "慌" },
  { "慎", "慎", "慎", "慎", "慎" },
  { "慌", "慌", "慌", "慌", "慌" },
  { "慺", "慺", "慺", "慺", "慺" },
  { "憎", "憎", "憎", "憎", "憎" },
  { "憲", "憲", "憲", "憲", "憲" },
  { "憤", "憤", "憤", "憤", "憤" },
  { "憯", "憯", "憯", "憯", "憯" },
  { "懞", "懞", "懞", "懞", "懞" },
  { "懲", "懲", "懲", "懲", "懲" },
  { "懶", "懶", "懶", "懶", "懶" },
  { "成", "成", "成", "成", "成" },
  { "戛", "戛", "戛", "戛", "戛" },
  { "扝", "扝", "扝", "扝", "扝" },
  { "抱", "抱", "抱", "抱", "抱" },
  { "拔", "拔", "拔", "拔", "拔" },
  { "捐", "捐", "捐", "捐", "捐" },
  { "𢬌", "𢬌", "𢬌", "𢬌", "𢬌" },
  { "挽", "挽", "挽", "挽", "挽" },
  { "拼", "拼", "拼", "拼", "拼" },
  { "捨", "捨", "捨", "捨", "捨" },
  { "掃", "掃", "掃", "掃", "掃" },
  { "揤", "揤", "揤", "揤", "揤" },
  { "𢯱", "𢯱", "𢯱", "𢯱", "𢯱" },
  { "搢", "搢", "搢", "搢", "搢" },
  { "揅", "揅", "揅", "揅", "揅" },
  { "掩", "掩", "掩", "掩", "掩" },
  { "㨮", "㨮", "㨮", "㨮", "㨮" },
  { "摩", "摩", "摩", "摩", "摩" },
  { "摾", "摾", "摾", "摾", "摾" },
  { "撝", "撝", "撝", "撝", "撝" },
  { "摷", "摷", "摷", "摷", "摷" },
  { "㩬", "㩬", "㩬", "㩬", "㩬" },
  { "敏", "敏", "敏", "敏", "敏" },
  { "敬", "敬", "敬", "敬", "敬" },
  { "𣀊", "𣀊", "𣀊", "𣀊", "𣀊" },
  { "旣", "旣", "旣", "旣", "旣" },
  { "書", "書", "書", "書", "書" },
  { "晉", "晉", "晉", "晉", "晉" },
  { "㬙", "㬙", "㬙", "㬙", "㬙" },
  { "暑", "暑", "暑", "暑", "暑" },
  { "㬈", "㬈", "㬈", "㬈", "㬈" },
  { "㫤", "㫤", "㫤", "㫤", "㫤" },
  { "冒", "冒", "冒", "冒", "冒" },
  { "冕", "冕", "冕", "冕", "冕" },
  { "最", "最", "最", "最", "最" },
  { "暜", "暜", "暜", "暜", "暜" },
  { "肭", "肭", "肭", "肭", "肭" },
  { "䏙", "䏙", "䏙", "䏙", "䏙" },
  { "朗", "朗", "朗", "朗", "朗" },
  { "望", "望", "望", "望", "望" },
  { "朡", "朡", "朡", "朡", "朡" },
  { "杞", "杞", "杞", "杞", "杞" },
  { "杓", "杓", "杓", "杓", "杓" },
  { "𣏃", "𣏃", "𣏃", "𣏃", "𣏃" },
  { "㭉", "㭉", "㭉", "㭉", "㭉" },
  { "柺", "柺", "柺", "柺", "柺" },
  { "枅", "枅", "枅", "枅", "枅" },
  { "桒", "桒", "桒", "桒", "桒" },
  { "梅", "梅", "梅", "梅", "梅" },
  { "𣑭", "𣑭", "𣑭", "𣑭", "𣑭" },
  { "梎", "梎", "梎", "梎", "梎" },
  { "栟", "栟", "栟", "栟", "栟" },
  { "椔", "椔", "椔", "椔", "椔" },
  { "㮝", "㮝", "㮝", "㮝", "㮝" },
  { "楂", "楂", "楂", "楂", "楂" },
  { "榣", "榣", "榣", "榣", "榣" },
  { "槪", "槪", "槪", "槪", "槪" },
  { "檨", "檨", "檨", "檨", "檨" },
  { "𣚣", "𣚣", "𣚣", "𣚣", "𣚣" },
  { "櫛", "櫛", "櫛", "櫛", "櫛" },
  { "㰘", "㰘", "㰘", "㰘", "㰘" },
  { "次", "次", "次", "次", "次" },
  { "𣢧", "𣢧", "𣢧", "𣢧", "𣢧" },
  { "歔", "歔", "歔", "歔", "歔" },
  { "㱎", "㱎", "㱎", "㱎", "㱎" },
  { "歲", "歲", "歲", "歲", "歲" },
  { "殟", "殟", "殟", "殟", "殟" },
  { "殺", "殺", "殺", "殺", "殺" },
  { "殻", "殻", "殻", "殻", "殻" },
  { "𣪍", "𣪍", "𣪍", "𣪍", "𣪍" },
  { "𡴋", "𡴋", "𡴋", "𡴋", "𡴋" },
  { "𣫺", "𣫺", "𣫺", "𣫺", "𣫺" },
  { "汎", "汎", "汎", "汎", "汎" },
  { "𣲼", "𣲼", "𣲼", "𣲼", "𣲼" },
  { "沿", "沿", "沿", "沿", "沿" },
  { "泍", "泍", "泍", "泍", "泍" },
  { "汧", "汧", "汧", "汧", "汧" },
  { "洖", "洖", "洖", "洖", "洖" },
  { "派", "派", "派", "派", "派" },
  { "海", "海", "海", "海", "海" },
  { "流", "流", "流", "流", "流" },
  { "浩", "浩", "浩", "浩", "浩" },
  { "浸", "浸", "浸", "浸", "浸" },
  { "涅", "涅", "涅", "涅", "涅" },
  { "𣴞", "𣴞", "𣴞", "𣴞", "𣴞" },
  { "洴", "洴", "洴", "洴", "洴" },
  { "港", "港", "港", "港", "港" },
  { "湮", "湮", "湮", "湮", "湮" },
  { "㴳", "㴳", "㴳", "㴳", "㴳" },
  { "滋", "滋", "滋", "滋", "滋" },
  { "滇", "滇", "滇", "滇", "滇" },
  { "𣻑", "𣻑", "𣻑", "𣻑", "𣻑" },
  { "淹", "淹", "淹", "淹", "淹" },
  { "潮", "潮", "潮", "潮", "潮" },
  { "𣽞", "𣽞", "𣽞", "𣽞", "𣽞" },
  { "𣾎", "𣾎", "𣾎", "𣾎", "𣾎" },
  { "濆", "濆", "濆", "濆", "濆" },
  { "瀹", "瀹", "瀹", "瀹", "瀹" },
  { "瀞", "瀞", "瀞", "瀞", "瀞" },
  { "瀛", "瀛", "瀛", "瀛", "瀛" },
  { "㶖", "㶖", "㶖", "㶖", "㶖" },
  { "灊", "灊", "灊", "灊", "灊" },
  { "災", "災", "災", "災", "災" },
  { "灷", "灷", "灷", "灷", "灷" },
  { "炭", "炭", "炭", "炭", "炭" },
  { "𠔥", "𠔥", "𠔥", "𠔥", "𠔥" },
  { "煅", "煅", "煅", "煅", "煅" },
  { "𤉣", "𤉣", "𤉣", "𤉣", "𤉣" },
  { "熜", "熜", "熜", "熜", "熜" },
  { "𤎫", "𤎫", "𤎫", "𤎫", "𤎫" },
  { "爨", "爨", "爨", "爨", "爨" },
  { "爵", "爵", "爵", "爵", "爵" },
  { "牐", "牐", "牐", "牐", "牐" },
  { "𤘈", "𤘈", "𤘈", "𤘈", "𤘈" },
  { "犀", "犀", "犀", "犀", "犀" },
  { "犕", "犕", "犕", "犕", "犕" },
  { "𤜵", "𤜵", "𤜵", "𤜵", "𤜵" },
  { "𤠔", "𤠔", "𤠔", "𤠔", "𤠔" },
  { "獺", "獺", "獺", "獺", "獺" },
  { "王", "王", "王", "王", "王" },
  { "㺬", "㺬", "㺬", "㺬", "㺬" },
  { "玥", "玥", "玥", "玥", "玥" },
  { "㺸", "㺸", "㺸", "㺸", "㺸" },
  { "㺸", "㺸", "㺸", "㺸", "㺸" },
  { "瑇", "瑇", "瑇", "瑇", "瑇" },
  { "瑜", "瑜", "瑜", "瑜", "瑜" },
  { "瑱", "瑱", "瑱", "瑱", "瑱" },
  { "璅", "璅", "璅", "璅", "璅" },
  { "瓊", "瓊", "瓊", "瓊", "瓊" },
  { "㼛", "㼛", "㼛", "㼛", "㼛" },
  { "甤", "甤", "甤", "甤", "甤" },
  { "𤰶", "𤰶", "𤰶", "𤰶", "𤰶" },
  { "甾", "甾", "甾", "甾", "甾" },
  { "𤲒", "𤲒", "𤲒", "𤲒", "𤲒" },
  { "異", "異", "異", "異", "異" },
  { "𢆟", "𢆟", "𢆟", "𢆟", "𢆟" },
  { "瘐", "瘐", "瘐", "瘐", "瘐" },
  { "𤾡", "𤾡", "𤾡", "𤾡", "𤾡" },
  { "𤾸", "𤾸", "𤾸", "𤾸", "𤾸" },
  { "𥁄", "𥁄", "𥁄", "𥁄", "𥁄" },
  { "㿼", "㿼", "㿼", "㿼", "㿼" },
  { "䀈", "䀈", "䀈", "䀈", "䀈" },
  { "直", "直", "直", "直", "直" },
  { "𥃳", "𥃳", "𥃳", "𥃳", "𥃳" },
  { "𥃲", "𥃲", "𥃲", "𥃲", "𥃲" },
  { "𥄙", "𥄙", "𥄙", "𥄙", "𥄙" },
  { "𥄳", "𥄳", "𥄳", "𥄳", "𥄳" },
  { "眞", "眞", "眞", "眞", "眞" },
  { "真", "真", "真", "真", "真" },
  { "真", "真", "真", "真", "真" },
  { "睊", "睊", "睊", "睊", "睊" },
  { "䀹", "䀹", "䀹", "䀹", "䀹" },
  { "瞋", "瞋", "瞋", "瞋", "瞋" },
  { "䁆", "䁆", "䁆", "䁆", "䁆" },
  { "䂖", "䂖", "䂖", "䂖", "䂖" },
  { "𥐝", "𥐝", "𥐝", "𥐝", "𥐝" },
  { "硎", "硎", "硎", "硎", "硎" },
  { "碌", "碌", "碌", "碌", "碌" },
  { "磌", "磌", "磌", "磌", "磌" },
  { "䃣", "䃣", "䃣", "䃣", "䃣" },
  { "𥘦", "𥘦", "𥘦", "𥘦", "𥘦" },
  { "祖", "祖", "祖", "祖", "祖" },
  { "𥚚", "𥚚", "𥚚", "𥚚", "𥚚" },
  { "𥛅", "𥛅", "𥛅", "𥛅", "𥛅" },
  { "福", "福", "福", "福", "福" },
  { "秫", "秫", "秫", "秫", "秫" },
  { "䄯", "䄯", "䄯", "䄯", "䄯" },
  { "穀", "穀", "穀", "穀", "穀" },
  { "穊", "穊", "穊", "穊", "穊" },
  { "穏", "穏", "穏", "穏", "穏" },
  { "𥥼", "𥥼", "𥥼", "𥥼", "𥥼" },
  { "𥪧", "𥪧", "𥪧", "𥪧", "𥪧" },
  { "𥪧", "𥪧", "𥪧", "𥪧", "𥪧" },
  { "竮", "竮", "竮", "竮", "竮" },
  { "䈂", "䈂", "䈂", "䈂", "䈂" },
  { "𥮫", "𥮫", "𥮫", "𥮫", "𥮫" },
  { "篆", "篆", "篆", "篆", "篆" },
  { "築", "築", "築", "築", "築" },
  { "䈧", "䈧", "䈧", "䈧", "䈧" },
  { "𥲀", "𥲀", "𥲀", "𥲀", "𥲀" },
  { "糒", "糒", "糒", "糒", "糒" },
  { "䊠", "䊠", "䊠", "䊠", "䊠" },
  { "糨", "糨", "糨", "糨", "糨" },
  { "糣", "糣", "糣", "糣", "糣" },
  { "紀", "紀", "紀", "紀", "紀" },
  { "𥾆", "𥾆", "𥾆", "𥾆", "𥾆" },
  { "絣", "絣", "絣", "絣", "絣" },
  { "䌁", "䌁", "䌁", "䌁", "䌁" },
  { "緇", "緇", "緇", "緇", "緇" },
  { "縂", "縂", "縂", "縂", "縂" },
  { "繅", "繅", "繅", "繅", "繅" },
  { "䌴", "䌴", "䌴", "䌴", "䌴" },
  { "𦈨", "𦈨", "𦈨", "𦈨", "𦈨" },
  { "𦉇", "𦉇", "𦉇", "𦉇", "𦉇" },
  { "䍙", "䍙", "䍙", "䍙", "䍙" },
  { "𦋙", "𦋙", "𦋙", "𦋙", "𦋙" },
  { "罺", "罺", "罺", "罺", "罺" },
  { "𦌾", "𦌾", "𦌾", "𦌾", "𦌾" },
  { "羕", "羕", "羕", "羕", "羕" },
  { "翺", "翺", "翺", "翺", "翺" },
  { "者", "者", "者", "者", "者" },
  { "𦓚", "𦓚", "𦓚", "𦓚", "𦓚" },
  { "𦔣", "𦔣", "𦔣", "𦔣", "𦔣" },
  { "聠", "聠", "聠", "聠", "聠" },
  { "𦖨", "𦖨", "𦖨", "𦖨", "𦖨" },
  { "聰", "聰", "聰", "聰", "聰" },
  { "𣍟", "𣍟", "𣍟", "𣍟", "𣍟" },
  { "䏕", "䏕", "䏕", "䏕", "䏕" },
  { "育", "育", "育", "育", "育" },
  { "脃", "脃", "脃", "脃", "脃" },
  { "䐋", "䐋", "䐋", "䐋", "䐋" },
  { "脾", "脾", "脾", "脾", "脾" },
  { "媵", "媵", "媵", "媵", "媵" },
  { "𦞧", "𦞧", "𦞧", "𦞧", "𦞧" },
  { "𦞵", "𦞵", "𦞵", "𦞵", "𦞵" },
  { "𣎓", "𣎓", "𣎓", "𣎓", "𣎓" },
  { "𣎜", "𣎜", "𣎜", "𣎜", "𣎜" },
  { "舁", "舁", "舁", "舁", "舁" },
  { "舄", "舄", "舄", "舄", "舄" },
  { "辞", "辞", "辞", "辞", "辞" },
  { "䑫", "䑫", "䑫", "䑫", "䑫" },
  { "芑", "芑", "芑", "芑", "芑" },
  { "芋", "芋", "芋", "芋", "芋" },
  { "芝", "芝", "芝", "芝", "芝" },
  { "劳", "劳", "劳", "劳", "劳" },
  { "花", "花", "花", "花", "花" },
  { "芳", "芳", "芳", "芳", "芳" },
  { "芽", "芽", "芽", "芽", "芽" },
  { "苦", "苦", "苦", "苦", "苦" },
  { "𦬼", "𦬼", "𦬼", "𦬼", "𦬼" },
  { "若", "若", "若", "若", "若" },
  { "茝", "茝", "茝", "茝", "茝" },
  { "荣", "荣", "荣", "荣", "荣" },
  { "莭", "莭", "莭", "莭", "莭" },
  { "茣", "茣", "茣", "茣", "茣" },
  { "莽", "莽", "莽", "莽", "莽" },
  { "菧", "菧", "菧", "菧", "菧" },
  { "著", "著", "著", "著", "著" },
  { "荓", "荓", "荓", "荓", "荓" },
  { "菊", "菊", "菊", "菊", "菊" },
  { "菌", "菌", "菌", "菌", "菌" },
  { "菜", "菜", "菜", "菜", "菜" },
  { "𦰶", "𦰶", "𦰶", "𦰶", "𦰶" },
  { "𦵫", "𦵫", "𦵫", "𦵫", "𦵫" },
  { "𦳕", "𦳕", "𦳕", "𦳕", "𦳕" },
  { "䔫", "䔫", "䔫", "䔫", "䔫" },
  { "蓱", "蓱", "蓱", "蓱", "蓱" },
  { "蓳", "蓳", "蓳", "蓳", "蓳" },
  { "蔖", "蔖", "蔖", "蔖", "蔖" },
  { "𧏊", "𧏊", "𧏊", "𧏊", "𧏊" },
  { "蕤", "蕤", "蕤", "蕤", "蕤" },
  { "𦼬", "𦼬", "𦼬", "𦼬", "𦼬" },
  { "䕝", "䕝", "䕝", "䕝", "䕝" },
  { "䕡", "䕡", "䕡", "䕡", "䕡" },
  { "𦾱", "𦾱", "𦾱", "𦾱", "𦾱" },
  { "𧃒", "𧃒", "𧃒", "𧃒", "𧃒" },
  { "䕫", "䕫", "䕫", "䕫", "䕫" },
  { "虐", "虐", "虐", "虐", "虐" },
  { "虜", "虜", "虜", "虜", "虜" },
  { "虧", "虧", "虧", "虧", "虧" },
  { "虩", "虩", "虩", "虩", "虩" },
  { "蚩", "蚩", "蚩", "蚩", "蚩" },
  { "蚈", "蚈", "蚈", "蚈", "蚈" },
  { "蜎", "蜎", "蜎", "蜎", "蜎" },
  { "蛢", "蛢", "蛢", "蛢", "蛢" },
  { "蝹", "蝹", "蝹", "蝹", "蝹" },
  { "蜨", "蜨", "蜨", "蜨", "蜨" },
  { "蝫", "蝫", "蝫", "蝫", "蝫" },
  { "螆", "螆", "螆", "螆", "螆" },
  { "䗗", "䗗", "䗗", "䗗", "䗗" },
  { "蟡", "蟡", "蟡", "蟡", "蟡" },
  { "蠁", "蠁", "蠁", "蠁", "蠁" },
  { "䗹", "䗹", "䗹", "䗹", "䗹" },
  { "衠", "衠", "衠", "衠", "衠" },
  { "衣", "衣", "衣", "衣", "衣" },
  { "𧙧", "𧙧", "𧙧", "𧙧", "𧙧" },
  { "裗", "裗", "裗", "裗", "裗" },
  { "裞", "裞", "裞", "裞", "裞" },
  { "䘵", "䘵", "䘵", "䘵", "䘵" },
  { "裺", "裺", "裺", "裺", "裺" },
  { "㒻", "㒻", "㒻", "㒻", "㒻" },
  { "𧢮", "𧢮", "𧢮", "𧢮", "𧢮" },
  { "𧥦", "𧥦", "𧥦", "𧥦", "𧥦" },
  { "䚾", "䚾", "䚾", "䚾", "䚾" },
  { "䛇", "䛇", "䛇", "䛇", "䛇" },
  { "誠", "誠", "誠", "誠", "誠" },
  { "諭", "諭", "諭", "諭", "諭" },
  { "變", "變", "變", "變", "變" },
  { "豕", "豕", "豕", "豕", "豕" },
  { "𧲨", "𧲨", "𧲨", "𧲨", "𧲨" },
  { "貫", "貫", "貫", "貫", "貫" },
  { "賁", "賁", "賁", "賁", "賁" },
  { "贛", "贛", "贛", "贛", "贛" },
  { "起", "起", "起", "起", "起" },
  { "𧼯", "𧼯", "𧼯", "𧼯", "𧼯" },
  { "𠠄", "𠠄", "𠠄", "𠠄", "𠠄" },
  { "跋", "跋", "跋", "跋", "跋" },
  { "趼", "趼", "趼", "趼", "趼" },
  { "跰", "跰", "跰", "跰", "跰" },
  { "𠣞", "𠣞", "𠣞", "𠣞", "𠣞" },
  { "軔", "軔", "軔", "軔", "軔" },
  { "輸", "輸", "輸", "輸", "輸" },
  { "𨗒", "𨗒", "𨗒", "𨗒", "𨗒" },
  { "𨗭", "𨗭", "𨗭", "𨗭", "𨗭" },
  { "邔", "邔", "邔", "邔", "邔" },
  { "郱", "郱", "郱", "郱", "郱" },
  { "鄑", "鄑", "鄑", "鄑", "鄑" },
  { "𨜮", "𨜮", "𨜮", "𨜮", "𨜮" },
  { "鄛", "鄛", "鄛", "鄛", "鄛" },
  { "鈸", "鈸", "鈸", "鈸", "鈸" },
  { "鋗", "鋗", "鋗", "鋗", "鋗" },
  { "鋘", "鋘", "鋘", "鋘", "鋘" },
  { "鉼", "鉼", "鉼", "鉼", "鉼" },
  { "鏹", "鏹", "鏹", "鏹", "鏹" },
  { "鐕", "鐕", "鐕", "鐕", "鐕" },
  { "𨯺", "𨯺", "𨯺", "𨯺", "𨯺" },
  { "開", "開", "開", "開", "開" },
  { "䦕", "䦕", "䦕", "䦕", "䦕" },
  { "閷", "閷", "閷", "閷", "閷" },
  { "𨵷", "𨵷", "𨵷", "𨵷", "𨵷" },
  { "䧦", "䧦", "䧦", "䧦", "䧦" },
  { "雃", "雃", "雃", "雃", "雃" },
  { "嶲", "嶲", "嶲", "嶲", "嶲" },
  { "霣", "霣", "霣", "霣", "霣" },
  { "𩅅", "𩅅", "𩅅", "𩅅", "𩅅" },
  { "𩈚", "𩈚", "𩈚", "𩈚", "𩈚" },
  { "䩮", "䩮", "䩮", "䩮", "䩮" },
  { "䩶", "䩶", "䩶", "䩶", "䩶" },
  { "韠", "韠", "韠", "韠", "韠" },
  { "𩐊", "𩐊", "𩐊", "𩐊", "𩐊" },
  { "䪲", "䪲", "䪲", "䪲", "䪲" },
  { "𩒖", "𩒖", "𩒖", "𩒖", "𩒖" },
  { "頋", "頋", "頋", "頋", "頋" },
  { "頋", "頋", "頋", "頋", "頋" },
  { "頩", "頩", "頩", "頩", "頩" },
  { "𩖶", "𩖶", "𩖶", "𩖶", "𩖶" },
  { "飢", "飢", "飢", "飢", "飢" },
  { "䬳", "䬳", "䬳", "䬳", "䬳" },
  { "餩", "餩", "餩", "餩", "餩" },
  { "馧", "馧", "馧", "馧", "馧" },
  { "駂", "駂", "駂", "駂", "駂" },
  { "駾", "駾", "駾", "駾", "駾" },
  { "䯎", "䯎", "䯎", "䯎", "䯎" },
  { "𩬰", "𩬰", "𩬰", "𩬰", "𩬰" },
  { "鬒", "鬒", "鬒", "鬒", "鬒" },
  { "鱀", "鱀", "鱀", "鱀", "鱀" },
  { "鳽", "鳽", "鳽", "鳽", "鳽" },
  { "䳎", "䳎", "䳎", "䳎", "䳎" },
  { "䳭", "䳭", "䳭", "䳭", "䳭" },
  { "鵧", "鵧", "鵧", "鵧", "鵧" },
  { "𪃎", "𪃎", "𪃎", "𪃎", "𪃎" },
  { "䳸", "䳸", "䳸", "䳸", "䳸" },
  { "𪄅", "𪄅", "𪄅", "𪄅", "𪄅" },
  { "𪈎", "𪈎", "𪈎", "𪈎", "𪈎" },
  { "𪊑", "𪊑", "𪊑", "𪊑", "𪊑" },
  { "麻", "麻", "麻", "麻", "麻" },
  { "䵖", "䵖", "䵖", "䵖", "䵖" },
  { "黹", "黹", "黹", "黹", "黹" },
  { "黾", "黾", "黾", "黾", "黾" },
  { "鼅", "鼅", "鼅", "鼅", "鼅" },
  { "鼏", "鼏", "鼏", "鼏", "鼏" },
  { "鼖", "鼖", "鼖", "鼖", "鼖" },
  { "鼻", "鼻", "鼻", "鼻", "鼻" },
  { "𪘀", "𪘀", "𪘀", "𪘀", "𪘀" },
  { "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b" },
  { "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b" },
  { "à֮́̕b", "à֮́̕b", "à֮́̕b", "à֮́̕b", "à֮́̕b" },
  { "á֮̀̕b", "á֮̀̕b", "á֮̀̕b", "á֮̀̕b", "á֮̀̕b" },
  { "à֮̂̕b", "à֮̂̕b", "à֮̂̕b", "à֮̂̕b", "à֮̂̕b" },
  { "ầ֮̕b", "ầ֮̕b", "ầ֮̕b", "ầ֮̕b", "ầ֮̕b" },
  { "à֮̃̕b", "à֮̃̕b", "à֮̃̕b", "à֮̃̕b", "à֮̃̕b" },
  { "ã֮̀̕b", "ã֮̀̕b", "ã֮̀̕b", "ã֮̀̕b", "ã֮̀̕b" },
  { "à֮̄̕b", "à֮̄̕b", "à֮̄̕b", "à֮̄̕b", "à֮̄̕b" },
  { "ā֮̀̕b", "ā֮̀̕b", "ā֮̀̕b", "ā֮̀̕b", "ā֮̀̕b" },
  { "à֮̅̕b", "à֮̅̕b", "à֮̅̕b", "à֮̅̕b", "à֮̅̕b" },
  { "a֮̅̀̕b", "a֮̅̀̕b", "a֮̅̀̕b", "a֮̅̀̕b", "a֮̅̀̕b" },
  { "à֮̆̕b", "à֮̆̕b", "à֮̆̕b", "à֮̆̕b", "à֮̆̕b" },
  { "ằ֮̕b", "ằ֮̕b", "ằ֮̕b", "ằ֮̕b", "ằ֮̕b" },
  { "à֮̇̕b", "à֮̇̕b", "à֮̇̕b", "à֮̇̕b", "à֮̇̕b" },
  { "ȧ֮̀̕b", "ȧ֮̀̕b", "ȧ֮̀̕b", "ȧ֮̀̕b", "ȧ֮̀̕b" },
  { "à֮̈̕b", "à֮̈̕b", "à֮̈̕b", "à֮̈̕b", "à֮̈̕b" },
  { "ä֮̀̕b", "ä֮̀̕b", "ä֮̀̕b", "ä֮̀̕b", "ä֮̀̕b" },
  { "à֮̉̕b", "à֮̉̕b", "à֮̉̕b", "à֮̉̕b", "à֮̉̕b" },
  { "ả֮̀̕b", "ả֮̀̕b", "ả֮̀̕b", "ả֮̀̕b", "ả֮̀̕b" },
  { "à֮̊̕b", "à֮̊̕b", "à֮̊̕b", "à֮̊̕b", "à֮̊̕b" },
  { "å֮̀̕b", "å֮̀̕b", "å֮̀̕b", "å֮̀̕b", "å֮̀̕b" },
  { "à֮̋̕b", "à֮̋̕b", "à֮̋̕b", "à֮̋̕b", "à֮̋̕b" },
  { "a֮̋̀̕b", "a֮̋̀̕b", "a֮̋̀̕b", "a֮̋̀̕b", "a֮̋̀̕b" },
  { "à֮̌̕b", "à֮̌̕b", "à֮̌̕b", "à֮̌̕b", "à֮̌̕b" },
  { "ǎ֮̀̕b", "ǎ֮̀̕b", "ǎ֮̀̕b", "ǎ֮̀̕b", "ǎ֮̀̕b" },
  { "à֮̍̕b", "à֮̍̕b", "à֮̍̕b", "à֮̍̕b", "à֮̍̕b" },
  { "a֮̍̀̕b", "a֮̍̀̕b", "a֮̍̀̕b", "a֮̍̀̕b", "a֮̍̀̕b" },
  { "à֮̎̕b", "à֮̎̕b", "à֮̎̕b", "à֮̎̕b", "à֮̎̕b" },
  { "a֮̎̀̕b", "a֮̎̀̕b", "a֮̎̀̕b", "a֮̎̀̕b", "a֮̎̀̕b" },
  { "à֮̏̕b", "à֮̏̕b", "à֮̏̕b", "à֮̏̕b", "à֮̏̕b" },
  { "ȁ֮̀̕b", "ȁ֮̀̕b", "ȁ֮̀̕b", "ȁ֮̀̕b", "ȁ֮̀̕b" },
  { "à֮̐̕b", "à֮̐̕b", "à֮̐̕b", "à֮̐̕b", "à֮̐̕b" },
  { "a֮̐̀̕b", "a֮̐̀̕b", "a֮̐̀̕b", "a֮̐̀̕b", "a֮̐̀̕b" },
  { "à֮̑̕b", "à֮̑̕b", "à֮̑̕b", "à֮̑̕b", "à֮̑̕b" },
  { "ȃ֮̀̕b", "ȃ֮̀̕b", "ȃ֮̀̕b", "ȃ֮̀̕b", "ȃ֮̀̕b" },
  { "à֮̒̕b", "à֮̒̕b", "à֮̒̕b", "à֮̒̕b", "à֮̒̕b" },
  { "a֮̒̀̕b", "a֮̒̀̕b", "a֮̒̀̕b", "a֮̒̀̕b", "a֮̒̀̕b" },
  { "à֮̓̕b", "à֮̓̕b", "à֮̓̕b", "à֮̓̕b", "à֮̓̕b" },
  { "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b" },
  { "à֮̔̕b", "à֮̔̕b", "à֮̔̕b", "à֮̔̕b", "à֮̔̕b" },
  { "a֮̔̀̕b", "a֮̔̀̕b", "a֮̔̀̕b", "a֮̔̀̕b", "a֮̔̀̕b" },
  { "à̕̕͜b", "à̕̕͜b", "à̕̕͜b", "à̕̕͜b", "à̕̕͜b" },
  { "à̕̕͜b", "à̕̕͜b", "à̕̕͜b", "à̕̕͜b", "à̕̕͜b" },
  { "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b" },
  { "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b", "a〪̖̖֚b" },
  { "a〪̖̗֚b", "a〪̖̗֚b", "a〪̖̗֚b", "a〪̖̗֚b", "a〪̖̗֚b" },
  { "a〪̗̖֚b", "a〪̗̖֚b", "a〪̗̖֚b", "a〪̗̖֚b", "a〪̗̖֚b" },
  { "a〪̖̘֚b", "a〪̖̘֚b", "a〪̖̘֚b", "a〪̖̘֚b", "a〪̖̘֚b" },
  { "a〪̘̖֚b", "a〪̘̖֚b", "a〪̘̖֚b", "a〪̘̖֚b", "a〪̘̖֚b" },
  { "a〪̖̙֚b", "a〪̖̙֚b", "a〪̖̙֚b", "a〪̖̙֚b", "a〪̖̙֚b" },
  { "a〪̙̖֚b", "a〪̙̖֚b", "a〪̙̖֚b", "a〪̙̖֚b", "a〪̙̖֚b" },
  { "à̕̚͜b", "à̕̚͜b", "à̕̚͜b", "à̕̚͜b", "à̕̚͜b" },
  { "à̚̕͜b", "à̚̕͜b", "à̚̕͜b", "à̚̕͜b", "à̚̕͜b" },
  { "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b" },
  { "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b", "a᷎̛̛〪b" },
  { "a〪̖̜֚b", "a〪̖̜֚b", "a〪̖̜֚b", "a〪̖̜֚b", "a〪̖̜֚b" },
  { "a〪̜̖֚b", "a〪̜̖֚b", "a〪̜̖֚b", "a〪̜̖֚b", "a〪̜̖֚b" },
  { "a〪̖̝֚b", "a〪̖̝֚b", "a〪̖̝֚b", "a〪̖̝֚b", "a〪̖̝֚b" },
  { "a〪̝̖֚b", "a〪̝̖֚b", "a〪̝̖֚b", "a〪̝̖֚b", "a〪̝̖֚b" },
  { "a〪̖̞֚b", "a〪̖̞֚b", "a〪̖̞֚b", "a〪̖̞֚b", "a〪̖̞֚b" },
  { "a〪̞̖֚b", "a〪̞̖֚b", "a〪̞̖֚b", "a〪̞̖֚b", "a〪̞̖֚b" },
  { "a〪̖̟֚b", "a〪̖̟֚b", "a〪̖̟֚b", "a〪̖̟֚b", "a〪̖̟֚b" },
  { "a〪̟̖֚b", "a〪̟̖֚b", "a〪̟̖֚b", "a〪̟̖֚b", "a〪̟̖֚b" },
  { "a〪̖̠֚b", "a〪̖̠֚b", "a〪̖̠֚b", "a〪̖̠֚b", "a〪̖̠֚b" },
  { "a〪̠̖֚b", "a〪̠̖֚b", "a〪̠̖֚b", "a〪̠̖֚b", "a〪̠̖֚b" },
  { "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b" },
  { "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b", "aུ̡̡᷎b" },
  { "aུ̡̢᷎b", "aུ̡̢᷎b", "aུ̡̢᷎b", "aུ̡̢᷎b", "aུ̡̢᷎b" },
  { "aུ̢̡᷎b", "aུ̢̡᷎b", "aུ̢̡᷎b", "aུ̢̡᷎b", "aུ̢̡᷎b" },
  { "a〪̖̣֚b", "a〪̖̣֚b", "a〪̖̣֚b", "a〪̖̣֚b", "a〪̖̣֚b" },
  { "ạ〪̖֚b", "ạ〪̖֚b", "ạ〪̖֚b", "ạ〪̖֚b", "ạ〪̖֚b" },
  { "a〪̖̤֚b", "a〪̖̤֚b", "a〪̖̤֚b", "a〪̖̤֚b", "a〪̖̤֚b" },
  { "a〪̤̖֚b", "a〪̤̖֚b", "a〪̤̖֚b", "a〪̤̖֚b", "a〪̤̖֚b" },
  { "a〪̖̥֚b", "a〪̖̥֚b", "a〪̖̥֚b", "a〪̖̥֚b", "a〪̖̥֚b" },
  { "ḁ〪̖֚b", "ḁ〪̖֚b", "ḁ〪̖֚b", "ḁ〪̖֚b", "ḁ〪̖֚b" },
  { "a〪̖̦֚b", "a〪̖̦֚b", "a〪̖̦֚b", "a〪̖̦֚b", "a〪̖̦֚b" },
  { "a〪̦̖֚b", "a〪̦̖֚b", "a〪̦̖֚b", "a〪̦̖֚b", "a〪̦̖֚b" },
  { "aུ̡̧᷎b", "aུ̡̧᷎b", "aུ̡̧᷎b", "aུ̡̧᷎b", "aུ̡̧᷎b" },
  { "aུ̧̡᷎b", "aུ̧̡᷎b", "aུ̧̡᷎b", "aུ̧̡᷎b", "aུ̧̡᷎b" },
  { "aུ̡̨᷎b", "aུ̡̨᷎b", "aུ̡̨᷎b", "aུ̡̨᷎b", "aུ̡̨᷎b" },
  { "ąུ̡᷎b", "ąུ̡᷎b", "ąུ̡᷎b", "ąུ̡᷎b", "ąུ̡᷎b" },
  { "a〪̖̩֚b", "a〪̖̩֚b", "a〪̖̩֚b", "a〪̖̩֚b", "a〪̖̩֚b" },
  { "a〪̩̖֚b", "a〪̩̖֚b", "a〪̩̖֚b", "a〪̩̖֚b", "a〪̩̖֚b" },
  { "a〪̖̪֚b", "a〪̖̪֚b", "a〪̖̪֚b", "a〪̖̪֚b", "a〪̖̪֚b" },
  { "a〪̪̖֚b", "a〪̪̖֚b", "a〪̪̖֚b", "a〪̪̖֚b", "a〪̪̖֚b" },
  { "a〪̖̫֚b", "a〪̖̫֚b", "a〪̖̫֚b", "a〪̖̫֚b", "a〪̖̫֚b" },
  { "a〪̫̖֚b", "a〪̫̖֚b", "a〪̫̖֚b", "a〪̫̖֚b", "a〪̫̖֚b" },
  { "a〪̖̬֚b", "a〪̖̬֚b", "a〪̖̬֚b", "a〪̖̬֚b", "a〪̖̬֚b" },
  { "a〪̬̖֚b", "a〪̬̖֚b", "a〪̬̖֚b", "a〪̬̖֚b", "a〪̬̖֚b" },
  { "a〪̖̭֚b", "a〪̖̭֚b", "a〪̖̭֚b", "a〪̖̭֚b", "a〪̖̭֚b" },
  { "a〪̭̖֚b", "a〪̭̖֚b", "a〪̭̖֚b", "a〪̭̖֚b", "a〪̭̖֚b" },
  { "a〪̖̮֚b", "a〪̖̮֚b", "a〪̖̮֚b", "a〪̖̮֚b", "a〪̖̮֚b" },
  { "a〪̮̖֚b", "a〪̮̖֚b", "a〪̮̖֚b", "a〪̮̖֚b", "a〪̮̖֚b" },
  { "a〪̖̯֚b", "a〪̖̯֚b", "a〪̖̯֚b", "a〪̖̯֚b", "a〪̖̯֚b" },
  { "a〪̯̖֚b", "a〪̯̖֚b", "a〪̯̖֚b", "a〪̯̖֚b", "a〪̯̖֚b" },
  { "a〪̖̰֚b", "a〪̖̰֚b", "a〪̖̰֚b", "a〪̖̰֚b", "a〪̖̰֚b" },
  { "a〪̰̖֚b", "a〪̰̖֚b", "a〪̰̖֚b", "a〪̰̖֚b", "a〪̰̖֚b" },
  { "a〪̖̱֚b", "a〪̖̱֚b", "a〪̖̱֚b", "a〪̖̱֚b", "a〪̖̱֚b" },
  { "a〪̱̖֚b", "a〪̱̖֚b", "a〪̱̖֚b", "a〪̱̖֚b", "a〪̱̖֚b" },
  { "a〪̖̲֚b", "a〪̖̲֚b", "a〪̖̲֚b", "a〪̖̲֚b", "a〪̖̲֚b" },
  { "a〪̲̖֚b", "a〪̲̖֚b", "a〪̲̖֚b", "a〪̲̖֚b", "a〪̲̖֚b" },
  { "a〪̖̳֚b", "a〪̖̳֚b", "a〪̖̳֚b", "a〪̖̳֚b", "a〪̖̳֚b" },
  { "a〪̳̖֚b", "a〪̳̖֚b", "a〪̳̖֚b", "a〪̳̖֚b", "a〪̳̖֚b" },
  { "a̴̴़b", "a̴̴़b", "a̴̴़b", "a̴̴़b", "a̴̴़b" },
  { "a̴̴़b", "a̴̴़b", "a̴̴़b", "a̴̴़b", "a̴̴़b" },
  { "a̴̵़b", "a̴̵़b", "a̴̵़b", "a̴̵़b", "a̴̵़b" },
  { "a̵̴़b", "a̵̴़b", "a̵̴़b", "a̵̴़b", "a̵̴़b" },
  { "a̴̶़b", "a̴̶़b", "a̴̶़b", "a̴̶़b", "a̴̶़b" },
  { "a̶̴़b", "a̶̴़b", "a̶̴़b", "a̶̴़b", "a̶̴़b" },
  { "a̴̷़b", "a̴̷़b", "a̴̷़b", "a̴̷़b", "a̴̷़b" },
  { "a̷̴़b", "a̷̴़b", "a̷̴़b", "a̷̴़b", "a̷̴़b" },
  { "a̴̸़b", "a̴̸़b", "a̴̸़b", "a̴̸़b", "a̴̸़b" },
  { "a̸̴़b", "a̸̴़b", "a̸̴़b", "a̸̴़b", "a̸̴़b" },
  { "a〪̖̹֚b", "a〪̖̹֚b", "a〪̖̹֚b", "a〪̖̹֚b", "a〪̖̹֚b" },
  { "a〪̹̖֚b", "a〪̹̖֚b", "a〪̹̖֚b", "a〪̹̖֚b", "a〪̹̖֚b" },
  { "a〪̖̺֚b", "a〪̖̺֚b", "a〪̖̺֚b", "a〪̖̺֚b", "a〪̖̺֚b" },
  { "a〪̺̖֚b", "a〪̺̖֚b", "a〪̺̖֚b", "a〪̺̖֚b", "a〪̺̖֚b" },
  { "a〪̖̻֚b", "a〪̖̻֚b", "a〪̖̻֚b", "a〪̖̻֚b", "a〪̖̻֚b" },
  { "a〪̻̖֚b", "a〪̻̖֚b", "a〪̻̖֚b", "a〪̻̖֚b", "a〪̻̖֚b" },
  { "a〪̖̼֚b", "a〪̖̼֚b", "a〪̖̼֚b", "a〪̖̼֚b", "a〪̖̼֚b" },
  { "a〪̼̖֚b", "a〪̼̖֚b", "a〪̼̖֚b", "a〪̼̖֚b", "a〪̼̖֚b" },
  { "à֮̽̕b", "à֮̽̕b", "à֮̽̕b", "à֮̽̕b", "à֮̽̕b" },
  { "a֮̽̀̕b", "a֮̽̀̕b", "a֮̽̀̕b", "a֮̽̀̕b", "a֮̽̀̕b" },
  { "à֮̾̕b", "à֮̾̕b", "à֮̾̕b", "à֮̾̕b", "à֮̾̕b" },
  { "a֮̾̀̕b", "a֮̾̀̕b", "a֮̾̀̕b", "a֮̾̀̕b", "a֮̾̀̕b" },
  { "à֮̿̕b", "à֮̿̕b", "à֮̿̕b", "à֮̿̕b", "à֮̿̕b" },
  { "a֮̿̀̕b", "a֮̿̀̕b", "a֮̿̀̕b", "a֮̿̀̕b", "a֮̿̀̕b" },
  { "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b" },
  { "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b", "à֮̀̕b" },
  { "à֮́̕b", "à֮́̕b", "à֮́̕b", "à֮́̕b", "à֮́̕b" },
  { "á֮̀̕b", "á֮̀̕b", "á֮̀̕b", "á֮̀̕b", "á֮̀̕b" },
  { "à֮͂̕b", "à֮͂̕b", "à֮͂̕b", "à֮͂̕b", "à֮͂̕b" },
  { "a֮͂̀̕b", "a֮͂̀̕b", "a֮͂̀̕b", "a֮͂̀̕b", "a֮͂̀̕b" },
  { "à֮̓̕b", "à֮̓̕b", "à֮̓̕b", "à֮̓̕b", "à֮̓̕b" },
  { "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b", "a֮̓̀̕b" },
  { "à֮̈́̕b", "à֮̈́̕b", "à֮̈́̕b", "à֮̈́̕b", "à֮̈́̕b" },
  { "ä֮́̀̕b", "ä֮́̀̕b", "ä֮́̀̕b", "ä֮́̀̕b", "ä֮́̀̕b" },
  { "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb" },
  { "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb", "a͝ͅͅb" },
  { "à֮͆̕b", "à֮͆̕b", "à֮͆̕b", "à֮͆̕b", "à֮͆̕b" },
  { "a֮͆̀̕b", "a֮͆̀̕b", "a֮͆̀̕b", "a֮͆̀̕b", "a֮͆̀̕b" },
  { "a〪̖͇֚b", "a〪̖͇֚b", "a〪̖͇֚b", "a〪̖͇֚b", "a〪̖͇֚b" },
  { "a〪͇̖֚b", "a〪͇̖֚b", "a〪͇̖֚b", "a〪͇̖֚b", "a〪͇̖֚b" },
  { "a〪̖͈֚b", "a〪̖͈֚b", "a〪̖͈֚b", "a〪̖͈֚b", "a〪̖͈֚b" },
  { "a〪͈̖֚b", "a〪͈̖֚b", "a〪͈̖֚b", "a〪͈̖֚b", "a〪͈̖֚b" },
  { "a〪̖͉֚b", "a〪̖͉֚b", "a〪̖͉֚b", "a〪̖͉֚b", "a〪̖͉֚b" },
  { "a〪͉̖֚b", "a〪͉̖֚b", "a〪͉̖֚b", "a〪͉̖֚b", "a〪͉̖֚b" },
  { "à֮͊̕b", "à֮͊̕b", "à֮͊̕b", "à֮͊̕b", "à֮͊̕b" },
  { "a֮͊̀̕b", "a֮͊̀̕b", "a֮͊̀̕b", "a֮͊̀̕b", "a֮͊̀̕b" },
  { "à֮͋̕b", "à֮͋̕b", "à֮͋̕b", "à֮͋̕b", "à֮͋̕b" },
  { "a֮͋̀̕b", "a֮͋̀̕b", "a֮͋̀̕b", "a֮͋̀̕b", "a֮͋̀̕b" },
  { "à֮͌̕b", "à֮͌̕b", "à֮͌̕b", "à֮͌̕b", "à֮͌̕b" },
  { "a֮͌̀̕b", "a֮͌̀̕b", "a֮͌̀̕b", "a֮͌̀̕b", "a֮͌̀̕b" },
  { "a〪̖͍֚b", "a〪̖͍֚b", "a〪̖͍֚b", "a〪̖͍֚b", "a〪̖͍֚b" },
  { "a〪͍̖֚b", "a〪͍̖֚b", "a〪͍̖֚b", "a〪͍̖֚b", "a〪͍̖֚b" },
  { "a〪̖͎֚b", "a〪̖͎֚b", "a〪̖͎֚b", "a〪̖͎֚b", "a〪̖͎֚b" },
  { "a〪͎̖֚b", "a〪͎̖֚b", "a〪͎̖֚b", "a〪͎̖֚b", "a〪͎̖֚b" },
  { "à֮͐̕b", "à֮͐̕b", "à֮͐̕b", "à֮͐̕b", "à֮͐̕b" },
  { "a֮͐̀̕b", "a֮͐̀̕b", "a֮͐̀̕b", "a֮͐̀̕b", "a֮͐̀̕b" },
  { "à֮͑̕b", "à֮͑̕b", "à֮͑̕b", "à֮͑̕b", "à֮͑̕b" },
  { "a֮͑̀̕b", "a֮͑̀̕b", "a֮͑̀̕b", "a֮͑̀̕b", "a֮͑̀̕b" },
  { "à֮͒̕b", "à֮͒̕b", "à֮͒̕b", "à֮͒̕b", "à֮͒̕b" },
  { "a֮͒̀̕b", "a֮͒̀̕b", "a֮͒̀̕b", "a֮͒̀̕b", "a֮͒̀̕b" },
  { "a〪̖͓֚b", "a〪̖͓֚b", "a〪̖͓֚b", "a〪̖͓֚b", "a〪̖͓֚b" },
  { "a〪͓̖֚b", "a〪͓̖֚b", "a〪͓̖֚b", "a〪͓̖֚b", "a〪͓̖֚b" },
  { "a〪̖͔֚b", "a〪̖͔֚b", "a〪̖͔֚b", "a〪̖͔֚b", "a〪̖͔֚b" },
  { "a〪͔̖֚b", "a〪͔̖֚b", "a〪͔̖֚b", "a〪͔̖֚b", "a〪͔̖֚b" },
  { "a〪̖͕֚b", "a〪̖͕֚b", "a〪̖͕֚b", "a〪̖͕֚b", "a〪̖͕֚b" },
  { "a〪͕̖֚b", "a〪͕̖֚b", "a〪͕̖֚b", "a〪͕̖֚b", "a〪͕̖֚b" },
  { "a〪̖͖֚b", "a〪̖͖֚b", "a〪̖͖֚b", "a〪̖͖֚b", "a〪̖͖֚b" },
  { "a〪͖̖֚b", "a〪͖̖֚b", "a〪͖̖֚b", "a〪͖̖֚b", "a〪͖̖֚b" },
  { "à֮͗̕b", "à֮͗̕b", "à֮͗̕b", "à֮͗̕b", "à֮͗̕b" },
  { "a֮͗̀̕b", "a֮͗̀̕b", "a֮͗̀̕b", "a֮͗̀̕b", "a֮͗̀̕b" },
  { "à̕͘͜b", "à̕͘͜b", "à̕͘͜b", "à̕͘͜b", "à̕͘͜b" },
  { "à͘̕͜b", "à͘̕͜b", "à͘̕͜b", "à͘̕͜b", "à͘̕͜b" },
  { "a〪̖͙֚b", "a〪̖͙֚b", "a〪̖͙֚b", "a〪̖͙֚b", "a〪̖͙֚b" },
  { "a〪͙̖֚b", "a〪͙̖֚b", "a〪͙̖֚b", "a〪͙̖֚b", "a〪͙̖֚b" },
  { "a〪̖͚֚b", "a〪̖͚֚b", "a〪̖͚֚b", "a〪̖͚֚b", "a〪̖͚֚b" },
  { "a〪͚̖֚b", "a〪͚̖֚b", "a〪͚̖֚b", "a〪͚̖֚b", "a〪͚̖֚b" },
  { "à֮͛̕b", "à֮͛̕b", "à֮͛̕b", "à֮͛̕b", "à֮͛̕b" },
  { "a֮͛̀̕b", "a֮͛̀̕b", "a֮͛̀̕b", "a֮͛̀̕b", "a֮͛̀̕b" },
  { "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b" },
  { "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b", "a̕͜͜͝b" },
  { "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb" },
  { "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb", "a͜͝͝ͅb" },
  { "a͜͝͞ͅb", "a͜͝͞ͅb", "a͜͝͞ͅb", "a͜͝͞ͅb", "a͜͝͞ͅb" },
  { "a͜͞͝ͅb", "a͜͞͝ͅb", "a͜͞͝ͅb", "a͜͞͝ͅb", "a͜͞͝ͅb" },
  { "a̕͜͟͝b", "a̕͜͟͝b", "a̕͜͟͝b", "a̕͜͟͝b", "a̕͜͟͝b" },
  { "a̕͟͜͝b", "a̕͟͜͝b", "a̕͟͜͝b", "a̕͟͜͝b", "a̕͟͜͝b" },
  { "a͜͝͠ͅb", "a͜͝͠ͅb", "a͜͝͠ͅb", "a͜͝͠ͅb", "a͜͝͠ͅb" },
  { "a͜͠͝ͅb", "a͜͠͝ͅb", "a͜͠͝ͅb", "a͜͠͝ͅb", "a͜͠͝ͅb" },
  { "a͜͝͡ͅb", "a͜͝͡ͅb", "a͜͝͡ͅb", "a͜͝͡ͅb", "a͜͝͡ͅb" },
  { "a͜͡͝ͅb", "a͜͡͝ͅb", "a͜͡͝ͅb", "a͜͡͝ͅb", "a͜͡͝ͅb" },
  { "a̕͜͢͝b", "a̕͜͢͝b", "a̕͜͢͝b", "a̕͜͢͝b", "a̕͜͢͝b" },
  { "a̕͢͜͝b", "a̕͢͜͝b", "a̕͢͜͝b", "a̕͢͜͝b", "a̕͢͜͝b" },
  { "à֮ͣ̕b", "à֮ͣ̕b", "à֮ͣ̕b", "à֮ͣ̕b", "à֮ͣ̕b" },
  { "a֮ͣ̀̕b", "a֮ͣ̀̕b", "a֮ͣ̀̕b", "a֮ͣ̀̕b", "a֮ͣ̀̕b" },
  { "à֮ͤ̕b", "à֮ͤ̕b", "à֮ͤ̕b", "à֮ͤ̕b", "à֮ͤ̕b" },
  { "a֮ͤ̀̕b", "a֮ͤ̀̕b", "a֮ͤ̀̕b", "a֮ͤ̀̕b", "a֮ͤ̀̕b" },
  { "à֮ͥ̕b", "à֮ͥ̕b", "à֮ͥ̕b", "à֮ͥ̕b", "à֮ͥ̕b" },
  { "a֮ͥ̀̕b", "a֮ͥ̀̕b", "a֮ͥ̀̕b", "a֮ͥ̀̕b", "a֮ͥ̀̕b" },
  { "à֮ͦ̕b", "à֮ͦ̕b", "à֮ͦ̕b", "à֮ͦ̕b", "à֮ͦ̕b" },
  { "a֮ͦ̀̕b", "a֮ͦ̀̕b", "a֮ͦ̀̕b", "a֮ͦ̀̕b", "a֮ͦ̀̕b" },
  { "à֮ͧ̕b", "à֮ͧ̕b", "à֮ͧ̕b", "à֮ͧ̕b", "à֮ͧ̕b" },
  { "a֮ͧ̀̕b", "a֮ͧ̀̕b", "a֮ͧ̀̕b", "a֮ͧ̀̕b", "a֮ͧ̀̕b" },
  { "à֮ͨ̕b", "à֮ͨ̕b", "à֮ͨ̕b", "à֮ͨ̕b", "à֮ͨ̕b" },
  { "a֮ͨ̀̕b", "a֮ͨ̀̕b", "a֮ͨ̀̕b", "a֮ͨ̀̕b", "a֮ͨ̀̕b" },
  { "à֮ͩ̕b", "à֮ͩ̕b", "à֮ͩ̕b", "à֮ͩ̕b", "à֮ͩ̕b" },
  { "a֮ͩ̀̕b", "a֮ͩ̀̕b", "a֮ͩ̀̕b", "a֮ͩ̀̕b", "a֮ͩ̀̕b" },
  { "à֮ͪ̕b", "à֮ͪ̕b", "à֮ͪ̕b", "à֮ͪ̕b", "à֮ͪ̕b" },
  { "a֮ͪ̀̕b", "a֮ͪ̀̕b", "a֮ͪ̀̕b", "a֮ͪ̀̕b", "a֮ͪ̀̕b" },
  { "à֮ͫ̕b", "à֮ͫ̕b", "à֮ͫ̕b", "à֮ͫ̕b", "à֮ͫ̕b" },
  { "a֮ͫ̀̕b", "a֮ͫ̀̕b", "a֮ͫ̀̕b", "a֮ͫ̀̕b", "a֮ͫ̀̕b" },
  { "à֮ͬ̕b", "à֮ͬ̕b", "à֮ͬ̕b", "à֮ͬ̕b", "à֮ͬ̕b" },
  { "a֮ͬ̀̕b", "a֮ͬ̀̕b", "a֮ͬ̀̕b", "a֮ͬ̀̕b", "a֮ͬ̀̕b" },
  { "à֮ͭ̕b", "à֮ͭ̕b", "à֮ͭ̕b", "à֮ͭ̕b", "à֮ͭ̕b" },
  { "a֮ͭ̀̕b", "a֮ͭ̀̕b", "a֮ͭ̀̕b", "a֮ͭ̀̕b", "a֮ͭ̀̕b" },
  { "à֮ͮ̕b", "à֮ͮ̕b", "à֮ͮ̕b", "à֮ͮ̕b", "à֮ͮ̕b" },
  { "a֮ͮ̀̕b", "a֮ͮ̀̕b", "a֮ͮ̀̕b", "a֮ͮ̀̕b", "a֮ͮ̀̕b" },
  { "à֮ͯ̕b", "à֮ͯ̕b", "à֮ͯ̕b", "à֮ͯ̕b", "à֮ͯ̕b" },
  { "a֮ͯ̀̕b", "a֮ͯ̀̕b", "a֮ͯ̀̕b", "a֮ͯ̀̕b", "a֮ͯ̀̕b" },
  { "à֮҃̕b", "à֮҃̕b", "à֮҃̕b", "à֮҃̕b", "à֮҃̕b" },
  { "a֮҃̀̕b", "a֮҃̀̕b", "a֮҃̀̕b", "a֮҃̀̕b", "a֮҃̀̕b" },
  { "à֮҄̕b", "à֮҄̕b", "à֮҄̕b", "à֮҄̕b", "à֮҄̕b" },
  { "a֮҄̀̕b", "a֮҄̀̕b", "a֮҄̀̕b", "a֮҄̀̕b", "a֮҄̀̕b" },
  { "à֮҅̕b", "à֮҅̕b", "à֮҅̕b", "à֮҅̕b", "à֮҅̕b" },
  { "a֮҅̀̕b", "a֮҅̀̕b", "a֮҅̀̕b", "a֮҅̀̕b", "a֮҅̀̕b" },
  { "à֮҆̕b", "à֮҆̕b", "à֮҆̕b", "à֮҆̕b", "à֮҆̕b" },
  { "a֮҆̀̕b", "a֮҆̀̕b", "a֮҆̀̕b", "a֮҆̀̕b", "a֮҆̀̕b" },
  { "à֮҇̕b", "à֮҇̕b", "à֮҇̕b", "à֮҇̕b", "à֮҇̕b" },
  { "a֮҇̀̕b", "a֮҇̀̕b", "a֮҇̀̕b", "a֮҇̀̕b", "a֮҇̀̕b" },
  { "a〪̖֑֚b", "a〪̖֑֚b", "a〪̖֑֚b", "a〪̖֑֚b", "a〪̖֑֚b" },
  { "a〪֑̖֚b", "a〪֑̖֚b", "a〪֑̖֚b", "a〪֑̖֚b", "a〪֑̖֚b" },
  { "à֮֒̕b", "à֮֒̕b", "à֮֒̕b", "à֮֒̕b", "à֮֒̕b" },
  { "a֮֒̀̕b", "a֮֒̀̕b", "a֮֒̀̕b", "a֮֒̀̕b", "a֮֒̀̕b" },
  { "à֮֓̕b", "à֮֓̕b", "à֮֓̕b", "à֮֓̕b", "à֮֓̕b" },
  { "a֮֓̀̕b", "a֮֓̀̕b", "a֮֓̀̕b", "a֮֓̀̕b", "a֮֓̀̕b" },
  { "à֮֔̕b", "à֮֔̕b", "à֮֔̕b", "à֮֔̕b", "à֮֔̕b" },
  { "a֮֔̀̕b", "a֮֔̀̕b", "a֮֔̀̕b", "a֮֔̀̕b", "a֮֔̀̕b" },
  { "à֮֕̕b", "à֮֕̕b", "à֮֕̕b", "à֮֕̕b", "à֮֕̕b" },
  { "a֮֕̀̕b", "a֮֕̀̕b", "a֮֕̀̕b", "a֮֕̀̕b", "a֮֕̀̕b" },
  { "a〪̖֖֚b", "a〪̖֖֚b", "a〪̖֖֚b", "a〪̖֖֚b", "a〪̖֖֚b" },
  { "a〪֖̖֚b", "a〪֖̖֚b", "a〪֖̖֚b", "a〪֖̖֚b", "a〪֖̖֚b" },
  { "à֮֗̕b", "à֮֗̕b", "à֮֗̕b", "à֮֗̕b", "à֮֗̕b" },
  { "a֮֗̀̕b", "a֮֗̀̕b", "a֮֗̀̕b", "a֮֗̀̕b", "a֮֗̀̕b" },
  { "à֮֘̕b", "à֮֘̕b", "à֮֘̕b", "à֮֘̕b", "à֮֘̕b" },
  { "a֮֘̀̕b", "a֮֘̀̕b", "a֮֘̀̕b", "a֮֘̀̕b", "a֮֘̀̕b" },
  { "à֮֙̕b", "à֮֙̕b", "à֮֙̕b", "à֮֙̕b", "à֮֙̕b" },
  { "a֮֙̀̕b", "a֮֙̀̕b", "a֮֙̀̕b", "a֮֙̀̕b", "a֮֙̀̕b" },
  { "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b" },
  { "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b", "a̖֚֚〮b" },
  { "a〪̖֛֚b", "a〪̖֛֚b", "a〪̖֛֚b", "a〪̖֛֚b", "a〪̖֛֚b" },
  { "a〪֛̖֚b", "a〪֛̖֚b", "a〪֛̖֚b", "a〪֛̖֚b", "a〪֛̖֚b" },
  { "à֮֜̕b", "à֮֜̕b", "à֮֜̕b", "à֮֜̕b", "à֮֜̕b" },
  { "a֮֜̀̕b", "a֮֜̀̕b", "a֮֜̀̕b", "a֮֜̀̕b", "a֮֜̀̕b" },
  { "à֮֝̕b", "à֮֝̕b", "à֮֝̕b", "à֮֝̕b", "à֮֝̕b" },
  { "a֮֝̀̕b", "a֮֝̀̕b", "a֮֝̀̕b", "a֮֝̀̕b", "a֮֝̀̕b" },
  { "à֮֞̕b", "à֮֞̕b", "à֮֞̕b", "à֮֞̕b", "à֮֞̕b" },
  { "a֮֞̀̕b", "a֮֞̀̕b", "a֮֞̀̕b", "a֮֞̀̕b", "a֮֞̀̕b" },
  { "à֮֟̕b", "à֮֟̕b", "à֮֟̕b", "à֮֟̕b", "à֮֟̕b" },
  { "a֮֟̀̕b", "a֮֟̀̕b", "a֮֟̀̕b", "a֮֟̀̕b", "a֮֟̀̕b" },
  { "à֮֠̕b", "à֮֠̕b", "à֮֠̕b", "à֮֠̕b", "à֮֠̕b" },
  { "a֮֠̀̕b", "a֮֠̀̕b", "a֮֠̀̕b", "a֮֠̀̕b", "a֮֠̀̕b" },
  { "à֮֡̕b", "à֮֡̕b", "à֮֡̕b", "à֮֡̕b", "à֮֡̕b" },
  { "a֮֡̀̕b", "a֮֡̀̕b", "a֮֡̀̕b", "a֮֡̀̕b", "a֮֡̀̕b" },
  { "a〪̖֢֚b", "a〪̖֢֚b", "a〪̖֢֚b", "a〪̖֢֚b", "a〪̖֢֚b" },
  { "a〪֢̖֚b", "a〪֢̖֚b", "a〪֢̖֚b", "a〪֢̖֚b", "a〪֢̖֚b" },
  { "a〪̖֣֚b", "a〪̖֣֚b", "a〪̖֣֚b", "a〪̖֣֚b", "a〪̖֣֚b" },
  { "a〪֣̖֚b", "a〪֣̖֚b", "a〪֣̖֚b", "a〪֣̖֚b", "a〪֣̖֚b" },
  { "a〪̖֤֚b", "a〪̖֤֚b", "a〪̖֤֚b", "a〪̖֤֚b", "a〪̖֤֚b" },
  { "a〪֤̖֚b", "a〪֤̖֚b", "a〪֤̖֚b", "a〪֤̖֚b", "a〪֤̖֚b" },
  { "a〪̖֥֚b", "a〪̖֥֚b", "a〪̖֥֚b", "a〪̖֥֚b", "a〪̖֥֚b" },
  { "a〪֥̖֚b", "a〪֥̖֚b", "a〪֥̖֚b", "a〪֥̖֚b", "a〪֥̖֚b" },
  { "a〪̖֦֚b", "a〪̖֦֚b", "a〪̖֦֚b", "a〪̖֦֚b", "a〪̖֦֚b" },
  { "a〪֦̖֚b", "a〪֦̖֚b", "a〪֦̖֚b", "a〪֦̖֚b", "a〪֦̖֚b" },
  { "a〪̖֧֚b", "a〪̖֧֚b", "a〪̖֧֚b", "a〪̖֧֚b", "a〪̖֧֚b" },
  { "a〪֧̖֚b", "a〪֧̖֚b", "a〪֧̖֚b", "a〪֧̖֚b", "a〪֧̖֚b" },
  { "à֮֨̕b", "à֮֨̕b", "à֮֨̕b", "à֮֨̕b", "à֮֨̕b" },
  { "a֮֨̀̕b", "a֮֨̀̕b", "a֮֨̀̕b", "a֮֨̀̕b", "a֮֨̀̕b" },
  { "à֮֩̕b", "à֮֩̕b", "à֮֩̕b", "à֮֩̕b", "à֮֩̕b" },
  { "a֮֩̀̕b", "a֮֩̀̕b", "a֮֩̀̕b", "a֮֩̀̕b", "a֮֩̀̕b" },
  { "a〪̖֪֚b", "a〪̖֪֚b", "a〪̖֪֚b", "a〪̖֪֚b", "a〪̖֪֚b" },
  { "a〪֪̖֚b", "a〪֪̖֚b", "a〪֪̖֚b", "a〪֪̖֚b", "a〪֪̖֚b" },
  { "à֮֫̕b", "à֮֫̕b", "à֮֫̕b", "à֮֫̕b", "à֮֫̕b" },
  { "a֮֫̀̕b", "a֮֫̀̕b", "a֮֫̀̕b", "a֮֫̀̕b", "a֮֫̀̕b" },
  { "à֮֬̕b", "à֮֬̕b", "à֮֬̕b", "à֮֬̕b", "à֮֬̕b" },
  { "a֮֬̀̕b", "a֮֬̀̕b", "a֮֬̀̕b", "a֮֬̀̕b", "a֮֬̀̕b" },
  { "a̖֚֭〮b", "a̖֚֭〮b", "a̖֚֭〮b", "a̖֚֭〮b", "a̖֚֭〮b" },
  { "a̖֭֚〮b", "a̖֭֚〮b", "a̖֭֚〮b", "a̖֭֚〮b", "a̖֭֚〮b" },
  { "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b" },
  { "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b", "à𝅭֮֮b" },
  { "à֮֯̕b", "à֮֯̕b", "à֮֯̕b", "à֮֯̕b", "à֮֯̕b" },
  { "a֮֯̀̕b", "a֮֯̀̕b", "a֮֯̀̕b", "a֮֯̀̕b", "a֮֯̀̕b" },
  { "a्ְְֱb", "a्ְְֱb", "a्ְְֱb", "a्ְְֱb", "a्ְְֱb" },
  { "a्ְְֱb", "a्ְְֱb", "a्ְְֱb", "a्ְְֱb", "a्ְְֱb" },
  { "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb" },
  { "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb", "aְֱֱֲb" },
  { "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb" },
  { "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb", "aֱֲֲֳb" },
  { "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb" },
  { "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb", "aֲֳֳִb" },
  { "aֳִִֵb", "aֳִִֵb", "aֳִִֵb", "aֳִִֵb", "aֳִִֵb" },
  { "aֳִִֵb", "aֳִִֵb", "aֳִִֵb", "aֳִִֵb", "aֳִִֵb" },
  { "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb" },
  { "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb", "aִֵֵֶb" },
  { "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb" },
  { "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb", "aֵֶֶַb" },
  { "aֶַַָb", "aֶַַָb", "aֶַַָb", "aֶַַָb", "aֶַַָb" },
  { "aֶַַָb", "aֶַַָb", "aֶַַָb", "aֶַַָb", "aֶַַָb" },
  { "aַָָֹb", "aַָָֹb", "aַָָֹb", "aַָָֹb", "aַָָֹb" },
  { "aַָָֹb", "aַָָֹb", "aַָָֹb", "aַָָֹb", "aַָָֹb" },
  { "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb" },
  { "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb", "aָֹֹֻb" },
  { "aָֹֺֻb", "aָֹֺֻb", "aָֹֺֻb", "aָֹֺֻb", "aָֹֺֻb" },
  { "aָֺֹֻb", "aָֺֹֻb", "aָֺֹֻb", "aָֺֹֻb", "aָֺֹֻb" },
  { "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb" },
  { "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb", "aֹֻֻּb" },
  { "aֻּּֽb", "aֻּּֽb", "aֻּּֽb", "aֻּּֽb", "aֻּּֽb" },
  { "aֻּּֽb", "aֻּּֽb", "aֻּּֽb", "aֻּּֽb", "aֻּּֽb" },
  { "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb" },
  { "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb", "aּֽֽֿb" },
  { "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb" },
  { "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb", "aֽֿֿׁb" },
  { "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb" },
  { "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb", "aֿׁׁׂb" },
  { "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb" },
  { "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb", "aׁׂׂﬞb" },
  { "à֮ׄ̕b", "à֮ׄ̕b", "à֮ׄ̕b", "à֮ׄ̕b", "à֮ׄ̕b" },
  { "a֮ׄ̀̕b", "a֮ׄ̀̕b", "a֮ׄ̀̕b", "a֮ׄ̀̕b", "a֮ׄ̀̕b" },
  { "a〪̖ׅ֚b", "a〪̖ׅ֚b", "a〪̖ׅ֚b", "a〪̖ׅ֚b", "a〪̖ׅ֚b" },
  { "a〪ׅ̖֚b", "a〪ׅ̖֚b", "a〪ׅ̖֚b", "a〪ׅ̖֚b", "a〪ׅ̖֚b" },
  { "aַָׇֹb", "aַָׇֹb", "aַָׇֹb", "aַָׇֹb", "aַָׇֹb" },
  { "aַׇָֹb", "aַׇָֹb", "aַׇָֹb", "aַׇָֹb", "aַׇָֹb" },
  { "à֮ؐ̕b", "à֮ؐ̕b", "à֮ؐ̕b", "à֮ؐ̕b", "à֮ؐ̕b" },
  { "a֮ؐ̀̕b", "a֮ؐ̀̕b", "a֮ؐ̀̕b", "a֮ؐ̀̕b", "a֮ؐ̀̕b" },
  { "à֮ؑ̕b", "à֮ؑ̕b", "à֮ؑ̕b", "à֮ؑ̕b", "à֮ؑ̕b" },
  { "a֮ؑ̀̕b", "a֮ؑ̀̕b", "a֮ؑ̀̕b", "a֮ؑ̀̕b", "a֮ؑ̀̕b" },
  { "à֮ؒ̕b", "à֮ؒ̕b", "à֮ؒ̕b", "à֮ؒ̕b", "à֮ؒ̕b" },
  { "a֮ؒ̀̕b", "a֮ؒ̀̕b", "a֮ؒ̀̕b", "a֮ؒ̀̕b", "a֮ؒ̀̕b" },
  { "à֮ؓ̕b", "à֮ؓ̕b", "à֮ؓ̕b", "à֮ؓ̕b", "à֮ؓ̕b" },
  { "a֮ؓ̀̕b", "a֮ؓ̀̕b", "a֮ؓ̀̕b", "a֮ؓ̀̕b", "a֮ؓ̀̕b" },
  { "à֮ؔ̕b", "à֮ؔ̕b", "à֮ؔ̕b", "à֮ؔ̕b", "à֮ؔ̕b" },
  { "a֮ؔ̀̕b", "a֮ؔ̀̕b", "a֮ؔ̀̕b", "a֮ؔ̀̕b", "a֮ؔ̀̕b" },
  { "à֮ؕ̕b", "à֮ؕ̕b", "à֮ؕ̕b", "à֮ؕ̕b", "à֮ؕ̕b" },
  { "a֮ؕ̀̕b", "a֮ؕ̀̕b", "a֮ؕ̀̕b", "a֮ؕ̀̕b", "a֮ؕ̀̕b" },
  { "à֮ؖ̕b", "à֮ؖ̕b", "à֮ؖ̕b", "à֮ؖ̕b", "à֮ؖ̕b" },
  { "a֮ؖ̀̕b", "a֮ؖ̀̕b", "a֮ؖ̀̕b", "a֮ؖ̀̕b", "a֮ؖ̀̕b" },
  { "à֮ؗ̕b", "à֮ؗ̕b", "à֮ؗ̕b", "à֮ؗ̕b", "à֮ؗ̕b" },
  { "a֮ؗ̀̕b", "a֮ؗ̀̕b", "a֮ؗ̀̕b", "a֮ؗ̀̕b", "a֮ؗ̀̕b" },
  { "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb" },
  { "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb", "aٍؘؘؙb" },
  { "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb" },
  { "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb", "aؘؙؙؚb" },
  { "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb" },
  { "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb", "aؙؚؚّb" },
  { "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb" },
  { "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb", "aﬞًًٌb" },
  { "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb" },
  { "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb", "aًٌٌٍb" },
  { "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb" },
  { "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb", "aٌٍٍؘb" },
  { "aٍؘَؙb", "aٍؘَؙb", "aٍؘَؙb", "aٍؘَؙb", "aٍؘَؙb" },
  { "aٍَؘؙb", "aٍَؘؙb", "aٍَؘؙb", "aٍَؘؙb", "aٍَؘؙb" },
  { "aؘؙُؚb", "aؘؙُؚb", "aؘؙُؚb", "aؘؙُؚb", "aؘؙُؚb" },
  { "aؘُؙؚb", "aؘُؙؚb", "aؘُؙؚb", "aؘُؙؚb", "aؘُؙؚb" },
  { "aؙؚِّb", "aؙؚِّb", "aؙؚِّb", "aؙؚِّb", "aؙؚِّb" },
  { "aؙِؚّb", "aؙِؚّb", "aؙِؚّb", "aؙِؚّb", "aؙِؚّb" },
  { "aؚّّْb", "aؚّّْb", "aؚّّْb", "aؚّّْb", "aؚّّْb" },
  { "aؚّّْb", "aؚّّْb", "aؚّّْb", "aؚّّْb", "aؚّّْb" },
  { "aّْْٰb", "aّْْٰb", "aّْْٰb", "aّْْٰb", "aّْْٰb" },
  { "aّْْٰb", "aّْْٰb", "aّْْٰb", "aّْْٰb", "aّْْٰb" },
  { "à֮ٓ̕b", "à֮ٓ̕b", "à֮ٓ̕b", "à֮ٓ̕b", "à֮ٓ̕b" },
  { "a֮ٓ̀̕b", "a֮ٓ̀̕b", "a֮ٓ̀̕b", "a֮ٓ̀̕b", "a֮ٓ̀̕b" },
  { "à֮ٔ̕b", "à֮ٔ̕b", "à֮ٔ̕b", "à֮ٔ̕b", "à֮ٔ̕b" },
  { "a֮ٔ̀̕b", "a֮ٔ̀̕b", "a֮ٔ̀̕b", "a֮ٔ̀̕b", "a֮ٔ̀̕b" },
  { "a〪̖ٕ֚b", "a〪̖ٕ֚b", "a〪̖ٕ֚b", "a〪̖ٕ֚b", "a〪̖ٕ֚b" },
  { "a〪ٕ̖֚b", "a〪ٕ̖֚b", "a〪ٕ̖֚b", "a〪ٕ̖֚b", "a〪ٕ̖֚b" },
  { "a〪̖ٖ֚b", "a〪̖ٖ֚b", "a〪̖ٖ֚b", "a〪̖ٖ֚b", "a〪̖ٖ֚b" },
  { "a〪ٖ̖֚b", "a〪ٖ̖֚b", "a〪ٖ̖֚b", "a〪ٖ̖֚b", "a〪ٖ̖֚b" },
  { "à֮ٗ̕b", "à֮ٗ̕b", "à֮ٗ̕b", "à֮ٗ̕b", "à֮ٗ̕b" },
  { "a֮ٗ̀̕b", "a֮ٗ̀̕b", "a֮ٗ̀̕b", "a֮ٗ̀̕b", "a֮ٗ̀̕b" },
  { "à֮٘̕b", "à֮٘̕b", "à֮٘̕b", "à֮٘̕b", "à֮٘̕b" },
  { "a֮٘̀̕b", "a֮٘̀̕b", "a֮٘̀̕b", "a֮٘̀̕b", "a֮٘̀̕b" },
  { "à֮ٙ̕b", "à֮ٙ̕b", "à֮ٙ̕b", "à֮ٙ̕b", "à֮ٙ̕b" },
  { "a֮ٙ̀̕b", "a֮ٙ̀̕b", "a֮ٙ̀̕b", "a֮ٙ̀̕b", "a֮ٙ̀̕b" },
  { "à֮ٚ̕b", "à֮ٚ̕b", "à֮ٚ̕b", "à֮ٚ̕b", "à֮ٚ̕b" },
  { "a֮ٚ̀̕b", "a֮ٚ̀̕b", "a֮ٚ̀̕b", "a֮ٚ̀̕b", "a֮ٚ̀̕b" },
  { "à֮ٛ̕b", "à֮ٛ̕b", "à֮ٛ̕b", "à֮ٛ̕b", "à֮ٛ̕b" },
  { "a֮ٛ̀̕b", "a֮ٛ̀̕b", "a֮ٛ̀̕b", "a֮ٛ̀̕b", "a֮ٛ̀̕b" },
  { "a〪̖ٜ֚b", "a〪̖ٜ֚b", "a〪̖ٜ֚b", "a〪̖ٜ֚b", "a〪̖ٜ֚b" },
  { "a〪ٜ̖֚b", "a〪ٜ̖֚b", "a〪ٜ̖֚b", "a〪ٜ̖֚b", "a〪ٜ̖֚b" },
  { "à֮ٝ̕b", "à֮ٝ̕b", "à֮ٝ̕b", "à֮ٝ̕b", "à֮ٝ̕b" },
  { "a֮ٝ̀̕b", "a֮ٝ̀̕b", "a֮ٝ̀̕b", "a֮ٝ̀̕b", "a֮ٝ̀̕b" },
  { "à֮ٞ̕b", "à֮ٞ̕b", "à֮ٞ̕b", "à֮ٞ̕b", "à֮ٞ̕b" },
  { "a֮ٞ̀̕b", "a֮ٞ̀̕b", "a֮ٞ̀̕b", "a֮ٞ̀̕b", "a֮ٞ̀̕b" },
  { "a〪̖ٟ֚b", "a〪̖ٟ֚b", "a〪̖ٟ֚b", "a〪̖ٟ֚b", "a〪̖ٟ֚b" },
  { "a〪ٟ̖֚b", "a〪ٟ̖֚b", "a〪ٟ̖֚b", "a〪ٟ̖֚b", "a〪ٟ̖֚b" },
  { "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb" },
  { "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb", "aْٰٰܑb" },
  { "à֮ۖ̕b", "à֮ۖ̕b", "à֮ۖ̕b", "à֮ۖ̕b", "à֮ۖ̕b" },
  { "a֮ۖ̀̕b", "a֮ۖ̀̕b", "a֮ۖ̀̕b", "a֮ۖ̀̕b", "a֮ۖ̀̕b" },
  { "à֮ۗ̕b", "à֮ۗ̕b", "à֮ۗ̕b", "à֮ۗ̕b", "à֮ۗ̕b" },
  { "a֮ۗ̀̕b", "a֮ۗ̀̕b", "a֮ۗ̀̕b", "a֮ۗ̀̕b", "a֮ۗ̀̕b" },
  { "à֮ۘ̕b", "à֮ۘ̕b", "à֮ۘ̕b", "à֮ۘ̕b", "à֮ۘ̕b" },
  { "a֮ۘ̀̕b", "a֮ۘ̀̕b", "a֮ۘ̀̕b", "a֮ۘ̀̕b", "a֮ۘ̀̕b" },
  { "à֮ۙ̕b", "à֮ۙ̕b", "à֮ۙ̕b", "à֮ۙ̕b", "à֮ۙ̕b" },
  { "a֮ۙ̀̕b", "a֮ۙ̀̕b", "a֮ۙ̀̕b", "a֮ۙ̀̕b", "a֮ۙ̀̕b" },
  { "à֮ۚ̕b", "à֮ۚ̕b", "à֮ۚ̕b", "à֮ۚ̕b", "à֮ۚ̕b" },
  { "a֮ۚ̀̕b", "a֮ۚ̀̕b", "a֮ۚ̀̕b", "a֮ۚ̀̕b", "a֮ۚ̀̕b" },
  { "à֮ۛ̕b", "à֮ۛ̕b", "à֮ۛ̕b", "à֮ۛ̕b", "à֮ۛ̕b" },
  { "a֮ۛ̀̕b", "a֮ۛ̀̕b", "a֮ۛ̀̕b", "a֮ۛ̀̕b", "a֮ۛ̀̕b" },
  { "à֮ۜ̕b", "à֮ۜ̕b", "à֮ۜ̕b", "à֮ۜ̕b", "à֮ۜ̕b" },
  { "a֮ۜ̀̕b", "a֮ۜ̀̕b", "a֮ۜ̀̕b", "a֮ۜ̀̕b", "a֮ۜ̀̕b" },
  { "à֮۟̕b", "à֮۟̕b", "à֮۟̕b", "à֮۟̕b", "à֮۟̕b" },
  { "a֮۟̀̕b", "a֮۟̀̕b", "a֮۟̀̕b", "a֮۟̀̕b", "a֮۟̀̕b" },
  { "à֮۠̕b", "à֮۠̕b", "à֮۠̕b", "à֮۠̕b", "à֮۠̕b" },
  { "a֮۠̀̕b", "a֮۠̀̕b", "a֮۠̀̕b", "a֮۠̀̕b", "a֮۠̀̕b" },
  { "à֮ۡ̕b", "à֮ۡ̕b", "à֮ۡ̕b", "à֮ۡ̕b", "à֮ۡ̕b" },
  { "a֮ۡ̀̕b", "a֮ۡ̀̕b", "a֮ۡ̀̕b", "a֮ۡ̀̕b", "a֮ۡ̀̕b" },
  { "à֮ۢ̕b", "à֮ۢ̕b", "à֮ۢ̕b", "à֮ۢ̕b", "à֮ۢ̕b" },
  { "a֮ۢ̀̕b", "a֮ۢ̀̕b", "a֮ۢ̀̕b", "a֮ۢ̀̕b", "a֮ۢ̀̕b" },
  { "a〪̖ۣ֚b", "a〪̖ۣ֚b", "a〪̖ۣ֚b", "a〪̖ۣ֚b", "a〪̖ۣ֚b" },
  { "a〪ۣ̖֚b", "a〪ۣ̖֚b", "a〪ۣ̖֚b", "a〪ۣ̖֚b", "a〪ۣ̖֚b" },
  { "à֮ۤ̕b", "à֮ۤ̕b", "à֮ۤ̕b", "à֮ۤ̕b", "à֮ۤ̕b" },
  { "a֮ۤ̀̕b", "a֮ۤ̀̕b", "a֮ۤ̀̕b", "a֮ۤ̀̕b", "a֮ۤ̀̕b" },
  { "à֮ۧ̕b", "à֮ۧ̕b", "à֮ۧ̕b", "à֮ۧ̕b", "à֮ۧ̕b" },
  { "a֮ۧ̀̕b", "a֮ۧ̀̕b", "a֮ۧ̀̕b", "a֮ۧ̀̕b", "a֮ۧ̀̕b" },
  { "à֮ۨ̕b", "à֮ۨ̕b", "à֮ۨ̕b", "à֮ۨ̕b", "à֮ۨ̕b" },
  { "a֮ۨ̀̕b", "a֮ۨ̀̕b", "a֮ۨ̀̕b", "a֮ۨ̀̕b", "a֮ۨ̀̕b" },
  { "a〪̖۪֚b", "a〪̖۪֚b", "a〪̖۪֚b", "a〪̖۪֚b", "a〪̖۪֚b" },
  { "a〪۪̖֚b", "a〪۪̖֚b", "a〪۪̖֚b", "a〪۪̖֚b", "a〪۪̖֚b" },
  { "à֮۫̕b", "à֮۫̕b", "à֮۫̕b", "à֮۫̕b", "à֮۫̕b" },
  { "a֮۫̀̕b", "a֮۫̀̕b", "a֮۫̀̕b", "a֮۫̀̕b", "a֮۫̀̕b" },
  { "à֮۬̕b", "à֮۬̕b", "à֮۬̕b", "à֮۬̕b", "à֮۬̕b" },
  { "a֮۬̀̕b", "a֮۬̀̕b", "a֮۬̀̕b", "a֮۬̀̕b", "a֮۬̀̕b" },
  { "a〪̖ۭ֚b", "a〪̖ۭ֚b", "a〪̖ۭ֚b", "a〪̖ۭ֚b", "a〪̖ۭ֚b" },
  { "a〪ۭ̖֚b", "a〪ۭ̖֚b", "a〪ۭ̖֚b", "a〪ۭ̖֚b", "a〪ۭ̖֚b" },
  { "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb" },
  { "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb", "aٰܑܑౕb" },
  { "à֮ܰ̕b", "à֮ܰ̕b", "à֮ܰ̕b", "à֮ܰ̕b", "à֮ܰ̕b" },
  { "a֮ܰ̀̕b", "a֮ܰ̀̕b", "a֮ܰ̀̕b", "a֮ܰ̀̕b", "a֮ܰ̀̕b" },
  { "a〪̖ܱ֚b", "a〪̖ܱ֚b", "a〪̖ܱ֚b", "a〪̖ܱ֚b", "a〪̖ܱ֚b" },
  { "a〪ܱ̖֚b", "a〪ܱ̖֚b", "a〪ܱ̖֚b", "a〪ܱ̖֚b", "a〪ܱ̖֚b" },
  { "à֮ܲ̕b", "à֮ܲ̕b", "à֮ܲ̕b", "à֮ܲ̕b", "à֮ܲ̕b" },
  { "a֮ܲ̀̕b", "a֮ܲ̀̕b", "a֮ܲ̀̕b", "a֮ܲ̀̕b", "a֮ܲ̀̕b" },
  { "à֮ܳ̕b", "à֮ܳ̕b", "à֮ܳ̕b", "à֮ܳ̕b", "à֮ܳ̕b" },
  { "a֮ܳ̀̕b", "a֮ܳ̀̕b", "a֮ܳ̀̕b", "a֮ܳ̀̕b", "a֮ܳ̀̕b" },
  { "a〪̖ܴ֚b", "a〪̖ܴ֚b", "a〪̖ܴ֚b", "a〪̖ܴ֚b", "a〪̖ܴ֚b" },
  { "a〪ܴ̖֚b", "a〪ܴ̖֚b", "a〪ܴ̖֚b", "a〪ܴ̖֚b", "a〪ܴ̖֚b" },
  { "à֮ܵ̕b", "à֮ܵ̕b", "à֮ܵ̕b", "à֮ܵ̕b", "à֮ܵ̕b" },
  { "a֮ܵ̀̕b", "a֮ܵ̀̕b", "a֮ܵ̀̕b", "a֮ܵ̀̕b", "a֮ܵ̀̕b" },
  { "à֮ܶ̕b", "à֮ܶ̕b", "à֮ܶ̕b", "à֮ܶ̕b", "à֮ܶ̕b" },
  { "a֮ܶ̀̕b", "a֮ܶ̀̕b", "a֮ܶ̀̕b", "a֮ܶ̀̕b", "a֮ܶ̀̕b" },
  { "a〪̖ܷ֚b", "a〪̖ܷ֚b", "a〪̖ܷ֚b", "a〪̖ܷ֚b", "a〪̖ܷ֚b" },
  { "a〪ܷ̖֚b", "a〪ܷ̖֚b", "a〪ܷ̖֚b", "a〪ܷ̖֚b", "a〪ܷ̖֚b" },
  { "a〪̖ܸ֚b", "a〪̖ܸ֚b", "a〪̖ܸ֚b", "a〪̖ܸ֚b", "a〪̖ܸ֚b" },
  { "a〪ܸ̖֚b", "a〪ܸ̖֚b", "a〪ܸ̖֚b", "a〪ܸ̖֚b", "a〪ܸ̖֚b" },
  { "a〪̖ܹ֚b", "a〪̖ܹ֚b", "a〪̖ܹ֚b", "a〪̖ܹ֚b", "a〪̖ܹ֚b" },
  { "a〪ܹ̖֚b", "a〪ܹ̖֚b", "a〪ܹ̖֚b", "a〪ܹ̖֚b", "a〪ܹ̖֚b" },
  { "à֮ܺ̕b", "à֮ܺ̕b", "à֮ܺ̕b", "à֮ܺ̕b", "à֮ܺ̕b" },
  { "a֮ܺ̀̕b", "a֮ܺ̀̕b", "a֮ܺ̀̕b", "a֮ܺ̀̕b", "a֮ܺ̀̕b" },
  { "a〪̖ܻ֚b", "a〪̖ܻ֚b", "a〪̖ܻ֚b", "a〪̖ܻ֚b", "a〪̖ܻ֚b" },
  { "a〪ܻ̖֚b", "a〪ܻ̖֚b", "a〪ܻ̖֚b", "a〪ܻ̖֚b", "a〪ܻ̖֚b" },
  { "a〪̖ܼ֚b", "a〪̖ܼ֚b", "a〪̖ܼ֚b", "a〪̖ܼ֚b", "a〪̖ܼ֚b" },
  { "a〪ܼ̖֚b", "a〪ܼ̖֚b", "a〪ܼ̖֚b", "a〪ܼ̖֚b", "a〪ܼ̖֚b" },
  { "à֮ܽ̕b", "à֮ܽ̕b", "à֮ܽ̕b", "à֮ܽ̕b", "à֮ܽ̕b" },
  { "a֮ܽ̀̕b", "a֮ܽ̀̕b", "a֮ܽ̀̕b", "a֮ܽ̀̕b", "a֮ܽ̀̕b" },
  { "a〪̖ܾ֚b", "a〪̖ܾ֚b", "a〪̖ܾ֚b", "a〪̖ܾ֚b", "a〪̖ܾ֚b" },
  { "a〪ܾ̖֚b", "a〪ܾ̖֚b", "a〪ܾ̖֚b", "a〪ܾ̖֚b", "a〪ܾ̖֚b" },
  { "à֮ܿ̕b", "à֮ܿ̕b", "à֮ܿ̕b", "à֮ܿ̕b", "à֮ܿ̕b" },
  { "a֮ܿ̀̕b", "a֮ܿ̀̕b", "a֮ܿ̀̕b", "a֮ܿ̀̕b", "a֮ܿ̀̕b" },
  { "à֮݀̕b", "à֮݀̕b", "à֮݀̕b", "à֮݀̕b", "à֮݀̕b" },
  { "a֮݀̀̕b", "a֮݀̀̕b", "a֮݀̀̕b", "a֮݀̀̕b", "a֮݀̀̕b" },
  { "à֮݁̕b", "à֮݁̕b", "à֮݁̕b", "à֮݁̕b", "à֮݁̕b" },
  { "a֮݁̀̕b", "a֮݁̀̕b", "a֮݁̀̕b", "a֮݁̀̕b", "a֮݁̀̕b" },
  { "a〪̖݂֚b", "a〪̖݂֚b", "a〪̖݂֚b", "a〪̖݂֚b", "a〪̖݂֚b" },
  { "a〪݂̖֚b", "a〪݂̖֚b", "a〪݂̖֚b", "a〪݂̖֚b", "a〪݂̖֚b" },
  { "à֮݃̕b", "à֮݃̕b", "à֮݃̕b", "à֮݃̕b", "à֮݃̕b" },
  { "a֮݃̀̕b", "a֮݃̀̕b", "a֮݃̀̕b", "a֮݃̀̕b", "a֮݃̀̕b" },
  { "a〪̖݄֚b", "a〪̖݄֚b", "a〪̖݄֚b", "a〪̖݄֚b", "a〪̖݄֚b" },
  { "a〪݄̖֚b", "a〪݄̖֚b", "a〪݄̖֚b", "a〪݄̖֚b", "a〪݄̖֚b" },
  { "à֮݅̕b", "à֮݅̕b", "à֮݅̕b", "à֮݅̕b", "à֮݅̕b" },
  { "a֮݅̀̕b", "a֮݅̀̕b", "a֮݅̀̕b", "a֮݅̀̕b", "a֮݅̀̕b" },
  { "a〪̖݆֚b", "a〪̖݆֚b", "a〪̖݆֚b", "a〪̖݆֚b", "a〪̖݆֚b" },
  { "a〪݆̖֚b", "a〪݆̖֚b", "a〪݆̖֚b", "a〪݆̖֚b", "a〪݆̖֚b" },
  { "à֮݇̕b", "à֮݇̕b", "à֮݇̕b", "à֮݇̕b", "à֮݇̕b" },
  { "a֮݇̀̕b", "a֮݇̀̕b", "a֮݇̀̕b", "a֮݇̀̕b", "a֮݇̀̕b" },
  { "a〪̖݈֚b", "a〪̖݈֚b", "a〪̖݈֚b", "a〪̖݈֚b", "a〪̖݈֚b" },
  { "a〪݈̖֚b", "a〪݈̖֚b", "a〪݈̖֚b", "a〪݈̖֚b", "a〪݈̖֚b" },
  { "à֮݉̕b", "à֮݉̕b", "à֮݉̕b", "à֮݉̕b", "à֮݉̕b" },
  { "a֮݉̀̕b", "a֮݉̀̕b", "a֮݉̀̕b", "a֮݉̀̕b", "a֮݉̀̕b" },
  { "à֮݊̕b", "à֮݊̕b", "à֮݊̕b", "à֮݊̕b", "à֮݊̕b" },
  { "a֮݊̀̕b", "a֮݊̀̕b", "a֮݊̀̕b", "a֮݊̀̕b", "a֮݊̀̕b" },
  { "à֮߫̕b", "à֮߫̕b", "à֮߫̕b", "à֮߫̕b", "à֮߫̕b" },
  { "a֮߫̀̕b", "a֮߫̀̕b", "a֮߫̀̕b", "a֮߫̀̕b", "a֮߫̀̕b" },
  { "à֮߬̕b", "à֮߬̕b", "à֮߬̕b", "à֮߬̕b", "à֮߬̕b" },
  { "a֮߬̀̕b", "a֮߬̀̕b", "a֮߬̀̕b", "a֮߬̀̕b", "a֮߬̀̕b" },
  { "à֮߭̕b", "à֮߭̕b", "à֮߭̕b", "à֮߭̕b", "à֮߭̕b" },
  { "a֮߭̀̕b", "a֮߭̀̕b", "a֮߭̀̕b", "a֮߭̀̕b", "a֮߭̀̕b" },
  { "à֮߮̕b", "à֮߮̕b", "à֮߮̕b", "à֮߮̕b", "à֮߮̕b" },
  { "a֮߮̀̕b", "a֮߮̀̕b", "a֮߮̀̕b", "a֮߮̀̕b", "a֮߮̀̕b" },
  { "à֮߯̕b", "à֮߯̕b", "à֮߯̕b", "à֮߯̕b", "à֮߯̕b" },
  { "a֮߯̀̕b", "a֮߯̀̕b", "a֮߯̀̕b", "a֮߯̀̕b", "a֮߯̀̕b" },
  { "à֮߰̕b", "à֮߰̕b", "à֮߰̕b", "à֮߰̕b", "à֮߰̕b" },
  { "a֮߰̀̕b", "a֮߰̀̕b", "a֮߰̀̕b", "a֮߰̀̕b", "a֮߰̀̕b" },
  { "à֮߱̕b", "à֮߱̕b", "à֮߱̕b", "à֮߱̕b", "à֮߱̕b" },
  { "a֮߱̀̕b", "a֮߱̀̕b", "a֮߱̀̕b", "a֮߱̀̕b", "a֮߱̀̕b" },
  { "a〪̖߲֚b", "a〪̖߲֚b", "a〪̖߲֚b", "a〪̖߲֚b", "a〪̖߲֚b" },
  { "a〪߲̖֚b", "a〪߲̖֚b", "a〪߲̖֚b", "a〪߲̖֚b", "a〪߲̖֚b" },
  { "à֮߳̕b", "à֮߳̕b", "à֮߳̕b", "à֮߳̕b", "à֮߳̕b" },
  { "a֮߳̀̕b", "a֮߳̀̕b", "a֮߳̀̕b", "a֮߳̀̕b", "a֮߳̀̕b" },
  { "a〪̖߽֚b", "a〪̖߽֚b", "a〪̖߽֚b", "a〪̖߽֚b", "a〪̖߽֚b" },
  { "a〪߽̖֚b", "a〪߽̖֚b", "a〪߽̖֚b", "a〪߽̖֚b", "a〪߽̖֚b" },
  { "à֮ࠖ̕b", "à֮ࠖ̕b", "à֮ࠖ̕b", "à֮ࠖ̕b", "à֮ࠖ̕b" },
  { "a֮ࠖ̀̕b", "a֮ࠖ̀̕b", "a֮ࠖ̀̕b", "a֮ࠖ̀̕b", "a֮ࠖ̀̕b" },
  { "à֮ࠗ̕b", "à֮ࠗ̕b", "à֮ࠗ̕b", "à֮ࠗ̕b", "à֮ࠗ̕b" },
  { "a֮ࠗ̀̕b", "a֮ࠗ̀̕b", "a֮ࠗ̀̕b", "a֮ࠗ̀̕b", "a֮ࠗ̀̕b" },
  { "à֮࠘̕b", "à֮࠘̕b", "à֮࠘̕b", "à֮࠘̕b", "à֮࠘̕b" },
  { "a֮࠘̀̕b", "a֮࠘̀̕b", "a֮࠘̀̕b", "a֮࠘̀̕b", "a֮࠘̀̕b" },
  { "à֮࠙̕b", "à֮࠙̕b", "à֮࠙̕b", "à֮࠙̕b", "à֮࠙̕b" },
  { "a֮࠙̀̕b", "a֮࠙̀̕b", "a֮࠙̀̕b", "a֮࠙̀̕b", "a֮࠙̀̕b" },
  { "à֮ࠛ̕b", "à֮ࠛ̕b", "à֮ࠛ̕b", "à֮ࠛ̕b", "à֮ࠛ̕b" },
  { "a֮ࠛ̀̕b", "a֮ࠛ̀̕b", "a֮ࠛ̀̕b", "a֮ࠛ̀̕b", "a֮ࠛ̀̕b" },
  { "à֮ࠜ̕b", "à֮ࠜ̕b", "à֮ࠜ̕b", "à֮ࠜ̕b", "à֮ࠜ̕b" },
  { "a֮ࠜ̀̕b", "a֮ࠜ̀̕b", "a֮ࠜ̀̕b", "a֮ࠜ̀̕b", "a֮ࠜ̀̕b" },
  { "à֮ࠝ̕b", "à֮ࠝ̕b", "à֮ࠝ̕b", "à֮ࠝ̕b", "à֮ࠝ̕b" },
  { "a֮ࠝ̀̕b", "a֮ࠝ̀̕b", "a֮ࠝ̀̕b", "a֮ࠝ̀̕b", "a֮ࠝ̀̕b" },
  { "à֮ࠞ̕b", "à֮ࠞ̕b", "à֮ࠞ̕b", "à֮ࠞ̕b", "à֮ࠞ̕b" },
  { "a֮ࠞ̀̕b", "a֮ࠞ̀̕b", "a֮ࠞ̀̕b", "a֮ࠞ̀̕b", "a֮ࠞ̀̕b" },
  { "à֮ࠟ̕b", "à֮ࠟ̕b", "à֮ࠟ̕b", "à֮ࠟ̕b", "à֮ࠟ̕b" },
  { "a֮ࠟ̀̕b", "a֮ࠟ̀̕b", "a֮ࠟ̀̕b", "a֮ࠟ̀̕b", "a֮ࠟ̀̕b" },
  { "à֮ࠠ̕b", "à֮ࠠ̕b", "à֮ࠠ̕b", "à֮ࠠ̕b", "à֮ࠠ̕b" },
  { "a֮ࠠ̀̕b", "a֮ࠠ̀̕b", "a֮ࠠ̀̕b", "a֮ࠠ̀̕b", "a֮ࠠ̀̕b" },
  { "à֮ࠡ̕b", "à֮ࠡ̕b", "à֮ࠡ̕b", "à֮ࠡ̕b", "à֮ࠡ̕b" },
  { "a֮ࠡ̀̕b", "a֮ࠡ̀̕b", "a֮ࠡ̀̕b", "a֮ࠡ̀̕b", "a֮ࠡ̀̕b" },
  { "à֮ࠢ̕b", "à֮ࠢ̕b", "à֮ࠢ̕b", "à֮ࠢ̕b", "à֮ࠢ̕b" },
  { "a֮ࠢ̀̕b", "a֮ࠢ̀̕b", "a֮ࠢ̀̕b", "a֮ࠢ̀̕b", "a֮ࠢ̀̕b" },
  { "à֮ࠣ̕b", "à֮ࠣ̕b", "à֮ࠣ̕b", "à֮ࠣ̕b", "à֮ࠣ̕b" },
  { "a֮ࠣ̀̕b", "a֮ࠣ̀̕b", "a֮ࠣ̀̕b", "a֮ࠣ̀̕b", "a֮ࠣ̀̕b" },
  { "à֮ࠥ̕b", "à֮ࠥ̕b", "à֮ࠥ̕b", "à֮ࠥ̕b", "à֮ࠥ̕b" },
  { "a֮ࠥ̀̕b", "a֮ࠥ̀̕b", "a֮ࠥ̀̕b", "a֮ࠥ̀̕b", "a֮ࠥ̀̕b" },
  { "à֮ࠦ̕b", "à֮ࠦ̕b", "à֮ࠦ̕b", "à֮ࠦ̕b", "à֮ࠦ̕b" },
  { "a֮ࠦ̀̕b", "a֮ࠦ̀̕b", "a֮ࠦ̀̕b", "a֮ࠦ̀̕b", "a֮ࠦ̀̕b" },
  { "à֮ࠧ̕b", "à֮ࠧ̕b", "à֮ࠧ̕b", "à֮ࠧ̕b", "à֮ࠧ̕b" },
  { "a֮ࠧ̀̕b", "a֮ࠧ̀̕b", "a֮ࠧ̀̕b", "a֮ࠧ̀̕b", "a֮ࠧ̀̕b" },
  { "à֮ࠩ̕b", "à֮ࠩ̕b", "à֮ࠩ̕b", "à֮ࠩ̕b", "à֮ࠩ̕b" },
  { "a֮ࠩ̀̕b", "a֮ࠩ̀̕b", "a֮ࠩ̀̕b", "a֮ࠩ̀̕b", "a֮ࠩ̀̕b" },
  { "à֮ࠪ̕b", "à֮ࠪ̕b", "à֮ࠪ̕b", "à֮ࠪ̕b", "à֮ࠪ̕b" },
  { "a֮ࠪ̀̕b", "a֮ࠪ̀̕b", "a֮ࠪ̀̕b", "a֮ࠪ̀̕b", "a֮ࠪ̀̕b" },
  { "à֮ࠫ̕b", "à֮ࠫ̕b", "à֮ࠫ̕b", "à֮ࠫ̕b", "à֮ࠫ̕b" },
  { "a֮ࠫ̀̕b", "a֮ࠫ̀̕b", "a֮ࠫ̀̕b", "a֮ࠫ̀̕b", "a֮ࠫ̀̕b" },
  { "à֮ࠬ̕b", "à֮ࠬ̕b", "à֮ࠬ̕b", "à֮ࠬ̕b", "à֮ࠬ̕b" },
  { "a֮ࠬ̀̕b", "a֮ࠬ̀̕b", "a֮ࠬ̀̕b", "a֮ࠬ̀̕b", "a֮ࠬ̀̕b" },
  { "à֮࠭̕b", "à֮࠭̕b", "à֮࠭̕b", "à֮࠭̕b", "à֮࠭̕b" },
  { "a֮࠭̀̕b", "a֮࠭̀̕b", "a֮࠭̀̕b", "a֮࠭̀̕b", "a֮࠭̀̕b" },
  { "a〪̖࡙֚b", "a〪̖࡙֚b", "a〪̖࡙֚b", "a〪̖࡙֚b", "a〪̖࡙֚b" },
  { "a〪࡙̖֚b", "a〪࡙̖֚b", "a〪࡙̖֚b", "a〪࡙̖֚b", "a〪࡙̖֚b" },
  { "a〪̖࡚֚b", "a〪̖࡚֚b", "a〪̖࡚֚b", "a〪̖࡚֚b", "a〪̖࡚֚b" },
  { "a〪࡚̖֚b", "a〪࡚̖֚b", "a〪࡚̖֚b", "a〪࡚̖֚b", "a〪࡚̖֚b" },
  { "a〪̖࡛֚b", "a〪̖࡛֚b", "a〪̖࡛֚b", "a〪̖࡛֚b", "a〪̖࡛֚b" },
  { "a〪࡛̖֚b", "a〪࡛̖֚b", "a〪࡛̖֚b", "a〪࡛̖֚b", "a〪࡛̖֚b" },
  { "a〪̖࣓֚b", "a〪̖࣓֚b", "a〪̖࣓֚b", "a〪̖࣓֚b", "a〪̖࣓֚b" },
  { "a〪࣓̖֚b", "a〪࣓̖֚b", "a〪࣓̖֚b", "a〪࣓̖֚b", "a〪࣓̖֚b" },
  { "à֮ࣔ̕b", "à֮ࣔ̕b", "à֮ࣔ̕b", "à֮ࣔ̕b", "à֮ࣔ̕b" },
  { "a֮ࣔ̀̕b", "a֮ࣔ̀̕b", "a֮ࣔ̀̕b", "a֮ࣔ̀̕b", "a֮ࣔ̀̕b" },
  { "à֮ࣕ̕b", "à֮ࣕ̕b", "à֮ࣕ̕b", "à֮ࣕ̕b", "à֮ࣕ̕b" },
  { "a֮ࣕ̀̕b", "a֮ࣕ̀̕b", "a֮ࣕ̀̕b", "a֮ࣕ̀̕b", "a֮ࣕ̀̕b" },
  { "à֮ࣖ̕b", "à֮ࣖ̕b", "à֮ࣖ̕b", "à֮ࣖ̕b", "à֮ࣖ̕b" },
  { "a֮ࣖ̀̕b", "a֮ࣖ̀̕b", "a֮ࣖ̀̕b", "a֮ࣖ̀̕b", "a֮ࣖ̀̕b" },
  { "à֮ࣗ̕b", "à֮ࣗ̕b", "à֮ࣗ̕b", "à֮ࣗ̕b", "à֮ࣗ̕b" },
  { "a֮ࣗ̀̕b", "a֮ࣗ̀̕b", "a֮ࣗ̀̕b", "a֮ࣗ̀̕b", "a֮ࣗ̀̕b" },
  { "à֮ࣘ̕b", "à֮ࣘ̕b", "à֮ࣘ̕b", "à֮ࣘ̕b", "à֮ࣘ̕b" },
  { "a֮ࣘ̀̕b", "a֮ࣘ̀̕b", "a֮ࣘ̀̕b", "a֮ࣘ̀̕b", "a֮ࣘ̀̕b" },
  { "à֮ࣙ̕b", "à֮ࣙ̕b", "à֮ࣙ̕b", "à֮ࣙ̕b", "à֮ࣙ̕b" },
  { "a֮ࣙ̀̕b", "a֮ࣙ̀̕b", "a֮ࣙ̀̕b", "a֮ࣙ̀̕b", "a֮ࣙ̀̕b" },
  { "à֮ࣚ̕b", "à֮ࣚ̕b", "à֮ࣚ̕b", "à֮ࣚ̕b", "à֮ࣚ̕b" },
  { "a֮ࣚ̀̕b", "a֮ࣚ̀̕b", "a֮ࣚ̀̕b", "a֮ࣚ̀̕b", "a֮ࣚ̀̕b" },
  { "à֮ࣛ̕b", "à֮ࣛ̕b", "à֮ࣛ̕b", "à֮ࣛ̕b", "à֮ࣛ̕b" },
  { "a֮ࣛ̀̕b", "a֮ࣛ̀̕b", "a֮ࣛ̀̕b", "a֮ࣛ̀̕b", "a֮ࣛ̀̕b" },
  { "à֮ࣜ̕b", "à֮ࣜ̕b", "à֮ࣜ̕b", "à֮ࣜ̕b", "à֮ࣜ̕b" },
  { "a֮ࣜ̀̕b", "a֮ࣜ̀̕b", "a֮ࣜ̀̕b", "a֮ࣜ̀̕b", "a֮ࣜ̀̕b" },
  { "à֮ࣝ̕b", "à֮ࣝ̕b", "à֮ࣝ̕b", "à֮ࣝ̕b", "à֮ࣝ̕b" },
  { "a֮ࣝ̀̕b", "a֮ࣝ̀̕b", "a֮ࣝ̀̕b", "a֮ࣝ̀̕b", "a֮ࣝ̀̕b" },
  { "à֮ࣞ̕b", "à֮ࣞ̕b", "à֮ࣞ̕b", "à֮ࣞ̕b", "à֮ࣞ̕b" },
  { "a֮ࣞ̀̕b", "a֮ࣞ̀̕b", "a֮ࣞ̀̕b", "a֮ࣞ̀̕b", "a֮ࣞ̀̕b" },
  { "à֮ࣟ̕b", "à֮ࣟ̕b", "à֮ࣟ̕b", "à֮ࣟ̕b", "à֮ࣟ̕b" },
  { "a֮ࣟ̀̕b", "a֮ࣟ̀̕b", "a֮ࣟ̀̕b", "a֮ࣟ̀̕b", "a֮ࣟ̀̕b" },
  { "à֮࣠̕b", "à֮࣠̕b", "à֮࣠̕b", "à֮࣠̕b", "à֮࣠̕b" },
  { "a֮࣠̀̕b", "a֮࣠̀̕b", "a֮࣠̀̕b", "a֮࣠̀̕b", "a֮࣠̀̕b" },
  { "à֮࣡̕b", "à֮࣡̕b", "à֮࣡̕b", "à֮࣡̕b", "à֮࣡̕b" },
  { "a֮࣡̀̕b", "a֮࣡̀̕b", "a֮࣡̀̕b", "a֮࣡̀̕b", "a֮࣡̀̕b" },
  { "a〪̖ࣣ֚b", "a〪̖ࣣ֚b", "a〪̖ࣣ֚b", "a〪̖ࣣ֚b", "a〪̖ࣣ֚b" },
  { "a〪ࣣ̖֚b", "a〪ࣣ̖֚b", "a〪ࣣ̖֚b", "a〪ࣣ̖֚b", "a〪ࣣ̖֚b" },
  { "à֮ࣤ̕b", "à֮ࣤ̕b", "à֮ࣤ̕b", "à֮ࣤ̕b", "à֮ࣤ̕b" },
  { "a֮ࣤ̀̕b", "a֮ࣤ̀̕b", "a֮ࣤ̀̕b", "a֮ࣤ̀̕b", "a֮ࣤ̀̕b" },
  { "à֮ࣥ̕b", "à֮ࣥ̕b", "à֮ࣥ̕b", "à֮ࣥ̕b", "à֮ࣥ̕b" },
  { "a֮ࣥ̀̕b", "a֮ࣥ̀̕b", "a֮ࣥ̀̕b", "a֮ࣥ̀̕b", "a֮ࣥ̀̕b" },
  { "a〪̖ࣦ֚b", "a〪̖ࣦ֚b", "a〪̖ࣦ֚b", "a〪̖ࣦ֚b", "a〪̖ࣦ֚b" },
  { "a〪ࣦ̖֚b", "a〪ࣦ̖֚b", "a〪ࣦ̖֚b", "a〪ࣦ̖֚b", "a〪ࣦ̖֚b" },
  { "à֮ࣧ̕b", "à֮ࣧ̕b", "à֮ࣧ̕b", "à֮ࣧ̕b", "à֮ࣧ̕b" },
  { "a֮ࣧ̀̕b", "a֮ࣧ̀̕b", "a֮ࣧ̀̕b", "a֮ࣧ̀̕b", "a֮ࣧ̀̕b" },
  { "à֮ࣨ̕b", "à֮ࣨ̕b", "à֮ࣨ̕b", "à֮ࣨ̕b", "à֮ࣨ̕b" },
  { "a֮ࣨ̀̕b", "a֮ࣨ̀̕b", "a֮ࣨ̀̕b", "a֮ࣨ̀̕b", "a֮ࣨ̀̕b" },
  { "a〪̖ࣩ֚b", "a〪̖ࣩ֚b", "a〪̖ࣩ֚b", "a〪̖ࣩ֚b", "a〪̖ࣩ֚b" },
  { "a〪ࣩ̖֚b", "a〪ࣩ̖֚b", "a〪ࣩ̖֚b", "a〪ࣩ̖֚b", "a〪ࣩ̖֚b" },
  { "à֮࣪̕b", "à֮࣪̕b", "à֮࣪̕b", "à֮࣪̕b", "à֮࣪̕b" },
  { "a֮࣪̀̕b", "a֮࣪̀̕b", "a֮࣪̀̕b", "a֮࣪̀̕b", "a֮࣪̀̕b" },
  { "à֮࣫̕b", "à֮࣫̕b", "à֮࣫̕b", "à֮࣫̕b", "à֮࣫̕b" },
  { "a֮࣫̀̕b", "a֮࣫̀̕b", "a֮࣫̀̕b", "a֮࣫̀̕b", "a֮࣫̀̕b" },
  { "à֮࣬̕b", "à֮࣬̕b", "à֮࣬̕b", "à֮࣬̕b", "à֮࣬̕b" },
  { "a֮࣬̀̕b", "a֮࣬̀̕b", "a֮࣬̀̕b", "a֮࣬̀̕b", "a֮࣬̀̕b" },
  { "a〪̖࣭֚b", "a〪̖࣭֚b", "a〪̖࣭֚b", "a〪̖࣭֚b", "a〪̖࣭֚b" },
  { "a〪࣭̖֚b", "a〪࣭̖֚b", "a〪࣭̖֚b", "a〪࣭̖֚b", "a〪࣭̖֚b" },
  { "a〪̖࣮֚b", "a〪̖࣮֚b", "a〪̖࣮֚b", "a〪̖࣮֚b", "a〪̖࣮֚b" },
  { "a〪࣮̖֚b", "a〪࣮̖֚b", "a〪࣮̖֚b", "a〪࣮̖֚b", "a〪࣮̖֚b" },
  { "a〪̖࣯֚b", "a〪̖࣯֚b", "a〪̖࣯֚b", "a〪̖࣯֚b", "a〪̖࣯֚b" },
  { "a〪࣯̖֚b", "a〪࣯̖֚b", "a〪࣯̖֚b", "a〪࣯̖֚b", "a〪࣯̖֚b" },
  { "aﬞًࣰٌb", "aﬞًࣰٌb", "aﬞًࣰٌb", "aﬞًࣰٌb", "aﬞًࣰٌb" },
  { "aﬞࣰًٌb", "aﬞࣰًٌb", "aﬞࣰًٌb", "aﬞࣰًٌb", "aﬞࣰًٌb" },
  { "aًٌࣱٍb", "aًٌࣱٍb", "aًٌࣱٍb", "aًٌࣱٍb", "aًٌࣱٍb" },
  { "aًࣱٌٍb", "aًࣱٌٍb", "aًࣱٌٍb", "aًࣱٌٍb", "aًࣱٌٍb" },
  { "aٌٍࣲؘb", "aٌٍࣲؘb", "aٌٍࣲؘb", "aٌٍࣲؘb", "aٌٍࣲؘb" },
  { "aٌࣲٍؘb", "aٌࣲٍؘb", "aٌࣲٍؘb", "aٌࣲٍؘb", "aٌࣲٍؘb" },
  { "à֮ࣳ̕b", "à֮ࣳ̕b", "à֮ࣳ̕b", "à֮ࣳ̕b", "à֮ࣳ̕b" },
  { "a֮ࣳ̀̕b", "a֮ࣳ̀̕b", "a֮ࣳ̀̕b", "a֮ࣳ̀̕b", "a֮ࣳ̀̕b" },
  { "à֮ࣴ̕b", "à֮ࣴ̕b", "à֮ࣴ̕b", "à֮ࣴ̕b", "à֮ࣴ̕b" },
  { "a֮ࣴ̀̕b", "a֮ࣴ̀̕b", "a֮ࣴ̀̕b", "a֮ࣴ̀̕b", "a֮ࣴ̀̕b" },
  { "à֮ࣵ̕b", "à֮ࣵ̕b", "à֮ࣵ̕b", "à֮ࣵ̕b", "à֮ࣵ̕b" },
  { "a֮ࣵ̀̕b", "a֮ࣵ̀̕b", "a֮ࣵ̀̕b", "a֮ࣵ̀̕b", "a֮ࣵ̀̕b" },
  { "a〪̖ࣶ֚b", "a〪̖ࣶ֚b", "a〪̖ࣶ֚b", "a〪̖ࣶ֚b", "a〪̖ࣶ֚b" },
  { "a〪ࣶ̖֚b", "a〪ࣶ̖֚b", "a〪ࣶ̖֚b", "a〪ࣶ̖֚b", "a〪ࣶ̖֚b" },
  { "à֮ࣷ̕b", "à֮ࣷ̕b", "à֮ࣷ̕b", "à֮ࣷ̕b", "à֮ࣷ̕b" },
  { "a֮ࣷ̀̕b", "a֮ࣷ̀̕b", "a֮ࣷ̀̕b", "a֮ࣷ̀̕b", "a֮ࣷ̀̕b" },
  { "à֮ࣸ̕b", "à֮ࣸ̕b", "à֮ࣸ̕b", "à֮ࣸ̕b", "à֮ࣸ̕b" },
  { "a֮ࣸ̀̕b", "a֮ࣸ̀̕b", "a֮ࣸ̀̕b", "a֮ࣸ̀̕b", "a֮ࣸ̀̕b" },
  { "a〪̖ࣹ֚b", "a〪̖ࣹ֚b", "a〪̖ࣹ֚b", "a〪̖ࣹ֚b", "a〪̖ࣹ֚b" },
  { "a〪ࣹ̖֚b", "a〪ࣹ̖֚b", "a〪ࣹ̖֚b", "a〪ࣹ̖֚b", "a〪ࣹ̖֚b" },
  { "a〪̖ࣺ֚b", "a〪̖ࣺ֚b", "a〪̖ࣺ֚b", "a〪̖ࣺ֚b", "a〪̖ࣺ֚b" },
  { "a〪ࣺ̖֚b", "a〪ࣺ̖֚b", "a〪ࣺ̖֚b", "a〪ࣺ̖֚b", "a〪ࣺ̖֚b" },
  { "à֮ࣻ̕b", "à֮ࣻ̕b", "à֮ࣻ̕b", "à֮ࣻ̕b", "à֮ࣻ̕b" },
  { "a֮ࣻ̀̕b", "a֮ࣻ̀̕b", "a֮ࣻ̀̕b", "a֮ࣻ̀̕b", "a֮ࣻ̀̕b" },
  { "à֮ࣼ̕b", "à֮ࣼ̕b", "à֮ࣼ̕b", "à֮ࣼ̕b", "à֮ࣼ̕b" },
  { "a֮ࣼ̀̕b", "a֮ࣼ̀̕b", "a֮ࣼ̀̕b", "a֮ࣼ̀̕b", "a֮ࣼ̀̕b" },
  { "à֮ࣽ̕b", "à֮ࣽ̕b", "à֮ࣽ̕b", "à֮ࣽ̕b", "à֮ࣽ̕b" },
  { "a֮ࣽ̀̕b", "a֮ࣽ̀̕b", "a֮ࣽ̀̕b", "a֮ࣽ̀̕b", "a֮ࣽ̀̕b" },
  { "à֮ࣾ̕b", "à֮ࣾ̕b", "à֮ࣾ̕b", "à֮ࣾ̕b", "à֮ࣾ̕b" },
  { "a֮ࣾ̀̕b", "a֮ࣾ̀̕b", "a֮ࣾ̀̕b", "a֮ࣾ̀̕b", "a֮ࣾ̀̕b" },
  { "à֮ࣿ̕b", "à֮ࣿ̕b", "à֮ࣿ̕b", "à֮ࣿ̕b", "à֮ࣿ̕b" },
  { "a֮ࣿ̀̕b", "a֮ࣿ̀̕b", "a֮ࣿ̀̕b", "a֮ࣿ̀̕b", "a֮ࣿ̀̕b" },
  { "a̴़़゙b", "a̴़़゙b", "a̴़़゙b", "a̴़़゙b", "a̴़़゙b" },
  { "a̴़़゙b", "a̴़़゙b", "a̴़़゙b", "a̴़़゙b", "a̴़़゙b" },
  { "a゙््ְb", "a゙््ְb", "a゙््ְb", "a゙््ְb", "a゙््ְb" },
  { "a゙््ְb", "a゙््ְb", "a゙््ְb", "a゙््ְb", "a゙््ְb" },
  { "à֮॑̕b", "à֮॑̕b", "à֮॑̕b", "à֮॑̕b", "à֮॑̕b" },
  { "a֮॑̀̕b", "a֮॑̀̕b", "a֮॑̀̕b", "a֮॑̀̕b", "a֮॑̀̕b" },
  { "a〪̖॒֚b", "a〪̖॒֚b", "a〪̖॒֚b", "a〪̖॒֚b", "a〪̖॒֚b" },
  { "a〪॒̖֚b", "a〪॒̖֚b", "a〪॒̖֚b", "a〪॒̖֚b", "a〪॒̖֚b" },
  { "à֮॓̕b", "à֮॓̕b", "à֮॓̕b", "à֮॓̕b", "à֮॓̕b" },
  { "a֮॓̀̕b", "a֮॓̀̕b", "a֮॓̀̕b", "a֮॓̀̕b", "a֮॓̀̕b" },
  { "à֮॔̕b", "à֮॔̕b", "à֮॔̕b", "à֮॔̕b", "à֮॔̕b" },
  { "a֮॔̀̕b", "a֮॔̀̕b", "a֮॔̀̕b", "a֮॔̀̕b", "a֮॔̀̕b" },
  { "a̴़়゙b", "a̴़়゙b", "a̴़়゙b", "a̴़়゙b", "a̴़়゙b" },
  { "a̴়़゙b", "a̴়़゙b", "a̴়़゙b", "a̴়़゙b", "a̴়़゙b" },
  { "a゙्্ְb", "a゙्্ְb", "a゙्্ְb", "a゙्্ְb", "a゙्্ְb" },
  { "a゙্्ְb", "a゙্्ְb", "a゙্्ְb", "a゙্्ְb", "a゙্्ְb" },
  { "à֮৾̕b", "à֮৾̕b", "à֮৾̕b", "à֮৾̕b", "à֮৾̕b" },
  { "a֮৾̀̕b", "a֮৾̀̕b", "a֮৾̀̕b", "a֮৾̀̕b", "a֮৾̀̕b" },
  { "a̴़਼゙b", "a̴़਼゙b", "a̴़਼゙b", "a̴़਼゙b", "a̴़਼゙b" },
  { "a̴਼़゙b", "a̴਼़゙b", "a̴਼़゙b", "a̴਼़゙b", "a̴਼़゙b" },
  { "a゙्੍ְb", "a゙्੍ְb", "a゙्੍ְb", "a゙्੍ְb", "a゙्੍ְb" },
  { "a゙੍्ְb", "a゙੍्ְb", "a゙੍्ְb", "a゙੍्ְb", "a゙੍्ְb" },
  { "a̴़઼゙b", "a̴़઼゙b", "a̴़઼゙b", "a̴़઼゙b", "a̴़઼゙b" },
  { "a̴઼़゙b", "a̴઼़゙b", "a̴઼़゙b", "a̴઼़゙b", "a̴઼़゙b" },
  { "a゙्્ְb", "a゙्્ְb", "a゙्્ְb", "a゙्્ְb", "a゙्્ְb" },
  { "a゙્्ְb", "a゙્्ְb", "a゙્्ְb", "a゙્्ְb", "a゙્्ְb" },
  { "a̴़଼゙b", "a̴़଼゙b", "a̴़଼゙b", "a̴़଼゙b", "a̴़଼゙b" },
  { "a̴଼़゙b", "a̴଼़゙b", "a̴଼़゙b", "a̴଼़゙b", "a̴଼़゙b" },
  { "a゙्୍ְb", "a゙्୍ְb", "a゙्୍ְb", "a゙्୍ְb", "a゙्୍ְb" },
  { "a゙୍्ְb", "a゙୍्ְb", "a゙୍्ְb", "a゙୍्ְb", "a゙୍्ְb" },
  { "a゙्்ְb", "a゙्்ְb", "a゙्்ְb", "a゙्்ְb", "a゙्்ְb" },
  { "a゙்्ְb", "a゙்्ְb", "a゙்्ְb", "a゙்्ְb", "a゙்्ְb" },
  { "a゙्్ְb", "a゙्్ְb", "a゙्్ְb", "a゙्్ְb", "a゙्్ְb" },
  { "a゙్्ְb", "a゙్्ְb", "a゙్्ְb", "a゙్्ְb", "a゙్्ְb" },
  { "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb" },
  { "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb", "aܑౕౕౖb" },
  { "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb" },
  { "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb", "aౕౖౖุb" },
  { "a̴़಼゙b", "a̴़಼゙b", "a̴़಼゙b", "a̴़಼゙b", "a̴़಼゙b" },
  { "a̴಼़゙b", "a̴಼़゙b", "a̴಼़゙b", "a̴಼़゙b", "a̴಼़゙b" },
  { "a゙्್ְb", "a゙्್ְb", "a゙्್ְb", "a゙्್ְb", "a゙्್ְb" },
  { "a゙್्ְb", "a゙್्ְb", "a゙್्ְb", "a゙್्ְb", "a゙್्ְb" },
  { "a゙्഻ְb", "a゙्഻ְb", "a゙्഻ְb", "a゙्഻ְb", "a゙्഻ְb" },
  { "a゙഻्ְb", "a゙഻्ְb", "a゙഻्ְb", "a゙഻्ְb", "a゙഻्ְb" },
  { "a゙्഼ְb", "a゙्഼ְb", "a゙्഼ְb", "a゙्഼ְb", "a゙्഼ְb" },
  { "a゙഼्ְb", "a゙഼्ְb", "a゙഼्ְb", "a゙഼्ְb", "a゙഼्ְb" },
  { "a゙्്ְb", "a゙्്ְb", "a゙्്ְb", "a゙्്ְb", "a゙्്ְb" },
  { "a゙്्ְb", "a゙്्ְb", "a゙്्ְb", "a゙്्ְb", "a゙്्ְb" },
  { "a゙्්ְb", "a゙्්ְb", "a゙्්ְb", "a゙्්ְb", "a゙्්ְb" },
  { "a゙්्ְb", "a゙්्ְb", "a゙්्ְb", "a゙්्ְb", "a゙්्ְb" },
  { "aౖุุ่b", "aౖุุ่b", "aౖุุ่b", "aౖุุ่b", "aౖุุ่b" },
  { "aౖุุ่b", "aౖุุ่b", "aౖุุ่b", "aౖุุ่b", "aౖุุ่b" },
  { "aౖุู่b", "aౖุู่b", "aౖุู่b", "aౖุู่b", "aౖุู่b" },
  { "aౖูุ่b", "aౖูุ่b", "aౖูุ่b", "aౖูุ่b", "aౖูุ่b" },
  { "a゙्ฺְb", "a゙्ฺְb", "a゙्ฺְb", "a゙्ฺְb", "a゙्ฺְb" },
  { "a゙ฺ्ְb", "a゙ฺ्ְb", "a゙ฺ्ְb", "a゙ฺ्ְb", "a゙ฺ्ְb" },
  { "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb" },
  { "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb", "aุ่่ຸb" },
  { "aุ่้ຸb", "aุ่้ຸb", "aุ่้ຸb", "aุ่้ຸb", "aุ่้ຸb" },
  { "aุ้่ຸb", "aุ้่ຸb", "aุ้่ຸb", "aุ้่ຸb", "aุ้่ຸb" },
  { "aุ่๊ຸb", "aุ่๊ຸb", "aุ่๊ຸb", "aุ่๊ຸb", "aุ่๊ຸb" },
  { "aุ๊่ຸb", "aุ๊่ຸb", "aุ๊่ຸb", "aุ๊่ຸb", "aุ๊่ຸb" },
  { "aุ่๋ຸb", "aุ่๋ຸb", "aุ่๋ຸb", "aุ่๋ຸb", "aุ่๋ຸb" },
  { "aุ๋่ຸb", "aุ๋่ຸb", "aุ๋่ຸb", "aุ๋่ຸb", "aุ๋่ຸb" },
  { "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b" },
  { "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b", "a่ຸຸ່b" },
  { "a่ຸູ່b", "a่ຸູ່b", "a่ຸູ່b", "a่ຸູ່b", "a่ຸູ່b" },
  { "a่ູຸ່b", "a่ູຸ່b", "a่ູຸ່b", "a่ູຸ່b", "a่ູຸ່b" },
  { "a゙्຺ְb", "a゙्຺ְb", "a゙्຺ְb", "a゙्຺ְb", "a゙्຺ְb" },
  { "a゙຺्ְb", "a゙຺्ְb", "a゙຺्ְb", "a゙຺्ְb", "a゙຺्ְb" },
  { "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb" },
  { "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb", "aຸ່່ཱb" },
  { "aຸ່້ཱb", "aຸ່້ཱb", "aຸ່້ཱb", "aຸ່້ཱb", "aຸ່້ཱb" },
  { "aຸ້່ཱb", "aຸ້່ཱb", "aຸ້່ཱb", "aຸ້່ཱb", "aຸ້່ཱb" },
  { "aຸ່໊ཱb", "aຸ່໊ཱb", "aຸ່໊ཱb", "aຸ່໊ཱb", "aຸ່໊ཱb" },
  { "aຸ໊່ཱb", "aຸ໊່ཱb", "aຸ໊່ཱb", "aຸ໊່ཱb", "aຸ໊່ཱb" },
  { "aຸ່໋ཱb", "aຸ່໋ཱb", "aຸ່໋ཱb", "aຸ່໋ཱb", "aຸ່໋ཱb" },
  { "aຸ໋່ཱb", "aຸ໋່ཱb", "aຸ໋່ཱb", "aຸ໋່ཱb", "aຸ໋່ཱb" },
  { "a〪̖༘֚b", "a〪̖༘֚b", "a〪̖༘֚b", "a〪̖༘֚b", "a〪̖༘֚b" },
  { "a〪༘̖֚b", "a〪༘̖֚b", "a〪༘̖֚b", "a〪༘̖֚b", "a〪༘̖֚b" },
  { "a〪̖༙֚b", "a〪̖༙֚b", "a〪̖༙֚b", "a〪̖༙֚b", "a〪̖༙֚b" },
  { "a〪༙̖֚b", "a〪༙̖֚b", "a〪༙̖֚b", "a〪༙̖֚b", "a〪༙̖֚b" },
  { "a〪̖༵֚b", "a〪̖༵֚b", "a〪̖༵֚b", "a〪̖༵֚b", "a〪̖༵֚b" },
  { "a〪༵̖֚b", "a〪༵̖֚b", "a〪༵̖֚b", "a〪༵̖֚b", "a〪༵̖֚b" },
  { "a〪̖༷֚b", "a〪̖༷֚b", "a〪̖༷֚b", "a〪̖༷֚b", "a〪̖༷֚b" },
  { "a〪༷̖֚b", "a〪༷̖֚b", "a〪༷̖֚b", "a〪༷̖֚b", "a〪༷̖֚b" },
  { "a᷎̛༹〪b", "a᷎̛༹〪b", "a᷎̛༹〪b", "a᷎̛༹〪b", "a᷎̛༹〪b" },
  { "a᷎༹̛〪b", "a᷎༹̛〪b", "a᷎༹̛〪b", "a᷎༹̛〪b", "a᷎༹̛〪b" },
  { "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb" },
  { "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb", "a່ཱཱིb" },
  { "aཱིིུb", "aཱིིུb", "aཱིིུb", "aཱིིུb", "aཱིིུb" },
  { "aཱིིུb", "aཱིིུb", "aཱིིུb", "aཱིིུb", "aཱིིུb" },
  { "aིུུ̡b", "aིུུ̡b", "aིུུ̡b", "aིུུ̡b", "aིུུ̡b" },
  { "aིུུ̡b", "aིུུ̡b", "aིུུ̡b", "aིུུ̡b", "aིུུ̡b" },
  { "aཱིེུb", "aཱིེུb", "aཱིེུb", "aཱིེུb", "aཱིེུb" },
  { "aཱེིུb", "aཱེིུb", "aཱེིུb", "aཱེིུb", "aཱེིུb" },
  { "aཱིཻུb", "aཱིཻུb", "aཱིཻུb", "aཱིཻུb", "aཱིཻུb" },
  { "aཱཻིུb", "aཱཻིུb", "aཱཻིུb", "aཱཻིུb", "aཱཻིུb" },
  { "aཱིོུb", "aཱིོུb", "aཱིོུb", "aཱིོུb", "aཱིོུb" },
  { "aཱོིུb", "aཱོིུb", "aཱོིུb", "aཱོིུb", "aཱོིུb" },
  { "aཱིཽུb", "aཱིཽུb", "aཱིཽུb", "aཱིཽུb", "aཱིཽུb" },
  { "aཱཽིུb", "aཱཽིུb", "aཱཽིུb", "aཱཽིུb", "aཱཽིུb" },
  { "aཱིྀུb", "aཱིྀུb", "aཱིྀུb", "aཱིྀུb", "aཱིྀུb" },
  { "aཱྀིུb", "aཱྀིུb", "aཱྀིུb", "aཱྀིུb", "aཱྀིུb" },
  { "à֮ྂ̕b", "à֮ྂ̕b", "à֮ྂ̕b", "à֮ྂ̕b", "à֮ྂ̕b" },
  { "a֮ྂ̀̕b", "a֮ྂ̀̕b", "a֮ྂ̀̕b", "a֮ྂ̀̕b", "a֮ྂ̀̕b" },
  { "à֮ྃ̕b", "à֮ྃ̕b", "à֮ྃ̕b", "à֮ྃ̕b", "à֮ྃ̕b" },
  { "a֮ྃ̀̕b", "a֮ྃ̀̕b", "a֮ྃ̀̕b", "a֮ྃ̀̕b", "a֮ྃ̀̕b" },
  { "a゙्྄ְb", "a゙्྄ְb", "a゙्྄ְb", "a゙्྄ְb", "a゙्྄ְb" },
  { "a゙྄्ְb", "a゙྄्ְb", "a゙྄्ְb", "a゙྄्ְb", "a゙྄्ְb" },
  { "à֮྆̕b", "à֮྆̕b", "à֮྆̕b", "à֮྆̕b", "à֮྆̕b" },
  { "a֮྆̀̕b", "a֮྆̀̕b", "a֮྆̀̕b", "a֮྆̀̕b", "a֮྆̀̕b" },
  { "à֮྇̕b", "à֮྇̕b", "à֮྇̕b", "à֮྇̕b", "à֮྇̕b" },
  { "a֮྇̀̕b", "a֮྇̀̕b", "a֮྇̀̕b", "a֮྇̀̕b", "a֮྇̀̕b" },
  { "a〪̖࿆֚b", "a〪̖࿆֚b", "a〪̖࿆֚b", "a〪̖࿆֚b", "a〪̖࿆֚b" },
  { "a〪࿆̖֚b", "a〪࿆̖֚b", "a〪࿆̖֚b", "a〪࿆̖֚b", "a〪࿆̖֚b" },
  { "a̴़့゙b", "a̴़့゙b", "a̴़့゙b", "a̴़့゙b", "a̴़့゙b" },
  { "a̴့़゙b", "a̴့़゙b", "a̴့़゙b", "a̴့़゙b", "a̴့़゙b" },
  { "a゙्္ְb", "a゙्္ְb", "a゙्္ְb", "a゙्္ְb", "a゙्္ְb" },
  { "a゙္्ְb", "a゙္्ְb", "a゙္्ְb", "a゙္्ְb", "a゙္्ְb" },
  { "a゙्်ְb", "a゙्်ְb", "a゙्်ְb", "a゙्်ְb", "a゙्်ְb" },
  { "a゙်्ְb", "a゙်्ְb", "a゙်्ְb", "a゙်्ְb", "a゙်्ְb" },
  { "a〪̖ႍ֚b", "a〪̖ႍ֚b", "a〪̖ႍ֚b", "a〪̖ႍ֚b", "a〪̖ႍ֚b" },
  { "a〪ႍ̖֚b", "a〪ႍ̖֚b", "a〪ႍ̖֚b", "a〪ႍ̖֚b", "a〪ႍ̖֚b" },
  { "à֮፝̕b", "à֮፝̕b", "à֮፝̕b", "à֮፝̕b", "à֮፝̕b" },
  { "a֮፝̀̕b", "a֮፝̀̕b", "a֮፝̀̕b", "a֮፝̀̕b", "a֮፝̀̕b" },
  { "à֮፞̕b", "à֮፞̕b", "à֮፞̕b", "à֮፞̕b", "à֮፞̕b" },
  { "a֮፞̀̕b", "a֮፞̀̕b", "a֮፞̀̕b", "a֮፞̀̕b", "a֮፞̀̕b" },
  { "à֮፟̕b", "à֮፟̕b", "à֮፟̕b", "à֮፟̕b", "à֮፟̕b" },
  { "a֮፟̀̕b", "a֮፟̀̕b", "a֮፟̀̕b", "a֮፟̀̕b", "a֮፟̀̕b" },
  { "a゙्᜔ְb", "a゙्᜔ְb", "a゙्᜔ְb", "a゙्᜔ְb", "a゙्᜔ְb" },
  { "a゙᜔्ְb", "a゙᜔्ְb", "a゙᜔्ְb", "a゙᜔्ְb", "a゙᜔्ְb" },
  { "a゙्᜴ְb", "a゙्᜴ְb", "a゙्᜴ְb", "a゙्᜴ְb", "a゙्᜴ְb" },
  { "a゙᜴्ְb", "a゙᜴्ְb", "a゙᜴्ְb", "a゙᜴्ְb", "a゙᜴्ְb" },
  { "a゙्្ְb", "a゙्្ְb", "a゙्្ְb", "a゙्្ְb", "a゙्្ְb" },
  { "a゙្्ְb", "a゙្्ְb", "a゙្्ְb", "a゙្्ְb", "a゙្्ְb" },
  { "à֮៝̕b", "à֮៝̕b", "à֮៝̕b", "à֮៝̕b", "à֮៝̕b" },
  { "a֮៝̀̕b", "a֮៝̀̕b", "a֮៝̀̕b", "a֮៝̀̕b", "a֮៝̀̕b" },
  { "à𝅭֮ᢩb", "à𝅭֮ᢩb", "à𝅭֮ᢩb", "à𝅭֮ᢩb", "à𝅭֮ᢩb" },
  { "à𝅭ᢩ֮b", "à𝅭ᢩ֮b", "à𝅭ᢩ֮b", "à𝅭ᢩ֮b", "à𝅭ᢩ֮b" },
  { "a̖֚᤹〮b", "a̖֚᤹〮b", "a̖֚᤹〮b", "a̖֚᤹〮b", "a̖֚᤹〮b" },
  { "a̖᤹֚〮b", "a̖᤹֚〮b", "a̖᤹֚〮b", "a̖᤹֚〮b", "a̖᤹֚〮b" },
  { "à֮᤺̕b", "à֮᤺̕b", "à֮᤺̕b", "à֮᤺̕b", "à֮᤺̕b" },
  { "a֮᤺̀̕b", "a֮᤺̀̕b", "a֮᤺̀̕b", "a֮᤺̀̕b", "a֮᤺̀̕b" },
  { "a〪̖᤻֚b", "a〪̖᤻֚b", "a〪̖᤻֚b", "a〪̖᤻֚b", "a〪̖᤻֚b" },
  { "a〪᤻̖֚b", "a〪᤻̖֚b", "a〪᤻̖֚b", "a〪᤻̖֚b", "a〪᤻̖֚b" },
  { "à֮ᨗ̕b", "à֮ᨗ̕b", "à֮ᨗ̕b", "à֮ᨗ̕b", "à֮ᨗ̕b" },
  { "a֮ᨗ̀̕b", "a֮ᨗ̀̕b", "a֮ᨗ̀̕b", "a֮ᨗ̀̕b", "a֮ᨗ̀̕b" },
  { "a〪̖ᨘ֚b", "a〪̖ᨘ֚b", "a〪̖ᨘ֚b", "a〪̖ᨘ֚b", "a〪̖ᨘ֚b" },
  { "a〪ᨘ̖֚b", "a〪ᨘ̖֚b", "a〪ᨘ̖֚b", "a〪ᨘ̖֚b", "a〪ᨘ̖֚b" },
  { "a゙्᩠ְb", "a゙्᩠ְb", "a゙्᩠ְb", "a゙्᩠ְb", "a゙्᩠ְb" },
  { "a゙᩠्ְb", "a゙᩠्ְb", "a゙᩠्ְb", "a゙᩠्ְb", "a゙᩠्ְb" },
  { "à֮᩵̕b", "à֮᩵̕b", "à֮᩵̕b", "à֮᩵̕b", "à֮᩵̕b" },
  { "a֮᩵̀̕b", "a֮᩵̀̕b", "a֮᩵̀̕b", "a֮᩵̀̕b", "a֮᩵̀̕b" },
  { "à֮᩶̕b", "à֮᩶̕b", "à֮᩶̕b", "à֮᩶̕b", "à֮᩶̕b" },
  { "a֮᩶̀̕b", "a֮᩶̀̕b", "a֮᩶̀̕b", "a֮᩶̀̕b", "a֮᩶̀̕b" },
  { "à֮᩷̕b", "à֮᩷̕b", "à֮᩷̕b", "à֮᩷̕b", "à֮᩷̕b" },
  { "a֮᩷̀̕b", "a֮᩷̀̕b", "a֮᩷̀̕b", "a֮᩷̀̕b", "a֮᩷̀̕b" },
  { "à֮᩸̕b", "à֮᩸̕b", "à֮᩸̕b", "à֮᩸̕b", "à֮᩸̕b" },
  { "a֮᩸̀̕b", "a֮᩸̀̕b", "a֮᩸̀̕b", "a֮᩸̀̕b", "a֮᩸̀̕b" },
  { "à֮᩹̕b", "à֮᩹̕b", "à֮᩹̕b", "à֮᩹̕b", "à֮᩹̕b" },
  { "a֮᩹̀̕b", "a֮᩹̀̕b", "a֮᩹̀̕b", "a֮᩹̀̕b", "a֮᩹̀̕b" },
  { "à֮᩺̕b", "à֮᩺̕b", "à֮᩺̕b", "à֮᩺̕b", "à֮᩺̕b" },
  { "a֮᩺̀̕b", "a֮᩺̀̕b", "a֮᩺̀̕b", "a֮᩺̀̕b", "a֮᩺̀̕b" },
  { "à֮᩻̕b", "à֮᩻̕b", "à֮᩻̕b", "à֮᩻̕b", "à֮᩻̕b" },
  { "a֮᩻̀̕b", "a֮᩻̀̕b", "a֮᩻̀̕b", "a֮᩻̀̕b", "a֮᩻̀̕b" },
  { "à֮᩼̕b", "à֮᩼̕b", "à֮᩼̕b", "à֮᩼̕b", "à֮᩼̕b" },
  { "a֮᩼̀̕b", "a֮᩼̀̕b", "a֮᩼̀̕b", "a֮᩼̀̕b", "a֮᩼̀̕b" },
  { "a〪̖᩿֚b", "a〪̖᩿֚b", "a〪̖᩿֚b", "a〪̖᩿֚b", "a〪̖᩿֚b" },
  { "a〪᩿̖֚b", "a〪᩿̖֚b", "a〪᩿̖֚b", "a〪᩿̖֚b", "a〪᩿̖֚b" },
  { "à֮᪰̕b", "à֮᪰̕b", "à֮᪰̕b", "à֮᪰̕b", "à֮᪰̕b" },
  { "a֮᪰̀̕b", "a֮᪰̀̕b", "a֮᪰̀̕b", "a֮᪰̀̕b", "a֮᪰̀̕b" },
  { "à֮᪱̕b", "à֮᪱̕b", "à֮᪱̕b", "à֮᪱̕b", "à֮᪱̕b" },
  { "a֮᪱̀̕b", "a֮᪱̀̕b", "a֮᪱̀̕b", "a֮᪱̀̕b", "a֮᪱̀̕b" },
  { "à֮᪲̕b", "à֮᪲̕b", "à֮᪲̕b", "à֮᪲̕b", "à֮᪲̕b" },
  { "a֮᪲̀̕b", "a֮᪲̀̕b", "a֮᪲̀̕b", "a֮᪲̀̕b", "a֮᪲̀̕b" },
  { "à֮᪳̕b", "à֮᪳̕b", "à֮᪳̕b", "à֮᪳̕b", "à֮᪳̕b" },
  { "a֮᪳̀̕b", "a֮᪳̀̕b", "a֮᪳̀̕b", "a֮᪳̀̕b", "a֮᪳̀̕b" },
  { "à֮᪴̕b", "à֮᪴̕b", "à֮᪴̕b", "à֮᪴̕b", "à֮᪴̕b" },
  { "a֮᪴̀̕b", "a֮᪴̀̕b", "a֮᪴̀̕b", "a֮᪴̀̕b", "a֮᪴̀̕b" },
  { "a〪̖᪵֚b", "a〪̖᪵֚b", "a〪̖᪵֚b", "a〪̖᪵֚b", "a〪̖᪵֚b" },
  { "a〪᪵̖֚b", "a〪᪵̖֚b", "a〪᪵̖֚b", "a〪᪵̖֚b", "a〪᪵̖֚b" },
  { "a〪̖᪶֚b", "a〪̖᪶֚b", "a〪̖᪶֚b", "a〪̖᪶֚b", "a〪̖᪶֚b" },
  { "a〪᪶̖֚b", "a〪᪶̖֚b", "a〪᪶̖֚b", "a〪᪶̖֚b", "a〪᪶̖֚b" },
  { "a〪̖᪷֚b", "a〪̖᪷֚b", "a〪̖᪷֚b", "a〪̖᪷֚b", "a〪̖᪷֚b" },
  { "a〪᪷̖֚b", "a〪᪷̖֚b", "a〪᪷̖֚b", "a〪᪷̖֚b", "a〪᪷̖֚b" },
  { "a〪̖᪸֚b", "a〪̖᪸֚b", "a〪̖᪸֚b", "a〪̖᪸֚b", "a〪̖᪸֚b" },
  { "a〪᪸̖֚b", "a〪᪸̖֚b", "a〪᪸̖֚b", "a〪᪸̖֚b", "a〪᪸̖֚b" },
  { "a〪̖᪹֚b", "a〪̖᪹֚b", "a〪̖᪹֚b", "a〪̖᪹֚b", "a〪̖᪹֚b" },
  { "a〪᪹̖֚b", "a〪᪹̖֚b", "a〪᪹̖֚b", "a〪᪹̖֚b", "a〪᪹̖֚b" },
  { "a〪̖᪺֚b", "a〪̖᪺֚b", "a〪̖᪺֚b", "a〪̖᪺֚b", "a〪̖᪺֚b" },
  { "a〪᪺̖֚b", "a〪᪺̖֚b", "a〪᪺̖֚b", "a〪᪺̖֚b", "a〪᪺̖֚b" },
  { "à֮᪻̕b", "à֮᪻̕b", "à֮᪻̕b", "à֮᪻̕b", "à֮᪻̕b" },
  { "a֮᪻̀̕b", "a֮᪻̀̕b", "a֮᪻̀̕b", "a֮᪻̀̕b", "a֮᪻̀̕b" },
  { "à֮᪼̕b", "à֮᪼̕b", "à֮᪼̕b", "à֮᪼̕b", "à֮᪼̕b" },
  { "a֮᪼̀̕b", "a֮᪼̀̕b", "a֮᪼̀̕b", "a֮᪼̀̕b", "a֮᪼̀̕b" },
  { "a〪̖᪽֚b", "a〪̖᪽֚b", "a〪̖᪽֚b", "a〪̖᪽֚b", "a〪̖᪽֚b" },
  { "a〪᪽̖֚b", "a〪᪽̖֚b", "a〪᪽̖֚b", "a〪᪽̖֚b", "a〪᪽̖֚b" },
  { "a̴़᬴゙b", "a̴़᬴゙b", "a̴़᬴゙b", "a̴़᬴゙b", "a̴़᬴゙b" },
  { "a̴᬴़゙b", "a̴᬴़゙b", "a̴᬴़゙b", "a̴᬴़゙b", "a̴᬴़゙b" },
  { "a゙्᭄ְb", "a゙्᭄ְb", "a゙्᭄ְb", "a゙्᭄ְb", "a゙्᭄ְb" },
  { "a゙᭄्ְb", "a゙᭄्ְb", "a゙᭄्ְb", "a゙᭄्ְb", "a゙᭄्ְb" },
  { "à֮᭫̕b", "à֮᭫̕b", "à֮᭫̕b", "à֮᭫̕b", "à֮᭫̕b" },
  { "a֮᭫̀̕b", "a֮᭫̀̕b", "a֮᭫̀̕b", "a֮᭫̀̕b", "a֮᭫̀̕b" },
  { "a〪̖᭬֚b", "a〪̖᭬֚b", "a〪̖᭬֚b", "a〪̖᭬֚b", "a〪̖᭬֚b" },
  { "a〪᭬̖֚b", "a〪᭬̖֚b", "a〪᭬̖֚b", "a〪᭬̖֚b", "a〪᭬̖֚b" },
  { "à֮᭭̕b", "à֮᭭̕b", "à֮᭭̕b", "à֮᭭̕b", "à֮᭭̕b" },
  { "a֮᭭̀̕b", "a֮᭭̀̕b", "a֮᭭̀̕b", "a֮᭭̀̕b", "a֮᭭̀̕b" },
  { "à֮᭮̕b", "à֮᭮̕b", "à֮᭮̕b", "à֮᭮̕b", "à֮᭮̕b" },
  { "a֮᭮̀̕b", "a֮᭮̀̕b", "a֮᭮̀̕b", "a֮᭮̀̕b", "a֮᭮̀̕b" },
  { "à֮᭯̕b", "à֮᭯̕b", "à֮᭯̕b", "à֮᭯̕b", "à֮᭯̕b" },
  { "a֮᭯̀̕b", "a֮᭯̀̕b", "a֮᭯̀̕b", "a֮᭯̀̕b", "a֮᭯̀̕b" },
  { "à֮᭰̕b", "à֮᭰̕b", "à֮᭰̕b", "à֮᭰̕b", "à֮᭰̕b" },
  { "a֮᭰̀̕b", "a֮᭰̀̕b", "a֮᭰̀̕b", "a֮᭰̀̕b", "a֮᭰̀̕b" },
  { "à֮᭱̕b", "à֮᭱̕b", "à֮᭱̕b", "à֮᭱̕b", "à֮᭱̕b" },
  { "a֮᭱̀̕b", "a֮᭱̀̕b", "a֮᭱̀̕b", "a֮᭱̀̕b", "a֮᭱̀̕b" },
  { "à֮᭲̕b", "à֮᭲̕b", "à֮᭲̕b", "à֮᭲̕b", "à֮᭲̕b" },
  { "a֮᭲̀̕b", "a֮᭲̀̕b", "a֮᭲̀̕b", "a֮᭲̀̕b", "a֮᭲̀̕b" },
  { "à֮᭳̕b", "à֮᭳̕b", "à֮᭳̕b", "à֮᭳̕b", "à֮᭳̕b" },
  { "a֮᭳̀̕b", "a֮᭳̀̕b", "a֮᭳̀̕b", "a֮᭳̀̕b", "a֮᭳̀̕b" },
  { "a゙्᮪ְb", "a゙्᮪ְb", "a゙्᮪ְb", "a゙्᮪ְb", "a゙्᮪ְb" },
  { "a゙᮪्ְb", "a゙᮪्ְb", "a゙᮪्ְb", "a゙᮪्ְb", "a゙᮪्ְb" },
  { "a゙्᮫ְb", "a゙्᮫ְb", "a゙्᮫ְb", "a゙्᮫ְb", "a゙्᮫ְb" },
  { "a゙᮫्ְb", "a゙᮫्ְb", "a゙᮫्ְb", "a゙᮫्ְb", "a゙᮫्ְb" },
  { "a̴़᯦゙b", "a̴़᯦゙b", "a̴़᯦゙b", "a̴़᯦゙b", "a̴़᯦゙b" },
  { "a̴᯦़゙b", "a̴᯦़゙b", "a̴᯦़゙b", "a̴᯦़゙b", "a̴᯦़゙b" },
  { "a゙्᯲ְb", "a゙्᯲ְb", "a゙्᯲ְb", "a゙्᯲ְb", "a゙्᯲ְb" },
  { "a゙᯲्ְb", "a゙᯲्ְb", "a゙᯲्ְb", "a゙᯲्ְb", "a゙᯲्ְb" },
  { "a゙्᯳ְb", "a゙्᯳ְb", "a゙्᯳ְb", "a゙्᯳ְb", "a゙्᯳ְb" },
  { "a゙᯳्ְb", "a゙᯳्ְb", "a゙᯳्ְb", "a゙᯳्ְb", "a゙᯳्ְb" },
  { "a̴़᰷゙b", "a̴़᰷゙b", "a̴़᰷゙b", "a̴़᰷゙b", "a̴़᰷゙b" },
  { "a̴᰷़゙b", "a̴᰷़゙b", "a̴᰷़゙b", "a̴᰷़゙b", "a̴᰷़゙b" },
  { "à֮᳐̕b", "à֮᳐̕b", "à֮᳐̕b", "à֮᳐̕b", "à֮᳐̕b" },
  { "a֮᳐̀̕b", "a֮᳐̀̕b", "a֮᳐̀̕b", "a֮᳐̀̕b", "a֮᳐̀̕b" },
  { "à֮᳑̕b", "à֮᳑̕b", "à֮᳑̕b", "à֮᳑̕b", "à֮᳑̕b" },
  { "a֮᳑̀̕b", "a֮᳑̀̕b", "a֮᳑̀̕b", "a֮᳑̀̕b", "a֮᳑̀̕b" },
  { "à֮᳒̕b", "à֮᳒̕b", "à֮᳒̕b", "à֮᳒̕b", "à֮᳒̕b" },
  { "a֮᳒̀̕b", "a֮᳒̀̕b", "a֮᳒̀̕b", "a֮᳒̀̕b", "a֮᳒̀̕b" },
  { "a̴᳔़b", "a̴᳔़b", "a̴᳔़b", "a̴᳔़b", "a̴᳔़b" },
  { "a᳔̴़b", "a᳔̴़b", "a᳔̴़b", "a᳔̴़b", "a᳔̴़b" },
  { "a〪̖᳕֚b", "a〪̖᳕֚b", "a〪̖᳕֚b", "a〪̖᳕֚b", "a〪̖᳕֚b" },
  { "a〪᳕̖֚b", "a〪᳕̖֚b", "a〪᳕̖֚b", "a〪᳕̖֚b", "a〪᳕̖֚b" },
  { "a〪̖᳖֚b", "a〪̖᳖֚b", "a〪̖᳖֚b", "a〪̖᳖֚b", "a〪̖᳖֚b" },
  { "a〪᳖̖֚b", "a〪᳖̖֚b", "a〪᳖̖֚b", "a〪᳖̖֚b", "a〪᳖̖֚b" },
  { "a〪̖᳗֚b", "a〪̖᳗֚b", "a〪̖᳗֚b", "a〪̖᳗֚b", "a〪̖᳗֚b" },
  { "a〪᳗̖֚b", "a〪᳗̖֚b", "a〪᳗̖֚b", "a〪᳗̖֚b", "a〪᳗̖֚b" },
  { "a〪̖᳘֚b", "a〪̖᳘֚b", "a〪̖᳘֚b", "a〪̖᳘֚b", "a〪̖᳘֚b" },
  { "a〪᳘̖֚b", "a〪᳘̖֚b", "a〪᳘̖֚b", "a〪᳘̖֚b", "a〪᳘̖֚b" },
  { "a〪̖᳙֚b", "a〪̖᳙֚b", "a〪̖᳙֚b", "a〪̖᳙֚b", "a〪̖᳙֚b" },
  { "a〪᳙̖֚b", "a〪᳙̖֚b", "a〪᳙̖֚b", "a〪᳙̖֚b", "a〪᳙̖֚b" },
  { "à֮᳚̕b", "à֮᳚̕b", "à֮᳚̕b", "à֮᳚̕b", "à֮᳚̕b" },
  { "a֮᳚̀̕b", "a֮᳚̀̕b", "a֮᳚̀̕b", "a֮᳚̀̕b", "a֮᳚̀̕b" },
  { "à֮᳛̕b", "à֮᳛̕b", "à֮᳛̕b", "à֮᳛̕b", "à֮᳛̕b" },
  { "a֮᳛̀̕b", "a֮᳛̀̕b", "a֮᳛̀̕b", "a֮᳛̀̕b", "a֮᳛̀̕b" },
  { "a〪̖᳜֚b", "a〪̖᳜֚b", "a〪̖᳜֚b", "a〪̖᳜֚b", "a〪̖᳜֚b" },
  { "a〪᳜̖֚b", "a〪᳜̖֚b", "a〪᳜̖֚b", "a〪᳜̖֚b", "a〪᳜̖֚b" },
  { "a〪̖᳝֚b", "a〪̖᳝֚b", "a〪̖᳝֚b", "a〪̖᳝֚b", "a〪̖᳝֚b" },
  { "a〪᳝̖֚b", "a〪᳝̖֚b", "a〪᳝̖֚b", "a〪᳝̖֚b", "a〪᳝̖֚b" },
  { "a〪̖᳞֚b", "a〪̖᳞֚b", "a〪̖᳞֚b", "a〪̖᳞֚b", "a〪̖᳞֚b" },
  { "a〪᳞̖֚b", "a〪᳞̖֚b", "a〪᳞̖֚b", "a〪᳞̖֚b", "a〪᳞̖֚b" },
  { "a〪̖᳟֚b", "a〪̖᳟֚b", "a〪̖᳟֚b", "a〪̖᳟֚b", "a〪̖᳟֚b" },
  { "a〪᳟̖֚b", "a〪᳟̖֚b", "a〪᳟̖֚b", "a〪᳟̖֚b", "a〪᳟̖֚b" },
  { "à֮᳠̕b", "à֮᳠̕b", "à֮᳠̕b", "à֮᳠̕b", "à֮᳠̕b" },
  { "a֮᳠̀̕b", "a֮᳠̀̕b", "a֮᳠̀̕b", "a֮᳠̀̕b", "a֮᳠̀̕b" },
  { "a̴᳢़b", "a̴᳢़b", "a̴᳢़b", "a̴᳢़b", "a̴᳢़b" },
  { "a᳢̴़b", "a᳢̴़b", "a᳢̴़b", "a᳢̴़b", "a᳢̴़b" },
  { "a̴᳣़b", "a̴᳣़b", "a̴᳣़b", "a̴᳣़b", "a̴᳣़b" },
  { "a᳣̴़b", "a᳣̴़b", "a᳣̴़b", "a᳣̴़b", "a᳣̴़b" },
  { "a̴᳤़b", "a̴᳤़b", "a̴᳤़b", "a̴᳤़b", "a̴᳤़b" },
  { "a᳤̴़b", "a᳤̴़b", "a᳤̴़b", "a᳤̴़b", "a᳤̴़b" },
  { "a̴᳥़b", "a̴᳥़b", "a̴᳥़b", "a̴᳥़b", "a̴᳥़b" },
  { "a᳥̴़b", "a᳥̴़b", "a᳥̴़b", "a᳥̴़b", "a᳥̴़b" },
  { "a̴᳦़b", "a̴᳦़b", "a̴᳦़b", "a̴᳦़b", "a̴᳦़b" },
  { "a᳦̴़b", "a᳦̴़b", "a᳦̴़b", "a᳦̴़b", "a᳦̴़b" },
  { "a̴᳧़b", "a̴᳧़b", "a̴᳧़b", "a̴᳧़b", "a̴᳧़b" },
  { "a᳧̴़b", "a᳧̴़b", "a᳧̴़b", "a᳧̴़b", "a᳧̴़b" },
  { "a̴᳨़b", "a̴᳨़b", "a̴᳨़b", "a̴᳨़b", "a̴᳨़b" },
  { "a᳨̴़b", "a᳨̴़b", "a᳨̴़b", "a᳨̴़b", "a᳨̴़b" },
  { "a〪̖᳭֚b", "a〪̖᳭֚b", "a〪̖᳭֚b", "a〪̖᳭֚b", "a〪̖᳭֚b" },
  { "a〪᳭̖֚b", "a〪᳭̖֚b", "a〪᳭̖֚b", "a〪᳭̖֚b", "a〪᳭̖֚b" },
  { "à֮᳴̕b", "à֮᳴̕b", "à֮᳴̕b", "à֮᳴̕b", "à֮᳴̕b" },
  { "a֮᳴̀̕b", "a֮᳴̀̕b", "a֮᳴̀̕b", "a֮᳴̀̕b", "a֮᳴̀̕b" },
  { "à֮᳸̕b", "à֮᳸̕b", "à֮᳸̕b", "à֮᳸̕b", "à֮᳸̕b" },
  { "a֮᳸̀̕b", "a֮᳸̀̕b", "a֮᳸̀̕b", "a֮᳸̀̕b", "a֮᳸̀̕b" },
  { "à֮᳹̕b", "à֮᳹̕b", "à֮᳹̕b", "à֮᳹̕b", "à֮᳹̕b" },
  { "a֮᳹̀̕b", "a֮᳹̀̕b", "a֮᳹̀̕b", "a֮᳹̀̕b", "a֮᳹̀̕b" },
  { "à֮᷀̕b", "à֮᷀̕b", "à֮᷀̕b", "à֮᷀̕b", "à֮᷀̕b" },
  { "a֮᷀̀̕b", "a֮᷀̀̕b", "a֮᷀̀̕b", "a֮᷀̀̕b", "a֮᷀̀̕b" },
  { "à֮᷁̕b", "à֮᷁̕b", "à֮᷁̕b", "à֮᷁̕b", "à֮᷁̕b" },
  { "a֮᷁̀̕b", "a֮᷁̀̕b", "a֮᷁̀̕b", "a֮᷁̀̕b", "a֮᷁̀̕b" },
  { "a〪̖᷂֚b", "a〪̖᷂֚b", "a〪̖᷂֚b", "a〪̖᷂֚b", "a〪̖᷂֚b" },
  { "a〪᷂̖֚b", "a〪᷂̖֚b", "a〪᷂̖֚b", "a〪᷂̖֚b", "a〪᷂̖֚b" },
  { "à֮᷃̕b", "à֮᷃̕b", "à֮᷃̕b", "à֮᷃̕b", "à֮᷃̕b" },
  { "a֮᷃̀̕b", "a֮᷃̀̕b", "a֮᷃̀̕b", "a֮᷃̀̕b", "a֮᷃̀̕b" },
  { "à֮᷄̕b", "à֮᷄̕b", "à֮᷄̕b", "à֮᷄̕b", "à֮᷄̕b" },
  { "a֮᷄̀̕b", "a֮᷄̀̕b", "a֮᷄̀̕b", "a֮᷄̀̕b", "a֮᷄̀̕b" },
  { "à֮᷅̕b", "à֮᷅̕b", "à֮᷅̕b", "à֮᷅̕b", "à֮᷅̕b" },
  { "a֮᷅̀̕b", "a֮᷅̀̕b", "a֮᷅̀̕b", "a֮᷅̀̕b", "a֮᷅̀̕b" },
  { "à֮᷆̕b", "à֮᷆̕b", "à֮᷆̕b", "à֮᷆̕b", "à֮᷆̕b" },
  { "a֮᷆̀̕b", "a֮᷆̀̕b", "a֮᷆̀̕b", "a֮᷆̀̕b", "a֮᷆̀̕b" },
  { "à֮᷇̕b", "à֮᷇̕b", "à֮᷇̕b", "à֮᷇̕b", "à֮᷇̕b" },
  { "a֮᷇̀̕b", "a֮᷇̀̕b", "a֮᷇̀̕b", "a֮᷇̀̕b", "a֮᷇̀̕b" },
  { "à֮᷈̕b", "à֮᷈̕b", "à֮᷈̕b", "à֮᷈̕b", "à֮᷈̕b" },
  { "a֮᷈̀̕b", "a֮᷈̀̕b", "a֮᷈̀̕b", "a֮᷈̀̕b", "a֮᷈̀̕b" },
  { "à֮᷉̕b", "à֮᷉̕b", "à֮᷉̕b", "à֮᷉̕b", "à֮᷉̕b" },
  { "a֮᷉̀̕b", "a֮᷉̀̕b", "a֮᷉̀̕b", "a֮᷉̀̕b", "a֮᷉̀̕b" },
  { "a〪̖᷊֚b", "a〪̖᷊֚b", "a〪̖᷊֚b", "a〪̖᷊֚b", "a〪̖᷊֚b" },
  { "a〪᷊̖֚b", "a〪᷊̖֚b", "a〪᷊̖֚b", "a〪᷊̖֚b", "a〪᷊̖֚b" },
  { "à֮᷋̕b", "à֮᷋̕b", "à֮᷋̕b", "à֮᷋̕b", "à֮᷋̕b" },
  { "a֮᷋̀̕b", "a֮᷋̀̕b", "a֮᷋̀̕b", "a֮᷋̀̕b", "a֮᷋̀̕b" },
  { "à֮᷌̕b", "à֮᷌̕b", "à֮᷌̕b", "à֮᷌̕b", "à֮᷌̕b" },
  { "a֮᷌̀̕b", "a֮᷌̀̕b", "a֮᷌̀̕b", "a֮᷌̀̕b", "a֮᷌̀̕b" },
  { "a͜͝᷍ͅb", "a͜͝᷍ͅb", "a͜͝᷍ͅb", "a͜͝᷍ͅb", "a͜͝᷍ͅb" },
  { "a͜᷍͝ͅb", "a͜᷍͝ͅb", "a͜᷍͝ͅb", "a͜᷍͝ͅb", "a͜᷍͝ͅb" },
  { "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b" },
  { "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b", "a̡᷎᷎̛b" },
  { "a〪̖᷏֚b", "a〪̖᷏֚b", "a〪̖᷏֚b", "a〪̖᷏֚b", "a〪̖᷏֚b" },
  { "a〪᷏̖֚b", "a〪᷏̖֚b", "a〪᷏̖֚b", "a〪᷏̖֚b", "a〪᷏̖֚b" },
  { "aུ̡᷐᷎b", "aུ̡᷐᷎b", "aུ̡᷐᷎b", "aུ̡᷐᷎b", "aུ̡᷐᷎b" },
  { "aུ᷐̡᷎b", "aུ᷐̡᷎b", "aུ᷐̡᷎b", "aུ᷐̡᷎b", "aུ᷐̡᷎b" },
  { "à֮᷑̕b", "à֮᷑̕b", "à֮᷑̕b", "à֮᷑̕b", "à֮᷑̕b" },
  { "a֮᷑̀̕b", "a֮᷑̀̕b", "a֮᷑̀̕b", "a֮᷑̀̕b", "a֮᷑̀̕b" },
  { "à֮᷒̕b", "à֮᷒̕b", "à֮᷒̕b", "à֮᷒̕b", "à֮᷒̕b" },
  { "a֮᷒̀̕b", "a֮᷒̀̕b", "a֮᷒̀̕b", "a֮᷒̀̕b", "a֮᷒̀̕b" },
  { "à֮ᷓ̕b", "à֮ᷓ̕b", "à֮ᷓ̕b", "à֮ᷓ̕b", "à֮ᷓ̕b" },
  { "a֮ᷓ̀̕b", "a֮ᷓ̀̕b", "a֮ᷓ̀̕b", "a֮ᷓ̀̕b", "a֮ᷓ̀̕b" },
  { "à֮ᷔ̕b", "à֮ᷔ̕b", "à֮ᷔ̕b", "à֮ᷔ̕b", "à֮ᷔ̕b" },
  { "a֮ᷔ̀̕b", "a֮ᷔ̀̕b", "a֮ᷔ̀̕b", "a֮ᷔ̀̕b", "a֮ᷔ̀̕b" },
  { "à֮ᷕ̕b", "à֮ᷕ̕b", "à֮ᷕ̕b", "à֮ᷕ̕b", "à֮ᷕ̕b" },
  { "a֮ᷕ̀̕b", "a֮ᷕ̀̕b", "a֮ᷕ̀̕b", "a֮ᷕ̀̕b", "a֮ᷕ̀̕b" },
  { "à֮ᷖ̕b", "à֮ᷖ̕b", "à֮ᷖ̕b", "à֮ᷖ̕b", "à֮ᷖ̕b" },
  { "a֮ᷖ̀̕b", "a֮ᷖ̀̕b", "a֮ᷖ̀̕b", "a֮ᷖ̀̕b", "a֮ᷖ̀̕b" },
  { "à֮ᷗ̕b", "à֮ᷗ̕b", "à֮ᷗ̕b", "à֮ᷗ̕b", "à֮ᷗ̕b" },
  { "a֮ᷗ̀̕b", "a֮ᷗ̀̕b", "a֮ᷗ̀̕b", "a֮ᷗ̀̕b", "a֮ᷗ̀̕b" },
  { "à֮ᷘ̕b", "à֮ᷘ̕b", "à֮ᷘ̕b", "à֮ᷘ̕b", "à֮ᷘ̕b" },
  { "a֮ᷘ̀̕b", "a֮ᷘ̀̕b", "a֮ᷘ̀̕b", "a֮ᷘ̀̕b", "a֮ᷘ̀̕b" },
  { "à֮ᷙ̕b", "à֮ᷙ̕b", "à֮ᷙ̕b", "à֮ᷙ̕b", "à֮ᷙ̕b" },
  { "a֮ᷙ̀̕b", "a֮ᷙ̀̕b", "a֮ᷙ̀̕b", "a֮ᷙ̀̕b", "a֮ᷙ̀̕b" },
  { "à֮ᷚ̕b", "à֮ᷚ̕b", "à֮ᷚ̕b", "à֮ᷚ̕b", "à֮ᷚ̕b" },
  { "a֮ᷚ̀̕b", "a֮ᷚ̀̕b", "a֮ᷚ̀̕b", "a֮ᷚ̀̕b", "a֮ᷚ̀̕b" },
  { "à֮ᷛ̕b", "à֮ᷛ̕b", "à֮ᷛ̕b", "à֮ᷛ̕b", "à֮ᷛ̕b" },
  { "a֮ᷛ̀̕b", "a֮ᷛ̀̕b", "a֮ᷛ̀̕b", "a֮ᷛ̀̕b", "a֮ᷛ̀̕b" },
  { "à֮ᷜ̕b", "à֮ᷜ̕b", "à֮ᷜ̕b", "à֮ᷜ̕b", "à֮ᷜ̕b" },
  { "a֮ᷜ̀̕b", "a֮ᷜ̀̕b", "a֮ᷜ̀̕b", "a֮ᷜ̀̕b", "a֮ᷜ̀̕b" },
  { "à֮ᷝ̕b", "à֮ᷝ̕b", "à֮ᷝ̕b", "à֮ᷝ̕b", "à֮ᷝ̕b" },
  { "a֮ᷝ̀̕b", "a֮ᷝ̀̕b", "a֮ᷝ̀̕b", "a֮ᷝ̀̕b", "a֮ᷝ̀̕b" },
  { "à֮ᷞ̕b", "à֮ᷞ̕b", "à֮ᷞ̕b", "à֮ᷞ̕b", "à֮ᷞ̕b" },
  { "a֮ᷞ̀̕b", "a֮ᷞ̀̕b", "a֮ᷞ̀̕b", "a֮ᷞ̀̕b", "a֮ᷞ̀̕b" },
  { "à֮ᷟ̕b", "à֮ᷟ̕b", "à֮ᷟ̕b", "à֮ᷟ̕b", "à֮ᷟ̕b" },
  { "a֮ᷟ̀̕b", "a֮ᷟ̀̕b", "a֮ᷟ̀̕b", "a֮ᷟ̀̕b", "a֮ᷟ̀̕b" },
  { "à֮ᷠ̕b", "à֮ᷠ̕b", "à֮ᷠ̕b", "à֮ᷠ̕b", "à֮ᷠ̕b" },
  { "a֮ᷠ̀̕b", "a֮ᷠ̀̕b", "a֮ᷠ̀̕b", "a֮ᷠ̀̕b", "a֮ᷠ̀̕b" },
  { "à֮ᷡ̕b", "à֮ᷡ̕b", "à֮ᷡ̕b", "à֮ᷡ̕b", "à֮ᷡ̕b" },
  { "a֮ᷡ̀̕b", "a֮ᷡ̀̕b", "a֮ᷡ̀̕b", "a֮ᷡ̀̕b", "a֮ᷡ̀̕b" },
  { "à֮ᷢ̕b", "à֮ᷢ̕b", "à֮ᷢ̕b", "à֮ᷢ̕b", "à֮ᷢ̕b" },
  { "a֮ᷢ̀̕b", "a֮ᷢ̀̕b", "a֮ᷢ̀̕b", "a֮ᷢ̀̕b", "a֮ᷢ̀̕b" },
  { "à֮ᷣ̕b", "à֮ᷣ̕b", "à֮ᷣ̕b", "à֮ᷣ̕b", "à֮ᷣ̕b" },
  { "a֮ᷣ̀̕b", "a֮ᷣ̀̕b", "a֮ᷣ̀̕b", "a֮ᷣ̀̕b", "a֮ᷣ̀̕b" },
  { "à֮ᷤ̕b", "à֮ᷤ̕b", "à֮ᷤ̕b", "à֮ᷤ̕b", "à֮ᷤ̕b" },
  { "a֮ᷤ̀̕b", "a֮ᷤ̀̕b", "a֮ᷤ̀̕b", "a֮ᷤ̀̕b", "a֮ᷤ̀̕b" },
  { "à֮ᷥ̕b", "à֮ᷥ̕b", "à֮ᷥ̕b", "à֮ᷥ̕b", "à֮ᷥ̕b" },
  { "a֮ᷥ̀̕b", "a֮ᷥ̀̕b", "a֮ᷥ̀̕b", "a֮ᷥ̀̕b", "a֮ᷥ̀̕b" },
  { "à֮ᷦ̕b", "à֮ᷦ̕b", "à֮ᷦ̕b", "à֮ᷦ̕b", "à֮ᷦ̕b" },
  { "a֮ᷦ̀̕b", "a֮ᷦ̀̕b", "a֮ᷦ̀̕b", "a֮ᷦ̀̕b", "a֮ᷦ̀̕b" },
  { "à֮ᷧ̕b", "à֮ᷧ̕b", "à֮ᷧ̕b", "à֮ᷧ̕b", "à֮ᷧ̕b" },
  { "a֮ᷧ̀̕b", "a֮ᷧ̀̕b", "a֮ᷧ̀̕b", "a֮ᷧ̀̕b", "a֮ᷧ̀̕b" },
  { "à֮ᷨ̕b", "à֮ᷨ̕b", "à֮ᷨ̕b", "à֮ᷨ̕b", "à֮ᷨ̕b" },
  { "a֮ᷨ̀̕b", "a֮ᷨ̀̕b", "a֮ᷨ̀̕b", "a֮ᷨ̀̕b", "a֮ᷨ̀̕b" },
  { "à֮ᷩ̕b", "à֮ᷩ̕b", "à֮ᷩ̕b", "à֮ᷩ̕b", "à֮ᷩ̕b" },
  { "a֮ᷩ̀̕b", "a֮ᷩ̀̕b", "a֮ᷩ̀̕b", "a֮ᷩ̀̕b", "a֮ᷩ̀̕b" },
  { "à֮ᷪ̕b", "à֮ᷪ̕b", "à֮ᷪ̕b", "à֮ᷪ̕b", "à֮ᷪ̕b" },
  { "a֮ᷪ̀̕b", "a֮ᷪ̀̕b", "a֮ᷪ̀̕b", "a֮ᷪ̀̕b", "a֮ᷪ̀̕b" },
  { "à֮ᷫ̕b", "à֮ᷫ̕b", "à֮ᷫ̕b", "à֮ᷫ̕b", "à֮ᷫ̕b" },
  { "a֮ᷫ̀̕b", "a֮ᷫ̀̕b", "a֮ᷫ̀̕b", "a֮ᷫ̀̕b", "a֮ᷫ̀̕b" },
  { "à֮ᷬ̕b", "à֮ᷬ̕b", "à֮ᷬ̕b", "à֮ᷬ̕b", "à֮ᷬ̕b" },
  { "a֮ᷬ̀̕b", "a֮ᷬ̀̕b", "a֮ᷬ̀̕b", "a֮ᷬ̀̕b", "a֮ᷬ̀̕b" },
  { "à֮ᷭ̕b", "à֮ᷭ̕b", "à֮ᷭ̕b", "à֮ᷭ̕b", "à֮ᷭ̕b" },
  { "a֮ᷭ̀̕b", "a֮ᷭ̀̕b", "a֮ᷭ̀̕b", "a֮ᷭ̀̕b", "a֮ᷭ̀̕b" },
  { "à֮ᷮ̕b", "à֮ᷮ̕b", "à֮ᷮ̕b", "à֮ᷮ̕b", "à֮ᷮ̕b" },
  { "a֮ᷮ̀̕b", "a֮ᷮ̀̕b", "a֮ᷮ̀̕b", "a֮ᷮ̀̕b", "a֮ᷮ̀̕b" },
  { "à֮ᷯ̕b", "à֮ᷯ̕b", "à֮ᷯ̕b", "à֮ᷯ̕b", "à֮ᷯ̕b" },
  { "a֮ᷯ̀̕b", "a֮ᷯ̀̕b", "a֮ᷯ̀̕b", "a֮ᷯ̀̕b", "a֮ᷯ̀̕b" },
  { "à֮ᷰ̕b", "à֮ᷰ̕b", "à֮ᷰ̕b", "à֮ᷰ̕b", "à֮ᷰ̕b" },
  { "a֮ᷰ̀̕b", "a֮ᷰ̀̕b", "a֮ᷰ̀̕b", "a֮ᷰ̀̕b", "a֮ᷰ̀̕b" },
  { "à֮ᷱ̕b", "à֮ᷱ̕b", "à֮ᷱ̕b", "à֮ᷱ̕b", "à֮ᷱ̕b" },
  { "a֮ᷱ̀̕b", "a֮ᷱ̀̕b", "a֮ᷱ̀̕b", "a֮ᷱ̀̕b", "a֮ᷱ̀̕b" },
  { "à֮ᷲ̕b", "à֮ᷲ̕b", "à֮ᷲ̕b", "à֮ᷲ̕b", "à֮ᷲ̕b" },
  { "a֮ᷲ̀̕b", "a֮ᷲ̀̕b", "a֮ᷲ̀̕b", "a֮ᷲ̀̕b", "a֮ᷲ̀̕b" },
  { "à֮ᷳ̕b", "à֮ᷳ̕b", "à֮ᷳ̕b", "à֮ᷳ̕b", "à֮ᷳ̕b" },
  { "a֮ᷳ̀̕b", "a֮ᷳ̀̕b", "a֮ᷳ̀̕b", "a֮ᷳ̀̕b", "a֮ᷳ̀̕b" },
  { "à֮ᷴ̕b", "à֮ᷴ̕b", "à֮ᷴ̕b", "à֮ᷴ̕b", "à֮ᷴ̕b" },
  { "a֮ᷴ̀̕b", "a֮ᷴ̀̕b", "a֮ᷴ̀̕b", "a֮ᷴ̀̕b", "a֮ᷴ̀̕b" },
  { "à֮᷵̕b", "à֮᷵̕b", "à֮᷵̕b", "à֮᷵̕b", "à֮᷵̕b" },
  { "a֮᷵̀̕b", "a֮᷵̀̕b", "a֮᷵̀̕b", "a֮᷵̀̕b", "a֮᷵̀̕b" },
  { "à̕᷶͜b", "à̕᷶͜b", "à̕᷶͜b", "à̕᷶͜b", "à̕᷶͜b" },
  { "à᷶̕͜b", "à᷶̕͜b", "à᷶̕͜b", "à᷶̕͜b", "à᷶̕͜b" },
  { "à𝅭֮᷷b", "à𝅭֮᷷b", "à𝅭֮᷷b", "à𝅭֮᷷b", "à𝅭֮᷷b" },
  { "à𝅭᷷֮b", "à𝅭᷷֮b", "à𝅭᷷֮b", "à𝅭᷷֮b", "à𝅭᷷֮b" },
  { "à𝅭֮᷸b", "à𝅭֮᷸b", "à𝅭֮᷸b", "à𝅭֮᷸b", "à𝅭֮᷸b" },
  { "à𝅭᷸֮b", "à𝅭᷸֮b", "à𝅭᷸֮b", "à𝅭᷸֮b", "à𝅭᷸֮b" },
  { "a〪̖᷹֚b", "a〪̖᷹֚b", "a〪̖᷹֚b", "a〪̖᷹֚b", "a〪̖᷹֚b" },
  { "a〪᷹̖֚b", "a〪᷹̖֚b", "a〪᷹̖֚b", "a〪᷹̖֚b", "a〪᷹̖֚b" },
  { "à֮᷻̕b", "à֮᷻̕b", "à֮᷻̕b", "à֮᷻̕b", "à֮᷻̕b" },
  { "a֮᷻̀̕b", "a֮᷻̀̕b", "a֮᷻̀̕b", "a֮᷻̀̕b", "a֮᷻̀̕b" },
  { "a̕͜᷼͝b", "a̕͜᷼͝b", "a̕͜᷼͝b", "a̕͜᷼͝b", "a̕͜᷼͝b" },
  { "a̕᷼͜͝b", "a̕᷼͜͝b", "a̕᷼͜͝b", "a̕᷼͜͝b", "a̕᷼͜͝b" },
  { "a〪̖᷽֚b", "a〪̖᷽֚b", "a〪̖᷽֚b", "a〪̖᷽֚b", "a〪̖᷽֚b" },
  { "a〪᷽̖֚b", "a〪᷽̖֚b", "a〪᷽̖֚b", "a〪᷽̖֚b", "a〪᷽̖֚b" },
  { "à֮᷾̕b", "à֮᷾̕b", "à֮᷾̕b", "à֮᷾̕b", "à֮᷾̕b" },
  { "a֮᷾̀̕b", "a֮᷾̀̕b", "a֮᷾̀̕b", "a֮᷾̀̕b", "a֮᷾̀̕b" },
  { "a〪̖᷿֚b", "a〪̖᷿֚b", "a〪̖᷿֚b", "a〪̖᷿֚b", "a〪̖᷿֚b" },
  { "a〪᷿̖֚b", "a〪᷿̖֚b", "a〪᷿̖֚b", "a〪᷿̖֚b", "a〪᷿̖֚b" },
  { "à֮⃐̕b", "à֮⃐̕b", "à֮⃐̕b", "à֮⃐̕b", "à֮⃐̕b" },
  { "a֮⃐̀̕b", "a֮⃐̀̕b", "a֮⃐̀̕b", "a֮⃐̀̕b", "a֮⃐̀̕b" },
  { "à֮⃑̕b", "à֮⃑̕b", "à֮⃑̕b", "à֮⃑̕b", "à֮⃑̕b" },
  { "a֮⃑̀̕b", "a֮⃑̀̕b", "a֮⃑̀̕b", "a֮⃑̀̕b", "a֮⃑̀̕b" },
  { "a̴⃒़b", "a̴⃒़b", "a̴⃒़b", "a̴⃒़b", "a̴⃒़b" },
  { "a⃒̴़b", "a⃒̴़b", "a⃒̴़b", "a⃒̴़b", "a⃒̴़b" },
  { "a̴⃓़b", "a̴⃓़b", "a̴⃓़b", "a̴⃓़b", "a̴⃓़b" },
  { "a⃓̴़b", "a⃓̴़b", "a⃓̴़b", "a⃓̴़b", "a⃓̴़b" },
  { "à֮⃔̕b", "à֮⃔̕b", "à֮⃔̕b", "à֮⃔̕b", "à֮⃔̕b" },
  { "a֮⃔̀̕b", "a֮⃔̀̕b", "a֮⃔̀̕b", "a֮⃔̀̕b", "a֮⃔̀̕b" },
  { "à֮⃕̕b", "à֮⃕̕b", "à֮⃕̕b", "à֮⃕̕b", "à֮⃕̕b" },
  { "a֮⃕̀̕b", "a֮⃕̀̕b", "a֮⃕̀̕b", "a֮⃕̀̕b", "a֮⃕̀̕b" },
  { "à֮⃖̕b", "à֮⃖̕b", "à֮⃖̕b", "à֮⃖̕b", "à֮⃖̕b" },
  { "a֮⃖̀̕b", "a֮⃖̀̕b", "a֮⃖̀̕b", "a֮⃖̀̕b", "a֮⃖̀̕b" },
  { "à֮⃗̕b", "à֮⃗̕b", "à֮⃗̕b", "à֮⃗̕b", "à֮⃗̕b" },
  { "a֮⃗̀̕b", "a֮⃗̀̕b", "a֮⃗̀̕b", "a֮⃗̀̕b", "a֮⃗̀̕b" },
  { "a̴⃘़b", "a̴⃘़b", "a̴⃘़b", "a̴⃘़b", "a̴⃘़b" },
  { "a⃘̴़b", "a⃘̴़b", "a⃘̴़b", "a⃘̴़b", "a⃘̴़b" },
  { "a̴⃙़b", "a̴⃙़b", "a̴⃙़b", "a̴⃙़b", "a̴⃙़b" },
  { "a⃙̴़b", "a⃙̴़b", "a⃙̴़b", "a⃙̴़b", "a⃙̴़b" },
  { "a̴⃚़b", "a̴⃚़b", "a̴⃚़b", "a̴⃚़b", "a̴⃚़b" },
  { "a⃚̴़b", "a⃚̴़b", "a⃚̴़b", "a⃚̴़b", "a⃚̴़b" },
  { "à֮⃛̕b", "à֮⃛̕b", "à֮⃛̕b", "à֮⃛̕b", "à֮⃛̕b" },
  { "a֮⃛̀̕b", "a֮⃛̀̕b", "a֮⃛̀̕b", "a֮⃛̀̕b", "a֮⃛̀̕b" },
  { "à֮⃜̕b", "à֮⃜̕b", "à֮⃜̕b", "à֮⃜̕b", "à֮⃜̕b" },
  { "a֮⃜̀̕b", "a֮⃜̀̕b", "a֮⃜̀̕b", "a֮⃜̀̕b", "a֮⃜̀̕b" },
  { "à֮⃡̕b", "à֮⃡̕b", "à֮⃡̕b", "à֮⃡̕b", "à֮⃡̕b" },
  { "a֮⃡̀̕b", "a֮⃡̀̕b", "a֮⃡̀̕b", "a֮⃡̀̕b", "a֮⃡̀̕b" },
  { "a̴⃥़b", "a̴⃥़b", "a̴⃥़b", "a̴⃥़b", "a̴⃥़b" },
  { "a⃥̴़b", "a⃥̴़b", "a⃥̴़b", "a⃥̴़b", "a⃥̴़b" },
  { "a̴⃦़b", "a̴⃦़b", "a̴⃦़b", "a̴⃦़b", "a̴⃦़b" },
  { "a⃦̴़b", "a⃦̴़b", "a⃦̴़b", "a⃦̴़b", "a⃦̴़b" },
  { "à֮⃧̕b", "à֮⃧̕b", "à֮⃧̕b", "à֮⃧̕b", "à֮⃧̕b" },
  { "a֮⃧̀̕b", "a֮⃧̀̕b", "a֮⃧̀̕b", "a֮⃧̀̕b", "a֮⃧̀̕b" },
  { "a〪̖⃨֚b", "a〪̖⃨֚b", "a〪̖⃨֚b", "a〪̖⃨֚b", "a〪̖⃨֚b" },
  { "a〪⃨̖֚b", "a〪⃨̖֚b", "a〪⃨̖֚b", "a〪⃨̖֚b", "a〪⃨̖֚b" },
  { "à֮⃩̕b", "à֮⃩̕b", "à֮⃩̕b", "à֮⃩̕b", "à֮⃩̕b" },
  { "a֮⃩̀̕b", "a֮⃩̀̕b", "a֮⃩̀̕b", "a֮⃩̀̕b", "a֮⃩̀̕b" },
  { "a̴⃪़b", "a̴⃪़b", "a̴⃪़b", "a̴⃪़b", "a̴⃪़b" },
  { "a⃪̴़b", "a⃪̴़b", "a⃪̴़b", "a⃪̴़b", "a⃪̴़b" },
  { "a̴⃫़b", "a̴⃫़b", "a̴⃫़b", "a̴⃫़b", "a̴⃫़b" },
  { "a⃫̴़b", "a⃫̴़b", "a⃫̴़b", "a⃫̴़b", "a⃫̴़b" },
  { "a〪̖⃬֚b", "a〪̖⃬֚b", "a〪̖⃬֚b", "a〪̖⃬֚b", "a〪̖⃬֚b" },
  { "a〪⃬̖֚b", "a〪⃬̖֚b", "a〪⃬̖֚b", "a〪⃬̖֚b", "a〪⃬̖֚b" },
  { "a〪̖⃭֚b", "a〪̖⃭֚b", "a〪̖⃭֚b", "a〪̖⃭֚b", "a〪̖⃭֚b" },
  { "a〪⃭̖֚b", "a〪⃭̖֚b", "a〪⃭̖֚b", "a〪⃭̖֚b", "a〪⃭̖֚b" },
  { "a〪̖⃮֚b", "a〪̖⃮֚b", "a〪̖⃮֚b", "a〪̖⃮֚b", "a〪̖⃮֚b" },
  { "a〪⃮̖֚b", "a〪⃮̖֚b", "a〪⃮̖֚b", "a〪⃮̖֚b", "a〪⃮̖֚b" },
  { "a〪̖⃯֚b", "a〪̖⃯֚b", "a〪̖⃯֚b", "a〪̖⃯֚b", "a〪̖⃯֚b" },
  { "a〪⃯̖֚b", "a〪⃯̖֚b", "a〪⃯̖֚b", "a〪⃯̖֚b", "a〪⃯̖֚b" },
  { "à֮⃰̕b", "à֮⃰̕b", "à֮⃰̕b", "à֮⃰̕b", "à֮⃰̕b" },
  { "a֮⃰̀̕b", "a֮⃰̀̕b", "a֮⃰̀̕b", "a֮⃰̀̕b", "a֮⃰̀̕b" },
  { "à֮⳯̕b", "à֮⳯̕b", "à֮⳯̕b", "à֮⳯̕b", "à֮⳯̕b" },
  { "a֮⳯̀̕b", "a֮⳯̀̕b", "a֮⳯̀̕b", "a֮⳯̀̕b", "a֮⳯̀̕b" },
  { "à֮⳰̕b", "à֮⳰̕b", "à֮⳰̕b", "à֮⳰̕b", "à֮⳰̕b" },
  { "a֮⳰̀̕b", "a֮⳰̀̕b", "a֮⳰̀̕b", "a֮⳰̀̕b", "a֮⳰̀̕b" },
  { "à֮⳱̕b", "à֮⳱̕b", "à֮⳱̕b", "à֮⳱̕b", "à֮⳱̕b" },
  { "a֮⳱̀̕b", "a֮⳱̀̕b", "a֮⳱̀̕b", "a֮⳱̀̕b", "a֮⳱̀̕b" },
  { "a゙्⵿ְb", "a゙्⵿ְb", "a゙्⵿ְb", "a゙्⵿ְb", "a゙्⵿ְb" },
  { "a゙⵿्ְb", "a゙⵿्ְb", "a゙⵿्ְb", "a゙⵿्ְb", "a゙⵿्ְb" },
  { "à֮ⷠ̕b", "à֮ⷠ̕b", "à֮ⷠ̕b", "à֮ⷠ̕b", "à֮ⷠ̕b" },
  { "a֮ⷠ̀̕b", "a֮ⷠ̀̕b", "a֮ⷠ̀̕b", "a֮ⷠ̀̕b", "a֮ⷠ̀̕b" },
  { "à֮ⷡ̕b", "à֮ⷡ̕b", "à֮ⷡ̕b", "à֮ⷡ̕b", "à֮ⷡ̕b" },
  { "a֮ⷡ̀̕b", "a֮ⷡ̀̕b", "a֮ⷡ̀̕b", "a֮ⷡ̀̕b", "a֮ⷡ̀̕b" },
  { "à֮ⷢ̕b", "à֮ⷢ̕b", "à֮ⷢ̕b", "à֮ⷢ̕b", "à֮ⷢ̕b" },
  { "a֮ⷢ̀̕b", "a֮ⷢ̀̕b", "a֮ⷢ̀̕b", "a֮ⷢ̀̕b", "a֮ⷢ̀̕b" },
  { "à֮ⷣ̕b", "à֮ⷣ̕b", "à֮ⷣ̕b", "à֮ⷣ̕b", "à֮ⷣ̕b" },
  { "a֮ⷣ̀̕b", "a֮ⷣ̀̕b", "a֮ⷣ̀̕b", "a֮ⷣ̀̕b", "a֮ⷣ̀̕b" },
  { "à֮ⷤ̕b", "à֮ⷤ̕b", "à֮ⷤ̕b", "à֮ⷤ̕b", "à֮ⷤ̕b" },
  { "a֮ⷤ̀̕b", "a֮ⷤ̀̕b", "a֮ⷤ̀̕b", "a֮ⷤ̀̕b", "a֮ⷤ̀̕b" },
  { "à֮ⷥ̕b", "à֮ⷥ̕b", "à֮ⷥ̕b", "à֮ⷥ̕b", "à֮ⷥ̕b" },
  { "a֮ⷥ̀̕b", "a֮ⷥ̀̕b", "a֮ⷥ̀̕b", "a֮ⷥ̀̕b", "a֮ⷥ̀̕b" },
  { "à֮ⷦ̕b", "à֮ⷦ̕b", "à֮ⷦ̕b", "à֮ⷦ̕b", "à֮ⷦ̕b" },
  { "a֮ⷦ̀̕b", "a֮ⷦ̀̕b", "a֮ⷦ̀̕b", "a֮ⷦ̀̕b", "a֮ⷦ̀̕b" },
  { "à֮ⷧ̕b", "à֮ⷧ̕b", "à֮ⷧ̕b", "à֮ⷧ̕b", "à֮ⷧ̕b" },
  { "a֮ⷧ̀̕b", "a֮ⷧ̀̕b", "a֮ⷧ̀̕b", "a֮ⷧ̀̕b", "a֮ⷧ̀̕b" },
  { "à֮ⷨ̕b", "à֮ⷨ̕b", "à֮ⷨ̕b", "à֮ⷨ̕b", "à֮ⷨ̕b" },
  { "a֮ⷨ̀̕b", "a֮ⷨ̀̕b", "a֮ⷨ̀̕b", "a֮ⷨ̀̕b", "a֮ⷨ̀̕b" },
  { "à֮ⷩ̕b", "à֮ⷩ̕b", "à֮ⷩ̕b", "à֮ⷩ̕b", "à֮ⷩ̕b" },
  { "a֮ⷩ̀̕b", "a֮ⷩ̀̕b", "a֮ⷩ̀̕b", "a֮ⷩ̀̕b", "a֮ⷩ̀̕b" },
  { "à֮ⷪ̕b", "à֮ⷪ̕b", "à֮ⷪ̕b", "à֮ⷪ̕b", "à֮ⷪ̕b" },
  { "a֮ⷪ̀̕b", "a֮ⷪ̀̕b", "a֮ⷪ̀̕b", "a֮ⷪ̀̕b", "a֮ⷪ̀̕b" },
  { "à֮ⷫ̕b", "à֮ⷫ̕b", "à֮ⷫ̕b", "à֮ⷫ̕b", "à֮ⷫ̕b" },
  { "a֮ⷫ̀̕b", "a֮ⷫ̀̕b", "a֮ⷫ̀̕b", "a֮ⷫ̀̕b", "a֮ⷫ̀̕b" },
  { "à֮ⷬ̕b", "à֮ⷬ̕b", "à֮ⷬ̕b", "à֮ⷬ̕b", "à֮ⷬ̕b" },
  { "a֮ⷬ̀̕b", "a֮ⷬ̀̕b", "a֮ⷬ̀̕b", "a֮ⷬ̀̕b", "a֮ⷬ̀̕b" },
  { "à֮ⷭ̕b", "à֮ⷭ̕b", "à֮ⷭ̕b", "à֮ⷭ̕b", "à֮ⷭ̕b" },
  { "a֮ⷭ̀̕b", "a֮ⷭ̀̕b", "a֮ⷭ̀̕b", "a֮ⷭ̀̕b", "a֮ⷭ̀̕b" },
  { "à֮ⷮ̕b", "à֮ⷮ̕b", "à֮ⷮ̕b", "à֮ⷮ̕b", "à֮ⷮ̕b" },
  { "a֮ⷮ̀̕b", "a֮ⷮ̀̕b", "a֮ⷮ̀̕b", "a֮ⷮ̀̕b", "a֮ⷮ̀̕b" },
  { "à֮ⷯ̕b", "à֮ⷯ̕b", "à֮ⷯ̕b", "à֮ⷯ̕b", "à֮ⷯ̕b" },
  { "a֮ⷯ̀̕b", "a֮ⷯ̀̕b", "a֮ⷯ̀̕b", "a֮ⷯ̀̕b", "a֮ⷯ̀̕b" },
  { "à֮ⷰ̕b", "à֮ⷰ̕b", "à֮ⷰ̕b", "à֮ⷰ̕b", "à֮ⷰ̕b" },
  { "a֮ⷰ̀̕b", "a֮ⷰ̀̕b", "a֮ⷰ̀̕b", "a֮ⷰ̀̕b", "a֮ⷰ̀̕b" },
  { "à֮ⷱ̕b", "à֮ⷱ̕b", "à֮ⷱ̕b", "à֮ⷱ̕b", "à֮ⷱ̕b" },
  { "a֮ⷱ̀̕b", "a֮ⷱ̀̕b", "a֮ⷱ̀̕b", "a֮ⷱ̀̕b", "a֮ⷱ̀̕b" },
  { "à֮ⷲ̕b", "à֮ⷲ̕b", "à֮ⷲ̕b", "à֮ⷲ̕b", "à֮ⷲ̕b" },
  { "a֮ⷲ̀̕b", "a֮ⷲ̀̕b", "a֮ⷲ̀̕b", "a֮ⷲ̀̕b", "a֮ⷲ̀̕b" },
  { "à֮ⷳ̕b", "à֮ⷳ̕b", "à֮ⷳ̕b", "à֮ⷳ̕b", "à֮ⷳ̕b" },
  { "a֮ⷳ̀̕b", "a֮ⷳ̀̕b", "a֮ⷳ̀̕b", "a֮ⷳ̀̕b", "a֮ⷳ̀̕b" },
  { "à֮ⷴ̕b", "à֮ⷴ̕b", "à֮ⷴ̕b", "à֮ⷴ̕b", "à֮ⷴ̕b" },
  { "a֮ⷴ̀̕b", "a֮ⷴ̀̕b", "a֮ⷴ̀̕b", "a֮ⷴ̀̕b", "a֮ⷴ̀̕b" },
  { "à֮ⷵ̕b", "à֮ⷵ̕b", "à֮ⷵ̕b", "à֮ⷵ̕b", "à֮ⷵ̕b" },
  { "a֮ⷵ̀̕b", "a֮ⷵ̀̕b", "a֮ⷵ̀̕b", "a֮ⷵ̀̕b", "a֮ⷵ̀̕b" },
  { "à֮ⷶ̕b", "à֮ⷶ̕b", "à֮ⷶ̕b", "à֮ⷶ̕b", "à֮ⷶ̕b" },
  { "a֮ⷶ̀̕b", "a֮ⷶ̀̕b", "a֮ⷶ̀̕b", "a֮ⷶ̀̕b", "a֮ⷶ̀̕b" },
  { "à֮ⷷ̕b", "à֮ⷷ̕b", "à֮ⷷ̕b", "à֮ⷷ̕b", "à֮ⷷ̕b" },
  { "a֮ⷷ̀̕b", "a֮ⷷ̀̕b", "a֮ⷷ̀̕b", "a֮ⷷ̀̕b", "a֮ⷷ̀̕b" },
  { "à֮ⷸ̕b", "à֮ⷸ̕b", "à֮ⷸ̕b", "à֮ⷸ̕b", "à֮ⷸ̕b" },
  { "a֮ⷸ̀̕b", "a֮ⷸ̀̕b", "a֮ⷸ̀̕b", "a֮ⷸ̀̕b", "a֮ⷸ̀̕b" },
  { "à֮ⷹ̕b", "à֮ⷹ̕b", "à֮ⷹ̕b", "à֮ⷹ̕b", "à֮ⷹ̕b" },
  { "a֮ⷹ̀̕b", "a֮ⷹ̀̕b", "a֮ⷹ̀̕b", "a֮ⷹ̀̕b", "a֮ⷹ̀̕b" },
  { "à֮ⷺ̕b", "à֮ⷺ̕b", "à֮ⷺ̕b", "à֮ⷺ̕b", "à֮ⷺ̕b" },
  { "a֮ⷺ̀̕b", "a֮ⷺ̀̕b", "a֮ⷺ̀̕b", "a֮ⷺ̀̕b", "a֮ⷺ̀̕b" },
  { "à֮ⷻ̕b", "à֮ⷻ̕b", "à֮ⷻ̕b", "à֮ⷻ̕b", "à֮ⷻ̕b" },
  { "a֮ⷻ̀̕b", "a֮ⷻ̀̕b", "a֮ⷻ̀̕b", "a֮ⷻ̀̕b", "a֮ⷻ̀̕b" },
  { "à֮ⷼ̕b", "à֮ⷼ̕b", "à֮ⷼ̕b", "à֮ⷼ̕b", "à֮ⷼ̕b" },
  { "a֮ⷼ̀̕b", "a֮ⷼ̀̕b", "a֮ⷼ̀̕b", "a֮ⷼ̀̕b", "a֮ⷼ̀̕b" },
  { "à֮ⷽ̕b", "à֮ⷽ̕b", "à֮ⷽ̕b", "à֮ⷽ̕b", "à֮ⷽ̕b" },
  { "a֮ⷽ̀̕b", "a֮ⷽ̀̕b", "a֮ⷽ̀̕b", "a֮ⷽ̀̕b", "a֮ⷽ̀̕b" },
  { "à֮ⷾ̕b", "à֮ⷾ̕b", "à֮ⷾ̕b", "à֮ⷾ̕b", "à֮ⷾ̕b" },
  { "a֮ⷾ̀̕b", "a֮ⷾ̀̕b", "a֮ⷾ̀̕b", "a֮ⷾ̀̕b", "a֮ⷾ̀̕b" },
  { "à֮ⷿ̕b", "à֮ⷿ̕b", "à֮ⷿ̕b", "à֮ⷿ̕b", "à֮ⷿ̕b" },
  { "a֮ⷿ̀̕b", "a֮ⷿ̀̕b", "a֮ⷿ̀̕b", "a֮ⷿ̀̕b", "a֮ⷿ̀̕b" },
  { "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b" },
  { "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b", "a̛〪〪̖b" },
  { "à𝅭֮〫b", "à𝅭֮〫b", "à𝅭֮〫b", "à𝅭֮〫b", "à𝅭֮〫b" },
  { "à𝅭〫֮b", "à𝅭〫֮b", "à𝅭〫֮b", "à𝅭〫֮b", "à𝅭〫֮b" },
  { "à̕〬͜b", "à̕〬͜b", "à̕〬͜b", "à̕〬͜b", "à̕〬͜b" },
  { "à〬̕͜b", "à〬̕͜b", "à〬̕͜b", "à〬̕͜b", "à〬̕͜b" },
  { "a̖֚〭〮b", "a̖֚〭〮b", "a̖֚〭〮b", "a̖֚〭〮b", "a̖֚〭〮b" },
  { "a̖〭֚〮b", "a̖〭֚〮b", "a̖〭֚〮b", "a̖〭֚〮b", "a̖〭֚〮b" },
  { "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b" },
  { "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b", "a֚〮〮𝅭b" },
  { "a֚〮〯𝅭b", "a֚〮〯𝅭b", "a֚〮〯𝅭b", "a֚〮〯𝅭b", "a֚〮〯𝅭b" },
  { "a֚〯〮𝅭b", "a֚〯〮𝅭b", "a֚〯〮𝅭b", "a֚〯〮𝅭b", "a֚〯〮𝅭b" },
  { "a़゙゙्b", "a़゙゙्b", "a़゙゙्b", "a़゙゙्b", "a़゙゙्b" },
  { "a़゙゙्b", "a़゙゙्b", "a़゙゙्b", "a़゙゙्b", "a़゙゙्b" },
  { "a़゙゚्b", "a़゙゚्b", "a़゙゚्b", "a़゙゚्b", "a़゙゚्b" },
  { "a़゚゙्b", "a़゚゙्b", "a़゚゙्b", "a़゚゙्b", "a़゚゙्b" },
  { "à֮꙯̕b", "à֮꙯̕b", "à֮꙯̕b", "à֮꙯̕b", "à֮꙯̕b" },
  { "a֮꙯̀̕b", "a֮꙯̀̕b", "a֮꙯̀̕b", "a֮꙯̀̕b", "a֮꙯̀̕b" },
  { "à֮ꙴ̕b", "à֮ꙴ̕b", "à֮ꙴ̕b", "à֮ꙴ̕b", "à֮ꙴ̕b" },
  { "a֮ꙴ̀̕b", "a֮ꙴ̀̕b", "a֮ꙴ̀̕b", "a֮ꙴ̀̕b", "a֮ꙴ̀̕b" },
  { "à֮ꙵ̕b", "à֮ꙵ̕b", "à֮ꙵ̕b", "à֮ꙵ̕b", "à֮ꙵ̕b" },
  { "a֮ꙵ̀̕b", "a֮ꙵ̀̕b", "a֮ꙵ̀̕b", "a֮ꙵ̀̕b", "a֮ꙵ̀̕b" },
  { "à֮ꙶ̕b", "à֮ꙶ̕b", "à֮ꙶ̕b", "à֮ꙶ̕b", "à֮ꙶ̕b" },
  { "a֮ꙶ̀̕b", "a֮ꙶ̀̕b", "a֮ꙶ̀̕b", "a֮ꙶ̀̕b", "a֮ꙶ̀̕b" },
  { "à֮ꙷ̕b", "à֮ꙷ̕b", "à֮ꙷ̕b", "à֮ꙷ̕b", "à֮ꙷ̕b" },
  { "a֮ꙷ̀̕b", "a֮ꙷ̀̕b", "a֮ꙷ̀̕b", "a֮ꙷ̀̕b", "a֮ꙷ̀̕b" },
  { "à֮ꙸ̕b", "à֮ꙸ̕b", "à֮ꙸ̕b", "à֮ꙸ̕b", "à֮ꙸ̕b" },
  { "a֮ꙸ̀̕b", "a֮ꙸ̀̕b", "a֮ꙸ̀̕b", "a֮ꙸ̀̕b", "a֮ꙸ̀̕b" },
  { "à֮ꙹ̕b", "à֮ꙹ̕b", "à֮ꙹ̕b", "à֮ꙹ̕b", "à֮ꙹ̕b" },
  { "a֮ꙹ̀̕b", "a֮ꙹ̀̕b", "a֮ꙹ̀̕b", "a֮ꙹ̀̕b", "a֮ꙹ̀̕b" },
  { "à֮ꙺ̕b", "à֮ꙺ̕b", "à֮ꙺ̕b", "à֮ꙺ̕b", "à֮ꙺ̕b" },
  { "a֮ꙺ̀̕b", "a֮ꙺ̀̕b", "a֮ꙺ̀̕b", "a֮ꙺ̀̕b", "a֮ꙺ̀̕b" },
  { "à֮ꙻ̕b", "à֮ꙻ̕b", "à֮ꙻ̕b", "à֮ꙻ̕b", "à֮ꙻ̕b" },
  { "a֮ꙻ̀̕b", "a֮ꙻ̀̕b", "a֮ꙻ̀̕b", "a֮ꙻ̀̕b", "a֮ꙻ̀̕b" },
  { "à֮꙼̕b", "à֮꙼̕b", "à֮꙼̕b", "à֮꙼̕b", "à֮꙼̕b" },
  { "a֮꙼̀̕b", "a֮꙼̀̕b", "a֮꙼̀̕b", "a֮꙼̀̕b", "a֮꙼̀̕b" },
  { "à֮꙽̕b", "à֮꙽̕b", "à֮꙽̕b", "à֮꙽̕b", "à֮꙽̕b" },
  { "a֮꙽̀̕b", "a֮꙽̀̕b", "a֮꙽̀̕b", "a֮꙽̀̕b", "a֮꙽̀̕b" },
  { "à֮ꚞ̕b", "à֮ꚞ̕b", "à֮ꚞ̕b", "à֮ꚞ̕b", "à֮ꚞ̕b" },
  { "a֮ꚞ̀̕b", "a֮ꚞ̀̕b", "a֮ꚞ̀̕b", "a֮ꚞ̀̕b", "a֮ꚞ̀̕b" },
  { "à֮ꚟ̕b", "à֮ꚟ̕b", "à֮ꚟ̕b", "à֮ꚟ̕b", "à֮ꚟ̕b" },
  { "a֮ꚟ̀̕b", "a֮ꚟ̀̕b", "a֮ꚟ̀̕b", "a֮ꚟ̀̕b", "a֮ꚟ̀̕b" },
  { "à֮꛰̕b", "à֮꛰̕b", "à֮꛰̕b", "à֮꛰̕b", "à֮꛰̕b" },
  { "a֮꛰̀̕b", "a֮꛰̀̕b", "a֮꛰̀̕b", "a֮꛰̀̕b", "a֮꛰̀̕b" },
  { "à֮꛱̕b", "à֮꛱̕b", "à֮꛱̕b", "à֮꛱̕b", "à֮꛱̕b" },
  { "a֮꛱̀̕b", "a֮꛱̀̕b", "a֮꛱̀̕b", "a֮꛱̀̕b", "a֮꛱̀̕b" },
  { "a゙्꠆ְb", "a゙्꠆ְb", "a゙्꠆ְb", "a゙्꠆ְb", "a゙्꠆ְb" },
  { "a゙꠆्ְb", "a゙꠆्ְb", "a゙꠆्ְb", "a゙꠆्ְb", "a゙꠆्ְb" },
  { "a゙्꣄ְb", "a゙्꣄ְb", "a゙्꣄ְb", "a゙्꣄ְb", "a゙्꣄ְb" },
  { "a゙꣄्ְb", "a゙꣄्ְb", "a゙꣄्ְb", "a゙꣄्ְb", "a゙꣄्ְb" },
  { "à֮꣠̕b", "à֮꣠̕b", "à֮꣠̕b", "à֮꣠̕b", "à֮꣠̕b" },
  { "a֮꣠̀̕b", "a֮꣠̀̕b", "a֮꣠̀̕b", "a֮꣠̀̕b", "a֮꣠̀̕b" },
  { "à֮꣡̕b", "à֮꣡̕b", "à֮꣡̕b", "à֮꣡̕b", "à֮꣡̕b" },
  { "a֮꣡̀̕b", "a֮꣡̀̕b", "a֮꣡̀̕b", "a֮꣡̀̕b", "a֮꣡̀̕b" },
  { "à֮꣢̕b", "à֮꣢̕b", "à֮꣢̕b", "à֮꣢̕b", "à֮꣢̕b" },
  { "a֮꣢̀̕b", "a֮꣢̀̕b", "a֮꣢̀̕b", "a֮꣢̀̕b", "a֮꣢̀̕b" },
  { "à֮꣣̕b", "à֮꣣̕b", "à֮꣣̕b", "à֮꣣̕b", "à֮꣣̕b" },
  { "a֮꣣̀̕b", "a֮꣣̀̕b", "a֮꣣̀̕b", "a֮꣣̀̕b", "a֮꣣̀̕b" },
  { "à֮꣤̕b", "à֮꣤̕b", "à֮꣤̕b", "à֮꣤̕b", "à֮꣤̕b" },
  { "a֮꣤̀̕b", "a֮꣤̀̕b", "a֮꣤̀̕b", "a֮꣤̀̕b", "a֮꣤̀̕b" },
  { "à֮꣥̕b", "à֮꣥̕b", "à֮꣥̕b", "à֮꣥̕b", "à֮꣥̕b" },
  { "a֮꣥̀̕b", "a֮꣥̀̕b", "a֮꣥̀̕b", "a֮꣥̀̕b", "a֮꣥̀̕b" },
  { "à֮꣦̕b", "à֮꣦̕b", "à֮꣦̕b", "à֮꣦̕b", "à֮꣦̕b" },
  { "a֮꣦̀̕b", "a֮꣦̀̕b", "a֮꣦̀̕b", "a֮꣦̀̕b", "a֮꣦̀̕b" },
  { "à֮꣧̕b", "à֮꣧̕b", "à֮꣧̕b", "à֮꣧̕b", "à֮꣧̕b" },
  { "a֮꣧̀̕b", "a֮꣧̀̕b", "a֮꣧̀̕b", "a֮꣧̀̕b", "a֮꣧̀̕b" },
  { "à֮꣨̕b", "à֮꣨̕b", "à֮꣨̕b", "à֮꣨̕b", "à֮꣨̕b" },
  { "a֮꣨̀̕b", "a֮꣨̀̕b", "a֮꣨̀̕b", "a֮꣨̀̕b", "a֮꣨̀̕b" },
  { "à֮꣩̕b", "à֮꣩̕b", "à֮꣩̕b", "à֮꣩̕b", "à֮꣩̕b" },
  { "a֮꣩̀̕b", "a֮꣩̀̕b", "a֮꣩̀̕b", "a֮꣩̀̕b", "a֮꣩̀̕b" },
  { "à֮꣪̕b", "à֮꣪̕b", "à֮꣪̕b", "à֮꣪̕b", "à֮꣪̕b" },
  { "a֮꣪̀̕b", "a֮꣪̀̕b", "a֮꣪̀̕b", "a֮꣪̀̕b", "a֮꣪̀̕b" },
  { "à֮꣫̕b", "à֮꣫̕b", "à֮꣫̕b", "à֮꣫̕b", "à֮꣫̕b" },
  { "a֮꣫̀̕b", "a֮꣫̀̕b", "a֮꣫̀̕b", "a֮꣫̀̕b", "a֮꣫̀̕b" },
  { "à֮꣬̕b", "à֮꣬̕b", "à֮꣬̕b", "à֮꣬̕b", "à֮꣬̕b" },
  { "a֮꣬̀̕b", "a֮꣬̀̕b", "a֮꣬̀̕b", "a֮꣬̀̕b", "a֮꣬̀̕b" },
  { "à֮꣭̕b", "à֮꣭̕b", "à֮꣭̕b", "à֮꣭̕b", "à֮꣭̕b" },
  { "a֮꣭̀̕b", "a֮꣭̀̕b", "a֮꣭̀̕b", "a֮꣭̀̕b", "a֮꣭̀̕b" },
  { "à֮꣮̕b", "à֮꣮̕b", "à֮꣮̕b", "à֮꣮̕b", "à֮꣮̕b" },
  { "a֮꣮̀̕b", "a֮꣮̀̕b", "a֮꣮̀̕b", "a֮꣮̀̕b", "a֮꣮̀̕b" },
  { "à֮꣯̕b", "à֮꣯̕b", "à֮꣯̕b", "à֮꣯̕b", "à֮꣯̕b" },
  { "a֮꣯̀̕b", "a֮꣯̀̕b", "a֮꣯̀̕b", "a֮꣯̀̕b", "a֮꣯̀̕b" },
  { "à֮꣰̕b", "à֮꣰̕b", "à֮꣰̕b", "à֮꣰̕b", "à֮꣰̕b" },
  { "a֮꣰̀̕b", "a֮꣰̀̕b", "a֮꣰̀̕b", "a֮꣰̀̕b", "a֮꣰̀̕b" },
  { "à֮꣱̕b", "à֮꣱̕b", "à֮꣱̕b", "à֮꣱̕b", "à֮꣱̕b" },
  { "a֮꣱̀̕b", "a֮꣱̀̕b", "a֮꣱̀̕b", "a֮꣱̀̕b", "a֮꣱̀̕b" },
  { "a〪̖꤫֚b", "a〪̖꤫֚b", "a〪̖꤫֚b", "a〪̖꤫֚b", "a〪̖꤫֚b" },
  { "a〪꤫̖֚b", "a〪꤫̖֚b", "a〪꤫̖֚b", "a〪꤫̖֚b", "a〪꤫̖֚b" },
  { "a〪̖꤬֚b", "a〪̖꤬֚b", "a〪̖꤬֚b", "a〪̖꤬֚b", "a〪̖꤬֚b" },
  { "a〪꤬̖֚b", "a〪꤬̖֚b", "a〪꤬̖֚b", "a〪꤬̖֚b", "a〪꤬̖֚b" },
  { "a〪̖꤭֚b", "a〪̖꤭֚b", "a〪̖꤭֚b", "a〪̖꤭֚b", "a〪̖꤭֚b" },
  { "a〪꤭̖֚b", "a〪꤭̖֚b", "a〪꤭̖֚b", "a〪꤭̖֚b", "a〪꤭̖֚b" },
  { "a゙्꥓ְb", "a゙्꥓ְb", "a゙्꥓ְb", "a゙्꥓ְb", "a゙्꥓ְb" },
  { "a゙꥓्ְb", "a゙꥓्ְb", "a゙꥓्ְb", "a゙꥓्ְb", "a゙꥓्ְb" },
  { "a̴़꦳゙b", "a̴़꦳゙b", "a̴़꦳゙b", "a̴़꦳゙b", "a̴़꦳゙b" },
  { "a̴꦳़゙b", "a̴꦳़゙b", "a̴꦳़゙b", "a̴꦳़゙b", "a̴꦳़゙b" },
  { "a゙्꧀ְb", "a゙्꧀ְb", "a゙्꧀ְb", "a゙्꧀ְb", "a゙्꧀ְb" },
  { "a゙꧀्ְb", "a゙꧀्ְb", "a゙꧀्ְb", "a゙꧀्ְb", "a゙꧀्ְb" },
  { "à֮ꪰ̕b", "à֮ꪰ̕b", "à֮ꪰ̕b", "à֮ꪰ̕b", "à֮ꪰ̕b" },
  { "a֮ꪰ̀̕b", "a֮ꪰ̀̕b", "a֮ꪰ̀̕b", "a֮ꪰ̀̕b", "a֮ꪰ̀̕b" },
  { "à֮ꪲ̕b", "à֮ꪲ̕b", "à֮ꪲ̕b", "à֮ꪲ̕b", "à֮ꪲ̕b" },
  { "a֮ꪲ̀̕b", "a֮ꪲ̀̕b", "a֮ꪲ̀̕b", "a֮ꪲ̀̕b", "a֮ꪲ̀̕b" },
  { "à֮ꪳ̕b", "à֮ꪳ̕b", "à֮ꪳ̕b", "à֮ꪳ̕b", "à֮ꪳ̕b" },
  { "a֮ꪳ̀̕b", "a֮ꪳ̀̕b", "a֮ꪳ̀̕b", "a֮ꪳ̀̕b", "a֮ꪳ̀̕b" },
  { "a〪̖ꪴ֚b", "a〪̖ꪴ֚b", "a〪̖ꪴ֚b", "a〪̖ꪴ֚b", "a〪̖ꪴ֚b" },
  { "a〪ꪴ̖֚b", "a〪ꪴ̖֚b", "a〪ꪴ̖֚b", "a〪ꪴ̖֚b", "a〪ꪴ̖֚b" },
  { "à֮ꪷ̕b", "à֮ꪷ̕b", "à֮ꪷ̕b", "à֮ꪷ̕b", "à֮ꪷ̕b" },
  { "a֮ꪷ̀̕b", "a֮ꪷ̀̕b", "a֮ꪷ̀̕b", "a֮ꪷ̀̕b", "a֮ꪷ̀̕b" },
  { "à֮ꪸ̕b", "à֮ꪸ̕b", "à֮ꪸ̕b", "à֮ꪸ̕b", "à֮ꪸ̕b" },
  { "a֮ꪸ̀̕b", "a֮ꪸ̀̕b", "a֮ꪸ̀̕b", "a֮ꪸ̀̕b", "a֮ꪸ̀̕b" },
  { "à֮ꪾ̕b", "à֮ꪾ̕b", "à֮ꪾ̕b", "à֮ꪾ̕b", "à֮ꪾ̕b" },
  { "a֮ꪾ̀̕b", "a֮ꪾ̀̕b", "a֮ꪾ̀̕b", "a֮ꪾ̀̕b", "a֮ꪾ̀̕b" },
  { "à֮꪿̕b", "à֮꪿̕b", "à֮꪿̕b", "à֮꪿̕b", "à֮꪿̕b" },
  { "a֮꪿̀̕b", "a֮꪿̀̕b", "a֮꪿̀̕b", "a֮꪿̀̕b", "a֮꪿̀̕b" },
  { "à֮꫁̕b", "à֮꫁̕b", "à֮꫁̕b", "à֮꫁̕b", "à֮꫁̕b" },
  { "a֮꫁̀̕b", "a֮꫁̀̕b", "a֮꫁̀̕b", "a֮꫁̀̕b", "a֮꫁̀̕b" },
  { "a゙्꫶ְb", "a゙्꫶ְb", "a゙्꫶ְb", "a゙्꫶ְb", "a゙्꫶ְb" },
  { "a゙꫶्ְb", "a゙꫶्ְb", "a゙꫶्ְb", "a゙꫶्ְb", "a゙꫶्ְb" },
  { "a゙्꯭ְb", "a゙्꯭ְb", "a゙्꯭ְb", "a゙्꯭ְb", "a゙्꯭ְb" },
  { "a゙꯭्ְb", "a゙꯭्ְb", "a゙꯭्ְb", "a゙꯭्ְb", "a゙꯭्ְb" },
  { "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb" },
  { "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb", "aׂﬞﬞًb" },
  { "à֮︠̕b", "à֮︠̕b", "à֮︠̕b", "à֮︠̕b", "à֮︠̕b" },
  { "a֮︠̀̕b", "a֮︠̀̕b", "a֮︠̀̕b", "a֮︠̀̕b", "a֮︠̀̕b" },
  { "à֮︡̕b", "à֮︡̕b", "à֮︡̕b", "à֮︡̕b", "à֮︡̕b" },
  { "a֮︡̀̕b", "a֮︡̀̕b", "a֮︡̀̕b", "a֮︡̀̕b", "a֮︡̀̕b" },
  { "à֮︢̕b", "à֮︢̕b", "à֮︢̕b", "à֮︢̕b", "à֮︢̕b" },
  { "a֮︢̀̕b", "a֮︢̀̕b", "a֮︢̀̕b", "a֮︢̀̕b", "a֮︢̀̕b" },
  { "à֮︣̕b", "à֮︣̕b", "à֮︣̕b", "à֮︣̕b", "à֮︣̕b" },
  { "a֮︣̀̕b", "a֮︣̀̕b", "a֮︣̀̕b", "a֮︣̀̕b", "a֮︣̀̕b" },
  { "à֮︤̕b", "à֮︤̕b", "à֮︤̕b", "à֮︤̕b", "à֮︤̕b" },
  { "a֮︤̀̕b", "a֮︤̀̕b", "a֮︤̀̕b", "a֮︤̀̕b", "a֮︤̀̕b" },
  { "à֮︥̕b", "à֮︥̕b", "à֮︥̕b", "à֮︥̕b", "à֮︥̕b" },
  { "a֮︥̀̕b", "a֮︥̀̕b", "a֮︥̀̕b", "a֮︥̀̕b", "a֮︥̀̕b" },
  { "à֮︦̕b", "à֮︦̕b", "à֮︦̕b", "à֮︦̕b", "à֮︦̕b" },
  { "a֮︦̀̕b", "a֮︦̀̕b", "a֮︦̀̕b", "a֮︦̀̕b", "a֮︦̀̕b" },
  { "a〪̖︧֚b", "a〪̖︧֚b", "a〪̖︧֚b", "a〪̖︧֚b", "a〪̖︧֚b" },
  { "a〪︧̖֚b", "a〪︧̖֚b", "a〪︧̖֚b", "a〪︧̖֚b", "a〪︧̖֚b" },
  { "a〪̖︨֚b", "a〪̖︨֚b", "a〪̖︨֚b", "a〪̖︨֚b", "a〪̖︨֚b" },
  { "a〪︨̖֚b", "a〪︨̖֚b", "a〪︨̖֚b", "a〪︨̖֚b", "a〪︨̖֚b" },
  { "a〪̖︩֚b", "a〪̖︩֚b", "a〪̖︩֚b", "a〪̖︩֚b", "a〪̖︩֚b" },
  { "a〪︩̖֚b", "a〪︩̖֚b", "a〪︩̖֚b", "a〪︩̖֚b", "a〪︩̖֚b" },
  { "a〪̖︪֚b", "a〪̖︪֚b", "a〪̖︪֚b", "a〪̖︪֚b", "a〪̖︪֚b" },
  { "a〪︪̖֚b", "a〪︪̖֚b", "a〪︪̖֚b", "a〪︪̖֚b", "a〪︪̖֚b" },
  { "a〪̖︫֚b", "a〪̖︫֚b", "a〪̖︫֚b", "a〪̖︫֚b", "a〪̖︫֚b" },
  { "a〪︫̖֚b", "a〪︫̖֚b", "a〪︫̖֚b", "a〪︫̖֚b", "a〪︫̖֚b" },
  { "a〪̖︬֚b", "a〪̖︬֚b", "a〪̖︬֚b", "a〪̖︬֚b", "a〪̖︬֚b" },
  { "a〪︬̖֚b", "a〪︬̖֚b", "a〪︬̖֚b", "a〪︬̖֚b", "a〪︬̖֚b" },
  { "a〪̖︭֚b", "a〪̖︭֚b", "a〪̖︭֚b", "a〪̖︭֚b", "a〪̖︭֚b" },
  { "a〪︭̖֚b", "a〪︭̖֚b", "a〪︭̖֚b", "a〪︭̖֚b", "a〪︭̖֚b" },
  { "à֮︮̕b", "à֮︮̕b", "à֮︮̕b", "à֮︮̕b", "à֮︮̕b" },
  { "a֮︮̀̕b", "a֮︮̀̕b", "a֮︮̀̕b", "a֮︮̀̕b", "a֮︮̀̕b" },
  { "à֮︯̕b", "à֮︯̕b", "à֮︯̕b", "à֮︯̕b", "à֮︯̕b" },
  { "a֮︯̀̕b", "a֮︯̀̕b", "a֮︯̀̕b", "a֮︯̀̕b", "a֮︯̀̕b" },
  { "a〪̖𐇽֚b", "a〪̖𐇽֚b", "a〪̖𐇽֚b", "a〪̖𐇽֚b", "a〪̖𐇽֚b" },
  { "a〪𐇽̖֚b", "a〪𐇽̖֚b", "a〪𐇽̖֚b", "a〪𐇽̖֚b", "a〪𐇽̖֚b" },
  { "a〪̖𐋠֚b", "a〪̖𐋠֚b", "a〪̖𐋠֚b", "a〪̖𐋠֚b", "a〪̖𐋠֚b" },
  { "a〪𐋠̖֚b", "a〪𐋠̖֚b", "a〪𐋠̖֚b", "a〪𐋠̖֚b", "a〪𐋠̖֚b" },
  { "à֮𐍶̕b", "à֮𐍶̕b", "à֮𐍶̕b", "à֮𐍶̕b", "à֮𐍶̕b" },
  { "a֮𐍶̀̕b", "a֮𐍶̀̕b", "a֮𐍶̀̕b", "a֮𐍶̀̕b", "a֮𐍶̀̕b" },
  { "à֮𐍷̕b", "à֮𐍷̕b", "à֮𐍷̕b", "à֮𐍷̕b", "à֮𐍷̕b" },
  { "a֮𐍷̀̕b", "a֮𐍷̀̕b", "a֮𐍷̀̕b", "a֮𐍷̀̕b", "a֮𐍷̀̕b" },
  { "à֮𐍸̕b", "à֮𐍸̕b", "à֮𐍸̕b", "à֮𐍸̕b", "à֮𐍸̕b" },
  { "a֮𐍸̀̕b", "a֮𐍸̀̕b", "a֮𐍸̀̕b", "a֮𐍸̀̕b", "a֮𐍸̀̕b" },
  { "à֮𐍹̕b", "à֮𐍹̕b", "à֮𐍹̕b", "à֮𐍹̕b", "à֮𐍹̕b" },
  { "a֮𐍹̀̕b", "a֮𐍹̀̕b", "a֮𐍹̀̕b", "a֮𐍹̀̕b", "a֮𐍹̀̕b" },
  { "à֮𐍺̕b", "à֮𐍺̕b", "à֮𐍺̕b", "à֮𐍺̕b", "à֮𐍺̕b" },
  { "a֮𐍺̀̕b", "a֮𐍺̀̕b", "a֮𐍺̀̕b", "a֮𐍺̀̕b", "a֮𐍺̀̕b" },
  { "a〪̖𐨍֚b", "a〪̖𐨍֚b", "a〪̖𐨍֚b", "a〪̖𐨍֚b", "a〪̖𐨍֚b" },
  { "a〪𐨍̖֚b", "a〪𐨍̖֚b", "a〪𐨍̖֚b", "a〪𐨍̖֚b", "a〪𐨍̖֚b" },
  { "à֮𐨏̕b", "à֮𐨏̕b", "à֮𐨏̕b", "à֮𐨏̕b", "à֮𐨏̕b" },
  { "a֮𐨏̀̕b", "a֮𐨏̀̕b", "a֮𐨏̀̕b", "a֮𐨏̀̕b", "a֮𐨏̀̕b" },
  { "à֮𐨸̕b", "à֮𐨸̕b", "à֮𐨸̕b", "à֮𐨸̕b", "à֮𐨸̕b" },
  { "a֮𐨸̀̕b", "a֮𐨸̀̕b", "a֮𐨸̀̕b", "a֮𐨸̀̕b", "a֮𐨸̀̕b" },
  { "a̴𐨹़b", "a̴𐨹़b", "a̴𐨹़b", "a̴𐨹़b", "a̴𐨹़b" },
  { "a𐨹̴़b", "a𐨹̴़b", "a𐨹̴़b", "a𐨹̴़b", "a𐨹̴़b" },
  { "a〪̖𐨺֚b", "a〪̖𐨺֚b", "a〪̖𐨺֚b", "a〪̖𐨺֚b", "a〪̖𐨺֚b" },
  { "a〪𐨺̖֚b", "a〪𐨺̖֚b", "a〪𐨺̖֚b", "a〪𐨺̖֚b", "a〪𐨺̖֚b" },
  { "a゙्𐨿ְb", "a゙्𐨿ְb", "a゙्𐨿ְb", "a゙्𐨿ְb", "a゙्𐨿ְb" },
  { "a゙𐨿्ְb", "a゙𐨿्ְb", "a゙𐨿्ְb", "a゙𐨿्ְb", "a゙𐨿्ְb" },
  { "à֮𐫥̕b", "à֮𐫥̕b", "à֮𐫥̕b", "à֮𐫥̕b", "à֮𐫥̕b" },
  { "a֮𐫥̀̕b", "a֮𐫥̀̕b", "a֮𐫥̀̕b", "a֮𐫥̀̕b", "a֮𐫥̀̕b" },
  { "a〪̖𐫦֚b", "a〪̖𐫦֚b", "a〪̖𐫦֚b", "a〪̖𐫦֚b", "a〪̖𐫦֚b" },
  { "a〪𐫦̖֚b", "a〪𐫦̖֚b", "a〪𐫦̖֚b", "a〪𐫦̖֚b", "a〪𐫦̖֚b" },
  { "à֮𐴤̕b", "à֮𐴤̕b", "à֮𐴤̕b", "à֮𐴤̕b", "à֮𐴤̕b" },
  { "a֮𐴤̀̕b", "a֮𐴤̀̕b", "a֮𐴤̀̕b", "a֮𐴤̀̕b", "a֮𐴤̀̕b" },
  { "à֮𐴥̕b", "à֮𐴥̕b", "à֮𐴥̕b", "à֮𐴥̕b", "à֮𐴥̕b" },
  { "a֮𐴥̀̕b", "a֮𐴥̀̕b", "a֮𐴥̀̕b", "a֮𐴥̀̕b", "a֮𐴥̀̕b" },
  { "à֮𐴦̕b", "à֮𐴦̕b", "à֮𐴦̕b", "à֮𐴦̕b", "à֮𐴦̕b" },
  { "a֮𐴦̀̕b", "a֮𐴦̀̕b", "a֮𐴦̀̕b", "a֮𐴦̀̕b", "a֮𐴦̀̕b" },
  { "à֮𐴧̕b", "à֮𐴧̕b", "à֮𐴧̕b", "à֮𐴧̕b", "à֮𐴧̕b" },
  { "a֮𐴧̀̕b", "a֮𐴧̀̕b", "a֮𐴧̀̕b", "a֮𐴧̀̕b", "a֮𐴧̀̕b" },
  { "a〪̖𐽆֚b", "a〪̖𐽆֚b", "a〪̖𐽆֚b", "a〪̖𐽆֚b", "a〪̖𐽆֚b" },
  { "a〪𐽆̖֚b", "a〪𐽆̖֚b", "a〪𐽆̖֚b", "a〪𐽆̖֚b", "a〪𐽆̖֚b" },
  { "a〪̖𐽇֚b", "a〪̖𐽇֚b", "a〪̖𐽇֚b", "a〪̖𐽇֚b", "a〪̖𐽇֚b" },
  { "a〪𐽇̖֚b", "a〪𐽇̖֚b", "a〪𐽇̖֚b", "a〪𐽇̖֚b", "a〪𐽇̖֚b" },
  { "à֮𐽈̕b", "à֮𐽈̕b", "à֮𐽈̕b", "à֮𐽈̕b", "à֮𐽈̕b" },
  { "a֮𐽈̀̕b", "a֮𐽈̀̕b", "a֮𐽈̀̕b", "a֮𐽈̀̕b", "a֮𐽈̀̕b" },
  { "à֮𐽉̕b", "à֮𐽉̕b", "à֮𐽉̕b", "à֮𐽉̕b", "à֮𐽉̕b" },
  { "a֮𐽉̀̕b", "a֮𐽉̀̕b", "a֮𐽉̀̕b", "a֮𐽉̀̕b", "a֮𐽉̀̕b" },
  { "à֮𐽊̕b", "à֮𐽊̕b", "à֮𐽊̕b", "à֮𐽊̕b", "à֮𐽊̕b" },
  { "a֮𐽊̀̕b", "a֮𐽊̀̕b", "a֮𐽊̀̕b", "a֮𐽊̀̕b", "a֮𐽊̀̕b" },
  { "a〪̖𐽋֚b", "a〪̖𐽋֚b", "a〪̖𐽋֚b", "a〪̖𐽋֚b", "a〪̖𐽋֚b" },
  { "a〪𐽋̖֚b", "a〪𐽋̖֚b", "a〪𐽋̖֚b", "a〪𐽋̖֚b", "a〪𐽋̖֚b" },
  { "à֮𐽌̕b", "à֮𐽌̕b", "à֮𐽌̕b", "à֮𐽌̕b", "à֮𐽌̕b" },
  { "a֮𐽌̀̕b", "a֮𐽌̀̕b", "a֮𐽌̀̕b", "a֮𐽌̀̕b", "a֮𐽌̀̕b" },
  { "a〪̖𐽍֚b", "a〪̖𐽍֚b", "a〪̖𐽍֚b", "a〪̖𐽍֚b", "a〪̖𐽍֚b" },
  { "a〪𐽍̖֚b", "a〪𐽍̖֚b", "a〪𐽍̖֚b", "a〪𐽍̖֚b", "a〪𐽍̖֚b" },
  { "a〪̖𐽎֚b", "a〪̖𐽎֚b", "a〪̖𐽎֚b", "a〪̖𐽎֚b", "a〪̖𐽎֚b" },
  { "a〪𐽎̖֚b", "a〪𐽎̖֚b", "a〪𐽎̖֚b", "a〪𐽎̖֚b", "a〪𐽎̖֚b" },
  { "a〪̖𐽏֚b", "a〪̖𐽏֚b", "a〪̖𐽏֚b", "a〪̖𐽏֚b", "a〪̖𐽏֚b" },
  { "a〪𐽏̖֚b", "a〪𐽏̖֚b", "a〪𐽏̖֚b", "a〪𐽏̖֚b", "a〪𐽏̖֚b" },
  { "a〪̖𐽐֚b", "a〪̖𐽐֚b", "a〪̖𐽐֚b", "a〪̖𐽐֚b", "a〪̖𐽐֚b" },
  { "a〪𐽐̖֚b", "a〪𐽐̖֚b", "a〪𐽐̖֚b", "a〪𐽐̖֚b", "a〪𐽐̖֚b" },
  { "a゙्𑁆ְb", "a゙्𑁆ְb", "a゙्𑁆ְb", "a゙्𑁆ְb", "a゙्𑁆ְb" },
  { "a゙𑁆्ְb", "a゙𑁆्ְb", "a゙𑁆्ְb", "a゙𑁆्ְb", "a゙𑁆्ְb" },
  { "a゙्𑁿ְb", "a゙्𑁿ְb", "a゙्𑁿ְb", "a゙्𑁿ְb", "a゙्𑁿ְb" },
  { "a゙𑁿्ְb", "a゙𑁿्ְb", "a゙𑁿्ְb", "a゙𑁿्ְb", "a゙𑁿्ְb" },
  { "a゙्𑂹ְb", "a゙्𑂹ְb", "a゙्𑂹ְb", "a゙्𑂹ְb", "a゙्𑂹ְb" },
  { "a゙𑂹्ְb", "a゙𑂹्ְb", "a゙𑂹्ְb", "a゙𑂹्ְb", "a゙𑂹्ְb" },
  { "a̴़𑂺゙b", "a̴़𑂺゙b", "a̴़𑂺゙b", "a̴़𑂺゙b", "a̴़𑂺゙b" },
  { "a̴𑂺़゙b", "a̴𑂺़゙b", "a̴𑂺़゙b", "a̴𑂺़゙b", "a̴𑂺़゙b" },
  { "à֮𑄀̕b", "à֮𑄀̕b", "à֮𑄀̕b", "à֮𑄀̕b", "à֮𑄀̕b" },
  { "a֮𑄀̀̕b", "a֮𑄀̀̕b", "a֮𑄀̀̕b", "a֮𑄀̀̕b", "a֮𑄀̀̕b" },
  { "à֮𑄁̕b", "à֮𑄁̕b", "à֮𑄁̕b", "à֮𑄁̕b", "à֮𑄁̕b" },
  { "a֮𑄁̀̕b", "a֮𑄁̀̕b", "a֮𑄁̀̕b", "a֮𑄁̀̕b", "a֮𑄁̀̕b" },
  { "à֮𑄂̕b", "à֮𑄂̕b", "à֮𑄂̕b", "à֮𑄂̕b", "à֮𑄂̕b" },
  { "a֮𑄂̀̕b", "a֮𑄂̀̕b", "a֮𑄂̀̕b", "a֮𑄂̀̕b", "a֮𑄂̀̕b" },
  { "a゙्𑄳ְb", "a゙्𑄳ְb", "a゙्𑄳ְb", "a゙्𑄳ְb", "a゙्𑄳ְb" },
  { "a゙𑄳्ְb", "a゙𑄳्ְb", "a゙𑄳्ְb", "a゙𑄳्ְb", "a゙𑄳्ְb" },
  { "a゙्𑄴ְb", "a゙्𑄴ְb", "a゙्𑄴ְb", "a゙्𑄴ְb", "a゙्𑄴ְb" },
  { "a゙𑄴्ְb", "a゙𑄴्ְb", "a゙𑄴्ְb", "a゙𑄴्ְb", "a゙𑄴्ְb" },
  { "a̴़𑅳゙b", "a̴़𑅳゙b", "a̴़𑅳゙b", "a̴़𑅳゙b", "a̴़𑅳゙b" },
  { "a̴𑅳़゙b", "a̴𑅳़゙b", "a̴𑅳़゙b", "a̴𑅳़゙b", "a̴𑅳़゙b" },
  { "a゙्𑇀ְb", "a゙्𑇀ְb", "a゙्𑇀ְb", "a゙्𑇀ְb", "a゙्𑇀ְb" },
  { "a゙𑇀्ְb", "a゙𑇀्ְb", "a゙𑇀्ְb", "a゙𑇀्ְb", "a゙𑇀्ְb" },
  { "a̴़𑇊゙b", "a̴़𑇊゙b", "a̴़𑇊゙b", "a̴़𑇊゙b", "a̴़𑇊゙b" },
  { "a̴𑇊़゙b", "a̴𑇊़゙b", "a̴𑇊़゙b", "a̴𑇊़゙b", "a̴𑇊़゙b" },
  { "a゙्𑈵ְb", "a゙्𑈵ְb", "a゙्𑈵ְb", "a゙्𑈵ְb", "a゙्𑈵ְb" },
  { "a゙𑈵्ְb", "a゙𑈵्ְb", "a゙𑈵्ְb", "a゙𑈵्ְb", "a゙𑈵्ְb" },
  { "a̴़𑈶゙b", "a̴़𑈶゙b", "a̴़𑈶゙b", "a̴़𑈶゙b", "a̴़𑈶゙b" },
  { "a̴𑈶़゙b", "a̴𑈶़゙b", "a̴𑈶़゙b", "a̴𑈶़゙b", "a̴𑈶़゙b" },
  { "a̴़𑋩゙b", "a̴़𑋩゙b", "a̴़𑋩゙b", "a̴़𑋩゙b", "a̴़𑋩゙b" },
  { "a̴𑋩़゙b", "a̴𑋩़゙b", "a̴𑋩़゙b", "a̴𑋩़゙b", "a̴𑋩़゙b" },
  { "a゙्𑋪ְb", "a゙्𑋪ְb", "a゙्𑋪ְb", "a゙्𑋪ְb", "a゙्𑋪ְb" },
  { "a゙𑋪्ְb", "a゙𑋪्ְb", "a゙𑋪्ְb", "a゙𑋪्ְb", "a゙𑋪्ְb" },
  { "a̴़𑌻゙b", "a̴़𑌻゙b", "a̴़𑌻゙b", "a̴़𑌻゙b", "a̴़𑌻゙b" },
  { "a̴𑌻़゙b", "a̴𑌻़゙b", "a̴𑌻़゙b", "a̴𑌻़゙b", "a̴𑌻़゙b" },
  { "a̴़𑌼゙b", "a̴़𑌼゙b", "a̴़𑌼゙b", "a̴़𑌼゙b", "a̴़𑌼゙b" },
  { "a̴𑌼़゙b", "a̴𑌼़゙b", "a̴𑌼़゙b", "a̴𑌼़゙b", "a̴𑌼़゙b" },
  { "a゙्𑍍ְb", "a゙्𑍍ְb", "a゙्𑍍ְb", "a゙्𑍍ְb", "a゙्𑍍ְb" },
  { "a゙𑍍्ְb", "a゙𑍍्ְb", "a゙𑍍्ְb", "a゙𑍍्ְb", "a゙𑍍्ְb" },
  { "à֮𑍦̕b", "à֮𑍦̕b", "à֮𑍦̕b", "à֮𑍦̕b", "à֮𑍦̕b" },
  { "a֮𑍦̀̕b", "a֮𑍦̀̕b", "a֮𑍦̀̕b", "a֮𑍦̀̕b", "a֮𑍦̀̕b" },
  { "à֮𑍧̕b", "à֮𑍧̕b", "à֮𑍧̕b", "à֮𑍧̕b", "à֮𑍧̕b" },
  { "a֮𑍧̀̕b", "a֮𑍧̀̕b", "a֮𑍧̀̕b", "a֮𑍧̀̕b", "a֮𑍧̀̕b" },
  { "à֮𑍨̕b", "à֮𑍨̕b", "à֮𑍨̕b", "à֮𑍨̕b", "à֮𑍨̕b" },
  { "a֮𑍨̀̕b", "a֮𑍨̀̕b", "a֮𑍨̀̕b", "a֮𑍨̀̕b", "a֮𑍨̀̕b" },
  { "à֮𑍩̕b", "à֮𑍩̕b", "à֮𑍩̕b", "à֮𑍩̕b", "à֮𑍩̕b" },
  { "a֮𑍩̀̕b", "a֮𑍩̀̕b", "a֮𑍩̀̕b", "a֮𑍩̀̕b", "a֮𑍩̀̕b" },
  { "à֮𑍪̕b", "à֮𑍪̕b", "à֮𑍪̕b", "à֮𑍪̕b", "à֮𑍪̕b" },
  { "a֮𑍪̀̕b", "a֮𑍪̀̕b", "a֮𑍪̀̕b", "a֮𑍪̀̕b", "a֮𑍪̀̕b" },
  { "à֮𑍫̕b", "à֮𑍫̕b", "à֮𑍫̕b", "à֮𑍫̕b", "à֮𑍫̕b" },
  { "a֮𑍫̀̕b", "a֮𑍫̀̕b", "a֮𑍫̀̕b", "a֮𑍫̀̕b", "a֮𑍫̀̕b" },
  { "à֮𑍬̕b", "à֮𑍬̕b", "à֮𑍬̕b", "à֮𑍬̕b", "à֮𑍬̕b" },
  { "a֮𑍬̀̕b", "a֮𑍬̀̕b", "a֮𑍬̀̕b", "a֮𑍬̀̕b", "a֮𑍬̀̕b" },
  { "à֮𑍰̕b", "à֮𑍰̕b", "à֮𑍰̕b", "à֮𑍰̕b", "à֮𑍰̕b" },
  { "a֮𑍰̀̕b", "a֮𑍰̀̕b", "a֮𑍰̀̕b", "a֮𑍰̀̕b", "a֮𑍰̀̕b" },
  { "à֮𑍱̕b", "à֮𑍱̕b", "à֮𑍱̕b", "à֮𑍱̕b", "à֮𑍱̕b" },
  { "a֮𑍱̀̕b", "a֮𑍱̀̕b", "a֮𑍱̀̕b", "a֮𑍱̀̕b", "a֮𑍱̀̕b" },
  { "à֮𑍲̕b", "à֮𑍲̕b", "à֮𑍲̕b", "à֮𑍲̕b", "à֮𑍲̕b" },
  { "a֮𑍲̀̕b", "a֮𑍲̀̕b", "a֮𑍲̀̕b", "a֮𑍲̀̕b", "a֮𑍲̀̕b" },
  { "à֮𑍳̕b", "à֮𑍳̕b", "à֮𑍳̕b", "à֮𑍳̕b", "à֮𑍳̕b" },
  { "a֮𑍳̀̕b", "a֮𑍳̀̕b", "a֮𑍳̀̕b", "a֮𑍳̀̕b", "a֮𑍳̀̕b" },
  { "à֮𑍴̕b", "à֮𑍴̕b", "à֮𑍴̕b", "à֮𑍴̕b", "à֮𑍴̕b" },
  { "a֮𑍴̀̕b", "a֮𑍴̀̕b", "a֮𑍴̀̕b", "a֮𑍴̀̕b", "a֮𑍴̀̕b" },
  { "a゙्𑑂ְb", "a゙्𑑂ְb", "a゙्𑑂ְb", "a゙्𑑂ְb", "a゙्𑑂ְb" },
  { "a゙𑑂्ְb", "a゙𑑂्ְb", "a゙𑑂्ְb", "a゙𑑂्ְb", "a゙𑑂्ְb" },
  { "a̴़𑑆゙b", "a̴़𑑆゙b", "a̴़𑑆゙b", "a̴़𑑆゙b", "a̴़𑑆゙b" },
  { "a̴𑑆़゙b", "a̴𑑆़゙b", "a̴𑑆़゙b", "a̴𑑆़゙b", "a̴𑑆़゙b" },
  { "à֮𑑞̕b", "à֮𑑞̕b", "à֮𑑞̕b", "à֮𑑞̕b", "à֮𑑞̕b" },
  { "a֮𑑞̀̕b", "a֮𑑞̀̕b", "a֮𑑞̀̕b", "a֮𑑞̀̕b", "a֮𑑞̀̕b" },
  { "a゙्𑓂ְb", "a゙्𑓂ְb", "a゙्𑓂ְb", "a゙्𑓂ְb", "a゙्𑓂ְb" },
  { "a゙𑓂्ְb", "a゙𑓂्ְb", "a゙𑓂्ְb", "a゙𑓂्ְb", "a゙𑓂्ְb" },
  { "a̴़𑓃゙b", "a̴़𑓃゙b", "a̴़𑓃゙b", "a̴़𑓃゙b", "a̴़𑓃゙b" },
  { "a̴𑓃़゙b", "a̴𑓃़゙b", "a̴𑓃़゙b", "a̴𑓃़゙b", "a̴𑓃़゙b" },
  { "a゙्𑖿ְb", "a゙्𑖿ְb", "a゙्𑖿ְb", "a゙्𑖿ְb", "a゙्𑖿ְb" },
  { "a゙𑖿्ְb", "a゙𑖿्ְb", "a゙𑖿्ְb", "a゙𑖿्ְb", "a゙𑖿्ְb" },
  { "a̴़𑗀゙b", "a̴़𑗀゙b", "a̴़𑗀゙b", "a̴़𑗀゙b", "a̴़𑗀゙b" },
  { "a̴𑗀़゙b", "a̴𑗀़゙b", "a̴𑗀़゙b", "a̴𑗀़゙b", "a̴𑗀़゙b" },
  { "a゙्𑘿ְb", "a゙्𑘿ְb", "a゙्𑘿ְb", "a゙्𑘿ְb", "a゙्𑘿ְb" },
  { "a゙𑘿्ְb", "a゙𑘿्ְb", "a゙𑘿्ְb", "a゙𑘿्ְb", "a゙𑘿्ְb" },
  { "a゙्𑚶ְb", "a゙्𑚶ְb", "a゙्𑚶ְb", "a゙्𑚶ְb", "a゙्𑚶ְb" },
  { "a゙𑚶्ְb", "a゙𑚶्ְb", "a゙𑚶्ְb", "a゙𑚶्ְb", "a゙𑚶्ְb" },
  { "a̴़𑚷゙b", "a̴़𑚷゙b", "a̴़𑚷゙b", "a̴़𑚷゙b", "a̴़𑚷゙b" },
  { "a̴𑚷़゙b", "a̴𑚷़゙b", "a̴𑚷़゙b", "a̴𑚷़゙b", "a̴𑚷़゙b" },
  { "a゙्𑜫ְb", "a゙्𑜫ְb", "a゙्𑜫ְb", "a゙्𑜫ְb", "a゙्𑜫ְb" },
  { "a゙𑜫्ְb", "a゙𑜫्ְb", "a゙𑜫्ְb", "a゙𑜫्ְb", "a゙𑜫्ְb" },
  { "a゙्𑠹ְb", "a゙्𑠹ְb", "a゙्𑠹ְb", "a゙्𑠹ְb", "a゙्𑠹ְb" },
  { "a゙𑠹्ְb", "a゙𑠹्ְb", "a゙𑠹्ְb", "a゙𑠹्ְb", "a゙𑠹्ְb" },
  { "a̴़𑠺゙b", "a̴़𑠺゙b", "a̴़𑠺゙b", "a̴़𑠺゙b", "a̴़𑠺゙b" },
  { "a̴𑠺़゙b", "a̴𑠺़゙b", "a̴𑠺़゙b", "a̴𑠺़゙b", "a̴𑠺़゙b" },
  { "a゙्𑧠ְb", "a゙्𑧠ְb", "a゙्𑧠ְb", "a゙्𑧠ְb", "a゙्𑧠ְb" },
  { "a゙𑧠्ְb", "a゙𑧠्ְb", "a゙𑧠्ְb", "a゙𑧠्ְb", "a゙𑧠्ְb" },
  { "a゙्𑨴ְb", "a゙्𑨴ְb", "a゙्𑨴ְb", "a゙्𑨴ְb", "a゙्𑨴ְb" },
  { "a゙𑨴्ְb", "a゙𑨴्ְb", "a゙𑨴्ְb", "a゙𑨴्ְb", "a゙𑨴्ְb" },
  { "a゙्𑩇ְb", "a゙्𑩇ְb", "a゙्𑩇ְb", "a゙्𑩇ְb", "a゙्𑩇ְb" },
  { "a゙𑩇्ְb", "a゙𑩇्ְb", "a゙𑩇्ְb", "a゙𑩇्ְb", "a゙𑩇्ְb" },
  { "a゙्𑪙ְb", "a゙्𑪙ְb", "a゙्𑪙ְb", "a゙्𑪙ְb", "a゙्𑪙ְb" },
  { "a゙𑪙्ְb", "a゙𑪙्ְb", "a゙𑪙्ְb", "a゙𑪙्ְb", "a゙𑪙्ְb" },
  { "a゙्𑰿ְb", "a゙्𑰿ְb", "a゙्𑰿ְb", "a゙्𑰿ְb", "a゙्𑰿ְb" },
  { "a゙𑰿्ְb", "a゙𑰿्ְb", "a゙𑰿्ְb", "a゙𑰿्ְb", "a゙𑰿्ְb" },
  { "a̴़𑵂゙b", "a̴़𑵂゙b", "a̴़𑵂゙b", "a̴़𑵂゙b", "a̴़𑵂゙b" },
  { "a̴𑵂़゙b", "a̴𑵂़゙b", "a̴𑵂़゙b", "a̴𑵂़゙b", "a̴𑵂़゙b" },
  { "a゙्𑵄ְb", "a゙्𑵄ְb", "a゙्𑵄ְb", "a゙्𑵄ְb", "a゙्𑵄ְb" },
  { "a゙𑵄्ְb", "a゙𑵄्ְb", "a゙𑵄्ְb", "a゙𑵄्ְb", "a゙𑵄्ְb" },
  { "a゙्𑵅ְb", "a゙्𑵅ְb", "a゙्𑵅ְb", "a゙्𑵅ְb", "a゙्𑵅ְb" },
  { "a゙𑵅्ְb", "a゙𑵅्ְb", "a゙𑵅्ְb", "a゙𑵅्ְb", "a゙𑵅्ְb" },
  { "a゙्𑶗ְb", "a゙्𑶗ְb", "a゙्𑶗ְb", "a゙्𑶗ְb", "a゙्𑶗ְb" },
  { "a゙𑶗्ְb", "a゙𑶗्ְb", "a゙𑶗्ְb", "a゙𑶗्ְb", "a゙𑶗्ְb" },
  { "a̴𖫰़b", "a̴𖫰़b", "a̴𖫰़b", "a̴𖫰़b", "a̴𖫰़b" },
  { "a𖫰̴़b", "a𖫰̴़b", "a𖫰̴़b", "a𖫰̴़b", "a𖫰̴़b" },
  { "a̴𖫱़b", "a̴𖫱़b", "a̴𖫱़b", "a̴𖫱़b", "a̴𖫱़b" },
  { "a𖫱̴़b", "a𖫱̴़b", "a𖫱̴़b", "a𖫱̴़b", "a𖫱̴़b" },
  { "a̴𖫲़b", "a̴𖫲़b", "a̴𖫲़b", "a̴𖫲़b", "a̴𖫲़b" },
  { "a𖫲̴़b", "a𖫲̴़b", "a𖫲̴़b", "a𖫲̴़b", "a𖫲̴़b" },
  { "a̴𖫳़b", "a̴𖫳़b", "a̴𖫳़b", "a̴𖫳़b", "a̴𖫳़b" },
  { "a𖫳̴़b", "a𖫳̴़b", "a𖫳̴़b", "a𖫳̴़b", "a𖫳̴़b" },
  { "a̴𖫴़b", "a̴𖫴़b", "a̴𖫴़b", "a̴𖫴़b", "a̴𖫴़b" },
  { "a𖫴̴़b", "a𖫴̴़b", "a𖫴̴़b", "a𖫴̴़b", "a𖫴̴़b" },
  { "à֮𖬰̕b", "à֮𖬰̕b", "à֮𖬰̕b", "à֮𖬰̕b", "à֮𖬰̕b" },
  { "a֮𖬰̀̕b", "a֮𖬰̀̕b", "a֮𖬰̀̕b", "a֮𖬰̀̕b", "a֮𖬰̀̕b" },
  { "à֮𖬱̕b", "à֮𖬱̕b", "à֮𖬱̕b", "à֮𖬱̕b", "à֮𖬱̕b" },
  { "a֮𖬱̀̕b", "a֮𖬱̀̕b", "a֮𖬱̀̕b", "a֮𖬱̀̕b", "a֮𖬱̀̕b" },
  { "à֮𖬲̕b", "à֮𖬲̕b", "à֮𖬲̕b", "à֮𖬲̕b", "à֮𖬲̕b" },
  { "a֮𖬲̀̕b", "a֮𖬲̀̕b", "a֮𖬲̀̕b", "a֮𖬲̀̕b", "a֮𖬲̀̕b" },
  { "à֮𖬳̕b", "à֮𖬳̕b", "à֮𖬳̕b", "à֮𖬳̕b", "à֮𖬳̕b" },
  { "a֮𖬳̀̕b", "a֮𖬳̀̕b", "a֮𖬳̀̕b", "a֮𖬳̀̕b", "a֮𖬳̀̕b" },
  { "à֮𖬴̕b", "à֮𖬴̕b", "à֮𖬴̕b", "à֮𖬴̕b", "à֮𖬴̕b" },
  { "a֮𖬴̀̕b", "a֮𖬴̀̕b", "a֮𖬴̀̕b", "a֮𖬴̀̕b", "a֮𖬴̀̕b" },
  { "à֮𖬵̕b", "à֮𖬵̕b", "à֮𖬵̕b", "à֮𖬵̕b", "à֮𖬵̕b" },
  { "a֮𖬵̀̕b", "a֮𖬵̀̕b", "a֮𖬵̀̕b", "a֮𖬵̀̕b", "a֮𖬵̀̕b" },
  { "à֮𖬶̕b", "à֮𖬶̕b", "à֮𖬶̕b", "à֮𖬶̕b", "à֮𖬶̕b" },
  { "a֮𖬶̀̕b", "a֮𖬶̀̕b", "a֮𖬶̀̕b", "a֮𖬶̀̕b", "a֮𖬶̀̕b" },
  { "a̴𛲞़b", "a̴𛲞़b", "a̴𛲞़b", "a̴𛲞़b", "a̴𛲞़b" },
  { "a𛲞̴़b", "a𛲞̴़b", "a𛲞̴़b", "a𛲞̴़b", "a𛲞̴़b" },
  { "a᷎̛𝅥〪b", "a᷎̛𝅥〪b", "a᷎̛𝅥〪b", "a᷎̛𝅥〪b", "a᷎̛𝅥〪b" },
  { "a᷎𝅥̛〪b", "a᷎𝅥̛〪b", "a᷎𝅥̛〪b", "a᷎𝅥̛〪b", "a᷎𝅥̛〪b" },
  { "a᷎̛𝅦〪b", "a᷎̛𝅦〪b", "a᷎̛𝅦〪b", "a᷎̛𝅦〪b", "a᷎̛𝅦〪b" },
  { "a᷎𝅦̛〪b", "a᷎𝅦̛〪b", "a᷎𝅦̛〪b", "a᷎𝅦̛〪b", "a᷎𝅦̛〪b" },
  { "a̴𝅧़b", "a̴𝅧़b", "a̴𝅧़b", "a̴𝅧़b", "a̴𝅧़b" },
  { "a𝅧̴़b", "a𝅧̴़b", "a𝅧̴़b", "a𝅧̴़b", "a𝅧̴़b" },
  { "a̴𝅨़b", "a̴𝅨़b", "a̴𝅨़b", "a̴𝅨़b", "a̴𝅨़b" },
  { "a𝅨̴़b", "a𝅨̴़b", "a𝅨̴़b", "a𝅨̴़b", "a𝅨̴़b" },
  { "a̴𝅩़b", "a̴𝅩़b", "a̴𝅩़b", "a̴𝅩़b", "a̴𝅩़b" },
  { "a𝅩̴़b", "a𝅩̴़b", "a𝅩̴़b", "a𝅩̴़b", "a𝅩̴़b" },
  { "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b" },
  { "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b", "a〮𝅭𝅭֮b" },
  { "a᷎̛𝅮〪b", "a᷎̛𝅮〪b", "a᷎̛𝅮〪b", "a᷎̛𝅮〪b", "a᷎̛𝅮〪b" },
  { "a᷎𝅮̛〪b", "a᷎𝅮̛〪b", "a᷎𝅮̛〪b", "a᷎𝅮̛〪b", "a᷎𝅮̛〪b" },
  { "a᷎̛𝅯〪b", "a᷎̛𝅯〪b", "a᷎̛𝅯〪b", "a᷎̛𝅯〪b", "a᷎̛𝅯〪b" },
  { "a᷎𝅯̛〪b", "a᷎𝅯̛〪b", "a᷎𝅯̛〪b", "a᷎𝅯̛〪b", "a᷎𝅯̛〪b" },
  { "a᷎̛𝅰〪b", "a᷎̛𝅰〪b", "a᷎̛𝅰〪b", "a᷎̛𝅰〪b", "a᷎̛𝅰〪b" },
  { "a᷎𝅰̛〪b", "a᷎𝅰̛〪b", "a᷎𝅰̛〪b", "a᷎𝅰̛〪b", "a᷎𝅰̛〪b" },
  { "a᷎̛𝅱〪b", "a᷎̛𝅱〪b", "a᷎̛𝅱〪b", "a᷎̛𝅱〪b", "a᷎̛𝅱〪b" },
  { "a᷎𝅱̛〪b", "a᷎𝅱̛〪b", "a᷎𝅱̛〪b", "a᷎𝅱̛〪b", "a᷎𝅱̛〪b" },
  { "a᷎̛𝅲〪b", "a᷎̛𝅲〪b", "a᷎̛𝅲〪b", "a᷎̛𝅲〪b", "a᷎̛𝅲〪b" },
  { "a᷎𝅲̛〪b", "a᷎𝅲̛〪b", "a᷎𝅲̛〪b", "a᷎𝅲̛〪b", "a᷎𝅲̛〪b" },
  { "a〪̖𝅻֚b", "a〪̖𝅻֚b", "a〪̖𝅻֚b", "a〪̖𝅻֚b", "a〪̖𝅻֚b" },
  { "a〪𝅻̖֚b", "a〪𝅻̖֚b", "a〪𝅻̖֚b", "a〪𝅻̖֚b", "a〪𝅻̖֚b" },
  { "a〪̖𝅼֚b", "a〪̖𝅼֚b", "a〪̖𝅼֚b", "a〪̖𝅼֚b", "a〪̖𝅼֚b" },
  { "a〪𝅼̖֚b", "a〪𝅼̖֚b", "a〪𝅼̖֚b", "a〪𝅼̖֚b", "a〪𝅼̖֚b" },
  { "a〪̖𝅽֚b", "a〪̖𝅽֚b", "a〪̖𝅽֚b", "a〪̖𝅽֚b", "a〪̖𝅽֚b" },
  { "a〪𝅽̖֚b", "a〪𝅽̖֚b", "a〪𝅽̖֚b", "a〪𝅽̖֚b", "a〪𝅽̖֚b" },
  { "a〪̖𝅾֚b", "a〪̖𝅾֚b", "a〪̖𝅾֚b", "a〪̖𝅾֚b", "a〪̖𝅾֚b" },
  { "a〪𝅾̖֚b", "a〪𝅾̖֚b", "a〪𝅾̖֚b", "a〪𝅾̖֚b", "a〪𝅾̖֚b" },
  { "a〪̖𝅿֚b", "a〪̖𝅿֚b", "a〪̖𝅿֚b", "a〪̖𝅿֚b", "a〪̖𝅿֚b" },
  { "a〪𝅿̖֚b", "a〪𝅿̖֚b", "a〪𝅿̖֚b", "a〪𝅿̖֚b", "a〪𝅿̖֚b" },
  { "a〪̖𝆀֚b", "a〪̖𝆀֚b", "a〪̖𝆀֚b", "a〪̖𝆀֚b", "a〪̖𝆀֚b" },
  { "a〪𝆀̖֚b", "a〪𝆀̖֚b", "a〪𝆀̖֚b", "a〪𝆀̖֚b", "a〪𝆀̖֚b" },
  { "a〪̖𝆁֚b", "a〪̖𝆁֚b", "a〪̖𝆁֚b", "a〪̖𝆁֚b", "a〪̖𝆁֚b" },
  { "a〪𝆁̖֚b", "a〪𝆁̖֚b", "a〪𝆁̖֚b", "a〪𝆁̖֚b", "a〪𝆁̖֚b" },
  { "a〪̖𝆂֚b", "a〪̖𝆂֚b", "a〪̖𝆂֚b", "a〪̖𝆂֚b", "a〪̖𝆂֚b" },
  { "a〪𝆂̖֚b", "a〪𝆂̖֚b", "a〪𝆂̖֚b", "a〪𝆂̖֚b", "a〪𝆂̖֚b" },
  { "à֮𝆅̕b", "à֮𝆅̕b", "à֮𝆅̕b", "à֮𝆅̕b", "à֮𝆅̕b" },
  { "a֮𝆅̀̕b", "a֮𝆅̀̕b", "a֮𝆅̀̕b", "a֮𝆅̀̕b", "a֮𝆅̀̕b" },
  { "à֮𝆆̕b", "à֮𝆆̕b", "à֮𝆆̕b", "à֮𝆆̕b", "à֮𝆆̕b" },
  { "a֮𝆆̀̕b", "a֮𝆆̀̕b", "a֮𝆆̀̕b", "a֮𝆆̀̕b", "a֮𝆆̀̕b" },
  { "à֮𝆇̕b", "à֮𝆇̕b", "à֮𝆇̕b", "à֮𝆇̕b", "à֮𝆇̕b" },
  { "a֮𝆇̀̕b", "a֮𝆇̀̕b", "a֮𝆇̀̕b", "a֮𝆇̀̕b", "a֮𝆇̀̕b" },
  { "à֮𝆈̕b", "à֮𝆈̕b", "à֮𝆈̕b", "à֮𝆈̕b", "à֮𝆈̕b" },
  { "a֮𝆈̀̕b", "a֮𝆈̀̕b", "a֮𝆈̀̕b", "a֮𝆈̀̕b", "a֮𝆈̀̕b" },
  { "à֮𝆉̕b", "à֮𝆉̕b", "à֮𝆉̕b", "à֮𝆉̕b", "à֮𝆉̕b" },
  { "a֮𝆉̀̕b", "a֮𝆉̀̕b", "a֮𝆉̀̕b", "a֮𝆉̀̕b", "a֮𝆉̀̕b" },
  { "a〪̖𝆊֚b", "a〪̖𝆊֚b", "a〪̖𝆊֚b", "a〪̖𝆊֚b", "a〪̖𝆊֚b" },
  { "a〪𝆊̖֚b", "a〪𝆊̖֚b", "a〪𝆊̖֚b", "a〪𝆊̖֚b", "a〪𝆊̖֚b" },
  { "a〪̖𝆋֚b", "a〪̖𝆋֚b", "a〪̖𝆋֚b", "a〪̖𝆋֚b", "a〪̖𝆋֚b" },
  { "a〪𝆋̖֚b", "a〪𝆋̖֚b", "a〪𝆋̖֚b", "a〪𝆋̖֚b", "a〪𝆋̖֚b" },
  { "à֮𝆪̕b", "à֮𝆪̕b", "à֮𝆪̕b", "à֮𝆪̕b", "à֮𝆪̕b" },
  { "a֮𝆪̀̕b", "a֮𝆪̀̕b", "a֮𝆪̀̕b", "a֮𝆪̀̕b", "a֮𝆪̀̕b" },
  { "à֮𝆫̕b", "à֮𝆫̕b", "à֮𝆫̕b", "à֮𝆫̕b", "à֮𝆫̕b" },
  { "a֮𝆫̀̕b", "a֮𝆫̀̕b", "a֮𝆫̀̕b", "a֮𝆫̀̕b", "a֮𝆫̀̕b" },
  { "à֮𝆬̕b", "à֮𝆬̕b", "à֮𝆬̕b", "à֮𝆬̕b", "à֮𝆬̕b" },
  { "a֮𝆬̀̕b", "a֮𝆬̀̕b", "a֮𝆬̀̕b", "a֮𝆬̀̕b", "a֮𝆬̀̕b" },
  { "à֮𝆭̕b", "à֮𝆭̕b", "à֮𝆭̕b", "à֮𝆭̕b", "à֮𝆭̕b" },
  { "a֮𝆭̀̕b", "a֮𝆭̀̕b", "a֮𝆭̀̕b", "a֮𝆭̀̕b", "a֮𝆭̀̕b" },
  { "à֮𝉂̕b", "à֮𝉂̕b", "à֮𝉂̕b", "à֮𝉂̕b", "à֮𝉂̕b" },
  { "a֮𝉂̀̕b", "a֮𝉂̀̕b", "a֮𝉂̀̕b", "a֮𝉂̀̕b", "a֮𝉂̀̕b" },
  { "à֮𝉃̕b", "à֮𝉃̕b", "à֮𝉃̕b", "à֮𝉃̕b", "à֮𝉃̕b" },
  { "a֮𝉃̀̕b", "a֮𝉃̀̕b", "a֮𝉃̀̕b", "a֮𝉃̀̕b", "a֮𝉃̀̕b" },
  { "à֮𝉄̕b", "à֮𝉄̕b", "à֮𝉄̕b", "à֮𝉄̕b", "à֮𝉄̕b" },
  { "a֮𝉄̀̕b", "a֮𝉄̀̕b", "a֮𝉄̀̕b", "a֮𝉄̀̕b", "a֮𝉄̀̕b" },
  { "à֮𞀀̕b", "à֮𞀀̕b", "à֮𞀀̕b", "à֮𞀀̕b", "à֮𞀀̕b" },
  { "a֮𞀀̀̕b", "a֮𞀀̀̕b", "a֮𞀀̀̕b", "a֮𞀀̀̕b", "a֮𞀀̀̕b" },
  { "à֮𞀁̕b", "à֮𞀁̕b", "à֮𞀁̕b", "à֮𞀁̕b", "à֮𞀁̕b" },
  { "a֮𞀁̀̕b", "a֮𞀁̀̕b", "a֮𞀁̀̕b", "a֮𞀁̀̕b", "a֮𞀁̀̕b" },
  { "à֮𞀂̕b", "à֮𞀂̕b", "à֮𞀂̕b", "à֮𞀂̕b", "à֮𞀂̕b" },
  { "a֮𞀂̀̕b", "a֮𞀂̀̕b", "a֮𞀂̀̕b", "a֮𞀂̀̕b", "a֮𞀂̀̕b" },
  { "à֮𞀃̕b", "à֮𞀃̕b", "à֮𞀃̕b", "à֮𞀃̕b", "à֮𞀃̕b" },
  { "a֮𞀃̀̕b", "a֮𞀃̀̕b", "a֮𞀃̀̕b", "a֮𞀃̀̕b", "a֮𞀃̀̕b" },
  { "à֮𞀄̕b", "à֮𞀄̕b", "à֮𞀄̕b", "à֮𞀄̕b", "à֮𞀄̕b" },
  { "a֮𞀄̀̕b", "a֮𞀄̀̕b", "a֮𞀄̀̕b", "a֮𞀄̀̕b", "a֮𞀄̀̕b" },
  { "à֮𞀅̕b", "à֮𞀅̕b", "à֮𞀅̕b", "à֮𞀅̕b", "à֮𞀅̕b" },
  { "a֮𞀅̀̕b", "a֮𞀅̀̕b", "a֮𞀅̀̕b", "a֮𞀅̀̕b", "a֮𞀅̀̕b" },
  { "à֮𞀆̕b", "à֮𞀆̕b", "à֮𞀆̕b", "à֮𞀆̕b", "à֮𞀆̕b" },
  { "a֮𞀆̀̕b", "a֮𞀆̀̕b", "a֮𞀆̀̕b", "a֮𞀆̀̕b", "a֮𞀆̀̕b" },
  { "à֮𞀈̕b", "à֮𞀈̕b", "à֮𞀈̕b", "à֮𞀈̕b", "à֮𞀈̕b" },
  { "a֮𞀈̀̕b", "a֮𞀈̀̕b", "a֮𞀈̀̕b", "a֮𞀈̀̕b", "a֮𞀈̀̕b" },
  { "à֮𞀉̕b", "à֮𞀉̕b", "à֮𞀉̕b", "à֮𞀉̕b", "à֮𞀉̕b" },
  { "a֮𞀉̀̕b", "a֮𞀉̀̕b", "a֮𞀉̀̕b", "a֮𞀉̀̕b", "a֮𞀉̀̕b" },
  { "à֮𞀊̕b", "à֮𞀊̕b", "à֮𞀊̕b", "à֮𞀊̕b", "à֮𞀊̕b" },
  { "a֮𞀊̀̕b", "a֮𞀊̀̕b", "a֮𞀊̀̕b", "a֮𞀊̀̕b", "a֮𞀊̀̕b" },
  { "à֮𞀋̕b", "à֮𞀋̕b", "à֮𞀋̕b", "à֮𞀋̕b", "à֮𞀋̕b" },
  { "a֮𞀋̀̕b", "a֮𞀋̀̕b", "a֮𞀋̀̕b", "a֮𞀋̀̕b", "a֮𞀋̀̕b" },
  { "à֮𞀌̕b", "à֮𞀌̕b", "à֮𞀌̕b", "à֮𞀌̕b", "à֮𞀌̕b" },
  { "a֮𞀌̀̕b", "a֮𞀌̀̕b", "a֮𞀌̀̕b", "a֮𞀌̀̕b", "a֮𞀌̀̕b" },
  { "à֮𞀍̕b", "à֮𞀍̕b", "à֮𞀍̕b", "à֮𞀍̕b", "à֮𞀍̕b" },
  { "a֮𞀍̀̕b", "a֮𞀍̀̕b", "a֮𞀍̀̕b", "a֮𞀍̀̕b", "a֮𞀍̀̕b" },
  { "à֮𞀎̕b", "à֮𞀎̕b", "à֮𞀎̕b", "à֮𞀎̕b", "à֮𞀎̕b" },
  { "a֮𞀎̀̕b", "a֮𞀎̀̕b", "a֮𞀎̀̕b", "a֮𞀎̀̕b", "a֮𞀎̀̕b" },
  { "à֮𞀏̕b", "à֮𞀏̕b", "à֮𞀏̕b", "à֮𞀏̕b", "à֮𞀏̕b" },
  { "a֮𞀏̀̕b", "a֮𞀏̀̕b", "a֮𞀏̀̕b", "a֮𞀏̀̕b", "a֮𞀏̀̕b" },
  { "à֮𞀐̕b", "à֮𞀐̕b", "à֮𞀐̕b", "à֮𞀐̕b", "à֮𞀐̕b" },
  { "a֮𞀐̀̕b", "a֮𞀐̀̕b", "a֮𞀐̀̕b", "a֮𞀐̀̕b", "a֮𞀐̀̕b" },
  { "à֮𞀑̕b", "à֮𞀑̕b", "à֮𞀑̕b", "à֮𞀑̕b", "à֮𞀑̕b" },
  { "a֮𞀑̀̕b", "a֮𞀑̀̕b", "a֮𞀑̀̕b", "a֮𞀑̀̕b", "a֮𞀑̀̕b" },
  { "à֮𞀒̕b", "à֮𞀒̕b", "à֮𞀒̕b", "à֮𞀒̕b", "à֮𞀒̕b" },
  { "a֮𞀒̀̕b", "a֮𞀒̀̕b", "a֮𞀒̀̕b", "a֮𞀒̀̕b", "a֮𞀒̀̕b" },
  { "à֮𞀓̕b", "à֮𞀓̕b", "à֮𞀓̕b", "à֮𞀓̕b", "à֮𞀓̕b" },
  { "a֮𞀓̀̕b", "a֮𞀓̀̕b", "a֮𞀓̀̕b", "a֮𞀓̀̕b", "a֮𞀓̀̕b" },
  { "à֮𞀔̕b", "à֮𞀔̕b", "à֮𞀔̕b", "à֮𞀔̕b", "à֮𞀔̕b" },
  { "a֮𞀔̀̕b", "a֮𞀔̀̕b", "a֮𞀔̀̕b", "a֮𞀔̀̕b", "a֮𞀔̀̕b" },
  { "à֮𞀕̕b", "à֮𞀕̕b", "à֮𞀕̕b", "à֮𞀕̕b", "à֮𞀕̕b" },
  { "a֮𞀕̀̕b", "a֮𞀕̀̕b", "a֮𞀕̀̕b", "a֮𞀕̀̕b", "a֮𞀕̀̕b" },
  { "à֮𞀖̕b", "à֮𞀖̕b", "à֮𞀖̕b", "à֮𞀖̕b", "à֮𞀖̕b" },
  { "a֮𞀖̀̕b", "a֮𞀖̀̕b", "a֮𞀖̀̕b", "a֮𞀖̀̕b", "a֮𞀖̀̕b" },
  { "à֮𞀗̕b", "à֮𞀗̕b", "à֮𞀗̕b", "à֮𞀗̕b", "à֮𞀗̕b" },
  { "a֮𞀗̀̕b", "a֮𞀗̀̕b", "a֮𞀗̀̕b", "a֮𞀗̀̕b", "a֮𞀗̀̕b" },
  { "à֮𞀘̕b", "à֮𞀘̕b", "à֮𞀘̕b", "à֮𞀘̕b", "à֮𞀘̕b" },
  { "a֮𞀘̀̕b", "a֮𞀘̀̕b", "a֮𞀘̀̕b", "a֮𞀘̀̕b", "a֮𞀘̀̕b" },
  { "à֮𞀛̕b", "à֮𞀛̕b", "à֮𞀛̕b", "à֮𞀛̕b", "à֮𞀛̕b" },
  { "a֮𞀛̀̕b", "a֮𞀛̀̕b", "a֮𞀛̀̕b", "a֮𞀛̀̕b", "a֮𞀛̀̕b" },
  { "à֮𞀜̕b", "à֮𞀜̕b", "à֮𞀜̕b", "à֮𞀜̕b", "à֮𞀜̕b" },
  { "a֮𞀜̀̕b", "a֮𞀜̀̕b", "a֮𞀜̀̕b", "a֮𞀜̀̕b", "a֮𞀜̀̕b" },
  { "à֮𞀝̕b", "à֮𞀝̕b", "à֮𞀝̕b", "à֮𞀝̕b", "à֮𞀝̕b" },
  { "a֮𞀝̀̕b", "a֮𞀝̀̕b", "a֮𞀝̀̕b", "a֮𞀝̀̕b", "a֮𞀝̀̕b" },
  { "à֮𞀞̕b", "à֮𞀞̕b", "à֮𞀞̕b", "à֮𞀞̕b", "à֮𞀞̕b" },
  { "a֮𞀞̀̕b", "a֮𞀞̀̕b", "a֮𞀞̀̕b", "a֮𞀞̀̕b", "a֮𞀞̀̕b" },
  { "à֮𞀟̕b", "à֮𞀟̕b", "à֮𞀟̕b", "à֮𞀟̕b", "à֮𞀟̕b" },
  { "a֮𞀟̀̕b", "a֮𞀟̀̕b", "a֮𞀟̀̕b", "a֮𞀟̀̕b", "a֮𞀟̀̕b" },
  { "à֮𞀠̕b", "à֮𞀠̕b", "à֮𞀠̕b", "à֮𞀠̕b", "à֮𞀠̕b" },
  { "a֮𞀠̀̕b", "a֮𞀠̀̕b", "a֮𞀠̀̕b", "a֮𞀠̀̕b", "a֮𞀠̀̕b" },
  { "à֮𞀡̕b", "à֮𞀡̕b", "à֮𞀡̕b", "à֮𞀡̕b", "à֮𞀡̕b" },
  { "a֮𞀡̀̕b", "a֮𞀡̀̕b", "a֮𞀡̀̕b", "a֮𞀡̀̕b", "a֮𞀡̀̕b" },
  { "à֮𞀣̕b", "à֮𞀣̕b", "à֮𞀣̕b", "à֮𞀣̕b", "à֮𞀣̕b" },
  { "a֮𞀣̀̕b", "a֮𞀣̀̕b", "a֮𞀣̀̕b", "a֮𞀣̀̕b", "a֮𞀣̀̕b" },
  { "à֮𞀤̕b", "à֮𞀤̕b", "à֮𞀤̕b", "à֮𞀤̕b", "à֮𞀤̕b" },
  { "a֮𞀤̀̕b", "a֮𞀤̀̕b", "a֮𞀤̀̕b", "a֮𞀤̀̕b", "a֮𞀤̀̕b" },
  { "à֮𞀦̕b", "à֮𞀦̕b", "à֮𞀦̕b", "à֮𞀦̕b", "à֮𞀦̕b" },
  { "a֮𞀦̀̕b", "a֮𞀦̀̕b", "a֮𞀦̀̕b", "a֮𞀦̀̕b", "a֮𞀦̀̕b" },
  { "à֮𞀧̕b", "à֮𞀧̕b", "à֮𞀧̕b", "à֮𞀧̕b", "à֮𞀧̕b" },
  { "a֮𞀧̀̕b", "a֮𞀧̀̕b", "a֮𞀧̀̕b", "a֮𞀧̀̕b", "a֮𞀧̀̕b" },
  { "à֮𞀨̕b", "à֮𞀨̕b", "à֮𞀨̕b", "à֮𞀨̕b", "à֮𞀨̕b" },
  { "a֮𞀨̀̕b", "a֮𞀨̀̕b", "a֮𞀨̀̕b", "a֮𞀨̀̕b", "a֮𞀨̀̕b" },
  { "à֮𞀩̕b", "à֮𞀩̕b", "à֮𞀩̕b", "à֮𞀩̕b", "à֮𞀩̕b" },
  { "a֮𞀩̀̕b", "a֮𞀩̀̕b", "a֮𞀩̀̕b", "a֮𞀩̀̕b", "a֮𞀩̀̕b" },
  { "à֮𞀪̕b", "à֮𞀪̕b", "à֮𞀪̕b", "à֮𞀪̕b", "à֮𞀪̕b" },
  { "a֮𞀪̀̕b", "a֮𞀪̀̕b", "a֮𞀪̀̕b", "a֮𞀪̀̕b", "a֮𞀪̀̕b" },
  { "à֮𞄰̕b", "à֮𞄰̕b", "à֮𞄰̕b", "à֮𞄰̕b", "à֮𞄰̕b" },
  { "a֮𞄰̀̕b", "a֮𞄰̀̕b", "a֮𞄰̀̕b", "a֮𞄰̀̕b", "a֮𞄰̀̕b" },
  { "à֮𞄱̕b", "à֮𞄱̕b", "à֮𞄱̕b", "à֮𞄱̕b", "à֮𞄱̕b" },
  { "a֮𞄱̀̕b", "a֮𞄱̀̕b", "a֮𞄱̀̕b", "a֮𞄱̀̕b", "a֮𞄱̀̕b" },
  { "à֮𞄲̕b", "à֮𞄲̕b", "à֮𞄲̕b", "à֮𞄲̕b", "à֮𞄲̕b" },
  { "a֮𞄲̀̕b", "a֮𞄲̀̕b", "a֮𞄲̀̕b", "a֮𞄲̀̕b", "a֮𞄲̀̕b" },
  { "à֮𞄳̕b", "à֮𞄳̕b", "à֮𞄳̕b", "à֮𞄳̕b", "à֮𞄳̕b" },
  { "a֮𞄳̀̕b", "a֮𞄳̀̕b", "a֮𞄳̀̕b", "a֮𞄳̀̕b", "a֮𞄳̀̕b" },
  { "à֮𞄴̕b", "à֮𞄴̕b", "à֮𞄴̕b", "à֮𞄴̕b", "à֮𞄴̕b" },
  { "a֮𞄴̀̕b", "a֮𞄴̀̕b", "a֮𞄴̀̕b", "a֮𞄴̀̕b", "a֮𞄴̀̕b" },
  { "à֮𞄵̕b", "à֮𞄵̕b", "à֮𞄵̕b", "à֮𞄵̕b", "à֮𞄵̕b" },
  { "a֮𞄵̀̕b", "a֮𞄵̀̕b", "a֮𞄵̀̕b", "a֮𞄵̀̕b", "a֮𞄵̀̕b" },
  { "à֮𞄶̕b", "à֮𞄶̕b", "à֮𞄶̕b", "à֮𞄶̕b", "à֮𞄶̕b" },
  { "a֮𞄶̀̕b", "a֮𞄶̀̕b", "a֮𞄶̀̕b", "a֮𞄶̀̕b", "a֮𞄶̀̕b" },
  { "à֮𞋬̕b", "à֮𞋬̕b", "à֮𞋬̕b", "à֮𞋬̕b", "à֮𞋬̕b" },
  { "a֮𞋬̀̕b", "a֮𞋬̀̕b", "a֮𞋬̀̕b", "a֮𞋬̀̕b", "a֮𞋬̀̕b" },
  { "à֮𞋭̕b", "à֮𞋭̕b", "à֮𞋭̕b", "à֮𞋭̕b", "à֮𞋭̕b" },
  { "a֮𞋭̀̕b", "a֮𞋭̀̕b", "a֮𞋭̀̕b", "a֮𞋭̀̕b", "a֮𞋭̀̕b" },
  { "à֮𞋮̕b", "à֮𞋮̕b", "à֮𞋮̕b", "à֮𞋮̕b", "à֮𞋮̕b" },
  { "a֮𞋮̀̕b", "a֮𞋮̀̕b", "a֮𞋮̀̕b", "a֮𞋮̀̕b", "a֮𞋮̀̕b" },
  { "à֮𞋯̕b", "à֮𞋯̕b", "à֮𞋯̕b", "à֮𞋯̕b", "à֮𞋯̕b" },
  { "a֮𞋯̀̕b", "a֮𞋯̀̕b", "a֮𞋯̀̕b", "a֮𞋯̀̕b", "a֮𞋯̀̕b" },
  { "a〪̖𞣐֚b", "a〪̖𞣐֚b", "a〪̖𞣐֚b", "a〪̖𞣐֚b", "a〪̖𞣐֚b" },
  { "a〪𞣐̖֚b", "a〪𞣐̖֚b", "a〪𞣐̖֚b", "a〪𞣐̖֚b", "a〪𞣐̖֚b" },
  { "a〪̖𞣑֚b", "a〪̖𞣑֚b", "a〪̖𞣑֚b", "a〪̖𞣑֚b", "a〪̖𞣑֚b" },
  { "a〪𞣑̖֚b", "a〪𞣑̖֚b", "a〪𞣑̖֚b", "a〪𞣑̖֚b", "a〪𞣑̖֚b" },
  { "a〪̖𞣒֚b", "a〪̖𞣒֚b", "a〪̖𞣒֚b", "a〪̖𞣒֚b", "a〪̖𞣒֚b" },
  { "a〪𞣒̖֚b", "a〪𞣒̖֚b", "a〪𞣒̖֚b", "a〪𞣒̖֚b", "a〪𞣒̖֚b" },
  { "a〪̖𞣓֚b", "a〪̖𞣓֚b", "a〪̖𞣓֚b", "a〪̖𞣓֚b", "a〪̖𞣓֚b" },
  { "a〪𞣓̖֚b", "a〪𞣓̖֚b", "a〪𞣓̖֚b", "a〪𞣓̖֚b", "a〪𞣓̖֚b" },
  { "a〪̖𞣔֚b", "a〪̖𞣔֚b", "a〪̖𞣔֚b", "a〪̖𞣔֚b", "a〪̖𞣔֚b" },
  { "a〪𞣔̖֚b", "a〪𞣔̖֚b", "a〪𞣔̖֚b", "a〪𞣔̖֚b", "a〪𞣔̖֚b" },
  { "a〪̖𞣕֚b", "a〪̖𞣕֚b", "a〪̖𞣕֚b", "a〪̖𞣕֚b", "a〪̖𞣕֚b" },
  { "a〪𞣕̖֚b", "a〪𞣕̖֚b", "a〪𞣕̖֚b", "a〪𞣕̖֚b", "a〪𞣕̖֚b" },
  { "a〪̖𞣖֚b", "a〪̖𞣖֚b", "a〪̖𞣖֚b", "a〪̖𞣖֚b", "a〪̖𞣖֚b" },
  { "a〪𞣖̖֚b", "a〪𞣖̖֚b", "a〪𞣖̖֚b", "a〪𞣖̖֚b", "a〪𞣖̖֚b" },
  { "à֮𞥄̕b", "à֮𞥄̕b", "à֮𞥄̕b", "à֮𞥄̕b", "à֮𞥄̕b" },
  { "a֮𞥄̀̕b", "a֮𞥄̀̕b", "a֮𞥄̀̕b", "a֮𞥄̀̕b", "a֮𞥄̀̕b" },
  { "à֮𞥅̕b", "à֮𞥅̕b", "à֮𞥅̕b", "à֮𞥅̕b", "à֮𞥅̕b" },
  { "a֮𞥅̀̕b", "a֮𞥅̀̕b", "a֮𞥅̀̕b", "a֮𞥅̀̕b", "a֮𞥅̀̕b" },
  { "à֮𞥆̕b", "à֮𞥆̕b", "à֮𞥆̕b", "à֮𞥆̕b", "à֮𞥆̕b" },
  { "a֮𞥆̀̕b", "a֮𞥆̀̕b", "a֮𞥆̀̕b", "a֮𞥆̀̕b", "a֮𞥆̀̕b" },
  { "à֮𞥇̕b", "à֮𞥇̕b", "à֮𞥇̕b", "à֮𞥇̕b", "à֮𞥇̕b" },
  { "a֮𞥇̀̕b", "a֮𞥇̀̕b", "a֮𞥇̀̕b", "a֮𞥇̀̕b", "a֮𞥇̀̕b" },
  { "à֮𞥈̕b", "à֮𞥈̕b", "à֮𞥈̕b", "à֮𞥈̕b", "à֮𞥈̕b" },
  { "a֮𞥈̀̕b", "a֮𞥈̀̕b", "a֮𞥈̀̕b", "a֮𞥈̀̕b", "a֮𞥈̀̕b" },
  { "à֮𞥉̕b", "à֮𞥉̕b", "à֮𞥉̕b", "à֮𞥉̕b", "à֮𞥉̕b" },
  { "a֮𞥉̀̕b", "a֮𞥉̀̕b", "a֮𞥉̀̕b", "a֮𞥉̀̕b", "a֮𞥉̀̕b" },
  { "a̴़𞥊゙b", "a̴़𞥊゙b", "a̴़𞥊゙b", "a̴़𞥊゙b", "a̴़𞥊゙b" },
  { "a̴𞥊़゙b", "a̴𞥊़゙b", "a̴𞥊़゙b", "a̴𞥊़゙b", "a̴𞥊़゙b" },
  { "ে̴া", "ে̴া", "ে̴া", "ে̴া", "ে̴া" },
  { "ে̴ৗ", "ে̴ৗ", "ে̴ৗ", "ে̴ৗ", "ে̴ৗ" },
  { "େ̴ା", "େ̴ା", "େ̴ା", "େ̴ା", "େ̴ା" },
  { "େ̴ୖ", "େ̴ୖ", "େ̴ୖ", "େ̴ୖ", "େ̴ୖ" },
  { "େ̴ୗ", "େ̴ୗ", "େ̴ୗ", "େ̴ୗ", "େ̴ୗ" },
  { "ஒ̴ௗ", "ஒ̴ௗ", "ஒ̴ௗ", "ஒ̴ௗ", "ஒ̴ௗ" },
  { "ெ̴ா", "ெ̴ா", "ெ̴ா", "ெ̴ா", "ெ̴ா" },
  { "ெ̴ௗ", "ெ̴ௗ", "ெ̴ௗ", "ெ̴ௗ", "ெ̴ௗ" },
  { "ே̴ா", "ே̴ா", "ே̴ா", "ே̴ா", "ே̴ா" },
  { "ಿ̴ೕ", "ಿ̴ೕ", "ಿ̴ೕ", "ಿ̴ೕ", "ಿ̴ೕ" },
  { "ೆ̴ೂ", "ೆ̴ೂ", "ೆ̴ೂ", "ೆ̴ೂ", "ೆ̴ೂ" },
  { "ೆ̴ೕ", "ೆ̴ೕ", "ೆ̴ೕ", "ೆ̴ೕ", "ೆ̴ೕ" },
  { "ೆ̴ೖ", "ೆ̴ೖ", "ೆ̴ೖ", "ೆ̴ೖ", "ೆ̴ೖ" },
  { "ೊ̴ೕ", "ೊ̴ೕ", "ೊ̴ೕ", "ೊ̴ೕ", "ೊ̴ೕ" },
  { "െ̴ാ", "െ̴ാ", "െ̴ാ", "െ̴ാ", "െ̴ാ" },
  { "െ̴ൗ", "െ̴ൗ", "െ̴ൗ", "െ̴ൗ", "െ̴ൗ" },
  { "േ̴ാ", "േ̴ാ", "േ̴ാ", "േ̴ാ", "േ̴ാ" },
  { "ෙ̴ා", "ෙ̴ා", "ෙ̴ා", "ෙ̴ා", "ෙ̴ා" },
  { "ෙ̴ෟ", "ෙ̴ෟ", "ෙ̴ෟ", "ෙ̴ෟ", "ෙ̴ෟ" },
  { "ཀ̴ྵ", "ཀ̴ྵ", "ཀ̴ྵ", "ཀ̴ྵ", "ཀ̴ྵ" },
  { "ག̴ྷ", "ག̴ྷ", "ག̴ྷ", "ག̴ྷ", "ག̴ྷ" },
  { "ཌ̴ྷ", "ཌ̴ྷ", "ཌ̴ྷ", "ཌ̴ྷ", "ཌ̴ྷ" },
  { "ད̴ྷ", "ད̴ྷ", "ད̴ྷ", "ད̴ྷ", "ད̴ྷ" },
  { "བ̴ྷ", "བ̴ྷ", "བ̴ྷ", "བ̴ྷ", "བ̴ྷ" },
  { "ཛ̴ྷ", "ཛ̴ྷ", "ཛ̴ྷ", "ཛ̴ྷ", "ཛ̴ྷ" },
  { "ྐ̴ྵ", "ྐ̴ྵ", "ྐ̴ྵ", "ྐ̴ྵ", "ྐ̴ྵ" },
  { "ྒ̴ྷ", "ྒ̴ྷ", "ྒ̴ྷ", "ྒ̴ྷ", "ྒ̴ྷ" },
  { "ྜ̴ྷ", "ྜ̴ྷ", "ྜ̴ྷ", "ྜ̴ྷ", "ྜ̴ྷ" },
  { "ྡ̴ྷ", "ྡ̴ྷ", "ྡ̴ྷ", "ྡ̴ྷ", "ྡ̴ྷ" },
  { "ྦ̴ྷ", "ྦ̴ྷ", "ྦ̴ྷ", "ྦ̴ྷ", "ྦ̴ྷ" },
  { "ྫ̴ྷ", "ྫ̴ྷ", "ྫ̴ྷ", "ྫ̴ྷ", "ྫ̴ྷ" },
  { "ဥ̴ီ", "ဥ̴ီ", "ဥ̴ီ", "ဥ̴ီ", "ဥ̴ီ" },
  { "ᄀ̴ᅡ", "ᄀ̴ᅡ", "ᄀ̴ᅡ", "ᄀ̴ᅡ", "ᄀ̴ᅡ" },
  { "ᄀ̴ᅮ", "ᄀ̴ᅮ", "ᄀ̴ᅮ", "ᄀ̴ᅮ", "ᄀ̴ᅮ" },
  { "ᄁ̴ᅦ", "ᄁ̴ᅦ", "ᄁ̴ᅦ", "ᄁ̴ᅦ", "ᄁ̴ᅦ" },
  { "ᄁ̴ᅳ", "ᄁ̴ᅳ", "ᄁ̴ᅳ", "ᄁ̴ᅳ", "ᄁ̴ᅳ" },
  { "ᄂ̴ᅫ", "ᄂ̴ᅫ", "ᄂ̴ᅫ", "ᄂ̴ᅫ", "ᄂ̴ᅫ" },
  { "ᄃ̴ᅣ", "ᄃ̴ᅣ", "ᄃ̴ᅣ", "ᄃ̴ᅣ", "ᄃ̴ᅣ" },
  { "ᄃ̴ᅰ", "ᄃ̴ᅰ", "ᄃ̴ᅰ", "ᄃ̴ᅰ", "ᄃ̴ᅰ" },
  { "ᄄ̴ᅨ", "ᄄ̴ᅨ", "ᄄ̴ᅨ", "ᄄ̴ᅨ", "ᄄ̴ᅨ" },
  { "ᄄ̴ᅵ", "ᄄ̴ᅵ", "ᄄ̴ᅵ", "ᄄ̴ᅵ", "ᄄ̴ᅵ" },
  { "ᄅ̴ᅭ", "ᄅ̴ᅭ", "ᄅ̴ᅭ", "ᄅ̴ᅭ", "ᄅ̴ᅭ" },
  { "ᄆ̴ᅥ", "ᄆ̴ᅥ", "ᄆ̴ᅥ", "ᄆ̴ᅥ", "ᄆ̴ᅥ" },
  { "ᄆ̴ᅲ", "ᄆ̴ᅲ", "ᄆ̴ᅲ", "ᄆ̴ᅲ", "ᄆ̴ᅲ" },
  { "ᄇ̴ᅪ", "ᄇ̴ᅪ", "ᄇ̴ᅪ", "ᄇ̴ᅪ", "ᄇ̴ᅪ" },
  { "ᄈ̴ᅢ", "ᄈ̴ᅢ", "ᄈ̴ᅢ", "ᄈ̴ᅢ", "ᄈ̴ᅢ" },
  { "ᄈ̴ᅯ", "ᄈ̴ᅯ", "ᄈ̴ᅯ", "ᄈ̴ᅯ", "ᄈ̴ᅯ" },
  { "ᄉ̴ᅧ", "ᄉ̴ᅧ", "ᄉ̴ᅧ", "ᄉ̴ᅧ", "ᄉ̴ᅧ" },
  { "ᄉ̴ᅴ", "ᄉ̴ᅴ", "ᄉ̴ᅴ", "ᄉ̴ᅴ", "ᄉ̴ᅴ" },
  { "ᄊ̴ᅬ", "ᄊ̴ᅬ", "ᄊ̴ᅬ", "ᄊ̴ᅬ", "ᄊ̴ᅬ" },
  { "ᄋ̴ᅤ", "ᄋ̴ᅤ", "ᄋ̴ᅤ", "ᄋ̴ᅤ", "ᄋ̴ᅤ" },
  { "ᄋ̴ᅱ", "ᄋ̴ᅱ", "ᄋ̴ᅱ", "ᄋ̴ᅱ", "ᄋ̴ᅱ" },
  { "ᄌ̴ᅩ", "ᄌ̴ᅩ", "ᄌ̴ᅩ", "ᄌ̴ᅩ", "ᄌ̴ᅩ" },
  { "ᄍ̴ᅡ", "ᄍ̴ᅡ", "ᄍ̴ᅡ", "ᄍ̴ᅡ", "ᄍ̴ᅡ" },
  { "ᄍ̴ᅮ", "ᄍ̴ᅮ", "ᄍ̴ᅮ", "ᄍ̴ᅮ", "ᄍ̴ᅮ" },
  { "ᄎ̴ᅦ", "ᄎ̴ᅦ", "ᄎ̴ᅦ", "ᄎ̴ᅦ", "ᄎ̴ᅦ" },
  { "ᄎ̴ᅳ", "ᄎ̴ᅳ", "ᄎ̴ᅳ", "ᄎ̴ᅳ", "ᄎ̴ᅳ" },
  { "ᄏ̴ᅫ", "ᄏ̴ᅫ", "ᄏ̴ᅫ", "ᄏ̴ᅫ", "ᄏ̴ᅫ" },
  { "ᄐ̴ᅣ", "ᄐ̴ᅣ", "ᄐ̴ᅣ", "ᄐ̴ᅣ", "ᄐ̴ᅣ" },
  { "ᄐ̴ᅰ", "ᄐ̴ᅰ", "ᄐ̴ᅰ", "ᄐ̴ᅰ", "ᄐ̴ᅰ" },
  { "ᄑ̴ᅨ", "ᄑ̴ᅨ", "ᄑ̴ᅨ", "ᄑ̴ᅨ", "ᄑ̴ᅨ" },
  { "ᄑ̴ᅵ", "ᄑ̴ᅵ", "ᄑ̴ᅵ", "ᄑ̴ᅵ", "ᄑ̴ᅵ" },
  { "ᄒ̴ᅭ", "ᄒ̴ᅭ", "ᄒ̴ᅭ", "ᄒ̴ᅭ", "ᄒ̴ᅭ" },
  { "ᬅ̴ᬵ", "ᬅ̴ᬵ", "ᬅ̴ᬵ", "ᬅ̴ᬵ", "ᬅ̴ᬵ" },
  { "ᬇ̴ᬵ", "ᬇ̴ᬵ", "ᬇ̴ᬵ", "ᬇ̴ᬵ", "ᬇ̴ᬵ" },
  { "ᬉ̴ᬵ", "ᬉ̴ᬵ", "ᬉ̴ᬵ", "ᬉ̴ᬵ", "ᬉ̴ᬵ" },
  { "ᬋ̴ᬵ", "ᬋ̴ᬵ", "ᬋ̴ᬵ", "ᬋ̴ᬵ", "ᬋ̴ᬵ" },
  { "ᬍ̴ᬵ", "ᬍ̴ᬵ", "ᬍ̴ᬵ", "ᬍ̴ᬵ", "ᬍ̴ᬵ" },
  { "ᬑ̴ᬵ", "ᬑ̴ᬵ", "ᬑ̴ᬵ", "ᬑ̴ᬵ", "ᬑ̴ᬵ" },
  { "ᬺ̴ᬵ", "ᬺ̴ᬵ", "ᬺ̴ᬵ", "ᬺ̴ᬵ", "ᬺ̴ᬵ" },
  { "ᬼ̴ᬵ", "ᬼ̴ᬵ", "ᬼ̴ᬵ", "ᬼ̴ᬵ", "ᬼ̴ᬵ" },
  { "ᬾ̴ᬵ", "ᬾ̴ᬵ", "ᬾ̴ᬵ", "ᬾ̴ᬵ", "ᬾ̴ᬵ" },
  { "ᬿ̴ᬵ", "ᬿ̴ᬵ", "ᬿ̴ᬵ", "ᬿ̴ᬵ", "ᬿ̴ᬵ" },
  { "ᭂ̴ᬵ", "ᭂ̴ᬵ", "ᭂ̴ᬵ", "ᭂ̴ᬵ", "ᭂ̴ᬵ" },
  { "걔̴ᆮ", "걔̴ᆮ", "걔̴ᆮ", "걔̴ᆮ", "걔̴ᆮ" },
  { "겨̴ᆵ", "겨̴ᆵ", "겨̴ᆵ", "겨̴ᆵ", "겨̴ᆵ" },
  { "과̴ᆼ", "과̴ᆼ", "과̴ᆼ", "과̴ᆼ", "과̴ᆼ" },
  { "귀̴ᆮ", "귀̴ᆮ", "귀̴ᆮ", "귀̴ᆮ", "귀̴ᆮ" },
  { "긔̴ᆵ", "긔̴ᆵ", "긔̴ᆵ", "긔̴ᆵ", "긔̴ᆵ" },
  { "깨̴ᆼ", "깨̴ᆼ", "깨̴ᆼ", "깨̴ᆼ", "깨̴ᆼ" },
  { "꼬̴ᆮ", "꼬̴ᆮ", "꼬̴ᆮ", "꼬̴ᆮ", "꼬̴ᆮ" },
  { "꾀̴ᆵ", "꾀̴ᆵ", "꾀̴ᆵ", "꾀̴ᆵ", "꾀̴ᆵ" },
  { "꿔̴ᆼ", "꿔̴ᆼ", "꿔̴ᆼ", "꿔̴ᆼ", "꿔̴ᆼ" },
  { "나̴ᆮ", "나̴ᆮ", "나̴ᆮ", "나̴ᆮ", "나̴ᆮ" },
  { "냬̴ᆵ", "냬̴ᆵ", "냬̴ᆵ", "냬̴ᆵ", "냬̴ᆵ" },
  { "녀̴ᆼ", "녀̴ᆼ", "녀̴ᆼ", "녀̴ᆼ", "녀̴ᆼ" },
  { "누̴ᆮ", "누̴ᆮ", "누̴ᆮ", "누̴ᆮ", "누̴ᆮ" },
  { "뉘̴ᆵ", "뉘̴ᆵ", "뉘̴ᆵ", "뉘̴ᆵ", "뉘̴ᆵ" },
  { "늬̴ᆼ", "늬̴ᆼ", "늬̴ᆼ", "늬̴ᆼ", "늬̴ᆼ" },
  { "데̴ᆮ", "데̴ᆮ", "데̴ᆮ", "데̴ᆮ", "데̴ᆮ" },
  { "도̴ᆵ", "도̴ᆵ", "도̴ᆵ", "도̴ᆵ", "도̴ᆵ" },
  { "되̴ᆼ", "되̴ᆼ", "되̴ᆼ", "되̴ᆼ", "되̴ᆼ" },
  { "드̴ᆮ", "드̴ᆮ", "드̴ᆮ", "드̴ᆮ", "드̴ᆮ" },
  { "따̴ᆵ", "따̴ᆵ", "따̴ᆵ", "따̴ᆵ", "따̴ᆵ" },
  { "떄̴ᆼ", "떄̴ᆼ", "떄̴ᆼ", "떄̴ᆼ", "떄̴ᆼ" },
  { "뙈̴ᆮ", "뙈̴ᆮ", "뙈̴ᆮ", "뙈̴ᆮ", "뙈̴ᆮ" },
  { "뚜̴ᆵ", "뚜̴ᆵ", "뚜̴ᆵ", "뚜̴ᆵ", "뚜̴ᆵ" },
  { "뛰̴ᆼ", "뛰̴ᆼ", "뛰̴ᆼ", "뛰̴ᆼ", "뛰̴ᆼ" },
  { "랴̴ᆮ", "랴̴ᆮ", "랴̴ᆮ", "랴̴ᆮ", "랴̴ᆮ" },
  { "레̴ᆵ", "레̴ᆵ", "레̴ᆵ", "레̴ᆵ", "레̴ᆵ" },
  { "로̴ᆼ", "로̴ᆼ", "로̴ᆼ", "로̴ᆼ", "로̴ᆼ" },
  { "뤠̴ᆮ", "뤠̴ᆮ", "뤠̴ᆮ", "뤠̴ᆮ", "뤠̴ᆮ" },
  { "르̴ᆵ", "르̴ᆵ", "르̴ᆵ", "르̴ᆵ", "르̴ᆵ" },
  { "마̴ᆼ", "마̴ᆼ", "마̴ᆼ", "마̴ᆼ", "마̴ᆼ" },
  { "몌̴ᆮ", "몌̴ᆮ", "몌̴ᆮ", "몌̴ᆮ", "몌̴ᆮ" },
  { "뫠̴ᆵ", "뫠̴ᆵ", "뫠̴ᆵ", "뫠̴ᆵ", "뫠̴ᆵ" },
  { "무̴ᆼ", "무̴ᆼ", "무̴ᆼ", "무̴ᆼ", "무̴ᆼ" },
  { "미̴ᆮ", "미̴ᆮ", "미̴ᆮ", "미̴ᆮ", "미̴ᆮ" },
  { "뱌̴ᆵ", "뱌̴ᆵ", "뱌̴ᆵ", "뱌̴ᆵ", "뱌̴ᆵ" },
  { "베̴ᆼ", "베̴ᆼ", "베̴ᆼ", "베̴ᆼ", "베̴ᆼ" },
  { "뵤̴ᆮ", "뵤̴ᆮ", "뵤̴ᆮ", "뵤̴ᆮ", "뵤̴ᆮ" },
  { "붸̴ᆵ", "붸̴ᆵ", "붸̴ᆵ", "붸̴ᆵ", "붸̴ᆵ" },
  { "브̴ᆼ", "브̴ᆼ", "브̴ᆼ", "브̴ᆼ", "브̴ᆼ" },
  { "뻐̴ᆮ", "뻐̴ᆮ", "뻐̴ᆮ", "뻐̴ᆮ", "뻐̴ᆮ" },
  { "뼤̴ᆵ", "뼤̴ᆵ", "뼤̴ᆵ", "뼤̴ᆵ", "뼤̴ᆵ" },
  { "뽸̴ᆼ", "뽸̴ᆼ", "뽸̴ᆼ", "뽸̴ᆼ", "뽸̴ᆼ" },
  { "쀼̴ᆮ", "쀼̴ᆮ", "쀼̴ᆮ", "쀼̴ᆮ", "쀼̴ᆮ" },
  { "삐̴ᆵ", "삐̴ᆵ", "삐̴ᆵ", "삐̴ᆵ", "삐̴ᆵ" },
  { "샤̴ᆼ", "샤̴ᆼ", "샤̴ᆼ", "샤̴ᆼ", "샤̴ᆼ" },
  { "솨̴ᆮ", "솨̴ᆮ", "솨̴ᆮ", "솨̴ᆮ", "솨̴ᆮ" },
  { "쇼̴ᆵ", "쇼̴ᆵ", "쇼̴ᆵ", "쇼̴ᆵ", "쇼̴ᆵ" },
  { "쉐̴ᆼ", "쉐̴ᆼ", "쉐̴ᆼ", "쉐̴ᆼ", "쉐̴ᆼ" },
  { "쌔̴ᆮ", "쌔̴ᆮ", "쌔̴ᆮ", "쌔̴ᆮ", "쌔̴ᆮ" },
  { "써̴ᆵ", "써̴ᆵ", "써̴ᆵ", "써̴ᆵ", "써̴ᆵ" },
  { "쎼̴ᆼ", "쎼̴ᆼ", "쎼̴ᆼ", "쎼̴ᆼ", "쎼̴ᆼ" },
  { "쒀̴ᆮ", "쒀̴ᆮ", "쒀̴ᆮ", "쒀̴ᆮ", "쒀̴ᆮ" },
  { "쓔̴ᆵ", "쓔̴ᆵ", "쓔̴ᆵ", "쓔̴ᆵ", "쓔̴ᆵ" },
  { "씨̴ᆼ", "씨̴ᆼ", "씨̴ᆼ", "씨̴ᆼ", "씨̴ᆼ" },
  { "여̴ᆮ", "여̴ᆮ", "여̴ᆮ", "여̴ᆮ", "여̴ᆮ" },
  { "와̴ᆵ", "와̴ᆵ", "와̴ᆵ", "와̴ᆵ", "와̴ᆵ" },
  { "요̴ᆼ", "요̴ᆼ", "요̴ᆼ", "요̴ᆼ", "요̴ᆼ" },
  { "의̴ᆮ", "의̴ᆮ", "의̴ᆮ", "의̴ᆮ", "의̴ᆮ" },
  { "재̴ᆵ", "재̴ᆵ", "재̴ᆵ", "재̴ᆵ", "재̴ᆵ" },
  { "저̴ᆼ", "저̴ᆼ", "저̴ᆼ", "저̴ᆼ", "저̴ᆼ" },
  { "죄̴ᆮ", "죄̴ᆮ", "죄̴ᆮ", "죄̴ᆮ", "죄̴ᆮ" },
  { "줘̴ᆵ", "줘̴ᆵ", "줘̴ᆵ", "줘̴ᆵ", "줘̴ᆵ" },
  { "쥬̴ᆼ", "쥬̴ᆼ", "쥬̴ᆼ", "쥬̴ᆼ", "쥬̴ᆼ" },
  { "쨰̴ᆮ", "쨰̴ᆮ", "쨰̴ᆮ", "쨰̴ᆮ", "쨰̴ᆮ" },
  { "쪄̴ᆵ", "쪄̴ᆵ", "쪄̴ᆵ", "쪄̴ᆵ", "쪄̴ᆵ" },
  { "쫘̴ᆼ", "쫘̴ᆼ", "쫘̴ᆼ", "쫘̴ᆼ", "쫘̴ᆼ" },
  { "쮜̴ᆮ", "쮜̴ᆮ", "쮜̴ᆮ", "쮜̴ᆮ", "쮜̴ᆮ" },
  { "쯰̴ᆵ", "쯰̴ᆵ", "쯰̴ᆵ", "쯰̴ᆵ", "쯰̴ᆵ" },
  { "채̴ᆼ", "채̴ᆼ", "채̴ᆼ", "채̴ᆼ", "채̴ᆼ" },
  { "초̴ᆮ", "초̴ᆮ", "초̴ᆮ", "초̴ᆮ", "초̴ᆮ" },
  { "최̴ᆵ", "최̴ᆵ", "최̴ᆵ", "최̴ᆵ", "최̴ᆵ" },
  { "춰̴ᆼ", "춰̴ᆼ", "춰̴ᆼ", "춰̴ᆼ", "춰̴ᆼ" },
  { "카̴ᆮ", "카̴ᆮ", "카̴ᆮ", "카̴ᆮ", "카̴ᆮ" },
  { "컈̴ᆵ", "컈̴ᆵ", "컈̴ᆵ", "컈̴ᆵ", "컈̴ᆵ" },
  { "켜̴ᆼ", "켜̴ᆼ", "켜̴ᆼ", "켜̴ᆼ", "켜̴ᆼ" },
  { "쿠̴ᆮ", "쿠̴ᆮ", "쿠̴ᆮ", "쿠̴ᆮ", "쿠̴ᆮ" },
  { "퀴̴ᆵ", "퀴̴ᆵ", "퀴̴ᆵ", "퀴̴ᆵ", "퀴̴ᆵ" },
  { "킈̴ᆼ", "킈̴ᆼ", "킈̴ᆼ", "킈̴ᆼ", "킈̴ᆼ" },
  { "테̴ᆮ", "테̴ᆮ", "테̴ᆮ", "테̴ᆮ", "테̴ᆮ" },
  { "토̴ᆵ", "토̴ᆵ", "토̴ᆵ", "토̴ᆵ", "토̴ᆵ" },
  { "퇴̴ᆼ", "퇴̴ᆼ", "퇴̴ᆼ", "퇴̴ᆼ", "퇴̴ᆼ" },
  { "트̴ᆮ", "트̴ᆮ", "트̴ᆮ", "트̴ᆮ", "트̴ᆮ" },
  { "파̴ᆵ", "파̴ᆵ", "파̴ᆵ", "파̴ᆵ", "파̴ᆵ" },
  { "퍠̴ᆼ", "퍠̴ᆼ", "퍠̴ᆼ", "퍠̴ᆼ", "퍠̴ᆼ" },
  { "퐤̴ᆮ", "퐤̴ᆮ", "퐤̴ᆮ", "퐤̴ᆮ", "퐤̴ᆮ" },
  { "푸̴ᆵ", "푸̴ᆵ", "푸̴ᆵ", "푸̴ᆵ", "푸̴ᆵ" },
  { "퓌̴ᆼ", "퓌̴ᆼ", "퓌̴ᆼ", "퓌̴ᆼ", "퓌̴ᆼ" },
  { "햐̴ᆮ", "햐̴ᆮ", "햐̴ᆮ", "햐̴ᆮ", "햐̴ᆮ" },
  { "헤̴ᆵ", "헤̴ᆵ", "헤̴ᆵ", "헤̴ᆵ", "헤̴ᆵ" },
  { "호̴ᆼ", "호̴ᆼ", "호̴ᆼ", "호̴ᆼ", "호̴ᆼ" },
  { "훼̴ᆮ", "훼̴ᆮ", "훼̴ᆮ", "훼̴ᆮ", "훼̴ᆮ" },
  { "흐̴ᆵ", "흐̴ᆵ", "흐̴ᆵ", "흐̴ᆵ", "흐̴ᆵ" },
  { "𑄱̴𑄧", "𑄱̴𑄧", "𑄱̴𑄧", "𑄱̴𑄧", "𑄱̴𑄧" },
  { "𑄲̴𑄧", "𑄲̴𑄧", "𑄲̴𑄧", "𑄲̴𑄧", "𑄲̴𑄧" },
  { "𑍇̴𑌾", "𑍇̴𑌾", "𑍇̴𑌾", "𑍇̴𑌾", "𑍇̴𑌾" },
  { "𑍇̴𑍗", "𑍇̴𑍗", "𑍇̴𑍗", "𑍇̴𑍗", "𑍇̴𑍗" },
  { "𑒹̴𑒰", "𑒹̴𑒰", "𑒹̴𑒰", "𑒹̴𑒰", "𑒹̴𑒰" },
  { "𑒹̴𑒺", "𑒹̴𑒺", "𑒹̴𑒺", "𑒹̴𑒺", "𑒹̴𑒺" },
  { "𑒹̴𑒽", "𑒹̴𑒽", "𑒹̴𑒽", "𑒹̴𑒽", "𑒹̴𑒽" },
  { "𑖸̴𑖯", "𑖸̴𑖯", "𑖸̴𑖯", "𑖸̴𑖯", "𑖸̴𑖯" },
  { "𑖹̴𑖯", "𑖹̴𑖯", "𑖹̴𑖯", "𑖹̴𑖯", "𑖹̴𑖯" },
};


INSTANTIATE_TEST_CASE_P( UnicodeTest, NormalizationTest, ValuesIn( tests ) );

} // namespace YouCompleteMe
