/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  mime_start.c

  (c) 2005,2006,2008 Bernhard R. Link

  call edit/view commands from mime types
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <errno.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <stdbool.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <X11/Intrinsic.h>
#include "global.h"
#include "Fm.h"
#include "Am.h"
#include "mime.h"
#include "mime_start.h"
#include "execute.h"

static char symlinknumber[11] = "XXXXXXXXXX";
static char *symlinkdirectory = NULL;

static char *concat(const char *directory,const char *name) {
	size_t dlen,nlen;
	char *r;

	dlen = (directory!=NULL)?strlen(directory):0;
	nlen = strlen(name);
	if( dlen == 0 )
		return strdup(name);
	if( dlen > 0 && directory[dlen-1] == '/' )
		dlen--;
	r = malloc(dlen+nlen+2);
	if( r == NULL )
		return r;
	memcpy(r,directory,dlen);
	r[dlen] = '/';
	memcpy(r+dlen+1,name,nlen+1);
	return r;
}

bool mime_start_init(const char *symlinkdir) {
	int r;

	strcpy(symlinknumber,"0000000000");
	symlinkdirectory = strdup(symlinkdir);
	if( symlinkdirectory == NULL ) {
		abortXfm("Out of Memory");
		return false;
	}
		
	/* what is not yet checked and therefore is the
	 * task of the user, is to make sure noone can
	 * hijack the directory by deleting it or moving
	 * it out of the way. I do not know if it is worthwile
	 * to check this (i.e. are all parent directories
	 * owned by us, or by root, and are sticky when they
	 * are writeable by all.) here, as at least group
	 * writeability should then also be warned but give
	 * false positives if it is a user-specific group,
	 * so just not set your resources wrong and LART your
	 * sysadm if /tmp is not sticky */
	r = mkdir(symlinkdir,0700);
	if( r != 0 ) {
		int e = errno;
		if( e == EEXIST ) {
			/* check if this is a directory and save */
			struct stat s;
			r = lstat(symlinkdir,&s);
			if( r == 0 && S_ISDIR(s.st_mode) ) {
				/* a directory it is */
				if( s.st_uid != getuid() ) {
					fprintf(stderr,"start directory %s is owned by someone else!\n",symlinkdir);
					return false;
				}
				r = chmod(symlinkdir,0700);
				if( r != 0 ) {
					e = errno;
					fprintf(stderr,"Error setting rights for %s: %d=%s\n",symlinkdir,e,strerror(e));
					return false;
				}
				return true;
			}
		}
		fprintf(stderr,"Error creating %s: %d=%s\n",symlinkdir,e,strerror(e));
		return false;
	} else
		return true;
}
void mime_start_done(const char *symlinkdir) {
	if( 0 != rmdir(symlinkdir) ) {
		int e = errno;
		fprintf(stderr,"while removing '%s': %d=%s\n",symlinkdir,e,strerror(e));
	}
}

static void program_finished(UNUSED(int status),void *data) {
	char *linkname = data;
	if( linkname != NULL) {
		unlink(linkname);
		free(linkname);
	}
}


static void nextsymlinkname(void) {
	int i = 9;

	assert( symlinknumber[i] != '\0' && symlinknumber[i+1] == '\0' );
	while( symlinknumber[i] == 'z'  ) {
		symlinknumber[i] = '0';
		i--;
		if( i < 0 ) 
			return;
	}
	if( symlinknumber[i] == '9' )
		symlinknumber[i] = 'a';
	else
		symlinknumber[i]++;
}

static char *unescape_insert(const char *template,const char *name) {
	bool inserted = false;
	char *r,*n;
	const char *t;
	size_t t_len,n_len;
	bool escaped = false;
	char c;

	if( template == NULL )
		return NULL;
	t_len = strlen(template);
	n_len = strlen(name);
	r = XtMalloc(t_len+n_len+1);
	n = r; t = template;
	while( (c=*(t++)) != '\0' ) {
		if( escaped ) {
			*(n++) = c;
			escaped = false;
		} else {
			if( c == '\\')
				escaped = true;
			else if( c != '%' || *t != 's' ) {
				*(n++) = c;
			} else if( inserted ) {
				fprintf(stderr,"Malformed nametemplate with two %%s: '%s'!\n",template);
				free(r);
				return NULL;
			} else {
				memcpy(n,name,n_len);
				n+=n_len;
				inserted = true;
				/* over 's', too */
				t++;
			}
		}
	}
	*(n++) = '\0';
	assert( (size_t)(n-r)  <= n_len + t_len+1 );
	if( !inserted ) {
		fprintf(stderr,"Malformed nametemplate without %%s: '%s'!\n",template);
		/* do not return a constant string */
		free(r);
		return NULL;
	}
	return r;
}

/* Test if a command template has %s within "'"s, so more characters can be allowed */
static bool isUnescaped(const char *template) {
	const char *t;
	bool escaped = false, singlequoted = false;
	char c;

	t = template;
	while( (c=*(t++)) != '\0' ) {
		if( escaped ) {
			escaped = false;
		} else if( c == '\\')
			escaped = true;
		else if( c == '\'' )
			singlequoted = !singlequoted;
		else if( c == '%' && *t == 's' ) {
			if( !singlequoted )
				return true;
		}
	}
	return false;
}

static char *calc_linkname(const char *prefix, const char *suffix) {
	size_t dlen,plen,nlen,slen;
	char *n;

	dlen = strlen(symlinkdirectory);
	if( dlen > 0 && symlinkdirectory[dlen-1] == '/' )
		dlen--;
	plen = prefix?strlen(prefix):0;
	nlen = strlen(symlinknumber);
	slen = suffix?strlen(suffix):0;

	n = malloc(dlen+1+plen+nlen+slen+1);
	if( n == NULL )
		return NULL;

	memcpy(n, symlinkdirectory, dlen);
	n[dlen] = '/';
	if( plen > 0 )
		memcpy(n+dlen+1, prefix, plen);
	memcpy(n+dlen+1+plen, symlinknumber, nlen);
	if( slen > 0 )
		memcpy(n+dlen+1+plen+nlen, suffix, slen);
	n[dlen+1+plen+nlen+slen] = '\0';
	return n;
}

static char *allocateLink(const char *filename,const char *prefix, const char *suffix) {
	while(1) {
		char *linkname;
		int err;

		nextsymlinkname();
		linkname = calc_linkname(prefix,suffix);
		if( linkname == NULL ) {
			abortXfm("Out of Memory");
			return NULL;
		}

		err = symlink(filename,linkname);
		if( err == 0 ) {
			//TODO: check here if the link
			//is the correct one in case of 
			//NFS-race conditions?
			return linkname;
		}
		if( err != EEXIST ) {
			int e = errno;
			fprintf(stderr,"Error creating symlink %s->%s: %d=%s\n",
					linkname,filename,e,strerror(e));
			return NULL;
		}
	}
}

/* check for filenames too dangerous to give even to
 * normal programs */
static bool veryDangerous(const char *filename) {
	/* check if it could be confused with an option,
	 * this is not dangerous by itself, but too
	 * many programs are ill written */
	if( filename[0] == '-' )
		return true;
	/* otherwise it is save */
	return false;
}
/* check for filenames too dangerous to give enclosed
 * in '' to a shell. */
static bool couldDangerous(const char *filename) {
	unsigned char c;
	while( (c=*filename) != '\0' ) {
		if( c == '\'' )
			return true;
		if( c == '\\' )
			return true;
		if( c == '!' )
			return true;
		filename++;
	}
	return false;
}
/* check for filenames too dangerous to give
 * to a shell unenclosed and unescaped */
static bool shellDangerous(const char *filename) {
	unsigned char c;

	while( (c=*filename) != '\0' ) {
		/* this is the other way around:
		 * only allow save characters */
		if( ( c < 'a' || c > 'z' ) 
			&& ( c < 'A' || c > 'Z' )
			&& ( c < '0' || c > '9' )
			&& ( c != '/' ) && ( c != '.' )
			&& ( c != '_' ) && ( c != '-' ) 
			&& ( c != ',' ) && ( c != ':' ))
			return true;
		filename++;
	}
	return false;
}

static size_t getArgCount(const char **l) {
	size_t c = 0;
	assert( l != NULL );
	while( *l != NULL ) {
		l++;
		c--;
	}
	return c;
}

static bool match(const char *fn, const char *prefix, const char *suffix) {
	size_t fnlen = strlen(fn),plen,slen;
	if( prefix == NULL )
		return true;
	assert(suffix != NULL);
	plen = strlen(prefix);
	slen = strlen(suffix);
	if( fnlen <= plen + slen )
		return false;
	if( slen > 0 && strcmp(fn+fnlen-slen,suffix) != 0 )
		return false;
	if( plen > 0 && strncmp(fn,prefix,plen) != 0 )
		return false;
	return true;
}

static const char *chooselinksuffix(const char *namesuffix, const char *filename) {
	const char *test;

	if( namesuffix != NULL )
		return namesuffix;

	namesuffix = strrchr(filename, '.');
	if( namesuffix == NULL )
		return namesuffix;
	/* if there is no namesuffix of the type, and the filename has a
	 * safe suffix, use that one: */
	for( test = namesuffix + 1 ; *test != '\0' ; test++ ) {
		if( *test < 'a' && *test > 'z' ) {
			namesuffix = NULL;
			break;
		}
	}
	return namesuffix;
}

static bool start(const struct mime_action *action,const char *command,const char *directory,const char *filename) {
	size_t l;
	char *unescaped,*fullfilename,*linkname;
	char *shellcommand = NULL;
	const char **args,**progargs;
	if( strcmp(command,"OPEN") == 0 ) {
		//TODO tell the file window to open this
		return false;
	} else if( strcmp(command,"LOAD") == 0 ) {
		Boolean b;
		fullfilename = concat(directory,filename);
		b = AmPushLoadFile(fullfilename,True);
		XtFree(fullfilename);
		return b?true:false;
	} else if( strcmp(command,"EXEC") == 0 ) {
		const char *argv[2];

		fullfilename = concat(directory,filename);
		if( fullfilename == NULL )
			abortXfm("Out of Memory");
		argv[0] = filename;
		argv[1] = NULL;

		execute_external_program(fullfilename,directory, argv, NULL, NULL);
		free(fullfilename);
		return true;
	}
	/* assume unescaping only shortens and there cannot be more arguments than characters */
	l = strlen(command);
	args = malloc((l+3+getArgCount(resources.xterminal)+getArgCount(resources.shell))*sizeof(char*));
	if( args == NULL )
		abortXfm("Out of Memory");
	if( action->needsterminal ) {
		int i;
		for( i = 0 ; resources.xterminal[i]!=NULL ; i++ ) {
			args[i] = resources.xterminal[i];
		}
		progargs = args + i;
	} else
		progargs = args;
	fullfilename = concat(directory,filename);
	if( fullfilename == NULL )
		abortXfm("Out of Memory");
	// TODO: look for a way to check normal files to
	// match nametemplates
	if( resources.alwaysSymlink || 
			!match(fullfilename, 
				action->nameprefix, action->namesuffix) ||
			veryDangerous(fullfilename) ) {
		linkname = allocateLink(fullfilename, action->nameprefix,
				chooselinksuffix(action->namesuffix, filename));
		if( linkname == NULL )
			return false;
	} else
		linkname = NULL;
	/* first try if we can get it working without a shell */
	unescaped = malloc(l+1);
	if( unescaped == NULL )
		abortXfm("Out of Memory");

	// TODO: implement this one...

	free(unescaped);
	/* if this does not work try with a shell */
	if( 1 ) {
		int i;

		if( linkname == NULL && (
				couldDangerous(fullfilename) ||
				( shellDangerous(fullfilename)
				  && isUnescaped(command) ) ) ) {
			linkname = allocateLink(fullfilename,
					action->nameprefix,
					chooselinksuffix(action->namesuffix,
						filename));
			if( linkname == NULL ) {
				free(args);
				free(fullfilename);
				return false;
			}
		}
		if( linkname == NULL )
			shellcommand = unescape_insert(command,fullfilename);
		else
			shellcommand = unescape_insert(command,linkname);
		if( shellcommand == NULL ) {
			free(args);
			free(fullfilename);
			free(linkname);
			return false;
		}
		for( i = 0 ; resources.shell[i] ; i++ ) {
			progargs[i] = resources.shell[i];
		}
		progargs[i] = shellcommand;
		progargs[i+1] = NULL;
	}
	/* actually call it */
	execute_external_program(args[0],directory, args, (linkname==NULL)?NULL:program_finished, linkname);
	free(args);
	free(fullfilename);
	// DO NOT free(linkname);
	free(shellcommand);
	return true;
}



bool mime_start(struct mime_filetype *mime_type,const char *directory,const char *filename,enum mime_start_type how ) {
	return mime_best_action(mime_type, directory, filename, how, start, resources.echo_mime_search);
	
}

