/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef VERSION
#define VERSION "UNKNOWN VERSION"
#endif

#include <stdio.h>
#include <string.h>

#include <libwps/libwps.h>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"

using namespace libwps;

class OdtOutputFileHelper : public OutputFileHelper
{
public:
	OdtOutputFileHelper(const char *outFileName, const char *password, const char *encoding) :
		OutputFileHelper(outFileName, password), mpPassword(password), mpEncoding(encoding) {};
	~OdtOutputFileHelper() {};
	bool convertDocument(librevenge::RVNGInputStream &input, bool isFlat)
	{
		OdtGenerator collector;
		collector.registerEmbeddedObjectHandler("image/wks-ods", &handleEmbeddedWKSObject);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		libwps::WPSResult result=WPSDocument::parse(&input, &collector, mpPassword, mpEncoding);
		if (result == WPS_ENCRYPTION_ERROR)
		{
			fprintf(stderr, "ERROR : encrypted file, problem with password\n");
			return false;
		}
		else if (result != WPS_OK)
		{
			fprintf(stderr, "ERROR : Couldn't convert the document\n");
			return false;
		}
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.text";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}
	bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		WPSCreator creator;
		WPSKind kind = WPS_TEXT;
		bool needEncoding;
		WPSConfidence confidence = WPSDocument::isFileFormatSupported(&input, kind, creator, needEncoding);
		if (confidence == WPS_CONFIDENCE_NONE || kind != WPS_TEXT)
			return false;
		if (!mpEncoding && needEncoding)
			fprintf(stderr, "WARNING: called without encoding, try to decode with basic character set encoding.\n");
		return true;
	}

private:

	static bool handleEmbeddedWKSObject(const librevenge::RVNGBinaryData &data, OdfDocumentHandler *pHandler,  const OdfStreamType streamType)
	{
		if (!data.size()) return false;
		OdsGenerator exporter;
		exporter.addDocumentHandler(pHandler, streamType);
		return WPSDocument::parse(data.getDataStream(), &exporter)==WPS_OK;
	}

	char const *mpPassword;
	char const *mpEncoding;
};

int printUsage(char *name)
{
	fprintf(stderr, "Usage: %s [--encoding encoding] [--password password] [--stdout] [--help] [--version] <infile> [outfile]\n", name);
	fprintf(stderr, "\twhere <infile> is the MS Works source document\n");
	fprintf(stderr, "\tand [outfile] is the odt target document. Alternately,\n");
	fprintf(stderr, "\tpass '--stdout' or simply omit the [outfile] to pipe the\n");
	fprintf(stderr, "\tresultant document as flat XML to standard output\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "\t--encoding encoding:    Define the file encoding where encoding can be\n");
	fprintf(stderr, "\t\t CP037, CP424, CP437, CP737, CP500, CP775, CP850, CP852, CP855, CP856, CP857,\n");
	fprintf(stderr, "\t\t CP860, CP861, CP862, CP863, CP864, CP865, CP866, CP869, CP874, CP875, CP1006, CP1026,\n");
	fprintf(stderr, "\t\t CP1250, CP1251, CP1252, CP1253, CP1254, CP1255, CP1256, CP1257, CP1258,\n");
	fprintf(stderr, "\t\t MacArabic, MacCeltic, MacCEurope, MacCroation, MacCyrillic, MacDevanage,\n");
	fprintf(stderr, "\t\t MacFarsi, MacGaelic, MacGreek, MacGujarati, MacGurmukhi, MacHebrew, MacIceland,\n");
	fprintf(stderr, "\t\t MacInuit, MacRoman, MacRomanian, MacThai, MacTurkish.\n");
	fprintf(stderr, "\t--help:                 Shows this help message\n");
	fprintf(stderr, "\t--password password:    Password to open the file\n");
	fprintf(stderr, "\t--stdout:               Always output the result to stdout\n");
	fprintf(stderr, "\t--version:              Output %s version \n", name);
	fprintf(stderr, "\n");
	return 1;
}

int printVersion(char const *name)
{
	printf("%s %s\n", name, VERSION);
	return 0;
}

int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(argv[0]);

	char *szInputFile = 0;
	char *szOutFile = 0;
	char *encoding = 0;
	char *password = 0;
	bool stdOutput = false;

	for (int i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "--stdout")==0)
			stdOutput = true;
		else if (strcmp(argv[i], "--version")==0)
			return printVersion(argv[0]);
		else if (strcmp(argv[i], "--help")==0)
			return printUsage(argv[0]);
		else if (strcmp(argv[i], "--encoding")==0)
		{
			if (i+1>=argc)
				return printUsage(argv[0]);
			encoding=argv[++i];
			continue;
		}
		else if (strcmp(argv[i], "--password")==0)
		{
			if (i+1>=argc)
				return printUsage(argv[0]);
			password=argv[++i];
			continue;
		}
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(argv[0]);
	}

	if (!szInputFile)
		return printUsage(argv[0]);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdtOutputFileHelper helper(szOutFile, password, encoding);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Microsoft Works document.\n");
		return 1;
	}

	if (!helper.convertDocument(input, szOutFile==0))
		return 1;

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
