/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtUserProfileHandler.h"

#include <control/profile/CUserProfileHandler.h>

#include <presentation/PFactory.h>
#include <presentation/qt/QtLanguage.h>
#include <presentation/qt/QtQuteCom.h>
#include <presentation/qt/login/QtLoginDialog.h>
#include <presentation/qt/QtSystray.h>
#include <presentation/qt/statusbar/QtStatusBar.h>

#include <qtutil/SafeConnect.h>
#include <util/Logger.h>
#include <util/SafeDelete.h>

#include <QtCore/QTimer>
#include <QtGui/QMessageBox>

QtUserProfileHandler::QtUserProfileHandler(CUserProfileHandler & cUserProfileHandler,
	QtQuteCom & qtQuteCom)
	: QObject(&qtQuteCom),
	_cUserProfileHandler(cUserProfileHandler),
	_qtQuteCom(qtQuteCom),_qtLoginDialog(0) {

	//Connection for UserProfile change
	SAFE_CONNECT_TYPE(this, SIGNAL(setCurrentUserProfileEventHandlerSignal()),
		SLOT(setCurrentUserProfileEventHandlerSlot()), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(noCurrentUserProfileSetEventHandlerSignal()),
		SLOT(noCurrentUserProfileSetEventHandlerSlot()), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(currentUserProfileWillDieEventHandlerSignal()),
		SLOT(currentUserProfileWillDieEventHandlerSlot()), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(userProfileInitializedEventHandlerSignal()),
		SLOT(userProfileInitializedEventHandlerSlot()), Qt::QueuedConnection);
	qRegisterMetaType<SipAccount>("SipAccount");
	qRegisterMetaType<QuteComAccount>("QuteComAccount");
	qRegisterMetaType<EnumSipLoginState::SipLoginState>("EnumSipLoginState::SipLoginState");
	SAFE_CONNECT_TYPE(this, SIGNAL(sipAccountConnectionFailedEventHandlerSignal(SipAccount*, EnumSipLoginState::SipLoginState)),
		SLOT(sipAccountConnectionFailedEventHandlerSlot(SipAccount*, EnumSipLoginState::SipLoginState)), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(profileLoadedFromBackupsEventHandlerSignal(QString)),
		SLOT(profileLoadedFromBackupsEventHandlerSlot(QString)), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(profileCannotBeLoadedEventHandlerSignal(QString)),
		SLOT(profileCannotBeLoadedEventHandlerSlot(QString)), Qt::QueuedConnection);
	SAFE_CONNECT_TYPE(this, SIGNAL(sipNetworkErrorEventHandlerSignal(SipAccount*)),
		SLOT(sipNetworkErrorEventHandlerSlot(SipAccount*)), Qt::QueuedConnection);
}

void QtUserProfileHandler::updatePresentation() {
}

void QtUserProfileHandler::noCurrentUserProfileSetEventHandler() {
	noCurrentUserProfileSetEventHandlerSignal();
}

void QtUserProfileHandler::currentUserProfileWillDieEventHandler() {
	currentUserProfileWillDieEventHandlerSignal();
}

void QtUserProfileHandler::userProfileInitializedEventHandler() {
	userProfileInitializedEventHandlerSignal();
}

void QtUserProfileHandler::sipAccountConnectionFailedEventHandler(const SipAccount & sipAccount, EnumSipLoginState::SipLoginState state) {
	// We need to clone the SipAccount because we don't want to pass a
	// reference of the original, and doing a simple copy would produce a
	// SipAccount instance, even if the reference is in fact a QuteComAccount.
	sipAccountConnectionFailedEventHandlerSignal(sipAccount.clone(), state);
}

void QtUserProfileHandler::profileLoadedFromBackupsEventHandler(std::string profileName) {
	profileLoadedFromBackupsEventHandlerSignal(QString::fromUtf8(profileName.c_str()));
}

void QtUserProfileHandler::profileCannotBeLoadedEventHandler(std::string profileName) {
	profileCannotBeLoadedEventHandlerSignal(QString::fromUtf8(profileName.c_str()));
}

void QtUserProfileHandler::networkErrorEventHandler(const SipAccount & sipAccount) {
	sipNetworkErrorEventHandlerSignal(sipAccount.clone());
}

void QtUserProfileHandler::noCurrentUserProfileSetEventHandlerSlot() {
	showLoginWindow();
}

void QtUserProfileHandler::setCurrentUserProfileEventHandler() {
	setCurrentUserProfileEventHandlerSignal();
}

void QtUserProfileHandler::setCurrentUserProfileEventHandlerSlot() {
	_qtQuteCom.setCurrentUserProfileEventHandlerSlot();
}

void QtUserProfileHandler::currentUserProfileWillDieEventHandlerSlot() {
	if(!_qtLoginDialog)
	{
		_qtQuteCom.currentUserProfileWillDieEventHandlerSlot();
		_cUserProfileHandler.currentUserProfileReleased();

	// For some reason, if we call showLoginWindow directly, the CPU is at 100%
	// on MacOS X. Delaying showLoginWindow with a timer solves it.
		QTimer::singleShot(0, this, SLOT(showLoginWindow()));
	}
}

void QtUserProfileHandler::userProfileInitializedEventHandlerSlot() {
	_qtQuteCom.userProfileInitializedEventHandlerSlot();
}

void QtUserProfileHandler::profileLoadedFromBackupsEventHandlerSlot(QString profileName) {
	QMessageBox::warning(_qtQuteCom.getWidget(), tr("@product@"),
		tr("A problem occured while loading your profile.\n"
			"The last backuped profile has been loaded: \n"
			"you may have lost last changes made"),
		QMessageBox::Ok, QMessageBox::NoButton);
}

void QtUserProfileHandler::profileCannotBeLoadedEventHandlerSlot(QString profileName) {
	QMessageBox::warning(_qtQuteCom.getWidget(), tr("@product@"),
		tr("Your profile could not be loaded.\n"
			"Choose another profile or create a new one."),
		QMessageBox::Ok, QMessageBox::NoButton);
}

void QtUserProfileHandler::showLoginWindow() {

	_qtQuteCom.uninstallQtBrowserWidget();
	if(!_qtLoginDialog)
	{
		if(!_qtLoginDialog)
		_qtLoginDialog = new QtLoginDialog(&_qtQuteCom, _cUserProfileHandler);
		if (_qtLoginDialog->exec() == QDialog::Accepted) {
			//_qtQuteCom.installQtBrowserWidget();
			_qtQuteCom.getQtStatusBar().updatePhoneLineState(EnumPhoneLineState::PhoneLineStateProgress);
			_qtQuteCom.getQtSystray().phoneLineStateChanged(EnumPhoneLineState::PhoneLineStateProgress);
		}
		delete _qtLoginDialog;
		_qtLoginDialog = 0;
	}
}

void QtUserProfileHandler::showLoginWindowWithQuteComAccount(const QuteComAccount & qutecomAccount) {

	if(!_qtLoginDialog )
	{
		_qtLoginDialog = new QtLoginDialog(&_qtQuteCom, _cUserProfileHandler);

		_qtLoginDialog->setValidAccount(qutecomAccount);
		if (_qtLoginDialog->exec() == QDialog::Accepted) {
			_qtQuteCom.installQtBrowserWidget();
			_qtQuteCom.getQtStatusBar().updatePhoneLineState(EnumPhoneLineState::PhoneLineStateProgress);
			_qtQuteCom.getQtSystray().phoneLineStateChanged(EnumPhoneLineState::PhoneLineStateProgress);
		}
		delete _qtLoginDialog;
		_qtLoginDialog = 0;
	}
}

void QtUserProfileHandler::sipNetworkErrorEventHandlerSlot(SipAccount* sipAccount) {
	/*_qtQuteCom.uninstallQtBrowserWidget();

	QtLoginDialog dlg(&_qtQuteCom, _cUserProfileHandler);
	dlg.setInvalidAccount(*sipAccount);
	dlg.setErrorMessage(tr("Network error."));
	if (dlg.exec() == QDialog::Accepted) {
		_qtQuteCom.installQtBrowserWidget();
	}
	delete sipAccount;*/
}

void QtUserProfileHandler::sipAccountConnectionFailedEventHandlerSlot(SipAccount* sipAccount, EnumSipLoginState::SipLoginState state) {
	LOG_DEBUG("error: %d", state);
	// FIXME: Why do we do different things depending on the account type?
	if (sipAccount->getType() == SipAccount::SipAccountTypeQuteCom) {
		_qtQuteCom.uninstallQtBrowserWidget();
	} else {
		_qtQuteCom.currentUserProfileWillDieEventHandlerSlot();
		_cUserProfileHandler.currentUserProfileReleased();
	}
	
	if(!_qtLoginDialog)
	{
		_qtLoginDialog =new QtLoginDialog(&_qtQuteCom, _cUserProfileHandler);
		_qtLoginDialog->setInvalidAccount(*sipAccount);
		QString message;
		if (state == EnumSipLoginState::SipLoginStatePasswordError) {
			if (sipAccount->getType() == SipAccount::SipAccountTypeQuteCom) {
				message = tr("Wrong email/password entered");
			} else {
				message = tr("Wrong login/password entered");
			}
		} else if (state == EnumSipLoginState::SipLoginStateNetworkError) {
			message = tr("Connection error");
		} else {
			message = tr("Unexpected error (code: %1").arg(int(state));
		}
		_qtLoginDialog->setErrorMessage(message);
		
		delete sipAccount;

		if (_qtLoginDialog->exec() == QDialog::Accepted) {
			_qtQuteCom.installQtBrowserWidget();
			_qtQuteCom.getQtStatusBar().updatePhoneLineState(EnumPhoneLineState::PhoneLineStateProgress);
			_qtQuteCom.getQtSystray().phoneLineStateChanged(EnumPhoneLineState::PhoneLineStateProgress);
		}
		delete _qtLoginDialog;
		_qtLoginDialog = 0;
	}
}
