// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/signin/profile_picker_handler.h"

#include <algorithm>
#include <vector>

#include "base/check.h"
#include "base/check_op.h"
#include "base/debug/dump_without_crashing.h"
#include "base/feature_list.h"
#include "base/files/file_path.h"
#include "base/functional/bind.h"
#include "base/functional/callback_helpers.h"
#include "base/json/values_util.h"
#include "base/metrics/histogram_functions.h"
#include "base/notreached.h"
#include "base/strings/utf_string_conversions.h"
#include "base/trace_event/trace_event.h"
#include "base/values.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/new_tab_page/chrome_colors/chrome_colors_service.h"
#include "chrome/browser/new_tab_page/chrome_colors/generated_colors_info.h"
#include "chrome/browser/profiles/keep_alive/profile_keep_alive_types.h"
#include "chrome/browser/profiles/keep_alive/scoped_profile_keep_alive.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profile_attributes_entry.h"
#include "chrome/browser/profiles/profile_attributes_storage.h"
#include "chrome/browser/profiles/profile_avatar_icon_util.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/profiles/profile_statistics.h"
#include "chrome/browser/profiles/profile_statistics_factory.h"
#include "chrome/browser/profiles/profile_window.h"
#include "chrome/browser/profiles/profiles_state.h"
#include "chrome/browser/signin/signin_util.h"
#include "chrome/browser/ui/browser_finder.h"
#include "chrome/browser/ui/browser_window.h"
#include "chrome/browser/ui/color/chrome_color_id.h"
#include "chrome/browser/ui/profiles/profile_colors_util.h"
#include "chrome/browser/ui/profiles/profile_picker.h"
#include "chrome/browser/ui/ui_features.h"
#include "chrome/browser/ui/webui/profile_helper.h"
#include "chrome/browser/ui/webui/signin/login_ui_service.h"
#include "chrome/browser/ui/webui/signin/login_ui_service_factory.h"
#include "chrome/browser/ui/webui/signin/signin_ui_error.h"
#include "chrome/browser/ui/webui/theme_source.h"
#include "chrome/common/pref_names.h"
#include "chrome/common/themes/autogenerated_theme_util.h"
#include "chrome/common/webui_url_constants.h"
#include "chrome/grit/generated_resources.h"
#include "components/signin/public/identity_manager/account_info.h"
#include "components/startup_metric_utils/browser/startup_metric_utils.h"
#include "components/supervised_user/core/common/features.h"
#include "content/public/browser/url_data_source.h"
#include "content/public/browser/web_ui.h"
#include "third_party/skia/include/core/SkBitmap.h"
#include "third_party/skia/include/core/SkColor.h"
#include "ui/base/l10n/l10n_util.h"
#include "ui/base/mojom/themes.mojom.h"
#include "ui/base/ui_base_features.h"
#include "ui/base/webui/web_ui_util.h"
#include "ui/color/color_id.h"
#include "ui/color/color_provider.h"
#include "ui/gfx/color_utils.h"
#include "ui/gfx/image/image.h"
#include "ui/webui/webui_util.h"

namespace {
const size_t kProfileCardAvatarSize = 74;
const size_t kProfileCreationAvatarSize = 100;

constexpr int kDefaultThemeColorId = -1;
constexpr int kManuallyPickedColorId = 0;

std::optional<SkColor> GetChromeColorColorById(int color_id) {
  for (chrome_colors::ColorInfo color_info :
       chrome_colors::kGeneratedColorsInfo) {
    if (color_id == color_info.id) {
      return color_info.color;
    }
  }

  return std::nullopt;
}

void RecordAskOnStartupChanged(bool value) {
  base::UmaHistogramBoolean("ProfilePicker.AskOnStartupChanged", value);
}

base::Value::Dict GetAutogeneratedProfileThemeInfoValue(
    int color_id,
    std::optional<SkColor> color,
    const ui::ColorProvider& color_provider,
    SkColor frame_color,
    SkColor active_tab_color,
    SkColor frame_text_color,
    float scale_factor) {
  base::Value::Dict dict;
  dict.Set("colorId", color_id);
  if (color.has_value()) {
    dict.Set("color", static_cast<int>(*color));
  }
  dict.Set("themeFrameColor", color_utils::SkColorToRgbaString(frame_color));
  dict.Set("themeShapeColor",
           color_utils::SkColorToRgbaString(active_tab_color));
  dict.Set("themeFrameTextColor",
           color_utils::SkColorToRgbaString(frame_text_color));
  DefaultAvatarColors avatar_colors =
      GetDefaultAvatarColors(color_provider, frame_color);
  gfx::Image icon = profiles::GetPlaceholderAvatarIconWithColors(
      /*fill_color=*/avatar_colors.fill_color,
      /*stroke_color=*/avatar_colors.stroke_color,
      kProfileCreationAvatarSize * scale_factor);
  dict.Set("themeGenericAvatar", webui::GetBitmapDataUrl(icon.AsBitmap()));
  return dict;
}

base::Value::Dict CreateDefaultProfileThemeInfo(
    const ui::ColorProvider& color_provider,
    float scale_factor) {
  SkColor frame_color = color_provider.GetColor(ui::kColorFrameActive);
  SkColor active_tab_color = color_provider.GetColor(kColorToolbar);
  SkColor frame_text_color =
      color_provider.GetColor(kColorTabForegroundInactiveFrameActive);
  return GetAutogeneratedProfileThemeInfoValue(
      kDefaultThemeColorId, std::nullopt, color_provider, frame_color,
      active_tab_color, frame_text_color, scale_factor);
}

base::Value::Dict CreateAutogeneratedProfileThemeInfo(
    int color_id,
    SkColor color,
    const ui::ColorProvider& color_provider,
    float scale_factor) {
  auto theme_colors = GetAutogeneratedThemeColors(color);
  SkColor frame_color = theme_colors.frame_color;
  SkColor active_tab_color = theme_colors.active_tab_color;
  SkColor frame_text_color = theme_colors.frame_text_color;
  return GetAutogeneratedProfileThemeInfoValue(color_id, color, color_provider,
                                               frame_color, active_tab_color,
                                               frame_text_color, scale_factor);
}

base::Value::Dict CreateProfileEntry(const ProfileAttributesEntry* entry,
                                     int avatar_icon_size) {
  base::Value::Dict profile_entry;
  profile_entry.Set("profilePath", base::FilePathToValue(entry->GetPath()));
  profile_entry.Set("localProfileName", entry->GetLocalProfileName());
  profile_entry.Set("hasEnterpriseLabel",
                    !entry->GetEnterpriseProfileLabel().empty());
  profile_entry.Set("isSyncing",
                    entry->GetSigninState() ==
                        SigninState::kSignedInWithConsentedPrimaryAccount);
  profile_entry.Set("needsSignin", entry->IsSigninRequired());
  // GAIA full name/user name can be empty, if the profile is not signed in to
  // chrome.
  profile_entry.Set("gaiaName", entry->GetGAIAName());
  profile_entry.Set("userName", entry->GetUserName());

  const auto local_profile_name = entry->GetLocalProfileName();
  std::u16string profileCardButtonLabel = l10n_util::GetStringFUTF16(
      IDS_PROFILE_PICKER_PROFILE_CARD_LABEL, local_profile_name);
  if (AccountInfo::IsManaged(entry->GetHostedDomain())) {
    profile_entry.Set("avatarBadge", "cr:domain");
#if BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  } else if (base::FeatureList::IsEnabled(
                 supervised_user::kShowKiteForSupervisedUsers) &&
             entry->IsSupervised()) {
    profileCardButtonLabel = l10n_util::GetStringFUTF16(
        IDS_PROFILE_PICKER_PROFILE_CARD_LABEL_SUPERVISED, local_profile_name);
    profile_entry.Set("avatarBadge", "cr:kite");
#endif  // BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC) || BUILDFLAG(IS_WIN)
  } else {
    profile_entry.Set("avatarBadge", "");
  }
  profile_entry.Set("profileCardButtonLabel", profileCardButtonLabel);

  gfx::Image icon =
      profiles::GetSizedAvatarIcon(entry->GetAvatarIcon(avatar_icon_size),
                                   avatar_icon_size, avatar_icon_size);
  std::string icon_url = webui::GetBitmapDataUrl(icon.AsBitmap());
  profile_entry.Set("avatarIcon", icon_url);
  return profile_entry;
}

// Opens the "Sign in to Chrome" Help Center URL.
void OpenSigninOnDesktopLearnMoreURL(Browser* browser) {
  // Browser may be closing if the Profile was locked after being loaded for
  // example.
  if (!browser || browser->IsBrowserClosing()) {
    return;
  }

  browser->OpenURL(
      content::OpenURLParams(GURL(chrome::kSigninOnDesktopLearnMoreURL),
                             content::Referrer(),
                             WindowOpenDisposition::NEW_FOREGROUND_TAB,
                             ui::PAGE_TRANSITION_LINK, false),
      /*navigation_handle_callback=*/{});
}

}  // namespace

ProfilePickerHandler::ProfilePickerHandler(bool is_glic_version)
    : is_glic_version_(is_glic_version) {}

ProfilePickerHandler::~ProfilePickerHandler() {
  OnJavascriptDisallowed();
}

void ProfilePickerHandler::EnableStartupMetrics() {
  DCHECK(creation_time_on_startup_.is_null());
  content::WebContents* contents = web_ui()->GetWebContents();
  if (contents->GetVisibility() == content::Visibility::VISIBLE) {
    // Only record paint event if the window is visible.
    creation_time_on_startup_ = base::TimeTicks::Now();
    Observe(web_ui()->GetWebContents());
  }
}

void ProfilePickerHandler::RegisterMessages() {
  web_ui()->RegisterMessageCallback(
      "mainViewInitialize",
      base::BindRepeating(&ProfilePickerHandler::HandleMainViewInitialize,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "launchSelectedProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchSelectedProfile,
                          base::Unretained(this), /*open_settings=*/false));
  web_ui()->RegisterMessageCallback(
      "openManageProfileSettingsSubPage",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchSelectedProfile,
                          base::Unretained(this), /*open_settings=*/true));
  web_ui()->RegisterMessageCallback(
      "launchGuestProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleLaunchGuestProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "askOnStartupChanged",
      base::BindRepeating(&ProfilePickerHandler::HandleAskOnStartupChanged,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getNewProfileSuggestedThemeInfo",
      base::BindRepeating(
          &ProfilePickerHandler::HandleGetNewProfileSuggestedThemeInfo,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getProfileThemeInfo",
      base::BindRepeating(&ProfilePickerHandler::HandleGetProfileThemeInfo,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "removeProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleRemoveProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getProfileStatistics",
      base::BindRepeating(&ProfilePickerHandler::HandleGetProfileStatistics,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "closeProfileStatistics",
      base::BindRepeating(&ProfilePickerHandler::HandleCloseProfileStatistics,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "selectNewAccount",
      base::BindRepeating(&ProfilePickerHandler::HandleSelectNewAccount,
                          base::Unretained(this)));
  // TODO(crbug.com/40144179): Consider renaming this message to
  // 'createLocalProfile' as this is only used for local profiles.
  web_ui()->RegisterMessageCallback(
      "getAvailableIcons",
      base::BindRepeating(&ProfilePickerHandler::HandleGetAvailableIcons,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "continueWithoutAccount",
      base::BindRepeating(&ProfilePickerHandler::HandleContinueWithoutAccount,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "getSwitchProfile",
      base::BindRepeating(&ProfilePickerHandler::HandleGetSwitchProfile,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "confirmProfileSwitch",
      base::BindRepeating(&ProfilePickerHandler::HandleConfirmProfileSwitch,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "cancelProfileSwitch",
      base::BindRepeating(&ProfilePickerHandler::HandleCancelProfileSwitch,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "setProfileName",
      base::BindRepeating(&ProfilePickerHandler::HandleSetProfileName,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "recordSignInPromoImpression",
      base::BindRepeating(
          &ProfilePickerHandler::HandleRecordSignInPromoImpression,
          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "updateProfileOrder",
      base::BindRepeating(&ProfilePickerHandler::HandleUpdateProfileOrder,
                          base::Unretained(this)));
  web_ui()->RegisterMessageCallback(
      "onLearnMoreClicked",
      base::BindRepeating(&ProfilePickerHandler::HandleOnLearnMoreClicked,
                          base::Unretained(this)));
  Profile* profile = Profile::FromWebUI(web_ui());
  content::URLDataSource::Add(profile, std::make_unique<ThemeSource>(profile));
}

void ProfilePickerHandler::OnJavascriptAllowed() {
  ProfileManager* profile_manager = g_browser_process->profile_manager();
  profile_attributes_storage_observation_.Observe(
      &profile_manager->GetProfileAttributesStorage());
}
void ProfilePickerHandler::OnJavascriptDisallowed() {
  profile_attributes_storage_observation_.Reset();
  weak_factory_.InvalidateWeakPtrs();
}

void ProfilePickerHandler::HandleMainViewInitialize(
    const base::Value::List& args) {
  if (!creation_time_on_startup_.is_null() && !main_view_initialized_) {
    // This function can be called multiple times if the page is reloaded. The
    // histogram is only recorded once.
    main_view_initialized_ = true;
    base::UmaHistogramTimes("ProfilePicker.StartupTime.MainViewInitialized",
                            base::TimeTicks::Now() - creation_time_on_startup_);
  }

  AllowJavascript();
  PushProfilesList();
}

void ProfilePickerHandler::HandleLaunchSelectedProfile(
    bool open_settings,
    const base::Value::List& args) {
  TRACE_EVENT1("browser", "ProfilePickerHandler::HandleLaunchSelectedProfile",
               "args", args.DebugString());
  if (args.empty()) {
    return;
  }
  const base::Value& profile_path_value = args[0];

  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(*profile_path);
  if (!entry) {
    NOTREACHED();
  }

  // If a browser window cannot be opened for profile, load the profile to
  // display a dialog.
  if (entry->IsSigninRequired()) {
    g_browser_process->profile_manager()->LoadProfileByPath(
        *profile_path, /*incognito=*/false,
        base::BindOnce(&ProfilePickerHandler::OnProfileForDialogLoaded,
                       weak_factory_.GetWeakPtr()));
    return;
  }

  bool should_record_startup_metrics = !creation_time_on_startup_.is_null();
  ProfilePicker::PickProfile(
      *profile_path,
      ProfilePicker::ProfilePickingArgs{
          .open_settings = open_settings,
          .should_record_startup_metrics = should_record_startup_metrics});
}

void ProfilePickerHandler::OnProfileForDialogLoaded(Profile* profile) {
  if (!profile) {
    return;
  }
#if BUILDFLAG(ENABLE_DICE_SUPPORT)
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile->GetPath());
  DCHECK(entry);
  if (entry->IsSigninRequired()) {
    DCHECK(signin_util::IsForceSigninEnabled());
    if (entry->CanBeManaged()) {
      ProfilePicker::SwitchToReauth(
          profile,
          base::BindOnce(&ProfilePickerHandler::DisplayForceSigninErrorDialog,
                         weak_factory_.GetWeakPtr(), profile->GetPath()));
    } else if (entry->GetActiveTime() != base::Time()) {
      // If force-sign-in is enabled, do not allow users to sign in to a
      // pre-existing locked profile, as this may force unexpected profile data
      // merge. We consider a profile as pre-existing if it has been active
      // previously. A pre-existed profile can still be used if it has been
      // signed in with an email address matched RestrictSigninToPattern policy
      // already.
      DisplayForceSigninErrorDialog(
          /*profile_path=*/base::FilePath(),
          ForceSigninUIError::ReauthNotAllowed());
    } else {
      // Fresh sign in via profile picker without existing email address.
      ProfilePicker::SwitchToDiceSignIn(
          profile->GetPath(),
          base::BindOnce(&ProfilePickerHandler::OnLoadSigninFinished,
                         weak_factory_.GetWeakPtr()));
    }
  }
#endif  // BUILDFLAG(ENABLE_DICE_SUPPORT)
}

void ProfilePickerHandler::DisplayForceSigninErrorDialog(
    const base::FilePath& profile_path,
    const ForceSigninUIError& error) {
  AllowJavascript();

  const auto& [title, body] = error.GetErrorTexts();
  FireWebUIListener("display-force-signin-error-dialog", base::Value(title),
                    base::Value(body),
                    base::Value(profile_path.AsUTF16Unsafe()));
}

void ProfilePickerHandler::HandleLaunchGuestProfile(
    const base::Value::List& args) {
  // TODO(crbug.com/40123459): Add check |IsGuestModeEnabled| once policy
  // checking has been added to the UI.
  ProfilePicker::PickProfile(
      ProfileManager::GetGuestProfilePath(),
      ProfilePicker::ProfilePickingArgs{
          .open_settings = false, .should_record_startup_metrics = false});
}

void ProfilePickerHandler::HandleAskOnStartupChanged(
    const base::Value::List& list) {
  if (list.empty() || !list[0].is_bool()) {
    return;
  }
  const bool show_on_startup = list[0].GetBool();

  PrefService* prefs = g_browser_process->local_state();
  prefs->SetBoolean(prefs::kBrowserShowProfilePickerOnStartup, show_on_startup);
  RecordAskOnStartupChanged(show_on_startup);
}

void ProfilePickerHandler::HandleGetNewProfileSuggestedThemeInfo(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];

  chrome_colors::ColorInfo color_info = GenerateNewProfileColor();
  base::Value::Dict dict = CreateAutogeneratedProfileThemeInfo(
      color_info.id, color_info.color,
      web_ui()->GetWebContents()->GetColorProvider(),
      web_ui()->GetDeviceScaleFactor());
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleGetProfileThemeInfo(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(2U, args.size());
  const base::Value& callback_id = args[0];
  const base::Value::Dict& user_theme_choice = args[1].GetDict();
  int color_id = user_theme_choice.FindInt("colorId").value();
  std::optional<SkColor> color = user_theme_choice.FindDouble("color");
  base::Value::Dict dict;
  switch (color_id) {
    case kDefaultThemeColorId:
      dict = CreateDefaultProfileThemeInfo(
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
    case kManuallyPickedColorId:
      dict = CreateAutogeneratedProfileThemeInfo(
          color_id, *color, web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
    default:
      dict = CreateAutogeneratedProfileThemeInfo(
          color_id, *GetChromeColorColorById(color_id),
          web_ui()->GetWebContents()->GetColorProvider(),
          web_ui()->GetDeviceScaleFactor());
      break;
  }
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleGetAvailableIcons(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];
  ResolveJavascriptCallback(callback_id,
                            profiles::GetCustomProfileAvatarIconsAndLabels());
}

void ProfilePickerHandler::HandleContinueWithoutAccount(
    const base::Value::List& args) {
  CHECK_EQ(1U, args.size());

  // profileColor is undefined for the default theme.
  std::optional<SkColor> profile_color;
  if (args[0].is_int()) {
    profile_color = args[0].GetInt();
  }

  RecordProfilePickerAction(ProfilePickerAction::kLaunchNewProfile);
  ProfileMetrics::LogProfileAddNewUser(
      ProfileMetrics::ADD_NEW_PROFILE_PICKER_LOCAL);
  ProfilePicker::SwitchToSignedOutPostIdentityFlow(
      profile_color,
      base::BindOnce(&ProfilePickerHandler::OnProfileCreationFinished,
                     // `OnProfileCreationFinished` is called when we want to
                     // close the profile picker.
                     weak_factory_.GetWeakPtr()));
}

void ProfilePickerHandler::HandleGetSwitchProfile(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& callback_id = args[0];
  int avatar_icon_size =
      kProfileCardAvatarSize * web_ui()->GetDeviceScaleFactor();
  base::FilePath profile_path = ProfilePicker::GetSwitchProfilePath();
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  base::Value::Dict dict = CreateProfileEntry(entry, avatar_icon_size);
  ResolveJavascriptCallback(callback_id, dict);
}

void ProfilePickerHandler::HandleConfirmProfileSwitch(
    const base::Value::List& args) {
  if (args.empty()) {
    return;
  }
  const base::Value& profile_path_value = args[0];

  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  // TODO(crbug.com/40751337): remove the profile used for the sign-in
  // flow.
  ProfilePicker::PickProfile(
      *profile_path,
      ProfilePicker::ProfilePickingArgs{
          .open_settings = false, .should_record_startup_metrics = false});
}

void ProfilePickerHandler::HandleCancelProfileSwitch(
    const base::Value::List& args) {
  ProfilePicker::CancelSignedInFlow();
}

void ProfilePickerHandler::OnProfileCreationFinished(
    bool finished_successfully) {
  FireWebUIListener("create-profile-finished", base::Value());
}

void ProfilePickerHandler::HandleRecordSignInPromoImpression(
    const base::Value::List& /*args*/) {
  signin_metrics::RecordSigninImpressionUserActionForAccessPoint(
      signin_metrics::AccessPoint::kUserManager);
}

void ProfilePickerHandler::HandleSetProfileName(const base::Value::List& args) {
  CHECK_EQ(2U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);

  if (!profile_path) {
    NOTREACHED();
  }
  std::u16string profile_name = base::UTF8ToUTF16(args[1].GetString());
  base::TrimWhitespace(profile_name, base::TRIM_ALL, &profile_name);
  CHECK(!profile_name.empty());
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path.value());
  CHECK(entry);
  entry->SetLocalProfileName(profile_name, /*is_default_name=*/false);
}

void ProfilePickerHandler::HandleRemoveProfile(const base::Value::List& args) {
  CHECK_EQ(1U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);

  if (!profile_path) {
    NOTREACHED();
  }

  RecordProfilePickerAction(ProfilePickerAction::kDeleteProfile);
  DCHECK(profile_statistics_keep_alive_);

  // Deleting the profile may delete `this` (see See https://crbug.com/1488267),
  // if the profile picker was shown in a tab. Keep the `ScopedProfileKeepAlive`
  // until the end of the function, to avoid the profile being unloaded and
  // reloaded.
  std::unique_ptr<ScopedProfileKeepAlive> profile_statistics_keep_alive =
      std::move(profile_statistics_keep_alive_);
  webui::DeleteProfileAtPath(*profile_path,
                             ProfileMetrics::DELETE_PROFILE_USER_MANAGER);
  // Do not use `this` after this point, it may be deleted.
}

void ProfilePickerHandler::HandleUpdateProfileOrder(
    const base::Value::List& args) {
  CHECK_EQ(2U, args.size());
  CHECK(args[0].is_int());
  CHECK(args[1].is_int());

  int from_index = args[0].GetInt();
  int to_index = args[1].GetInt();
  CHECK(from_index >= 0 && to_index >= 0);

  g_browser_process->profile_manager()
      ->GetProfileAttributesStorage()
      .UpdateProfilesOrderPref(from_index, to_index);
}

void ProfilePickerHandler::HandleOnLearnMoreClicked(
    const base::Value::List& args) {
  CHECK_EQ(0U, args.size());

  // Loads the last used profile and open/uses a browser to show the help page.
  profiles::SwitchToProfile(
      g_browser_process->profile_manager()->GetLastUsedProfileDir(),
      /*always_create=*/false,
      base::BindOnce(&OpenSigninOnDesktopLearnMoreURL));
}

void ProfilePickerHandler::HandleCloseProfileStatistics(
    const base::Value::List& args) {
  CHECK_EQ(0U, args.size());
  DCHECK(profile_statistics_keep_alive_);
  profile_statistics_keep_alive_.reset();
}

void ProfilePickerHandler::HandleGetProfileStatistics(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  const base::Value& profile_path_value = args[0];
  std::optional<base::FilePath> profile_path =
      base::ValueToFilePath(profile_path_value);
  if (!profile_path) {
    return;
  }

  Profile* profile =
      g_browser_process->profile_manager()->GetProfileByPath(*profile_path);

  if (profile) {
    GatherProfileStatistics(profile);
  } else {
    g_browser_process->profile_manager()->LoadProfileByPath(
        *profile_path, false,
        base::BindOnce(&ProfilePickerHandler::GatherProfileStatistics,
                       weak_factory_.GetWeakPtr()));
  }
}

void ProfilePickerHandler::GatherProfileStatistics(Profile* profile) {
  if (!profile) {
    return;
  }

  profile_statistics_keep_alive_ = std::make_unique<ScopedProfileKeepAlive>(
      profile, ProfileKeepAliveOrigin::kProfileStatistics);

  ProfileStatisticsFactory::GetForProfile(profile)->GatherStatistics(
      base::BindRepeating(&ProfilePickerHandler::OnProfileStatisticsReceived,
                          weak_factory_.GetWeakPtr(), profile->GetPath()));
}

void ProfilePickerHandler::OnProfileStatisticsReceived(
    const base::FilePath& profile_path,
    profiles::ProfileCategoryStats result) {
  base::Value::Dict dict;
  dict.Set("profilePath", base::FilePathToValue(profile_path));
  base::Value::Dict stats;
  // Categories are defined in |kProfileStatisticsCategories|
  // {"BrowsingHistory", "Passwords", "Bookmarks", "Autofill"}.
  for (const auto& item : result) {
    stats.Set(item.category, item.count);
  }
  dict.Set("statistics", std::move(stats));
  FireWebUIListener("profile-statistics-received", dict);
}

void ProfilePickerHandler::HandleSelectNewAccount(
    const base::Value::List& args) {
  AllowJavascript();
  CHECK_EQ(1U, args.size());
  std::optional<SkColor> profile_color = args[0].GetIfInt();
#if BUILDFLAG(ENABLE_DICE_SUPPORT)
  if (signin_util::IsForceSigninEnabled()) {
    // Force sign-in policy uses a separate flow that doesn't initialize the
    // profile color. Generate a new profile color here.
    profile_color = GenerateNewProfileColor().color;
  }
  ProfilePicker::SwitchToDiceSignIn(
      profile_color, base::BindOnce(&ProfilePickerHandler::OnLoadSigninFinished,
                                    weak_factory_.GetWeakPtr()));
#else
  NOTERACHED();
#endif
}

void ProfilePickerHandler::OnLoadSigninFinished(bool success) {
  AllowJavascript();
  FireWebUIListener("load-signin-finished", base::Value(success));
}

void ProfilePickerHandler::PushProfilesList() {
  DCHECK(IsJavascriptAllowed());
  FireWebUIListener("profiles-list-changed", GetProfilesList());
}

void ProfilePickerHandler::SetProfilesOrder(
    const std::vector<ProfileAttributesEntry*>& entries) {
  profiles_order_.clear();
  size_t index = 0;
  for (const ProfileAttributesEntry* entry : entries) {
    profiles_order_[entry->GetPath()] = index++;
  }
}

std::vector<ProfileAttributesEntry*>
ProfilePickerHandler::GetProfileAttributes() {
  std::vector<ProfileAttributesEntry*> ordered_entries =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetAllProfilesAttributesSortedByLocalProfileNameWithCheck();
  std::erase_if(ordered_entries, [](const ProfileAttributesEntry* entry) {
    return entry->IsOmitted();
  });
  size_t number_of_profiles = ordered_entries.size();

  if (profiles_order_.size() != number_of_profiles) {
    // Should only happen the first time the function is called.
    // Profile creation and deletion are handled at
    // 'OnProfileAdded', 'OnProfileWasRemoved'.
    DCHECK(!profiles_order_.size());
    SetProfilesOrder(ordered_entries);
    return ordered_entries;
  }

  // Vector of nullptr entries.
  std::vector<ProfileAttributesEntry*> entries(number_of_profiles);
  for (ProfileAttributesEntry* entry : ordered_entries) {
    DCHECK(profiles_order_.find(entry->GetPath()) != profiles_order_.end());
    size_t index = profiles_order_[entry->GetPath()];
    DCHECK_LT(index, number_of_profiles);
    DCHECK(!entries[index]);
    entries[index] = entry;
  }
  return entries;
}

base::Value::List ProfilePickerHandler::GetProfilesList() {
  base::Value::List profiles_list;
  std::vector<ProfileAttributesEntry*> entries = GetProfileAttributes();

  // In Glic version, only allow profile entries that are eligible. This may
  // cause the returned profile list to be empty, and will display different
  // strings in the Ui.
  if (is_glic_version_) {
    std::erase_if(entries, [](const ProfileAttributesEntry* entry) {
      return !entry->IsGlicEligible();
    });
  }

  const int avatar_icon_size =
      kProfileCardAvatarSize * web_ui()->GetDeviceScaleFactor();
  for (const ProfileAttributesEntry* entry : entries) {
    profiles_list.Append(CreateProfileEntry(entry, avatar_icon_size));
  }
  return profiles_list;
}

void ProfilePickerHandler::AddProfileToListAndPushUpdates(
    const base::FilePath& profile_path) {
  size_t number_of_profiles = profiles_order_.size();
  auto it_and_whether_inserted =
      profiles_order_.insert({profile_path, number_of_profiles});
  // We shouldn't add the same profile to the list more than once. Use
  // `insert()` to not corrput the map in case this happens.
  // https://crbug.com/1195784
  DCHECK(it_and_whether_inserted.second);

  MaybeUpdateGuestMode();
  PushProfilesList();
}

void ProfilePickerHandler::RemoveProfileFromListAndPushUpdates(
    const base::FilePath& profile_path) {
  auto remove_it = profiles_order_.find(profile_path);
  // Guest and omitted profiles aren't added to the list.
  // It's possible that a profile gets marked as guest or as omitted after it
  // had been added to the list. In that case, the profile gets removed from the
  // list once in `OnProfileIsOmittedChanged()` but not the second time when
  // `OnProfileWasRemoved()` is called.
  if (remove_it == profiles_order_.end()) {
    return;
  }

  size_t index = remove_it->second;
  profiles_order_.erase(remove_it);
  for (auto& it : profiles_order_) {
    if (it.second > index) {
      --it.second;
    }
  }
  MaybeUpdateGuestMode();
  FireWebUIListener("profile-removed", base::FilePathToValue(profile_path));
}

void ProfilePickerHandler::OnProfileAdded(const base::FilePath& profile_path) {
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  if (entry->IsOmitted()) {
    return;
  }

  AddProfileToListAndPushUpdates(profile_path);
}

void ProfilePickerHandler::OnProfileWasRemoved(
    const base::FilePath& profile_path,
    const std::u16string& profile_name) {
  DCHECK(IsJavascriptAllowed());
  RemoveProfileFromListAndPushUpdates(profile_path);
}

void ProfilePickerHandler::OnProfileIsOmittedChanged(
    const base::FilePath& profile_path) {
  ProfileAttributesEntry* entry =
      g_browser_process->profile_manager()
          ->GetProfileAttributesStorage()
          .GetProfileAttributesWithPath(profile_path);
  CHECK(entry);
  if (entry->IsOmitted()) {
    RemoveProfileFromListAndPushUpdates(profile_path);
  } else {
    AddProfileToListAndPushUpdates(profile_path);
  }
}

void ProfilePickerHandler::OnProfileAvatarChanged(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileHighResAvatarLoaded(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileNameChanged(
    const base::FilePath& profile_path,
    const std::u16string& old_profile_name) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileHostedDomainChanged(
    const base::FilePath& profile_path) {
  PushProfilesList();
}

void ProfilePickerHandler::OnProfileSupervisedUserIdChanged(
    const base::FilePath& profile_path) {
  MaybeUpdateGuestMode();
  PushProfilesList();
}

void ProfilePickerHandler::DidFirstVisuallyNonEmptyPaint() {
  DCHECK(!creation_time_on_startup_.is_null());
  auto now = base::TimeTicks::Now();
  base::UmaHistogramTimes("ProfilePicker.StartupTime.FirstPaint",
                          now - creation_time_on_startup_);
  startup_metric_utils::GetBrowser().RecordExternalStartupMetric(
      "ProfilePicker.StartupTime.FirstPaint.FromApplicationStart", now,
      /*set_non_browser_ui_displayed=*/true);
  // Stop observing so that the histogram is only recorded once.
  Observe(nullptr);
}

void ProfilePickerHandler::OnVisibilityChanged(content::Visibility visibility) {
  // If the profile picker is hidden, the first paint will be delayed until the
  // picker is visible again. Stop monitoring the first paint to avoid polluting
  // the metrics.
  if (visibility != content::Visibility::VISIBLE) {
    Observe(nullptr);
  }
}

void ProfilePickerHandler::MaybeUpdateGuestMode() {
  CHECK(IsJavascriptAllowed());
  FireWebUIListener("guest-mode-availability-updated",
                    base::Value(profiles::IsGuestModeEnabled()));
}

void RecordProfilePickerAction(ProfilePickerAction action) {
  base::UmaHistogramEnumeration("ProfilePicker.UserAction", action);
}
