///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/NavigationModes.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager. 
ViewportActionsHandler::ViewportActionsHandler()
{
	connect(addCommandAction(ACTION_VIEWPORT_MAXIMIZE, tr("Maximize Active Viewport"), ":/core/viewport/maximize_viewport.png", tr("Enlarge/reduce the active viewport.")), SIGNAL(triggered(bool)), this, SLOT(onViewportMaximize()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS, tr("Zoom Scene Extents"), ":/core/viewport/zoom_scene_extents.png", tr("Zoom at the full scene.")), SIGNAL(triggered(bool)), this, SLOT(onViewportZoomSceneExtents()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS, tr("Zoom Selection Extents"), ":/core/viewport/zoom_selection_extents.png", tr("Zoom at the selected object(s).")), SIGNAL(triggered(bool)), this, SLOT(onViewportZoomSelectionExtents()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL, tr("Zoom All Scene Extents"), ":/core/viewport/zoom_all_scene_extents.png", tr("Zoom at the full scene in all viewports.")), SIGNAL(triggered(bool)), this, SLOT(onViewportZoomSceneExtentsAll()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL, tr("Zoom All Selection Extents"), ":/core/viewport/zoom_all_selection_extents.png", tr("Zoom at the selected object(s) in all viewports.")), SIGNAL(triggered(bool)), this, SLOT(onViewportZoomSelectionExtentsAll()));

	addViewportModeAction(ACTION_VIEWPORT_ZOOM, ZoomMode::instance(), tr("Zoom"), ":/core/viewport/mode_zoom.png", tr("Activate zoom mode to zoom into or out of objects."));
	addViewportModeAction(ACTION_VIEWPORT_PAN, PanMode::instance(), tr("Pan"), ":/core/viewport/mode_pan.png", tr("Activate pan mode to shift the region visible in the viewports."));
	addViewportModeAction(ACTION_VIEWPORT_ORBIT, OrbitMode::instance(), tr("Orbit"), ":/core/viewport/mode_orbit.png", tr("Activate orbit mode to rotate the camera around the scene."));
	addViewportModeAction(ACTION_VIEWPORT_FOV, FOVMode::instance(), tr("Field Of View"), ":/core/viewport/mode_fov.png", tr("Activate field of view mode to change the perspective projection."));
	addViewportModeAction(ACTION_VIEWPORT_PICK_ORBIT_CENTER, PickOrbitCenterMode::instance(), tr("Set Orbit Center"), ":/core/viewport/mode_set_orbit_center.png", tr("Set the center of rotation for orbit mode."));
}

/// Creates and registers a single command action.
ActionProxy* ViewportActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath, const QString& statusTip)
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	proxy->setStatusTip(statusTip);
	return proxy;
}

/// Creates and registers a single viewport input mode action.
ViewportModeAction* ViewportActionsHandler::addViewportModeAction(const QString& id, ViewportInputHandler* handler, const QString& title, const char* iconPath, const QString& statusTip)
{
	ViewportModeAction* action = new ViewportModeAction(id, handler, QColor(255, 255, 200));
	ActionProxy* proxy = ACTION_MANAGER.addAction(action, title, iconPath);
	proxy->setStatusTip(statusTip);
	return action;
}

/// Handles ACTION_VIEWPORT_MAXIMIZE command.
void ViewportActionsHandler::onViewportMaximize()
{
	if(VIEWPORT_MANAGER.maximizedViewport())
		VIEWPORT_MANAGER.setMaximizedViewport(NULL);
	else if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.setMaximizedViewport(VIEWPORT_MANAGER.activeViewport());
}

/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS commands.
void ViewportActionsHandler::onViewportZoomSceneExtents()
{
	if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.activeViewport()->zoomToExtents(SceneRenderer::ALL);
}

/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS commands.
void ViewportActionsHandler::onViewportZoomSelectionExtents()
{
	if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.activeViewport()->zoomToExtents(SceneRenderer::SELECTED_OBJECTS);
}

/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL commands.
void ViewportActionsHandler::onViewportZoomSceneExtentsAll()
{
	Q_FOREACH(Viewport* vp, VIEWPORT_MANAGER.viewports())
		vp->zoomToExtents(SceneRenderer::ALL);
}

/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL commands.
void ViewportActionsHandler::onViewportZoomSelectionExtentsAll()
{
	Q_FOREACH(Viewport* vp, VIEWPORT_MANAGER.viewports())
		vp->zoomToExtents(SceneRenderer::SELECTED_OBJECTS);
}

};
