/*
 * Copyright (C) 2005-2019 Centre National d'Etudes Spatiales (CNES)
 *
 * This file is part of Orfeo Toolbox
 *
 *     https://www.orfeo-toolbox.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mvdPixelDescriptionWidget.h"
#include "ui_mvdPixelDescriptionWidget.h"


/*****************************************************************************/
/* INCLUDE SECTION                                                           */

//
// Qt includes (sorted by alphabetic order)
//// Must be included before system/custom includes.

//
// System includes (sorted by alphabetic order)

//
// ITK includes (sorted by alphabetic order)

//
// OTB includes (sorted by alphabetic order)

//
// Monteverdi includes (sorted by alphabetic order)
#include "mvdAlgorithm.h"

namespace mvd
{
/*
  TRANSLATOR mvd::PixelDescriptionWidget

  Necessary for lupdate to be aware of C++ namespaces.

  Context comment for translator.
*/


/*****************************************************************************/
/* CONSTANTS                                                                 */


/*****************************************************************************/
/* STATIC IMPLEMENTATION SECTION                                             */


/*****************************************************************************/
/* CLASS IMPLEMENTATION SECTION                                              */

/*******************************************************************************/
PixelDescriptionWidget::PixelDescriptionWidget(QWidget* p, Qt::WindowFlags flags)
  : QWidget(p, flags),
    m_UI(new mvd::Ui::PixelDescriptionWidget()),
    m_CartographicRootItem(NULL),
    m_GeographicRootItem(NULL),
    m_PixelValueRootItem(NULL),
    m_CartographicItem(NULL),
    m_GeographicItem(NULL)
{
  m_UI->setupUi(this);

  SetupUI();
}

/*******************************************************************************/
PixelDescriptionWidget::~PixelDescriptionWidget()
{
  delete m_UI;
  m_UI = NULL;
}

/*******************************************************************************/
void PixelDescriptionWidget::SetupUI()
{
  //
  // Cartographic coordiantes
  m_CartographicRootItem = new QTreeWidgetItem(GetDescriptionTree());
  // m_CartographicRootItem->setText(0, tr("Cartographic"));
  m_CartographicRootItem->setExpanded(true);

  // m_CartographicItem = new QTreeWidgetItem( m_CartographicRootItem );
  // m_CartographicItem->setText(0, tr("Coordinates"));

  //
  // Geographic coordinates
  m_GeographicRootItem = new QTreeWidgetItem(GetDescriptionTree());
  m_GeographicRootItem->setText(0, tr("Geographic"));
  m_GeographicRootItem->setExpanded(true);

  // m_GeographicItem = new QTreeWidgetItem( m_GeographicRootItem );
  // m_GeographicItem->setText(0, tr("Coordinates"));

  //
  // Child items will be created + updated in a dedicated slot
  m_PixelValueRootItem = new QTreeWidgetItem(GetDescriptionTree());
  m_PixelValueRootItem->setText(0, tr("Pixel Values"));
  m_PixelValueRootItem->setExpanded(true);
}

/*******************************************************************************/
QTreeWidget* PixelDescriptionWidget::GetDescriptionTree()
{
  return m_UI->m_DescriptionTree;
}

/*******************************************************************************/
/* SLOTS                                                                       */
/*******************************************************************************/
void PixelDescriptionWidget::OnCurrentPhysicalUpdated(const QStringList& currentPhysical)
{
  // qDebug() << this << "::OnCurrentPhysicalUpdated(" << currentPhysical << ")";

  // m_CartographicItem->setText(1, currentPhysical);

  if (!currentPhysical.empty())
  {
    // remove the previous QTreeWidgetItem  of m_GeographicRootItem
    while (m_CartographicRootItem->childCount() > 0)
    {
      // Remove QTreeWidgetItem
      QTreeWidgetItem* child = m_CartographicRootItem->takeChild(0);

      // Delete it from memory.
      delete child;
      child = NULL;
    }

    m_CartographicRootItem->setText(0, currentPhysical[0]);

    // fill with the new values
    QTreeWidgetItem* iCartoXItem = new QTreeWidgetItem(m_CartographicRootItem);
    iCartoXItem->setText(0, tr("X"));
    iCartoXItem->setText(1, currentPhysical[1]);

    QTreeWidgetItem* iCartoYItem = new QTreeWidgetItem(m_CartographicRootItem);
    iCartoYItem->setText(0, tr("Y"));
    iCartoYItem->setText(1, currentPhysical[2]);
  }
}

/*******************************************************************************/
void PixelDescriptionWidget::OnCurrentGeographicUpdated(const QStringList& /*const QString &*/ currentGeo)
{
  // qDebug() << this << "::OnCurrentGeographicUpdated(" << currentGeo << ")";

  if (!currentGeo.empty())
  {
    // remove the previous QTreeWidgetItem  of m_GeographicRootItem
    while (m_GeographicRootItem->childCount() > 0)
    {
      // Remove QTreeWidgetItem
      QTreeWidgetItem* child = m_GeographicRootItem->takeChild(0);

      // Delete it from memory.
      delete child;
      child = NULL;
    }

    m_GeographicRootItem->setText(0, currentGeo[0]);

    // fill with the new values
    QTreeWidgetItem* iGeoLongItem = new QTreeWidgetItem(m_GeographicRootItem);
    iGeoLongItem->setText(0, tr("Long"));
    iGeoLongItem->setText(1, currentGeo[1]);

    QTreeWidgetItem* iGeoLatItem = new QTreeWidgetItem(m_GeographicRootItem);
    iGeoLatItem->setText(0, tr("Lat"));
    iGeoLatItem->setText(1, currentGeo[2]);

    QTreeWidgetItem* iGeoElevationItem = new QTreeWidgetItem(m_GeographicRootItem);
    iGeoElevationItem->setText(0, tr("Elevation"));
    iGeoElevationItem->setText(1, currentGeo.size() > 3 ? currentGeo[3] : tr("Not available"));
  }
}

/*******************************************************************************/
void PixelDescriptionWidget::OnCurrentPixelValueUpdated(const VectorImageType::PixelType& currentPixel, const QStringList& bandNames)
{
  // qDebug() << this << "::OnCurrentPixelValueUpdated(" << bandNames << ")";

  if (!bandNames.empty() || currentPixel.GetSize() != 0)
  {
    //
    // remove the previous QTreeWidgetItem  of m_PixelValueRootItem
    while (m_PixelValueRootItem->childCount() > 0)
    {
      // Remove QTreeWidgetItem
      QTreeWidgetItem* child = m_PixelValueRootItem->takeChild(0);

      // Delete it from memory.
      delete child;
      child = NULL;
    }

    // qDebug() << "Bands:" << bandNames;

    // assert( bandNames.size()==currentPixel.GetSize() );

    // fill with the new values
    for (unsigned int idx = 0; idx < currentPixel.GetSize(); idx++)
    {
      QTreeWidgetItem* iBandItem = new QTreeWidgetItem(m_PixelValueRootItem);

      // figure out if a band name is available, if not use Band idx
      if (!bandNames[idx].isEmpty() && static_cast<unsigned int>(bandNames.size()) == currentPixel.GetSize())
      {
        iBandItem->setText(0, bandNames[idx]);
      }
      else
      {
        iBandItem->setText(0, tr("Band %1").arg(idx + 1));
      }
      // set the value
      iBandItem->setText(1, QString("%1").arg(currentPixel.GetElement(idx)));
    }
  }
}

/*******************************************************************************/

} // end namespace 'mvd'
