//*******************************************************************
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Author:  Garrett Potts
//
//*******************************************************************
//  $Id: ossim-info.cpp 15287 2009-08-29 14:01:30Z dburken $

#include <iostream>
#include <vector>
using namespace std;

#include <ossim/ossimConfig.h>

#include <ossim/base/ossimArgumentParser.h>
#include <ossim/base/ossimApplicationUsage.h>
#include <ossim/base/ossimContainerProperty.h>
#include <ossim/base/ossimConstants.h>
#include <ossim/base/ossimDatum.h>
#include <ossim/base/ossimDatumFactory.h>
#include <ossim/base/ossimEllipsoid.h>
#include <ossim/base/ossimFilename.h>
#include <ossim/base/ossimGpt.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimNBandLutDataObject.h>
#include <ossim/base/ossimNotifyContext.h>
#include <ossim/base/ossimObjectFactoryRegistry.h>
#include <ossim/base/ossimRefPtr.h>
#include <ossim/base/ossimString.h>
#include <ossim/base/ossimXmlDocument.h>

#include <ossim/imaging/ossimImageData.h>
#include <ossim/imaging/ossimImageHandler.h>
#include <ossim/imaging/ossimImageHandlerRegistry.h>
#include <ossim/imaging/ossimImageSourceFactory.h>
#include <ossim/imaging/ossimImageWriterFactoryRegistry.h>


#include <ossim/init/ossimInit.h>

#include <ossim/plugin/ossimSharedPluginRegistry.h>

#include <ossim/projection/ossimMapProjection.h>
#include <ossim/projection/ossimMapProjectionInfo.h>
#include <ossim/projection/ossimProjection.h>
#include <ossim/projection/ossimProjectionFactoryRegistry.h>

#include <ossim/support_data/ossimInfoBase.h>
#include <ossim/support_data/ossimInfoFactoryRegistry.h>


//**************************************************************************
// usage()
//**************************************************************************
void usage()
{
   cout << " examples:\n\n"
        << "    ossim-info -i ./myfile.tif\n"
        << "      prints out only general image information\n\n"
        << "    ossim-info -p ./myfile.tif\n"
        << "      prints out only image projection information\n\n"
        << "    ossim-info -p -s wge ./myfile.tif\n"
        << "      prints out only image projection information and shifts to wgs84\n\n"
        << "    ossim-info -p -i ./myfile.tif\n"
        << "      prints out both image and projection information\n\n"
        << "    ossim-info -p -i ./myfile.tif -o ./myfile.geom\n"
        << "      writes geometry file with both image and projection information\n\n"
        << "    ossim-info -p -i ./myfile.tif -v -o ./myfile.geom\n"
        << "      writes geometry file with both image and projection information\n"
        << "      while overwriting exisitng .geom file.\n\n"
        << "    ossim-info -f XML ./myfile.tif\n"
        << "      prints out image and projection information as an XML document\n\n"
        << "    ossim-info -d myfile.ntf\n"
        << "      Dumps all data available, in this case, all nitf tags, from file.\n\n"
        << "    ossim-info -d a.toc\n"
        << "      Dumps all data available, in this case, all nitf and rpf tags, from file.\n\n"
        << "    ossim-info --dno a.toc\n"
        << "      \"dno\" for \"dump no overviws\" Dumps all data available,\n"
        << "       in this case, all nitf and rpf tags, from file ignoring overviews.\n\n"
        << "    ossim-info -d -i -p myfile.ntf\n"
        << "      Typical usage case, i.e. do a dump of tags and print out image and\n"
        << "      projection information.\n\n"
        << endl;
}

// Note be sure to ih->setCurrentEntry before calling.
bool isOverview(const ossimImageHandler* ih)
{
   bool result = false; // Have to prove it.
   if (ih)
   {
      ossimString s = "imag";
      ossimRefPtr<ossimProperty> prop = ih->getProperty(s);
      if (prop.valid())
      {
         ossimString s;
         prop->valueToString(s);
         if (s.toFloat32() < 1.0)
         {
            result = true;
         }
      }
   }
   return result;
}

void outputMetadata(ossimRefPtr<ossimImageHandler> ih,
                    ossimKeywordlist& kwl)
{
   if(ih.valid())
   {
      std::vector< ossimRefPtr< ossimProperty > > list;
      ih->getPropertyList(list);
      std::vector< ossimRefPtr< ossimProperty > >::const_iterator i =
         list.begin();
      while (i != list.end())
      {
         if ( (*i).valid() )
         {
            ossimString key;
            ossimString value;

            // Check for one level of nested container.
            if ((*i)->getClassName() == "ossimContainerProperty")
            {
               ossimContainerProperty* ptr = PTR_CAST(ossimContainerProperty,
                                                      (*i).get());
               if (ptr)
               {
                  std::vector< ossimRefPtr< ossimProperty > > list2;    
                  ptr->getPropertyList(list2);

                  std::vector< ossimRefPtr< ossimProperty > >::const_iterator
                     i2 = list2.begin();
                  while (i2 != list2.end())
                  {
                     key   = (*i2)->getName();
                     value = (*i2)->valueToString();
                     kwl.add(key.c_str(), value.c_str(), true);
                     ++i2;
                  }
               }
            }
            else // Not a container.
            {
               key   = (*i)->getName();
               value = (*i)->valueToString();
               kwl.add(key.c_str(), value.c_str(), true);
            }
         }
         ++i;
      }
   }
}
   
void outputGeometryEntry(ossimRefPtr<ossimImageHandler> ih,
                         ossimKeywordlist& kwl,
                         const ossimString& prefix,
                         const ossimString& datumCode="")
{
   const ossimDatum* datum = NULL;
   if(datumCode != "")
   {
      datum = ossimDatumFactory::instance()->create(datumCode.c_str());
   }
   ossimKeywordlist kwl2;
   ossimDrect outputRect = ih->getBoundingRect();
   
   ih->getImageGeometry(kwl2);
   
   ossimProjection* imageProj
      = ossimProjectionFactoryRegistry::instance()->createProjection(kwl2);
   
   if(imageProj)
   {
      imageProj->saveState(kwl,
                           prefix);
      
      ossimGpt ulg;
      ossimGpt llg;
      ossimGpt lrg;
      ossimGpt urg;
      imageProj->lineSampleToWorld(outputRect.ul(), ulg);
      imageProj->lineSampleToWorld(outputRect.ll(), llg);
      imageProj->lineSampleToWorld(outputRect.lr(), lrg);
      imageProj->lineSampleToWorld(outputRect.ur(), urg);
      
      if(datum)
      {
         ulg.changeDatum(datum);
         llg.changeDatum(datum);
         lrg.changeDatum(datum);
         urg.changeDatum(datum);
      }
      kwl.add(prefix, "ul_lat", ulg.latd(), true);
      kwl.add(prefix, "ul_lon", ulg.lond(), true);
      kwl.add(prefix, "ll_lat", llg.latd(), true);
      kwl.add(prefix, "ll_lon", llg.lond(), true);
      kwl.add(prefix, "lr_lat", lrg.latd(), true);
      kwl.add(prefix, "lr_lon", lrg.lond(), true);
      kwl.add(prefix, "ur_lat", urg.latd(), true);
      kwl.add(prefix, "ur_lon", urg.lond(), true);

      // ESH 01/2009:  We've brought back the following from
      // OSSIM 1.666 which is needed by EW 4.4.
      if(!kwl.find(ossimKeywordNames::TIE_POINT_LAT_KW))
      {
         kwl.add(prefix, ossimKeywordNames::TIE_POINT_LAT_KW,
                 ulg.latd(), true);
         kwl.add(prefix, ossimKeywordNames::TIE_POINT_LON_KW,
                 ulg.lond(), true);
         // ESH 2/08 -- prevent bad values
         if ( outputRect.height()-1.0 > DBL_EPSILON )
         {
            kwl.add(prefix, ossimKeywordNames::DECIMAL_DEGREES_PER_PIXEL_LAT,
                    fabs(ulg.latd()-llg.latd())/(outputRect.height()-1.0),
                    true);
         }
         // ESH 2/08 -- prevent bad values
         if ( outputRect.width()-1.0 > DBL_EPSILON )
         {
            kwl.add(prefix, ossimKeywordNames::DECIMAL_DEGREES_PER_PIXEL_LON,
                    fabs(ulg.lond()-urg.lond())/(outputRect.width()-1.0),
                    true);
         }
      }
      ossimDpt gsd = imageProj->getMetersPerPixel();
      kwl.add(prefix,
              ossimKeywordNames::METERS_PER_PIXEL_X_KW,
              gsd.x,
              true);
      kwl.add(prefix,
              ossimKeywordNames::METERS_PER_PIXEL_Y_KW,
              gsd.y,
              true);
      // ESH 01/2009: End of EW 4.4 hack
      
   }
   else
   {
      cerr << "No projection geometry for file " << ih->getFilename() << endl;
   }
}

//**************************************************************************
// outputGeometry
//**************************************************************************
void outputGeometry(ossimRefPtr<ossimImageHandler> ih,
                    ossimKeywordlist& kwl,
                    const ossimString& datumCode="",
                    bool dnoFlag=false)
{
   if(ih.valid())
   {
      ossim_uint32 numEntries = 0;
      
      ossim_uint32 entryIdx = 0;
      std::vector<ossim_uint32> entryList;
      ih->getEntryList(entryList);
      for(entryIdx = 0; entryIdx < ih->getNumberOfEntries();++entryIdx)
      {
         ih->setCurrentEntry(entryList[entryIdx]);

         bool outputEntry = true;
         if (dnoFlag) // caller doesn't want overviews.
         {
            if ( isOverview( ih.get()) )
            {
               if (numEntries > 0)
               {
                  --numEntries;
               }
               outputEntry = false;
            }
         }

         if (outputEntry)
         {
            ++numEntries;
            ossimString prefix = "image";
            prefix = prefix + ossimString::toString(entryIdx) + ".";
            kwl.add(prefix,
                    ossimKeywordNames::ENTRY_KW,
                    entryList[entryIdx],
                    true);
            outputGeometryEntry(ih, kwl, prefix, datumCode);
         }
      }

      kwl.add(ossimKeywordNames::NUMBER_ENTRIES_KW, numEntries, true);
   }
}

void outputGeneralImageInfo(ossimRefPtr<ossimImageHandler> ih,
                            ossimKeywordlist& kwl,
                            const ossimString& prefix)
{
   ossimDrect boundingRect = ih->getBoundingRect();
   int i = 0;
   kwl.add(prefix,ossimKeywordNames::UL_X_KW, boundingRect.ul().x, true);
   kwl.add(prefix,ossimKeywordNames::UL_Y_KW, boundingRect.ul().y, true);
   kwl.add(prefix,ossimKeywordNames::LR_X_KW, boundingRect.lr().x, true);
   kwl.add(prefix,ossimKeywordNames::LR_Y_KW, boundingRect.lr().y, true);
   
   kwl.add(prefix,ossimKeywordNames::NUMBER_INPUT_BANDS_KW,
           ih->getNumberOfInputBands(), true);
   kwl.add(prefix,ossimKeywordNames::NUMBER_OUTPUT_BANDS_KW,
           ih->getNumberOfOutputBands(), true);
   kwl.add(prefix,ossimKeywordNames::NUMBER_LINES_KW,
           boundingRect.height(), true);
   kwl.add(prefix,ossimKeywordNames::NUMBER_SAMPLES_KW,
           boundingRect.width(), true);
   
   ossimScalarType scalar = ih->getOutputScalarType();
   
   for(i = 0; i < (int)ih->getNumberOfInputBands(); ++i)
   {
      ossimString band = ossimString("band") + ossimString::toString(i) + ".";
      
      kwl.add(prefix,band+"null_value",
              ih->getNullPixelValue(i),
              true);
      kwl.add(prefix,band+"min_value",
              ih->getMinPixelValue(i),
              true);
      kwl.add(prefix,band+"max_value",
              ih->getMaxPixelValue(i),
              true);
   }
   
   // Output Radiometry.
   switch(scalar)
   {
      case OSSIM_UINT8:
      {
         kwl.add(prefix,"radiometry", "8-bit", true);
         break;
      }
      case OSSIM_USHORT11:
      {
         kwl.add(prefix,"radiometry", "11-bit", true);
         break;
      }
      case OSSIM_UINT16:
      {
         kwl.add(prefix,"radiometry", "16-bit unsigned", true);
         break;
      }
      case OSSIM_SINT16:
      {
         kwl.add(prefix,"radiometry", "16-bit signed", true);
         break;
      }
      case OSSIM_UINT32:
      {
         kwl.add(prefix,"radiometry", "32-bit unsigned", true);
         break;
      }
      case OSSIM_FLOAT32:
      {
         kwl.add(prefix,"radiometry", "float", true);
         break;
      }
      case OSSIM_NORMALIZED_FLOAT:
      {
         kwl.add(prefix,"radiometry", "normalized float", true);
         break;
      }
      default:
      {
         kwl.add(prefix,"radiometry", "unknown", true);
         break;
      }
   }

   // ESH 06/2009 -- Add in number of decimation levels
   kwl.add(prefix,"number_decimation_levels",
           ih->getNumberOfDecimationLevels(), true);
}


//**************************************************************************
// outputGeneralImageInfo
//**************************************************************************
ossimKeywordlist outputGeneralImageInfo(ossimRefPtr<ossimImageHandler> ih,
                                        ossimKeywordlist& kwl,
                                        bool dnoFlag)
{
   // If ImageHandler exists, output image information.
   if(ih.valid())
   {
      ossim_uint32 numEntries = 0;
      
      ossim_uint32 entryIdx = 0;
      std::vector<ossim_uint32> entryList;
      ih->getEntryList(entryList);
      for(entryIdx = 0; entryIdx < ih->getNumberOfEntries();++entryIdx)
      {
         ih->setCurrentEntry(entryList[entryIdx]);

         bool outputEntry = true;
         if (dnoFlag) // caller doesn't want overviews.
         {
            if ( isOverview( ih.get()) )
            {
               outputEntry = false;
            }
         }

         if (outputEntry)
         {
            ++numEntries;
            ossimString prefix = "image";
            prefix = prefix + ossimString::toString(entryIdx) + ".";
            kwl.add(prefix,
                    ossimKeywordNames::ENTRY_KW,
                    entryList[entryIdx],
                    true);
            outputGeneralImageInfo(ih, kwl, prefix);
         }
      }
      
      kwl.add(ossimKeywordNames::NUMBER_ENTRIES_KW, numEntries, true);
   }

   return kwl;
}

void writeToOssimGeom(const ossimKeywordlist& okwl, const ossimFilename& ofile)
{
   ossimString prefix = "image";
   const char* type = okwl.find("image0.", "type");
   unsigned int idx = 0;
   ossimFilename f = ofile;
   ossimString ext = f.ext();
   ossimFilename baseFile = f;
   baseFile = baseFile.setExtension("");
   prefix = ossimString("image0");
   ossimString strIdx = "0";
   
   while(type)
   {
      ossimKeywordlist temp;
      temp.add(okwl,
               prefix+".",
               true);
      ossimFilename outputFile = (baseFile + "_e" + strIdx + ".geom");
      temp.write(outputFile.c_str());
      ++idx;
      strIdx = ossimString::toString(idx);
      prefix = ossimString("image");
      type = okwl.find(prefix+strIdx+".", "type");
   }
}

void outputColorPalette(ossimRefPtr<ossimImageHandler> ih,
                        ossimKeywordlist& kwl)
{
   // If ImageHandler exists, output image information.
   if(ih.valid())
   {
      if(ih->getLut().valid())
      {
         ossim_uint32 entryIdx = 0;
         std::vector<ossim_uint32> entryList;
         ih->getEntryList(entryList);
         for(entryIdx = 0; entryIdx < ih->getNumberOfEntries();++entryIdx)
         {
            ih->setCurrentEntry(entryList[entryIdx]);
            ossimString prefix = "image";
            prefix = prefix + ossimString::toString(entryIdx) + ".lut.";
            if(ih->getLut().valid())
            {
               ih->getLut()->saveState(kwl, prefix);
            }
         }
      }
   }
}

void printDatums()
{
   std::vector<ossimString> datumList =
      ossimDatumFactory::instance()->getList();
   
   std::vector<ossimString>::const_iterator i;
   
   for (i = datumList.begin(); i != datumList.end(); ++i)
   {
      const ossimDatum* datum = ossimDatumFactory::instance()->create(*i);
      
      if (datum)
      {
         cout << setiosflags(ios::left)
              << setw(7)
              << datum->code().c_str()
              << setw(33)
              << datum->name().c_str()
              << setw(10) 
              << "Ellipse:"
              << datum->ellipsoid()->name()
              << endl;
      }
   }
}

void printFactories(const ossimString& factorTypeName,
                    const ossimString& factoryObjectName,
                    bool factoryOutputKeywordListFlag)
{
   vector<ossimString> typeList;
   if(factoryObjectName =="")
   {
      ossimObjectFactoryRegistry::instance()->getTypeNameList(typeList,
                                                              factorTypeName);
   }
   else
   {
      typeList.push_back(factoryObjectName);
   }
   
   for(ossim_uint32 i = 0; i < typeList.size(); ++i)
   {
      if(factoryOutputKeywordListFlag)
      {
         ossimObject* obj = ossimObjectFactoryRegistry::instance()->createObject(typeList[i]);
         if(obj)
         {
            cout << typeList[i] << endl;
            cout << "______________________________________________________" << endl;
            ossimKeywordlist kwl;
            obj->saveState(kwl);
            cout << kwl << endl;
            cout << "______________________________________________________" << endl;
            delete obj;
         }
      }
      else
      {
         cout << typeList[i] << endl;
      }
   }
}

void printPlugins()
{
   if(ossimSharedPluginRegistry::instance()->getNumberOfPlugins() > 0)
   {
      ossimSharedPluginRegistry::instance()->
         printAllPluginInformation(std::cout);
   }
   else
   {
      std::cout << "No plugins loaded in the OSSIM core library" << std::endl;
   }
}

void ft2mtrs(const ossimString& ftString, bool use_us_survey)
{
   double feet = ftString.toDouble();
   double meters = 0.0;
   std::string conversionString;
   if (use_us_survey)
   {
      meters = feet * US_METERS_PER_FT;
      conversionString = "0.3048006096";
   }
   else
   {
      meters = feet * MTRS_PER_FT;
      conversionString = "0.3048";
   }
   
   cout << setiosflags(ios::fixed) << setprecision(15)
        << feet << " * " << conversionString << " = "
        << meters << " meters." << endl;
}

void mtrs2ft(const ossimString& mtrsString, bool use_us_survey)
{
   double meters = mtrsString.toDouble();
   double feet = 0.0;
   std::string conversionString;

   if (use_us_survey)
   {
      feet = meters / US_METERS_PER_FT;
      conversionString = "0.3048006096";
   }
   else
   {
      feet = meters / MTRS_PER_FT;
      conversionString = "0.3048";
   }

   cout << setiosflags(ios::fixed) << setprecision(15)
        << meters << " / " << conversionString << " = "
        << feet << " feet." << endl;   
}

void deg2rad(const ossimString& degString)
{
   double degrees = degString.toDouble();
   double radians = degrees * RAD_PER_DEG;
   
   std::cout << std::setiosflags(std::ios::fixed) << std::setprecision(15)
             << "\n" << degrees << " degrees = "
             << radians << " radians.\n" << std::endl;
}

void rad2deg(const ossimString& radString)
{
   double radians = radString.toDouble();
   double degrees = radians * DEG_PER_RAD;
   
   std::cout << std::setiosflags(std::ios::fixed) << std::setprecision(15)
             << "\n" << radians << " radians = "
             << degrees << " degrees.\n" << std::endl;
}

void mtrsPerDeg(const ossimString& latitudeString)
{
   ossim_float64 lat = latitudeString.toDouble();
   ossim_float64 lon = 0.0;
   
   ossimGpt gpt(lat, lon);
   
   ossimDpt      mpd          = gpt.metersPerDegree();  
   ossim_float64 radius       = gpt.datum()->ellipsoid()->geodeticRadius(lat);
   ossim_float64 arcLengthLat = mpd.y/60.0;
   ossim_float64 arcLengthLon = mpd.x/60.0;

   cout << setiosflags(ios::fixed) << setprecision(15)
        << "Meters per degree and minute at latitude of " << lat << ":\n"
        << "Meters per degree latitude:   "
        << setw(20) << mpd.y << endl
        << "Meters per degree longitude:  "
        << setw(20) << mpd.x << endl
        << "Meters per minute latitude:   "
        << setw(20) << arcLengthLat << endl
        << "Meters per minute longitude:  "
        << setw(20) << arcLengthLon << endl
        << "Geodetic radius:              "
        << setw(20) << radius << endl
        << endl;
}

void testPlugin(const ossimFilename& plugin)
{
   if( ossimSharedPluginRegistry::instance()->registerPlugin(plugin.expand()) )
   {
      std::cout << "Plugin loaded: " << plugin << std::endl;
   }
   else
   {
      std::cout << "Unable to load plugin: " << plugin << std::endl;
   }
}
   

//**************************************************************************
// Main Application
//**************************************************************************
int main(int argc, char *argv[])
{
   std::string tempString;
   ossimArgumentParser::ossimParameter stringParam(tempString);
   ossimArgumentParser argumentParser(&argc, argv);
   ossimInit::instance()->addOptions(argumentParser);
   ossimInit::instance()->initialize(argumentParser);

   argumentParser.getApplicationUsage()->
      setApplicationName(argumentParser.getApplicationName());

   argumentParser.getApplicationUsage()->
      setDescription(argumentParser.getApplicationName()+
                     " displays image and geometry information.");

   argumentParser.getApplicationUsage()->
      setCommandLineUsage(argumentParser.getApplicationName()+
                          "ossim-info [-i] [-p] <full path to file> [-v] [-o <geom file>]");
                          
   argumentParser.getApplicationUsage()->
      addCommandLineOption("-h", "Display this information");
   
   argumentParser.getApplicationUsage()->
      addCommandLineOption("-i",
                           "Will print out the general image information.");

   argumentParser.getApplicationUsage()->
      addCommandLineOption("-m",
                           "Will print out meta data information.");

   argumentParser.getApplicationUsage()->
      addCommandLineOption("-p", "Will print out the projection information.");

   argumentParser.getApplicationUsage()->
      addCommandLineOption("--palette", "Will print out the color palette if one exists.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("-o", "Will output the information to the file specified.  Default is to standard out.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("-v", "Overwrite existing geometry.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("-s", "Force the ground rect to be the specified datum");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--ogeom-format", "Formats the data to an ossim gemetry file, seperate files for multi image file formats");
     
  argumentParser.getApplicationUsage()->
 	addCommandLineOption("-f", "Will output the information specified format [KWL | XML].  Default is KWL.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("-d", "A generic dump if one is available.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--dno", "A generic dump if one is available.  This option ignores overviews.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--datums", "Prints datum list.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--factories", "Prints factory list.");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--factory-type", "The base class you wish to query. No argument defaults to all registered type\n");

  argumentParser.getApplicationUsage()->
     addCommandLineOption("--factory-object", "object type to instantiate");
  
  argumentParser.getApplicationUsage()->
     addCommandLineOption("--factory-keyword-list", "Optional, will output the keyword list from a saveState on each object\n");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--plugins", "Prints plugin list.");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--plugin-test", "Test plugin passed to option.");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--ft2mtrs", "Gives meters from feet (0.3048 meters per foot).");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--ft2mtrs-us-survey",
     "Gives meters from feet (0.3048006096 meters per foot).");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--mtrs2ft", "Gives feet from meters (0.3048 meters per foot).");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--mtrs2ft-us-survey",
     "Gives feet from meters (0.3048006096 meters per foot).");
  
  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--deg2rad", "Gives radians from degrees.");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--rad2deg", "Gives degrees from radians.");

  argumentParser.getApplicationUsage()->addCommandLineOption(
     "--mtrsPerDeg",
     "Gives meters per degree and meters per minute for a given latitude.");
  
   bool theImageGeometryOption = false;
   bool theImageInformationOption = false;
   bool theOutputOption = false;
   bool theFileOverwriteFlag = false;
   bool theFormatToOssimGeomFlag = false;
   bool thePaletteOption = false;
   bool dumpOption = false;
   bool dumpNoOverviewsOption = false;
   ossimString datumCode;
   ossimString formatCode = "KWL";
   ossimFilename oFile = "";
   ossimString  factorTypeName = "";
   ossimString  factoryObjectName = "";
   bool         factoryOutputKeywordListFlag = false;
   bool         dumpFactories = false;

   bool metadataOption = false;

   // check for # of args or -h
   if (argumentParser.read("-h") || argumentParser.read("--help")||(argc==1))
   {
      argumentParser.getApplicationUsage()->write(std::cout);
      usage();
      return 0;
   }
   if(argumentParser.read("--palette"))
   {
      thePaletteOption = true;
   }
   if(argumentParser.read("--ogeom-format"))
   {
      theFormatToOssimGeomFlag = true;
   }
   if(argumentParser.read("-p"))
   {
      theImageGeometryOption = true;
   }
   if(argumentParser.read("-i"))
   {
      theImageInformationOption = true;
   }
   if(argumentParser.read("-m"))
   {
      metadataOption = true;
   }
   if(argumentParser.read("-v"))
   {
      theFileOverwriteFlag = true;
   }
   if(argumentParser.read("-o", stringParam))
   {
      theOutputOption = true;
      oFile = ossimFilename(tempString); 
   }
   if(argumentParser.read("-s", stringParam))
   {
      datumCode = tempString;
      datumCode = datumCode.upcase();
   }
   if(argumentParser.read("-f", stringParam))
   {
      formatCode = tempString;
      formatCode = formatCode.upcase();
   }   
   if (argumentParser.read("-d"))
   {
      dumpOption = true;
   }
   if (argumentParser.read("--dno"))
   {
      dumpNoOverviewsOption = true;
   }
   if (argumentParser.read("--datums"))
   {
      printDatums();
      if (argc == 1)
      {
         return 0;
      }
   }
   
   while( argumentParser.read("--factory-type", stringParam) )
   {
      factorTypeName = tempString;
      dumpFactories = true;
   }
   if(argumentParser.read("--factory-keyword-list"))
   {
      factoryOutputKeywordListFlag = true;
      dumpFactories = true;
   }
   while( argumentParser.read("--factory-object", stringParam) )
   {
      factoryObjectName = tempString;
      dumpFactories = true;
   }
   if ( argumentParser.read("--factories") )
   {
      dumpFactories = true;
   }

   if ( dumpFactories )
   {
      printFactories(factorTypeName,
                     factoryObjectName,
                     factoryOutputKeywordListFlag);
      if (argc == 1)
      {
         return 0;
      }
   }

   if (argumentParser.read("--plugins"))
   {
      printPlugins();
      if (argc == 1)
      {
         return 0;
      }
   }

   if ( argumentParser.read("--plugin-test", stringParam) )
   {
      testPlugin(ossimFilename(tempString));
      if (argc == 1)
      {
         return 0;
      }
   }

   if ( argumentParser.read("--ft2mtrs", stringParam) )
   {
      ft2mtrs(tempString, false);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--ft2mtrs-us-survey", stringParam) )
   {
      ft2mtrs(tempString, true);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--mtrs2ft", stringParam) )
   {
      mtrs2ft(tempString, false);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--mtrs2ft-us-survey", stringParam) )
   {
      mtrs2ft(tempString, true);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--deg2rad", stringParam) )
   {
      deg2rad(tempString);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--rad2deg", stringParam) )
   {
      rad2deg(tempString);
      if (argc == 1)
      {
         return 0;
      }
   }
   
   if ( argumentParser.read("--mtrsPerDeg", stringParam) )
   {
      mtrsPerDeg(tempString);
      if (argc == 1)
      {
         return 0;
      }
   }
   if ( argumentParser.read("--mtrsPerDeg") )
   {
      cout << argv[0] << "--mtrsPerDeg <latitude>" << endl;
      if (argc == 1)
      {
         return 0;
      }
   }

   // Default to -i -p if no options are used...
   if( !theImageGeometryOption && !theImageInformationOption &&
       !dumpOption && !dumpNoOverviewsOption)
   {
      theImageGeometryOption    = true;
      theImageInformationOption = true;
   }

   //---
   // One of the dump options picked.
   // 
   // NOTE:  The ossimInfo's don't necessarily need an image handler so we
   // put this code block before the image handler check which will return
   // false if it can't get one.
   //---     
   if (dumpOption || dumpNoOverviewsOption)
   {
      ossimInfoBase* info = ossimInfoFactoryRegistry::instance()->
         create(ossimFilename(argv[argc - 1]));
      if (info)
      {
         if (dumpNoOverviewsOption)
         {
            info->setProcessOverviewFlag(false);
         }
         info->print(std::cout);
         delete info;
         info = 0;
      }
      else
      {
         cout << "No dump available for:  " << argv[argc - 1] << endl;
      }
   }

   // Open up an image handler.
   ossimRefPtr<ossimImageHandler> ih
      = (ossimImageHandler*)ossimImageHandlerRegistry::instance()
      ->open(ossimFilename(argv[argc - 1]));
   if (!ih)
   {
      cerr << "ossim-info:main Could not open:  "
           << argv[argc - 1] << endl;
      return 1;
   }

   // Output keyword list.
   ossimKeywordlist okwl;

   // Image Geometry
   if(theImageGeometryOption)
   {
      ossimKeywordlist geom_kwl; 
      outputGeometry(ih, geom_kwl, datumCode, dumpNoOverviewsOption);
      okwl.addList(geom_kwl, true);
   }
   
   // Image Information
   if(theImageInformationOption)
   {
      ossimKeywordlist img_kwl;
      outputGeneralImageInfo(ih, img_kwl, dumpNoOverviewsOption);
      okwl.addList(img_kwl, true);
      
   }

   // Palette info
   if (thePaletteOption)
   {
      ossimKeywordlist palette_kwl;
      outputColorPalette(ih, palette_kwl);
      okwl.addList(palette_kwl, true);
   }

   // metadata info
   if (metadataOption)
   {
      ossimKeywordlist metadata_kwl;
      outputMetadata(ih, metadata_kwl);
      okwl.addList(metadata_kwl, true);
   }

   if(theOutputOption)
   {
      // check to make sure geometry doesn't already exist
      if(oFile.exists() && !theFileOverwriteFlag)
      {
         cout << "ERROR:  Geometry already exists - " << oFile << endl;
         return 1;
      }

      
      cout << "\nWriting geometry file: " << oFile << endl;

      if(theFormatToOssimGeomFlag)
      {
         writeToOssimGeom(okwl, oFile);
      }
      else
      {
         okwl.write(oFile);
      }
   }

   if ( formatCode == "XML" )
   {
	   ossimXmlDocument document;
	   
	   document.fromKwl( okwl );
	   cout << document << std::endl;
   }
   else 
   {
	   cout << okwl << std::endl;
   }
   
// Output to screen if not writing.
//    if(theImageGeometryOption && !theOutputOption)
//       cout << geom_kwl << endl;
   
//    if(theImageInformationOption && !theOutputOption)
//       cout << img_kwl << endl;
      
   exit(0);
}
