// license:BSD-3-Clause
// copyright-holders:Phil Stroffolino
/*
Konami "Twin16"

driver by Phil Stroffolino

CPU   : 68000(10M)x2 Z-80
SOUND : YM2151 007232 uPD7759C
OSC.  : 3.579545MHz 18432.00KHz

Main processors are a pair of 68000 CPUs
Sounds are generated by a Z80, a Yamaha 2151 and 3012, a Konami custom IC and a uPD7759C

    Dark Adventure / Devil World / Majuu no Ohkoku
    Vulcan Venture / Gradius II
    Cuebrick
    MIA (Japan)
    Final Round / Hard Puncher (Japan)

68000 Memory Map for Konami Twin System

    CPUA                                        CPUB
    0x000000..0x03ffff  ROM                     0x000000..0x03ffff
    0x040000..0x043fff  communication RAM       0x040000..0x043fff (shared)
    0x060000..0x063fff  work RAM                0x060000..0x063fff
    0x080000..0x080fff  palette
    0x080000..0x09ffff  ROM (extra tile data)
    0x0a0000..0x0a0001  IRQ control             0x0a0000..0x0a0001
    0x0a0008..0x0a0009  sound command
    0x0a0010..0xa00011  watchdog
    0x0c0000..0x0c0001  screenflip
    0x0c0002..0x0c000f  scroll registers

    0x100000..0x103fff  FIXRAM (text layer)
    0x120000..0x123fff  VIDRAM (tilemaps)       0x480000..0x483fff (shared)
    0x140000..0x143fff  OBJRAM (sprites)        0x400000..0x403fff (shared)
                        ZIP RAM (tiles)         0x500000..0x53ffff
                        gfx ROM (banked)        0x600000..0x77ffff
                        sprite gfx RAM          0x780000..0x79ffff

Known Issues:
- repeated uPD7759C samples in fround, disconnecting reset helps but doesn't fix it
- see video/twin16.c for graphics related issues

*/

#include "emu.h"
#include "includes/twin16.h"
#include "includes/konamipt.h"

#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"
#include "machine/gen_latch.h"
#include "machine/nvram.h"
#include "machine/watchdog.h"
#include "sound/ym2151.h"
#include "speaker.h"


#define CPUA_IRQ_ENABLE (m_CPUA_register & 0x20)
#define CPUB_IRQ_ENABLE (m_CPUB_register & 0x02)




int twin16_state::spriteram_process_enable(  )
{
	return (m_CPUA_register & 0x40) == 0;
}

/******************************************************************************************/

/* Read/Write Handlers */

WRITE16_MEMBER(twin16_state::CPUA_register_w)
{
	/*
	7   6   5   4   3   2   1   0
	    X                           sprite processing disable
	        X                       IRQ5 enable (CPUA)
	            X                   0->1 trigger IRQ6 on CPUB
	                X               0->1 trigger IRQ on sound CPU
	                    x   x   x   coin counters
	*/
	int old = m_CPUA_register;
	COMBINE_DATA(&m_CPUA_register);

	if (m_CPUA_register != old)
	{
		int rising_edge = ~old & m_CPUA_register;
		int falling_edge = old & ~m_CPUA_register;

		if (rising_edge & 0x08)
			m_audiocpu->set_input_line_and_vector(0, HOLD_LINE, 0xff);

		if (falling_edge & 0x40)
			spriteram_process();

		if (rising_edge & 0x10)
			m_subcpu->set_input_line(M68K_IRQ_6, HOLD_LINE);

		machine().bookkeeping().coin_counter_w(0, m_CPUA_register & 0x01);
		machine().bookkeeping().coin_counter_w(1, m_CPUA_register & 0x02);
		machine().bookkeeping().coin_counter_w(2, m_CPUA_register & 0x04);
	}
}

WRITE16_MEMBER(twin16_state::CPUB_register_w)
{
	/*
	7   6   5   4   3   2   1   0
	                    X           gfx bank select
	                        X       IRQ5 enable
	                            X   0->1 trigger IRQ6 on CPUA
	*/
	int old = m_CPUB_register;
	COMBINE_DATA(&m_CPUB_register);

	if (m_CPUB_register != old)
	{
		int rising_edge = ~old & m_CPUB_register;
		if (rising_edge & 0x01)
			m_maincpu->set_input_line(M68K_IRQ_6, HOLD_LINE);

		m_gfxrombank->set_entry(BIT(m_CPUB_register, 2));
	}
}

WRITE16_MEMBER(fround_state::fround_CPU_register_w)
{
	/*
	7   6   5   4   3   2   1   0
	        X                       IRQ5 enable
	                X               0->1 trigger IRQ on sound CPU
	                        x   x   coin counters
	*/
	uint16_t old = m_CPUA_register;
	COMBINE_DATA(&m_CPUA_register);
	if (m_CPUA_register != old)
	{
		if ((old & 0x08) == 0 && (m_CPUA_register & 0x08))
			m_audiocpu->set_input_line_and_vector(0, HOLD_LINE, 0xff);

		machine().bookkeeping().coin_counter_w(0, m_CPUA_register & 0x01);
		machine().bookkeeping().coin_counter_w(1, m_CPUA_register & 0x02);
	}
}

READ8_MEMBER(twin16_state::upd_busy_r)
{
	return m_upd7759->busy_r();
}

WRITE8_MEMBER(twin16_state::upd_reset_w)
{
	m_upd7759->reset_w(data & 2);
}

WRITE8_MEMBER(twin16_state::upd_start_w)
{
	m_upd7759->start_w(data & 1);
}

/* Memory Maps */

ADDRESS_MAP_START(twin16_state::sound_map)
	AM_RANGE(0x0000, 0x7fff) AM_ROM
	AM_RANGE(0x8000, 0x87ff) AM_RAM
	AM_RANGE(0x9000, 0x9000) AM_WRITE(upd_reset_w)
	AM_RANGE(0xa000, 0xa000) AM_DEVREAD("soundlatch", generic_latch_8_device, read)
	AM_RANGE(0xb000, 0xb00d) AM_DEVREADWRITE("k007232", k007232_device, read, write)
	AM_RANGE(0xc000, 0xc001) AM_DEVREADWRITE("ymsnd", ym2151_device, read, write)
	AM_RANGE(0xd000, 0xd000) AM_DEVWRITE("upd", upd7759_device, port_w)
	AM_RANGE(0xe000, 0xe000) AM_WRITE(upd_start_w)
	AM_RANGE(0xf000, 0xf000) AM_READ(upd_busy_r) // miaj writes 0 to it
	ADDRESS_MAP_END

ADDRESS_MAP_START(twin16_state::main_map)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x040000, 0x043fff) AM_RAM AM_SHARE("comram")
//  AM_RANGE(0x044000, 0x04ffff) AM_NOP             // miaj
	AM_RANGE(0x060000, 0x063fff) AM_RAM
	AM_RANGE(0x080000, 0x080fff) AM_DEVREADWRITE8("palette", palette_device, read8, write8, 0x00ff) AM_SHARE("palette")
	AM_RANGE(0x081000, 0x081fff) AM_WRITENOP
	AM_RANGE(0x0a0000, 0x0a0001) AM_READ_PORT("SYSTEM") AM_WRITE(CPUA_register_w)
	AM_RANGE(0x0a0002, 0x0a0003) AM_READ_PORT("P1")
	AM_RANGE(0x0a0004, 0x0a0005) AM_READ_PORT("P2")
	AM_RANGE(0x0a0006, 0x0a0007) AM_READ_PORT("P3")
	AM_RANGE(0x0a0008, 0x0a0009) AM_DEVWRITE8("soundlatch", generic_latch_8_device, write, 0x00ff)
	AM_RANGE(0x0a0010, 0x0a0011) AM_READ_PORT("DSW2") AM_DEVWRITE("watchdog", watchdog_timer_device, reset16_w)
	AM_RANGE(0x0a0012, 0x0a0013) AM_READ_PORT("DSW1")
	AM_RANGE(0x0a0018, 0x0a0019) AM_READ_PORT("DSW3")
	AM_RANGE(0x0c0000, 0x0c000f) AM_WRITE(video_register_w)
	AM_RANGE(0x0c000e, 0x0c000f) AM_READ(sprite_status_r)
	AM_RANGE(0x100000, 0x103fff) AM_RAM_WRITE(fixram_w) AM_SHARE("fixram")
//  AM_RANGE(0x104000, 0x105fff) AM_NOP             // miaj
	AM_RANGE(0x120000, 0x121fff) AM_RAM_WRITE(videoram0_w) AM_SHARE("videoram.0")
	AM_RANGE(0x122000, 0x123fff) AM_RAM_WRITE(videoram1_w) AM_SHARE("videoram.1")
	AM_RANGE(0x140000, 0x143fff) AM_RAM AM_SHARE("spriteram")
ADDRESS_MAP_END

ADDRESS_MAP_START(twin16_state::sub_map)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x040000, 0x043fff) AM_RAM AM_SHARE("comram")
//  AM_RANGE(0x044000, 0x04ffff) AM_NOP             // miaj
	AM_RANGE(0x060000, 0x063fff) AM_RAM
	AM_RANGE(0x080000, 0x09ffff) AM_ROM AM_REGION("data", 0)
	AM_RANGE(0x0a0000, 0x0a0001) AM_WRITE(CPUB_register_w)
	AM_RANGE(0x400000, 0x403fff) AM_RAM AM_SHARE("spriteram")
	AM_RANGE(0x480000, 0x481fff) AM_RAM_WRITE(videoram0_w) AM_SHARE("videoram.0")
	AM_RANGE(0x482000, 0x483fff) AM_RAM_WRITE(videoram1_w) AM_SHARE("videoram.1")
	AM_RANGE(0x500000, 0x53ffff) AM_RAM_WRITE(zipram_w) AM_SHARE("zipram")
	AM_RANGE(0x600000, 0x6fffff) AM_ROM AM_REGION("gfxrom", 0)
	AM_RANGE(0x700000, 0x77ffff) AM_ROMBANK("gfxrombank")
	AM_RANGE(0x780000, 0x79ffff) AM_RAM AM_SHARE("sprite_gfx_ram")
ADDRESS_MAP_END

ADDRESS_MAP_START(fround_state::fround_map)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x040000, 0x043fff) AM_RAM AM_SHARE("comram")
	AM_RANGE(0x060000, 0x063fff) AM_RAM
	AM_RANGE(0x080000, 0x080fff) AM_DEVREADWRITE8("palette", palette_device, read8, write8, 0x00ff) AM_SHARE("palette")
	AM_RANGE(0x0a0000, 0x0a0001) AM_READ_PORT("SYSTEM") AM_WRITE(fround_CPU_register_w)
	AM_RANGE(0x0a0002, 0x0a0003) AM_READ_PORT("P1")
	AM_RANGE(0x0a0004, 0x0a0005) AM_READ_PORT("P2")
	AM_RANGE(0x0a0008, 0x0a0009) AM_DEVWRITE8("soundlatch", generic_latch_8_device, write, 0x00ff)
	AM_RANGE(0x0a0010, 0x0a0011) AM_READ_PORT("DSW2") AM_DEVWRITE("watchdog", watchdog_timer_device, reset16_w)
	AM_RANGE(0x0a0012, 0x0a0013) AM_READ_PORT("DSW1")
	AM_RANGE(0x0a0018, 0x0a0019) AM_READ_PORT("DSW3")
	AM_RANGE(0x0c0000, 0x0c000f) AM_WRITE(video_register_w)
	AM_RANGE(0x0c000e, 0x0c000f) AM_READ(sprite_status_r)
	AM_RANGE(0x0e0000, 0x0e0001) AM_WRITE(gfx_bank_w)
	AM_RANGE(0x100000, 0x103fff) AM_RAM_WRITE(fixram_w) AM_SHARE("fixram")
	AM_RANGE(0x120000, 0x121fff) AM_RAM_WRITE(videoram0_w) AM_SHARE("videoram.0")
	AM_RANGE(0x122000, 0x123fff) AM_RAM_WRITE(videoram1_w) AM_SHARE("videoram.1")
	AM_RANGE(0x140000, 0x143fff) AM_RAM AM_SHARE("spriteram")
	AM_RANGE(0x500000, 0x5fffff) AM_ROM AM_REGION("tiles", 0)
	AM_RANGE(0x600000, 0x6fffff) AM_ROM AM_REGION("gfxrom", 0)
ADDRESS_MAP_END

/* Input Ports */

static INPUT_PORTS_START( devilw )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) PORT_NAME("Map Button") // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123_UNK(1) // button1 = start/powerup, button2 = attack, button3 = jump

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW1")
	KONAMI_COINAGE_LOC(DEF_STR( Free_Play ), "Invalid", SW1)
	/* "Invalid" = both coin slots disabled */

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) ) PORT_DIPLOCATION("SW2:1,2")
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPUNUSED_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW2:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW2:4" )
	PORT_DIPUNUSED_DIPLOC( 0x10, IP_ACTIVE_LOW, "SW2:5" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPUNUSED_DIPLOC( 0x02, IP_ACTIVE_LOW, "SW3:2" )
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW3:4" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( darkadv )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) PORT_NAME("Map Button") // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE2 )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123(1)     // button1 = start/jump, button2 = attack, button3 = dynamite
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE3 )

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	KONAMI8_B123_UNK(3)

	PORT_START("DSW1")  /* Coinage */
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coinage ) ) PORT_DIPLOCATION("SW1:1,2,3,4")
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) )
	PORT_DIPSETTING(    0x00, "Invalid" )
	PORT_DIPUNUSED_DIPLOC( 0x10, IP_ACTIVE_LOW, "SW1:5" )
	PORT_DIPUNUSED_DIPLOC( 0x20, IP_ACTIVE_LOW, "SW1:6" )
	PORT_DIPUNUSED_DIPLOC( 0x40, IP_ACTIVE_LOW, "SW1:7" )
	PORT_DIPUNUSED_DIPLOC( 0x80, IP_ACTIVE_LOW, "SW1:8" )

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) ) PORT_DIPLOCATION("SW2:1,2")
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPUNUSED_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW2:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW2:4" )
	PORT_DIPUNUSED_DIPLOC( 0x10, IP_ACTIVE_LOW, "SW2:5" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPUNUSED_DIPLOC( 0x02, IP_ACTIVE_LOW, "SW3:2" )
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW3:4" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED)
INPUT_PORTS_END

static INPUT_PORTS_START( vulcan )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123_UNK(1) // button1 = powerup, button2 = shoot, button3 = missile

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW1")
	KONAMI_COINAGE_LOC(DEF_STR( Free_Play ), DEF_STR( None ), SW1)
	/* "None" = coin slot B disabled */

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) ) PORT_DIPLOCATION("SW2:1,2")
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) ) PORT_DIPLOCATION("SW2:3")
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Bonus_Life ) ) PORT_DIPLOCATION("SW2:4,5")
	PORT_DIPSETTING(    0x18, "20K, Every 70K" )
	PORT_DIPSETTING(    0x10, "30K, Every 80K" )
	PORT_DIPSETTING(    0x08, "20K Only" )
	PORT_DIPSETTING(    0x00, "70K Only" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Upright Controls" ) PORT_DIPLOCATION("SW3:2")
	PORT_DIPSETTING(    0x02, DEF_STR( Single ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Dual ) )
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW3:4" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED)
INPUT_PORTS_END


static INPUT_PORTS_START( gradius2 )    // same as vulcan, different bonus
	PORT_INCLUDE( vulcan )

	PORT_MODIFY("DSW2")
	PORT_DIPNAME( 0x18, 0x18, DEF_STR (Bonus_Life ) ) PORT_DIPLOCATION("SW2:4,5")
	PORT_DIPSETTING(    0x18, "20K, Every 150K" )
	PORT_DIPSETTING(    0x10, "30K, Every 200K" )
	PORT_DIPSETTING(    0x08, "20K Only" )
	PORT_DIPSETTING(    0x00, "70K Only" )
INPUT_PORTS_END

static INPUT_PORTS_START( fround )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123_UNK(1) // button1 = face punch, button2 = body punch, button3 = defend

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW1")
	KONAMI_COINAGE_LOC(DEF_STR( Free_Play ), "No Coin B", SW1)
	/* "No Coin B" = coins produce sound, but no effect on coin counter */

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x02, "Energy" ) PORT_DIPLOCATION("SW2:1,2")
	PORT_DIPSETTING(    0x03, "18" )
	PORT_DIPSETTING(    0x02, "20" ) // US and Japan factory default = "20"
	PORT_DIPSETTING(    0x01, "22" )
	PORT_DIPSETTING(    0x00, "24" )
	PORT_DIPUNUSED_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW2:3" ) // manual says "not used"
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW2:4" ) // ditto
	PORT_DIPUNUSED_DIPLOC( 0x10, IP_ACTIVE_LOW, "SW2:5" ) // ditto
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPUNUSED_DIPLOC( 0x02, IP_ACTIVE_LOW, "SW3:2" ) // manual says "not used"
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW3:4" ) // ditto
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( miaj )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123_UNK(1) // button1 = knife, button2 = weapon, button3 = choice

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW1")
	KONAMI_COINAGE_LOC(DEF_STR( Free_Play ), "Invalid", SW1)
	/* "Invalid" = both coin slots disabled */

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )        PORT_DIPLOCATION("SW2:1,2")
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPUNUSED_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW2:3" )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Bonus_Life ) )   PORT_DIPLOCATION("SW2:4,5")
	PORT_DIPSETTING(    0x18, "30K, Every 80K" )        // JP default
	PORT_DIPSETTING(    0x10, "50K, Every 100K" )
	PORT_DIPSETTING(    0x08, "50K Only" )              // US default
	PORT_DIPSETTING(    0x00, "100K Only" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )   PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )       // JP default
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )    // US default
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPUNUSED_DIPLOC( 0x02, IP_ACTIVE_LOW, "SW3:2" )
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPUNUSED_DIPLOC( 0x08, IP_ACTIVE_LOW, "SW3:4" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( cuebrickj )
	PORT_START("SYSTEM")    /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) // advance through tests
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")    /* 0xa0003 */
	KONAMI8_B123_UNK(1) // button1 = move, button2 = warp, button3 = stop

	PORT_START("P2")    /* 0xa0005 */
	KONAMI8_B123_UNK(2)

	PORT_START("P3")    /* 0xa0007 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW1")
	KONAMI_COINAGE_LOC(DEF_STR( Free_Play ), "Invalid", SW1)
	/* "Invalid" = both coin slots disabled */

	PORT_START("DSW2")
	PORT_DIPUNUSED_DIPLOC( 0x01, IP_ACTIVE_LOW, "SW2:1" ) // manual says "not used"
	PORT_DIPUNUSED_DIPLOC( 0x02, IP_ACTIVE_LOW, "SW2:2" ) // manual says "not used"
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) ) PORT_DIPLOCATION("SW2:3")
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x08, "Machine Name" ) PORT_DIPLOCATION("SW2:4,5")
	PORT_DIPSETTING(    0x18, DEF_STR( None ) )
	PORT_DIPSETTING(    0x10, "Lewis" )
	PORT_DIPSETTING(    0x08, "Johnson" )
	PORT_DIPSETTING(    0x00, "George" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:6,7")
	PORT_DIPSETTING(    0x60, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Difficult ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Very_Difficult ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW2:8")
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW3")  /* 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) ) PORT_DIPLOCATION("SW3:1")
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Upright Controls" ) PORT_DIPLOCATION("SW3:2")
	PORT_DIPSETTING(    0x02, DEF_STR( Single ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Dual ) )
	PORT_SERVICE_DIPLOC( 0x04, IP_ACTIVE_LOW, "SW3:3" )
	PORT_DIPNAME( 0x08, 0x08, "Stop Time" ) PORT_DIPLOCATION("SW3:4")
	PORT_DIPSETTING(    0x08, "200" )
	PORT_DIPSETTING(    0x00, "150" )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

/* Graphics Layouts */

static const gfx_layout tile_layout =
{
	8,8,
	RGN_FRAC(1,1),
	4,
	{ 0,1,2,3 },
	{ STEP8(0, 4) },
	{ STEP8(0, 4*8) },
	4*8*8
};

/* Graphics Decode Info */

static GFXDECODE_START( twin16 )
	GFXDECODE_ENTRY( "fixed", 0, tile_layout,   0, 16 )
	GFXDECODE_RAM(  "zipram", 0, tile_layout, 512, 16 )
GFXDECODE_END

static GFXDECODE_START( fround )
	GFXDECODE_ENTRY( "fixed", 0, tile_layout,   0, 16 )
	GFXDECODE_ENTRY( "tiles", 0, tile_layout, 512, 16 )
GFXDECODE_END

/* Sound Interfaces */

WRITE8_MEMBER(twin16_state::volume_callback)
{
	m_k007232->set_volume(0,(data >> 4) * 0x11,0);
	m_k007232->set_volume(1,0,(data & 0x0f) * 0x11);
}

/* Interrupt Generators */

INTERRUPT_GEN_MEMBER(twin16_state::CPUA_interrupt)
{
	if (CPUA_IRQ_ENABLE) device.execute().set_input_line(5, HOLD_LINE);
}

INTERRUPT_GEN_MEMBER(twin16_state::CPUB_interrupt)
{
	if (CPUB_IRQ_ENABLE) device.execute().set_input_line(5, HOLD_LINE);
}

/* Machine Drivers */

void twin16_state::machine_reset()
{
	m_CPUA_register = 0;
	m_CPUB_register = 0;
}

void twin16_state::machine_start()
{
	/* register for savestates */
	save_item(NAME(m_CPUA_register));
	save_item(NAME(m_CPUB_register));
}

MACHINE_CONFIG_START(twin16_state::twin16)
	// basic machine hardware
	MCFG_CPU_ADD("maincpu", M68000, XTAL(18'432'000)/2)
	MCFG_CPU_PROGRAM_MAP(main_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", twin16_state, CPUA_interrupt)

	MCFG_CPU_ADD("sub", M68000, XTAL(18'432'000)/2)
	MCFG_CPU_PROGRAM_MAP(sub_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", twin16_state, CPUB_interrupt)

	MCFG_CPU_ADD("audiocpu", Z80, XTAL(3'579'545))
	MCFG_CPU_PROGRAM_MAP(sound_map)

	MCFG_QUANTUM_TIME(attotime::from_hz(6000))

	MCFG_WATCHDOG_ADD("watchdog")

	// video hardware
	MCFG_BUFFERED_SPRITERAM16_ADD("spriteram")

	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_RAW_PARAMS(XTAL(18'432'000)/2, 576, 0, 40*8, 264, 2*8, 30*8)
	MCFG_SCREEN_UPDATE_DRIVER(twin16_state, screen_update_twin16)
	MCFG_SCREEN_VBLANK_CALLBACK(WRITELINE(twin16_state, screen_vblank_twin16))
	MCFG_SCREEN_PALETTE("palette")

	MCFG_GFXDECODE_ADD("gfxdecode", "palette", twin16)

	MCFG_PALETTE_ADD("palette", 1024)
	MCFG_PALETTE_FORMAT(xBBBBBGGGGGRRRRR)
	MCFG_PALETTE_MEMBITS(8)
	MCFG_PALETTE_ENABLE_SHADOWS()

	// sound hardware
	MCFG_SPEAKER_STANDARD_STEREO("lspeaker", "rspeaker")

	MCFG_GENERIC_LATCH_8_ADD("soundlatch")

	MCFG_YM2151_ADD("ymsnd", XTAL(3'579'545))
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)

	MCFG_SOUND_ADD("k007232", K007232, XTAL(3'579'545))
	MCFG_K007232_PORT_WRITE_HANDLER(WRITE8(twin16_state, volume_callback))
	MCFG_SOUND_ROUTE(0, "lspeaker", 0.12) // estimated with gradius2 OST
	MCFG_SOUND_ROUTE(0, "rspeaker", 0.12)
	MCFG_SOUND_ROUTE(1, "lspeaker", 0.12)
	MCFG_SOUND_ROUTE(1, "rspeaker", 0.12)

	MCFG_SOUND_ADD("upd", UPD7759, UPD7759_STANDARD_CLOCK)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "lspeaker", 0.20)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "rspeaker", 0.20)
MACHINE_CONFIG_END

MACHINE_CONFIG_START(twin16_state::devilw)
	twin16(config);
	MCFG_QUANTUM_TIME(attotime::from_hz(60000)) // watchdog reset otherwise
MACHINE_CONFIG_END

MACHINE_CONFIG_START(fround_state::fround)
	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M68000, XTAL(18'432'000)/2)
	MCFG_CPU_PROGRAM_MAP(fround_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", twin16_state, CPUA_interrupt)

	MCFG_CPU_ADD("audiocpu", Z80, XTAL(3'579'545))
	MCFG_CPU_PROGRAM_MAP(sound_map)

	MCFG_QUANTUM_TIME(attotime::from_hz(6000))

	MCFG_WATCHDOG_ADD("watchdog")

	/* video hardware */
	MCFG_BUFFERED_SPRITERAM16_ADD("spriteram")

	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_RAW_PARAMS(XTAL(18'432'000)/2, 576, 0, 40*8, 264, 2*8, 30*8)
	MCFG_SCREEN_UPDATE_DRIVER(twin16_state, screen_update_twin16)
	MCFG_SCREEN_VBLANK_CALLBACK(WRITELINE(twin16_state, screen_vblank_twin16))
	MCFG_SCREEN_PALETTE("palette")

	MCFG_GFXDECODE_ADD("gfxdecode", "palette", fround)

	MCFG_PALETTE_ADD("palette", 1024)
	MCFG_PALETTE_FORMAT(xBBBBBGGGGGRRRRR)
	MCFG_PALETTE_MEMBITS(8)
	MCFG_PALETTE_ENABLE_SHADOWS()

	/* sound hardware */
	MCFG_SPEAKER_STANDARD_STEREO("lspeaker", "rspeaker")

	MCFG_GENERIC_LATCH_8_ADD("soundlatch")

	MCFG_YM2151_ADD("ymsnd", XTAL(3'579'545))
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)

	MCFG_SOUND_ADD("k007232", K007232, XTAL(3'579'545))
	MCFG_K007232_PORT_WRITE_HANDLER(WRITE8(twin16_state, volume_callback))
	MCFG_SOUND_ROUTE(0, "lspeaker", 0.12)
	MCFG_SOUND_ROUTE(0, "rspeaker", 0.12)
	MCFG_SOUND_ROUTE(1, "lspeaker", 0.12)
	MCFG_SOUND_ROUTE(1, "rspeaker", 0.12)

	MCFG_SOUND_ADD("upd", UPD7759, UPD7759_STANDARD_CLOCK)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "lspeaker", 0.20)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "rspeaker", 0.20)
MACHINE_CONFIG_END

MACHINE_CONFIG_START(twin16_state::miaj)
	twin16(config);
	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_RAW_PARAMS(XTAL(18'432'000)/2, 576, 1*8, 39*8, 264, 2*8, 30*8)
MACHINE_CONFIG_END

MACHINE_CONFIG_START(cuebrickj_state::cuebrickj)
	twin16(config);
	MCFG_SCREEN_MODIFY("screen")
	MCFG_SCREEN_RAW_PARAMS(XTAL(18'432'000)/2, 576, 1*8, 39*8, 264, 2*8, 30*8)
	MCFG_NVRAM_ADD_0FILL("nvram")
MACHINE_CONFIG_END

/* ROMs */

ROM_START( devilw )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "687_t05.6n",  0x00000, 0x10000, CRC(8ab7dc61) SHA1(de8e93f04f22fa07ea6c379eeae12f8fcb48f751) )
	ROM_LOAD16_BYTE( "687_t04.4n",  0x00001, 0x10000, CRC(c69924da) SHA1(9b7220eac95fd92f32fec92b21f4f02f871c5d43) )
	ROM_LOAD16_BYTE( "687_t09.6r",  0x20000, 0x10000, CRC(fae97de0) SHA1(25b085c9de0943a0112d76447ca0504ed57cad85) )
	ROM_LOAD16_BYTE( "687_t08.4r",  0x20001, 0x10000, CRC(8c898d67) SHA1(0dd1a4467f97c993ea8d40398f46c3fc88c889bd) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "687_q07.10n", 0x00000, 0x10000, CRC(53110c0b) SHA1(a473486f7c99ab758187376d5bd89581e0b7cbdf) )
	ROM_LOAD16_BYTE( "687_q06.8n",  0x00001, 0x10000, CRC(9c53a0c5) SHA1(3b195a88af2b0198af47e4f7140087bf5a7feee6) )
	ROM_LOAD16_BYTE( "687_q13.10s", 0x20000, 0x10000, CRC(36ae6014) SHA1(102d4c3215fb5f199ce6f93ce92725b5cce0b01d) )
	ROM_LOAD16_BYTE( "687_q12.8s",  0x20001, 0x10000, CRC(6d012167) SHA1(d608e29e0b2e834c099386b6ebb667e4bfc40ce7) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "687_l03.10a", 0x00000,  0x8000, CRC(7201983c) SHA1(06d089406d2f702e8d51ffdfbf34e4727a28d506) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "687_m14.d8", 0x0000, 0x4000, CRC(d7338557) SHA1(9b384baafabaab3888a0139674f0b530303684ca) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "687i15.p13", 0x000000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD32_WORD_SWAP( "687i17.p16", 0x000002, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD32_WORD_SWAP( "687i16.p15", 0x100000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )
	ROM_LOAD32_WORD_SWAP( "687i18.p18", 0x100002, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )

	ROM_REGION16_BE( 0x20000, "data", 0 )   // tile data; mapped at 0x80000 on CPUB
	ROM_LOAD16_BYTE( "687_l11.10r", 0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",  0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "687_i01.5a", 0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "687_i02.7c", 0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( majuu )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "687_s05.6n",  0x00000, 0x10000, CRC(bd99b434) SHA1(4dbcf3a57ff0b4c3a160e2a60c2ee3fd38f5a95d) )
	ROM_LOAD16_BYTE( "687_s04.4n",  0x00001, 0x10000, CRC(3df732e2) SHA1(32e42edbc6f9dfba6ff881c72b8b014b6905e7f3) )
	ROM_LOAD16_BYTE( "687_s09.6r",  0x20000, 0x10000, CRC(1f6efec3) SHA1(4177d300877e3ffcb481f534ef7b3e4bdc160be0) )
	ROM_LOAD16_BYTE( "687_s08.4r",  0x20001, 0x10000, CRC(8a16c8c6) SHA1(df3af1c137632eab98b1eea538cc4ef5b134f44f) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "687_q07.10n", 0x00000, 0x10000, CRC(53110c0b) SHA1(a473486f7c99ab758187376d5bd89581e0b7cbdf) )
	ROM_LOAD16_BYTE( "687_q06.8n",  0x00001, 0x10000, CRC(9c53a0c5) SHA1(3b195a88af2b0198af47e4f7140087bf5a7feee6) )
	ROM_LOAD16_BYTE( "687_q13.10s", 0x20000, 0x10000, CRC(36ae6014) SHA1(102d4c3215fb5f199ce6f93ce92725b5cce0b01d) )
	ROM_LOAD16_BYTE( "687_q12.8s",  0x20001, 0x10000, CRC(6d012167) SHA1(d608e29e0b2e834c099386b6ebb667e4bfc40ce7) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "687_l03.10a", 0x00000,  0x8000, CRC(7201983c) SHA1(06d089406d2f702e8d51ffdfbf34e4727a28d506) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "687_l14.d8", 0x0000, 0x4000, CRC(20ecccd6) SHA1(b8ac3186de5ea81ae1c64b9511b7a0718aabab48) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "687i15.p13", 0x000000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD32_WORD_SWAP( "687i17.p16", 0x000002, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD32_WORD_SWAP( "687i16.p15", 0x100000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )
	ROM_LOAD32_WORD_SWAP( "687i18.p18", 0x100002, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )

	ROM_REGION16_BE( 0x20000, "data", 0 )   // tile data; mapped at 0x80000 on CPUB
	ROM_LOAD16_BYTE( "687_l11.10r", 0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",  0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "687_i01.5a", 0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "687_i02.7c", 0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( darkadv )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "687_n05.6n",  0x00000, 0x10000, CRC(a9195b0b) SHA1(a5b30c93dc962b5aef33bab7e257ed606f00c9f2) )
	ROM_LOAD16_BYTE( "687_n04.4n",  0x00001, 0x10000, CRC(65b55105) SHA1(ef102eebf34cceebafb55b04003ab9173af43458) )
	ROM_LOAD16_BYTE( "687_n09.6r",  0x20000, 0x10000, CRC(1c6b594c) SHA1(06e4eab7f544170ac40e8c77b9ebb236a6094367) )
	ROM_LOAD16_BYTE( "687_n08.4r",  0x20001, 0x10000, CRC(a9603196) SHA1(a3ab00a446531d370809d8e405cabe2d74cb612b) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "687_n07.10n", 0x00000, 0x10000, CRC(6154322a) SHA1(e84c75b4d346ade2cfb5d56454282fafb04c2d77) )
	ROM_LOAD16_BYTE( "687_n06.8n",  0x00001, 0x10000, CRC(37a72e8b) SHA1(574d655ba7cda2f8eb628155fb8662a4ee5f23e5) )
	ROM_LOAD16_BYTE( "687_n13.10s", 0x20000, 0x10000, CRC(f1c252af) SHA1(dbe78352ae5b284870becc92139773f66c52ed7d) )
	ROM_LOAD16_BYTE( "687_n12.8s",  0x20001, 0x10000, CRC(da221944) SHA1(c288ee583414a077f0d8b815b8e01b7f19a5fafe) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "687_n03.10a", 0x00000,  0x8000, CRC(a24c682f) SHA1(cf053270d3f77448ff802db832598ae3cf7dae6c) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "687_n14.d8", 0x0000, 0x4000, CRC(c76ac6d2) SHA1(d8fec255f1f7177a3716a5894fb679cbe172b6ea) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "687i15.p13", 0x000000, 0x80000, CRC(eec8c5b2) SHA1(c400dd8c7bb85d233815572acd547acf626e4c73) )
	ROM_LOAD32_WORD_SWAP( "687i17.p16", 0x000002, 0x80000, CRC(66cb3923) SHA1(3c1fc1dff77201cf8d8c4594c965695066c1701c) )
	ROM_LOAD32_WORD_SWAP( "687i16.p15", 0x100000, 0x80000, CRC(746cf48b) SHA1(8f51df931b1de928f402f51bbaf02e37dfec1d6d) )
	ROM_LOAD32_WORD_SWAP( "687i18.p18", 0x100002, 0x80000, CRC(a1c7d0db) SHA1(901bc6e1982b9a8c2150a802995256d0168cc605) )

	ROM_REGION16_BE( 0x20000, "data", 0 )   // tile data; mapped at 0x80000 on CPUB
	ROM_LOAD16_BYTE( "687_l11.10r", 0x00000, 0x10000, CRC(399deee8) SHA1(dcc65e95f28ae4e9b671e70ce0bd5ba0fe178506) )
	ROM_LOAD16_BYTE( "687_l10.8r",  0x00001, 0x10000, CRC(117c91ee) SHA1(dcf8efb25fc73cff916b66b7bcfd3c1fb2556a53) )

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "687_i01.5a", 0x00000, 0x20000, CRC(d4992dfb) SHA1(c65bef07b6adb9ab6328d679595450945dbf6a88) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "687_i02.7c", 0x00000, 0x20000, CRC(e5947501) SHA1(9ec7319bd97711981db60af6f477c2ba2ec7e6b8) )
ROM_END

ROM_START( vulcan )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_w05.6n",  0x00000, 0x10000, CRC(6e0e99cd) SHA1(f9f4ef617aee83397b0ac98a5cc92f60d3a3d5ed) )
	ROM_LOAD16_BYTE( "785_w04.4n",  0x00001, 0x10000, CRC(23ec74ca) SHA1(901c7c798b113232f20e24f61940ed6ef5148599) )
	ROM_LOAD16_BYTE( "785_w09.6r",  0x20000, 0x10000, CRC(377e4f28) SHA1(d3af2e52c38b5d618cac2e66df99ab7950638fbc) )
	ROM_LOAD16_BYTE( "785_w08.4r",  0x20001, 0x10000, CRC(813d41ea) SHA1(58ba0e4b1afec3e5674286c44da14244591f1312) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_p07.10n", 0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n",  0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s", 0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",  0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_h14.d8", 0x0000, 0x4000, CRC(02f4b16f) SHA1(45addc99f520770f38c6aa69aef9af59cfc410b5) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( vulcana )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_r05.6n",  0x00000, 0x10000, CRC(1777a57b) SHA1(9696fa9c6d53587972d213d5c12245e40fbc2bda) )
	ROM_LOAD16_BYTE( "785_r04.4n",  0x00001, 0x10000, CRC(790cbdf3) SHA1(e247c2be00169490aea945518c1a68e94799f5f4) )
	ROM_LOAD16_BYTE( "785_u09.6r",  0x20000, 0x10000, CRC(1aba7010) SHA1(bcf28640d5ba7864ed7112db23ff344434b2d944) )
	ROM_LOAD16_BYTE( "785_k08.4r",  0x20001, 0x10000, CRC(36bb1be1) SHA1(ce9a98443a453b1c05dfb8d0f8ba0f94292b9d30) ) /* Same as 785_r08.4r */

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_p07.10n", 0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n",  0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s", 0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",  0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_h14.d8", 0x0000, 0x4000, CRC(02f4b16f) SHA1(45addc99f520770f38c6aa69aef9af59cfc410b5) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( vulcanb )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_n05.6n",  0x00000, 0x10000, CRC(6371d475) SHA1(82e6adc38de8363c126cd073dbc95eeec576db98) )
	ROM_LOAD16_BYTE( "785_n04.4n",  0x00001, 0x10000, CRC(b6f24b9e) SHA1(01685f89955385714062cc70316edce1a6a64e35) )
	ROM_LOAD16_BYTE( "785_r09.6r",  0x20000, 0x10000, CRC(fbfd1543) SHA1(8ef41a0a9d3d1de9395bdd5288199841c61d7045) )
	ROM_LOAD16_BYTE( "785_r08.4r",  0x20001, 0x10000, CRC(36bb1be1) SHA1(ce9a98443a453b1c05dfb8d0f8ba0f94292b9d30) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_g07.10n", 0x00000, 0x10000, CRC(ee09dd5d) SHA1(9b6fb12c2cb7930df12d9876810811540fd560ee) ) /* requires older CPU B code compared to other sets */
	ROM_LOAD16_BYTE( "785_g06.8n",  0x00001, 0x10000, CRC(85ab7af7) SHA1(5cb36918a5cdfd16611da76f07450ae1d115f2c7) )
	ROM_LOAD16_BYTE( "785_g13.10s", 0x20000, 0x10000, CRC(274f325d) SHA1(1076efa204eff0fc8a8788706b17b9a128023d35) )
	ROM_LOAD16_BYTE( "785_g12.8s",  0x20001, 0x10000, CRC(1625f933) SHA1(3f25d7396af46e75e3ae8456414e31935de43d34) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_h14.d8", 0x0000, 0x4000, CRC(02f4b16f) SHA1(45addc99f520770f38c6aa69aef9af59cfc410b5) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( gradius2 )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_x05.6n",  0x00000, 0x10000, CRC(8a23a7b8) SHA1(9846272c011715b17f146d5265a09e790d6e37cd) )
	ROM_LOAD16_BYTE( "785_x04.4n",  0x00001, 0x10000, CRC(88e466ce) SHA1(3a7d098b851bcf6dc8e7ff601d11563406c56666) )
	ROM_LOAD16_BYTE( "785_x09.6r",  0x20000, 0x10000, CRC(3f3d7d7a) SHA1(8e3acf980179e749ec642b6ff211c4ba57c6c592) )
	ROM_LOAD16_BYTE( "785_x08.4r",  0x20001, 0x10000, CRC(c39c8efd) SHA1(372dbb254525a7334fe79a42ba9851eafa856847) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_p07.10n", 0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n",  0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s", 0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",  0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_g14.d8", 0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( gradius2a )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_p05.6n",  0x00000, 0x10000, CRC(4db0e736) SHA1(45edf094da448d039af18a7b47a592a3de02f17b) )
	ROM_LOAD16_BYTE( "785_p04.4n",  0x00001, 0x10000, CRC(765b99e6) SHA1(4acae0cf2bf23f8a1cf7b650b018d22a9ae947c9) )
	ROM_LOAD16_BYTE( "785_t09.6r",  0x20000, 0x10000, CRC(4e3f4965) SHA1(130da6db0590779948a29fd5010f0e628e4f2ee6) )
	ROM_LOAD16_BYTE( "785_j08.4r",  0x20001, 0x10000, CRC(2b1c9108) SHA1(3d367a1b60f95798b478dc4a84daa9b362271d4d) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_p07.10n", 0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n",  0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s", 0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",  0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_g14.d8", 0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( gradius2b )
	ROM_REGION( 0x40000, "maincpu", 0 )     // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "785_p05.6n",  0x00000, 0x10000, CRC(4db0e736) SHA1(45edf094da448d039af18a7b47a592a3de02f17b) )
	ROM_LOAD16_BYTE( "785_p04.4n",  0x00001, 0x10000, CRC(765b99e6) SHA1(4acae0cf2bf23f8a1cf7b650b018d22a9ae947c9) )
	ROM_LOAD16_BYTE( "785_j09.6r",  0x20000, 0x10000, CRC(6d96a7e3) SHA1(cc1362f5922cf1e348a58d54d03320f394a2aeab) )
	ROM_LOAD16_BYTE( "785_j08.4r",  0x20001, 0x10000, CRC(2b1c9108) SHA1(3d367a1b60f95798b478dc4a84daa9b362271d4d) )

	ROM_REGION( 0x40000, "sub", 0 )     // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "785_p07.10n", 0x00000, 0x10000, CRC(686d549d) SHA1(9687be801c4fb963bf6b0199e2ae9f5051213f7a) )
	ROM_LOAD16_BYTE( "785_p06.8n",  0x00001, 0x10000, CRC(70c94bee) SHA1(951e00ca4d3a47a21b4db05bcdc8ead45b42c3f1) )
	ROM_LOAD16_BYTE( "785_p13.10s", 0x20000, 0x10000, CRC(478fdb0a) SHA1(2e285ad6dcfc67f3e24d231e0e1be19036ce64d2) )
	ROM_LOAD16_BYTE( "785_p12.8s",  0x20001, 0x10000, CRC(38ea402a) SHA1(90ff2bd71435988cde967704ce3b1401de206683) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "785_g03.10a", 0x00000,  0x8000, CRC(67a3b50d) SHA1(3c83f3b0df73d9361ec3cda26a6c4c603a088419) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "785_g14.d8", 0x0000, 0x4000, CRC(9dcdad9d) SHA1(22f457408b453a71b7e91974aee1b1e735ff887f) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "785f15.p13", 0x000000, 0x80000, CRC(5bd239ac) SHA1(9c96f6069d06a1d80c04650ed56bc5e1508db657) )
	ROM_LOAD32_WORD_SWAP( "785f17.p16", 0x000002, 0x80000, CRC(4e7a7b82) SHA1(520b5ebd400954042d55cf243842a6d6b6d10210) )
	ROM_LOAD32_WORD_SWAP( "785f16.p15", 0x100000, 0x80000, CRC(95c6b8a3) SHA1(0a906af72d08993fd83f23d72ffb919801aa17af) )
	ROM_LOAD32_WORD_SWAP( "785f18.p18", 0x100002, 0x80000, CRC(3f604e9a) SHA1(c5e0b6f6dd506209e8c07cbae89c821828f488ff) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "785_f01.5a", 0x00000, 0x20000, CRC(a0d8d69e) SHA1(2994e5740b7c099d55fb162a363a26ef1995c756) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "785_f02.7c", 0x00000, 0x20000, CRC(c39f5ca4) SHA1(92ebea11b7d3a72d0f58c456f16a6ff36d6b3062) )
ROM_END

ROM_START( fround )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "870_m21.bin", 0x00000, 0x20000, CRC(436dbffb) SHA1(be8c8544f4d8ae86f216095753c6178a3cbf0e8d) )
	ROM_LOAD16_BYTE( "870_m20.bin", 0x00001, 0x20000, CRC(b1c79d6a) SHA1(76e95e87eaa96ba694675fde7706540c584a36fb) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "870_f03.10a", 0x00000,  0x8000, CRC(a645c727) SHA1(e173e79130f187d090766664f1c478d47062749c) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "870_f14.d8", 0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) )

	ROM_REGION16_BE( 0x100000, "tiles", 0 )  // tile gfx data
	ROM_LOAD32_WORD_SWAP( "870c16.p15", 0x00000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )
	ROM_LOAD32_WORD_SWAP( "870c18.p18", 0x00002, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )

	ROM_REGION16_BE( 0x100000, "gfxrom", 0 )  // sprite gfx data
	ROM_LOAD32_WORD_SWAP( "870c15.p13", 0x00000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )
	ROM_LOAD32_WORD_SWAP( "870c17.p16", 0x00002, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "870_c01.5a", 0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "870_c02.7c", 0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( froundl )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "870_l21.bin", 0x00000, 0x20000, CRC(e21a3a19) SHA1(5f5793c88093bc8632eab673f5e0d954ac9b9177) )
	ROM_LOAD16_BYTE( "870_l20.bin", 0x00001, 0x20000, CRC(0ce9786f) SHA1(0fb0eef999ed09dd50eed403255887103fc35461) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "870_f03.10a", 0x00000,  0x8000, CRC(a645c727) SHA1(e173e79130f187d090766664f1c478d47062749c) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "870_f14.d8", 0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) )

	ROM_REGION16_BE( 0x100000, "tiles", 0 )  // tile gfx data
	ROM_LOAD32_WORD_SWAP( "870c16.p15", 0x00000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )
	ROM_LOAD32_WORD_SWAP( "870c18.p18", 0x00002, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )

	ROM_REGION16_BE( 0x100000, "gfxrom", 0 )  // sprite gfx data
	ROM_LOAD32_WORD_SWAP( "870c15.p13", 0x00000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )
	ROM_LOAD32_WORD_SWAP( "870c17.p16", 0x00002, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "870_c01.5a", 0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, "upd", 0 ) // samples
	ROM_LOAD( "870_c02.7c", 0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( hpuncher )
	ROM_REGION( 0x40000, "maincpu", 0 ) // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "870_h05.6n",  0x00000, 0x10000, CRC(2bcfeef3) SHA1(8843e795a310e1a79b7b65226d9526197c2144ef) )
	ROM_LOAD16_BYTE( "870_h04.4n",  0x00001, 0x10000, CRC(b9f97fd3) SHA1(d15a659f753f9f475dfd4f7e96789bacc54cb1f9) )
	ROM_LOAD16_BYTE( "870_h09.6r",  0x20000, 0x10000, CRC(96a4f8b1) SHA1(2709f5236235b1b69c3367054fb998c1beb35625) )
	ROM_LOAD16_BYTE( "870_h08.4r",  0x20001, 0x10000, CRC(46d65156) SHA1(6dd9d97a8e3394db668da1fc6a46c166bca1f456) )

	ROM_REGION( 0x40000, "sub", 0 ) // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "870_h07.10n", 0x00000, 0x10000, CRC(b4dda612) SHA1(e9612af11a151de1f0236629c84679f80936cae9) )
	ROM_LOAD16_BYTE( "870_h06.8n",  0x00001, 0x10000, CRC(696ba702) SHA1(e35c9aeab4305d828a347eaddf4abc806a889fa3) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "870_g03.10a", 0x00000,  0x8000, CRC(db9c10c8) SHA1(b2ec4d6800a4ab00dfcc8d3cd4abf9b9d7c5544a) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "870_f14.d8", 0x0000, 0x4000, CRC(c9b46615) SHA1(c0cddb1af47b8e0865055624cf4e89a111ac1b0f) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP( "870c15.p13", 0x000000, 0x80000, CRC(8c9281df) SHA1(5e3d80be414db108d5363d0ea1b74021ba942c33) )
	ROM_LOAD32_WORD_SWAP( "870c17.p16", 0x000002, 0x80000, CRC(2bc99ff8) SHA1(9a06502317a71ca5662b79aedf47379b8e5434a9) )
	ROM_LOAD32_WORD_SWAP( "870c16.p15", 0x100000, 0x80000, CRC(41df6a1b) SHA1(32e0fdeb53628d18adde851e4496dd01ac6ec68f) )
	ROM_LOAD32_WORD_SWAP( "870c18.p18", 0x100002, 0x80000, CRC(07927fe8) SHA1(0ab5e0e785347fbed5c4b930a32876d6ce2bef4a) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD( "870_c01.5a", 0x00000, 0x20000, CRC(6af96546) SHA1(63b49b28c0f2ef8f52bc4c5955ad6a633dd553cf) )

	ROM_REGION( 0x20000, "upd", 0 )     // samples
	ROM_LOAD( "870_c02.7c", 0x00000, 0x20000, CRC(54e12c6d) SHA1(8171441aeb1a1dc4d65fae59c7c217f992f70cbc) )
ROM_END

ROM_START( miaj )
	ROM_REGION( 0x40000, "maincpu", 0 )     // 68000 code (CPU A)
	ROM_LOAD16_BYTE("808_r05.6n", 0x00000, 0x10000, CRC(91fd83f4) SHA1(12cba5828d1d752efe4fc1972bed0205ac277850) )
	ROM_LOAD16_BYTE("808_r04.4n", 0x00001, 0x10000, CRC(f1c8c597) SHA1(4e8e87aedc9dac4ed05816989210de49f4dd60d2) )
	ROM_LOAD16_BYTE("808_r09.6r", 0x20000, 0x10000, CRC(f74d4467) SHA1(2b2491ee7ece979f2ec849075b0c30b6c1ed6bae) )
	ROM_LOAD16_BYTE("808_r08.4r", 0x20001, 0x10000, CRC(26f21704) SHA1(123cb2e4bcb25d82faa75e4505f9859c10bca754) )

	ROM_REGION( 0x40000, "sub", 0 )     // 68000 code (CPU B)
	ROM_LOAD16_BYTE("808_e07.10n", 0x00000, 0x10000, CRC(297bdcea) SHA1(41ed650cbc4f56777fa352812e9a560554459f90) )
	ROM_LOAD16_BYTE("808_e06.8n",  0x00001, 0x10000, CRC(8f576b33) SHA1(cd8c9a45626c30571ca12d0241cccc092c0903a0) )
	ROM_LOAD16_BYTE("808_e13.10s", 0x20000, 0x10000, CRC(1fa708f4) SHA1(9511a19f50fb61571c2986c72d1a85e87b8d0495) )
	ROM_LOAD16_BYTE("808_e12.8s",  0x20001, 0x10000, CRC(d62f1fde) SHA1(1e55084f1294b6ac7c152fcd1800511fcab5d360) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "808_e03.10a", 0x00000,  0x8000, CRC(3d93a7cd) SHA1(dcdd327e78f32436b276d0666f62a5b733b296e8) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD("808_e14.d8", 0x0000, 0x4000, CRC(b9d36525) SHA1(53291c4911d7e1a5110539e4c57a11d0d530dc6f) )

	ROM_REGION16_BE( 0x200000, "gfxrom", 0 )  // gfx data used at runtime
	ROM_LOAD32_WORD_SWAP("808d15.p13", 0x000000, 0x80000, CRC(2b22a6b6) SHA1(8e1af0627a4eac045128c4096e2cfb59c3d2f5ef) )
	ROM_LOAD32_WORD_SWAP("808d17.p16", 0x000002, 0x80000, CRC(d1299082) SHA1(c3c07b0517e7428ccd1cdf9e15aaf16d98e7c4cd) )

	ROM_REGION16_BE( 0x20000, "data", ROMREGION_ERASE00 )    // tile data; mapped at 0x80000 on CPUB

	ROM_REGION( 0x20000, "k007232", 0 )  // samples
	ROM_LOAD("808_d01.5a", 0x00000, 0x20000, CRC(fd4d37c0) SHA1(ef91c6e7bb57c27a9a51729fffd1bfe3e806fb61) )

	ROM_REGION( 0x20000, "upd", ROMREGION_ERASE00 )     // samples
ROM_END

ROM_START( cuebrickj )
	ROM_REGION( 0x40000, "maincpu", 0 )     // 68000 code (CPU A)
	ROM_LOAD16_BYTE( "903_e05.6n",  0x00000, 0x10000, CRC(8b556220) SHA1(dbe24133e74018c4fe9332519394cbb882c4ed5a) )
	ROM_LOAD16_BYTE( "903_e04.4n",  0x00001, 0x10000, CRC(bf9c7927) SHA1(3a594b8846f7e6074ca54f8cd5fe2ba3b64ba740) )
	ROM_LOAD16_BYTE( "903_e09.6r",  0x20000, 0x10000, CRC(2a77554d) SHA1(53eee99737333c8c4bbb9c757fd12488b6c4d3d8) )
	ROM_LOAD16_BYTE( "903_e08.4r",  0x20001, 0x10000, CRC(c0a430c1) SHA1(bef737f12e9a79781a4cd795ffc108899757901d) )

	ROM_REGION( 0x40000, "sub", 0 )     // 68000 code (CPU B)
	ROM_LOAD16_BYTE( "903_d07.10n", 0x00000, 0x10000, CRC(fc0edce7) SHA1(57e969a42c48a9464ec370d4605eaecb690d58ab) )
	ROM_LOAD16_BYTE( "903_d06.8n",  0x00001, 0x10000, CRC(b2cef6fe) SHA1(27d1d15f3a408e6166745cb6a159704a8cbb4584) )
	ROM_LOAD16_BYTE( "903_e13.10s", 0x20000, 0x10000, CRC(4fb5fb80) SHA1(3a59dae3846341289c31aa106684ebc45488ca45) )
	ROM_LOAD16_BYTE( "903_e12.8s",  0x20001, 0x10000, CRC(883e3097) SHA1(fe0fa1a2881a67223d741c400bb8c1a0c67946c1) )

	ROM_REGION( 0x8000, "audiocpu", 0 )    // Z80 code (sound CPU)
	ROM_LOAD( "903_d03.10a", 0x00000,  0x8000, CRC(455e855a) SHA1(cfdd54a5071862653ee94c0455301f4a7245fbd8) )

	ROM_REGION( 0x4000, "fixed", 0 )  // title screen graphics & characters
	ROM_LOAD( "903_e14.d8", 0x0000, 0x4000, CRC(ddbebbd5) SHA1(df532f19a08cabb28d693e4cd445c47c9e35e5ff) )

	ROM_REGION16_BE( 0x200000, "gfxrom", ROMREGION_ERASE00 )   // gfx data used at runtime
	// unpopulated

	ROM_REGION16_BE( 0x20000, "data", 0 )   // tile data; mapped at 0x80000 on CPUB
	ROM_LOAD16_BYTE( "903_e11.10r", 0x00000, 0x10000, CRC(5c41faf8) SHA1(f9eee6a7b92d3b3aa4320747da6390310522a2cf) )
	ROM_LOAD16_BYTE( "903_e10.8r",  0x00001, 0x10000, CRC(417576d4) SHA1(e84762743e3a1117b6ef7ea0b304877e4a719f75) )

	ROM_REGION( 0x20000, "k007232", ROMREGION_ERASE00 )  // samples
	// unpopulated

	ROM_REGION( 0x20000, "upd", ROMREGION_ERASE00 )     // samples
	// unpopulated
ROM_END

/* Driver Initialization */

DRIVER_INIT_MEMBER(twin16_state,twin16)
{
	m_is_fround = false;
	m_gfxrombank->configure_entries(0, 2, memregion("gfxrom")->base() + 0x100000, 0x80000);
	m_gfxrombank->set_entry(0);
}

DRIVER_INIT_MEMBER(fround_state,fround)
{
	m_is_fround = true;
}

WRITE8_MEMBER(cuebrickj_state::nvram_bank_w)
{
	membank("nvrambank")->set_entry(data);
}

DRIVER_INIT_MEMBER(cuebrickj_state,cuebrickj)
{
	DRIVER_INIT_CALL(twin16);

	address_space &space = m_maincpu->space(AS_PROGRAM);

	space.install_readwrite_bank(0x0b0000, 0x0b03ff, "nvrambank");
	space.install_write_handler( 0x0b0400, 0x0b0401, WRITE8_DELEGATE(cuebrickj_state, nvram_bank_w), 0xff00);

	membank("nvrambank")->configure_entries(0, 0x20, m_nvram, 0x400);

	machine().device<nvram_device>("nvram")->set_base(m_nvram, sizeof(m_nvram));

	save_item(NAME(m_nvram));
}

/* Game Drivers */

//    YEAR, NAME,      PARENT,   MACHINE,   INPUT,     STATE,           INIT,      MONITOR,COMPANY,  FULLNAME,FLAGS
GAME( 1987, devilw,    0,        devilw,    devilw,    twin16_state,    twin16,    ROT0,   "Konami", "Devil World", MACHINE_SUPPORTS_SAVE )
GAME( 1987, majuu,     devilw,   devilw,    devilw,    twin16_state,    twin16,    ROT0,   "Konami", "Majuu no Ohkoku", MACHINE_SUPPORTS_SAVE )
GAME( 1987, darkadv,   devilw,   devilw,    darkadv,   twin16_state,    twin16,    ROT0,   "Konami", "Dark Adventure", MACHINE_SUPPORTS_SAVE )
GAME( 1988, vulcan,    0,        twin16,    vulcan,    twin16_state,    twin16,    ROT0,   "Konami", "Vulcan Venture (New)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, vulcana,   vulcan,   twin16,    vulcan,    twin16_state,    twin16,    ROT0,   "Konami", "Vulcan Venture (Old)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, vulcanb,   vulcan,   twin16,    vulcan,    twin16_state,    twin16,    ROT0,   "Konami", "Vulcan Venture (Oldest)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, gradius2,  vulcan,   twin16,    gradius2,  twin16_state,    twin16,    ROT0,   "Konami", "Gradius II - GOFER no Yabou (Japan New Ver.)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, gradius2a, vulcan,   twin16,    vulcan,    twin16_state,    twin16,    ROT0,   "Konami", "Gradius II - GOFER no Yabou (Japan Old Ver.)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, gradius2b, vulcan,   twin16,    vulcan,    twin16_state,    twin16,    ROT0,   "Konami", "Gradius II - GOFER no Yabou (Japan Older Ver.)", MACHINE_SUPPORTS_SAVE )

GAME( 1988, fround,    0,        fround,    fround,    fround_state,    fround,    ROT0,   "Konami", "The Final Round (version M)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, froundl,   fround,   fround,    fround,    fround_state,    fround,    ROT0,   "Konami", "The Final Round (version L)", MACHINE_SUPPORTS_SAVE )
GAME( 1988, hpuncher,  fround,   twin16,    fround,    twin16_state,    twin16,    ROT0,   "Konami", "Hard Puncher (Japan)", MACHINE_SUPPORTS_SAVE )
GAME( 1989, miaj,      mia,      miaj,      miaj,      twin16_state,    twin16,    ROT0,   "Konami", "M.I.A. - Missing in Action (version R) (Japan)", MACHINE_SUPPORTS_SAVE )
GAME( 1989, cuebrickj, cuebrick, cuebrickj, cuebrickj, cuebrickj_state, cuebrickj, ROT0,   "Konami", "Cue Brick (Japan)", MACHINE_SUPPORTS_SAVE )
