/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.cache.factories;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.ConfigurationException;
import org.jboss.cache.xml.XmlHelper;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Parser able to parse a series of cache configurations stored in an
 * XML document with the following structure:
 * <pre>
 * <cache-configs>
 *    <cache-config name="configA">
 *     ....
 *    </cache-config>
 *    <cache-config name="configB">
 *     ....
 *    </cache-config>
 * </cache-configs>
 * </pre>
 * <p/>
 * The '....' represents the normal content of the mbean element in a 
 * JBC -service.xml config file.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1.1 $
 */
public class CacheConfigsXmlParser 
 {
    /** Name of the root element in a cache configs XML document*/
    public static final String DOCUMENT_ROOT = "cache-configs";
    /** 
     * Name of the element that represents an individual cache configuration 
     * in a cache configs XML document.
     */
    public static final String CONFIG_ROOT = "cache-config";
    /** 
     * Name of the attribute in a {@link #CONFIG_ROOT cache-config} element that specifies
     * the name of the configuration.
     */
    public static final String CONFIG_NAME = "name";
    
    private static final Log log = LogFactory.getLog(CacheConfigsXmlParser.class);
    
    private final XmlConfigurationParser parser = new XmlConfigurationParser();

    public Map<String, Configuration> parseConfigs(String fileName) throws CloneNotSupportedException 
    {
       InputStream is = getAsInputStreamFromClassLoader(fileName);
       if (is == null)
       {
          if (log.isDebugEnabled())
             log.debug("Unable to find configuration file " + fileName + " in classpath; searching for this file on the filesystem instead.");
          try
          {
             is = new FileInputStream(fileName);
          }
          catch (FileNotFoundException e)
          {
             throw new ConfigurationException("Unable to find config file " + fileName + " either in classpath or on the filesystem!", e);
          }
       }

       return parseConfigs(is);
    }
     
    public Map<String, Configuration> parseConfigs(InputStream stream) throws CloneNotSupportedException 
    {            
       // loop through all elements in XML.
       Element root = XmlHelper.getDocumentRoot(stream);
       NodeList list = root.getElementsByTagName(CONFIG_ROOT);
       if (list == null || list.getLength() == 0) 
          throw new ConfigurationException("Can't find " + CONFIG_ROOT + " tag");
         
       Map<String, Configuration> result = new HashMap<String, Configuration>();
         
       for (int i = 0; i < list.getLength(); i++)
       {
          Node node = list.item(i);
          if (node.getNodeType() != Node.ELEMENT_NODE)
          {
             continue;
          }
            
          Element element = (Element) node;
          String name = element.getAttribute(CONFIG_NAME);
          if (name == null || name.trim().length() == 0)
              throw new ConfigurationException("Element " + element + " has no name attribute");
            
          Configuration c = parser.parseConfiguration(element);
          // Prove that we can successfully clone it
          c = c.clone();
          result.put(name.trim(), c);
       }
         
       return result;
    }

    protected InputStream getAsInputStreamFromClassLoader(String filename)
    {
       InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(filename);
       if (is == null)
       {
          // check system class loader
          is = getClass().getClassLoader().getResourceAsStream(filename);
       }
       return is;
    }
 }