[Home](../index.md) > [Documentation](index.md) > JavaCC BNF

---

This page contains the complete syntax of the JavaCC grammar files with detailed explanations of each construct.

### <a name="toc"></a>Contents

- [**Tokens**](#tokens)
  * [Reserved words](#reserved-words)
  * [Whitespace](#whitespace)
  * [Comments](#comments)
  * [Reserved words and literals](#reserved-and-literals)
  * [Separators](#separators)
  * [Operators](#operators)
  * [Identifiers](#identifiers)
- [**Non-Terminals**](#non-terminals)
  * [JavaCC grammar](#javacc-grammar)
  * [Java grammar](#java-grammar)
    * [Java identifiers](#java-identifiers)
    * [Program structure](#program-structure)
    * [Modifiers](#modifiers)
    * [Declarations](#declarations)
    * [Types](#types)
    * [Expressions](#expressions)
    * [Statements](#statements)
    * [Annotations](#annotations)
    * [Annotation types](#annotation-types)

## <a name="tokens"></a>Tokens

The following is the JavaCC token specification.

### <a name="reserved-words"></a>Reserved words

---

```java
/*
 * These are the only tokens in JavaCC but not in Java.
 */
<DEFAULT> TOKEN : { <_LOOKAHEAD: "LOOKAHEAD">
                  | <_IGNORE_CASE: "IGNORE_CASE">
                  | <_PARSER_BEGIN: "PARSER_BEGIN">
                  | <_PARSER_END: "PARSER_END">
                  | <_JAVACODE: "JAVACODE">
                  | <_CPPCODE: "CPPCODE">
                  | <_TOKEN: "TOKEN">
                  | <_SPECIAL_TOKEN: "SPECIAL_TOKEN">
                  | <_MORE: "MORE">
                  | <_SKIP: "SKIP">
                  | <_TOKEN_MGR_DECLS: "TOKEN_MGR_DECLS">
                  | <_EOF: "EOF">
                  | <_DCL_PARSER_BEGIN: "DCL_PARSER_BEGIN">
                  | <_DCL_PARSER_END: "DCL_PARSER_END">
                  | <_INC_PARSER_BEGIN: "INC_PARSER_BEGIN">
                  | <_INC_PARSER_END: "INC_PARSER_END">
                  | <_DEF_PARSER_BEGIN: "DEF_PARSER_BEGIN">
                  | <_DEF_PARSER_END: "DEF_PARSER_END">
}
```

<br>

### <a name="whitespace"></a>Whitespace

---

```java
/*
 * Whitespace
 */
<DEFAULT> SKIP : { " "
                 | "\t"
                 | "\n"
                 | "\r"
                 | "\f"
                 | "/*@egen*/" : AFTER_EGEN
}

<AFTER_EGEN> SKIP : {
  <~[]> : DEFAULT
}
```

<br>

### <a name="comments"></a>Comments

---

```java
/*
 * Comments
 */
<DEFAULT> MORE : { "//" : IN_SINGLE_LINE_COMMENT
                 | <"/**" ~["/"]> : IN_FORMAL_COMMENT
                 | "/*" : IN_MULTI_LINE_COMMENT
                 | "/*@bgen(jjtree" : IN_MULTI_LINE_COMMENT
}

<IN_SINGLE_LINE_COMMENT> SPECIAL : {
  <SINGLE_LINE_COMMENT: "\n" | "\r" | "\r\n"> : DEFAULT
}

<IN_FORMAL_COMMENT> SPECIAL : {
  <FORMAL_COMMENT: "*/"> : DEFAULT
}

<IN_MULTI_LINE_COMMENT> SPECIAL : {
  <MULTI_LINE_COMMENT: "*/"> : DEFAULT
}

<IN_SINGLE_LINE_COMMENT,IN_FORMAL_COMMENT,IN_MULTI_LINE_COMMENT> MORE : {
  <~[]>
}
```

<br>

### <a name="reserved-and-literals"></a>Reserved words and literals

---

```java
/*
 * Java reserved words and literals
 */
<DEFAULT> TOKEN : { <ABSTRACT: "abstract">
                  | <ASSERT: "assert">
                  | <BOOLEAN: "boolean">
                  | <BREAK: "break">
                  | <BYTE: "byte">
                  | <CASE: "case">
                  | <CATCH: "catch">
                  | <CHAR: "char">
                  | <CLASS: "class">
                  | <CONST: "const">
                  | <CONTINUE: "continue">
                  | <_DEFAULT: "default">
                  | <DO: "do">
                  | <DOUBLE: "double">
                  | <ELSE: "else">
                  | <ENUM: "enum">
                  | <EXTENDS: "extends">
                  | <FALSE: "false">
                  | <FINAL: "final">
                  | <FINALLY: "finally">
                  | <FLOAT: "float">
                  | <FOR: "for">
                  | <GOTO: "goto">
                  | <IF: "if">
                  | <IMPLEMENTS: "implements">
                  | <IMPORT: "import">
                  | <INSTANCEOF: "instanceof">
                  | <INT: "int">
                  | <INTERFACE: "interface">
                  | <LONG: "long">
                  | <NATIVE: "native">
                  | <NEW: "new">
                  | <NULL: "null">
                  | <PACKAGE: "package">
                  | <PRIVATE: "private">
                  | <PROTECTED: "protected">
                  | <PUBLIC: "public">
                  | <RETURN: "return">
                  | <SHORT: "short">
                  | <STATIC: "static">
                  | <STRICTFP: "strictfp">
                  | <SUPER: "super">
                  | <SWITCH: "switch">
                  | <SYNCHRONIZED: "synchronized">
                  | <THIS: "this">
                  | <THROW: "throw">
                  | <THROWS: "throws">
                  | <TRANSIENT: "transient">
                  | <TRUE: "true">
                  | <TRY: "try">
                  | <VOID: "void">
                  | <VOLATILE: "volatile">
                  | <WHILE: "while">
}
```

```java
/*
 * C++ reserved words and literals
 */
<DEFAULT> TOKEN : { <TEMPLATE: "template">
                  | <TYPENAME: "typename">
}
```

```java
/*
 * Java literals
 */
<DEFAULT> TOKEN : {
    <INTEGER_LITERAL: <DECIMAL_LITERAL> (["l","L"])?
  | <HEX_LITERAL> (["l","L"])?
  | <OCTAL_LITERAL> (["l","L"])?
  | <BINARY_LITERAL> (["l","L"])?>
  | <#DECIMAL_LITERAL: ["1"-"9"] (("_")* ["0"-"9"])*>
  | <#HEX_LITERAL: "0" ["x","X"] ["0"-"9","a"-"f","A"-"F"] (("_")* ["0"-"9","a"-"f","A"-"F"])*>
  | <#OCTAL_LITERAL: "0" (("_")* ["0"-"7"])*>
  | <#BINARY_LITERAL: "0" ["b","B"] ["0","1"] (("_")* ["0","1"])*>
  | <FLOATING_POINT_LITERAL: <DECIMAL_FLOATING_POINT_LITERAL>
  | <HEXADECIMAL_FLOATING_POINT_LITERAL>>
  | <#DECIMAL_FLOATING_POINT_LITERAL:
    ["0"-"9"] (("_")* ["0"-"9"])* "." (["0"-"9"] (("_")* ["0"-"9"])*)?
    (<DECIMAL_EXPONENT>)? (["f","F","d","D"])?
  | "." ["0"-"9"] (("_")* ["0"-"9"])* (<DECIMAL_EXPONENT>)? (["f","F","d","D"])?
  | ["0"-"9"] (("_")* ["0"-"9"])* <DECIMAL_EXPONENT> (["f","F","d","D"])?
  | ["0"-"9"] (("_")* ["0"-"9"])* (<DECIMAL_EXPONENT>)? ["f","F","d","D"]>
  | <#DECIMAL_EXPONENT: ["e","E"] (["+","-"])? ["0"-"9"] (("_")* ["0"-"9"])*>
  | <#HEXADECIMAL_FLOATING_POINT_LITERAL:
    "0" ["x","X"] ["0"-"9","a"-"f","A"-"F"] (("_")* ["0"-"9","a"-"f","A"-"F"])* (".")?
    <HEXADECIMAL_EXPONENT> (["f","F","d","D"])?
  | "0" ["x","X"] (["0"-"9","a"-"f","A"-"F"] (("_")* ["0"-"9","a"-"f","A"-"F"])*)?
    "." ["0"-"9","a"-"f","A"-"F"] (("_")* ["0"-"9","a"-"f","A"-"F"])*
    <HEXADECIMAL_EXPONENT> (["f","F","d","D"])?>
  | <#HEXADECIMAL_EXPONENT: ["p","P"] (["+","-"])? ["0"-"9"] (("_")* ["0"-"9"])*>
  | <CHARACTER_LITERAL: "\'" (~["\'","\\","\n","\r"]
  | "\\" (["n","t","b","r","f","\\","\'","\""]
  | ["0"-"7"] (["0"-"7"])?
  | ["0"-"3"] ["0"-"7"] ["0"-"7"])) "\'">
  | <STRING_LITERAL: "\"" (~["\"","\\","\n","\r"]
  | "\\" (["n","t","b","r","f","\\","\'","\""]
  | ["0"-"7"] (["0"-"7"])?
  | ["0"-"3"] ["0"-"7"] ["0"-"7"]))* "\"">
}
```

<br>

### <a name="separators"></a>Separators

---

```java
/*
 * Separators
 */
<DEFAULT> TOKEN : { <LPAREN: "(">
                  | <RPAREN: ")">
                  | <LBRACE: "{">
                  | <RBRACE: "}">
                  | <LBRACKET: "[">
                  | <RBRACKET: "]">
                  | <SEMICOLON: ";">
                  | <COMMA: ",">
                  | <DOT: ".">
}
```

<br>

### <a name="operators"></a>Operators

---

```java
/*
 * Operators
 */
<DEFAULT> TOKEN : { <ASSIGN: "=">
                  | <LT: "<">
                  | <BANG: "!">
                  | <TILDE: "~">
                  | <HOOK: "?">
                  | <COLON: ":">
                  | <DOUBLECOLON: "::">
                  | <EQ: "==">
                  | <LE: "<=">
                  | <GE: ">=">
                  | <NE: "!=">
                  | <SC_OR: "||">
                  | <SC_AND: "&&">
                  | <INCR: "++">
                  | <DECR: "--">
                  | <PLUS: "+">
                  | <MINUS: "-">
                  | <STAR: "*">
                  | <SLASH: "/">
                  | <BIT_AND: "&">
                  | <BIT_OR: "|">
                  | <XOR: "^">
                  | <REM: "%">
                  | <PLUSASSIGN: "+=">
                  | <MINUSASSIGN: "-=">
                  | <STARASSIGN: "*=">
                  | <SLASHASSIGN: "/=">
                  | <ANDASSIGN: "&=">
                  | <ORASSIGN: "|=">
                  | <XORASSIGN: "^=">
                  | <REMASSIGN: "%=">
}

/*
 * >'s need special attention due to generics syntax.
 */

<DEFAULT> TOKEN : { <RUNSIGNEDSHIFT: ">>>"> : {
                  | <RSIGNEDSHIFT: ">>"> : {
                  | <GT: ">">
}
```

### <a name="identifiers"></a>Identifiers

---

```java
/*
 * Identifiers
 */
<DEFAULT> TOKEN : {
  <IDENTIFIER: <LETTER> (<PART_LETTER>)*>
| <#LETTER: ["$","A"-"Z","_","a"-"z","\u00a2"-"\u00a5","\u00aa","\u00b5","\u00ba",
  "\u00c0"-"\u00d6","\u00d8"-"\u00f6","\u00f8"-"\u021f","\u0222"-"\u0233","\u0250"-"\u02ad",
  "\u02b0"-"\u02b8","\u02bb"-"\u02c1","\u02d0"-"\u02d1","\u02e0"-"\u02e4","\u02ee","\u037a",
  "\u0386","\u0388"-"\u038a","\u038c","\u038e"-"\u03a1","\u03a3"-"\u03ce","\u03d0"-"\u03d7",
  "\u03da"-"\u03f3","\u0400"-"\u0481","\u048c"-"\u04c4","\u04c7"-"\u04c8","\u04cb"-"\u04cc",
  "\u04d0"-"\u04f5","\u04f8"-"\u04f9","\u0531"-"\u0556","\u0559","\u0561"-"\u0587",
  "\u05d0"-"\u05ea","\u05f0"-"\u05f2","\u0621"-"\u063a","\u0640"-"\u064a","\u0671"-"\u06d3",
  "\u06d5","\u06e5"-"\u06e6","\u06fa"-"\u06fc","\u0710","\u0712"-"\u072c","\u0780"-"\u07a5",
  "\u0905"-"\u0939","\u093d","\u0950","\u0958"-"\u0961","\u0985"-"\u098c","\u098f"-"\u0990",
  "\u0993"-"\u09a8","\u09aa"-"\u09b0","\u09b2","\u09b6"-"\u09b9","\u09dc"-"\u09dd",
  "\u09df"-"\u09e1","\u09f0"-"\u09f3","\u0a05"-"\u0a0a","\u0a0f"-"\u0a10","\u0a13"-"\u0a28",
  "\u0a2a"-"\u0a30","\u0a32"-"\u0a33","\u0a35"-"\u0a36","\u0a38"-"\u0a39","\u0a59"-"\u0a5c",
  "\u0a5e","\u0a72"-"\u0a74","\u0a85"-"\u0a8b","\u0a8d","\u0a8f"-"\u0a91","\u0a93"-"\u0aa8",
  "\u0aaa"-"\u0ab0","\u0ab2"-"\u0ab3","\u0ab5"-"\u0ab9","\u0abd","\u0ad0","\u0ae0",
  "\u0b05"-"\u0b0c","\u0b0f"-"\u0b10","\u0b13"-"\u0b28","\u0b2a"-"\u0b30","\u0b32"-"\u0b33",
  "\u0b36"-"\u0b39","\u0b3d","\u0b5c"-"\u0b5d","\u0b5f"-"\u0b61","\u0b85"-"\u0b8a",
  "\u0b8e"-"\u0b90","\u0b92"-"\u0b95","\u0b99"-"\u0b9a","\u0b9c","\u0b9e"-"\u0b9f",
  "\u0ba3"-"\u0ba4","\u0ba8"-"\u0baa","\u0bae"-"\u0bb5","\u0bb7"-"\u0bb9","\u0c05"-"\u0c0c",
  "\u0c0e"-"\u0c10","\u0c12"-"\u0c28","\u0c2a"-"\u0c33","\u0c35"-"\u0c39","\u0c60"-"\u0c61",
  "\u0c85"-"\u0c8c","\u0c8e"-"\u0c90","\u0c92"-"\u0ca8","\u0caa"-"\u0cb3","\u0cb5"-"\u0cb9",
  "\u0cde","\u0ce0"-"\u0ce1","\u0d05"-"\u0d0c","\u0d0e"-"\u0d10","\u0d12"-"\u0d28",
  "\u0d2a"-"\u0d39","\u0d60"-"\u0d61","\u0d85"-"\u0d96","\u0d9a"-"\u0db1","\u0db3"-"\u0dbb",
  "\u0dbd","\u0dc0"-"\u0dc6","\u0e01"-"\u0e30","\u0e32"-"\u0e33","\u0e3f"-"\u0e46",
  "\u0e81"-"\u0e82","\u0e84","\u0e87"-"\u0e88","\u0e8a","\u0e8d","\u0e94"-"\u0e97",
  "\u0e99"-"\u0e9f","\u0ea1"-"\u0ea3","\u0ea5","\u0ea7","\u0eaa"-"\u0eab","\u0ead"-"\u0eb0",
  "\u0eb2"-"\u0eb3","\u0ebd","\u0ec0"-"\u0ec4","\u0ec6","\u0edc"-"\u0edd","\u0f00",
  "\u0f40"-"\u0f47","\u0f49"-"\u0f6a","\u0f88"-"\u0f8b","\u1000"-"\u1021","\u1023"-"\u1027",
  "\u1029"-"\u102a","\u1050"-"\u1055","\u10a0"-"\u10c5","\u10d0"-"\u10f6","\u1100"-"\u1159",
  "\u115f"-"\u11a2","\u11a8"-"\u11f9","\u1200"-"\u1206","\u1208"-"\u1246","\u1248",
  "\u124a"-"\u124d","\u1250"-"\u1256","\u1258","\u125a"-"\u125d","\u1260"-"\u1286","\u1288",
  "\u128a"-"\u128d","\u1290"-"\u12ae","\u12b0","\u12b2"-"\u12b5","\u12b8"-"\u12be","\u12c0",
  "\u12c2"-"\u12c5","\u12c8"-"\u12ce","\u12d0"-"\u12d6","\u12d8"-"\u12ee","\u12f0"-"\u130e",
  "\u1310","\u1312"-"\u1315","\u1318"-"\u131e","\u1320"-"\u1346","\u1348"-"\u135a",
  "\u13a0"-"\u13f4","\u1401"-"\u166c","\u166f"-"\u1676","\u1681"-"\u169a","\u16a0"-"\u16ea",
  "\u1780"-"\u17b3","\u17db","\u1820"-"\u1877","\u1880"-"\u18a8","\u1e00"-"\u1e9b",
  "\u1ea0"-"\u1ef9","\u1f00"-"\u1f15","\u1f18"-"\u1f1d","\u1f20"-"\u1f45","\u1f48"-"\u1f4d",
  "\u1f50"-"\u1f57","\u1f59","\u1f5b","\u1f5d","\u1f5f"-"\u1f7d","\u1f80"-"\u1fb4",
  "\u1fb6"-"\u1fbc","\u1fbe","\u1fc2"-"\u1fc4","\u1fc6"-"\u1fcc","\u1fd0"-"\u1fd3",
  "\u1fd6"-"\u1fdb","\u1fe0"-"\u1fec","\u1ff2"-"\u1ff4","\u1ff6"-"\u1ffc","\u203f"-"\u2040",
  "\u207f","\u20a0"-"\u20af","\u2102","\u2107","\u210a"-"\u2113","\u2115","\u2119"-"\u211d",
  "\u2124","\u2126","\u2128","\u212a"-"\u212d","\u212f"-"\u2131","\u2133"-"\u2139",
  "\u2160"-"\u2183","\u3005"-"\u3007","\u3021"-"\u3029","\u3031"-"\u3035","\u3038"-"\u303a",
  "\u3041"-"\u3094","\u309d"-"\u309e","\u30a1"-"\u30fe","\u3105"-"\u312c","\u3131"-"\u318e",
  "\u31a0"-"\u31b7","\u3400"-"\u4db5","\u4e00"-"\u9fa5","\ua000"-"\ua48c","\uac00"-"\ud7a3",
  "\uf900"-"\ufa2d","\ufb00"-"\ufb06","\ufb13"-"\ufb17","\ufb1d","\ufb1f"-"\ufb28",
  "\ufb2a"-"\ufb36","\ufb38"-"\ufb3c","\ufb3e","\ufb40"-"\ufb41","\ufb43"-"\ufb44",
  "\ufb46"-"\ufbb1","\ufbd3"-"\ufd3d","\ufd50"-"\ufd8f","\ufd92"-"\ufdc7","\ufdf0"-"\ufdfb",
  "\ufe33"-"\ufe34","\ufe4d"-"\ufe4f","\ufe69","\ufe70"-"\ufe72","\ufe74","\ufe76"-"\ufefc",
  "\uff04","\uff21"-"\uff3a","\uff3f","\uff41"-"\uff5a","\uff65"-"\uffbe","\uffc2"-"\uffc7",
  "\uffca"-"\uffcf","\uffd2"-"\uffd7","\uffda"-"\uffdc","\uffe0"-"\uffe1","\uffe5"-"\uffe6"]>
| <#PART_LETTER: ["\u0000"-"\b","\u000e"-"\u001b","$","0"-"9","A"-"Z","_","a"-"z",
  "\u007f"-"\u009f","\u00a2"-"\u00a5","\u00aa","\u00b5","\u00ba","\u00c0"-"\u00d6",
  "\u00d8"-"\u00f6","\u00f8"-"\u021f","\u0222"-"\u0233","\u0250"-"\u02ad","\u02b0"-"\u02b8",
  "\u02bb"-"\u02c1","\u02d0"-"\u02d1","\u02e0"-"\u02e4","\u02ee","\u0300"-"\u034e",
  "\u0360"-"\u0362","\u037a","\u0386","\u0388"-"\u038a","\u038c","\u038e"-"\u03a1",
  "\u03a3"-"\u03ce","\u03d0"-"\u03d7","\u03da"-"\u03f3","\u0400"-"\u0481","\u0483"-"\u0486",
  "\u048c"-"\u04c4","\u04c7"-"\u04c8","\u04cb"-"\u04cc","\u04d0"-"\u04f5","\u04f8"-"\u04f9",
  "\u0531"-"\u0556","\u0559","\u0561"-"\u0587","\u0591"-"\u05a1","\u05a3"-"\u05b9",
  "\u05bb"-"\u05bd","\u05bf","\u05c1"-"\u05c2","\u05c4","\u05d0"-"\u05ea","\u05f0"-"\u05f2",
  "\u0621"-"\u063a","\u0640"-"\u0655","\u0660"-"\u0669","\u0670"-"\u06d3","\u06d5"-"\u06dc",
  "\u06df"-"\u06e8","\u06ea"-"\u06ed","\u06f0"-"\u06fc","\u070f"-"\u072c","\u0730"-"\u074a",
  "\u0780"-"\u07b0","\u0901"-"\u0903","\u0905"-"\u0939","\u093c"-"\u094d","\u0950"-"\u0954",
  "\u0958"-"\u0963","\u0966"-"\u096f","\u0981"-"\u0983","\u0985"-"\u098c","\u098f"-"\u0990",
  "\u0993"-"\u09a8","\u09aa"-"\u09b0","\u09b2","\u09b6"-"\u09b9","\u09bc","\u09be"-"\u09c4",
  "\u09c7"-"\u09c8","\u09cb"-"\u09cd","\u09d7","\u09dc"-"\u09dd","\u09df"-"\u09e3",
  "\u09e6"-"\u09f3","\u0a02","\u0a05"-"\u0a0a","\u0a0f"-"\u0a10","\u0a13"-"\u0a28",
  "\u0a2a"-"\u0a30","\u0a32"-"\u0a33","\u0a35"-"\u0a36","\u0a38"-"\u0a39","\u0a3c",
  "\u0a3e"-"\u0a42","\u0a47"-"\u0a48","\u0a4b"-"\u0a4d","\u0a59"-"\u0a5c","\u0a5e",
  "\u0a66"-"\u0a74","\u0a81"-"\u0a83","\u0a85"-"\u0a8b","\u0a8d","\u0a8f"-"\u0a91",
  "\u0a93"-"\u0aa8","\u0aaa"-"\u0ab0","\u0ab2"-"\u0ab3","\u0ab5"-"\u0ab9","\u0abc"-"\u0ac5",
  "\u0ac7"-"\u0ac9","\u0acb"-"\u0acd","\u0ad0","\u0ae0","\u0ae6"-"\u0aef","\u0b01"-"\u0b03",
  "\u0b05"-"\u0b0c","\u0b0f"-"\u0b10","\u0b13"-"\u0b28","\u0b2a"-"\u0b30","\u0b32"-"\u0b33",
  "\u0b36"-"\u0b39","\u0b3c"-"\u0b43","\u0b47"-"\u0b48","\u0b4b"-"\u0b4d","\u0b56"-"\u0b57",
  "\u0b5c"-"\u0b5d","\u0b5f"-"\u0b61","\u0b66"-"\u0b6f","\u0b82"-"\u0b83","\u0b85"-"\u0b8a",
  "\u0b8e"-"\u0b90","\u0b92"-"\u0b95","\u0b99"-"\u0b9a","\u0b9c","\u0b9e"-"\u0b9f",
  "\u0ba3"-"\u0ba4","\u0ba8"-"\u0baa","\u0bae"-"\u0bb5","\u0bb7"-"\u0bb9","\u0bbe"-"\u0bc2",
  "\u0bc6"-"\u0bc8","\u0bca"-"\u0bcd","\u0bd7","\u0be7"-"\u0bef","\u0c01"-"\u0c03",
  "\u0c05"-"\u0c0c","\u0c0e"-"\u0c10","\u0c12"-"\u0c28","\u0c2a"-"\u0c33","\u0c35"-"\u0c39",
  "\u0c3e"-"\u0c44","\u0c46"-"\u0c48","\u0c4a"-"\u0c4d","\u0c55"-"\u0c56","\u0c60"-"\u0c61",
  "\u0c66"-"\u0c6f","\u0c82"-"\u0c83","\u0c85"-"\u0c8c","\u0c8e"-"\u0c90","\u0c92"-"\u0ca8",
  "\u0caa"-"\u0cb3","\u0cb5"-"\u0cb9","\u0cbe"-"\u0cc4","\u0cc6"-"\u0cc8","\u0cca"-"\u0ccd",
  "\u0cd5"-"\u0cd6","\u0cde","\u0ce0"-"\u0ce1","\u0ce6"-"\u0cef","\u0d02"-"\u0d03",
  "\u0d05"-"\u0d0c","\u0d0e"-"\u0d10","\u0d12"-"\u0d28","\u0d2a"-"\u0d39","\u0d3e"-"\u0d43",
  "\u0d46"-"\u0d48","\u0d4a"-"\u0d4d","\u0d57","\u0d60"-"\u0d61","\u0d66"-"\u0d6f",
  "\u0d82"-"\u0d83","\u0d85"-"\u0d96","\u0d9a"-"\u0db1","\u0db3"-"\u0dbb","\u0dbd",
  "\u0dc0"-"\u0dc6","\u0dca","\u0dcf"-"\u0dd4","\u0dd6","\u0dd8"-"\u0ddf","\u0df2"-"\u0df3",
  "\u0e01"-"\u0e3a","\u0e3f"-"\u0e4e","\u0e50"-"\u0e59","\u0e81"-"\u0e82","\u0e84",
  "\u0e87"-"\u0e88","\u0e8a","\u0e8d","\u0e94"-"\u0e97","\u0e99"-"\u0e9f","\u0ea1"-"\u0ea3",
  "\u0ea5","\u0ea7","\u0eaa"-"\u0eab","\u0ead"-"\u0eb9","\u0ebb"-"\u0ebd","\u0ec0"-"\u0ec4",
  "\u0ec6","\u0ec8"-"\u0ecd","\u0ed0"-"\u0ed9","\u0edc"-"\u0edd","\u0f00","\u0f18"-"\u0f19",
  "\u0f20"-"\u0f29","\u0f35","\u0f37","\u0f39","\u0f3e"-"\u0f47","\u0f49"-"\u0f6a",
  "\u0f71"-"\u0f84","\u0f86"-"\u0f8b","\u0f90"-"\u0f97","\u0f99"-"\u0fbc","\u0fc6",
  "\u1000"-"\u1021","\u1023"-"\u1027","\u1029"-"\u102a","\u102c"-"\u1032","\u1036"-"\u1039",
  "\u1040"-"\u1049","\u1050"-"\u1059","\u10a0"-"\u10c5","\u10d0"-"\u10f6","\u1100"-"\u1159",
  "\u115f"-"\u11a2","\u11a8"-"\u11f9","\u1200"-"\u1206","\u1208"-"\u1246","\u1248",
  "\u124a"-"\u124d","\u1250"-"\u1256","\u1258","\u125a"-"\u125d","\u1260"-"\u1286",
  "\u1288","\u128a"-"\u128d","\u1290"-"\u12ae","\u12b0","\u12b2"-"\u12b5","\u12b8"-"\u12be",
  "\u12c0","\u12c2"-"\u12c5","\u12c8"-"\u12ce","\u12d0"-"\u12d6","\u12d8"-"\u12ee",
  "\u12f0"-"\u130e","\u1310","\u1312"-"\u1315","\u1318"-"\u131e","\u1320"-"\u1346",
  "\u1348"-"\u135a","\u1369"-"\u1371","\u13a0"-"\u13f4","\u1401"-"\u166c","\u166f"-"\u1676",
  "\u1681"-"\u169a","\u16a0"-"\u16ea","\u1780"-"\u17d3","\u17db","\u17e0"-"\u17e9",
  "\u180b"-"\u180e","\u1810"-"\u1819","\u1820"-"\u1877","\u1880"-"\u18a9","\u1e00"-"\u1e9b",
  "\u1ea0"-"\u1ef9","\u1f00"-"\u1f15","\u1f18"-"\u1f1d","\u1f20"-"\u1f45","\u1f48"-"\u1f4d",
  "\u1f50"-"\u1f57","\u1f59","\u1f5b","\u1f5d","\u1f5f"-"\u1f7d","\u1f80"-"\u1fb4",
  "\u1fb6"-"\u1fbc","\u1fbe","\u1fc2"-"\u1fc4","\u1fc6"-"\u1fcc","\u1fd0"-"\u1fd3",
  "\u1fd6"-"\u1fdb","\u1fe0"-"\u1fec","\u1ff2"-"\u1ff4","\u1ff6"-"\u1ffc","\u200c"-"\u200f",
  "\u202a"-"\u202e","\u203f"-"\u2040","\u206a"-"\u206f","\u207f","\u20a0"-"\u20af",
  "\u20d0"-"\u20dc","\u20e1","\u2102","\u2107","\u210a"-"\u2113","\u2115","\u2119"-"\u211d",
  "\u2124","\u2126","\u2128","\u212a"-"\u212d","\u212f"-"\u2131","\u2133"-"\u2139",
  "\u2160"-"\u2183","\u3005"-"\u3007","\u3021"-"\u302f","\u3031"-"\u3035","\u3038"-"\u303a",
  "\u3041"-"\u3094","\u3099"-"\u309a","\u309d"-"\u309e","\u30a1"-"\u30fe","\u3105"-"\u312c",
  "\u3131"-"\u318e","\u31a0"-"\u31b7","\u3400"-"\u4db5","\u4e00"-"\u9fa5","\ua000"-"\ua48c",
  "\uac00"-"\ud7a3","\uf900"-"\ufa2d","\ufb00"-"\ufb06","\ufb13"-"\ufb17","\ufb1d"-"\ufb28",
  "\ufb2a"-"\ufb36","\ufb38"-"\ufb3c","\ufb3e","\ufb40"-"\ufb41","\ufb43"-"\ufb44",
  "\ufb46"-"\ufbb1","\ufbd3"-"\ufd3d","\ufd50"-"\ufd8f","\ufd92"-"\ufdc7","\ufdf0"-"\ufdfb",
  "\ufe20"-"\ufe23","\ufe33"-"\ufe34","\ufe4d"-"\ufe4f","\ufe69","\ufe70"-"\ufe72","\ufe74",
  "\ufe76"-"\ufefc","\ufeff","\uff04","\uff10"-"\uff19","\uff21"-"\uff3a","\uff3f",
  "\uff41"-"\uff5a","\uff65"-"\uffbe","\uffc2"-"\uffc7","\uffca"-"\uffcf","\uffd2"-"\uffd7",
  "\uffda"-"\uffdc","\uffe0"-"\uffe1","\uffe5"-"\uffe6","\ufff9"-"\ufffb"]>
}

<DEFAULT> TOKEN : {
  <ACT_TOK: ~[]>
}
```

<br>

## <a name="non-terminals"></a>Non-Terminals

### <a name="javacc-grammar"></a>JavaCC grammar

---

```java
javacc_input ::= javacc_options
                 "PARSER_BEGIN" "(" identifier ")"
                 CompilationUnit
                 "PARSER_END" "(" identifier ")"
                 ( production )+
                 <EOF>
```

```java
javacc_options ::= ( <IDENTIFIER> "{" ( option_binding )* "}" )?
```

```java
option_binding ::= ( <IDENTIFIER> | "LOOKAHEAD" | "IGNORE_CASE" | "static" | "PARSER_BEGIN" )
                   "="
                   ( IntegerLiteral | BooleanLiteral | StringLiteral | StringList )
                   ";"
```

```java
StringList ::= "(" StringLiteral ( "," StringLiteral )* ")"
```

```java
production ::= javacode_production
             | cppcode_production
             | regular_expr_production
             | token_manager_decls
             | bnf_production
```

```java
javacode_production ::= "JAVACODE"
                        AccessModifier
                        ResultType
                        identifier
                        FormalParameters ( "throws" Name ( "," Name )* )?
                        Block
```

```java
cppcode_production ::= "CPPCODE"
                       AccessModifier
                       ResultType
                       identifier
                       FormalParameters ( "throws" Name ( "," Name )* )?
                       Block
```

```java
bnf_production ::= AccessModifier
                   ResultType
                   identifier
                   FormalParameters ( "throws" Name ( "," Name )* )?
                   ":"
                   Block
                   "{" expansion_choices "}"
```

```java
AccessModifier ::= ( "public" | "protected" | "private" )?
```

```java
regular_expr_production ::= ( "<" "*" ">"
                            | "<" <IDENTIFIER> ( "," <IDENTIFIER> )* ">"
                            )?                     
                            regexpr_kind ( "[" "IGNORE_CASE" "]" )? ":" "{"
                            regexpr_spec ( "|" regexpr_spec )* "}"
```

```java
token_manager_decls ::= "TOKEN_MGR_DECLS" ":" ( ClassOrInterfaceBody )?
```

```java
regexpr_kind ::= "TOKEN"
               | "SPECIAL_TOKEN"
               | "SKIP"
               | "MORE"
```

```java
regexpr_spec ::= regular_expression
                 ( Block )?
                 ( ":" <IDENTIFIER> )?
```

```java
expansion_choices ::= expansion ( "|" expansion )*
```

```java
expansion ::= ( "LOOKAHEAD" "(" local_lookahead ")" )?
              ( expansion_unit )+
```

```java
local_lookahead ::= ( IntegerLiteral )?
                    ( "," )?
                    ( expansion_choices )?
                    ( "," )?
                    ( "{" ( Expression )? "}" )?
```

```java
expansion_unit ::= "LOOKAHEAD" "(" local_lookahead ")"
                 | Block
                 | "[" expansion_choices "]"
                 | "try" "{" expansion_choices "}"
                 ( "catch" "(" ( Name <IDENTIFIER> )? ")" Block )*
                 ( "finally" Block )?
                 | ( PrimaryExpression "=" )?
                 (
                   identifier ( TypeArguments )? Arguments
                 | regular_expression ( "." <IDENTIFIER> )?
                 )
                 | "(" expansion_choices ")"
                 ( "+" | "*" | "?" )?
```

```java
regular_expression ::= StringLiteral
                     | <LANGLE: "<">
                       ( ( "#" )? identifier ":" )?
                       complex_regular_expression_choices <RANGLE: ">">
                     | "<" identifier ">"
                     | "<" "EOF" ">"
```

```java
complex_regular_expression_choices ::= complex_regular_expression
                                       ( "|" complex_regular_expression )*
```

```java
complex_regular_expression ::= ( complex_regular_expression_unit )+
```

```java
complex_regular_expression_unit ::= StringLiteral
                                  | "<" identifier ">"
                                  | character_list
                                  | "(" complex_regular_expression_choices ")"
                                  (
                                    "+" | "*" | "?" | "{"
                                    IntegerLiteral
                                    ( "," ( IntegerLiteral )? )?
                                    "}"
                                  )?
```

```java
character_list ::= ( "~" )? "[" ( character_descriptor ( "," character_descriptor )* )? "]"
```

```java
character_descriptor ::= StringLiteral ( "-" StringLiteral )?
```

```java
identifier ::= <IDENTIFIER>
```

<br>

## <a name="java-grammar"></a>Java Grammar

The Java grammar is modified to use sequences of tokens for the missing tokens - those that include "<<" and ">>".

### <a name="java-identifiers"></a>Java identifiers

---

```java
/*
 * The following production defines Java identifiers - it includes the reserved words of JavaCC also.
 */
JavaIdentifier ::= ( <IDENTIFIER>
                   | "LOOKAHEAD"
                   | "IGNORE_CASE"
                   | "PARSER_BEGIN"
                   | "PARSER_END"
                   | "JAVACODE"
                   | "TOKEN"
                   | "SPECIAL_TOKEN"
                   | "MORE"
                   | "SKIP"
                   | "TOKEN_MGR_DECLS"
                   | "EOF"
                   | "template"
                   | "DCL_PARSER_BEGIN"
                   | "DCL_PARSER_END"
                   | "INC_PARSER_BEGIN"
                   | "INC_PARSER_END"
                   | "DEF_PARSER_BEGIN"
                   | "DEF_PARSER_END"
)
```

<br>

### <a name="program-structure"></a>Program structure

---

```java
CompilationUnit ::= ( PackageDeclaration )?
                    ( ImportDeclaration )*
                    ( TypeDeclaration )*
```

```java
PackageDeclaration ::= Modifiers "package" Name ";"
```

```java
ImportDeclaration ::= "import"
                      ( "static" )?
                      Name
                      ( "." "*" )?
                      ";"
```

<br>

### <a name="modifiers"></a>Modifiers

---

```java
/*
 * Modifiers - we match all modifiers in a single rule to reduce the chances of
 * syntax errors for simple modifier mistakes. It will also enable us to give
 * better error messages.
 */
Modifiers ::= (
  (
      "public"
    | "static"
    | "protected"
    | "private"
    | "final"
    | "abstract"
    | "synchronized"
    | "native"
    | "transient"
    | "volatile"
    | "strictfp"
    | Annotation
  )
)*
```

<br>

### <a name="declarations"></a>Declarations

---

```java
TypeDeclaration ::= ";"
                  | Modifiers
                  ( ClassOrInterfaceDeclaration | EnumDeclaration | AnnotationTypeDeclaration )
```

```java
ClassOrInterfaceDeclaration ::= ( "class" | "interface" )
                                JavaIdentifier ( TypeParameters )?
                                ( ExtendsList )?
                                ( ImplementsList )?
                                ClassOrInterfaceBody
```

```java
ExtendsList ::= "extends"
                ClassOrInterfaceType
                ( "," ClassOrInterfaceType )*
```

```java
ImplementsList ::= "implements"
                   ClassOrInterfaceType
                   ( "," ClassOrInterfaceType )*
```

```java
EnumDeclaration ::= "enum"
                    JavaIdentifier
                    ( ImplementsList )?
                    EnumBody
```

```java
EnumBody ::= "{"
             ( EnumConstant ( "," EnumConstant )* )?
             ( "," )?
             ( ";" ( ClassOrInterfaceBodyDeclaration )* )?
             "}"
```

```java
EnumConstant ::= Modifiers
                 JavaIdentifier
                 ( Arguments )?
                 ( ClassOrInterfaceBody )?
```

```java
TypeParameters ::= "<" TypeParameter ( "," TypeParameter )* ">"
```

```java
TypeParameter ::= JavaIdentifier ( TypeBound )?
```

```java
TypeBound ::= "extends"
              ClassOrInterfaceType
              ( "&" ClassOrInterfaceType )*
```

```java
ClassOrInterfaceBody ::= "{" ( ClassOrInterfaceBodyDeclaration )* "}"
```

```java
ClassOrInterfaceBodyDeclaration ::= Initializer
                                  | Modifiers
                                  (
                                    ClassOrInterfaceDeclaration
                                  | EnumDeclaration
                                  | ConstructorDeclaration
                                  | FieldDeclaration
                                  | MethodDeclaration
                                  )
                                  | ";"
```

```java
FieldDeclaration ::= Type VariableDeclarator ( "," VariableDeclarator )* ";"
```

```java
VariableDeclarator ::= VariableDeclaratorId ( "=" VariableInitializer )?
```

```java
VariableDeclaratorId ::= JavaIdentifier ( "[" "]" )*
```

```java
VariableInitializer ::= ArrayInitializer
                      | Expression
```

```java
ArrayInitializer ::= "{"
                     ( VariableInitializer ( "," VariableInitializer )* )?
                     ( "," )?
                     "}"
```

```java
MethodDeclaration ::= ( TypeParameters )?
                      ResultType
                      MethodDeclarator
                      ( "throws" NameList )?
                      ( Block | ";" )
```

```java
MethodDeclarator ::= JavaIdentifier FormalParameters ( "[" "]" )*
```

```java
FormalParameters ::= "(" ( FormalParameter ( "," FormalParameter )* )? ")"
```

```java
FormalParameter ::= Modifiers
                    Type
                    ( ( "&" | "*" ) | "..." )?
                    VariableDeclaratorId
```

```java
ConstructorDeclaration ::= ( TypeParameters )?
                           JavaIdentifier
                           FormalParameters
                           ( "throws" NameList )?
                           "{"
                           ( ExplicitConstructorInvocation )?
                           ( BlockStatement )*
                           "}"
```

```java
ExplicitConstructorInvocation ::= "this" Arguments ";"
                                | ( PrimaryExpression "." )? "super" Arguments ";"
```

```java
Initializer ::= ( "static" )? Block
```

### <a name="types"></a>Types

```java
Type ::= ReferenceType | PrimitiveType
```

```java
ReferenceType ::= PrimitiveType ( "[" "]" )+
                | ( ( Template )? ClassOrInterfaceType ) ( "[" "]" )*
```

```java
Template ::= "template"
             "<"
             TemplateBase
             ( "," TemplateBase )*
             ">"
```

```java
TemplateBase ::= TemplatePack ( "..." )? <IDENTIFIER>
```

```java
TemplatePack ::= ( "class" | "typename" )
```

```java
ClassOrInterfaceType ::= ( "::" )?
                         <IDENTIFIER>
                         ( TypeArguments )?
                         (
                           ( "." | "::" )
                           <IDENTIFIER>
                           ( TypeArguments )?
                         )*
```

```java
TypeArguments ::= "<"
                  ( TypeArgument
                    ( "," TypeArgument
                      ( "..." )?
                    )*
                  )?
                  ">"
```

```java
TypeArgument ::= ReferenceType | "?" ( WildcardBounds )?
```

```java
WildcardBounds ::= "extends" ReferenceType | "super" ReferenceType
```

```java
PrimitiveType ::= "boolean"
                | "char"
                | "byte"
                | "short"
                | "int"
                | "long"
                | "float"
                | "double"
```

```java
ResultType ::= ( "void" ( "*" )?
               | ( "const" )? Type ( "*" | "&" )? )
```

```java
Name ::= JavaIdentifier ( "." JavaIdentifier )*
```

```java
NameList ::= Name ( "," Name )*
```

<br>

---

### <a name="expressions"></a>Expressions

```java
Expression ::= ConditionalExpression ( AssignmentOperator Expression )?
```

```java
AssignmentOperator ::= "="
                     | "*="
                     | "/="
                     | "%="
                     | "+="
                     | "-="
                     | "<<="
                     | ">>="
                     | ">>>="
                     | "&="
                     | "^="
                     | "|="
```

```java
ConditionalExpression ::= ConditionalOrExpression ( "?" Expression ":" Expression )?
```

```java
ConditionalOrExpression ::= ConditionalAndExpression ( "||" ConditionalAndExpression )*
```

```java
ConditionalAndExpression ::= InclusiveOrExpression ( "&&" InclusiveOrExpression )*
```

```java
InclusiveOrExpression ::= ExclusiveOrExpression ( "|" ExclusiveOrExpression )*
```

```java
ExclusiveOrExpression ::= AndExpression ( "^" AndExpression )*
```

```java
AndExpression ::= EqualityExpression ( "&" EqualityExpression )*
```

```java
EqualityExpression ::= InstanceOfExpression ( ( "==" | "!=" ) InstanceOfExpression )*
```

```java
InstanceOfExpression ::= RelationalExpression ( "instanceof" Type )?
```

```java
RelationalExpression ::= ShiftExpression ( ( "<" | ">" | "<=" | ">=" ) ShiftExpression )*
```

```java
ShiftExpression ::= AdditiveExpression
                    (
                      ( "<<" | RSIGNEDSHIFT | RUNSIGNEDSHIFT )
                      AdditiveExpression
                    )*
```

```java
AdditiveExpression ::= MultiplicativeExpression
                       (
                         ( "+" | "-" )
                         MultiplicativeExpression
                       )*
```

```java
MultiplicativeExpression ::= UnaryExpression ( ( "*" | "/" | "%" ) UnaryExpression )*
```

```java
UnaryExpression ::= ( "+" | "-" )
                    UnaryExpression
                  | PreIncrementExpression
                  | PreDecrementExpression
                  | UnaryExpressionNotPlusMinus
```

```java
PreIncrementExpression ::= "++" PrimaryExpression
```

```java
PreDecrementExpression ::= "--" PrimaryExpression
```

```java
UnaryExpressionNotPlusMinus ::= ( "~" | "!" )
                                UnaryExpression
                              | CastExpression
                              | PostfixExpression
```

```java
/*
 * This production is to determine lookahead only. The LOOKAHEAD specifications
 * below are not used, but they are there just to indicate that we know about this.
 */
CastLookahead ::= "(" PrimitiveType
                | "(" Type "[" "]"
                | "(" Type ")"
                ( "~" | "!" | "(" | JavaIdentifier | "this" | "super" | "new" | Literal )
```

```java
PostfixExpression ::= PrimaryExpression ( "++" | "--" )?
```

```java
CastExpression ::= "(" Type ")" UnaryExpression
                 | "(" Type ")" UnaryExpressionNotPlusMinus
```

```java
PrimaryExpression ::= PrimaryPrefix ( PrimarySuffix )*
```

```java
MemberSelector ::= "." TypeArguments JavaIdentifier
```

```java
PrimaryPrefix ::= Literal
                | "this"
                | "super" "." JavaIdentifier
                | "(" Expression ")"
                | AllocationExpression
                | ResultType "." "class"
                | Name
```

```java
PrimarySuffix ::= "." "this"
                | "." AllocationExpression
                | MemberSelector
                | "[" Expression "]"
                | "." JavaIdentifier
                | Arguments
```

```java
Literal ::= <INTEGER_LITERAL>
          | <FLOATING_POINT_LITERAL>
          | <CHARACTER_LITERAL>
          | <STRING_LITERAL>
          | BooleanLiteral
          | NullLiteral
```

```java
IntegerLiteral ::= <INTEGER_LITERAL>
```

```java
BooleanLiteral ::= "true" | "false"
```

```java
StringLiteral ::= <STRING_LITERAL>
```

```java
NullLiteral ::= "null"
```

```java
Arguments ::= "(" ( ArgumentList )? ")"
```

```java
ArgumentList ::= Expression ( "," Expression )*
```

```java
AllocationExpression ::= "new" PrimitiveType ArrayDimsAndInits
                       | "new" ClassOrInterfaceType ( TypeArguments )?
                       ( ArrayDimsAndInits | Arguments ( ClassOrInterfaceBody )? )
```

```java
/*
 * This LOOKAHEAD specification is to parse to PrimarySuffix
 * if there is an expression between the "[...]".
 */
ArrayDimsAndInits ::= ( "[" Expression "]" )+ ( "[" "]" )*
                    | ( "[" "]" )+ ArrayInitializer
```

<br>

### <a name="statements"></a>Statements

---

```java
Statement ::= LabeledStatement
            | AssertStatement
            | Block
            | EmptyStatement
            | StatementExpression ";"
            | SwitchStatement
            | IfStatement
            | WhileStatement
            | DoStatement
            | ForStatement
            | BreakStatement
            | ContinueStatement
            | ReturnStatement
            | ThrowStatement
            | SynchronizedStatement
            | TryStatement
```

```java
AssertStatement ::= "assert" Expression ( ":" Expression )? ";"
```

```java
LabeledStatement ::= JavaIdentifier ":" Statement
```

```java
Block ::= "{" ( BlockStatement )* "}"
```

```java
BlockStatement ::= LocalVariableDeclaration ";"
                 | Statement
                 | ClassOrInterfaceDeclaration
```

```java
LocalVariableDeclaration ::= Modifiers Type VariableDeclarator ( "," VariableDeclarator )*
```

```java
EmptyStatement ::= ";"
```

```java
StatementExpression ::= PreIncrementExpression
                      | PreDecrementExpression
                      | PrimaryExpression
                      ( "++" | "--" | AssignmentOperator Expression )?
```

```java
SwitchStatement ::= "switch" "(" Expression ")" "{" ( SwitchLabel ( BlockStatement )* )* "}"
```

```java
SwitchLabel ::= "case" Expression ":"
              | "default" ":"
```

```java
IfStatement ::= "if" "(" Expression ")" Statement ( "else" Statement )?
```

```java
WhileStatement ::= "while" "(" Expression ")" Statement
```

```java
DoStatement ::= "do" Statement "while" "(" Expression ")" ";"
```

```java
ForStatement ::= "for" "("
                 ( Modifiers Type JavaIdentifier ":" Expression
                   | ( ForInit )? ";" ( Expression )? ";" ( ForUpdate )?
                 )
                 ")"
                 Statement
```

```java
ForInit ::= LocalVariableDeclaration | StatementExpressionList
```

```java
StatementExpressionList ::= StatementExpression ( "," StatementExpression )*
```

```java
ForUpdate ::= StatementExpressionList
```

```java
BreakStatement ::= "break" ( JavaIdentifier )? ";"
```

```java
ContinueStatement ::= "continue" ( JavaIdentifier )? ";"
```

```java
ReturnStatement ::= "return" ( Expression )? ";"
```

```java
ThrowStatement ::= "throw" Expression ";"
```

```java
SynchronizedStatement ::= "synchronized" "(" Expression ")" Block
```

```java
ResourceDeclaration ::= Type VariableDeclaratorId "=" Expression
```

```java
CatchParameter ::= Modifiers
                   Type
                   ( ( "&" | "*" ) | "..." )? ( "|" Type )*
                   VariableDeclaratorId
```

```java
TryStatement ::= "try" ( "(" ResourceDeclaration
                   ( ";" ResourceDeclaration )* ( ";" )? ")" )?
                   Block
                   ( "catch" "(" CatchParameter ")" Block )*
                   ( "finally" Block )?
```

```java
/*
 * We use productions to match >>>, >> and > so that we can
 * keep the type declaration syntax with generics clean.
 */

RUNSIGNEDSHIFT ::= ( ">" ">" ">" )

RSIGNEDSHIFT ::= ( ">" ">" )
```

<br>

### <a name="annotations"></a>Annotations

---

```java
Annotation ::= NormalAnnotation | SingleMemberAnnotation | MarkerAnnotation
```

```java
NormalAnnotation ::= "@" Name "(" ( MemberValuePairs )? ")"
```

```java
MarkerAnnotation ::= "@" Name
```

```java
SingleMemberAnnotation ::= "@" Name "(" MemberValue ")"
```

```java
MemberValuePairs ::= MemberValuePair ( "," MemberValuePair )*
```

```java
MemberValuePair ::= JavaIdentifier "=" MemberValue
```

```java
MemberValue ::= Annotation | MemberValueArrayInitializer | ConditionalExpression
```

```java
MemberValueArrayInitializer ::= "{" MemberValue ( "," MemberValue )* ( "," )? "}"
```

<br>

### <a name="annotation-types"></a>Annotation types

---

```java
AnnotationTypeDeclaration ::= "@" "interface" JavaIdentifier AnnotationTypeBody
```

```java
AnnotationTypeBody ::= "{" ( AnnotationTypeMemberDeclaration )* "}"
```

```java
AnnotationTypeMemberDeclaration ::= Modifiers
                                    ( Type JavaIdentifier "(" ")" ( DefaultValue )? ";"
                                    | ClassOrInterfaceDeclaration
                                    | EnumDeclaration
                                    | AnnotationTypeDeclaration
                                    | FieldDeclaration
                                    )
                                    | ( ";" )
```

```java
DefaultValue ::= "default" MemberValue
```

<br>

---

[NEXT >>](api.md)

<br>
