/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

/**
 * SECTION: gnome-scan-utils
 * @title: Miscellaneous utils
 * 
 **/
#include "gnome-scan-utils.h"
#include "gnome-scan-types.h"

#define	GS_WARN_UNSUPPORTED_UNIT(unit)	g_warning("%s: Unsupported unit %s.", __FUNCTION__, gs_enum_get_nick_from_value (GNOME_TYPE_SCAN_UNIT, unit))

const gchar*
gs_enum_get_nick_from_value (GType type, guint value)
{
	GEnumClass *klass;
	GEnumValue *val;
	
	klass = g_type_class_ref (type);
	val = g_enum_get_value (klass, value);
	g_type_class_unref (klass);
	
	return g_strdup (val->value_nick);
}

/**
 * gs_convert:
 * @val:	the value to convert
 * @from:	the current @val unit
 * @to:		the target unit fro @val
 * @res:	the resolution in dpi
 *
 * Convert @val from @from unit to @to at @res resolution. Useful for
 * preview area, #GnomeScanner, etc.
 **/
gdouble
gs_convert (gdouble val,
			GnomeScanUnit from,
			GnomeScanUnit to,
			gdouble res)
{
	if (from == to)
		return val;
	
	switch (from) {
	case GS_UNIT_NONE:
	case GS_UNIT_BIT:
	case GS_UNIT_PERCENT:
	case GS_UNIT_MICROSECOND:
	case GS_UNIT_DPI:
		GS_WARN_UNSUPPORTED_UNIT(from);
		return val;
	default:
		switch (to) {
		case GS_UNIT_NONE:
		case GS_UNIT_BIT:
		case GS_UNIT_PERCENT:
		case GS_UNIT_MICROSECOND:
		case GS_UNIT_DPI:
			GS_WARN_UNSUPPORTED_UNIT(from);
			return val;
		default:
			return gs_convert_from_mm (gs_convert_to_mm (val, from, res),
									   to, res);
		}
	}
}


gdouble
gs_convert_to_mm (gdouble val,
				  GnomeScanUnit unit,
				  gdouble res)
{
	switch (unit) {
	case GS_UNIT_NONE:
	case GS_UNIT_BIT:
	case GS_UNIT_PERCENT:
	case GS_UNIT_MICROSECOND:
	case GS_UNIT_DPI:
		GS_WARN_UNSUPPORTED_UNIT(unit);
		return val;
	case GS_UNIT_MM:
		return val;
	case GS_UNIT_PIXEL:
		return val * (MM_PER_INCH / res);
	}
}

gdouble
gs_convert_from_mm (gdouble val,
					GnomeScanUnit unit,
					gdouble res)
{
	switch (unit) {
	case GS_UNIT_NONE:
	case GS_UNIT_BIT:
	case GS_UNIT_PERCENT:
	case GS_UNIT_MICROSECOND:
	case GS_UNIT_DPI:
		GS_WARN_UNSUPPORTED_UNIT(unit);
		return val;
	case GS_UNIT_MM:
		return val;
	case GS_UNIT_PIXEL:
		return val / (MM_PER_INCH / res);
	}
}

GSRectangle*
gs_rectangle_convert (GSRectangle *r,
					  GnomeScanUnit from,
					  GnomeScanUnit to,
					  gdouble res)
{
	return gs_rectangle_convert_from_mm (gs_rectangle_convert_to_mm (r, from, res),
										 to, res);
}

GSRectangle*
gs_rectangle_convert_to_mm (GSRectangle *r,
							GnomeScanUnit unit,
							gdouble res)
{
	GSRectangle*rect = g_new0 (GSRectangle, 1);
#define conv(var)	rect->var = gs_convert_to_mm (r->var, unit, res)
	conv(x);
	conv(y);
	conv(width);
	conv(height);
#undef	conv
	return rect;
}

GSRectangle*
gs_rectangle_convert_from_mm (GSRectangle *r,
							  GnomeScanUnit unit,
							  gdouble res)
{
	GSRectangle*rect = g_new0 (GSRectangle, 1);
#define conv(var)	rect->var = gs_convert_from_mm (r->var, unit, res)
	conv(x);
	conv(y);
	conv(width);
	conv(height);
#undef	conv
	return rect;
}

GSRectangle*
gs_rectangle_rotate(GSRectangle *r,
					GSRectangle *a,
					guint angle)
{
	GSRectangle *res = g_memdup(r, sizeof(GSRectangle));
	angle%= 360;
	
	switch (angle)
	{
		case 0:
			break;
		case 270:
			res->width = r->height;
			res->height = r->width;
			res->x = r->y;
			res->y = MAX(a->width - r->x - r->width, a->x);
			break;
		case 180:
			res->x = MAX(a->width - r->x - r->width, a->x);
			res->y = MAX(a->height - r->y - r->height, a->y);
			break;
		case 90:
			res->width = r->height;
			res->height = r->width;
			res->y = r->x;
			res->x = MAX(a->height - r->y - r->height, a->y);
			break;
		default:
			g_warning("%s: %i degree rotation is not supported",
					  __FUNCTION__, angle);
			break;
	}
	return res;
}
