/* flower.c
**
**    code for a flower
**
**    Updated Version foobillard++ started at 12/2010
**    Copyright (C) 2010 - 2013 Holger Schaekel (foobillardplus@go4more.de)
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License Version 2 as
**    published by the Free Software Foundation;
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include "flower.h"

static int flower_id = -1;

/***********************************************************************
 *        This is a general routine to make a new material active      *
 ***********************************************************************/

void flower_Material_Parameters(float shininess, float r_diffuseColor, float g_diffuseColor, float b_diffuseColor, float r_specularColor, float g_specularColor, float b_specularColor) {
	GLfloat glfModelSpecular[4];

	// This causes the polygon's ambient & diffuse colors to be derived from the glColor() value.
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess);
	/* Set the specular shading color. */
	glfModelSpecular[0] = r_specularColor;
	glfModelSpecular[1] = g_specularColor;
	glfModelSpecular[2] = b_specularColor;
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, glfModelSpecular);
	glColor3f(r_diffuseColor, g_diffuseColor, b_diffuseColor);
}

/***********************************************************************
 *                          Make an glcallist                          *
 ***********************************************************************/

void init_flower(void) {
	 if(flower_id != -1) {
	 	glDeleteLists(flower_id,1);
	 }
  fprintf(stderr,"Initialize flower graphics\n");
  flower_id = glGenLists(1);
  glNewList(flower_id, GL_COMPILE);
  flower();
  glEndList();
}

/***********************************************************************
 *                         display the flower                          *
 ***********************************************************************/

void display_flower(void) {
	 if(flower_id == -1) {
	 	init_flower();
	 } else {
	 	glCallList(flower_id);
	 }
}

/***********************************************************************
 *                           build the flower                          *
 *                 Polygons for object (2484 polygons)                 *
 ***********************************************************************/

void flower(void) {
	static const GLfloat VertexData[] = {
		-1.15726f, -0.171149f, 3.68038f, -1.13963f, -0.148076f, 3.68600f, -1.13876f, -0.153079f, 3.67196f, 
		-1.13519f, -0.133426f, 3.70669f, -1.13217f, -0.142503f, 3.67866f, 
		-1.13198f, -0.132926f, 4.35952f, -1.12980f, -0.129402f, 3.70182f, 
		-1.12808f, -0.119259f, 4.36400f, -1.12712f, -0.165060f, 4.36963f, 
		-1.12467f, -0.0721900f, 4.33576f, -1.12408f, -0.132552f, 4.37110f, 
		-1.11862f, -0.129532f, 4.37886f, -1.11779f, -0.148335f, 4.38181f, 
		-1.11607f, -0.0744300f, 4.34886f, -1.10743f, -0.0830630f, 4.36265f, 
		-1.10273f, -0.108911f, 3.70417f, -1.09929f, -0.0902290f, 3.72537f, 
		-1.09838f, -0.0869250f, 4.37666f, -1.09750f, -0.00310900f, 4.28264f, 
		-1.09622f, -0.0715810f, 3.74619f, -1.09495f, -0.110251f, 3.68985f, 
		-1.09048f, -0.0958970f, 4.38950f, -1.08725f, -0.109987f, 3.67805f, 
		-1.08133f, -0.0101710f, 4.30805f, -1.06501f, -0.0159240f, 4.33369f, 
		-1.06398f, -0.0172830f, 3.79416f, -1.05551f, -0.0423540f, 3.76562f, 
		-1.04938f, 0.0588000f, 4.21483f, -1.04840f, -0.0190610f, 4.35980f, 
		-1.04456f, -0.0618090f, 3.73898f, -1.03165f, -0.0208890f, 4.38613f, 
		-1.02865f, -0.0700340f, 3.71617f, -1.02596f, 0.0518500f, 4.25164f, 
		-1.02549f, 0.0183250f, 3.85081f, -1.01026f, -0.0726440f, 3.69527f, 
		-1.00206f, 0.0473750f, 4.28920f, -0.998896f, -0.00209700f, 3.81629f, 
		-0.979253f, 0.0980230f, 4.14700f, -0.977219f, 0.0478490f, 4.32824f, 
		-0.976878f, -0.364339f, 3.97533f, -0.972640f, -0.0192090f, 3.78230f, 
		-0.969399f, -0.354723f, 3.96753f, -0.967768f, 0.0241000f, 3.91633f, 
		-0.967057f, -0.352061f, 3.97463f, -0.964821f, -0.349095f, 3.96954f, 
		-0.960785f, -0.344265f, 3.98399f, -0.959078f, -0.342123f, 3.98052f, 
		-0.952067f, 0.0899660f, 4.18972f, -0.951904f, 0.0507990f, 4.36802f, 
		-0.947065f, -0.0297010f, 3.74935f, -0.945233f, -0.331219f, 3.97634f, 
		-0.944218f, -0.331932f, 3.96773f, -0.942738f, -0.331792f, 3.96038f, 
		-0.939356f, -0.321277f, 3.98614f, -0.933732f, -0.311354f, 3.99583f, 
		-0.929878f, 0.0142220f, 3.87111f, -0.923801f, 0.0853450f, 4.23414f, 
		-0.921831f, -0.0368830f, 3.71692f, -0.909780f, -0.295801f, 3.99740f, 
		-0.909620f, -0.306154f, 3.98208f, -0.908337f, -0.282459f, 4.01318f, 
		-0.906256f, -0.310531f, 3.96766f, -0.903917f, -0.445619f, 4.01510f, 
		-0.903917f, -0.432733f, 4.01510f, -0.903917f, -0.397979f, 4.01510f, 
		-0.903917f, -0.363420f, 4.01510f, -0.903917f, -0.351120f, 4.01510f, 
		-0.901288f, -0.311920f, 3.95371f, -0.900921f, -0.446874f, 4.01510f, 
		-0.899645f, 0.104234f, 4.08602f, -0.899487f, -0.350451f, 4.01510f, 
		-0.898059f, -0.445619f, 4.07401f, -0.898059f, -0.432733f, 4.07401f, 
		-0.898059f, -0.397979f, 4.07401f, -0.898059f, -0.363420f, 4.07401f, 
		-0.898059f, -0.351120f, 4.07401f, -0.898058f, -0.445619f, 3.95620f, 
		-0.898058f, -0.432733f, 3.95620f, -0.898058f, -0.397979f, 3.95620f, 
		-0.898058f, -0.363420f, 3.95620f, -0.898058f, -0.351120f, 3.95620f, 
		-0.895121f, -0.446874f, 4.07342f, -0.895120f, -0.446874f, 3.95678f, 
		-0.894462f, 0.00645400f, 3.82600f, -0.893715f, -0.350451f, 4.07314f, 
		-0.893714f, -0.350451f, 3.95706f, -0.893373f, 0.0875960f, 4.28196f, 
		-0.893119f, 0.00775300f, 3.98007f, -0.891933f, -0.446734f, 4.01510f, 
		-0.886307f, -0.446734f, 3.95854f, -0.886307f, -0.446734f, 4.07167f, 
		-0.886198f, -0.350786f, 4.01510f, -0.882887f, -0.446408f, 4.01510f, 
		-0.880929f, -0.445619f, 3.89954f, -0.880929f, -0.445619f, 4.13067f, 
		-0.880929f, -0.432733f, 3.89954f, -0.880929f, -0.432733f, 4.13067f, 
		-0.880929f, -0.397979f, 3.89954f, -0.880929f, -0.397979f, 4.13067f, 
		-0.880929f, -0.363420f, 3.89954f, -0.880929f, -0.363420f, 4.13067f, 
		-0.880929f, -0.351120f, 3.89954f, -0.880929f, -0.351120f, 4.13067f, 
		-0.880684f, -0.350786f, 4.07055f, -0.880683f, -0.350786f, 3.95966f, 
		-0.879716f, -0.447106f, 4.01510f, -0.878162f, -0.446874f, 4.12952f, 
		-0.878161f, -0.446874f, 3.90068f, -0.878141f, -0.263510f, 4.03359f, 
		-0.877436f, -0.446408f, 3.96030f, -0.877436f, -0.446408f, 4.06990f, 
		-0.876837f, -0.350451f, 4.12897f, -0.876836f, -0.350451f, 3.90123f, 
		-0.874327f, -0.447106f, 3.96092f, -0.874327f, -0.447106f, 4.06929f, 
		-0.872892f, -0.351588f, 4.01510f, -0.871802f, -0.274378f, 4.01128f, 
		-0.871792f, 0.0957810f, 4.13165f, -0.869858f, -0.446734f, 4.12608f, 
		-0.869857f, -0.446734f, 3.90412f, -0.868412f, -0.352318f, 4.01510f, 
		-0.868139f, -0.507296f, 4.01510f, -0.867636f, -0.351588f, 3.96225f, 
		-0.867636f, -0.351588f, 4.06796f, -0.866689f, -0.370015f, 4.01510f, 
		-0.865529f, -0.283484f, 3.98930f, -0.864560f, -0.350786f, 4.12389f, 
		-0.864559f, -0.350786f, 3.90632f, -0.863996f, 0.00290400f, 3.78109f, 
		-0.863242f, -0.352318f, 3.96312f, -0.863242f, -0.352318f, 4.06708f, 
		-0.862974f, -0.507296f, 3.96318f, -0.862974f, -0.507296f, 4.06703f, 
		-0.861864f, 0.0932810f, 4.33147f, -0.861655f, -0.421718f, 4.01510f, 
		-0.861553f, -0.370015f, 4.06675f, -0.861552f, -0.370015f, 3.96346f, 
		-0.861500f, -0.446408f, 3.90758f, -0.861500f, -0.446408f, 4.12262f, 
		-0.859388f, -0.289067f, 3.96797f, -0.858571f, -0.447106f, 4.12141f, 
		-0.858570f, -0.447106f, 3.90880f, -0.856616f, -0.421718f, 3.96444f, 
		-0.856616f, -0.421718f, 4.06576f, -0.854242f, 0.00635200f, 3.92522f, 
		-0.853312f, -0.292889f, 3.94696f, -0.852976f, -0.445619f, 4.18285f, 
		-0.852976f, -0.432733f, 4.18285f, -0.852976f, -0.397979f, 4.18285f, 
		-0.852976f, -0.363420f, 4.18285f, -0.852976f, -0.351120f, 4.18285f, 
		-0.852975f, -0.445619f, 3.84736f, -0.852975f, -0.432733f, 3.84736f, 
		-0.852975f, -0.397979f, 3.84736f, -0.852975f, -0.363420f, 3.84736f, 
		-0.852975f, -0.351120f, 3.84736f, -0.852267f, -0.351588f, 4.11880f, 
		-0.852266f, -0.351588f, 3.91141f, -0.850485f, -0.446874f, 4.18118f, 
		-0.850484f, -0.446874f, 3.84902f, -0.849293f, -0.350451f, 4.18039f, 
		-0.849292f, -0.350451f, 3.84982f, -0.848127f, -0.352318f, 3.91312f, 
		-0.848127f, -0.352318f, 4.11708f, -0.847875f, -0.507296f, 4.11698f, 
		-0.847874f, -0.507296f, 3.91323f, -0.846535f, -0.370015f, 3.91378f, 
		-0.846535f, -0.370015f, 4.11642f, -0.843013f, -0.446734f, 4.17619f, 
		-0.843012f, -0.446734f, 3.85402f, -0.842123f, 0.0910360f, 4.17920f, 
		-0.841884f, -0.421718f, 3.91571f, -0.841884f, -0.421718f, 4.11450f, 
		-0.838246f, -0.350786f, 4.17301f, -0.838245f, -0.350786f, 3.85720f, 
		-0.836688f, -0.260437f, 4.05458f, -0.836004f, 0.00146300f, 3.73627f, 
		-0.835493f, -0.446408f, 4.17117f, -0.835492f, -0.446408f, 3.85904f, 
		-0.833871f, -0.685447f, 4.01510f, -0.832857f, -0.447106f, 4.16940f, 
		-0.832856f, -0.447106f, 3.86080f, -0.831019f, -0.473248f, 3.96953f, 
		-0.831019f, -0.473248f, 3.96953f, -0.831019f, -0.473248f, 3.96953f, 
		-0.831019f, -0.473248f, 4.06067f, -0.831019f, -0.473248f, 4.06067f, 
		-0.831019f, -0.473248f, 4.06067f, -0.830177f, -0.473248f, 4.01510f, 
		-0.830177f, -0.473248f, 4.01510f, -0.830177f, -0.473248f, 4.01510f, 
		-0.829372f, -0.685447f, 3.96986f, -0.829372f, -0.685447f, 4.06035f, 
		-0.827184f, -0.351588f, 3.86459f, -0.827184f, -0.351588f, 4.16562f, 
		-0.826878f, -0.265693f, 4.02476f, -0.823460f, -0.352318f, 4.16313f, 
		-0.823459f, -0.352318f, 3.86708f, -0.823233f, -0.507296f, 4.16297f, 
		-0.823232f, -0.507296f, 3.86723f, -0.823099f, 0.0671100f, 4.03878f, 
		-0.822027f, -0.370015f, 3.86804f, -0.822027f, -0.370015f, 4.16217f, 
		-0.818272f, -0.269827f, 3.99548f, -0.817842f, -0.421718f, 3.87083f, 
		-0.817842f, -0.421718f, 4.15937f, -0.817743f, 0.00677100f, 3.87098f, 
		-0.816216f, -0.685447f, 4.10386f, -0.816215f, -0.685447f, 3.92634f, 
		-0.815467f, -0.445619f, 4.22864f, -0.815467f, -0.432733f, 4.22864f, 
		-0.815467f, -0.397979f, 4.22864f, -0.815467f, -0.363420f, 4.22864f, 
		-0.815467f, -0.351120f, 4.22864f, -0.815466f, -0.445619f, 3.80156f, 
		-0.815466f, -0.432733f, 3.80156f, -0.815466f, -0.397979f, 3.80156f, 
		-0.815466f, -0.363420f, 3.80156f, -0.815466f, -0.351120f, 3.80156f, 
		-0.813348f, -0.446874f, 4.22652f, -0.813347f, -0.446874f, 3.80368f, 
		-0.812803f, -0.473248f, 4.10245f, -0.812803f, -0.473248f, 4.10245f, 
		-0.812803f, -0.473248f, 4.10245f, -0.812802f, -0.473248f, 3.92776f, 
		-0.812802f, -0.473248f, 3.92776f, -0.812802f, -0.473248f, 3.92776f, 
		-0.812334f, -0.350451f, 4.22551f, -0.812333f, -0.350451f, 3.80470f, 
		-0.812071f, -0.271717f, 3.96727f, -0.808821f, 0.0937070f, 4.23060f, 
		-0.807835f, -0.395531f, 4.28986f, -0.807075f, -0.272483f, 3.93960f, 
		-0.806993f, -0.446734f, 4.22017f, -0.806992f, -0.446734f, 3.81004f, 
		-0.805405f, -0.364269f, 4.26667f, -0.803856f, -0.0230040f, 4.03137f, 
		-0.803014f, -0.395661f, 4.29378f, -0.802938f, -0.350786f, 4.21611f, 
		-0.802937f, -0.350786f, 3.81409f, -0.802910f, -0.424252f, 4.30068f, 
		-0.802418f, -0.417469f, 4.30119f, -0.800596f, -0.446408f, 4.21377f, 
		-0.800595f, -0.446408f, 3.81644f, -0.799488f, -0.864199f, 4.01510f, 
		-0.798354f, -0.447106f, 4.21153f, -0.798353f, -0.447106f, 3.81868f, 
		-0.798011f, -0.420634f, 4.30311f, -0.797907f, -0.335724f, 4.23596f, 
		-0.797270f, 0.0608970f, 4.08678f, -0.796827f, -0.395171f, 4.29777f, 
		-0.796691f, -0.364470f, 4.27567f, -0.796142f, -0.473248f, 4.14487f, 
		-0.796142f, -0.473248f, 4.14487f, -0.796142f, -0.473248f, 4.14487f, 
		-0.796141f, -0.473248f, 3.88533f, -0.796141f, -0.473248f, 3.88533f, 
		-0.796141f, -0.473248f, 3.88533f, -0.795656f, -0.864199f, 3.97657f, 
		-0.795656f, -0.864199f, 4.05364f, -0.795409f, -0.434973f, 4.30394f, 
		-0.794746f, -0.685447f, 4.14394f, -0.794745f, -0.685447f, 3.88627f, 
		-0.793886f, -0.409460f, 4.30369f, -0.793529f, -0.351588f, 4.20670f, 
		-0.793528f, -0.351588f, 3.82350f, -0.790361f, -0.352318f, 4.20354f, 
		-0.790360f, -0.352318f, 3.82667f, -0.790168f, -0.507296f, 4.20334f, 
		-0.790167f, -0.507296f, 3.82686f, -0.790111f, -0.412238f, 4.30443f, 
		-0.789936f, -0.384880f, 4.30086f, -0.789142f, -0.370015f, 3.82789f, 
		-0.789142f, -0.370015f, 4.20232f, -0.787788f, -0.361798f, 4.28474f, 
		-0.787628f, -0.315156f, 4.20256f, -0.787564f, -0.926190f, 4.01510f, 
		-0.787222f, -0.269136f, 4.07136f, -0.786000f, 0.0108320f, 3.81794f, 
		-0.785583f, -0.421718f, 4.19876f, -0.785582f, -0.421718f, 3.83145f, 
		-0.784856f, -0.332493f, 4.24915f, -0.784450f, -0.864199f, 3.93950f, 
		-0.784450f, -0.864199f, 4.09071f, -0.783963f, -0.926190f, 3.97889f, 
		-0.783963f, -0.926190f, 4.05131f, -0.782745f, -0.374971f, 4.30370f, 
		-0.778823f, -0.269882f, 4.03659f, -0.778507f, -0.353382f, 4.29398f, 
		-0.777773f, -0.0259600f, 3.97377f, -0.773703f, 0.100085f, 4.28394f, 
		-0.773434f, -0.926190f, 3.94406f, -0.773434f, -0.926190f, 4.08614f, 
		-0.771990f, -0.328045f, 4.26300f, -0.771481f, -0.269659f, 4.00258f, 
		-0.769980f, -0.303908f, 4.16860f, -0.769673f, -0.445619f, 4.26615f, 
		-0.769673f, -0.432733f, 4.26615f, -0.769673f, -0.397979f, 4.26615f, 
		-0.769673f, -0.363420f, 4.26615f, -0.769673f, -0.351120f, 4.26615f, 
		-0.769672f, -0.445619f, 3.76406f, -0.769672f, -0.432733f, 3.76406f, 
		-0.769672f, -0.397979f, 3.76406f, -0.769672f, -0.363420f, 3.76406f, 
		-0.769672f, -0.351120f, 3.76406f, -0.769036f, -0.342094f, 4.30330f, 
		-0.768918f, 0.0574940f, 4.13505f, -0.768913f, -0.308332f, 4.21651f, 
		-0.768339f, -0.928898f, 3.98200f, -0.768339f, -0.928898f, 4.04820f, 
		-0.768009f, -0.446874f, 4.26366f, -0.768007f, -0.446874f, 3.76655f, 
		-0.767704f, -0.928898f, 4.01510f, -0.767347f, -0.490425f, 3.98220f, 
		-0.767347f, -0.490425f, 3.98220f, -0.767347f, -0.490425f, 3.98220f, 
		-0.767347f, -0.490425f, 4.04801f, -0.767347f, -0.490425f, 4.04801f, 
		-0.767347f, -0.490425f, 4.04801f, -0.767212f, -0.350451f, 4.26247f, 
		-0.767211f, -0.350451f, 3.76774f, -0.766253f, -0.267498f, 3.97011f, 
		-0.766163f, -0.864199f, 4.12484f, -0.766162f, -0.864199f, 3.90537f, 
		-0.765937f, -0.685447f, 4.17911f, -0.765936f, -0.685447f, 3.85109f, 
		-0.763325f, -0.473248f, 4.17650f, -0.763325f, -0.473248f, 4.17650f, 
		-0.763325f, -0.473248f, 4.17650f, -0.763324f, -0.473248f, 3.85371f, 
		-0.763324f, -0.473248f, 3.85371f, -0.763324f, -0.473248f, 3.85371f, 
		-0.763016f, -0.446734f, 4.25619f, -0.763015f, -0.446734f, 3.77402f, 
		-0.762081f, -0.264368f, 3.93840f, -0.759831f, -0.350786f, 4.25142f, 
		-0.759829f, -0.350786f, 3.77879f, -0.759496f, -0.321159f, 4.27820f, 
		-0.757991f, -0.446408f, 4.24867f, -0.757990f, -0.446408f, 3.78154f, 
		-0.756634f, 0.0167130f, 3.76550f, -0.756250f, -0.926190f, 3.91199f, 
		-0.756250f, -0.926190f, 4.11822f, -0.756230f, -0.447106f, 4.24603f, 
		-0.756229f, -0.447106f, 3.78417f, -0.755085f, -0.928898f, 3.95166f, 
		-0.755085f, -0.928898f, 4.07854f, -0.753925f, -0.00444800f, 4.00802f, 
		-0.752440f, -0.351588f, 4.24036f, -0.752439f, -0.351588f, 3.78985f, 
		-0.750528f, -0.301437f, 4.23151f, -0.750196f, -0.0266640f, 3.91814f, 
		-0.749951f, -0.352318f, 4.23664f, -0.749950f, -0.352318f, 3.79357f, 
		-0.749799f, -0.507296f, 4.23641f, -0.749798f, -0.507296f, 3.79380f, 
		-0.749124f, -0.490425f, 4.01510f, -0.749124f, -0.490425f, 4.01510f, 
		-0.749124f, -0.490425f, 4.01510f, -0.748994f, -0.370015f, 4.23520f, 
		-0.748993f, -0.370015f, 3.79500f, -0.747552f, -0.295964f, 4.18242f, 
		-0.747189f, -0.313055f, 4.29408f, -0.746198f, -0.421718f, 4.23102f, 
		-0.746196f, -0.421718f, 3.79919f, -0.743005f, -0.928898f, 3.92084f, 
		-0.743005f, -0.928898f, 4.10937f, -0.742163f, -0.490425f, 3.92140f, 
		-0.742163f, -0.490425f, 3.92140f, -0.742163f, -0.490425f, 3.92140f, 
		-0.742163f, -0.490425f, 4.10881f, -0.742163f, -0.490425f, 4.10881f, 
		-0.742163f, -0.490425f, 4.10881f, -0.741624f, -0.864199f, 3.87541f, 
		-0.741624f, -0.864199f, 4.15480f, -0.740377f, -0.303319f, 4.13621f, 
		-0.737919f, -0.490425f, 3.95877f, -0.737919f, -0.490425f, 3.95877f, 
		-0.737919f, -0.490425f, 3.95877f, -0.737919f, -0.490425f, 4.07143f, 
		-0.737919f, -0.490425f, 4.07143f, -0.737919f, -0.490425f, 4.07143f, 
		-0.736177f, -0.0309700f, 4.29154f, -0.736127f, -0.0680940f, 4.21095f, 
		-0.735520f, 0.0597110f, 4.18383f, -0.735036f, -0.00738800f, 4.05597f, 
		-0.733193f, -0.926190f, 3.88384f, -0.733193f, -0.926190f, 4.14637f, 
		-0.732985f, -0.285504f, 4.07911f, -0.732801f, -0.294403f, 4.24866f, 
		-0.731698f, -0.473248f, 4.20932f, -0.731698f, -0.473248f, 4.20932f, 
		-0.731698f, -0.473248f, 4.20932f, -0.731697f, -0.473248f, 3.82089f, 
		-0.731697f, -0.473248f, 3.82089f, -0.731697f, -0.473248f, 3.82089f, 
		-0.731454f, -0.287077f, 4.04550f, -0.730765f, -0.685447f, 4.20792f, 
		-0.730764f, -0.685447f, 3.82229f, -0.729518f, -0.0371810f, 4.38859f, 
		-0.728794f, -0.287452f, 4.01257f, -0.725312f, -0.288398f, 4.19719f, 
		-0.723879f, -0.285429f, 3.98100f, -0.723568f, -0.139652f, 4.14475f, 
		-0.722949f, -0.0871820f, 4.06515f, -0.722831f, -0.929801f, 3.99106f, 
		-0.722831f, -0.929801f, 4.03915f, -0.719633f, -0.0228650f, 3.86647f, 
		-0.719460f, -0.299372f, 4.15156f, -0.719149f, -0.928898f, 3.89788f, 
		-0.719149f, -0.928898f, 4.13233f, -0.717836f, -0.282209f, 3.95012f, 
		-0.717495f, -0.445619f, 4.29411f, -0.717495f, -0.432733f, 4.29411f, 
		-0.717495f, -0.397979f, 4.29411f, -0.717495f, -0.363420f, 4.29411f, 
		-0.717495f, -0.351120f, 4.29411f, -0.717493f, -0.445619f, 3.73610f, 
		-0.717493f, -0.432733f, 3.73610f, -0.717493f, -0.397979f, 3.73610f, 
		-0.717493f, -0.363420f, 3.73610f, -0.717493f, -0.351120f, 3.73610f, 
		-0.716348f, -0.446874f, 4.29134f, -0.716347f, -0.446874f, 3.73887f, 
		-0.715799f, -0.350451f, 4.29001f, -0.715798f, -0.350451f, 3.74020f, 
		-0.715403f, -0.287299f, 4.26687f, -0.714906f, -0.0938750f, 4.00960f, 
		-0.713282f, -0.0764040f, 4.48480f, -0.712909f, -0.446734f, 4.28303f, 
		-0.712907f, -0.446734f, 3.74717f, -0.711918f, -0.00898000f, 4.10151f, 
		-0.711666f, -0.864199f, 3.85087f, -0.711666f, -0.864199f, 4.17934f, 
		-0.710714f, -0.350786f, 4.27774f, -0.710713f, -0.350786f, 3.75247f, 
		-0.709512f, -0.929801f, 4.01510f, -0.709447f, -0.446408f, 4.27468f, 
		-0.709445f, -0.446408f, 3.75553f, -0.709135f, 0.142957f, 3.76660f, 
		-0.708233f, -0.447106f, 4.27175f, -0.708232f, -0.447106f, 3.75846f, 
		-0.706897f, -0.317810f, 4.10826f, -0.706011f, -0.490425f, 3.91102f, 
		-0.706011f, -0.490425f, 3.91102f, -0.706011f, -0.490425f, 3.91102f, 
		-0.706011f, -0.490425f, 4.11919f, -0.706011f, -0.490425f, 4.11919f, 
		-0.706011f, -0.490425f, 4.11919f, -0.705622f, -0.351588f, 4.26544f, 
		-0.705621f, -0.351588f, 3.76477f, -0.705042f, -0.926190f, 3.86078f, 
		-0.705042f, -0.926190f, 4.16943f, -0.704493f, 0.112200f, 3.86945f, 
		-0.704424f, -0.929801f, 3.94662f, -0.704424f, -0.929801f, 4.08359f, 
		-0.703907f, -0.352318f, 4.26130f, -0.703906f, -0.352318f, 3.76890f, 
		-0.703803f, -0.507296f, 4.26105f, -0.703802f, -0.507296f, 3.76916f, 
		-0.703449f, -0.281587f, 4.21387f, -0.703248f, -0.370015f, 4.25971f, 
		-0.703247f, -0.370015f, 3.77050f, -0.701322f, -0.929801f, 3.97393f, 
		-0.701322f, -0.929801f, 4.05627f, -0.701321f, -0.421718f, 4.25506f, 
		-0.701320f, -0.421718f, 3.77515f, -0.699843f, -0.0980800f, 3.95837f, 
		-0.699599f, 0.0647390f, 4.23286f, -0.698254f, -0.295352f, 4.16694f, 
		-0.696430f, -0.101537f, 3.99651f, -0.696192f, -0.928898f, 3.87403f, 
		-0.696192f, -0.928898f, 4.15618f, -0.695980f, -0.325468f, 4.02245f, 
		-0.695630f, -0.490425f, 3.87487f, -0.695630f, -0.490425f, 3.87487f, 
		-0.695630f, -0.490425f, 3.87487f, -0.695630f, -0.490425f, 4.15534f, 
		-0.695630f, -0.490425f, 4.15534f, -0.695630f, -0.490425f, 4.15534f, 
		-0.695437f, 0.159304f, 3.66940f, -0.693313f, -0.323218f, 4.05082f, 
		-0.692702f, -0.236741f, 4.09083f, -0.691623f, -0.101775f, 4.04004f, 
		-0.691359f, -0.318996f, 4.12587f, -0.690689f, -0.685447f, 4.22939f, 
		-0.690688f, -0.685447f, 3.80082f, -0.690021f, -0.325094f, 3.99544f, 
		-0.689558f, 0.0480220f, 3.95585f, -0.689275f, -0.473248f, 4.22598f, 
		-0.689275f, -0.473248f, 4.22598f, -0.689275f, -0.473248f, 4.22598f, 
		-0.689274f, -0.473248f, 3.80423f, -0.689274f, -0.473248f, 3.80423f, 
		-0.689274f, -0.473248f, 3.80423f, -0.687577f, -0.0168130f, 3.81678f, 
		-0.686334f, -0.319656f, 4.07985f, -0.684605f, -0.138313f, 4.56285f, 
		-0.682078f, -0.0394230f, 4.29285f, -0.681773f, -0.275153f, 4.23151f, 
		-0.680341f, -0.00787500f, 4.14224f, -0.680082f, -0.208552f, 4.02560f, 
		-0.679749f, -0.323409f, 3.96910f, -0.679048f, -0.0743070f, 4.21065f, 
		-0.678910f, -0.0452380f, 4.39036f, -0.678851f, -0.102085f, 4.07841f, 
		-0.678810f, -0.385968f, 4.02923f, -0.678001f, -0.929801f, 3.93903f, 
		-0.678001f, -0.929801f, 4.09118f, -0.677620f, -0.351801f, 4.08767f, 
		-0.677531f, -0.864199f, 3.83258f, -0.677531f, -0.864199f, 4.19763f, 
		-0.676468f, -0.291184f, 4.18238f, -0.674706f, -0.211703f, 3.98256f, 
		-0.674470f, -0.319304f, 4.14180f, -0.673370f, -0.203788f, 4.07633f, 
		-0.672968f, -0.926190f, 3.84360f, -0.672968f, -0.926190f, 4.18661f, 
		-0.672947f, -0.384243f, 4.05185f, -0.672376f, -0.0685840f, 4.00368f, 
		-0.671671f, -0.445915f, 4.03318f, -0.671149f, 0.153529f, 3.58553f, 
		-0.670738f, -0.0973100f, 3.91577f, -0.670414f, -0.929801f, 3.91261f, 
		-0.670414f, -0.929801f, 4.11760f, -0.669971f, -0.355274f, 4.10728f, 
		-0.669749f, -0.386074f, 4.00889f, -0.669677f, -0.0833540f, 4.48632f, 
		-0.669532f, -0.142865f, 4.14809f, -0.668953f, -0.482270f, 4.03455f, 
		-0.668445f, -0.506829f, 4.03463f, -0.665367f, -0.928898f, 3.86195f, 
		-0.665367f, -0.928898f, 4.16826f, -0.664307f, -0.445484f, 4.05085f, 
		-0.662660f, -0.330708f, 4.05613f, -0.662595f, -0.221216f, 4.03152f, 
		-0.661343f, -0.482270f, 4.05005f, -0.661334f, -0.332841f, 4.02868f, 
		-0.661217f, -0.506829f, 4.04935f, -0.660832f, -0.445619f, 4.31123f, 
		-0.660832f, -0.432733f, 4.31123f, -0.660832f, -0.397979f, 4.31123f, 
		-0.660832f, -0.363420f, 4.31123f, -0.660832f, -0.351120f, 4.31123f, 
		-0.660831f, -0.445619f, 3.71897f, -0.660831f, -0.432733f, 3.71897f, 
		-0.660831f, -0.397979f, 3.71897f, -0.660831f, -0.363420f, 3.71897f, 
		-0.660831f, -0.351120f, 3.71897f, -0.660737f, -0.333516f, 3.99523f, 
		-0.660248f, -0.446874f, 4.30830f, -0.660247f, -0.446874f, 3.72191f, 
		-0.659968f, -0.350451f, 4.30689f, -0.659967f, -0.350451f, 3.72332f, 
		-0.659671f, -0.445942f, 4.01878f, -0.659624f, -0.381708f, 4.07560f, 
		-0.659172f, -0.191102f, 4.02588f, -0.659115f, -0.220718f, 4.06075f, 
		-0.659079f, -0.357261f, 4.12281f, -0.658495f, -0.446734f, 4.29948f, 
		-0.658493f, -0.446734f, 3.73073f, -0.658257f, -0.490425f, 3.87911f, 
		-0.658257f, -0.490425f, 3.87911f, -0.658257f, -0.490425f, 3.87911f, 
		-0.658257f, -0.490425f, 4.15109f, -0.658257f, -0.490425f, 4.15109f, 
		-0.658257f, -0.490425f, 4.15109f, -0.657376f, -0.350786f, 4.29386f, 
		-0.657375f, -0.350786f, 3.73635f, -0.656730f, -0.446408f, 4.29061f, 
		-0.656729f, -0.446408f, 3.73959f, -0.656517f, -0.506829f, 4.02322f, 
		-0.656394f, -0.482270f, 4.02253f, -0.656190f, -0.207394f, 4.61531f, 
		-0.656172f, -0.273010f, 4.03539f, -0.656112f, -0.447106f, 4.28750f, 
		-0.656111f, -0.447106f, 3.74270f, -0.655313f, -0.222871f, 3.99564f, 
		-0.655108f, -0.396413f, 4.07259f, -0.654879f, -0.317858f, 4.15441f, 
		-0.654781f, -0.351588f, 4.28081f, -0.654780f, -0.351588f, 3.74940f, 
		-0.654552f, -0.400138f, 4.09315f, -0.654508f, -0.145375f, 4.56390f, 
		-0.654392f, -0.286943f, 4.19785f, -0.653907f, -0.352318f, 4.27642f, 
		-0.653906f, -0.352318f, 3.75379f, -0.653854f, -0.507296f, 4.27615f, 
		-0.653852f, -0.507296f, 3.75406f, -0.653571f, -0.370015f, 4.27473f, 
		-0.653570f, -0.370015f, 3.75548f, -0.653242f, -0.238069f, 4.10950f, 
		-0.653228f, -0.385370f, 3.98968f, -0.653067f, -0.319144f, 4.03233f, 
		-0.652589f, -0.421718f, 4.26979f, -0.652588f, -0.421718f, 3.76042f, 
		-0.652573f, -0.392897f, 4.04756f, -0.650153f, -0.102538f, 4.10645f, 
		-0.649025f, -0.402006f, 4.10771f, -0.648421f, -0.500690f, 4.06203f, 
		-0.647694f, -0.331651f, 4.07255f, -0.647522f, -0.427623f, 4.05763f, 
		-0.647497f, -0.473248f, 4.24419f, -0.647497f, -0.473248f, 4.24419f, 
		-0.647497f, -0.473248f, 4.24419f, -0.647496f, -0.473248f, 3.78601f, 
		-0.647496f, -0.473248f, 3.78601f, -0.647496f, -0.473248f, 3.78601f, 
		-0.647260f, -0.444850f, 4.07015f, -0.647169f, -0.685447f, 3.78766f, 
		-0.647169f, -0.685447f, 4.24255f, -0.646696f, -0.429594f, 4.02579f, 
		-0.645157f, -0.211629f, 3.95492f, -0.644534f, -0.00542200f, 4.18055f, 
		-0.644362f, -0.445514f, 4.08082f, -0.644022f, 0.117921f, 3.52264f, 
		-0.643884f, -0.429504f, 4.00091f, -0.643647f, -0.482270f, 4.06729f, 
		-0.643518f, -0.364664f, 4.02294f, -0.643098f, -0.929801f, 3.91571f, 
		-0.643098f, -0.929801f, 4.11450f, -0.642102f, -0.363777f, 4.04356f, 
		-0.641993f, 0.141556f, 3.76996f, -0.641924f, -0.442764f, 4.06120f, 
		-0.641894f, 0.109244f, 3.86053f, -0.641705f, -0.356274f, 4.13017f, 
		-0.641252f, -0.446322f, 4.09428f, -0.640742f, -0.268130f, 4.63478f, 
		-0.640523f, -0.209634f, 4.61564f, -0.640463f, -0.864199f, 3.82138f, 
		-0.640463f, -0.864199f, 4.20883f, -0.640072f, -0.500690f, 4.01560f, 
		-0.639549f, -0.334337f, 4.01686f, -0.638950f, -0.379192f, 4.01136f, 
		-0.638495f, -0.429594f, 4.06144f, -0.638379f, 0.0413290f, 3.93281f, 
		-0.638362f, -0.331040f, 4.08904f, -0.638137f, -0.926190f, 3.83307f, 
		-0.638137f, -0.926190f, 4.19714f, -0.637989f, -0.445766f, 4.00601f, 
		-0.636948f, 0.149426f, 3.67710f, -0.636920f, -0.491833f, 4.04367f, 
		-0.636786f, -0.491147f, 4.06940f, -0.636379f, -0.478471f, 4.05848f, 
		-0.635271f, -0.367161f, 3.99683f, -0.635029f, -0.928898f, 3.84869f, 
		-0.635029f, -0.928898f, 4.18152f, -0.634841f, -0.282561f, 4.00364f, 
		-0.634840f, -0.254463f, 4.63410f, -0.634832f, -0.490425f, 4.18052f, 
		-0.634832f, -0.490425f, 4.18052f, -0.634832f, -0.490425f, 4.18052f, 
		-0.634831f, -0.490425f, 3.84968f, -0.634831f, -0.490425f, 3.84968f, 
		-0.634831f, -0.490425f, 3.84968f, -0.634620f, -0.490425f, 4.08168f, 
		-0.634612f, -0.0940520f, 3.87748f, -0.634114f, -0.512579f, 4.01991f, 
		-0.634109f, -0.222532f, 4.07670f, -0.633937f, -0.315536f, 4.16535f, 
		-0.633749f, -0.482270f, 4.01226f, -0.633556f, -0.400160f, 4.11029f, 
		-0.633337f, -0.272872f, 4.00422f, -0.633175f, -0.0733480f, 3.97078f, 
		-0.633077f, -0.392897f, 4.08088f, -0.632590f, -0.439198f, 4.07887f, 
		-0.631666f, -0.333157f, 3.98263f, -0.630191f, -0.192258f, 3.98954f, 
		-0.629659f, -0.300264f, 4.63650f, -0.629417f, -0.520115f, 4.01432f, 
		-0.629214f, -0.386096f, 4.00494f, -0.629055f, -0.483283f, 4.00231f, 
		-0.628799f, -0.409732f, 4.01963f, -0.627837f, -0.333920f, 3.99420f, 
		-0.627751f, 0.133107f, 3.58944f, -0.627046f, -0.443247f, 4.09543f, 
		-0.626723f, -0.267756f, 4.63483f, -0.626452f, -0.519050f, 4.02660f, 
		-0.626291f, -0.0498590f, 4.39219f, -0.625977f, -0.929801f, 3.89420f, 
		-0.625977f, -0.929801f, 4.13601f, -0.625774f, -0.302990f, 4.06879f, 
		-0.625589f, -0.512579f, 4.00309f, -0.625542f, -0.0441680f, 4.29415f, 
		-0.625191f, -0.0878290f, 4.48787f, -0.624266f, -0.218267f, 4.61631f, 
		-0.624135f, -0.151128f, 4.56496f, -0.622151f, -0.370159f, 4.06487f, 
		-0.621358f, -0.366111f, 4.05322f, -0.621356f, -0.485935f, 4.08306f, 
		-0.621088f, -0.353800f, 4.13344f, -0.621050f, -0.0777100f, 4.20997f, 
		-0.620914f, -0.380168f, 4.05055f, -0.620857f, 0.0636230f, 3.46969f, 
		-0.619045f, -0.429232f, 3.99758f, -0.618327f, -0.255083f, 3.96317f, 
		-0.618322f, -0.410124f, 4.03916f, -0.618299f, -0.157230f, 3.98952f, 
		-0.617993f, -0.415231f, 4.00421f, -0.617606f, -0.332014f, 4.00667f, 
		-0.617539f, -0.408577f, 4.02879f, -0.617237f, -0.264736f, 4.63472f, 
		-0.616200f, -0.144555f, 4.14790f, -0.615392f, 0.0928500f, 3.51447f, 
		-0.615159f, -0.211741f, 4.08067f, -0.615082f, -0.284500f, 4.06273f, 
		-0.614377f, -0.238773f, 4.11622f, -0.614337f, -0.283539f, 4.63570f, 
		-0.613983f, -0.331324f, 4.10257f, -0.613492f, -0.103063f, 4.12933f, 
		-0.613421f, -0.393169f, 4.09459f, -0.613116f, -0.396458f, 4.10689f, 
		-0.612775f, -0.439661f, 4.08945f, -0.611793f, -0.280564f, 4.01480f, 
		-0.610255f, -0.521484f, 4.03288f, -0.608732f, -0.271710f, 3.98890f, 
		-0.608456f, 0.00177800f, 3.41561f, -0.608433f, -0.470822f, 4.08287f, 
		-0.607596f, -0.222129f, 4.61679f, -0.607292f, -0.438230f, 4.09042f, 
		-0.606858f, -0.512579f, 4.00615f, -0.604935f, 0.0449750f, 3.45592f, 
		-0.604694f, -0.356237f, 4.00829f, -0.603520f, -0.209943f, 3.93499f, 
		-0.602383f, -0.293708f, 3.99831f, -0.602305f, -0.389780f, 4.05998f, 
		-0.601928f, -0.929801f, 3.90752f, -0.601928f, -0.929801f, 4.01510f, 
		-0.601928f, -0.929801f, 4.12269f, -0.601928f, -0.928898f, 3.84933f, 
		-0.601928f, -0.928898f, 4.18088f, -0.601928f, -0.926190f, 3.82947f, 
		-0.601928f, -0.926190f, 4.20074f, -0.601928f, -0.864199f, 3.81754f, 
		-0.601928f, -0.864199f, 4.21266f, -0.601928f, -0.685447f, 4.24705f, 
		-0.601928f, -0.507296f, 4.28131f, -0.601928f, -0.490425f, 4.01510f, 
		-0.601928f, -0.490425f, 4.16230f, -0.601928f, -0.490425f, 4.16230f, 
		-0.601928f, -0.490425f, 4.16230f, -0.601928f, -0.473248f, 4.24335f, 
		-0.601928f, -0.473248f, 4.24335f, -0.601928f, -0.473248f, 4.24335f, 
		-0.601928f, -0.447106f, 4.29289f, -0.601928f, -0.446874f, 4.31410f, 
		-0.601928f, -0.446734f, 4.30511f, -0.601928f, -0.446408f, 4.29606f, 
		-0.601928f, -0.445619f, 4.31709f, -0.601928f, -0.432733f, 4.31709f, 
		-0.601928f, -0.421718f, 4.27483f, -0.601928f, -0.397979f, 4.31709f, 
		-0.601928f, -0.370015f, 4.27986f, -0.601928f, -0.363420f, 4.31709f, 
		-0.601928f, -0.352318f, 4.28159f, -0.601928f, -0.351588f, 4.28607f, 
		-0.601928f, -0.351120f, 4.31709f, -0.601928f, -0.350786f, 4.29937f, 
		-0.601928f, -0.350451f, 4.31266f, -0.601927f, -0.685447f, 3.78316f, 
		-0.601927f, -0.507296f, 3.74889f, -0.601927f, -0.490425f, 3.86791f, 
		-0.601927f, -0.490425f, 3.86791f, -0.601927f, -0.490425f, 3.86791f, 
		-0.601927f, -0.473248f, 3.78685f, -0.601927f, -0.473248f, 3.78685f, 
		-0.601927f, -0.473248f, 3.78685f, -0.601927f, -0.447106f, 3.73731f, 
		-0.601927f, -0.446874f, 3.71611f, -0.601927f, -0.446734f, 3.72510f, 
		-0.601927f, -0.446408f, 3.73414f, -0.601927f, -0.445619f, 3.71311f, 
		-0.601927f, -0.432733f, 3.71311f, -0.601927f, -0.421718f, 3.75538f, 
		-0.601927f, -0.397979f, 3.71311f, -0.601927f, -0.370015f, 3.75034f, 
		-0.601927f, -0.363420f, 3.71311f, -0.601927f, -0.352318f, 3.74862f, 
		-0.601927f, -0.351588f, 3.74414f, -0.601927f, -0.351120f, 3.71311f, 
		-0.601927f, -0.350786f, 3.73083f, -0.601927f, -0.350451f, 3.71755f, 
		-0.601913f, -0.281534f, 4.04434f, -0.601447f, 0.00580200f, 3.41754f, 
		-0.600960f, -0.493286f, 4.07919f, -0.600479f, -0.192778f, 3.97081f, 
		-0.599992f, -0.465520f, 4.08487f, -0.599256f, -0.395606f, 3.98378f, 
		-0.598812f, -0.448490f, 3.99627f, -0.598341f, -0.225501f, 4.08601f, 
		-0.598098f, -0.0359450f, 3.38287f, -0.597799f, -0.279808f, 4.02776f, 
		-0.597634f, -0.483140f, 4.00227f, -0.596340f, -0.170824f, 3.92116f, 
		-0.595422f, -0.161377f, 4.07779f, -0.594942f, -0.508789f, 4.07929f, 
		-0.594253f, -0.423567f, 4.10116f, -0.594093f, -0.300913f, 4.06051f, 
		-0.594022f, -0.485718f, 4.04065f, -0.593478f, -0.339823f, 3.96390f, 
		-0.593464f, -0.0765890f, 3.95205f, -0.593443f, -0.393983f, 4.06909f, 
		-0.593374f, -0.0128720f, 3.40076f, -0.593213f, -0.154265f, 4.56603f, 
		-0.593073f, -0.441049f, 4.06760f, -0.593017f, -0.380499f, 4.02881f, 
		-0.592536f, -0.231101f, 4.61753f, -0.592294f, -0.119879f, 4.10942f, 
		-0.592075f, -0.156541f, 4.00660f, -0.591154f, -0.0405910f, 3.96202f, 
		-0.590951f, -0.370055f, 4.05740f, -0.590704f, -0.500764f, 4.00514f, 
		-0.590451f, -0.386096f, 3.99996f, -0.589710f, -0.424942f, 4.00081f, 
		-0.589444f, -0.331512f, 4.07732f, -0.589222f, -0.325598f, 4.01945f, 
		-0.589024f, -0.491651f, 4.01237f, -0.588900f, 0.0262930f, 3.44165f, 
		-0.588382f, -0.334199f, 3.99333f, -0.588360f, -0.332282f, 3.98045f, 
		-0.587042f, 0.0733950f, 3.50941f, -0.586941f, 0.0371010f, 3.91798f, 
		-0.586004f, -0.385384f, 4.04950f, -0.584975f, 0.115995f, 3.59334f, 
		-0.584829f, -0.209420f, 3.99240f, -0.584564f, -0.381092f, 4.12101f, 
		-0.583193f, -0.428869f, 4.00503f, -0.583185f, -0.00729900f, 3.40310f, 
		-0.582586f, -0.211500f, 4.06642f, -0.582398f, -0.296273f, 4.02527f, 
		-0.581063f, -0.0178750f, 3.39395f, -0.580690f, -0.256197f, 3.97511f, 
		-0.580637f, -0.158615f, 4.05074f, -0.580586f, -0.268502f, 4.00527f, 
		-0.580161f, 0.108540f, 3.85394f, -0.580092f, -0.493286f, 4.08080f, 
		-0.579873f, 0.141658f, 3.68275f, -0.579724f, -0.513957f, 4.08034f, 
		-0.578945f, -0.0873550f, 4.48949f, -0.578822f, -0.465668f, 4.08425f, 
		-0.578309f, -0.430480f, 4.03766f, -0.577879f, -0.929801f, 3.89420f, 
		-0.577879f, -0.929801f, 4.13601f, -0.576705f, -0.238229f, 4.09905f, 
		-0.576634f, 0.141975f, 3.77163f, -0.576104f, -0.156715f, 4.02618f, 
		-0.575989f, -0.296475f, 3.93575f, -0.575329f, -0.298715f, 4.04755f, 
		-0.575310f, -0.393631f, 3.99355f, -0.574889f, -0.444447f, 4.00132f, 
		-0.574275f, -0.424159f, 4.09370f, -0.574050f, -0.477859f, 4.00563f, 
		-0.573305f, -0.395069f, 4.02398f, -0.573000f, -0.502569f, 4.00822f, 
		-0.572639f, 0.0249530f, 3.44051f, -0.571740f, -0.351764f, 4.13735f, 
		-0.571328f, -0.493286f, 4.04262f, -0.570741f, -0.386096f, 4.07130f, 
		-0.569945f, -0.508789f, 4.04844f, -0.569647f, -0.0476080f, 4.39417f, 
		-0.569305f, -0.192025f, 3.98728f, -0.569230f, -0.340104f, 3.97937f, 
		-0.569024f, -0.490425f, 4.18052f, -0.569024f, -0.490425f, 4.18052f, 
		-0.569024f, -0.490425f, 4.18052f, -0.569023f, -0.490425f, 3.84968f, 
		-0.569023f, -0.490425f, 3.84968f, -0.569023f, -0.490425f, 3.84968f, 
		-0.568826f, -0.928898f, 3.84869f, -0.568826f, -0.928898f, 4.18152f, 
		-0.568666f, -0.415230f, 4.06019f, -0.567420f, -0.210340f, 4.02228f, 
		-0.566936f, -0.0428090f, 3.99169f, -0.566647f, -0.381388f, 4.10637f, 
		-0.566296f, -0.513957f, 4.07677f, -0.565954f, -0.493285f, 4.07703f, 
		-0.565719f, -0.926190f, 3.83307f, -0.565719f, -0.926190f, 4.19714f, 
		-0.565652f, -0.513957f, 4.06298f, -0.565570f, -0.465520f, 4.04239f, 
		-0.565276f, -0.493285f, 4.06251f, -0.564824f, -0.331652f, 4.03269f, 
		-0.564486f, -0.120993f, 4.08311f, -0.564360f, -0.424942f, 4.00632f, 
		-0.564278f, -0.143201f, 4.14064f, -0.564134f, -0.0414980f, 4.29540f, 
		-0.563935f, -0.465742f, 4.07863f, -0.563782f, -0.306035f, 4.03609f, 
		-0.563761f, -0.386096f, 4.00577f, -0.563393f, -0.864199f, 3.82138f, 
		-0.563393f, -0.864199f, 4.20883f, -0.562508f, -0.211090f, 4.04826f, 
		-0.562015f, -0.156093f, 4.56712f, -0.561942f, -0.424942f, 4.03232f, 
		-0.561611f, -0.465668f, 4.06301f, -0.561216f, -0.0754930f, 4.20849f, 
		-0.561215f, -0.386096f, 4.03314f, -0.560757f, -0.929801f, 3.91571f, 
		-0.560757f, -0.929801f, 4.11450f, -0.560305f, -0.174303f, 3.94723f, 
		-0.560164f, -0.334338f, 4.00253f, -0.559913f, -0.333920f, 4.07129f, 
		-0.559254f, 0.0651700f, 3.51058f, -0.558560f, -0.424456f, 4.08290f, 
		-0.558547f, 0.0252170f, 3.44068f, -0.557529f, -0.102380f, 3.87751f, 
		-0.556687f, -0.685447f, 4.24255f, -0.556686f, -0.685447f, 3.78766f, 
		-0.556599f, -0.442225f, 4.01007f, -0.556534f, -0.0490530f, 4.09780f, 
		-0.556435f, -0.474876f, 4.01238f, -0.556359f, -0.473248f, 4.24419f, 
		-0.556359f, -0.473248f, 4.24419f, -0.556359f, -0.473248f, 4.24419f, 
		-0.556358f, -0.473248f, 3.78601f, -0.556358f, -0.473248f, 3.78601f, 
		-0.556358f, -0.473248f, 3.78601f, -0.556132f, -0.349983f, 4.11949f, 
		-0.555505f, -0.498999f, 4.01397f, -0.555235f, -0.392516f, 4.00629f, 
		-0.554420f, -0.334199f, 4.03187f, -0.554374f, -0.334933f, 4.14929f, 
		-0.552735f, -0.0767880f, 3.96167f, -0.551623f, -0.0438060f, 4.14723f, 
		-0.551369f, -0.424159f, 4.06543f, -0.551268f, -0.421718f, 4.26979f, 
		-0.551266f, -0.421718f, 3.76042f, -0.551054f, -0.480400f, 4.04284f, 
		-0.550856f, -0.381582f, 4.08991f, -0.550773f, -0.119879f, 3.98407f, 
		-0.550759f, -0.476489f, 4.02592f, -0.550668f, -0.264271f, 4.03747f, 
		-0.550571f, -0.256754f, 3.99391f, -0.550565f, -0.298558f, 3.95581f, 
		-0.550500f, -0.0449760f, 4.02350f, -0.550286f, -0.370015f, 4.27473f, 
		-0.550284f, -0.370015f, 3.75548f, -0.550003f, -0.507296f, 4.27615f, 
		-0.550002f, -0.507296f, 3.75406f, -0.549950f, -0.352318f, 4.27642f, 
		-0.549948f, -0.352318f, 3.75379f, -0.549626f, -0.0470400f, 4.05958f, 
		-0.549574f, -0.443643f, 4.02623f, -0.549309f, -0.497488f, 4.03602f, 
		-0.549201f, -0.500497f, 4.02482f, -0.549076f, -0.351588f, 4.28081f, 
		-0.549074f, -0.351588f, 3.74940f, -0.548441f, -0.423567f, 4.04462f, 
		-0.548180f, -0.446882f, 4.04611f, -0.547745f, -0.447106f, 4.28750f, 
		-0.547743f, -0.447106f, 3.74270f, -0.547491f, -0.269942f, 3.90211f, 
		-0.547126f, -0.446408f, 4.29061f, -0.547125f, -0.446408f, 3.73959f, 
		-0.547021f, -0.340106f, 3.99653f, -0.546480f, -0.350786f, 4.29386f, 
		-0.546479f, -0.350786f, 3.73635f, -0.546240f, 0.0602180f, 3.92891f, 
		-0.545599f, -0.490425f, 4.15109f, -0.545599f, -0.490425f, 4.15109f, 
		-0.545599f, -0.490425f, 4.15109f, -0.545598f, -0.490425f, 3.87911f, 
		-0.545598f, -0.490425f, 3.87911f, -0.545598f, -0.490425f, 3.87911f, 
		-0.545392f, -0.121550f, 4.05429f, -0.545362f, -0.446734f, 4.29948f, 
		-0.545360f, -0.446734f, 3.73073f, -0.543888f, -0.350451f, 4.30689f, 
		-0.543887f, -0.350451f, 3.72332f, -0.543726f, -0.120993f, 4.02044f, 
		-0.543609f, -0.446874f, 4.30830f, -0.543607f, -0.446874f, 3.72191f, 
		-0.543444f, 0.105503f, 3.59722f, -0.543024f, -0.445619f, 4.31123f, 
		-0.543024f, -0.432733f, 4.31123f, -0.543024f, -0.397979f, 4.31123f, 
		-0.543024f, -0.363420f, 4.31123f, -0.543024f, -0.351120f, 4.31123f, 
		-0.543023f, -0.445619f, 3.71897f, -0.543023f, -0.432733f, 3.71897f, 
		-0.543023f, -0.397979f, 3.71897f, -0.543023f, -0.363420f, 3.71897f, 
		-0.543023f, -0.351120f, 3.71897f, -0.542900f, -0.393120f, 4.02498f, 
		-0.541853f, -0.349279f, 4.09972f, -0.540690f, -0.331041f, 4.13156f, 
		-0.539628f, -0.237062f, 4.06992f, -0.539320f, -0.381574f, 4.06980f, 
		-0.538488f, -0.928898f, 3.86195f, -0.538488f, -0.928898f, 4.16826f, 
		-0.537400f, -0.190637f, 4.02135f, -0.535489f, -0.256197f, 4.02641f, 
		-0.534984f, 0.0378030f, 3.91954f, -0.534435f, -0.394583f, 4.04665f, 
		-0.533442f, -0.929801f, 3.91261f, -0.533442f, -0.929801f, 4.11760f, 
		-0.531881f, -0.0431920f, 4.10808f, -0.531819f, -0.0844050f, 4.49113f, 
		-0.531747f, 0.0625600f, 3.51488f, -0.531052f, -0.329950f, 4.15596f, 
		-0.530888f, -0.926190f, 3.84360f, -0.530888f, -0.926190f, 4.18661f, 
		-0.530236f, -0.350727f, 4.07613f, -0.529910f, -0.381464f, 4.04786f, 
		-0.528893f, -0.339548f, 4.01709f, -0.528020f, -0.329023f, 4.11162f, 
		-0.527925f, -0.255083f, 4.06576f, -0.527465f, -0.177066f, 3.97617f, 
		-0.526639f, -0.299350f, 3.97625f, -0.526325f, -0.864199f, 4.19763f, 
		-0.526324f, -0.864199f, 3.83258f, -0.525855f, -0.929801f, 3.93903f, 
		-0.525855f, -0.929801f, 4.09118f, -0.525785f, 0.0544480f, 3.97296f, 
		-0.525624f, 0.138108f, 3.68435f, -0.521796f, -0.272989f, 3.92439f, 
		-0.520159f, 0.112339f, 3.85198f, -0.520156f, -0.110728f, 3.91162f, 
		-0.519949f, -0.353252f, 4.05064f, -0.518283f, -0.325658f, 4.14107f, 
		-0.517376f, -0.330753f, 4.08731f, -0.515987f, -0.0418620f, 4.06777f, 
		-0.514844f, 0.146036f, 3.76994f, -0.514581f, -0.473248f, 4.22598f, 
		-0.514581f, -0.473248f, 4.22598f, -0.514581f, -0.473248f, 4.22598f, 
		-0.514580f, -0.473248f, 3.80423f, -0.514580f, -0.473248f, 3.80423f, 
		-0.514580f, -0.473248f, 3.80423f, -0.513167f, -0.685447f, 4.22939f, 
		-0.513166f, -0.685447f, 3.80082f, -0.513060f, -0.140325f, 4.12984f, 
		-0.512805f, -0.338711f, 4.03933f, -0.511497f, -0.0754640f, 3.98547f, 
		-0.510992f, -0.0419230f, 4.39621f, -0.509543f, 0.0500260f, 4.01795f, 
		-0.508687f, -0.264603f, 3.86578f, -0.508226f, -0.490425f, 4.15534f, 
		-0.508226f, -0.490425f, 4.15534f, -0.508226f, -0.490425f, 4.15534f, 
		-0.508225f, -0.490425f, 3.87487f, -0.508225f, -0.490425f, 3.87487f, 
		-0.508225f, -0.490425f, 3.87487f, -0.507789f, -0.0390990f, 4.02513f, 
		-0.507744f, -0.334357f, 4.06080f, -0.507663f, -0.928898f, 3.87403f, 
		-0.507663f, -0.928898f, 4.15618f, -0.505828f, -0.322295f, 4.12490f, 
		-0.505710f, -0.297557f, 3.99745f, -0.504394f, -0.336950f, 4.16031f, 
		-0.503439f, -0.0356200f, 3.98133f, -0.502536f, 0.0983210f, 3.60110f, 
		-0.502535f, -0.421718f, 4.25506f, -0.502534f, -0.421718f, 3.77515f, 
		-0.502533f, -0.929801f, 3.97393f, -0.502533f, -0.929801f, 4.05627f, 
		-0.501727f, 0.0483030f, 4.06485f, -0.501015f, -0.178396f, 4.01086f, 
		-0.501010f, -0.0627740f, 3.82877f, -0.500609f, -0.370015f, 4.25971f, 
		-0.500607f, -0.370015f, 3.77050f, -0.500463f, -0.0704650f, 4.20662f, 
		-0.500289f, -0.0351190f, 4.29663f, -0.500054f, -0.507296f, 4.26105f, 
		-0.500053f, -0.507296f, 3.76916f, -0.499949f, -0.352318f, 4.26130f, 
		-0.499948f, -0.352318f, 3.76890f, -0.499431f, -0.929801f, 3.94662f, 
		-0.499431f, -0.929801f, 4.08359f, -0.498813f, -0.926190f, 3.86078f, 
		-0.498813f, -0.926190f, 4.16943f, -0.498235f, -0.351588f, 4.26544f, 
		-0.498233f, -0.351588f, 3.76477f, -0.498124f, 0.0479280f, 4.11268f, 
		-0.497845f, -0.490425f, 4.11919f, -0.497845f, -0.490425f, 4.11919f, 
		-0.497845f, -0.490425f, 4.11919f, -0.497844f, -0.490425f, 3.91102f, 
		-0.497844f, -0.490425f, 3.91102f, -0.497844f, -0.490425f, 3.91102f, 
		-0.497527f, -0.274144f, 3.94617f, -0.495623f, -0.447106f, 4.27175f, 
		-0.495622f, -0.447106f, 3.75846f, -0.494410f, -0.446408f, 4.27468f, 
		-0.494409f, -0.446408f, 3.75553f, -0.494344f, -0.929801f, 4.01510f, 
		-0.493999f, -0.320788f, 4.10619f, -0.493143f, -0.350786f, 4.27774f, 
		-0.493141f, -0.350786f, 3.75247f, -0.493112f, -0.333829f, 4.14937f, 
		-0.492190f, -0.864199f, 3.85087f, -0.492190f, -0.864199f, 4.17934f, 
		-0.490948f, -0.446734f, 4.28303f, -0.490947f, -0.446734f, 3.74717f, 
		-0.488057f, -0.350451f, 4.29001f, -0.488056f, -0.350451f, 3.74020f, 
		-0.487590f, 7.90000e-005f, 4.18629f, -0.487509f, -0.446874f, 4.29134f, 
		-0.487507f, -0.446874f, 3.73887f, -0.486362f, -0.445619f, 4.29411f, 
		-0.486362f, -0.432733f, 4.29411f, -0.486362f, -0.397979f, 4.29411f, 
		-0.486362f, -0.363420f, 4.29411f, -0.486362f, -0.351120f, 4.29411f, 
		-0.486361f, -0.445619f, 3.73610f, -0.486361f, -0.432733f, 3.73610f, 
		-0.486361f, -0.397979f, 3.73610f, -0.486361f, -0.363420f, 3.73610f, 
		-0.486361f, -0.351120f, 3.73610f, -0.486280f, -0.294471f, 4.01903f, 
		-0.485405f, -0.267392f, 3.88663f, -0.484706f, -0.928898f, 3.89788f, 
		-0.484706f, -0.928898f, 4.13233f, -0.483328f, -0.118995f, 3.94560f, 
		-0.482768f, 0.0409700f, 3.92931f, -0.482484f, -0.320209f, 4.08621f, 
		-0.481664f, -0.329883f, 4.13820f, -0.481025f, -0.929801f, 3.99106f, 
		-0.481025f, -0.929801f, 4.03915f, -0.477759f, -0.179010f, 4.04842f, 
		-0.477015f, -0.356064f, 4.16530f, -0.476858f, 0.00810400f, 4.13883f, 
		-0.476392f, 0.138058f, 3.89794f, -0.476109f, -0.271513f, 3.96691f, 
		-0.473091f, -0.685447f, 3.82229f, -0.473091f, -0.685447f, 4.20792f, 
		-0.472788f, 0.136667f, 3.68390f, -0.472159f, -0.473248f, 4.20932f, 
		-0.472159f, -0.473248f, 4.20932f, -0.472159f, -0.473248f, 4.20932f, 
		-0.472158f, -0.473248f, 3.82089f, -0.472158f, -0.473248f, 3.82089f, 
		-0.472158f, -0.473248f, 3.82089f, -0.470663f, -0.926190f, 3.88384f, 
		-0.470663f, -0.926190f, 4.14637f, -0.469882f, -0.324288f, 4.12654f, 
		-0.469380f, -0.355545f, 4.15781f, -0.466299f, 0.0162090f, 4.09083f, 
		-0.465937f, -0.490425f, 3.95878f, -0.465937f, -0.490425f, 3.95878f, 
		-0.465937f, -0.490425f, 3.95878f, -0.465937f, -0.490425f, 4.07143f, 
		-0.465937f, -0.490425f, 4.07143f, -0.465937f, -0.490425f, 4.07143f, 
		-0.465163f, -0.278332f, 3.83258f, -0.464376f, -0.0759960f, 3.86601f, 
		-0.463124f, -0.268386f, 3.90697f, -0.462231f, -0.864199f, 3.87541f, 
		-0.462231f, -0.864199f, 4.15480f, -0.461918f, -0.352577f, 4.15014f, 
		-0.461693f, -0.490425f, 4.10881f, -0.461693f, -0.490425f, 4.10881f, 
		-0.461693f, -0.490425f, 4.10881f, -0.461692f, -0.490425f, 3.92140f, 
		-0.461692f, -0.490425f, 3.92140f, -0.461692f, -0.490425f, 3.92140f, 
		-0.461024f, 0.118391f, 3.85235f, -0.460851f, -0.928898f, 3.92084f, 
		-0.460851f, -0.928898f, 4.10937f, -0.458029f, 0.129046f, 3.95331f, 
		-0.457933f, -0.317868f, 4.11464f, -0.457659f, -0.421718f, 4.23102f, 
		-0.457658f, -0.421718f, 3.79919f, -0.456118f, -0.266990f, 3.98714f, 
		-0.456087f, 0.0244760f, 4.04177f, -0.454863f, -0.370015f, 4.23520f, 
		-0.454862f, -0.370015f, 3.79501f, -0.454839f, 0.151917f, 3.76656f, 
		-0.454802f, -0.344710f, 4.14210f, -0.454732f, -0.490425f, 4.01510f, 
		-0.454732f, -0.490425f, 4.01510f, -0.454732f, -0.490425f, 4.01510f, 
		-0.454656f, -0.380700f, 4.16748f, -0.454057f, -0.507296f, 4.23641f, 
		-0.454056f, -0.507296f, 3.79380f, -0.453905f, -0.352318f, 3.79357f, 
		-0.453905f, -0.352318f, 4.23664f, -0.451417f, -0.351588f, 4.24036f, 
		-0.451416f, -0.351588f, 3.78985f, -0.451327f, -0.380517f, 4.16334f, 
		-0.448771f, -0.928898f, 3.95166f, -0.448771f, -0.928898f, 4.07854f, 
		-0.447999f, -0.379811f, 4.15803f, -0.447858f, -0.334395f, 4.13388f, 
		-0.447626f, -0.447106f, 4.24603f, -0.447625f, -0.447106f, 3.78417f, 
		-0.447606f, -0.926190f, 3.91199f, -0.447606f, -0.926190f, 4.11822f, 
		-0.447590f, -0.127100f, 3.97933f, -0.446559f, -0.279724f, 3.84943f, 
		-0.446264f, -0.370848f, 4.15205f, -0.446049f, 0.0328240f, 3.99218f, 
		-0.445865f, -0.446408f, 4.24867f, -0.445864f, -0.446408f, 3.78154f, 
		-0.444707f, -0.362232f, 4.14582f, -0.444026f, -0.350786f, 4.25142f, 
		-0.444025f, -0.350786f, 3.77879f, -0.443191f, -0.398454f, 4.16295f, 
		-0.443057f, -0.404265f, 4.16342f, -0.442848f, -0.265787f, 3.92629f, 
		-0.441786f, 0.122910f, 4.00779f, -0.441773f, -0.391494f, 4.15559f, 
		-0.441315f, -0.401019f, 4.15920f, -0.440930f, -0.393824f, 4.15238f, 
		-0.440840f, -0.446734f, 4.25619f, -0.440839f, -0.446734f, 3.77402f, 
		-0.440532f, -0.473248f, 4.17650f, -0.440532f, -0.473248f, 4.17650f, 
		-0.440532f, -0.473248f, 4.17650f, -0.440531f, -0.473248f, 3.85371f, 
		-0.440531f, -0.473248f, 3.85371f, -0.440531f, -0.473248f, 3.85371f, 
		-0.439416f, -0.413169f, 4.15707f, -0.437919f, -0.685447f, 3.85109f, 
		-0.437919f, -0.685447f, 4.17911f, -0.437693f, -0.864199f, 3.90537f, 
		-0.437693f, -0.864199f, 4.12484f, -0.436644f, -0.350451f, 4.26247f, 
		-0.436643f, -0.350451f, 3.76774f, -0.436509f, -0.490425f, 3.98220f, 
		-0.436509f, -0.490425f, 3.98220f, -0.436509f, -0.490425f, 3.98220f, 
		-0.436509f, -0.490425f, 4.04801f, -0.436509f, -0.490425f, 4.04801f, 
		-0.436509f, -0.490425f, 4.04801f, -0.436152f, -0.928898f, 4.01510f, 
		-0.435848f, -0.446874f, 4.26366f, -0.435847f, -0.446874f, 3.76655f, 
		-0.435516f, -0.928898f, 3.98200f, -0.435516f, -0.928898f, 4.04820f, 
		-0.434184f, -0.445619f, 4.26615f, -0.434184f, -0.432733f, 4.26615f, 
		-0.434184f, -0.397979f, 4.26615f, -0.434184f, -0.363420f, 4.26615f, 
		-0.434184f, -0.351120f, 4.26615f, -0.434183f, -0.445619f, 3.76406f, 
		-0.434183f, -0.432733f, 3.76406f, -0.434183f, -0.397979f, 3.76406f, 
		-0.434183f, -0.363420f, 3.76406f, -0.434183f, -0.351120f, 3.76406f, 
		-0.430422f, -0.926190f, 3.94406f, -0.430422f, -0.926190f, 4.08614f, 
		-0.429782f, 0.122527f, 4.06054f, -0.428454f, -0.279821f, 3.86612f, 
		-0.427023f, -0.0895280f, 3.90146f, -0.425897f, -0.0650290f, 3.77148f, 
		-0.423574f, -0.261393f, 3.94510f, -0.422505f, -0.309005f, 3.80832f, 
		-0.419898f, 0.125021f, 4.11241f, -0.419893f, -0.926190f, 3.97889f, 
		-0.419893f, -0.926190f, 4.05131f, -0.419406f, -0.864199f, 3.93950f, 
		-0.419406f, -0.864199f, 4.09071f, -0.418274f, -0.421718f, 4.19876f, 
		-0.418273f, -0.421718f, 3.83145f, -0.416291f, -0.926190f, 4.01510f, 
		-0.414714f, -0.370015f, 4.20232f, -0.414713f, -0.370015f, 3.82789f, 
		-0.413689f, -0.507296f, 4.20334f, -0.413688f, -0.507296f, 3.82686f, 
		-0.413495f, -0.352318f, 3.82667f, -0.413495f, -0.352318f, 4.20354f, 
		-0.412396f, -0.135125f, 4.01294f, -0.412318f, 0.0170540f, 4.21541f, 
		-0.411348f, -0.277326f, 3.88249f, -0.410426f, -0.307943f, 3.81972f, 
		-0.410327f, -0.351588f, 3.82350f, -0.410327f, -0.351588f, 4.20671f, 
		-0.409110f, -0.685447f, 3.88627f, -0.409110f, -0.685447f, 4.14394f, 
		-0.408200f, -0.864199f, 3.97657f, -0.408200f, -0.864199f, 4.05364f, 
		-0.407714f, -0.473248f, 3.88533f, -0.407714f, -0.473248f, 3.88533f, 
		-0.407714f, -0.473248f, 3.88533f, -0.407714f, -0.473248f, 4.14487f, 
		-0.407714f, -0.473248f, 4.14487f, -0.407714f, -0.473248f, 4.14487f, 
		-0.406860f, 0.0315210f, 4.16608f, -0.405502f, -0.447106f, 3.81868f, 
		-0.405502f, -0.447106f, 4.21153f, -0.404367f, -0.864199f, 4.01510f, 
		-0.403260f, -0.446408f, 3.81644f, -0.403260f, -0.446408f, 4.21377f, 
		-0.400919f, -0.350786f, 4.21611f, -0.400918f, -0.350786f, 3.81410f, 
		-0.399490f, 0.0456760f, 4.11643f, -0.398537f, -0.306195f, 3.83109f, 
		-0.397098f, -0.0806310f, 3.80816f, -0.396864f, -0.446734f, 4.22017f, 
		-0.396863f, -0.446734f, 3.81004f, -0.394741f, -0.273535f, 3.89870f, 
		-0.391522f, -0.350451f, 4.22551f, -0.391521f, -0.350451f, 3.80470f, 
		-0.391053f, -0.473248f, 3.92776f, -0.391053f, -0.473248f, 3.92776f, 
		-0.391053f, -0.473248f, 3.92776f, -0.391053f, -0.473248f, 4.10245f, 
		-0.391053f, -0.473248f, 4.10245f, -0.391053f, -0.473248f, 4.10245f, 
		-0.390508f, -0.446874f, 3.80368f, -0.390508f, -0.446874f, 4.22652f, 
		-0.388677f, 0.180486f, 3.87139f, -0.388390f, -0.445619f, 4.22864f, 
		-0.388390f, -0.432733f, 4.22864f, -0.388390f, -0.397979f, 4.22864f, 
		-0.388390f, -0.363420f, 4.22864f, -0.388390f, -0.351120f, 4.22864f, 
		-0.388389f, -0.445619f, 3.80157f, -0.388389f, -0.432733f, 3.80157f, 
		-0.388389f, -0.397979f, 3.80157f, -0.388389f, -0.363420f, 3.80157f, 
		-0.388389f, -0.351120f, 3.80157f, -0.388294f, 0.0592080f, 4.06617f, 
		-0.388258f, -0.343688f, 3.79415f, -0.388231f, -0.103684f, 3.93330f, 
		-0.387640f, -0.685447f, 3.92634f, -0.387640f, -0.685447f, 4.10387f, 
		-0.387027f, -0.303076f, 3.84244f, -0.386014f, -0.421718f, 4.15937f, 
		-0.386013f, -0.421718f, 3.87084f, -0.383011f, -0.338951f, 3.80025f, 
		-0.381829f, -0.370015f, 4.16217f, -0.381828f, -0.370015f, 3.86804f, 
		-0.380624f, -0.507296f, 4.16297f, -0.380623f, -0.507296f, 3.86723f, 
		-0.380396f, -0.352318f, 3.86708f, -0.380396f, -0.352318f, 4.16313f, 
		-0.376700f, -0.336860f, 3.80647f, -0.376672f, -0.351588f, 4.16562f, 
		-0.376671f, -0.351588f, 3.86459f, -0.375707f, -0.299272f, 3.85377f, 
		-0.375185f, 0.0724300f, 4.01560f, -0.374484f, -0.685447f, 3.96986f, 
		-0.374484f, -0.685447f, 4.06035f, -0.373679f, -0.473248f, 4.01510f, 
		-0.373679f, -0.473248f, 4.01510f, -0.373679f, -0.473248f, 4.01510f, 
		-0.372837f, -0.473248f, 3.96954f, -0.372837f, -0.473248f, 3.96954f, 
		-0.372837f, -0.473248f, 3.96954f, -0.372837f, -0.473248f, 4.06067f, 
		-0.372837f, -0.473248f, 4.06067f, -0.372837f, -0.473248f, 4.06067f, 
		-0.371841f, 0.169748f, 3.93187f, -0.370999f, -0.447106f, 3.86080f, 
		-0.370999f, -0.447106f, 4.16941f, -0.370926f, -0.332286f, 3.81294f, 
		-0.370320f, -0.359652f, 3.79378f, -0.369984f, -0.685447f, 4.01510f, 
		-0.369970f, -0.369448f, 3.79119f, -0.368363f, -0.446408f, 3.85904f, 
		-0.368363f, -0.446408f, 4.17117f, -0.367413f, -0.0956540f, 3.84302f, 
		-0.366271f, -0.361165f, 3.79684f, -0.365610f, -0.350786f, 3.85720f, 
		-0.365610f, -0.350786f, 4.17301f, -0.364868f, -0.331040f, 3.81866f, 
		-0.364085f, -0.354204f, 3.80079f, -0.362477f, -0.378054f, 3.79515f, 
		-0.361972f, -0.421718f, 4.11450f, -0.361971f, -0.421718f, 3.91571f, 
		-0.360843f, -0.446734f, 3.85402f, -0.360843f, -0.446734f, 4.17619f, 
		-0.360617f, -0.361276f, 3.80171f, -0.357321f, -0.370015f, 3.91378f, 
		-0.357321f, -0.370015f, 4.11642f, -0.356023f, 0.162880f, 3.99019f, 
		-0.355981f, -0.507296f, 3.91323f, -0.355981f, -0.507296f, 4.11698f, 
		-0.355729f, -0.352318f, 3.91313f, -0.355729f, -0.352318f, 4.11708f, 
		-0.354563f, -0.350451f, 3.84982f, -0.354563f, -0.350451f, 4.18039f, 
		-0.353371f, -0.446874f, 3.84902f, -0.353371f, -0.446874f, 4.18118f, 
		-0.352696f, -0.0955130f, 3.72419f, -0.351590f, -0.351588f, 4.11880f, 
		-0.351589f, -0.351588f, 3.91141f, -0.350881f, -0.445619f, 4.18285f, 
		-0.350881f, -0.432733f, 4.18285f, -0.350881f, -0.397979f, 4.18285f, 
		-0.350881f, -0.363420f, 4.18285f, -0.350881f, -0.351120f, 4.18285f, 
		-0.350880f, -0.445619f, 3.84736f, -0.350880f, -0.432733f, 3.84736f, 
		-0.350880f, -0.397979f, 3.84736f, -0.350880f, -0.363420f, 3.84736f, 
		-0.350880f, -0.351120f, 3.84736f, -0.348719f, -0.118150f, 3.96335f, 
		-0.347240f, -0.421718f, 3.96444f, -0.347240f, -0.421718f, 4.06576f, 
		-0.345285f, -0.447106f, 3.90880f, -0.345285f, -0.447106f, 4.12141f, 
		-0.342356f, -0.446408f, 3.90759f, -0.342356f, -0.446408f, 4.12262f, 
		-0.342303f, -0.370015f, 3.96346f, -0.342303f, -0.370015f, 4.06675f, 
		-0.342243f, 0.163753f, 4.04421f, -0.342201f, -0.421718f, 4.01510f, 
		-0.340881f, -0.507296f, 3.96318f, -0.340881f, -0.507296f, 4.06703f, 
		-0.340613f, -0.352318f, 3.96312f, -0.340613f, -0.352318f, 4.06708f, 
		-0.339296f, -0.350786f, 3.90632f, -0.339296f, -0.350786f, 4.12389f, 
		-0.337929f, 0.0123990f, 4.22342f, -0.337167f, -0.370015f, 4.01510f, 
		-0.336220f, -0.351588f, 3.96225f, -0.336220f, -0.351588f, 4.06796f, 
		-0.335955f, -0.109519f, 3.87425f, -0.335717f, -0.507296f, 4.01510f, 
		-0.335444f, -0.352318f, 4.01510f, -0.333998f, -0.446734f, 3.90412f, 
		-0.333998f, -0.446734f, 4.12608f, -0.332312f, -0.113251f, 3.75463f, 
		-0.330963f, -0.351588f, 4.01510f, -0.329528f, -0.447106f, 3.96092f, 
		-0.329528f, -0.447106f, 4.06929f, -0.329482f, 0.168497f, 4.09607f, 
		-0.329333f, 0.0256850f, 4.18055f, -0.327019f, -0.350451f, 3.90123f, 
		-0.327019f, -0.350451f, 4.12897f, -0.326419f, -0.446408f, 3.96030f, 
		-0.326419f, -0.446408f, 4.06991f, -0.325695f, -0.446874f, 4.12952f, 
		-0.325694f, -0.446874f, 3.90068f, -0.324139f, -0.447106f, 4.01510f, 
		-0.323172f, -0.350786f, 3.95966f, -0.323172f, -0.350786f, 4.07055f, 
		-0.322927f, -0.445619f, 4.13067f, -0.322927f, -0.432733f, 4.13067f, 
		-0.322927f, -0.397979f, 4.13067f, -0.322927f, -0.363420f, 4.13067f, 
		-0.322927f, -0.351120f, 4.13067f, -0.322926f, -0.445619f, 3.89954f, 
		-0.322926f, -0.432733f, 3.89954f, -0.322926f, -0.397979f, 3.89954f, 
		-0.322926f, -0.363420f, 3.89954f, -0.322926f, -0.351120f, 3.89954f, 
		-0.320969f, -0.446408f, 4.01510f, -0.318722f, 0.0395500f, 4.13752f, 
		-0.317658f, -0.350786f, 4.01510f, -0.317549f, -0.446734f, 3.95854f, 
		-0.317549f, -0.446734f, 4.07167f, -0.311923f, -0.446734f, 4.01510f, 
		-0.311735f, -0.128477f, 3.78428f, -0.310141f, -0.350451f, 3.95706f, 
		-0.310141f, -0.350451f, 4.07314f, -0.308735f, -0.446874f, 3.95678f, 
		-0.308735f, -0.446874f, 4.07342f, -0.305797f, -0.445619f, 3.95620f, 
		-0.305797f, -0.445619f, 4.07401f, -0.305797f, -0.432733f, 3.95620f, 
		-0.305797f, -0.432733f, 4.07401f, -0.305797f, -0.397979f, 3.95620f, 
		-0.305797f, -0.397979f, 4.07401f, -0.305797f, -0.363420f, 3.95620f, 
		-0.305797f, -0.363420f, 4.07401f, -0.305797f, -0.351120f, 3.95620f, 
		-0.305797f, -0.351120f, 4.07401f, -0.304369f, -0.350451f, 4.01510f, 
		-0.304084f, 0.0545730f, 4.09413f, -0.303610f, -0.122805f, 3.90366f, 
		-0.302935f, -0.446874f, 4.01510f, -0.301910f, -0.140594f, 3.70542f, 
		-0.299939f, -0.445619f, 4.01510f, -0.299939f, -0.432733f, 4.01510f, 
		-0.299939f, -0.397979f, 4.01510f, -0.299939f, -0.363420f, 4.01510f, 
		-0.299939f, -0.351120f, 4.01510f, -0.290771f, -0.138679f, 3.81237f, 
		-0.290165f, 0.175057f, 3.85153f, -0.287431f, 0.0701750f, 4.05057f, 
		-0.284008f, -0.162473f, 3.72198f, -0.275390f, 0.165316f, 3.90777f, 
		-0.269613f, -0.146369f, 3.83967f, -0.267501f, -0.197641f, 3.69889f, 
		-0.267228f, -0.179104f, 3.73923f, -0.265565f, -0.0111650f, 4.21968f, 
		-0.261045f, 0.159141f, 3.96231f, -0.253027f, -0.00347500f, 4.18749f, 
		-0.252689f, -0.185236f, 3.75786f, -0.252343f, -0.215205f, 3.70332f, 
		-0.247558f, 0.160101f, 4.01340f, -0.243430f, -0.264022f, 3.68830f, 
		-0.239716f, 0.00672700f, 4.15507f, -0.239272f, -0.186119f, 3.77718f, 
		-0.237700f, -0.233621f, 3.70798f, -0.237473f, -0.258355f, 3.69175f, 
		-0.234501f, 0.164627f, 4.06278f, -0.227532f, -0.304670f, 3.68381f, 
		-0.227322f, -0.234559f, 3.71713f, -0.226961f, -0.278116f, 3.69066f, 
		-0.224859f, 0.0219530f, 4.12218f, -0.220282f, -0.272018f, 3.69659f, 
		-0.219608f, -0.234640f, 3.72590f, -0.217643f, -0.284517f, 3.69556f, 
		-0.209229f, 0.0396910f, 4.08904f, -0.196365f, -0.0509150f, 4.21357f, 
		-0.194314f, 0.130845f, 3.84125f, -0.186695f, -0.0500320f, 4.19213f, 
		-0.182405f, 0.124633f, 3.88655f, -0.178191f, -0.0439000f, 4.17008f, 
		-0.172023f, -0.0272690f, 4.14682f, -0.170660f, 0.120951f, 3.93116f, 
		-0.167023f, -0.00539000f, 4.12295f, -0.159245f, 0.122329f, 3.97440f, 
		-0.147994f, 0.126238f, 4.01694f, -0.143731f, -0.0624370f, 4.14911f, 
		-0.142275f, -0.0994360f, 4.20408f, -0.139738f, -0.0800010f, 4.16439f, 
		-0.138717f, -0.0993550f, 4.19295f, -0.136209f, -0.0984170f, 4.17935f, 
		-0.122274f, -0.128818f, 4.16431f, -0.122166f, -0.123151f, 4.17119f, 
		-0.117477f, -0.136814f, 4.18843f, -0.115832f, -0.142912f, 4.17965f, 
		-0.115241f, -0.149313f, 4.19016f, -0.114587f, 0.0569230f, 3.84348f, 
		-0.110246f, -0.169466f, 4.17564f, -0.106612f, 0.0565820f, 3.87372f, 
		-0.0986760f, 0.0575660f, 3.90377f, -0.0908180f, 0.0611980f, 3.93347f, 
		-0.0829980f, 0.0661550f, 3.96298f, -0.0593390f, -0.0202290f, 3.85318f, 
		-0.0555080f, -0.0123230f, 3.86837f, -0.0510720f, -0.00962700f, 3.88469f, 
		-0.0468390f, -0.00214200f, 3.90097f, -0.0425910f, -0.000996000f, 3.91618f, 
		-0.0369290f, -0.0741300f, 3.86534f, -0.0365890f, -0.0555700f, 3.86962f, 
		-0.0338260f, -0.0592420f, 3.87845f, -0.0321620f, -0.0918820f, 3.87865f, 
		-0.0318690f, -0.0465430f, 3.88729f, -0.0298480f, -0.0605900f, 3.89183f
	};
	static const GLfloat NormalData[] = {
		-0.729457f, 0.625234f, -0.277443f, -0.644768f, 0.681602f, -0.345966f, -0.638431f, 0.682496f, -0.355815f, 
		-0.660852f, 0.655606f, -0.365316f, -0.628316f, 0.704072f, -0.330910f, 
		-0.851810f, 0.181953f, 0.491236f, -0.605000f, 0.674144f, -0.423681f, 
		-0.775306f, 0.245960f, 0.581726f, -0.825960f, 0.0516807f, 0.561354f, 
		-0.734040f, 0.482735f, 0.477654f, -0.766032f, 0.212015f, 0.606831f, 
		-0.742063f, 0.227922f, 0.630392f, -0.720443f, 0.153904f, 0.676221f, 
		-0.655602f, 0.492797f, 0.572134f, -0.638264f, 0.493694f, 0.590665f, 
		-0.427466f, 0.766368f, -0.479534f, -0.382204f, 0.723081f, -0.575391f, 
		-0.663787f, 0.450948f, 0.596685f, -0.498239f, 0.718079f, 0.485922f, 
		-0.405638f, 0.718598f, -0.564866f, -0.469202f, 0.796776f, -0.380785f, 
		-0.619335f, 0.410108f, 0.669504f, -0.449486f, 0.800627f, -0.396180f, 
		-0.502285f, 0.710927f, 0.492231f, -0.544484f, 0.695125f, 0.469403f, 
		-0.0979478f, 0.773290f, -0.626441f, -0.135515f, 0.778652f, -0.612647f, 
		-0.319414f, 0.867412f, 0.381537f, -0.592622f, 0.670667f, 0.446100f, 
		-0.236135f, 0.816014f, -0.527600f, -0.603655f, 0.656143f, 0.452854f, 
		-0.307239f, 0.858688f, -0.410194f, -0.342700f, 0.865556f, 0.365198f, 
		0.0552682f, 0.897170f, -0.438214f, -0.316646f, 0.874444f, -0.367536f, 
		-0.401578f, 0.860278f, 0.314096f, -0.0132660f, 0.894779f, -0.446312f, 
		-0.0874132f, 0.967434f, 0.237551f, -0.455487f, 0.849998f, 0.264640f, 
		-0.780349f, 0.625216f, 0.0126873f, -0.148211f, 0.895384f, -0.419906f, 
		-0.724838f, 0.682494f, -0.0938706f, 0.128715f, 0.981923f, -0.138776f, 
		-0.727074f, 0.681601f, -0.0823633f, -0.706212f, 0.704072f, -0.0744746f, 
		-0.749200f, 0.655591f, -0.0943384f, -0.718942f, 0.674141f, -0.169283f, 
		-0.109202f, 0.971092f, 0.212264f, -0.474258f, 0.844060f, 0.250284f, 
		-0.247727f, 0.897660f, -0.364469f, -0.574730f, 0.766377f, -0.286971f, 
		-0.576906f, 0.796778f, -0.179788f, -0.564294f, 0.800623f, -0.201431f, 
		-0.568226f, 0.723088f, -0.392764f, -0.586120f, 0.718607f, -0.374256f, 
		0.0696515f, 0.984581f, -0.160468f, -0.167907f, 0.975973f, 0.138868f, 
		-0.281065f, 0.896295f, -0.343012f, -0.352916f, 0.778652f, -0.518799f, 
		-0.414854f, 0.816016f, -0.402509f, -0.323145f, 0.773291f, -0.545526f, 
		-0.437386f, 0.858686f, -0.267119f, -0.831803f, -0.555071f, -7.08511e-006f, 
		-1.00000f, -2.97301e-007f, -8.79201e-006f, -1.00000f, -2.97301e-007f, -8.79201e-006f, 
		-1.00000f, -2.97301e-007f, -8.79201e-006f, -0.756714f, 0.653746f, -6.43471e-006f, 
		-0.430306f, 0.874439f, -0.224037f, -0.189328f, -0.981914f, -1.29161e-006f, 
		0.221141f, 0.974421f, 0.0400122f, -0.0622960f, 0.998058f, -3.77488e-007f, 
		-0.815876f, -0.554985f, 0.162292f, -0.980785f, -7.01387e-007f, 0.195094f, 
		-0.980785f, -7.01387e-007f, 0.195094f, -0.980785f, -7.01387e-007f, 0.195094f, 
		-0.742233f, 0.653675f, 0.147643f, -0.815870f, -0.554994f, -0.162292f, 
		-0.980784f, -1.73896e-007f, -0.195095f, -0.980784f, -1.73896e-007f, -0.195095f, 
		-0.980784f, -1.73896e-007f, -0.195095f, -0.742233f, 0.653675f, -0.147643f, 
		-0.185739f, -0.981904f, 0.0369473f, -0.185729f, -0.981906f, -0.0369452f, 
		-0.0546217f, 0.983156f, -0.174414f, -0.0611129f, 0.998057f, 0.0121568f, 
		-0.0611103f, 0.998057f, -0.0121557f, -0.221176f, 0.972779f, 0.0691509f, 
		0.220215f, 0.973116f, 0.0674631f, 0.0257947f, -0.999667f, 6.74374e-008f, 
		0.0253043f, -0.999667f, 0.00503324f, 0.0253033f, -0.999667f, -0.00503315f, 
		0.0426902f, 0.999088f, 3.73405e-007f, -0.0901925f, -0.995924f, -5.25793e-008f, 
		-0.768488f, -0.555067f, -0.318319f, -0.768500f, -0.555048f, 0.318323f, 
		-0.923880f, -2.26602e-007f, -0.382683f, -0.923879f, 9.79967e-007f, 0.382684f, 
		-0.923880f, -2.26602e-007f, -0.382683f, -0.923879f, 9.79967e-007f, 0.382684f, 
		-0.923880f, -2.26602e-007f, -0.382683f, -0.923879f, 9.79967e-007f, 0.382684f, 
		-0.699112f, 0.653747f, -0.289581f, -0.699115f, 0.653743f, 0.289584f, 
		0.0418774f, 0.999088f, -0.00832953f, 0.0418785f, 0.999088f, 0.00833022f, 
		-0.715430f, -0.698684f, -2.18506e-007f, -0.174940f, -0.981909f, 0.0724615f, 
		-0.174921f, -0.981913f, -0.0724544f, -0.111078f, 0.897168f, -0.427495f, 
		-0.0884733f, -0.995923f, -0.0175988f, -0.0884811f, -0.995922f, 0.0175996f, 
		-0.0575590f, 0.998057f, 0.0238417f, -0.0575534f, 0.998058f, -0.0238389f, 
		-0.701744f, -0.698620f, -0.139591f, -0.701752f, -0.698613f, 0.139585f, 
		0.110624f, 0.993862f, 1.42219e-007f, -0.177737f, 0.894776f, -0.409616f, 
		0.200146f, 0.979693f, 0.0119538f, 0.0238310f, -0.999667f, -0.00987106f, 
		0.0238317f, -0.999667f, 0.00987163f, 0.728273f, 0.685287f, 3.43595e-006f, 
		-0.981999f, -0.188885f, -8.94581e-007f, 0.108518f, 0.993860f, 0.0215854f, 
		0.108516f, 0.993860f, -0.0215843f, 0.995294f, 0.0969024f, 5.47861e-006f, 
		-0.293289f, 0.895382f, -0.335071f, 0.0394404f, 0.999088f, -0.0163369f, 
		0.0394409f, 0.999088f, 0.0163372f, -0.166810f, 0.971554f, -0.168097f, 
		0.714336f, 0.685225f, 0.142092f, 0.714345f, 0.685217f, -0.142092f, 
		-0.963135f, -0.188852f, -0.191587f, -0.963138f, -0.188854f, 0.191573f, 
		-0.237852f, 0.970126f, 0.0477696f, 0.954546f, 0.298065f, 2.72636e-006f, 
		0.976171f, 0.0968819f, -0.194175f, 0.976170f, 0.0968893f, 0.194176f, 
		-0.0833219f, -0.995925f, -0.0345144f, -0.0833315f, -0.995924f, 0.0345148f, 
		-0.365168f, 0.897659f, -0.246699f, -0.660976f, -0.698681f, 0.273780f, 
		-0.660966f, -0.698688f, -0.273784f, 0.935792f, 0.299404f, 0.186147f, 
		0.935791f, 0.299408f, -0.186142f, 0.205160f, 0.976065f, 0.0721564f, 
		-0.388176f, 0.896294f, -0.214420f, -0.691664f, -0.555002f, 0.462141f, 
		-0.831476f, 4.98542e-007f, 0.555561f, -0.831476f, 4.98542e-007f, 0.555561f, 
		-0.831476f, 4.98542e-007f, 0.555561f, -0.629241f, 0.653674f, 0.420435f, 
		-0.691656f, -0.555005f, -0.462149f, -0.831470f, 3.20314e-007f, -0.555570f, 
		-0.831470f, 3.20314e-007f, -0.555570f, -0.831470f, 3.20314e-007f, -0.555570f, 
		-0.629235f, 0.653676f, -0.420442f, 0.102199f, 0.993863f, -0.0423327f, 
		0.102205f, 0.993862f, 0.0423353f, -0.157448f, -0.981908f, 0.105198f, 
		-0.157445f, -0.981909f, -0.105199f, -0.0518095f, 0.998057f, 0.0346167f, 
		-0.0518059f, 0.998057f, -0.0346153f, 0.672835f, 0.685290f, 0.278695f, 
		0.672833f, 0.685292f, -0.278696f, -0.907250f, -0.188885f, 0.375791f, 
		-0.907248f, -0.188886f, -0.375798f, 0.919532f, 0.0969086f, 0.380880f, 
		0.919531f, 0.0969077f, -0.380883f, 0.0214517f, -0.999667f, -0.0143333f, 
		0.0214519f, -0.999667f, 0.0143341f, 0.143710f, 0.987223f, -0.0688350f, 
		0.881883f, 0.298067f, 0.365291f, 0.881885f, 0.298065f, -0.365288f, 
		0.0355019f, 0.999088f, -0.0237214f, 0.0355020f, 0.999088f, 0.0237223f, 
		0.0681161f, 0.981924f, -0.176593f, -0.214058f, 0.962038f, -0.169298f, 
		-0.0750070f, -0.995923f, 0.0501191f, -0.0750050f, -0.995923f, -0.0501191f, 
		-0.981999f, -0.188885f, -8.94581e-007f, -0.594915f, -0.698615f, 0.397510f, 
		-0.594912f, -0.698617f, -0.397510f, 0.855019f, 0.489914f, 0.170080f, 
		0.235312f, 0.970237f, -0.0571690f, 0.195521f, 0.970237f, 0.142871f, 
		0.855021f, 0.489912f, -0.170075f, 0.195523f, 0.970237f, -0.142867f, 
		0.235312f, 0.970237f, 0.0571688f, 0.874008f, 0.485911f, -1.86400e-008f, 
		0.235312f, 0.970237f, -0.0571690f, 0.235312f, 0.970237f, 0.0571688f, 
		-0.963137f, -0.188854f, -0.191576f, -0.963138f, -0.188854f, 0.191573f, 
		0.0920022f, 0.993859f, 0.0614743f, 0.0919981f, 0.993860f, -0.0614709f, 
		0.00522013f, 0.984581f, -0.174853f, 0.605587f, 0.685226f, -0.404637f, 
		0.605593f, 0.685213f, 0.404648f, -0.816508f, -0.188849f, 0.545573f, 
		-0.816507f, -0.188852f, -0.545574f, 0.530572f, 0.832304f, -0.160510f, 
		0.827556f, 0.0968865f, 0.552960f, 0.827559f, 0.0968955f, -0.552953f, 
		-0.115369f, 0.983156f, -0.141756f, 0.793324f, 0.299413f, 0.530085f, 
		0.793330f, 0.299404f, -0.530080f, 0.158929f, 0.982709f, 0.0949977f, 
		-0.907250f, -0.188885f, 0.375791f, -0.907248f, -0.188886f, -0.375798f, 
		-0.588166f, -0.555094f, 0.588160f, -0.707111f, -3.04551e-007f, 0.707103f, 
		-0.707111f, -3.04551e-007f, 0.707103f, -0.707111f, -3.04551e-007f, 0.707103f, 
		-0.535078f, 0.653748f, 0.535075f, -0.588167f, -0.555085f, -0.588166f, 
		-0.707108f, 4.74049e-007f, -0.707106f, -0.707108f, 4.74049e-007f, -0.707106f, 
		-0.707108f, 4.74049e-007f, -0.707106f, -0.535076f, 0.653746f, -0.535079f, 
		-0.133857f, -0.981919f, 0.133855f, -0.133863f, -0.981917f, -0.133863f, 
		0.807481f, 0.485907f, -0.334467f, 0.195523f, 0.970237f, -0.142867f, 
		0.239276f, 0.970238f, -0.0372332f, 0.807473f, 0.485917f, 0.334474f, 
		0.239278f, 0.970237f, 0.0372342f, 0.195521f, 0.970237f, 0.142871f, 
		-0.0440484f, 0.998058f, 0.0440488f, -0.0440497f, 0.998058f, -0.0440501f, 
		-0.217226f, 0.971556f, -0.0942997f, 0.0920647f, 0.985200f, -0.144589f, 
		-0.677999f, 0.333990f, 0.654804f, -0.261556f, 0.962040f, -0.0778883f, 
		0.0182396f, -0.999667f, -0.0182398f, 0.0182401f, -0.999667f, 0.0182403f, 
		-0.591852f, 0.577317f, 0.562509f, 0.470307f, 0.872863f, 0.130083f, 
		-0.611862f, 0.345028f, 0.711745f, 0.0301862f, 0.999088f, -0.0301860f, 
		0.0301864f, 0.999088f, 0.0301868f, -0.598212f, 0.0646486f, 0.798726f, 
		-0.543664f, 0.125233f, 0.829908f, -0.0637819f, -0.995923f, 0.0637838f, 
		-0.0637785f, -0.995924f, -0.0637780f, -0.981999f, -0.188885f, -8.94581e-007f, 
		-0.505893f, -0.698676f, 0.505889f, -0.505886f, -0.698683f, -0.505887f, 
		-0.423606f, 0.107368f, 0.899461f, -0.559408f, 0.721554f, 0.407949f, 
		0.516532f, 0.834163f, -0.193304f, -0.637655f, 0.327190f, 0.697383f, 
		-0.623573f, 0.555119f, 0.550454f, 0.724856f, 0.489910f, -0.484326f, 
		0.125966f, 0.970237f, -0.206815f, 0.239276f, 0.970238f, -0.0372332f, 
		0.724847f, 0.489919f, 0.484331f, 0.239278f, 0.970237f, 0.0372342f, 
		0.125968f, 0.970237f, 0.206815f, -0.963137f, -0.188854f, -0.191576f, 
		-0.963138f, -0.188854f, 0.191573f, -0.308817f, -0.00423718f, 0.951112f, 
		-0.816507f, -0.188853f, 0.545574f, -0.816507f, -0.188854f, -0.545574f, 
		-0.404150f, 0.112809f, 0.907710f, 0.0782244f, 0.993862f, -0.0782236f, 
		0.0782247f, 0.993862f, 0.0782241f, 0.514966f, 0.685291f, -0.514962f, 
		0.514970f, 0.685280f, 0.514974f, -0.694381f, -0.188885f, 0.694375f, 
		-0.694375f, -0.188888f, -0.694380f, -0.284217f, 0.0718467f, 0.956064f, 
		-0.679869f, 0.279064f, 0.678160f, 0.703775f, 0.0968997f, 0.703784f, 
		0.703782f, 0.0969088f, -0.703775f, -0.722482f, 0.512348f, 0.464240f, 
		-0.463712f, 0.860500f, 0.210974f, -0.691935f, -0.721960f, 1.55435e-007f, 
		0.229544f, 0.973114f, -0.0189562f, 0.0977925f, 0.988297f, 0.117072f, 
		0.674970f, 0.298064f, -0.674962f, 0.674962f, 0.298069f, 0.674967f, 
		-0.586863f, 0.709725f, 0.389721f, -0.907248f, -0.188886f, -0.375798f, 
		-0.907250f, -0.188885f, 0.375791f, -0.678185f, -0.722405f, -0.134894f, 
		-0.678186f, -0.722404f, 0.134894f, -0.628533f, 0.259617f, 0.733174f, 
		0.217302f, 0.976063f, -0.00901810f, -0.792655f, 0.489355f, 0.363633f, 
		0.490631f, 0.854080f, 0.172710f, 0.0761323f, 0.982883f, -0.167763f, 
		-0.639266f, -0.721958f, -0.264793f, -0.639265f, -0.721960f, 0.264790f, 
		-0.647554f, 0.691837f, 0.319429f, 0.182824f, 0.982708f, 0.0293313f, 
		-0.274128f, 0.960733f, -0.0429717f, -0.462147f, -0.555017f, 0.691648f, 
		-0.555573f, -1.54653e-007f, 0.831468f, -0.555573f, -1.54653e-007f, 0.831468f, 
		-0.555573f, -1.54653e-007f, 0.831468f, -0.420438f, 0.653680f, 0.629233f, 
		-0.462144f, -0.555010f, -0.691655f, -0.555568f, 5.84541e-007f, -0.831471f, 
		-0.555568f, 5.84541e-007f, -0.831471f, -0.555568f, 5.84541e-007f, -0.831471f, 
		-0.420438f, 0.653675f, -0.629238f, -0.801778f, 0.491021f, 0.340662f, 
		0.466771f, 0.837705f, -0.283506f, -0.469518f, 0.860016f, 0.199816f, 
		-0.0812511f, -0.996563f, -0.0161611f, -0.0812509f, -0.996563f, 0.0161615f, 
		-0.105194f, -0.981911f, 0.157431f, -0.105195f, -0.981910f, -0.157439f, 
		-0.0842478f, -0.996445f, 2.05349e-007f, 0.000000f, 1.00000f, 0.000000f, 
		0.235312f, 0.970237f, -0.0571690f, 0.195521f, 0.970237f, 0.142871f, 
		0.000000f, 1.00000f, 0.000000f, 0.195523f, 0.970237f, -0.142867f, 
		0.235312f, 0.970237f, 0.0571688f, -0.0346120f, 0.998057f, 0.0518019f, 
		-0.0346169f, 0.998057f, -0.0518087f, 0.134215f, 0.988297f, 0.0724969f, 
		-0.816507f, -0.188853f, 0.545573f, -0.816508f, -0.188849f, -0.545573f, 
		-0.694377f, -0.188888f, 0.694379f, -0.694380f, -0.188886f, -0.694376f, 
		0.618020f, 0.485913f, -0.618013f, 0.125966f, 0.970237f, -0.206815f, 
		0.206813f, 0.970238f, -0.125966f, 0.618017f, 0.485910f, 0.618017f, 
		0.206816f, 0.970237f, 0.125966f, 0.125968f, 0.970237f, 0.206815f, 
		0.0143337f, -0.999667f, -0.0214513f, 0.0143341f, -0.999667f, 0.0214513f, 
		0.107805f, 0.990346f, 0.0871404f, 0.0237219f, 0.999088f, -0.0355021f, 
		0.0237220f, 0.999088f, 0.0355026f, -0.681671f, 0.696963f, 0.222639f, 
		-0.0501277f, -0.995921f, 0.0750198f, -0.0501166f, -0.995923f, -0.0750056f, 
		0.0678393f, 0.990346f, 0.120882f, -0.574940f, -0.722401f, -0.384162f, 
		-0.574937f, -0.722404f, 0.384160f, -0.397519f, -0.698607f, 0.594918f, 
		-0.397505f, -0.698619f, -0.594912f, -0.0778349f, -0.996445f, -0.0322398f, 
		-0.0778343f, -0.996445f, 0.0322402f, 0.763245f, 0.606367f, -0.223108f, 
		0.0614715f, 0.993860f, -0.0919979f, 0.0614705f, 0.993860f, 0.0919955f, 
		-0.480732f, 0.861647f, 0.162670f, 0.487006f, 0.828044f, 0.277793f, 
		0.404643f, 0.685221f, -0.605587f, 0.404646f, 0.685216f, 0.605591f, 
		-0.545580f, -0.188854f, 0.816503f, -0.545569f, -0.188854f, -0.816510f, 
		0.000000f, 1.00000f, 0.000000f, 0.235312f, 0.970237f, -0.0571690f, 
		0.235312f, 0.970237f, 0.0571688f, 0.552958f, 0.0968867f, -0.827557f, 
		0.552957f, 0.0968832f, 0.827558f, -0.258637f, 0.963879f, -0.0635872f, 
		-0.682881f, 0.708457f, 0.178216f, 0.530084f, 0.299408f, -0.793326f, 
		0.530081f, 0.299409f, 0.793328f, -0.0688814f, -0.996563f, -0.0460258f, 
		-0.0688811f, -0.996563f, 0.0460246f, 0.000000f, 1.00000f, 0.000000f, 
		0.239278f, 0.970237f, 0.0372342f, 0.125968f, 0.970237f, 0.206815f, 
		0.000000f, 1.00000f, 0.000000f, 0.125966f, 0.970237f, -0.206815f, 
		0.239276f, 0.970238f, -0.0372332f, -0.694380f, -0.188886f, -0.694376f, 
		-0.694374f, -0.188886f, 0.694382f, 0.0396423f, 0.965580f, -0.257067f, 
		0.000000f, 1.00000f, 0.000000f, 0.239278f, 0.970237f, 0.0372342f, 
		0.195521f, 0.970237f, 0.142871f, 0.000000f, 1.00000f, 0.000000f, 
		0.195523f, 0.970237f, -0.142867f, 0.239276f, 0.970238f, -0.0372332f, 
		0.142572f, 0.973661f, -0.177923f, 0.102717f, 0.811355f, -0.575459f, 
		0.406637f, 0.839057f, -0.361427f, 0.757471f, 0.589962f, -0.279612f, 
		-0.489272f, -0.721959f, -0.489273f, -0.489269f, -0.721959f, 0.489275f, 
		0.485027f, 0.872861f, -0.0534906f, -0.476866f, 0.871924f, 0.111116f, 
		0.484328f, 0.489920f, -0.724849f, 0.0372334f, 0.970237f, -0.239278f, 
		0.206813f, 0.970238f, -0.125966f, 0.484328f, 0.489910f, 0.724855f, 
		0.206816f, 0.970237f, 0.125966f, 0.0372306f, 0.970237f, 0.239279f, 
		0.519731f, 0.854060f, -0.0214796f, -0.545580f, -0.188854f, 0.816503f, 
		-0.545569f, -0.188854f, -0.816510f, 0.146257f, 0.967496f, 0.206301f, 
		0.555645f, 0.827828f, 0.0771951f, -0.225122f, 0.969035f, -0.101451f, 
		0.536857f, 0.825496f, 0.174188f, 0.152562f, 0.580012f, -0.800194f, 
		0.818907f, 0.571458f, 0.0531624f, -0.0108563f, -0.999939f, -0.00215864f, 
		-0.0108563f, -0.999939f, 0.00216005f, 0.432674f, 0.826176f, 0.360869f, 
		0.0692413f, 0.949894f, -0.304806f, -0.0595715f, -0.996445f, -0.0595736f, 
		-0.0595718f, -0.996445f, 0.0595710f, 0.517534f, 0.830788f, 0.204819f, 
		-0.318315f, -0.555078f, 0.768482f, -0.382682f, -2.99908e-007f, 0.923880f, 
		-0.382682f, -2.99908e-007f, 0.923880f, -0.382682f, -2.99908e-007f, 0.923880f, 
		-0.289578f, 0.653752f, 0.699109f, -0.318314f, -0.555069f, -0.768489f, 
		-0.382681f, 1.82627e-007f, -0.923881f, -0.382681f, 1.82627e-007f, -0.923881f, 
		-0.382681f, 1.82627e-007f, -0.923881f, -0.289580f, 0.653746f, -0.699113f, 
		-0.0724498f, -0.981915f, 0.174910f, -0.0724526f, -0.981913f, -0.174921f, 
		-0.0238362f, 0.998058f, 0.0575465f, -0.0238398f, 0.998058f, -0.0575537f, 
		-0.467488f, 0.879848f, 0.0855680f, 0.832277f, 0.537230f, 0.136746f, 
		0.133792f, 0.867413f, 0.479265f, 0.00987161f, -0.999667f, -0.0238303f, 
		0.00987163f, -0.999667f, 0.0238314f, 0.693773f, 0.575650f, -0.432790f, 
		-0.545569f, -0.188854f, -0.816510f, -0.545580f, -0.188854f, 0.816503f, 
		0.0163368f, 0.999088f, -0.0394403f, 0.0163368f, 0.999088f, 0.0394408f, 
		-0.00889644f, -0.999960f, 7.52891e-007f, -0.0345201f, -0.995923f, 0.0833363f, 
		-0.0345132f, -0.995925f, -0.0833244f, 0.0613093f, 0.973116f, 0.222007f, 
		-0.273787f, -0.698679f, 0.660975f, -0.273782f, -0.698686f, -0.660970f, 
		0.461335f, 0.819375f, -0.340285f, 0.000000f, 1.00000f, 0.000000f, 
		0.206816f, 0.970237f, 0.125966f, 0.125968f, 0.970237f, 0.206815f, 
		0.000000f, 1.00000f, 0.000000f, 0.125966f, 0.970237f, -0.206815f, 
		0.206813f, 0.970238f, -0.125966f, 0.0423357f, 0.993862f, -0.102209f, 
		0.0423346f, 0.993862f, 0.102206f, -0.384157f, -0.722405f, -0.574938f, 
		-0.384163f, -0.722402f, 0.574938f, 0.142745f, 0.872862f, 0.466623f, 
		-0.00920343f, -0.999939f, -0.00615014f, -0.00920340f, -0.999939f, 0.00614885f, 
		0.278697f, 0.685290f, -0.672834f, 0.278697f, 0.685287f, 0.672837f, 
		-0.375800f, -0.188887f, 0.907246f, -0.375794f, -0.188887f, -0.907249f, 
		-0.192117f, 0.972284f, -0.133245f, 0.380882f, 0.0969101f, -0.919531f, 
		0.380879f, 0.0969096f, 0.919532f, -0.00821914f, -0.999960f, -0.00340451f, 
		-0.00821911f, -0.999960f, 0.00340458f, 0.365288f, 0.298065f, -0.881885f, 
		0.365282f, 0.298069f, 0.881887f, 0.770860f, 0.512943f, 0.377710f, 
		0.383313f, 0.840627f, -0.382645f, 0.107465f, 0.915638f, -0.387374f, 
		0.918858f, 0.383964f, -0.0909444f, -0.0460243f, -0.996563f, -0.0688822f, 
		-0.0460247f, -0.996563f, 0.0688810f, 0.857066f, 0.511402f, 0.0625057f, 
		0.000000f, 1.00000f, 0.000000f, 0.206816f, 0.970237f, 0.125966f, 
		0.0372306f, 0.970237f, 0.239279f, 0.000000f, 1.00000f, 0.000000f, 
		0.0372334f, 0.970237f, -0.239278f, 0.206813f, 0.970238f, -0.125966f, 
		0.186182f, 0.981924f, 0.0340912f, 0.823699f, 0.537212f, -0.181448f, 
		0.417564f, 0.419968f, -0.805771f, 0.915250f, 0.353878f, -0.192583f, 
		0.467223f, 0.778839f, -0.418465f, -0.375800f, -0.188887f, 0.907246f, 
		-0.375794f, -0.188887f, -0.907249f, 0.784114f, 0.542817f, 0.300859f, 
		0.394787f, 0.571456f, 0.719431f, 0.334468f, 0.485918f, -0.807474f, 
		0.0372334f, 0.970237f, -0.239278f, 0.142868f, 0.970237f, -0.195521f, 
		0.334461f, 0.485911f, 0.807481f, 0.142865f, 0.970237f, 0.195524f, 
		0.0372306f, 0.970237f, 0.239279f, 0.402809f, 0.831749f, 0.382018f, 
		0.780292f, 0.571462f, -0.254119f, 0.118655f, 0.718082f, 0.685769f, 
		0.110119f, 0.979115f, -0.170900f, -0.176977f, 0.973050f, -0.147829f, 
		0.583935f, 0.593059f, -0.554347f, 0.972040f, 0.234799f, 0.00268410f, 
		0.732452f, 0.571839f, 0.369478f, 0.0752497f, 0.816398f, -0.572565f, 
		0.113183f, 0.971106f, 0.210099f, 0.799779f, 0.334879f, -0.498206f, 
		0.967420f, 0.240031f, 0.0805227f, -0.00629072f, -0.999960f, -0.00629149f, 
		-0.00629070f, -0.999960f, 0.00628940f, 0.801439f, 0.553898f, -0.225592f, 
		-0.375794f, -0.188887f, -0.907249f, -0.375800f, -0.188887f, 0.907246f, 
		0.116416f, 0.893207f, -0.434313f, 0.858394f, 0.252080f, 0.446783f, 
		0.395002f, 0.715092f, -0.576730f, 0.958607f, 0.257181f, -0.122192f, 
		-0.264789f, -0.721960f, -0.639265f, -0.264792f, -0.721960f, 0.639265f, 
		0.901810f, 0.228657f, -0.366680f, 0.623554f, 0.254348f, 0.739248f, 
		0.982033f, 0.134899f, 0.131960f, 0.399376f, 0.897171f, -0.188634f, 
		0.609093f, 0.547739f, 0.573574f, -0.00614960f, -0.999939f, -0.00920477f, 
		-0.00614961f, -0.999939f, 0.00920276f, 0.778614f, 0.520548f, -0.350415f, 
		0.798960f, 0.311942f, 0.514155f, 0.107138f, 0.865556f, 0.489219f, 
		0.102287f, 0.553999f, -0.826210f, 0.983897f, 0.0637330f, 0.166985f, 
		0.984410f, 0.0209460f, 0.174639f, -0.0322403f, -0.996445f, -0.0778345f, 
		-0.0322395f, -0.996445f, 0.0778340f, 0.855700f, 0.112976f, -0.504988f, 
		0.718122f, 0.269301f, -0.641699f, 0.981161f, 0.192346f, -0.0180407f, 
		0.818458f, 0.0523302f, -0.572178f, 0.984172f, 0.152709f, -0.0899192f, 
		0.809421f, 0.0208271f, -0.586860f, -0.162286f, -0.555007f, 0.815862f, 
		-0.195090f, -1.24137e-007f, 0.980785f, -0.195090f, -1.24137e-007f, 0.980785f, 
		-0.195090f, -1.24137e-007f, 0.980785f, -0.147640f, 0.653677f, 0.742233f, 
		-0.162284f, -0.554994f, -0.815871f, -0.195088f, -1.43786e-007f, -0.980786f, 
		-0.195088f, -1.43786e-007f, -0.980786f, -0.195088f, -1.43786e-007f, -0.980786f, 
		-0.147637f, 0.653677f, -0.742233f, 0.815685f, 0.185781f, 0.547854f, 
		-0.0369414f, -0.981909f, 0.185713f, -0.0369438f, -0.981906f, -0.185730f, 
		-0.0121553f, 0.998057f, 0.0611084f, -0.0121553f, 0.998057f, -0.0611086f, 
		0.672902f, 0.189980f, 0.714920f, 0.840150f, 0.255044f, -0.478645f, 
		0.752001f, 0.126920f, 0.646828f, 0.826765f, 0.207771f, -0.522772f, 
		0.544277f, 0.500768f, -0.673048f, 0.00503334f, -0.999667f, -0.0253041f, 
		0.00503336f, -0.999667f, 0.0253050f, 0.000000f, 1.00000f, 0.000000f, 
		0.142865f, 0.970237f, 0.195524f, 0.0372306f, 0.970237f, 0.239279f, 
		0.000000f, 1.00000f, 0.000000f, 0.0372334f, 0.970237f, -0.239278f, 
		0.142868f, 0.970237f, -0.195521f, 0.00833000f, 0.999088f, -0.0418770f, 
		0.00832974f, 0.999088f, 0.0418782f, -0.0175994f, -0.995922f, 0.0884821f, 
		-0.0175998f, -0.995922f, -0.0884822f, 0.560070f, 0.0202888f, 0.828197f, 
		0.581334f, 0.0800143f, 0.809721f, -0.0216199f, 0.482734f, 0.875500f, 
		0.792257f, 0.0603684f, 0.607194f, -0.139583f, -0.698615f, 0.701751f, 
		-0.139587f, -0.698613f, -0.701751f, 0.970354f, 0.203628f, 0.130190f, 
		0.945519f, 0.325272f, -0.0138331f, 0.263229f, 0.701910f, -0.661840f, 
		0.0215882f, 0.993859f, -0.108530f, 0.0215865f, 0.993859f, 0.108524f, 
		0.928562f, 0.325751f, -0.177931f, 0.121560f, 0.710928f, 0.692679f, 
		0.116770f, 0.887336f, -0.446094f, 0.142094f, 0.685222f, -0.714339f, 
		0.142087f, 0.685223f, 0.714340f, -0.191575f, -0.188853f, 0.963137f, 
		-0.191584f, -0.188854f, -0.963135f, 0.194173f, 0.0968998f, -0.976170f, 
		0.194166f, 0.0968977f, 0.976171f, 0.306765f, 0.352291f, -0.884187f, 
		0.700320f, 0.362176f, 0.615126f, 0.793148f, 0.0312681f, 0.608226f, 
		0.186138f, 0.299404f, -0.935794f, 0.186136f, 0.299408f, 0.935793f, 
		0.837001f, 0.0902177f, -0.539713f, 0.572925f, 0.368798f, -0.731946f, 
		0.587742f, 0.383401f, -0.712435f, 0.697701f, 0.0239299f, -0.715989f, 
		0.958404f, 0.152987f, -0.240951f, 0.846115f, 0.00477864f, -0.532978f, 
		0.170072f, 0.489913f, -0.855021f, -0.0571655f, 0.970238f, -0.235311f, 
		0.142868f, 0.970237f, -0.195521f, 0.170070f, 0.489913f, 0.855021f, 
		0.142865f, 0.970237f, 0.195524f, -0.0571694f, 0.970237f, 0.235312f, 
		0.762540f, 0.131753f, -0.633383f, -0.191584f, -0.188854f, -0.963135f, 
		-0.191575f, -0.188853f, 0.963137f, 0.988559f, 0.149456f, -0.0203479f, 
		0.521000f, 0.325287f, 0.789144f, 0.536216f, 0.611388f, -0.581959f, 
		0.976304f, 0.209160f, -0.0555147f, 0.475913f, 0.773294f, -0.418954f, 
		0.690441f, 0.168864f, 0.703404f, 0.709695f, 0.0608735f, -0.701874f, 
		0.991476f, 0.110124f, 0.0696269f, -0.00340457f, -0.999960f, -0.00821983f, 
		-0.00340445f, -0.999960f, 0.00821899f, 0.811432f, 0.156680f, -0.563054f, 
		0.0492667f, 0.976065f, 0.211825f, 0.976586f, 0.178694f, 0.119787f, 
		0.117745f, 0.854060f, 0.506673f, 0.182457f, 0.519348f, -0.834857f, 
		0.604633f, 0.326055f, -0.726710f, -0.0770777f, 0.181953f, 0.980282f, 
		0.100679f, 0.492799f, 0.864299f, -0.191584f, -0.188854f, -0.963135f, 
		-0.191575f, -0.188853f, 0.963137f, 0.413033f, 0.0233135f, 0.910417f, 
		0.644926f, 0.0978930f, 0.757950f, -0.443393f, 0.0910374f, 0.891692f, 
		0.972010f, 0.126855f, -0.197748f, 0.331446f, 0.537208f, 0.775597f, 
		0.578308f, 0.222504f, -0.784889f, -0.134901f, -0.722404f, -0.678184f, 
		-0.134890f, -0.722404f, 0.678186f, 0.536231f, 0.229810f, 0.812185f, 
		0.172775f, 0.984581f, -0.0273843f, 0.984524f, 0.117195f, -0.130300f, 
		0.985207f, 0.170236f, -0.0196747f, 0.978897f, 0.130568f, 0.157200f, 
		0.961855f, 0.0968553f, 0.255839f, -0.0161622f, -0.996563f, -0.0812515f, 
		-0.0161613f, -0.996563f, 0.0812518f, -0.471067f, 0.120075f, 0.873887f, 
		0.0408405f, 0.245964f, 0.968418f, 0.000000f, 1.00000f, 0.000000f, 
		-0.0571655f, 0.970238f, -0.235311f, 0.142868f, 0.970237f, -0.195521f, 
		0.000000f, 1.00000f, 0.000000f, 0.142865f, 0.970237f, 0.195524f, 
		-0.0571694f, 0.970237f, 0.235312f, 0.612632f, 0.304238f, -0.729466f, 
		0.532329f, 0.578797f, 0.617754f, 0.986965f, 0.146983f, 0.0655474f, 
		0.438643f, 0.231828f, -0.868244f, 0.209488f, 0.712828f, -0.669322f, 
		0.434884f, 0.0954302f, 0.895416f, -0.00740572f, 0.376553f, -0.926365f, 
		0.656967f, 0.106042f, 0.746424f, 0.523966f, 0.232155f, 0.819490f, 
		0.696937f, 0.0991320f, -0.710248f, 0.703552f, 0.0393126f, -0.709556f, 
		0.256259f, 0.216252f, 0.942107f, 0.634857f, 0.131681f, 0.761326f, 
		-0.00466656f, 0.0516958f, 0.998652f, 0.993430f, 0.0890052f, 0.0719320f, 
		0.101075f, 0.0931277f, 0.990511f, 0.958109f, 0.0684718f, 0.278100f, 
		-0.556270f, 0.101823f, 0.824740f, -0.0279481f, 0.237871f, 0.970895f, 
		0.369418f, 0.894779f, -0.250800f, -0.118507f, 0.285056f, -0.951157f, 
		0.0667641f, 0.212030f, 0.974980f, 0.793307f, 0.141868f, -0.592062f, 
		0.0195751f, 0.975952f, 0.217103f, -0.00215949f, -0.999939f, -0.0108564f, 
		-0.00215954f, -0.999939f, 0.0108564f, -0.285169f, 0.0838368f, -0.954804f, 
		0.598886f, 0.148119f, 0.787017f, 0.0179977f, 0.986710f, -0.161491f, 
		0.0316809f, 0.860277f, 0.508841f, 0.125757f, 0.493700f, 0.860491f, 
		0.0788472f, 0.695123f, 0.714553f, -0.236592f, 0.0346103f, -0.970993f, 
		0.321461f, 0.162372f, -0.932898f, -0.139955f, 0.256272f, -0.956419f, 
		0.0590148f, 0.548558f, -0.834027f, -0.00599367f, 0.826257f, -0.563262f, 
		-0.825851f, 0.124506f, -0.549971f, 0.258755f, 0.718603f, -0.645489f, 
		-0.0162713f, 0.126899f, 0.991782f, -0.263230f, 0.379771f, 0.886840f, 
		-0.921826f, 0.135157f, -0.363277f, -0.577589f, 0.214907f, 0.787531f, 
		0.127386f, 0.0239210f, 0.991565f, -0.00673412f, 0.259371f, 0.965754f, 
		-0.932366f, 0.142813f, 0.332112f, 0.0997257f, 0.227939f, 0.968555f, 
		-0.0590910f, 0.535328f, -0.842575f, 0.444120f, 0.778651f, -0.443238f, 
		-0.434178f, 0.226430f, -0.871905f, -0.837305f, 0.154592f, -0.524425f, 
		-0.114005f, 0.292852f, -0.949337f, 0.149739f, 0.153922f, 0.976671f, 
		-0.142598f, 0.188305f, -0.971703f, 0.483368f, 0.401686f, -0.777820f, 
		-0.152644f, 0.114819f, -0.981589f, -0.166126f, 0.384220f, -0.908172f, 
		-0.159195f, 0.0719389f, -0.984623f, -0.588873f, 0.126225f, 0.798308f, 
		0.284330f, 0.142196f, -0.948123f, 0.00922565f, 0.243100f, 0.969957f, 
		-0.0465996f, 0.655606f, -0.753664f, -0.288695f, 0.228496f, -0.929755f, 
		0.116253f, 0.450956f, 0.884943f, -0.273348f, 0.265217f, -0.924630f, 
		-0.138581f, 0.0795599f, 0.987150f, 0.280249f, 0.723079f, -0.631361f, 
		-0.791400f, 0.124892f, 0.598404f, 0.383045f, 0.376906f, 0.843338f, 
		-0.804288f, 0.154856f, 0.573708f, -0.405701f, 0.0448965f, -0.912902f, 
		1.93258e-008f, -0.999960f, -0.00889659f, 0.000000f, -1.00000f, 1.18019e-007f, 
		2.22363e-008f, -0.999960f, 0.00889656f, -1.01741e-007f, -0.996445f, -0.0842484f, 
		1.84814e-007f, -0.996445f, 0.0842499f, 3.01727e-007f, -0.721961f, -0.691934f, 
		3.51557e-006f, -0.721960f, 0.691935f, 6.71872e-007f, -0.188890f, -0.981998f, 
		2.78480e-006f, -0.188890f, 0.981998f, -7.35502e-007f, -0.188887f, 0.981999f, 
		1.60950e-007f, -0.188887f, 0.981999f, 0.000000f, 1.00000f, 0.000000f, 
		0.000000f, 1.00000f, 0.000000f, -0.0571655f, 0.970238f, -0.235311f, 
		0.0571688f, 0.970237f, -0.235312f, -1.64405e-006f, 0.485905f, -0.874011f, 
		-0.0571655f, 0.970238f, -0.235311f, 0.0571688f, 0.970237f, -0.235312f, 
		1.59715e-006f, -0.698681f, 0.715433f, 3.23617e-007f, -0.981916f, 0.189317f, 
		2.93965e-008f, -0.999667f, -0.0257949f, 3.69464e-007f, -0.995924f, 0.0902005f, 
		1.90702e-007f, -0.555080f, 0.831797f, -2.50826e-007f, -7.84595e-008f, 1.00000f, 
		8.35651e-007f, 0.298064f, -0.954546f, 6.92579e-007f, -8.27910e-008f, 1.00000f, 
		3.35792e-006f, 0.0969167f, -0.995292f, 6.92579e-007f, -8.27910e-008f, 1.00000f, 
		3.16018e-006f, 0.685289f, -0.728271f, 9.10671e-007f, 0.993861f, -0.110632f, 
		-4.76337e-007f, 0.653745f, 0.756715f, 7.98740e-008f, 0.999088f, -0.0426899f, 
		-1.47256e-007f, 0.998058f, 0.0622976f, 4.52905e-007f, -0.188888f, -0.981999f, 
		5.80167e-008f, -0.188890f, -0.981998f, 0.000000f, 1.00000f, 0.000000f, 
		0.0571694f, 0.970237f, 0.235312f, -0.0571694f, 0.970237f, 0.235312f, 
		-3.00104e-007f, 0.485911f, 0.874008f, 0.0571694f, 0.970237f, 0.235312f, 
		-0.0571694f, 0.970237f, 0.235312f, 5.14123e-008f, -0.698675f, -0.715440f, 
		-2.62701e-007f, -0.981914f, -0.189327f, 1.46400e-008f, -0.999667f, 0.0257958f, 
		1.47410e-007f, -0.995923f, -0.0902093f, -7.89728e-007f, -0.555072f, -0.831802f, 
		-1.24103e-006f, 1.83194e-008f, -1.00000f, 6.26739e-007f, 0.298066f, 0.954545f, 
		-2.88262e-007f, 2.97250e-008f, -1.00000f, 1.55542e-006f, 0.0969155f, 0.995293f, 
		8.94736e-007f, 1.85696e-008f, -1.00000f, 9.04933e-007f, 0.685295f, 0.728266f, 
		-1.24500e-007f, 0.993862f, 0.110625f, 5.56086e-007f, 0.653746f, -0.756714f, 
		-7.31208e-008f, 0.999088f, 0.0426906f, 6.42782e-008f, 0.998058f, -0.0622956f, 
		-0.921136f, 0.158227f, -0.355630f, 0.0326462f, 0.674144f, -0.737879f, 
		0.0506448f, 0.0927586f, -0.994400f, 0.0522437f, 0.206071f, 0.977141f, 
		-0.0774938f, 0.237792f, -0.968220f, -0.382618f, 0.246143f, 0.890515f, 
		-0.255800f, 0.160704f, 0.953279f, 0.308973f, 0.251613f, -0.917184f, 
		-0.157553f, 0.625239f, -0.764365f, -0.940187f, 0.161338f, 0.300031f, 
		-0.180460f, 0.118680f, 0.976396f, -0.410317f, 0.544000f, 0.731918f, 
		-0.884544f, 0.253786f, -0.391375f, 0.0770226f, 0.0233085f, -0.996757f, 
		-0.320761f, 0.357927f, -0.876927f, -0.446696f, 0.0854121f, -0.890599f, 
		0.156118f, 0.138516f, -0.977978f, -0.455182f, 0.459597f, 0.762614f, 
		0.100579f, 0.251589f, 0.962594f, -0.834097f, 0.0705274f, -0.547091f, 
		-0.0542689f, 0.681608f, -0.729703f, 0.0323905f, 0.670666f, 0.741052f, 
		-0.843076f, 0.0129165f, -0.537639f, -0.873979f, 0.127545f, 0.468928f, 
		0.201447f, 0.410117f, 0.889507f, -0.617131f, 0.431008f, -0.658317f, 
		-0.684169f, 0.203291f, 0.700418f, -0.701819f, 0.368670f, 0.609535f, 
		0.185721f, 0.163523f, -0.968901f, -0.161491f, 0.0961612f, 0.982178f, 
		-0.0693556f, 0.0784416f, 0.994503f, -0.0431952f, 0.0208141f, 0.998850f, 
		-0.741754f, 0.198836f, -0.640519f, -0.593303f, 0.271183f, 0.757925f, 
		-0.279833f, 0.0919360f, 0.955637f, 0.175086f, 0.766369f, -0.618081f, 
		-0.188651f, 0.198260f, 0.961823f, 0.0826506f, 0.252770f, 0.963990f, 
		0.318351f, 0.816012f, -0.482470f, 0.0979609f, 0.511399f, 0.853742f, 
		-0.942906f, 0.113906f, -0.312975f, 0.274694f, 0.895384f, -0.350472f, 
		-0.859816f, 0.247532f, 0.446592f, -0.535359f, 0.464543f, -0.705401f, 
		-0.175267f, 0.144200f, 0.973903f, -0.0581556f, 0.704078f, -0.707737f, 
		-0.570000f, 0.201122f, -0.796649f, -0.886995f, 0.150319f, 0.436627f, 
		-0.0425715f, 0.682501f, -0.729644f, -0.380221f, 0.333596f, 0.862639f, 
		-0.937850f, 0.234501f, -0.255826f, -0.599584f, 0.266926f, 0.754487f, 
		0.0274829f, 0.827827f, 0.560310f, -0.117362f, 0.0779081f, -0.990029f, 
		0.117825f, 0.983156f, -0.139717f, -0.0914015f, 0.0202915f, -0.995607f, 
		-0.0395971f, 0.849997f, 0.525297f, -0.234828f, 0.197686f, -0.951723f, 
		-0.906114f, -0.0116014f, -0.422874f, 0.00215941f, -0.999939f, -0.0108567f, 
		0.00215950f, -0.999939f, 0.0108563f, -0.490755f, 0.354889f, -0.795747f, 
		0.00517478f, 0.982709f, 0.185084f, -0.941285f, 0.214375f, 0.260817f, 
		-0.369944f, 0.739018f, 0.563022f, -0.940109f, 0.138033f, -0.311675f, 
		-0.458190f, 0.213609f, 0.862806f, -0.354870f, 0.128410f, 0.926055f, 
		-0.438221f, 0.301710f, -0.846719f, -0.279570f, 0.0977243f, 0.955139f, 
		-0.898144f, 0.0680727f, -0.434400f, -0.254717f, 0.0815131f, 0.963574f, 
		0.0693517f, 0.796779f, -0.600278f, -0.593577f, 0.646150f, -0.479747f, 
		-0.959796f, 0.0931373f, 0.264798f, -0.972036f, 0.0927652f, -0.215735f, 
		-0.956450f, 0.0239368f, 0.290912f, -0.0680977f, 0.972775f, 0.221521f, 
		-0.541071f, 0.263483f, 0.798636f, -0.500642f, 0.442413f, 0.744062f, 
		0.000000f, 1.00000f, 0.000000f, -0.142868f, 0.970237f, -0.195521f, 
		0.0571688f, 0.970237f, -0.235312f, 0.000000f, 1.00000f, 0.000000f, 
		0.0571694f, 0.970237f, 0.235312f, -0.142868f, 0.970237f, 0.195521f, 
		0.0161621f, -0.996563f, -0.0812522f, 0.0161617f, -0.996563f, 0.0812522f, 
		-0.969961f, 0.0233020f, -0.242144f, -0.926715f, 0.221316f, 0.303675f, 
		-0.769827f, 0.344220f, 0.537474f, -0.603892f, 0.410209f, -0.683405f, 
		-0.778259f, 0.0209556f, -0.627593f, -0.776238f, 0.0743628f, -0.626039f, 
		0.134902f, -0.722404f, -0.678185f, 0.134895f, -0.722404f, 0.678186f, 
		-0.992071f, 0.0208323f, 0.123943f, -0.961966f, 0.237871f, 0.134308f, 
		-0.992146f, 0.0784509f, 0.0974226f, -0.813389f, 0.243886f, -0.528126f, 
		-0.708652f, 0.380951f, -0.593876f, -0.748552f, 0.0209202f, 0.662746f, 
		-0.213505f, 0.580279f, -0.785933f, -0.0684645f, 0.985020f, -0.158267f, 
		-0.766484f, 0.172838f, -0.618570f, -0.698294f, 0.240909f, 0.674054f, 
		-0.746683f, 0.0743557f, 0.661011f, 0.191581f, -0.188855f, -0.963136f, 
		0.191571f, -0.188851f, 0.963139f, -0.930042f, 0.201885f, -0.307025f, 
		0.0317163f, 0.656144f, 0.753969f, -0.996913f, 0.0202701f, -0.0758522f, 
		-0.979814f, 0.198257f, -0.0256680f, -0.0844298f, 0.835527f, -0.542924f, 
		-0.995739f, 0.0779112f, -0.0493277f, 0.00340448f, -0.999960f, -0.00821922f, 
		0.00340439f, -0.999960f, 0.00821946f, -0.460129f, 0.528417f, 0.713482f, 
		-0.737691f, 0.172851f, 0.652636f, -0.967588f, 0.237795f, -0.0850099f, 
		0.181125f, 0.858687f, -0.479428f, -0.760209f, 0.245393f, -0.601552f, 
		0.0929351f, 0.800629f, -0.591909f, -0.311945f, 0.767174f, 0.560477f, 
		0.191571f, -0.188851f, 0.963139f, 0.191581f, -0.188855f, -0.963136f, 
		-0.708382f, 0.0816727f, 0.701088f, -0.900955f, 0.406866f, -0.150800f, 
		-0.691677f, 0.0365594f, 0.721281f, -0.170071f, 0.489908f, -0.855024f, 
		-0.142868f, 0.970237f, -0.195521f, 0.0571688f, 0.970237f, -0.235312f, 
		-0.170075f, 0.489910f, 0.855022f, 0.0571694f, 0.970237f, 0.235312f, 
		-0.142868f, 0.970237f, 0.195521f, -0.632829f, 0.606433f, -0.481422f, 
		-0.671683f, 0.0130011f, 0.740724f, -0.689650f, 0.182305f, 0.700819f, 
		-0.977551f, 0.197696f, 0.0728716f, -0.417865f, 0.895457f, -0.153442f, 
		-0.350060f, 0.323797f, 0.878985f, -0.648892f, 0.689931f, -0.320835f, 
		-0.929886f, 0.283325f, -0.234606f, -0.186137f, 0.299405f, -0.935794f, 
		-0.186142f, 0.299406f, 0.935792f, -0.999884f, -0.00908179f, 0.0122658f, 
		-0.764385f, 0.331746f, -0.552866f, -0.902665f, 0.381431f, 0.199262f, 
		-0.975653f, -0.00863541f, 0.219151f, -0.707477f, 0.252481f, 0.660098f, 
		-0.706140f, 0.301863f, 0.640503f, -0.396799f, 0.736590f, 0.547710f, 
		-0.907504f, 0.334308f, 0.254313f, -0.194171f, 0.0968926f, -0.976171f, 
		-0.194173f, 0.0968990f, 0.976170f, 0.191571f, -0.188851f, 0.963139f, 
		0.191581f, -0.188855f, -0.963136f, -0.142093f, 0.685220f, -0.714342f, 
		-0.142091f, 0.685224f, 0.714338f, -0.925927f, 0.373292f, -0.0575580f, 
		-0.949931f, 0.0940387f, 0.297974f, -0.996898f, -0.0768968f, -0.0167485f, 
		-0.974551f, -0.0611344f, 0.215668f, -0.0215870f, 0.993859f, -0.108523f, 
		-0.0215874f, 0.993859f, 0.108524f, -0.936080f, 0.332825f, -0.113938f, 
		-0.981394f, 0.125556f, 0.145261f, 0.139582f, -0.698612f, 0.701753f, 
		0.139590f, -0.698607f, -0.701757f, -0.145274f, 0.943878f, 0.296632f, 
		0.0176001f, -0.995922f, 0.0884832f, 0.0176024f, -0.995921f, -0.0884910f, 
		-0.625077f, 0.427859f, 0.652852f, -0.00833003f, 0.999088f, -0.0418782f, 
		-0.00833016f, 0.999088f, 0.0418775f, -0.695219f, 0.576958f, 0.428708f, 
		0.000000f, 1.00000f, 0.000000f, -0.142868f, 0.970237f, -0.195521f, 
		-0.0372333f, 0.970238f, -0.239275f, 0.000000f, 1.00000f, 0.000000f, 
		-0.0372341f, 0.970238f, 0.239275f, -0.142868f, 0.970237f, 0.195521f, 
		-0.902565f, 0.326196f, -0.281022f, -0.00503316f, -0.999667f, -0.0253033f, 
		-0.00503355f, -0.999667f, 0.0253059f, 0.0121549f, 0.998057f, 0.0611093f, 
		0.0121567f, 0.998057f, -0.0611134f, -0.934380f, 0.345024f, 0.0888434f, 
		0.0369410f, -0.981908f, 0.185721f, 0.0369440f, -0.981907f, -0.185723f, 
		0.174502f, 0.897659f, -0.404670f, 0.162282f, -0.555001f, 0.815866f, 
		0.195085f, -2.24550e-008f, 0.980786f, 0.195085f, -2.24550e-008f, 0.980786f, 
		0.195085f, -2.24550e-008f, 0.980786f, 0.147635f, 0.653676f, 0.742234f, 
		0.162290f, -0.555000f, -0.815866f, 0.195095f, 1.64574e-007f, -0.980784f, 
		0.195095f, 1.64574e-007f, -0.980784f, 0.195095f, 1.64574e-007f, -0.980784f, 
		0.147642f, 0.653674f, -0.742234f, -0.862824f, 0.221904f, 0.454195f, 
		-0.716042f, 0.537362f, -0.445562f, -0.435780f, 0.881528f, -0.181672f, 
		-0.563849f, 0.423200f, -0.709209f, -0.866984f, 0.324380f, -0.378306f, 
		0.0322405f, -0.996445f, -0.0778360f, 0.0322397f, -0.996445f, 0.0778353f, 
		-0.658909f, 0.286751f, 0.695422f, -0.886533f, 0.359909f, 0.290733f, 
		-0.149802f, 0.542820f, 0.826381f, -0.895137f, 0.263926f, 0.359268f, 
		0.00614947f, -0.999939f, -0.00920362f, 0.00614952f, -0.999939f, 0.00920331f, 
		-0.709996f, 0.637972f, -0.298157f, -0.0620542f, 0.844059f, 0.532648f, 
		0.140096f, 0.874443f, -0.464460f, -0.0389108f, 0.970273f, 0.238863f, 
		0.264790f, -0.721960f, -0.639265f, 0.264793f, -0.721958f, 0.639266f, 
		-0.775075f, 0.500437f, -0.385773f, -0.884574f, 0.343199f, -0.315823f, 
		-0.722834f, 0.448660f, 0.525562f, -0.468010f, 0.852339f, -0.233420f, 
		-0.892515f, 0.417027f, 0.171769f, -0.578144f, 0.541738f, 0.610138f, 
		-0.473085f, 0.734189f, 0.486987f, 0.375794f, -0.188887f, 0.907249f, 
		0.375793f, -0.188889f, -0.907249f, 0.00629060f, -0.999960f, -0.00629021f, 
		0.00629069f, -0.999960f, 0.00629073f, -0.729739f, 0.563495f, 0.387240f, 
		0.0522676f, 0.971554f, -0.230977f, -0.166842f, 0.945819f, 0.278550f, 
		-0.0713135f, 0.825495f, 0.559886f, -0.310713f, 0.775163f, 0.550073f, 
		-0.790539f, 0.492808f, -0.363577f, -0.0489817f, 0.977140f, 0.206877f, 
		-0.496143f, 0.826275f, -0.266669f, -0.811090f, 0.556035f, -0.181545f, 
		-0.0462697f, 0.988297f, 0.145356f, -0.334469f, 0.485908f, -0.807480f, 
		-0.142868f, 0.970237f, -0.195521f, -0.0372333f, 0.970238f, -0.239275f, 
		-0.334474f, 0.485907f, 0.807479f, -0.0372341f, 0.970238f, 0.239275f, 
		-0.142868f, 0.970237f, 0.195521f, 0.375787f, -0.188885f, 0.907252f, 
		0.375797f, -0.188888f, -0.907248f, -0.256646f, 0.618801f, -0.742441f, 
		-0.744915f, 0.468820f, 0.474668f, -0.466471f, 0.370926f, 0.803006f, 
		-0.0952294f, 0.970126f, 0.223129f, -0.793519f, 0.558429f, 0.241838f, 
		0.101038f, 0.994828f, 0.0103936f, 0.000000f, 1.00000f, 0.000000f, 
		-0.206816f, 0.970237f, -0.125966f, -0.0372333f, 0.970238f, -0.239275f, 
		0.000000f, 1.00000f, 0.000000f, -0.0372341f, 0.970238f, 0.239275f, 
		-0.206815f, 0.970237f, 0.125966f, -0.842674f, 0.536976f, -0.0394590f, 
		-0.509649f, 0.814790f, -0.276359f, 0.0460251f, -0.996563f, -0.0688826f, 
		0.0460251f, -0.996563f, 0.0688817f, -0.0844393f, 0.989363f, 0.118449f, 
		-0.543176f, 0.731505f, 0.412140f, 0.294717f, 0.816369f, 0.496673f, 
		-0.836256f, 0.548258f, 0.00944074f, 0.138508f, 0.896293f, -0.421278f, 
		-0.365290f, 0.298061f, -0.881886f, -0.365293f, 0.298064f, 0.881884f, 
		0.00821920f, -0.999960f, -0.00340401f, 0.00821923f, -0.999960f, 0.00340439f, 
		-0.808573f, 0.582436f, 0.0835299f, -0.634423f, 0.625185f, 0.454588f, 
		0.00729628f, 0.941696f, 0.336386f, -0.380881f, 0.0969187f, -0.919531f, 
		-0.380886f, 0.0969175f, 0.919528f, -0.109137f, 0.841070f, -0.529802f, 
		-0.0952318f, 0.982882f, -0.157712f, 0.375787f, -0.188885f, 0.907252f, 
		0.375797f, -0.188888f, -0.907248f, -0.278694f, 0.685292f, -0.672833f, 
		-0.278700f, 0.685290f, 0.672832f, 0.00920347f, -0.999939f, -0.00614868f, 
		0.00920361f, -0.999939f, 0.00614921f, 0.384159f, -0.722404f, -0.574938f, 
		0.384164f, -0.722401f, 0.574940f, -0.0423388f, 0.993861f, -0.102214f, 
		-0.0423374f, 0.993861f, 0.102211f, -0.798812f, 0.600803f, 0.0305697f, 
		0.000000f, 1.00000f, 0.000000f, -0.206816f, 0.970237f, -0.125966f, 
		-0.125963f, 0.970237f, -0.206817f, 0.000000f, 1.00000f, 0.000000f, 
		-0.206815f, 0.970237f, 0.125966f, -0.125965f, 0.970237f, 0.206816f, 
		-0.236295f, 0.946759f, 0.218661f, 0.273781f, -0.698676f, 0.660981f, 
		0.273790f, -0.698678f, -0.660975f, 0.0345189f, -0.995923f, 0.0833375f, 
		0.0345198f, -0.995923f, -0.0833363f, 0.00889640f, -0.999960f, -2.92681e-007f, 
		-0.157303f, 0.987420f, 0.0160247f, -0.0163366f, 0.999088f, -0.0394399f, 
		-0.0163367f, 0.999088f, 0.0394400f, 0.281315f, 0.816389f, 0.504352f, 
		0.545571f, -0.188855f, -0.816509f, 0.545573f, -0.188855f, 0.816507f, 
		-0.00987124f, -0.999667f, -0.0238316f, -0.00987154f, -0.999667f, 0.0238323f, 
		0.0238402f, 0.998058f, 0.0575546f, 0.0238440f, 0.998057f, -0.0575625f, 
		-0.458118f, 0.888645f, 0.0209323f, 0.0724481f, -0.981916f, 0.174907f, 
		0.0724600f, -0.981911f, -0.174928f, 0.318315f, -0.555081f, 0.768480f, 
		0.382682f, -2.00236e-008f, 0.923880f, 0.382682f, -2.00236e-008f, 0.923880f, 
		0.382682f, -2.00236e-008f, 0.923880f, 0.289581f, 0.653746f, 0.699113f, 
		0.318327f, -0.555059f, -0.768490f, 0.382690f, 2.52929e-007f, -0.923877f, 
		0.382690f, 2.52929e-007f, -0.923877f, 0.382690f, 2.52929e-007f, -0.923877f, 
		0.289590f, 0.653739f, -0.699116f, -0.564902f, 0.730437f, 0.383859f, 
		0.0791844f, 0.996843f, -0.00574245f, 0.0595729f, -0.996445f, -0.0595736f, 
		0.0595728f, -0.996445f, 0.0595725f, -0.298238f, 0.806940f, 0.509806f, 
		-0.226853f, 0.571843f, 0.788374f, -0.203925f, 0.978536f, -0.0296846f, 
		0.214231f, 0.834201f, 0.508148f, 0.0108563f, -0.999939f, -0.00215971f, 
		0.0108563f, -0.999939f, 0.00215948f, -0.622923f, 0.680140f, 0.386493f, 
		0.503050f, 0.641237f, 0.579445f, -0.507682f, 0.861465f, 0.0116851f, 
		-0.503720f, 0.806027f, 0.310784f, -0.310523f, 0.938013f, 0.153972f, 
		0.545571f, -0.188855f, -0.816509f, 0.545573f, -0.188855f, 0.816507f, 
		0.0279155f, 0.962038f, -0.271484f, -0.484332f, 0.489914f, -0.724850f, 
		-0.206816f, 0.970237f, -0.125966f, -0.0372333f, 0.970238f, -0.239275f, 
		-0.484336f, 0.489910f, 0.724849f, -0.0372341f, 0.970238f, 0.239275f, 
		-0.206815f, 0.970237f, 0.125966f, 0.489271f, -0.721961f, -0.489271f, 
		0.489270f, -0.721959f, 0.489274f, 0.0771628f, 0.878530f, 0.471415f, 
		0.495596f, 0.620080f, 0.608182f, -0.587880f, 0.808918f, 0.00696152f, 
		0.000000f, 1.00000f, 0.000000f, -0.195521f, 0.970237f, 0.142868f, 
		-0.239278f, 0.970237f, 0.0372309f, 0.000000f, 1.00000f, 0.000000f, 
		-0.239275f, 0.970238f, -0.0372338f, -0.195522f, 0.970237f, -0.142867f, 
		0.289764f, 0.921770f, -0.257636f, 0.0243913f, 0.945491f, 0.324733f, 
		0.00961079f, 0.998194f, -0.0592971f, 0.694378f, -0.188890f, -0.694377f, 
		0.694377f, -0.188889f, 0.694379f, 0.417582f, 0.580073f, 0.699386f, 
		0.000000f, 1.00000f, 0.000000f, -0.239275f, 0.970238f, -0.0372338f, 
		-0.125963f, 0.970237f, -0.206817f, 0.000000f, 1.00000f, 0.000000f, 
		-0.125965f, 0.970237f, 0.206816f, -0.239278f, 0.970237f, 0.0372309f, 
		-0.105017f, 0.830787f, 0.546594f, 0.0688816f, -0.996563f, -0.0460255f, 
		0.0688817f, -0.996563f, 0.0460256f, -0.519433f, 0.806062f, 0.283642f, 
		-0.00633619f, 0.900013f, 0.435816f, -0.530083f, 0.299403f, -0.793328f, 
		-0.530088f, 0.299405f, 0.793324f, -0.337321f, 0.932155f, 0.131536f, 
		-0.629597f, 0.776775f, 0.0150974f, -0.552951f, 0.0969128f, -0.827558f, 
		-0.552956f, 0.0968995f, 0.827556f, -0.0655629f, 0.990346f, 0.122131f, 
		0.308268f, 0.570228f, 0.761454f, 0.000000f, 1.00000f, 0.000000f, 
		-0.235312f, 0.970237f, 0.0571686f, -0.235312f, 0.970237f, -0.0571693f, 
		0.622095f, 0.402477f, 0.671573f, 0.545573f, -0.188855f, 0.816507f, 
		0.545576f, -0.188853f, -0.816506f, -0.404639f, 0.685224f, 0.605587f, 
		-0.404633f, 0.685231f, -0.605583f, -0.0614784f, 0.993858f, -0.0920104f, 
		-0.0614748f, 0.993859f, 0.0920026f, 0.677330f, 0.418474f, 0.605065f, 
		0.0778354f, -0.996445f, -0.0322400f, 0.0778351f, -0.996445f, 0.0322405f, 
		0.664934f, 0.399554f, 0.631046f, 0.271807f, 0.583155f, 0.765540f, 
		0.397512f, -0.698610f, 0.594918f, 0.397512f, -0.698614f, -0.594914f, 
		0.574939f, -0.722404f, -0.384158f, 0.574937f, -0.722405f, 0.384159f, 
		-0.253538f, 0.860424f, 0.442029f, 0.267346f, 0.926129f, -0.266105f, 
		0.650738f, 0.350152f, 0.673746f, -0.639963f, 0.768180f, 0.0186446f, 
		0.0501234f, -0.995922f, 0.0750169f, 0.0501211f, -0.995922f, -0.0750116f, 
		0.707101f, 0.335741f, 0.622323f, -0.0237214f, 0.999088f, -0.0355018f, 
		-0.0237214f, 0.999088f, 0.0355018f, 0.816141f, 0.210727f, 0.538059f, 
		0.791320f, 0.147835f, 0.593260f, -0.0651736f, 0.991108f, -0.116004f, 
		-0.552654f, 0.810978f, 0.192065f, 0.894213f, 0.204836f, 0.398026f, 
		0.886595f, 0.198755f, 0.417667f, 0.945775f, 0.167759f, 0.278149f, 
		-0.0143337f, -0.999667f, -0.0214522f, -0.0143338f, -0.999667f, 0.0214519f, 
		-0.618015f, 0.485913f, -0.618017f, -0.206816f, 0.970237f, -0.125966f, 
		-0.125963f, 0.970237f, -0.206817f, -0.618017f, 0.485910f, 0.618017f, 
		-0.206815f, 0.970237f, 0.125966f, -0.125965f, 0.970237f, 0.206816f, 
		0.948413f, 0.0917253f, 0.303477f, 0.694378f, -0.188890f, -0.694377f, 
		0.694381f, -0.188886f, 0.694375f, 0.816510f, -0.188850f, -0.545570f, 
		0.816505f, -0.188854f, 0.545577f, 0.0346220f, 0.998056f, 0.0518141f, 
		0.0346230f, 0.998056f, -0.0518178f, 0.000000f, 1.00000f, 0.000000f, 
		-0.195521f, 0.970237f, 0.142868f, -0.235312f, 0.970237f, -0.0571693f, 
		0.000000f, 1.00000f, 0.000000f, -0.235312f, 0.970237f, 0.0571686f, 
		-0.195522f, 0.970237f, -0.142867f, 0.0842481f, -0.996445f, -5.95045e-008f, 
		0.105212f, -0.981905f, 0.157458f, 0.105215f, -0.981903f, -0.157468f, 
		0.0812523f, -0.996563f, -0.0161620f, 0.0812517f, -0.996563f, 0.0161621f, 
		0.462161f, -0.554988f, 0.691661f, 0.555577f, -5.51795e-008f, 0.831465f, 
		0.555577f, -5.51795e-008f, 0.831465f, 0.555577f, -5.51795e-008f, 0.831465f, 
		0.420450f, 0.653669f, 0.629236f, 0.462157f, -0.554980f, -0.691670f, 
		0.555568f, -1.44640e-007f, -0.831471f, 0.555568f, -1.44640e-007f, -0.831471f, 
		0.555568f, -1.44640e-007f, -0.831471f, 0.420445f, 0.653666f, -0.629243f, 
		0.639266f, -0.721959f, -0.264790f, 0.639265f, -0.721960f, 0.264791f, 
		-0.571722f, 0.815320f, 0.0915775f, 0.201600f, 0.932662f, -0.299164f, 
		0.0619053f, 0.952218f, 0.299079f, 0.340048f, 0.929180f, 0.144885f, 
		-0.0921942f, 0.986495f, -0.135382f, 0.383239f, 0.772290f, -0.506652f, 
		-0.574539f, 0.816560f, 0.0559967f, 0.678185f, -0.722403f, -0.134903f, 
		0.678184f, -0.722404f, 0.134903f, 0.907250f, -0.188888f, -0.375791f, 
		0.907249f, -0.188888f, 0.375792f, -0.674963f, 0.298066f, -0.674968f, 
		-0.674964f, 0.298068f, 0.674966f, 0.691934f, -0.721961f, -3.36015e-007f, 
		-0.703775f, 0.0969292f, -0.703779f, -0.703776f, 0.0969138f, 0.703781f, 
		0.694381f, -0.188886f, 0.694375f, 0.694378f, -0.188890f, -0.694377f, 
		-0.514964f, 0.685288f, 0.514968f, -0.514957f, 0.685301f, -0.514959f, 
		-0.219762f, 0.888218f, 0.403453f, -0.152371f, 0.960943f, 0.231024f, 
		0.131244f, 0.933683f, -0.333184f, 0.372704f, 0.784446f, -0.495717f, 
		-0.0782272f, 0.993862f, 0.0782275f, -0.0782283f, 0.993861f, -0.0782292f, 
		0.816510f, -0.188850f, -0.545570f, 0.816505f, -0.188854f, 0.545577f, 
		0.963135f, -0.188854f, -0.191585f, 0.963135f, -0.188854f, 0.191585f, 
		-0.724848f, 0.489916f, 0.484332f, -0.125965f, 0.970237f, 0.206816f, 
		-0.239278f, 0.970237f, 0.0372309f, -0.724851f, 0.489909f, -0.484334f, 
		-0.239275f, 0.970238f, -0.0372338f, -0.125963f, 0.970237f, -0.206817f, 
		-0.177284f, 0.958294f, 0.224150f, 0.505889f, -0.698680f, -0.505887f, 
		0.505889f, -0.698682f, 0.505885f, 0.981998f, -0.188889f, -4.77235e-007f, 
		0.0637789f, -0.995924f, -0.0637798f, 0.0637811f, -0.995924f, 0.0637816f, 
		-0.0301863f, 0.999088f, -0.0301871f, -0.0301864f, 0.999088f, 0.0301864f, 
		-0.224695f, 0.952220f, 0.206857f, 0.317613f, 0.803584f, -0.503363f, 
		0.354218f, 0.927380f, 0.120403f, -0.0182400f, -0.999667f, -0.0182397f, 
		-0.0182401f, -0.999667f, 0.0182400f, 0.104457f, 0.933532f, -0.342939f, 
		0.0440591f, 0.998057f, 0.0440584f, 0.0440561f, 0.998057f, -0.0440561f, 
		-0.807477f, 0.485916f, 0.334464f, -0.195521f, 0.970237f, 0.142868f, 
		-0.239278f, 0.970237f, 0.0372309f, -0.807481f, 0.485905f, -0.334470f, 
		-0.239275f, 0.970238f, -0.0372338f, -0.195522f, 0.970237f, -0.142867f, 
		0.133877f, -0.981913f, -0.133877f, 0.133900f, -0.981907f, 0.133901f, 
		-0.140289f, 0.967262f, 0.211478f, 0.588187f, -0.555041f, 0.588189f, 
		0.707106f, -1.26305e-007f, 0.707108f, 0.707106f, -1.26305e-007f, 0.707108f, 
		0.707107f, -2.10978e-009f, 0.707107f, 0.535084f, 0.653736f, 0.535083f, 
		0.588174f, -0.555069f, -0.588175f, 0.707107f, -5.19338e-007f, -0.707107f, 
		0.707106f, -3.33137e-007f, -0.707108f, 0.707106f, -5.37105e-007f, -0.707108f, 
		0.535081f, 0.653739f, -0.535083f, -0.265988f, 0.945491f, 0.187876f, 
		0.370924f, 0.550301f, -0.748053f, 0.101110f, 0.958293f, 0.267303f, 
		0.907250f, -0.188888f, -0.375791f, 0.907246f, -0.188887f, 0.375800f, 
		0.269365f, 0.815005f, -0.513039f, -0.793325f, 0.299406f, -0.530086f, 
		-0.793325f, 0.299409f, 0.530084f, 0.427636f, 0.590823f, -0.684146f, 
		-0.827557f, 0.0969020f, -0.552955f, -0.827560f, 0.0968881f, 0.552952f, 
		0.816505f, -0.188854f, 0.545577f, 0.816510f, -0.188850f, -0.545570f, 
		-0.605592f, 0.685219f, 0.404639f, -0.605586f, 0.685225f, -0.404638f, 
		0.399692f, 0.628769f, -0.667006f, -0.0920016f, 0.993860f, -0.0614732f, 
		-0.0919995f, 0.993860f, 0.0614731f, 0.266286f, 0.820940f, -0.505124f, 
		-0.284771f, 0.941696f, 0.179202f, 0.963135f, -0.188854f, -0.191585f, 
		0.963136f, -0.188854f, 0.191582f, -0.874008f, 0.485912f, 1.26752e-007f, 
		-0.235312f, 0.970237f, 0.0571686f, -0.235312f, 0.970237f, -0.0571693f, 
		-0.855020f, 0.489915f, 0.170071f, -0.195521f, 0.970237f, 0.142868f, 
		-0.235312f, 0.970237f, -0.0571693f, -0.855021f, 0.489911f, -0.170076f, 
		-0.235312f, 0.970237f, 0.0571686f, -0.195522f, 0.970237f, -0.142867f, 
		-0.149136f, 0.971392f, 0.184811f, 0.594919f, -0.698611f, -0.397510f, 
		0.594914f, -0.698613f, 0.397513f, 0.421248f, 0.628819f, -0.653557f, 
		0.520195f, 0.385091f, -0.762301f, 0.981998f, -0.188889f, -4.77235e-007f, 
		0.496296f, 0.312872f, -0.809815f, 0.0750125f, -0.995922f, -0.0501222f, 
		0.0750132f, -0.995922f, 0.0501227f, 0.367249f, 0.927535f, 0.0693297f, 
		0.551170f, 0.370873f, -0.747439f, -0.0355023f, 0.999088f, 0.0237217f, 
		-0.0355016f, 0.999088f, -0.0237219f, 0.522194f, 0.622234f, -0.583214f, 
		0.562589f, 0.403545f, -0.721557f, 0.639356f, 0.217615f, -0.737474f, 
		-0.881884f, 0.298065f, -0.365291f, -0.881887f, 0.298064f, 0.365284f, 
		-0.0214517f, -0.999667f, 0.0143340f, -0.0214518f, -0.999667f, -0.0143334f, 
		0.666998f, 0.343839f, -0.660975f, -0.919533f, 0.0969094f, 0.380878f, 
		-0.919529f, 0.0969145f, -0.380886f, -0.175963f, 0.979623f, 0.0968255f, 
		0.907249f, -0.188887f, -0.375794f, 0.907246f, -0.188887f, 0.375800f, 
		-0.672831f, 0.685294f, 0.278694f, -0.672835f, 0.685288f, -0.278700f, 
		0.0518091f, 0.998057f, -0.0346171f, 0.0518159f, 0.998056f, 0.0346222f, 
		0.157444f, -0.981909f, -0.105199f, 0.157468f, -0.981903f, 0.105219f, 
		0.589805f, 0.805014f, 0.0638902f, -0.102208f, 0.993862f, -0.0423360f, 
		-0.102203f, 0.993862f, 0.0423342f, 0.691667f, -0.554979f, 0.462163f, 
		0.831466f, -1.93635e-007f, 0.555575f, 0.831466f, -1.93635e-007f, 0.555575f, 
		0.831466f, -1.93635e-007f, 0.555575f, 0.629240f, 0.653667f, 0.420448f, 
		0.691656f, -0.555005f, -0.462149f, 0.831469f, -6.31755e-008f, -0.555571f, 
		0.831469f, -6.31755e-008f, -0.555571f, 0.831469f, -6.31755e-008f, -0.555571f, 
		0.629236f, 0.653674f, -0.420442f, 0.119805f, 0.960946f, 0.249460f, 
		-0.935793f, 0.299406f, 0.186139f, -0.935791f, 0.299408f, -0.186143f, 
		0.660977f, -0.698677f, -0.273788f, 0.660975f, -0.698677f, 0.273791f, 
		0.0833362f, -0.995923f, -0.0345195f, 0.0833363f, -0.995923f, 0.0345202f, 
		-0.976170f, 0.0968968f, 0.194174f, -0.976170f, 0.0968951f, -0.194175f, 
		-0.203948f, 0.978981f, 0.00133203f, -0.954545f, 0.298067f, 6.25785e-007f, 
		0.963135f, -0.188854f, -0.191585f, 0.963136f, -0.188854f, 0.191582f, 
		-0.714333f, 0.685229f, 0.142091f, -0.714339f, 0.685222f, -0.142093f, 
		-0.0394405f, 0.999088f, 0.0163366f, -0.0394394f, 0.999088f, -0.0163363f, 
		0.167527f, 0.933512f, 0.317003f, -0.995293f, 0.0969132f, 8.87209e-007f, 
		-0.108519f, 0.993860f, 0.0215857f, -0.108521f, 0.993860f, -0.0215869f, 
		0.362096f, 0.931623f, 0.0310497f, 0.981998f, -0.188889f, -4.77235e-007f, 
		-0.728267f, 0.685294f, 8.71943e-008f, -0.0238312f, -0.999667f, 0.00987191f, 
		-0.0238315f, -0.999667f, -0.00987155f, 0.596982f, 0.802038f, 0.0186525f, 
		-0.110625f, 0.993862f, -2.71383e-007f, 0.701754f, -0.698610f, -0.139590f, 
		0.701755f, -0.698609f, 0.139589f, -0.214302f, 0.976222f, -0.0326282f, 
		0.159535f, 0.931622f, 0.326541f, 0.0575529f, 0.998058f, -0.0238390f, 
		0.0575597f, 0.998057f, 0.0238425f, 0.0884879f, -0.995922f, -0.0176015f, 
		0.0884889f, -0.995922f, 0.0176017f, 0.174920f, -0.981913f, 0.0724562f, 
		0.174917f, -0.981914f, -0.0724535f, 0.715439f, -0.698676f, -3.96769e-007f, 
		-0.0418780f, 0.999088f, 0.00832985f, -0.0418775f, 0.999088f, -0.00832998f, 
		0.768485f, -0.555067f, 0.318325f, 0.923876f, 2.05384e-007f, 0.382692f, 
		0.923876f, 2.05384e-007f, 0.382692f, 0.923876f, 2.05384e-007f, 0.382692f, 
		0.699114f, 0.653741f, 0.289590f, 0.768486f, -0.555070f, -0.318318f, 
		0.923879f, 3.72371e-007f, -0.382684f, 0.923879f, 3.72371e-007f, -0.382684f, 
		0.923879f, 3.72371e-007f, -0.382684f, 0.699112f, 0.653747f, -0.289581f, 
		0.0902074f, -0.995923f, 1.56330e-007f, 0.129349f, 0.927536f, 0.350636f, 
		-0.0426908f, 0.999088f, -2.17849e-007f, -0.0253044f, -0.999667f, 0.00503330f, 
		-0.0253046f, -0.999667f, -0.00503340f, -0.0257957f, -0.999667f, 2.50889e-008f, 
		0.578905f, 0.811254f, -0.0820732f, 0.0611093f, 0.998057f, -0.0121551f, 
		0.0611119f, 0.998057f, 0.0121558f, 0.185726f, -0.981907f, -0.0369433f, 
		0.185722f, -0.981908f, 0.0369426f, 0.815868f, -0.554998f, -0.162284f, 
		0.815866f, -0.555000f, 0.162287f, 0.980786f, 1.66092e-007f, -0.195087f, 
		0.980785f, 2.53355e-007f, 0.195092f, 0.980786f, 1.66092e-007f, -0.195087f, 
		0.980785f, 2.53355e-007f, 0.195092f, 0.980786f, 1.66092e-007f, -0.195087f, 
		0.980785f, 2.53355e-007f, 0.195092f, 0.742233f, 0.653677f, -0.147636f, 
		0.742234f, 0.653675f, 0.147640f, 0.0622952f, 0.998058f, -2.96615e-007f, 
		0.0788403f, 0.927380f, 0.365719f, 0.358018f, 0.933516f, 0.0192854f, 
		0.189327f, -0.981914f, -2.47472e-007f, 0.762493f, 0.637699f, -0.109290f, 
		0.831802f, -0.555072f, -2.91661e-007f, 1.00000f, -1.53562e-007f, -9.35917e-008f, 
		1.00000f, -1.53562e-007f, -9.35917e-008f, 1.00000f, -1.53562e-007f, -9.35917e-008f, 
		0.756714f, 0.653747f, -1.39175e-007f, 0.527468f, 0.834776f, -0.157884f, 
		0.253203f, 0.962877f, 0.0935796f, 0.0505573f, 0.929181f, 0.366152f, 
		0.757257f, 0.639335f, -0.133467f, 0.243691f, 0.967445f, 0.0682986f, 
		0.501605f, 0.846504f, -0.178391f, 0.686106f, 0.465635f, -0.558966f, 
		0.700344f, 0.672428f, -0.239496f, 0.410898f, 0.846503f, 0.338520f, 
		0.214018f, 0.976725f, -0.0142959f, 0.406604f, 0.834774f, 0.371248f, 
		0.610271f, 0.718691f, -0.333245f, 0.725171f, 0.480300f, -0.493396f, 
		0.180553f, 0.978187f, -0.102716f, 0.410532f, 0.283882f, -0.866530f, 
		0.368026f, 0.811254f, 0.454338f, 0.577792f, 0.738855f, -0.346769f, 
		0.679697f, 0.525850f, -0.511365f, 0.514523f, 0.328896f, -0.791893f, 
		0.167654f, 0.976721f, -0.133820f, 0.445173f, 0.211682f, -0.870064f, 
		0.654013f, 0.542888f, -0.526821f, 0.533191f, 0.324334f, -0.781354f, 
		0.290928f, 0.802039f, 0.521626f, 0.637531f, 0.342818f, -0.689949f, 
		0.711750f, 0.529207f, -0.461900f, 0.695120f, 0.284320f, -0.660281f, 
		0.248438f, 0.805013f, 0.538732f, 0.594487f, 0.738865f, 0.317275f, 
		0.548036f, 0.835534f, -0.0392328f, 0.599582f, 0.718693f, 0.352110f, 
		0.534681f, 0.843049f, -0.0581740f, 0.565485f, 0.672426f, 0.477566f, 
		0.503811f, 0.639338f, 0.580880f, 0.498674f, 0.858325f, -0.120840f, 
		0.485759f, 0.637699f, 0.597811f, 0.458311f, 0.868915f, -0.186917f, 
		0.441156f, 0.872661f, -0.209388f, 0.832165f, 0.465641f, 0.301129f, 
		0.762111f, 0.529212f, 0.372991f, 0.796029f, 0.480303f, 0.368302f, 
		0.788105f, 0.542887f, 0.290112f, 0.788064f, 0.525846f, 0.320063f, 
		0.954299f, 0.283895f, -0.0933659f, 0.943747f, 0.328900f, 0.0341655f, 
		0.919557f, 0.342799f, 0.192103f, 0.944312f, 0.324322f, 0.0555843f, 
		0.923720f, 0.284294f, 0.256748f, 0.728473f, 0.653693f, -0.204971f, 
		0.975144f, 0.211665f, -0.0655060f, 0.716259f, 0.668281f, -0.200933f, 
		0.678115f, 0.696017f, -0.236050f, 0.645267f, 0.714841f, -0.269506f, 
		0.639654f, 0.721802f, -0.264281f, 0.803530f, 0.419498f, -0.422327f, 
		0.824647f, 0.454342f, -0.336943f, 0.799536f, 0.497750f, -0.336135f, 
		0.810377f, 0.495090f, -0.313329f, 0.856031f, 0.476498f, -0.200403f, 
		0.895717f, 0.159969f, -0.414851f, 0.904112f, 0.230367f, -0.359878f, 
		0.919379f, 0.212189f, -0.331239f, 0.958744f, 0.0471796f, -0.280328f, 
		0.921373f, 0.244267f, -0.302334f, 0.965107f, 0.172162f, -0.197302f
	};
	static const GLfloat TexCoordData[] = {
		0.752582f, 1.14465f, 0.752689f, 1.08990f, 0.779817f, 1.08770f, 
		0.720234f, 1.06974f, 0.770994f, 1.06728f, 
		0.547135f, -0.662492f, 0.731272f, 1.05530f, 
		0.551940f, -0.626227f, 0.567953f, -0.737880f, 
		0.517436f, -0.497404f, 0.565102f, -0.656017f, 
		0.576754f, -0.644823f, 0.584356f, -0.690386f, 
		0.538155f, -0.497993f, 0.561019f, -0.514697f, 
		0.749330f, 0.994512f, 0.718786f, 0.967398f, 
		0.583618f, -0.520355f, 0.474394f, -0.275800f, 
		0.689350f, 0.940091f, 0.776127f, 0.982417f, 
		0.605260f, -0.539803f, 0.799067f, 0.968631f, 
		0.515020f, -0.286387f, 0.555981f, -0.295446f, 
		0.666508f, 0.832853f, 0.705192f, 0.858981f, 
		0.440889f, -0.0482150f, 0.597614f, -0.301448f, 
		0.742153f, 0.875088f, 0.639583f, -0.305923f, 
		0.775668f, 0.871156f, 0.495679f, -0.0551170f, 
		0.651192f, 0.740125f, 0.807461f, 0.857204f, 
		0.551349f, -0.0599200f, 0.694737f, 0.746483f, 
		0.439804f, 0.134816f, 0.608777f, -0.0605230f, 
		0.147088f, 1.09010f, 0.737073f, 0.751057f, 
		0.124982f, 1.05296f, 0.642889f, 0.654007f, 
		0.147058f, 1.04246f, 0.132073f, 1.03100f, 
		0.176809f, 1.01407f, 0.166624f, 1.00508f, 
		0.493279f, 0.132106f, 0.667084f, -0.0590260f, 
		0.776992f, 0.752062f, 0.151409f, 0.960342f, 
		0.127595f, 0.958866f, 0.107867f, 0.955318f, 
		0.180591f, 0.928755f, 0.209539f, 0.898337f, 
		0.691669f, 0.646340f, 0.548482f, 0.130225f, 
		0.815703f, 0.751284f, 0.199146f, 0.841397f, 
		0.159938f, 0.858097f, 0.240672f, 0.816373f, 
		0.125370f, 0.858149f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.0931210f, 0.849876f, 0.827969f, 0.0466290f, 
		0.454659f, 0.288361f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.740009f, 0.641847f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.607139f, 0.130003f, 
		0.640277f, 0.567616f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.265844f, 0.743452f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.214376f, 0.747195f, 
		0.500405f, 0.288302f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.164659f, 0.749349f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.787472f, 0.643703f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.667523f, 0.130611f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.118445f, 0.748324f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.693073f, 0.555590f, 
		0.0739820f, 0.745710f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.548284f, 0.287761f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.280690f, 0.654845f, 0.834497f, 0.648733f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.218369f, 0.650338f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.468978f, 0.427488f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.157583f, 0.647717f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.745944f, 0.548273f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.0998670f, 0.648866f, 0.600427f, 0.286254f, 
		0.416452f, 0.829188f, 0.0436860f, 0.651900f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.420618f, 0.768336f, 0.642034f, 0.474070f, 
		0.407040f, 0.830610f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.411470f, 0.879107f, 
		0.408041f, 0.868347f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.401650f, 0.873908f, 0.426951f, 0.704266f, 
		0.509641f, 0.426493f, 0.395285f, 0.831500f, 
		0.402075f, 0.772613f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.402495f, 0.897040f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.391297f, 0.856897f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.386353f, 0.861931f, 
		0.379336f, 0.817521f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.381843f, 0.773114f, 
		0.438437f, 0.644695f, 0.827969f, 0.0466290f, 
		0.287952f, 0.553867f, 0.798967f, 0.550376f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.398487f, 0.704695f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.363137f, 0.804440f, 
		0.218406f, 0.549738f, 0.358236f, 0.766059f, 
		0.696036f, 0.471271f, 0.654704f, 0.284265f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.368907f, 0.704329f, 0.150301f, 0.548248f, 
		0.446847f, 0.583262f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.332940f, 0.755227f, 
		0.551654f, 0.425460f, 0.401620f, 0.637192f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0850800f, 0.552041f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.0213000f, 0.558474f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.337094f, 0.702375f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.852065f, 0.557189f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.481703f, 0.564697f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.364058f, 0.630726f, 0.749861f, 0.471148f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.405129f, 0.569155f, 
		0.304164f, 0.699627f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.443950f, 0.513611f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.878721f, 0.490563f, 0.878818f, 0.399245f, 
		0.596366f, 0.424355f, 0.518075f, 0.560475f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.290372f, 0.443834f, 0.325007f, 0.626333f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.221769f, 0.444303f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.868548f, 0.591167f, 
		0.154407f, 0.445990f, 0.362947f, 0.556722f, 
		0.0895280f, 0.450116f, 0.870169f, 0.306122f, 
		0.651137f, 0.370508f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.803333f, 0.476378f, 
		0.402672f, 0.499632f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0258910f, 0.455460f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.285212f, 0.622976f, 0.703026f, 0.379702f, 
		0.852565f, 0.698515f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.554745f, 0.557827f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0329870f, 0.578459f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.438319f, 0.437071f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0347470f, 0.481456f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.319837f, 0.547637f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.754378f, 0.388511f, 
		0.642428f, 0.423213f, 0.361226f, 0.486732f, 
		0.491776f, 0.712487f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.160062f, 0.331157f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.0358120f, 0.666865f, 0.221822f, 0.326417f, 
		0.854104f, 0.200105f, 0.522794f, 0.704049f, 
		0.399654f, 0.426432f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.100109f, 0.334439f, 
		0.0444050f, 0.372122f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.856630f, 0.484286f, 
		0.284486f, 0.320948f, 0.835039f, 0.810066f, 
		0.826125f, 0.497932f, 0.276263f, 0.540226f, 
		0.592012f, 0.558326f, 0.716512f, 0.267201f, 
		0.0410590f, 0.336991f, 0.823697f, 0.404190f, 
		0.819843f, 0.600119f, 0.553710f, 0.699365f, 
		0.157432f, 0.193963f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.438525f, 0.354976f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.319444f, 0.475990f, 0.762114f, 0.278401f, 
		0.361081f, 0.415644f, 0.670560f, 0.254065f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.211932f, 0.188465f, 0.0652720f, 0.246721f, 
		0.151572f, 0.0717290f, 0.0407880f, 0.754318f, 
		0.804655f, 0.396549f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.401482f, 0.347366f, 
		0.103722f, 0.196358f, 0.808702f, 0.709385f, 
		0.815801f, 0.309955f, 0.147541f, 0.00177700f, 
		0.144424f, -0.0420340f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.199329f, 0.0666650f, 
		0.837683f, 0.105503f, 0.526573f, 0.864286f, 
		0.191244f, -0.00276300f, 0.734390f, 0.158995f, 
		0.185419f, -0.0460040f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.772449f, 0.168806f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.103731f, 0.0734900f, 0.266827f, 0.181415f, 
		0.0887650f, 0.133000f, 0.551538f, 0.858289f, 
		0.364701f, 0.338346f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.103153f, -0.0401740f, 
		0.103502f, 0.00345100f, 0.811184f, 0.928574f, 
		0.106299f, 0.0587040f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.501535f, 0.875601f, 
		0.440729f, 0.271273f, 0.322693f, 0.404558f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.405440f, 0.266286f, 0.796554f, 0.824366f, 
		0.277496f, 0.466327f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.805680f, 0.206287f, 
		0.0504070f, 0.197202f, 0.123048f, 0.0188490f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.584422f, 0.702187f, 
		0.370383f, 0.259101f, 0.218187f, -0.0391990f, 
		0.696311f, 0.146774f, 0.815655f, 0.0177070f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.247043f, 0.0599490f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.754557f, 0.0803100f, 
		0.807016f, 0.285726f, 0.629578f, 0.560399f, 
		0.408815f, 0.187044f, 0.0454790f, 0.848462f, 
		0.784764f, 0.0877170f, 0.234779f, -0.00873500f, 
		0.532188f, 1.04826f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.551220f, 1.04282f, 
		0.0951490f, 0.566359f, 0.441093f, 0.189913f, 
		0.0942310f, 0.473688f, 0.328446f, 0.326506f, 
		0.376425f, 0.182171f, 0.776214f, 1.05680f, 
		0.783046f, 0.933384f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0714830f, -0.0288050f, 
		0.144184f, 0.00845000f, 0.529839f, 0.0371510f, 
		0.724397f, 0.0706640f, 0.0983400f, 0.370062f, 
		0.796878f, 0.114381f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0558480f, 0.0736000f, 
		0.0966420f, 0.655077f, 0.751892f, 0.0268350f, 
		0.411898f, 0.108722f, 0.433516f, 0.129521f, 
		0.513320f, 1.05878f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.518891f, 0.113183f, 
		0.767829f, 1.02476f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.382646f, 0.106400f, 
		0.854396f, 0.404202f, 0.775869f, 0.0163850f, 
		0.576357f, 0.862930f, 0.284397f, 0.393323f, 
		0.0592940f, 0.00369200f, 0.335790f, 0.247521f, 
		0.137219f, 0.0631830f, 0.111928f, 0.248478f, 
		0.792940f, 0.0554340f, 0.791434f, 0.0129650f, 
		0.810467f, 0.173797f, 0.127414f, 0.136873f, 
		0.739627f, 1.13253f, 0.538722f, 1.24409f, 
		0.468271f, 0.0471460f, 0.526070f, 1.20172f, 
		0.543364f, 0.0145880f, 0.455805f, 0.107902f, 
		0.0994810f, 0.752375f, 0.343810f, 0.173291f, 
		0.744720f, 1.05615f, 0.551829f, 1.24014f, 
		0.769242f, 0.609296f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.205561f, 0.0939350f, 
		0.798067f, 0.0208800f, 0.771183f, 0.506473f, 
		0.764057f, 0.719662f, 0.751224f, 0.953891f, 
		0.757861f, 0.838127f, 0.194239f, 0.0314640f, 
		0.570580f, 1.04754f, 0.352168f, 0.101438f, 
		0.292454f, 0.313255f, 0.767651f, 0.410637f, 
		0.583769f, 0.0369640f, 0.0679150f, 0.968683f, 
		0.815061f, 0.0906480f, 0.441618f, 0.206296f, 
		0.570950f, 0.0145880f, 0.506682f, 0.214308f, 
		0.487194f, 0.0175500f, 0.170644f, 0.0112940f, 
		0.557668f, 0.0166730f, 0.724302f, 1.04928f, 
		0.761782f, 0.314627f, 0.104437f, 0.870788f, 
		0.211111f, 0.187074f, 0.596683f, 0.112435f, 
		0.756717f, 0.211284f, 0.710923f, 1.09357f, 
		0.754143f, 0.120126f, 0.615032f, 0.708764f, 
		0.755751f, 0.0606740f, 0.301429f, 0.233743f, 
		0.759449f, 0.0170740f, 0.538681f, 0.118333f, 
		0.565946f, 1.24409f, 0.169900f, 0.0650800f, 
		0.126124f, 1.13436f, 0.326351f, 0.113539f, 
		0.720069f, 0.964513f, 0.311083f, 0.162406f, 
		0.820227f, 0.0218900f, 0.116596f, 0.994435f, 
		0.0854500f, 0.0374860f, 0.851567f, 0.291115f, 
		0.0722690f, 0.0892490f, 0.173049f, 0.0215020f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.577577f, 0.117959f, 0.141041f, 1.10743f, 
		0.344484f, 0.761948f, 0.166649f, 0.137660f, 
		0.344697f, 0.706478f, 0.343883f, 0.511998f, 
		0.346564f, 0.471689f, 0.601104f, 0.872890f, 
		0.186670f, 1.23045f, 0.558300f, 0.120814f, 
		0.348135f, 0.445578f, 0.427467f, 0.318292f, 
		0.609601f, 0.212871f, 0.348902f, 0.790970f, 
		0.347129f, 0.617630f, 0.173623f, 0.102388f, 
		0.579295f, 1.20136f, 0.340862f, 0.557638f, 
		0.158385f, 0.247828f, 0.714242f, 0.0587830f, 
		0.178793f, 1.14906f, 0.718753f, 0.850808f, 
		0.723843f, 0.0147030f, 0.112676f, 0.0272580f, 
		0.690670f, 0.987767f, 0.214990f, 0.289035f, 
		0.532277f, 0.219513f, 0.489371f, 0.312151f, 
		0.590103f, 1.05736f, 0.353234f, 0.432352f, 
		0.512558f, 0.0568260f, 0.520755f, 0.0126370f, 
		0.707550f, 0.119605f, 0.203366f, 0.0111640f, 
		0.837591f, 0.0388860f, 0.165877f, 1.02182f, 
		0.503106f, 0.118468f, 0.848463f, 0.176377f, 
		0.158388f, 0.881155f, 0.151556f, 0.367124f, 
		0.144893f, 0.0310880f, 0.155305f, 0.748566f, 
		0.0628920f, 0.169528f, 0.350062f, 0.524283f, 
		0.845404f, 0.0913410f, 0.201037f, 1.11244f, 
		0.175465f, 0.188708f, 0.106977f, 0.100045f, 
		0.216117f, 1.14189f, 0.491142f, 0.215017f, 
		0.583737f, 0.218794f, 0.206103f, 0.0618090f, 
		0.152608f, 0.467085f, 0.360826f, 0.760246f, 
		0.155607f, 0.645573f, 0.360362f, 0.800472f, 
		0.717848f, 0.728751f, 0.362154f, 0.704576f, 
		0.691818f, 0.0215550f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.706675f, 0.213909f, 
		0.155854f, 0.556854f, 0.557939f, 0.221936f, 
		0.338266f, 0.599745f, 0.140992f, 0.106028f, 
		0.363806f, 0.513713f, 0.365198f, 0.474641f, 
		0.365212f, 0.618021f, 0.365996f, 0.449250f, 
		0.670573f, 0.0533280f, 0.366498f, 0.430700f, 
		0.201063f, 0.998651f, 0.351779f, 0.455312f, 
		0.408873f, 0.773156f, 0.648032f, 0.0655860f, 
		0.402030f, 0.800227f, 0.714848f, 0.618926f, 
		0.207057f, 0.132271f, 0.362749f, 0.558040f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.635466f, 0.0327710f, 0.101355f, 0.179935f, 
		0.522432f, 0.315563f, 0.368300f, 0.526742f, 
		0.375259f, 0.801635f, 0.377045f, 0.760796f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.389987f, 0.805636f, 0.413437f, 0.718388f, 
		0.393020f, 0.765850f, 0.659026f, 0.115950f, 
		0.179845f, 0.285898f, 0.561861f, 0.0184730f, 
		0.708462f, 0.320975f, 0.711550f, 0.517361f, 
		0.379475f, 0.705114f, 0.231467f, 0.0215540f, 
		0.557173f, 0.0639710f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.139115f, 0.187332f, 
		0.679438f, 0.862950f, 0.603420f, 0.0211260f, 
		0.396524f, 0.710531f, 0.709754f, 0.420085f, 
		0.602439f, 0.0660460f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.479793f, 0.323897f, 
		0.550777f, 0.126206f, 0.647972f, 0.127541f, 
		0.202324f, 0.867603f, 0.383193f, 0.619749f, 
		0.229350f, 0.976490f, 0.415108f, 0.419856f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.382948f, 0.475759f, 0.622434f, 0.310141f, 
		0.382438f, 0.450880f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.369718f, 0.456900f, 
		0.381873f, 0.432769f, 0.383349f, 0.514175f, 
		0.599189f, 0.128288f, 0.354789f, 0.401606f, 
		0.204448f, 0.242365f, 0.220123f, 0.388956f, 
		0.400969f, 0.624148f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.408222f, 0.443927f, 
		0.386606f, 0.530031f, 0.0470800f, 0.265304f, 
		0.396040f, 0.448425f, 0.244066f, 0.0559560f, 
		0.540968f, 0.221916f, 0.362955f, 0.599192f, 
		0.555595f, 0.317018f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.588963f, 0.314558f, 
		0.398931f, 0.473214f, 0.401762f, 0.431939f, 
		0.393304f, 0.430700f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.418642f, 0.629883f, 
		0.414030f, 0.468836f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.338111f, 0.640661f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.384439f, 0.557868f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.472102f, 0.406678f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.141283f, 0.281363f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0958900f, 0.274033f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.205392f, 0.741024f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.402131f, 0.512131f, 
		0.388120f, 0.461295f, 0.371691f, 0.400388f, 
		0.656093f, 0.215348f, 0.405046f, 0.534979f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.248052f, 0.123795f, 0.591401f, 0.225172f, 
		0.203332f, 0.362433f, 0.420532f, 0.508834f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.185552f, 0.383396f, 0.670858f, 0.737246f, 
		0.241252f, 0.842091f, 0.361601f, 0.354055f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.407445f, 0.471303f, 0.423553f, 0.540756f, 
		0.405734f, 0.556550f, 0.389364f, 0.403596f, 
		0.642138f, 0.226607f, 0.532244f, 0.329344f, 
		0.387019f, 0.598577f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.513442f, 0.407724f, 
		0.210842f, 0.640638f, 0.364161f, 0.639756f, 
		0.208769f, 0.462813f, 0.472185f, 0.422534f, 
		0.427234f, 0.484119f, 0.376444f, 0.349102f, 
		0.408580f, 0.415656f, 0.144414f, 0.377610f, 
		0.213644f, 0.552538f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.655490f, 0.328163f, 
		0.426832f, 0.554658f, 0.250313f, 0.234496f, 
		0.658557f, 0.628781f, 0.554476f, 0.407965f, 
		0.342408f, 0.682727f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0901410f, 0.371368f, 
		0.428566f, 0.432142f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.391969f, 0.346993f, 
		0.409837f, 0.597840f, 0.368174f, 0.303289f, 
		0.0293020f, 0.364899f, 0.252611f, 0.731616f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.594896f, 0.406596f, 0.584948f, 0.334475f, 
		0.412496f, 0.520595f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.650932f, 0.431034f, 
		0.649571f, 0.529421f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.635009f, 0.404421f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.389182f, 0.639007f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.408860f, 0.350572f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.379943f, 0.295308f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.229437f, 0.475971f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.432031f, 0.597041f, 
		0.366108f, 0.682319f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.529102f, 0.426733f, 
		0.254390f, 0.356864f, 0.426435f, 0.356995f, 
		0.392275f, 0.288704f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.637777f, 0.339449f, 
		0.370467f, 0.239939f, 0.191377f, 0.470641f, 
		0.460015f, 0.497855f, 0.412143f, 0.638569f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.264212f, 0.637988f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.405730f, 0.284853f, 
		0.378158f, 0.231271f, 0.145421f, 0.465559f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.350494f, 0.725356f, 0.471614f, 0.519881f, 
		0.389021f, 0.681861f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.386622f, 0.225944f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.263823f, 0.459706f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.509605f, 0.497235f, 
		0.419747f, 0.282378f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.434075f, 0.638286f, 
		0.0836720f, 0.460973f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.269976f, 0.550816f, 
		0.396631f, 0.227297f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.370886f, 0.175888f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.374780f, 0.172204f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.379657f, 0.168468f, 0.407412f, 0.231992f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.585700f, 0.433972f, 0.369287f, 0.725297f, 
		0.386610f, 0.179019f, 0.0140250f, 0.456634f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.393643f, 0.188541f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.373503f, 0.133322f, 
		0.371835f, 0.123018f, 0.410358f, 0.681304f, 
		0.557793f, 0.496679f, 0.380715f, 0.141130f, 
		0.376027f, 0.126030f, 0.382652f, 0.134730f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.375147f, 0.102307f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.603177f, 0.496252f, 0.387747f, 0.724932f, 
		0.530208f, 0.521544f, 0.427585f, 0.630117f, 
		0.430908f, 0.680696f, 0.361701f, 0.767962f, 
		0.647159f, 0.495888f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.642138f, 0.442732f, 0.238123f, 0.567687f, 
		0.405541f, 0.723959f, 0.374188f, 0.767106f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.194386f, 0.562713f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.143239f, 0.558431f, 0.386573f, 0.766014f, 
		0.481374f, 0.624888f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.423002f, 0.722681f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.452993f, 0.595515f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.0772740f, 0.555532f, 
		0.372390f, 0.804119f, 0.587751f, 0.527964f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.398753f, 0.764450f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.378795f, 0.801018f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.385513f, 0.800053f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.410830f, 0.762650f, 
		0.00389900f, 0.553326f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.509161f, 0.591436f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.392287f, 0.797373f, 
		0.381089f, 0.820305f, 0.827969f, 0.0466290f, 
		0.378917f, 0.827403f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.534224f, 0.621239f, 
		0.384581f, 0.821996f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.398456f, 0.797201f, 
		0.388440f, 0.817323f, 0.383897f, 0.834709f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.389944f, 0.822931f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.562578f, 0.589049f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.448229f, 0.729222f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.644768f, 0.536762f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.610494f, 0.590048f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.241373f, 0.681709f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.585192f, 0.620752f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.495116f, 0.731956f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.655660f, 0.592740f, 
		0.191643f, 0.674693f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.136803f, 0.669445f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.540293f, 0.732136f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.0717410f, 0.667733f, 0.635219f, 0.621845f, 
		0.827969f, 0.0466290f, 0.462285f, 0.798707f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.827969f, 0.0466290f, 
		0.827969f, 0.0466290f, 0.582050f, 0.727207f, 
		0.450915f, 0.709492f, 0.00156900f, 0.667789f, 
		0.506490f, 0.835484f, 0.510348f, 0.697667f, 
		0.622096f, 0.719725f, 0.480564f, 0.888770f, 
		0.547558f, 0.860552f, 0.237304f, 0.805971f, 
		0.565861f, 0.690965f, 0.186286f, 0.792074f, 
		0.582351f, 0.862204f, 0.517302f, 0.937519f, 
		0.613535f, 0.694508f, 0.513877f, 1.04961f, 
		0.132150f, 0.782849f, 0.614007f, 0.852148f, 
		0.553279f, 0.988792f, 0.529356f, 1.04011f, 
		0.657289f, 0.703173f, 0.551215f, 1.16628f, 
		0.581396f, 0.993685f, 0.554715f, 1.09916f, 
		0.0717790f, 0.782967f, 0.574485f, 1.08959f, 
		0.603112f, 0.994917f, 0.581571f, 1.12384f, 
		0.00829100f, 0.787757f, 0.224034f, 0.928405f, 
		0.455296f, 0.831185f, 0.181451f, 0.900348f, 
		0.513054f, 0.812823f, 0.135321f, 0.882889f, 
		0.0820940f, 0.886628f, 0.566721f, 0.801645f, 
		0.0253190f, 0.900964f, 0.612208f, 0.804836f, 
		0.653604f, 0.815211f, 0.0535730f, 0.995655f, 
		0.202417f, 1.02697f, 0.0972590f, 0.970437f, 
		0.174986f, 0.986912f, 0.140321f, 0.967149f, 
		0.0939760f, 1.06008f, 0.111849f, 1.02612f, 
		0.164737f, 1.03917f, 0.141159f, 1.03321f, 
		0.173308f, 1.07964f, 0.467649f, 0.951991f, 
		0.137048f, 1.08961f, 0.517163f, 0.933797f, 
		0.564234f, 0.920310f, 0.606420f, 0.916239f, 
		0.646163f, 0.916876f, 0.487065f, 1.07139f, 
		0.522380f, 1.04735f, 0.559159f, 1.03963f, 
		0.595646f, 1.02329f, 0.627228f, 1.02573f, 
		0.512635f, 1.18885f, 0.528410f, 1.14023f, 
		0.552253f, 1.15229f, 0.546064f, 1.23972f, 
		0.579361f, 1.12057f, 0.589062f, 1.15935f
	};
	static const GLint Indices[] = {
		1156, 1141, 1156, 1297, 1156, 1275, 1155, 1274, 1155, 1212, 1155, 1080, 1222, 1172, 1222, 1338, 
		1222, 1341, 1221, 1340, 1221, 1296, 1221, 1140, 1225, 1144, 1225, 1300, 1225, 1344, 1224, 1343, 
		1224, 1337, 1224, 1171, 1153, 1077, 1153, 1209, 1153, 1278, 1152, 1277, 1152, 1299, 1152, 1143, 
		1040, 961, 1040, 1026, 1040, 1130, 1039, 1129, 1039, 1208, 1039, 1076, 865, 741, 865, 744, 
		865, 914, 864, 913, 864, 1025, 864, 960, 652, 567, 652, 492, 652, 605, 651, 604, 
		651, 743, 651, 740, 480, 449, 480, 329, 480, 393, 479, 392, 479, 491, 479, 566, 
		373, 382, 373, 223, 373, 254, 372, 253, 372, 328, 372, 448, 319, 359, 319, 190, 
		319, 187, 318, 186, 318, 222, 318, 381, 316, 379, 316, 226, 316, 184, 315, 183, 
		315, 189, 315, 358, 370, 446, 370, 332, 370, 257, 369, 256, 369, 225, 369, 378, 
		477, 564, 477, 495, 477, 396, 476, 395, 476, 331, 476, 445, 655, 764, 655, 767, 
		655, 608, 654, 607, 654, 494, 654, 563, 868, 964, 868, 1029, 868, 917, 867, 916, 
		867, 766, 867, 763, 1043, 1079, 1043, 1211, 1043, 1133, 1042, 1132, 1042, 1028, 1042, 963, 
		1041, 738, 1078, 738, 1154, 738, 1139, 738, 1220, 738, 1170, 738, 1223, 738, 1142, 738, 
		1151, 738, 1075, 738, 1038, 738, 959, 738, 863, 738, 739, 738, 650, 738, 565, 738, 
		478, 738, 447, 738, 371, 738, 380, 738, 317, 738, 357, 738, 314, 738, 377, 738, 
		368, 738, 444, 738, 475, 738, 562, 738, 653, 738, 762, 738, 866, 738, 962, 1041, 
		1070, 1011, 1091, 1125, 1148, 1214, 1269, 1260, 1326, 1346, 1369, 1393, 1418, 1395, 1424, 1444, 
		1441, 1446, 1445, 1465, 1451, 1453, 1426, 1431, 1380, 1432, 1381, 1382, 1305, 1306, 1233, 1307, 
		1234, 1235, 1103, 1104, 978, 1095, 968, 956, 759, 758, 568, 756, 581, 586, 450, 457, 
		353, 462, 360, 364, 273, 279, 205, 327, 205, 252, 205, 221, 172, 143, 167, 135, 
		130, 124, 120, 129, 115, 122, 104, 157, 127, 174, 112, 161, 155, 228, 218, 305, 
		217, 304, 303, 421, 420, 547, 419, 546, 545, 773, 772, 979, 769, 972, 967, 1099, 
		1094, 1206, 1085, 1194, 1186, 1283, 1280, 1346, 1280, 1260, 1186, 1175, 1065, 1125, 1031, 1011, 
		908, 890, 734, 877, 732, 730, 638, 646, 532, 680, 620, 728, 524, 508, 410, 455, 
		366, 346, 342, 291, 282, 284, 258, 244, 191, 179, 121, 192, 132, 164, 114, 140, 
		138, 180, 177, 242, 168, 233, 219, 333, 311, 423, 296, 413, 414, 540, 541, 750, 
		542, 752, 754, 976, 977, 1103, 977, 978, 754, 757, 544, 759, 553, 568, 425, 435, 
		336, 450, 349, 353, 264, 266, 196, 273, 202, 205, 167, 172, 508, 728, 455, 464, 
		346, 406, 309, 313, 284, 276, 244, 285, 259, 283, 192, 207, 164, 261, 198, 268, 
		180, 245, 242, 344, 339, 438, 333, 430, 423, 560, 551, 746, 540, 749, 750, 974, 
		975, 1101, 976, 1102, 1103, 1233, 1234, 464, 728, 406, 437, 313, 407, 310, 347, 285, 
		292, 283, 343, 323, 375, 261, 325, 268, 398, 355, 459, 344, 441, 438, 576, 570, 
		748, 560, 747, 746, 966, 971, 1098, 974, 1100, 1101, 1231, 1232, 1305, 1232, 1233, 1101, 
		1102, 437, 728, 407, 465, 347, 456, 367, 411, 343, 388, 375, 453, 434, 512, 398, 
		486, 459, 611, 588, 737, 576, 745, 748, 950, 953, 1084, 966, 1093, 1098, 1205, 1227, 
		1302, 1231, 1304, 1305, 1380, 1381, 465, 728, 456, 509, 411, 525, 473, 533, 453, 518, 
		512, 639, 631, 735, 611, 736, 737, 907, 938, 1064, 950, 1082, 1084, 1185, 1193, 1284, 
		1205, 1290, 1302, 1364, 1376, 1426, 1376, 1380, 1302, 1304, 509, 728, 525, 621, 533, 681, 
		647, 731, 639, 733, 735, 878, 891, 1010, 907, 1030, 1064, 1126, 1174, 1259, 1185, 1281, 
		1284, 1347, 1353, 1396, 1364, 1415, 1426, 1445, 1451, 621, 728, 681, 729, 731, 843, 870, 
		990, 878, 1002, 1010, 1071, 1092, 1149, 1126, 1215, 1259, 1270, 1325, 1370, 1347, 1394, 1396, 
		1419, 1425, 1441, 1425, 1445, 1396, 1415, 729, 728, 843, 899, 990, 996, 1047, 1113, 1071, 
		1135, 1149, 1188, 1217, 1253, 1270, 1318, 1370, 1335, 1402, 1412, 1419, 1428, 1441, 1418, 1424, 
		899, 728, 996, 1013, 1113, 1069, 1159, 1182, 1188, 1242, 1253, 1251, 1272, 1282, 1335, 1350, 
		1412, 1334, 1401, 1369, 1401, 1418, 1412, 1428, 1013, 728, 1069, 1056, 1182, 1119, 1230, 1226, 
		1251, 1256, 1282, 1250, 1271, 1252, 1334, 1317, 1369, 1269, 1326, 1056, 728, 1119, 1086, 1226, 
		1118, 1229, 1181, 1250, 1241, 1252, 1187, 1216, 1148, 1216, 1269, 1252, 1317, 1086, 728, 1118, 
		1055, 1181, 1068, 1158, 1112, 1187, 1134, 1148, 1070, 1091, 1055, 728, 1068, 1012, 1112, 995, 
		1046, 989, 1070, 1001, 1011, 877, 890, 532, 620, 532, 524, 472, 410, 452, 387, 374, 
		342, 324, 282, 262, 208, 165, 191, 131, 121, 113, 105, 92, 114, 110, 138, 90, 
		118, 106, 168, 158, 219, 146, 209, 210, 296, 297, 414, 298, 415, 416, 542, 543, 
		754, 543, 544, 416, 417, 300, 425, 320, 336, 227, 238, 173, 264, 194, 196, 156, 
		163, 130, 163, 167, 196, 202, 646, 730, 680, 727, 728, 842, 898, 989, 898, 995, 
		728, 1012, 727, 730, 842, 869, 989, 877, 1001, 1283, 1194, 1283, 1206, 1291, 1301, 1363, 
		1375, 1427, 1385, 1436, 1437, 1452, 1454, 1468, 1456, 1469, 1470, 1457, 1459, 1434, 1461, 1435, 
		1423, 1384, 1374, 1293, 1357, 1285, 1268, 1196, 1178, 1072, 1177, 1066, 1060, 940, 936, 753, 
		926, 751, 742, 595, 603, 595, 490, 466, 364, 466, 462, 595, 590, 753, 586, 755, 
		756, 940, 946, 1072, 956, 1088, 1095, 1196, 1218, 1293, 1235, 1308, 1307, 1384, 1383, 1434, 
		1382, 1433, 1432, 1457, 1455, 1468, 1453, 1467, 1465, 1452, 1450, 1427, 1444, 1414, 1395, 1363, 
		1352, 1283, 1352, 1346, 1395, 1393, 1094, 1085, 967, 954, 771, 951, 768, 761, 577, 589, 
		460, 610, 487, 511, 399, 433, 374, 433, 452, 511, 517, 638, 517, 532, 452, 472, 
		92, 110, 92, 90, 88, 68, 89, 82, 107, 76, 93, 95, 151, 152, 215, 153, 
		216, 217, 216, 303, 215, 302, 301, 419, 418, 545, 424, 552, 561, 769, 770, 967, 
		770, 771, 561, 571, 439, 577, 442, 460, 345, 356, 269, 399, 326, 374, 262, 324, 
		92, 88, 92, 89, 109, 137, 113, 141, 165, 181, 199, 269, 262, 326, 118, 138, 
		168, 177, 233, 242, 333, 339, 430, 438, 560, 570, 747, 748, 966, 953, 1093, 1084, 
		1205, 1193, 1290, 1284, 1364, 1353, 1450, 1465, 1444, 1446, 1283, 1291, 1363, 1427, 1414, 1363, 
		228, 161, 228, 174, 239, 265, 337, 350, 451, 354, 458, 463, 587, 591, 776, 596, 
		774, 765, 927, 915, 927, 1027, 1054, 1163, 1061, 1167, 1176, 1258, 1261, 1327, 1267, 1331, 
		1356, 1379, 1405, 1429, 1422, 1448, 1460, 1462, 1471, 1461, 1470, 1459, 228, 239, 228, 337, 
		321, 426, 305, 422, 421, 549, 548, 777, 547, 775, 773, 981, 980, 1106, 979, 1105, 
		1099, 1236, 1228, 1301, 1228, 1206, 1099, 127, 112, 104, 85, 70, 80, 66, 65, 75, 
		74, 100, 73, 98, 96, 148, 147, 210, 147, 146, 96, 94, 71, 106, 81, 90, 
		81, 68, 71, 62, 63, 76, 77, 95, 78, 97, 99, 153, 154, 217, 155, 218, 
		194, 156, 173, 126, 111, 103, 84, 70, 75, 66, 126, 156, 103, 123, 115, 130, 
		120, 349, 264, 336, 238, 581, 450, 568, 435, 946, 756, 956, 758, 1293, 1285, 1196, 
		1072, 1088, 1196, 1374, 1423, 1357, 1406, 1378, 1430, 1410, 1417, 1404, 1413, 1408, 1403, 1397, 
		1366, 1391, 1362, 1295, 1321, 1273, 1321, 1210, 1255, 1163, 1258, 1167, 1406, 1423, 1430, 1449, 
		1462, 1449, 1461, 1423, 1462, 1448, 1462, 1429, 1443, 1417, 1443, 1430, 1462, 1405, 1422, 1356, 
		1373, 1294, 1389, 1313, 1312, 1240, 1239, 1108, 1238, 1107, 1106, 1107, 981, 1108, 982, 983, 
		777, 780, 549, 782, 554, 569, 426, 436, 337, 436, 451, 569, 582, 779, 587, 778, 
		776, 941, 937, 1061, 927, 1054, 1394, 1370, 1419, 1402, 1281, 1259, 1347, 1325, 1082, 1064, 
		1185, 1174, 745, 737, 950, 938, 441, 459, 576, 588, 245, 268, 344, 355, 140, 164, 
		180, 198, 105, 121, 114, 132, 92, 109, 113, 165, 131, 113, 141, 137, 181, 178, 
		243, 169, 234, 220, 334, 312, 424, 301, 418, 243, 234, 243, 334, 340, 439, 345, 
		442, 178, 137, 169, 119, 107, 119, 89, 137, 243, 340, 243, 345, 246, 269, 246, 
		181, 243, 571, 771, 577, 768, 954, 1085, 951, 1083, 1065, 1083, 1186, 1085, 1375, 1301, 
		1385, 1309, 1310, 1309, 1236, 1301, 1427, 1436, 1452, 1468, 1467, 1452, 1455, 1432, 1453, 1431, 
		1100, 1098, 1231, 1227, 749, 746, 974, 971, 413, 423, 540, 551, 209, 219, 296, 311, 
		94, 106, 146, 158, 82, 68, 76, 62, 107, 93, 107, 151, 159, 220, 159, 169, 
		107, 312, 220, 301, 214, 215, 214, 151, 220, 772, 769, 545, 552, 773, 980, 979, 
		1099, 972, 979, 1105, 1106, 1236, 1237, 1310, 1237, 1238, 1106, 1373, 1422, 1389, 1440, 1439, 
		1440, 1460, 1422, 1454, 1437, 1456, 1438, 1439, 1387, 1388, 1312, 1388, 1389, 1439, 1438, 1437, 
		1387, 1386, 1310, 1386, 1385, 1437, 1471, 1470, 1460, 1458, 1439, 1458, 1456, 1470, 1468, 1469, 
		1457, 1434, 1433, 1457, 1383, 1307, 1382, 1306, 1434, 1435, 1384, 1293, 1308, 1384, 1218, 1095, 
		1235, 1104, 752, 750, 976, 975, 968, 759, 978, 757, 415, 414, 542, 541, 553, 425, 
		544, 417, 210, 297, 210, 298, 211, 212, 148, 149, 100, 150, 102, 111, 75, 84, 
		320, 227, 300, 213, 212, 150, 149, 210, 211, 148, 100, 98, 148, 213, 227, 150, 
		160, 111, 160, 173, 227, 74, 65, 73, 64, 63, 78, 77, 100, 102, 75, 64, 
		65, 78, 79, 99, 79, 80, 65, 85, 112, 80, 101, 99, 155, 154, 152, 95, 
		153, 97, 101, 112, 155, 420, 419, 303, 302, 228, 321, 305, 421, 304, 305, 421, 
		548, 547, 773, 546, 547, 422, 426, 549, 554, 775, 777, 981, 982, 780, 782, 983, 
		969, 1096, 957, 1089, 1073, 1197, 1179, 1267, 1176, 1261, 1096, 1089, 1096, 1197, 1219, 1294, 
		1240, 1313, 969, 782, 957, 781, 779, 781, 569, 782, 1239, 1312, 1238, 1311, 1310, 1311, 
		1387, 1312, 1096, 1219, 1096, 1240, 1109, 1108, 1109, 983, 1096, 1379, 1331, 1379, 1327, 1371, 
		1366, 1371, 1403, 1379, 1409, 1429, 1409, 1417, 1403, 1413, 1258, 1255, 1258, 1321, 1324, 1366, 
		1324, 1327, 1258, 1362, 1366, 1321, 1391, 1295, 1391, 1339, 1391, 1336, 1400, 1392, 1408, 1398, 
		1404, 1367, 1372, 1328, 1378, 1330, 1357, 1330, 1268, 1328, 1262, 1257, 1177, 1166, 1060, 1162, 
		1053, 1024, 926, 912, 742, 1400, 1408, 1391, 1397, 1372, 1378, 1404, 1410, 1392, 1336, 1392, 
		1342, 1392, 1298, 1361, 1320, 1367, 1323, 1328, 1323, 1257, 1320, 1254, 1207, 1162, 1128, 1024, 
		1361, 1367, 1392, 1398, 1320, 1298, 1320, 1276, 1207, 1178, 1268, 1177, 1262, 1166, 1257, 1162, 
		1254, 1072, 1066, 940, 753, 755, 940, 936, 1060, 926, 1053, 590, 462, 586, 457, 753, 
		751, 595, 360, 273, 353, 266, 364, 490, 364, 391, 364, 327, 279, 123, 156, 130, 
		143, 221, 143, 185, 143, 188, 134, 142, 124, 136, 129, 166, 162, 197, 157, 193, 
		174, 193, 265, 197, 267, 272, 354, 361, 463, 365, 467, 493, 596, 606, 765, 134, 
		124, 143, 135, 162, 157, 129, 122, 142, 188, 142, 182, 142, 224, 171, 204, 166, 
		201, 197, 201, 272, 204, 280, 330, 365, 394, 493, 171, 166, 142, 136, 204, 224, 
		204, 255, 330, 350, 265, 354, 267, 361, 272, 365, 280, 582, 451, 587, 458, 591, 
		463, 596, 467, 779, 778, 779, 941, 947, 1073, 947, 957, 779, 937, 776, 927, 774, 
		1179, 1073, 1176, 1067, 1061, 1067, 941, 1073, 1163, 1027, 1163, 1131, 1210, 1241, 1181, 1187, 
		1158, 1256, 1226, 1250, 1229, 1350, 1282, 1334, 1271, 1242, 1182, 1251, 1230, 1318, 1253, 1335, 
		1272, 1135, 1113, 1188, 1159, 1215, 1149, 1270, 1217, 1002, 990, 1071, 1047, 1030, 1010, 1126, 
		1092, 733, 731, 878, 870, 736, 735, 907, 891, 518, 533, 639, 647, 486, 512, 611, 
		631, 388, 411, 453, 473, 325, 375, 398, 434, 292, 347, 343, 367, 207, 283, 261, 
		323, 276, 313, 285, 310, 179, 244, 192, 259, 291, 346, 284, 309, 208, 282, 191, 
		258, 387, 410, 342, 366, 165, 199, 262, 356, 460, 399, 487, 734, 732, 734, 638, 
		630, 511, 630, 610, 734, 760, 908, 761, 939, 951, 939, 1065, 908, 1031, 589, 761, 
		610, 760, 869, 730, 877, 1175, 1260, 1125, 1214, 1134, 1112, 1070, 1046, 73, 63, 72, 
		71, 72, 96, 73, 300, 212, 299, 298, 299, 416, 300, 1286, 1267, 1286, 1356, 1286, 
		1294, 1286, 1197, 1267, 103, 115, 91, 104, 91, 70, 103, 334, 424, 431, 561, 431, 
		439, 334, 665, 535, 637, 592, 709, 707, 844, 703, 885, 896, 1032, 896, 1062, 896, 
		1063, 886, 860, 684, 679, 505, 685, 528, 584, 429, 584, 498, 584, 574, 629, 649, 
		686, 677, 702, 669, 702, 708, 702, 807, 719, 804, 686, 687, 584, 687, 685, 804, 
		839, 998, 860, 1035, 1063, 592, 535, 592, 483, 592, 404, 529, 504, 703, 692, 896, 
		684, 886, 529, 703, 592, 707, 885, 1032, 844, 987, 844, 882, 815, 802, 709, 711, 
		665, 713, 665, 666, 665, 602, 665, 597, 535, 815, 709, 844, 504, 404, 504, 384, 
		504, 383, 499, 505, 499, 684, 504, 692, 505, 383, 505, 400, 505, 429, 528, 679, 
		685, 860, 839, 629, 686, 584, 804, 807, 804, 893, 998, 649, 574, 649, 628, 649, 
		669, 677, 719, 686, 702, 802, 882, 802, 853, 802, 841, 802, 805, 802, 713, 711, 
		637, 709, 665, 897, 902, 921, 992, 901, 933, 832, 1008, 900, 1018, 794, 1018, 906, 
		1018, 1059, 1146, 1289, 1245, 1354, 1411, 1447, 1472, 1483, 1477, 1483, 1488, 1483, 1497, 1493, 
		1496, 1489, 1494, 1490, 1492, 1490, 1486, 1490, 1478, 1484, 1475, 1489, 1479, 1483, 1479, 1447, 
		1475, 1416, 1377, 1289, 1246, 1059, 992, 902, 992, 1007, 992, 1120, 1058, 1189, 1008, 1114, 
		1018, 1245, 1146, 1058, 1008, 992, 933, 900, 794, 832, 696, 832, 674, 819, 813, 901, 
		889, 897, 884, 897, 894, 897, 871, 897, 858, 902, 819, 901, 832, 1189, 1120, 1189, 
		1263, 1189, 1390, 1316, 1411, 1316, 1245, 1189, 1114, 1411, 1390, 1411, 1464, 1411, 1477, 1472, 
		1354, 1447, 1289, 1416, 1493, 1489, 1483, 1475, 1478, 1475, 1466, 1377, 1496, 1497, 1496, 1498, 
		1496, 1492, 1494, 1484, 1489, 1490, 813, 674, 813, 671, 813, 699, 813, 814, 813, 884, 
		889, 921, 901, 897, 695, 820, 667, 613, 550, 514, 502, 468, 428, 289, 405, 289, 
		236, 289, 87, 144, 55, 206, 83, 128, 40, 49, 31, 57, 31, 34, 31, 22, 
		20, 4, 15, 1, 6, 0, 6, 3, 6, 19, 16, 26, 15, 29, 31, 29, 
		40, 26, 36, 33, 55, 42, 87, 613, 820, 613, 724, 613, 657, 523, 408, 468, 
		352, 289, 206, 144, 523, 468, 613, 514, 428, 405, 502, 516, 502, 601, 538, 612, 
		550, 617, 695, 683, 695, 721, 695, 817, 695, 827, 820, 538, 550, 502, 408, 657, 
		408, 496, 408, 341, 278, 128, 278, 206, 408, 352, 128, 341, 128, 176, 128, 57, 
		49, 83, 40, 55, 36, 20, 15, 31, 26, 19, 26, 25, 33, 4, 22, 4, 
		2, 4, 0, 1, 16, 15, 6, 612, 601, 612, 635, 612, 642, 612, 658, 612, 
		683, 617, 667, 550, 695, 834, 991, 861, 923, 786, 801, 664, 822, 636, 625, 489, 
		625, 454, 625, 440, 623, 641, 845, 837, 1015, 824, 973, 903, 1052, 903, 999, 903, 
		905, 855, 828, 818, 803, 784, 791, 784, 717, 784, 694, 722, 704, 818, 821, 903, 
		821, 824, 704, 675, 522, 641, 481, 440, 923, 991, 923, 1034, 923, 1115, 993, 1017, 
		822, 835, 625, 845, 623, 993, 822, 923, 801, 636, 489, 664, 520, 664, 557, 668, 
		663, 786, 716, 834, 700, 834, 816, 834, 888, 834, 932, 991, 668, 786, 664, 1017, 
		1115, 1017, 1157, 1017, 1168, 1023, 1015, 1023, 845, 1017, 835, 1015, 1168, 1015, 1127, 1015, 
		1052, 973, 837, 824, 641, 675, 855, 818, 903, 704, 694, 704, 616, 522, 828, 905, 
		828, 831, 828, 791, 803, 722, 818, 784, 663, 557, 663, 575, 663, 594, 663, 633, 
		663, 700, 716, 861, 786, 834, 798, 705, 829, 883, 892, 965, 970, 1022, 1044, 1165, 
		1051, 1165, 1192, 1165, 1333, 1314, 1463, 1287, 1442, 1421, 1487, 1482, 1502, 1480, 1502, 1500, 
		1502, 1511, 1513, 1517, 1514, 1518, 1516, 1521, 1516, 1515, 1516, 1510, 1512, 1505, 1514, 1504, 
		1502, 1504, 1487, 1505, 1495, 1499, 1463, 1474, 1333, 883, 705, 883, 808, 883, 924, 997, 
		1122, 1022, 1138, 1165, 1287, 1314, 997, 1022, 883, 965, 1044, 1051, 970, 930, 970, 825, 
		872, 830, 892, 848, 798, 823, 798, 726, 798, 688, 798, 682, 705, 872, 892, 970, 
		1122, 924, 1122, 1097, 1122, 1264, 1279, 1421, 1279, 1287, 1122, 1138, 1421, 1264, 1421, 1407, 
		1421, 1480, 1482, 1442, 1487, 1463, 1495, 1513, 1514, 1502, 1505, 1510, 1505, 1507, 1499, 1517, 
		1511, 1517, 1519, 1517, 1521, 1518, 1512, 1514, 1516, 830, 825, 830, 725, 830, 723, 830, 
		806, 830, 823, 848, 829, 892, 798, 619, 578, 536, 484, 558, 506, 598, 432, 501, 
		385, 614, 385, 469, 385, 290, 230, 86, 170, 56, 47, 35, 32, 23, 27, 23, 
		18, 23, 9, 13, 7, 14, 10, 11, 8, 11, 12, 11, 21, 17, 28, 14, 
		24, 23, 24, 35, 28, 38, 48, 86, 133, 290, 484, 578, 484, 471, 484, 348, 
		386, 249, 432, 307, 385, 170, 230, 386, 432, 484, 506, 501, 614, 598, 710, 598, 
		790, 659, 689, 558, 622, 619, 678, 619, 670, 619, 672, 619, 645, 578, 659, 558, 
		598, 249, 348, 249, 200, 249, 69, 117, 47, 117, 170, 249, 307, 47, 69, 47, 
		37, 47, 27, 32, 56, 35, 86, 38, 13, 14, 23, 28, 21, 28, 30, 48, 
		7, 9, 7, 5, 7, 8, 10, 17, 14, 11, 689, 790, 689, 811, 689, 799, 
		689, 715, 689, 678, 622, 536, 558, 619, 931, 949, 943, 984, 909, 920, 849, 955, 
		862, 934, 800, 934, 847, 934, 952, 1016, 1111, 1081, 1147, 1200, 1244, 1265, 1319, 1292, 
		1319, 1332, 1319, 1358, 1348, 1359, 1329, 1355, 1349, 1360, 1349, 1351, 1349, 1315, 1322, 1266, 
		1329, 1288, 1319, 1288, 1244, 1266, 1190, 1145, 1111, 1037, 952, 984, 949, 984, 994, 984, 
		1033, 1005, 1049, 955, 1009, 934, 1081, 1016, 1005, 955, 984, 920, 862, 800, 849, 788, 
		849, 789, 850, 852, 909, 911, 931, 919, 931, 945, 931, 944, 931, 928, 949, 850, 
		909, 849, 1049, 1033, 1049, 1110, 1049, 1164, 1124, 1200, 1124, 1081, 1049, 1009, 1200, 1164, 
		1200, 1247, 1200, 1292, 1265, 1147, 1244, 1111, 1190, 1348, 1329, 1319, 1266, 1315, 1266, 1248, 
		1145, 1359, 1358, 1359, 1365, 1359, 1360, 1355, 1322, 1329, 1349, 852, 789, 852, 793, 852, 
		812, 852, 854, 852, 919, 911, 943, 909, 931, 615, 579, 583, 526, 599, 559, 626, 
		515, 580, 513, 660, 513, 585, 513, 500, 461, 390, 402, 351, 308, 293, 281, 251, 
		248, 251, 235, 251, 231, 237, 241, 250, 247, 263, 260, 263, 270, 263, 286, 271, 
		288, 250, 274, 251, 274, 293, 288, 338, 363, 390, 427, 500, 526, 579, 526, 510, 
		526, 443, 485, 409, 515, 470, 513, 402, 461, 485, 515, 526, 559, 580, 660, 626, 
		691, 626, 712, 662, 676, 599, 627, 615, 656, 615, 643, 615, 644, 615, 624, 579, 
		662, 599, 626, 409, 443, 409, 376, 409, 295, 362, 308, 362, 402, 409, 470, 308, 
		295, 308, 275, 308, 248, 281, 351, 293, 390, 338, 237, 250, 251, 288, 286, 288, 
		306, 363, 241, 231, 241, 240, 241, 260, 247, 271, 250, 263, 676, 712, 676, 720, 
		676, 718, 676, 690, 676, 656, 627, 583, 599, 615, 573, 640, 555, 527, 521, 507, 
		519, 474, 482, 397, 497, 397, 389, 397, 277, 287, 195, 294, 203, 229, 125, 139, 
		61, 145, 61, 67, 61, 52, 51, 44, 50, 43, 46, 39, 46, 45, 46, 54, 
		53, 58, 50, 59, 61, 59, 125, 58, 116, 108, 195, 175, 277, 527, 640, 527, 
		593, 527, 503, 488, 403, 474, 401, 397, 294, 287, 488, 474, 527, 507, 482, 497, 
		519, 556, 519, 609, 534, 537, 521, 530, 573, 531, 573, 572, 573, 632, 573, 661, 
		640, 534, 521, 519, 403, 503, 403, 412, 403, 335, 322, 229, 322, 294, 403, 401, 
		229, 335, 229, 232, 229, 145, 139, 203, 125, 195, 116, 51, 50, 61, 58, 54, 
		58, 60, 108, 44, 52, 44, 41, 44, 39, 43, 53, 50, 46, 537, 609, 537, 
		618, 537, 600, 537, 539, 537, 531, 530, 555, 521, 573, 836, 787, 840, 851, 887, 
		904, 925, 929, 988, 1003, 1004, 1003, 1019, 1003, 1045, 1021, 1087, 1006, 1048, 1020, 1117, 
		1090, 1137, 1050, 1137, 1121, 1137, 1173, 1180, 1198, 1183, 1203, 1202, 1213, 1202, 1204, 1202, 
		1195, 1191, 1169, 1183, 1150, 1137, 1150, 1117, 1169, 1136, 1161, 1087, 1116, 1045, 851, 787, 
		851, 797, 851, 826, 874, 918, 929, 985, 1003, 1006, 1021, 874, 929, 851, 904, 988, 
		1004, 925, 948, 925, 880, 895, 881, 887, 876, 836, 875, 836, 838, 836, 796, 836, 
		785, 787, 895, 887, 925, 918, 826, 918, 856, 918, 922, 986, 1020, 986, 1006, 918, 
		985, 1020, 922, 1020, 1000, 1020, 1050, 1090, 1048, 1117, 1087, 1136, 1180, 1183, 1137, 1169, 
		1195, 1169, 1184, 1161, 1198, 1173, 1198, 1199, 1198, 1213, 1203, 1191, 1183, 1202, 881, 880, 
		881, 857, 881, 859, 881, 879, 881, 875, 876, 840, 887, 836, 783, 795, 833, 942, 
		846, 935, 873, 1036, 1014, 1160, 958, 1160, 1123, 1160, 1303, 1345, 1476, 1368, 1481, 1485, 
		1506, 1508, 1524, 1509, 1524, 1525, 1524, 1530, 1529, 1535, 1528, 1533, 1532, 1534, 1532, 1531, 
		1532, 1526, 1527, 1522, 1528, 1523, 1524, 1523, 1506, 1522, 1503, 1501, 1476, 1473, 1303, 942, 
		795, 942, 910, 942, 1074, 1057, 1243, 1036, 1201, 1160, 1368, 1345, 1057, 1036, 942, 935, 
		1014, 958, 873, 810, 873, 698, 809, 714, 846, 792, 783, 701, 783, 697, 783, 693, 
		783, 706, 795, 809, 846, 873, 1243, 1074, 1243, 1249, 1243, 1420, 1399, 1485, 1399, 1368, 
		1243, 1201, 1485, 1420, 1485, 1491, 1485, 1509, 1508, 1481, 1506, 1476, 1503, 1529, 1528, 1524, 
		1522, 1526, 1522, 1520, 1501, 1535, 1530, 1535, 1536, 1535, 1534, 1533, 1527, 1528, 1532, 714, 
		698, 714, 648, 714, 634, 714, 673, 714, 701, 792, 833, 846, 783
	};

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2, GL_FLOAT, 0, TexCoordData);
	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 0, NormalData);
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 0, VertexData);


	// Material attributes for surface 'tierra_maceta'
	flower_Material_Parameters(52.0000f, 0.200000f, 0.150685f, 0.106683f, 0.200000f, 0.200000f, 0.200000f);

	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[0]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[6]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[12]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[18]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[24]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[30]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[36]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[42]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[48]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[54]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[60]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[66]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[72]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[78]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[84]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[90]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[96]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[102]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[108]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[114]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[120]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[126]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[132]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[138]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[144]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[150]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[156]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[162]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[168]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[174]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[180]);
	glDrawElements(GL_TRIANGLE_STRIP, 6, GL_UNSIGNED_INT, &Indices[186]);
	glDrawElements(GL_TRIANGLE_STRIP, 64, GL_UNSIGNED_INT, &Indices[192]);

	// Material attributes for surface 'maceta'
	flower_Material_Parameters(96.0000f, 0.200000f, 0.114493f, 0.0376813f, 0.200000f, 0.200000f, 0.200000f);

	glDrawElements(GL_TRIANGLE_STRIP, 188, GL_UNSIGNED_INT, &Indices[256]);
	glDrawElements(GL_TRIANGLE_STRIP, 43, GL_UNSIGNED_INT, &Indices[444]);
	glDrawElements(GL_TRIANGLE_STRIP, 42, GL_UNSIGNED_INT, &Indices[487]);
	glDrawElements(GL_TRIANGLE_STRIP, 37, GL_UNSIGNED_INT, &Indices[529]);
	glDrawElements(GL_TRIANGLE_STRIP, 36, GL_UNSIGNED_INT, &Indices[566]);
	glDrawElements(GL_TRIANGLE_STRIP, 31, GL_UNSIGNED_INT, &Indices[602]);
	glDrawElements(GL_TRIANGLE_STRIP, 30, GL_UNSIGNED_INT, &Indices[633]);
	glDrawElements(GL_TRIANGLE_STRIP, 25, GL_UNSIGNED_INT, &Indices[663]);
	glDrawElements(GL_TRIANGLE_STRIP, 24, GL_UNSIGNED_INT, &Indices[688]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[712]);
	glDrawElements(GL_TRIANGLE_STRIP, 18, GL_UNSIGNED_INT, &Indices[731]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[749]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[762]);
	glDrawElements(GL_TRIANGLE_STRIP, 63, GL_UNSIGNED_INT, &Indices[775]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[838]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[850]);
	glDrawElements(GL_TRIANGLE_STRIP, 93, GL_UNSIGNED_INT, &Indices[857]);
	glDrawElements(GL_TRIANGLE_STRIP, 26, GL_UNSIGNED_INT, &Indices[950]);
	glDrawElements(GL_TRIANGLE_STRIP, 48, GL_UNSIGNED_INT, &Indices[976]);
	glDrawElements(GL_TRIANGLE_STRIP, 14, GL_UNSIGNED_INT, &Indices[1024]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1038]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1042]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1046]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1050]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1054]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1058]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1062]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1066]);
	glDrawElements(GL_TRIANGLE_STRIP, 44, GL_UNSIGNED_INT, &Indices[1072]);
	glDrawElements(GL_TRIANGLE_STRIP, 27, GL_UNSIGNED_INT, &Indices[1116]);
	glDrawElements(GL_TRIANGLE_STRIP, 41, GL_UNSIGNED_INT, &Indices[1143]);
	glDrawElements(GL_TRIANGLE_STRIP, 10, GL_UNSIGNED_INT, &Indices[1184]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[1194]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1201]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1205]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1209]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1213]);
	glDrawElements(GL_TRIANGLE_STRIP, 25, GL_UNSIGNED_INT, &Indices[1219]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1244]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1252]);
	glDrawElements(GL_TRIANGLE_STRIP, 41, GL_UNSIGNED_INT, &Indices[1261]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1302]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1306]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1310]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1314]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1318]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1322]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1326]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1330]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1334]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[1340]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1353]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1361]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1369]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1378]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1382]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1390]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1398]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1404]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1408]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1412]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1416]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1420]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1424]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1428]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1432]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1441]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1449]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1453]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1459]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1467]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[1475]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1486]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1494]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1502]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1508]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1512]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1518]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1522]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1526]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1530]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1534]);
	glDrawElements(GL_TRIANGLE_STRIP, 14, GL_UNSIGNED_INT, &Indices[1538]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[1552]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1559]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1565]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[1573]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[1580]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1583]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[1591]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1598]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[1602]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1605]);
	glDrawElements(GL_TRIANGLES, 12, GL_UNSIGNED_INT, &Indices[1609]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1621]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1625]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[1629]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1642]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1650]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1658]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1666]);
	glDrawElements(GL_TRIANGLE_STRIP, 15, GL_UNSIGNED_INT, &Indices[1675]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1690]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[1699]);
	glDrawElements(GL_TRIANGLE_STRIP, 31, GL_UNSIGNED_INT, &Indices[1702]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1733]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1737]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[1741]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1760]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[1764]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1769]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1773]);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_INT, &Indices[1777]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1783]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1787]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[1791]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1794]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[1798]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[1805]);
	glDrawElements(GL_TRIANGLE_STRIP, 31, GL_UNSIGNED_INT, &Indices[1808]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1839]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1843]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[1847]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1866]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[1870]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1875]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1879]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1883]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1887]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[1891]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1900]);
	glDrawElements(GL_TRIANGLE_STRIP, 8, GL_UNSIGNED_INT, &Indices[1904]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[1912]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1917]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1921]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1925]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1929]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1933]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1937]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1941]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1945]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1949]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1953]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1957]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1961]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1965]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1969]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1973]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1977]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1981]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1985]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1989]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1993]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[1997]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2001]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2005]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2008]);
	glDrawElements(GL_TRIANGLE_STRIP, 18, GL_UNSIGNED_INT, &Indices[2012]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2030]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2034]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2037]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2041]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2045]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2052]);
	glDrawElements(GL_TRIANGLE_STRIP, 9, GL_UNSIGNED_INT, &Indices[2059]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2068]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2075]);

	// Material attributes for surface 'leaf_paloagua.jpg'
	flower_Material_Parameters(128.000f, 0.0403672f, 0.200000f, 0.00367001f, 0.200000f, 0.200000f, 0.200000f);

	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2082]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2133]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2146]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2150]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2169]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2172]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2184]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2191]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2195]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2198]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2203]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2210]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2213]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2224]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2227]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2278]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2291]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2295]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2314]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2317]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2329]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2336]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2340]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2343]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2348]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2355]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2358]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2369]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2372]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2423]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2436]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2440]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2459]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2462]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2474]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2481]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2485]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2488]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2493]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2500]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2503]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2514]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2517]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2568]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2581]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2585]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2604]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2607]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2619]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2626]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2630]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2633]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2638]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2645]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2648]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2659]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2662]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2713]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2726]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2730]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2749]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2752]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2764]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2771]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2775]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2778]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2783]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2790]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2793]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2804]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2807]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[2858]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2871]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[2875]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2894]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[2897]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2909]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[2916]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2920]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[2923]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[2928]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2935]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[2938]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[2949]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[2952]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[3003]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3016]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[3020]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3039]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[3042]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3054]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3061]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3065]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[3068]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3073]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3080]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[3083]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3094]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[3097]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[3148]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3161]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[3165]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3184]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[3187]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3199]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3206]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3210]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[3213]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3218]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3225]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[3228]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3239]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[3242]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[3293]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3306]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[3310]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3329]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[3332]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3344]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3351]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3355]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[3358]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3363]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3370]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[3373]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3384]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[3387]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[3438]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3451]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[3455]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3474]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[3477]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3489]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3496]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3500]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[3503]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3508]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3515]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[3518]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3529]);
	glDrawElements(GL_TRIANGLE_STRIP, 51, GL_UNSIGNED_INT, &Indices[3532]);
	glDrawElements(GL_TRIANGLE_STRIP, 13, GL_UNSIGNED_INT, &Indices[3583]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3596]);
	glDrawElements(GL_TRIANGLE_STRIP, 19, GL_UNSIGNED_INT, &Indices[3600]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3619]);
	glDrawElements(GL_TRIANGLE_STRIP, 12, GL_UNSIGNED_INT, &Indices[3622]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3634]);
	glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_INT, &Indices[3641]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3645]);
	glDrawElements(GL_TRIANGLE_STRIP, 5, GL_UNSIGNED_INT, &Indices[3648]);
	glDrawElements(GL_TRIANGLE_STRIP, 7, GL_UNSIGNED_INT, &Indices[3653]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3660]);
	glDrawElements(GL_TRIANGLE_STRIP, 11, GL_UNSIGNED_INT, &Indices[3663]);
	glDrawElements(GL_TRIANGLES, 3, GL_UNSIGNED_INT, &Indices[3674]);
}	// End of polygons for object 'flower'
