// =====================================================================
//
//  time_table.cxx
//
//  Author(s):
//    Robert Stiles, KK5VD, Copyright (C) 2013, 2014
//
// This file is part of FLAMP and FLMSG.
//
// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// =====================================================================


// Doxygen
/** ********************************************************
 \page page_modem_timing_table Modem Timing Table Structure

	A table and functions to calculate the number of seconds or 
	minutes needed to transmit a string of data.

 \par char *modem_name
	The string ID of the modem.<br>
	Example:<br>
	8PSK125, 8PSK250, etc. <br>
	
 \par float scale
	A value used to scale the resulting toble in an attemp to adjust for 
	errors. However, because of the differential errors for each charater 
	and overhead time, it's not recommended for use.

 \par float overhead
 	Each modem contains a leading and trailing pre/postamble. The time for
	both a accumulated and stored in this variable.

 \par table[256]
 	Floating point representation of the amount of time it takes to 
	transmit the character in seconds. Character value range 0 to 255.

 \verbatim
 	typedef struct {
		char *mode_name;
		float scale;
		float overhead;
		float table[256];
	} MODE_TIME_TABLE;

	static MODE_TIME_TABLE mode_time_table[] = {
		{
			(char *) "8PSK125", 1.0, 2.973000,
 			{
 				0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875,
 				0.042500, 0.058500, 0.058875, 0.058500, 0.058500, 0.042875, 0.058500, 0.058500,
 				0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500,
 				0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.063625, 0.064312,
 				0.016000, 0.047625, 0.048313, 0.053437, 0.053062, 0.053437, 0.053437, 0.047625,
				...
			},...
		},...
	};

\endverbatim

 ***********************************************************/

#include "debug.h"
#include "time_table.h"

typedef struct {
	char *mode_name;
	float scale;
	float overhead;
	float table[256];
} MODE_TIME_TABLE;

static MODE_TIME_TABLE mode_time_table[] = {
	{
		(char *) "8PSK125", 1.0, 2.973000,
		{
			0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875,
			0.042500, 0.058500, 0.058875, 0.058500, 0.058500, 0.042875, 0.058500, 0.058500,
			0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500,
			0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.063625, 0.064312,
			0.016000, 0.047625, 0.048313, 0.053437, 0.053062, 0.053437, 0.053437, 0.047625,
			0.048313, 0.048000, 0.053062, 0.048313, 0.042500, 0.047625, 0.048313, 0.048000,
			0.042500, 0.042875, 0.048000, 0.047625, 0.048313, 0.048000, 0.047625, 0.048313,
			0.048000, 0.047625, 0.048313, 0.048000, 0.053062, 0.048313, 0.053437, 0.047625,
			0.053437, 0.042500, 0.047625, 0.042875, 0.042500, 0.042500, 0.042875, 0.048000,
			0.047625, 0.042875, 0.048000, 0.047625, 0.042875, 0.042500, 0.042500, 0.042875,
			0.042500, 0.047625, 0.042875, 0.042500, 0.042500, 0.048313, 0.048000, 0.047625,
			0.048313, 0.048000, 0.047625, 0.053437, 0.053437, 0.053062, 0.053437, 0.053437,
			0.053062, 0.026875, 0.037437, 0.031625, 0.032313, 0.021438, 0.037062, 0.037437,
			0.032000, 0.026500, 0.042875, 0.037437, 0.031625, 0.037437, 0.026500, 0.026500,
			0.037437, 0.037437, 0.031625, 0.032313, 0.021438, 0.031625, 0.037437, 0.037437,
			0.037062, 0.037437, 0.037437, 0.053062, 0.053437, 0.053437, 0.053062, 0.064312,
			0.064000, 0.063625, 0.064312, 0.064000, 0.063625, 0.064312, 0.064000, 0.063625,
			0.064312, 0.064000, 0.063625, 0.064312, 0.064000, 0.063625, 0.064312, 0.064000,
			0.063625, 0.064312, 0.064000, 0.063625, 0.064312, 0.064000, 0.063625, 0.064312,
			0.064000, 0.063625, 0.064312, 0.064000, 0.063625, 0.064312, 0.064000, 0.063625,
			0.053437, 0.053437, 0.053062, 0.053437, 0.053437, 0.053062, 0.053437, 0.053437,
			0.053062, 0.053437, 0.053437, 0.053062, 0.053437, 0.053437, 0.053062, 0.053437,
			0.053437, 0.053062, 0.053437, 0.053437, 0.053062, 0.053437, 0.053437, 0.053062,
			0.053437, 0.053437, 0.053062, 0.053437, 0.053437, 0.053062, 0.053437, 0.053437,
			0.053062, 0.053437, 0.053437, 0.053062, 0.058875, 0.058500, 0.058500, 0.058875,
			0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500,
			0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500,
			0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875,
			0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500,
			0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500,
			0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875,
			0.058500, 0.058500, 0.058875, 0.058500, 0.058500, 0.058875, 0.058500, 0.058500
		}
	},
	{
		(char *) "8PSK250", 1.0, 2.028875,
		{
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.021438, 0.029437, 0.029437, 0.029437, 0.029437, 0.021438, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.032000, 0.032000,
			0.008000, 0.024000, 0.024000, 0.026687, 0.026687, 0.026687, 0.026687, 0.024000,
			0.024000, 0.024000, 0.026687, 0.024000, 0.021438, 0.024000, 0.024000, 0.024000,
			0.021438, 0.021438, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.026687, 0.024000, 0.026687, 0.024000,
			0.026687, 0.021438, 0.024000, 0.021438, 0.021438, 0.021438, 0.021438, 0.024000,
			0.024000, 0.021438, 0.024000, 0.024000, 0.021438, 0.021438, 0.021438, 0.021438,
			0.021438, 0.024000, 0.021438, 0.021438, 0.021438, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687,
			0.026687, 0.013437, 0.018687, 0.016000, 0.016000, 0.010688, 0.018687, 0.018687,
			0.016000, 0.013437, 0.021438, 0.018687, 0.016000, 0.018687, 0.013437, 0.013437,
			0.018687, 0.018687, 0.016000, 0.016000, 0.010688, 0.016000, 0.018687, 0.018687,
			0.018687, 0.018687, 0.018687, 0.026687, 0.026687, 0.026687, 0.026687, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687,
			0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687,
			0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687,
			0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687, 0.026687,
			0.026687, 0.026687, 0.026687, 0.026687, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437,
			0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437, 0.029437
		}
	},
	{
		(char *) "8PSK500", 1.0, 1.206000,
		{
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.008000, 0.010937, 0.010937, 0.010937, 0.010937, 0.008000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000,
			0.003000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.009000,
			0.009000, 0.009000, 0.010000, 0.008937, 0.008000, 0.009000, 0.009000, 0.009000,
			0.008000, 0.008000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.009000, 0.010000, 0.008937, 0.010000, 0.009000,
			0.010000, 0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000, 0.009000,
			0.009000, 0.008000, 0.008937, 0.008937, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008937, 0.008937, 0.008937,
			0.008937, 0.008937, 0.008937, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.004938, 0.006938, 0.006000, 0.006000, 0.004000, 0.007000, 0.007000,
			0.006000, 0.004938, 0.008000, 0.007000, 0.006000, 0.006938, 0.004938, 0.004938,
			0.006938, 0.006938, 0.006000, 0.006000, 0.004000, 0.006000, 0.006938, 0.006938,
			0.006938, 0.006938, 0.006938, 0.010000, 0.010000, 0.010000, 0.010000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000
		}
	},
	{
		(char *) "8PSK1000", 1.0, 1.089000,
		{
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.004000, 0.005500, 0.005500, 0.005500, 0.005500, 0.004000, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.006000, 0.006000,
			0.001437, 0.004437, 0.004437, 0.005000, 0.005000, 0.005000, 0.005000, 0.004437,
			0.004437, 0.004437, 0.005000, 0.004500, 0.004000, 0.004437, 0.004437, 0.004437,
			0.004000, 0.004000, 0.004437, 0.004437, 0.004437, 0.004437, 0.004437, 0.004437,
			0.004437, 0.004437, 0.004437, 0.004437, 0.005000, 0.004500, 0.005000, 0.004437,
			0.005000, 0.004000, 0.004437, 0.004000, 0.004000, 0.004000, 0.004000, 0.004437,
			0.004437, 0.004000, 0.004500, 0.004500, 0.004000, 0.004000, 0.004000, 0.004000,
			0.004000, 0.004437, 0.004000, 0.004000, 0.004000, 0.004500, 0.004500, 0.004500,
			0.004500, 0.004500, 0.004500, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.002500, 0.003500, 0.003000, 0.003000, 0.002000, 0.003438, 0.003438,
			0.003000, 0.002500, 0.004000, 0.003438, 0.003000, 0.003500, 0.002500, 0.002500,
			0.003500, 0.003500, 0.003000, 0.003000, 0.002000, 0.003000, 0.003500, 0.003500,
			0.003500, 0.003500, 0.003500, 0.005000, 0.005000, 0.005000, 0.005000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437,
			0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437, 0.005437
		}
	},
	{
		(char *) "8PSK1600", 1.0, 0.658750,
		{
			0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750,
			0.002000, 0.002750, 0.002750, 0.002687, 0.002750, 0.002000, 0.002750, 0.002750,
			0.002750, 0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750,
			0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750, 0.003000, 0.003000,
			0.000750, 0.002250, 0.002250, 0.002500, 0.002500, 0.002500, 0.002437, 0.002250,
			0.002250, 0.002250, 0.002437, 0.002250, 0.002000, 0.002250, 0.002250, 0.002188,
			0.002000, 0.002000, 0.002250, 0.002250, 0.002250, 0.002250, 0.002188, 0.002250,
			0.002250, 0.002250, 0.002250, 0.002188, 0.002500, 0.002250, 0.002500, 0.002250,
			0.002500, 0.002000, 0.002250, 0.002000, 0.002000, 0.002000, 0.002000, 0.002188,
			0.002250, 0.002000, 0.002250, 0.002250, 0.002000, 0.002000, 0.002000, 0.002000,
			0.002000, 0.002250, 0.002000, 0.002000, 0.002000, 0.002188, 0.002250, 0.002250,
			0.002250, 0.002250, 0.002188, 0.002500, 0.002500, 0.002437, 0.002500, 0.002500,
			0.002500, 0.001250, 0.001687, 0.001500, 0.001500, 0.001000, 0.001687, 0.001750,
			0.001500, 0.001187, 0.002000, 0.001750, 0.001500, 0.001750, 0.001250, 0.001187,
			0.001750, 0.001750, 0.001437, 0.001500, 0.001000, 0.001500, 0.001750, 0.001687,
			0.001750, 0.001750, 0.001750, 0.002500, 0.002500, 0.002437, 0.002500, 0.003000,
			0.003000, 0.003000, 0.003000, 0.002938, 0.003000, 0.003000, 0.003000, 0.003000,
			0.002938, 0.003000, 0.003000, 0.003000, 0.003000, 0.002938, 0.003000, 0.003000,
			0.003000, 0.003000, 0.002938, 0.003000, 0.003000, 0.003000, 0.003000, 0.002938,
			0.003000, 0.003000, 0.003000, 0.003000, 0.002938, 0.003000, 0.003000, 0.003000,
			0.002500, 0.002500, 0.002437, 0.002500, 0.002500, 0.002500, 0.002500, 0.002437,
			0.002500, 0.002500, 0.002500, 0.002500, 0.002437, 0.002500, 0.002500, 0.002500,
			0.002500, 0.002437, 0.002500, 0.002500, 0.002500, 0.002500, 0.002437, 0.002500,
			0.002500, 0.002500, 0.002500, 0.002437, 0.002500, 0.002500, 0.002500, 0.002500,
			0.002437, 0.002500, 0.002500, 0.002500, 0.002750, 0.002750, 0.002750, 0.002750,
			0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750,
			0.002750, 0.002750, 0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687,
			0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750, 0.002750,
			0.002750, 0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750,
			0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750, 0.002750, 0.002750,
			0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687, 0.002750, 0.002750,
			0.002750, 0.002750, 0.002687, 0.002750, 0.002750, 0.002750, 0.002750, 0.002687
		}
	},
	{
		(char *) "BPSK31", 1.0, 2.048000,
		{
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.320000, 0.224000, 0.384000, 0.384000, 0.224000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.096000, 0.352000, 0.352000, 0.352000, 0.352000, 0.384000, 0.384000, 0.352000,
			0.320000, 0.320000, 0.352000, 0.352000, 0.288000, 0.256000, 0.288000, 0.352000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.320000, 0.352000, 0.352000, 0.288000, 0.352000, 0.384000,
			0.384000, 0.288000, 0.320000, 0.320000, 0.320000, 0.288000, 0.320000, 0.320000,
			0.352000, 0.288000, 0.352000, 0.352000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.352000, 0.320000, 0.288000, 0.288000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.384000, 0.352000, 0.352000, 0.352000, 0.384000, 0.352000,
			0.384000, 0.192000, 0.288000, 0.256000, 0.256000, 0.128000, 0.256000, 0.288000,
			0.256000, 0.192000, 0.352000, 0.320000, 0.224000, 0.256000, 0.192000, 0.160000,
			0.256000, 0.352000, 0.224000, 0.224000, 0.160000, 0.256000, 0.288000, 0.288000,
			0.320000, 0.288000, 0.352000, 0.384000, 0.352000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000
		}
	},
	{
		(char *) "BPSK63", 1.0, 2.048000,
		{
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.160000, 0.112000, 0.192000, 0.192000, 0.112000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.048000, 0.176000, 0.176000, 0.176000, 0.176000, 0.192000, 0.192000, 0.176000,
			0.160000, 0.160000, 0.176000, 0.176000, 0.144000, 0.128000, 0.144000, 0.176000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.160000, 0.176000, 0.176000, 0.144000, 0.176000, 0.192000,
			0.192000, 0.144000, 0.160000, 0.160000, 0.160000, 0.144000, 0.160000, 0.160000,
			0.176000, 0.144000, 0.176000, 0.176000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.176000, 0.160000, 0.144000, 0.144000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.192000, 0.176000, 0.176000, 0.176000, 0.192000, 0.176000,
			0.192000, 0.096000, 0.144000, 0.128000, 0.128000, 0.064000, 0.128000, 0.144000,
			0.128000, 0.096000, 0.176000, 0.160000, 0.112000, 0.128000, 0.096000, 0.080000,
			0.128000, 0.176000, 0.112000, 0.112000, 0.080000, 0.128000, 0.144000, 0.144000,
			0.160000, 0.144000, 0.176000, 0.192000, 0.176000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000
		}
	},
	{
		(char *) "BPSK63F", 1.0, 4.096000,
		{
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.256000, 0.352000, 0.352000, 0.352000, 0.352000, 0.256000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.384000, 0.384000,
			0.096000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.288000,
			0.288000, 0.288000, 0.320000, 0.288000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.256000, 0.256000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.288000, 0.320000, 0.288000, 0.320000, 0.288000,
			0.320000, 0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000, 0.288000,
			0.288000, 0.256000, 0.288000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.160000, 0.224000, 0.192000, 0.192000, 0.128000, 0.224000, 0.224000,
			0.192000, 0.160000, 0.256000, 0.224000, 0.192000, 0.224000, 0.160000, 0.160000,
			0.224000, 0.224000, 0.192000, 0.192000, 0.128000, 0.192000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.320000, 0.320000, 0.320000, 0.320000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000
		}
	},
	{
		(char *) "BPSK125", 1.0, 2.048000,
		{
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.080000, 0.056000, 0.096000, 0.096000, 0.056000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.024000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000, 0.088000,
			0.080000, 0.080000, 0.088000, 0.088000, 0.072000, 0.064000, 0.072000, 0.088000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.080000, 0.088000, 0.088000, 0.072000, 0.088000, 0.096000,
			0.096000, 0.072000, 0.080000, 0.080000, 0.080000, 0.072000, 0.080000, 0.080000,
			0.088000, 0.072000, 0.088000, 0.088000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.088000, 0.080000, 0.072000, 0.072000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.096000, 0.088000, 0.088000, 0.088000, 0.096000, 0.088000,
			0.096000, 0.048000, 0.072000, 0.064000, 0.064000, 0.032000, 0.064000, 0.072000,
			0.064000, 0.048000, 0.088000, 0.080000, 0.056000, 0.064000, 0.048000, 0.040000,
			0.064000, 0.088000, 0.056000, 0.056000, 0.040000, 0.064000, 0.072000, 0.072000,
			0.080000, 0.072000, 0.088000, 0.096000, 0.088000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000
		}
	},
	{
		(char *) "BPSK250", 1.0, 2.048000,
		{
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.040000, 0.028000, 0.048000, 0.048000, 0.028000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.012000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000, 0.044000,
			0.040000, 0.040000, 0.044000, 0.044000, 0.036000, 0.032000, 0.036000, 0.044000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.040000, 0.044000, 0.044000, 0.036000, 0.044000, 0.048000,
			0.048000, 0.036000, 0.040000, 0.040000, 0.040000, 0.036000, 0.040000, 0.040000,
			0.044000, 0.036000, 0.044000, 0.044000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.044000, 0.040000, 0.036000, 0.036000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.048000, 0.044000, 0.044000, 0.044000, 0.048000, 0.044000,
			0.048000, 0.024000, 0.036000, 0.032000, 0.032000, 0.016000, 0.032000, 0.036000,
			0.032000, 0.024000, 0.044000, 0.040000, 0.028000, 0.032000, 0.024000, 0.020000,
			0.032000, 0.044000, 0.028000, 0.028000, 0.020000, 0.032000, 0.036000, 0.036000,
			0.040000, 0.036000, 0.044000, 0.048000, 0.044000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000
		}
	},
	{
		(char *) "BPSK500", 1.0, 2.048000,
		{
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.020000, 0.014000, 0.024000, 0.024000, 0.014000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.006000, 0.022000, 0.022000, 0.022000, 0.022000, 0.024000, 0.024000, 0.022000,
			0.020000, 0.020000, 0.022000, 0.022000, 0.018000, 0.016000, 0.018000, 0.022000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.020000, 0.022000, 0.022000, 0.018000, 0.022000, 0.024000,
			0.024000, 0.018000, 0.020000, 0.020000, 0.020000, 0.018000, 0.020000, 0.020000,
			0.022000, 0.018000, 0.022000, 0.022000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.022000, 0.020000, 0.018000, 0.018000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.024000, 0.022000, 0.022000, 0.022000, 0.024000, 0.022000,
			0.024000, 0.012000, 0.018000, 0.016000, 0.016000, 0.008000, 0.016000, 0.018000,
			0.016000, 0.012000, 0.022000, 0.020000, 0.014000, 0.016000, 0.012000, 0.010000,
			0.016000, 0.022000, 0.014000, 0.014000, 0.010000, 0.016000, 0.018000, 0.018000,
			0.020000, 0.018000, 0.022000, 0.024000, 0.022000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000
		}
	},
	{
		(char *) "BPSK1000", 1.0, 0.256000,
		{
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.010000, 0.007000, 0.012000, 0.012000, 0.007000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.003000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000, 0.011000,
			0.010000, 0.010000, 0.011000, 0.011000, 0.009000, 0.008000, 0.009000, 0.011000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.010000, 0.011000, 0.011000, 0.009000, 0.011000, 0.012000,
			0.012000, 0.009000, 0.010000, 0.010000, 0.010000, 0.009000, 0.010000, 0.010000,
			0.011000, 0.009000, 0.011000, 0.011000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.011000, 0.010000, 0.009000, 0.009000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.012000, 0.011000, 0.011000, 0.011000, 0.012000, 0.011000,
			0.012000, 0.006000, 0.009000, 0.008000, 0.008000, 0.004000, 0.008000, 0.009000,
			0.008000, 0.006000, 0.011000, 0.010000, 0.007000, 0.008000, 0.006000, 0.005000,
			0.008000, 0.011000, 0.007000, 0.007000, 0.005000, 0.008000, 0.009000, 0.009000,
			0.010000, 0.009000, 0.011000, 0.012000, 0.011000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000
		}
	},
	{
		(char *) "DOMX22", 1.0, 1.346757,
		{
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.092880, 0.139320, 0.139320, 0.139320, 0.139320, 0.092880, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.046440, 0.092880, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.092880,
			0.139320, 0.139320, 0.139320, 0.139320, 0.092880, 0.092880, 0.092880, 0.139320,
			0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880,
			0.092880, 0.092880, 0.139320, 0.139320, 0.139320, 0.092880, 0.139320, 0.092880,
			0.139320, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880,
			0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880,
			0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880, 0.092880,
			0.092880, 0.092880, 0.092880, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.046440, 0.092880, 0.092880, 0.092880, 0.046440, 0.092880, 0.092880,
			0.092880, 0.046440, 0.092880, 0.092880, 0.092880, 0.092880, 0.046440, 0.046440,
			0.092880, 0.092880, 0.046440, 0.092880, 0.046440, 0.092880, 0.092880, 0.092880,
			0.092880, 0.092880, 0.092880, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320,
			0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320, 0.139320
		}
	},
	{
		(char *) "DOMX44", 1.0, 0.673379,
		{
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.046440, 0.069660, 0.069660, 0.069660, 0.069660, 0.046440, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.023220, 0.046440, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.046440,
			0.069660, 0.069660, 0.069660, 0.069660, 0.046440, 0.046440, 0.046440, 0.069660,
			0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440,
			0.046440, 0.046440, 0.069660, 0.069660, 0.069660, 0.046440, 0.069660, 0.046440,
			0.069660, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440,
			0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440,
			0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440, 0.046440,
			0.046440, 0.046440, 0.046440, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.023220, 0.046440, 0.046440, 0.046440, 0.023220, 0.046440, 0.046440,
			0.046440, 0.023220, 0.046440, 0.046440, 0.046440, 0.046440, 0.023220, 0.023220,
			0.046440, 0.046440, 0.023220, 0.046440, 0.023220, 0.046440, 0.046440, 0.046440,
			0.046440, 0.046440, 0.046440, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660,
			0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660, 0.069660
		}
	},
	{
		(char *) "DOMX88", 1.0, 0.336689,
		{
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.023220, 0.034830, 0.034830, 0.034830, 0.034830, 0.023220, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.011610, 0.023220, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.023220,
			0.034830, 0.034830, 0.034830, 0.034830, 0.023220, 0.023220, 0.023220, 0.034830,
			0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220,
			0.023220, 0.023220, 0.034830, 0.034830, 0.034830, 0.023220, 0.034830, 0.023220,
			0.034830, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220,
			0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220,
			0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220, 0.023220,
			0.023220, 0.023220, 0.023220, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.011610, 0.023220, 0.023220, 0.023220, 0.011610, 0.023220, 0.023220,
			0.023220, 0.011610, 0.023220, 0.023220, 0.023220, 0.023220, 0.011610, 0.011610,
			0.023220, 0.023220, 0.011610, 0.023220, 0.011610, 0.023220, 0.023220, 0.023220,
			0.023220, 0.023220, 0.023220, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830,
			0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830, 0.034830
		}
	},
	{
		(char *) "MFSK16", 1.0, 6.336000,
		{
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.256000, 0.352000, 0.352000, 0.352000, 0.352000, 0.256000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.384000, 0.384000,
			0.096000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.288000,
			0.288000, 0.288000, 0.320000, 0.288000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.256000, 0.256000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.288000, 0.320000, 0.288000, 0.320000, 0.288000,
			0.320000, 0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000, 0.288000,
			0.288000, 0.256000, 0.288000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.160000, 0.224000, 0.192000, 0.192000, 0.128000, 0.224000, 0.224000,
			0.192000, 0.160000, 0.256000, 0.224000, 0.192000, 0.224000, 0.160000, 0.160000,
			0.224000, 0.224000, 0.192000, 0.192000, 0.128000, 0.192000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.320000, 0.320000, 0.320000, 0.320000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000
		}
	},
	{
		(char *) "MFSK22", 1.0, 4.597551,
		{
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.185760, 0.255420, 0.255420, 0.255420, 0.255420, 0.185760, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.278639, 0.278639,
			0.069660, 0.208980, 0.208980, 0.232200, 0.232200, 0.232200, 0.232200, 0.208980,
			0.208980, 0.208980, 0.232200, 0.208980, 0.185760, 0.208980, 0.208980, 0.208980,
			0.185760, 0.185760, 0.208980, 0.208980, 0.208980, 0.208980, 0.208980, 0.208980,
			0.208980, 0.208980, 0.208980, 0.208980, 0.232200, 0.208980, 0.232200, 0.208980,
			0.232200, 0.185760, 0.208980, 0.185760, 0.185760, 0.185760, 0.185760, 0.208980,
			0.208980, 0.185760, 0.208980, 0.208980, 0.185760, 0.185760, 0.185760, 0.185760,
			0.185760, 0.208980, 0.185760, 0.185760, 0.185760, 0.208980, 0.208980, 0.208980,
			0.208980, 0.208980, 0.208980, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.116100, 0.162540, 0.139320, 0.139320, 0.092880, 0.162540, 0.162540,
			0.139320, 0.116100, 0.185760, 0.162540, 0.139320, 0.162540, 0.116100, 0.116100,
			0.162540, 0.162540, 0.139320, 0.139320, 0.092880, 0.139320, 0.162540, 0.162540,
			0.162540, 0.162540, 0.162540, 0.232200, 0.232200, 0.232200, 0.232200, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420
		}
	},
	{
		(char *) "MFSK31", 1.0, 4.214000,
		{
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.170625, 0.234625, 0.234625, 0.234625, 0.234625, 0.170625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.256000, 0.256000,
			0.064000, 0.192000, 0.192000, 0.213250, 0.213250, 0.213250, 0.213250, 0.192000,
			0.192000, 0.192000, 0.213250, 0.192000, 0.170625, 0.192000, 0.192000, 0.192000,
			0.170625, 0.170625, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.213250, 0.192000, 0.213250, 0.192000,
			0.213250, 0.170625, 0.192000, 0.170625, 0.170625, 0.170625, 0.170625, 0.192000,
			0.192000, 0.170625, 0.192000, 0.192000, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.192000, 0.170625, 0.170625, 0.170625, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250,
			0.213250, 0.106625, 0.149250, 0.128000, 0.128000, 0.085250, 0.149250, 0.149250,
			0.128000, 0.106625, 0.170625, 0.149250, 0.128000, 0.149250, 0.106625, 0.106625,
			0.149250, 0.149250, 0.128000, 0.128000, 0.085250, 0.128000, 0.149250, 0.149250,
			0.149250, 0.149250, 0.149250, 0.213250, 0.213250, 0.213250, 0.213250, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250,
			0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250,
			0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250,
			0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250, 0.213250,
			0.213250, 0.213250, 0.213250, 0.213250, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625,
			0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625, 0.234625
		}
	},
	{
		(char *) "MFSK32", 1.0, 3.168000,
		{
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.128000, 0.176000, 0.176000, 0.176000, 0.176000, 0.128000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.192000, 0.192000,
			0.048000, 0.144000, 0.144000, 0.160000, 0.160000, 0.160000, 0.160000, 0.144000,
			0.144000, 0.144000, 0.160000, 0.144000, 0.128000, 0.144000, 0.144000, 0.144000,
			0.128000, 0.128000, 0.144000, 0.144000, 0.144000, 0.144000, 0.144000, 0.144000,
			0.144000, 0.144000, 0.144000, 0.144000, 0.160000, 0.144000, 0.160000, 0.144000,
			0.160000, 0.128000, 0.144000, 0.128000, 0.128000, 0.128000, 0.128000, 0.144000,
			0.144000, 0.128000, 0.144000, 0.144000, 0.128000, 0.128000, 0.128000, 0.128000,
			0.128000, 0.144000, 0.128000, 0.128000, 0.128000, 0.144000, 0.144000, 0.144000,
			0.144000, 0.144000, 0.144000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.080000, 0.112000, 0.096000, 0.096000, 0.064000, 0.112000, 0.112000,
			0.096000, 0.080000, 0.128000, 0.112000, 0.096000, 0.112000, 0.080000, 0.080000,
			0.112000, 0.112000, 0.096000, 0.096000, 0.064000, 0.096000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.160000, 0.160000, 0.160000, 0.160000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000
		}
	},
	{
		(char *) "MFSK64", 1.0, 2.352000,
		{
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.064000, 0.088000, 0.088000, 0.088000, 0.088000, 0.064000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000,
			0.024000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.072000,
			0.072000, 0.072000, 0.080000, 0.072000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.064000, 0.064000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.072000, 0.080000, 0.072000, 0.080000, 0.072000,
			0.080000, 0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000, 0.072000,
			0.072000, 0.064000, 0.072000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.040000, 0.056000, 0.048000, 0.048000, 0.032000, 0.056000, 0.056000,
			0.048000, 0.040000, 0.064000, 0.056000, 0.048000, 0.056000, 0.040000, 0.040000,
			0.056000, 0.056000, 0.048000, 0.048000, 0.032000, 0.048000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.080000, 0.080000, 0.080000, 0.080000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000
		}
	},
	{
		(char *) "MFSK64L", 1.0, 20.432000,
		{
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.064000, 0.088000, 0.088000, 0.088000, 0.088000, 0.064000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000,
			0.024000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.072000,
			0.072000, 0.072000, 0.080000, 0.072000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.064000, 0.064000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.072000, 0.080000, 0.072000, 0.080000, 0.072000,
			0.080000, 0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000, 0.072000,
			0.072000, 0.064000, 0.072000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.040000, 0.056000, 0.048000, 0.048000, 0.032000, 0.056000, 0.056000,
			0.048000, 0.040000, 0.064000, 0.056000, 0.048000, 0.056000, 0.040000, 0.040000,
			0.056000, 0.056000, 0.048000, 0.048000, 0.032000, 0.048000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.080000, 0.080000, 0.080000, 0.080000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000
		}
	},
	{
		(char *) "MFSK128", 1.0, 1.360000,
		{
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.032000, 0.044000, 0.044000, 0.044000, 0.044000, 0.032000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000,
			0.012000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.036000,
			0.036000, 0.036000, 0.040000, 0.036000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.032000, 0.032000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.036000, 0.040000, 0.036000, 0.040000, 0.036000,
			0.040000, 0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000, 0.036000,
			0.036000, 0.032000, 0.036000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.020000, 0.028000, 0.024000, 0.024000, 0.016000, 0.028000, 0.028000,
			0.024000, 0.020000, 0.032000, 0.028000, 0.024000, 0.028000, 0.020000, 0.020000,
			0.028000, 0.028000, 0.024000, 0.024000, 0.016000, 0.024000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.040000, 0.040000, 0.040000, 0.040000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000
		}
	},
	{
		(char *) "MFSK128L", 1.0, 20.216000,
		{
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.032000, 0.044000, 0.044000, 0.044000, 0.044000, 0.032000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000,
			0.012000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.036000,
			0.036000, 0.036000, 0.040000, 0.036000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.032000, 0.032000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.036000, 0.040000, 0.036000, 0.040000, 0.036000,
			0.040000, 0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000, 0.036000,
			0.036000, 0.032000, 0.036000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.020000, 0.028000, 0.024000, 0.024000, 0.016000, 0.028000, 0.028000,
			0.024000, 0.020000, 0.032000, 0.028000, 0.024000, 0.028000, 0.020000, 0.020000,
			0.028000, 0.028000, 0.024000, 0.024000, 0.016000, 0.024000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.040000, 0.040000, 0.040000, 0.040000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000
		}
	},
	{
		(char *) "MT63-500L", 1.0, 15.000000,
		{
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000
		}
	},
	{
		(char *) "MT63-500S", 1.0, 8.600000,
		{
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000,
			0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000, 0.400000
		}
	},
	{
		(char *) "MT63-1KL", 1.0, 8.500000,
		{
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000
		}
	},
	{
		(char *) "MT63-1KS", 1.0, 5.300000,
		{
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000
		}
	},
	{
		(char *) "MT63-2KL", 1.0, 5.300000,
		{
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000
		}
	},
	{
		(char *) "MT63-2KS", 1.0, 3.700000,
		{
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000
		}
	},
	{
		(char *) "Olivia-4-250", 1.0, 3.088000,
		{
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000
		}
	},
	{
		(char *) "Olivia-4-500", 1.0, 2.568000,
		{
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000
		}
	},
	{
		(char *) "Olivia-8-250", 1.0, 3.446000,
		{
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250,
			1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250, 1.365250
		}
	},
	{
		(char *) "Olivia-8-500", 1.0, 2.748000,
		{
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625
		}
	},
	{
		(char *) "Olivia-8-1K", 1.0, 2.398000,
		{
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625, 0.170625,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250
		}
	},
	{
		(char *) "Olivia-16-500", 1.0, 4.128000,
		{
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000,
			1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000, 1.024000
		}
	},
	{
		(char *) "Olivia-16-1K", 1.0, 3.088000,
		{
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000,
			0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000, 0.512000
		}
	},
	{
		(char *) "Olivia-32-1K", 1.0, 4.812000,
		{
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625
		}
	},
	{
		(char *) "Olivia-64-2K", 1.0, 4.812000,
		{
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250, 0.341250,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625,
			0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625, 0.682625
		}
	},
	{
		(char *) "PSK63RC4", 1.0, 2.048000,
		{
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.064000, 0.088000, 0.088000, 0.088000, 0.088000, 0.064000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000,
			0.024000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.072000,
			0.072000, 0.072000, 0.080000, 0.072000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.064000, 0.064000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.072000, 0.080000, 0.072000, 0.080000, 0.072000,
			0.080000, 0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000, 0.072000,
			0.072000, 0.064000, 0.072000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.040000, 0.056000, 0.048000, 0.048000, 0.032000, 0.056000, 0.056000,
			0.048000, 0.040000, 0.064000, 0.056000, 0.048000, 0.056000, 0.040000, 0.040000,
			0.056000, 0.056000, 0.048000, 0.048000, 0.032000, 0.048000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.080000, 0.080000, 0.080000, 0.080000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000
		}
	},
	{
		(char *) "PSK63RC5", 1.0, 6.572750,
		{
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.051125, 0.070375, 0.070375, 0.070375, 0.070375, 0.051125, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.076750, 0.076750,
			0.019125, 0.057500, 0.057500, 0.064000, 0.064000, 0.064000, 0.064000, 0.057500,
			0.057500, 0.057500, 0.064000, 0.057500, 0.051125, 0.057500, 0.057500, 0.057500,
			0.051125, 0.051125, 0.057500, 0.057500, 0.057500, 0.057500, 0.057500, 0.057500,
			0.057500, 0.057500, 0.057500, 0.057500, 0.064000, 0.057500, 0.064000, 0.057500,
			0.064000, 0.051125, 0.057500, 0.051125, 0.051125, 0.051125, 0.051125, 0.057500,
			0.057500, 0.051125, 0.057500, 0.057500, 0.051125, 0.051125, 0.051125, 0.051125,
			0.051125, 0.057500, 0.051125, 0.051125, 0.051125, 0.057500, 0.057500, 0.057500,
			0.057500, 0.057500, 0.057500, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.032000, 0.044750, 0.038375, 0.038375, 0.025500, 0.044750, 0.044750,
			0.038375, 0.032000, 0.051125, 0.044750, 0.038375, 0.044750, 0.032000, 0.032000,
			0.044750, 0.044750, 0.038375, 0.038375, 0.025500, 0.038375, 0.044750, 0.044750,
			0.044750, 0.044750, 0.044750, 0.064000, 0.064000, 0.064000, 0.064000, 0.076750,
			0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750,
			0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750,
			0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750,
			0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750, 0.076750,
			0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.064000, 0.064000, 0.064000, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375,
			0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375, 0.070375
		}
	},
	{
		(char *) "PSK63RC10", 1.0, 3.286375,
		{
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.025500, 0.035125, 0.035125, 0.035125, 0.035125, 0.025500, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.038375, 0.038375,
			0.009500, 0.028750, 0.028750, 0.032000, 0.032000, 0.032000, 0.032000, 0.028750,
			0.028750, 0.028750, 0.032000, 0.028750, 0.025500, 0.028750, 0.028750, 0.028750,
			0.025500, 0.025500, 0.028750, 0.028750, 0.028750, 0.028750, 0.028750, 0.028750,
			0.028750, 0.028750, 0.028750, 0.028750, 0.032000, 0.028750, 0.032000, 0.028750,
			0.032000, 0.025500, 0.028750, 0.025500, 0.025500, 0.025500, 0.025500, 0.028750,
			0.028750, 0.025500, 0.028750, 0.028750, 0.025500, 0.025500, 0.025500, 0.025500,
			0.025500, 0.028750, 0.025500, 0.025500, 0.025500, 0.028750, 0.028750, 0.028750,
			0.028750, 0.028750, 0.028750, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.016000, 0.022375, 0.019125, 0.019125, 0.012750, 0.022375, 0.022375,
			0.019125, 0.016000, 0.025500, 0.022375, 0.019125, 0.022375, 0.016000, 0.016000,
			0.022375, 0.022375, 0.019125, 0.019125, 0.012750, 0.019125, 0.022375, 0.022375,
			0.022375, 0.022375, 0.022375, 0.032000, 0.032000, 0.032000, 0.032000, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125
		}
	},
	{
		(char *) "PSK63RC20", 1.0, 1.651125,
		{
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.012750, 0.017500, 0.017500, 0.017500, 0.017500, 0.012750, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.019125, 0.019125,
			0.004750, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.014375,
			0.014375, 0.014375, 0.016000, 0.014375, 0.012750, 0.014375, 0.014375, 0.014375,
			0.012750, 0.012750, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.014375, 0.016000, 0.014375, 0.016000, 0.014375,
			0.016000, 0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750, 0.014375,
			0.014375, 0.012750, 0.014375, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750,
			0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.008000, 0.011125, 0.009500, 0.009500, 0.006375, 0.011125, 0.011125,
			0.009500, 0.008000, 0.012750, 0.011125, 0.009500, 0.011125, 0.008000, 0.008000,
			0.011125, 0.011125, 0.009500, 0.009500, 0.006375, 0.009500, 0.011125, 0.011125,
			0.011125, 0.011125, 0.011125, 0.016000, 0.016000, 0.016000, 0.016000, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500
		}
	},
	{
		(char *) "PSK63RC32", 1.0, 1.024000,
		{
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.008000, 0.011000, 0.011000, 0.011000, 0.011000, 0.008000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000,
			0.003000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.009000,
			0.009000, 0.009000, 0.010000, 0.009000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.008000, 0.008000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.009000, 0.010000, 0.009000, 0.010000, 0.009000,
			0.010000, 0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000, 0.009000,
			0.009000, 0.008000, 0.009000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.005000, 0.007000, 0.006000, 0.006000, 0.004000, 0.007000, 0.007000,
			0.006000, 0.005000, 0.008000, 0.007000, 0.006000, 0.007000, 0.005000, 0.005000,
			0.007000, 0.007000, 0.006000, 0.006000, 0.004000, 0.006000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.010000, 0.010000, 0.010000, 0.010000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000
		}
	},
	{
		(char *) "PSK125R", 1.0, 4.096000,
		{
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.128000, 0.176000, 0.176000, 0.176000, 0.176000, 0.128000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.192000, 0.192000,
			0.048000, 0.144000, 0.144000, 0.160000, 0.160000, 0.160000, 0.160000, 0.144000,
			0.144000, 0.144000, 0.160000, 0.144000, 0.128000, 0.144000, 0.144000, 0.144000,
			0.128000, 0.128000, 0.144000, 0.144000, 0.144000, 0.144000, 0.144000, 0.144000,
			0.144000, 0.144000, 0.144000, 0.144000, 0.160000, 0.144000, 0.160000, 0.144000,
			0.160000, 0.128000, 0.144000, 0.128000, 0.128000, 0.128000, 0.128000, 0.144000,
			0.144000, 0.128000, 0.144000, 0.144000, 0.128000, 0.128000, 0.128000, 0.128000,
			0.128000, 0.144000, 0.128000, 0.128000, 0.128000, 0.144000, 0.144000, 0.144000,
			0.144000, 0.144000, 0.144000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.080000, 0.112000, 0.096000, 0.096000, 0.064000, 0.112000, 0.112000,
			0.096000, 0.080000, 0.128000, 0.112000, 0.096000, 0.112000, 0.080000, 0.080000,
			0.112000, 0.112000, 0.096000, 0.096000, 0.064000, 0.096000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.160000, 0.160000, 0.160000, 0.160000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000, 0.176000
		}
	},
	{
		(char *) "PSK125RC4", 1.0, 4.096000,
		{
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.032000, 0.044000, 0.044000, 0.044000, 0.044000, 0.032000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000,
			0.012000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.036000,
			0.036000, 0.036000, 0.040000, 0.036000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.032000, 0.032000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.036000, 0.040000, 0.036000, 0.040000, 0.036000,
			0.040000, 0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000, 0.036000,
			0.036000, 0.032000, 0.036000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.020000, 0.028000, 0.024000, 0.024000, 0.016000, 0.028000, 0.028000,
			0.024000, 0.020000, 0.032000, 0.028000, 0.024000, 0.028000, 0.020000, 0.020000,
			0.028000, 0.028000, 0.024000, 0.024000, 0.016000, 0.024000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.040000, 0.040000, 0.040000, 0.040000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000
		}
	},
	{
		(char *) "PSK125RC5", 1.0, 3.286375,
		{
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.025500, 0.035125, 0.035125, 0.035125, 0.035125, 0.025500, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.038375, 0.038375,
			0.009500, 0.028750, 0.028750, 0.032000, 0.032000, 0.032000, 0.032000, 0.028750,
			0.028750, 0.028750, 0.032000, 0.028750, 0.025500, 0.028750, 0.028750, 0.028750,
			0.025500, 0.025500, 0.028750, 0.028750, 0.028750, 0.028750, 0.028750, 0.028750,
			0.028750, 0.028750, 0.028750, 0.028750, 0.032000, 0.028750, 0.032000, 0.028750,
			0.032000, 0.025500, 0.028750, 0.025500, 0.025500, 0.025500, 0.025500, 0.028750,
			0.028750, 0.025500, 0.028750, 0.028750, 0.025500, 0.025500, 0.025500, 0.025500,
			0.025500, 0.028750, 0.025500, 0.025500, 0.025500, 0.028750, 0.028750, 0.028750,
			0.028750, 0.028750, 0.028750, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.016000, 0.022375, 0.019125, 0.019125, 0.012750, 0.022375, 0.022375,
			0.019125, 0.016000, 0.025500, 0.022375, 0.019125, 0.022375, 0.016000, 0.016000,
			0.022375, 0.022375, 0.019125, 0.019125, 0.012750, 0.019125, 0.022375, 0.022375,
			0.022375, 0.022375, 0.022375, 0.032000, 0.032000, 0.032000, 0.032000, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375, 0.038375,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125,
			0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125, 0.035125
		}
	},
	{
		(char *) "PSK125C12", 1.0, 0.170625,
		{
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.006625, 0.004625, 0.008000, 0.008000, 0.004625, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.002000, 0.007250, 0.007250, 0.007250, 0.007250, 0.008000, 0.008000, 0.007250,
			0.006625, 0.006625, 0.007250, 0.007250, 0.006000, 0.005250, 0.006000, 0.007250,
			0.006625, 0.006625, 0.006625, 0.006625, 0.007250, 0.007250, 0.007250, 0.007250,
			0.007250, 0.007250, 0.006625, 0.007250, 0.007250, 0.006000, 0.007250, 0.008000,
			0.008000, 0.006000, 0.006625, 0.006625, 0.006625, 0.006000, 0.006625, 0.006625,
			0.007250, 0.006000, 0.007250, 0.007250, 0.006625, 0.006625, 0.006625, 0.006625,
			0.006625, 0.007250, 0.006625, 0.006000, 0.006000, 0.007250, 0.007250, 0.007250,
			0.007250, 0.007250, 0.008000, 0.007250, 0.007250, 0.007250, 0.008000, 0.007250,
			0.008000, 0.004000, 0.006000, 0.005250, 0.005250, 0.002625, 0.005250, 0.006000,
			0.005250, 0.004000, 0.007250, 0.006625, 0.004625, 0.005250, 0.004000, 0.003250,
			0.005250, 0.007250, 0.004625, 0.004625, 0.003250, 0.005250, 0.006000, 0.006000,
			0.006625, 0.006000, 0.007250, 0.008000, 0.007250, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250
		}
	},
	{
		(char *) "PSK125RC10", 1.0, 1.643125,
		{
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.012750, 0.017500, 0.017500, 0.017500, 0.017500, 0.012750, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.019125, 0.019125,
			0.004750, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.014375,
			0.014375, 0.014375, 0.016000, 0.014375, 0.012750, 0.014375, 0.014375, 0.014375,
			0.012750, 0.012750, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.014375, 0.016000, 0.014375, 0.016000, 0.014375,
			0.016000, 0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750, 0.014375,
			0.014375, 0.012750, 0.014375, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750,
			0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.008000, 0.011125, 0.009500, 0.009500, 0.006375, 0.011125, 0.011125,
			0.009500, 0.008000, 0.012750, 0.011125, 0.009500, 0.011125, 0.008000, 0.008000,
			0.011125, 0.011125, 0.009500, 0.009500, 0.006375, 0.009500, 0.011125, 0.011125,
			0.011125, 0.011125, 0.011125, 0.016000, 0.016000, 0.016000, 0.016000, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500
		}
	},
	{
		(char *) "PSK125RC12", 1.0, 1.370625,
		{
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.010625, 0.014625, 0.014625, 0.014625, 0.014625, 0.010625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.016000, 0.016000,
			0.004000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.012000,
			0.012000, 0.012000, 0.013250, 0.012000, 0.010625, 0.012000, 0.012000, 0.012000,
			0.010625, 0.010625, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.013250, 0.012000, 0.013250, 0.012000,
			0.013250, 0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625, 0.012000,
			0.012000, 0.010625, 0.012000, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625,
			0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.006625, 0.009250, 0.008000, 0.008000, 0.005250, 0.009250, 0.009250,
			0.008000, 0.006625, 0.010625, 0.009250, 0.008000, 0.009250, 0.006625, 0.006625,
			0.009250, 0.009250, 0.008000, 0.008000, 0.005250, 0.008000, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.013250, 0.013250, 0.013250, 0.013250, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625
		}
	},
	{
		(char *) "PSK125RC16", 1.0, 1.024000,
		{
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.008000, 0.011000, 0.011000, 0.011000, 0.011000, 0.008000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000,
			0.003000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.009000,
			0.009000, 0.009000, 0.010000, 0.009000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.008000, 0.008000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.009000, 0.010000, 0.009000, 0.010000, 0.009000,
			0.010000, 0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000, 0.009000,
			0.009000, 0.008000, 0.009000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.005000, 0.007000, 0.006000, 0.006000, 0.004000, 0.007000, 0.007000,
			0.006000, 0.005000, 0.008000, 0.007000, 0.006000, 0.007000, 0.005000, 0.005000,
			0.007000, 0.007000, 0.006000, 0.006000, 0.004000, 0.006000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.010000, 0.010000, 0.010000, 0.010000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000
		}
	},
	{
		(char *) "PSK250R", 1.0, 4.096000,
		{
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.064000, 0.088000, 0.088000, 0.088000, 0.088000, 0.064000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000,
			0.024000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.072000,
			0.072000, 0.072000, 0.080000, 0.072000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.064000, 0.064000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.072000, 0.080000, 0.072000, 0.080000, 0.072000,
			0.080000, 0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000, 0.072000,
			0.072000, 0.064000, 0.072000, 0.072000, 0.064000, 0.064000, 0.064000, 0.064000,
			0.064000, 0.072000, 0.064000, 0.064000, 0.064000, 0.072000, 0.072000, 0.072000,
			0.072000, 0.072000, 0.072000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.040000, 0.056000, 0.048000, 0.048000, 0.032000, 0.056000, 0.056000,
			0.048000, 0.040000, 0.064000, 0.056000, 0.048000, 0.056000, 0.040000, 0.040000,
			0.056000, 0.056000, 0.048000, 0.048000, 0.032000, 0.048000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.080000, 0.080000, 0.080000, 0.080000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000, 0.088000
		}
	},
	{
		(char *) "PSK250C6", 1.0, 0.682625,
		{
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.006625, 0.004625, 0.008000, 0.008000, 0.004625, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.002000, 0.007250, 0.007250, 0.007250, 0.007250, 0.008000, 0.008000, 0.007250,
			0.006625, 0.006625, 0.007250, 0.007250, 0.006000, 0.005250, 0.006000, 0.007250,
			0.006625, 0.006625, 0.006625, 0.006625, 0.007250, 0.007250, 0.007250, 0.007250,
			0.007250, 0.007250, 0.006625, 0.007250, 0.007250, 0.006000, 0.007250, 0.008000,
			0.008000, 0.006000, 0.006625, 0.006625, 0.006625, 0.006000, 0.006625, 0.006625,
			0.007250, 0.006000, 0.007250, 0.007250, 0.006625, 0.006625, 0.006625, 0.006625,
			0.006625, 0.007250, 0.006625, 0.006000, 0.006000, 0.007250, 0.007250, 0.007250,
			0.007250, 0.007250, 0.008000, 0.007250, 0.007250, 0.007250, 0.008000, 0.007250,
			0.008000, 0.004000, 0.006000, 0.005250, 0.005250, 0.002625, 0.005250, 0.006000,
			0.005250, 0.004000, 0.007250, 0.006625, 0.004625, 0.005250, 0.004000, 0.003250,
			0.005250, 0.007250, 0.004625, 0.004625, 0.003250, 0.005250, 0.006000, 0.006000,
			0.006625, 0.006000, 0.007250, 0.008000, 0.007250, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008000, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625, 0.008625,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250, 0.009250
		}
	},
	{
		(char *) "PSK250RC2", 1.0, 4.096000,
		{
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.032000, 0.044000, 0.044000, 0.044000, 0.044000, 0.032000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000,
			0.012000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.036000,
			0.036000, 0.036000, 0.040000, 0.036000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.032000, 0.032000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.036000, 0.040000, 0.036000, 0.040000, 0.036000,
			0.040000, 0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000, 0.036000,
			0.036000, 0.032000, 0.036000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.020000, 0.028000, 0.024000, 0.024000, 0.016000, 0.028000, 0.028000,
			0.024000, 0.020000, 0.032000, 0.028000, 0.024000, 0.028000, 0.020000, 0.020000,
			0.028000, 0.028000, 0.024000, 0.024000, 0.016000, 0.024000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.040000, 0.040000, 0.040000, 0.040000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000
		}
	},
	{
		(char *) "PSK250RC3", 1.0, 2.733250,
		{
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.021250, 0.029250, 0.029250, 0.029250, 0.029250, 0.021250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.032000, 0.032000,
			0.008000, 0.024000, 0.024000, 0.026625, 0.026625, 0.026625, 0.026625, 0.024000,
			0.024000, 0.024000, 0.026625, 0.024000, 0.021250, 0.024000, 0.024000, 0.024000,
			0.021250, 0.021250, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.026625, 0.024000, 0.026625, 0.024000,
			0.026625, 0.021250, 0.024000, 0.021250, 0.021250, 0.021250, 0.021250, 0.024000,
			0.024000, 0.021250, 0.024000, 0.024000, 0.021250, 0.021250, 0.021250, 0.021250,
			0.021250, 0.024000, 0.021250, 0.021250, 0.021250, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625,
			0.026625, 0.013250, 0.018625, 0.016000, 0.016000, 0.010625, 0.018625, 0.018625,
			0.016000, 0.013250, 0.021250, 0.018625, 0.016000, 0.018625, 0.013250, 0.013250,
			0.018625, 0.018625, 0.016000, 0.016000, 0.010625, 0.016000, 0.018625, 0.018625,
			0.018625, 0.018625, 0.018625, 0.026625, 0.026625, 0.026625, 0.026625, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625,
			0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625,
			0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625,
			0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625, 0.026625,
			0.026625, 0.026625, 0.026625, 0.026625, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250,
			0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250, 0.029250
		}
	},
	{
		(char *) "PSK250RC5", 1.0, 3.278375,
		{
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.012750, 0.017500, 0.017500, 0.017500, 0.017500, 0.012750, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.019125, 0.019125,
			0.004750, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.014375,
			0.014375, 0.014375, 0.016000, 0.014375, 0.012750, 0.014375, 0.014375, 0.014375,
			0.012750, 0.012750, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.014375, 0.016000, 0.014375, 0.016000, 0.014375,
			0.016000, 0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750, 0.014375,
			0.014375, 0.012750, 0.014375, 0.014375, 0.012750, 0.012750, 0.012750, 0.012750,
			0.012750, 0.014375, 0.012750, 0.012750, 0.012750, 0.014375, 0.014375, 0.014375,
			0.014375, 0.014375, 0.014375, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.008000, 0.011125, 0.009500, 0.009500, 0.006375, 0.011125, 0.011125,
			0.009500, 0.008000, 0.012750, 0.011125, 0.009500, 0.011125, 0.008000, 0.008000,
			0.011125, 0.011125, 0.009500, 0.009500, 0.006375, 0.009500, 0.011125, 0.011125,
			0.011125, 0.011125, 0.011125, 0.016000, 0.016000, 0.016000, 0.016000, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125, 0.019125,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500,
			0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500, 0.017500
		}
	},
	{
		(char *) "PSK250RC6", 1.0, 2.733250,
		{
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.010625, 0.014625, 0.014625, 0.014625, 0.014625, 0.010625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.016000, 0.016000,
			0.004000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.012000,
			0.012000, 0.012000, 0.013250, 0.012000, 0.010625, 0.012000, 0.012000, 0.012000,
			0.010625, 0.010625, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.013250, 0.012000, 0.013250, 0.012000,
			0.013250, 0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625, 0.012000,
			0.012000, 0.010625, 0.012000, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625,
			0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.006625, 0.009250, 0.008000, 0.008000, 0.005250, 0.009250, 0.009250,
			0.008000, 0.006625, 0.010625, 0.009250, 0.008000, 0.009250, 0.006625, 0.006625,
			0.009250, 0.009250, 0.008000, 0.008000, 0.005250, 0.008000, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.013250, 0.013250, 0.013250, 0.013250, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625
		}
	},
	{
		(char *) "PSK250RC7", 1.0, 2.345625,
		{
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.009125, 0.012500, 0.012500, 0.012500, 0.012500, 0.009125, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.013625, 0.013625,
			0.003375, 0.010250, 0.010250, 0.011375, 0.011375, 0.011375, 0.011375, 0.010250,
			0.010250, 0.010250, 0.011375, 0.010250, 0.009125, 0.010250, 0.010250, 0.010250,
			0.009125, 0.009125, 0.010250, 0.010250, 0.010250, 0.010250, 0.010250, 0.010250,
			0.010250, 0.010250, 0.010250, 0.010250, 0.011375, 0.010250, 0.011375, 0.010250,
			0.011375, 0.009125, 0.010250, 0.009125, 0.009125, 0.009125, 0.009125, 0.010250,
			0.010250, 0.009125, 0.010250, 0.010250, 0.009125, 0.009125, 0.009125, 0.009125,
			0.009125, 0.010250, 0.009125, 0.009125, 0.009125, 0.010250, 0.010250, 0.010250,
			0.010250, 0.010250, 0.010250, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375,
			0.011375, 0.005625, 0.008000, 0.006750, 0.006750, 0.004500, 0.008000, 0.008000,
			0.006750, 0.005625, 0.009125, 0.008000, 0.006750, 0.008000, 0.005625, 0.005625,
			0.008000, 0.008000, 0.006750, 0.006750, 0.004500, 0.006750, 0.008000, 0.008000,
			0.008000, 0.008000, 0.008000, 0.011375, 0.011375, 0.011375, 0.011375, 0.013625,
			0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625,
			0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625,
			0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625,
			0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625, 0.013625,
			0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375,
			0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375,
			0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375,
			0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375, 0.011375,
			0.011375, 0.011375, 0.011375, 0.011375, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500
		}
	},
	{
		(char *) "PSK500R", 1.0, 4.096000,
		{
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.032000, 0.044000, 0.044000, 0.044000, 0.044000, 0.032000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000,
			0.012000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.036000,
			0.036000, 0.036000, 0.040000, 0.036000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.032000, 0.032000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.036000, 0.040000, 0.036000, 0.040000, 0.036000,
			0.040000, 0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000, 0.036000,
			0.036000, 0.032000, 0.036000, 0.036000, 0.032000, 0.032000, 0.032000, 0.032000,
			0.032000, 0.036000, 0.032000, 0.032000, 0.032000, 0.036000, 0.036000, 0.036000,
			0.036000, 0.036000, 0.036000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.020000, 0.028000, 0.024000, 0.024000, 0.016000, 0.028000, 0.028000,
			0.024000, 0.020000, 0.032000, 0.028000, 0.024000, 0.028000, 0.020000, 0.020000,
			0.028000, 0.028000, 0.024000, 0.024000, 0.016000, 0.024000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.040000, 0.040000, 0.040000, 0.040000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000, 0.044000
		}
	},
	{
		(char *) "PSK500C2", 1.0, 1.024000,
		{
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.010000, 0.007000, 0.012000, 0.012000, 0.007000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.003000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000, 0.011000,
			0.010000, 0.010000, 0.011000, 0.011000, 0.009000, 0.008000, 0.009000, 0.011000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.010000, 0.011000, 0.011000, 0.009000, 0.011000, 0.012000,
			0.012000, 0.009000, 0.010000, 0.010000, 0.010000, 0.009000, 0.010000, 0.010000,
			0.011000, 0.009000, 0.011000, 0.011000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.011000, 0.010000, 0.009000, 0.009000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.012000, 0.011000, 0.011000, 0.011000, 0.012000, 0.011000,
			0.012000, 0.006000, 0.009000, 0.008000, 0.008000, 0.004000, 0.008000, 0.009000,
			0.008000, 0.006000, 0.011000, 0.010000, 0.007000, 0.008000, 0.006000, 0.005000,
			0.008000, 0.011000, 0.007000, 0.007000, 0.005000, 0.008000, 0.009000, 0.009000,
			0.010000, 0.009000, 0.011000, 0.012000, 0.011000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000, 0.013000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000, 0.014000
		}
	},
	{
		(char *) "PSK500C4", 1.0, 0.512000,
		{
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.005000, 0.003500, 0.006000, 0.006000, 0.003500, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.001500, 0.005500, 0.005500, 0.005500, 0.005500, 0.006000, 0.006000, 0.005500,
			0.005000, 0.005000, 0.005500, 0.005500, 0.004500, 0.004000, 0.004500, 0.005500,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005500, 0.005500, 0.005500, 0.005500,
			0.005500, 0.005500, 0.005000, 0.005500, 0.005500, 0.004500, 0.005500, 0.006000,
			0.006000, 0.004500, 0.005000, 0.005000, 0.005000, 0.004500, 0.005000, 0.005000,
			0.005500, 0.004500, 0.005500, 0.005500, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005500, 0.005000, 0.004500, 0.004500, 0.005500, 0.005500, 0.005500,
			0.005500, 0.005500, 0.006000, 0.005500, 0.005500, 0.005500, 0.006000, 0.005500,
			0.006000, 0.003000, 0.004500, 0.004000, 0.004000, 0.002000, 0.004000, 0.004500,
			0.004000, 0.003000, 0.005500, 0.005000, 0.003500, 0.004000, 0.003000, 0.002500,
			0.004000, 0.005500, 0.003500, 0.003500, 0.002500, 0.004000, 0.004500, 0.004500,
			0.005000, 0.004500, 0.005500, 0.006000, 0.005500, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000
		}
	},
	{
		(char *) "PSK500RC2", 1.0, 4.096000,
		{
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.016000, 0.022000, 0.022000, 0.022000, 0.022000, 0.016000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.024000, 0.024000,
			0.006000, 0.018000, 0.018000, 0.020000, 0.020000, 0.020000, 0.020000, 0.018000,
			0.018000, 0.018000, 0.020000, 0.018000, 0.016000, 0.018000, 0.018000, 0.018000,
			0.016000, 0.016000, 0.018000, 0.018000, 0.018000, 0.018000, 0.018000, 0.018000,
			0.018000, 0.018000, 0.018000, 0.018000, 0.020000, 0.018000, 0.020000, 0.018000,
			0.020000, 0.016000, 0.018000, 0.016000, 0.016000, 0.016000, 0.016000, 0.018000,
			0.018000, 0.016000, 0.018000, 0.018000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.018000, 0.016000, 0.016000, 0.016000, 0.018000, 0.018000, 0.018000,
			0.018000, 0.018000, 0.018000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.010000, 0.014000, 0.012000, 0.012000, 0.008000, 0.014000, 0.014000,
			0.012000, 0.010000, 0.016000, 0.014000, 0.012000, 0.014000, 0.010000, 0.010000,
			0.014000, 0.014000, 0.012000, 0.012000, 0.008000, 0.012000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.020000, 0.020000, 0.020000, 0.020000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000
		}
	},
	{
		(char *) "PSK500RC3", 1.0, 2.733250,
		{
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.010625, 0.014625, 0.014625, 0.014625, 0.014625, 0.010625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.016000, 0.016000,
			0.004000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.012000,
			0.012000, 0.012000, 0.013250, 0.012000, 0.010625, 0.012000, 0.012000, 0.012000,
			0.010625, 0.010625, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.013250, 0.012000, 0.013250, 0.012000,
			0.013250, 0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625, 0.012000,
			0.012000, 0.010625, 0.012000, 0.012000, 0.010625, 0.010625, 0.010625, 0.010625,
			0.010625, 0.012000, 0.010625, 0.010625, 0.010625, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.006625, 0.009250, 0.008000, 0.008000, 0.005250, 0.009250, 0.009250,
			0.008000, 0.006625, 0.010625, 0.009250, 0.008000, 0.009250, 0.006625, 0.006625,
			0.009250, 0.009250, 0.008000, 0.008000, 0.005250, 0.008000, 0.009250, 0.009250,
			0.009250, 0.009250, 0.009250, 0.013250, 0.013250, 0.013250, 0.013250, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250, 0.013250,
			0.013250, 0.013250, 0.013250, 0.013250, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625,
			0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625, 0.014625
		}
	},
	{
		(char *) "PSK500RC4", 1.0, 2.048000,
		{
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.008000, 0.011000, 0.011000, 0.011000, 0.011000, 0.008000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000,
			0.003000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.009000,
			0.009000, 0.009000, 0.010000, 0.009000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.008000, 0.008000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.009000, 0.010000, 0.009000, 0.010000, 0.009000,
			0.010000, 0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000, 0.009000,
			0.009000, 0.008000, 0.009000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.005000, 0.007000, 0.006000, 0.006000, 0.004000, 0.007000, 0.007000,
			0.006000, 0.005000, 0.008000, 0.007000, 0.006000, 0.007000, 0.005000, 0.005000,
			0.007000, 0.007000, 0.006000, 0.006000, 0.004000, 0.006000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.010000, 0.010000, 0.010000, 0.010000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000
		}
	},
	{
		(char *) "PSK800C2", 1.0, 0.640000,
		{
			0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500,
			0.007500, 0.006250, 0.004375, 0.007500, 0.007500, 0.004375, 0.007500, 0.007500,
			0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500,
			0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500,
			0.001875, 0.006875, 0.006875, 0.006875, 0.006875, 0.007500, 0.007500, 0.006875,
			0.006250, 0.006250, 0.006875, 0.006875, 0.005625, 0.005000, 0.005625, 0.006875,
			0.006250, 0.006250, 0.006250, 0.006250, 0.006875, 0.006875, 0.006875, 0.006875,
			0.006875, 0.006875, 0.006250, 0.006875, 0.006875, 0.005625, 0.006875, 0.007500,
			0.007500, 0.005625, 0.006250, 0.006250, 0.006250, 0.005625, 0.006250, 0.006250,
			0.006875, 0.005625, 0.006875, 0.006875, 0.006250, 0.006250, 0.006250, 0.006250,
			0.006250, 0.006875, 0.006250, 0.005625, 0.005625, 0.006875, 0.006875, 0.006875,
			0.006875, 0.006875, 0.007500, 0.006875, 0.006875, 0.006875, 0.007500, 0.006875,
			0.007500, 0.003750, 0.005625, 0.005000, 0.005000, 0.002500, 0.005000, 0.005625,
			0.005000, 0.003750, 0.006875, 0.006250, 0.004375, 0.005000, 0.003750, 0.003125,
			0.005000, 0.006875, 0.004375, 0.004375, 0.003125, 0.005000, 0.005625, 0.005625,
			0.006250, 0.005625, 0.006875, 0.007500, 0.006875, 0.007500, 0.007500, 0.007500,
			0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500,
			0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.007500, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125, 0.008125,
			0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750,
			0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750,
			0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750, 0.008750
		}
	},
	{
		(char *) "PSK800RC2", 1.0, 2.560000,
		{
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.010000, 0.013750, 0.013750, 0.013750, 0.013750, 0.010000, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.015000, 0.015000,
			0.003750, 0.011250, 0.011250, 0.012500, 0.012500, 0.012500, 0.012500, 0.011250,
			0.011250, 0.011250, 0.012500, 0.011250, 0.010000, 0.011250, 0.011250, 0.011250,
			0.010000, 0.010000, 0.011250, 0.011250, 0.011250, 0.011250, 0.011250, 0.011250,
			0.011250, 0.011250, 0.011250, 0.011250, 0.012500, 0.011250, 0.012500, 0.011250,
			0.012500, 0.010000, 0.011250, 0.010000, 0.010000, 0.010000, 0.010000, 0.011250,
			0.011250, 0.010000, 0.011250, 0.011250, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.011250, 0.010000, 0.010000, 0.010000, 0.011250, 0.011250, 0.011250,
			0.011250, 0.011250, 0.011250, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.006250, 0.008750, 0.007500, 0.007500, 0.005000, 0.008750, 0.008750,
			0.007500, 0.006250, 0.010000, 0.008750, 0.007500, 0.008750, 0.006250, 0.006250,
			0.008750, 0.008750, 0.007500, 0.007500, 0.005000, 0.007500, 0.008750, 0.008750,
			0.008750, 0.008750, 0.008750, 0.012500, 0.012500, 0.012500, 0.012500, 0.015000,
			0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000,
			0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000,
			0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000,
			0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000, 0.015000,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500, 0.012500,
			0.012500, 0.012500, 0.012500, 0.012500, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750,
			0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750, 0.013750
		}
	},
	{
		(char *) "PSK1000C2", 1.0, 1.024000,
		{
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.005000, 0.003500, 0.006000, 0.006000, 0.003500, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.001500, 0.005500, 0.005500, 0.005500, 0.005500, 0.006000, 0.006000, 0.005500,
			0.005000, 0.005000, 0.005500, 0.005500, 0.004500, 0.004000, 0.004500, 0.005500,
			0.005000, 0.005000, 0.005000, 0.005000, 0.005500, 0.005500, 0.005500, 0.005500,
			0.005500, 0.005500, 0.005000, 0.005500, 0.005500, 0.004500, 0.005500, 0.006000,
			0.006000, 0.004500, 0.005000, 0.005000, 0.005000, 0.004500, 0.005000, 0.005000,
			0.005500, 0.004500, 0.005500, 0.005500, 0.005000, 0.005000, 0.005000, 0.005000,
			0.005000, 0.005500, 0.005000, 0.004500, 0.004500, 0.005500, 0.005500, 0.005500,
			0.005500, 0.005500, 0.006000, 0.005500, 0.005500, 0.005500, 0.006000, 0.005500,
			0.006000, 0.003000, 0.004500, 0.004000, 0.004000, 0.002000, 0.004000, 0.004500,
			0.004000, 0.003000, 0.005500, 0.005000, 0.003500, 0.004000, 0.003000, 0.002500,
			0.004000, 0.005500, 0.003500, 0.003500, 0.002500, 0.004000, 0.004500, 0.004500,
			0.005000, 0.004500, 0.005500, 0.006000, 0.005500, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000,
			0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006000, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500, 0.006500,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000, 0.007000
		}
	},
	{
		(char *) "PSK1000R", 1.0, 2.048000,
		{
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.016000, 0.022000, 0.022000, 0.022000, 0.022000, 0.016000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.024000, 0.024000,
			0.006000, 0.018000, 0.018000, 0.020000, 0.020000, 0.020000, 0.020000, 0.018000,
			0.018000, 0.018000, 0.020000, 0.018000, 0.016000, 0.018000, 0.018000, 0.018000,
			0.016000, 0.016000, 0.018000, 0.018000, 0.018000, 0.018000, 0.018000, 0.018000,
			0.018000, 0.018000, 0.018000, 0.018000, 0.020000, 0.018000, 0.020000, 0.018000,
			0.020000, 0.016000, 0.018000, 0.016000, 0.016000, 0.016000, 0.016000, 0.018000,
			0.018000, 0.016000, 0.018000, 0.018000, 0.016000, 0.016000, 0.016000, 0.016000,
			0.016000, 0.018000, 0.016000, 0.016000, 0.016000, 0.018000, 0.018000, 0.018000,
			0.018000, 0.018000, 0.018000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.010000, 0.014000, 0.012000, 0.012000, 0.008000, 0.014000, 0.014000,
			0.012000, 0.010000, 0.016000, 0.014000, 0.012000, 0.014000, 0.010000, 0.010000,
			0.014000, 0.014000, 0.012000, 0.012000, 0.008000, 0.012000, 0.014000, 0.014000,
			0.014000, 0.014000, 0.014000, 0.020000, 0.020000, 0.020000, 0.020000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000, 0.022000
		}
	},
	{
		(char *) "PSK1000RC2", 1.0, 2.048000,
		{
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.008000, 0.011000, 0.011000, 0.011000, 0.011000, 0.008000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.012000, 0.012000,
			0.003000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.009000,
			0.009000, 0.009000, 0.010000, 0.009000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.008000, 0.008000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.009000, 0.010000, 0.009000, 0.010000, 0.009000,
			0.010000, 0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000, 0.009000,
			0.009000, 0.008000, 0.009000, 0.009000, 0.008000, 0.008000, 0.008000, 0.008000,
			0.008000, 0.009000, 0.008000, 0.008000, 0.008000, 0.009000, 0.009000, 0.009000,
			0.009000, 0.009000, 0.009000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.005000, 0.007000, 0.006000, 0.006000, 0.004000, 0.007000, 0.007000,
			0.006000, 0.005000, 0.008000, 0.007000, 0.006000, 0.007000, 0.005000, 0.005000,
			0.007000, 0.007000, 0.006000, 0.006000, 0.004000, 0.006000, 0.007000, 0.007000,
			0.007000, 0.007000, 0.007000, 0.010000, 0.010000, 0.010000, 0.010000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000, 0.012000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000, 0.010000,
			0.010000, 0.010000, 0.010000, 0.010000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000,
			0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000, 0.011000
		}
	},
	{
		(char *) "QPSK31", 1.0, 3.072000,
		{
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.320000, 0.224000, 0.384000, 0.384000, 0.224000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.096000, 0.352000, 0.352000, 0.352000, 0.352000, 0.384000, 0.384000, 0.352000,
			0.320000, 0.320000, 0.352000, 0.352000, 0.288000, 0.256000, 0.288000, 0.352000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.320000, 0.352000, 0.352000, 0.288000, 0.352000, 0.384000,
			0.384000, 0.288000, 0.320000, 0.320000, 0.320000, 0.288000, 0.320000, 0.320000,
			0.352000, 0.288000, 0.352000, 0.352000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.352000, 0.320000, 0.288000, 0.288000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.384000, 0.352000, 0.352000, 0.352000, 0.384000, 0.352000,
			0.384000, 0.192000, 0.288000, 0.256000, 0.256000, 0.128000, 0.256000, 0.288000,
			0.256000, 0.192000, 0.352000, 0.320000, 0.224000, 0.256000, 0.192000, 0.160000,
			0.256000, 0.352000, 0.224000, 0.224000, 0.160000, 0.256000, 0.288000, 0.288000,
			0.320000, 0.288000, 0.352000, 0.384000, 0.352000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000, 0.416000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000,
			0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000, 0.448000
		}
	},
	{
		(char *) "QPSK63", 1.0, 3.072000,
		{
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.160000, 0.112000, 0.192000, 0.192000, 0.112000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.048000, 0.176000, 0.176000, 0.176000, 0.176000, 0.192000, 0.192000, 0.176000,
			0.160000, 0.160000, 0.176000, 0.176000, 0.144000, 0.128000, 0.144000, 0.176000,
			0.160000, 0.160000, 0.160000, 0.160000, 0.176000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.160000, 0.176000, 0.176000, 0.144000, 0.176000, 0.192000,
			0.192000, 0.144000, 0.160000, 0.160000, 0.160000, 0.144000, 0.160000, 0.160000,
			0.176000, 0.144000, 0.176000, 0.176000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.176000, 0.160000, 0.144000, 0.144000, 0.176000, 0.176000, 0.176000,
			0.176000, 0.176000, 0.192000, 0.176000, 0.176000, 0.176000, 0.192000, 0.176000,
			0.192000, 0.096000, 0.144000, 0.128000, 0.128000, 0.064000, 0.128000, 0.144000,
			0.128000, 0.096000, 0.176000, 0.160000, 0.112000, 0.128000, 0.096000, 0.080000,
			0.128000, 0.176000, 0.112000, 0.112000, 0.080000, 0.128000, 0.144000, 0.144000,
			0.160000, 0.144000, 0.176000, 0.192000, 0.176000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000,
			0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.192000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000, 0.208000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000, 0.224000
		}
	},
	{
		(char *) "QPSK125", 1.0, 3.072000,
		{
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.080000, 0.056000, 0.096000, 0.096000, 0.056000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.024000, 0.088000, 0.088000, 0.088000, 0.088000, 0.096000, 0.096000, 0.088000,
			0.080000, 0.080000, 0.088000, 0.088000, 0.072000, 0.064000, 0.072000, 0.088000,
			0.080000, 0.080000, 0.080000, 0.080000, 0.088000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.080000, 0.088000, 0.088000, 0.072000, 0.088000, 0.096000,
			0.096000, 0.072000, 0.080000, 0.080000, 0.080000, 0.072000, 0.080000, 0.080000,
			0.088000, 0.072000, 0.088000, 0.088000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.088000, 0.080000, 0.072000, 0.072000, 0.088000, 0.088000, 0.088000,
			0.088000, 0.088000, 0.096000, 0.088000, 0.088000, 0.088000, 0.096000, 0.088000,
			0.096000, 0.048000, 0.072000, 0.064000, 0.064000, 0.032000, 0.064000, 0.072000,
			0.064000, 0.048000, 0.088000, 0.080000, 0.056000, 0.064000, 0.048000, 0.040000,
			0.064000, 0.088000, 0.056000, 0.056000, 0.040000, 0.064000, 0.072000, 0.072000,
			0.080000, 0.072000, 0.088000, 0.096000, 0.088000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000,
			0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.096000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000, 0.104000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000,
			0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000, 0.112000
		}
	},
	{
		(char *) "QPSK250", 1.0, 3.072000,
		{
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.040000, 0.028000, 0.048000, 0.048000, 0.028000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.012000, 0.044000, 0.044000, 0.044000, 0.044000, 0.048000, 0.048000, 0.044000,
			0.040000, 0.040000, 0.044000, 0.044000, 0.036000, 0.032000, 0.036000, 0.044000,
			0.040000, 0.040000, 0.040000, 0.040000, 0.044000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.040000, 0.044000, 0.044000, 0.036000, 0.044000, 0.048000,
			0.048000, 0.036000, 0.040000, 0.040000, 0.040000, 0.036000, 0.040000, 0.040000,
			0.044000, 0.036000, 0.044000, 0.044000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.044000, 0.040000, 0.036000, 0.036000, 0.044000, 0.044000, 0.044000,
			0.044000, 0.044000, 0.048000, 0.044000, 0.044000, 0.044000, 0.048000, 0.044000,
			0.048000, 0.024000, 0.036000, 0.032000, 0.032000, 0.016000, 0.032000, 0.036000,
			0.032000, 0.024000, 0.044000, 0.040000, 0.028000, 0.032000, 0.024000, 0.020000,
			0.032000, 0.044000, 0.028000, 0.028000, 0.020000, 0.032000, 0.036000, 0.036000,
			0.040000, 0.036000, 0.044000, 0.048000, 0.044000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000,
			0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.048000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000, 0.052000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000,
			0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000, 0.056000
		}
	},
	{
		(char *) "QPSK500", 1.0, 3.072000,
		{
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.020000, 0.014000, 0.024000, 0.024000, 0.014000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.006000, 0.022000, 0.022000, 0.022000, 0.022000, 0.024000, 0.024000, 0.022000,
			0.020000, 0.020000, 0.022000, 0.022000, 0.018000, 0.016000, 0.018000, 0.022000,
			0.020000, 0.020000, 0.020000, 0.020000, 0.022000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.020000, 0.022000, 0.022000, 0.018000, 0.022000, 0.024000,
			0.024000, 0.018000, 0.020000, 0.020000, 0.020000, 0.018000, 0.020000, 0.020000,
			0.022000, 0.018000, 0.022000, 0.022000, 0.020000, 0.020000, 0.020000, 0.020000,
			0.020000, 0.022000, 0.020000, 0.018000, 0.018000, 0.022000, 0.022000, 0.022000,
			0.022000, 0.022000, 0.024000, 0.022000, 0.022000, 0.022000, 0.024000, 0.022000,
			0.024000, 0.012000, 0.018000, 0.016000, 0.016000, 0.008000, 0.016000, 0.018000,
			0.016000, 0.012000, 0.022000, 0.020000, 0.014000, 0.016000, 0.012000, 0.010000,
			0.016000, 0.022000, 0.014000, 0.014000, 0.010000, 0.016000, 0.018000, 0.018000,
			0.020000, 0.018000, 0.022000, 0.024000, 0.022000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000,
			0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.024000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000, 0.026000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000,
			0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000, 0.028000
		}
	},
	{
		(char *) "THOR16", 1.0, 4.480000,
		{
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.256000, 0.352000, 0.352000, 0.352000, 0.352000, 0.256000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.384000, 0.384000,
			0.096000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.288000,
			0.288000, 0.288000, 0.320000, 0.288000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.256000, 0.256000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.288000, 0.320000, 0.288000, 0.320000, 0.288000,
			0.320000, 0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000, 0.288000,
			0.288000, 0.256000, 0.288000, 0.288000, 0.256000, 0.256000, 0.256000, 0.256000,
			0.256000, 0.288000, 0.256000, 0.256000, 0.256000, 0.288000, 0.288000, 0.288000,
			0.288000, 0.288000, 0.288000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.160000, 0.224000, 0.192000, 0.192000, 0.128000, 0.224000, 0.224000,
			0.192000, 0.160000, 0.256000, 0.224000, 0.192000, 0.224000, 0.160000, 0.160000,
			0.224000, 0.224000, 0.192000, 0.192000, 0.128000, 0.192000, 0.224000, 0.224000,
			0.224000, 0.224000, 0.224000, 0.320000, 0.320000, 0.320000, 0.320000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000, 0.384000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000, 0.320000,
			0.320000, 0.320000, 0.320000, 0.320000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000,
			0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000, 0.352000
		}
	},
	{
		(char *) "THOR22", 1.0, 3.250794,
		{
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.185760, 0.255420, 0.255420, 0.255420, 0.255420, 0.185760, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.278639, 0.278639,
			0.069660, 0.208980, 0.208980, 0.232200, 0.232200, 0.232200, 0.232200, 0.208980,
			0.208980, 0.208980, 0.232200, 0.208980, 0.185760, 0.208980, 0.208980, 0.208980,
			0.185760, 0.185760, 0.208980, 0.208980, 0.208980, 0.208980, 0.208980, 0.208980,
			0.208980, 0.208980, 0.208980, 0.208980, 0.232200, 0.208980, 0.232200, 0.208980,
			0.232200, 0.185760, 0.208980, 0.185760, 0.185760, 0.185760, 0.185760, 0.208980,
			0.208980, 0.185760, 0.208980, 0.208980, 0.185760, 0.185760, 0.185760, 0.185760,
			0.185760, 0.208980, 0.185760, 0.185760, 0.185760, 0.208980, 0.208980, 0.208980,
			0.208980, 0.208980, 0.208980, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.116100, 0.162540, 0.139320, 0.139320, 0.092880, 0.162540, 0.162540,
			0.139320, 0.116100, 0.185760, 0.162540, 0.139320, 0.162540, 0.116100, 0.116100,
			0.162540, 0.162540, 0.139320, 0.139320, 0.092880, 0.139320, 0.162540, 0.162540,
			0.162540, 0.162540, 0.162540, 0.232200, 0.232200, 0.232200, 0.232200, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639, 0.278639,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200, 0.232200,
			0.232200, 0.232200, 0.232200, 0.232200, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420,
			0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420, 0.255420
		}
	},
	{
		(char *) "THOR25x4", 1.0, 10.720000,
		{
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.160000, 0.220000, 0.220000, 0.220000, 0.220000, 0.160000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.240000, 0.240000,
			0.060000, 0.180000, 0.180000, 0.200000, 0.200000, 0.200000, 0.200000, 0.180000,
			0.180000, 0.180000, 0.200000, 0.180000, 0.160000, 0.180000, 0.180000, 0.180000,
			0.160000, 0.160000, 0.180000, 0.180000, 0.180000, 0.180000, 0.180000, 0.180000,
			0.180000, 0.180000, 0.180000, 0.180000, 0.200000, 0.180000, 0.200000, 0.180000,
			0.200000, 0.160000, 0.180000, 0.160000, 0.160000, 0.160000, 0.160000, 0.180000,
			0.180000, 0.160000, 0.180000, 0.180000, 0.160000, 0.160000, 0.160000, 0.160000,
			0.160000, 0.180000, 0.160000, 0.160000, 0.160000, 0.180000, 0.180000, 0.180000,
			0.180000, 0.180000, 0.180000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.100000, 0.140000, 0.120000, 0.120000, 0.080000, 0.140000, 0.140000,
			0.120000, 0.100000, 0.160000, 0.140000, 0.120000, 0.140000, 0.100000, 0.100000,
			0.140000, 0.140000, 0.120000, 0.120000, 0.080000, 0.120000, 0.140000, 0.140000,
			0.140000, 0.140000, 0.140000, 0.200000, 0.200000, 0.200000, 0.200000, 0.240000,
			0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000,
			0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000,
			0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000,
			0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000, 0.240000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000, 0.200000,
			0.200000, 0.200000, 0.200000, 0.200000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000,
			0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000, 0.220000
		}
	},
	{
		(char *) "THOR50x1", 1.0, 5.360000,
		{
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.080000, 0.110000, 0.110000, 0.110000, 0.110000, 0.080000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.120000, 0.120000,
			0.030000, 0.090000, 0.090000, 0.100000, 0.100000, 0.100000, 0.100000, 0.090000,
			0.090000, 0.090000, 0.100000, 0.090000, 0.080000, 0.090000, 0.090000, 0.090000,
			0.080000, 0.080000, 0.090000, 0.090000, 0.090000, 0.090000, 0.090000, 0.090000,
			0.090000, 0.090000, 0.090000, 0.090000, 0.100000, 0.090000, 0.100000, 0.090000,
			0.100000, 0.080000, 0.090000, 0.080000, 0.080000, 0.080000, 0.080000, 0.090000,
			0.090000, 0.080000, 0.090000, 0.090000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.090000, 0.080000, 0.080000, 0.080000, 0.090000, 0.090000, 0.090000,
			0.090000, 0.090000, 0.090000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.050000, 0.070000, 0.060000, 0.060000, 0.040000, 0.070000, 0.070000,
			0.060000, 0.050000, 0.080000, 0.070000, 0.060000, 0.070000, 0.050000, 0.050000,
			0.070000, 0.070000, 0.060000, 0.060000, 0.040000, 0.060000, 0.070000, 0.070000,
			0.070000, 0.070000, 0.070000, 0.100000, 0.100000, 0.100000, 0.100000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000
		}
	},
	{
		(char *) "THOR50x2", 1.0, 5.360000,
		{
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.080000, 0.110000, 0.110000, 0.110000, 0.110000, 0.080000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.120000, 0.120000,
			0.030000, 0.090000, 0.090000, 0.100000, 0.100000, 0.100000, 0.100000, 0.090000,
			0.090000, 0.090000, 0.100000, 0.090000, 0.080000, 0.090000, 0.090000, 0.090000,
			0.080000, 0.080000, 0.090000, 0.090000, 0.090000, 0.090000, 0.090000, 0.090000,
			0.090000, 0.090000, 0.090000, 0.090000, 0.100000, 0.090000, 0.100000, 0.090000,
			0.100000, 0.080000, 0.090000, 0.080000, 0.080000, 0.080000, 0.080000, 0.090000,
			0.090000, 0.080000, 0.090000, 0.090000, 0.080000, 0.080000, 0.080000, 0.080000,
			0.080000, 0.090000, 0.080000, 0.080000, 0.080000, 0.090000, 0.090000, 0.090000,
			0.090000, 0.090000, 0.090000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.050000, 0.070000, 0.060000, 0.060000, 0.040000, 0.070000, 0.070000,
			0.060000, 0.050000, 0.080000, 0.070000, 0.060000, 0.070000, 0.050000, 0.050000,
			0.070000, 0.070000, 0.060000, 0.060000, 0.040000, 0.060000, 0.070000, 0.070000,
			0.070000, 0.070000, 0.070000, 0.100000, 0.100000, 0.100000, 0.100000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000, 0.120000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000, 0.100000,
			0.100000, 0.100000, 0.100000, 0.100000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000,
			0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000, 0.110000
		}
	},
	{
		(char *) "THOR100", 1.0, 2.680000,
		{
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.040000, 0.055000, 0.055000, 0.055000, 0.055000, 0.040000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.060000, 0.060000,
			0.015000, 0.045000, 0.045000, 0.050000, 0.050000, 0.050000, 0.050000, 0.045000,
			0.045000, 0.045000, 0.050000, 0.045000, 0.040000, 0.045000, 0.045000, 0.045000,
			0.040000, 0.040000, 0.045000, 0.045000, 0.045000, 0.045000, 0.045000, 0.045000,
			0.045000, 0.045000, 0.045000, 0.045000, 0.050000, 0.045000, 0.050000, 0.045000,
			0.050000, 0.040000, 0.045000, 0.040000, 0.040000, 0.040000, 0.040000, 0.045000,
			0.045000, 0.040000, 0.045000, 0.045000, 0.040000, 0.040000, 0.040000, 0.040000,
			0.040000, 0.045000, 0.040000, 0.040000, 0.040000, 0.045000, 0.045000, 0.045000,
			0.045000, 0.045000, 0.045000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.025000, 0.035000, 0.030000, 0.030000, 0.020000, 0.035000, 0.035000,
			0.030000, 0.025000, 0.040000, 0.035000, 0.030000, 0.035000, 0.025000, 0.025000,
			0.035000, 0.035000, 0.030000, 0.030000, 0.020000, 0.030000, 0.035000, 0.035000,
			0.035000, 0.035000, 0.035000, 0.050000, 0.050000, 0.050000, 0.050000, 0.060000,
			0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000,
			0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000,
			0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000,
			0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000, 0.060000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000, 0.050000,
			0.050000, 0.050000, 0.050000, 0.050000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000,
			0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000, 0.055000
		}
	}
};


static float time_length(MODE_TIME_TABLE *mTable, const char *string, int length);
static int str_cnt(char * str, int count_limit);

/** ********************************************************
 * \brief Determine the length of the string with a count
 * limitation.
 * \return signed integer. The number of characters in the 
 * array not to excede count limit.
 ***********************************************************/
static int str_cnt(char * str, int count_limit)
{
	if(!str || (count_limit < 1))
		return 0;

	int value = 0;

	for(int index = 0; index < count_limit; index++) {
		if(str[index] == 0) break;
		value++;
	}

	return value;
}

/** ********************************************************
 * \brief Calulate the time it take to transmit the given 
 * text with the selected modem.
 ***********************************************************/
static float time_length(MODE_TIME_TABLE *mTable, const char *string, int length)
{
	unsigned int character = 0;
	float accum = 0.0;
	
	while (length > 0) {
		character = (unsigned int) *string++;
		character &= 0xFF;
		accum += mTable->table[character];
		length--;
	}
	
	return (accum * mTable->scale);
}

/** ********************************************************
 * \brief Returns the time it take to transmit the given
 * text with the selected modem. 'C' interface.
 * \return float Duration in seconds.
 ***********************************************************/
float seconds_from_c_string(const char *mode, const char *string, int length, float *overhead)
{
	int mode_index = 0;
	int max_mode_name = 16;
	int mode_count = sizeof(mode_time_table) / sizeof(MODE_TIME_TABLE);
	static int last_mode = 0;
	
	if(!mode || !string || length < 1) return 0.0;
	
	if(strncmp(mode, mode_time_table[last_mode].mode_name, max_mode_name) == 0) {
		if(overhead) {
			*overhead = mode_time_table[last_mode].overhead;
		}
		return time_length(&mode_time_table[last_mode], string, length);
	} else {
		for(mode_index = 0; mode_index < mode_count; mode_index++) {
			if(strncmp((char *) mode, (char *) mode_time_table[mode_index].mode_name, max_mode_name) == 0) {
				last_mode = mode_index;
				if(overhead) {
					*overhead = mode_time_table[last_mode].overhead;
				}
				return time_length(&mode_time_table[mode_index], string, length);
			}
		}
	}
	return 0.0;
}

/** ********************************************************
 * \brief Returns the time it take to transmit the given
 * text with the selected modem. 'C' interface.
 * \return float Duration in minutes.
 ***********************************************************/
float minutes_from_c_string(const char *mode, const char *string, int length, float *overhead)
{
	float time = 0.0;
	
	time = seconds_from_c_string(mode, string, length, overhead) / 60.0;
	if(overhead)
		*overhead /= 60.0;
	return time;
}

/** ********************************************************
 * \brief Returns the time it take to transmit the given
 * text with the selected modem. 'C++' std::string interface.
 * \return float Duration in seconds.
 ***********************************************************/
float seconds_from_string(std::string mode, std::string& str, float *overhead)
{
	try {
		return seconds_from_c_string(mode.c_str(), str.c_str(), str.length(), overhead);
	} catch ( ... ) {
		LOG_ERROR("%s", "String Ref Error");
		return 0.0;
	}
}

/** ********************************************************
 * \brief Returns the time it take to transmit the given
 * text with the selected modem. 'C++' std::string interface.
 * \return float Duration in minutes.
 ***********************************************************/
float minutes_from_string(std::string mode, std::string& str, float *overhead)
{
	float time = 0.0;
	try {
		time = seconds_from_c_string(mode.c_str(), str.c_str(), str.length(), overhead) / 60.0;
		if(overhead)
			*overhead /= 60.0;
	} catch ( ... ) {
		LOG_ERROR("%s", "String Ref Error");
		return 0.0;
	}
	
	return time;
}


/** ********************************************************
 * \brief Search the modem table for a modem string match.
 * 'C++' std::string interface.
 * \return bool Match found = true.
 ***********************************************************/
bool modem_available(std::string modem)
{
	if(modem.empty())
		return false;
	return modem_available((char *) modem.c_str(), (int) modem.size());
}


/** ********************************************************
 * \brief Search the modem table for a modem string match.
 * 'C' interface.
 * \return bool Match found = true.
 ***********************************************************/
bool modem_available(char *modem, int search_limit)
{
	if(!modem || (search_limit < 1))
		return false;

	int mode_count = (int) sizeof(mode_time_table) / sizeof(MODE_TIME_TABLE);
	int index = 0;
	int results = 0;
	bool found = false;

	for(index = 0; index < mode_count; index++) {
		results = strncmp(modem, mode_time_table[index].mode_name, search_limit);
		if(results == 0) {
			results = str_cnt(mode_time_table[index].mode_name, 16);
			if(results == search_limit) {
				found = true;
				break;
			}
		}
	}

	return found;
}

/** ********************************************************
 * \brief Return the number of table entires.
 * \return integer
 ***********************************************************/
int mode_table_count(void)
{
	return (int) sizeof(mode_time_table) / sizeof(MODE_TIME_TABLE);
}

/** ********************************************************
 * \brief Return modem ID string at position 'x' in the 
 * Table.
 * \return char *.
 ***********************************************************/
char * modem_at_index(int index)
{
	static char modem_name[] = "Bad Index";

	int index_limit = mode_table_count();

	if(index < 0 || index >= index_limit)
		return (char *) &modem_name[0];

	return (char *) mode_time_table[index].mode_name;
}

