/* $Id: glue-png.c,v 1.8 2012-06-26 22:01:31 sidekell Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "glue.h"
#include "glue-png.h"

#ifdef HAVE_PNG
#include <png.h>

#ifndef Z_BEST_COMPRESSION
#include <zlib.h>
#endif

#endif


bool
png_read(
	uint32_t **dest, 
	unsigned int *w,
	unsigned int *h,
	const char* filename
)
{
#ifdef HAVE_PNG
	png_structp p;
	png_infop pi;
	png_uint_32 width, height;
	int bit_depth, color_type, interlace_type, compression_type;
	int filter_method;

	uint8_t header[8];
	size_t r;
	int b, i;
	png_bytepp rows;
	uint32_t* saved_dest;

	FILE* f = fopen(filename, "r");

	if (f == NULL) {
		faum_log(FAUM_LOG_WARNING, "glue-png", "", 
			"couldn't open %s for reading.\n", filename);
		return false;
	}

	/* first read header bytes into header */
	r = fread(header, sizeof(header), 1, f);
	if (r < 1) {
		faum_log(FAUM_LOG_WARNING, "glue-png", "",
			"couldn't read header of png file %s.\n", filename);
		goto png_err_fd_out;
	}

	/* create png structures */
	p = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (p == NULL) {
		faum_log(FAUM_LOG_DEBUG, "glue-png", "", 
			"png_create_read_struct failed in %s\n", 
			__FUNCTION__);
		goto png_err_fd_out;
	}

	pi = png_create_info_struct(p);
	if (pi == NULL) {
		faum_log(FAUM_LOG_DEBUG, "glue-png", "", 
			"png_create_info_struct failed in %s\n",
			__FUNCTION__);
		goto png_err_out;
	}

	/* check if its a png */
	b = png_sig_cmp((png_bytep)header, 0, sizeof(header));
	if (b) {
		faum_log(FAUM_LOG_DEBUG, "glue-png", "",
			"%s is not a png file.\n", filename);
		goto png_err_out;
	}

	/* tell libpng that we read the header already */
	png_set_sig_bytes(p, sizeof(header));

	/* check if an error occured in png_* functions, then we'll end
	 * up right here with setjmp returning 1 */
	if (setjmp(png_jmpbuf(p))) {
		faum_log(FAUM_LOG_WARNING, "glue-png", "", 
			"%s: an error occured in %s.\n", filename,
			__FUNCTION__);

		goto png_err_out;
	}

	png_init_io(p, f);
	png_read_info(p, pi);

    	png_get_IHDR(p, pi, &width, &height,
		       &bit_depth, &color_type, &interlace_type,
		       &compression_type, &filter_method);

	switch(color_type) {
	case PNG_COLOR_TYPE_PALETTE:
		png_set_palette_to_rgb(p);
		png_set_add_alpha(p, 0xFF, PNG_FILLER_AFTER);
		break;
	case PNG_COLOR_TYPE_RGB:
		png_set_add_alpha(p, 0xFF, PNG_FILLER_AFTER);
		break;
	case PNG_COLOR_TYPE_GRAY:
		png_set_gray_to_rgb(p);
		png_set_add_alpha(p, 0xFF, PNG_FILLER_AFTER);
		break;
	case PNG_COLOR_TYPE_GRAY_ALPHA:
		png_set_gray_to_rgb(p);
		break;
	case PNG_COLOR_TYPE_RGB_ALPHA:
		break;
	default:
		faum_log(FAUM_LOG_DEBUG, "glue-png", "",
			"color type of %s not supported.\n", filename);
		goto png_err_out;
		return false;
	}

	if (bit_depth == 16) {
		png_set_strip_16(p);
	}

	/* flip rgb to bgr */
	png_set_bgr(p);

	png_read_update_info(p, pi);
	/* after the transformations, one row *must* exactly fit 
	 * one row of ARGB pixels, each stored in an uint32_t */
	assert(png_get_rowbytes(p, pi) == width * sizeof(uint32_t));

	/* allocate memory for row pointers */
	rows = malloc(height * png_sizeof(png_bytep));
	assert(rows);

	/* allocate memory in dest */
	*dest = shm_alloc(width * height * sizeof(uint32_t));
	assert(*dest);

	*w = width;
	*h = height;

	saved_dest = *dest;
	for (i = 0; i < height; i++) {
		rows[i] = (png_bytep)*dest;
		(*dest) += width;
	}
	*dest = saved_dest;

	png_set_rows(p, pi, rows);
	png_read_image(p, rows);

	png_destroy_read_struct(&p, &pi, NULL);
	free(rows);
	fclose(f);
	return true;

png_err_out:
	png_destroy_read_struct(&p, &pi, NULL);
png_err_fd_out:
	fclose(f);
	return false;
#else
	faum_log(FAUM_LOG_WARNING, "glue-png", "", 
			"png files not supported.\n");
	return false;
#endif	/* HAVE_PNG */
}


void
png_write(
	const uint32_t* source,
	unsigned int w,
	unsigned int h,
	unsigned int x,
	unsigned int y,
	unsigned int dw,
	unsigned int dh,
	const char* filename
)
{
#ifdef HAVE_PNG
	FILE *fp;
	png_structp png_ptr;
	png_infop info_ptr;
	png_bytep * rowptrs;
	unsigned int yc;
	unsigned int xc;
	int ret;
	const uint32_t* pixel;

	/* don't write empty images */
	assert(0 < w);
	assert(0 < h);
	assert(0 < dw);
	assert(0 < dh);
	/* rectangle inside buffer at all? */
	assert(x + dw <= w);
	assert(y + dh <= h);

	/*
	 * Create screenshot file.
	 */
	fp = fopen(filename, "w");
	if (! fp) {
		faum_log(FAUM_LOG_ERROR, "glue-png", "",
			"Can't create %s: %s.\n", filename, strerror(errno));
		return;
	}

	/* Since our internal structure is a mess, we have to convert it
	 * for libpng. */
	rowptrs = malloc(h * sizeof(png_bytep));
	assert(rowptrs);
	pixel = source + y * w + x;
	for (yc = 0; yc < dh; yc++) {
		rowptrs[yc] = malloc(w * sizeof(png_byte) * 3);
		assert(rowptrs[yc]);
		for (xc = 0; xc < dw ; xc++) {
			rowptrs[yc][xc*3 + 0] = GET_RED(*pixel);
			rowptrs[yc][xc*3 + 1] = GET_GREEN(*pixel);
			rowptrs[yc][xc*3 + 2] = GET_BLUE(*pixel);
			pixel++;
		}
		pixel += w - dw;
	}

	/* Now comes the mess: libpng. This code is mostly copied
	 * from the libpng documentation... */
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
					  NULL, NULL, NULL);
	assert(png_ptr);

	info_ptr = png_create_info_struct(png_ptr);
	assert(info_ptr);

	if (setjmp(png_jmpbuf(png_ptr))) {
		assert(0);
	}

	png_init_io(png_ptr, fp);

	/* set the zlib compression level */
	png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);

	/* Fill in inportant header info */
	png_set_IHDR(png_ptr, info_ptr, dw, dh, 8,
			PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
			PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	/* Tell it where our data is */
	png_set_rows(png_ptr, info_ptr, rowptrs);

	/* And let it write the crap */
	png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);

	/* Clean up our temporary memory */
	for (yc = 0; yc < dh; yc++) {
		free(rowptrs[yc]);
	}
	free(rowptrs);

	/*
	 * Close screenshot file.
	 */
	ret = fclose(fp);
	assert(0 <= ret);

	return;
#else  /* HAVE_PNG else */
	faum_log(FAUM_LOG_WARNING, "glue-png", "",
			"%s: png files not supported\n.", __FUNCTION__);
#endif /* HAVE_PNG */
}
