/* $Id$ 
 *
 * Generate C-Code from intermediate Code.
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __GEN_CCODE_HPP_INCLUDED
#define __GEN_CCODE_HPP_INCLUDED

#include "intermediate/visitor/Visitor.hpp"
#include "intermediate/operands/RegisterFactory.hpp"
#include <iostream>

namespace intermediate {

//! generate C code from intermediate representation
/** This visitor can be used to transform the intermediate
 *  representation into C-code.
 */
class GenCCode : public Visitor {
public:
	//! c'tor
	/** @param destination destination target stream, to
	 *   which code will get written to.
	 */
	GenCCode(std::ostream &destination);

private:
	//! visit a CodeContainer node
	/** @param node node that gets visited.
	 */
	virtual void visit(CodeContainer &node);

	//! visit a ImmediateOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(ImmediateOperand &node);

	//! visit a BeginTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(BeginTransfer &node);

	//! visit an EndTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(EndTransfer &node);

	//! visit a SetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(SetParam &node);

	//! visit a GetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetParam &node);

	//! visit a Connect node
	/** @param node node that gets visited.
	 */
	virtual void visit(Connect &node);

	//! visit a Mov node
	/** @param node node that gets visited.
	 */
	virtual void visit(Mov &node);

	//! visit a Je node
	/** @param node node that gets visited.
	 */
	virtual void visit(Je &node);

	//! visit a Jbe node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jbe &node);

	//! visit a Jne node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jne &node);

	//! visit a Jb node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jb &node);

	//! visit a Jmp node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jmp &node);

	//! visit a Label node
	/** @param node node that gets visited.
	 */
	virtual void visit(Label &node);

	//! visit an Abort node
	/** @param node node that gets visited.
	 */
	virtual void visit(Abort &node);

	//! visit an Add node
	/** @param node node that gets visited.
	 */
	virtual void visit(Add &node);

	//! visit a Sub node
	/** @param node node that gets visited.
	 */
	virtual void visit(Sub &node);

	//! visit a Call node
	/** @param node node that gets visited.
	 */
	virtual void visit(Call &node);

	//! visit a Return node
	/** @param node node that gets visited.
	 */
	virtual void visit(Return &node);

	//! visit a Proc node
	/** @param node node that gets visited.
	 */
	virtual void visit(Proc &node);

	//! visit an Update node
	/** @param node node that gets visited.
	 */
	virtual void visit(Update &node);

	//! visit a GetSig node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetSig &node);

	//! visit a GetSimTime node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetSimTime &node);

	//! visit an IMul node
	/** @param node node that gets visited.
	 */
	virtual void visit(IMul &node);

	//! visit a Div node
	/** @param node node that gets visited.
	 */
	virtual void visit(Div &node);

	//! visit an ROffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(ROffset &node);

	//! visit an AOffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(AOffset &node);

	//! visit a Suspend node
	/** @param node node that gets visited.
	 */
	virtual void visit(Suspend &node);

	//! visit a WakeOn node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeOn &node);

	//! visit a WakeAt node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeAt &node);

	//! visit a Log node
	/** @param node node that gets visited.
	 */
	virtual void visit(Log &node);

	//! visit an IndirectOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(IndirectOperand &node);

	//! visit a Reference node
	/** @param node node that gets visited.
	 */
	virtual void visit(Reference &node);

	//! visit a Register node
	/** @param node node that gets visited.
	 */
	virtual void visit(Register &node);

	//! visit a Data node
	/** @param node node that gets visited.
	 */
	virtual void visit(Data &node);

	//! visit a TypeElement node
	/** @param node node that gets visited.
	 */
	virtual void visit(TypeElement &node);

	//! visit a Type node
	/** @param node node that gets visited.
	 */
	virtual void visit(Type &node);

	//! process a conditional jmp
	/** @param node conditional jump opcode. 
	 *  @oaram op operator symbol to draw.
	 */
	template <typename T>
	void processCondJmp(T &node, const char *op);

	//! process an arithmetic node.
	/** @param node arithmetic node
	 *  @param op operator symbol to draw.
	 */
	template <typename T>
	void processArith(T &node, const char *op);

	//! process a record type
	/** @param node record type to process */
	void processRecord(Type &node);

	//! process an array type
	/** @param node array type to process */
	void processArray(Type &node);

	//! process a record type element
	/** @param node type element of a record type to process.
	 *  @param number element number.
	 */
	void processRecordElement(TypeElement &node, unsigned int number);

	//! generate variables for virtual Registers of a CodeContainer
	/** @param regFab RegisterFactory instance of the CodeContainer, 
	 *         which keeps track of the virtual registers associated
	 *         with the container.
	 */
	void genRegisters(const RegisterFactory &regFab);

	//! create statements to initialize data and signals
	/** @param container Code container in question.
	 */
	void initializeContainerData(const CodeContainer &container);

	//! create statements to initializie one data element.
	/** @param element data element in question.
	 */
	void initializeData(const Data &data);

	//! create the initializer for a variable
	/** @param data Variable declaration in question.
	 */
	void createVariable(const Data &data);

	//! create a driver
	/** @param data Driver declaration.
	 */
	void createDriver (const Data &data);

	//! create a signal
	/** @param data Signal declaration.
	 */
	void createSignal (const Data &data);

	//! put an OpType to this->dst
	/** @param t print it on this->dst.
	 */
	void putOpType(const enum OpType t);

	//! chain stack frames.
	/** @param callee Reference to the callee
	 */
	void chainStackFrames(const Reference &callee);

	//! create the prelude (includes etc.).
	void prelude(void);

	//! create the final function that maps vhdl names to function calls
	/* FIXME not yet used */
	void finale(void);

	//! stream to put nodes to.
	std::ostream &dst;

	//! nesting level of current code container.
	unsigned int nestingLevel;

	//! list of functions that can be called by name
	/** currently the functions are restricted to those without 
	 *  parameters.
	 */
	std::list<const std::string *> funcs;

	//! add the finale?
	bool addFinale;
}; 

}; /* namespace intermediate */

#endif /* __GEN_CCODE_HPP_INCLUDED */
