/*
 * Copyright 2011, Blender Foundation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contributor: 
 *		Jeroen Bakker 
 *		Monique Dewanchand
 */

#ifndef __COM_NODEBASE_H__
#define __COM_NODEBASE_H__

#include "COM_InputSocket.h"
#include "COM_OutputSocket.h"
#include "DNA_node_types.h"
#include "BKE_text.h"
#include <vector>
#include <string>

using namespace std;


class NodeOperation;
class ExecutionSystem;

/**
 * @brief The NodeBase class is the super-class of all node related objects like @see Node @see NodeOperation
 * the reason for the existence of this class is to support graph-nodes when using ExecutionSystem
 * the NodeBase also contains the reference to InputSocket and OutputSocket.
 * @ingroup Model
 */
class NodeBase {
private:
	/**
	 * @brief the list of actual inputsockets @see InputSocket
	 */
	vector<InputSocket *> m_inputsockets;

	/**
	 * @brief the list of actual outputsockets @see OutputSocket
	 */
	vector<OutputSocket *> m_outputsockets;

	/**
	 * @brief stores the reference to the SDNA bNode struct
	 */
	bNode *m_editorNode;

	/**
	 * @brief stores the reference to the SDNA bNode struct
	 */
	bNodeTree *m_editorNodeTree;

protected:
	/**
	 * @brief get access to the vector of input sockets
	 */
	inline vector<InputSocket *>& getInputSockets() { return this->m_inputsockets; }
	
	/**
	 * @brief get access to the vector of input sockets
	 */
	inline vector<OutputSocket *>& getOutputSockets() { return this->m_outputsockets; }


protected:
	/**
	 * @brief destructor
	 * clean up memory related to this NodeBase.
	 */
	virtual ~NodeBase();
	
public:
	/**
	 * @brief get the reference to the SDNA bNode struct
	 */
	bNode *getbNode() const {return m_editorNode;}

	/**
	 * @brief get the reference to the SDNA bNodeTree struct
	 */
	bNodeTree *getbNodeTree() const {return m_editorNodeTree;}
	
	/**
	 * @brief set the reference to the bNode
	 * @note used in Node instances to receive the storage/settings and complex node for highlight during execution
	 * @param bNode
	 */
	void setbNode(bNode *node) {this->m_editorNode = node;}
	
	/**
	 * @brief set the reference to the bNodeTree
	 * @param bNodeTree
	 */
	void setbNodeTree(bNodeTree *nodetree) {this->m_editorNodeTree = nodetree;}
	
	/**
	 * @brief is this node an operation?
	 * This is true when the instance is of the subclass NodeOperation.
	 * @return [true:false]
	 * @see NodeOperation
	 */
	virtual const bool isOperation() const { return false; }
	
	/**
	 * @brief check if this is an input node
	 * An input node is a node that only has output sockets and no input sockets
	 * @return [false..true]
	 */
	const bool isInputNode() const;
	
	/**
	 * @brief Return the number of input sockets of this node.
	 */
	const unsigned int getNumberOfInputSockets() const { return this->m_inputsockets.size(); }

	/**
	 * @brief Return the number of output sockets of this node.
	 */
	const unsigned int getNumberOfOutputSockets() const { return this->m_outputsockets.size(); }

	/**
	 * get the reference to a certain outputsocket
	 * @param index
	 * the index of the needed outputsocket
	 */
	OutputSocket *getOutputSocket(const unsigned int index);
	
	/**
	 * get the reference to the first outputsocket
	 * @param index
	 * the index of the needed outputsocket
	 */
	inline OutputSocket *getOutputSocket() { return getOutputSocket(0); }
	
	/**
	 * get the reference to a certain inputsocket
	 * @param index
	 * the index of the needed inputsocket
	 */
	InputSocket *getInputSocket(const unsigned int index);
	
	virtual bool isStatic() const { return false; }
	void getStaticValues(float *result) const { }

protected:
	NodeBase();
	
	/**
	 * @brief add an InputSocket to the collection of inputsockets
	 * @note may only be called in an constructor
	 * @param socket the InputSocket to add
	 */
	void addInputSocket(DataType datatype);
	void addInputSocket(DataType datatype, InputSocketResizeMode resizeMode);
	void addInputSocket(DataType datatype, InputSocketResizeMode resizeMode, bNodeSocket *socket);
	
	/**
	 * @brief add an OutputSocket to the collection of outputsockets
	 * @note may only be called in an constructor
	 * @param socket the OutputSocket to add
	 */
	void addOutputSocket(DataType datatype);
	void addOutputSocket(DataType datatype, bNodeSocket *socket);


#ifdef WITH_CXX_GUARDEDALLOC
	MEM_CXX_CLASS_ALLOC_FUNCS("COM:NodeBase")
#endif
};

#endif  /* __COM_NODEBASE_H__ */
