// -*- Mode: C++; tab-width: 2; -*-
// vi: set ts=2:
//

#include <BALL/FORMAT/molFileFactory.h>
#include <BALL/FORMAT/dockResultFile.h>
#include <BALL/FORMAT/commandlineParser.h>
#include <BALL/DOCKING/COMMON/structurePreparer.h>
#include <BALL/DOCKING/COMMON/dockingAlgorithm.h>
#include <BALL/SCORING/FUNCTIONS/gridedMM.h>
#include <BALL/SCORING/FUNCTIONS/MMScoring.h>
#include <BALL/SCORING/FUNCTIONS/PBScoring.h>
#include <BALL/SCORING/FUNCTIONS/gridedPLP.h>
#include <BALL/SCORING/FUNCTIONS/PLPScoring.h>
#include <BALL/SCORING/FUNCTIONS/rescoring3D.h>
#include <BALL/SCORING/FUNCTIONS/rescoring4D.h>
#include <BALL/SCORING/FUNCTIONS/rescoring1D.h>
#include <BALL/SCORING/COMMON/rescorer.h>
#include "version.h"

using namespace BALL;

int main(int argc, char* argv[])
{
	CommandlineParser par("TaGRes", "Target-specific Grid-Rescoring", VERSION, String(__DATE__), "Rescoring");
	par.registerParameter("rec", "receptor pdb-file", INFILE, true);
	par.registerParameter("rl", "reference-ligand", INFILE, true);
	par.registerParameter(DockingAlgorithm::OPTION_FILE_PARAMETER_NAME, "configuration file", INFILE);
	par.registerParameter("i", "compounds to be rescored", INFILE, true);
	par.registerParameter("mod", "model-file generated by TaGRes-train", INFILE, true);
	par.registerParameter("tf", "top-scored fraction of compounds not to be rescored", DOUBLE);
	par.registerParameter("o", "rescored compounds", OUTFILE, true);
	par.registerParameter("write_ini", "write ini-file w/ default parameters (and don't do anything else)", OUTFILE);
	par.registerParameter("method", "rescoring type: 'Rescoring3D' or 'Rescoring4D', or 'Rescoring1D'", STRING, true);
	par.registerParameter("function", "scoring function: 'MM' or 'PLP'", STRING);
	par.registerFlag("rm", "remove input file when finished");

	String man = "This tool rescores docking output poses using Target-specific Grid-Rescoring.\nPlease generate a regression model for binding-affinity approximation for your protein target by use of the tool TaGRes-train before using this tool.\nAs input TaGRes needs:\n\n\
    * a file containing a protonated protein in pdb-format\n\
    * a file containing a reference ligand. This reference ligand should be located in the binding pocket. Supported formats are mol2, sdf or drf (DockResultFile, xml-based).\n\
    * a file containing the compounds that are to be rescored. Supported formats are mol2, sdf or drf (DockResultFile, xml-based). Those compound should have been docked into the specified protein.\n\
    * a regression model file as generated by TaGRes-train for same protein target than the one specified here.\n\nTaGRes will evaluate each given input pose with a scoring function and apply the specified regression model to the score contributions generated this way, resulting in a re-score value, i.e. a (probably) enhanced approximation of the compound's binding-free-energy.\n\nOutput of this tool is a file in the same format as the input ligand file containing all compounds with scores obtained by rescoring in form of a property 're-score'.";
	par.setToolManual(man);
	list<String> slist;
	slist.push_back("Rescoring3D");
	slist.push_back("Rescoring4D");
	slist.push_back("Rescoring1D");
	par.setParameterRestrictions("method",slist);
	slist.clear();
	slist.push_back("MM");
	slist.push_back("PLP");
	par.setParameterRestrictions("function",slist);
	par.setParameterRestrictions("tf",0,1);
	par.setSupportedFormats("rec","pdb");
	par.setSupportedFormats("rl",MolFileFactory::getSupportedFormats());
	par.setSupportedFormats(DockingAlgorithm::OPTION_FILE_PARAMETER_NAME,"ini");
	par.setSupportedFormats("i",MolFileFactory::getSupportedFormats());
	par.setSupportedFormats("mod","mod");
	par.setSupportedFormats("o","mol2,sdf,drf");
	par.setSupportedFormats("write_ini","ini");
	par.setOutputFormatSource("o","i");
	Options default_options;
	ScoringFunction::getDefaultOptions(default_options);
	par.registerAdvancedParameters(default_options);
	par.setSupportedFormats("filename","ini");
	par.parse(argc, argv);


	int status = Rescorer::runRescoring(par, false, false);

	if (status == 0 && par.has("rm"))
	{
		File::remove(par.get("i"));
	}
	return status;
}
