/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_hybrid_fp32_mla_4x24_a55 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const float *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const float *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 124f\n"
      "cmp %x[M], #0x2\n"
      "bgt 83f\n"
      "beq 42f\n"
      "mov x17, %x[bias]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "cbz x17, 3f\n"
      "ldr q8, [x17, #0x0]\n"
      "ldr q9, [x17, #0x10]\n"
      "ldr q10, [x17, #0x20]\n"
      "ldr q11, [x17, #0x30]\n"
      "ldr q12, [x17, #0x40]\n"
      "ldr q13, [x17, #0x50]\n"
      "add x17, x17, #0x60\n"
      "b 18f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 17f\n"
      "cmp x16, #0x18\n"
      "bge 16f\n"
      "tbz x16, #4, 7f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v11.4s }, [x14], #0x10\n"
      "tbz x16, #2, 5f\n"
      "ld1 { v12.4s }, [x14], #0x10\n"
      "tbz x16, #1, 4f\n"
      "ldr d13, [x14], #0x8\n"
      "mov x20, #0x58\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v13.s }[2], [x14]\n"
      "b 15f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x16, #0, 15f\n"
      "ldr s13, [x14, #0x0]\n"
      "b 15f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_16
      "tbz x16, #1, 6f\n"
      "ldr d12, [x14], #0x8\n"
      "mov x20, #0x48\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v12.s }[2], [x14]\n"
      "b 15f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x16, #0, 15f\n"
      "ldr s12, [x14, #0x0]\n"
      "b 15f\n"
      "7:"  // Height 1: Partial accumulate: partial_8_0
      "tbz x16, #3, 11f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "tbz x16, #2, 9f\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "tbz x16, #1, 8f\n"
      "ldr d11, [x14], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v11.s }[2], [x14]\n"
      "b 15f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x16, #0, 15f\n"
      "ldr s11, [x14, #0x0]\n"
      "b 15f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x16, #1, 10f\n"
      "ldr d10, [x14], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v10.s }[2], [x14]\n"
      "b 15f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x16, #0, 15f\n"
      "ldr s10, [x14, #0x0]\n"
      "b 15f\n"
      "11:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x16, #2, 13f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "tbz x16, #1, 12f\n"
      "ldr d9, [x14], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v9.s }[2], [x14]\n"
      "b 15f\n"
      "12:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x16, #0, 15f\n"
      "ldr s9, [x14, #0x0]\n"
      "b 15f\n"
      "13:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x16, #1, 14f\n"
      "ldr d8, [x14], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x16, #0, 15f\n"
      "ld1 { v8.s }[2], [x14]\n"
      "b 15f\n"
      "14:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s8, [x14, #0x0]\n"
      "mov x20, #0x0\n"
      "15:"  // Height 1: Partial accumulate: Done
      "sub x14, x14, x20\n"
      "b 18f\n"
      "16:"  // Height 1: full accumulate
      "ldr q8, [x14, #0x0]\n"
      "ldr q9, [x14, #0x10]\n"
      "ldr q10, [x14, #0x20]\n"
      "ldr q11, [x14, #0x30]\n"
      "ldr q12, [x14, #0x40]\n"
      "ldr q13, [x14, #0x50]\n"
      "b 18f\n"
      "17:"  // Height 1: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "18:"  // Height 1: setup done
      "mov x13, #0x0\n"
      "19:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w12, [x20, x13, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 20f\n"
      "ldr x20, [%x[input_ptr], x13, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x11, [x20, #0x0]\n"
      "cbnz x13, 21f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x11, x11, x20, LSL #2\n"
      "b 21f\n"
      "20:"  // Height 1: setup direct input
      "mov x11, %x[input_ptr]\n"
      "21:"  // Height 1: input setup done
      "cmp x12, #0x4\n"
      "blt 24f\n"
      "ldr q0, [x11, #0x0]\n"
      "cmp x12, #0x8\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "blt 23f\n"
      "22:"  // Height 1: Multiply loop: Main loop head
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "ldr d19, [x15, #0x40]\n"
      "ldr x20, [x15, #0x48]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "ldr d18, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "ldr d17, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "ldr d16, [x15, #0x70]\n"
      "mov v19.d[1], x20\n"
      "ldr x20, [x15, #0x58]\n"
      "mov v18.d[1], x20\n"
      "ldr x20, [x15, #0x68]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x15, #0x78]\n"
      "mov v16.d[1], x20\n"
      "fmla v12.4s, v19.4s, v0.s[0]\n"
      "ldr d19, [x15, #0x80]\n"
      "ldr x20, [x15, #0x88]\n"
      "fmla v13.4s, v18.4s, v0.s[0]\n"
      "ldr d18, [x15, #0x90]\n"
      "fmla v8.4s, v17.4s, v0.s[1]\n"
      "ldr d17, [x15, #0xa0]\n"
      "fmla v9.4s, v16.4s, v0.s[1]\n"
      "ldr d16, [x15, #0xb0]\n"
      "mov v19.d[1], x20\n"
      "ldr x20, [x15, #0x98]\n"
      "mov v18.d[1], x20\n"
      "ldr x20, [x15, #0xa8]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x15, #0xb8]\n"
      "mov v16.d[1], x20\n"
      "fmla v10.4s, v19.4s, v0.s[1]\n"
      "ldr d19, [x15, #0xc0]\n"
      "ldr x20, [x15, #0xc8]\n"
      "fmla v11.4s, v18.4s, v0.s[1]\n"
      "ldr d18, [x15, #0xd0]\n"
      "fmla v12.4s, v17.4s, v0.s[1]\n"
      "ldr d17, [x15, #0xe0]\n"
      "fmla v13.4s, v16.4s, v0.s[1]\n"
      "ldr d16, [x15, #0xf0]\n"
      "mov v19.d[1], x20\n"
      "ldr x20, [x15, #0xd8]\n"
      "mov v18.d[1], x20\n"
      "ldr x20, [x15, #0xe8]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x15, #0xf8]\n"
      "mov v16.d[1], x20\n"
      "fmla v8.4s, v19.4s, v0.s[2]\n"
      "ldr d19, [x15, #0x100]\n"
      "ldr x20, [x15, #0x108]\n"
      "fmla v9.4s, v18.4s, v0.s[2]\n"
      "ldr d18, [x15, #0x110]\n"
      "fmla v10.4s, v17.4s, v0.s[2]\n"
      "ldr d17, [x15, #0x120]\n"
      "fmla v11.4s, v16.4s, v0.s[2]\n"
      "ldr d16, [x15, #0x130]\n"
      "mov v19.d[1], x20\n"
      "ldr x20, [x15, #0x118]\n"
      "mov v18.d[1], x20\n"
      "ldr x20, [x15, #0x128]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x15, #0x138]\n"
      "mov v16.d[1], x20\n"
      "fmla v12.4s, v19.4s, v0.s[2]\n"
      "ldr d19, [x15, #0x140]\n"
      "ldr x20, [x15, #0x148]\n"
      "fmla v13.4s, v18.4s, v0.s[2]\n"
      "ldr d18, [x15, #0x150]\n"
      "fmla v8.4s, v17.4s, v0.s[3]\n"
      "ldr d17, [x15, #0x160]\n"
      "fmla v9.4s, v16.4s, v0.s[3]\n"
      "ldr d16, [x15, #0x170]\n"
      "mov v19.d[1], x20\n"
      "ldr x20, [x15, #0x158]\n"
      "mov v18.d[1], x20\n"
      "ldr x20, [x15, #0x168]\n"
      "mov v17.d[1], x20\n"
      "ldr x20, [x15, #0x178]\n"
      "mov v16.d[1], x20\n"
      "add x11, x11, #0x10\n"
      "add x15, x15, #0x180\n"
      "fmla v10.4s, v19.4s, v0.s[3]\n"
      "ldr d4, [x15, #0x0]\n"
      "ldr x20, [x15, #0x8]\n"
      "fmla v11.4s, v18.4s, v0.s[3]\n"
      "ldr d5, [x15, #0x10]\n"
      "fmla v12.4s, v17.4s, v0.s[3]\n"
      "ldr d6, [x15, #0x20]\n"
      "fmla v13.4s, v16.4s, v0.s[3]\n"
      "ldr d0, [x11, #0x0]\n"
      "sub x12, x12, #0x4\n"
      "ldr d7, [x15, #0x30]\n"
      "cmp x12, #0x8\n"
      "ldr x21, [x15, #0x18]\n"
      "mov v4.d[1], x20\n"
      "ldr x20, [x15, #0x28]\n"
      "mov v5.d[1], x21\n"
      "ldr x21, [x11, #0x8]\n"
      "mov v6.d[1], x20\n"
      "ldr x20, [x15, #0x38]\n"
      "mov v0.d[1], x21\n"
      "mov v7.d[1], x20\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "bge 22b\n"
      "23:"  // Height 1: Multiply loop: Single iteration only
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "ldr q19, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "ldr q18, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "ldr q17, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "ldr q16, [x15, #0x70]\n"
      "fmla v12.4s, v19.4s, v0.s[0]\n"
      "ldr q19, [x15, #0x80]\n"
      "fmla v13.4s, v18.4s, v0.s[0]\n"
      "ldr q18, [x15, #0x90]\n"
      "fmla v8.4s, v17.4s, v0.s[1]\n"
      "ldr q17, [x15, #0xa0]\n"
      "fmla v9.4s, v16.4s, v0.s[1]\n"
      "ldr q16, [x15, #0xb0]\n"
      "fmla v10.4s, v19.4s, v0.s[1]\n"
      "ldr q19, [x15, #0xc0]\n"
      "fmla v11.4s, v18.4s, v0.s[1]\n"
      "ldr q18, [x15, #0xd0]\n"
      "fmla v12.4s, v17.4s, v0.s[1]\n"
      "ldr q17, [x15, #0xe0]\n"
      "fmla v13.4s, v16.4s, v0.s[1]\n"
      "ldr q16, [x15, #0xf0]\n"
      "fmla v8.4s, v19.4s, v0.s[2]\n"
      "ldr q19, [x15, #0x100]\n"
      "fmla v9.4s, v18.4s, v0.s[2]\n"
      "ldr q18, [x15, #0x110]\n"
      "fmla v10.4s, v17.4s, v0.s[2]\n"
      "ldr q17, [x15, #0x120]\n"
      "fmla v11.4s, v16.4s, v0.s[2]\n"
      "ldr q16, [x15, #0x130]\n"
      "fmla v12.4s, v19.4s, v0.s[2]\n"
      "ldr q19, [x15, #0x140]\n"
      "fmla v13.4s, v18.4s, v0.s[2]\n"
      "ldr q18, [x15, #0x150]\n"
      "fmla v8.4s, v17.4s, v0.s[3]\n"
      "ldr q17, [x15, #0x160]\n"
      "fmla v9.4s, v16.4s, v0.s[3]\n"
      "ldr q16, [x15, #0x170]\n"
      "add x11, x11, #0x10\n"
      "sub x12, x12, #0x4\n"
      "fmla v10.4s, v19.4s, v0.s[3]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "fmla v11.4s, v18.4s, v0.s[3]\n"
      "add x15, x15, #0x180\n"
      "fmla v12.4s, v17.4s, v0.s[3]\n"
      "fmla v13.4s, v16.4s, v0.s[3]\n"
      "24:"  // Height 1: Multiply loop: Main loop skip
      "cbz x12, 26f\n"
      "25:"  // Height 1: Multiply loop: Odd block loop
      "ldr s17, [x11], #0x4\n"
      "sub x12, x12, #0x1\n"
      "ldr q16, [x15, #0x0]\n"
      "fmla v8.4s, v16.4s, v17.s[0]\n"
      "ldr q16, [x15, #0x10]\n"
      "fmla v9.4s, v16.4s, v17.s[0]\n"
      "ldr q16, [x15, #0x20]\n"
      "fmla v10.4s, v16.4s, v17.s[0]\n"
      "ldr q16, [x15, #0x30]\n"
      "fmla v11.4s, v16.4s, v17.s[0]\n"
      "ldr q16, [x15, #0x40]\n"
      "fmla v12.4s, v16.4s, v17.s[0]\n"
      "ldr q16, [x15, #0x50]\n"
      "fmla v13.4s, v16.4s, v17.s[0]\n"
      "add x15, x15, #0x60\n"
      "cbnz x12, 25b\n"
      "26:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x13, x13, #0x1\n"
      "cmp x13, x20\n"
      "bne 19b\n"
      "prfm pstl1keep, [x14, #0x0]\n"
      "tbz %x[flags], #1, 27f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v16.4s\n"
      "fmin v9.4s, v9.4s, v16.4s\n"
      "fmin v10.4s, v10.4s, v16.4s\n"
      "fmin v11.4s, v11.4s, v16.4s\n"
      "fmin v12.4s, v12.4s, v16.4s\n"
      "fmin v13.4s, v13.4s, v16.4s\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmax v8.4s, v8.4s, v16.4s\n"
      "fmax v9.4s, v9.4s, v16.4s\n"
      "fmax v10.4s, v10.4s, v16.4s\n"
      "fmax v11.4s, v11.4s, v16.4s\n"
      "fmax v12.4s, v12.4s, v16.4s\n"
      "fmax v13.4s, v13.4s, v16.4s\n"
      "27:"  // Height 1: No activation
      "cmp x16, #0x18\n"
      "bge 40f\n"
      "tbz x16, #4, 31f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v11.4s }, [x14], #0x10\n"
      "tbz x16, #2, 29f\n"
      "st1 { v12.4s }, [x14], #0x10\n"
      "tbz x16, #1, 28f\n"
      "str d13, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v13.s }[2], [x14]\n"
      "b 39f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_20
      "tbz x16, #0, 39f\n"
      "str s13, [x14, #0x0]\n"
      "b 39f\n"
      "29:"  // Height 1: Partial direct writeback: partial_2_16
      "tbz x16, #1, 30f\n"
      "str d12, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v12.s }[2], [x14]\n"
      "b 39f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_16
      "tbz x16, #0, 39f\n"
      "str s12, [x14, #0x0]\n"
      "b 39f\n"
      "31:"  // Height 1: Partial direct writeback: partial_8_0
      "tbz x16, #3, 35f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "tbz x16, #2, 33f\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "tbz x16, #1, 32f\n"
      "str d11, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v11.s }[2], [x14]\n"
      "b 39f\n"
      "32:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x16, #0, 39f\n"
      "str s11, [x14, #0x0]\n"
      "b 39f\n"
      "33:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x16, #1, 34f\n"
      "str d10, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v10.s }[2], [x14]\n"
      "b 39f\n"
      "34:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x16, #0, 39f\n"
      "str s10, [x14, #0x0]\n"
      "b 39f\n"
      "35:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x16, #2, 37f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "tbz x16, #1, 36f\n"
      "str d9, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v9.s }[2], [x14]\n"
      "b 39f\n"
      "36:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x16, #0, 39f\n"
      "str s9, [x14, #0x0]\n"
      "b 39f\n"
      "37:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x16, #1, 38f\n"
      "str d8, [x14], #0x8\n"
      "tbz x16, #0, 39f\n"
      "st1 { v8.s }[2], [x14]\n"
      "b 39f\n"
      "38:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x14, #0x0]\n"
      "39:"  // Height 1: Partial direct writeback: Done
      "b 41f\n"
      "40:"  // Height 1: Full writeback
      "str q8, [x14, #0x0]\n"
      "str q9, [x14, #0x10]\n"
      "str q10, [x14, #0x20]\n"
      "str q11, [x14, #0x30]\n"
      "str q12, [x14, #0x40]\n"
      "str q13, [x14, #0x50]\n"
      "add x14, x14, #0x60\n"
      "41:"  // Height 1: Writeback done
      "subs x16, x16, #0x18\n"
      "bgt 2b\n"
      "b 166f\n"
      "42:"  // Height 2
      "mov x17, %x[bias]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[output_ptr]\n"
      "43:"  // Height 2: Column loop
      "cbz x17, 44f\n"
      "ldr q8, [x17, #0x0]\n"
      "mov v14.16b, v8.16b\n"
      "ldr q9, [x17, #0x10]\n"
      "mov v15.16b, v9.16b\n"
      "ldr q10, [x17, #0x20]\n"
      "mov v16.16b, v10.16b\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v17.16b, v11.16b\n"
      "ldr q12, [x17, #0x40]\n"
      "mov v18.16b, v12.16b\n"
      "ldr q13, [x17, #0x50]\n"
      "mov v19.16b, v13.16b\n"
      "add x17, x17, #0x60\n"
      "b 59f\n"
      "44:"  // Height 2: no bias
      "tbz %x[flags], #0, 58f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x16, #0x18\n"
      "add x23, x14, x20, LSL #2\n"
      "bge 57f\n"
      "tbz x16, #4, 48f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v11.4s }, [x14], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "tbz x16, #2, 46f\n"
      "ld1 { v12.4s }, [x14], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "tbz x16, #1, 45f\n"
      "ldr d13, [x14], #0x8\n"
      "mov x20, #0x58\n"
      "ldr d19, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v13.s }[2], [x14]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "b 56f\n"
      "45:"  // Height 2: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x16, #0, 56f\n"
      "ldr s13, [x14, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "b 56f\n"
      "46:"  // Height 2: Partial accumulate: partial_2_16
      "tbz x16, #1, 47f\n"
      "ldr d12, [x14], #0x8\n"
      "mov x20, #0x48\n"
      "ldr d18, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v12.s }[2], [x14]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "b 56f\n"
      "47:"  // Height 2: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x16, #0, 56f\n"
      "ldr s12, [x14, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "b 56f\n"
      "48:"  // Height 2: Partial accumulate: partial_8_0
      "tbz x16, #3, 52f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "tbz x16, #2, 50f\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "tbz x16, #1, 49f\n"
      "ldr d11, [x14], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d17, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v11.s }[2], [x14]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "b 56f\n"
      "49:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x16, #0, 56f\n"
      "ldr s11, [x14, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "b 56f\n"
      "50:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x16, #1, 51f\n"
      "ldr d10, [x14], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d16, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v10.s }[2], [x14]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "b 56f\n"
      "51:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x16, #0, 56f\n"
      "ldr s10, [x14, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "b 56f\n"
      "52:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x16, #2, 54f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "tbz x16, #1, 53f\n"
      "ldr d9, [x14], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d15, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v9.s }[2], [x14]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "b 56f\n"
      "53:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x16, #0, 56f\n"
      "ldr s9, [x14, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "b 56f\n"
      "54:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x16, #1, 55f\n"
      "ldr d8, [x14], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "tbz x16, #0, 56f\n"
      "ld1 { v8.s }[2], [x14]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "b 56f\n"
      "55:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s8, [x14, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s14, [x23, #0x0]\n"
      "56:"  // Height 2: Partial accumulate: Done
      "sub x14, x14, x20\n"
      "b 59f\n"
      "57:"  // Height 2: full accumulate
      "ldr q8, [x14, #0x0]\n"
      "ldr q9, [x14, #0x10]\n"
      "ldr q10, [x14, #0x20]\n"
      "ldr q11, [x14, #0x30]\n"
      "ldr q12, [x14, #0x40]\n"
      "ldr q13, [x14, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "b 59f\n"
      "58:"  // Height 2: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "59:"  // Height 2: setup done
      "mov x13, #0x0\n"
      "60:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w12, [x20, x13, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 61f\n"
      "ldr x20, [%x[input_ptr], x13, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x11, [x20, #0x0]\n"
      "ldr x10, [x20, #0x8]\n"
      "cbnz x13, 62f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x11, x11, x20, LSL #2\n"
      "add x10, x10, x20, LSL #2\n"
      "b 62f\n"
      "61:"  // Height 2: setup direct input
      "mov x11, %x[input_ptr]\n"
      "add x10, x11, x21, LSL #2\n"
      "62:"  // Height 2: input setup done
      "cmp x12, #0x4\n"
      "blt 65f\n"
      "ldr q0, [x11, #0x0]\n"
      "cmp x12, #0x8\n"
      "ldr q1, [x10, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "blt 64f\n"
      "63:"  // Height 2: Multiply loop: Main loop head
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "ldr x23, [x15, #0x48]\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "ldr d23, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "ldr x22, [x15, #0x58]\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "ldr d22, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "ldr x21, [x15, #0x68]\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "ldr d21, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "ldr x20, [x15, #0x78]\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "ldr d20, [x15, #0x70]\n"
      "mov v23.d[1], x23\n"
      "fmla v12.4s, v23.4s, v0.s[0]\n"
      "mov v22.d[1], x22\n"
      "fmla v18.4s, v23.4s, v1.s[0]\n"
      "ldr d23, [x15, #0x80]\n"
      "mov v21.d[1], x21\n"
      "mov v20.d[1], x20\n"
      "ldr x23, [x15, #0x88]\n"
      "fmla v13.4s, v22.4s, v0.s[0]\n"
      "ldr x22, [x15, #0x98]\n"
      "fmla v19.4s, v22.4s, v1.s[0]\n"
      "ldr d22, [x15, #0x90]\n"
      "fmla v8.4s, v21.4s, v0.s[1]\n"
      "ldr x21, [x15, #0xa8]\n"
      "fmla v14.4s, v21.4s, v1.s[1]\n"
      "ldr d21, [x15, #0xa0]\n"
      "fmla v9.4s, v20.4s, v0.s[1]\n"
      "ldr x20, [x15, #0xb8]\n"
      "fmla v15.4s, v20.4s, v1.s[1]\n"
      "ldr d20, [x15, #0xb0]\n"
      "mov v23.d[1], x23\n"
      "fmla v10.4s, v23.4s, v0.s[1]\n"
      "mov v22.d[1], x22\n"
      "fmla v16.4s, v23.4s, v1.s[1]\n"
      "ldr d23, [x15, #0xc0]\n"
      "mov v21.d[1], x21\n"
      "mov v20.d[1], x20\n"
      "ldr x23, [x15, #0xc8]\n"
      "fmla v11.4s, v22.4s, v0.s[1]\n"
      "ldr x22, [x15, #0xd8]\n"
      "fmla v17.4s, v22.4s, v1.s[1]\n"
      "ldr d22, [x15, #0xd0]\n"
      "fmla v12.4s, v21.4s, v0.s[1]\n"
      "ldr x21, [x15, #0xe8]\n"
      "fmla v18.4s, v21.4s, v1.s[1]\n"
      "ldr d21, [x15, #0xe0]\n"
      "fmla v13.4s, v20.4s, v0.s[1]\n"
      "ldr x20, [x15, #0xf8]\n"
      "fmla v19.4s, v20.4s, v1.s[1]\n"
      "ldr d20, [x15, #0xf0]\n"
      "mov v23.d[1], x23\n"
      "fmla v8.4s, v23.4s, v0.s[2]\n"
      "mov v22.d[1], x22\n"
      "fmla v14.4s, v23.4s, v1.s[2]\n"
      "ldr d23, [x15, #0x100]\n"
      "mov v21.d[1], x21\n"
      "mov v20.d[1], x20\n"
      "ldr x23, [x15, #0x108]\n"
      "fmla v9.4s, v22.4s, v0.s[2]\n"
      "ldr x22, [x15, #0x118]\n"
      "fmla v15.4s, v22.4s, v1.s[2]\n"
      "ldr d22, [x15, #0x110]\n"
      "fmla v10.4s, v21.4s, v0.s[2]\n"
      "ldr x21, [x15, #0x128]\n"
      "fmla v16.4s, v21.4s, v1.s[2]\n"
      "ldr d21, [x15, #0x120]\n"
      "fmla v11.4s, v20.4s, v0.s[2]\n"
      "ldr x20, [x15, #0x138]\n"
      "fmla v17.4s, v20.4s, v1.s[2]\n"
      "ldr d20, [x15, #0x130]\n"
      "mov v23.d[1], x23\n"
      "fmla v12.4s, v23.4s, v0.s[2]\n"
      "mov v22.d[1], x22\n"
      "fmla v18.4s, v23.4s, v1.s[2]\n"
      "ldr d23, [x15, #0x140]\n"
      "mov v21.d[1], x21\n"
      "mov v20.d[1], x20\n"
      "ldr x23, [x15, #0x148]\n"
      "fmla v13.4s, v22.4s, v0.s[2]\n"
      "ldr x22, [x15, #0x158]\n"
      "fmla v19.4s, v22.4s, v1.s[2]\n"
      "ldr d22, [x15, #0x150]\n"
      "fmla v8.4s, v21.4s, v0.s[3]\n"
      "ldr x21, [x15, #0x168]\n"
      "fmla v14.4s, v21.4s, v1.s[3]\n"
      "ldr d21, [x15, #0x160]\n"
      "fmla v9.4s, v20.4s, v0.s[3]\n"
      "ldr x20, [x15, #0x178]\n"
      "fmla v15.4s, v20.4s, v1.s[3]\n"
      "ldr d20, [x15, #0x170]\n"
      "mov v23.d[1], x23\n"
      "add x11, x11, #0x10\n"
      "mov v22.d[1], x22\n"
      "add x10, x10, #0x10\n"
      "mov v21.d[1], x21\n"
      "add x15, x15, #0x180\n"
      "mov v20.d[1], x20\n"
      "fmla v10.4s, v23.4s, v0.s[3]\n"
      "fmla v16.4s, v23.4s, v1.s[3]\n"
      "ldr d4, [x15, #0x0]\n"
      "ldr x21, [x15, #0x8]\n"
      "fmla v11.4s, v22.4s, v0.s[3]\n"
      "fmla v17.4s, v22.4s, v1.s[3]\n"
      "ldr d5, [x15, #0x10]\n"
      "ldr x20, [x15, #0x18]\n"
      "fmla v12.4s, v21.4s, v0.s[3]\n"
      "fmla v18.4s, v21.4s, v1.s[3]\n"
      "ldr d6, [x15, #0x20]\n"
      "ldr x23, [x15, #0x28]\n"
      "fmla v13.4s, v20.4s, v0.s[3]\n"
      "ldr d0, [x11, #0x0]\n"
      "fmla v19.4s, v20.4s, v1.s[3]\n"
      "ldr d1, [x10, #0x0]\n"
      "sub x12, x12, #0x4\n"
      "ldr d7, [x15, #0x30]\n"
      "cmp x12, #0x8\n"
      "ldr x22, [x11, #0x8]\n"
      "mov v4.d[1], x21\n"
      "ldr x21, [x10, #0x8]\n"
      "mov v5.d[1], x20\n"
      "ldr x20, [x15, #0x38]\n"
      "mov v6.d[1], x23\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "mov v0.d[1], x22\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "mov v1.d[1], x21\n"
      "mov v7.d[1], x20\n"
      "bge 63b\n"
      "64:"  // Height 2: Multiply loop: Single iteration only
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "add x11, x11, #0x10\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "ldr q23, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "add x10, x10, #0x10\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "ldr q22, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "sub x12, x12, #0x4\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "ldr q21, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "ldr q20, [x15, #0x70]\n"
      "fmla v12.4s, v23.4s, v0.s[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "fmla v18.4s, v23.4s, v1.s[0]\n"
      "ldr q23, [x15, #0x80]\n"
      "fmla v13.4s, v22.4s, v0.s[0]\n"
      "fmla v19.4s, v22.4s, v1.s[0]\n"
      "ldr q22, [x15, #0x90]\n"
      "fmla v8.4s, v21.4s, v0.s[1]\n"
      "fmla v14.4s, v21.4s, v1.s[1]\n"
      "ldr q21, [x15, #0xa0]\n"
      "fmla v9.4s, v20.4s, v0.s[1]\n"
      "fmla v15.4s, v20.4s, v1.s[1]\n"
      "ldr q20, [x15, #0xb0]\n"
      "fmla v10.4s, v23.4s, v0.s[1]\n"
      "fmla v16.4s, v23.4s, v1.s[1]\n"
      "ldr q23, [x15, #0xc0]\n"
      "fmla v11.4s, v22.4s, v0.s[1]\n"
      "fmla v17.4s, v22.4s, v1.s[1]\n"
      "ldr q22, [x15, #0xd0]\n"
      "fmla v12.4s, v21.4s, v0.s[1]\n"
      "fmla v18.4s, v21.4s, v1.s[1]\n"
      "ldr q21, [x15, #0xe0]\n"
      "fmla v13.4s, v20.4s, v0.s[1]\n"
      "fmla v19.4s, v20.4s, v1.s[1]\n"
      "ldr q20, [x15, #0xf0]\n"
      "fmla v8.4s, v23.4s, v0.s[2]\n"
      "fmla v14.4s, v23.4s, v1.s[2]\n"
      "ldr q23, [x15, #0x100]\n"
      "fmla v9.4s, v22.4s, v0.s[2]\n"
      "fmla v15.4s, v22.4s, v1.s[2]\n"
      "ldr q22, [x15, #0x110]\n"
      "fmla v10.4s, v21.4s, v0.s[2]\n"
      "fmla v16.4s, v21.4s, v1.s[2]\n"
      "ldr q21, [x15, #0x120]\n"
      "fmla v11.4s, v20.4s, v0.s[2]\n"
      "fmla v17.4s, v20.4s, v1.s[2]\n"
      "ldr q20, [x15, #0x130]\n"
      "fmla v12.4s, v23.4s, v0.s[2]\n"
      "fmla v18.4s, v23.4s, v1.s[2]\n"
      "ldr q23, [x15, #0x140]\n"
      "fmla v13.4s, v22.4s, v0.s[2]\n"
      "fmla v19.4s, v22.4s, v1.s[2]\n"
      "ldr q22, [x15, #0x150]\n"
      "fmla v8.4s, v21.4s, v0.s[3]\n"
      "fmla v14.4s, v21.4s, v1.s[3]\n"
      "ldr q21, [x15, #0x160]\n"
      "fmla v9.4s, v20.4s, v0.s[3]\n"
      "fmla v15.4s, v20.4s, v1.s[3]\n"
      "ldr q20, [x15, #0x170]\n"
      "fmla v10.4s, v23.4s, v0.s[3]\n"
      "add x15, x15, #0x180\n"
      "fmla v16.4s, v23.4s, v1.s[3]\n"
      "fmla v11.4s, v22.4s, v0.s[3]\n"
      "fmla v17.4s, v22.4s, v1.s[3]\n"
      "fmla v12.4s, v21.4s, v0.s[3]\n"
      "fmla v18.4s, v21.4s, v1.s[3]\n"
      "fmla v13.4s, v20.4s, v0.s[3]\n"
      "fmla v19.4s, v20.4s, v1.s[3]\n"
      "65:"  // Height 2: Multiply loop: Main loop skip
      "cbz x12, 67f\n"
      "66:"  // Height 2: Multiply loop: Odd block loop
      "ldr s25, [x11], #0x4\n"
      "sub x12, x12, #0x1\n"
      "ldr s24, [x10], #0x4\n"
      "ldr q21, [x15, #0x0]\n"
      "fmla v8.4s, v21.4s, v25.s[0]\n"
      "ldr q20, [x15, #0x10]\n"
      "fmla v14.4s, v21.4s, v24.s[0]\n"
      "ldr q23, [x15, #0x20]\n"
      "fmla v9.4s, v20.4s, v25.s[0]\n"
      "ldr q22, [x15, #0x30]\n"
      "fmla v15.4s, v20.4s, v24.s[0]\n"
      "ldr q21, [x15, #0x40]\n"
      "fmla v10.4s, v23.4s, v25.s[0]\n"
      "ldr q20, [x15, #0x50]\n"
      "fmla v16.4s, v23.4s, v24.s[0]\n"
      "fmla v11.4s, v22.4s, v25.s[0]\n"
      "add x15, x15, #0x60\n"
      "fmla v17.4s, v22.4s, v24.s[0]\n"
      "fmla v12.4s, v21.4s, v25.s[0]\n"
      "fmla v18.4s, v21.4s, v24.s[0]\n"
      "fmla v13.4s, v20.4s, v25.s[0]\n"
      "fmla v19.4s, v20.4s, v24.s[0]\n"
      "cbnz x12, 66b\n"
      "67:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x13, x13, #0x1\n"
      "cmp x13, x20\n"
      "bne 60b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x14, x20, LSL #2\n"
      "prfm pstl1keep, [x14, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "tbz %x[flags], #1, 68f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v20.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v20.4s\n"
      "fmin v9.4s, v9.4s, v20.4s\n"
      "fmin v10.4s, v10.4s, v20.4s\n"
      "fmin v11.4s, v11.4s, v20.4s\n"
      "fmin v12.4s, v12.4s, v20.4s\n"
      "fmin v13.4s, v13.4s, v20.4s\n"
      "fmin v14.4s, v14.4s, v20.4s\n"
      "fmin v15.4s, v15.4s, v20.4s\n"
      "fmin v16.4s, v16.4s, v20.4s\n"
      "fmin v17.4s, v17.4s, v20.4s\n"
      "fmin v18.4s, v18.4s, v20.4s\n"
      "fmin v19.4s, v19.4s, v20.4s\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v20.4s }, [x20]\n"
      "fmax v8.4s, v8.4s, v20.4s\n"
      "fmax v9.4s, v9.4s, v20.4s\n"
      "fmax v10.4s, v10.4s, v20.4s\n"
      "fmax v11.4s, v11.4s, v20.4s\n"
      "fmax v12.4s, v12.4s, v20.4s\n"
      "fmax v13.4s, v13.4s, v20.4s\n"
      "fmax v14.4s, v14.4s, v20.4s\n"
      "fmax v15.4s, v15.4s, v20.4s\n"
      "fmax v16.4s, v16.4s, v20.4s\n"
      "fmax v17.4s, v17.4s, v20.4s\n"
      "fmax v18.4s, v18.4s, v20.4s\n"
      "fmax v19.4s, v19.4s, v20.4s\n"
      "68:"  // Height 2: No activation
      "cmp x16, #0x18\n"
      "bge 81f\n"
      "tbz x16, #4, 72f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v11.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "tbz x16, #2, 70f\n"
      "st1 { v12.4s }, [x14], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "tbz x16, #1, 69f\n"
      "str d13, [x14], #0x8\n"
      "str d19, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v13.s }[2], [x14]\n"
      "st1 { v19.s }[2], [x23]\n"
      "b 80f\n"
      "69:"  // Height 2: Partial direct writeback: partial_1_20
      "tbz x16, #0, 80f\n"
      "str s13, [x14, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "b 80f\n"
      "70:"  // Height 2: Partial direct writeback: partial_2_16
      "tbz x16, #1, 71f\n"
      "str d12, [x14], #0x8\n"
      "str d18, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v12.s }[2], [x14]\n"
      "st1 { v18.s }[2], [x23]\n"
      "b 80f\n"
      "71:"  // Height 2: Partial direct writeback: partial_1_16
      "tbz x16, #0, 80f\n"
      "str s12, [x14, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "b 80f\n"
      "72:"  // Height 2: Partial direct writeback: partial_8_0
      "tbz x16, #3, 76f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "tbz x16, #2, 74f\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "tbz x16, #1, 73f\n"
      "str d11, [x14], #0x8\n"
      "str d17, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v11.s }[2], [x14]\n"
      "st1 { v17.s }[2], [x23]\n"
      "b 80f\n"
      "73:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x16, #0, 80f\n"
      "str s11, [x14, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "b 80f\n"
      "74:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x16, #1, 75f\n"
      "str d10, [x14], #0x8\n"
      "str d16, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v10.s }[2], [x14]\n"
      "st1 { v16.s }[2], [x23]\n"
      "b 80f\n"
      "75:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x16, #0, 80f\n"
      "str s10, [x14, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "b 80f\n"
      "76:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x16, #2, 78f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "tbz x16, #1, 77f\n"
      "str d9, [x14], #0x8\n"
      "str d15, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v9.s }[2], [x14]\n"
      "st1 { v15.s }[2], [x23]\n"
      "b 80f\n"
      "77:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x16, #0, 80f\n"
      "str s9, [x14, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "b 80f\n"
      "78:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x16, #1, 79f\n"
      "str d8, [x14], #0x8\n"
      "str d14, [x23], #0x8\n"
      "tbz x16, #0, 80f\n"
      "st1 { v8.s }[2], [x14]\n"
      "st1 { v14.s }[2], [x23]\n"
      "b 80f\n"
      "79:"  // Height 2: Partial direct writeback: partial_1_0
      "str s8, [x14, #0x0]\n"
      "str s14, [x23, #0x0]\n"
      "80:"  // Height 2: Partial direct writeback: Done
      "b 82f\n"
      "81:"  // Height 2: Full writeback
      "str q8, [x14, #0x0]\n"
      "str q9, [x14, #0x10]\n"
      "str q10, [x14, #0x20]\n"
      "str q11, [x14, #0x30]\n"
      "str q12, [x14, #0x40]\n"
      "str q13, [x14, #0x50]\n"
      "add x14, x14, #0x60\n"
      "str q14, [x23, #0x0]\n"
      "str q15, [x23, #0x10]\n"
      "str q16, [x23, #0x20]\n"
      "str q17, [x23, #0x30]\n"
      "str q18, [x23, #0x40]\n"
      "str q19, [x23, #0x50]\n"
      "82:"  // Height 2: Writeback done
      "subs x16, x16, #0x18\n"
      "bgt 43b\n"
      "b 166f\n"
      "83:"  // Height 3
      "mov x17, %x[bias]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[output_ptr]\n"
      "84:"  // Height 3: Column loop
      "cbz x17, 85f\n"
      "ldr q8, [x17, #0x0]\n"
      "mov v14.16b, v8.16b\n"
      "ldr q9, [x17, #0x10]\n"
      "mov v15.16b, v9.16b\n"
      "ldr q10, [x17, #0x20]\n"
      "mov v16.16b, v10.16b\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v17.16b, v11.16b\n"
      "ldr q12, [x17, #0x40]\n"
      "mov v18.16b, v12.16b\n"
      "ldr q13, [x17, #0x50]\n"
      "mov v19.16b, v13.16b\n"
      "mov v20.16b, v8.16b\n"
      "add x17, x17, #0x60\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v12.16b\n"
      "mov v25.16b, v13.16b\n"
      "b 100f\n"
      "85:"  // Height 3: no bias
      "tbz %x[flags], #0, 99f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x14, x20, LSL #2\n"
      "cmp x16, #0x18\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 98f\n"
      "tbz x16, #4, 89f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v11.4s }, [x14], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "tbz x16, #2, 87f\n"
      "ld1 { v12.4s }, [x14], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v24.4s }, [x22], #0x10\n"
      "tbz x16, #1, 86f\n"
      "ldr d13, [x14], #0x8\n"
      "mov x20, #0x58\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d25, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v13.s }[2], [x14]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v25.s }[2], [x22]\n"
      "b 97f\n"
      "86:"  // Height 3: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x16, #0, 97f\n"
      "ldr s13, [x14, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s25, [x22, #0x0]\n"
      "b 97f\n"
      "87:"  // Height 3: Partial accumulate: partial_2_16
      "tbz x16, #1, 88f\n"
      "ldr d12, [x14], #0x8\n"
      "mov x20, #0x48\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d24, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v12.s }[2], [x14]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v24.s }[2], [x22]\n"
      "b 97f\n"
      "88:"  // Height 3: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x16, #0, 97f\n"
      "ldr s12, [x14, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s24, [x22, #0x0]\n"
      "b 97f\n"
      "89:"  // Height 3: Partial accumulate: partial_8_0
      "tbz x16, #3, 93f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "tbz x16, #2, 91f\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "tbz x16, #1, 90f\n"
      "ldr d11, [x14], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v11.s }[2], [x14]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "b 97f\n"
      "90:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x16, #0, 97f\n"
      "ldr s11, [x14, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "b 97f\n"
      "91:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x16, #1, 92f\n"
      "ldr d10, [x14], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v10.s }[2], [x14]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "b 97f\n"
      "92:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x16, #0, 97f\n"
      "ldr s10, [x14, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "b 97f\n"
      "93:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x16, #2, 95f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "tbz x16, #1, 94f\n"
      "ldr d9, [x14], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d15, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v9.s }[2], [x14]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "b 97f\n"
      "94:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x16, #0, 97f\n"
      "ldr s9, [x14, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "b 97f\n"
      "95:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x16, #1, 96f\n"
      "ldr d8, [x14], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "tbz x16, #0, 97f\n"
      "ld1 { v8.s }[2], [x14]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "b 97f\n"
      "96:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s8, [x14, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s14, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "97:"  // Height 3: Partial accumulate: Done
      "sub x14, x14, x20\n"
      "b 100f\n"
      "98:"  // Height 3: full accumulate
      "ldr q8, [x14, #0x0]\n"
      "ldr q9, [x14, #0x10]\n"
      "ldr q10, [x14, #0x20]\n"
      "ldr q11, [x14, #0x30]\n"
      "ldr q12, [x14, #0x40]\n"
      "ldr q13, [x14, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x22, #0x40]\n"
      "ldr q25, [x22, #0x50]\n"
      "b 100f\n"
      "99:"  // Height 3: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "100:"  // Height 3: setup done
      "mov x13, #0x0\n"
      "101:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w12, [x20, x13, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 102f\n"
      "ldr x20, [%x[input_ptr], x13, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x11, [x20, #0x0]\n"
      "ldr x10, [x20, #0x8]\n"
      "ldr x9, [x20, #0x10]\n"
      "cbnz x13, 103f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x11, x11, x20, LSL #2\n"
      "add x10, x10, x20, LSL #2\n"
      "add x9, x9, x20, LSL #2\n"
      "b 103f\n"
      "102:"  // Height 3: setup direct input
      "mov x11, %x[input_ptr]\n"
      "add x10, x11, x21, LSL #2\n"
      "add x9, x10, x21, LSL #2\n"
      "103:"  // Height 3: input setup done
      "cmp x12, #0x4\n"
      "blt 106f\n"
      "ldr q0, [x11, #0x0]\n"
      "cmp x12, #0x8\n"
      "ldr q1, [x10, #0x0]\n"
      "ldr q2, [x9, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "blt 105f\n"
      "104:"  // Height 3: Multiply loop: Main loop head
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "ldr x23, [x15, #0x48]\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "ldr x22, [x15, #0x58]\n"
      "fmla v20.4s, v4.4s, v2.s[0]\n"
      "ldr d29, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "ldr x21, [x15, #0x68]\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "ldr x20, [x15, #0x78]\n"
      "fmla v21.4s, v5.4s, v2.s[0]\n"
      "ldr d28, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "mov v29.d[1], x23\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "mov v28.d[1], x22\n"
      "fmla v22.4s, v6.4s, v2.s[0]\n"
      "ldr d27, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "mov v27.d[1], x21\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "ldr x23, [x15, #0x88]\n"
      "fmla v23.4s, v7.4s, v2.s[0]\n"
      "ldr d26, [x15, #0x70]\n"
      "mov v26.d[1], x20\n"
      "fmla v12.4s, v29.4s, v0.s[0]\n"
      "fmla v18.4s, v29.4s, v1.s[0]\n"
      "ldr x22, [x15, #0x98]\n"
      "fmla v24.4s, v29.4s, v2.s[0]\n"
      "ldr d29, [x15, #0x80]\n"
      "fmla v13.4s, v28.4s, v0.s[0]\n"
      "ldr x21, [x15, #0xa8]\n"
      "fmla v19.4s, v28.4s, v1.s[0]\n"
      "ldr x20, [x15, #0xb8]\n"
      "fmla v25.4s, v28.4s, v2.s[0]\n"
      "ldr d28, [x15, #0x90]\n"
      "fmla v8.4s, v27.4s, v0.s[1]\n"
      "mov v29.d[1], x23\n"
      "fmla v14.4s, v27.4s, v1.s[1]\n"
      "mov v28.d[1], x22\n"
      "fmla v20.4s, v27.4s, v2.s[1]\n"
      "ldr d27, [x15, #0xa0]\n"
      "fmla v9.4s, v26.4s, v0.s[1]\n"
      "mov v27.d[1], x21\n"
      "fmla v15.4s, v26.4s, v1.s[1]\n"
      "ldr x23, [x15, #0xc8]\n"
      "fmla v21.4s, v26.4s, v2.s[1]\n"
      "ldr d26, [x15, #0xb0]\n"
      "mov v26.d[1], x20\n"
      "fmla v10.4s, v29.4s, v0.s[1]\n"
      "fmla v16.4s, v29.4s, v1.s[1]\n"
      "ldr x22, [x15, #0xd8]\n"
      "fmla v22.4s, v29.4s, v2.s[1]\n"
      "ldr d29, [x15, #0xc0]\n"
      "fmla v11.4s, v28.4s, v0.s[1]\n"
      "ldr x21, [x15, #0xe8]\n"
      "fmla v17.4s, v28.4s, v1.s[1]\n"
      "ldr x20, [x15, #0xf8]\n"
      "fmla v23.4s, v28.4s, v2.s[1]\n"
      "ldr d28, [x15, #0xd0]\n"
      "fmla v12.4s, v27.4s, v0.s[1]\n"
      "mov v29.d[1], x23\n"
      "fmla v18.4s, v27.4s, v1.s[1]\n"
      "mov v28.d[1], x22\n"
      "fmla v24.4s, v27.4s, v2.s[1]\n"
      "ldr d27, [x15, #0xe0]\n"
      "fmla v13.4s, v26.4s, v0.s[1]\n"
      "mov v27.d[1], x21\n"
      "fmla v19.4s, v26.4s, v1.s[1]\n"
      "ldr x23, [x15, #0x108]\n"
      "fmla v25.4s, v26.4s, v2.s[1]\n"
      "ldr d26, [x15, #0xf0]\n"
      "mov v26.d[1], x20\n"
      "fmla v8.4s, v29.4s, v0.s[2]\n"
      "fmla v14.4s, v29.4s, v1.s[2]\n"
      "ldr x22, [x15, #0x118]\n"
      "fmla v20.4s, v29.4s, v2.s[2]\n"
      "ldr d29, [x15, #0x100]\n"
      "fmla v9.4s, v28.4s, v0.s[2]\n"
      "ldr x21, [x15, #0x128]\n"
      "fmla v15.4s, v28.4s, v1.s[2]\n"
      "ldr x20, [x15, #0x138]\n"
      "fmla v21.4s, v28.4s, v2.s[2]\n"
      "ldr d28, [x15, #0x110]\n"
      "fmla v10.4s, v27.4s, v0.s[2]\n"
      "mov v29.d[1], x23\n"
      "fmla v16.4s, v27.4s, v1.s[2]\n"
      "mov v28.d[1], x22\n"
      "fmla v22.4s, v27.4s, v2.s[2]\n"
      "ldr d27, [x15, #0x120]\n"
      "fmla v11.4s, v26.4s, v0.s[2]\n"
      "mov v27.d[1], x21\n"
      "fmla v17.4s, v26.4s, v1.s[2]\n"
      "ldr x23, [x15, #0x148]\n"
      "fmla v23.4s, v26.4s, v2.s[2]\n"
      "ldr d26, [x15, #0x130]\n"
      "mov v26.d[1], x20\n"
      "fmla v12.4s, v29.4s, v0.s[2]\n"
      "fmla v18.4s, v29.4s, v1.s[2]\n"
      "ldr x22, [x15, #0x158]\n"
      "fmla v24.4s, v29.4s, v2.s[2]\n"
      "ldr d29, [x15, #0x140]\n"
      "fmla v13.4s, v28.4s, v0.s[2]\n"
      "ldr x21, [x15, #0x168]\n"
      "fmla v19.4s, v28.4s, v1.s[2]\n"
      "ldr x20, [x15, #0x178]\n"
      "fmla v25.4s, v28.4s, v2.s[2]\n"
      "ldr d28, [x15, #0x150]\n"
      "fmla v8.4s, v27.4s, v0.s[3]\n"
      "mov v29.d[1], x23\n"
      "fmla v14.4s, v27.4s, v1.s[3]\n"
      "mov v28.d[1], x22\n"
      "fmla v20.4s, v27.4s, v2.s[3]\n"
      "ldr d27, [x15, #0x160]\n"
      "fmla v9.4s, v26.4s, v0.s[3]\n"
      "mov v27.d[1], x21\n"
      "fmla v15.4s, v26.4s, v1.s[3]\n"
      "add x11, x11, #0x10\n"
      "fmla v21.4s, v26.4s, v2.s[3]\n"
      "ldr d26, [x15, #0x170]\n"
      "mov v26.d[1], x20\n"
      "add x10, x10, #0x10\n"
      "add x9, x9, #0x10\n"
      "add x15, x15, #0x180\n"
      "fmla v10.4s, v29.4s, v0.s[3]\n"
      "ldr x26, [x15, #0x8]\n"
      "fmla v16.4s, v29.4s, v1.s[3]\n"
      "ldr x25, [x15, #0x18]\n"
      "fmla v22.4s, v29.4s, v2.s[3]\n"
      "ldr d4, [x15, #0x0]\n"
      "fmla v11.4s, v28.4s, v0.s[3]\n"
      "ldr x24, [x15, #0x28]\n"
      "fmla v17.4s, v28.4s, v1.s[3]\n"
      "ldr x23, [x11, #0x8]\n"
      "fmla v23.4s, v28.4s, v2.s[3]\n"
      "ldr d5, [x15, #0x10]\n"
      "fmla v12.4s, v27.4s, v0.s[3]\n"
      "ldr x22, [x10, #0x8]\n"
      "fmla v18.4s, v27.4s, v1.s[3]\n"
      "ldr x21, [x9, #0x8]\n"
      "fmla v24.4s, v27.4s, v2.s[3]\n"
      "ldr d6, [x15, #0x20]\n"
      "fmla v13.4s, v26.4s, v0.s[3]\n"
      "ldr d0, [x11, #0x0]\n"
      "fmla v19.4s, v26.4s, v1.s[3]\n"
      "ldr d1, [x10, #0x0]\n"
      "fmla v25.4s, v26.4s, v2.s[3]\n"
      "ldr d2, [x9, #0x0]\n"
      "ldr d7, [x15, #0x30]\n"
      "sub x12, x12, #0x4\n"
      "ldr x20, [x15, #0x38]\n"
      "cmp x12, #0x8\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "mov v4.d[1], x26\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "mov v5.d[1], x25\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "mov v6.d[1], x24\n"
      "mov v0.d[1], x23\n"
      "mov v1.d[1], x22\n"
      "mov v2.d[1], x21\n"
      "mov v7.d[1], x20\n"
      "bge 104b\n"
      "105:"  // Height 3: Multiply loop: Single iteration only
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "add x11, x11, #0x10\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "add x10, x10, #0x10\n"
      "fmla v20.4s, v4.4s, v2.s[0]\n"
      "ldr q29, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "add x9, x9, #0x10\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "sub x12, x12, #0x4\n"
      "fmla v21.4s, v5.4s, v2.s[0]\n"
      "ldr q28, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "fmla v22.4s, v6.4s, v2.s[0]\n"
      "ldr q27, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "fmla v23.4s, v7.4s, v2.s[0]\n"
      "ldr q26, [x15, #0x70]\n"
      "fmla v12.4s, v29.4s, v0.s[0]\n"
      "fmla v18.4s, v29.4s, v1.s[0]\n"
      "fmla v24.4s, v29.4s, v2.s[0]\n"
      "ldr q29, [x15, #0x80]\n"
      "fmla v13.4s, v28.4s, v0.s[0]\n"
      "fmla v19.4s, v28.4s, v1.s[0]\n"
      "fmla v25.4s, v28.4s, v2.s[0]\n"
      "ldr q28, [x15, #0x90]\n"
      "fmla v8.4s, v27.4s, v0.s[1]\n"
      "fmla v14.4s, v27.4s, v1.s[1]\n"
      "fmla v20.4s, v27.4s, v2.s[1]\n"
      "ldr q27, [x15, #0xa0]\n"
      "fmla v9.4s, v26.4s, v0.s[1]\n"
      "fmla v15.4s, v26.4s, v1.s[1]\n"
      "fmla v21.4s, v26.4s, v2.s[1]\n"
      "ldr q26, [x15, #0xb0]\n"
      "fmla v10.4s, v29.4s, v0.s[1]\n"
      "fmla v16.4s, v29.4s, v1.s[1]\n"
      "fmla v22.4s, v29.4s, v2.s[1]\n"
      "ldr q29, [x15, #0xc0]\n"
      "fmla v11.4s, v28.4s, v0.s[1]\n"
      "fmla v17.4s, v28.4s, v1.s[1]\n"
      "fmla v23.4s, v28.4s, v2.s[1]\n"
      "ldr q28, [x15, #0xd0]\n"
      "fmla v12.4s, v27.4s, v0.s[1]\n"
      "fmla v18.4s, v27.4s, v1.s[1]\n"
      "fmla v24.4s, v27.4s, v2.s[1]\n"
      "ldr q27, [x15, #0xe0]\n"
      "fmla v13.4s, v26.4s, v0.s[1]\n"
      "fmla v19.4s, v26.4s, v1.s[1]\n"
      "fmla v25.4s, v26.4s, v2.s[1]\n"
      "ldr q26, [x15, #0xf0]\n"
      "fmla v8.4s, v29.4s, v0.s[2]\n"
      "fmla v14.4s, v29.4s, v1.s[2]\n"
      "fmla v20.4s, v29.4s, v2.s[2]\n"
      "ldr q29, [x15, #0x100]\n"
      "fmla v9.4s, v28.4s, v0.s[2]\n"
      "fmla v15.4s, v28.4s, v1.s[2]\n"
      "fmla v21.4s, v28.4s, v2.s[2]\n"
      "ldr q28, [x15, #0x110]\n"
      "fmla v10.4s, v27.4s, v0.s[2]\n"
      "fmla v16.4s, v27.4s, v1.s[2]\n"
      "fmla v22.4s, v27.4s, v2.s[2]\n"
      "ldr q27, [x15, #0x120]\n"
      "fmla v11.4s, v26.4s, v0.s[2]\n"
      "fmla v17.4s, v26.4s, v1.s[2]\n"
      "fmla v23.4s, v26.4s, v2.s[2]\n"
      "ldr q26, [x15, #0x130]\n"
      "fmla v12.4s, v29.4s, v0.s[2]\n"
      "fmla v18.4s, v29.4s, v1.s[2]\n"
      "fmla v24.4s, v29.4s, v2.s[2]\n"
      "ldr q29, [x15, #0x140]\n"
      "fmla v13.4s, v28.4s, v0.s[2]\n"
      "fmla v19.4s, v28.4s, v1.s[2]\n"
      "fmla v25.4s, v28.4s, v2.s[2]\n"
      "ldr q28, [x15, #0x150]\n"
      "fmla v8.4s, v27.4s, v0.s[3]\n"
      "fmla v14.4s, v27.4s, v1.s[3]\n"
      "fmla v20.4s, v27.4s, v2.s[3]\n"
      "ldr q27, [x15, #0x160]\n"
      "fmla v9.4s, v26.4s, v0.s[3]\n"
      "fmla v15.4s, v26.4s, v1.s[3]\n"
      "fmla v21.4s, v26.4s, v2.s[3]\n"
      "ldr q26, [x15, #0x170]\n"
      "fmla v10.4s, v29.4s, v0.s[3]\n"
      "add x15, x15, #0x180\n"
      "fmla v16.4s, v29.4s, v1.s[3]\n"
      "fmla v22.4s, v29.4s, v2.s[3]\n"
      "fmla v11.4s, v28.4s, v0.s[3]\n"
      "fmla v17.4s, v28.4s, v1.s[3]\n"
      "fmla v23.4s, v28.4s, v2.s[3]\n"
      "fmla v12.4s, v27.4s, v0.s[3]\n"
      "fmla v18.4s, v27.4s, v1.s[3]\n"
      "fmla v24.4s, v27.4s, v2.s[3]\n"
      "fmla v13.4s, v26.4s, v0.s[3]\n"
      "fmla v19.4s, v26.4s, v1.s[3]\n"
      "fmla v25.4s, v26.4s, v2.s[3]\n"
      "106:"  // Height 3: Multiply loop: Main loop skip
      "cbz x12, 108f\n"
      "107:"  // Height 3: Multiply loop: Odd block loop
      "ldr s0, [x11], #0x4\n"
      "sub x12, x12, #0x1\n"
      "ldr s31, [x10], #0x4\n"
      "ldr s30, [x9], #0x4\n"
      "ldr q27, [x15, #0x0]\n"
      "fmla v8.4s, v27.4s, v0.s[0]\n"
      "ldr q26, [x15, #0x10]\n"
      "fmla v14.4s, v27.4s, v31.s[0]\n"
      "ldr q29, [x15, #0x20]\n"
      "fmla v20.4s, v27.4s, v30.s[0]\n"
      "ldr q28, [x15, #0x30]\n"
      "fmla v9.4s, v26.4s, v0.s[0]\n"
      "ldr q27, [x15, #0x40]\n"
      "fmla v15.4s, v26.4s, v31.s[0]\n"
      "fmla v21.4s, v26.4s, v30.s[0]\n"
      "ldr q26, [x15, #0x50]\n"
      "fmla v10.4s, v29.4s, v0.s[0]\n"
      "add x15, x15, #0x60\n"
      "fmla v16.4s, v29.4s, v31.s[0]\n"
      "fmla v22.4s, v29.4s, v30.s[0]\n"
      "fmla v11.4s, v28.4s, v0.s[0]\n"
      "fmla v17.4s, v28.4s, v31.s[0]\n"
      "fmla v23.4s, v28.4s, v30.s[0]\n"
      "fmla v12.4s, v27.4s, v0.s[0]\n"
      "fmla v18.4s, v27.4s, v31.s[0]\n"
      "fmla v24.4s, v27.4s, v30.s[0]\n"
      "fmla v13.4s, v26.4s, v0.s[0]\n"
      "fmla v19.4s, v26.4s, v31.s[0]\n"
      "fmla v25.4s, v26.4s, v30.s[0]\n"
      "cbnz x12, 107b\n"
      "108:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x13, x13, #0x1\n"
      "cmp x13, x20\n"
      "bne 101b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x14, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x14, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "tbz %x[flags], #1, 109f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v26.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v26.4s\n"
      "fmin v9.4s, v9.4s, v26.4s\n"
      "fmin v10.4s, v10.4s, v26.4s\n"
      "fmin v11.4s, v11.4s, v26.4s\n"
      "fmin v12.4s, v12.4s, v26.4s\n"
      "fmin v13.4s, v13.4s, v26.4s\n"
      "fmin v14.4s, v14.4s, v26.4s\n"
      "fmin v15.4s, v15.4s, v26.4s\n"
      "fmin v16.4s, v16.4s, v26.4s\n"
      "fmin v17.4s, v17.4s, v26.4s\n"
      "fmin v18.4s, v18.4s, v26.4s\n"
      "fmin v19.4s, v19.4s, v26.4s\n"
      "fmin v20.4s, v20.4s, v26.4s\n"
      "fmin v21.4s, v21.4s, v26.4s\n"
      "fmin v22.4s, v22.4s, v26.4s\n"
      "fmin v23.4s, v23.4s, v26.4s\n"
      "fmin v24.4s, v24.4s, v26.4s\n"
      "fmin v25.4s, v25.4s, v26.4s\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v26.4s }, [x20]\n"
      "fmax v8.4s, v8.4s, v26.4s\n"
      "fmax v9.4s, v9.4s, v26.4s\n"
      "fmax v10.4s, v10.4s, v26.4s\n"
      "fmax v11.4s, v11.4s, v26.4s\n"
      "fmax v12.4s, v12.4s, v26.4s\n"
      "fmax v13.4s, v13.4s, v26.4s\n"
      "fmax v14.4s, v14.4s, v26.4s\n"
      "fmax v15.4s, v15.4s, v26.4s\n"
      "fmax v16.4s, v16.4s, v26.4s\n"
      "fmax v17.4s, v17.4s, v26.4s\n"
      "fmax v18.4s, v18.4s, v26.4s\n"
      "fmax v19.4s, v19.4s, v26.4s\n"
      "fmax v20.4s, v20.4s, v26.4s\n"
      "fmax v21.4s, v21.4s, v26.4s\n"
      "fmax v22.4s, v22.4s, v26.4s\n"
      "fmax v23.4s, v23.4s, v26.4s\n"
      "fmax v24.4s, v24.4s, v26.4s\n"
      "fmax v25.4s, v25.4s, v26.4s\n"
      "109:"  // Height 3: No activation
      "cmp x16, #0x18\n"
      "bge 122f\n"
      "tbz x16, #4, 113f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v11.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x22], #0x10\n"
      "tbz x16, #2, 111f\n"
      "st1 { v12.4s }, [x14], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v24.4s }, [x22], #0x10\n"
      "tbz x16, #1, 110f\n"
      "str d13, [x14], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d25, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v13.s }[2], [x14]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v25.s }[2], [x22]\n"
      "b 121f\n"
      "110:"  // Height 3: Partial direct writeback: partial_1_20
      "tbz x16, #0, 121f\n"
      "str s13, [x14, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s25, [x22, #0x0]\n"
      "b 121f\n"
      "111:"  // Height 3: Partial direct writeback: partial_2_16
      "tbz x16, #1, 112f\n"
      "str d12, [x14], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v12.s }[2], [x14]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v24.s }[2], [x22]\n"
      "b 121f\n"
      "112:"  // Height 3: Partial direct writeback: partial_1_16
      "tbz x16, #0, 121f\n"
      "str s12, [x14, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s24, [x22, #0x0]\n"
      "b 121f\n"
      "113:"  // Height 3: Partial direct writeback: partial_8_0
      "tbz x16, #3, 117f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "tbz x16, #2, 115f\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "tbz x16, #1, 114f\n"
      "str d11, [x14], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v11.s }[2], [x14]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "b 121f\n"
      "114:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x16, #0, 121f\n"
      "str s11, [x14, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "b 121f\n"
      "115:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x16, #1, 116f\n"
      "str d10, [x14], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v10.s }[2], [x14]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "b 121f\n"
      "116:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x16, #0, 121f\n"
      "str s10, [x14, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "b 121f\n"
      "117:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x16, #2, 119f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "tbz x16, #1, 118f\n"
      "str d9, [x14], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v9.s }[2], [x14]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "b 121f\n"
      "118:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x16, #0, 121f\n"
      "str s9, [x14, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "b 121f\n"
      "119:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x16, #1, 120f\n"
      "str d8, [x14], #0x8\n"
      "str d14, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "tbz x16, #0, 121f\n"
      "st1 { v8.s }[2], [x14]\n"
      "st1 { v14.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "b 121f\n"
      "120:"  // Height 3: Partial direct writeback: partial_1_0
      "str s8, [x14, #0x0]\n"
      "str s14, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "121:"  // Height 3: Partial direct writeback: Done
      "b 123f\n"
      "122:"  // Height 3: Full writeback
      "str q8, [x14, #0x0]\n"
      "str q9, [x14, #0x10]\n"
      "str q10, [x14, #0x20]\n"
      "str q11, [x14, #0x30]\n"
      "str q12, [x14, #0x40]\n"
      "str q13, [x14, #0x50]\n"
      "add x14, x14, #0x60\n"
      "str q14, [x23, #0x0]\n"
      "str q15, [x23, #0x10]\n"
      "str q16, [x23, #0x20]\n"
      "str q17, [x23, #0x30]\n"
      "str q18, [x23, #0x40]\n"
      "str q19, [x23, #0x50]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x22, #0x40]\n"
      "str q25, [x22, #0x50]\n"
      "123:"  // Height 3: Writeback done
      "subs x16, x16, #0x18\n"
      "bgt 84b\n"
      "b 166f\n"
      "124:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x10\n"
      "mov x17, %x[bias]\n"
      "ldr x16, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x14, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "125:"  // Height 4: Column loop
      "cbz x17, 126f\n"
      "ldr q8, [x17, #0x0]\n"
      "mov v14.16b, v8.16b\n"
      "ldr q9, [x17, #0x10]\n"
      "mov v15.16b, v9.16b\n"
      "ldr q10, [x17, #0x20]\n"
      "mov v16.16b, v10.16b\n"
      "ldr q11, [x17, #0x30]\n"
      "mov v17.16b, v11.16b\n"
      "ldr q12, [x17, #0x40]\n"
      "mov v18.16b, v12.16b\n"
      "ldr q13, [x17, #0x50]\n"
      "mov v19.16b, v13.16b\n"
      "mov v20.16b, v8.16b\n"
      "add x17, x17, #0x60\n"
      "mov v21.16b, v9.16b\n"
      "mov v22.16b, v10.16b\n"
      "mov v23.16b, v11.16b\n"
      "mov v24.16b, v12.16b\n"
      "mov v25.16b, v13.16b\n"
      "mov v26.16b, v8.16b\n"
      "mov v27.16b, v9.16b\n"
      "mov v28.16b, v10.16b\n"
      "mov v29.16b, v11.16b\n"
      "mov v30.16b, v12.16b\n"
      "mov v31.16b, v13.16b\n"
      "b 141f\n"
      "126:"  // Height 4: no bias
      "tbz %x[flags], #0, 140f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x14, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x16, #0x18\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 139f\n"
      "tbz x16, #4, 130f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v28.4s }, [x21], #0x10\n"
      "ld1 { v11.4s }, [x14], #0x10\n"
      "ld1 { v17.4s }, [x23], #0x10\n"
      "ld1 { v23.4s }, [x22], #0x10\n"
      "ld1 { v29.4s }, [x21], #0x10\n"
      "tbz x16, #2, 128f\n"
      "ld1 { v12.4s }, [x14], #0x10\n"
      "ld1 { v18.4s }, [x23], #0x10\n"
      "ld1 { v24.4s }, [x22], #0x10\n"
      "ld1 { v30.4s }, [x21], #0x10\n"
      "tbz x16, #1, 127f\n"
      "ldr d13, [x14], #0x8\n"
      "mov x20, #0x58\n"
      "ldr d19, [x23], #0x8\n"
      "ldr d25, [x22], #0x8\n"
      "ldr d31, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v13.s }[2], [x14]\n"
      "ld1 { v19.s }[2], [x23]\n"
      "ld1 { v25.s }[2], [x22]\n"
      "ld1 { v31.s }[2], [x21]\n"
      "b 138f\n"
      "127:"  // Height 4: Partial accumulate: partial_1_20
      "mov x20, #0x50\n"
      "tbz x16, #0, 138f\n"
      "ldr s13, [x14, #0x0]\n"
      "ldr s19, [x23, #0x0]\n"
      "ldr s25, [x22, #0x0]\n"
      "ldr s31, [x21, #0x0]\n"
      "b 138f\n"
      "128:"  // Height 4: Partial accumulate: partial_2_16
      "tbz x16, #1, 129f\n"
      "ldr d12, [x14], #0x8\n"
      "mov x20, #0x48\n"
      "ldr d18, [x23], #0x8\n"
      "ldr d24, [x22], #0x8\n"
      "ldr d30, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v12.s }[2], [x14]\n"
      "ld1 { v18.s }[2], [x23]\n"
      "ld1 { v24.s }[2], [x22]\n"
      "ld1 { v30.s }[2], [x21]\n"
      "b 138f\n"
      "129:"  // Height 4: Partial accumulate: partial_1_16
      "mov x20, #0x40\n"
      "tbz x16, #0, 138f\n"
      "ldr s12, [x14, #0x0]\n"
      "ldr s18, [x23, #0x0]\n"
      "ldr s24, [x22, #0x0]\n"
      "ldr s30, [x21, #0x0]\n"
      "b 138f\n"
      "130:"  // Height 4: Partial accumulate: partial_8_0
      "tbz x16, #3, 134f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "ld1 { v9.4s }, [x14], #0x10\n"
      "ld1 { v15.4s }, [x23], #0x10\n"
      "ld1 { v21.4s }, [x22], #0x10\n"
      "ld1 { v27.4s }, [x21], #0x10\n"
      "tbz x16, #2, 132f\n"
      "ld1 { v10.4s }, [x14], #0x10\n"
      "ld1 { v16.4s }, [x23], #0x10\n"
      "ld1 { v22.4s }, [x22], #0x10\n"
      "ld1 { v28.4s }, [x21], #0x10\n"
      "tbz x16, #1, 131f\n"
      "ldr d11, [x14], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d17, [x23], #0x8\n"
      "ldr d23, [x22], #0x8\n"
      "ldr d29, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v11.s }[2], [x14]\n"
      "ld1 { v17.s }[2], [x23]\n"
      "ld1 { v23.s }[2], [x22]\n"
      "ld1 { v29.s }[2], [x21]\n"
      "b 138f\n"
      "131:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x16, #0, 138f\n"
      "ldr s11, [x14, #0x0]\n"
      "ldr s17, [x23, #0x0]\n"
      "ldr s23, [x22, #0x0]\n"
      "ldr s29, [x21, #0x0]\n"
      "b 138f\n"
      "132:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x16, #1, 133f\n"
      "ldr d10, [x14], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d16, [x23], #0x8\n"
      "ldr d22, [x22], #0x8\n"
      "ldr d28, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v10.s }[2], [x14]\n"
      "ld1 { v16.s }[2], [x23]\n"
      "ld1 { v22.s }[2], [x22]\n"
      "ld1 { v28.s }[2], [x21]\n"
      "b 138f\n"
      "133:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x16, #0, 138f\n"
      "ldr s10, [x14, #0x0]\n"
      "ldr s16, [x23, #0x0]\n"
      "ldr s22, [x22, #0x0]\n"
      "ldr s28, [x21, #0x0]\n"
      "b 138f\n"
      "134:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x16, #2, 136f\n"
      "ld1 { v8.4s }, [x14], #0x10\n"
      "ld1 { v14.4s }, [x23], #0x10\n"
      "ld1 { v20.4s }, [x22], #0x10\n"
      "ld1 { v26.4s }, [x21], #0x10\n"
      "tbz x16, #1, 135f\n"
      "ldr d9, [x14], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d15, [x23], #0x8\n"
      "ldr d21, [x22], #0x8\n"
      "ldr d27, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v9.s }[2], [x14]\n"
      "ld1 { v15.s }[2], [x23]\n"
      "ld1 { v21.s }[2], [x22]\n"
      "ld1 { v27.s }[2], [x21]\n"
      "b 138f\n"
      "135:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x16, #0, 138f\n"
      "ldr s9, [x14, #0x0]\n"
      "ldr s15, [x23, #0x0]\n"
      "ldr s21, [x22, #0x0]\n"
      "ldr s27, [x21, #0x0]\n"
      "b 138f\n"
      "136:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x16, #1, 137f\n"
      "ldr d8, [x14], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d14, [x23], #0x8\n"
      "ldr d20, [x22], #0x8\n"
      "ldr d26, [x21], #0x8\n"
      "tbz x16, #0, 138f\n"
      "ld1 { v8.s }[2], [x14]\n"
      "ld1 { v14.s }[2], [x23]\n"
      "ld1 { v20.s }[2], [x22]\n"
      "ld1 { v26.s }[2], [x21]\n"
      "b 138f\n"
      "137:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s8, [x14, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s14, [x23, #0x0]\n"
      "ldr s20, [x22, #0x0]\n"
      "ldr s26, [x21, #0x0]\n"
      "138:"  // Height 4: Partial accumulate: Done
      "sub x14, x14, x20\n"
      "b 141f\n"
      "139:"  // Height 4: full accumulate
      "ldr q8, [x14, #0x0]\n"
      "ldr q9, [x14, #0x10]\n"
      "ldr q10, [x14, #0x20]\n"
      "ldr q11, [x14, #0x30]\n"
      "ldr q12, [x14, #0x40]\n"
      "ldr q13, [x14, #0x50]\n"
      "ldr q14, [x23, #0x0]\n"
      "ldr q15, [x23, #0x10]\n"
      "ldr q16, [x23, #0x20]\n"
      "ldr q17, [x23, #0x30]\n"
      "ldr q18, [x23, #0x40]\n"
      "ldr q19, [x23, #0x50]\n"
      "ldr q20, [x22, #0x0]\n"
      "ldr q21, [x22, #0x10]\n"
      "ldr q22, [x22, #0x20]\n"
      "ldr q23, [x22, #0x30]\n"
      "ldr q24, [x22, #0x40]\n"
      "ldr q25, [x22, #0x50]\n"
      "ldr q26, [x21, #0x0]\n"
      "ldr q27, [x21, #0x10]\n"
      "ldr q28, [x21, #0x20]\n"
      "ldr q29, [x21, #0x30]\n"
      "ldr q30, [x21, #0x40]\n"
      "ldr q31, [x21, #0x50]\n"
      "b 141f\n"
      "140:"  // Height 4: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "141:"  // Height 4: setup done
      "mov x13, #0x0\n"
      "142:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w12, [x20, x13, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 143f\n"
      "ldr x20, [%x[input_ptr], x13, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x11, [x20, #0x0]\n"
      "ldr x10, [x20, #0x8]\n"
      "ldr x9, [x20, #0x10]\n"
      "ldr x28, [x20, #0x18]\n"
      "cbnz x13, 144f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x11, x11, x20, LSL #2\n"
      "add x10, x10, x20, LSL #2\n"
      "add x9, x9, x20, LSL #2\n"
      "add x28, x28, x20, LSL #2\n"
      "b 144f\n"
      "143:"  // Height 4: setup direct input
      "mov x11, %x[input_ptr]\n"
      "add x10, x11, x21, LSL #2\n"
      "add x9, x10, x21, LSL #2\n"
      "add x28, x9, x21, LSL #2\n"
      "144:"  // Height 4: input setup done
      "cmp x12, #0x4\n"
      "blt 147f\n"
      "ldr q0, [x11, #0x0]\n"
      "cmp x12, #0x8\n"
      "ldr q1, [x10, #0x0]\n"
      "ldr q2, [x9, #0x0]\n"
      "ldr q3, [x28, #0x0]\n"
      "ldr q4, [x15, #0x0]\n"
      "ldr q5, [x15, #0x10]\n"
      "ldr q6, [x15, #0x20]\n"
      "ldr q7, [x15, #0x30]\n"
      "blt 146f\n"
      "145:"  // Height 4: Multiply loop: Main loop head
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "ldr x23, [x15, #0x48]\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "ldr x22, [x15, #0x58]\n"
      "fmla v20.4s, v4.4s, v2.s[0]\n"
      "ldr x21, [x15, #0x68]\n"
      "fmla v26.4s, v4.4s, v3.s[0]\n"
      "ldr d4, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "ldr x20, [x15, #0x78]\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "mov v4.d[1], x23\n"
      "fmla v21.4s, v5.4s, v2.s[0]\n"
      "ldr x23, [x15, #0x88]\n"
      "fmla v27.4s, v5.4s, v3.s[0]\n"
      "ldr d5, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "mov v5.d[1], x22\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "ldr x22, [x15, #0x98]\n"
      "fmla v22.4s, v6.4s, v2.s[0]\n"
      "add x11, x11, #0x10\n"
      "fmla v28.4s, v6.4s, v3.s[0]\n"
      "ldr d6, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "mov v6.d[1], x21\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "ldr x21, [x15, #0xa8]\n"
      "fmla v23.4s, v7.4s, v2.s[0]\n"
      "add x10, x10, #0x10\n"
      "fmla v29.4s, v7.4s, v3.s[0]\n"
      "ldr d7, [x15, #0x70]\n"
      "mov v7.d[1], x20\n"
      "fmla v12.4s, v4.4s, v0.s[0]\n"
      "fmla v18.4s, v4.4s, v1.s[0]\n"
      "ldr x20, [x15, #0xb8]\n"
      "fmla v24.4s, v4.4s, v2.s[0]\n"
      "add x9, x9, #0x10\n"
      "fmla v30.4s, v4.4s, v3.s[0]\n"
      "ldr d4, [x15, #0x80]\n"
      "fmla v13.4s, v5.4s, v0.s[0]\n"
      "mov v4.d[1], x23\n"
      "fmla v19.4s, v5.4s, v1.s[0]\n"
      "ldr x23, [x15, #0xc8]\n"
      "fmla v25.4s, v5.4s, v2.s[0]\n"
      "add x28, x28, #0x10\n"
      "fmla v31.4s, v5.4s, v3.s[0]\n"
      "ldr d5, [x15, #0x90]\n"
      "fmla v8.4s, v6.4s, v0.s[1]\n"
      "mov v5.d[1], x22\n"
      "fmla v14.4s, v6.4s, v1.s[1]\n"
      "ldr x22, [x15, #0xd8]\n"
      "fmla v20.4s, v6.4s, v2.s[1]\n"
      "ldr x27, [x11, #0x8]\n"
      "fmla v26.4s, v6.4s, v3.s[1]\n"
      "ldr d6, [x15, #0xa0]\n"
      "fmla v9.4s, v7.4s, v0.s[1]\n"
      "mov v6.d[1], x21\n"
      "fmla v15.4s, v7.4s, v1.s[1]\n"
      "ldr x21, [x15, #0xe8]\n"
      "fmla v21.4s, v7.4s, v2.s[1]\n"
      "ldr x26, [x10, #0x8]\n"
      "fmla v27.4s, v7.4s, v3.s[1]\n"
      "ldr d7, [x15, #0xb0]\n"
      "mov v7.d[1], x20\n"
      "fmla v10.4s, v4.4s, v0.s[1]\n"
      "fmla v16.4s, v4.4s, v1.s[1]\n"
      "ldr x20, [x15, #0xf8]\n"
      "fmla v22.4s, v4.4s, v2.s[1]\n"
      "ldr x25, [x9, #0x8]\n"
      "fmla v28.4s, v4.4s, v3.s[1]\n"
      "ldr d4, [x15, #0xc0]\n"
      "fmla v11.4s, v5.4s, v0.s[1]\n"
      "mov v4.d[1], x23\n"
      "fmla v17.4s, v5.4s, v1.s[1]\n"
      "ldr x23, [x15, #0x108]\n"
      "fmla v23.4s, v5.4s, v2.s[1]\n"
      "ldr x24, [x28, #0x8]\n"
      "fmla v29.4s, v5.4s, v3.s[1]\n"
      "ldr d5, [x15, #0xd0]\n"
      "fmla v12.4s, v6.4s, v0.s[1]\n"
      "mov v5.d[1], x22\n"
      "fmla v18.4s, v6.4s, v1.s[1]\n"
      "ldr x22, [x15, #0x118]\n"
      "fmla v24.4s, v6.4s, v2.s[1]\n"
      "sub x12, x12, #0x4\n"
      "fmla v30.4s, v6.4s, v3.s[1]\n"
      "ldr d6, [x15, #0xe0]\n"
      "fmla v13.4s, v7.4s, v0.s[1]\n"
      "mov v6.d[1], x21\n"
      "fmla v19.4s, v7.4s, v1.s[1]\n"
      "ldr x21, [x15, #0x128]\n"
      "fmla v25.4s, v7.4s, v2.s[1]\n"
      "cmp x12, #0x8\n"
      "fmla v31.4s, v7.4s, v3.s[1]\n"
      "ldr d7, [x15, #0xf0]\n"
      "mov v7.d[1], x20\n"
      "fmla v8.4s, v4.4s, v0.s[2]\n"
      "fmla v14.4s, v4.4s, v1.s[2]\n"
      "ldr x20, [x15, #0x138]\n"
      "fmla v20.4s, v4.4s, v2.s[2]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "fmla v26.4s, v4.4s, v3.s[2]\n"
      "ldr d4, [x15, #0x100]\n"
      "fmla v9.4s, v5.4s, v0.s[2]\n"
      "mov v4.d[1], x23\n"
      "fmla v15.4s, v5.4s, v1.s[2]\n"
      "ldr x23, [x15, #0x148]\n"
      "fmla v21.4s, v5.4s, v2.s[2]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "fmla v27.4s, v5.4s, v3.s[2]\n"
      "ldr d5, [x15, #0x110]\n"
      "fmla v10.4s, v6.4s, v0.s[2]\n"
      "mov v5.d[1], x22\n"
      "fmla v16.4s, v6.4s, v1.s[2]\n"
      "ldr x22, [x15, #0x158]\n"
      "fmla v22.4s, v6.4s, v2.s[2]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "fmla v28.4s, v6.4s, v3.s[2]\n"
      "ldr d6, [x15, #0x120]\n"
      "fmla v11.4s, v7.4s, v0.s[2]\n"
      "mov v6.d[1], x21\n"
      "fmla v17.4s, v7.4s, v1.s[2]\n"
      "ldr x21, [x15, #0x168]\n"
      "fmla v23.4s, v7.4s, v2.s[2]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v29.4s, v7.4s, v3.s[2]\n"
      "ldr d7, [x15, #0x130]\n"
      "mov v7.d[1], x20\n"
      "fmla v12.4s, v4.4s, v0.s[2]\n"
      "fmla v18.4s, v4.4s, v1.s[2]\n"
      "ldr x20, [x15, #0x178]\n"
      "fmla v24.4s, v4.4s, v2.s[2]\n"
      "fmla v30.4s, v4.4s, v3.s[2]\n"
      "ldr d4, [x15, #0x140]\n"
      "fmla v13.4s, v5.4s, v0.s[2]\n"
      "mov v4.d[1], x23\n"
      "fmla v19.4s, v5.4s, v1.s[2]\n"
      "fmla v25.4s, v5.4s, v2.s[2]\n"
      "fmla v31.4s, v5.4s, v3.s[2]\n"
      "ldr d5, [x15, #0x150]\n"
      "fmla v8.4s, v6.4s, v0.s[3]\n"
      "mov v5.d[1], x22\n"
      "fmla v14.4s, v6.4s, v1.s[3]\n"
      "fmla v20.4s, v6.4s, v2.s[3]\n"
      "fmla v26.4s, v6.4s, v3.s[3]\n"
      "ldr d6, [x15, #0x160]\n"
      "fmla v9.4s, v7.4s, v0.s[3]\n"
      "mov v6.d[1], x21\n"
      "fmla v15.4s, v7.4s, v1.s[3]\n"
      "fmla v21.4s, v7.4s, v2.s[3]\n"
      "fmla v27.4s, v7.4s, v3.s[3]\n"
      "ldr d7, [x15, #0x170]\n"
      "mov v7.d[1], x20\n"
      "add x15, x15, #0x180\n"
      "fmla v10.4s, v4.4s, v0.s[3]\n"
      "ldr x23, [x15, #0x8]\n"
      "fmla v16.4s, v4.4s, v1.s[3]\n"
      "ldr x22, [x15, #0x18]\n"
      "fmla v22.4s, v4.4s, v2.s[3]\n"
      "ldr x21, [x15, #0x28]\n"
      "fmla v28.4s, v4.4s, v3.s[3]\n"
      "ldr d4, [x15, #0x0]\n"
      "fmla v11.4s, v5.4s, v0.s[3]\n"
      "ldr x20, [x15, #0x38]\n"
      "fmla v17.4s, v5.4s, v1.s[3]\n"
      "mov v4.d[1], x23\n"
      "fmla v23.4s, v5.4s, v2.s[3]\n"
      "fmla v29.4s, v5.4s, v3.s[3]\n"
      "ldr d5, [x15, #0x10]\n"
      "fmla v12.4s, v6.4s, v0.s[3]\n"
      "mov v5.d[1], x22\n"
      "fmla v18.4s, v6.4s, v1.s[3]\n"
      "fmla v24.4s, v6.4s, v2.s[3]\n"
      "fmla v30.4s, v6.4s, v3.s[3]\n"
      "ldr d6, [x15, #0x20]\n"
      "fmla v13.4s, v7.4s, v0.s[3]\n"
      "ldr d0, [x11, #0x0]\n"
      "fmla v19.4s, v7.4s, v1.s[3]\n"
      "ldr d1, [x10, #0x0]\n"
      "fmla v25.4s, v7.4s, v2.s[3]\n"
      "ldr d2, [x9, #0x0]\n"
      "fmla v31.4s, v7.4s, v3.s[3]\n"
      "ldr d3, [x28, #0x0]\n"
      "ldr d7, [x15, #0x30]\n"
      "mov v6.d[1], x21\n"
      "mov v0.d[1], x27\n"
      "mov v1.d[1], x26\n"
      "mov v2.d[1], x25\n"
      "mov v3.d[1], x24\n"
      "mov v7.d[1], x20\n"
      "bge 145b\n"
      "146:"  // Height 4: Multiply loop: Single iteration only
      "fmla v8.4s, v4.4s, v0.s[0]\n"
      "add x11, x11, #0x10\n"
      "fmla v14.4s, v4.4s, v1.s[0]\n"
      "add x10, x10, #0x10\n"
      "fmla v20.4s, v4.4s, v2.s[0]\n"
      "add x9, x9, #0x10\n"
      "fmla v26.4s, v4.4s, v3.s[0]\n"
      "ldr q4, [x15, #0x40]\n"
      "fmla v9.4s, v5.4s, v0.s[0]\n"
      "add x28, x28, #0x10\n"
      "fmla v15.4s, v5.4s, v1.s[0]\n"
      "sub x12, x12, #0x4\n"
      "fmla v21.4s, v5.4s, v2.s[0]\n"
      "prfm pldl1keep, [x11, #0x80]\n"
      "fmla v27.4s, v5.4s, v3.s[0]\n"
      "ldr q5, [x15, #0x50]\n"
      "fmla v10.4s, v6.4s, v0.s[0]\n"
      "prfm pldl1keep, [x10, #0x80]\n"
      "fmla v16.4s, v6.4s, v1.s[0]\n"
      "prfm pldl1keep, [x9, #0x80]\n"
      "fmla v22.4s, v6.4s, v2.s[0]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v28.4s, v6.4s, v3.s[0]\n"
      "ldr q6, [x15, #0x60]\n"
      "fmla v11.4s, v7.4s, v0.s[0]\n"
      "fmla v17.4s, v7.4s, v1.s[0]\n"
      "fmla v23.4s, v7.4s, v2.s[0]\n"
      "fmla v29.4s, v7.4s, v3.s[0]\n"
      "ldr q7, [x15, #0x70]\n"
      "fmla v12.4s, v4.4s, v0.s[0]\n"
      "fmla v18.4s, v4.4s, v1.s[0]\n"
      "fmla v24.4s, v4.4s, v2.s[0]\n"
      "fmla v30.4s, v4.4s, v3.s[0]\n"
      "ldr q4, [x15, #0x80]\n"
      "fmla v13.4s, v5.4s, v0.s[0]\n"
      "fmla v19.4s, v5.4s, v1.s[0]\n"
      "fmla v25.4s, v5.4s, v2.s[0]\n"
      "fmla v31.4s, v5.4s, v3.s[0]\n"
      "ldr q5, [x15, #0x90]\n"
      "fmla v8.4s, v6.4s, v0.s[1]\n"
      "fmla v14.4s, v6.4s, v1.s[1]\n"
      "fmla v20.4s, v6.4s, v2.s[1]\n"
      "fmla v26.4s, v6.4s, v3.s[1]\n"
      "ldr q6, [x15, #0xa0]\n"
      "fmla v9.4s, v7.4s, v0.s[1]\n"
      "fmla v15.4s, v7.4s, v1.s[1]\n"
      "fmla v21.4s, v7.4s, v2.s[1]\n"
      "fmla v27.4s, v7.4s, v3.s[1]\n"
      "ldr q7, [x15, #0xb0]\n"
      "fmla v10.4s, v4.4s, v0.s[1]\n"
      "fmla v16.4s, v4.4s, v1.s[1]\n"
      "fmla v22.4s, v4.4s, v2.s[1]\n"
      "fmla v28.4s, v4.4s, v3.s[1]\n"
      "ldr q4, [x15, #0xc0]\n"
      "fmla v11.4s, v5.4s, v0.s[1]\n"
      "fmla v17.4s, v5.4s, v1.s[1]\n"
      "fmla v23.4s, v5.4s, v2.s[1]\n"
      "fmla v29.4s, v5.4s, v3.s[1]\n"
      "ldr q5, [x15, #0xd0]\n"
      "fmla v12.4s, v6.4s, v0.s[1]\n"
      "fmla v18.4s, v6.4s, v1.s[1]\n"
      "fmla v24.4s, v6.4s, v2.s[1]\n"
      "fmla v30.4s, v6.4s, v3.s[1]\n"
      "ldr q6, [x15, #0xe0]\n"
      "fmla v13.4s, v7.4s, v0.s[1]\n"
      "fmla v19.4s, v7.4s, v1.s[1]\n"
      "fmla v25.4s, v7.4s, v2.s[1]\n"
      "fmla v31.4s, v7.4s, v3.s[1]\n"
      "ldr q7, [x15, #0xf0]\n"
      "fmla v8.4s, v4.4s, v0.s[2]\n"
      "fmla v14.4s, v4.4s, v1.s[2]\n"
      "fmla v20.4s, v4.4s, v2.s[2]\n"
      "fmla v26.4s, v4.4s, v3.s[2]\n"
      "ldr q4, [x15, #0x100]\n"
      "fmla v9.4s, v5.4s, v0.s[2]\n"
      "fmla v15.4s, v5.4s, v1.s[2]\n"
      "fmla v21.4s, v5.4s, v2.s[2]\n"
      "fmla v27.4s, v5.4s, v3.s[2]\n"
      "ldr q5, [x15, #0x110]\n"
      "fmla v10.4s, v6.4s, v0.s[2]\n"
      "fmla v16.4s, v6.4s, v1.s[2]\n"
      "fmla v22.4s, v6.4s, v2.s[2]\n"
      "fmla v28.4s, v6.4s, v3.s[2]\n"
      "ldr q6, [x15, #0x120]\n"
      "fmla v11.4s, v7.4s, v0.s[2]\n"
      "fmla v17.4s, v7.4s, v1.s[2]\n"
      "fmla v23.4s, v7.4s, v2.s[2]\n"
      "fmla v29.4s, v7.4s, v3.s[2]\n"
      "ldr q7, [x15, #0x130]\n"
      "fmla v12.4s, v4.4s, v0.s[2]\n"
      "fmla v18.4s, v4.4s, v1.s[2]\n"
      "fmla v24.4s, v4.4s, v2.s[2]\n"
      "fmla v30.4s, v4.4s, v3.s[2]\n"
      "ldr q4, [x15, #0x140]\n"
      "fmla v13.4s, v5.4s, v0.s[2]\n"
      "fmla v19.4s, v5.4s, v1.s[2]\n"
      "fmla v25.4s, v5.4s, v2.s[2]\n"
      "fmla v31.4s, v5.4s, v3.s[2]\n"
      "ldr q5, [x15, #0x150]\n"
      "fmla v8.4s, v6.4s, v0.s[3]\n"
      "fmla v14.4s, v6.4s, v1.s[3]\n"
      "fmla v20.4s, v6.4s, v2.s[3]\n"
      "fmla v26.4s, v6.4s, v3.s[3]\n"
      "ldr q6, [x15, #0x160]\n"
      "fmla v9.4s, v7.4s, v0.s[3]\n"
      "fmla v15.4s, v7.4s, v1.s[3]\n"
      "fmla v21.4s, v7.4s, v2.s[3]\n"
      "fmla v27.4s, v7.4s, v3.s[3]\n"
      "ldr q7, [x15, #0x170]\n"
      "fmla v10.4s, v4.4s, v0.s[3]\n"
      "add x15, x15, #0x180\n"
      "fmla v16.4s, v4.4s, v1.s[3]\n"
      "fmla v22.4s, v4.4s, v2.s[3]\n"
      "fmla v28.4s, v4.4s, v3.s[3]\n"
      "fmla v11.4s, v5.4s, v0.s[3]\n"
      "fmla v17.4s, v5.4s, v1.s[3]\n"
      "fmla v23.4s, v5.4s, v2.s[3]\n"
      "fmla v29.4s, v5.4s, v3.s[3]\n"
      "fmla v12.4s, v6.4s, v0.s[3]\n"
      "fmla v18.4s, v6.4s, v1.s[3]\n"
      "fmla v24.4s, v6.4s, v2.s[3]\n"
      "fmla v30.4s, v6.4s, v3.s[3]\n"
      "fmla v13.4s, v7.4s, v0.s[3]\n"
      "fmla v19.4s, v7.4s, v1.s[3]\n"
      "fmla v25.4s, v7.4s, v2.s[3]\n"
      "fmla v31.4s, v7.4s, v3.s[3]\n"
      "147:"  // Height 4: Multiply loop: Main loop skip
      "cbz x12, 149f\n"
      "148:"  // Height 4: Multiply loop: Odd block loop
      "ldr s7, [x11], #0x4\n"
      "sub x12, x12, #0x1\n"
      "ldr s6, [x10], #0x4\n"
      "ldr s5, [x9], #0x4\n"
      "ldr s4, [x28], #0x4\n"
      "ldr q1, [x15, #0x0]\n"
      "fmla v8.4s, v1.4s, v7.s[0]\n"
      "ldr q0, [x15, #0x10]\n"
      "fmla v14.4s, v1.4s, v6.s[0]\n"
      "ldr q3, [x15, #0x20]\n"
      "fmla v20.4s, v1.4s, v5.s[0]\n"
      "ldr q2, [x15, #0x30]\n"
      "fmla v26.4s, v1.4s, v4.s[0]\n"
      "ldr q1, [x15, #0x40]\n"
      "fmla v9.4s, v0.4s, v7.s[0]\n"
      "fmla v15.4s, v0.4s, v6.s[0]\n"
      "fmla v21.4s, v0.4s, v5.s[0]\n"
      "fmla v27.4s, v0.4s, v4.s[0]\n"
      "ldr q0, [x15, #0x50]\n"
      "fmla v10.4s, v3.4s, v7.s[0]\n"
      "add x15, x15, #0x60\n"
      "fmla v16.4s, v3.4s, v6.s[0]\n"
      "fmla v22.4s, v3.4s, v5.s[0]\n"
      "fmla v28.4s, v3.4s, v4.s[0]\n"
      "fmla v11.4s, v2.4s, v7.s[0]\n"
      "fmla v17.4s, v2.4s, v6.s[0]\n"
      "fmla v23.4s, v2.4s, v5.s[0]\n"
      "fmla v29.4s, v2.4s, v4.s[0]\n"
      "fmla v12.4s, v1.4s, v7.s[0]\n"
      "fmla v18.4s, v1.4s, v6.s[0]\n"
      "fmla v24.4s, v1.4s, v5.s[0]\n"
      "fmla v30.4s, v1.4s, v4.s[0]\n"
      "fmla v13.4s, v0.4s, v7.s[0]\n"
      "fmla v19.4s, v0.4s, v6.s[0]\n"
      "fmla v25.4s, v0.4s, v5.s[0]\n"
      "fmla v31.4s, v0.4s, v4.s[0]\n"
      "cbnz x12, 148b\n"
      "149:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x13, x13, #0x1\n"
      "cmp x13, x20\n"
      "bne 142b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x14, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "prfm pstl1keep, [x14, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "tbz %x[flags], #1, 150f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v0.4s\n"
      "fmin v9.4s, v9.4s, v0.4s\n"
      "fmin v10.4s, v10.4s, v0.4s\n"
      "fmin v11.4s, v11.4s, v0.4s\n"
      "fmin v12.4s, v12.4s, v0.4s\n"
      "fmin v13.4s, v13.4s, v0.4s\n"
      "fmin v14.4s, v14.4s, v0.4s\n"
      "fmin v15.4s, v15.4s, v0.4s\n"
      "fmin v16.4s, v16.4s, v0.4s\n"
      "fmin v17.4s, v17.4s, v0.4s\n"
      "fmin v18.4s, v18.4s, v0.4s\n"
      "fmin v19.4s, v19.4s, v0.4s\n"
      "fmin v20.4s, v20.4s, v0.4s\n"
      "fmin v21.4s, v21.4s, v0.4s\n"
      "fmin v22.4s, v22.4s, v0.4s\n"
      "fmin v23.4s, v23.4s, v0.4s\n"
      "fmin v24.4s, v24.4s, v0.4s\n"
      "fmin v25.4s, v25.4s, v0.4s\n"
      "fmin v26.4s, v26.4s, v0.4s\n"
      "fmin v27.4s, v27.4s, v0.4s\n"
      "fmin v28.4s, v28.4s, v0.4s\n"
      "fmin v29.4s, v29.4s, v0.4s\n"
      "fmin v30.4s, v30.4s, v0.4s\n"
      "fmin v31.4s, v31.4s, v0.4s\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "fmax v28.4s, v28.4s, v0.4s\n"
      "fmax v29.4s, v29.4s, v0.4s\n"
      "fmax v30.4s, v30.4s, v0.4s\n"
      "fmax v31.4s, v31.4s, v0.4s\n"
      "150:"  // Height 4: No activation
      "cmp x16, #0x18\n"
      "bge 163f\n"
      "tbz x16, #4, 154f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v11.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v17.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v23.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "st1 { v27.4s }, [x21], #0x10\n"
      "st1 { v28.4s }, [x21], #0x10\n"
      "st1 { v29.4s }, [x21], #0x10\n"
      "tbz x16, #2, 152f\n"
      "st1 { v12.4s }, [x14], #0x10\n"
      "st1 { v18.4s }, [x23], #0x10\n"
      "st1 { v24.4s }, [x22], #0x10\n"
      "st1 { v30.4s }, [x21], #0x10\n"
      "tbz x16, #1, 151f\n"
      "str d13, [x14], #0x8\n"
      "str d19, [x23], #0x8\n"
      "str d25, [x22], #0x8\n"
      "str d31, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v13.s }[2], [x14]\n"
      "st1 { v19.s }[2], [x23]\n"
      "st1 { v25.s }[2], [x22]\n"
      "st1 { v31.s }[2], [x21]\n"
      "b 162f\n"
      "151:"  // Height 4: Partial direct writeback: partial_1_20
      "tbz x16, #0, 162f\n"
      "str s13, [x14, #0x0]\n"
      "str s19, [x23, #0x0]\n"
      "str s25, [x22, #0x0]\n"
      "str s31, [x21, #0x0]\n"
      "b 162f\n"
      "152:"  // Height 4: Partial direct writeback: partial_2_16
      "tbz x16, #1, 153f\n"
      "str d12, [x14], #0x8\n"
      "str d18, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "str d30, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v12.s }[2], [x14]\n"
      "st1 { v18.s }[2], [x23]\n"
      "st1 { v24.s }[2], [x22]\n"
      "st1 { v30.s }[2], [x21]\n"
      "b 162f\n"
      "153:"  // Height 4: Partial direct writeback: partial_1_16
      "tbz x16, #0, 162f\n"
      "str s12, [x14, #0x0]\n"
      "str s18, [x23, #0x0]\n"
      "str s24, [x22, #0x0]\n"
      "str s30, [x21, #0x0]\n"
      "b 162f\n"
      "154:"  // Height 4: Partial direct writeback: partial_8_0
      "tbz x16, #3, 158f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v9.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v15.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v21.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "st1 { v27.4s }, [x21], #0x10\n"
      "tbz x16, #2, 156f\n"
      "st1 { v10.4s }, [x14], #0x10\n"
      "st1 { v16.4s }, [x23], #0x10\n"
      "st1 { v22.4s }, [x22], #0x10\n"
      "st1 { v28.4s }, [x21], #0x10\n"
      "tbz x16, #1, 155f\n"
      "str d11, [x14], #0x8\n"
      "str d17, [x23], #0x8\n"
      "str d23, [x22], #0x8\n"
      "str d29, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v11.s }[2], [x14]\n"
      "st1 { v17.s }[2], [x23]\n"
      "st1 { v23.s }[2], [x22]\n"
      "st1 { v29.s }[2], [x21]\n"
      "b 162f\n"
      "155:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x16, #0, 162f\n"
      "str s11, [x14, #0x0]\n"
      "str s17, [x23, #0x0]\n"
      "str s23, [x22, #0x0]\n"
      "str s29, [x21, #0x0]\n"
      "b 162f\n"
      "156:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x16, #1, 157f\n"
      "str d10, [x14], #0x8\n"
      "str d16, [x23], #0x8\n"
      "str d22, [x22], #0x8\n"
      "str d28, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v10.s }[2], [x14]\n"
      "st1 { v16.s }[2], [x23]\n"
      "st1 { v22.s }[2], [x22]\n"
      "st1 { v28.s }[2], [x21]\n"
      "b 162f\n"
      "157:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x16, #0, 162f\n"
      "str s10, [x14, #0x0]\n"
      "str s16, [x23, #0x0]\n"
      "str s22, [x22, #0x0]\n"
      "str s28, [x21, #0x0]\n"
      "b 162f\n"
      "158:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x16, #2, 160f\n"
      "st1 { v8.4s }, [x14], #0x10\n"
      "st1 { v14.4s }, [x23], #0x10\n"
      "st1 { v20.4s }, [x22], #0x10\n"
      "st1 { v26.4s }, [x21], #0x10\n"
      "tbz x16, #1, 159f\n"
      "str d9, [x14], #0x8\n"
      "str d15, [x23], #0x8\n"
      "str d21, [x22], #0x8\n"
      "str d27, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v9.s }[2], [x14]\n"
      "st1 { v15.s }[2], [x23]\n"
      "st1 { v21.s }[2], [x22]\n"
      "st1 { v27.s }[2], [x21]\n"
      "b 162f\n"
      "159:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x16, #0, 162f\n"
      "str s9, [x14, #0x0]\n"
      "str s15, [x23, #0x0]\n"
      "str s21, [x22, #0x0]\n"
      "str s27, [x21, #0x0]\n"
      "b 162f\n"
      "160:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x16, #1, 161f\n"
      "str d8, [x14], #0x8\n"
      "str d14, [x23], #0x8\n"
      "str d20, [x22], #0x8\n"
      "str d26, [x21], #0x8\n"
      "tbz x16, #0, 162f\n"
      "st1 { v8.s }[2], [x14]\n"
      "st1 { v14.s }[2], [x23]\n"
      "st1 { v20.s }[2], [x22]\n"
      "st1 { v26.s }[2], [x21]\n"
      "b 162f\n"
      "161:"  // Height 4: Partial direct writeback: partial_1_0
      "str s8, [x14, #0x0]\n"
      "str s14, [x23, #0x0]\n"
      "str s20, [x22, #0x0]\n"
      "str s26, [x21, #0x0]\n"
      "162:"  // Height 4: Partial direct writeback: Done
      "b 164f\n"
      "163:"  // Height 4: Full writeback
      "str q8, [x14, #0x0]\n"
      "str q9, [x14, #0x10]\n"
      "str q10, [x14, #0x20]\n"
      "str q11, [x14, #0x30]\n"
      "str q12, [x14, #0x40]\n"
      "str q13, [x14, #0x50]\n"
      "add x14, x14, #0x60\n"
      "str q14, [x23, #0x0]\n"
      "str q15, [x23, #0x10]\n"
      "str q16, [x23, #0x20]\n"
      "str q17, [x23, #0x30]\n"
      "str q18, [x23, #0x40]\n"
      "str q19, [x23, #0x50]\n"
      "str q20, [x22, #0x0]\n"
      "str q21, [x22, #0x10]\n"
      "str q22, [x22, #0x20]\n"
      "str q23, [x22, #0x30]\n"
      "str q24, [x22, #0x40]\n"
      "str q25, [x22, #0x50]\n"
      "str q26, [x21, #0x0]\n"
      "str q27, [x21, #0x10]\n"
      "str q28, [x21, #0x20]\n"
      "str q29, [x21, #0x30]\n"
      "str q30, [x21, #0x40]\n"
      "str q31, [x21, #0x50]\n"
      "164:"  // Height 4: Writeback done
      "subs x16, x16, #0x18\n"
      "bgt 125b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 166f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 165f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "165:"  // Update direct input
      "mov x20, #0x10\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "166:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
