/*
 * Copyright (c) 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <cstddef>
#include <cstdint>

#if defined(ARM_COMPUTE_ENABLE_SME2)

namespace arm_conv {
namespace depthwise {

void sme2_fp16_nhwc_3x3_s1_output4x4_mla_depthfirst_direct_impl(
  const unsigned int n_tile_rows,
  const unsigned int n_tile_cols,
  const __fp16 *inptr,
  int64_t ld_input_row,
  int64_t ld_input_col,
  __fp16 *outptr,
  int64_t ld_output_row,
  int64_t ld_output_col,
  const void *params,
  unsigned int n_channels,
  const __fp16 activation_min,
  const __fp16 activation_max
)
{
  struct Args
  {
    const uint64_t n_tile_rows, n_tile_cols;
    const __fp16 *inptr;
    const uint64_t ld_input_row;
    const uint64_t ld_input_col;
    __fp16 *outptr;
    const uint64_t ld_output_row;
    const uint64_t ld_output_col;
    const void *params;
    const __fp16 min, max;

    uint64_t tile_i = 0, tile_j = 0;

    Args(
      const unsigned int n_tile_rows,
      const unsigned int n_tile_cols,
      const __fp16 *inptr,
      int64_t ld_input_row,
      int64_t ld_input_col,
      __fp16 *outptr,
      int64_t ld_output_row,
      int64_t ld_output_col,
      const void *params,
      const float activation_min,
      const float activation_max
    ) : n_tile_rows(n_tile_rows), n_tile_cols(n_tile_cols), inptr(inptr),
        ld_input_row(ld_input_row), ld_input_col(ld_input_col), outptr(outptr),
        ld_output_row(ld_output_row), ld_output_col(ld_output_col),
        params(params), min(activation_min), max(activation_max)
    {
    }
  };

  Args params_struct(
    n_tile_rows, n_tile_cols,
    inptr, ld_input_row, ld_input_col,
    outptr, ld_output_row, ld_output_col,
    params, activation_min, activation_max
  );

  __asm__ __volatile__(
    ".inst 0xd503477f  // SMSTART ZA\n"
    "mov x1, #0x0\n"
    "mov x2, #0x0\n"
    "ptrue p3.b\n"
    ".inst 0x25207810  // ptrue pn8.b\n"
    "1:"  // Tile loop
    "str x1, [%x[params_struct], %[offsetof_args_tile_i]]\n"
    "mov x22, #0x4\n"
    "str x2, [%x[params_struct], %[offsetof_args_tile_j]]\n"
    "ldr x21, [%x[params_struct], %[offsetof_args_ld_input_row]]\n"
    "ldr x3, [%x[params_struct], %[offsetof_args_ld_input_col]]\n"
    "ldr x4, [%x[params_struct], %[offsetof_args_inptr]]\n"
    "mul x20, x1, x21\n"  // offset = tile_i * ld_input_row
    "ldr x5, [%x[params_struct], %[offsetof_args_params]]\n"
    "madd x20, x2, x3, x20\n"  // offset += tile_j * ld_input_col
    "mul x20, x20, x22\n"  // offset *= kernel_stride * output_size
    "add x6, x3, x3\n"
    "add x4, x4, x20, LSL #1\n"  // inptr[0] += offset * sizeof(__fp16)
    "add x7, x4, x21, LSL #1\n"
    "add x8, x6, x3\n"
    "add x17, x7, x21, LSL #1\n"
    "add x16, x8, x3\n"
    "add x15, x17, x21, LSL #1\n"
    "add x14, x16, x3\n"
    "add x13, x15, x21, LSL #1\n"
    "add x12, x13, x21, LSL #1\n"
    "cbnz x2, 2f\n"
    "ldr x11, [%x[params_struct], %[offsetof_args_n_tile_cols]]\n"
    "lsl x10, %x[n_channels], #0x1\n"
    "mov x21, #0x8\n"
    "mul x21, x21, x3\n"
    "add x9, x17, x6, LSL #1\n"
    "add x28, x4, x14, LSL #1\n"
    "add x27, x17, x8, LSL #1\n"
    "sub x20, x11, x2\n"
    "add x26, x12, x14, LSL #1\n"
    "sub x20, x20, #0x1\n"
    "add x25, x15, x6, LSL #1\n"
    "and x20, x20, #0x3fffff\n"
    "add x24, x4, x3, LSL #1\n"
    "orr x10, x10, x20, LSL #22\n"
    "add x23, x4, x16, LSL #1\n"
    "orr x10, x10, x21, LSL #38\n"
    "add x22, x15, x8, LSL #1\n"
    "add x21, x7, x14, LSL #1\n"
    "add x20, x7, x6, LSL #1\n"
    ".inst 0xf8aa493a  // rprfm pldonce, x10, [x9]\n"
    "add x9, x13, x14, LSL #1\n"
    ".inst 0xf8aa489a  // rprfm pldonce, x10, [x4]\n"
    ".inst 0xf8aa4b9a  // rprfm pldonce, x10, [x28]\n"
    "add x28, x7, x8, LSL #1\n"
    ".inst 0xf8aa4b7a  // rprfm pldonce, x10, [x27]\n"
    "add x27, x12, x3, LSL #1\n"
    ".inst 0xf8aa499a  // rprfm pldonce, x10, [x12]\n"
    ".inst 0xf8aa4b5a  // rprfm pldonce, x10, [x26]\n"
    "add x26, x17, x3, LSL #1\n"
    ".inst 0xf8aa4b3a  // rprfm pldonce, x10, [x25]\n"
    "add x25, x12, x16, LSL #1\n"
    ".inst 0xf8aa4b1a  // rprfm pldonce, x10, [x24]\n"
    "add x24, x17, x16, LSL #1\n"
    ".inst 0xf8aa4afa  // rprfm pldonce, x10, [x23]\n"
    "add x23, x4, x6, LSL #1\n"
    ".inst 0xf8aa4ada  // rprfm pldonce, x10, [x22]\n"
    "add x22, x15, x3, LSL #1\n"
    ".inst 0xf8aa48fa  // rprfm pldonce, x10, [x7]\n"
    ".inst 0xf8aa4aba  // rprfm pldonce, x10, [x21]\n"
    "add x21, x4, x8, LSL #1\n"
    ".inst 0xf8aa49ba  // rprfm pldonce, x10, [x13]\n"
    ".inst 0xf8aa4a9a  // rprfm pldonce, x10, [x20]\n"
    "add x20, x15, x16, LSL #1\n"
    ".inst 0xf8aa493a  // rprfm pldonce, x10, [x9]\n"
    "add x9, x17, x14, LSL #1\n"
    ".inst 0xf8aa4b9a  // rprfm pldonce, x10, [x28]\n"
    "add x28, x13, x6, LSL #1\n"
    ".inst 0xf8aa4b7a  // rprfm pldonce, x10, [x27]\n"
    "add x27, x15, x14, LSL #1\n"
    ".inst 0xf8aa4b5a  // rprfm pldonce, x10, [x26]\n"
    "add x26, x12, x6, LSL #1\n"
    ".inst 0xf8aa4b3a  // rprfm pldonce, x10, [x25]\n"
    "add x25, x13, x8, LSL #1\n"
    ".inst 0xf8aa4b1a  // rprfm pldonce, x10, [x24]\n"
    "add x24, x12, x8, LSL #1\n"
    ".inst 0xf8aa4afa  // rprfm pldonce, x10, [x23]\n"
    "add x23, x7, x3, LSL #1\n"
    ".inst 0xf8aa4ada  // rprfm pldonce, x10, [x22]\n"
    "add x22, x7, x16, LSL #1\n"
    ".inst 0xf8aa4aba  // rprfm pldonce, x10, [x21]\n"
    "add x21, x13, x3, LSL #1\n"
    ".inst 0xf8aa4a3a  // rprfm pldonce, x10, [x17]\n"
    ".inst 0xf8aa4a9a  // rprfm pldonce, x10, [x20]\n"
    "add x20, x13, x16, LSL #1\n"
    ".inst 0xf8aa493a  // rprfm pldonce, x10, [x9]\n"
    ".inst 0xf8aa49fa  // rprfm pldonce, x10, [x15]\n"
    ".inst 0xf8aa4b9a  // rprfm pldonce, x10, [x28]\n"
    ".inst 0xf8aa4b7a  // rprfm pldonce, x10, [x27]\n"
    ".inst 0xf8aa4b5a  // rprfm pldonce, x10, [x26]\n"
    ".inst 0xf8aa4b3a  // rprfm pldonce, x10, [x25]\n"
    ".inst 0xf8aa4b1a  // rprfm pldonce, x10, [x24]\n"
    ".inst 0xf8aa4afa  // rprfm pldonce, x10, [x23]\n"
    ".inst 0xf8aa4ada  // rprfm pldonce, x10, [x22]\n"
    ".inst 0xf8aa4aba  // rprfm pldonce, x10, [x21]\n"
    ".inst 0xf8aa4a9a  // rprfm pldonce, x10, [x20]\n"
    "2:"  // Tile loop: Prefetch input rows: End
    "ldr x22, [%x[params_struct], %[offsetof_args_ld_output_row]]\n"
    "mov x21, #0x4\n"
    "ld1h { z15.h }, p3/Z, [x5]\n"
    "addvl x5, x5, #1\n"
    "ldr x9, [%x[params_struct], %[offsetof_args_ld_output_col]]\n"
    "cnth x28\n"
    ".inst 0xa040a0a0  // ld1h { z0.h-z3.h }, pn8.b/Z, [x5]\n"
    "addvl x5, x5, #4\n"
    "ldr x27, [%x[params_struct], %[offsetof_args_outptr]]\n"
    "whilelt p2.h, XZR, %x[n_channels]\n"
    ".inst 0xa040a0a4  // ld1h { z4.h-z7.h }, pn8.b/Z, [x5]\n"
    "addvl x5, x5, #4\n"
    "mul x20, x1, x22\n"  // offset = tile_i * ld_output_row
    "cmp x28, %x[n_channels]\n"
    "ld1rh { z14.h }, p3/Z, [%x[params_struct], %[offsetof_args_min]]\n"
    "madd x20, x2, x9, x20\n"  // offset += tile_j * ld_output_col
    "add x26, x9, x9\n"
    "ld1rh { z13.h }, p3/Z, [%x[params_struct], %[offsetof_args_max]]\n"
    "mul x20, x20, x21\n"  // offset *= output_tile_size
    "add x25, x26, x9\n"
    "ld1h { z8.h }, p3/Z, [x5]\n"
    "add x27, x27, x20, LSL #1\n"  // outptrs[0] += offset * sizeof(__fp16)
    "mov x21, #0x0\n"
    "ld1h { z9.h }, p2/Z, [x17, x6, LSL #1]\n"
    "add x24, x27, x22, LSL #1\n"
    "sub x20, XZR, x28\n"
    "ld1h { z10.h }, p2/Z, [x4]\n"
    "add x23, x24, x22, LSL #1\n"
    "ld1h { z11.h }, p2/Z, [x4, x14, LSL #1]\n"
    "addvl x5, x5, #1\n"
    "add x22, x23, x22, LSL #1\n"
    "ld1h { z12.h }, p2/Z, [x17, x8, LSL #1]\n"
    "bge 4f\n"
    "3:"  // Tile loop: Channel loop
    "movprfx z21, z15\n fmla z21.h, p3/M, z4.h, z9.h\n"
    "movprfx z16, z15\n fmla z16.h, p3/M, z8.h, z9.h\n"
    "whilelt p1.h, x28, %x[n_channels]\n"
    "inch x21\n"
    "movprfx z22, z15\n fmla z22.h, p3/M, z3.h, z9.h\n"
    "movprfx z25, z15\n fmla z25.h, p3/M, z1.h, z9.h\n"
    "inch x28\n"
    "mov p0.b, p2.b\n"
    "movprfx z26, z15\n fmla z26.h, p3/M, z0.h, z9.h\n"
    "movprfx z17, z15\n fmla z17.h, p3/M, z7.h, z9.h\n"
    "inch x20\n"
    "movprfx z18, z15\n fmla z18.h, p3/M, z6.h, z9.h\n"
    "movprfx z20, z15\n fmla z20.h, p3/M, z5.h, z9.h\n"
    "fmla z21.h, p3/M, z5.h, z12.h\n"
    "movprfx z24, z15\n fmla z24.h, p3/M, z2.h, z9.h\n"
    "ld1h { z9.h }, p2/Z, [x15, x6, LSL #1]\n"
    "fmla z16.h, p3/M, z0.h, z10.h\n"
    "movprfx z19, z15\n fmla z19.h, p3/M, z2.h, z11.h\n"
    "ld1h { z10.h }, p2/Z, [x12]\n"
    "fmla z22.h, p3/M, z4.h, z12.h\n"
    "fmla z25.h, p3/M, z2.h, z12.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x14, LSL #1]\n"
    "fmla z26.h, p3/M, z1.h, z12.h\n"
    "fmla z17.h, p3/M, z8.h, z12.h\n"
    "movprfx z28, z15\n fmla z28.h, p3/M, z6.h, z10.h\n"
    "fmla z21.h, p3/M, z7.h, z9.h\n"
    "ld1h { z10.h }, p2/Z, [x15, x8, LSL #1]\n"
    "fmla z18.h, p3/M, z7.h, z12.h\n"
    "fmla z19.h, p3/M, z6.h, z12.h\n"
    "movprfx z23, z15\n fmla z23.h, p3/M, z3.h, z12.h\n"
    "movprfx z27, z15\n fmla z27.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x4, x3, LSL #1]\n"
    "movprfx z31, z15\n fmla z31.h, p3/M, z8.h, z11.h\n"
    "fmla z22.h, p3/M, z6.h, z9.h\n"
    "ld1h { z11.h }, p2/Z, [x4, x16, LSL #1]\n"
    "fmla z25.h, p3/M, z4.h, z9.h\n"
    "fmla z26.h, p3/M, z3.h, z9.h\n"
    "movprfx z29, z15\n fmla z29.h, p3/M, z1.h, z9.h\n"
    "movprfx z30, z15\n fmla z30.h, p3/M, z0.h, z9.h\n"
    "ld1h { z15.h }, p3/Z, [x5]\n"
    "addvl x5, x5, #1\n"
    "fmla z20.h, p3/M, z8.h, z9.h\n"
    "fmla z24.h, p3/M, z5.h, z9.h\n"
    "fmla z28.h, p3/M, z2.h, z9.h\n"
    "fmla z21.h, p3/M, z8.h, z10.h\n"
    "ld1h { z9.h }, p2/Z, [x7]\n"
    "fmla z16.h, p3/M, z1.h, z12.h\n"
    "fmla z17.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x7, x14, LSL #1]\n"
    "fmla z18.h, p3/M, z2.h, z11.h\n"
    "fmla z19.h, p3/M, z1.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13]\n"
    "fmla z22.h, p3/M, z7.h, z10.h\n"
    "fmla z23.h, p3/M, z6.h, z10.h\n"
    "fmla z25.h, p3/M, z5.h, z10.h\n"
    "fmla z26.h, p3/M, z4.h, z10.h\n"
    "fmla z27.h, p3/M, z3.h, z10.h\n"
    "fmla z29.h, p3/M, z2.h, z10.h\n"
    "fmla z30.h, p3/M, z1.h, z10.h\n"
    "fmla z31.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x7, x6, LSL #1]\n"
    "fmla z20.h, p3/M, z0.h, z9.h\n"
    "fmla z24.h, p3/M, z6.h, z11.h\n"
    "fmla z28.h, p3/M, z3.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x14, LSL #1]\n"
    "fmla z16.h, p3/M, z3.h, z9.h\n"
    "fmla z21.h, p3/M, z1.h, z10.h\n"
    "fmla z19.h, p3/M, z5.h, z12.h\n"
    "fmla z23.h, p3/M, z2.h, z12.h\n"
    "fmla z17.h, p3/M, z4.h, z10.h\n"
    "ld1h { z12.h }, p2/Z, [x7, x8, LSL #1]\n"
    "fmla z18.h, p3/M, z3.h, z10.h\n"
    "fmla z22.h, p3/M, z0.h, z10.h\n"
    "fmla z27.h, p3/M, z8.h, z11.h\n"
    "fmla z31.h, p3/M, z5.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x3, LSL #1]\n"
    "fmla z20.h, p3/M, z2.h, z10.h\n"
    "fmla z21.h, p3/M, z2.h, z12.h\n"
    "fmla z16.h, p3/M, z5.h, z10.h\n"
    "fmla z17.h, p3/M, z5.h, z12.h\n"
    "ld1h { z10.h }, p2/Z, [x17, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z4.h, z12.h\n"
    "fmla z19.h, p3/M, z3.h, z12.h\n"
    "fmla z22.h, p3/M, z1.h, z12.h\n"
    "fmla z23.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x17, x16, LSL #1]\n"
    "fmla z28.h, p3/M, z7.h, z11.h\n"
    "fmla z29.h, p3/M, z6.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x16, LSL #1]\n"
    "fmla z20.h, p3/M, z4.h, z10.h\n"
    "fmla z21.h, p3/M, z3.h, z10.h\n"
    "fmla z24.h, p3/M, z1.h, z10.h\n"
    "fmla z25.h, p3/M, z0.h, z10.h\n"
    "fmla z16.h, p3/M, z7.h, z10.h\n"
    "fmla z17.h, p3/M, z6.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x4, x6, LSL #1]\n"
    "fmla z30.h, p3/M, z8.h, z11.h\n"
    "fmla z31.h, p3/M, z7.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x15, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z8.h, z12.h\n"
    "fmla z19.h, p3/M, z7.h, z12.h\n"
    "fmla z22.h, p3/M, z5.h, z12.h\n"
    "fmla z23.h, p3/M, z4.h, z12.h\n"
    "fmla z26.h, p3/M, z2.h, z12.h\n"
    "fmla z27.h, p3/M, z1.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x4, x8, LSL #1]\n"
    "addvl x4, x4, #1\n"
    "fmla z20.h, p3/M, z7.h, z11.h\n"
    "fmla z21.h, p3/M, z6.h, z11.h\n"
    "fmla z24.h, p3/M, z4.h, z11.h\n"
    "fmla z25.h, p3/M, z3.h, z11.h\n"
    "fmla z28.h, p3/M, z1.h, z11.h\n"
    "fmla z29.h, p3/M, z0.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x15, x16, LSL #1]\n"
    "fmla z16.h, p3/M, z2.h, z10.h\n"
    "fmla z17.h, p3/M, z1.h, z10.h\n"
    "fmla z18.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x17]\n"
    "fmla z19.h, p3/M, z0.h, z12.h\n"
    "fmla z30.h, p3/M, z2.h, z11.h\n"
    "fmla z22.h, p3/M, z8.h, z11.h\n"
    "fmla z23.h, p3/M, z7.h, z11.h\n"
    "fmla z26.h, p3/M, z5.h, z11.h\n"
    "fmla z20.h, p3/M, z3.h, z10.h\n"
    "fmla z24.h, p3/M, z0.h, z10.h\n"
    "fmla z27.h, p3/M, z4.h, z11.h\n"
    "fmla z31.h, p3/M, z1.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x6, LSL #1]\n"
    "fmla z17.h, p3/M, z2.h, z12.h\n"
    "fmla z18.h, p3/M, z1.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x17, x14, LSL #1]\n"
    "addvl x17, x17, #1\n"
    "fmla z16.h, p3/M, z6.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x15]\n"
    "fmla z29.h, p3/M, z4.h, z11.h\n"
    "fmla z30.h, p3/M, z3.h, z11.h\n"
    "fmla z25.h, p3/M, z7.h, z11.h\n"
    "ld1h { z9.h }, p1/Z, [x17, x6, LSL #1]\n"
    "fmla z19.h, p3/M, z8.h, z12.h\n"
    "fmla z23.h, p3/M, z5.h, z12.h\n"
    "fmla z27.h, p3/M, z2.h, z12.h\n"
    "fmla z20.h, p3/M, z6.h, z10.h\n"
    "ld1h { z12.h }, p2/Z, [x15, x14, LSL #1]\n"
    "addvl x15, x15, #1\n"
    "fmla z24.h, p3/M, z3.h, z10.h\n"
    "fmla z28.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x12, x6, LSL #1]\n"
    "fmla z26.h, p3/M, z6.h, z11.h\n"
    "fmla z31.h, p3/M, z2.h, z12.h\n"
    "fmla z23.h, p3/M, z8.h, z12.h\n"
    "fmla z29.h, p3/M, z7.h, z10.h\n"
    "fmla z30.h, p3/M, z6.h, z10.h\n"
    "fmla z27.h, p3/M, z5.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x12, x8, LSL #1]\n"
    "addvl x12, x12, #1\n"
    "fmla z24.h, p3/M, z8.h, z11.h\n"
    "fmla z28.h, p3/M, z5.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x8, LSL #1]\n"
    "fmla z29.h, p3/M, z5.h, z11.h\n"
    "fmla z30.h, p3/M, z4.h, z11.h\n"
    "fmla z31.h, p3/M, z3.h, z11.h\n"
    "fmla z25.h, p3/M, z8.h, z11.h\n"
    "fmla z26.h, p3/M, z7.h, z11.h\n"
    "fmla z27.h, p3/M, z6.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x7, x16, LSL #1]\n"
    "fmla z28.h, p3/M, z8.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x7, x3, LSL #1]\n"
    "addvl x7, x7, #1\n"
    "fmla z29.h, p3/M, z8.h, z12.h\n"
    "fmla z30.h, p3/M, z7.h, z12.h\n"
    "fmla z31.h, p3/M, z6.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x13, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z5.h, z11.h\n"
    "fmla z19.h, p3/M, z4.h, z11.h\n"
    "fmla z16.h, p3/M, z4.h, z10.h\n"
    "fmla z17.h, p3/M, z3.h, z10.h\n"
    "fmla z20.h, p3/M, z1.h, z10.h\n"
    "fmla z21.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x13, x16, LSL #1]\n"
    "whilelt p2.h, x21, %x[n_channels]\n"
    "fmla z22.h, p3/M, z2.h, z11.h\n"
    "fmla z23.h, p3/M, z1.h, z11.h\n"
    "cmp x28, %x[n_channels]\n"
    "addvl x13, x13, #1\n"
    "fmla z24.h, p3/M, z7.h, z12.h\n"
    "fmla z25.h, p3/M, z6.h, z12.h\n"
    "ld1h { z11.h }, p1/Z, [x4, x14, LSL #1]\n"
    "fmla z28.h, p3/M, z4.h, z12.h\n"
    "fmla z29.h, p3/M, z3.h, z12.h\n"
    ".inst 0xa040a0a0  // ld1h { z0.h-z3.h }, pn8.b/Z, [x5]\n"
    "addvl x5, x5, #4\n"
    "fmla z26.h, p3/M, z8.h, z10.h\n"
    "fmla z27.h, p3/M, z7.h, z10.h\n"
    "ld1h { z12.h }, p1/Z, [x17, x8, LSL #1]\n"
    "fmla z30.h, p3/M, z5.h, z10.h\n"
    "fmla z31.h, p3/M, z4.h, z10.h\n"
    ".inst 0xa040a0a4  // ld1h { z4.h-z7.h }, pn8.b/Z, [x5]\n"
    "addvl x5, x5, #4\n"
    ".inst 0xc16dc9d0  // fclamp { z16.h-z19.h }, z14.h, z13.h\n"
    ".inst 0xc16dc9d4  // fclamp { z20.h-z23.h }, z14.h, z13.h\n"
    "ld1h { z10.h }, p1/Z, [x4]\n"
    "ld1h { z8.h }, p3/Z, [x5]\n"
    "addvl x5, x5, #1\n"
    ".inst 0xc16dc9d8  // fclamp { z24.h-z27.h }, z14.h, z13.h\n"
    ".inst 0xc16dc9dc  // fclamp { z28.h-z31.h }, z14.h, z13.h\n"
    "st1h { z16.h }, p0, [x27]\n"
    "st1h { z17.h }, p0, [x27, x9, LSL #1]\n"
    "st1h { z18.h }, p0, [x27, x26, LSL #1]\n"
    "st1h { z19.h }, p0, [x27, x25, LSL #1]\n"
    "addvl x27, x27, #1\n"
    "st1h { z20.h }, p0, [x24]\n"
    "st1h { z21.h }, p0, [x24, x9, LSL #1]\n"
    "st1h { z22.h }, p0, [x24, x26, LSL #1]\n"
    "st1h { z23.h }, p0, [x24, x25, LSL #1]\n"
    "addvl x24, x24, #1\n"
    "st1h { z24.h }, p0, [x23]\n"
    "st1h { z25.h }, p0, [x23, x9, LSL #1]\n"
    "st1h { z26.h }, p0, [x23, x26, LSL #1]\n"
    "st1h { z27.h }, p0, [x23, x25, LSL #1]\n"
    "addvl x23, x23, #1\n"
    "st1h { z28.h }, p0, [x22]\n"
    "st1h { z29.h }, p0, [x22, x9, LSL #1]\n"
    "st1h { z30.h }, p0, [x22, x26, LSL #1]\n"
    "st1h { z31.h }, p0, [x22, x25, LSL #1]\n"
    "addvl x22, x22, #1\n"
    "blt 3b\n"
    "4:"  // Tile loop: Channel tail
    "movprfx z21, z15\n fmla z21.h, p3/M, z4.h, z9.h\n"
    "movprfx z16, z15\n fmla z16.h, p3/M, z8.h, z9.h\n"
    "ldr x2, [%x[params_struct], %[offsetof_args_tile_j]]\n"
    "mov p0.b, p2.b\n"
    "movprfx z22, z15\n fmla z22.h, p3/M, z3.h, z9.h\n"
    "movprfx z25, z15\n fmla z25.h, p3/M, z1.h, z9.h\n"
    "ldr x1, [%x[params_struct], %[offsetof_args_tile_i]]\n"
    "movprfx z26, z15\n fmla z26.h, p3/M, z0.h, z9.h\n"
    "movprfx z17, z15\n fmla z17.h, p3/M, z7.h, z9.h\n"
    "ldr x11, [%x[params_struct], %[offsetof_args_n_tile_cols]]\n"
    "movprfx z18, z15\n fmla z18.h, p3/M, z6.h, z9.h\n"
    "movprfx z20, z15\n fmla z20.h, p3/M, z5.h, z9.h\n"
    "ldr x21, [%x[params_struct], %[offsetof_args_n_tile_rows]]\n"
    "add x2, x2, #0x1\n"
    "fmla z21.h, p3/M, z5.h, z12.h\n"
    "movprfx z24, z15\n fmla z24.h, p3/M, z2.h, z9.h\n"
    "ld1h { z9.h }, p2/Z, [x15, x6, LSL #1]\n"
    "add x20, x1, #0x1\n"
    "fmla z16.h, p3/M, z0.h, z10.h\n"
    "movprfx z19, z15\n fmla z19.h, p3/M, z2.h, z11.h\n"
    "ld1h { z10.h }, p2/Z, [x12]\n"
    "cmp x2, x11\n"
    "fmla z22.h, p3/M, z4.h, z12.h\n"
    "fmla z25.h, p3/M, z2.h, z12.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x14, LSL #1]\n"
    "csel x1, x1, x20, LT\n"
    "fmla z26.h, p3/M, z1.h, z12.h\n"
    "fmla z17.h, p3/M, z8.h, z12.h\n"
    "csel x2, x2, XZR, LT\n"
    "cmp x1, x21\n"
    "movprfx z28, z15\n fmla z28.h, p3/M, z6.h, z10.h\n"
    "fmla z21.h, p3/M, z7.h, z9.h\n"
    "ld1h { z10.h }, p2/Z, [x15, x8, LSL #1]\n"
    "fmla z18.h, p3/M, z7.h, z12.h\n"
    "fmla z19.h, p3/M, z6.h, z12.h\n"
    "movprfx z23, z15\n fmla z23.h, p3/M, z3.h, z12.h\n"
    "movprfx z27, z15\n fmla z27.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x4, x3, LSL #1]\n"
    "movprfx z31, z15\n fmla z31.h, p3/M, z8.h, z11.h\n"
    "fmla z22.h, p3/M, z6.h, z9.h\n"
    "ld1h { z11.h }, p2/Z, [x4, x16, LSL #1]\n"
    "fmla z25.h, p3/M, z4.h, z9.h\n"
    "fmla z26.h, p3/M, z3.h, z9.h\n"
    "movprfx z29, z15\n fmla z29.h, p3/M, z1.h, z9.h\n"
    "movprfx z30, z15\n fmla z30.h, p3/M, z0.h, z9.h\n"
    "fmla z20.h, p3/M, z8.h, z9.h\n"
    "fmla z24.h, p3/M, z5.h, z9.h\n"
    "fmla z28.h, p3/M, z2.h, z9.h\n"
    "fmla z21.h, p3/M, z8.h, z10.h\n"
    "ld1h { z9.h }, p2/Z, [x7]\n"
    "fmla z16.h, p3/M, z1.h, z12.h\n"
    "fmla z17.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x7, x14, LSL #1]\n"
    "fmla z18.h, p3/M, z2.h, z11.h\n"
    "fmla z19.h, p3/M, z1.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13]\n"
    "fmla z22.h, p3/M, z7.h, z10.h\n"
    "fmla z23.h, p3/M, z6.h, z10.h\n"
    "fmla z25.h, p3/M, z5.h, z10.h\n"
    "fmla z26.h, p3/M, z4.h, z10.h\n"
    "fmla z27.h, p3/M, z3.h, z10.h\n"
    "fmla z29.h, p3/M, z2.h, z10.h\n"
    "fmla z30.h, p3/M, z1.h, z10.h\n"
    "fmla z31.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x7, x6, LSL #1]\n"
    "fmla z20.h, p3/M, z0.h, z9.h\n"
    "fmla z24.h, p3/M, z6.h, z11.h\n"
    "fmla z28.h, p3/M, z3.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x14, LSL #1]\n"
    "fmla z16.h, p3/M, z3.h, z9.h\n"
    "fmla z21.h, p3/M, z1.h, z10.h\n"
    "fmla z19.h, p3/M, z5.h, z12.h\n"
    "fmla z23.h, p3/M, z2.h, z12.h\n"
    "fmla z17.h, p3/M, z4.h, z10.h\n"
    "ld1h { z12.h }, p2/Z, [x7, x8, LSL #1]\n"
    "fmla z18.h, p3/M, z3.h, z10.h\n"
    "fmla z22.h, p3/M, z0.h, z10.h\n"
    "fmla z27.h, p3/M, z8.h, z11.h\n"
    "fmla z31.h, p3/M, z5.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x3, LSL #1]\n"
    "fmla z20.h, p3/M, z2.h, z10.h\n"
    "fmla z21.h, p3/M, z2.h, z12.h\n"
    "fmla z16.h, p3/M, z5.h, z10.h\n"
    "fmla z17.h, p3/M, z5.h, z12.h\n"
    "ld1h { z10.h }, p2/Z, [x17, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z4.h, z12.h\n"
    "fmla z19.h, p3/M, z3.h, z12.h\n"
    "fmla z22.h, p3/M, z1.h, z12.h\n"
    "fmla z23.h, p3/M, z0.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x17, x16, LSL #1]\n"
    "fmla z28.h, p3/M, z7.h, z11.h\n"
    "fmla z29.h, p3/M, z6.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x12, x16, LSL #1]\n"
    "fmla z20.h, p3/M, z4.h, z10.h\n"
    "fmla z21.h, p3/M, z3.h, z10.h\n"
    "fmla z24.h, p3/M, z1.h, z10.h\n"
    "fmla z25.h, p3/M, z0.h, z10.h\n"
    "fmla z16.h, p3/M, z7.h, z10.h\n"
    "fmla z17.h, p3/M, z6.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x4, x6, LSL #1]\n"
    "fmla z30.h, p3/M, z8.h, z11.h\n"
    "fmla z31.h, p3/M, z7.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x15, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z8.h, z12.h\n"
    "fmla z19.h, p3/M, z7.h, z12.h\n"
    "fmla z22.h, p3/M, z5.h, z12.h\n"
    "fmla z23.h, p3/M, z4.h, z12.h\n"
    "fmla z26.h, p3/M, z2.h, z12.h\n"
    "fmla z27.h, p3/M, z1.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x4, x8, LSL #1]\n"
    "fmla z20.h, p3/M, z7.h, z11.h\n"
    "fmla z21.h, p3/M, z6.h, z11.h\n"
    "fmla z24.h, p3/M, z4.h, z11.h\n"
    "fmla z25.h, p3/M, z3.h, z11.h\n"
    "fmla z28.h, p3/M, z1.h, z11.h\n"
    "fmla z29.h, p3/M, z0.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x15, x16, LSL #1]\n"
    "fmla z16.h, p3/M, z2.h, z10.h\n"
    "fmla z17.h, p3/M, z1.h, z10.h\n"
    "fmla z18.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x17]\n"
    "fmla z19.h, p3/M, z0.h, z12.h\n"
    "fmla z30.h, p3/M, z2.h, z11.h\n"
    "fmla z22.h, p3/M, z8.h, z11.h\n"
    "fmla z23.h, p3/M, z7.h, z11.h\n"
    "fmla z26.h, p3/M, z5.h, z11.h\n"
    "fmla z20.h, p3/M, z3.h, z10.h\n"
    "fmla z24.h, p3/M, z0.h, z10.h\n"
    "fmla z27.h, p3/M, z4.h, z11.h\n"
    "fmla z31.h, p3/M, z1.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x6, LSL #1]\n"
    "fmla z17.h, p3/M, z2.h, z12.h\n"
    "fmla z18.h, p3/M, z1.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x17, x14, LSL #1]\n"
    "fmla z16.h, p3/M, z6.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x15]\n"
    "fmla z29.h, p3/M, z4.h, z11.h\n"
    "fmla z30.h, p3/M, z3.h, z11.h\n"
    "fmla z25.h, p3/M, z7.h, z11.h\n"
    "fmla z19.h, p3/M, z8.h, z12.h\n"
    "fmla z23.h, p3/M, z5.h, z12.h\n"
    "fmla z27.h, p3/M, z2.h, z12.h\n"
    "fmla z20.h, p3/M, z6.h, z10.h\n"
    "ld1h { z12.h }, p2/Z, [x15, x14, LSL #1]\n"
    "fmla z24.h, p3/M, z3.h, z10.h\n"
    "fmla z28.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x12, x6, LSL #1]\n"
    "fmla z26.h, p3/M, z6.h, z11.h\n"
    "fmla z31.h, p3/M, z2.h, z12.h\n"
    "fmla z23.h, p3/M, z8.h, z12.h\n"
    "fmla z29.h, p3/M, z7.h, z10.h\n"
    "fmla z30.h, p3/M, z6.h, z10.h\n"
    "fmla z27.h, p3/M, z5.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x12, x8, LSL #1]\n"
    "fmla z24.h, p3/M, z8.h, z11.h\n"
    "fmla z28.h, p3/M, z5.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x13, x8, LSL #1]\n"
    "fmla z29.h, p3/M, z5.h, z11.h\n"
    "fmla z30.h, p3/M, z4.h, z11.h\n"
    "fmla z31.h, p3/M, z3.h, z11.h\n"
    "fmla z25.h, p3/M, z8.h, z11.h\n"
    "fmla z26.h, p3/M, z7.h, z11.h\n"
    "fmla z27.h, p3/M, z6.h, z11.h\n"
    "ld1h { z11.h }, p2/Z, [x7, x16, LSL #1]\n"
    "fmla z28.h, p3/M, z8.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x7, x3, LSL #1]\n"
    "fmla z29.h, p3/M, z8.h, z12.h\n"
    "fmla z30.h, p3/M, z7.h, z12.h\n"
    "fmla z31.h, p3/M, z6.h, z12.h\n"
    "ld1h { z12.h }, p2/Z, [x13, x3, LSL #1]\n"
    "fmla z18.h, p3/M, z5.h, z11.h\n"
    "fmla z19.h, p3/M, z4.h, z11.h\n"
    "fmla z16.h, p3/M, z4.h, z10.h\n"
    "fmla z17.h, p3/M, z3.h, z10.h\n"
    "fmla z20.h, p3/M, z1.h, z10.h\n"
    "fmla z21.h, p3/M, z0.h, z10.h\n"
    "ld1h { z10.h }, p2/Z, [x13, x16, LSL #1]\n"
    "fmla z22.h, p3/M, z2.h, z11.h\n"
    "fmla z23.h, p3/M, z1.h, z11.h\n"
    "fmla z24.h, p3/M, z7.h, z12.h\n"
    "fmla z25.h, p3/M, z6.h, z12.h\n"
    "fmla z28.h, p3/M, z4.h, z12.h\n"
    "fmla z29.h, p3/M, z3.h, z12.h\n"
    "fmla z26.h, p3/M, z8.h, z10.h\n"
    "fmla z27.h, p3/M, z7.h, z10.h\n"
    "fmla z30.h, p3/M, z5.h, z10.h\n"
    "fmla z31.h, p3/M, z4.h, z10.h\n"
    ".inst 0xc16dc9d0  // fclamp { z16.h-z19.h }, z14.h, z13.h\n"
    ".inst 0xc16dc9d4  // fclamp { z20.h-z23.h }, z14.h, z13.h\n"
    ".inst 0xc16dc9d8  // fclamp { z24.h-z27.h }, z14.h, z13.h\n"
    ".inst 0xc16dc9dc  // fclamp { z28.h-z31.h }, z14.h, z13.h\n"
    "st1h { z16.h }, p0, [x27]\n"
    "st1h { z17.h }, p0, [x27, x9, LSL #1]\n"
    "st1h { z18.h }, p0, [x27, x26, LSL #1]\n"
    "st1h { z19.h }, p0, [x27, x25, LSL #1]\n"
    "st1h { z20.h }, p0, [x24]\n"
    "st1h { z21.h }, p0, [x24, x9, LSL #1]\n"
    "st1h { z22.h }, p0, [x24, x26, LSL #1]\n"
    "st1h { z23.h }, p0, [x24, x25, LSL #1]\n"
    "st1h { z24.h }, p0, [x23]\n"
    "st1h { z25.h }, p0, [x23, x9, LSL #1]\n"
    "st1h { z26.h }, p0, [x23, x26, LSL #1]\n"
    "st1h { z27.h }, p0, [x23, x25, LSL #1]\n"
    "st1h { z28.h }, p0, [x22]\n"
    "st1h { z29.h }, p0, [x22, x9, LSL #1]\n"
    "st1h { z30.h }, p0, [x22, x26, LSL #1]\n"
    "st1h { z31.h }, p0, [x22, x25, LSL #1]\n"
    "blt 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [n_channels] "r" ((unsigned long) n_channels), [offsetof_args_inptr] "I" (offsetof(Args, inptr)), [offsetof_args_ld_input_col] "I" (offsetof(Args, ld_input_col)), [offsetof_args_ld_input_row] "I" (offsetof(Args, ld_input_row)), [offsetof_args_ld_output_col] "I" (offsetof(Args, ld_output_col)), [offsetof_args_ld_output_row] "I" (offsetof(Args, ld_output_row)), [offsetof_args_max] "I" (offsetof(Args, max)), [offsetof_args_min] "I" (offsetof(Args, min)), [offsetof_args_n_tile_cols] "I" (offsetof(Args, n_tile_cols)), [offsetof_args_n_tile_rows] "I" (offsetof(Args, n_tile_rows)), [offsetof_args_outptr] "I" (offsetof(Args, outptr)), [offsetof_args_params] "I" (offsetof(Args, params)), [offsetof_args_tile_i] "I" (offsetof(Args, tile_i)), [offsetof_args_tile_j] "I" (offsetof(Args, tile_j)), [params_struct] "r" (&params_struct)
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif // defined(ARM_COMPUTE_ENABLE_SME2)
