// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_RENDERER_MEDIA_RTC_VIDEO_DECODER_H_
#define CONTENT_RENDERER_MEDIA_RTC_VIDEO_DECODER_H_

#include <stddef.h>
#include <stdint.h>

#include <deque>
#include <list>
#include <map>
#include <set>
#include <utility>

#include "base/gtest_prod_util.h"
#include "base/macros.h"
#include "base/memory/weak_ptr.h"
#include "base/synchronization/lock.h"
#include "base/threading/thread.h"
#include "content/common/content_export.h"
#include "media/base/bitstream_buffer.h"
#include "media/base/video_decoder.h"
#include "media/video/picture.h"
#include "media/video/video_decode_accelerator.h"
#include "third_party/webrtc/modules/video_coding/include/video_codec_interface.h"
#include "ui/gfx/geometry/rect.h"

namespace base {
class WaitableEvent;
};

namespace media {
class DecoderBuffer;
class GpuVideoAcceleratorFactories;
}

namespace gpu {
struct SyncToken;
}

namespace content {

// This class uses hardware accelerated video decoder to decode video for
// WebRTC. |vda_message_loop_| is the message loop proxy of the media thread,
// which VDA::Client methods run on. webrtc::VideoDecoder methods run on WebRTC
// DecodingThread or Chrome_libJingle_WorkerThread, which are trampolined to
// |vda_message_loop_|. Decode() is non-blocking and queues the buffers. Decoded
// frames are delivered to WebRTC on |vda_message_loop_|.
class CONTENT_EXPORT RTCVideoDecoder
    : NON_EXPORTED_BASE(public webrtc::VideoDecoder),
      public media::VideoDecodeAccelerator::Client {
 public:
  ~RTCVideoDecoder() override;

  // Creates a RTCVideoDecoder on the message loop of |factories|. Returns NULL
  // if failed. The video decoder will run on the message loop of |factories|.
  static scoped_ptr<RTCVideoDecoder> Create(
      webrtc::VideoCodecType type,
      media::GpuVideoAcceleratorFactories* factories);
  // Destroys |decoder| on the loop of |factories|
  static void Destroy(webrtc::VideoDecoder* decoder,
                      media::GpuVideoAcceleratorFactories* factories);

  // webrtc::VideoDecoder implementation.
  // Called on WebRTC DecodingThread.
  int32_t InitDecode(const webrtc::VideoCodec* codecSettings,
                     int32_t numberOfCores) override;
  // Called on WebRTC DecodingThread.
  int32_t Decode(const webrtc::EncodedImage& inputImage,
                 bool missingFrames,
                 const webrtc::RTPFragmentationHeader* fragmentation,
                 const webrtc::CodecSpecificInfo* codecSpecificInfo = NULL,
                 int64_t renderTimeMs = -1) override;
  // Called on WebRTC DecodingThread.
  int32_t RegisterDecodeCompleteCallback(
      webrtc::DecodedImageCallback* callback) override;
  // Called on Chrome_libJingle_WorkerThread. The child thread is blocked while
  // this runs.
  int32_t Release() override;
  // Called on Chrome_libJingle_WorkerThread. The child thread is blocked while
  // this runs.
  int32_t Reset() override;

  // VideoDecodeAccelerator::Client implementation.
  void ProvidePictureBuffers(uint32_t count,
                             const gfx::Size& size,
                             uint32_t texture_target) override;
  void DismissPictureBuffer(int32_t id) override;
  void PictureReady(const media::Picture& picture) override;
  void NotifyEndOfBitstreamBuffer(int32_t id) override;
  void NotifyFlushDone() override;
  void NotifyResetDone() override;
  void NotifyError(media::VideoDecodeAccelerator::Error error) override;

 private:
  // Metadata of a bitstream buffer.
  struct BufferData {
    BufferData(int32_t bitstream_buffer_id,
               uint32_t timestamp,
               size_t size,
               const gfx::Rect& visible_rect);
    BufferData();
    ~BufferData();
    int32_t bitstream_buffer_id;
    uint32_t timestamp;  // in 90KHz
    size_t size;  // buffer size
    gfx::Rect visible_rect;
  };

  FRIEND_TEST_ALL_PREFIXES(RTCVideoDecoderTest, IsBufferAfterReset);
  FRIEND_TEST_ALL_PREFIXES(RTCVideoDecoderTest, IsFirstBufferAfterReset);

  RTCVideoDecoder(webrtc::VideoCodecType type,
                  media::GpuVideoAcceleratorFactories* factories);

  // Requests a buffer to be decoded by VDA.
  void RequestBufferDecode();

  bool CanMoreDecodeWorkBeDone();

  // Returns true if bitstream buffer id |id_buffer| comes after |id_reset|.
  // This handles the wraparound.
  bool IsBufferAfterReset(int32_t id_buffer, int32_t id_reset);

  // Returns true if bitstream buffer |id_buffer| is the first buffer after
  // |id_reset|.
  bool IsFirstBufferAfterReset(int32_t id_buffer, int32_t id_reset);

  // Saves a WebRTC buffer in |decode_buffers_| for decode.
  void SaveToDecodeBuffers_Locked(const webrtc::EncodedImage& input_image,
                                  scoped_ptr<base::SharedMemory> shm_buffer,
                                  const BufferData& buffer_data);

  // Saves a WebRTC buffer in |pending_buffers_| waiting for SHM available.
  // Returns true on success.
  bool SaveToPendingBuffers_Locked(const webrtc::EncodedImage& input_image,
                                   const BufferData& buffer_data);

  // Gets SHM and moves pending buffers to decode buffers.
  void MovePendingBuffersToDecodeBuffers();

  scoped_refptr<media::VideoFrame> CreateVideoFrame(
      const media::Picture& picture,
      const media::PictureBuffer& pb,
      uint32_t timestamp,
      const gfx::Rect& visible_rect);

  // Resets VDA.
  void ResetInternal();

  // Static method is to allow it to run even after RVD is deleted.
  static void ReleaseMailbox(base::WeakPtr<RTCVideoDecoder> decoder,
                             media::GpuVideoAcceleratorFactories* factories,
                             int64_t picture_buffer_id,
                             uint32_t texture_id,
                             const gpu::SyncToken& release_sync_token);
  // Tells VDA that a picture buffer can be recycled.
  void ReusePictureBuffer(int64_t picture_buffer_id);

  // Create |vda_| on |vda_loop_proxy_|.
  void CreateVDA(media::VideoCodecProfile profile, base::WaitableEvent* waiter);

  void DestroyTextures();
  void DestroyVDA();

  // Gets a shared-memory segment of at least |min_size| bytes from
  // |available_shm_segments_|. Returns NULL if there is no buffer or the
  // buffer is not big enough.
  scoped_ptr<base::SharedMemory> GetSHM_Locked(size_t min_size);

  // Returns a shared-memory segment to the available pool.
  void PutSHM_Locked(scoped_ptr<base::SharedMemory> shm_buffer);

  // Allocates |count| shared memory buffers of |size| bytes.
  void CreateSHM(size_t count, size_t size);

  // Stores the buffer metadata to |input_buffer_data_|.
  void RecordBufferData(const BufferData& buffer_data);
  // Gets the buffer metadata from |input_buffer_data_|.
  void GetBufferData(int32_t bitstream_buffer_id,
                     uint32_t* timestamp,
                     gfx::Rect* visible_rect);

  // Records the result of InitDecode to UMA and returns |status|.
  int32_t RecordInitDecodeUMA(int32_t status);

  // Assert the contract that this class is operated on the right thread.
  void DCheckGpuVideoAcceleratorFactoriesTaskRunnerIsCurrent() const;

  // Query factories_ whether |profile| is supported and return true is so,
  // false otherwise. If true, also set resolution limits for |profile|
  // in min/max_resolution_.
  bool IsProfileSupported(media::VideoCodecProfile profile);

  // Clear the pending_buffers_ queue, freeing memory.
  void ClearPendingBuffers();

  enum State {
    UNINITIALIZED,  // The decoder has not initialized.
    INITIALIZED,    // The decoder has initialized.
    RESETTING,      // The decoder is being reset.
    DECODE_ERROR,   // Decoding error happened.
  };

  static const int32_t ID_LAST;     // maximum bitstream buffer id
  static const int32_t ID_HALF;     // half of the maximum bitstream buffer id
  static const int32_t ID_INVALID;  // indicates Reset or Release never occurred

  // The hardware video decoder.
  scoped_ptr<media::VideoDecodeAccelerator> vda_;

  // The video codec type, as reported by WebRTC.
  const webrtc::VideoCodecType video_codec_type_;

  // The size of the incoming video frames.
  gfx::Size frame_size_;

  media::GpuVideoAcceleratorFactories* const factories_;

  // The texture target used for decoded pictures.
  uint32_t decoder_texture_target_;

  // Metadata of the buffers that have been sent for decode.
  std::list<BufferData> input_buffer_data_;

  // A map from bitstream buffer IDs to bitstream buffers that are being
  // processed by VDA. The map owns SHM buffers.
  std::map<int32_t, base::SharedMemory*> bitstream_buffers_in_decoder_;

  // A map from picture buffer IDs to texture-backed picture buffers.
  std::map<int32_t, media::PictureBuffer> assigned_picture_buffers_;

  // PictureBuffers given to us by VDA via PictureReady, which we sent forward
  // as VideoFrames to be rendered via read_cb_, and which will be returned
  // to us via ReusePictureBuffer.
  typedef std::map<int32_t /* picture_buffer_id */, uint32_t /* texture_id */>
      PictureBufferTextureMap;
  PictureBufferTextureMap picture_buffers_at_display_;

  // The id that will be given to the next picture buffer.
  int32_t next_picture_buffer_id_;

  // Protects |state_|, |decode_complete_callback_| , |num_shm_buffers_|,
  // |available_shm_segments_|, |pending_buffers_|, |decode_buffers_|,
  // |next_bitstream_buffer_id_| and |reset_bitstream_buffer_id_|.
  base::Lock lock_;

  // The state of RTCVideoDecoder. Guarded by |lock_|.
  State state_;

  // Guarded by |lock_|.
  webrtc::DecodedImageCallback* decode_complete_callback_;

  // Total number of allocated SHM buffers. Guarded by |lock_|.
  size_t num_shm_buffers_;

  // Shared-memory buffer pool.  Since allocating SHM segments requires a
  // round-trip to the browser process, we keep allocation out of the
  // steady-state of the decoder. The vector owns SHM buffers. Guarded by
  // |lock_|.
  std::vector<base::SharedMemory*> available_shm_segments_;

  // A queue storing WebRTC encoding images (and their metadata) that are
  // waiting for the shared memory. Guarded by |lock_|.
  std::deque<std::pair<webrtc::EncodedImage, BufferData>> pending_buffers_;

  // A queue storing buffers (and their metadata) that will be sent to VDA for
  // decode. The queue owns SHM buffers. Guarded by |lock_|.
  std::deque<std::pair<base::SharedMemory*, BufferData>> decode_buffers_;

  // The id that will be given to the next bitstream buffer. Guarded by |lock_|.
  int32_t next_bitstream_buffer_id_;

  // A buffer that has an id less than this should be dropped because Reset or
  // Release has been called. Guarded by |lock_|.
  int32_t reset_bitstream_buffer_id_;

  // Minimum and maximum supported resolutions for the current profile/VDA.
  gfx::Size min_resolution_;
  gfx::Size max_resolution_;

  // Must be destroyed, or invalidated, on |vda_loop_proxy_|
  // NOTE: Weak pointers must be invalidated before all other member variables.
  base::WeakPtrFactory<RTCVideoDecoder> weak_factory_;

  DISALLOW_COPY_AND_ASSIGN(RTCVideoDecoder);
};

}  // namespace content

#endif  // CONTENT_RENDERER_MEDIA_RTC_VIDEO_DECODER_H_
