# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Space objects."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    "DEFAULT_SPACE_NAME",
    "Space",
    ]

import re

from django.core.exceptions import ValidationError
from django.db.models import CharField
from maasserver import DefaultMeta
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel


def validate_space_name(value):
    """Django validator: `value` must be either `None`, or valid."""
    if value is None:
        return
    namespec = re.compile('^[ \w-]+$')
    if not namespec.search(value):
        raise ValidationError("Invalid space name: %s." % value)

# Name of the special, default space.  This space cannot be deleted.
DEFAULT_SPACE_NAME = 'space-0'


class Space(CleanSave, TimestampedModel):
    """A `Space`.

    :ivar name: The short-human-identifiable name for this space.
    :ivar objects: An instance of the class :class:`SpaceManager`.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""
        verbose_name = "Space"
        verbose_name_plural = "Spaces"

    name = CharField(
        max_length=256, editable=True, null=True, blank=True, unique=False,
        validators=[validate_space_name])
