/*
 * Copyright © 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Thomas Voß <thomas.voss@canonical.com>
 */

#include "com/ubuntu/location/service/session/stub.h"

#include "com/ubuntu/location/logging.h"

#include <core/dbus/stub.h>

#include <functional>

namespace cul = com::ubuntu::location;
namespace culs = com::ubuntu::location::service;
namespace culss = com::ubuntu::location::service::session;

namespace dbus = core::dbus;

struct culss::Stub::Private
{
    void update_heading(const core::dbus::Message::Ptr& msg);
    void update_position(const core::dbus::Message::Ptr& msg);
    void update_velocity(const core::dbus::Message::Ptr& msg);

    Stub* parent;
    dbus::types::ObjectPath session_path;
    dbus::Object::Ptr object;
};

culss::Stub::Stub(const dbus::Bus::Ptr& bus,
                  const dbus::types::ObjectPath& session_path)
        : dbus::Stub<culss::Interface>(bus),
        d(new Private{this,
                        session_path,
                        access_service()->add_object_for_path(session_path)})
{
    d->object->install_method_handler<culss::Interface::UpdatePosition>(
        std::bind(&Stub::Private::update_position,
                  std::ref(d),
                  std::placeholders::_1));
    d->object->install_method_handler<culss::Interface::UpdateHeading>(
        std::bind(&Stub::Private::update_heading,
                  std::ref(d),
                  std::placeholders::_1));
    d->object->install_method_handler<culss::Interface::UpdateVelocity>(
        std::bind(&Stub::Private::update_velocity,
                  std::ref(d),
                  std::placeholders::_1));
}

culss::Stub::~Stub() noexcept {}

const dbus::types::ObjectPath& culss::Stub::path() const
{
    return d->session_path;
}

void culss::Stub::start_position_updates()
{
    auto result = d->object->transact_method<Interface::StartPositionUpdates,void>();

    if (result.is_error())
        throw std::runtime_error(result.error().print());
}

void culss::Stub::stop_position_updates() noexcept
{
    try {
        auto result = d->object->transact_method<Interface::StopPositionUpdates,void>();

        if (result.is_error())
            LOG(WARNING) << result.error();
    } catch(const std::runtime_error& e)
    {
        LOG(WARNING) << e.what();
    }
}

void culss::Stub::start_velocity_updates()
{
    auto result = d->object->transact_method<Interface::StartVelocityUpdates,void>();

    if (result.is_error())
        throw std::runtime_error(result.error().print());
}

void culss::Stub::stop_velocity_updates() noexcept
{
    try {
        auto result = d->object->transact_method<Interface::StopVelocityUpdates,void>();

        if (result.is_error())
            LOG(WARNING) << result.error();
    } catch(const std::runtime_error& e)
    {
        LOG(WARNING) << e.what();
    }
}

void culss::Stub::start_heading_updates()
{
    auto result = d->object->transact_method<Interface::StartHeadingUpdates,void>();

    if (result.is_error())
        throw std::runtime_error(result.error().print());
}

void culss::Stub::stop_heading_updates() noexcept
{
    try {
        auto result = d->object->transact_method<Interface::StopHeadingUpdates,void>();

        if (result.is_error())
            LOG(WARNING) << result.error();
    } catch(const std::runtime_error& e)
    {
        LOG(WARNING) << e.what();
    }
}

void culss::Stub::Private::update_heading(const core::dbus::Message::Ptr& incoming)
{
    try
    {
        Update<Heading> update; incoming->reader() >> update;
        parent->access_heading_updates_channel()(update);
        parent->access_bus()->send(dbus::Message::make_method_return(incoming));
    } catch(const std::runtime_error& e)
    {
        parent->access_bus()->send(dbus::Message::make_error(incoming, Interface::Errors::ErrorParsingUpdate::name(), e.what()));
    }
}

void culss::Stub::Private::update_position(const core::dbus::Message::Ptr& incoming)
{
    try
    {
        Update<Position> update; incoming->reader() >> update;
        parent->access_position_updates_channel()(update);
        parent->access_bus()->send(dbus::Message::make_method_return(incoming));
    } catch(const std::runtime_error& e)
    {
        parent->access_bus()->send(dbus::Message::make_error(incoming, Interface::Errors::ErrorParsingUpdate::name(), e.what()));
    }
}

void culss::Stub::Private::update_velocity(const core::dbus::Message::Ptr& incoming)
{
    try
    {
        Update<Velocity> update; incoming->reader() >> update;
        parent->access_velocity_updates_channel()(update);
        parent->access_bus()->send(dbus::Message::make_method_return(incoming));
    } catch(const std::runtime_error& e)
    {
        parent->access_bus()->send(dbus::Message::make_error(incoming, Interface::Errors::ErrorParsingUpdate::name(), e.what()));
    }
}
