import numpy as np

class PerformanceCurve:

  def __init__(self, values, max_value, num_bins):

    if max_value <= 0.0:
      raise ValueError("max_value must be larger 0.0!")

    if num_bins < 1:
      raise ValueError("num_points must be larger 0!")

    if len(values) == 0:
      raise ValueError("number of provided values must be larger 0!")

    self.max_value = max_value
    self.num_bins = num_bins
    self.bin_size = max_value / num_bins
    self.curve = list([0] * self.num_bins)

    num_values = len(values)
    for v in values:
      if v < self.max_value:
        start_bin = int(float(v) / self.bin_size)
        for i in range(start_bin, self.num_bins):
          self.curve[i] += 1

    for i in range(self.num_bins):
      self.curve[i] = float(self.curve[i]) / num_values     
    

  def GetValueAt(self, value):

    if value >= self.max_value or value < 0:
      raise RuntimeError("Invalid value!")

    value_bin = int(float(value)/self.bin_size)
    return self.curve[value_bin]
  
  def Subtract(self, other_curve):
    
    if self.max_value != other_curve.max_value:
      raise ValueError("max_size parameters must match!")

    if self.num_bins != other_curve.num_bins:
      raise ValueError("num_bins parameters must match!")

    for i in range(self.num_bins):
      self.curve[i] -= other_curve.curve[i]


  def Add(self, other_curve):

    if self.max_value != other_curve.max_value:
      raise ValueError("max_size parameters must match!")

    if self.num_bins != other_curve.num_bins:
      raise ValueError("num_bins parameters must match!")

    for i in range(self.num_bins):
      self.curve[i] += other_curve.curve[i]


  def AUC(self):

    integral = 0.0
    for i in range(self.num_bins):
      integral += self.curve[i] * self.bin_size

    return integral

  def Copy(self):

    dummy_values = [0.5]
    curve = PerformanceCurve(dummy_values, self.max_value, self.num_bins)
    curve.curve = list(self.curve)

    return curve




def CreateAVGPerformanceCurve(curves):

  num_curves = len(curves)
  
  if num_curves == 0:
    raise RuntimeError("Number of curves must be at least 1!")

  # check whether all curves are consistent
  max_value = curves[0].max_value
  num_bins = curves[0].num_bins
  for c in curves[1:]:
    if max_value != c.max_value:
      raise RuntimeError("All curves must be consistently parametrized")
    if num_bins != c.num_bins:
      raise RuntimeError("All curves must be consistently parametrized")

  #create dummy curve
  dummy_values = [0.5]
  curve = PerformanceCurve(dummy_values, max_value, num_bins)
  for i in range(len(curve.curve)):
    curve.curve[i] = 0.0

  #Add all curves
  for c in curves:
    curve.Add(c)

  #normalize
  for i in range(len(curve.curve)):
    curve.curve[i] /= num_curves

  return curve
