/* Pdlzip - LZMA lossless data compressor
   Copyright (C) 2010-2025 Antonio Diaz Diaz.

   This program is free software. Redistribution and use in source and
   binary forms, with or without modification, are permitted provided
   that the following conditions are met:

   1. Redistributions of source code must retain the above copyright
   notice, this list of conditions, and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions, and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef max
  #define max(x,y) ((x) >= (y) ? (x) : (y))
#endif

typedef int State;

enum {
  min_dictionary_bits = 12,
  min_dictionary_size = 1 << min_dictionary_bits,	/* >= modeled_distances */
  max_dictionary_bits = 29,
  max_dictionary_size = 1 << max_dictionary_bits,
  max_dictionary_bits_c = 27,			/* kDicLogSizeMaxCompress */
  max_dictionary_size_c = 1 << max_dictionary_bits_c,
  literal_context_bits = 3,
  literal_pos_state_bits = 0,				/* not used */
  pos_state_bits = 2,

  len_low_bits = 3,
  len_mid_bits = 3,
  len_high_bits = 8,
  len_low_symbols = 1 << len_low_bits,
  len_mid_symbols = 1 << len_mid_bits,
  len_high_symbols = 1 << len_high_bits,
  max_len_symbols = len_low_symbols + len_mid_symbols + len_high_symbols,

  min_match_len = 2,					/* must be 2 */
  max_match_len = min_match_len + max_len_symbols - 1,	/* 273 */
  min_match_len_limit = 5 };


typedef uint32_t CRC32[256];	/* Table of CRCs of all 8-bit messages. */

extern CRC32 crc32;

static inline void CRC32_init( void )
  {
  unsigned n;
  for( n = 0; n < 256; ++n )
    {
    unsigned c = n;
    int k;
    for( k = 0; k < 8; ++k )
      { if( c & 1 ) c = 0xEDB88320U ^ ( c >> 1 ); else c >>= 1; }
    crc32[n] = c;
    }
  }

/* about as fast as it is possible without messing with endianness */
static inline void CRC32_update_buf( uint32_t * const crc,
                                     const uint8_t * const buffer,
                                     const int size )
  {
  int i;
  uint32_t c = *crc;
  for( i = 0; i < size; ++i )
    c = crc32[(c^buffer[i])&0xFF] ^ ( c >> 8 );
  *crc = c;
  }


static inline bool isvalid_ds( const unsigned dictionary_size )
  { return dictionary_size >= min_dictionary_size &&
           dictionary_size <= max_dictionary_size; }


static inline int real_bits( unsigned value )
  {
  int bits = 0;
  while( value > 0 ) { value >>= 1; ++bits; }
  return bits;
  }


static const uint8_t lzip_magic[4] = { 0x4C, 0x5A, 0x49, 0x50 }; /* "LZIP" */

enum { Lh_size = 6 };
typedef uint8_t Lzip_header[Lh_size];	/* 0-3 magic bytes */
					/*   4 version */
					/*   5 coded dictionary size */

static inline void Lh_set_magic( Lzip_header data )
  { memcpy( data, lzip_magic, 4 ); data[4] = 1; }

static inline bool Lh_check_magic( const Lzip_header data )
  { return memcmp( data, lzip_magic, 4 ) == 0; }

/* detect (truncated) header */
static inline bool Lh_check_prefix( const Lzip_header data, const int sz )
  {
  int i; for( i = 0; i < sz && i < 4; ++i )
    if( data[i] != lzip_magic[i] ) return false;
  return sz > 0;
  }

/* detect corrupt header */
static inline bool Lh_check_corrupt( const Lzip_header data )
  {
  int matches = 0;
  int i; for( i = 0; i < 4; ++i )
    if( data[i] == lzip_magic[i] ) ++matches;
  return matches > 1 && matches < 4;
  }

static inline uint8_t Lh_version( const Lzip_header data )
  { return data[4]; }

static inline bool Lh_check_version( const Lzip_header data )
  { return data[4] == 1; }

static inline unsigned Lh_get_dictionary_size( const Lzip_header data )
  {
  unsigned sz = 1 << ( data[5] & 0x1F );
  if( sz > min_dictionary_size )
    sz -= ( sz / 16 ) * ( ( data[5] >> 5 ) & 7 );
  return sz;
  }

static inline bool Lh_set_dictionary_size( Lzip_header data, const unsigned sz )
  {
  if( !isvalid_ds( sz ) ) return false;
  data[5] = real_bits( sz - 1 );
  if( sz > min_dictionary_size )
    {
    const unsigned base_size = 1 << data[5];
    const unsigned fraction = base_size / 16;
    unsigned i;
    for( i = 7; i >= 1; --i )
      if( base_size - ( i * fraction ) >= sz )
        { data[5] |= i << 5; break; }
    }
  return true;
  }


enum { Lt_size = 20 };
typedef uint8_t Lzip_trailer[Lt_size];
			/*  0-3  CRC32 of the uncompressed data */
			/*  4-11 size of the uncompressed data */
			/* 12-19 member size including header and trailer */

static inline unsigned Lt_get_data_crc( const Lzip_trailer data )
  {
  unsigned tmp = 0;
  int i; for( i = 3; i >= 0; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Lt_set_data_crc( Lzip_trailer data, unsigned crc )
  { int i; for( i = 0; i <= 3; ++i ) { data[i] = (uint8_t)crc; crc >>= 8; } }

static inline unsigned long long Lt_get_data_size( const Lzip_trailer data )
  {
  unsigned long long tmp = 0;
  int i; for( i = 11; i >= 4; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Lt_set_data_size( Lzip_trailer data, unsigned long long sz )
  { int i; for( i = 4; i <= 11; ++i ) { data[i] = (uint8_t)sz; sz >>= 8; } }

static inline unsigned long long Lt_get_member_size( const Lzip_trailer data )
  {
  unsigned long long tmp = 0;
  int i; for( i = 19; i >= 12; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Lt_set_member_size( Lzip_trailer data, unsigned long long sz )
  { int i; for( i = 12; i <= 19; ++i ) { data[i] = (uint8_t)sz; sz >>= 8; } }


typedef struct Cl_options	/* command-line options */
  {
  bool ignore_trailing;
  bool loose_trailing;
  } Cl_options;

static inline void Cl_options_init( Cl_options * cl_opts )
  { cl_opts->ignore_trailing = true; cl_opts->loose_trailing = false; }


static inline void set_retval( int * retval, const int new_val )
  { if( *retval < new_val ) *retval = new_val; }

static const char * const empty_msg = "Empty member not allowed.";
static const char * const mem_msg = "Not enough memory.";
static const char * const trailing_msg = "Trailing data not allowed.";

/* defined in main.c */
extern int verbosity;
int readblock( const int fd, uint8_t * const buf, const int size );
int writeblock( const int fd, const uint8_t * const buf, const int size );

#define SZ_OK 0

#define SZ_ERROR_READ 8
#define SZ_ERROR_WRITE 9
