/*
 * lws-api-test-jose - RFC8152 cose_key tests
 *
 * Written in 2010-2021 by Andy Green <andy@warmcat.com>
 *
 * This file is made available under the Creative Commons CC0 1.0
 * Universal Public Domain Dedication.
 *
 * Raw key CBOR created from descriptions at
 *
 * https://github.com/cose-wg/Examples/blob/master/KeySet.txt
 */

#include <libwebsockets.h>
#include <stdlib.h>
#include <fcntl.h>

static int
key_import_cb(struct lws_cose_key *s, void *user)
{
	lwsl_notice("%s: key type %lld\n", __func__, (long long)s->kty);

	return 0;
}

static const uint8_t
	cose_key1[] = {
			0xa6, 0x01, 0x02, 0x02, 0x62,
			0x31, 0x31, 0x20, 0x01, 0x21,
			0x58, 0x20, 0xba, 0xc5, 0xb1,
			0x1c, 0xad, 0x8f, 0x99, 0xf9,
			0xc7, 0x2b, 0x05, 0xcf, 0x4b,
			0x9e, 0x26, 0xd2, 0x44, 0xdc,
			0x18, 0x9f, 0x74, 0x52, 0x28,
			0x25, 0x5a, 0x21, 0x9a, 0x86,
			0xd6, 0xa0, 0x9e, 0xff, 0x22,
			0x58, 0x20, 0x20, 0x13, 0x8b,
			0xf8, 0x2d, 0xc1, 0xb6, 0xd5,
			0x62, 0xbe, 0x0f, 0xa5, 0x4a,
			0xb7, 0x80, 0x4a, 0x3a, 0x64,
			0xb6, 0xd7, 0x2c, 0xcf, 0xed,
			0x6b, 0x6f, 0xb6, 0xed, 0x28,
			0xbb, 0xfc, 0x11, 0x7e, 0x23,
			0x58, 0x20, 0x57, 0xc9, 0x20,
			0x77, 0x66, 0x41, 0x46, 0xe8,
			0x76, 0x76, 0x0c, 0x95, 0x20,
			0xd0, 0x54, 0xaa, 0x93, 0xc3,
			0xaf, 0xb0, 0x4e, 0x30, 0x67,
			0x05, 0xdb, 0x60, 0x90, 0x30,
			0x85, 0x07, 0xb4, 0xd3 },
	cose_key2[] = {
			0xa6, 0x01, 0x02, 0x02, 0x78,
			0x24, 0x6d, 0x65, 0x72, 0x69,
			0x61, 0x64, 0x6f, 0x63, 0x2e,
			0x62, 0x72, 0x61, 0x6e, 0x64,
			0x79, 0x62, 0x75, 0x63, 0x6b,
			0x40, 0x62, 0x75, 0x63, 0x6b,
			0x6c, 0x61, 0x6e, 0x64, 0x2e,
			0x65, 0x78, 0x61, 0x6d, 0x70,
			0x6c, 0x65, 0x20, 0x01, 0x21,
			0x58, 0x20, 0x65, 0xed, 0xa5,
			0xa1, 0x25, 0x77, 0xc2, 0xba,
			0xe8, 0x29, 0x43, 0x7f, 0xe3,
			0x38, 0x70, 0x1a, 0x10, 0xaa,
			0xa3, 0x75, 0xe1, 0xbb, 0x5b,
			0x5d, 0xe1, 0x08, 0xde, 0x43,
			0x9c, 0x08, 0x55, 0x1d, 0x22,
			0x58, 0x20, 0x1e, 0x52, 0xed,
			0x75, 0x70, 0x11, 0x63, 0xf7,
			0xf9, 0xe4, 0x0d, 0xdf, 0x9f,
			0x34, 0x1b, 0x3d, 0xc9, 0xba,
			0x86, 0x0a, 0xf7, 0xe0, 0xca,
			0x7c, 0xa7, 0xe9, 0xee, 0xcd,
			0x00, 0x84, 0xd1, 0x9c, 0x23,
			0x58, 0x20, 0xaf, 0xf9, 0x07,
			0xc9, 0x9f, 0x9a, 0xd3, 0xaa,
			0xe6, 0xc4, 0xcd, 0xf2, 0x11,
			0x22, 0xbc, 0xe2, 0xbd, 0x68,
			0xb5, 0x28, 0x3e, 0x69, 0x07,
			0x15, 0x4a, 0xd9, 0x11, 0x84,
			0x0f, 0xa2, 0x08, 0xcf },

	cose_key3[] = { 0xa3, 0x01, 0x04, 0x02, 0x6a,
			0x6f, 0x75, 0x72, 0x2d, 0x73,
			0x65, 0x63, 0x72, 0x65, 0x74,
			0x20, 0x58, 0x20, 0x84, 0x9b,
			0x57, 0x21, 0x9d, 0xae, 0x48,
			0xde, 0x64, 0x6d, 0x07, 0xdb,
			0xb5, 0x33, 0x56, 0x6e, 0x97,
			0x66, 0x86, 0x45, 0x7c, 0x14,
			0x91, 0xbe, 0x3a, 0x76, 0xdc,
			0xea, 0x6c, 0x42, 0x71, 0x88 },

	cose_key4[] = { 0xa6, 0x01, 0x02, 0x02, 0x78,
			0x1e, 0x62, 0x69, 0x6c, 0x62,
			0x6f, 0x2e, 0x62, 0x61, 0x67,
			0x67, 0x69, 0x6e, 0x73, 0x40,
			0x68, 0x6f, 0x62, 0x62, 0x69,
			0x74, 0x6f, 0x6e, 0x2e, 0x65,
			0x78, 0x61, 0x6d, 0x70, 0x6c,
			0x65, 0x20, 0x03, 0x21, 0x58,
			0x42, 0x00, 0x72, 0x99, 0x2c,
			0xb3, 0xac, 0x08, 0xec, 0xf3,
			0xe5, 0xc6, 0x3d, 0xed, 0xec,
			0x0d, 0x51, 0xa8, 0xc1, 0xf7,
			0x9e, 0xf2, 0xf8, 0x2f, 0x94,
			0xf3, 0xc7, 0x37, 0xbf, 0x5d,
			0xe7, 0x98, 0x66, 0x71, 0xea,
			0xc6, 0x25, 0xfe, 0x82, 0x57,
			0xbb, 0xd0, 0x39, 0x46, 0x44,
			0xca, 0xaa, 0x3a, 0xaf, 0x8f,
			0x27, 0xa4, 0x58, 0x5f, 0xbb,
			0xca, 0xd0, 0xf2, 0x45, 0x76,
			0x20, 0x08, 0x5e, 0x5c, 0x8f,
			0x42, 0xad, 0x22, 0x58, 0x42,
			0x01, 0xdc, 0xa6, 0x94, 0x7b,
			0xce, 0x88, 0xbc, 0x57, 0x90,
			0x48, 0x5a, 0xc9, 0x74, 0x27,
			0x34, 0x2b, 0xc3, 0x5f, 0x88,
			0x7d, 0x86, 0xd6, 0x5a, 0x08,
			0x93, 0x77, 0xe2, 0x47, 0xe6,
			0x0b, 0xaa, 0x55, 0xe4, 0xe8,
			0x50, 0x1e, 0x2a, 0xda, 0x57,
			0x24, 0xac, 0x51, 0xd6, 0x90,
			0x90, 0x08, 0x03, 0x3e, 0xbc,
			0x10, 0xac, 0x99, 0x9b, 0x9d,
			0x7f, 0x5c, 0xc2, 0x51, 0x9f,
			0x3f, 0xe1, 0xea, 0x1d, 0x94,
			0x75, 0x23, 0x58, 0x42, 0x00,
			0x08, 0x51, 0x38, 0xdd, 0xab,
			0xf5, 0xca, 0x97, 0x5f, 0x58,
			0x60, 0xf9, 0x1a, 0x08, 0xe9,
			0x1d, 0x6d, 0x5f, 0x9a, 0x76,
			0xad, 0x40, 0x18, 0x76, 0x6a,
			0x47, 0x66, 0x80, 0xb5, 0x5c,
			0xd3, 0x39, 0xe8, 0xab, 0x6c,
			0x72, 0xb5, 0xfa, 0xcd, 0xb2,
			0xa2, 0xa5, 0x0a, 0xc2, 0x5b,
			0xd0, 0x86, 0x64, 0x7d, 0xd3,
			0xe2, 0xe6, 0xe9, 0x9e, 0x84,
			0xca, 0x2c, 0x36, 0x09, 0xfd,
			0xf1, 0x77, 0xfe, 0xb2, 0x6d },
	cose_key5[] = { 0xa3, 0x01, 0x04, 0x02, 0x6b,
			0x6f, 0x75, 0x72, 0x2d, 0x73,
			0x65, 0x63, 0x72, 0x65, 0x74,
			0x32, 0x20, 0x50, 0x84, 0x9b,
			0x57, 0x86, 0x45, 0x7c, 0x14,
			0x91, 0xbe, 0x3a, 0x76, 0xdc,
			0xea, 0x6c, 0x42, 0x71 },

	cose_key6[] = { 0xa6, 0x01, 0x02, 0x02, 0x78,
			0x21, 0x70, 0x65, 0x72, 0x65,
			0x67, 0x72, 0x69, 0x6e, 0x2e,
			0x74, 0x6f, 0x6f, 0x6b, 0x40,
			0x74, 0x75, 0x63, 0x6b, 0x62,
			0x6f, 0x72, 0x6f, 0x75, 0x67,
			0x68, 0x2e, 0x65, 0x78, 0x61,
			0x6d, 0x70, 0x6c, 0x65, 0x20,
			0x01, 0x21, 0x58, 0x20, 0x98,
			0xf5, 0x0a, 0x4f, 0xf6, 0xc0,
			0x58, 0x61, 0xc8, 0x86, 0x0d,
			0x13, 0xa6, 0x38, 0xea, 0x56,
			0xc3, 0xf5, 0xad, 0x75, 0x90,
			0xbb, 0xfb, 0xf0, 0x54, 0xe1,
			0xc7, 0xb4, 0xd9, 0x1d, 0x62,
			0x80, 0x22, 0x58, 0x20, 0xf0,
			0x14, 0x00, 0xb0, 0x89, 0x86,
			0x78, 0x04, 0xb8, 0xe9, 0xfc,
			0x96, 0xc3, 0x93, 0x21, 0x61,
			0xf1, 0x93, 0x4f, 0x42, 0x23,
			0x06, 0x91, 0x70, 0xd9, 0x24,
			0xb7, 0xe0, 0x3b, 0xf8, 0x22,
			0xbb, 0x23, 0x58, 0x20, 0x02,
			0xd1, 0xf7, 0xe6, 0xf2, 0x6c,
			0x43, 0xd4, 0x86, 0x8d, 0x87,
			0xce, 0xb2, 0x35, 0x31, 0x61,
			0x74, 0x0a, 0xac, 0xf1, 0xf7,
			0x16, 0x36, 0x47, 0x98, 0x4b,
			0x52, 0x2a, 0x84, 0x8d, 0xf1,
			0xc3 },
	cose_key7[] = { 0xa3, 0x01, 0x04, 0x02, 0x58,
			0x24, 0x30, 0x31, 0x38, 0x63,
			0x30, 0x61, 0x65, 0x35, 0x2d,
			0x34, 0x64, 0x39, 0x62, 0x2d,
			0x34, 0x37, 0x31, 0x62, 0x2d,
			0x62, 0x66, 0x64, 0x36, 0x2d,
			0x65, 0x65, 0x66, 0x33, 0x31,
			0x34, 0x62, 0x63, 0x37, 0x30,
			0x33, 0x37, 0x20, 0x58, 0x20,
			0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88 },

	cose_key8[] = {
			/* kid "sec-48" for hmac 384 */

			0xa3, 0x01, 0x04, 0x02, 0x66,
			0x73, 0x65, 0x63, 0x2d, 0x34,
			0x38, 0x20, 0x58, 0x30, 0x84,
			0x9b, 0x57, 0x21, 0x9d, 0xae,
			0x48, 0xde, 0x64, 0x6d, 0x07,
			0xdb, 0xb5, 0x33, 0x56, 0x6e,
			0x97, 0x66, 0x86, 0x45, 0x7c,
			0x14, 0x91, 0xbe, 0x3a, 0x76,
			0xdc, 0xea, 0x6c, 0x42, 0x71,
			0x88, 0x00, 0x11, 0x22, 0x33,
			0x77, 0x88, 0x99, 0xaa, 0x21,
			0x22, 0x23, 0x24, 0x25, 0x26,
			0x27, 0x28
	},

	cose_key9[] = {
			/* kid "sec-64" for hmac 512 */

			0xa3, 0x01, 0x04, 0x02, 0x46,
			0x73, 0x65, 0x63, 0x2d, 0x36,
			0x34, 0x20, 0x58, 0x40, 0x84,
			0x9b, 0x57, 0x21, 0x9d, 0xae,
			0x48, 0xde, 0x64, 0x6d, 0x07,
			0xdb, 0xb5, 0x33, 0x56, 0x6e,
			0x97, 0x66, 0x86, 0x45, 0x7c,
			0x14, 0x91, 0xbe, 0x3a, 0x76,
			0xdc, 0xea, 0x6c, 0x42, 0x71,
			0x88, 0x00, 0x11, 0x22, 0x33,
			0x77, 0x88, 0x99, 0xaa, 0x21,
			0x22, 0x23, 0x24, 0x25, 0x26,
			0x27, 0x28, 0xaa, 0xbb, 0xcc,
			0xdd, 0xee, 0xff, 0xa5, 0xa6,
			0xa7, 0xa8, 0xa9, 0xa0, 0xb1,
			0xb2, 0xb3, 0xb4,
	},

	cose_key10[] = { /* kid "11" (again) ed22519 OKP key */
			0xa5, 0x01, 0x01, 0x02, 0x42,
			0x31, 0x31, 0x20, 0x06, 0x21,
			0x58, 0x20, 0xd7, 0x5a, 0x98,
			0x01, 0x82, 0xb1, 0x0a, 0xb7,
			0xd5, 0x4b, 0xfe, 0xd3, 0xc9,
			0x64, 0x07, 0x3a, 0x0e, 0xe1,
			0x72, 0xf3, 0xda, 0xa6, 0x23,
			0x25, 0xaf, 0x02, 0x1a, 0x68,
			0xf7, 0x07, 0x51, 0x1a, 0x23,
			0x58, 0x20, 0x9d, 0x61, 0xb1,
			0x9d, 0xef, 0xfd, 0x5a, 0x60,
			0xba, 0x84, 0x4a, 0xf4, 0x92,
			0xec, 0x2c, 0xc4, 0x44, 0x49,
			0xc5, 0x69, 0x7b, 0x32, 0x69,
			0x19, 0x70, 0x3b, 0xac, 0x03,
			0x1c, 0xae, 0x7f, 0x60
	},

	cose_key_set1[] = {

			0x89,

			0xa6, 0x01, 0x02, 0x02, 0x42,
			0x31, 0x31, 0x20, 0x01, 0x21,
			0x58, 0x20, 0xba, 0xc5, 0xb1,
			0x1c, 0xad, 0x8f, 0x99, 0xf9,
			0xc7, 0x2b, 0x05, 0xcf, 0x4b,
			0x9e, 0x26, 0xd2, 0x44, 0xdc,
			0x18, 0x9f, 0x74, 0x52, 0x28,
			0x25, 0x5a, 0x21, 0x9a, 0x86,
			0xd6, 0xa0, 0x9e, 0xff, 0x22,
			0x58, 0x20, 0x20, 0x13, 0x8b,
			0xf8, 0x2d, 0xc1, 0xb6, 0xd5,
			0x62, 0xbe, 0x0f, 0xa5, 0x4a,
			0xb7, 0x80, 0x4a, 0x3a, 0x64,
			0xb6, 0xd7, 0x2c, 0xcf, 0xed,
			0x6b, 0x6f, 0xb6, 0xed, 0x28,
			0xbb, 0xfc, 0x11, 0x7e, 0x23,
			0x58, 0x20, 0x57, 0xc9, 0x20,
			0x77, 0x66, 0x41, 0x46, 0xe8,
			0x76, 0x76, 0x0c, 0x95, 0x20,
			0xd0, 0x54, 0xaa, 0x93, 0xc3,
			0xaf, 0xb0, 0x4e, 0x30, 0x67,
			0x05, 0xdb, 0x60, 0x90, 0x30,
			0x85, 0x07, 0xb4, 0xd3,

			0xa6, 0x01, 0x02, 0x02, 0x58,
			0x24, 0x6d, 0x65, 0x72, 0x69,
			0x61, 0x64, 0x6f, 0x63, 0x2e,
			0x62, 0x72, 0x61, 0x6e, 0x64,
			0x79, 0x62, 0x75, 0x63, 0x6b,
			0x40, 0x62, 0x75, 0x63, 0x6b,
			0x6c, 0x61, 0x6e, 0x64, 0x2e,
			0x65, 0x78, 0x61, 0x6d, 0x70,
			0x6c, 0x65, 0x20, 0x01, 0x21,
			0x58, 0x20, 0x65, 0xed, 0xa5,
			0xa1, 0x25, 0x77, 0xc2, 0xba,
			0xe8, 0x29, 0x43, 0x7f, 0xe3,
			0x38, 0x70, 0x1a, 0x10, 0xaa,
			0xa3, 0x75, 0xe1, 0xbb, 0x5b,
			0x5d, 0xe1, 0x08, 0xde, 0x43,
			0x9c, 0x08, 0x55, 0x1d, 0x22,
			0x58, 0x20, 0x1e, 0x52, 0xed,
			0x75, 0x70, 0x11, 0x63, 0xf7,
			0xf9, 0xe4, 0x0d, 0xdf, 0x9f,
			0x34, 0x1b, 0x3d, 0xc9, 0xba,
			0x86, 0x0a, 0xf7, 0xe0, 0xca,
			0x7c, 0xa7, 0xe9, 0xee, 0xcd,
			0x00, 0x84, 0xd1, 0x9c, 0x23,
			0x58, 0x20, 0xaf, 0xf9, 0x07,
			0xc9, 0x9f, 0x9a, 0xd3, 0xaa,
			0xe6, 0xc4, 0xcd, 0xf2, 0x11,
			0x22, 0xbc, 0xe2, 0xbd, 0x68,
			0xb5, 0x28, 0x3e, 0x69, 0x07,
			0x15, 0x4a, 0xd9, 0x11, 0x84,
			0x0f, 0xa2, 0x08, 0xcf,

			0xa3, 0x01, 0x04, 0x02, 0x4a,
			0x6f, 0x75, 0x72, 0x2d, 0x73,
			0x65, 0x63, 0x72, 0x65, 0x74,
			0x20, 0x58, 0x20, 0x84, 0x9b,
			0x57, 0x21, 0x9d, 0xae, 0x48,
			0xde, 0x64, 0x6d, 0x07, 0xdb,
			0xb5, 0x33, 0x56, 0x6e, 0x97,
			0x66, 0x86, 0x45, 0x7c, 0x14,
			0x91, 0xbe, 0x3a, 0x76, 0xdc,
			0xea, 0x6c, 0x42, 0x71, 0x88,

			0xa6, 0x01, 0x02, 0x02, 0x58,
			0x1e, 0x62, 0x69, 0x6c, 0x62,
			0x6f, 0x2e, 0x62, 0x61, 0x67,
			0x67, 0x69, 0x6e, 0x73, 0x40,
			0x68, 0x6f, 0x62, 0x62, 0x69,
			0x74, 0x6f, 0x6e, 0x2e, 0x65,
			0x78, 0x61, 0x6d, 0x70, 0x6c,
			0x65, 0x20, 0x03, 0x21, 0x58,
			0x42, 0x00, 0x72, 0x99, 0x2c,
			0xb3, 0xac, 0x08, 0xec, 0xf3,
			0xe5, 0xc6, 0x3d, 0xed, 0xec,
			0x0d, 0x51, 0xa8, 0xc1, 0xf7,
			0x9e, 0xf2, 0xf8, 0x2f, 0x94,
			0xf3, 0xc7, 0x37, 0xbf, 0x5d,
			0xe7, 0x98, 0x66, 0x71, 0xea,
			0xc6, 0x25, 0xfe, 0x82, 0x57,
			0xbb, 0xd0, 0x39, 0x46, 0x44,
			0xca, 0xaa, 0x3a, 0xaf, 0x8f,
			0x27, 0xa4, 0x58, 0x5f, 0xbb,
			0xca, 0xd0, 0xf2, 0x45, 0x76,
			0x20, 0x08, 0x5e, 0x5c, 0x8f,
			0x42, 0xad, 0x22, 0x58, 0x42,
			0x01, 0xdc, 0xa6, 0x94, 0x7b,
			0xce, 0x88, 0xbc, 0x57, 0x90,
			0x48, 0x5a, 0xc9, 0x74, 0x27,
			0x34, 0x2b, 0xc3, 0x5f, 0x88,
			0x7d, 0x86, 0xd6, 0x5a, 0x08,
			0x93, 0x77, 0xe2, 0x47, 0xe6,
			0x0b, 0xaa, 0x55, 0xe4, 0xe8,
			0x50, 0x1e, 0x2a, 0xda, 0x57,
			0x24, 0xac, 0x51, 0xd6, 0x90,
			0x90, 0x08, 0x03, 0x3e, 0xbc,
			0x10, 0xac, 0x99, 0x9b, 0x9d,
			0x7f, 0x5c, 0xc2, 0x51, 0x9f,
			0x3f, 0xe1, 0xea, 0x1d, 0x94,
			0x75, 0x23, 0x58, 0x42, 0x00,
			0x08, 0x51, 0x38, 0xdd, 0xab,
			0xf5, 0xca, 0x97, 0x5f, 0x58,
			0x60, 0xf9, 0x1a, 0x08, 0xe9,
			0x1d, 0x6d, 0x5f, 0x9a, 0x76,
			0xad, 0x40, 0x18, 0x76, 0x6a,
			0x47, 0x66, 0x80, 0xb5, 0x5c,
			0xd3, 0x39, 0xe8, 0xab, 0x6c,
			0x72, 0xb5, 0xfa, 0xcd, 0xb2,
			0xa2, 0xa5, 0x0a, 0xc2, 0x5b,
			0xd0, 0x86, 0x64, 0x7d, 0xd3,
			0xe2, 0xe6, 0xe9, 0x9e, 0x84,
			0xca, 0x2c, 0x36, 0x09, 0xfd,
			0xf1, 0x77, 0xfe, 0xb2, 0x6d,

			0xa3, 0x01, 0x04, 0x02, 0x4b,
			0x6f, 0x75, 0x72, 0x2d, 0x73,
			0x65, 0x63, 0x72, 0x65, 0x74,
			0x32, 0x20, 0x50, 0x84, 0x9b,
			0x57, 0x86, 0x45, 0x7c, 0x14,
			0x91, 0xbe, 0x3a, 0x76, 0xdc,
			0xea, 0x6c, 0x42, 0x71,

			0xa6, 0x01, 0x02, 0x02, 0x58,
			0x21, 0x70, 0x65, 0x72, 0x65,
			0x67, 0x72, 0x69, 0x6e, 0x2e,
			0x74, 0x6f, 0x6f, 0x6b, 0x40,
			0x74, 0x75, 0x63, 0x6b, 0x62,
			0x6f, 0x72, 0x6f, 0x75, 0x67,
			0x68, 0x2e, 0x65, 0x78, 0x61,
			0x6d, 0x70, 0x6c, 0x65, 0x20,
			0x01, 0x21, 0x58, 0x20, 0x98,
			0xf5, 0x0a, 0x4f, 0xf6, 0xc0,
			0x58, 0x61, 0xc8, 0x86, 0x0d,
			0x13, 0xa6, 0x38, 0xea, 0x56,
			0xc3, 0xf5, 0xad, 0x75, 0x90,
			0xbb, 0xfb, 0xf0, 0x54, 0xe1,
			0xc7, 0xb4, 0xd9, 0x1d, 0x62,
			0x80, 0x22, 0x58, 0x20, 0xf0,
			0x14, 0x00, 0xb0, 0x89, 0x86,
			0x78, 0x04, 0xb8, 0xe9, 0xfc,
			0x96, 0xc3, 0x93, 0x21, 0x61,
			0xf1, 0x93, 0x4f, 0x42, 0x23,
			0x06, 0x91, 0x70, 0xd9, 0x24,
			0xb7, 0xe0, 0x3b, 0xf8, 0x22,
			0xbb, 0x23, 0x58, 0x20, 0x02,
			0xd1, 0xf7, 0xe6, 0xf2, 0x6c,
			0x43, 0xd4, 0x86, 0x8d, 0x87,
			0xce, 0xb2, 0x35, 0x31, 0x61,
			0x74, 0x0a, 0xac, 0xf1, 0xf7,
			0x16, 0x36, 0x47, 0x98, 0x4b,
			0x52, 0x2a, 0x84, 0x8d, 0xf1,
			0xc3,

			0xa3, 0x01, 0x04, 0x02, 0x58,
			0x24, 0x30, 0x31, 0x38, 0x63,
			0x30, 0x61, 0x65, 0x35, 0x2d,
			0x34, 0x64, 0x39, 0x62, 0x2d,
			0x34, 0x37, 0x31, 0x62, 0x2d,
			0x62, 0x66, 0x64, 0x36, 0x2d,
			0x65, 0x65, 0x66, 0x33, 0x31,
			0x34, 0x62, 0x63, 0x37, 0x30,
			0x33, 0x37, 0x04, 0x58, 0x20,
			0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88,

			/* kid "sec-48" for hmac 384 */

			0xa3, 0x01, 0x04, 0x02, 0x46,
			0x73, 0x65, 0x63, 0x2d, 0x34,
			0x38, 0x20, 0x58, 0x30, 0x84,
			0x9b, 0x57, 0x21, 0x9d, 0xae,
			0x48, 0xde, 0x64, 0x6d, 0x07,
			0xdb, 0xb5, 0x33, 0x56, 0x6e,
			0x97, 0x66, 0x86, 0x45, 0x7c,
			0x14, 0x91, 0xbe, 0x3a, 0x76,
			0xdc, 0xea, 0x6c, 0x42, 0x71,
			0x88, 0x00, 0x11, 0x22, 0x33,
			0x77, 0x88, 0x99, 0xaa, 0x21,
			0x22, 0x23, 0x24, 0x25, 0x26,
			0x27, 0x28,

			/* kid "sec-64" for hmac 512 */

			0xa3, 0x01, 0x04, 0x02, 0x46,
			0x73, 0x65, 0x63, 0x2d, 0x36,
			0x34, 0x20, 0x58, 0x40, 0x84,
			0x9b, 0x57, 0x21, 0x9d, 0xae,
			0x48, 0xde, 0x64, 0x6d, 0x07,
			0xdb, 0xb5, 0x33, 0x56, 0x6e,
			0x97, 0x66, 0x86, 0x45, 0x7c,
			0x14, 0x91, 0xbe, 0x3a, 0x76,
			0xdc, 0xea, 0x6c, 0x42, 0x71,
			0x88, 0x00, 0x11, 0x22, 0x33,
			0x77, 0x88, 0x99, 0xaa, 0x21,
			0x22, 0x23, 0x24, 0x25, 0x26,
			0x27, 0x28, 0xaa, 0xbb, 0xcc,
			0xdd, 0xee, 0xff, 0xa5, 0xa6,
			0xa7, 0xa8, 0xa9, 0xa0, 0xb1,
			0xb2, 0xb3, 0xb4,
}
;

struct keyinfo {
	const uint8_t		*set;
	size_t			len;
};

struct keyinfo keyset1 = { cose_key_set1, sizeof(cose_key_set1) },
		key3 = { cose_key3, sizeof(cose_key3) },
		key8 = { cose_key8, sizeof(cose_key8) },
		key9 = { cose_key9, sizeof(cose_key9) },
		key10 = { cose_key10, sizeof(cose_key10) }
;

/* key pieces */

static const uint8_t
	key1_x[] = { 0xba, 0xc5, 0xb1, 0x1c, 0xad,
			0x8f, 0x99, 0xf9, 0xc7, 0x2b,
			0x05, 0xcf, 0x4b, 0x9e, 0x26,
			0xd2, 0x44, 0xdc, 0x18, 0x9f,
			0x74, 0x52, 0x28, 0x25, 0x5a,
			0x21, 0x9a, 0x86, 0xd6, 0xa0,
			0x9e, 0xff },
	key1_y[] = { 0x20, 0x13, 0x8b, 0xf8, 0x2d,
			0xc1, 0xb6, 0xd5, 0x62, 0xbe,
			0x0f, 0xa5, 0x4a, 0xb7, 0x80,
			0x4a, 0x3a, 0x64, 0xb6, 0xd7,
			0x2c, 0xcf, 0xed, 0x6b, 0x6f,
			0xb6, 0xed, 0x28, 0xbb, 0xfc,
			0x11, 0x7e },
	key1_d[] = { 0x57, 0xc9, 0x20, 0x77, 0x66,
			0x41, 0x46, 0xe8, 0x76, 0x76,
			0x0c, 0x95, 0x20, 0xd0, 0x54,
			0xaa, 0x93, 0xc3, 0xaf, 0xb0,
			0x4e, 0x30, 0x67, 0x05, 0xdb,
			0x60, 0x90, 0x30, 0x85, 0x07,
			0xb4, 0xd3 },

	key2_x[] = { 0x65, 0xed, 0xa5, 0xa1, 0x25,
			0x77, 0xc2, 0xba, 0xe8, 0x29,
			0x43, 0x7f, 0xe3, 0x38, 0x70,
			0x1a, 0x10, 0xaa, 0xa3, 0x75,
			0xe1, 0xbb, 0x5b, 0x5d, 0xe1,
			0x08, 0xde, 0x43, 0x9c, 0x08,
			0x55, 0x1d },
	key2_y[] = { 0x1e, 0x52, 0xed, 0x75, 0x70,
			0x11, 0x63, 0xf7, 0xf9, 0xe4,
			0x0d, 0xdf, 0x9f, 0x34, 0x1b,
			0x3d, 0xc9, 0xba, 0x86, 0x0a,
			0xf7, 0xe0, 0xca, 0x7c, 0xa7,
			0xe9, 0xee, 0xcd, 0x00, 0x84,
			0xd1, 0x9c },
	key2_d[] = { 0xaf, 0xf9, 0x07, 0xc9, 0x9f,
			0x9a, 0xd3, 0xaa, 0xe6, 0xc4,
			0xcd, 0xf2, 0x11, 0x22, 0xbc,
			0xe2, 0xbd, 0x68, 0xb5, 0x28,
			0x3e, 0x69, 0x07, 0x15, 0x4a,
			0xd9, 0x11, 0x84, 0x0f, 0xa2,
			0x08, 0xcf },

	key3_k[] = { 0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88 },

	key4_x[] = { 0x00, 0x72, 0x99, 0x2c, 0xb3,
			0xac, 0x08, 0xec, 0xf3, 0xe5,
			0xc6, 0x3d, 0xed, 0xec, 0x0d,
			0x51, 0xa8, 0xc1, 0xf7, 0x9e,
			0xf2, 0xf8, 0x2f, 0x94, 0xf3,
			0xc7, 0x37, 0xbf, 0x5d, 0xe7,
			0x98, 0x66, 0x71, 0xea, 0xc6,
			0x25, 0xfe, 0x82, 0x57, 0xbb,
			0xd0, 0x39, 0x46, 0x44, 0xca,
			0xaa, 0x3a, 0xaf, 0x8f, 0x27,
			0xa4, 0x58, 0x5f, 0xbb, 0xca,
			0xd0, 0xf2, 0x45, 0x76, 0x20,
			0x08, 0x5e, 0x5c, 0x8f, 0x42,
			0xad },
	key4_y[] = { 0x01, 0xdc, 0xa6, 0x94, 0x7b,
			0xce, 0x88, 0xbc, 0x57, 0x90,
			0x48, 0x5a, 0xc9, 0x74, 0x27,
			0x34, 0x2b, 0xc3, 0x5f, 0x88,
			0x7d, 0x86, 0xd6, 0x5a, 0x08,
			0x93, 0x77, 0xe2, 0x47, 0xe6,
			0x0b, 0xaa, 0x55, 0xe4, 0xe8,
			0x50, 0x1e, 0x2a, 0xda, 0x57,
			0x24, 0xac, 0x51, 0xd6, 0x90,
			0x90, 0x08, 0x03, 0x3e, 0xbc,
			0x10, 0xac, 0x99, 0x9b, 0x9d,
			0x7f, 0x5c, 0xc2, 0x51, 0x9f,
			0x3f, 0xe1, 0xea, 0x1d, 0x94,
			0x75 },
	key4_d[] = { 0x00, 0x08, 0x51, 0x38, 0xdd,
			0xab, 0xf5, 0xca, 0x97, 0x5f,
			0x58, 0x60, 0xf9, 0x1a, 0x08,
			0xe9, 0x1d, 0x6d, 0x5f, 0x9a,
			0x76, 0xad, 0x40, 0x18, 0x76,
			0x6a, 0x47, 0x66, 0x80, 0xb5,
			0x5c, 0xd3, 0x39, 0xe8, 0xab,
			0x6c, 0x72, 0xb5, 0xfa, 0xcd,
			0xb2, 0xa2, 0xa5, 0x0a, 0xc2,
			0x5b, 0xd0, 0x86, 0x64, 0x7d,
			0xd3, 0xe2, 0xe6, 0xe9, 0x9e,
			0x84, 0xca, 0x2c, 0x36, 0x09,
			0xfd, 0xf1, 0x77, 0xfe, 0xb2,
			0x6d },
	key5_k[] = { 0x84, 0x9b, 0x57, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71 },

	key6_x[] = { 0x98, 0xf5, 0x0a, 0x4f, 0xf6,
			0xc0, 0x58, 0x61, 0xc8, 0x86,
			0x0d, 0x13, 0xa6, 0x38, 0xea,
			0x56, 0xc3, 0xf5, 0xad, 0x75,
			0x90, 0xbb, 0xfb, 0xf0, 0x54,
			0xe1, 0xc7, 0xb4, 0xd9, 0x1d,
			0x62, 0x80 },
	key6_y[] = { 0xf0, 0x14, 0x00, 0xb0, 0x89,
			0x86, 0x78, 0x04, 0xb8, 0xe9,
			0xfc, 0x96, 0xc3, 0x93, 0x21,
			0x61, 0xf1, 0x93, 0x4f, 0x42,
			0x23, 0x06, 0x91, 0x70, 0xd9,
			0x24, 0xb7, 0xe0, 0x3b, 0xf8,
			0x22, 0xbb },
	key6_d[] = { 0x02, 0xd1, 0xf7, 0xe6, 0xf2,
			0x6c, 0x43, 0xd4, 0x86, 0x8d,
			0x87, 0xce, 0xb2, 0x35, 0x31,
			0x61, 0x74, 0x0a, 0xac, 0xf1,
			0xf7, 0x16, 0x36, 0x47, 0x98,
			0x4b, 0x52, 0x2a, 0x84, 0x8d,
			0xf1, 0xc3 },

	key7_k[] = { 0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88 },

	key8_k[] = { 0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88, 0x00, 0x11, 0x22,
			0x33, 0x77, 0x88, 0x99, 0xaa,
			0x21, 0x22, 0x23, 0x24, 0x25,
			0x26, 0x27, 0x28 },

	key9_k[] = {    0x84, 0x9b, 0x57, 0x21, 0x9d,
			0xae, 0x48, 0xde, 0x64, 0x6d,
			0x07, 0xdb, 0xb5, 0x33, 0x56,
			0x6e, 0x97, 0x66, 0x86, 0x45,
			0x7c, 0x14, 0x91, 0xbe, 0x3a,
			0x76, 0xdc, 0xea, 0x6c, 0x42,
			0x71, 0x88, 0x00, 0x11, 0x22,
			0x33, 0x77, 0x88, 0x99, 0xaa,
			0x21, 0x22, 0x23, 0x24, 0x25,
			0x26, 0x27, 0x28, 0xaa, 0xbb,
			0xcc, 0xdd, 0xee, 0xff, 0xa5,
			0xa6, 0xa7, 0xa8, 0xa9, 0xa0,
			0xb1, 0xb2, 0xb3, 0xb4 }
#if 0
			,
	key10_x[] = {
			0xd7, 0x5a, 0x98, 0x01, 0x82,
			0xb1, 0x0a, 0xb7, 0xd5, 0x4b,
			0xfe, 0xd3, 0xc9, 0x64, 0x07,
			0x3a, 0x0e, 0xe1, 0x72, 0xf3,
			0xda, 0xa6, 0x23, 0x25, 0xaf,
			0x02, 0x1a, 0x68, 0xf7, 0x07,
			0x51, 0x1a
	}, key10_d[] = {
			0x9d, 0x61, 0xb1, 0x9d, 0xef,
			0xfd, 0x5a, 0x60, 0xba, 0x84,
			0x4a, 0xf4, 0x92, 0xec, 0x2c,
			0xc4, 0x44, 0x49, 0xc5, 0x69,
			0x7b, 0x32, 0x69, 0x19, 0x70,
			0x3b, 0xac, 0x03, 0x1c, 0xae,
			0x7f, 0x60
	}
#endif
;

int
test_cose_keys(struct lws_context *context)
{
	struct lws_cose_key *ck;
	lws_dll2_owner_t set;
	lws_lec_pctx_t wc;
	uint8_t buf[4096];
	int n;

#if 0
	{
		int fd = open("set1.cks",
			      LWS_O_CREAT | LWS_O_TRUNC | LWS_O_WRONLY, 0600);

		if (fd >= 0) {
			write(fd, cose_key_set1, sizeof(cose_key_set1));
			close(fd);
		}
	}
#endif

#if 0
	lws_lec_pctx_t wx;
	uint8_t dump[8192];

	lws_lec_init(&wx, buf, sizeof(buf));

	if (lws_lec_printf(&wx,
		"{%d:%d, %d:%.*b, %d:%d, %d:%.*b, %d:%.*b}",
		LWSCOSE_WKK_KTY, LWSCOSE_WKKTV_OKP,
		LWSCOSE_WKK_KID, 2, "11",
		LWSCOSE_WKOKP_CRV, LWSCOSE_WKEC_ED25519,
		LWSCOSE_WKECKP_X, (int)sizeof(key10_x), key10_x,
//		LWSCOSE_WKECKP_Y, (int)sizeof(key6_y), key6_y,
		LWSCOSE_WKECKP_D, (int)sizeof(key10_d), key10_d) !=
			LWS_LECPCTX_RET_FINISHED)
		return 1;

	lws_hex_from_byte_array(buf, wx.used, (char *)dump, sizeof(dump));
	puts((const char *)dump);
#endif
#if 0
	lws_lec_pctx_t wx;
	uint8_t dump[8192];

	lws_lec_init(&wx, buf, sizeof(buf));

	if (lws_lec_printf(&wx,
		"{%d:%d, %d:%.*b, %d:%.*b}",
		LWSCOSE_WKK_KTY, LWSCOSE_WKKTV_SYMMETRIC,
		LWSCOSE_WKK_KID, 6, "sec-64",
		-1, (int)sizeof(key9_k), key9_k) !=
			LWS_LECPCTX_RET_FINISHED)
		return 1;

	lws_hex_from_byte_array(buf, wx.used, (char *)dump, sizeof(dump));
	puts((const char *)dump);
#endif

	/* key1 import */

	lwsl_user("%s: key 1 import\n", __func__);

	ck = lws_cose_key_import(NULL, key_import_cb, NULL, cose_key1, sizeof(cose_key1));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_EC2 ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_EC ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_X].len != sizeof(key1_x) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].len != sizeof(key1_y) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_D].len != sizeof(key1_d) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_X].buf, key1_x, sizeof(key1_x)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].buf, key1_y, sizeof(key1_y)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_D].buf, key1_d, sizeof(key1_d)))
		goto bail;

	// lws_cose_key_dump(ck);

	/* key 1 export */

	lwsl_user("%s: key 1 export\n", __func__);

	lws_lec_init(&wc, buf, sizeof(buf));
	n = (int)lws_cose_key_export(ck, &wc, LWSJWKF_EXPORT_PRIVATE);
	lws_cose_key_destroy(&ck);
	if (n != LWS_LECPCTX_RET_FINISHED)
		goto bail;

	// lwsl_hexdump_notice(buf, wc.used);

	/* key2 import */

	lwsl_user("%s: key 2 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key2, sizeof(cose_key2));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_EC2 ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_EC ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_X].len != sizeof(key2_x) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].len != sizeof(key2_y) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_D].len != sizeof(key2_d) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_X].buf, key2_x, sizeof(key2_x)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].buf, key2_y, sizeof(key2_y)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_D].buf, key2_d, sizeof(key2_d)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key3 import */

	lwsl_user("%s: key 3 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key3, sizeof(cose_key3));
	if (!ck) {
		lwsl_err("%s: key 3 import failed\n", __func__);
		goto bail;
	}

	if (ck->kty != LWSCOSE_WKKTV_SYMMETRIC ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_OCT ||
	    ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len != sizeof(key3_k) ||
	    memcmp(ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].buf, key3_k, sizeof(key3_k))) {
		lwsl_err("%s: key 3 checks failed %d %d %d\n", __func__,
				(int)ck->kty, (int)ck->gencrypto_kty,
				(int)ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len);
		goto bail;
	}

	lws_cose_key_destroy(&ck);

	/* key4 import */

	lwsl_user("%s: key 4 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key4, sizeof(cose_key4));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_EC2 ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_EC ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_X].len != sizeof(key4_x) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].len != sizeof(key4_y) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_D].len != sizeof(key4_d) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_X].buf, key4_x, sizeof(key4_x)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].buf, key4_y, sizeof(key4_y)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_D].buf, key4_d, sizeof(key4_d)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key5 import */

	lwsl_user("%s: key 5 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key5, sizeof(cose_key5));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_SYMMETRIC ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_OCT ||
	    ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len != sizeof(key5_k) ||
	    memcmp(ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].buf, key5_k, sizeof(key5_k)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key6 import */

	lwsl_user("%s: key 6 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key6, sizeof(cose_key6));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_EC2 ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_EC ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_X].len != sizeof(key6_x) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].len != sizeof(key6_y) ||
	    ck->e[LWS_GENCRYPTO_EC_KEYEL_D].len != sizeof(key6_d) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_X].buf, key6_x, sizeof(key6_x)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_Y].buf, key6_y, sizeof(key6_y)) ||
	    memcmp(ck->e[LWS_GENCRYPTO_EC_KEYEL_D].buf, key6_d, sizeof(key6_d)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key7 import */

	lwsl_user("%s: key 7 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key7, sizeof(cose_key7));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_SYMMETRIC ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_OCT ||
	    ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len != sizeof(key7_k) ||
	    memcmp(ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].buf, key7_k, sizeof(key7_k)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key8 import */

	lwsl_user("%s: key 8 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key8, sizeof(cose_key8));
	if (!ck)
		return 1;

	if (ck->kty != LWSCOSE_WKKTV_SYMMETRIC ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_OCT ||
	    ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len != sizeof(key8_k) ||
	    memcmp(ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].buf, key8_k, sizeof(key8_k)))
		goto bail;

	lws_cose_key_destroy(&ck);

	/* key9 import */

	lwsl_user("%s: key 9 import\n", __func__);

	ck = lws_cose_key_import(NULL, NULL, NULL, cose_key9, sizeof(cose_key9));
	if (!ck) {
		lwsl_err("%s: cose9 import fail\n", __func__);
		goto bail;
	}

	if (ck->kty != LWSCOSE_WKKTV_SYMMETRIC ||
	    ck->gencrypto_kty != LWS_GENCRYPTO_KTY_OCT ||
	    ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].len != sizeof(key9_k) ||
	    memcmp(ck->e[LWS_GENCRYPTO_OCT_KEYEL_K].buf, key9_k, sizeof(key9_k))) {
		lwsl_notice("%s: key9 check fails\n", __func__);
		goto bail;
	}

	lws_cose_key_destroy(&ck);

	/* key set 1 */

	lwsl_user("%s: key_set1\n", __func__);
	lws_dll2_owner_clear(&set);
	ck = lws_cose_key_import(&set, NULL, NULL, cose_key_set1, sizeof(cose_key_set1));
	if (!ck)
		return 1;

	lws_cose_key_set_destroy(&set);

	/* generate */

	ck = lws_cose_key_generate(context, LWSCOSE_WKKTV_EC2,
				   (1 << LWSCOSE_WKKO_SIGN) |
				   (1 << LWSCOSE_WKKO_VERIFY) |
				   (1 << LWSCOSE_WKKO_ENCRYPT) |
				   (1 << LWSCOSE_WKKO_DECRYPT),
				   0, "P-256", (const uint8_t *)"the-keyid", 9);
	if (!ck)
		return 1;

	// lws_cose_key_dump(ck);

	lws_cose_key_destroy(&ck);

	return 0;

bail:
	lwsl_err("%s: selftest failed ++++++++++++++++++++\n", __func__);

	return 1;
}
