/***********************************************************************************

    Copyright (C) 2007-2024 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_HELPERS_HEADER
#define LIFEOGRAPH_HELPERS_HEADER


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <sstream>
#include <set>

#include <libintl.h>

#include <gtkmm.h>
#include <gcrypt.h>

// DEFINITIONS FOR LIBGETTEXT
#define _(String)               gettext(String)
#define gettext_noop(String)    String
#define N_(String)              gettext_noop(String)
// END OF LIBGETTEXT DEFINITIONS

namespace LIFEO
{
    static const char   MIME_ENTRY[]        = "application/LoGentry";
    static const char   MIME_THEME[]        = "application/LoGtheme";
    static const char   MIME_CHART[]        = "application/LoGchart";
    static const char   MIME_TABLE[]        = "application/LoGtable";
    static const char   MIME_TEXT_LOG[]     = "application/LoGtext";
    static const char   MIME_TEXT_PLAIN[]   = "text/plain;charset=utf-8";
    static const char   MIME_LOG_FILE[]     = "application/x-lifeographdiary";
} // end of namespace LIFEO

namespace HELPERS
{
// CONSTANTS
static constexpr double MI_TO_KM_RATIO{ 1.609344 };
static constexpr double PI = 3.141592653589793;

// TYPE DEFINITIONS
using date_t_old        = unsigned long;
using Value             = double;

using String            = std::string;
using StringSize        = std::string::size_type;
using VecStrings        = std::vector< String >;
using ListStrings       = std::list< String >;
using SetStrings        = std::set< String >;
using StrStream         = std::stringstream;

using Ustring           = Glib::ustring;
using UstringSize       = Glib::ustring::size_type;
using PairConstStrings  = std::pair< const Ustring, const Ustring >;
using VecUstrings       = std::vector< Ustring >;
using ListUstrings      = std::list< Ustring >;
using SetUstrings       = std::set< Ustring >;

using Wchar             = gunichar;
using R2Pixbuf          = Glib::RefPtr< Gdk::Pixbuf >;
using R2Icon            = Glib::RefPtr< Gtk::IconPaintable >;
using MapPathPixbufs    = std::map< String, R2Pixbuf >;
using Color             = Gdk::RGBA;
using R2ActionGroup     = Glib::RefPtr< Gio::SimpleActionGroup >;
using R2Action          = Glib::RefPtr< Gio::SimpleAction >;

using FuncVoid          = std::function< void( void ) >;
using FuncVoidInt       = std::function< void( int ) >;
using FuncVoidString    = std::function< void( const String& ) >;
using FuncVoidUstring   = std::function< void( const Ustring& ) >;
using SignalVoid        = sigc::signal< void( void ) >;
using SignalVoidInt     = sigc::signal< void( int ) >;
using SignalVoidBool    = sigc::signal< void( bool ) >;
using SignalVoidString  = sigc::signal< void( const String& ) >;
using SignalVoidUstring = sigc::signal< void( const Ustring& ) >;
using SignalBoolUstring = sigc::signal< bool( const Ustring& ) >;
using SignalVoidColor   = sigc::signal< void( const Color& ) >;

enum class ArrayPosition { FIRST, INTERMEDIATE, LAST };

// ENUM HELPERS
template< typename T >
constexpr auto get_underlying( T const v )
    //-> typename std::underlying_type<Enumeration>::type
{
    return static_cast< typename std::underlying_type< T >::type >( v );
}
template< typename T >
constexpr auto set_from_underlying( typename std::underlying_type< T >::type v )
    //-> typename std::underlying_type<Enumeration>::type
{
    return static_cast< T >( v );
}

// ERROR for throw-ing
class Error
{
    public:
                        Error( const Ustring& );
};

// RESULT
enum Result
{
    OK,
    ABORTED,
    SUCCESS,
    FAILURE,
    COULD_NOT_START,
    COULD_NOT_FINISH,
    WRONG_PASSWORD,
    //APPARENTLY_ENCRYTED_FILE,
    //APPARENTLY_PLAIN_FILE,
    INCOMPATIBLE_FILE_OLD,
    INCOMPATIBLE_FILE_NEW,
    CORRUPT_FILE,
    // EMPTY_DATABASE, // not used anymore

    // RESULTS USED BY set_path():
    FILE_NOT_FOUND,
    FILE_NOT_READABLE,
    FILE_NOT_WRITABLE,
    FILE_LOCKED,
    FILE_READ_ONLY,

    // RESULTS USED BY Date
    OUT_OF_RANGE,
    INVALID
};

// CONSTANTS =======================================================================================
struct Constants
{
    static constexpr double INFINITY_PLS = std::numeric_limits< double >::infinity();
    static constexpr double INFINITY_MNS = -std::numeric_limits< double >::infinity();
};

// STRING OPERATIONS ===============================================================================
enum class LetterCase { CASE_SENTENCE, CASE_TITLE, CASE_LOWER, CASE_UPPER };

struct FuncCmpStrings
{
    bool operator()( const Ustring& l, const Ustring& r )
    const { return( l < r ); }
};

class STR
{
    private:
        static void             print_internal( String& ) {}

        template< typename Arg1, typename... Args >
        static void             print_internal( String& str, Arg1 arg1, Args... args )
        {
            str += format( arg1 );
            print_internal( str, args... );
        }
        static const char*      format( const char* str ){ return str; }
        static const String     format( const String& str ){ return str; }
        static const char       format( const char ch ){ return ch; }
        static const String     format( int num ){ return std::to_string( num ); }
        static const String     format( unsigned int num ){ return std::to_string( num ); }
        static const String     format( long num ){ return std::to_string( num ); }
        static const String     format( unsigned long num ){ return std::to_string( num ); }
        static const String     format( long long num ){ return std::to_string( num ); }
        static const String     format( unsigned long long num ){ return std::to_string( num ); }
        static const String     format( double num ){ return format_number( num ); }

    public:
        template< typename... Args >
        static Ustring          compose( Args... args )
        {
            String str;
            print_internal( str, args... );

            return str;
        }

        static String           format_percentage( double );
        static String           format_hex( int );
        static String           format_number( double, int = -1 );
        // to be able to use above in template functions:
        static const String&    format_number( const String& s ) { return s; }
        static String           format_number_no_inf( double num )
        {
        	return( num == Constants::INFINITY_MNS || num == Constants::INFINITY_PLS ?
        			"" : format_number( num ) );
        }
        static String           format_number_roman( int, bool = false );

        static bool             begins_with( const String&, const String& );
        static bool             ends_with( const String&, const String& );
        static bool             ends_with( const String&, const char );
        static char             get_end_char( const String& );
        static bool             get_line( const String&, StringSize&, String& );
        static bool             get_line( const char*, unsigned int&, String& );
        static int              replace( String&, const String&, const String& );
        static String           replace_spaces( const String& );

        static Ustring          lowercase( const Ustring& );
        static Ustring          sentencecase( const Ustring& );
        static Ustring          titlecase( const Ustring& );

        static int64_t          get_i64( const String&, int& i );
        static int32_t          get_i32( const String&, int& i );
        static int32_t          get_i32( const String& ); // guarded std::stol
        static double           get_d( const String& ); // std::stod creates locale issues
        static double           get_d( const String&, int& i ); // version 2

        static int              get_pos_c( const char* str, char c, int def_value = -1 )
        {
            const char* str_sub = strchr( str, c );
            if( !str_sub ) return def_value;
            else           return( int( str_sub - str ) );
        }

        static VecUstrings      make_substr_vector( const Ustring&, int = 1 );
        static int              find_sentence_start_backwards( const String&, StringSize );
        static int              find_word_start_backwards( const String&, StringSize );

        static bool             is_char_space( Wchar c )
        { return( c == ' ' || c == '\t' || c == '\n' ); }
        static bool             is_char_name( Wchar c )
        { return( c == '_' || c == '-' || Glib::Unicode::isalnum( c ) ); }
        static bool             is_char_not_name( Wchar c )
        { return( !is_char_name( c ) ); }

        static Ustring          create_unique_name( const Ustring& name0,
                std::function< bool( const Ustring& ) >&& is_taken )
        {
            Ustring name = name0;
            for( int i = 0; is_taken( name ); i++ )
            {
                name = compose( name0, "_", i );
            }

            return name;
        }
};

// OTHER STRING OPERATIONS =========================================================================
String              get_env_lang();
#ifdef _WIN32
wchar_t*            convert_utf8_to_16( const Ustring& );
Ustring             convert_utf16_to_8( const wchar_t* );
#endif

// SOME BASICS =====================================================================================
template< typename T >
bool is_value_in_range_excl( const T& value, const T& low, const T& high )
{
    return( ( value > low ) && ( value < high ) );
}

// OLD DATE ========================================================================================
// order: 10 bits
// day:    5 bits
// month:  4 bits
// year:  12 bits
// ordinal flag:  1 bit (32nd bit)

class DateOld
{
    public:
        static const unsigned long  NOT_SET              = 0xFFFFFFFF;
        static const unsigned long  DATE_MAX             = 0xFFFFFFFF;

        static const unsigned int   YEAR_MIN             = 1900;
        static const unsigned int   YEAR_MAX             = 2199;
        static const unsigned long  ORDER_MAX            = 1023;
        static const unsigned long  ORDER_1ST_MAX        = 0x3FF00000; // bits 21..30
        static const unsigned long  ORDER_2ND_MAX        =    0xFFC00; // bits 11..20
        static const unsigned int   ORDER_3RD_MAX        =      0x3FF; // bits 01..10

        static const unsigned long  ORDER_1ST_STEP       =   0x100000; // bit 21
        static const unsigned long  ORDER_2ND_STEP       =      0x400; // bit 11

        static const unsigned long  FILTER_DAY           =     0x7C00; // bits 11..15
        static const unsigned long  FILTER_MONTH         =    0x78000; // bits 16..19
        static const unsigned long  FILTER_YEAR          = 0x7FF80000; // bits 20..31
        static const unsigned long  FILTER_YEARMONTH     = FILTER_YEAR|FILTER_MONTH;
        static const unsigned long  FILTER_ORDER_1ST_INV = DATE_MAX ^ ORDER_1ST_MAX;
        static const unsigned long  FILTER_ORDER_2ND_INV = DATE_MAX ^ ORDER_2ND_MAX;
        static const unsigned long  FILTER_ORDER_3RD_INV = DATE_MAX ^ ORDER_3RD_MAX; // FILTER_PURE
        static const unsigned long  FILTER_DAY_INV       = DATE_MAX ^ FILTER_DAY;
        static const unsigned long  FILTER_MONTH_INV     = DATE_MAX ^ FILTER_MONTH;
        static const unsigned long  FILTER_YEAR_INV      = DATE_MAX ^ FILTER_YEAR;

        // hidden elements' sequence numbers are not shown
        static const unsigned long  FLAG_VISIBLE         = 0x40000000; // only for ordinal items
        static const unsigned long  FLAG_ORDINAL         = 0x80000000; // 32nd bit

        static const unsigned long  NUMBERED_MIN         = FLAG_VISIBLE|FLAG_ORDINAL;
        static const unsigned long  FREE_MIN             = FLAG_ORDINAL;

        explicit                    DateOld( date_t_old date )
            :   m_date( date ) {}
        explicit                    DateOld( unsigned int y, unsigned int m, unsigned int d,
                                          unsigned int o = 0 )
            :   m_date( ( y << 19 ) | ( m << 15 ) | ( d << 10 ) | o ) {}
        // ORDINAL C'TOR
        explicit                    DateOld( bool f_n,
                                          unsigned int o1, unsigned int o2, unsigned int o3 )
            : m_date( ( f_n ? NUMBERED_MIN : FREE_MIN ) | ( o1 << 20 ) | ( o2 << 10 ) | o3 ) {}

        // TEMPORAL METHODS
        static unsigned int         get_year( const date_t_old d )
        { return ( ( d & FILTER_YEAR ) >> 19 ); }

        static unsigned int         get_month( const date_t_old d )
        { return( ( d & FILTER_MONTH ) >> 15 ); }

        static unsigned int         get_day( const date_t_old d )
        { return( ( d & FILTER_DAY ) >> 10 ); }

        // ORDINAL METHODS
        static unsigned int         get_order_3rd( const date_t_old d )
        { return( d & ORDER_3RD_MAX ); }
        static unsigned int         get_order_2nd( const date_t_old d )
        { return( ( d & ORDER_2ND_MAX ) >> 10 ); }
        static unsigned int         get_order_1st( const date_t_old d )
        { return( ( d & ORDER_1ST_MAX ) >> 20 ); }

        static bool                 is_1st_level( const date_t_old d )
        { return( ( d & ( ORDER_2ND_MAX | ORDER_3RD_MAX ) ) == 0 ); }
        static bool                 is_2nd_level( const date_t_old d )
        { return( ( d & ORDER_2ND_MAX ) != 0 && ( d & ORDER_3RD_MAX ) == 0 ); }
        static bool                 is_3rd_level( const date_t_old d )
        { return( ( d & ORDER_2ND_MAX ) != 0 && ( d & ORDER_3RD_MAX ) != 0 ); }

        static int                  get_level( const date_t_old d )
        {
            if( not( is_ordinal( d ) ) ) return 4; // temporal
            if( is_1st_level( d ) ) return 1;
            if( is_2nd_level( d ) ) return 2;
            return 3;
        }

        // RELATIONSHIP METHODS
        static date_t_old           get_parent( const date_t_old d )
        {
            switch( get_level( d ) )
            {
                case 1: return( DATE_MAX );
                case 2: return( d & FILTER_ORDER_2ND_INV & FILTER_ORDER_3RD_INV );
                default: return( d & FILTER_ORDER_3RD_INV );
            }
        }

        static bool                 is_set( const date_t_old date )
        { return( date != NOT_SET ); }

        static bool                 is_ordinal( const date_t_old d )
        { return( d & FLAG_ORDINAL ); }

        static bool                 is_hidden( const date_t_old d )
        { return( is_ordinal( d ) && !( d & FLAG_VISIBLE ) ); }

        static bool                 is_sibling( const date_t_old d1, const date_t_old d2 )
        { return( is_ordinal( d1 ) && get_parent( d1 ) == get_parent( d2 ) ); }
        static bool                 is_child_of( const date_t_old dc, const date_t_old dp )
        {
            if( dc == dp ) return false;
            if( dp == DATE_MAX ) return true; // DATE_MAX means all dates
            if( ( dc & FLAG_VISIBLE ) != ( dp & FLAG_VISIBLE ) ) return false;
            if( is_1st_level( dp ) ) return( get_order_1st( dc ) == get_order_1st( dp ) );
            if( is_2nd_level( dp ) ) return( get_order_1st( dc ) == get_order_1st( dp ) &&
                                             get_order_2nd( dc ) == get_order_2nd( dp ) );
            return false;
        }

        // MAKE METHODS
        static date_t_old           make( unsigned int y, unsigned int m, unsigned int d,
                                          unsigned int o = 0 )
        { return( ( y << 19 ) | ( m << 15 ) | ( d << 10 ) | o ); }
        static date_t_old           make_ordinal( bool f_num,
                                                  unsigned int o1,
                                                  unsigned int o2,
                                                  unsigned int o3 = 0 )
        { return( ( f_num ? NUMBERED_MIN : FREE_MIN ) | ( o1 << 20 ) | ( o2 << 10 ) | o3 ); }

        date_t_old                  m_date{ 0 };
};

// DATE & TIME =====================================================================================
using DateV                 = int64_t;
using SignalVoidDate        = sigc::signal< void( DateV ) > ;
using SignalVoidDateUstring = sigc::signal< void( DateV, const Ustring& ) > ;
namespace Date
{
    extern String   s_format_order;
    extern char     s_format_separator;
    extern int      s_week_start_day;

    // CONSTANTS
    static const unsigned int  TIME_MAX = 235959;
    static const unsigned int  YEAR_MIN = 1800;
    static const unsigned int  YEAR_MAX = 2299;

    static constexpr DateV NOT_SET = 0x0;
    static constexpr DateV LATEST  =  229912310235959;
    static constexpr DateV MIN_1ST =              100;
    static constexpr DateV HOR_1ST =            10000;
    static constexpr DateV TIM_SEP =          1000000; // time separator
    static constexpr DateV DAY_1ST =         10000000;
    static constexpr DateV MNT_1ST =       1000000000;
    static constexpr DateV YER_1ST =     100000000000;
    static constexpr DateV DAT_SEP = 1000000000000000; // date separator

    static constexpr DateV BCK_MON = 10000001000000000; // backwards month
    static constexpr DateV FWD_MON = 10000002000000000; // backwards month
    static constexpr DateV THS_MON = 10000003000000000; // this month

    // MAKERS
    inline DateV make_min( int m )      { return( m * MIN_1ST ); }
    inline DateV make_hour( int h )     { return( h * HOR_1ST ); }
    inline DateV make_year( int y )     { return( y * YER_1ST ); }
    inline DateV make_month( int m )    { return( m * MNT_1ST ); }
    inline DateV make_day( int d )      { return( d * DAY_1ST ); }
    inline DateV make( unsigned y, unsigned m, unsigned d )
    { return( make_year( y ) + make_month( m ) + make_day( d ) ); }
    inline DateV make( unsigned y, unsigned M, unsigned d, unsigned h, unsigned m, unsigned s )
    { return( make_year( y ) + make_month( M ) + make_day( d ) +
              make_hour( h ) + make_min( m ) + s ); }
    DateV make( const Ustring& );
    DateV make( const Glib::DateTime& );
    inline DateV make_from_ctime( const tm* ti )
    {
        return make( ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday,
                     ti->tm_hour, ti->tm_min, ti->tm_sec );
    }
    inline DateV make_from_ctime( const time_t t )
    { return make_from_ctime( localtime( &t ) ); }

    // ISOLATORS
    inline DateV isolate_mins( DateV d )  { return( d % HOR_1ST - d % MIN_1ST ); }
    inline DateV isolate_hours( DateV d ) { return( d % TIM_SEP - d % HOR_1ST ); }
    inline DateV isolate_day( DateV d )   { return( d % MNT_1ST - d % DAY_1ST ); }
    inline DateV isolate_month( DateV d ) { return( d % YER_1ST - d % MNT_1ST ); }
    inline DateV isolate_year( DateV d )  { return( d % DAT_SEP - d % YER_1ST ); }
    inline DateV isolate_MD( DateV d )    { return( d % YER_1ST - d % DAY_1ST ); }
    inline DateV isolate_YM( DateV d )    { return( d % DAT_SEP - d % MNT_1ST ); }
    inline DateV isolate_YMD( DateV d )   { return( d % DAT_SEP - d % DAY_1ST ); }

    // GETTERS
    inline unsigned get_secs( DateV d )  { return(   d % MIN_1ST ); } // same as isolating
    inline unsigned get_mins( DateV d )  { return( ( d % HOR_1ST - d % MIN_1ST ) / MIN_1ST ); }
    inline unsigned get_hours( DateV d ) { return( ( d % TIM_SEP - d % HOR_1ST ) / HOR_1ST ); }
    inline unsigned get_time( DateV d )  { return(   d % TIM_SEP ); } // same as isolating
    inline unsigned get_day( DateV d )   { return( ( d % MNT_1ST - d % DAY_1ST ) / DAY_1ST ); }
    inline unsigned get_month( DateV d ) { return( ( d % YER_1ST - d % MNT_1ST ) / MNT_1ST ); }
    inline unsigned get_year( DateV d )  { return( ( d % DAT_SEP - d % YER_1ST ) / YER_1ST ); }

    bool is_leap_year_gen( const unsigned );
    inline bool is_leap_year( const DateV d ) { return is_leap_year_gen( get_year( d ) ); }

    Glib::Date::Month get_month_glib( const DateV d );
    unsigned get_week( const DateV );
    unsigned get_weekday( const DateV );
    unsigned get_yearday( const DateV );
    unsigned get_days_in_month( const DateV );
    inline unsigned get_days_in_year( const DateV d )
    { return( is_leap_year( d ) ? 366 : 365 ); }
    inline unsigned get_days_since_min( const DateV d )
    {
        unsigned    result { get_yearday( d ) };
        auto        year   { get_year( d ) };
        for( auto y = YEAR_MIN; y < year; ++y )
            result += ( is_leap_year_gen( y ) ? 366 : 365 );
        return result;
    }
    inline int64_t get_secs_since_min( const DateV d )
    { return( get_secs( d ) + 60 * get_mins( d )
                            + 3600 * get_hours( d )
                            + 86400 * int64_t( get_days_since_min( d ) ) ); }

    String get_duration_str( const DateV, const DateV );

    inline Glib::Date get_glib( const DateV d )
    { return Glib::Date( get_day( d ), get_month_glib( d ), get_year( d ) ); }

    inline DateV get_today()
    {
        time_t t = time( NULL );
        struct tm* ti = localtime( &t );
        return make( ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday );
    }
    inline DateV get_now()
    {
        time_t t = time( NULL );
        struct tm* ti = localtime( &t );
        return make_from_ctime( ti );
    }

    // SETTERS
    inline bool set_sec( DateV& date, const unsigned s )
    {
        if( s > 59 ) return false;
        date = ( date - ( date % MIN_1ST ) + s );
        return true;
    }
    inline bool set_min( DateV& date, const unsigned m )
    {
        if( m > 59 ) return false;
        date = ( date - ( date % HOR_1ST ) + ( date % MIN_1ST ) + make_min( m ) );
        return true;
    }
    inline bool set_hour( DateV& date, const unsigned h )
    {
        if( h > 23 ) return false;
        date = ( date - ( date % TIM_SEP ) + ( date % HOR_1ST ) + make_hour( h ) );
        return true;
    }

    inline void set_time( DateV& date, unsigned t )
    { if( t <= TIME_MAX ) date = ( date - ( date % TIM_SEP ) + ( t % TIM_SEP ) ); }

    inline void set_day( DateV& date, unsigned d )
    { if( d < 32 ) date = ( date - ( date % MNT_1ST - date % DAY_1ST ) + make_day( d ) ); }

    inline void set_month( DateV& date, unsigned m )
    { if( m < 13 ) date = ( date - ( date % YER_1ST - date % MNT_1ST ) + make_month( m ) ); }

    inline void set_year( DateV& date, unsigned y )
    { if( y >= YEAR_MIN && y <= YEAR_MAX ) date = ( ( date % YER_1ST ) + make_year( y ) ); }

    // CHECKERS
    inline bool is_set( const DateV date ) { return( date != NOT_SET ); }
    inline bool is_valid( const DateV d )
    {
        const auto day    { get_day( d ) };
        const auto month  { get_month( d ) };
        return( day > 0 && month > 0 && month < 13 && day <= get_days_in_month( d ) );
    }
    inline bool is_legit( const DateV d )
    { return( is_valid( d ) && get_year( d ) <= YEAR_MAX && get_year( d ) >= YEAR_MIN ); }

    // SHIFTERS
    void forward_months( DateV&, unsigned int );
    void forward_days( DateV&, unsigned int );
    void backward_months( DateV&, unsigned int );
    void backward_days( DateV&, unsigned int );
    void backward_to_week_start( DateV& );
    inline void backward_to_month_start( DateV& date )
    { date = ( isolate_year( date ) + isolate_month( date ) + make_day( 1 ) ); }
    inline void backward_to_year_start( DateV& date )
    { date = ( isolate_year( date ) + make_month( 1 ) + make_day( 1 ) ); }

    // CONVERSION TO STRING
    Ustring format_string( const DateV, const String&, const char = s_format_separator );
    inline Ustring format_string( const DateV date )
    { return format_string( date, s_format_order, s_format_separator ); }
    Ustring format_string_adv( const DateV, const String& );
    inline String format_string_time( const DateV time )
    {
        char buffer[ 9 ] = "";

        if( time >= 0 )
            g_snprintf( buffer, 9, "%02d:%02d:%02d", get_hours( time ),
                                                     get_mins( time ),
                                                     get_secs( time ) );
        return buffer;
    }
    String  get_format_str_default();
    //Ustring format_string_ctime();
    Ustring get_weekday_str( const DateV );
    Ustring get_month_name( const DateV );
    inline String get_month_str( const DateV d ) { return std::to_string( get_month( d ) ); };
    inline String get_year_str( const DateV d ) { return std::to_string( get_year( d ) ); };
    Ustring get_day_name( int );

    // CALCULATORS
    uint64_t     calculate_secs_between_abs( const DateV, const DateV );
    unsigned int calculate_days_between_abs( const DateV, const DateV );
    inline int   calculate_days_between( const DateV d1, const DateV d2 )
    {
        const auto dist_abs{ calculate_days_between_abs( d1, d2 ) };
        return( d1 > d2 ? -dist_abs : dist_abs );
    }
    unsigned int calculate_weeks_between_abs( const DateV, const DateV );
    int          calculate_months_between( const DateV, const DateV );
    inline unsigned int calculate_months_between_abs( const DateV d1, const DateV d2 )
    { return( d1 > d2 ? -calculate_months_between( d1, d2 )
                      : calculate_months_between( d1, d2 ) ); }

} // end of namespace DateTime

typedef bool( *FuncCompareDates )( const DateV&, const DateV& ) ;

inline bool
compare_dates( const DateV& date_l, const DateV& date_r )
{
    return( date_l > date_r );
}

// CONSOLE MESSAGES ================================================================================
class Console
{
    private:
        static void print( std::ostream& os )
        {
            os << std::endl;
        }

        template< typename Arg1, typename... Args >
        static void print( std::ostream& os, Arg1 arg1, Args... args )
        {
            try{ os << arg1; } // this is necessary for Windows iconv exceptions
            catch( Glib::Error& er )
            { os << "Error printing text:" << er.what() << std::endl; }

            print( os, args... );
        }

#if LIFEOGRAPH_DEBUG_BUILD
        static unsigned DEBUG_LINE_I;
#endif

    template< typename... Args >
    friend void print_error( Args... );

    template< typename... Args >
    friend void print_info( Args... );

    template< typename... Args >
    friend void PRINT_DEBUG( Args... );
};

template< typename... Args >
void print_info( Args... args )
{
    Console::print( std::cout, "INFO: ", args... );
}

template< typename... Args >
void print_error( Args... args )
{
    Console::print( std::cerr, "ERROR: ", args... );
}

#if LIFEOGRAPH_DEBUG_BUILD
    template< typename... Args >
    void PRINT_DEBUG( Args... args )
    {
        Console::print( std::cout, "* DBG:", Console::DEBUG_LINE_I++, " * ", args... );
    }
#else
#define PRINT_DEBUG( ... ) ;
#endif

// COLOR OPERATIONS ================================================================================
struct ColorBasic
{
    ColorBasic() { }
    ColorBasic( double rn, double gn, double bn ) : r( rn ), g( gn ), b( bn ) { }

    double r = -1.0;
    double g = -1.0;
    double b = -1.0;

    // bool is_set() const
    // { return( r >= 0.0 && g >= 0.0 && b >= 0.0 ); }

    void set( double rn, double gn, double bn )
    {
        r = rn;
        g = gn;
        b = bn;
    }

    // void unset()
    // {
    //     r = g = b = -1.0;
    // }
};

inline float
get_color_diff( const Color& c1, const Color c2 )
{
    return( fabs( c2.get_red() - c1.get_red() ) +
            fabs( c2.get_green() - c1.get_green() ) +
            fabs( c2.get_blue() - c1.get_blue() ) );
}

inline Ustring
convert_gdkcolor_to_html( const Color& gdkcolor )
{
    // this function's source of inspiration is Geany
    char buffer[ 8 ];

    g_snprintf( buffer, 8, "#%02X%02X%02X",
            gdkcolor.get_red_u() >> 8,
            gdkcolor.get_green_u() >> 8,
            gdkcolor.get_blue_u() >> 8 );
    return buffer;
}

inline Ustring
convert_gdkrgba_to_html( const Color& gdkcolor )
{
    char buffer[ 14 ];
    g_snprintf( buffer, 14, "#%04X%04X%04X",
            int( gdkcolor.get_red() * 0xFFFF ),
            int( gdkcolor.get_green() * 0xFFFF ),
            int( gdkcolor.get_blue() * 0xFFFF ) );
            //int( gdkcolor.get_alpha() * 0xFFFF ) );
    return buffer;
}

Color               contrast2( const Color&, const Color&, const Color& );
Color               midtone( const Color&, const Color& );
Color               midtone( const Color&, const Color&, double );

// FILE OPERATIONS =================================================================================
#ifdef _WIN32
String              get_exec_path();
#endif

std::ios::pos_type  get_file_size( std::ifstream& );

bool                copy_file( const String&, const String&, bool );
bool                copy_file_suffix( const String&, const String&, int, bool );

bool                is_dir( const String& );
bool                is_dir( const Glib::RefPtr< Gio::File >& );

bool                check_path_exists( const String& );
bool                check_uri_writable( const String& );

String              evaluate_path( const String& );

inline String
get_filename_base( const String& path )
{
    auto file { Gio::File::create_for_commandline_arg( path ) };
    return file->get_basename();
}

#ifdef _WIN32

String
convert_filename_to_win32_locale( const String& );

inline String
convert_locale_to_utf8( const String& str )
{
    return Glib::locale_to_utf8( str );
}

#define PATH( A ) convert_filename_to_win32_locale( A )
#define PATH2( A ) convert_utf8_to_16( A )
#else
#define PATH( A ) ( A )
#define PATH2( A ) ( A )
#define convert_locale_to_utf8( A ) ( A )
#endif

// the following is not used right now but may be helpful in the future
void                get_all_files_in_path( const String&, SetStrings& );

// ENCRYPTION ======================================================================================
class Cipher
{
    public:
        static const int    cCIPHER_ALGORITHM   = GCRY_CIPHER_AES256;
        static const int    cCIPHER_MODE        = GCRY_CIPHER_MODE_CFB;
        static const int    cIV_SIZE            = 16; // = 128 bits
        static const int    cSALT_SIZE          = 16; // = 128 bits
        static const int    cKEY_SIZE           = 32; // = 256 bits
        static const int    cHASH_ALGORITHM     = GCRY_MD_SHA256;

        static bool         init();

        static void         create_iv( unsigned char** );
        static void         expand_key( char const*,
                                        const unsigned char*,
                                        unsigned char** );
        static void         create_new_key( char const*,
                                            unsigned char**,
                                            unsigned char** );
        static void         encrypt_buffer( unsigned char*,
                                            size_t&,
                                            const unsigned char*,
                                            const unsigned char* );
        static void         decrypt_buffer( unsigned char*,
                                            size_t,
                                            const unsigned char*,
                                            const unsigned char* );

    protected:

    private:

};

struct CipherBuffers
{
    CipherBuffers()
    :   buffer( NULL ), salt( NULL ), iv( NULL ), key( NULL ) {}

    unsigned char* buffer;
    unsigned char* salt;
    unsigned char* iv;
    unsigned char* key;

    void clear()
    {
        if( buffer ) delete[] buffer;
        if( salt ) delete[] salt;
        if( iv ) delete[] iv;
        if( key ) delete[] key;
    }
};

// DROP POSITIONS ==================================================================================
enum class DropPosition { NONE, BEFORE, INTO, AFTER };

// ENTRY WIDGET WITH SELF-HANDLED CLEAR ICON =======================================================
class EntryClear : public Gtk::Entry
{
    public:
        EntryClear() : Gtk::Entry() { init(); }

        EntryClear( BaseObjectType* cobject, const Glib::RefPtr< Gtk::Builder >& )
        : Gtk::Entry( cobject ) { init(); }

        virtual ~EntryClear() {}  // does not seem to work...
                                  // ...probably Gtk::Entry's destructor is not virtual

        bool is_empty() const { return get_text().empty(); }

        void clear() { set_text ( "" ); }

    protected:
        void                init();

        void                handle_icon_press( Gtk::Entry::IconPosition );
        void                handle_icon_release( Gtk::Entry::IconPosition );
        virtual void        on_changed() override;
        virtual void        on_key_release_event( guint, guint, Gdk::ModifierType );
        bool                m_F_pressed{ false };
};

// ADVANCED MESSAGE DIALOG LIKE IN THE GOOD OLD DAYS ===============================================
class DialogMessage : public Gtk::Window
{
    public:
        static DialogMessage* init( Gtk::Window*, const Ustring&,
                                                  const Ustring& = _( "_Cancel" ) );

        DialogMessage*        add_extra_info( const Ustring&, bool = false );
        DialogMessage*        add_extra_widget( Gtk::Widget* );
        Gtk::Button*          add_button_b( const Ustring&, const FuncVoid& );
        DialogMessage*        add_cancel_handler( const FuncVoid& );
        DialogMessage*        add_button( const Ustring& l, const FuncVoid& h )
        { add_button_b( l, h ); return this; }
        DialogMessage*        add_button( const Ustring& l, const FuncVoid& h, const Ustring& c )
        { add_button_b( l, h )->add_css_class( c ); return this; }

        Gtk::Box*             m_Bx_content;
        Gtk::Box*             m_Bx_buttons;
        Gtk::Label*           m_L_msg;
        Gtk::Button*          m_B_cancel;

    protected:
        DialogMessage( const Ustring&, const Ustring& );

        static std::vector< DialogMessage* > s_dialog_store;
};

// DIALOGEVENT =====================================================================================
class DialogEvent : public Gtk::Dialog
{
    public:
                            DialogEvent( const Glib::ustring& );
                            DialogEvent( BaseObjectType*,
                                         const Glib::RefPtr< Gtk::Builder >& );

    protected:
        void                handle_logout();

};

// FILE CHOOSER BUTTON REPLACEMENT =================================================================
class FileChooserButton : public Gtk::Button
{
    public:
                            FileChooserButton( const Ustring& name ) : Gtk::Button( name )
                            { init(); }
                            FileChooserButton( BaseObjectType* obj,
                                               const Glib::RefPtr< Gtk::Builder >& builder )
                            : Gtk::Button( obj )
                            { init(); }

        void                set_icon( const String& );

        //String              get_path() const;
        void                set_uri( const String& );

        void                set_info_text( const String& );

        void                add_file_filters( const Ustring&,
                                              const Ustring&,
                                              const Ustring& );
        void                add_image_file_filters();
        void                add_diary_file_filters();
        void                set_pick_folder_mode()
        { m_F_folder_mode = true; }

        static void         add_file_filters( Glib::RefPtr< Gtk::FileDialog >&,
                                              const Ustring&,
                                              const Ustring&,
                                              const Ustring& );
        static void         add_image_file_filters( Glib::RefPtr< Gtk::FileDialog >& );
        static void         add_diary_file_filters( Glib::RefPtr< Gtk::FileDialog >& );

        SignalVoidString    m_signal_file_set;

    protected:
        void                init();
        void                handle_click();

        Gtk::Label*         m_label;
        Gtk::Image*         m_icon;
        Glib::RefPtr< Gtk::FileDialog >
                            m_dlg;
        bool                m_F_folder_mode { false };
        Ustring             m_empty_text;
        String              m_uri;
};

// FRAME FOR PRINTING ==============================================================================
//Gtk::Frame* create_frame( const Glib::ustring&, Gtk::Widget& );

// TREEVIEW ========================================================================================
// scroll_to_row( Path& ) does not work for some reason. So, implemented our own
bool is_treepath_less( const Gtk::TreePath&, const Gtk::TreePath& );
bool is_treepath_more( const Gtk::TreePath&, const Gtk::TreePath& );

// OTHER GTK HELPERS ===============================================================================
static constexpr int CTRL_SHIFT_MASK
{
    ( static_cast< int >( Gdk::ModifierType::CONTROL_MASK ) |
      static_cast< int >( Gdk::ModifierType::SHIFT_MASK ) )
};
static constexpr int ALT_SHIFT_MASK
{
    ( static_cast< int >( Gdk::ModifierType::ALT_MASK ) |
      static_cast< int >( Gdk::ModifierType::SHIFT_MASK ) )
};
static constexpr int CTRL_ALT_SHIFT_MASK
{
    ( static_cast< int >( Gdk::ModifierType::CONTROL_MASK ) |
      static_cast< int >( Gdk::ModifierType::ALT_MASK ) |
#ifdef __APPLE__
      static_cast< int >( Gdk::ModifierType::META_MASK ) |
#endif
      static_cast< int >( Gdk::ModifierType::SHIFT_MASK ) )
};
static constexpr int CTRL_META_MASK
{
    ( static_cast< int >( Gdk::ModifierType::CONTROL_MASK ) |
      static_cast< int >( Gdk::ModifierType::META_MASK ) )
};
static constexpr int CTRL_META_SHIFT_MASK
{
    ( static_cast< int >( Gdk::ModifierType::CONTROL_MASK ) |
      static_cast< int >( Gdk::ModifierType::META_MASK ) |
      static_cast< int >( Gdk::ModifierType::SHIFT_MASK ) )
};

// Gtk::MenuItem*      create_menuitem_markup( const Ustring&,
//                                             const Glib::SignalProxy0< void >::SlotType& );

void                remove_all_children_from_Bx( Gtk::Box* );
void                remove_all_children_from_LBx( Gtk::ListBox* );
void                remove_all_children_from_FBx( Gtk::FlowBox* );

void                flush_gtk_event_queue();

int                 get_LB_item_count( Gtk::ListBox* LB );
void                scroll_to_selected_LB_row( Gtk::ListBox* LB );
void                select_LB_item_prev( Gtk::ListBox* );
void                select_LB_item_next( Gtk::ListBox* );

R2Pixbuf            lookup_default_icon_pixbuf( const Ustring& name, int size );

inline R2Icon
lookup_default_icon( const Ustring& name, int size )
{
    return Gtk::IconTheme::get_for_display( Gdk::Display::get_default() )->lookup_icon( name,
                                                                                        size );
}

} // end of namespace HELPERS

#endif

