#!/usr/bin/env python3

# Generates docs/download.html
# Run:
# ./generate-download.py > docs/download.html

import json
import sys

if len(sys.argv) > 1:
  debian_text = '''<p>The Debian version <tt>%s</tt>, which this documentation
 was built for, corresponds to the subsequently mentioned release (plus
 patches as applicable). Note that, in the Debian package, most of the
 following links will not work for hopefully obvious reasons.</p>
  '''.strip() % sys.argv[1]
else:
  debian_text = ''

with open('releases.json', 'rt',
          encoding='UTF-8', errors='strict', newline=None) as infile:
  releases = json.load(infile)

def file_links(release):
  v = release['version']
  if 'distsubdir' in release and release['distsubdir'] == False:
    sd = ''
  else:
    sd = 'dist/'
  return ['<a href="%(v)s/%(sd)s%(f)s">%(f)s</a>' % {
    'f': f, 'sd': sd, 'v': v} for f in release['files']]

# Validation of releases.json
for idx, release in enumerate(releases):
  if idx == 0: continue
  assert 'version' in release, 'Release missing version: %s' % release
  assert 'files' in release, 'Release missing files: %s' % release
  assert release['version'] < releases[idx - 1]['version'], (
      'Releases should be in reverse chronological order in releases.json')

current_html = '<p>' + ('</p><p>'.join(file_links(releases[0]))) + '</p>'
current_link = 'https://github.com/danvk/dygraphs/releases/tag/v' + releases[0]['version']

previous_lis = []
for release in releases[1:]:
  previous_lis.append('<li>%(v)s: %(files)s (<a href="%(v)s/">%(v)s docs</a>)' % {
      'v': release['version'],
      'files': ', '.join(file_links(release))
    })

print('''<!--#set var="pagetitle" value="downloads list" -->
<!--#include virtual="header.html" -->

<!--
  DO NOT EDIT THIS FILE!

  This file is generated by generate-download.py.
-->

%(debian_text)s

<p>The current version of dygraphs is <b>%(version)s</b>.
 Most users will want to download minified files for this version:</p>

<div id="current-release" class="panel">
%(current_html)s
</div>

<p>For recent releases, <a href="%(current_link)s">GitHub
 Releases</a> hosts both source tarball
 (<tt>dygraphs_%(version)s.orig.tar.gz</tt>, basically
 the git repository packaged up) and binary release
 (<tt>dygraphs-%(version)s.tgz</tt>, identical with
 what was uploaded to NPM). The latter contains the precompiled
 CSS and JS files in both readable and minified forms as well as
 a copy of the documentation site, ideal for end users.
 The former only contains the source code needed to build all of it;
 distro packagers will want that file.</p>

<p>There's a hosted version of dygraphs on <a
 href="https://unpkg.com/dygraphs/">UNPKG</a>:</p>

<pre>&lt;script type="text/javascript" src="https://unpkg.com/dygraphs@%(version)s/dist/dygraph.min.js"&gt;&lt;/script&gt;
&lt;link rel="stylesheet" type="text/css" href="https://unpkg.com/dygraphs@%(version)s/dist/dygraph.min.css" /&gt;
</pre>

<p>There's a hosted version of dygraphs on <a
 href="https://cdnjs.com/libraries/dygraph">cdnjs.com</a>:</p>

<pre>&lt;script type="text/javascript" src="//cdnjs.cloudflare.com/ajax/libs/dygraph/%(version)s/dygraph.min.js"&gt;&lt;/script&gt;
&lt;link rel="stylesheet" type="text/css" href="//cdnjs.cloudflare.com/ajax/libs/dygraph/%(version)s/dygraph.min.css" /&gt;
</pre>

<p>But note that use of CDNs violates the EU-GDPR. Besides the tarballs
 (see above) you can also install dygraphs locally into your project
 instead using <a href="https://www.npmjs.org/package/dygraphs">NPM</a>:</p>

<pre>$ npm install dygraphs
  # dygraphs is now in node_modules/dygraphs/dist/dygraph.{css,js} for
  # the browser, and node_modules/dygraphs/index{,.es5}.js for nodejs</pre>

<p>Most distributions include a source map to facilitate debugging.</p>

<p>To generate your own minified JS, install the prerequisites…</p><ul>
<li><tt>mksh</tt></li>
<li><tt>pax</tt></li>
<li><tt>python3</tt></li>
</ul><p>… and run:</p>

<pre>git clone https://github.com/danvk/dygraphs.git
cd dygraphs
npm install
npm run build-jsonly
</pre>

<p>This will create a <tt>dygraph.min.css</tt> and a <tt>dygraph.min.js</tt>
 file in the <code>dist</code> directory.</p>

<p>You may also download files for previously-released versions:</p>

<ul>
%(previous_lis)s
</ul>

<p>See <a href="https://dygraphs.com/versions.html">Version History</a>
 for more information on each release.</p>

<!--#include virtual="footer.html" -->''' % {
    'version': releases[0]['version'],
    'current_html': current_html,
    'current_link': current_link,
    'debian_text': debian_text,
    'previous_lis': '\n'.join(previous_lis)
    })
