from uuid import UUID

from bleak.backends.device import BLEDevice
from bluetooth_data_tools import monotonic_time_coarse
from habluetooth import BluetoothServiceInfo, BluetoothServiceInfoBleak

from bluetooth_sensor_state_data import BluetoothData


def make_bluetooth_service_info(  # noqa: PLR0913
    name: str,
    manufacturer_data: dict[int, bytes],
    service_uuids: list[str],
    address: str,
    rssi: int,
    service_data: dict[UUID, bytes],
    source: str,
    tx_power: int = 0,
    raw: bytes | None = None,
) -> BluetoothServiceInfoBleak:
    return BluetoothServiceInfoBleak(
        name=name,
        manufacturer_data=manufacturer_data,
        service_uuids=service_uuids,
        address=address,
        rssi=rssi,
        service_data=service_data,
        source=source,
        device=BLEDevice(
            name=name,
            address=address,
            details={},
            rssi=rssi,
        ),
        time=monotonic_time_coarse(),
        advertisement=None,
        connectable=True,
        tx_power=tx_power,
        raw=raw,
    )


SERVICE_INFO_1 = BluetoothServiceInfo(
    name="SensorPush HTP.xw 241",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        6400: b"\xe9\x95\xbbK\xb5",
        17667: b"\xad\x00\x01\x00\x00",
        9984: b"\xe7ICo\xb5",
        1536: b"\x80\x8a\xf58\xb5",
        20992: b"\xa2L\x87^\xb5",
        31744: b"\x8bl\x0fK\xb5",
        30208: b"\x03\xd1\x1c\xc9\xb5",
        23040: b"\xe7\x82\xd6Y\xb5",
        41728: b"(rxT\xb5",
        54528: b"\xb2\x81\xb9\xbf\xb5",
        768: b"\xfc\xba^F\xb5",
        42752: b"T\x7f\xc8\xc9\xb5",
        15360: b"\xb0SLu\xb5",
        5120: b":V.a\xb5",
        10752: b"z\xa6'W\xb5",
        10240: b",\xff\xceY\xb5",
        12288: b"$\xe3\x00|\xb5",
        59136: b".2\x8eb\xb5",
        32768: b"i\xbd3_\xb5",
        14848: b"\xe6\x9a\xbf\xc3\xb5",
        54272: b"\xf3\xce\x10\xc1\xb5",
        45056: b"\x12c\xd4Y\xb5",
        18432: b"\xce\xcfcJ\xb5",
        14592: b"o\x02\xd2[\xb5",
        48384: b"\xdd\x02\xab?\xb5",
        51456: b"\xda~\xbaK\xb5",
        2048: b"\xe5\xab\xa6x\xb5",
        10496: b"I\xe9\x13\xc3\xb5",
        56320: b"M\x154_\xb5",
        23296: b"\x1a|\xc2\xc5\xb5",
        46592: b"[\xce\x0cI\xb5",
        50688: b"\xc0\t{V\xb5",
        42240: b"\xb7LY\xb8\xb5",
        8448: b"H\xccc\x85\xb5",
        62208: b"|X&W\xb5",
        31488: b"\xdf\x1c-\x9a\xb5",
        61440: b"\x97\xa1\xb6\xbd\xb5",
        56064: b">\xf2\x04\xb9\xb5",
        3584: b"!\xf4]F\xb5",
        8192: b"~\xc3c\x85\xb5",
        50176: b"\x99\xc8\xaeC\xb5",
        12032: b"\x0fn\xad\xb7\xb5",
        25088: b"\xe8\xe7iN\xb5",
        7424: b"w?_\xbc\xb5",
        22016: b"u\x8c\nG\xb5",
        23552: b"\x01f~X\xb5",
        16128: b"\xe6\x19\xfc\xb2\xb5",
        17408: b"\x1f*\xc6S\xb5",
        54016: b"'\xde\xaa\xb5\xb5",
        52992: b"\xc2\xd9\xaa\xb5\xb5",
        42496: b"\xb4\xf7\xbaK\xb5",
        36096: b"i\x8aJ\xae\xb5",
        41216: b"\xda\xfc\xd9a\xb5",
        33536: b"\xe7\x05\xc2O\xb5",
        65280: b"\x8a\x97vT\xb5",
        22784: b"\xe8\xb7\x98\xa9\xb5",
        62464: b"\xbd\x08\xff>\xb5",
        57088: b"\xcc\xdctV\xb5",
        60160: b"9\xb6\x7f^\xb5",
        512: b"\x9c\x1e\tG\xb5",
        39424: b"~\x1d~X\xb5",
        17920: b".\x1d\xb0C\xb5",
        11008: b"\x1e*\xb2\x80\xb5",
        61184: b"O\x9a\x92\xa5\xb5",
        4352: b"h\x98x\x93\xb5",
        22272: b"\n\x83V@\xb5",
        28160: b"\x19\x10\xa7=\xb5",
        16896: b"\xc7\xcb\xb8I\xb5",
        46080: b"7c\xc9U\xb5",
        27136: b"\x02\xb4~X\xb5",
        29184: b"D\x1b\x83`\xb5",
        20224: b"+\x89\xe1\xa0\xb5",
        34560: b'"l\x95\xa7\xb5',
        63744: b"\x03\xd8\xabA\xb5",
        62720: b"\xc3B\x85\\\xb5",
        30464: b"k#\xceU\xb5",
        65024: b"\xb6T>\xa6\xb5",
        47360: b"\xa8\xa9\x95\xa7\xb5",
        21504: b"\x17\xc5\x98\xa9\xb5",
        7168: b"\xe0\xa7\x85b\xb5",
        38400: b"Y\xe4~X\xb5",
        40704: b"3_\t\x82\xb5",
        27392: b"E6\xac|\xb5",
        34304: b"\xa1P\xc0M\xb5",
        30720: b"xFOw\xb5",
        45568: b"j\xbeR>\xb5",
        26368: b"\xf7Z\x0fK\xb5",
        36864: b"\x84\xee\xdb]\xb5",
        63232: b"\xbajcJ\xb5",
        35840: b"\xfc\xff]\x81\xb5",
        32256: b"u\xc3\x04C\xb5",
        43264: b"B#?\xa6\xb5",
        40960: b"B!\xb8\x84\xb5",
        26880: b"\xf4)6\xa0\xb5",
        51712: b"\xbc\xec,[\xb5",
        8704: b"\x91O4e\xb5",
        54784: b"\xc08\x1dU\xb5",
        34816: b"#\xcd\x17Q\xb5",
        256: b"`)~\x97\xb5",
        24576: b"1\xa1\x17Q\xb5",
        21248: b"\x00\x7f\x84\x9b\xb5",
        19712: b"\x90$-\x9a\xb5",
        28416: b"%\xed\xe1\xa0\xb5",
        52736: b"\x83\xd7aH\xb5",
        55040: b"l\x96xV\xb5",
        53760: b"\x90\x91\xb7I\xb5",
        26112: b"\xfa\xe0\xfdy\xb5",
        38144: b"O\x10~X\xb5",
        53504: b"r\x19\xd5Y\xb5",
        61952: b"\xc7\x93)Y\xb5",
        51200: b"\xc3%$\x94\xb5",
        58112: b"\xdcQ\xb5G\xb5",
        57600: b"BB*\x98\xb5",
        37632: b"D\xcf\x1dU\xb5",
        1792: b"\x80B\x15\x8a\xb5",
        56832: b"\x05\xa9l\x8b\xb5",
        25344: b"!\x00\xc7\x8e\xb5",
        44288: b"\xc1\x06\xc4\x8c\xb5",
        6912: b"yX\x15\x8a\xb5",
        43520: b"\x18\x7f\x06\x80\xb5",
        33792: b"@1\x0cI\xb5",
        29952: b"]a\xaa?\xb5",
        40448: b"\xa4\x05\xdaa\xb5",
        32512: b"\xdd\xc9\x8bf\xb5",
        37376: b")\xd6\xeeo\xb5",
        7680: b"Ca\x15\x8a\xb5",
        12800: b"\x8b\xcff\x87\xb5",
        36352: b"\xd6\xfb\xbd\x88\xb5",
        56576: b"\t+Lu\xb5",
        41984: b"x\x18\xb8\x84\xb5",
        13056: b"\xd5\xa2\xd0U\xb5",
        44800: b"\xaa\xdaZ\x7f\xb5",
        60928: b"&*\xaf~\xb5",
        13824: b"bx\xafC\xb5",
        49152: b"c\xd1\xaeC\xb5",
        29696: b"\t\xb8\xb5G\xb5",
        20736: b"\xc5/\xb8I\xb5",
        16384: b"l\xd1\x1fS\xb5",
        30976: b"\xe5\x1a\xb8I\xb5",
        15872: b"\x1bCU{\xb5",
        31232: b"\xaeo\xa7=\xb5",
        29440: b"^\xc6\xb1E\xb5",
        17664: b"\xb8\xf4\x00|\xb5",
        14080: b"\x10\xed\xfdy\xb5",
        20480: b"\x05g\x11K\xb5",
        51968: b"\x02Y[\x7f\xb5",
        48128: b"\xb1*\tG\xb5",
        60672: b"\xbc\xba\xfe>\xb5",
        11776: b'"\xac\x91j\xb5',
        60416: b"\x12LYB\xb5",
        14336: b"Pm*Y\xb5",
        55808: b"\xdcR4e\xb5",
        50944: b"\xa9\xd8]F\xb5",
        7936: b"u8\xc1O\xb5",
        17152: b"\xc3\xc8\x15O\xb5",
        24320: b'\xdd\xf7"Y\xb5',
        37120: b">\x99.a\xb5",
        15104: b"c\xc7.a\xb5",
        2304: b"\x9d\x8a\xd7_\xb5",
        48896: b"\xa7\xd6L:\xb5",
        2560: b"\x1br}X\xb5",
        55296: b"\xe8\x104_\xb5",
        19968: b"\xd0\x85\x0fK\xb5",
        37888: b"w\xaabJ\xb5",
        27648: b"j\xb6nR\xb5",
        46848: b"\xd9\xb1}\\\xb5",
        64768: b"q\x17\xc6S\xb5",
        64512: b"{n&[\xb5",
        4864: b"H\xe1lP\xb5",
        34048: b"\xd6\xdbhN\xb5",
        5376: b"\xb4\xd3\x18O\xb5",
        55552: b"\n\xf4eL\xb5",
        24064: b"\xd7\xf2nR\xb5",
        53248: b"j\xdc W\xb5",
        49408: b"D\x06\x18Q\xb5",
        11520: b",\x82\xb7G\xb5",
        47616: b"G\xbb\xbaK\xb5",
        18944: b"\xf5miN\xb5",
        64256: b"\x1eY\xd9[\xb5",
        18176: b"i|ZD\xb5",
        50432: b"\xa8\x8a]F\xb5",
        47104: b"\xdd\xb7\xc4Q\xb5",
        58624: b"W\x02dJ\xb5",
        15616: b"\xe3\x13gL\xb5",
        52480: b"p{\xc4Q\xb5",
        48640: b"\xd85oP\xb5",
        47872: b"\xbb\x1bYB\xb5",
        59904: b"\x8c_\xfe>\xb5",
        52224: b"c\xd4R>\xb5",
        40192: b"Y\x95\x01A\xb5",
        39680: b"\x03\xea\xfb<\xb5",
        3072: b"\x8c}V@\xb5",
        18688: b"\xde*\xf9:\xb5",
        21760: b"o\x87V@\xb5",
        9216: b"\xf1\xd0_F\xb5",
        35584: b"'W\xc6Q\xb5",
        19456: b"%\x01lN\xb5",
        58880: b"\xf4\xe9uT\xb5",
        59392: b"D\x90\xc7Q\xb5",
        64000: b"\xc3\xac\xcdU\xb5",
        28928: b"x\xe0\xc7Q\xb5",
        12544: b"\x03\xca|X\xb5",
        44032: b"\x85#\xc5O\xb5",
        9728: b"\x0cP}X\xb5",
        4096: b"\x8b\xcc\xd4Y\xb5",
        4608: b"*\xcbwT\xb5",
        25856: b"~\x97zV\xb5",
        5888: b"\x1f\xc4)Y\xb5",
        27904: b"n\xf1\x87^\xb5",
        36608: b"\x05\x074_\xb5",
        35328: b"\xa3\xb7\xd6Y\xb5",
    },
    service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
)

SERVICE_INFO_2 = BluetoothServiceInfo(
    name="SensorPush HTP.xw 241",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        6400: b"\xe9\x95\xbbK\xb5",
        17667: b"\xad\x00\x01\x00\x00",
        9984: b"\xe7ICo\xb5",
        1536: b"\x80\x8a\xf58\xb5",
        20992: b"\xa2L\x87^\xb5",
        31744: b"\x8bl\x0fK\xb5",
        30208: b"\x03\xd1\x1c\xc9\xb5",
        23040: b"\xe7\x82\xd6Y\xb5",
        41728: b"(rxT\xb5",
        54528: b"\xb2\x81\xb9\xbf\xb5",
        768: b"\xfc\xba^F\xb5",
        42752: b"T\x7f\xc8\xc9\xb5",
        15360: b"\xb0SLu\xb5",
        5120: b":V.a\xb5",
        10752: b"z\xa6'W\xb5",
        10240: b",\xff\xceY\xb5",
        12288: b"$\xe3\x00|\xb5",
        59136: b".2\x8eb\xb5",
        32768: b"i\xbd3_\xb5",
        14848: b"\xe6\x9a\xbf\xc3\xb5",
        54272: b"\xf3\xce\x10\xc1\xb5",
        45056: b"\x12c\xd4Y\xb5",
        18432: b"\xce\xcfcJ\xb5",
        14592: b"o\x02\xd2[\xb5",
        48384: b"\xdd\x02\xab?\xb5",
        51456: b"\xda~\xbaK\xb5",
        2048: b"\xe5\xab\xa6x\xb5",
        10496: b"I\xe9\x13\xc3\xb5",
        56320: b"M\x154_\xb5",
        23296: b"\x1a|\xc2\xc5\xb5",
        46592: b"[\xce\x0cI\xb5",
        50688: b"\xc0\t{V\xb5",
        42240: b"\xb7LY\xb8\xb5",
        8448: b"H\xccc\x85\xb5",
        62208: b"|X&W\xb5",
        31488: b"\xdf\x1c-\x9a\xb5",
        61440: b"\x97\xa1\xb6\xbd\xb5",
        56064: b">\xf2\x04\xb9\xb5",
        3584: b"!\xf4]F\xb5",
        8192: b"~\xc3c\x85\xb5",
        50176: b"\x99\xc8\xaeC\xb5",
        12032: b"\x0fn\xad\xb7\xb5",
        25088: b"\xe8\xe7iN\xb5",
        7424: b"w?_\xbc\xb5",
        22016: b"u\x8c\nG\xb5",
        23552: b"\x01f~X\xb5",
        16128: b"\xe6\x19\xfc\xb2\xb5",
        17408: b"\x1f*\xc6S\xb5",
        54016: b"'\xde\xaa\xb5\xb5",
        52992: b"\xc2\xd9\xaa\xb5\xb5",
        42496: b"\xb4\xf7\xbaK\xb5",
        36096: b"i\x8aJ\xae\xb5",
        41216: b"\xda\xfc\xd9a\xb5",
        33536: b"\xe7\x05\xc2O\xb5",
        65280: b"\x8a\x97vT\xb5",
        22784: b"\xe8\xb7\x98\xa9\xb5",
        62464: b"\xbd\x08\xff>\xb5",
        57088: b"\xcc\xdctV\xb5",
        60160: b"9\xb6\x7f^\xb5",
        512: b"\x9c\x1e\tG\xb5",
        39424: b"~\x1d~X\xb5",
        17920: b".\x1d\xb0C\xb5",
        11008: b"\x1e*\xb2\x80\xb5",
        61184: b"O\x9a\x92\xa5\xb5",
        4352: b"h\x98x\x93\xb5",
        22272: b"\n\x83V@\xb5",
        28160: b"\x19\x10\xa7=\xb5",
        16896: b"\xc7\xcb\xb8I\xb5",
        46080: b"7c\xc9U\xb5",
        27136: b"\x02\xb4~X\xb5",
        29184: b"D\x1b\x83`\xb5",
        20224: b"+\x89\xe1\xa0\xb5",
        34560: b'"l\x95\xa7\xb5',
        63744: b"\x03\xd8\xabA\xb5",
        62720: b"\xc3B\x85\\\xb5",
        30464: b"k#\xceU\xb5",
        65024: b"\xb6T>\xa6\xb5",
        47360: b"\xa8\xa9\x95\xa7\xb5",
        21504: b"\x17\xc5\x98\xa9\xb5",
        7168: b"\xe0\xa7\x85b\xb5",
        38400: b"Y\xe4~X\xb5",
        40704: b"3_\t\x82\xb5",
        27392: b"E6\xac|\xb5",
        34304: b"\xa1P\xc0M\xb5",
        30720: b"xFOw\xb5",
        45568: b"j\xbeR>\xb5",
        26368: b"\xf7Z\x0fK\xb5",
        36864: b"\x84\xee\xdb]\xb5",
        63232: b"\xbajcJ\xb5",
        35840: b"\xfc\xff]\x81\xb5",
        32256: b"u\xc3\x04C\xb5",
        43264: b'\xf90"S\xb5',
        40960: b"B!\xb8\x84\xb5",
        26880: b"\xf4)6\xa0\xb5",
        51712: b"\xbc\xec,[\xb5",
        8704: b"\x91O4e\xb5",
        54784: b"\xc08\x1dU\xb5",
        34816: b"#\xcd\x17Q\xb5",
        256: b"`)~\x97\xb5",
        24576: b"1\xa1\x17Q\xb5",
        21248: b"\x00\x7f\x84\x9b\xb5",
        19712: b"\x90$-\x9a\xb5",
        28416: b"%\xed\xe1\xa0\xb5",
        52736: b"\x83\xd7aH\xb5",
        55040: b"l\x96xV\xb5",
        53760: b"\x90\x91\xb7I\xb5",
        26112: b"\xfa\xe0\xfdy\xb5",
        38144: b"O\x10~X\xb5",
        53504: b"r\x19\xd5Y\xb5",
        61952: b"\xc7\x93)Y\xb5",
        51200: b"\xc3%$\x94\xb5",
        58112: b"\xdcQ\xb5G\xb5",
        57600: b"BB*\x98\xb5",
        37632: b"D\xcf\x1dU\xb5",
        1792: b"\x80B\x15\x8a\xb5",
        56832: b"\x05\xa9l\x8b\xb5",
        25344: b"!\x00\xc7\x8e\xb5",
        44288: b"\xc1\x06\xc4\x8c\xb5",
        6912: b"yX\x15\x8a\xb5",
        43520: b"\x18\x7f\x06\x80\xb5",
        33792: b"@1\x0cI\xb5",
        29952: b"]a\xaa?\xb5",
        40448: b"\xa4\x05\xdaa\xb5",
        32512: b"\xdd\xc9\x8bf\xb5",
        37376: b")\xd6\xeeo\xb5",
        7680: b"Ca\x15\x8a\xb5",
        12800: b"\x8b\xcff\x87\xb5",
        36352: b"\xd6\xfb\xbd\x88\xb5",
        56576: b"\t+Lu\xb5",
        41984: b"x\x18\xb8\x84\xb5",
        13056: b"\xd5\xa2\xd0U\xb5",
        44800: b"\xaa\xdaZ\x7f\xb5",
        60928: b"&*\xaf~\xb5",
        13824: b"bx\xafC\xb5",
        49152: b"c\xd1\xaeC\xb5",
        29696: b"\t\xb8\xb5G\xb5",
        20736: b"\xc5/\xb8I\xb5",
        16384: b"l\xd1\x1fS\xb5",
        30976: b"\xe5\x1a\xb8I\xb5",
        15872: b"\x1bCU{\xb5",
        31232: b"\xaeo\xa7=\xb5",
        29440: b"^\xc6\xb1E\xb5",
        17664: b"\xb8\xf4\x00|\xb5",
        14080: b"\x10\xed\xfdy\xb5",
        20480: b"\x05g\x11K\xb5",
        51968: b"\x02Y[\x7f\xb5",
        48128: b"\xb1*\tG\xb5",
        60672: b"\xbc\xba\xfe>\xb5",
        11776: b'"\xac\x91j\xb5',
        60416: b"\x12LYB\xb5",
        14336: b"Pm*Y\xb5",
        55808: b"\xdcR4e\xb5",
        50944: b"\xa9\xd8]F\xb5",
        7936: b"u8\xc1O\xb5",
        17152: b"\xc3\xc8\x15O\xb5",
        24320: b'\xdd\xf7"Y\xb5',
        37120: b">\x99.a\xb5",
        15104: b"c\xc7.a\xb5",
        2304: b"\x9d\x8a\xd7_\xb5",
        48896: b"\xa7\xd6L:\xb5",
        2560: b"\x1br}X\xb5",
        55296: b"\xe8\x104_\xb5",
        19968: b"\xd0\x85\x0fK\xb5",
        37888: b"w\xaabJ\xb5",
        27648: b"j\xb6nR\xb5",
        46848: b"\xd9\xb1}\\\xb5",
        64768: b"q\x17\xc6S\xb5",
        64512: b"{n&[\xb5",
        4864: b"H\xe1lP\xb5",
        34048: b"\xd6\xdbhN\xb5",
        5376: b"\xb4\xd3\x18O\xb5",
        55552: b"\n\xf4eL\xb5",
        24064: b"\xd7\xf2nR\xb5",
        53248: b"j\xdc W\xb5",
        49408: b"D\x06\x18Q\xb5",
        11520: b",\x82\xb7G\xb5",
        47616: b"G\xbb\xbaK\xb5",
        18944: b"\xf5miN\xb5",
        64256: b"\x1eY\xd9[\xb5",
        18176: b"i|ZD\xb5",
        50432: b"\xa8\x8a]F\xb5",
        47104: b"\xdd\xb7\xc4Q\xb5",
        58624: b"W\x02dJ\xb5",
        15616: b"\xe3\x13gL\xb5",
        52480: b"p{\xc4Q\xb5",
        48640: b"\xd85oP\xb5",
        47872: b"\xbb\x1bYB\xb5",
        59904: b"\x8c_\xfe>\xb5",
        52224: b"c\xd4R>\xb5",
        40192: b"Y\x95\x01A\xb5",
        39680: b"\x03\xea\xfb<\xb5",
        3072: b"\x8c}V@\xb5",
        18688: b"\xde*\xf9:\xb5",
        21760: b"o\x87V@\xb5",
        9216: b"\xf1\xd0_F\xb5",
        35584: b"'W\xc6Q\xb5",
        19456: b"%\x01lN\xb5",
        58880: b"\xf4\xe9uT\xb5",
        59392: b"D\x90\xc7Q\xb5",
        64000: b"\xc3\xac\xcdU\xb5",
        28928: b"x\xe0\xc7Q\xb5",
        12544: b"\x03\xca|X\xb5",
        44032: b"\x85#\xc5O\xb5",
        9728: b"\x0cP}X\xb5",
        4096: b"\x8b\xcc\xd4Y\xb5",
        4608: b"*\xcbwT\xb5",
        25856: b"~\x97zV\xb5",
        5888: b"\x1f\xc4)Y\xb5",
        27904: b"n\xf1\x87^\xb5",
        36608: b"\x05\x074_\xb5",
        35328: b"\xa3\xb7\xd6Y\xb5",
    },
    service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
)

SERVICE_INFO_3 = BluetoothServiceInfo(
    name="SensorPush HTP.xw 241",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        6400: b"\xe9\x95\xbbK\xb5",
        17667: b"\xad\x00\x01\x00\x00",
        9984: b"\xe7ICo\xb5",
        1536: b"\x80\x8a\xf58\xb5",
        20992: b"\xa2L\x87^\xb5",
        31744: b"\x8bl\x0fK\xb5",
        30208: b"\x03\xd1\x1c\xc9\xb5",
        23040: b"\xe7\x82\xd6Y\xb5",
        41728: b"(rxT\xb5",
        54528: b"\xb2\x81\xb9\xbf\xb5",
        768: b"\xfc\xba^F\xb5",
        42752: b"T\x7f\xc8\xc9\xb5",
        15360: b"\xb0SLu\xb5",
        5120: b":V.a\xb5",
        10752: b"z\xa6'W\xb5",
        10240: b",\xff\xceY\xb5",
        12288: b"$\xe3\x00|\xb5",
        59136: b".2\x8eb\xb5",
        32768: b"i\xbd3_\xb5",
        14848: b"\xe6\x9a\xbf\xc3\xb5",
        54272: b"\xf3\xce\x10\xc1\xb5",
        45056: b"\x12c\xd4Y\xb5",
        18432: b"\xce\xcfcJ\xb5",
        14592: b"o\x02\xd2[\xb5",
        48384: b"\xdd\x02\xab?\xb5",
        51456: b"\xda~\xbaK\xb5",
        2048: b"\xe5\xab\xa6x\xb5",
        10496: b"I\xe9\x13\xc3\xb5",
        56320: b"M\x154_\xb5",
        23296: b"\x1a|\xc2\xc5\xb5",
        46592: b"[\xce\x0cI\xb5",
        50688: b"\xc0\t{V\xb5",
        42240: b"\xb7LY\xb8\xb5",
        8448: b"H\xccc\x85\xb5",
        62208: b"|X&W\xb5",
        31488: b"\xdf\x1c-\x9a\xb5",
        61440: b"\x97\xa1\xb6\xbd\xb5",
        56064: b">\xf2\x04\xb9\xb5",
        3584: b"!\xf4]F\xb5",
        8192: b"~\xc3c\x85\xb5",
        50176: b"\x99\xc8\xaeC\xb5",
        12032: b"\x0fn\xad\xb7\xb5",
        25088: b"\xe8\xe7iN\xb5",
        7424: b"w?_\xbc\xb5",
        22016: b"u\x8c\nG\xb5",
        23552: b"\x01f~X\xb5",
        16128: b"\xe6\x19\xfc\xb2\xb5",
        17408: b"\x1f*\xc6S\xb5",
        54016: b"'\xde\xaa\xb5\xb5",
        52992: b"\xc2\xd9\xaa\xb5\xb5",
        42496: b"\xb4\xf7\xbaK\xb5",
        36096: b"i\x8aJ\xae\xb5",
        41216: b"\xda\xfc\xd9a\xb5",
        33536: b"\xe7\x05\xc2O\xb5",
        65280: b"\x8a\x97vT\xb5",
        22784: b"\xe8\xb7\x98\xa9\xb5",
        62464: b"\xbd\x08\xff>\xb5",
        57088: b"\xcc\xdctV\xb5",
        60160: b"9\xb6\x7f^\xb5",
        512: b"\x9c\x1e\tG\xb5",
        39424: b"~\x1d~X\xb5",
        17920: b".\x1d\xb0C\xb5",
        11008: b"\x1e*\xb2\x80\xb5",
        61184: b"O\x9a\x92\xa5\xb5",
        4352: b"h\x98x\x93\xb5",
        22272: b"\n\x83V@\xb5",
        28160: b"\x19\x10\xa7=\xb5",
        16896: b"\xc7\xcb\xb8I\xb5",
        46080: b"7c\xc9U\xb5",
        27136: b"\x02\xb4~X\xb5",
        29184: b"D\x1b\x83`\xb5",
        20224: b"+\x89\xe1\xa0\xb5",
        34560: b'"l\x95\xa7\xb5',
        63744: b"\x03\xd8\xabA\xb5",
        62720: b"\xc3B\x85\\\xb5",
        30464: b"k#\xceU\xb5",
        65024: b"\xb6T>\xa6\xb5",
        47360: b"\xa8\xa9\x95\xa7\xb5",
        21504: b"\x17\xc5\x98\xa9\xb5",
        7168: b"\xe0\xa7\x85b\xb5",
        38400: b"Y\xe4~X\xb5",
        40704: b"3_\t\x82\xb5",
        27392: b"E6\xac|\xb5",
        34304: b"\xa1P\xc0M\xb5",
        30720: b"xFOw\xb5",
        45568: b"j\xbeR>\xb5",
        26368: b"\xf7Z\x0fK\xb5",
        36864: b"\x84\xee\xdb]\xb5",
        63232: b"\xbajcJ\xb5",
        35840: b"\xfc\xff]\x81\xb5",
        32256: b"u\xc3\x04C\xb5",
        43264: b'\xf90"S\xb5',
        40960: b"B!\xb8\x84\xb5",
        26880: b"\xf4)6\xa0\xb5",
        51712: b"\xbc\xec,[\xb5",
        8704: b"\x91O4e\xb5",
        54784: b"\xc08\x1dU\xb5",
        34816: b"#\xcd\x17Q\xb5",
        256: b"`)~\x97\xb5",
        24576: b"1\xa1\x17Q\xb5",
        21248: b"\x00\x7f\x84\x9b\xb5",
        19712: b"\x90$-\x9a\xb5",
        28416: b"%\xed\xe1\xa0\xb5",
        52736: b"\x83\xd7aH\xb5",
        55040: b"l\x96xV\xb5",
        53760: b"\x90\x91\xb7I\xb5",
        26112: b"\xfa\xe0\xfdy\xb5",
        38144: b"O\x10~X\xb5",
        53504: b"r\x19\xd5Y\xb5",
        61952: b"\xc7\x93)Y\xb5",
        51200: b"\xc3%$\x94\xb5",
        58112: b"\xdcQ\xb5G\xb5",
        57600: b"BB*\x98\xb5",
        37632: b"D\xcf\x1dU\xb5",
        1792: b"\x80B\x15\x8a\xb5",
        56832: b"\x05\xa9l\x8b\xb5",
        25344: b"!\x00\xc7\x8e\xb5",
        44288: b"\xc1\x06\xc4\x8c\xb5",
        6912: b"yX\x15\x8a\xb5",
        43520: b"\x18\x7f\x06\x80\xb5",
        33792: b"@1\x0cI\xb5",
        29952: b"]a\xaa?\xb5",
        40448: b"\xa4\x05\xdaa\xb5",
        32512: b"\xdd\xc9\x8bf\xb5",
        37376: b")\xd6\xeeo\xb5",
        7680: b"Ca\x15\x8a\xb5",
        12800: b"\x8b\xcff\x87\xb5",
        36352: b"\xd6\xfb\xbd\x88\xb5",
        56576: b"\t+Lu\xb5",
        41984: b"x\x18\xb8\x84\xb5",
        13056: b"\xd5\xa2\xd0U\xb5",
        44800: b"\xaa\xdaZ\x7f\xb5",
        60928: b"&*\xaf~\xb5",
        13824: b"bx\xafC\xb5",
        49152: b"c\xd1\xaeC\xb5",
        29696: b"\t\xb8\xb5G\xb5",
        20736: b"\xc5/\xb8I\xb5",
        16384: b"l\xd1\x1fS\xb5",
        30976: b"\xe5\x1a\xb8I\xb5",
        15872: b"\x1bCU{\xb5",
        31232: b"\xaeo\xa7=\xb5",
        29440: b"^\xc6\xb1E\xb5",
        17664: b"\xb8\xf4\x00|\xb5",
        14080: b"\x10\xed\xfdy\xb5",
        20480: b"\x05g\x11K\xb5",
        51968: b"\x02Y[\x7f\xb5",
        48128: b"\xb1*\tG\xb5",
        60672: b"\xbc\xba\xfe>\xb5",
        11776: b'"\xac\x91j\xb5',
        60416: b"\x12LYB\xb5",
        14336: b"Pm*Y\xb5",
        55808: b"\xdcR4e\xb5",
        50944: b"\xa9\xd8]F\xb5",
        7936: b"u8\xc1O\xb5",
        17152: b"\xc3\xc8\x15O\xb5",
        24320: b'\xdd\xf7"Y\xb5',
        37120: b">\x99.a\xb5",
        15104: b"c\xc7.a\xb5",
        2304: b"\x9d\x8a\xd7_\xb5",
        48896: b"\xa7\xd6L:\xb5",
        2560: b"\x1br}X\xb5",
        55296: b"\xe8\x104_\xb5",
        19968: b"\xd0\x85\x0fK\xb5",
        37888: b"w\xaabJ\xb5",
        27648: b"j\xb6nR\xb5",
        46848: b"\xd9\xb1}\\\xb5",
        64768: b"q\x17\xc6S\xb5",
        64512: b"{n&[\xb5",
        4864: b"H\xe1lP\xb5",
        34048: b"\xd6\xdbhN\xb5",
        5376: b"\xb4\xd3\x18O\xb5",
        55552: b"\n\xf4eL\xb5",
        24064: b"\xd7\xf2nR\xb5",
        53248: b"j\xdc W\xb5",
        49408: b"D\x06\x18Q\xb5",
        11520: b",\x82\xb7G\xb5",
        47616: b"G\xbb\xbaK\xb5",
        18944: b"\xf5miN\xb5",
        64256: b"\x1eY\xd9[\xb5",
        18176: b"i|ZD\xb5",
        50432: b"\xa8\x8a]F\xb5",
        47104: b"\xdd\xb7\xc4Q\xb5",
        58624: b"W\x02dJ\xb5",
        15616: b"\xe3\x13gL\xb5",
        52480: b"p{\xc4Q\xb5",
        48640: b"\xd85oP\xb5",
        47872: b"\xbb\x1bYB\xb5",
        59904: b"\x8c_\xfe>\xb5",
        52224: b"c\xd4R>\xb5",
        40192: b"Y\x95\x01A\xb5",
        39680: b"\x03\xea\xfb<\xb5",
        3072: b"\x8c}V@\xb5",
        18688: b"\xde*\xf9:\xb5",
        21760: b"o\x87V@\xb5",
        9216: b"\xf1\xd0_F\xb5",
        35584: b"'W\xc6Q\xb5",
        19456: b"%\x01lN\xb5",
        58880: b"\xf4\xe9uT\xb5",
        59392: b"D\x90\xc7Q\xb5",
        64000: b"\xc3\xac\xcdU\xb5",
        28928: b"x\xe0\xc7Q\xb5",
        12544: b"\x03\xca|X\xb5",
        44032: b"\x85#\xc5O\xb5",
        9728: b"\x0cP}X\xb5",
        4096: b"\x8b\xcc\xd4Y\xb5",
        4608: b"*\xcbwT\xb5",
        25856: b"~\x97zV\xb5",
        5888: b"\x1f\xc4)Y\xb5",
        27904: b"n\xf1\x87^\xb5",
        36608: b"\x05\x074_\xb5",
        35328: b"\xa3\xb7\xd6Y\xb5",
        45312: b"\xd60.[\xb5",
    },
    service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
)


def test_init_class():
    BluetoothData()


def test_changed_manufacturer_data():
    data = BluetoothData()
    first_data = data.changed_manufacturer_data(SERVICE_INFO_1)
    assert first_data == {}

    second_data = data.changed_manufacturer_data(SERVICE_INFO_2)
    assert second_data == {43264: b'\xf90"S\xb5'}

    third_data = data.changed_manufacturer_data(SERVICE_INFO_3)
    assert third_data == {45312: b"\xd60.[\xb5"}


SERVICE_INFO_WITH_EXCLUDE_1 = BluetoothServiceInfo(
    name="xBBQ",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        1: b"\x00\x00,\x11\x00\x00m\xd3\x12\x01\x12\x01",
        2: b"\x00\x00,\x11\x00\x00m\xd3\x80\x03\x80\x03",
    },
    service_uuids=["0000fff0-0000-1000-8000-00805f9b34fb"],
)
SERVICE_INFO_WITH_EXCLUDE_2 = BluetoothServiceInfo(
    name="xBBQ",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        1: b"\x00\x00,\x11\x00\x00m\xd3\x11\x01\x12\x01",
        2: b"\x00\x00,\x11\x00\x00m\xd3\x80\x03\x80\x03",
    },
    service_uuids=["0000fff0-0000-1000-8000-00805f9b34fb"],
)
SERVICE_INFO_WITH_EXCLUDE_3 = BluetoothServiceInfo(
    name="xBBQ",
    address="61DE521B-F0BF-9F44-64D4-75BBE1738105",
    rssi=-63,
    service_data={},
    source="local",
    manufacturer_data={
        1: b"\x00\x00,\x11\x00\x00m\xd3\x11\x01\x12\x01",
        2: b"\x00\x00,\x11\x00\x00m\xd3\x80\x03\x80\x03",
    },
    service_uuids=["0000fff0-0000-1000-8000-00805f9b34fb"],
)


def test_changed_manufacturer_data_with_exclude():
    data = BluetoothData()
    first_data = data.changed_manufacturer_data(SERVICE_INFO_WITH_EXCLUDE_1, {2})
    assert first_data == {1: b"\x00\x00,\x11\x00\x00m\xd3\x12\x01\x12\x01"}

    second_data = data.changed_manufacturer_data(SERVICE_INFO_WITH_EXCLUDE_2, {2})
    assert second_data == {1: b"\x00\x00,\x11\x00\x00m\xd3\x11\x01\x12\x01"}

    third_data = data.changed_manufacturer_data(SERVICE_INFO_WITH_EXCLUDE_3, {2})
    assert third_data == {1: b"\x00\x00,\x11\x00\x00m\xd3\x11\x01\x12\x01"}


def test_changed_manufacturer_data_raw():
    service_info = make_bluetooth_service_info(
        name="SensorPush HT.w 0CA1",
        manufacturer_data={1: b""},  # anything here
        service_data={},
        service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
        address="aa:bb:cc:dd:ee:ff",
        rssi=-60,
        source="local",
        raw=b"\x06\xff\x04\x9a\xc9\xa5\x46",
    )

    data = BluetoothData()
    assert data.changed_manufacturer_data(service_info, {2}) == {39428: b"\xc9\xa5F"}
    assert data.changed_manufacturer_data(service_info, {39428}) == {1: b""}
    assert data.changed_manufacturer_data(service_info) == {39428: b"\xc9\xa5F"}


def test_changed_manufacturer_data_raw_multiple():
    service_info = make_bluetooth_service_info(
        name="SensorPush HT.w 0CA1",
        manufacturer_data={1: b"", 2: b""},  # anything here
        service_data={},
        service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
        address="aa:bb:cc:dd:ee:ff",
        rssi=-60,
        source="local",
        raw=b"\x06\xff\x04\x9a\xc9\xa5\x46",
    )

    data = BluetoothData()
    assert data.changed_manufacturer_data(service_info, {2}) == {39428: b"\xc9\xa5F"}
    assert data.changed_manufacturer_data(service_info, {39428}) == {}
    assert data.changed_manufacturer_data(service_info) == {39428: b"\xc9\xa5F"}


def test_single_changed_manufacturer_data_raw():
    service_info = make_bluetooth_service_info(
        name="SensorPush HT.w 0CA1",
        manufacturer_data={39428: b"\xc9\xa5F"},
        service_data={},
        service_uuids=["ef090000-11d6-42ba-93b8-9dd7ec090ab0"],
        address="aa:bb:cc:dd:ee:ff",
        rssi=-60,
        source="local",
        raw=b"\x06",
    )

    data = BluetoothData()
    assert data.changed_manufacturer_data(service_info, {2}) == {39428: b"\xc9\xa5F"}
    assert data.changed_manufacturer_data(service_info, {39428}) == {}
    assert data.changed_manufacturer_data(service_info) == {39428: b"\xc9\xa5F"}
